/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TW_SEEK implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "ccache.h"
#include "cas.h"
#include "cal_alert.h"
#include "decoder_obj.h"
#include "cm.h"
#include "report_obj.h"

#include "seek.h"
#include "seek_content.h"
#include "tw_seek_content.h"
#include "tw_seek.h"
#include "_tw_seek.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/



/*****************************************************************************
*   eAvailableToRegister
*
* This API is used to query if content playing on a channel is available for
* registration for seek alerts.
*
* Inputs:
*
*   hDecoder   A handle to a valid Decoder whose Traffic/Weather Seek Service
*              the caller wishes to know if content is available to be
*              registered with.
*   tChannelId  The id of the channel whose content the caller wished to know
*               the availability to register
*   peTrafficWeatherAvailablility - destination pointer where the availability
*                                   of registration will be copied
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAvailableToRegister (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    SEEK_AVAILABILTY_ENUM *peTrafficWeatherAvailablility
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    // did the caller provide us with a valid pointer?
    if ( peTrafficWeatherAvailablility == NULL )
    {
        // Error!
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock the object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // Get this seek service's handle from the provided DECODER
        eReturnCode = DECODER_eSeekService(hDecoder,
                          SEEK_SERVICE_TRAFFIC_WEATHER, &hSeekService);

        // did the caller provide us with a valid pointer?
        if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
        {
            CD_OBJECT hCDO;
            CID_OBJECT hMarketCID;
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            hCDO = hGetCDOAndCids(
                       hDecoder,
                       tChannelId,
                       &hMarketCID
                           );

            if (hCDO == CD_INVALID_OBJECT)
            {
                // we got an invalid cdo so we cannot identify content
                // so seeks are currently unavailable for this channel.
                *peTrafficWeatherAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;

            }
            else
            {
                // we were able to retrieve cids

                do
                {
                    BOOLEAN bRegistered = FALSE;

                    // did we retrieve a cid?
                    if (hMarketCID == CID_INVALID_OBJECT)
                    {
                        // no, so we cannot identify the market
                        *peTrafficWeatherAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;
                        break;
                    }

                    // we can identify the market, now see if the market is
                    // registered already
                    bRegistered = CAL.bExists(
                                      psObj->hSeekList,
                                      hMarketCID);

                    if (bRegistered == TRUE)
                    {
                        // already registered
                        *peTrafficWeatherAvailablility =
                            SEEK_AVAILIBILITY_ALREADY_REGISTERED;
                    }
                    else
                    {
                        // not already registered
                        *peTrafficWeatherAvailablility = SEEK_AVAILABILTY_AVAILABLE;
                    }
                }
                while(0);
            }
        }
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    else
    {
        // couldn't lock object
        eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    }


    return eReturnCode;
}

/*****************************************************************************
*   eRegister
*
* This API is used to register content for Traffic/Weather Seek Alerts
*
* Inputs:
*
*   hDecoder   A handle to a valid Decoder whose Traffic/Weather Seek Service
*              the caller wishes to register content with
*   tChannelId      The id of the channel on which content to be registered is
*                   playing on
*
* Outputs:
*   phSeekContent  A pointer to store the registered content object.
*   Returns a error code.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegister (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Lock the service
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        do
        {
            CD_OBJECT hCDO;
            CID_OBJECT hMarketCID;
            SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
            SEEK_SERVICE_OBJECT_STRUCT *psObj;

            eReturnCode =
                DECODER_eSeekService(hDecoder,
                    SEEK_SERVICE_TRAFFIC_WEATHER, &hSeekService);

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            psObj = (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // get the channels cdo and its cids
            hCDO = hGetCDOAndCids(
                        hDecoder,
                        tChannelId,
                        &hMarketCID
                            );

            if (hCDO == CD_INVALID_OBJECT)
            {
                // failure to get cids
                eReturnCode =  SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
                break;
            }

            eReturnCode = eRegisterMkt(psObj, hMarketCID, phSeekContent);

        } while (0);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}


/*****************************************************************************
*   eRegisterMarket
*
*
*
* This API is used to register a specific market for
* Traffic/Weather Seek Alerts
*
* Inputs:
*
*   hDecoder   A handle to a valid Decoder whose Traffic/Weather Seek Service
*              the caller wishes to register content with
*   hReportId  The id to register
*
* Outputs:
*   phSeekContent  A pointer to store the registered content object.
*   Returns a error code.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegisterMarket (
    DECODER_OBJECT hDecoder,
    CID_OBJECT hReportId,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Lock the service
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        eReturnCode =
            DECODER_eSeekService(hDecoder, SEEK_SERVICE_TRAFFIC_WEATHER, &hSeekService);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            eReturnCode = eRegisterMkt(psObj, hReportId, phSeekContent);
        }
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*   bHandleSeekAlert
*
* This is the specific seek service interface function used to
* handle seek alerts
*
* Returns TRUE if a Seek Alert Event should be generated. FALSE if not
*****************************************************************************/
static BOOLEAN bHandleSeekAlert(
    SEEK_SERVICE_OBJECT hSeek,
    CAL_ALERT_OBJECT hAlert
        )
{
    BOOLEAN bSetDecoderSeekAlertEvent = FALSE;
    CAL_CONTENT_OBJECT hContent;

    // get the content and channel that caused the alert
    hContent = CAL_ALERT.hContent(hAlert);

    // should we allow the seek alert
    bSetDecoderSeekAlertEvent = bSeekAlertAllowed(
                                 hSeek,
                                 hContent
                                     );

    return bSetDecoderSeekAlertEvent;
}

/*****************************************************************************
*   bSeekAlertAllowed
*
* This function implements the rules to determine if the application should be
* informed of a content match via a seek alert
*
* Returns TRUE is a Seek Alert is allowed, FALSE if it is not
*****************************************************************************/
static BOOLEAN bSeekAlertAllowed(
    SEEK_SERVICE_OBJECT hSeek,
    CAL_CONTENT_OBJECT hContent
        )
{
    // we start out assuming the alert is allowed, then we'll check the rules
    // to see if it is not allowed
    BOOLEAN bValid, bEnabled, bAlertAllowed = TRUE;

    SEEK_SERVICE_OBJECT_STRUCT *psObj;

    bValid = SMSO_bValid((SMS_OBJECT)hSeek);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    psObj = (SEEK_SERVICE_OBJECT_STRUCT*)hSeek;

    do
    {
        // 1. The service must be enabled for a seek alert to be allowed
        printf(TW_SEEK_OBJECT_NAME": Service is :%s\n",
               psObj->eState == SEEK_STATE_ENABLED ? "ENABLED" : "NOT_ENABLED");
        // is service enabled?
        if (psObj->eState != SEEK_STATE_ENABLED)
        {
            // service is disabled, so seek alert is not allowed
            bAlertAllowed = FALSE;
            break;
        }

        // 2. The registered item must be enabled for a seek alert to be allowed
        bEnabled = SEEK_CONTENT.bEnabled(hContent);
        printf(TW_SEEK_OBJECT_NAME":Item is :%s\n",
               bEnabled == TRUE ? "ENABLED" : "DISABLED");
        // is item enabled?
        if (bEnabled == FALSE)
        {
            // item is disabled, so seek alert is not allowed
            bAlertAllowed = FALSE;
            break;
        }

    } while (0);

    printf(TW_SEEK_OBJECT_NAME": Seek Alert %s \n",
           bAlertAllowed == TRUE ? "ALLOWED" : "NOT ALLOWED");

    if (bAlertAllowed == TRUE)
    {
        // TW Seeks get disabled after a match is reported.
        SEEK_CONTENT.eDisable(hContent);
    }

    return bAlertAllowed;
}

/*****************************************************************************
*   bInit
*
* This function implements the service specific initialization
*
* Returns TRUE if successful, FALSE if not
*****************************************************************************/
static BOOLEAN bInit(
    SEEK_SERVICE_OBJECT hSeek
        )
{
    BOOLEAN bValid, bReturn = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hSeek);
    if (bValid == TRUE)
    {
        SEEK_SERVICE_OBJECT_STRUCT *psObj;

        psObj = (SEEK_SERVICE_OBJECT_STRUCT*)hSeek;

        bReturn = bRegisterContentFromConfigFile(psObj);
    }

    return bReturn;
}

/*****************************************************************************
*   eRegisterMkt
*
* This local function registers the market with the service
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegisterMkt(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    CID_OBJECT hMarketCID,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bOk;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    TW_SEEK_REPORT_ITERATOR_STRUCT sIterator;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem = NULL;
    STRING_OBJECT hMarketIdString = STRING_INVALID_OBJECT,
                  hMarketNameString = STRING_INVALID_OBJECT;

    sIterator.hCid = hMarketCID;
    sIterator.phMarketId = &hMarketIdString;
    sIterator.phMarketName = &hMarketNameString;
    sIterator.bFound = FALSE;

    do
    {
        bOk = REPORT.bIterateContent(
            bReportIteratorCallback,
            &sIterator);
        if (bOk == FALSE)
        {
           // couln't iterate REPORTS
           break;
        }

		if (sIterator.bFound == FALSE)
        {
           // couln't find a match for market cid
           break;
        }

        psItem = psRegisterContent(
            psObj,
            hMarketCID,
            hMarketIdString,
            hMarketNameString,
            &eReturnCode,
            phSeekContent
                );

        if (psItem == NULL)
        {
            // couldn't register
            break;
        }

        // add registered item to config file
        bOk = bAddToConfigFile(
                  psObj,
                  psItem,
                  hMarketIdString,
                  hMarketNameString
                      );
        if (bOk == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

    } while (0);

    if (hMarketIdString != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hMarketIdString);
    }

    if (hMarketNameString != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hMarketNameString);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hGetCDOAndCids
*
*****************************************************************************/
static CD_OBJECT hGetCDOAndCids(
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    CID_OBJECT *phMarketCID
        )
{
    BOOLEAN bReturn = FALSE;
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT hChannel;
    CD_OBJECT hCDO;

    // did caller give us a valid id?
    if (tChannelId == CHANNEL_INVALID_ID)
    {
        // no, so instead we use the currently tuned channel id
        tChannelId = DECODER.tCurrentChannelId(hDecoder);
    }

    // From the DECODER handle, determine the channel cache handle
    hCCache = DECODER_hCCache(hDecoder);

    // get the channel obj for this requested chan id
    hChannel = CCACHE_hChannelFromIds (
        hCCache,
        SERVICE_INVALID_ID,
        tChannelId,
        FALSE
            );

    // get the channel's cdo
    hCDO = CHANNEL.hCDO(hChannel);
    if (hCDO != CD_INVALID_OBJECT)
    {
        // get the cids from the cdo
        bReturn = bRetrieveCidsForSeeks(
                         hCDO,
                         phMarketCID
                             );
        if (bReturn == FALSE)
        {
            // couldn't get cids, so this content is not valid for us
            hCDO = CD_INVALID_OBJECT;
        }
    }

    return hCDO;
}

/*****************************************************************************
*
*    bRetrieveCidsForSeeks
*
*    This is a local function which looks at a CDO and extracts the various
*    CIDs that can be used for registering.
*
*****************************************************************************/
static BOOLEAN bRetrieveCidsForSeeks(
    CD_OBJECT hCDO,
    CID_OBJECT *phMarketCID
        )
{
    CDO_TYPE_ENUM eCDOType;

    // clear the values
    *phMarketCID = CID_INVALID_OBJECT;

    // get the type of cdo
    eCDOType = CDO.eType(hCDO);

    // extract cids based on cdo type
    switch (eCDOType)
    {
        case CDO_REPORT:
        {
            // reports can be registered for traffic/weather seeks
            *phMarketCID = REPORT.hId(hCDO);
        }
        break;

        case CDO_NEWS:
        case CDO_ENTERTAINMENT:
        case CDO_MUSIC:
        case CDO_SPORTS:
        case CDO_NON_PROGRAM:
        case CDO_UNKNOWN:
        case CDO_INVALID:
        default:
        {
            // can't seek on these types of cdos cdo
            return FALSE;
        }
    }

    // if the content didn't have a valid market id cid
    if (*phMarketCID == CID_INVALID_OBJECT)
    {
        puts(TW_SEEK_OBJECT_NAME": No Market CID.");
    }

    return TRUE;
}

/*****************************************************************************
*
*    bAddToConfigFile
*
*****************************************************************************/
static BOOLEAN bAddToConfigFile(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem,
    STRING_OBJECT hMarketIdString,
    STRING_OBJECT hMarketNameString
        )
{
    TAG_OBJECT hTag;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    if (psObj->hTag == TAG_INVALID_OBJECT)
    {
        // must not be a config file
        return TRUE;
    }

    // add a new Registered Content Tag
    // this will be the parent tag of all tags needed to persist an
    // instance of content registered with this service
    eReturn = TAG_eAdd(
          SEEK_REGISTERED_CONTENT,
          psObj->hTag,
          &hTag,
          NULL
              );

    if ( eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // add all the tags needed to persist an
        // instance of at seek content
        eReturn = eAddToConfigFile (
            hTag,
            psItem,
            hMarketIdString,
            hMarketNameString
            );
    }

    // successful?
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // yes
        return TRUE;
    }

    // no
    return FALSE;
}

/*****************************************************************************
*
*    psRegisterContent
*
*****************************************************************************/
static SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisterContent(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    CID_OBJECT hMarketCID,
    STRING_OBJECT hMarketIdString,
    STRING_OBJECT hMarketNameString,
    SMSAPI_RETURN_CODE_ENUM *peReturn,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem = NULL;
    *peReturn = SMSAPI_RETURN_CODE_ERROR;

    // create a seek content item of type that is applicable to the
    // traffic/weather service
    psItem = SEEK_CONTENT_psCreateRegisteredItem((SEEK_SERVICE_OBJECT)psObj, FALSE);
    if (psItem != NULL)
    {
        // initialize the service specific content portion
        TW_SEEK_CONTENT_vSetServiceSpecificInfo(
            (void *)psItem,
            hMarketCID,
            (SEEK_SERVICE_OBJECT)psObj
                );

        // we have all the information we need, now we can
        // register with the CAL
        *peReturn = CAL.eRegister(
            psObj->hSeekList,
            hMarketCID,
            (void *)psItem,
            TW_SEEK_CAL_REGISTRATION_OPTIONS,
            hMarketNameString,
            hMarketIdString,
            (CAL_CONTENT_OBJECT *)phSeekContent
                );

        if (*peReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // we couldn't successfully register for some reason, destroy the
            // item we created
            SEEK_CONTENT_vDestroyRegisteredItem(psItem);
            psItem = NULL;
        }
    }

    return psItem;
}

/*****************************************************************************
*
*   bRegisterContentFromConfigFile
*
*****************************************************************************/
static BOOLEAN bRegisterContentFromConfigFile(
    SEEK_SERVICE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bContinue;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;

    if (psObj->hTag == TAG_INVALID_OBJECT)
    {
        // must not be a config file
        return TRUE;
    }

    eResult = TAG_eFirstChild(psObj->hTag, &hTag);
    if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TW_SEEK_OBJECT_NAME": Failed to get Tag: %s (#%d)",
                SMSAPI_DEBUG_pacReturnCodeText(eResult), eResult);
        return FALSE;
    }

    // the seek service object holds the parent tag of all registered content
    // tags. Iterate through all the child tags, and for each registered
    // content tag extract the information and register and configure.
    while (hTag != TAG_INVALID_OBJECT)
    {
        bContinue = bRegisteredContentTagIterator(hTag, psObj);
        if (bContinue == FALSE)
        {
            break;
        }

        eResult = TAG_eNextSibling(hTag, &hTag);
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TW_SEEK_OBJECT_NAME": Failed to get Tag: %s (#%d)",
                    SMSAPI_DEBUG_pacReturnCodeText(eResult), eResult);
            break;
        }
    }

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bRegisteredContentTagIterator
*
*****************************************************************************/
static BOOLEAN bRegisteredContentTagIterator(
    TAG_OBJECT hTag,
    SEEK_SERVICE_OBJECT_STRUCT *psObj
        )
{
    STRING_OBJECT hTagName;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    CID_OBJECT hCID = CID_INVALID_OBJECT;
    STRING_OBJECT hMarketIdString = STRING_INVALID_OBJECT,
                  hMarketNameString = STRING_INVALID_OBJECT,
                  hMarketIdStringNew = STRING_INVALID_OBJECT,
                  hMarketNameStringNew = STRING_INVALID_OBJECT,
                  hMarketIdStringTag = STRING_INVALID_OBJECT,
                  hMarketNameStringTag = STRING_INVALID_OBJECT;

    // get the tag name
    hTagName = TAG_hTagName(hTag);
    // see if the tag name is one that we're looking for
    if (STRING.n16CompareCStr(SEEK_REGISTERED_CONTENT, 0, hTagName) == 0)
    {
        do
        {
            BOOLEAN bOk;
            TW_SEEK_REPORT_ITERATOR_STRUCT sIterator;

            // get the CID
            eReturn = eGetCidFromTags(
                          hTag,
                          &hCID
                              );

            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            eReturn = eGetMarketNameIdTextFromTags(
                hTag,
                &hMarketIdStringTag,
                &hMarketNameStringTag
                    );
            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            hMarketIdString = hMarketIdStringTag;
            hMarketNameString = hMarketNameStringTag;

            // Prepare iterator struct for searching
            sIterator.bFound = FALSE;
            sIterator.hCid = hCID;
            sIterator.phMarketId = &hMarketIdStringNew;
            sIterator.phMarketName = &hMarketNameStringNew;

            // Search for a corresponding report by cid
            bOk = REPORT.bIterateContent(
                bReportIteratorCallback,
                &sIterator
                    );

            if (bOk == FALSE)
            {
                // Couldn't iterate reports
                break;
            }

            // At this point, hMarketIdStringTag and hMarketNameStringTag shall
            // be a valid strings. Otherwise we would have never got here.

            if (sIterator.bFound == TRUE)
            {
                N16 n16Equal;

                do
                {
                    n16Equal = STRING.n16Compare(hMarketIdStringTag, hMarketIdStringNew, TRUE);
                    if (n16Equal != 0)
                    {
                        hMarketIdString = hMarketIdStringNew;
                        hMarketNameString = hMarketNameStringNew;

                        break;
                    }

                    n16Equal = STRING.n16Compare(hMarketNameStringTag, hMarketNameStringNew, TRUE);
                    if (n16Equal != 0)
                    {
                        hMarketIdString = hMarketIdStringNew;
                        hMarketNameString = hMarketNameStringNew;

                        break;
                    }

                } while (FALSE);

                if (n16Equal != 0)
                {
                    eReturn = eSetMarketNameIdTextToTags(
                        hTag,
                        hMarketIdString,
                        hMarketNameString
                            );

                    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        break;
                    }
                }
            }

            psItem = psRegisterContent(
                         psObj,
                         hCID,
                         hMarketIdString,
                         hMarketNameString,
                         &eReturn,
                         NULL
                             );

            if (psItem != NULL)
            {
                psItem->bEnabled = FALSE; // when starting up we should alwyas be disabled
                psItem->hTag = hTag;
            }

        } while (0);
    }

    // destroy anything allocated
    if (hCID != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hCID);
    }

    if (hMarketIdStringTag != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hMarketIdStringTag);
    }
    if (hMarketNameStringTag != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hMarketNameStringTag);
    }
    if (hMarketIdStringNew != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hMarketIdStringNew);
    }
    if (hMarketNameStringNew != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hMarketNameStringNew);
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   eGetCidsFromTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetCidFromTags(
    TAG_OBJECT hTag,
    CID_OBJECT *phCID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hChildTag;
    STRING_OBJECT hCIDString = STRING_INVALID_OBJECT,
                  *phString;
    size_t tSize, tStringSize = 0;
    char *pacCStrBuffer = NULL;
    void *pvVoid;

    *phCID = CID_INVALID_OBJECT;

    // get market cid tag
    eReturn = TAG_eGet(
        TW_SEEK_CID,
        hTag,
        &hChildTag,
        NULL,
        FALSE
            );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // get the market cid
        phString = &hCIDString;
        tSize = sizeof(STRING_OBJECT);
        eReturn = TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize);
        tStringSize = STRING.tSize(hCIDString);
        tStringSize++; // add one for NULL

        // allocate memory for the read buffer
        pacCStrBuffer =
            (char *) SMSO_hCreate(
                        TW_SEEK_OBJECT_NAME":CSTR",
                        tStringSize,
                        SMS_INVALID_OBJECT,
                        FALSE // No lock necessary
                         );
        if (pacCStrBuffer != NULL)
        {
            if (hCIDString != STRING_INVALID_OBJECT)
            {
                // copy to a c-string so that we can create a CID from it

                STRING.tCopyToCStr(hCIDString, pacCStrBuffer, tStringSize);
                pvVoid = (void *)pacCStrBuffer;
                // Read from memory assumes memory provided as an input is
                // NULL terminated. It is in fact since we just did a
                // string copy above into a buffer we know is at least one
                // larger than the source (thus it will be NULL terminated).
                *phCID = CID_hReadFromMemory((void const**)&pvVoid);
                STRING.vDestroy(hCIDString);
            }

            // we're done with the buffer
            SMSO_vDestroy((SMS_OBJECT)pacCStrBuffer);
        }
        else
        {
            // couldn't allocate memory
            eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eGetMarketNameIdTextFromTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetMarketNameIdTextFromTags(
    TAG_OBJECT hTag,
    STRING_OBJECT *phMarketIdText,
    STRING_OBJECT *phMarketNameText
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;

    if ((phMarketIdText == NULL) || (phMarketNameText == NULL))
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // get title text
    eReturn = TAG_eGet(
        TW_SEEK_MARKET_ID_TEXT,
        hTag,
        &hChildTag,
        NULL,
        TRUE
            );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        size_t tSize;
        tSize = sizeof(STRING_OBJECT);

        // get title text
        eReturn = TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phMarketIdText,
            &tSize);

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // get artist text tag
            eReturn = TAG_eGet(
                          TW_SEEK_MARKET_NAME_TEXT,
                          hTag,
                          &hChildTag,
                          NULL,
                          TRUE
                              );
            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // get artist text
                eReturn = TAG_eGetTagValue(
                    hChildTag,
                    TAG_TYPE_STRING,
                    (void **)&phMarketNameText,
                    &tSize);
            }
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eSetMarketNameIdTextToTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetMarketNameIdTextToTags (
    TAG_OBJECT hTag,
    STRING_OBJECT hMarketIdText,
    STRING_OBJECT hMarketNameText
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    do
    {
        TAG_OBJECT hMarketIdTag;
        TAG_OBJECT hMarketNameTag;

        // Verify inputs
        if ((hTag == TAG_INVALID_OBJECT) ||
            (hMarketIdText == STRING_INVALID_OBJECT) ||
            (hMarketNameText == STRING_INVALID_OBJECT))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        eReturnCode = TAG_eGet(
            TW_SEEK_MARKET_ID_TEXT,
            hTag,
            &hMarketIdTag,
            NULL,
            FALSE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eReturnCode = TAG_eSetTagValue(
            hMarketIdTag,
            TAG_TYPE_STRING,
            &hMarketIdText,
            sizeof(STRING_OBJECT),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eReturnCode = TAG_eGet(
            TW_SEEK_MARKET_NAME_TEXT,
            hTag,
            &hMarketNameTag,
            NULL,
            FALSE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eReturnCode = TAG_eSetTagValue(
            hMarketNameTag,
            TAG_TYPE_STRING,
            &hMarketNameText,
            sizeof(STRING_OBJECT),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eWriteCIDsToTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteCIDToTag(
    TAG_OBJECT hParentTag,
    TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psTWSpecificInfo
        )
{
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hCidTag = TAG_INVALID_OBJECT;
    size_t tCIDSize;
    char *pacCIDBuffer;
    void *pvVoid;

    do
    {
        // get the cid sizes
        tCIDSize = CID_tSize(psTWSpecificInfo->hMarketCID);

        // allocate memory for the read buffer
        pacCIDBuffer =
            (char *) SMSO_hCreate(
                         TW_SEEK_OBJECT_NAME":CB",
                         tCIDSize,
                         SMS_INVALID_OBJECT,
                         FALSE // No lock necessary
                          );
        if (pacCIDBuffer == NULL)
        {
            eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
            break;
        }

        pvVoid = pacCIDBuffer;

        // add a tag for the artist cid
        eReturn = TAG_eAdd(
                      TW_SEEK_CID,
                      hParentTag,
                      &hCidTag,
                      NULL
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // create a string containing the market cid
        CID_n32FWriteToMemory(psTWSpecificInfo->hMarketCID, (void **)&pvVoid);
        hValueString = STRING.hCreate(pacCIDBuffer, tCIDSize);
        // set the tag value
        eReturn = TAG_eSetTagValue(
                      hCidTag,
                      TAG_TYPE_STRING,
                      &hValueString,
                      sizeof(STRING_OBJECT),
                      FALSE
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

    } while(0);

    if(pacCIDBuffer != NULL)
    {
        // if we allocated this buffer, delete it
        SMSO_vDestroy((SMS_OBJECT)pacCIDBuffer);
    }

    if (hValueString != STRING_INVALID_OBJECT)
    {
        // if we created the string, delete it
        STRING.vDestroy(hValueString);
    }

    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // something went wrong, so remove the tags we added
        if (TAG_INVALID_OBJECT != hCidTag)
        {
            TAG_eRemove(hCidTag, FALSE);
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eWriteTextToTag
*
* common fxn that can be used for ArtistText, TitleText
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteTextToTag(
    TAG_OBJECT hParentTag,
    const char *pacTagName,
    STRING_OBJECT hText
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hTextTag = TAG_INVALID_OBJECT;

    // add the text tag
    eReturn = TAG_eAdd(
          pacTagName,
          hParentTag,
          &hTextTag,
          NULL
              );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the artist text tag value
        eReturn = TAG_eSetTagValue(
            hTextTag,
            TAG_TYPE_STRING,
            &hText,
            sizeof(STRING_OBJECT),
            FALSE
                );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // something went wrong, so remove the tag we added
            TAG_eRemove(hTextTag, FALSE);
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eAddToConfigFile
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddToConfigFile (
    TAG_OBJECT hParentTag,
    void *pvItem,
    STRING_OBJECT hMarketIdString,
    STRING_OBJECT hMarketNameString
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psTWSpecificInfo;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psGenericItem;

    psGenericItem = (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)pvItem;
    psTWSpecificInfo = (TW_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                           &psGenericItem->uServiceSpecific;

    // verify that the pointer to the data isn't NULL

    if ( psTWSpecificInfo == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        // set the parent tag
        psGenericItem->hTag = hParentTag;

        // write the cid tags
        eReturn = eWriteCIDToTag(hParentTag, psTWSpecificInfo);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write artist text
        eReturn = eWriteTextToTag(
                      hParentTag,
                      TW_SEEK_MARKET_NAME_TEXT,
                      hMarketNameString);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write title text
        eReturn = eWriteTextToTag(
                      hParentTag,
                      TW_SEEK_MARKET_ID_TEXT,
                      hMarketIdString);

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            eReturn = CM_eCommitChangesToFile();
        }

    } while (0);

    return eReturn;
}

/*****************************************************************************
*
*   bReportIteratorCallback
*
*****************************************************************************/
static BOOLEAN bReportIteratorCallback(
    const char *pacId,
    const char *pacName,
    CID_OBJECT hId,
    void *pvContentIteratorCallbackArg
        )
{
    TW_SEEK_REPORT_ITERATOR_STRUCT *psStruct =
        (TW_SEEK_REPORT_ITERATOR_STRUCT *)pvContentIteratorCallbackArg;

    // did we find what we're looking for?
    if (CID.n16Compare(psStruct->hCid, hId) == 0)
    {
        // found the one we're looking for
        STRING_OBJECT hMarketId, hMarketName;

        hMarketId = STRING.hCreate(pacId, 0);
        hMarketName = STRING.hCreate(pacName, 0);
        *(psStruct->phMarketId) = hMarketId;
        *(psStruct->phMarketName) = hMarketName;

        psStruct->bFound = TRUE;

        // done iterating
        return FALSE;
    }

    // keep iterating
    return TRUE;
}
