/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TRAFFIC_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sql_interface_obj.h"
#include "srm_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "alertc_event_obj.h"
#include "location_obj.h"
#include "traffic_msg_obj.h"
#include "traffic_db_constants.h"
#include "dsrl_obj.h"
#include "db_util.h"

#include "traffic_mgr_obj.h"
#include "_traffic_mgr_obj.h"

#include "traffic_interface.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static TRAFFIC_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    TRAFFIC_POSCODE_LOOKUP_CALLBACK vTrafficPosLookUpCallback,
    void *pvPosLookUpCallbackArg,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    TRAFFIC_MGR_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOk;
    DATASERVICE_CREATE_STRUCT sCreate;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    bOk = DATASERVICE_IMPL_bProcessOptions(
        TRAFFIC_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
    // restore stack (push)
    if (bOk == FALSE)
    {
        // Bad options!
        return TRAFFIC_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = TRAFFIC_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(TRAFFIC_MGR_OBJECT_STRUCT);
    sCreate.tDataID = (DATASERVICE_ID)GsTrafficIntf.tDSI;

    // Disable all DMIs for this service
    if (GsTrafficIntf.bConfigureDataFilter == NULL)
    {
        // No control over the radio data filter, so enable all DMIs
        sCreate.bEnableAll = TRUE;
    }
    else
    {
        // Keep the DMIs disabled until we enable the market DMIs of
        // our choosing.
        sCreate.bEnableAll = FALSE;
    }

    // Suggest an OTA buffer size
    sCreate.tSuggestedOTABufferByteSize = GsTrafficIntf.tOTABufferByteSize;

    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = (
        DATASERVICE_EVENT_ALL |
        DATASERVICE_INTERNAL_EVENT_DSRL );

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psObj = (TRAFFIC_MGR_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hCreateNewService( &sCreate );
    if (psObj == NULL)
    {
        // Can't create the service, fail out!
        return TRAFFIC_SERVICE_INVALID_OBJECT;
    }

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        DATASERVICE_EVENT_NONE,
        tEventRequestMask,
        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
        pvAppEventCallbackArg);

    // Save the PosCode look-up callback
    psObj->vTrafficPosLookUpCallback = vTrafficPosLookUpCallback;
    psObj->pvPosLookUpCallbackArg = pvPosLookUpCallbackArg;

    eReturnCode = OSAL.eLinkedListCreate(
               &psObj->hDSRLList,
               TRAFFIC_MGR_OBJECT_NAME":DSRLList",
               (OSAL_LL_COMPARE_HANDLER)n16CompareDSRLDesc,
               (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE)
                   );

    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vUninitObject( psObj, TRUE );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

        return TRAFFIC_SERVICE_INVALID_OBJECT;
    }

    // initialize the interface-specific object, if we need to
    if (GsTrafficIntf.hInitParser != NULL)
    {
        // Make the call to init the broadcast-specific parser
        psObj->hTrafficInterfaceData =
             GsTrafficIntf.hInitParser(
             DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));

        if (psObj->hTrafficInterfaceData ==
            TRAFFIC_INTERFACE_INVALID_OBJECT)
        {
            // Error!
            vUninitObject( psObj, TRUE );
            DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

            return TRAFFIC_SERVICE_INVALID_OBJECT;
        }
    }

    // Initialize the app facing object
    bOk = bInitAppFacingObject(psObj);

    if (bOk == TRUE)
    {
        // The service may now start
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

        if (bOk == FALSE)
        {
            // Error!
            vUninitObject( psObj, TRUE );
            DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

            return TRAFFIC_SERVICE_INVALID_OBJECT;
        }
    }

    return (TRAFFIC_SERVICE_OBJECT)psObj;
}

/*****************************************************************************
*
*   eAddMarket
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddMarket(
    TRAFFIC_SERVICE_OBJECT hTrafficService,
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj = (TRAFFIC_APP_OBJECT_STRUCT *)NULL;
    SMSAPI_RETURN_CODE_ENUM eErrorCode;
    TRAFFIC_LOCID_OBJECT hLocIDCopy = TRAFFIC_LOCID_INVALID_OBJECT;

    do
    {
        TRAFFIC_MARKET tMarket;
        BOOLEAN bValid;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Validate the traffic service handle
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hTrafficService);
        if (FALSE == bValid)
        {
            eErrorCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Validate locid handle
        if (TRAFFIC_LOCID_INVALID_OBJECT == hTrafficLocID  )
        {
            eErrorCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Validate the traffic market
        tMarket = TRAFFIC_LOCID.tMarket(hTrafficLocID);

        if ((0 == tMarket) || (tMarket >= NUM_TRAFFIC_MARKETS))
        {
            // ERROR
            eErrorCode = SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
            break;
        }

        // Attempt to copy this locid
        hLocIDCopy = (TRAFFIC_LOCID_OBJECT)LOCID.hDuplicate(hTrafficLocID);

        if (TRAFFIC_LOCID_INVALID_OBJECT == hLocIDCopy)
        {
            eErrorCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
            break;
        }

        // Get a pointer to the locked app-facing object
        psAppObj = psGetAppFacingObject(hTrafficService);
        if ((TRAFFIC_APP_OBJECT_STRUCT *)NULL == psAppObj)
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // We just add the TRAFFIC_LOCID to the list.
        // We let the linked list api check duplicates for us.
        eReturnCode = OSAL.eLinkedListAdd(
            psAppObj->hMarketFilterList,
            &hEntry, hLocIDCopy);

        // If we already track this market it's not big deal.
        // If there was an error then stop
        if ((OSAL_ERROR_LIST_ITEM_NOT_UNIQUE != eReturnCode) &&
            (OSAL_SUCCESS != eReturnCode))
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Only update the filter if necessary
        if (OSAL_SUCCESS == eReturnCode)
        {
            BOOLEAN bFilterUpdated;

            // Update the filter byte array used by the chipset
            bFilterUpdated = bUpdateFilterForMarket(
                (TRAFFIC_MGR_OBJECT_STRUCT *)hTrafficService,
                tMarket, TRUE);
            if (FALSE == bFilterUpdated)
            {
                // Remove the entry from the LL
                OSAL.eLinkedListRemove(hEntry);

                eErrorCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }

            // Clear this handle so the object doesn't get freed below
            hLocIDCopy = TRAFFIC_LOCID_INVALID_OBJECT;

            printf(TRAFFIC_MGR_OBJECT_NAME
                "Market %u added to filter\n", tMarket);
        }

        // We're all good
        eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (NULL != psAppObj)
    {
        // We're done with this object
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    // Free any locid we've left behind
    TRAFFIC_LOCID.vDestroy(hLocIDCopy);

    return eErrorCode;
}

/*****************************************************************************
*
*   eRemoveMarket
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveMarket(
    TRAFFIC_SERVICE_OBJECT hTrafficService,
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj = (TRAFFIC_APP_OBJECT_STRUCT *)NULL;
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        TRAFFIC_MARKET tMarket;
        BOOLEAN bValid;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Validate the traffic service handle
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hTrafficService);
        if (FALSE == bValid)
        {
            eErrorCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Validate locid handle
        if (TRAFFIC_LOCID_INVALID_OBJECT == hTrafficLocID  )
        {
            eErrorCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Validate the traffic market
        tMarket = TRAFFIC_LOCID.tMarket(hTrafficLocID);

        if ((0 == tMarket) || (tMarket >= NUM_TRAFFIC_MARKETS))
        {
            // ERROR
            eErrorCode = SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
            break;
        }

        // Get a pointer to the locked app-facing object
        psAppObj = psGetAppFacingObject(hTrafficService);

        if ((TRAFFIC_APP_OBJECT_STRUCT *)NULL == psAppObj)
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Search for the matching entry so we
        // can remove it
        eReturnCode = OSAL.eLinkedListSearch(
            psAppObj->hMarketFilterList,
            &hEntry,
            (void *) hTrafficLocID);

        if (OSAL_SUCCESS == eReturnCode)
        {
            TRAFFIC_LOCID_OBJECT hTrafficLocIDToRemove;

            hTrafficLocIDToRemove = (TRAFFIC_LOCID_OBJECT)
                OSAL.pvLinkedListThis(hEntry);

            // Remove the linked list entry
            eReturnCode = OSAL.eLinkedListRemove( hEntry );

            if (OSAL_SUCCESS == eReturnCode)
            {
                vReleaseMarketFilterListNode(hTrafficLocIDToRemove);
            }
        }

        if ((OSAL_SUCCESS != eReturnCode) &&
            (OSAL_OBJECT_NOT_FOUND != eReturnCode))
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        if (OSAL_SUCCESS == eReturnCode)
        {
            BOOLEAN bFilterUpdated;

            // Update the filter byte array used by the chipset
            bFilterUpdated = bUpdateFilterForMarket(
                (TRAFFIC_MGR_OBJECT_STRUCT *)hTrafficService,
                tMarket, FALSE);

            if (FALSE == bFilterUpdated)
            {
                eErrorCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }

            printf(TRAFFIC_MGR_OBJECT_NAME
                "Market %u removed from filter\n", tMarket);
        }

        // We're all good
        eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (NULL != psAppObj)
    {
        // We're done with this object
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return eErrorCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   TRAFFIC_MGR_hGetAlertCEvent
*
*****************************************************************************/
ALERTC_EVENT_OBJECT TRAFFIC_MGR_hGetAlertCEvent (
    TRAFFIC_SERVICE_OBJECT hTrafficService,
    TRAFFIC_EVENT_CODE tEventCode
        )
{
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;
    ALERTC_EVENT_OBJECT hAlertCEvent = ALERTC_EVENT_INVALID_OBJECT;

    // Validate the hTrafficService, then get a pointer to the
    // locked app-facing object
    psAppObj = psGetAppFacingObject(hTrafficService);

    if (psAppObj != NULL)
    {
        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            printf(TRAFFIC_MGR_OBJECT_NAME": fetching event data for %u from ",tEventCode);

            if (psAppObj->hAlertCEventList != OSAL_INVALID_OBJECT_HDL)
            {
                OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                BOOLEAN bSuccess;

                bSuccess = ALERTC_EVENT_bSetEventCode(
                    psAppObj->hDummyAlertCEvent, tEventCode);
                if (bSuccess == TRUE)
                {
                    // Search through the list looking for a cached event
                    eReturnCode = OSAL.eLinkedListSearch(
                        psAppObj->hAlertCEventList, &hEntry,
                        psAppObj->hDummyAlertCEvent);

                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        hAlertCEvent = (ALERTC_EVENT_OBJECT)
                            OSAL.pvLinkedListThis(hEntry);

                        printf("memory\n");
                    }
                }
            }

            // Didn't find it in our cached list, do a DB look-up
            if (hAlertCEvent == ALERTC_EVENT_INVALID_OBJECT)
            {
                TRAFFIC_DB_QUERY_RESULT_STRUCT sQueryResult;
                BOOLEAN bOk;

                // Prepare the Query iterator
                OSAL.bMemSet(&sQueryResult, 0, sizeof(sQueryResult));
                sQueryResult.bResultantRows = FALSE;
                sQueryResult.uDbRow.sEvents.tCode = tEventCode;
                sQueryResult.uDbRow.sEvents.eType = ALERTC_EVENT_TYPE_UNKNOWN;
                sQueryResult.uDbRow.sEvents.eClass = TRAFFIC_MSG_CLASS_UNKNOWN;
                sQueryResult.uDbRow.sEvents.eQuantType = ALERTC_QUANTIFIER_TYPE_UNKNOWN;
                sQueryResult.uDbRow.sEvents.hText = STRING_INVALID_OBJECT;
                sQueryResult.uDbRow.sEvents.hTextMod = STRING_INVALID_OBJECT;

                // Build our query string
                snprintf( &psAppObj->acBuffer[0], sizeof(psAppObj->acBuffer),
                          TRAFFIC_SELECT_EVENT_INFO, tEventCode );

                // Perform the SQL query and process the result
                // (it will provide us with a data row)
                bOk = SQL_INTERFACE.bQuery(
                        psAppObj->hSQLRefConnection, &psAppObj->acBuffer[0],
                        bProcessSelectEventResult, &sQueryResult ) ;

                if (bOk == FALSE)
                {
                    break;
                }

                // Create our ALERTC_EVENT_OBJECT even if there is no information about
                // it in the reference database
                hAlertCEvent = ALERTC_EVENT_hCreate(
                    (SMS_OBJECT)psAppObj,
                    sQueryResult.uDbRow.sEvents.tCode,
                    sQueryResult.uDbRow.sEvents.eType,
                    sQueryResult.uDbRow.sEvents.eClass,
                    sQueryResult.uDbRow.sEvents.eQuantType,
                    sQueryResult.uDbRow.sEvents.hText,
                    sQueryResult.uDbRow.sEvents.hTextMod);

                // If we didn't create it, fail out
                if (hAlertCEvent == ALERTC_EVENT_INVALID_OBJECT)
                {
                    break;
                }

                // Create our event list if it doesn't exist
                if (psAppObj->hAlertCEventList == OSAL_INVALID_OBJECT_HDL)
                {
                    eReturnCode = OSAL.eLinkedListCreate(
                        &psAppObj->hAlertCEventList,
                        TRAFFIC_MGR_OBJECT_NAME":AlertCEventCache",
                        (OSAL_LL_COMPARE_HANDLER)ALERTC_EVENT_n16Compare,
                        OSAL_LL_OPTION_LINEAR
                            );

                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        ALERTC_EVENT_vDestroy(hAlertCEvent);
                        hAlertCEvent = ALERTC_EVENT_INVALID_OBJECT;
                        break;
                    }
                }

                // Add cached event to our list
                eReturnCode =
                    OSAL.eLinkedListAdd(psAppObj->hAlertCEventList,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hAlertCEvent);

                if (eReturnCode != OSAL_SUCCESS)
                {
                    ALERTC_EVENT_vDestroy(hAlertCEvent);
                    hAlertCEvent = ALERTC_EVENT_INVALID_OBJECT;
                    break;
                }

                printf("database\n");
            }
        } while (FALSE);

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return hAlertCEvent;
}

/*****************************************************************************
*
*   TRAFFIC_MGR_hGetSupplInfoText
*
*****************************************************************************/
const char *TRAFFIC_MGR_pcGetSupplInfoText (
    TRAFFIC_SERVICE_OBJECT hTrafficService,
    ALERTC_SUPPL_INFO_CODE tCode
        )
{
    char *pcSupplInfoText = NULL;
    SUPPL_INFO_STRUCT *psSupplInfoStruct = NULL;
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;

    // Validate the hTrafficService, then get a pointer to the
    // locked app-facing object
    psAppObj = psGetAppFacingObject(hTrafficService);

    if (psAppObj != NULL)
    {
        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            printf(TRAFFIC_MGR_OBJECT_NAME": fetching suppl info for %u from ",tCode);

            if (psAppObj->hSupplInfoList != OSAL_INVALID_OBJECT_HDL)
            {
                OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                SUPPL_INFO_STRUCT sDummySupplInfo;

                sDummySupplInfo.tCode = tCode;

                // Search through the list looking for a cached event
                eReturnCode = OSAL.eLinkedListSearch(
                    psAppObj->hSupplInfoList, &hEntry,
                    (void *)&sDummySupplInfo);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    psSupplInfoStruct = (SUPPL_INFO_STRUCT *)
                        OSAL.pvLinkedListThis(hEntry);

                    pcSupplInfoText = psSupplInfoStruct->pcText;

                    printf("memory\n");
                }
            }

            // Didn't find it in our cached list, do a DB look-up
            if (psSupplInfoStruct == NULL)
            {
                TRAFFIC_DB_QUERY_RESULT_STRUCT sQueryResult;
                BOOLEAN bOk;
                size_t tLength;

                // Build our query string
                snprintf( &psAppObj->acBuffer[0], sizeof(psAppObj->acBuffer),
                          TRAFFIC_SELECT_SUPPL_INFO, tCode );

                // Perform the SQL query and process the result
                // (it will provide us with a data row)
                bOk = SQL_INTERFACE.bQuery(
                    psAppObj->hSQLRefConnection, &psAppObj->acBuffer[0],
                        bProcessSelectSupplInfoResult, &sQueryResult ) ;

                if (bOk == FALSE)
                {
                    break;
                }

                if (sQueryResult.bResultantRows == FALSE)
                {
                    break;
                }

                // Create our SUPPL_INFO_STRUCT
                psSupplInfoStruct = (SUPPL_INFO_STRUCT *)
                    SMSO_hCreate(
                        TRAFFIC_MGR_OBJECT_NAME":SupplInfo",
                        sizeof(SUPPL_INFO_STRUCT),
                        (SMS_OBJECT)psAppObj,
                        FALSE);

                // If we didn't create it, fail out
                if (psSupplInfoStruct == NULL)
                {
                    break;
                }

                // Make some space to store the string
                tLength = strlen(sQueryResult.uDbRow.sSupplInfo.acText) + 1;
                psSupplInfoStruct->pcText =
                    (char *) SMSO_hCreate(
                        TRAFFIC_MGR_OBJECT_NAME":SupplInfoText",
                        tLength,
                        (SMS_OBJECT)psAppObj,
                        FALSE);

                // If we didn't create it, fail out
                if (psSupplInfoStruct->pcText == NULL)
                {
                    vReleaseSupplInfo(psSupplInfoStruct);
                    break;
                }

                // Store our variables
                psSupplInfoStruct->tCode =
                    sQueryResult.uDbRow.sSupplInfo.tCode;
                strncpy(psSupplInfoStruct->pcText,
                    &sQueryResult.uDbRow.sSupplInfo.acText[0], tLength);

                // Create our suppl info list if it doesn't exist
                if (psAppObj->hSupplInfoList == OSAL_INVALID_OBJECT_HDL)
                {
                    eReturnCode = OSAL.eLinkedListCreate(
                        &psAppObj->hSupplInfoList,
                        TRAFFIC_MGR_OBJECT_NAME":SupplInfoCache",
                        (OSAL_LL_COMPARE_HANDLER)n16CompareSupplInfoCode,
                        OSAL_LL_OPTION_LINEAR
                            );

                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        vReleaseSupplInfo(psSupplInfoStruct);
                        break;
                    }
                }

                // Add cached suppl info to our list
                eReturnCode =
                    OSAL.eLinkedListAdd(psAppObj->hSupplInfoList,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psSupplInfoStruct);

                if (eReturnCode != OSAL_SUCCESS)
                {
                    vReleaseSupplInfo(psSupplInfoStruct);
                    break;
                }

                pcSupplInfoText = psSupplInfoStruct->pcText;

                printf("database\n");
            }
        } while (FALSE);

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return pcSupplInfoText;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   psGetAppFacingObject
*
*****************************************************************************/
static TRAFFIC_APP_OBJECT_STRUCT *psGetAppFacingObject(
    TRAFFIC_SERVICE_OBJECT hTrafficService
        )
{
    TRAFFIC_MGR_OBJECT_STRUCT *psObj =
        (TRAFFIC_MGR_OBJECT_STRUCT *)hTrafficService;
    BOOLEAN bValid, bLocked;

    do
    {
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hTrafficService);

        if (bValid == FALSE)
        {
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    TRAFFIC_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our traffic handle from the callback argument
    psObj = (TRAFFIC_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == TRUE)
    {
        switch( tCurrentEvent )
        {
            // Handle Traffic Service events here...

            // State has changed
            case DATASERVICE_EVENT_STATE:
            {
                BOOLEAN bStateChanged;
                DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                    (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

                // Process the state transition
                bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psObj,
                    psStateChange,
                    &GsTrafficStateHandlers,
                    (void *)psObj);

                if (bStateChanged == TRUE)
                {
                    // The state has been updated
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    // Is the service stopped now?
                    if (psStateChange->eCurrentState ==
                            DATASERVICE_STATE_STOPPED)
                    {
                        bStopEvent = TRUE;
                    }
                }
            }
            break;

            // This service has a message to process
            case DATASERVICE_EVENT_NEW_DATA:
            {
                OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
                BOOLEAN bOk = TRUE;

                // Ensure the payload handle is valid
                // If it isn't, there's a problem with
                // SMS
                if (hPayload == OSAL_INVALID_BUFFER_HDL)
                {
                    bOk = FALSE;

                    // Set the error
                    vSetError( psObj,
                               DATASERVICE_ERROR_CODE_GENERAL);
                }
                else
                {
                    BOOLEAN bPayloadOk;
                    printf(TRAFFIC_MGR_OBJECT_NAME": Payload Received (%u)\n",
                        OSAL.tBufferGetSize(hPayload));


                    bPayloadOk = bProcessPayload(psObj, hPayload);

                    if (bPayloadOk == TRUE)
                    {
                        puts(TRAFFIC_MGR_OBJECT_NAME": Message Payload Processed Ok");
                    }
                    else
                    {
                        DATASERVICE_IMPL_vLog(
                            TRAFFIC_MGR_OBJECT_NAME": Failed to process message\n");
                    }

                }

                if (bOk != TRUE)
                {
                    // If an error occurred, indicate a state change
                    // to the application
                    tEventMask |= DATASERVICE_EVENT_STATE;
                }
            }
            break;

            // We need to do some work with our DSRLs
            case DATASERVICE_INTERNAL_EVENT_DSRL:
            {
                BOOLEAN bSuccess = FALSE;
                DSRL_ARG_STRUCT *psDSRLArg =
                    (DSRL_ARG_STRUCT *)pvEventArg;
                TRAFFIC_APP_OBJECT_STRUCT *psAppObj;

                psAppObj = psGetAppFacingObject((TRAFFIC_SERVICE_OBJECT)psObj);

                if (psAppObj != NULL)
                {
                    switch (psDSRLArg->eAction)
                    {
                        case DSRL_ACTION_ADD:
                        {
                            bSuccess = bCreateList(psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_MODIFY:
                        {
                            bSuccess = bModifyList(psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_REFRESH:
                        {
                            bSuccess = bRefreshList(psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_REMOVE:
                        {
                            bSuccess = bDestroyList(psObj, psDSRLArg);
                        }
                        break;

                        case DSRL_ACTION_INVALID:
                        default:
                        break;
                    }

                    if (bSuccess == TRUE)
                    {
                        // We succeeded, clear the target list
                        // since we own those handles now
                        OSAL.bMemSet(
                            &psDSRLArg->ahTargetList[0], 0,
                            sizeof(DSRL_TARGET_OBJECT) * psDSRLArg->tNumTargets);
                    }

                    SMSO_vUnlock((SMS_OBJECT)psAppObj);
                }

                // Go to the error state if the create/modify failed
                // or if we aren't ready
                if (bSuccess == FALSE)
                {
                    BOOLEAN bLocked;

                    bLocked = SMSO_bLock((SMS_OBJECT)psDSRLArg->hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);

                    if (bLocked == TRUE)
                    {
                        DSRL_vSetState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);

                        SMSO_vUnlock((SMS_OBJECT)psDSRLArg->hDSRL);
                    }
                    else
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            TRAFFIC_MGR_OBJECT_NAME": unable to set DSRL to the error state");
                    }
                }
            }
            break;

            case DATASERVICE_EVENT_TIMEOUT:
            {
                printf(TRAFFIC_MGR_OBJECT_NAME
                    ": checking for msgs to cancel\n");
                vCheckForMessagesToCancel(psObj);
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFFIC_MGR_OBJECT_NAME": Got unknown event (%u)",
                    tCurrentEvent);
            }
            break;
        }

        if (bStopEvent == TRUE)
        {
            // Uninitialize the object, but leave
            // enough to be useful to the callback
            vUninitObject( psObj, FALSE );
        }

        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psObj->sEvent, tEventMask);

        // Notify of any change via any registered callback which may be present
        SMSU_bNotify(&psObj->sEvent);

        if (bStopEvent == TRUE)
        {
            // Filter out all further traffic manager updates
            SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

            vDestroyObject(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   bUpdateFilterForMarket
*
*   Assumes a VALID handle for the traffic manager
*
*****************************************************************************/
static BOOLEAN bUpdateFilterForMarket (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket,
    BOOLEAN bAddMarket
        )
{
    BOOLEAN bOk = FALSE;

    if (GsTrafficIntf.bConfigureDataFilter == NULL)
    {
        // Don't need to do anything, just return TRUE
        return TRUE;
    }

    if (tMarket != TRAFFIC_INVALID_MARKET)
    {
        // Update the data filter
        bOk = GsTrafficIntf.bConfigureDataFilter(
            psObj->hTrafficInterfaceData,
            (TRAFFIC_SERVICE_OBJECT)psObj,
            bAddMarket, tMarket);
    }

    // And we're done
    return bOk;
}

/*****************************************************************************
*
*   bCreateList
*
*****************************************************************************/
static BOOLEAN bCreateList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DSRL_DESC_STRUCT *psDSRLDesc;

    do
    {
        // create the DSRL desc entry and add it
        // to the list
        psDSRLDesc = (DSRL_DESC_STRUCT *)
            SMSO_hCreate(
                TRAFFIC_MGR_OBJECT_NAME":DSRLDesc",
                sizeof(DSRL_DESC_STRUCT),
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
                FALSE);

        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }

        // Set up our internal record
        psDSRLDesc->hDSRL = psDSRLArg->hDSRL;
        psDSRLDesc->eCurrentRequestState = DSRL_STATE_UNKNOWN;
        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hTargetList,
            TRAFFIC_MGR_OBJECT_NAME":DSRLDesc:TgtList",
            NULL,
            OSAL_LL_OPTION_LINEAR
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Add this to our list of tracked Targets for this service
        eReturnCode =
            OSAL.eLinkedListAdd(psObj->hDSRLList,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                psDSRLDesc);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Link up the DSRL
        bSuccess = bCreateLinksWithDSRLAndTable(
            psObj,
            psDSRLArg,
            psDSRLDesc);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Set the default sort function
        bSuccess = DSRL_bSetDefaultSortFunction (psDSRLDesc->hDSRL,
            n16SortDSRLByDistance, psDSRLDesc);
        if (FALSE == bSuccess)
        {
            break;
        }

        // Looks good
        bSuccess = TRUE;

        // Set the DSRL state to READY
        DSRL_vSetState(psDSRLDesc->hDSRL, DSRL_STATE_READY);

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bModifyList
*
*****************************************************************************/
static BOOLEAN bModifyList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE, bAdd = FALSE, bSort = FALSE;
    DSRL_DESC_STRUCT sDSRLDescSearch;
    DSRL_DESC_STRUCT *psDSRLDescToModify;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        // get the DSRL desc for this item
        sDSRLDescSearch.hDSRL = psDSRLArg->hDSRL;

        eReturnCode =
            OSAL.eLinkedListSearch(psObj->hDSRLList,
                                   &hEntry,
                                   &sDSRLDescSearch);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Get the DSRL Desc from the Posistions
        psDSRLDescToModify = (DSRL_DESC_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        // set the DSRL state to UPDATING
        DSRL_vSetState(psDSRLDescToModify->hDSRL,
            DSRL_STATE_UPDATING);

        // MOdify the DSRL as the user specifed
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                // Add the specified entries
                bAdd = TRUE;
            }
            break;

            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                // Remove all items from the DSRL target list
                eReturnCode = OSAL.eLinkedListIterate(
                    psDSRLDescToModify->hTargetList,
                    (OSAL_LL_ITERATOR_HANDLER)bIterateTargetDescToRemoveDSRLDesc,
                    psDSRLDescToModify);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    eReturnCode = OSAL.eLinkedListRemoveAll(
                        psDSRLDescToModify->hTargetList,
                        (OSAL_LL_RELEASE_HANDLER)vReleaseTargetNode);

                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        // We passed
                        // Add the specified entries
                        bAdd = TRUE;

                        // We need to re-sort the list after
                        // we've added the new targets
                        bSort = TRUE;
                    }
                    else
                    {
                        // some sort of error
                        bSuccess = FALSE;
                    }
                }
                else if (eReturnCode == OSAL_NO_OBJECTS)
                {
                    // Empty descriptor from before, just populate it now
                    bAdd = TRUE;
                }
                else
                {
                    // some sort of error
                    bSuccess = FALSE;
                }
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                // Remove the specified targets
                bSuccess = bHandleRemoveTargets(
                    psObj,
                    psDSRLArg,
                    psDSRLDescToModify, &bSort);
            }
            break;

            case DSRL_MODIFY_OPERATION_INVALID:
            default:
                break;

        }

        if (bAdd == TRUE)
        {
            bSuccess = bCreateLinksWithDSRLAndTable(
                psObj,
                psDSRLArg,
                psDSRLDescToModify);
        }

        // Do we need to re-sort the DSRL?
        if ((TRUE == bSuccess) && (TRUE == bSort))
        {
            // Our default sort uses the target list
            // as the sort criteria...it just changed so
            // it's time for a re-sort
            bSuccess = DSRL_bSort(psDSRLDescToModify->hDSRL);
        }

        // If we got here and didn't succeed at all,
        // then bail out
        if (bSuccess == FALSE)
        {
            DSRL_vSetState(psDSRLDescToModify->hDSRL,
                DSRL_STATE_ERROR);
            break;
        }

        // Set the DSRL state to ready state
        DSRL_vSetState(psDSRLDescToModify->hDSRL,
            DSRL_STATE_READY);

        // All done
        bSuccess = TRUE;

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bRefreshList
*
*****************************************************************************/
static BOOLEAN bRefreshList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    DSRL_DESC_STRUCT sDSRLDescSearch;
    DSRL_STATE_ENUM eOrigState;
    DSRL_DESC_STRUCT *psDSRLDescToModify;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        // get the DSRL desc for this item
        sDSRLDescSearch.hDSRL = psDSRLArg->hDSRL;

        eReturnCode =
            OSAL.eLinkedListSearch(psObj->hDSRLList,
                                   &hEntry,
                                   &sDSRLDescSearch);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Get the DSRL Desc from the Posistions
        psDSRLDescToModify = (DSRL_DESC_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        eOrigState = DSRL.eState(psDSRLDescToModify->hDSRL);

        DSRL_vSetState(psDSRLDescToModify->hDSRL,
            DSRL_STATE_UPDATING);

        DSRL_vRemoveAllEntries(psDSRLDescToModify->hDSRL);

        // Refresh the targets
        eReturnCode = OSAL.eLinkedListIterate(
            psDSRLDescToModify->hTargetList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsForRefresh,
            (void *)psDSRLDescToModify);

        // If we got here and didn't succeed at all,
        // then bail out
        if (eReturnCode != OSAL_SUCCESS)
        {
            DSRL_vSetState(psDSRLDescToModify->hDSRL,
                DSRL_STATE_ERROR);
            break;
        }

        if (eOrigState == DSRL_STATE_READY)
        {
            DSRL_vSetState(psDSRLDescToModify->hDSRL,
                DSRL_STATE_READY);
        }

        // All done
        bSuccess = TRUE;

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bDestroyList
*
*****************************************************************************/
static BOOLEAN bDestroyList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    DSRL_DESC_STRUCT sDSRLDescSearch;
    DSRL_DESC_STRUCT *psDSRLDescToRemove;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // get the DSRL desc for this item
    sDSRLDescSearch.hDSRL = psDSRLArg->hDSRL;

    eReturnCode =
        OSAL.eLinkedListSearch(psObj->hDSRLList,
                               &hEntry,
                               &sDSRLDescSearch);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // Remove the DSRL Desc from the Posistions
        psDSRLDescToRemove = OSAL.pvLinkedListThis(hEntry);

        OSAL.eLinkedListIterate(
            psDSRLDescToRemove->hTargetList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateTargetDescToRemoveDSRLDesc,
            psDSRLDescToRemove);

        // Remove the entry from our DSRL list
        OSAL.eLinkedListRemove(hEntry);

        // Free memory
        vReleaseDSRLNode(psDSRLDescToRemove);

        bSuccess = TRUE;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessPayload
*
*****************************************************************************/
static BOOLEAN bProcessPayload (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bMarketInFilter = FALSE, bSuccess = FALSE;
    TRAFFIC_LOCID_OBJECT hTrafficLocID = TRAFFIC_LOCID_INVALID_OBJECT;
    TRAFFIC_MSG_TYPE_ENUM eMsgType;
    size_t tNumAlertCBytes = 0, tBytesInPayload;
    TRAFFIC_MARKET_PROC_STATE_ENUM eMarketProcState;
    TRAFFIC_MARKET tMarket;
    BOOLEAN bNewPayload = TRUE;

    do
    {
        if (GsTrafficIntf.bParseTrafficServiceBits == NULL)
        {
            // ERROR!
            break;
        }

        do
        {

            bSuccess = GsTrafficIntf.bParseTrafficServiceBits(
                psObj->hTrafficInterfaceData, &hPayload, bNewPayload,
                &hTrafficLocID, &eMsgType, &eMarketProcState, &tNumAlertCBytes);

            if (bSuccess == FALSE || tNumAlertCBytes == 0)
            {
                // Error!
                break;
            }

            // If we have any more data, this is no longer a new payload
            bNewPayload = FALSE;

            if (eMarketProcState == TRAFFIC_MARKET_PROC_STATE_NATION_COMPLETE)
            {
                // Mark all markets as complete before we start processing
                // the new alert-c messages.  The flag indicates that everything
                // we have up until this point is a complete picture
                // of the nation's traffic market.
                vMarkNationComplete(psObj);
            }

            tMarket = TRAFFIC_LOCID.tMarket(hTrafficLocID);

            if (tMarket == TRAFFIC_INVALID_MARKET)
            {
                // Error!
                break;
            }

            // See if this market is in our filter
            bMarketInFilter = bCheckMarketList(psObj, hTrafficLocID);

            if (bMarketInFilter == FALSE)
            {
                size_t tBytesSkipped;

                // Skip past the alert-c messages
                tBytesSkipped = OSAL.tBufferSeekHead(hPayload, tNumAlertCBytes);
                if (tBytesSkipped != tNumAlertCBytes)
                {
                    // Error!
                    break;
                }
            }
            else
            {
                BOOLEAN bOk;

                bOk = bEnsureTrafficTableExists(psObj->psAppObj, tMarket);
                if (FALSE == bOk)
                {
                    break;
                }

                puts(TRAFFIC_MGR_OBJECT_NAME
                    ": Pass Off Alert-C Messages for Processing\n");

                bSuccess =
                    bProcessTrafficMessagesFromPayload(psObj, hPayload,
                        hTrafficLocID, tNumAlertCBytes, eMarketProcState);

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TRAFFIC_MGR_OBJECT_NAME": Error Processing "
                        "Alert-C Messages");
                }
                else
                {
                    if (eMarketProcState == TRAFFIC_MARKET_PROC_STATE_COMPLETE)
                    {
                        // Mark table for market as complete
                        bMarkMarketAsComplete(psObj, tMarket);

                        // Tell the DSRLs to fire their callbacks if
                        // they detected any changes
                        OSAL.eLinkedListIterate(
                           psObj->hDSRLList,
                           (OSAL_LL_ITERATOR_HANDLER)bNotifyTargets,
                           (void *)(size_t)DSRL_STATE_READY);
                    }
                }
            }

            // Get our payload size to see if there is still data
            // to process
            tBytesInPayload = OSAL.tBufferGetSize(hPayload);

            // We don't need the traffic locid anymore
            TRAFFIC_LOCID.vDestroy(hTrafficLocID);
            hTrafficLocID = TRAFFIC_LOCID_INVALID_OBJECT;

        } while (tBytesInPayload > 0);

        // Mark the last time we saw data
        OSAL.eTimeGet(&psObj->un32LastTimeDataRXd);

    } while (FALSE);

    // If we still have a payload handle, free it
    if (hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        DATASERVICE_IMPL_bFreeDataPayload(hPayload);
    }

    // Make sure there aren't any dangling traffic loc IDs
    if (hTrafficLocID != TRAFFIC_LOCID_INVALID_OBJECT)
    {
        TRAFFIC_LOCID.vDestroy(hTrafficLocID);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vMarkNationComplete
*
*****************************************************************************/
static void vMarkNationComplete (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    TRAFFIC_MARKET tMarket;

    for (tMarket = 1; tMarket <= NUM_TRAFFIC_MARKETS; tMarket++)
    {
        // Mark table for market as complete
        bMarkMarketAsComplete(psObj, tMarket);
    }

    // Tell the DSRLs to fire their callbacks if
    // they detected any changes
    OSAL.eLinkedListIterate(
       psObj->hDSRLList,
       (OSAL_LL_ITERATOR_HANDLER)bNotifyTargets,
       (void *)(size_t)DSRL_STATE_READY);

    DATASERVICE_IMPL_vLog(TRAFFIC_MGR_OBJECT_NAME
        ": Nation marked as complete\n");

    return;
}

/*****************************************************************************
*
*   bProcessTrafficMessagesFromPayload
*
*****************************************************************************/
static BOOLEAN bProcessTrafficMessagesFromPayload (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TRAFFIC_LOCID_OBJECT hTrafficLocID,
    size_t tNumAlertCBytes,
    TRAFFIC_MARKET_PROC_STATE_ENUM eMarketProcState
        )
{
    BOOLEAN bOk = TRUE;
    TRAFFIC_MSG_OBJECT hCurTrafficMsg;
    size_t tBytesInBuffer, tInitialBytesInBuffer;
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;

    puts(
        TRAFFIC_MGR_OBJECT_NAME": Processing Alert-C From Payload\n");

    tInitialBytesInBuffer =
        tBytesInBuffer = OSAL.tBufferGetSize(hPayload);

    // Get the app facing object
    psAppObj = psGetAppFacingObject((TRAFFIC_SERVICE_OBJECT)psObj);

    if (psAppObj == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFFIC_MGR_OBJECT_NAME
            ": Couldn't get app facing object in order to process Traffic Msg");
        return FALSE;
    }

    do
    {
        if (tBytesInBuffer < ALERTC_MESSAGE_MIN_BYTE_LENGTH)
        {
            printf(TRAFFIC_MGR_OBJECT_NAME": Parsing Alert-C -- not enough bytes\n");
            break;
        }

        hCurTrafficMsg = TRAFFIC_MSG_hCreate(
            (SMS_OBJECT)psObj->psAppObj,
            hPayload,
            hTrafficLocID);

        // Get the buffer size to see update the number of bytes
        // processed and to make sure there is still more bytes
        // to process
        tBytesInBuffer = OSAL.tBufferGetSize(hPayload);

        printf(TRAFFIC_MGR_OBJECT_NAME
                        ": %u bytes processed. %u bytes left read\n",
                        (tInitialBytesInBuffer - tBytesInBuffer), tBytesInBuffer);

        // Make sure we didn't go to far
        if ((tInitialBytesInBuffer - tBytesInBuffer) > tNumAlertCBytes)
        {
            // We've processed more than we should have
            // There was a problem in the parsing that happened
            // before here.  Mark this as a failure.
            printf(TRAFFIC_MGR_OBJECT_NAME
                ": Error parsing Alert-C -- too many bytes (%u/%u) read\n",
                (tInitialBytesInBuffer - tBytesInBuffer), tNumAlertCBytes);
            bOk = FALSE;
            break;
        }

        if (hCurTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            // We had a problem reading an AlertC Message
            // Report the error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": Reading Alert-C from buffer.");
            bOk = FALSE;
            break;
        }
        else
        {
            bOk = bUpdateTrafficTable(psObj, hCurTrafficMsg);

            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TRAFFIC_MGR_OBJECT_NAME
                    ": Couldn't update traffic table with this msg.");

                TRAFFIC_MSG_vDestroy(hCurTrafficMsg);
            }
        }

    } while ((tInitialBytesInBuffer - tBytesInBuffer) < tNumAlertCBytes);

    SMSO_vUnlock((SMS_OBJECT)psAppObj);

    return bOk;
}

/*****************************************************************************
*
*   bUpdateTrafficTable
*
*****************************************************************************/
static BOOLEAN bUpdateTrafficTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg
        )
{
    BOOLEAN  bOk;
    OSAL_LINKED_LIST_ENTRY hStartPosEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (psObj->vTrafficPosLookUpCallback == NULL)
    {
        // We can't do any sort of traffic table stuff
        // (expand here in the future if add a raw alert-c mode)
        return FALSE;
    }

    do
    {
        // Update our traffic table with the positions from
        // this traffic message
        bOk = bUpdatePositionsInTable(psObj, hTrafficMsg, &hStartPosEntry);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": Failed to update positions in the table.");
            break;
        }

        // Update the positions in our table with the
        // latest traffic events.
        bUpdateTrafficMsgsInTable(psObj, hTrafficMsg,
            hStartPosEntry);

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bUpdatePositionsInTable
*
*****************************************************************************/
static BOOLEAN bUpdatePositionsInTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    OSAL_LINKED_LIST_ENTRY *phStartPosEntry
        )
{
    TRAFFIC_POS_CODE tCurPosCode;
    TRAFFIC_MARKET tMarket;
    UN8 un8NumLocsReq, un8NumLocsKnown, un8NumLocsToFind, un8NumLocsFound = 0;
    UN8 un8Index;
    BOOLEAN bFinalLoc = FALSE;
    TRAFFIC_DIRECTION_ENUM eDirection;
    LOCATION_OBJECT ahLocations[MAX_EXTENT_VALUE+1];    // +1 for initial location
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    POS_EVENT_DESC_STRUCT *psPosDesc, **ppsLastPosDesc = NULL;
    POS_EVENT_DESC_STRUCT sPosSearch;

    if (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT || phStartPosEntry == NULL)
    {
        // Error! Stop processing
        return FALSE;
    }

    printf(
        TRAFFIC_MGR_OBJECT_NAME": Updating Positions in Traffic Table\n");

    // Get the variables from the traffic message
    tCurPosCode = TRAFFIC_MSG_tInitialPosCode(hTrafficMsg);
    tMarket = TRAFFIC_MSG_tMarket(hTrafficMsg);
    if (tMarket == TRAFFIC_INVALID_MARKET)
    {
        // Should not happen, but who knows...
        return FALSE;
    }
    eDirection = TRAFFIC_MSG.eDirection(hTrafficMsg);

    // We want to get extent+1 so we get the position of
    // the message and the adjacent positions it covers
    un8NumLocsReq = TRAFFIC_MSG_un8InitialNumLocs(hTrafficMsg) + 1;

    printf(TRAFFIC_MGR_OBJECT_NAME
        ": looking for locations starting at %d and extending %d in %d direction\n",
        tCurPosCode, un8NumLocsReq, eDirection);

    un8NumLocsKnown =
        un8FindNumLocationsKnown(psObj, &hEntry, &tCurPosCode, tMarket,
                                 eDirection, un8NumLocsReq, &bFinalLoc);

    printf(TRAFFIC_MGR_OBJECT_NAME
            ": number locations known: %d\n",
            un8NumLocsKnown);

    if (un8NumLocsKnown > 0)
    {
        // We know one entry, so store it to aid in later steps
        *phStartPosEntry = hEntry;
    }

    // if we don't known all the locations requested, fetch them from the
    // app and add them to our table
    if (un8NumLocsKnown < un8NumLocsReq && bFinalLoc == FALSE)
    {
        if (un8NumLocsKnown == 1)
        {
            // We only found our initial location
            // So we don't know how it links up to the rest
            // of the locations.
            un8NumLocsToFind = un8NumLocsReq;
        }
        else
        {
            un8NumLocsToFind = un8NumLocsReq - un8NumLocsKnown;
        }

        // Call our pos code look-up handler to get the data
        psObj->vTrafficPosLookUpCallback((TRAFFIC_SERVICE_OBJECT)psObj,
                                         tMarket,
                                         tCurPosCode,
                                         eDirection,
                                         un8NumLocsToFind,
                                         &un8NumLocsFound,
                                         &ahLocations[0],
                                         psObj->pvPosLookUpCallbackArg);

        printf(TRAFFIC_MGR_OBJECT_NAME
            ": Found %d locations for %d-%d (requested %d of %d)\n",
            un8NumLocsFound,
            tMarket,
            tCurPosCode,
            un8NumLocsToFind,
            un8NumLocsReq);

        if (un8NumLocsFound != 0)
        {
            // Loop through the extents and add them as need be
            for (un8Index = 0;
                 un8Index < un8NumLocsFound;
                 un8Index++)
            {
                TRAFFIC_LOCID_OBJECT hTrafficLocID;

                // Find the pos code entry, starting our search from the hEntry value
                // that was set by un8FindNumLocationsKnown
                hTrafficLocID =
                    (TRAFFIC_LOCID_OBJECT)LOCATION.hLocID(ahLocations[un8Index]);
                sPosSearch.tPosCode = TRAFFIC_LOCID.tPosCode(hTrafficLocID);

                eReturnCode = OSAL.eLinkedListSearch(
                    psObj->psAppObj->ahTrafficTable[tMarket-1], &hEntry,
                    (void *) &sPosSearch);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    // In this case, we already knew about the location
                    // but we were missing some portion of the chain
                    // since un8FindNumLocationsKnown didn't find it

                    // grab the pointer to our entry
                    psPosDesc = (POS_EVENT_DESC_STRUCT *)
                        OSAL.pvLinkedListThis(hEntry);
                }
                else
                {
                    // create the entry and add it
                    // to the list
                    psPosDesc = psCreatePosDesc(psObj, ahLocations[un8Index]);

                    if (psPosDesc == NULL)
                    {
                        // Error! Stop processing
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            TRAFFIC_MGR_OBJECT_NAME
                            ": Failed to alloc memory for PosEntry.");
                        return FALSE;
                    }

                    hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                    eReturnCode =
                        OSAL.eLinkedListAdd(
                            psObj->psAppObj->ahTrafficTable[tMarket-1],
                            &hEntry,
                            psPosDesc);

                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        // ERROR!
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            TRAFFIC_MGR_OBJECT_NAME
                            ": Failed to add PosEntry to traffic table.");
                        return FALSE;
                    }

                    if (*phStartPosEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                    {
                        // This is our start entry
                        *phStartPosEntry = hEntry;
                    }
                }

                // If we are looking to set the link for the previous
                // struct, set the link to this one
                if (ppsLastPosDesc != NULL)
                {
                    *ppsLastPosDesc = psPosDesc;
                }

                // Save our link address for the next go round
                if (eDirection == TRAFFIC_DIRECTION_POSITIVE)
                {
                    ppsLastPosDesc = &psPosDesc->psPositive;
                }
                else
                {
                    ppsLastPosDesc = &psPosDesc->psNegative;
                }

                if ( ((un8Index+1) == un8NumLocsFound) &&
                     (un8NumLocsFound < un8NumLocsToFind)
                   )
                {
                    // This is the last location we were working with
                    // and it looks like there aren't any more
                    // left in the database, so set the flag
                    // to prevent searching past this point in the future
                    if (eDirection == TRAFFIC_DIRECTION_POSITIVE)
                    {
                        psPosDesc->bPosFinalLocation = TRUE;
                    }
                    else
                    {
                        psPosDesc->bNegFinalLocation = TRUE;
                    }

                    printf(TRAFFIC_MGR_OBJECT_NAME
                     ": Found less locations than expected\n");
                }

                // Destroy the user-created LOCATION
                LOCATION.vDestroy(ahLocations[un8Index]);
            }
        }
    }

    // Keep iterating through all Traffic Messages
    return TRUE;
}

/*****************************************************************************
*
*   psCreatePosDesc
*
*****************************************************************************/
static POS_EVENT_DESC_STRUCT *psCreatePosDesc (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hLocation
        )
{
    POS_EVENT_DESC_STRUCT *psPosDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    TRAFFIC_LOCID_OBJECT hLocID;

    do
    {
        psPosDesc = (POS_EVENT_DESC_STRUCT *)
            SMSO_hCreate(
                TRAFFIC_MGR_OBJECT_NAME":PosEntry",
                sizeof(POS_EVENT_DESC_STRUCT),
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj), FALSE);

        if (psPosDesc == NULL)
        {
            // Error! Stop processing
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": Failed to alloc memory for PosEntry.");
            break;
        }

        // Copy the location and make us the owner
        psPosDesc->hLocation =
            LOCATION_hDuplicate((SMS_OBJECT)psObj->psAppObj,
                                 hLocation);

        psPosDesc->psPositive = NULL;
        psPosDesc->psNegative = NULL;

        hLocID = (TRAFFIC_LOCID_OBJECT)LOCATION.hLocID(psPosDesc->hLocation);
        psPosDesc->tPosCode = TRAFFIC_LOCID.tPosCode(hLocID);

        eReturnCode = OSAL.eLinkedListCreate(
            &psPosDesc->hEventList,
            TRAFFIC_MGR_OBJECT_NAME":PosEntry:EvtList",
            NULL,
            (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE)
                );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error! Stop processing
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": Failed to create PosEntry event list.");
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psPosDesc->hDSRLList,
            TRAFFIC_MGR_OBJECT_NAME":PosEntry:DSRLList",
            n16CompareDSRLDesc,
            (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE)
                );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error! Stop processing
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": Failed to create PosEntry DSRL list.");
            break;
        }

        // Update the position with the current targets
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->hDSRLList,
            (OSAL_LL_ITERATOR_HANDLER)bLinkNewPosDescToDSRLDesc,
            (void *)psPosDesc );

        if(eReturnCode != OSAL_SUCCESS &&
           eReturnCode != OSAL_NO_OBJECTS)
        {
            // Error! Stop processing
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": Failed to add PosEntry to our current targets.");
            break;
        }

#if DEBUG_OBJECT == 1
        printf(TRAFFIC_MGR_OBJECT_NAME
            ": created PosEntry (0x%p) for location (0x%p) at PosCode: %d\n",
            psPosDesc, psPosDesc->hLocation, psPosDesc->tPosCode);
#endif

        return psPosDesc;

    } while(FALSE);

    // Must have errored out
    // free memory and return null
    vReleasePosDesc(psPosDesc);

    return NULL;
}

/*****************************************************************************
*
*   bUpdateTrafficMsgsInTable
*
*****************************************************************************/
static BOOLEAN bUpdateTrafficMsgsInTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    OSAL_LINKED_LIST_ENTRY hStartPosEntry
        )
{
    BOOLEAN bOk;
    TRAFFIC_DIRECTION_ENUM eDirection;
    TRAFFIC_MSG_CLASS_ENUM eClass;
    POS_EVENT_DESC_STRUCT *psPosDesc;
    EVENT_ENTRY_STRUCT *psEventDesc = NULL;
    ALERTC_EVENT_TYPE_ENUM eType;

    TRAFFIC_POS_CODE tCurPosCode = TRAFFIC_MSG_tInitialPosCode(hTrafficMsg);
#if DEBUG_OBJECT == 1

    TRAFFIC_MARKET tMarket = TRAFFIC_MSG_tMarket(hTrafficMsg);
    TRAFFIC_EVENT_CODE tEventCode = TRAFFIC_MSG.tEventCode(hTrafficMsg);

    printf(TRAFFIC_MGR_OBJECT_NAME
        ": Updating table with hTrafficMsg: 0x%p -- %u,%u,%u\n",
        hTrafficMsg, tMarket, tCurPosCode, tEventCode);
#endif

    if (hTrafficMsg == TRAFFIC_MSG_INVALID_OBJECT)
    {
        // Error! Stop processing
        return FALSE;
    }

    // Get the variables from traffic message
    eType = TRAFFIC_MSG_eAlertCType(hTrafficMsg);
    eClass = TRAFFIC_MSG.eClass(hTrafficMsg);
    eDirection = TRAFFIC_MSG.eDirection(hTrafficMsg);

    if (eDirection == TRAFFIC_DIRECTION_UNKNOWN)
    {
        // Error! Stop processing
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFFIC_MGR_OBJECT_NAME": Error updating traffic table: "
            "Unknown Alert-C direction");
        return FALSE;
    }

    // Grab the position description from our list
    psPosDesc = (POS_EVENT_DESC_STRUCT *)
        OSAL.pvLinkedListThis(hStartPosEntry);

    if (psPosDesc == NULL)
    {
        // ERROR!
        return FALSE;
    }

    // Mark the DSRLs for this position as being in the updating state
    OSAL.eLinkedListIterate(
        psPosDesc->hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bNotifyTargets,
        (void *)(size_t)DSRL_STATE_UPDATING);

    // Grab the event
    psEventDesc = psGetEventForPos(psObj, psPosDesc, eDirection, eClass);

    if (psEventDesc == NULL)
    {
        // ERROR!
        return FALSE;
    }

    if (eType == ALERTC_EVENT_TYPE_CANCELLED)
    {
        printf(TRAFFIC_MGR_OBJECT_NAME
            ": Canceling an event at %d for class %d\n",
            tCurPosCode, eClass);

        bOk = bHandleCanceledTrafficMessage(psObj, psPosDesc, psEventDesc);

        if (bOk == FALSE)
        {
            // ERROR!
            printf(TRAFFIC_MGR_OBJECT_NAME
                ": failed to cancel a message\n");
            return FALSE;
        }

        TRAFFIC_MSG_vDestroy(hTrafficMsg);
    }
    else
    {
        if (psEventDesc->hUnstableMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            printf(TRAFFIC_MGR_OBJECT_NAME
                ": Adding an event at %d for class %d\n",
                tCurPosCode, eClass);

            // This is a new message
            bOk = bHandleNewTrafficMessage(psObj, hTrafficMsg, eDirection,
                eClass, psPosDesc, psEventDesc);

            if (bOk == FALSE)
            {
                // ERROR!
                printf(TRAFFIC_MGR_OBJECT_NAME
                    ": failed to add a new message\n");
                return FALSE;
            }
        }
        else
        {
            printf(TRAFFIC_MGR_OBJECT_NAME
                ": Updating an event at %d for class %d\n",
                tCurPosCode, eClass);

            // Handle the udpate
            bOk = bHandleUpdatedTrafficMessage(psObj, hTrafficMsg, eDirection,
                eClass, psPosDesc, psEventDesc);

            if (bOk == FALSE)
            {
                // ERROR!
                printf(TRAFFIC_MGR_OBJECT_NAME
                    ": failed to add a new message\n");
                return FALSE;
            }
        }
    }

    // Make sure the cancellation timer is running
    // since we have messages to process
    vStartCancelTimer(psObj);

    return TRUE;
}

/*****************************************************************************
*
*   psGetEventForPos
*
*****************************************************************************/
static EVENT_ENTRY_STRUCT *psGetEventForPos (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    TRAFFIC_DIRECTION_ENUM eDirection,
    TRAFFIC_MSG_CLASS_ENUM eClass
        )
{
    EVENT_ENTRY_STRUCT *psEventDesc = NULL;
    EVENT_ENTRY_STRUCT sEventDescToFind;
    OSAL_LINKED_LIST_ENTRY hEventEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Construct our search object
    sEventDescToFind.eClass = eClass;
    sEventDescToFind.eDirection = eDirection;

    eReturnCode = OSAL.eLinkedListLinearSearch(
        psPosDesc->hEventList, &hEventEntry,
       (OSAL_LL_COMPARE_HANDLER)n16CompareEventDesc,
       (void *)&sEventDescToFind);

    if (eReturnCode == OSAL_SUCCESS)
    {
        psEventDesc = (EVENT_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(hEventEntry);
    }
    else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        psEventDesc = (EVENT_ENTRY_STRUCT *)
            SMSO_hCreate(
                TRAFFIC_MGR_OBJECT_NAME":EventDesc",
                sizeof(EVENT_ENTRY_STRUCT),
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj), FALSE);

        // Set up the members if we were able to create the memory
        if (psEventDesc != NULL)
        {
            psEventDesc->eClass = eClass;
            psEventDesc->eDirection = eDirection;

            // Add this message to our event list
            OSAL.eLinkedListAdd(psPosDesc->hEventList, &hEventEntry, psEventDesc);
        }
    }

    return psEventDesc;
}

/*****************************************************************************
*
*   bHandleCanceledTrafficMessage
*
*****************************************************************************/
static BOOLEAN bHandleCanceledTrafficMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EVENT_ENTRY_STRUCT *psHeadEventDesc
        )
{
    UN8 un8NumLocsReq, un8Index;
    BOOLEAN bSuccess = TRUE;
    EVENT_ENTRY_STRUCT *psCurEventDesc = psHeadEventDesc;
    TRAFFIC_LOCID_OBJECT hLocID;
    TRAFFIC_POS_CODE tOriginalPosCode, tPosCodeToCancel;

    printf(TRAFFIC_MGR_OBJECT_NAME
        ": Canceling Message\n");

    if (psHeadEventDesc->hUnstableMsg != TRAFFIC_MSG_INVALID_OBJECT)
    {
        TRAFFIC_MSG_OBJECT hTrafficMsgToRemove =
            psHeadEventDesc->hUnstableMsg;

        // Grab the original  inital location for this message
        tOriginalPosCode = TRAFFIC_MSG_tInitialPosCode(hTrafficMsgToRemove);

        // Get the PosCode we are trying to cancel
        hLocID = LOCATION.hLocID(psPosDesc->hLocation);
        tPosCodeToCancel = TRAFFIC_LOCID.tPosCode(hLocID);

        if (tOriginalPosCode != tPosCodeToCancel)
        {
            // Just ingore this cancelation message
            return TRUE;
        }

        // Find the number of extents for the original traffic message
        un8NumLocsReq =
            TRAFFIC_MSG_un8InitialNumLocs(hTrafficMsgToRemove)+1;

        printf(TRAFFIC_MGR_OBJECT_NAME
            ": Num locations to cancel:%d\n", un8NumLocsReq);

        // Clear the handles to the head entry in all
        // the psEventDesc for this chain
        for (un8Index = 0;
             un8Index < un8NumLocsReq && psPosDesc != NULL;
             un8Index++)
        {
            if (psCurEventDesc->hUnstableMsg != TRAFFIC_MSG_INVALID_OBJECT)
            {
                BOOLEAN bMsgDestroyed = FALSE;

                // Remove the location
                TRAFFIC_MSG_bRemoveLocation(psCurEventDesc->hUnstableMsg,
                    psPosDesc->hLocation, &bMsgDestroyed);

                printf(TRAFFIC_MGR_OBJECT_NAME
                    ": Canceled message at %u\n", TRAFFIC_LOCID.tPosCode(hLocID));

                if (bMsgDestroyed == TRUE)
                {
                    printf(TRAFFIC_MGR_OBJECT_NAME
                        ": Removing 0x%p from DSRL\n", psCurEventDesc->hUnstableMsg);
                    // We cancled a message, so remove the pointer from the DSRL
                    // which should be equal to the unstable message since that is
                    // copied to the stable message
                    // Why am I doing this here instead of in bMarkEventAsComplete?
                    vUpdateTargetList (
                        psPosDesc->hDSRLList,
                        psCurEventDesc->hUnstableMsg,
                        TRAFFIC_MSG_INVALID_OBJECT,
                        DSRL_ENTRY_STATUS_REMOVED);
                }

                psCurEventDesc->hUnstableMsg = TRAFFIC_MSG_INVALID_OBJECT;
            }

            // Move to the next event
            if (psCurEventDesc->eDirection == TRAFFIC_DIRECTION_POSITIVE)
            {
                psPosDesc = psPosDesc->psPositive;
            }
            else
            {
                psPosDesc = psPosDesc->psNegative;
            }

            if (psPosDesc == NULL)
            {
                // Ran out of positions
                break;
            }

            // Grab the next event
            psCurEventDesc =
                psGetEventForPos(psObj, psPosDesc,
                    psCurEventDesc->eDirection,
                    psCurEventDesc->eClass);

            if (psCurEventDesc == NULL)
            {
                // Some sort of error
                bSuccess = FALSE;
                break;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleNewTrafficMessage
*
*****************************************************************************/
static BOOLEAN bHandleNewTrafficMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFFIC_DIRECTION_ENUM eDirection,
    TRAFFIC_MSG_CLASS_ENUM eClass,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EVENT_ENTRY_STRUCT *psHeadEventDesc
        )
{
    UN8 un8NumExtent;
    BOOLEAN bSuccess = TRUE;
    EVENT_ENTRY_STRUCT *psCurEventDesc;

#if DEBUG_OBJECT == 1
    TRAFFIC_LOCID_OBJECT hLocID;
#endif

    printf(TRAFFIC_MGR_OBJECT_NAME
        ": Creating Message\n");

    // Set the traffic message in the head entry
    psHeadEventDesc->hUnstableMsg = hTrafficMsg;

    // Add the initial location to our traffic message
    // so the app can access it
    TRAFFIC_MSG_bAddLocation(psHeadEventDesc->hUnstableMsg, psPosDesc->hLocation);

    // Figure out how many items to modify
    un8NumExtent =
        TRAFFIC_MSG_un8InitialNumLocs(psHeadEventDesc->hUnstableMsg);

    while (un8NumExtent > 0)
    {
        if (eDirection == TRAFFIC_DIRECTION_POSITIVE)
        {
            psPosDesc = psPosDesc->psPositive;
        }
        else
        {
            psPosDesc = psPosDesc->psNegative;
        }

        if (psPosDesc == NULL)
        {
            // Hit the end of the road, exit the loop
            break;
        }

        // Grab the event
        psCurEventDesc =
            psGetEventForPos(psObj, psPosDesc, eDirection, eClass);

        if (psCurEventDesc == NULL)
        {
            bSuccess = FALSE;
            break;
        }

        if (psCurEventDesc->hUnstableMsg != TRAFFIC_MSG_INVALID_OBJECT)
        {
            printf(TRAFFIC_MGR_OBJECT_NAME
                ": found a member of the chain (while creating) that is part of a different message (0x%p)\n",
                psCurEventDesc->hUnstableMsg);

            vHandleOverlappedMessage(psObj,
                hTrafficMsg, psCurEventDesc);
        }

        // Set our event
        psCurEventDesc->hUnstableMsg = psHeadEventDesc->hUnstableMsg;

#if DEBUG_OBJECT == 1
        hLocID = LOCATION.hLocID(psPosDesc->hLocation);
        printf(TRAFFIC_MGR_OBJECT_NAME
            ": Set message at %u\n", TRAFFIC_LOCID.tPosCode(hLocID));
#endif

        // Add this location to our traffic message
        // so the app can access it
        TRAFFIC_MSG_bAddLocation(psCurEventDesc->hUnstableMsg,
            psPosDesc->hLocation);

        // Move on
        un8NumExtent--;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleUpdatedTrafficMessage
*
*****************************************************************************/
static BOOLEAN bHandleUpdatedTrafficMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFFIC_DIRECTION_ENUM eDirection,
    TRAFFIC_MSG_CLASS_ENUM eClass,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EVENT_ENTRY_STRUCT *psHeadEventDesc
        )
{
    N16 n16Compare;
    BOOLEAN bSuccess = TRUE;
#if DEBUG_OBJECT == 1
    TRAFFIC_LOCID_OBJECT hLocID;
#endif

    printf(TRAFFIC_MGR_OBJECT_NAME
        ": Updating Message\n");

    // We have an active event at this posistion
    // Are they equal?
    n16Compare = TRAFFIC_MSG_n16Equal(hTrafficMsg, psHeadEventDesc->hUnstableMsg);
    if (n16Compare == 0)
    {
        // Clear the flag so that this message gets written to the DB again
        TRAFFIC_MSG_vSetWrittenToDBFlag(psHeadEventDesc->hUnstableMsg, FALSE);

        // Modify the timestamp for the message
        TRAFFIC_MSG_vUpdateTimeStamp(psHeadEventDesc->hUnstableMsg);

        // This is a repeated message, so destroy it
        TRAFFIC_MSG_vDestroy(hTrafficMsg);

        printf(TRAFFIC_MGR_OBJECT_NAME
            ": repeat message for %d\n",
            TRAFFIC_MSG.tEventCode(psHeadEventDesc->hUnstableMsg));
    }
    else
    {
        // This is a different message
        UN8 un8Index;
        EVENT_ENTRY_STRUCT *psCurEventDesc = psHeadEventDesc;

        // Find the number of extents for the new traffic message
        UN8 un8NumLocsReq =
            TRAFFIC_MSG_un8InitialNumLocs(hTrafficMsg)+1;

        printf(TRAFFIC_MGR_OBJECT_NAME
            ": num locations to update is %d\n",un8NumLocsReq);

        // Modify the handles in the chain
        for (un8Index = 0;
             un8Index < un8NumLocsReq && psPosDesc != NULL;
             un8Index++)
        {
            if (psCurEventDesc->hUnstableMsg != TRAFFIC_MSG_INVALID_OBJECT)
            {

                printf(TRAFFIC_MGR_OBJECT_NAME
                    ": previous message was 0x%p\n", psCurEventDesc->hUnstableMsg);

                vHandleOverlappedMessage(psObj,
                    hTrafficMsg, psCurEventDesc);
            }
            else
            {
                printf(TRAFFIC_MGR_OBJECT_NAME": no previous message\n");
            }
#if DEBUG_OBJECT == 1
            hLocID = LOCATION.hLocID(psPosDesc->hLocation);
            printf(TRAFFIC_MGR_OBJECT_NAME
                ": Updated message at %u\n", TRAFFIC_LOCID.tPosCode(hLocID));
#endif

            // Save the handle
            psCurEventDesc->hUnstableMsg = hTrafficMsg;

            // Add this location to our traffic message
            // so the app can access it
            TRAFFIC_MSG_bAddLocation(hTrafficMsg, psPosDesc->hLocation);

            // Move to the next event
            if (psHeadEventDesc->eDirection == TRAFFIC_DIRECTION_POSITIVE)
            {
                psPosDesc = psPosDesc->psPositive;
            }
            else
            {
                psPosDesc = psPosDesc->psNegative;
            }

            if (psPosDesc == NULL)
            {
                // Ran out of positions
                break;
            }

            // Grab the next event
            psCurEventDesc =
                psGetEventForPos(psObj, psPosDesc,
                    eDirection, eClass);

            if (psCurEventDesc == NULL)
            {
                // Some sort of error
                bSuccess = FALSE;
                break;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vHandleOverlappedMessage
*
*****************************************************************************/
static void vHandleOverlappedMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    EVENT_ENTRY_STRUCT *psOverlapedEventDesc
        )
{

    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    TRAFFIC_MARKET tMarket;
    POS_EVENT_DESC_STRUCT *psPosDesc;
    EVENT_ENTRY_STRUCT *psHeadEventDesc;

    // Do some sort of conditioning here?

    // Find the position of the original message
    tMarket = TRAFFIC_MSG_tMarket(psOverlapedEventDesc->hUnstableMsg);

    if ( tMarket != TRAFFIC_INVALID_MARKET )
    {
        POS_EVENT_DESC_STRUCT sPosSearch;

        sPosSearch.tPosCode =
            TRAFFIC_MSG_tInitialPosCode(psOverlapedEventDesc->hUnstableMsg);
        eReturnCode =
            OSAL.eLinkedListSearch(psObj->psAppObj->ahTrafficTable[tMarket-1],
                                   &hEntry,
                                   (void*)&sPosSearch);

        if (eReturnCode == OSAL_SUCCESS)
        {
            // grab the pointer to our entry
            psPosDesc = (POS_EVENT_DESC_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

            // Find the event description for this position
            psHeadEventDesc = psGetEventForPos (
                psObj, psPosDesc, psOverlapedEventDesc->eDirection,
                psOverlapedEventDesc->eClass);

            if (psHeadEventDesc != NULL)
            {
                // Wipe it out
                bHandleCanceledTrafficMessage(psObj, psPosDesc, psHeadEventDesc);
            }
        }
    }
}

/*****************************************************************************
*
*   bNotifyTargets
*
*****************************************************************************/
static BOOLEAN bNotifyTargets (
    DSRL_DESC_STRUCT *psDSRLDesc,
    void *pvArg
        )
{
    DSRL_STATE_ENUM eState =
        (DSRL_STATE_ENUM)(size_t)pvArg;
    BOOLEAN bLocked;

    if (psDSRLDesc != NULL)
    {
        bLocked = SMSO_bLock((SMS_OBJECT)psDSRLDesc->hDSRL,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == TRUE)
        {
            // Tell the DSRL to notify the listener if there
            // are any pending changes
            DSRL_vSetState(psDSRLDesc->hDSRL, eState);

            SMSO_vUnlock((SMS_OBJECT)psDSRLDesc->hDSRL);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   vUpdateTargetList
*
*****************************************************************************/
static void vUpdateTargetList (
    OSAL_OBJECT_HDL hDSRLList,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFFIC_MSG_OBJECT hOldTrafficMsg,
    DSRL_ENTRY_STATUS_ENUM eStatus
        )
{
    TARGET_UPDATE_STRUCT sUpdateInfo;

    sUpdateInfo.hMsg = hTrafficMsg;
    sUpdateInfo.hOldMsg = hOldTrafficMsg;
    sUpdateInfo.eStatus = eStatus;

    // Iterate through the list of targets and tell each DSRL
    OSAL.eLinkedListIterate(hDSRLList,
        (OSAL_LL_ITERATOR_HANDLER)bUpdateTarget, (void *)&sUpdateInfo);

    return;
}

/*****************************************************************************
*
*   bUpdateTarget
*
*****************************************************************************/
static BOOLEAN bUpdateTarget (
    DSRL_DESC_STRUCT *psDSRLDesc,
    TARGET_UPDATE_STRUCT *psUpdate
        )
{
    DSRL_ADD_REPLACE_RESULT_ENUM eResult = DSRL_ADD_REPLACE_ERROR;

    switch (psUpdate->eStatus)
    {
        case DSRL_ENTRY_STATUS_NEW:
        {
            // Make the call to add the entry to the DSRL
            eResult = DSRL_eAddEntry(psDSRLDesc->hDSRL, (DSRL_ENTRY_OBJECT)psUpdate->hMsg);

            if (eResult == DSRL_ADD_REPLACE_ERROR)
            {
                printf(TRAFFIC_MGR_OBJECT_NAME": ERROR, can't add 0x%p to DSRL\n", psUpdate->hMsg);
            }
        }
        break;

        case DSRL_ENTRY_STATUS_CHANGED:
        {
            eResult = DSRL_eReplaceEntry(psDSRLDesc->hDSRL,
                (DSRL_ENTRY_OBJECT)psUpdate->hOldMsg, (DSRL_ENTRY_OBJECT)psUpdate->hMsg);

            if (eResult == DSRL_ADD_REPLACE_ERROR)
            {
                printf(TRAFFIC_MGR_OBJECT_NAME": ERROR, can't replace 0x%p with 0x%p in DSRL\n",
                    psUpdate->hOldMsg, psUpdate->hMsg);

                // We can't replace the old entry with this new entry,
                // so remove the old entry
                DSRL_vRemoveEntry(psDSRLDesc->hDSRL,
                    (DSRL_ENTRY_OBJECT)psUpdate->hMsg);
            }
        }
        break;

        case DSRL_ENTRY_STATUS_REMOVED:
        {
            // Remove this message the DSRL
            DSRL_vRemoveEntry(psDSRLDesc->hDSRL, (DSRL_ENTRY_OBJECT)psUpdate->hMsg);
        }
        break;

        case DSRL_ENTRY_STATUS_UNCHANGED:
        case DSRL_ENTRY_STATUS_UNKNOWN:
        default:
        break;
    }

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bCheckMarketList
*
*****************************************************************************/
static BOOLEAN bCheckMarketList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
                OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bInFilter = FALSE;
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;

    psAppObj = psGetAppFacingObject((TRAFFIC_SERVICE_OBJECT)psObj);

    if (psAppObj != NULL)
    {
        // Look to see if this TRAFFIC_LOCID matches anything
        // in our market filter list
        eReturnCode = OSAL.eLinkedListSearch(
            psAppObj->hMarketFilterList, &hEntry,
            (void *) hTrafficLocID);

        if (eReturnCode == OSAL_SUCCESS)
        {
            bInFilter = TRUE;
        }

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return bInFilter;
}

/*****************************************************************************
*
*   bInitAppFacingObject
*
*****************************************************************************/
static BOOLEAN bInitAppFacingObject(
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;
    BOOLEAN bResult = FALSE;

    // Begin DSRL config
    DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

    // Set service type
    sDSRLConfig.eServiceType = DATASERVICE_TYPE_TRAFFIC;

    // Set the parent object size
    sDSRLConfig.tParentObjectSize = sizeof(TRAFFIC_APP_OBJECT_STRUCT);

    // Configure the device feature
    sDSRLConfig.bDeviceEnabled = TRUE;
    sDSRLConfig.un32DeviceNotifyDistance = TRAFFIC_DEVICE_DISTANCE_THRESHOLD;
    sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

    do 
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Create it
        psObj->psAppObj = (TRAFFIC_APP_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hConfigureDSRL(
            (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);
        if (psObj->psAppObj == NULL)
        {
            break;
        }

        // Create the linked-list used to store the market/BSA filter
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->psAppObj->hMarketFilterList,
            TRAFFIC_MGR_OBJECT_NAME":MarketFilterList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareTrafficLocID,
            (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE)
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // create dummy event for search operations
        psObj->psAppObj->hDummyAlertCEvent = ALERTC_EVENT_hCreate(
            (SMS_OBJECT)psObj->psAppObj, TRAFFIC_INVALID_EVENT_CODE, 
            ALERTC_EVENT_TYPE_UNKNOWN, TRAFFIC_MSG_CLASS_UNKNOWN, 
            ALERTC_QUANTIFIER_TYPE_UNKNOWN, STRING_INVALID_OBJECT,
            STRING_INVALID_OBJECT);
        if (psObj->psAppObj->hDummyAlertCEvent == ALERTC_EVENT_INVALID_OBJECT)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if(bResult != TRUE)
    {
        vUninitAppFacingObject(psObj);
    }
    else
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)psObj->psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   vUninitAppFacingObject
*
*****************************************************************************/
static void vUninitAppFacingObject(
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bLocked;
    UN8 un8MarketIdx;

    // Lock the app object
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TRAFFIC_MGR_OBJECT_NAME": Unable to lock app object");
        return;
    }

    if (psObj->psAppObj->hMarketFilterList != OSAL_INVALID_OBJECT_HDL)
    {

        // Destroy the market linked list
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->psAppObj->hMarketFilterList,
            vReleaseMarketFilterListNode);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psObj->psAppObj->hMarketFilterList);
            psObj->psAppObj->hMarketFilterList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Disconnect from the database application
    if (psObj->psAppObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect( psObj->psAppObj->hSQLRefConnection );
        psObj->psAppObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psObj->psAppObj->pacFullyQualifiedRefDatabaseFilePath != NULL)
    {
        // Remove database path resources
        SMSO_vDestroy( (SMS_OBJECT)
            &psObj->psAppObj->pacFullyQualifiedRefDatabaseFilePath[0]);
    }

    if (psObj->psAppObj->hSQLPersistantConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->psAppObj->hSQLPersistantConnection,
            psObj->psAppObj->hInsertMsgIntoDBStmt);
        psObj->psAppObj->hInsertMsgIntoDBStmt = SQL_INTERFACE_INVALID_OBJECT;

        SQL_INTERFACE.vDisconnect( psObj->psAppObj->hSQLPersistantConnection );
        psObj->psAppObj->hSQLPersistantConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psObj->psAppObj->pacFullyQualifiedPersistantDatabaseFilePath != NULL)
    {
        // Remove database path resources
        SMSO_vDestroy( (SMS_OBJECT)
            &psObj->psAppObj->pacFullyQualifiedPersistantDatabaseFilePath[0]);
    }

    if (psObj->psAppObj->hDummyAlertCEvent != ALERTC_EVENT_INVALID_OBJECT)
    {
        ALERTC_EVENT_vDestroy(psObj->psAppObj->hDummyAlertCEvent);
        psObj->psAppObj->hDummyAlertCEvent = ALERTC_EVENT_INVALID_OBJECT;
    }

    // Remove the cache of Alert-C Event Info that was pulled
    // from the database
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psObj->psAppObj->hAlertCEventList,
            vReleaseAlertCEventInfo);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->psAppObj->hAlertCEventList);
        psObj->psAppObj->hAlertCEventList = OSAL_INVALID_OBJECT_HDL;
    }

    // Remove the cache of Alert-C Supplemental Info that was pulled
    // from the database
    eReturnCode = OSAL.eLinkedListRemoveAll(psObj->psAppObj->hSupplInfoList,
            vReleaseSupplInfo);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->psAppObj->hSupplInfoList);
        psObj->psAppObj->hSupplInfoList = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the unprocessed traffic table and active traffic table
    for (un8MarketIdx = 0;
         un8MarketIdx <NUM_TRAFFIC_MARKETS;
         un8MarketIdx++)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->psAppObj->ahTrafficTable[un8MarketIdx],
            vReleasePosDesc);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(
                psObj->psAppObj->ahTrafficTable[un8MarketIdx]);
        }
    }

    // Destroy the app object
    DATASERVICE_IMPL_vDestroyDSRLParent((DATASERVICE_IMPL_HDL)psObj, (SMS_OBJECT)psObj->psAppObj);
    psObj->psAppObj = NULL;

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Destroy the target list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psObj->hDSRLList, (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLNode);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->hDSRLList);
        psObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
    }

    // Drain the Traffic Message Pool
    TRAFFIC_MSG_vUninstall();

    // Uninit parser here and null it out.
    if (GsTrafficIntf.vUninitParser != NULL)
    {
        // Make the call to uninit the broadcast-specific parser?
        GsTrafficIntf.vUninitParser(psObj->hTrafficInterfaceData);
        psObj->hTrafficInterfaceData = TRAFFIC_INTERFACE_INVALID_OBJECT;
    }

    // Clear the cancel event handle, we're done with it
    psObj->hCancelEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

    // Remove any broadcast-specific parsing data
    if (GsTrafficIntf.vUninitParser != NULL)
    {
        // Make the call to uninit the broadcast-specific parser?
        GsTrafficIntf.vUninitParser((SMS_OBJECT)psObj->hTrafficInterfaceData);
        psObj->hTrafficInterfaceData = TRAFFIC_INTERFACE_INVALID_OBJECT;
    }

    // Uninit the app object
    vUninitAppFacingObject(psObj);

    if (bFullDelete == TRUE)
    {
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    // Destroy the SMS Update Event object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   vReleaseMarketFilterListNode
*
*****************************************************************************/
static void vReleaseMarketFilterListNode (
    void *pvData
        )
{
    TRAFFIC_LOCID_OBJECT hTrafficLocID =
        (TRAFFIC_LOCID_OBJECT)pvData;

    TRAFFIC_LOCID.vDestroy(hTrafficLocID);

    return;
}

/*****************************************************************************
*
*   vReleasePosDesc
*
*****************************************************************************/
static void vReleasePosDesc (
    void *pvData
        )
{
    POS_EVENT_DESC_STRUCT *psObj =
        (POS_EVENT_DESC_STRUCT *)pvData;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psObj != NULL)
    {
        LOCATION.vDestroy(psObj->hLocation);

        eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hDSRLList, NULL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psObj->hDSRLList);
        }

        // Release the event description.  The traffic message
        // refrenced will be release when we drain the pool
        eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hEventList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseEventDesc);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psObj->hEventList);
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   vReleaseEventDesc
*
*****************************************************************************/
static void vReleaseEventDesc (
    EVENT_ENTRY_STRUCT *psEventDesc
        )
{
    SMSO_vDestroy((SMS_OBJECT)psEventDesc);

    return;
}

/*****************************************************************************
*
*   vReleaseDSRLNode
*
*****************************************************************************/
static void vReleaseDSRLNode (
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    DSRL_vDestroy(psDSRLDesc->hDSRL);

    eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetList,
        (OSAL_LL_RELEASE_HANDLER)vReleaseTargetNode);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psDSRLDesc->hTargetList);
    }

    SMSO_vDestroy((SMS_OBJECT)psDSRLDesc);

    return;
}

/*****************************************************************************
*
*   vReleaseTargetNode
*
*****************************************************************************/
static void vReleaseTargetNode (
    TARGET_DESC_STRUCT *psTargetDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    LOCATION.vDestroy(psTargetDesc->hLocation);

    eReturnCode = OSAL.eLinkedListRemoveAll(psTargetDesc->hPosList, NULL);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psTargetDesc->hPosList);
    }

    SMSO_vDestroy((SMS_OBJECT)psTargetDesc);

    return;
}

/*****************************************************************************
*
*   vReleaseAlertCEventInfo
*
*****************************************************************************/
static void vReleaseAlertCEventInfo (
    void *pvData
        )
{
    ALERTC_EVENT_vDestroy((ALERTC_EVENT_OBJECT)pvData);

    return;
}

/*****************************************************************************
*
*   vReleaseSupplInfo
*
*****************************************************************************/
static void vReleaseSupplInfo (
    void *pvData
        )
{
    SUPPL_INFO_STRUCT *psObj =
        (SUPPL_INFO_STRUCT *)pvData;

    if (psObj != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pcText);
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;

    do
    {
        // Validate the hTrafficService, then get a pointer to the
        // locked app-facing object
        psAppObj = psGetAppFacingObject((TRAFFIC_SERVICE_OBJECT) psObj);

        if (psAppObj == NULL)
        {
            // Error, not sure why this would happen
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Register for a timed event
        psObj->hCancelEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);

        if(psObj->hCancelEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Initilize the Traffic Message Pool
        bOk = TRAFFIC_MSG_bInstall((SMS_OBJECT)psAppObj,
            (TRAFFIC_SERVICE_OBJECT) psObj);

        if (bOk == FALSE)
        {
            printf(TRAFFIC_MGR_OBJECT_NAME
                "Traffic Message Pool Not Available\n");

            // Error, not sure why this would happen
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Construct the paths needed by the traffic data service
        // (based on information gathered from the config manager)
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            TRAFFIC_DATABASE_FOLDER,
            TRAFFIC_REF_DATABASE_FILENAME,
            &psAppObj->pacFullyQualifiedRefDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Connect to the traffic ref database (read-only)
        psAppObj->hSQLRefConnection =
            DB_UTIL_hConnectToReference(
                &psAppObj->pacFullyQualifiedRefDatabaseFilePath[0],
                bVerifyDBVersion,
                NULL,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY);

        // If the connection succeeded, continue with
        // initial processing
        if (psAppObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            vSetError( psObj, eErrorCode );

            bOk = FALSE;
            break;
        }

        // Build the path to the persitant storage DB
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            TRAFFIC_DATABASE_FOLDER,
            TRAFFIC_PERSIST_DATABASE_FILENAME,
            &psAppObj->pacFullyQualifiedPersistantDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        // Connect to the traffic persitant storage database
        psAppObj->hSQLPersistantConnection =
            DB_UTIL_hConnectToPersistent(
                &psAppObj->pacFullyQualifiedPersistantDatabaseFilePath[0],
                bCreatePersistDBTables,
                (void *)NULL,
                bVerifyDBVersion,
                NULL,
                &eErrorCode);

        if (psAppObj->hSQLPersistantConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            vSetError( psObj, eErrorCode );

            bOk = FALSE;
            break;
        }

        // Clean up the DB from expired message
        bOk = bRemoveExpiredMessagesFromDatabase(psAppObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": failed to clean up DB from expired messages");
            break;
        }

        // Restore markets
        vRestoreAllMarketsFromDatabase(psObj, psAppObj);

    } while(FALSE);

    if (NULL != psAppObj)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return bOk;
}

/*****************************************************************************
*
*   bHandleServiceError
*
*****************************************************************************/
static BOOLEAN bHandleServiceError (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    vStopCancelTimer(psObj);

    return TRUE;
}

/*****************************************************************************
*
*   n16CompareTrafficLocID
*
*****************************************************************************/
static N16 n16CompareTrafficLocID (
    void *pvObj1,
    void *pvObj2
        )
{
    TRAFFIC_LOCID_OBJECT hTrafficLocID1 =
        (TRAFFIC_LOCID_OBJECT)pvObj1;
    TRAFFIC_LOCID_OBJECT hTrafficLocID2 =
        (TRAFFIC_LOCID_OBJECT)pvObj2;

    N16 n16Result =
        TRAFFIC_LOCID.n16CompareBSAs(hTrafficLocID1, hTrafficLocID2);

    if (n16Result != 0)
    {
        n16Result =
            TRAFFIC_LOCID.n16CompareMarkets(hTrafficLocID1, hTrafficLocID2);
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareDSRLDesc
*
*****************************************************************************/
static N16 n16CompareDSRLDesc (
    void *pvObj1,
    void *pvObj2
        )
{
    DSRL_DESC_STRUCT *psObj1 =
        (DSRL_DESC_STRUCT *)pvObj1;
    DSRL_DESC_STRUCT *psObj2 =
        (DSRL_DESC_STRUCT *)pvObj2;
    N16 n16Result = N16_MIN;

    if (psObj1 != NULL && psObj2 != NULL)
    {
        // Just do a pointer-compare of the DSRL handles
        n16Result = COMPARE(psObj1->hDSRL, psObj2->hDSRL);
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareTargetDesc
*
*****************************************************************************/
static N16 n16CompareTargetDesc (
    void *pvObj1,
    void *pvObj2
        )
{
    TARGET_DESC_STRUCT *psObj1 =
        (TARGET_DESC_STRUCT *)pvObj1;
    TARGET_DESC_STRUCT *psObj2 =
        (TARGET_DESC_STRUCT *)pvObj2;
    N16 n16Result = N16_MIN;

    if (psObj1 != NULL && psObj2 != NULL)
    {
        // Compare the locations
        n16Result = (N16)LOCATION_bCompare(psObj1->hLocation, psObj2->hLocation);
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareEventDesc
*
*****************************************************************************/
static N16 n16CompareEventDesc (
    void *pvObj1,
    void *pvObj2
        )
{
    EVENT_ENTRY_STRUCT *psObj1 =
        (EVENT_ENTRY_STRUCT *)pvObj1;
    EVENT_ENTRY_STRUCT *psObj2 =
        (EVENT_ENTRY_STRUCT *)pvObj2;
    N16 n16Result = N16_MIN;

    if (psObj1 != NULL && psObj2 != NULL)
    {
        // Compare the direction first, then the class
        if (psObj1->eDirection == psObj2->eDirection)
        {
            if (psObj1->eClass == psObj2->eClass)
            {
                n16Result = 0;
            }
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16ComparePosCode
*
*****************************************************************************/
static N16 n16ComparePosCode (
    void *pvObj1,
    void *pvObj2
        )
{
    POS_EVENT_DESC_STRUCT *psPosEventDesc1 =
        (POS_EVENT_DESC_STRUCT *)pvObj1;
    POS_EVENT_DESC_STRUCT *psPosEventDesc2 =
        (POS_EVENT_DESC_STRUCT *)pvObj2;

    if (NULL == psPosEventDesc1)
    {
        if (NULL == psPosEventDesc2)
        {
            return 0;
        }

        return -1;
    }
    else if (NULL == psPosEventDesc2)
    {
        return 1;
    }

    return COMPARE(psPosEventDesc1->tPosCode, psPosEventDesc2->tPosCode);
}

/*****************************************************************************
*
*   n16CompareSupplInfoCode
*
*****************************************************************************/
static N16 n16CompareSupplInfoCode (
    SUPPL_INFO_STRUCT *psSupplInfo1,
    SUPPL_INFO_STRUCT *psSupplInfo2
        )
{
    N16 n16Return = N16_MIN;

    if ((psSupplInfo1 != NULL) &&
        (psSupplInfo2 != NULL))
    {
        n16Return = COMPARE(psSupplInfo1->tCode, psSupplInfo2->tCode);
    }

    return n16Return;
}

/*****************************************************************************
*
*   un8FindNumLocationsKnown
*
*****************************************************************************/
static UN8 un8FindNumLocationsKnown (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    TRAFFIC_POS_CODE *ptPosCode,
    TRAFFIC_MARKET tMarket,
    TRAFFIC_DIRECTION_ENUM eDirection,
    UN8 un8NumLocsReq,
    BOOLEAN *pbFinalLocation
        )
{
    UN8 un8NumLocationsKnown = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    POS_EVENT_DESC_STRUCT *psPosDesc;
    TRAFFIC_LOCID_OBJECT hCurLocID;
    POS_EVENT_DESC_STRUCT sPosSearch;

    // Find the location in the list
    sPosSearch.tPosCode = *ptPosCode;

    eReturnCode =
        OSAL.eLinkedListSearch(psObj->psAppObj->ahTrafficTable[tMarket-1],
                               phEntry,
                               (void*)&sPosSearch);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // grab the pointer to our entry
        psPosDesc = (POS_EVENT_DESC_STRUCT *)
            OSAL.pvLinkedListThis(*phEntry);
        if (eDirection == TRAFFIC_DIRECTION_POSITIVE)
        {
            *pbFinalLocation = psPosDesc->bPosFinalLocation;
        }
        else
        {
            *pbFinalLocation = psPosDesc->bNegFinalLocation;
        }

        un8NumLocationsKnown++;

        // Find the rest of the locations
        while (un8NumLocationsKnown < un8NumLocsReq &&
               psPosDesc != NULL)
        {
            // Grab the next description
            if (eDirection == TRAFFIC_DIRECTION_POSITIVE)
            {
                psPosDesc = psPosDesc->psPositive;
            }
            else
            {
                psPosDesc = psPosDesc->psNegative;
            }

            if (psPosDesc != NULL)
            {
                un8NumLocationsKnown++;
                hCurLocID = LOCATION.hLocID(psPosDesc->hLocation);
                *ptPosCode = TRAFFIC_LOCID.tPosCode(hCurLocID);

                if (eDirection == TRAFFIC_DIRECTION_POSITIVE)
                {
                    *pbFinalLocation = psPosDesc->bPosFinalLocation;
                }
                else
                {
                    *pbFinalLocation = psPosDesc->bNegFinalLocation;
                }
            }
        }
    }

    return un8NumLocationsKnown;
}

/*****************************************************************************
*
*   bCreateLinksWithDSRLAndTable
*
*****************************************************************************/
static BOOLEAN bCreateLinksWithDSRLAndTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    OSAL_FIXED_OBJECT hLat, hLon;
    LOCID_OBJECT hLocID;
    TRAFFIC_MARKET tMarket;
    DSRL_TARGET_TYPE_ENUM eTargetType;
    UN8 un8Index;
    size_t tDSRLTargetIndex;
    TARGET_DESC_STRUCT *psCurTargetDesc;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = TRUE;
    TRAFFIC_LINK_TARGET_DESC_ITERATOR_STRUCT sIterator;

    sIterator.psDSRLDesc = psDSRLDesc;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        // Make sure the data target type is a location
        eTargetType = DSRL_TARGET.eType(
            psDSRLArg->ahTargetList[tDSRLTargetIndex]);

        if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
        {
            // Error
            bSuccess = FALSE;
            break;
        }

        // // Extract the lat/lon/locid from this target
        hLat = LOCATION.hLat(psDSRLArg->ahTargetList[tDSRLTargetIndex]);
        hLon = LOCATION.hLon(psDSRLArg->ahTargetList[tDSRLTargetIndex]);
        hLocID = LOCATION.hLocID(psDSRLArg->ahTargetList[tDSRLTargetIndex]);

        // See what we have here
        if ((hLat == OSAL_FIXED_INVALID_OBJECT) &&
            (hLon == OSAL_FIXED_INVALID_OBJECT) &&
            (hLocID == LOCID_INVALID_OBJECT))
        {
            // Incomplete location -- just skip it for now
            continue;
        }


        // Create a target descrption for this location
        psCurTargetDesc = (TARGET_DESC_STRUCT *)
            SMSO_hCreate(
                TRAFFIC_MGR_OBJECT_NAME":TargetDesc",
                sizeof(TARGET_DESC_STRUCT),
                (SMS_OBJECT)psDSRLDesc,
                FALSE);

        if (psCurTargetDesc == NULL)
        {
            bSuccess = FALSE;
            break;
        }

        // Create a linked-list to keep track of the positions
        // for this target
        eReturnCode = OSAL.eLinkedListCreate(
            &psCurTargetDesc->hPosList,
            TRAFFIC_MGR_OBJECT_NAME":TargetDesc:PosList",
            NULL,
            OSAL_LL_OPTION_LINEAR
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": failed to create a list of positions (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            SMSO_vDestroy((SMS_OBJECT)psCurTargetDesc);
            bSuccess = FALSE;
            break;
        }

        // Save our location handle
        psCurTargetDesc->hLocation =
            psDSRLArg->ahTargetList[tDSRLTargetIndex];

        sIterator.psTargetDesc = psCurTargetDesc;

        if ((hLat == OSAL_FIXED_INVALID_OBJECT) &&
            (hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            if (LOCID.eType(hLocID) == LOCID_TYPE_TRAFFIC)
            {
                tMarket =
                    TRAFFIC_LOCID.tMarket((TRAFFIC_LOCID_OBJECT)hLocID);

                if (tMarket >= NUM_TRAFFIC_MARKETS ||
                    tMarket == 0)
                {
                    // Invalid traffic market
                    SMSO_vDestroy((SMS_OBJECT)psCurTargetDesc);
                    bSuccess = FALSE;
                    break;
                }

                // just set this DSRL for positions in one market
                OSAL.eLinkedListIterate(
                        psObj->psAppObj->ahTrafficTable[tMarket-1],
                        (OSAL_LL_ITERATOR_HANDLER)bLinkTargetDescToPosDesc,
                        (void *)&sIterator);
            }
            else
            {
                SMSO_vDestroy((SMS_OBJECT)psCurTargetDesc);
                bSuccess = FALSE;
                break;
            }
        }
        else
        {
            // Update all our positions if they should be watched by
            // this target
            for (un8Index = 0; un8Index < NUM_TRAFFIC_MARKETS; un8Index++)
            {
                OSAL.eLinkedListIterate(
                    psObj->psAppObj->ahTrafficTable[un8Index],
                    (OSAL_LL_ITERATOR_HANDLER)bLinkTargetDescToPosDesc,
                    (void *)&sIterator);
            }
        }

        // Add this target description to our DSRL description
        eReturnCode = OSAL.eLinkedListAdd(psDSRLDesc->hTargetList,
                            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psCurTargetDesc);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSO_vDestroy((SMS_OBJECT)psCurTargetDesc);
            bSuccess = FALSE;
            break;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleRemoveTargets
*
*****************************************************************************/
static BOOLEAN bHandleRemoveTargets (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN *pbSortNeeded
        )
{
    size_t tDSRLTargetIndex;
    TARGET_DESC_STRUCT sTargetDescSearch;
    TARGET_DESC_STRUCT *psCurTargetDesc;
    POS_EVENT_DESC_STRUCT *psCurPosDesc;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = TRUE, bPosFound;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY,
                           hEntryToRemove = OSAL_INVALID_LINKED_LIST_ENTRY,
                           hDSRLDescToRemove = OSAL_INVALID_LINKED_LIST_ENTRY,
                           hFirstTarget;

    // Initialize sort flag
    *pbSortNeeded = FALSE;

    // Get the first target entry in the descriptor.  If we end up removing
    // this target it means we need to re-sort the DSRL (the default sort is
    // by the first target's location)
    hFirstTarget = OSAL.hLinkedListFirst(
        psDSRLDesc->hTargetList, (void **)NULL);

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        // Find the current target in our list
        sTargetDescSearch.hLocation =
            psDSRLArg->ahTargetList[tDSRLTargetIndex];
        eReturnCode =
            OSAL.eLinkedListLinearSearch(psDSRLDesc->hTargetList,
                                         &hEntry,
                                         n16CompareTargetDesc,
                                         &sTargetDescSearch);

        // If this target isn't in our list, contiune
        if (eReturnCode != OSAL_SUCCESS)
        {
            continue;
        }

        // Extract target descriptor
        psCurTargetDesc = (TARGET_DESC_STRUCT *)OSAL.pvLinkedListThis(hEntry);

        // Is this our first target?
        if (hEntry == hFirstTarget)
        {
            // Yes, we need to re-sort the DSRL
            // when we're done processing this request
            *pbSortNeeded = TRUE;
        }

        // Remove the target entry from our list of targets for this DSRL
        OSAL.eLinkedListRemove(hEntry);
        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Iterate the positions and unlink them from the DSRL
        hEntryToRemove = OSAL.hLinkedListFirst(
            psCurTargetDesc->hPosList,
            (void **)&psCurPosDesc);
        while (hEntryToRemove != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Remove the entry from the list
            OSAL.eLinkedListRemove(hEntryToRemove);

            bPosFound = bFindPosDescInDSRLDesc(psDSRLDesc, psCurPosDesc);

            if (bPosFound == FALSE)
            {
                // Iterate the events and remove them from the DSRL
                OSAL.eLinkedListIterate(
                   psCurPosDesc->hEventList,
                   (OSAL_LL_ITERATOR_HANDLER)bRemoveEventFromTarget,
                   psDSRLDesc);

                // Remove this DSRL from the position code's list of DSRLs
                eReturnCode =
                        OSAL.eLinkedListSearch(psCurPosDesc->hDSRLList,
                                               &hDSRLDescToRemove,
                                               psDSRLDesc);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    OSAL.eLinkedListRemove(hDSRLDescToRemove);
                    hDSRLDescToRemove = OSAL_INVALID_LINKED_LIST_ENTRY;
                }
            }

            hEntryToRemove = OSAL.hLinkedListFirst(
                psCurTargetDesc->hPosList,
                (void**)&psCurPosDesc);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bFindPosDescInDSRLDesc
*
*****************************************************************************/
static BOOLEAN bFindPosDescInDSRLDesc(
    DSRL_DESC_STRUCT *psDSRLDesc,
    POS_EVENT_DESC_STRUCT *psPosDesc
        )
{
    DSRL_REMOVE_TARGET_ITERATOR_STRUCT sIterator;

    sIterator.bFound = FALSE;
    sIterator.psPosDesc = psPosDesc;

    // Iterate targets looking for this position
    OSAL.eLinkedListIterate(
        psDSRLDesc->hTargetList,
        (OSAL_LL_ITERATOR_HANDLER)bFindPosDescInTargetDesc,
        (void *)&sIterator );

    return sIterator.bFound;
}

/*****************************************************************************
*
*   bFindPosDescInDSRLDesc
*
*****************************************************************************/
static BOOLEAN bFindPosDescInTargetDesc(
    TARGET_DESC_STRUCT *psTargetDesc,
    DSRL_REMOVE_TARGET_ITERATOR_STRUCT *psPosDescIterator
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    eReturnCode =
            OSAL.eLinkedListSearch(psTargetDesc->hPosList,
                                   &hEntry,
                                   psPosDescIterator->psPosDesc);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // Stop looking
        psPosDescIterator->bFound = TRUE;
        return FALSE;
    }
    else
    {
        // Keep looking
        return TRUE;
    }
}

/*****************************************************************************
*
*   bLinkNewPosDescToDSRLDesc
*
*   Itterator that goes through a list of DSRL descriptions and creates a
*   two-way link to a posdesc if the new posotion matches the DSRL.
*   This is called as new positions are discovered
*****************************************************************************/

static BOOLEAN bLinkNewPosDescToDSRLDesc (
    DSRL_DESC_STRUCT *psDSRLDesc,
    POS_EVENT_DESC_STRUCT *psPosDesc
        )
{
    TRAFFIC_LINK_POS_DESC_ITERATOR_STRUCT sIterator;

    sIterator.psDSRLDesc = psDSRLDesc;
    sIterator.psPosDesc = psPosDesc;

    // Iterate this DSRL Desc's list of targets and see
    // if this position matches any of them
    OSAL.eLinkedListIterate(psDSRLDesc->hTargetList,
        (OSAL_LL_ITERATOR_HANDLER)bLinkPosDescToTargetDesc,
        (void *)&sIterator);

    return TRUE;
}

/*****************************************************************************
*
*   bLinkPosDescToTargetDesc
*
*****************************************************************************/
static BOOLEAN bLinkPosDescToTargetDesc (
    TARGET_DESC_STRUCT *psTargetDesc,
    TRAFFIC_LINK_POS_DESC_ITERATOR_STRUCT *psIterator
        )
{
    if (psIterator != NULL)
    {
        TRAFFIC_LINK_TARGET_DESC_ITERATOR_STRUCT sIterator;

        sIterator.psDSRLDesc = psIterator->psDSRLDesc;
        sIterator.psTargetDesc = psTargetDesc;

        // We are just making a two-way link, so flip the args and re-use
        // the other iterator
        return bLinkTargetDescToPosDesc(psIterator->psPosDesc, &sIterator);
    }

    return FALSE;
}

/*****************************************************************************
*
*   bLinkTargetDescToPosDesc
*
*   Iterator that goes through a list of psPosDesc and
*   adds targets to the PosDesc if need-be
*   This is called as new DSRLs are created
*****************************************************************************/
static BOOLEAN bLinkTargetDescToPosDesc (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    TRAFFIC_LINK_TARGET_DESC_ITERATOR_STRUCT *psIterator
        )
{
    LOCATION_OBJECT hLocTarget, hLocPosition;
    BOOLEAN bMatch = FALSE;

    if (psPosDesc != NULL && psIterator != NULL)
    {
        // Get the location values
        hLocPosition = psPosDesc->hLocation;
        hLocTarget = psIterator->psTargetDesc->hLocation;

        bMatch = bShouldLinkLocations(hLocTarget, hLocPosition);

        if (bMatch == TRUE)
        {
            // Add DSRL description to the position
            OSAL.eLinkedListAdd(psPosDesc->hDSRLList,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                psIterator->psDSRLDesc
                               );

            // Add position to the target
            OSAL.eLinkedListAdd(psIterator->psTargetDesc->hPosList,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                psPosDesc
                               );

            OSAL.eLinkedListIterate(
                psPosDesc->hEventList,
               (OSAL_LL_ITERATOR_HANDLER)bAddEventToTarget,
               psIterator->psDSRLDesc);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateTargetsForRefresh
*
*****************************************************************************/
static BOOLEAN bIterateTargetsForRefresh (
    TARGET_DESC_STRUCT *psTargetDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    OSAL.eLinkedListIterate(
        psTargetDesc->hPosList,
       (OSAL_LL_ITERATOR_HANDLER)bIteratePosDescsForRefresh,
       psDSRLDesc);

    return TRUE;
}


/*****************************************************************************
*
*   bIteratePosDescsForRefresh
*
*****************************************************************************/
static BOOLEAN bIteratePosDescsForRefresh (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    OSAL.eLinkedListIterate(
        psPosDesc->hEventList,
       (OSAL_LL_ITERATOR_HANDLER)bAddEventToTarget,
       psDSRLDesc);

    return TRUE;
}

/*****************************************************************************
*
*   bAddEventToTarget
*
*****************************************************************************/
static BOOLEAN bAddEventToTarget (
    EVENT_ENTRY_STRUCT *psCurEventDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    if (psCurEventDesc->hStableMsg != TRAFFIC_MSG_INVALID_OBJECT)
    {
        DSRL_eAddEntry(psDSRLDesc->hDSRL, (DSRL_ENTRY_OBJECT)psCurEventDesc->hStableMsg);
    }

    // Keep on truckin'
    return TRUE;
}

/*****************************************************************************
*
*   bRemoveEventFromTarget
*
*****************************************************************************/
static BOOLEAN bRemoveEventFromTarget (
    EVENT_ENTRY_STRUCT *psCurEventDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    if (psCurEventDesc->hStableMsg != TRAFFIC_MSG_INVALID_OBJECT)
    {
        DSRL_vRemoveEntry(psDSRLDesc->hDSRL, (DSRL_ENTRY_OBJECT)psCurEventDesc->hStableMsg);
    }

    // Keep on truckin'
    return TRUE;
}

/*****************************************************************************
*
*   bIterateTargetDescToRemoveDSRLDesc
*
*****************************************************************************/
static BOOLEAN bIterateTargetDescToRemoveDSRLDesc (
    TARGET_DESC_STRUCT *psTargetDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    // Loop through the posisions in the target and remove this DSRL from it
    OSAL.eLinkedListIterate(
        psTargetDesc->hPosList,
       (OSAL_LL_ITERATOR_HANDLER)bRemoveDSRLDescFromPos,
       psDSRLDesc);

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveDSRLDescFromPos
*
*****************************************************************************/
static BOOLEAN bRemoveDSRLDescFromPos (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (psPosDesc != NULL && psDSRLDesc != NULL)
    {
        // Iterate the events and remove them from the DSRL
        OSAL.eLinkedListIterate(
            psPosDesc->hEventList,
           (OSAL_LL_ITERATOR_HANDLER)bRemoveEventFromTarget,
           psDSRLDesc);

        // Remove this DSRL from the position code's list of DSRLs
        eReturnCode =
                OSAL.eLinkedListSearch(psPosDesc->hDSRLList,
                                       &hEntry,
                                       psDSRLDesc);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListRemove(hEntry);
        }
    }

    return TRUE;

}

/*****************************************************************************
*
*   bShouldLinkLocations
*
*****************************************************************************/
static BOOLEAN bShouldLinkLocations (
    LOCATION_OBJECT hLocationDSRL,
    LOCATION_OBJECT hLocationPosDesc
        )
{
    BOOLEAN bShouldLink = FALSE;
    BOOLEAN bIsPoint = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    eReturn = LOCATION.eIsPoint(hLocationDSRL, &bIsPoint);

    if ((eReturn == SMSAPI_RETURN_CODE_SUCCESS) &&
        (bIsPoint == FALSE))
    {
        OSAL_FIXED_OBJECT hLat = LOCATION.hLat(hLocationPosDesc);
        OSAL_FIXED_OBJECT hLon = LOCATION.hLon(hLocationPosDesc);

        bShouldLink =
            LOCATION.bCoordinatesWithinArea(hLocationDSRL, hLat, hLon);
    }
    else
    {
        N16 n16Compare;
        LOCID_OBJECT hLocIDDSRL = LOCATION.hLocID(hLocationDSRL);
        LOCID_OBJECT hLocIDPosDesc = LOCATION.hLocID(hLocationPosDesc);

        if (LOCID.eType(hLocIDDSRL) == LOCID_TYPE_TRAFFIC &&
            LOCID.eType(hLocIDPosDesc) == LOCID_TYPE_TRAFFIC)
        {
            n16Compare =
                TRAFFIC_LOCID.n16CompareMarkets(hLocIDDSRL, hLocIDPosDesc);

            if (n16Compare == 0)
            {
                bShouldLink = TRUE;
            }
        }
    }

    return bShouldLink;
}

/*****************************************************************************
*
*   bMarkMarketAsComplete
*
*****************************************************************************/
static BOOLEAN bMarkMarketAsComplete (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket
        )
{
    BOOLEAN bOk = FALSE;
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;

    if (psObj->psAppObj->ahTrafficTable[tMarket-1] == NULL)
    {
        // Nothing to do
        return TRUE;
    }

    // Get a pointer to the locked app-facing object
    psAppObj = psGetAppFacingObject((TRAFFIC_SERVICE_OBJECT)psObj);

    if (psAppObj != NULL)
    {
        // Build our query string to clear this market from the database
        snprintf( &psAppObj->acBuffer[0], sizeof(psAppObj->acBuffer),
            TRAFFIC_DELETE_MARKET, tMarket );

        SQL_INTERFACE.bStartTransaction(psAppObj->hSQLPersistantConnection);

        // Clear the old traffic table
        bOk = SQL_INTERFACE.bExecuteCommand(
            psAppObj->hSQLPersistantConnection,
            &psAppObj->acBuffer[0]);

        if (bOk == TRUE)
        {
            DATABASE_STORAGE_ITERATOR_STRUCT sIteratorArg;

            // Setup our iterator argument
            sIteratorArg.psMgr = psObj;
            sIteratorArg.tMarket = tMarket;
            OSAL.eTimeGet(&sIteratorArg.un32UTCsec);

            OSAL.eLinkedListIterate(
                psObj->psAppObj->ahTrafficTable[tMarket-1],
                (OSAL_LL_ITERATOR_HANDLER)bMarkPositionAsComplete,
                &sIteratorArg);

            bOk = TRUE;
        }

        SQL_INTERFACE.bEndTransaction(psAppObj->hSQLPersistantConnection);

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return bOk;
}

/*****************************************************************************
*
*   bMarkPositionAsComplete
*
*****************************************************************************/
static BOOLEAN bMarkPositionAsComplete (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    DATABASE_STORAGE_ITERATOR_STRUCT *psIteratorArg
        )
{
    if (psPosDesc != NULL)
    {
        // Update our iterator argument with the current position
        psIteratorArg->psPosDesc = psPosDesc;

        OSAL.eLinkedListIterate(
            psPosDesc->hEventList,
           (OSAL_LL_ITERATOR_HANDLER)bMarkEventAsComplete,
           psIteratorArg);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bMarkEventAsComplete
*
*   Assumes that psObj->psAppObj is LOCKED
*****************************************************************************/
static BOOLEAN bMarkEventAsComplete (
    EVENT_ENTRY_STRUCT *psCurEventEntry,
    DATABASE_STORAGE_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bWrittenToDB = FALSE, bOk;
    TRAFFIC_LOCID_OBJECT hLocID;
    TRAFFIC_POS_CODE tUnstablePosCode, tCurPosCode;

    do
    {
        if (psCurEventEntry == NULL)
        {
            break;
        }

        if (psCurEventEntry->hUnstableMsg == TRAFFIC_MSG_INVALID_OBJECT &&
            psCurEventEntry->hStableMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            // No new message, no previous message either, so go do something else
            break;
        }

        if (psCurEventEntry->hUnstableMsg == TRAFFIC_MSG_INVALID_OBJECT &&
            psCurEventEntry->hStableMsg != TRAFFIC_MSG_INVALID_OBJECT)
        {
            // Modify the stable message pointer
            psCurEventEntry->hStableMsg = psCurEventEntry->hUnstableMsg;

            // We are all done
            // so exit and keep on moving through the events
            break;
        }

        hLocID = LOCATION.hLocID(psIteratorArg->psPosDesc->hLocation);

        if (hLocID == TRAFFIC_LOCID_INVALID_OBJECT)
        {
            break;
        }

        tCurPosCode = TRAFFIC_LOCID.tPosCode(hLocID);
        tUnstablePosCode = TRAFFIC_MSG_tInitialPosCode(psCurEventEntry->hUnstableMsg);

        // See if we need to write this message to the DSRL/DB
        bWrittenToDB = TRAFFIC_MSG_bWrittenToDB(psCurEventEntry->hUnstableMsg);

        if ( (bWrittenToDB == TRUE) ||
             (tCurPosCode != tUnstablePosCode)
           )
        {
            // Set the unstable msg to the stable msg now that
            // we know the market is complete.  We already added
            // this message to the DSRL/DB
            psCurEventEntry->hStableMsg = psCurEventEntry->hUnstableMsg;

            // We are all done
            // so exit and keep on moving through the events
            break;
        }

        printf(TRAFFIC_MGR_OBJECT_NAME": updating event(%u) in DSRL/DB at pos %u -- \n",
            tUnstablePosCode, tCurPosCode);

        if (psCurEventEntry->hStableMsg == TRAFFIC_MSG_INVALID_OBJECT &&
            psCurEventEntry->hUnstableMsg != TRAFFIC_MSG_INVALID_OBJECT)
        {
            puts(TRAFFIC_MGR_OBJECT_NAME": NEW");

            // This is a new message in the unstable pointer,
            // so add it to the DSRL
            vUpdateTargetList (
                psIteratorArg->psPosDesc->hDSRLList,
                psCurEventEntry->hUnstableMsg,
                TRAFFIC_MSG_INVALID_OBJECT,
                DSRL_ENTRY_STATUS_NEW);
        }
        else
        {
            // We are doing an update, so replace our previous entry
            // but only if it is a new handle
            if (psCurEventEntry->hStableMsg != psCurEventEntry->hUnstableMsg)
            {
                puts(TRAFFIC_MGR_OBJECT_NAME": CHANGED");

                vUpdateTargetList (
                    psIteratorArg->psPosDesc->hDSRLList,
                    psCurEventEntry->hUnstableMsg,  // New Message
                    psCurEventEntry->hStableMsg,    // Old Message
                    DSRL_ENTRY_STATUS_CHANGED);
            }
        }

        // Set the unstable msg to the stable msg now that
        // we know the market is complete
        psCurEventEntry->hStableMsg = psCurEventEntry->hUnstableMsg;

        // Insert info into DB
        bOk = TRAFFIC_MSG_bInsertIntoDB(
            psCurEventEntry->hStableMsg, psIteratorArg->un32UTCsec,
            psIteratorArg->psMgr->psAppObj->hSQLPersistantConnection,
            &psIteratorArg->psMgr->psAppObj->hInsertMsgIntoDBStmt,
            psIteratorArg->psMgr->psAppObj->asBindParams);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": ERROR -- Couldn't build DB String to save TrafficMsg\n");
            break;
        }

        TRAFFIC_MSG_vSetWrittenToDBFlag(psCurEventEntry->hStableMsg, TRUE);

    } while (FALSE);

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveExpiredMessagesFromDatabase
*
*****************************************************************************/
static BOOLEAN bRemoveExpiredMessagesFromDatabase(
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk = FALSE;
    UN32 un32ExpirationTime;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Request current time
        eReturnCode = OSAL.eTimeGet(&un32ExpirationTime);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME": failed to get current time (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Adjust expiration time since the messages which ain't meet condition
        //           current_time >= msg_time + MSG_TIMEOUT i.e.
        //           current_time - MSG_TIMEOUT >= msg_time
        // shall be removed as expired ones
        un32ExpirationTime -= MSG_TIMEOUT;

        // Prepare query
        snprintf(psAppObj->acBuffer, sizeof(psAppObj->acBuffer),
            TRAFFIC_REMOVE_EXPIRED_MESSAGES, un32ExpirationTime);

        // Do SQL query to remove all expire messages
        bOk = SQL_INTERFACE.bExecuteCommand(psAppObj->hSQLPersistantConnection,
                                            psAppObj->acBuffer);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME": failed to exec query: '%s'",
                psAppObj->acBuffer);
        }
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   vRestoreAllMarketsFromDatabase
*
*   Assumes a valid and locked psAppObj
*****************************************************************************/
static void vRestoreAllMarketsFromDatabase (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj
        )
{
    TRAFFIC_MARKET tMarket;

    for (tMarket = 1; tMarket <= NUM_TRAFFIC_MARKETS; tMarket++)
    {
        // Attempt to restore a market from the database
        vRestoreMarketFromDatabase(psObj, psAppObj, tMarket);
    }

    return;
}

/*****************************************************************************
*
*   vRestoreMarketFromDatabase
*
*****************************************************************************/
static void vRestoreMarketFromDatabase (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj,
    TRAFFIC_MARKET tMarket
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32UTCsec;
    TRAFFIC_STORAGE_RESTORATION_ARG_STRUCT sRestorationArg;

    do
    {
        eReturnCode = OSAL.eTimeGet(&un32UTCsec);

        if (eReturnCode != OSAL_SUCCESS)
        {
            printf("Result: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Determine how old our data should be, which would be
        // 15 minutes earlier than now
        if (un32UTCsec < RESTORED_MSGS_TIMEOUT_SECS)
        {
            // Just to be safe, though I don't think this will hapen
            break;
        }

        // We don't want any messages whose timestamp is less than this
        un32UTCsec = un32UTCsec - RESTORED_MSGS_TIMEOUT_SECS;

        // Prepare our restoration argument for processing the records
        sRestorationArg.psMgr = psObj;
        sRestorationArg.tMarket = tMarket;
        sRestorationArg.un32UTCsec = un32UTCsec;
        sRestorationArg.bResultantRows = FALSE;

        // Build our query string to fetch the records for
        // this market that have a valid timestamp
        snprintf( &psAppObj->acBuffer[0], sizeof(psAppObj->acBuffer),
            TRAFFIC_SELECT_STORED_MESSAGES, tMarket);

        SQL_INTERFACE.bStartTransaction(psAppObj->hSQLPersistantConnection);

        // Process the records and add the records to the incident message list
        bOk = SQL_INTERFACE.bQuery(
            psAppObj->hSQLPersistantConnection, &psAppObj->acBuffer[0],
                bProcessStoredMessages, &sRestorationArg) ;

        if (bOk == FALSE)
        {
            // Something really bad must have happened
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME": Error Processing "
                "stored messages from the database");
            break;
        }

        SQL_INTERFACE.bEndTransaction(psAppObj->hSQLPersistantConnection);

        // If we processed any messages, we need to get a work request
        if (sRestorationArg.bResultantRows == TRUE)
        {
            // Mark table for market as complete
            bMarkMarketAsComplete(psObj, tMarket);
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vStartCancelTimer
*
*****************************************************************************/
static void vStartCancelTimer (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    // Kick off the event, override any event already set
    bOk = DATASERVICE_IMPL_bSetTimedEvent(
        psObj->hCancelEvent, TRUE, FALSE, CANCEL_MSG_TIMER_PERIOD);

    if (bOk == TRUE)
    {
        puts(TRAFFIC_MGR_OBJECT_NAME": starting timer");
    }

    return;
}

/*****************************************************************************
*
*   vStopCancelTimer
*
*****************************************************************************/
static void vStopCancelTimer (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    puts(TRAFFIC_MGR_OBJECT_NAME": stopping cancel timer");

    // Stop the cancel event from occurring
    DATASERVICE_IMPL_bStopTimedEvent(psObj->hCancelEvent);

    return;
}

/*****************************************************************************
*
*   vCheckForMessagesToCancel
*
*****************************************************************************/
static void vCheckForMessagesToCancel (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        )
{
    EXPIRED_MESSAGE_ITERATOR_STRUCT sIteratorArg;
    UN8 un8Index;

    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;

    psAppObj = psGetAppFacingObject((TRAFFIC_SERVICE_OBJECT)psObj);

    if (psAppObj != NULL)
    {

        sIteratorArg.psMgr = psObj;
        sIteratorArg.bMsgsStillActive = FALSE;

        OSAL.eTimeGet(&sIteratorArg.un32UTCsec);

        // If we are in some sort of "no data" situation.
        // Set the updating flag
        if (sIteratorArg.un32UTCsec >=
            (psObj->un32LastTimeDataRXd + MSG_TIMEOUT))
        {
            OSAL.eLinkedListIterate(
                psObj->hDSRLList,
                (OSAL_LL_ITERATOR_HANDLER)bNotifyTargets,
                (void *)(size_t)DSRL_STATE_UPDATING);
        }

        for (un8Index = 0;
             un8Index < NUM_TRAFFIC_MARKETS;
             un8Index++)
        {
            OSAL.eLinkedListIterate(psObj->psAppObj->ahTrafficTable[un8Index],
                                   (OSAL_LL_ITERATOR_HANDLER)bCheckTimeForMarket,
                                   &sIteratorArg);
        }

        // Tell the DSRLs to fire their callbacks if
        // any cancellations happened.  Note, we will only
        // do this if we are living in some sort of
        // "no data" situation.
        if (sIteratorArg.un32UTCsec >=
            (psObj->un32LastTimeDataRXd + MSG_TIMEOUT))
        {
            OSAL.eLinkedListIterate(
                psObj->hDSRLList,
                (OSAL_LL_ITERATOR_HANDLER)bNotifyTargets,
                (void *)(size_t)DSRL_STATE_READY);
        }

        if (sIteratorArg.bMsgsStillActive == TRUE)
        {
            vStartCancelTimer(psObj);
        }

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }
    else
    {
        // Couldn't get the app object for some reason, start
        // the timer to try again
        vStartCancelTimer(psObj);
    }

    return;
}

/*****************************************************************************
*
*   bCheckTimeForMarket
*
*****************************************************************************/
static BOOLEAN bCheckTimeForMarket (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EXPIRED_MESSAGE_ITERATOR_STRUCT *psIteratorArg
        )
{
    psIteratorArg->psPosDesc = psPosDesc;

    OSAL.eLinkedListIterate(
        psPosDesc->hEventList,
       (OSAL_LL_ITERATOR_HANDLER)bCheckTimeForEvent,
       psIteratorArg);

    return TRUE;
}

/*****************************************************************************
*
*   bCheckTimeForEvent
*
*****************************************************************************/
static BOOLEAN bCheckTimeForEvent (
    EVENT_ENTRY_STRUCT *psCurEventEntry,
    EXPIRED_MESSAGE_ITERATOR_STRUCT *psIteratorArg
        )
{
    BOOLEAN bContinue = FALSE;

    do 
    {
        TRAFFIC_POS_CODE tMsgInitalPosCode, tPosCode;
        BOOLEAN bExpired, bMsgDestroyed = FALSE;
        TRAFFIC_LOCID_OBJECT hLocID;
        UN8 un8NumLocsReq, un8Index;
        POS_EVENT_DESC_STRUCT *psPosDesc;
        EVENT_ENTRY_STRUCT *psEventDesc = psCurEventEntry;

        if ((psCurEventEntry == NULL ) || (psIteratorArg == NULL))
        {
            break;
        }

        if (psCurEventEntry->hStableMsg == TRAFFIC_MSG_INVALID_OBJECT)
        {
            // no stable message, nothing to expire, but valid case
            bContinue = TRUE;
            break;
        }

        // we check expiration if message initial poscode 
        // (or, in other words, starting position) is equal to 
        // currently iterated position code 

        psPosDesc = psIteratorArg->psPosDesc;

        // Grab the initial location for this message
        tMsgInitalPosCode = TRAFFIC_MSG_tInitialPosCode(
            psCurEventEntry->hStableMsg);

        // Grab the LOCID object for currently iterated position
        hLocID = LOCATION.hLocID(psPosDesc->hLocation);
        // Get the poscode for current position
        tPosCode = TRAFFIC_LOCID.tPosCode(hLocID);
        if (tPosCode != tMsgInitalPosCode)
        {
            bContinue = TRUE;
            break;
        }

        // checking expired status
        bExpired = bCheckTimestampAndDuration(psCurEventEntry->hStableMsg,
            psIteratorArg->un32UTCsec);
        if (bExpired == FALSE)
        {
            //not expired
            bContinue = TRUE;
            break;
        }

        printf(TRAFFIC_MGR_OBJECT_NAME": removing stable message "
            "%p due to timeout\n", psCurEventEntry->hStableMsg);

        // Find the number of extents for the original traffic message
        un8NumLocsReq = TRAFFIC_MSG_un8InitialNumLocs(
            psCurEventEntry->hStableMsg) + 1;

        // Clear the handles to the head entry in all
        // the psCurEventEntry for this chain
        for (un8Index = 0; ((un8Index < un8NumLocsReq) && 
            (psPosDesc != NULL)); ++un8Index)
        {
            printf(TRAFFIC_MGR_OBJECT_NAME": Removing %p from DSRLs\n",
                psEventDesc->hStableMsg);

            vUpdateTargetList(psPosDesc->hDSRLList,
                psEventDesc->hStableMsg,
                TRAFFIC_MSG_INVALID_OBJECT,
                DSRL_ENTRY_STATUS_REMOVED);

            // Remove the location
            TRAFFIC_MSG_bRemoveLocation(psEventDesc->hStableMsg,
                psPosDesc->hLocation, &bMsgDestroyed);
            psEventDesc->hStableMsg = TRAFFIC_MSG_INVALID_OBJECT;

            if (bMsgDestroyed != TRUE)
            {
                // Move to the next position for this event
                if (psEventDesc->eDirection == TRAFFIC_DIRECTION_POSITIVE)
                {
                    psPosDesc = psPosDesc->psPositive;
                }
                else
                {
                    psPosDesc = psPosDesc->psNegative;
                }

                if (psPosDesc == NULL)
                {
                    printf(TRAFFIC_MGR_OBJECT_NAME": Failed to get next "
                        "position for event\n");
                    break;
                }

                // Grab the event from the next position
                psEventDesc = psGetEventForPos(psIteratorArg->psMgr, 
                    psPosDesc, psEventDesc->eDirection,
                    psEventDesc->eClass);
                if ((psEventDesc == NULL) ||
                    (psEventDesc->hStableMsg == 
                    TRAFFIC_MSG_INVALID_OBJECT))
                {
                    printf(TRAFFIC_MGR_OBJECT_NAME": Failed to get the "
                        "event for position id %u\n",
                        TRAFFIC_LOCID.tPosCode(LOCATION.hLocID(
                        psPosDesc->hLocation)));
                    break;
                }
            }
        }

        bContinue = TRUE;
    } while (FALSE);
    
    if ((psCurEventEntry != NULL) && (psIteratorArg != NULL))
    {
        if (psCurEventEntry->hStableMsg != TRAFFIC_MSG_INVALID_OBJECT
            && psIteratorArg->bMsgsStillActive == FALSE)
        {
            psIteratorArg->bMsgsStillActive = TRUE;
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   un32GetNextDayStartTimestamp
*
*****************************************************************************/
static UN32 un32GetNextDayStartTimestamp(
    void
        )
{
    UN32 un32Result = 0;
    UN32 un32CurrentTime;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Get current time
    eReturnCode = OSAL.eTimeGet(&un32CurrentTime);
    if (eReturnCode == OSAL_SUCCESS)
    {
        // To make sure this new time will relate to the next day.
        un32Result = (un32CurrentTime + TRAFFIC_DURATION_24_HOURS_AS_SECONDS);

        // Exclude all extra hours to align timestamp by day frame
        un32Result -= (un32Result % TRAFFIC_DURATION_24_HOURS_AS_SECONDS);
    }
    else
    {
        printf(TRAFFIC_MGR_OBJECT_NAME": failed to get current time (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return un32Result;
}

/*****************************************************************************
*
*   bCheckTimestampAndDuration
*
*****************************************************************************/
static BOOLEAN bCheckTimestampAndDuration (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    UN32 un32CurrentTime
        )
{
    UN32 un32CurMsgTimeStamp;
    TRAFFIC_DURATION_ENUM eDuration;
    BOOLEAN bExprired;

    // Get the timestamp for the traffic message
    un32CurMsgTimeStamp = TRAFFIC_MSG_un32TimeStamp(hTrafficMsg);

    // get the duration
    eDuration = TRAFFIC_MSG.eDuration(hTrafficMsg);

    // Adjust the timestamp based on the duration
    switch (eDuration)
    {
        case TRAFFIC_DURATION_15_MINUTES:
        {
            un32CurMsgTimeStamp += TRAFFIC_DURATION_15_MINUTES_AS_SECONDS;
        }
        break;

        case TRAFFIC_DURATION_30_MINUTES:
        {
            un32CurMsgTimeStamp += TRAFFIC_DURATION_30_MINUTES_AS_SECONDS;
        }
        break;

        case TRAFFIC_DURATION_1_HOUR:
        {
            un32CurMsgTimeStamp += TRAFFIC_DURATION_1_HOUR_AS_SECONDS;
        }
        break;

        case TRAFFIC_DURATION_2_HOURS:
        {
            un32CurMsgTimeStamp += TRAFFIC_DURATION_2_HOURS_AS_SECONDS;
        }
        break;

        case TRAFFIC_DURATION_3_HOURS:
        {
            un32CurMsgTimeStamp += TRAFFIC_DURATION_3_HOURS_AS_SECONDS;
        }
        break;

        case TRAFFIC_DURATION_4_HOURS:
        {
            un32CurMsgTimeStamp += TRAFFIC_DURATION_4_HOURS_AS_SECONDS;
        }
        break;

        case TRAFFIC_DURATION_REST_OF_DAY:
        {
            UN32 un32NextDayTimestamp;

            // Get the timestamp related to the start of the next day
            un32NextDayTimestamp = un32GetNextDayStartTimestamp();
            if (un32NextDayTimestamp > 0)
            {
                un32CurMsgTimeStamp = un32NextDayTimestamp;
            }
            else
            {
                // Let's use 12 hours duration
                un32CurMsgTimeStamp += TRAFFIC_DURATION_REST_OF_DAY_AS_SECONDS;
            }
        }
        break;

        case TRAFFIC_DURATION_UNKNOWN:
        {
            // Let's use default timeout
            un32CurMsgTimeStamp += MSG_TIMEOUT;
        }
        break;

        case TRAFFIC_DURATION_COUNT:
        {
            // Nothing! If someone has it - burn it in hell of expired messages
        }
        break;
    }

    // Check expiration timestamp
    bExprired = (un32CurrentTime < un32CurMsgTimeStamp) ? FALSE : TRUE;

    return bExprired;
}

/*****************************************************************************
*
*   bCreatePersistDBTables
*
*****************************************************************************/
static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    BOOLEAN bSuccess = FALSE, bTransactionStarted = FALSE;

    do
    {
        bTransactionStarted = SQL_INTERFACE.bStartTransaction(hSQLConnection);

        if (bTransactionStarted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": failed to start transaction to recreate persistant storage");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            TRAFFIC_CREATE_STORAGE_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": failed to create traffic_stroage table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            TRAFFIC_CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": failed to create db_version in persistant storage");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            TRAFFIC_INSERT_VERSION_ROW);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": failed to set db version in persistant storage");
            break;
        }

    }
    while (FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        bTransactionEnded = SQL_INTERFACE.bEndTransaction(hSQLConnection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": failed to end transaction");

            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bVerifyDBVersion
*
*****************************************************************************/
static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    TRAFFIC_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
            hSQLConnection, TRAFFIC_SELECT_DB_VERSION,
            bProcessSelectDBVersionResult, &sQueryResult ) ;

    if (bOk == TRUE)
    {
        if (sQueryResult.bResultantRows == TRUE)
        {
            TRAFFIC_VERSION_ROW_STRUCT *psVersionRow =
                &sQueryResult.uDbRow.sVersion;
            UN8 un8ActualDBVersion;

            // Grab the actual database version
            un8ActualDBVersion = (UN8)atoi(TRAFFIC_DATABASE_FILE_VERSION);

            // Verify that the db schema version matched
            if (psVersionRow->un8DBVer == un8ActualDBVersion)
            {
                // Version match!
                return TRUE;
            }
            else
            {
                // Version mismatch!
                return FALSE;
            }
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectDBVersionResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.  It populates a
*   pointer to a TRAFFIC_VERSION_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    DB_VERSION_FIELDS_ENUM eCurrentField =
        (DB_VERSION_FIELDS_ENUM)0;
    TRAFFIC_DB_QUERY_RESULT_STRUCT *psResult =
        (TRAFFIC_DB_QUERY_RESULT_STRUCT *)pvArg;
    TRAFFIC_VERSION_ROW_STRUCT *psVersionRow;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == DB_VERSION_MAX_FIELDS )
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psVersionRow = &psResult->uDbRow.sVersion;

    // Clear the version structure
    OSAL.bMemSet(psVersionRow, 0, sizeof(TRAFFIC_VERSION_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the version of the database.
    // Process the results (should be only one row)
    do
    {

        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case DB_VERSION_FIELD_DB_VER:
            {
                psVersionRow->un8DBVer =
                    (UN8)atoi((char *)psCurrentCol->uData.sCString.pcData);
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < DB_VERSION_MAX_FIELDS);

    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectEventResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on Events table.  It populates a
*   pointer to a TRAFFIC_EVENTS_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectEventResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    EVENTS_FIELDS_ENUM eCurrentField =
        (EVENTS_FIELDS_ENUM)0;
    TRAFFIC_DB_QUERY_RESULT_STRUCT *psResult =
        (TRAFFIC_DB_QUERY_RESULT_STRUCT *)pvArg;
    TRAFFIC_EVENTS_ROW_STRUCT *psEventsRow;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there is are the correct
    // number of columns, then we have good results
    if ( n32NumberOfColumns == EVENTS_MAX_FIELDS )
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;
        return FALSE;
    }

    // Get a more convenient pointer
    psEventsRow = &psResult->uDbRow.sEvents;

    // Clear the event structure
    OSAL.bMemSet(psEventsRow, 0, sizeof(TRAFFIC_EVENTS_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the details for an Alert-C event code.
    // Process the results (should be only one.)
    do
    {
        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case EVENTS_FIELD_CODE:
            {
                psEventsRow->tCode =
                    (TRAFFIC_EVENT_CODE)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case EVENTS_FIELD_CLASS:
            {
                psEventsRow->eClass =
                    (TRAFFIC_MSG_CLASS_ENUM)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case EVENTS_FIELD_QUANT_TYPE:
            {
                psEventsRow->eQuantType =
                    (ALERTC_QUANTIFIER_TYPE_ENUM)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case EVENTS_FIELD_TEXT_US:
            {
                psEventsRow->hText =
                    STRING.hCreate((char *)psCurrentCol->uData.sCString.pcData,
                        strlen((char *)psCurrentCol->uData.sCString.pcData));
            }
            break;

            case EVENTS_FIELD_TEXT_MOD_US:
            {
                psEventsRow->hTextMod =
                    STRING.hCreate((char *)psCurrentCol->uData.sCString.pcData,
                        strlen((char *)psCurrentCol->uData.sCString.pcData));
            }
            break;

            case EVENTS_FIELD_TYPE:
            {
                if (strcmp((char *)psCurrentCol->uData.sCString.pcData,
                        EVENTS_TABLE_ALERTC_TYPE_SPEED_AND_FLOW) == 0)
                {
                    psEventsRow->eType =
                        ALERTC_EVENT_TYPE_SPEED_AND_FLOW;
                }
                else if (strcmp((char *)psCurrentCol->uData.sCString.pcData,
                    EVENTS_TABLE_ALERTC_TYPE_CANCLLED) == 0)
                {
                    psEventsRow->eType =
                        ALERTC_EVENT_TYPE_CANCELLED;
                }
                else if (strcmp((char *)psCurrentCol->uData.sCString.pcData,
                    EVENTS_TABLE_ALERTC_TYPE_INCIDENT) == 0)
                {
                    psEventsRow->eType =
                        ALERTC_EVENT_TYPE_INCIDENT;
                }
                else
                {
                    // Shouldn't get here -- DB error?
                    psEventsRow->eType =
                        ALERTC_EVENT_TYPE_UNKNOWN;
                }
            }
            break;


            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < EVENTS_MAX_FIELDS);

    // We only want one row
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectSupplInfoResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on Sup_Info table.  It populates a
*   pointer to a TRAFFIC_SUPPL_INFO_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectSupplInfoResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    SUPPL_INFO_FIELDS_ENUM eCurrentField =
        (SUPPL_INFO_FIELDS_ENUM)0;
    TRAFFIC_DB_QUERY_RESULT_STRUCT *psResult =
        (TRAFFIC_DB_QUERY_RESULT_STRUCT *)pvArg;
    TRAFFIC_SUPPL_INFO_ROW_STRUCT *psSupplInfoRow;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if ( n32NumberOfColumns == SUPPL_INFO_MAX_FIELDS )
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psSupplInfoRow = &psResult->uDbRow.sSupplInfo;

    // Clear the event structure
    OSAL.bMemSet(psSupplInfoRow, 0, sizeof(TRAFFIC_SUPPL_INFO_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the details for an Alert-C event code.
    // Process the results (should be only one.)
    do
    {
        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on its type
        switch (eCurrentField)
        {
            case SUPPL_INFO_FIELD_CODE:
            {
                psSupplInfoRow->tCode =
                    (TRAFFIC_EVENT_CODE)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case SUPPL_INFO_FIELD_TEXT_US:
            {
                strncpy(psSupplInfoRow->acText,
                     (char *)psCurrentCol->uData.sCString.pcData,
                     sizeof(psSupplInfoRow->acText) - 1);
            }
            break;


            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < SUPPL_INFO_MAX_FIELDS);

    return FALSE;
}

/*******************************************************************************
*
*   bProcessStoredMessages
*
*******************************************************************************/
static BOOLEAN bProcessStoredMessages (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    TRAFFIC_MSG_OBJECT hCurTrafficMsg;
    BOOLEAN bOk;
    TRAFFIC_STORAGE_ROW_STRUCT sStorageRow;
    TRAFFIC_STORAGE_FIELDS_ENUM eCurrentField = (TRAFFIC_STORAGE_FIELDS_ENUM)0;
    TRAFFIC_STORAGE_RESTORATION_ARG_STRUCT *psRestorationArg =
        (TRAFFIC_STORAGE_RESTORATION_ARG_STRUCT *)pvArg;

    // Verify input
    if (psRestorationArg == NULL)
    {
        return FALSE;
    }

    // If there is the correct number of columns,
    // then we have good results
    if (n32NumberOfColumns == TRAFFIC_STORAGE_MAX_FIELDS )
    {
        // Indicate that we had at least one row
        psRestorationArg->bResultantRows = TRUE;
    }
    else
    {
        return FALSE;
    }

    // We have at least one row, so make sure we have a table entry
    // for this market
    bOk = bEnsureTrafficTableExists(
        psRestorationArg->psMgr->psAppObj,
        psRestorationArg->tMarket);
    if (FALSE == bOk)
    {
        // Big error
        return FALSE;
    }

    // Initializing local structure
    OSAL.bMemSet(&sStorageRow, 0, sizeof(TRAFFIC_STORAGE_ROW_STRUCT));

    do
    {
        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case TRAFFIC_STORAGE_TIMESTAMP:
            {
                sStorageRow.un32UTCsec =
                    psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_MARKET:
            {
                sStorageRow.tMarket =
                    (TRAFFIC_MARKET)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_BSA:
            {
                sStorageRow.tBSA =
                    (TRAFFIC_BSA)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_POS_CODE:
            {
                sStorageRow.tPosCode =
                    (TRAFFIC_POS_CODE)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_EXTENT:
            {
                sStorageRow.un8Extent =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_EVENT_CLASS:
            {
                // Do nothing, this was just used as part of
                // the primary key value in the database
            }
            break;
            case TRAFFIC_STORAGE_EVENT_CODE:
            {
                sStorageRow.tEventCode =
                    (TRAFFIC_EVENT_CODE)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_DIRECTION:
            {
                sStorageRow.eDirection =
                    (TRAFFIC_DIRECTION_ENUM)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_DURATION:
            {
                sStorageRow.un8Duration =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_DIVERSION:
            {
                sStorageRow.bDiversionAdvised =
                    (BOOLEAN)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_MULTI_GROUP_COUNT:
            {
                sStorageRow.un8MultiGroupCount =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_FREE_FORM_DATA_COUNT:
            {
                sStorageRow.un8FreeFormDataCount =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;
            case TRAFFIC_STORAGE_FREE_FORM_DATA_BLOB:
            {
                if (psCurrentCol->uData.sBLOB.tSize ==
                    (sStorageRow.un8FreeFormDataCount * sizeof(UN16) * 2))
                {
                    OSAL.bMemCpy(
                        &sStorageRow.aun16FreeFormData[0],
                        psCurrentCol->uData.sBLOB.pvData,
                        psCurrentCol->uData.sBLOB.tSize);
                }
            }
            break;
            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < TRAFFIC_STORAGE_MAX_FIELDS);

    // Modify our timestamp so it represents a second offset instead
    // of a whole unixtime
    sStorageRow.un32UTCsec -= psRestorationArg->un32UTCsec;

    // Create the traffic message and add it to our list
    // to get processed by the traffic manager
    hCurTrafficMsg = TRAFFIC_MSG_hCreateFromDB(
        (SMS_OBJECT)psRestorationArg->psMgr->psAppObj,
        &sStorageRow);

    if (hCurTrafficMsg != TRAFFIC_MSG_INVALID_OBJECT)
    {
        // Process the message into our table
        BOOLEAN bOk = bUpdateTrafficTable(
            psRestorationArg->psMgr, hCurTrafficMsg);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TRAFFIC_MGR_OBJECT_NAME
                ": Couldn't update traffic table with this msg from the DB.");

            TRAFFIC_MSG_vDestroy(hCurTrafficMsg);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   n16SortDSRLByDistance
*
*****************************************************************************/
static N16 n16SortDSRLByDistance (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    void *pvSortArgs
        )
{
    TRAFFIC_MSG_OBJECT hTrafficMsg1 =
        (TRAFFIC_MSG_OBJECT)DSRL_ENTRY.hTrafficMsg(hEntry1);
    LOCATION_OBJECT hInitialLocation1 =
        TRAFFIC_MSG.hGetLocation(hTrafficMsg1, 0);
    TRAFFIC_MSG_OBJECT hTrafficMsg2 =
        (TRAFFIC_MSG_OBJECT)DSRL_ENTRY.hTrafficMsg(hEntry2);
    LOCATION_OBJECT hInitialLocation2 =
        TRAFFIC_MSG.hGetLocation(hTrafficMsg2, 0);
    LOCATION_OBJECT hClosestLoc;
    DSRL_DESC_STRUCT *psDSRLDesc= (DSRL_DESC_STRUCT *)pvSortArgs;
    TARGET_DESC_STRUCT *psTargetDesc;
    N16 n16Return = N16_MIN;

    // Fetch the first Target Description
    OSAL.hLinkedListFirst(psDSRLDesc->hTargetList, (void**)&psTargetDesc);

    if (psTargetDesc != NULL)
    {
        // Determine the closest location
        hClosestLoc = LOCATION.hClosest(psTargetDesc->hLocation,
            hInitialLocation1, hInitialLocation2);

        if (hClosestLoc != LOCATION_INVALID_OBJECT)
        {
            if (hClosestLoc == hInitialLocation1)
            {
                n16Return = -1;
            }
            else
            {
                n16Return = 1;
            }
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}

/*****************************************************************************
*
*   bEnsureTrafficTableExists
*
*****************************************************************************/
static BOOLEAN bEnsureTrafficTableExists (
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj,
    TRAFFIC_MARKET tMarket
        )
{
    char acListName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psAppObj->ahTrafficTable[tMarket-1] != OSAL_INVALID_OBJECT_HDL)
    {
        return TRUE;    /* already created */
    }

    snprintf(&acListName[0], sizeof(acListName),
        "%s:TrafficTable(%d)", TRAFFIC_MGR_OBJECT_NAME, tMarket);

    eReturnCode = OSAL.eLinkedListCreate(
        &psAppObj->ahTrafficTable[tMarket-1],
        &acListName[0], n16ComparePosCode, OSAL_LL_OPTION_RBTREE);

    return (eReturnCode == OSAL_SUCCESS) ? TRUE : FALSE;
}
