/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TRAFFIC_LOCID implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "locid_obj.h"
#include "traffic_locid_obj.h"
#include "_traffic_locid_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreate
*
* This object interface method is used to create a TRAFFIC_LOCID object that
* is used to the Alert-C releated identiciation for a particular position.
*
* The creation of the TRAFFIC_LOCID will fail if a market is not specified.
* The market is the only requried parameter to create a TRAFFIC_LOCID
*
* Inputs:
*
*   tPosCode - The TRAFFIC_POS_CODE that is the aggreed upon position code
*                used to describe the position in Alert-C messages
*     tMarket - The TRAFFIC_MARKET that is represents the market this
*               TRAFFIC_LOCID_OBJECT is covering
*     tBSAQ - The TRAFFIC_BSA that is the Broadcast Service Area that this
*             TRAFFIC_LOCID_OBJECT is covering.  The BSA is a subdivion
*             of a TRAFFIC_MARKET.
*
* Outputs:
*
*   A handle to a TRAFFIC_LOCID_OBJECT
*
*****************************************************************************/
static TRAFFIC_LOCID_OBJECT hCreate (
    TRAFFIC_POS_CODE tPosCode,
    TRAFFIC_MARKET tMarket,
    TRAFFIC_BSA tBSA
        )
{
    TRAFFIC_LOCID_OBJECT hLocID = TRAFFIC_LOCID_INVALID_OBJECT;

    hLocID = TRAFFIC_LOCID.hCreateExt(tPosCode, tMarket, tBSA, 0);

    return (TRAFFIC_LOCID_OBJECT)hLocID;
}


/*****************************************************************************
*
*   hCreateExt
*
*****************************************************************************/
static TRAFFIC_LOCID_OBJECT hCreateExt (
    TRAFFIC_POS_CODE tPosCode,
    TRAFFIC_MARKET tMarket,
    TRAFFIC_BSA tBSA,
    TRAFFIC_OFFSET tOffset
        )
{
    LOCID_OBJECT hLocIDSuper;
    LOCID_UNION *puSubClassData;

    // Lets make sure we aren't trying to make a TRAFFIC_LOCID
    // for an Market.  The market is the minimal we need
    if(tMarket == TRAFFIC_INVALID_MARKET)
    {
        // Error!
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    // Create an instance of the LOCID super-class
    hLocIDSuper = LOCID_hCreate((LOC_ID)tPosCode,
        LOCID_TYPE_TRAFFIC, &gsTrafficLocIDInterface);

    if (hLocIDSuper == LOCID_INVALID_OBJECT)
    {
        // Error!
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    // Get a pointer to the union to fill in our
    // child-class data
    puSubClassData = LOCID_puSubClassData(hLocIDSuper);

    if (puSubClassData == LOCID_INVALID_UNION)
    {
        // Error!
        LOCID.vDestroy(hLocIDSuper);
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    // Fill in subclass data
    puSubClassData->sTrafficLocID.tMarket = tMarket;
    puSubClassData->sTrafficLocID.tBSA = tBSA;
    puSubClassData->sTrafficLocID.tOffset = tOffset;

    return (TRAFFIC_LOCID_OBJECT)hLocIDSuper;
}


/*****************************************************************************
*
*   tPosCode
*
* This object interface method is used to return the TRAFFIC_POS_CODE for a
* given TRAFFIC_LOCID_object.
*
* Inputs:
*
*   hTrafficLocID - The TRAFFIC_LOCID_OBJECT to get this value from
*
* Outputs:
*
*   The TRAFFIC_POS_CODE value on success.
*     TRAFFIC_INVALID_POS_CODE on failure or if there is no position code
*     associated with this object
*
*****************************************************************************/
static TRAFFIC_POS_CODE tPosCode (
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    BOOLEAN bValid;
    TRAFFIC_POS_CODE tPosCode = TRAFFIC_INVALID_POS_CODE;
    LOCID_TYPE_ENUM eType;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);
    if(bValid == FALSE)
    {
        // Error!
        return TRAFFIC_INVALID_POS_CODE;
    }

    eType = LOCID.eType(hTrafficLocID);

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        tPosCode = (TRAFFIC_POS_CODE)LOCID.tID((LOCID_OBJECT)hTrafficLocID);
    }

    return tPosCode;
}

/*****************************************************************************
*
*   tMarket
*
* This object interface method is used to return the TRAFFIC_MARKET for a
* given TRAFFIC_LOCID_object.
*
* Inputs:
*
*   hTrafficLocID - The TRAFFIC_LOCID_OBJECT to get this value from
*
* Outputs:
*
*   The TRAFFIC_MARKET value on success.
*     TRAFFIC_INVALID_MARKET on failure
*
*****************************************************************************/
static TRAFFIC_MARKET tMarket (
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    BOOLEAN bValid;
    TRAFFIC_MARKET tMarket = TRAFFIC_INVALID_MARKET;
    LOCID_TYPE_ENUM eType;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);
    if(bValid == FALSE)
    {
        // Error!
        return TRAFFIC_INVALID_MARKET;
    }

    eType = LOCID.eType((LOCID_OBJECT)hTrafficLocID);

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        LOCID_UNION *puSubClassData;
        // Get a pointer to the union to fill in our
        // child-class data
        puSubClassData = LOCID_puSubClassData(hTrafficLocID);

        if (puSubClassData == LOCID_INVALID_UNION)
        {
            // Error!
            return TRAFFIC_INVALID_MARKET;
        }

        tMarket = puSubClassData->sTrafficLocID.tMarket;
    }

    return tMarket;
}

/*****************************************************************************
*
*   tBSA
*
* This object interface method is used to return the TRAFFIC_BSA for a
* given TRAFFIC_LOCID_object.
*
* Inputs:
*
*   hTrafficLocID - The TRAFFIC_LOCID_OBJECT to get this value from
*
* Outputs:
*
*   The TRAFFIC_BSA value on success.
*     TRAFFIC_INVALID_BSA on failure or if there is no BSA
*     associated with this object
*
*****************************************************************************/
static TRAFFIC_BSA tBSA (
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    BOOLEAN bValid;
    TRAFFIC_BSA tBSA = TRAFFIC_INVALID_BSA;
    LOCID_TYPE_ENUM eType;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);
    if(bValid == FALSE)
    {
        // Error!
        return TRAFFIC_INVALID_BSA;
    }

    eType = LOCID.eType((LOCID_OBJECT)hTrafficLocID);

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        LOCID_UNION *puSubClassData;
        // Get a pointer to the union to fill in our
        // child-class data
        puSubClassData = LOCID_puSubClassData(hTrafficLocID);

        if (puSubClassData == LOCID_INVALID_UNION)
        {
            // Error!
            return TRAFFIC_INVALID_BSA;
        }

        tBSA = puSubClassData->sTrafficLocID.tBSA;
    }

    return tBSA;
}


/*****************************************************************************
*
*   tOffset
*
*****************************************************************************/
static TRAFFIC_OFFSET tOffset (
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    BOOLEAN bValid;
    TRAFFIC_OFFSET tOffset = 0;
    LOCID_TYPE_ENUM eType;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);
    if(bValid == FALSE)
    {
        // Error!
        return 0;
    }

    eType = LOCID.eType((LOCID_OBJECT)hTrafficLocID);

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        LOCID_UNION *puSubClassData;
        // Get a pointer to the union to fill in our
        // child-class data
        puSubClassData = LOCID_puSubClassData(hTrafficLocID);
        if (puSubClassData == LOCID_INVALID_UNION)
        {
            // Error!
            return 0;
        }

        tOffset = puSubClassData->sTrafficLocID.tOffset;
    }

    return tOffset;
}


/*****************************************************************************
*
*   eType
*
*****************************************************************************/
static TRAFFIC_LOCID_TYPE_ENUM eType (
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    TRAFFIC_LOCID_TYPE_ENUM eTrafficLocIDType = TRAFFIC_LOCID_TYPE_INVALID;

    do
    {
        BOOLEAN bValid;
        LOCID_TYPE_ENUM eType;
        TRAFFIC_POS_CODE tPosCode = TRAFFIC_INVALID_POS_CODE;
        LOCID_UNION *puSubClassData;

        // Verify inputs. Object handle must be valid.
        bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);
        if(bValid == FALSE)
        {
            // Error!
            break;
        }

        eType = LOCID.eType((LOCID_OBJECT)hTrafficLocID);
        if (eType != LOCID_TYPE_TRAFFIC)
        {
            break;
        }

        // Get a pointer to the union to fill in our
        // child-class data
        puSubClassData = LOCID_puSubClassData(hTrafficLocID);
        if (puSubClassData == LOCID_INVALID_UNION)
        {
            // Error!
            break;
        }

        if (puSubClassData->sTrafficLocID.tMarket == TRAFFIC_INVALID_MARKET)
        {
            break;
        }

        eTrafficLocIDType = TRAFFIC_LOCID_TYPE_MARKET;

        if (puSubClassData->sTrafficLocID.tBSA == TRAFFIC_INVALID_BSA)
        {
            break;
        }

        eTrafficLocIDType = TRAFFIC_LOCID_TYPE_BSA;

        tPosCode = (TRAFFIC_POS_CODE)LOCID.tID((LOCID_OBJECT)hTrafficLocID);
        if (tPosCode == TRAFFIC_INVALID_POS_CODE)
        {
            break;
        }

        eTrafficLocIDType = TRAFFIC_LOCID_TYPE_POINT;
    } while (FALSE);

    return eTrafficLocIDType;
}


/*****************************************************************************
*
*   n16CompareMarkets
*
* This interface method compares the relationship of two TRAFFIC_LOCIDs.
* This performs a relationship (lt, gt, eq) compare typically used for sorting.
*
*       Outputs:
*               0   - TRAFFIC_LOCIDs have the same Market value
*               > 0 - hTrafficLocID1 is greater than (after) hTrafficLocID2
*               < 0 - hTrafficLocID1 is less than (before) hTrafficLocID2
*                       (or error)
*
*****************************************************************************/
static N16 n16CompareMarkets (
    TRAFFIC_LOCID_OBJECT hTrafficLocID1,
    TRAFFIC_LOCID_OBJECT hTrafficLocID2
    )
{
    return n16CompareTrafficLocIDs(hTrafficLocID1,
                                   hTrafficLocID2,
                                   TRAFFIC_LOCID_COMPARE_STYLE_MARKET);
}

/*****************************************************************************
*
*   n16CompareBSAs
*
* This interface method compares the relationship of two TRAFFIC_LOCIDs.
* This performs a relationship (lt, gt, eq) compare typically used for sorting.
* Since we are comparing BSA
*
*       Outputs:
*               0   - TRAFFIC_LOCIDs have the same market and BSA value
*               > 0 - hTrafficLocID1 is greater than (after) hTrafficLocID2
*               < 0 - hTrafficLocID1 is less than (before) hTrafficLocID2
*                       (or error)
*
*****************************************************************************/
static N16 n16CompareBSAs (
    TRAFFIC_LOCID_OBJECT hTrafficLocID1,
    TRAFFIC_LOCID_OBJECT hTrafficLocID2
    )
{
    return n16CompareTrafficLocIDs(hTrafficLocID1,
                                   hTrafficLocID2,
                                   TRAFFIC_LOCID_COMPARE_STYLE_BSA);
}



/*****************************************************************************
*
*   hFRead
*
* This object interface method is used to read from a specified device psFile
* and generate a TRAFFIC_LOCID from that information. The data read from the device
* must have been previously written by the n32FWrite method. Upon
* successful execution of this method a new TRAFFIC_LOCID is created (created by the
* caller) which may be used to register for events or presented to the
* UI for display, etc. This method allows the caller to effectively read
* a previously stored TRAFFIC_LOCID regenerating the original TRAFFIC_LOCID
* written(saved) at an earlier time.
*
* Inputs:
*
*   psFile - The device to read the TRAFFIC_LOCID contents from.
*
* Outputs:
*
*   A new TRAFFIC_LOCID on success,
*   otherwise TRAFFIC_LOCID_INVALID_OBJECT on failure.
*
*****************************************************************************/
static TRAFFIC_LOCID_OBJECT hFRead (
    FILE *psFile
        )
{
    TRAFFIC_POS_CODE tPosCode = TRAFFIC_INVALID_POS_CODE;
    LOCID_TYPE_ENUM    eType;
    UN32 un32Type = 0;

    size_t tTotal, tLen;

    // Verify input
    if(psFile == NULL)
    {
        return LOCID_INVALID_OBJECT;
    }

    // Read TRAFFIC_LOCID from file...

    // Read tPosCode
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&tPosCode) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(TRAFFIC_POS_CODE)));

    if(tTotal != sizeof(TRAFFIC_POS_CODE))
    {
        // End of file or error occurred
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    // Read the type of LOCID
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&un32Type) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(UN32)));

    if(tTotal != sizeof(UN32))
    {
        // End of file or error occurred
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    // Cast type
    eType = (LOCID_TYPE_ENUM)un32Type;

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        return TRAFFIC_LOCID_hFRead((LOC_ID)tPosCode, psFile);
    }
    else
    {
        // Unknown type, return an error
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }


}


/*****************************************************************************
*
*   vDestroy
*
* This object interface method is used by the caller to destroy the specified
* TRAFFIC_LOCID object and all members of the object
*

* Inputs:
*
*   hTrafficLocID - A handle to a valid TRAFFIC_LOCID_OBJECT that the caller
*                    wants to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
static void vDestroy (
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    BOOLEAN bValid;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);
    if(bValid == FALSE)
    {
        // Error!
        return;
    }

    printf(TRAFFIC_LOCID_OBJECT_NAME": Traffic LocID destroyed 0x%p\n", hTrafficLocID);

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hTrafficLocID);

    return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroy
*
*****************************************************************************/
LOCID_OBJECT TRAFFIC_LOCID_hFRead (
    LOC_ID tID,
    FILE *psFile
        )
{
    TRAFFIC_POS_CODE tPosCode = tID;
    TRAFFIC_MARKET tMarket = TRAFFIC_INVALID_MARKET;
    TRAFFIC_BSA tBSA = TRAFFIC_INVALID_BSA;

    size_t tTotal, tLen;

    // Verify input
    if(psFile == NULL)
    {
        return LOCID_INVALID_OBJECT;
    }

    // Read rest of the TRAFFIC_LOCID from file...

    // Read tMarket
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&tMarket) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(TRAFFIC_MARKET)));

    if(tTotal != sizeof(TRAFFIC_MARKET))
    {
        // End of file or error occured
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    // Read tBSA
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&tBSA) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(TRAFFIC_BSA)));

    if(tTotal != sizeof(TRAFFIC_BSA))
    {
        // End of file or error occurred
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    return TRAFFIC_LOCID.hCreate(tPosCode, tMarket, tBSA);
}
/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/


/*****************************************************************************
*
*   hDuplicate
*
* This LOCID interface method is used to access the do a "deep-copy" of a
* TRAFFIC_LOCID.
*
* Inputs:
*
*   hLocID - The LOCID_OBJECT object to duplicate
*
* Outputs:
*
*   A LOCID that is the duplicate of hLocID
*
*****************************************************************************/
static TRAFFIC_LOCID_OBJECT hDuplicate (
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        )
{
    LOCID_UNION *puSubClassData;
    TRAFFIC_POS_CODE tPosCode;
    LOCID_TYPE_ENUM eType;
    TRAFFIC_LOCID_OBJECT hTrafficLocIDCopy = TRAFFIC_LOCID_INVALID_OBJECT;

    // Verify inputs
    if(SMSO_bValid((SMS_OBJECT)hTrafficLocID) == FALSE )
    {
        // Error!
        return TRAFFIC_LOCID_INVALID_OBJECT;
    }

    eType = LOCID.eType((LOCID_OBJECT)hTrafficLocID);

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        // Get Position Code via super class
        tPosCode = (TRAFFIC_POS_CODE)LOCID.tID(hTrafficLocID);

        // Get a pointer to the union to fill in our
        // child-class data
        puSubClassData = LOCID_puSubClassData(hTrafficLocID);

        if (puSubClassData == LOCID_INVALID_UNION)
        {
            // Error!
            return TRAFFIC_LOCID_INVALID_OBJECT;
        }

        hTrafficLocIDCopy =
            TRAFFIC_LOCID.hCreate(tPosCode,
                puSubClassData->sTrafficLocID.tMarket,
                puSubClassData->sTrafficLocID.tBSA);
    }

    return hTrafficLocIDCopy;
}

/*****************************************************************************
*
*   n32FWrite
*
* This LOCID interface method is used to write the contents of a LOCID
* (performing a deep write) to a device specified by psFile. The intent of
* this method is to effectively store the contents of a LOCID for later
* retrieval (perhaps after a power cycle). The entire LOCID contents are written
* to the device, with enough information to recreate the exact LOCID
* when the hFRead() method is called.
*
* Inputs:
*
*   hLocID - The LOCID handle the caller wishes to write.
*   psFile - The device to write the LOCID contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FWrite (
    TRAFFIC_LOCID_OBJECT hTrafficLocID,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    BOOLEAN bValid;
    LOCID_TYPE_ENUM eType;
    UN32 un32BSA;
    UN8 un8ByteToWrite;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);
    if((bValid == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Call superclass write function
    n32Return = LOCID_n32FWrite(hTrafficLocID, psFile);

    eType = LOCID.eType((LOCID_OBJECT)hTrafficLocID);

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        LOCID_UNION *puSubClassData;

        // Get a pointer to the union to fill in our
        // child-class data
        puSubClassData = LOCID_puSubClassData(hTrafficLocID);

        if (puSubClassData == LOCID_INVALID_UNION)
        {
            // Error!
            return EOF;
        }

        un8ByteToWrite = BYTE0(puSubClassData->sTrafficLocID.tMarket);
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);

        un32BSA = (UN32)puSubClassData->sTrafficLocID.tBSA;
        un8ByteToWrite = BYTE0(un32BSA);
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
        un8ByteToWrite = BYTE1(un32BSA);
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
        un8ByteToWrite = BYTE2(un32BSA);
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
        un8ByteToWrite = BYTE3(un32BSA);
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);


    }

    return n32Return;
}

/*****************************************************************************
*
*   n16Compare
*
* This LOCID interface method compares the relationship of two TRAFFIC_LOCIDs.
* This performs a relationship (lt, gt, eq) compare typically used for sorting.
*
*       Outputs:
*               0   - TRAFFIC_LOCIDs have the same value
*               > 0 - hTrafficLocID1 is greater than (after) hTrafficLocID2
*               < 0 - hTrafficLocID1 is less than (before) hTrafficLocID2
*                       (or error)
*
*****************************************************************************/
static N16 n16Compare (
    TRAFFIC_LOCID_OBJECT hTrafficLocID1,
    TRAFFIC_LOCID_OBJECT hTrafficLocID2
    )
{
    return n16CompareTrafficLocIDs(hTrafficLocID1,
                                   hTrafficLocID2,
                                   TRAFFIC_LOCID_COMPARE_STYLE_FULL);
}

/*****************************************************************************
*
*   n32FPrintf
*
* This LOCID interface method is used by the caller to send formatted
* output of a TRAFFIC_LOCID's contents to a specified file or device.
* This is mainly helpful during debugging of TRAFFIC_LOCID's but could be used by
* an application for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a TRAFFIC_LOCID to a file for the
* purposes of later re-generating the TRAFFIC_LOCID (for storage of the object).
* This API instead sends the TRAFFIC_LOCID as a verbose formatted output version.
*
* Inputs:
*
*   hTrafficLocID - The TRAFFIC_LOCID handle the caller wishes to write.
*   psFile - The device to write the TRAFFIC_LOCID contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    TRAFFIC_LOCID_OBJECT hTrafficLocID,
    FILE *psFile
        )
{
    LOCID_UNION *puSubClassData;
    TRAFFIC_POS_CODE tPosCode;
    LOCID_TYPE_ENUM eType;
    N32 n32Return = 0;
    BOOLEAN bValid;

    // Determine if the caller owns this resource
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bValid == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    eType = LOCID.eType((LOCID_OBJECT)hTrafficLocID);

    if (eType == LOCID_TYPE_TRAFFIC)
    {
        // Get the ID (Position Code)
        tPosCode = (TRAFFIC_POS_CODE)LOCID.tID(hTrafficLocID);
        puSubClassData = LOCID_puSubClassData(hTrafficLocID);

        if (puSubClassData == LOCID_INVALID_UNION)
        {
            // Error!
            return EOF;
        }

        // Print TRAFFIC_LOCID information header
        n32Return += fprintf(psFile, "TRAFFIC_LOCID: hTrafficLocID = 0x%p\n",
            hTrafficLocID);

        // Print TRAFFIC_LOCID information body
        n32Return += fprintf(psFile, "\tPosCode = %d\n",
            tPosCode);
        n32Return += fprintf(psFile, "\tMarket = %d\n",
            puSubClassData->sTrafficLocID.tMarket);
        n32Return += fprintf(psFile, "\tBSA = %d\n",
            puSubClassData->sTrafficLocID.tBSA);

    }
    else
    {
        // This isn't a TRAFFIC_LOCID so send it to the superclass
        n32Return = LOCID.n32FPrintf(hTrafficLocID, psFile);
    }

    return n32Return;
}

/*****************************************************************************
*
*   n16CompareTrafficLocIDs
*
* This private function is used to compare to TRAFFIC_LOCID objects
*
* Inputs:
*
*   hTrafficLocID1 - The TRAFFIC_LOCID handle the caller wishes to write.
*   hTrafficLocID2 - The device to write the TRAFFIC_LOCID contents to.
*     eCompareStyle - The style of compare to do:
*                     TRAFFIC_LOCID_COMPARE_STYLE_FULL -- Compare all object
*                                                         feild values
*                    TRAFFIC_LOCID_COMPARE_STYLE_MARKET -- Compare the market
*                                                         feild values
*                     TRAFFIC_LOCID_COMPARE_STYLE_BSA -- Compare the BSA
*                                                         feild values
*
* Outputs:
*
*               0   - TRAFFIC_LOCIDs have the same value based on
*                       eCompareStyle
*               > 0 - hTrafficLocID1 is greater than (after) hTrafficLocID2
*               < 0 - hTrafficLocID1 is less than (before) hTrafficLocID2
*                       (or error)
*
*****************************************************************************/
static N16 n16CompareTrafficLocIDs (
    TRAFFIC_LOCID_OBJECT hTrafficLocID1,
    TRAFFIC_LOCID_OBJECT hTrafficLocID2,
    TRAFFIC_LOCID_COMPARE_STYLE_ENUM eCompareStyle
        )
{
    N16 n16Result = N16_MIN;
    LOCID_TYPE_ENUM eType1, eType2;
    LOCID_UNION *puSubClassData1, *puSubClassData2;

    BOOLEAN bValid = FALSE;

    // Verify inputs. Object handles must be valid and caller own both.
    bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID1);
    if(bValid == TRUE)
    {
        bValid = SMSO_bValid((SMS_OBJECT)hTrafficLocID2);
    }
    if(bValid == FALSE)
    {
        // Error!
        return N16_MIN;
    }

    eType1 = LOCID.eType(hTrafficLocID1);
    eType2 = LOCID.eType(hTrafficLocID2);

    if (eType1 != LOCID_TYPE_TRAFFIC ||
        eType2 != LOCID_TYPE_TRAFFIC)
    {
        // These two objects are not both TRAFFIC_LOCID
        // so return an error
        return N16_MIN;
    }

    // Fetch the subclass data
    puSubClassData1 = LOCID_puSubClassData(hTrafficLocID1);
    puSubClassData2 = LOCID_puSubClassData(hTrafficLocID2);

    if (puSubClassData1 == LOCID_INVALID_UNION ||
        puSubClassData2 == LOCID_INVALID_UNION)
    {
        // Error!
        return N16_MIN;
    }

    // if the markets are not equal, return the realtionship
    // based on the numerical order of the markets
    // i.e, Market 8 (Detroit) is greater thatn Market 1
    n16Result = COMPARE(puSubClassData1->sTrafficLocID.tMarket, 
        puSubClassData2->sTrafficLocID.tMarket);
    if (n16Result == 0)
    {
        // Markets are equal
        // If we are just comparing markets, we would stop here
        if (eCompareStyle != TRAFFIC_LOCID_COMPARE_STYLE_MARKET)
        {
            // Perform further comparisons
            //
            // A TRAFFIC_LOCID without a BSA is
            // "less-than" one with a BSA
            if (puSubClassData1->sTrafficLocID.tBSA == TRAFFIC_INVALID_BSA &&
                puSubClassData2->sTrafficLocID.tBSA != TRAFFIC_INVALID_BSA)
            {
                n16Result = -1;
            }
            else
            {
                if (puSubClassData1->sTrafficLocID.tBSA <
                    puSubClassData2->sTrafficLocID.tBSA)
                {
                    n16Result = -1;
                }
                else if (puSubClassData1->sTrafficLocID.tBSA >
                         puSubClassData2->sTrafficLocID.tBSA)
                {
                    n16Result = 1;
                }
                else
                {
                    n16Result = 0;
                }
            }
            if (n16Result == 0)
            {
                // BSAs are equal
                // If we are just comparing BSAs, we would stop here
                if (eCompareStyle != TRAFFIC_LOCID_COMPARE_STYLE_BSA)
                {
                    // Perform further comparisons
                    //
                    TRAFFIC_POS_CODE tPosCode1, tPosCode2;

                    tPosCode1 = LOCID.tID(hTrafficLocID1);
                    tPosCode2 = LOCID.tID(hTrafficLocID2);

                    // A TRAFFIC_LOCID without a PosCode is
                    // "less-than" one with a PosCode

                    if (tPosCode1 == TRAFFIC_INVALID_POS_CODE &&
                        tPosCode2 != TRAFFIC_INVALID_POS_CODE)
                    {
                        n16Result = -1;
                    }
                    else
                    {
                        if (tPosCode1 < tPosCode2)
                        {
                            n16Result = -1;
                        }
                        else if (tPosCode1 > tPosCode2)
                        {
                            n16Result = 1;
                        }
                        else
                        {
                            n16Result = 0;
                        }
                    }

                }
            }
        }
    }

    return n16Result;
}



