/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the traffic
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef TRAFFIC_DB_CONSTANTS_H_
#define TRAFFIC_DB_CONSTANTS_H_

  /**************/
 /** INCLUDES **/
/**************/
#include "sms_api.h"
#include "alertc_event_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* Database File Name */
#define TRAFFIC_DATABASE_FOLDER "traffic"
#define TRAFFIC_REF_DATABASE_FILENAME "traffic_ref.db"
#define TRAFFIC_PERSIST_DATABASE_FILENAME "traffic.db"

/* Database Table Names */
#define DATABASE_VERSION_TABLE_NAME "db_version"
#define EVENTS_TABLE_NAME "Events"
#define SUPPLEMENTAL_INFO_TABLE_NAME "Sup_Info"
#define TRAFFIC_STORAGE_TABLE_NAME "traffic_storage"

/* Database Table Attributes */
#define TRAFFIC_DATABASE_FILE_VERSION "4"

/* Constants used by the Events table to signify the event type
   which matches with ALERTC_EVENT_TYPE_ENUM */
#define EVENTS_TABLE_ALERTC_TYPE_CANCLLED "C"
#define EVENTS_TABLE_ALERTC_TYPE_INCIDENT "I"
#define EVENTS_TABLE_ALERTC_TYPE_SPEED_AND_FLOW "SF"

/* Database Queries */

// Select all from version
#define TRAFFIC_SELECT_DB_VERSION \
	"select * from "DATABASE_VERSION_TABLE_NAME";"

// Select alert-c event information
#define TRAFFIC_SELECT_EVENT_INFO \
    "select * from "EVENTS_TABLE_NAME" where code=%4u;"

// Select supplemental code information
#define TRAFFIC_SELECT_SUPPL_INFO \
    "select * from "SUPPLEMENTAL_INFO_TABLE_NAME" where code=%3u;"

// Remove the traffic events for a market from the traffic storage table
#define TRAFFIC_DELETE_MARKET \
    "DELETE FROM "TRAFFIC_STORAGE_TABLE_NAME" WHERE market=%2d;"

// Add a traffic message to the traffic storage table
#define TRAFFIC_ADD_MSG_TO_TRAFFIC_TABLE \
    "INSERT OR REPLACE INTO "TRAFFIC_STORAGE_TABLE_NAME" VALUES " \
    "(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"

#define TRAFFIC_ADD_STMT_TIME_STAMP_PARAM   0
#define TRAFFIC_ADD_STMT_MARKET_PARAM       1
#define TRAFFIC_ADD_STMT_BSA_PARAM          2
#define TRAFFIC_ADD_STMT_POS_CODE_PARAM     3
#define TRAFFIC_ADD_STMT_EXTENT_PARAM       4
#define TRAFFIC_ADD_STMT_EVENT_CLASS_PARAM  5
#define TRAFFIC_ADD_STMT_EVENT_CODE_PARAM   6
#define TRAFFIC_ADD_STMT_DIRECTION_PARAM    7
#define TRAFFIC_ADD_STMT_DURATION_PARAM     8
#define TRAFFIC_ADD_STMT_DIVERSION_PARAM    9
#define TRAFFIC_ADD_STMT_MGR_COUNT_PARAM   10
#define TRAFFIC_ADD_STMT_FFD_COUNT_PARAM   11
#define TRAFFIC_ADD_STMT_FFD               12
#define TRAFFIC_ADD_STMT_PARAMS_COUNT      13

#define TRAFFIC_SELECT_STORED_MESSAGES \
    "SELECT * FROM "TRAFFIC_STORAGE_TABLE_NAME" WHERE " \
    "market=%2d;"

#define TRAFFIC_REMOVE_EXPIRED_MESSAGES \
    "DELETE FROM "TRAFFIC_STORAGE_TABLE_NAME" WHERE " \
    " UnixTimeStamp <= %u;"

#define TRAFFIC_CREATE_STORAGE_TABLE \
     "CREATE TABLE "TRAFFIC_STORAGE_TABLE_NAME" ( " \
     "UnixTimeStamp int, " \
     "Market tinyint, " \
     "BSA int, " \
     "PosCode int, " \
     "Extent tinyint, " \
     "EventClass int," \
     "EventCode int, " \
     "Direction tinyint, " \
     "Duration int, " \
     "Diversion tinyint, " \
     "MultiGroupCount tinyint," \
     "FreeFormDataCount tinyint, " \
     "FreeFormData blob, " \
     "primary key (Market, PosCode, Direction, EventClass));"

#define TRAFFIC_CREATE_VERSION_TABLE \
    "CREATE TABLE "DATABASE_VERSION_TABLE_NAME" (dbVer);"

#define TRAFFIC_INSERT_VERSION_ROW \
    "INSERT INTO "DATABASE_VERSION_TABLE_NAME" VALUES('" \
    TRAFFIC_DATABASE_FILE_VERSION \
    "');"

// Constant used to size buffers used to perform
// DB queries.  Size is based on largest string
// size (all defines above specify field lengths)
#define TRAFFIC_MAX_SQL_STRING_LENGTH (80)

// Constant used to store the supplemental info string
// Size is based on largest field in database.
#define TRAFFIC_MAX_SUPPL_INFO_STRING_LENGTH (60)

// Each free form data entry has a label and a value.
// The smallest length of a value is zero bits.  Therefore, the max
// number of freeform entires we could have would be when the free form
// data field is filled with values that are zero bits.  This means it will
// just be filled with labels.
// All but the first message in a multi-group message can contain freeform data
#define MAX_FREEFORM_ENTRIES          (28)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Enumeration specifying all the available fields
// in the db_version database relation
typedef enum db_version_fields_enum {
    DB_VERSION_FIELD_DB_VER = 0,
    DB_VERSION_MAX_FIELDS

} DB_VERSION_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the Events table
typedef enum events_fields_enum {
    EVENTS_FIELD_CODE = 0,
    EVENTS_FIELD_TEXT_US,
    EVENTS_FIELD_TEXT_MOD_US,
    EVENTS_FIELD_TYPE,
    EVENTS_FIELD_QUANT_TYPE,
    EVENTS_FIELD_CLASS,
    EVENTS_MAX_FIELDS
} EVENTS_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the Supplemental Info table
typedef enum suppl_info_fields_enum {
    SUPPL_INFO_FIELD_CODE = 0,
    SUPPL_INFO_FIELD_TEXT_US,
    SUPPL_INFO_MAX_FIELDS
} SUPPL_INFO_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the Traffic Storage table
typedef enum traffic_storage_fields_enum {
    TRAFFIC_STORAGE_TIMESTAMP = 0,
    TRAFFIC_STORAGE_MARKET,
    TRAFFIC_STORAGE_BSA,
    TRAFFIC_STORAGE_POS_CODE,
    TRAFFIC_STORAGE_EXTENT,
    TRAFFIC_STORAGE_EVENT_CLASS,
    TRAFFIC_STORAGE_EVENT_CODE,
    TRAFFIC_STORAGE_DIRECTION,
    TRAFFIC_STORAGE_DURATION,
    TRAFFIC_STORAGE_DIVERSION,
    TRAFFIC_STORAGE_MULTI_GROUP_COUNT,
    TRAFFIC_STORAGE_FREE_FORM_DATA_COUNT,
    TRAFFIC_STORAGE_FREE_FORM_DATA_BLOB,
    TRAFFIC_STORAGE_MAX_FIELDS
} TRAFFIC_STORAGE_FIELDS_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// Structure which specifies all attributes
// of a single database version row that is
// available within the traffic database
// These are the raw values that are retrieved from the database
typedef struct traffic_version_row_struct
{
    UN8 un8DBVer;           // Database schema version

} TRAFFIC_VERSION_ROW_STRUCT;

typedef struct traffic_events_row_struct
{
    TRAFFIC_EVENT_CODE tCode;
    ALERTC_EVENT_TYPE_ENUM eType;
    TRAFFIC_MSG_CLASS_ENUM eClass;
    ALERTC_QUANTIFIER_TYPE_ENUM eQuantType;
    STRING_OBJECT hText;
    STRING_OBJECT hTextMod;

} TRAFFIC_EVENTS_ROW_STRUCT;

typedef struct traffic_suppl_info_row_struct
{
    ALERTC_SUPPL_INFO_CODE tCode;
    char acText[TRAFFIC_MAX_SUPPL_INFO_STRING_LENGTH];

} TRAFFIC_SUPPL_INFO_ROW_STRUCT;

typedef struct traffic_storage_row_struct
{
    UN32 un32UTCsec;
    TRAFFIC_MARKET tMarket;
    TRAFFIC_BSA tBSA;
    TRAFFIC_POS_CODE tPosCode;
    UN8 un8Extent;
    TRAFFIC_EVENT_CODE tEventCode;
    TRAFFIC_DIRECTION_ENUM eDirection;
    UN8 un8Duration;
    BOOLEAN bDiversionAdvised;
    UN8 un8MultiGroupCount;
    UN8 un8FreeFormDataCount;
    UN16 aun16FreeFormData[MAX_FREEFORM_ENTRIES*2];
} TRAFFIC_STORAGE_ROW_STRUCT;

// Union to group all the row definitions
// in a convenient manner
typedef union traffic_db_row_union
{
    TRAFFIC_VERSION_ROW_STRUCT sVersion;
    TRAFFIC_EVENTS_ROW_STRUCT sEvents;
    TRAFFIC_SUPPL_INFO_ROW_STRUCT sSupplInfo;

} TRAFFIC_DB_ROW_UNION;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif // TRAFFIC_DB_CONSTANTS_H_
