/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:THEATER_TIMES implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "movies_mgr_obj.h"
#include "dsrl_entry_obj.h"
#include "movies_db_constants.h"
#include "movie_obj.h"

#include "theater_times_obj.h"
#include "_theater_times_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*   TODO: Update this header
*   THEATER_TIMES_hCreate
*
* This object interface method is used by the caller to create a MOVIE
* object with data received from the movie times protocl.
*
* This function assumes that the STRING_OBJECTs were created for our use
* by hMoviesService, so there is no need to duplicate
*
* Inputs:
*
*   hMoviesService - The data service object that is the parent to this object
*   tID - The service-defined ID for this movie
*   hName - The name of the movie
*   un8RunTime - The nubmer of minutes this movie runs for
*   hActors - Text containing the actors in this movie
*   hSynopis - The text description for the movie
*   eRating - The rating as an enum, if available
*   hRatingText - The rating as text if available
*
* Outputs:
*
*   A handle to a MOVIE object on success
*   A MOVIE_INVALID_OBJECT on failrue
*
*****************************************************************************/
THEATER_TIMES_OBJECT THEATER_TIMES_hCreate (
    MOVIES_SERVICE_OBJECT hMoviesService,
    SMS_OBJECT hParent,
    THEATER_ID tID,
    UN32 un32InitialTime,
    UN8 un8NumMovies
        )
{
    THEATER_TIMES_OBJECT_STRUCT *psObj =
        (THEATER_TIMES_OBJECT_STRUCT *)THEATER_TIMES_INVALID_OBJECT;
    BOOLEAN bSuccess;

    // Create an instance of the THEATER_TIMES object
    psObj = (THEATER_TIMES_OBJECT_STRUCT *)
        SMSO_hCreate(
            THEATER_TIMES_OBJECT_NAME,
            sizeof(THEATER_TIMES_OBJECT_STRUCT),
            (SMS_OBJECT)hParent,
            FALSE);

    // If we have a pointer, set it up, otherwise return
    // the invalid handle
    if(psObj != NULL)
    {
        // Save the service handle
        psObj->hMoviesService = hMoviesService;

        // Initialize the object
        bSuccess = bSetupTheaterTimes(
            psObj, tID, un32InitialTime, un8NumMovies);

        if (bSuccess == FALSE)
        {
            vDestroyObject(psObj);
            psObj = (THEATER_TIMES_OBJECT_STRUCT *)
                THEATER_TIMES_INVALID_OBJECT;
        }
    }

    return (THEATER_TIMES_OBJECT)psObj;
}

/*****************************************************************************
*
*   THEATER_TIMES_hCreateFromDB
*
* This object interface method is used by the caller to create a MOVIE
* object with data received from the movie times protocl.
*
* This function assumes that the STRING_OBJECTs were created for our use
* by hMoviesService, so there is no need to duplicate
*
* Inputs:
*
*   hMoviesService - The data service object that is the parent to this object
*   hParent - The SMS object to use as this object's parent
*   psTimesRow - The row of data which defines this object
*
* Outputs:
*
*   A handle to a MOVIE object on success
*   A MOVIE_INVALID_OBJECT on failrue
*
*****************************************************************************/
THEATER_TIMES_OBJECT THEATER_TIMES_hCreateFromDB (
    MOVIES_SERVICE_OBJECT hMoviesService,
    SMS_OBJECT hParent,
    THEATER_TIMES_ROW_STRUCT *psTimesRow
        )
{
    THEATER_TIMES_OBJECT_STRUCT *psObj =
            (THEATER_TIMES_OBJECT_STRUCT *)THEATER_TIMES_INVALID_OBJECT;
    BOOLEAN bSuccess;

    if (psTimesRow == NULL)
    {
        return THEATER_TIMES_INVALID_OBJECT;
    }

    // Create an instance of the THEATER_TIMES object
    psObj = (THEATER_TIMES_OBJECT_STRUCT *)
        SMSO_hCreate(
            THEATER_TIMES_OBJECT_NAME,
            sizeof(THEATER_TIMES_OBJECT_STRUCT),
            (SMS_OBJECT)hParent,
            FALSE);

    // If we have a pointer, set it up, otherwise return
    // the invalid handle
    if(psObj != NULL)
    {
        // Save the service handle
        psObj->hMoviesService = hMoviesService;

        // Initialize the object
        bSuccess =
            bSetupTheaterTimes(psObj,
                psTimesRow->tID,
                psTimesRow->un32InitialTime,
                psTimesRow->un8NumMovies);

        if (bSuccess == TRUE)
        {
            // only process times if setup successful
            bSuccess =
                bProcessTimesFromRow(psObj, psTimesRow);
        }

        // if either bSetupTheaterTimes or bProcessTimesFromRow returned FALSE....
        if (bSuccess == FALSE)
        {
            vDestroyObject(psObj);
            psObj =
                (THEATER_TIMES_OBJECT_STRUCT *)THEATER_TIMES_INVALID_OBJECT;
        }
    }

    return (THEATER_TIMES_OBJECT)psObj;
}

/*****************************************************************************
*
*   THEATER_TIMES_bReuse
*
*****************************************************************************/
BOOLEAN THEATER_TIMES_bReuse (
    THEATER_TIMES_OBJECT hTheaterTimes,
    THEATER_ID tID,
    UN32 un32InitialTime,
    UN8 un8NumMovies
        )
{
    BOOLEAN bOwner, bSuccess;
    THEATER_TIMES_OBJECT_STRUCT *psObj =
        (THEATER_TIMES_OBJECT_STRUCT *)hTheaterTimes;

    do
    {
        // Determine if we own this object
        bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);

        if (bOwner == FALSE || tID == THEATER_INVALID_ID)
        {
            break;
        }

        // Reset the initial values
        bSuccess = bResetObject(psObj);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Setup our object with the new values
        bSuccess = bSetupTheaterTimes(
            psObj, tID, un32InitialTime, un8NumMovies);

        if (bSuccess == FALSE)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   THEATER_TIMES_un8NumMovies
*
* This friend function reaturns the number of movies contained in this object

* Inputs:
*
*   hTheaterTimes - A handle to a valid THEATER_TIMES object that the
*                 caller wants to get the number of movies represented
*
* Outputs:
*
*    Returns a valid THEATER_ID upon success.
*    0 on error or if there are no movies
*
*****************************************************************************/
UN8 THEATER_TIMES_un8NumMovies (
    THEATER_TIMES_OBJECT hTheaterTimes
        )
{
    UN8 un8NumMovies = 0;
    BOOLEAN bOwner;

    // Determine if we own this object
    bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);
    if (bOwner == TRUE)
    {
        THEATER_TIMES_OBJECT_STRUCT *psObj =
            (THEATER_TIMES_OBJECT_STRUCT *)hTheaterTimes;

        un8NumMovies = psObj->un8NumMovies;
    }

    return un8NumMovies;
}

/*****************************************************************************
*
*   THEATER_TIMES_hAddMovie
*
*****************************************************************************/
THEATER_TIMES_MOVIE_ENTRY_OBJECT THEATER_TIMES_hAddMovie (
    THEATER_TIMES_OBJECT hTheaterTimes,
    MOVIE_ID tID,
    MOVIE_RATING_SYS_CODE tRatingExceptionSystem,
    MOVIE_RATING_CODE tRatingException
        )
{
    do
    {
        THEATER_TIMES_OBJECT_STRUCT *psObj =
            (THEATER_TIMES_OBJECT_STRUCT *)hTheaterTimes;
        MOVIE_TIMES_ENTRY_STRUCT *psTimesEntry;
        OSAL_LINKED_LIST_ENTRY hLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bHaveException = FALSE;
        MOVIE_RATING_STRUCT sRating = {
            MOVIE_RATING_SYSTEM_UNKNOWN,
            STRING_INVALID_OBJECT,
            MOVIE_RATING_UNKNOWN,
            STRING_INVALID_OBJECT,
            MOVIE_RATING_INVALID_SYS_CODE,
            MOVIE_RATING_INVALID_CODE
        };

        BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);

        // Verify inputs
        if ((bOwner == FALSE) || (tID == MOVIE_INVALID_ID))
        {
            break;
        }

        // Do we have a ratings exception for this movie?
        if (tRatingException != MOVIE_RATING_INVALID_CODE)
        {
            // Attempt to grab exception data
            bHaveException = MOVIES_MGR_bGetRatingInfo(
                psObj->hMoviesService,
                tRatingExceptionSystem,
                tRatingException,
                &sRating);
        }

        // Get an entry from our list that is free to use
        psTimesEntry = psFindMovieEntry(
            psObj, FALSE, MOVIE_INVALID_ID, &hLLEntry);

        if (psTimesEntry == NULL)
        {
            // We couldn't get a movie entry for some reason
            // so fail
            break;
        }

        // Set the Movie ID and mark it as in use
        psTimesEntry->bInUse = TRUE;
        psTimesEntry->tID = tID;
        psTimesEntry->bHaveException = bHaveException;

        // Do we have rating exception data?
        if (bHaveException == TRUE)
        {
            psTimesEntry->sException.eSystem = sRating.eRatingSystem;
            psTimesEntry->sException.eRating = sRating.eRating;
            psTimesEntry->sException.hRatingText = sRating.hRatingText;
            psTimesEntry->sException.tSysCode = tRatingExceptionSystem;
            psTimesEntry->sException.tRatingCode = tRatingException;
        }

        // TODO: move this so it only happens one time
        // Replace the entry so that it is resorted
        eReturnCode =
            OSAL.eLinkedListReplaceEntry(
                psObj->hMovieList, hLLEntry, (void*)psTimesEntry);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        return (THEATER_TIMES_MOVIE_ENTRY_OBJECT)psTimesEntry;

    } while (FALSE);

    return THEATER_TIMES_MOVIE_ENTRY_INVALID_OBJECT;
}

/*****************************************************************************
*
*   THEATER_TIMES_bAddShowTimeOffset
*
*****************************************************************************/
BOOLEAN THEATER_TIMES_bAddShowTimeOffset (
    THEATER_TIMES_OBJECT hTheaterTimes,
    THEATER_TIMES_MOVIE_ENTRY_OBJECT hMovieEntry,
    UN16 un16ShowTimeOffset
        )
{
    do
    {
        MOVIE_TIMES_ENTRY_STRUCT *psEntry =
            (MOVIE_TIMES_ENTRY_STRUCT *)hMovieEntry;

        BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);

        // Verify inputs
        if (bOwner == FALSE || hMovieEntry == NULL)
        {
            break;
        }

        // Verify this a in-use movie entry
        if (psEntry->bInUse == FALSE)
        {
            break;
        }

        // Make sure we aren't adding more show times than we are
        // supposed to.
        if (psEntry->un8NumShowTimes >= MOVIES_THEATER_TIMES_MAX_NUM_TIME_OFFSETS)
        {
            break;
        }

        // Add the show time
        psEntry->aun16TimeOffsets[psEntry->un8NumShowTimes++] = un16ShowTimeOffset;

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   THEATER_TIMES_eIterateMovies
*
****************************************************************************/
SMSAPI_RETURN_CODE_ENUM THEATER_TIMES_eIterateMovies (
    THEATER_TIMES_OBJECT hTheaterTimes,
    THEATER_OBJECT hTheater,
    THEATER_MOVIES_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eSMSReturnCode = SMSAPI_RETURN_CODE_ERROR;
    THEATER_TIMES_OBJECT_STRUCT *psObj =
        (THEATER_TIMES_OBJECT_STRUCT *)hTheaterTimes;
    BOOLEAN bOwner;

    // Determine if we own this object
    bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);
    if (bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        MOVIE_ITERATOR_STRUCT sIterator;

        sIterator.hParent = SMSO_hParent((SMS_OBJECT)hTheaterTimes);
        sIterator.psTheaterTimes = psObj;
        sIterator.hTheater = hTheater;
        sIterator.bIterator = bIterator;
        sIterator.pvIteratorArg = pvIteratorArg;

        // Call our private iterator function on our movie list
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->hMovieList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateMovies,
            (void *)&sIterator);

        // Condition the retun code
        if (eReturnCode == OSAL_SUCCESS)
        {
            eSMSReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else if (eReturnCode == OSAL_NO_OBJECTS)
        {
            eSMSReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        // Clear our current entry since we are done iterating
        psObj->psCurMovieEntry = NULL;
        psObj->hCurMovie = MOVIE_INVALID_OBJECT;
    }

    return eSMSReturnCode;
}

/*****************************************************************************
*
*   THEATER_TIMES_eIterateMovies
*   TODO: Add a friend function header block
****************************************************************************/
SMSAPI_RETURN_CODE_ENUM THEATER_TIMES_eIterateTimes (
    THEATER_TIMES_OBJECT hTheaterTimes,
    THEATER_OBJECT hTheater,
    THEATER_MOVIE_TIMES_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eSMSReturnCode = SMSAPI_RETURN_CODE_ERROR;
    THEATER_TIMES_OBJECT_STRUCT *psObj =
        (THEATER_TIMES_OBJECT_STRUCT *)hTheaterTimes;

    do
    {
        BOOLEAN bContinue, bOwner;
        UN8 un8Idx;
        UN32 un32StartTime;

        // Determine if we own this object
        bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);
        if (bOwner == FALSE)
        {
            break;
        }

        if (psObj->psCurMovieEntry == NULL)
        {
            // We need to have an iterated movie
            eSMSReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            break;
        }

        if (psObj->psCurMovieEntry->un8NumShowTimes == 0)
        {
            // There are no showtimes for this movie.
            // This isn't an error, the app just needs to tell the
            // user to call the theater
            eSMSReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
            break;
        }

        for (un8Idx = 0;
             un8Idx < psObj->psCurMovieEntry->un8NumShowTimes;
             un8Idx++)
        {
            un32StartTime = psObj->un32InitialTime +
                (psObj->psCurMovieEntry->aun16TimeOffsets[un8Idx] *
                                SECONDS_IN_FIVE_MINUTES);

            // Call the application's iterator callback
            bContinue = bIterator(hTheater, psObj->hCurMovie,
                un32StartTime, pvIteratorArg);

            if (bContinue == FALSE)
            {
                break;
            }
        }

        // All is good
        eSMSReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eSMSReturnCode;
}

/*****************************************************************************
*
*   THEATER_TIMES_bInsertIntoDB
*
****************************************************************************/
BOOLEAN THEATER_TIMES_bInsertIntoDB (
    THEATER_TIMES_OBJECT hTheaterTimes,
    SQL_INTERFACE_OBJECT hSQL,
    SQL_PREPARED_STATEMENT_HANDLE hStmt,
    SQL_BIND_PARAMETER_STRUCT *psBindParams,
    UN16 *pun16BlobData,
    size_t tBlobDataSize
        )
{
    THEATER_TIMES_OBJECT_STRUCT *psObj =
        (THEATER_TIMES_OBJECT_STRUCT *)hTheaterTimes;
    BOOLEAN bOwner, bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CREATE_THEATER_TIMES_BLOB_ITERATOR_STRUCT sIteratorArg;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);

    do
    {
        // Verify ownership
        if(bOwner != TRUE)
        {
            break;
        }

        // Prepare our iterator arg

        // We haven't written any UN16s in the blob field yet
        sIteratorArg.tNumUN16sWritten = 0;

        // Memory location for blob writing
        sIteratorArg.pun16BlobData = pun16BlobData;

        // Size of the blob data
        sIteratorArg.tBlobDataSize = tBlobDataSize;

        // No errors yet
        sIteratorArg.bError = FALSE;

        if (psObj->hMovieList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->hMovieList,
                (OSAL_LL_ITERATOR_HANDLER)bAddMovieTimesDataToBlob,
                (void*)&sIteratorArg);

            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        if (sIteratorArg.bError == TRUE)
        {
            break;
        }

        // Make sure that the blob is properly cleared
        // if we haven't written any times data
        if (0 == sIteratorArg.tNumUN16sWritten)
        {
            // Clear the blob data
            OSAL.bMemSet(
                &pun16BlobData[0], 0, 
                THEATER_TIMES_BLOB_NUM_OVERHEAD_UN16 * sizeof(UN16));

            // We have written in the overhead values
            sIteratorArg.tNumUN16sWritten = 
                THEATER_TIMES_BLOB_NUM_OVERHEAD_UN16;
        }

        psBindParams[MOVIES_INSERT_TIMES_STMT_ID_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psBindParams[MOVIES_INSERT_TIMES_STMT_ID_PARAM].pvData =
            (void *)(size_t)psObj->tID;

        psBindParams[MOVIES_INSERT_TIMES_STMT_INIT_TIME_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psBindParams[MOVIES_INSERT_TIMES_STMT_INIT_TIME_PARAM].pvData =
            (void *)(size_t)psObj->un32InitialTime;

        psBindParams[MOVIES_INSERT_TIMES_STMT_NUM_MOVIES_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psBindParams[MOVIES_INSERT_TIMES_STMT_NUM_MOVIES_PARAM].pvData =
            (void *)(size_t)psObj->un8NumMovies;

        psBindParams[MOVIES_INSERT_TIMES_STMT_NUM_UN16S_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psBindParams[MOVIES_INSERT_TIMES_STMT_NUM_UN16S_PARAM].pvData =
            (void *)(size_t)sIteratorArg.tNumUN16sWritten;

        psBindParams[MOVIES_INSERT_TIMES_STMT_SHOW_TIMES_PARAM].eType =
            SQL_BIND_TYPE_BLOB;
        psBindParams[MOVIES_INSERT_TIMES_STMT_SHOW_TIMES_PARAM].pvData =
            (void *)&pun16BlobData[0];
        psBindParams[MOVIES_INSERT_TIMES_STMT_SHOW_TIMES_PARAM].tSize =
            sIteratorArg.tNumUN16sWritten * sizeof(UN16);

        bOk = SQL_INTERFACE.bExecutePreparedStatement(
            hSQL, hStmt, NULL, NULL,
            MOVIES_INSERT_TIMES_STMT_PARAMS_COUNT,
            psBindParams);

        return bOk;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   THEATER_TIMES_vDestroy
*
* This object interface method is used by the caller to destroy the specified
* THEATER_TIME object and all members of the object
*

* Inputs:
*
*   hTheaterTimes - A handle to a valid THEATER_TIMES object that the
*                 caller wants to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
void THEATER_TIMES_vDestroy (
    THEATER_TIMES_OBJECT hTheaterTimes
        )
{
    THEATER_TIMES_OBJECT_STRUCT *psObj =
        (THEATER_TIMES_OBJECT_STRUCT *)hTheaterTimes;

    // Determine if we own this object
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hTheaterTimes);

    if (bOwner == TRUE)
    {
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bSetupTheaterTimes
*
*****************************************************************************/
static BOOLEAN bSetupTheaterTimes (
    THEATER_TIMES_OBJECT_STRUCT *psObj,
    THEATER_ID tID,
    UN32 un32InitialTime,
    UN8 un8NumMovies
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        psObj->tID = tID;
        psObj->un32InitialTime = un32InitialTime;
        psObj->un8NumMovies = un8NumMovies;

        // A value of 0 for un8NumMovies indicates the theater should
        // be marked as closed in the UI
        if (un8NumMovies > 0)
        {
            UN8 un8NumBlocksNeeded;

            if (psObj->hMovieList == OSAL_INVALID_OBJECT_HDL)
            {
                // create the movie list
                eReturnCode =
                    OSAL.eLinkedListCreate(
                        &psObj->hMovieList,
                        THEATER_TIMES_OBJECT_NAME":MvList",
                        (OSAL_LL_COMPARE_HANDLER)n16CompareMovieEntry,
                        OSAL_LL_OPTION_LINEAR);

                if (eReturnCode != OSAL_SUCCESS)
                {
                    break;
                }
            }

            // Determine the number of blocks we'll need for the number
            // of movies needed in this object
            un8NumBlocksNeeded =
                (psObj->un8NumMovies / NUM_MOVIE_TIMES_ENTRYS_IN_BLOCK) + 1;

            // We may already have some blocks allocated in our linked list
            // if we are reusing this object, so take that into account
            if (un8NumBlocksNeeded > psObj->un8NumBlocks)
            {
                MOVIE_TIMES_ENTRY_BLOCK_STRUCT *psBlock;
                BOOLEAN bSuccess = TRUE;
                UN8 un8Idx;

                // Only create the blocks we need
                un8NumBlocksNeeded -= psObj->un8NumBlocks;

                // Populate our linked list with more blocks if need be
                while (un8NumBlocksNeeded > 0)
                {
                    // allocate memory for a block
                    psBlock = (MOVIE_TIMES_ENTRY_BLOCK_STRUCT *)
                        SMSO_hCreate(
                            THEATER_TIMES_OBJECT_NAME":MovieBlock",
                            sizeof(MOVIE_TIMES_ENTRY_BLOCK_STRUCT),
                            (SMS_OBJECT)psObj,
                            FALSE);

                    if (psBlock == NULL)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    // add the entries to the linked list
                    for (un8Idx = 0;
                         un8Idx < NUM_MOVIE_TIMES_ENTRYS_IN_BLOCK;
                         un8Idx++)
                    {
                        eReturnCode = OSAL.eLinkedListAdd(
                            psObj->hMovieList,
                            &psBlock->ahEntries[un8Idx],
                            &psBlock->asData[un8Idx]);

                        if (eReturnCode != OSAL_SUCCESS)
                        {
                            bSuccess = FALSE;
                            break;
                        }

                        psBlock->asData[un8Idx].psHeader = psBlock;
                    }

                    un8NumBlocksNeeded--;
                    psObj->un8NumBlocks++;
                }

                if (bSuccess == FALSE)
                {
                    // Error populating the movie list
                    break;
                }
            }
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bResetObject
*
*   ASSUMES psObj and psTimesRow are NOT NULL
*****************************************************************************/
static BOOLEAN bProcessTimesFromRow (
    THEATER_TIMES_OBJECT_STRUCT *psObj,
    THEATER_TIMES_ROW_STRUCT *psTimesRow
        )
{
    BOOLEAN bSuccess = TRUE;
    UN8 un8MvIdx, un8ShowTimeIdx;
    size_t tCurUN16Idx = 0;
    MOVIE_ID tCurID;
    UN8 un8CurNumShowTimes;
    THEATER_TIMES_MOVIE_ENTRY_OBJECT hCurMovieEntry;
    UN16 un16CurShowTime;
    MOVIE_RATING_SYS_CODE tSysCode;
    MOVIE_RATING_CODE tRatingCode;
    BOOLEAN bHaveException;

    for (un8MvIdx = 0; un8MvIdx < psObj->un8NumMovies; un8MvIdx++)
    {
        if ((tCurUN16Idx + THEATER_TIMES_BLOB_NUM_OVERHEAD_UN16) > 
             psTimesRow->tNumUN16s)
        {
            // Not enough to read a movie entry
            bSuccess = FALSE;
            break;
        }

        // Read movie ID and number of show times
        tCurID = (MOVIE_ID)psTimesRow->pun16BlobData[tCurUN16Idx++];

        // Read the rating exception flag
        bHaveException = (BOOLEAN)psTimesRow->pun16BlobData[tCurUN16Idx++];

        if (bHaveException == FALSE)
        {
            // Invalidate these
            tSysCode = MOVIE_RATING_INVALID_SYS_CODE;
            tRatingCode = MOVIE_RATING_INVALID_CODE;

            // Skip the next two entries
            tCurUN16Idx += 2;
        }
        else
        {
            // Read the exception data
            tSysCode = (MOVIE_RATING_SYS_CODE)psTimesRow->pun16BlobData[tCurUN16Idx++];
            tRatingCode = (MOVIE_RATING_CODE)psTimesRow->pun16BlobData[tCurUN16Idx++];
        }

        un8CurNumShowTimes = (UN8)psTimesRow->pun16BlobData[tCurUN16Idx++];

        // Add this movie to the theater times object now
        hCurMovieEntry = THEATER_TIMES_hAddMovie(
            (THEATER_TIMES_OBJECT)psObj, 
            tCurID, 
            tSysCode,
            tRatingCode);

        if (hCurMovieEntry == THEATER_TIMES_MOVIE_ENTRY_INVALID_OBJECT)
        {
            bSuccess = FALSE;
            break;
        }

        if ((tCurUN16Idx + un8CurNumShowTimes) > psTimesRow->tNumUN16s)
        {
            // Not enough to read all the showtimes
            bSuccess = FALSE;
            break;
        }

        for (un8ShowTimeIdx = 0;
             un8ShowTimeIdx < un8CurNumShowTimes && bSuccess == TRUE;
             un8ShowTimeIdx++)
        {
            un16CurShowTime = psTimesRow->pun16BlobData[tCurUN16Idx++];

            bSuccess =
                THEATER_TIMES_bAddShowTimeOffset(
                    (THEATER_TIMES_OBJECT)psObj, hCurMovieEntry, un16CurShowTime);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bResetObject
*
*   ASSUMES psObj IS NOT NULL
*****************************************************************************/
static BOOLEAN bResetObject (
    THEATER_TIMES_OBJECT_STRUCT *psObj
        )
{
    do
    {
        psObj->tID = THEATER_INVALID_ID;
        psObj->un32InitialTime = 0;
        psObj->un8NumMovies = 0;
        psObj->un8NumBlocks = 0;

        if (psObj->hMovieList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode =
                OSAL.eLinkedListIterate(psObj->hMovieList,
                    (OSAL_LL_ITERATOR_HANDLER)bResetMovieEntry,
                    NULL);

            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    THEATER_TIMES_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        if (psObj->hMovieList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode =
                OSAL.eLinkedListRemoveAll(psObj->hMovieList,
                    (OSAL_LL_RELEASE_HANDLER)vReleaseBlock);

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListDelete(psObj->hMovieList);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    psObj->hMovieList = OSAL_INVALID_OBJECT_HDL;
                }
            }
        }

        psObj->tID = THEATER_INVALID_ID;
        psObj->un32InitialTime = 0;
        psObj->un8NumMovies = 0;
        psObj->un8NumBlocks = 0;

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   vDestoryObject
*
*****************************************************************************/
static void vReleaseBlock (
    MOVIE_TIMES_ENTRY_STRUCT *psEntry
        )
{
    if (psEntry != NULL)
    {
        UN8 un8Idx;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
        MOVIE_TIMES_ENTRY_BLOCK_STRUCT *psBlock = psEntry->psHeader;

        // Remove the linked list entries for items in this block
        // except for this entry
        for (un8Idx = 0;
             un8Idx < NUM_MOVIE_TIMES_ENTRYS_IN_BLOCK;
             un8Idx++)
        {
            if (&psBlock->asData[un8Idx] != psEntry)
            {
                eReturnCode = OSAL.eLinkedListRemove(
                    psBlock->ahEntries[un8Idx]);

                if (eReturnCode != OSAL_SUCCESS)
                {
                    // We should keep the block dangling if it exists
                    break;
                }
            }
        }

        if (eReturnCode == OSAL_SUCCESS)
        {
            // All the eLinkedListRemove calls worked, so free
            // this block of memory
            SMSO_vDestroy((SMS_OBJECT)psBlock);
        }
    }

    return;
}

/*****************************************************************************
*
*   psFindMovieEntry
*
*   ASSUMES psObj IS NOT NULL
*****************************************************************************/
static MOVIE_TIMES_ENTRY_STRUCT *psFindMovieEntry (
    THEATER_TIMES_OBJECT_STRUCT *psObj,
    BOOLEAN bInUse,
    MOVIE_ID tID,
    OSAL_LINKED_LIST_ENTRY *phEntry
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    MOVIE_TIMES_ENTRY_STRUCT sEntryToFind;
    MOVIE_TIMES_ENTRY_STRUCT *psFoundEntry = NULL;

    // Condition our inputs
    if (phEntry == NULL)
    {
        phEntry = &hEntry;
    }

    // Prepare search record
    sEntryToFind.bInUse = bInUse;
    sEntryToFind.tID = tID;

    eReturnCode = OSAL.eLinkedListSearch(
        psObj->hMovieList, phEntry, &sEntryToFind);

    if (eReturnCode == OSAL_SUCCESS)
    {
        psFoundEntry = (MOVIE_TIMES_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(*phEntry);
    }

    return psFoundEntry;
}

/*****************************************************************************
*
*   bResetMovieEntry
*
*****************************************************************************/
static BOOLEAN bResetMovieEntry (
    MOVIE_TIMES_ENTRY_STRUCT *psEntry,
    void *pvArg
        )
{
    if (psEntry != NULL)
    {
        psEntry->bInUse = FALSE;
        psEntry->tID = MOVIE_INVALID_ID;
        psEntry->un8NumShowTimes = 0;

        OSAL.bMemSet(&psEntry->aun16TimeOffsets[0],
            sizeof(psEntry->aun16TimeOffsets), 0);
    }

    // Keep going through the list
    return TRUE;
}

/*****************************************************************************
*
*   bIterateMovies
*
*****************************************************************************/
static BOOLEAN bIterateMovies (
    MOVIE_TIMES_ENTRY_STRUCT *psEntry,
    MOVIE_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = TRUE;

    if (psEntry != NULL && psIterator != NULL)
    {
        if (psEntry->bInUse == TRUE)
        {
            MOVIE_OBJECT hMovie;
            
            hMovie = MOVIES_MGR_hGetMovieForID(
                psIterator->hParent, psEntry->tID);

            if (hMovie != MOVIE_INVALID_OBJECT)
            {
                // Set our current movie entry so that the application
                // can iterate the times for it
                psIterator->psTheaterTimes->psCurMovieEntry = psEntry;
                psIterator->psTheaterTimes->hCurMovie = hMovie;

                // Do we have an exception for this movie's ratings?
                if (psEntry->bHaveException == TRUE)
                {
                    // Yes, set the exception now
                    MOVIE_vSetException(hMovie, 
                        psEntry->sException.eSystem,
                        psEntry->sException.eRating,
                        psEntry->sException.hRatingText);
                }

                // Call the private iterator function
                bContinue = psIterator->bIterator(
                    psIterator->hTheater, hMovie, psIterator->pvIteratorArg);
                
                // Clear the exception if necessary
                if (psEntry->bHaveException == TRUE)
                {
                    MOVIE_vClearException(hMovie);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    THEATER_TIMES_OBJECT_NAME
                    ": Unknown MVID(%u) found while iterating theater %u",
                    psEntry->tID, psIterator->psTheaterTimes->tID);
            }
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   bAddMovieTimesDataToBlob
*
*****************************************************************************/
static BOOLEAN bAddMovieTimesDataToBlob (
    MOVIE_TIMES_ENTRY_STRUCT *psEntry,
    CREATE_THEATER_TIMES_BLOB_ITERATOR_STRUCT *psIterator
        )
{
    UN8 un8Idx;

    // Ensure we have valid arguments
    if ((psEntry != NULL) && (psIterator != NULL))
    {
        // Only serialize entries which are in use
        if (psEntry->bInUse == TRUE)
        {
            // Size of the "header" for this entry
            size_t tHeaderSize = (
                THEATER_TIMES_BLOB_NUM_OVERHEAD_UN16 +
                psEntry->un8NumShowTimes
                 ) * sizeof(UN16);

            // Size of the data for this entry
            size_t tDataSize = psIterator->tNumUN16sWritten * sizeof(UN16);

            // Ensure we're within our bounds
            if ( (tHeaderSize + tDataSize) >= psIterator->tBlobDataSize)
            {
                // We wrote all we could, stop iterator and flag an error
                psIterator->bError = TRUE;
                return FALSE;
            }

            // Write the Movie ID as a UN16
            *psIterator->pun16BlobData++ = (UN16)psEntry->tID;

            // Write the rating exception flag as a UN16
            *psIterator->pun16BlobData++ = (UN16)psEntry->bHaveException;

            // Write the rating exception system as a UN16
            *psIterator->pun16BlobData++ = (UN16)psEntry->sException.tSysCode;
            
            // Write the rating exception as a UN16
            *psIterator->pun16BlobData++ = (UN16)psEntry->sException.tRatingCode;

            // Write the number of times as a UN16
            *psIterator->pun16BlobData++ = (UN16)psEntry->un8NumShowTimes;

            // loop through the available times and add them to the array
            for (un8Idx = 0; un8Idx < psEntry->un8NumShowTimes; un8Idx++)
            {
                *psIterator->pun16BlobData++ = psEntry->aun16TimeOffsets[un8Idx];
            }

            // Some UN16s for overhead, one UN16 for each time
            psIterator->tNumUN16sWritten += 
                THEATER_TIMES_BLOB_NUM_OVERHEAD_UN16 + psEntry->un8NumShowTimes;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   n16CompareMovieEntry
*
*****************************************************************************/
static N16 n16CompareMovieEntry (
    MOVIE_TIMES_ENTRY_STRUCT *psEntry1,
    MOVIE_TIMES_ENTRY_STRUCT *psEntry2
        )
{
    N16 n16Return = N16_MIN;

    if (psEntry1 != NULL && psEntry2 != NULL)
    {
        if (psEntry1->bInUse == TRUE && psEntry2->bInUse == TRUE)
        {
            // Both are in use, so return the difference between their
            // IDs
            n16Return = psEntry1->tID - psEntry2->tID;
        }
        else
        {
            if (psEntry1->bInUse == FALSE && psEntry2->bInUse == FALSE)
            {
                // Both are free, so return 0
                n16Return = 0;
            }
            else if(psEntry1->bInUse == FALSE)
            {
                // psEntry1 is less than (before) psEntry2
                n16Return = -1;
            }
            else
            {
                // psEntry1 is greater than (after) psEntry2
                n16Return = 1;
            }
        }
    }

    return n16Return;
}
