/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:THEATER implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "dsrl_entry_obj.h"
#include "location_obj.h"
#include "locid_obj.h"
#include "string_obj.h"
#include "movies_db_constants.h"
#include "movies_mgr_obj.h"
#include "_theater_obj.h"
#include "theater_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hLocation
*
*****************************************************************************/
static LOCATION_OBJECT hLocation (
    THEATER_OBJECT hTheater
        )
{
    BOOLEAN bOwner;
    LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
    if (bOwner == TRUE)
    {
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;

        hLocation = psObj->hLocation;
    }

    return hLocation;
}

/*****************************************************************************
*
*   un8NumMovies
*
*****************************************************************************/
static UN8 un8NumMovies (
    THEATER_OBJECT hTheater
        )
{
    BOOLEAN bOwner;
    UN8 un8NumMovies = 0;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
    if (bOwner == TRUE)
    {
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;

        if (psObj->hShowTimes != THEATER_TIMES_INVALID_OBJECT)
        {
            un8NumMovies = THEATER_TIMES_un8NumMovies(psObj->hShowTimes);
        }
    }

    return un8NumMovies;
}

/*****************************************************************************
*
*   eIterateMovies
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateMovies (
    THEATER_OBJECT hTheater,
    THEATER_MOVIES_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    do
    {
        BOOLEAN bOwner;
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;

        // Validate function pointer
        if ((THEATER_MOVIES_ITERATOR)NULL == bIterator)
        {
            // Error!
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Check ownership of the theater now
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
        if (FALSE == bOwner)
        {
            // Error!
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Do we have any movies to report?
        if (THEATER_TIMES_INVALID_OBJECT == psObj->hShowTimes)
        {
            // We have no show times, so indicate this
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
            break;
        }

        // Get the theater times object to iterate its movies for us
        eReturnCode = THEATER_TIMES_eIterateMovies(
            psObj->hShowTimes, hTheater, bIterator, pvIteratorArg);

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eIterateTimes
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateTimes (
    THEATER_OBJECT hTheater,
    THEATER_MOVIE_TIMES_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    do
    {
        BOOLEAN bOwner;
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;

        // Validate function pointer
        if ((THEATER_MOVIE_TIMES_ITERATOR)NULL == bIterator)
        {
            // Error!
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Check ownership of the theater now
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
        if (FALSE == bOwner)
        {
            // Error!
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Do we have any times to report?
        if (THEATER_TIMES_INVALID_OBJECT == psObj->hShowTimes)
        {
            // We have no show times, so indicate this
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
            break;
        }

        // Get the theater times object to iterate its times for us
        eReturnCode = THEATER_TIMES_eIterateTimes(
                psObj->hShowTimes, hTheater, bIterator, pvIteratorArg);

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAmenities
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAmenities (
    THEATER_OBJECT hTheater,
    size_t tNumAmenities,
    AMENITY_STRUCT *pasTheaterAmenities
        )
{
    BOOLEAN bOwner;

    if (tNumAmenities < THEATER_NUM_AMENITIES)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else if (pasTheaterAmenities == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Check ownership of the object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
    if (bOwner == FALSE)
    {
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    else
    {
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;
        UN8 un8AmenityOffset = 0;
        SMSAPI_RETURN_CODE_ENUM eReturnCode =
            SMSAPI_RETURN_CODE_SUCCESS;
        THEATER_AMENITY_ENUM eCurAmenity = THEATER_BASE_AMENITY;
        UN8 un8RawAmen;

        do
        {
            switch (eCurAmenity)
            {
                case THEATER_AMENITY_STADIUM_SEATING:
                {
                    un8RawAmen = (psObj->un16Amenities & THEATER_AMENITY_BITMASK);
                    if ((un8RawAmen == THEATER_AMENITY_AVAILABLE) ||
                        (un8RawAmen == THEATER_AMENITY_NOT_AVAILABLE))
                    {
                        pasTheaterAmenities[un8AmenityOffset].uAmenity.eTheaterAmenity =
                            THEATER_AMENITY_STADIUM_SEATING;
                        if (un8RawAmen == THEATER_AMENITY_AVAILABLE)
                        {
                            pasTheaterAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasTheaterAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case THEATER_AMENITY_STADIUM_ROCKER_SEATING:
                {
                    un8RawAmen = (psObj->un16Amenities & THEATER_AMENITY_BITMASK);
                    if ((un8RawAmen == THEATER_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == THEATER_AMENITY_NOT_AVAILABLE))
                    {
                        pasTheaterAmenities[un8AmenityOffset].uAmenity.eTheaterAmenity =
                            THEATER_AMENITY_STADIUM_ROCKER_SEATING;

                        if (un8RawAmen == THEATER_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasTheaterAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasTheaterAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }
                }
                break;

                default:
                {
                    // We can't identify these amenities
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                    break;
                }
            }
        } while (++eCurAmenity <= THEATER_MAX_AMENITY);

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (un8AmenityOffset == 0))
        {
            // We didn't find any amenities, let the caller know
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        return eReturnCode;
    }
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    THEATER_OBJECT hTheater,
    FILE *psFile
        )
{
    N32 n32Return = 0, n32Temp;
    BOOLEAN bOwner;
    THEATER_OBJECT_STRUCT *psObj =
        (THEATER_OBJECT_STRUCT *)hTheater;
    OSAL_FIXED_OBJECT hLat, hLon;
    LOCID_OBJECT hLocID;

    // Ensure we own this object and we have a valid file pointer
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
    if ( (bOwner == FALSE) || (psFile == NULL) )
    {
        return EOF;
    }

    hLocID = LOCATION.hLocID(psObj->hLocation);

    // Print the theater data
    n32Return += fprintf(psFile, "Theater Name: ");
    n32Temp = STRING.n32FWrite(LOCATION.hDescription(psObj->hLocation), psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n\tTheater Id (LOCID): %d\n",LOCID.tID(hLocID));
    n32Return += fprintf(psFile, "\n\tAddress: ");
    n32Temp = STRING.n32FWrite(LOCATION.hStreetName(psObj->hLocation), psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, ", ");
    n32Temp = STRING.n32FWrite(LOCATION.hCity(psObj->hLocation), psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, ", ");
    n32Temp = STRING.n32FWrite(LOCATION.hState(psObj->hLocation), psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, ", ");
    n32Temp = STRING.n32FWrite(LOCATION.hZipCode(psObj->hLocation), psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n\tPhone: ");
    n32Temp = STRING.n32FWrite(LOCATION.hPhone(psObj->hLocation), psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    hLat = LOCATION.hLat(psObj->hLocation);
    hLon = LOCATION.hLon(psObj->hLocation);

    n32Return += fprintf(psFile,
        "\n\tLat: Fixed Val: %10d Bits: %2u\n\tLon: Fixed Val: %9d Bits: %2u\n",
        OSAL_FIXED.n32Value(hLat),
        OSAL_FIXED.un8NumFractionalBits(hLat),
        OSAL_FIXED.n32Value(hLon),
        OSAL_FIXED.un8NumFractionalBits(hLon));

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   THEATER_hCreate
*
*   Create a new object based on the identifiers in the theater row.
*   Ownership of all string objects found in the theater row is transfered
*   to the newly created object.
*
*****************************************************************************/
THEATER_OBJECT THEATER_hCreate(
    SMS_OBJECT hOwner,
    THEATER_ROW_STRUCT *psTheaterRow,
    size_t tServiceDescriptorSize
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    THEATER_OBJECT_STRUCT *psObj;
    LOCATION_ATTRIBUTE_STRUCT sLocAttrs;
    BOOLEAN bOwner;

    // Verify ownership of parent object
    bOwner = SMSO_bOwner((SMS_OBJECT)hOwner);

    // Validate inputs
    if ((psTheaterRow == NULL) || (bOwner == FALSE))
    {
        return THEATER_INVALID_OBJECT;
    }

    OSAL.bMemSet(&sLocAttrs, 0, sizeof(sLocAttrs));

    // Generate a name for this theater
    snprintf( &acName[0], sizeof(acName),
              THEATER_OBJECT_NAME": id:%u",
              psTheaterRow->tID );

    // Create an instance of this object
    psObj = (THEATER_OBJECT_STRUCT *)
        DSRL_ENTRY_hCreate(
            &acName[0],
            DSRL_ENTRY_TYPE_THEATER,
            sizeof(THEATER_OBJECT_STRUCT),
            tServiceDescriptorSize,
            hOwner, FALSE);
    if(psObj == NULL)
    {
        // Error!
        return THEATER_INVALID_OBJECT;
    }

    do
    {
        LOCID_OBJECT hLocId;
        SMS_OBJECT hSMSTheater;
        N32 n32Lat, n32Lon;

        // Store amenities info
        psObj->un16Amenities = psTheaterRow->un16Amenities;

        // Get the underlying SMS Object for this theater
        hSMSTheater = DSRL_ENTRY_hGetSMSObject((DSRL_ENTRY_OBJECT)psObj);

        // Create a loc id for this theater
        // Just use the standard LocID Object
        hLocId = LOCID_hCreate(psTheaterRow->tID,
            LOCID_TYPE_THEATER,
            (const LOCID_INTERFACE_STRUCT *)NULL);

        if (hLocId == LOCID_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Populate the location attribute structure
        sLocAttrs.hDescription = psTheaterRow->hName;
        sLocAttrs.hStreetNum = STRING_INVALID_OBJECT;
        sLocAttrs.hCity = psTheaterRow->hCity;
        sLocAttrs.hPhone = psTheaterRow->hPhone;
        sLocAttrs.hState = LOCATION_hStateAbbrvForID(psTheaterRow->tStateID);
        sLocAttrs.hStreetName = psTheaterRow->hAddr;
        sLocAttrs.hZipCode = psTheaterRow->hZIP;

        n32Lat =
            OSAL_FIXED.n32ScaledValue(psTheaterRow->hLat, LOCATION_BINPOINT);

        n32Lon =
            OSAL_FIXED.n32ScaledValue(psTheaterRow->hLon, LOCATION_BINPOINT);

        // Create the location object
        psObj->hLocation = LOCATION_hCreate(
            hSMSTheater, hLocId,
            n32Lat, LOCATION_BINPOINT,
            n32Lon, LOCATION_BINPOINT,
            DISTANCE_INVALID_OBJECT,
            &sLocAttrs, FALSE );

        if (psObj->hLocation == LOCATION_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Provide the caller with the object
        return (THEATER_OBJECT)psObj;

    } while (FALSE);

    THEATER_vDestroy((THEATER_OBJECT)psObj);

    return THEATER_INVALID_OBJECT;
}

#if 0

// Leaving this code here for future reference
/*****************************************************************************
*
*   THEATER_bUpdateFields
*
*   Updates an existing object based on the identifiers in the theater row.
*   Ownership of all string objects found in the theater row is transfered
*   to the object.
*
*****************************************************************************/
BOOLEAN THEATER_bUpdateFields (
    THEATER_OBJECT hTheater,
    THEATER_ROW_STRUCT *psTheaterRow
        )
{
    BOOLEAN bOwner,
            bSuccess = FALSE;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
    if (bOwner == TRUE)
    {
        LOCATION_ATTRIBUTE_STRUCT sLocAttrs;
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;

        OSAL.bMemSet(&sLocAttrs, 0, sizeof(sLocAttrs));

        sLocAttrs.hCity = psTheaterRow->hCity;
        sLocAttrs.hDescription = psTheaterRow->hName;
        sLocAttrs.hPhone = psTheaterRow->hPhone;
        sLocAttrs.hState = LOCATION_hStateAbbrvForID(psTheaterRow->tStateID);
        sLocAttrs.hZipCode = psTheaterRow->hZIP;
        sLocAttrs.hStreetName = psTheaterRow->hAddr;

        bSuccess = LOCATION_bUpdateDescriptiveAttributes(
            psObj->hLocation, &sLocAttrs, FALSE, FALSE);

        if (bSuccess == TRUE)
        {
            if (psTheaterRow->hLat != OSAL_FIXED_INVALID_OBJECT &&
                psTheaterRow->hLon != OSAL_FIXED_INVALID_OBJECT)
            {
                N32 n32Lat, n32Lon;

                // Convert the FIXED objects to fixed-point numbers
                n32Lat =
                    OSAL_FIXED.n32ScaledValue(psTheaterRow->hLat, LOCATION_BINPOINT);

                n32Lon =
                    OSAL_FIXED.n32ScaledValue(psTheaterRow->hLon, LOCATION_BINPOINT);

                bSuccess = LOCATION_bUpdateCoordinates(
                    psObj->hLocation, n32Lat, LOCATION_BINPOINT,
                    n32Lon, LOCATION_BINPOINT);
            }
        }
    }

    return bSuccess;
}
#endif

/*****************************************************************************
*
*   THEATER_vDestroy
*
*****************************************************************************/
void THEATER_vDestroy (
    THEATER_OBJECT hTheater
        )
{
    BOOLEAN bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
    if (bOwner == TRUE)
    {
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;

        LOCATION.vDestroy(psObj->hLocation);

        // Destroy the object itself
        DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   THEATER_bSetShowTimes
*
*****************************************************************************/
BOOLEAN THEATER_bSetShowTimes (
    THEATER_OBJECT hTheater,
    THEATER_TIMES_OBJECT hShowTimes,
    BOOLEAN *pbChanged
        )
{
    BOOLEAN bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hTheater);
    if (bOwner == TRUE)
    {
        THEATER_OBJECT_STRUCT *psObj =
            (THEATER_OBJECT_STRUCT *)hTheater;

        // Is this new information?
        if (psObj->hShowTimes != hShowTimes)
        {
            if (pbChanged != NULL)
            {
                *pbChanged = TRUE;
            }

            psObj->hShowTimes = hShowTimes;
        }
    }

    return bOwner;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
