/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TEMPERATURE implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_temperature_obj.h"
#include "temperature_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eHigh
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eHigh (
    TEMPERATURE_OBJECT hTemperature,
    N16 *n16HighTemperature
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    TEMPERATURE_OBJECT_STRUCT *psObj = (TEMPERATURE_OBJECT_STRUCT*)hTemperature;

    do
    {
        if (n16HighTemperature == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hTemperature);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bHighAvailable != TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        //TODO: add check and conversion to Celsius
        *n16HighTemperature = psObj->n16High;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eLow
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eLow (
    TEMPERATURE_OBJECT hTemperature,
    N16 *n16LowTemperature
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    TEMPERATURE_OBJECT_STRUCT *psObj = (TEMPERATURE_OBJECT_STRUCT*)hTemperature;

    do
    {
        if (n16LowTemperature == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hTemperature);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bLowAvailable != TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        //TODO: add check and conversion to Celsius
        *n16LowTemperature = psObj->n16Low;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eCurrent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCurrent (
    TEMPERATURE_OBJECT hTemperature,
    N16 *n16CurrentTemperature
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    TEMPERATURE_OBJECT_STRUCT *psObj = (TEMPERATURE_OBJECT_STRUCT*)hTemperature;

    do
    {
        if (n16CurrentTemperature == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hTemperature);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bCurAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        //TODO: add check and conversion to Celsius
        *n16CurrentTemperature = psObj->n16Current;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    TEMPERATURE_OBJECT hTemperature,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    TEMPERATURE_OBJECT_STRUCT *psObj = (TEMPERATURE_OBJECT_STRUCT*)hTemperature;
    BOOLEAN bOwner;

    // Determine if we are the owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hTemperature);

    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print TEMPERATURE information header
    n32Return += fprintf(psFile, "\nTemperature (Object: 0x%X):",hTemperature);

    if (psObj->bHighAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\n\tHigh: %d ",
                             psObj->n16High);
    }

    if (psObj->bLowAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\n\tLow: %d ",
                             psObj->n16Low);
    }

    if (psObj->bCurAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\n\tCurrent: %d",
                             psObj->n16Current);
    }

    n32Return += fprintf(psFile, "\n");

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   TEMPERATURE_hCreate
*
*****************************************************************************/
TEMPERATURE_OBJECT TEMPERATURE_hCreate(
        SMS_OBJECT hParent,
        BOOLEAN bHighAvailable,
        N16 n16High,
        BOOLEAN bLowAvailable,
        N16 n16Low,
        BOOLEAN bCurAvailable,
        N16 n16Current
            )
{
    TEMPERATURE_OBJECT_STRUCT *psObj =
        (TEMPERATURE_OBJECT_STRUCT*)TEMPERATURE_INVALID_OBJECT;

    // Create an instance of the DISTANCE object
    psObj = (TEMPERATURE_OBJECT_STRUCT*)
        SMSO_hCreate(
            TEMPERATURE_OBJECT_NAME,
            sizeof(TEMPERATURE_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return TEMPERATURE_INVALID_OBJECT;
    }

    // Initialize object per inputs
    psObj->bHighAvailable = bHighAvailable;
    psObj->bLowAvailable = bLowAvailable;
    psObj->bCurAvailable = bCurAvailable;
    psObj->n16Current = n16Current;
    psObj->n16High = n16High;
    psObj->n16Low = n16Low;

    return (TEMPERATURE_OBJECT)psObj;
}

/*****************************************************************************
*
*   TEMPERATURE_vDestroy
*
*****************************************************************************/
void TEMPERATURE_vDestroy (
    TEMPERATURE_OBJECT hTemperature
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hTemperature);
    if(bOwner == TRUE)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hTemperature);
    }
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
