/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
*  This module contains the Object:TEAM implementation for the
*  Sirius Module Services (SMS)
*
******************************************************************************/
#include <stdio.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "sports.h"
#include "cid_obj.h"
#include "cid_integer.h"
#include "cdo_obj.h"
#include "league_obj.h"
#include "sql_interface_obj.h"
#include "db_util.h"

#include "team_obj.h"
#include "_team_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
PUBLIC FUNCTIONS
*****************************************************************************/

#ifdef SXM_LEGACY_SSP
/*****************************************************************************
*
*   hCreate
*
* Create a Team Object.
*
*****************************************************************************/
static TEAM_OBJECT hCreate(
    LEAGUE_ENUM eLeague,
    CID_OBJECT hTeamId,
    const char *pacAbbrev,
    const char *pacName,
    const char *pacNickname
    )
{
    TEAM_OBJECT hTeam = TEAM_INVALID_OBJECT;
    TEAM_STRUCT sTeam;
    LEAGUE_OBJECT hLeague;
    CID_OBJECT hCopyTeamId;

    // Check input names and id
    if ( (pacAbbrev == NULL) ||
        (pacName == NULL) ||
        (pacNickname == NULL) ||
        (hTeamId == CID_INVALID_OBJECT)
        )
    {
        // the caller did not provide valid names
        return TEAM_INVALID_OBJECT;
    }

    // Make a copy of the CID which was provided to us
    hCopyTeamId = CID.hDuplicate(hTeamId);
    if(hCopyTeamId != CID_INVALID_OBJECT)
    {
        // Add this team (copy what is provided)
        hTeam = TEAM_hAdd(
            hCopyTeamId, TEAM_INVALID_VERSION, 
            pacName, pacNickname, pacAbbrev, 0, NULL, 0, NULL);
        if(hTeam == TEAM_INVALID_OBJECT)
        {
            // Could not add new team, so destroy the id we created.
            CID.vDestroy(hCopyTeamId);
        }
    }

    return hTeam;
}
#endif

/*****************************************************************************
*
*   hName
*
* Retrieve the team name given the provided TEAM_OBJECT handle.
*
*****************************************************************************/
static STRING_OBJECT hName (
    TEAM_OBJECT hTeam
    )
{
    STRING_OBJECT hName = STRING_INVALID_OBJECT;
    TEAM_OBJECT_STRUCT const *psObj =
        (TEAM_OBJECT_STRUCT const *)hTeam;

    if(psObj != NULL)
    {
        hName = psObj->psTeam->hName;
    }

    return hName;
}

/*****************************************************************************
*
*   hNickname
*
* Retrieve the team nickname given the provided TEAM_OBJECT handle.
*
*****************************************************************************/
static STRING_OBJECT hNickname (
    TEAM_OBJECT hTeam
    )
{
    STRING_OBJECT hNickname = STRING_INVALID_OBJECT;
    TEAM_OBJECT_STRUCT const *psObj =
        (TEAM_OBJECT_STRUCT const *)hTeam;

    if(psObj != NULL)
    {
        hNickname = psObj->psTeam->hNickname;
    }

    return hNickname;
}

/*****************************************************************************
*
*   hAbbreviation
*
* Retrieve the team abbreviation given the provided TEAM_OBJECT handle.
*
*****************************************************************************/
static STRING_OBJECT hAbbreviation (
    TEAM_OBJECT hTeam
    )
{
    STRING_OBJECT hAbbreviation = STRING_INVALID_OBJECT;
    TEAM_OBJECT_STRUCT const *psObj =
        (TEAM_OBJECT_STRUCT const *)hTeam;

    if(psObj != NULL)
    {
        hAbbreviation = psObj->psTeam->hAbbreviation;
    }

    return hAbbreviation;
}

/*****************************************************************************
*
*   hId
*
* Retrieve the unique team CID given the provided TEAM_OBJECT handle.
*
*****************************************************************************/
static CID_OBJECT hId (
    TEAM_OBJECT hTeam
    )
{
    CID_OBJECT hCID = CID_INVALID_OBJECT;
    TEAM_OBJECT_STRUCT const *psObj =
        (TEAM_OBJECT_STRUCT const *)hTeam;

    if(psObj != NULL)
    {
        // Extract CID
        hCID = psObj->hId;
    }

    return hCID;
}

/*****************************************************************************
*
*   bIterateContent
*
* Iterates the available TEAMs within a provided LEAGUE. For each available
* TEAM the caller's provided callback and callback argument are called.
*
*****************************************************************************/
static BOOLEAN bIterateContent (
    LEAGUE_OBJECT hLeague,
    TEAM_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
    )
{
    BOOLEAN bSuccess;

    // Have the league object perform this action now
    bSuccess = LEAGUE_bIterateTeams(
        hLeague,
        TRUE,
        bContentIteratorCallback,
        pvContentIteratorCallbackArg);

    return bSuccess;
}

/*****************************************************************************
*
*   n32Version
*
*****************************************************************************/
static N32 n32Version (
    TEAM_OBJECT hTeam
    )
{
    N32 n32Version = TEAM_INVALID_VERSION;

    TEAM_OBJECT_STRUCT const *psObj =
        (TEAM_OBJECT_STRUCT const *)hTeam;

    if(psObj != NULL)
    {
        n32Version = psObj->psTeam->n32Version;
    }

    return n32Version;
}

/*****************************************************************************
*
*   bGetTableVersion
*
*****************************************************************************/
static BOOLEAN bGetTableVersion (
    N32 *pn32Version,
    BOOLEAN *pbComplete
    )
{
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bLocked;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsTeamCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if(pn32Version != NULL)
        {
            *pn32Version = gpsTeamCtrl->n32Version;
        }

        if(pbComplete != NULL)
        {
            *pbComplete = gpsTeamCtrl->bComplete;
        }

        bSuccess = TRUE;

        SMSO_vUnlock((SMS_OBJECT)gpsTeamCtrl);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   un16Id
*
*****************************************************************************/
static UN16 un16Id (
    TEAM_OBJECT hTeam
        )
{
    UN16 un16Id = UN16_MAX;

    TEAM_OBJECT_STRUCT const *psObj =
        (TEAM_OBJECT_STRUCT const *)hTeam;

    if(psObj != NULL)
    {
        UN32 un32Id = UN32_MAX;
        void *pvId = &un32Id;
        CID_n32GetValue(psObj->hId, &pvId);
        un16Id = (UN16)un32Id;
    }

    return un16Id;
}

/*****************************************************************************
*
*   eLeagueMembership
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eLeagueMembership (
    TEAM_OBJECT hTeam,
    LEAGUE_MEMBERSHIP_STRUCT *psLeagueMembership
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;

    TEAM_OBJECT_STRUCT const *psObj =
        (TEAM_OBJECT_STRUCT const *)hTeam;

    if( (psObj != NULL) && (psLeagueMembership != NULL) )
    {
        *psLeagueMembership = psObj->sLeagueMembership;
        eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturn;
}

/*****************************************************************************
*
*   bIsSportsFlashEligible
*
*****************************************************************************/
static BOOLEAN bIsSportsFlashEligible (
    LEAGUE_OBJECT hLeague,
    TEAM_OBJECT hTeam
        )
{
    BOOLEAN bEligible = FALSE;

    do
    {
        UN8 un8LeagueId;
        TEAM_OBJECT_STRUCT const *psTeamObj =
            (TEAM_OBJECT_STRUCT const *)hTeam;
        size_t tIndex;
        BOOLEAN bSportsFlashEnabled;

        if (psTeamObj == NULL)
        {
            break;
        }

        bSportsFlashEnabled = LEAGUE.bIsSportsFlashEnabled(hLeague);
        if (bSportsFlashEnabled == FALSE)
        {
            break;
        }

        un8LeagueId = LEAGUE.un8Id(hLeague);
        if (un8LeagueId == LEAGUE_INVALID_ID)
        {
            break;
        }

        // looking for league in our array
        for (tIndex = 0; tIndex < MAX_NUM_MEMBER_LEAGUES; ++tIndex)
        {
            if (psTeamObj->asLeagueTiers[tIndex].un8LeagueId == 
                un8LeagueId)
            {
                UN8 un8LeagueTiers;

                un8LeagueTiers = LEAGUE_un8SportsFlashTiers(hLeague);
                if (un8LeagueTiers >= 
                    psTeamObj->asLeagueTiers[tIndex].un8Tier)
                {
                    bEligible = TRUE;
                }

                break;
            }
        }

    } while (FALSE);

    return bEligible;
}

/*****************************************************************************
FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   TEAM_bInitialize
*
* Initialize the available TEAM list. This needs to be called once
* when SMS is started. The TEAM objects being initialized here can be
* used by multiple SMS objects.
*
* NOTE: This function assumes it will be called only once,
* and by a single context. Without first uninitializing it again.
*
*****************************************************************************/
BOOLEAN TEAM_bInitialize ( SMS_OBJECT hParent )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOk;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;

    // Initialize the known league list.
    // Creates an empty league list to be populated later.

    // Create an SMS object which can be locked to enforce exclusive
    // access to the elements within this structure.
    gpsTeamCtrl = (TEAM_CTRL_STRUCT *)
        SMSO_hCreate(
        TEAM_OBJECT_NAME":Ctrl",
        sizeof(TEAM_CTRL_STRUCT),
        hParent,
        TRUE // We want the lock feature (inherited)
        );
    if(gpsTeamCtrl == NULL)
    {
        // Error! Something went wrong.
        return FALSE;
    }

    do
    {
        // Create the team list of entries.
        eReturnCode =
            OSAL.eLinkedListCreate(
            &gpsTeamCtrl->hTeamList,
            TEAM_OBJECT_NAME":List",
            (OSAL_LL_COMPARE_HANDLER)n16Compare,
            OSAL_LL_OPTION_UNIQUE | 
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
            OSAL_LL_OPTION_RBTREE
            );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create the team trash bin.
        eReturnCode =
            OSAL.eLinkedListCreate(
            &gpsTeamCtrl->hTeamTrashBin,
            TEAM_OBJECT_NAME":TrashBin",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR |
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
            );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize CID objects
        gpsTeamCtrl->hWorkingLeagueId = CID_INVALID_OBJECT;
        gpsTeamCtrl->hWorkingTeamId = CID_INVALID_OBJECT;

        // Initialize last processed version
        gpsTeamCtrl->n32Version = TEAM_INVALID_VERSION;
        gpsTeamCtrl->bComplete = FALSE;

        // Initialize dummy object
        gpsTeamCtrl->sDummyTeam.bSaved = FALSE;
        gpsTeamCtrl->sDummyTeam.hId = CID_INVALID_OBJECT;
        gpsTeamCtrl->sDummyTeam.psTeam = NULL;
        OSAL.bMemSet(
            &gpsTeamCtrl->sDummyTeam.sLeagueMembership, 0, 
            sizeof(gpsTeamCtrl->sDummyTeam.sLeagueMembership));

        // Initialize SQL
        gpsTeamCtrl->pacFileName = NULL;

        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            SPORTS_DB_FOLDER,
            SPORTS_DB_FILENAME,
            &gpsTeamCtrl->pacFileName);

        if (bOk == TRUE)
        {
            gpsTeamCtrl->hSQLConnection =
                DB_UTIL_hConnectToReference(
                &gpsTeamCtrl->pacFileName[0],
                bVerifyDBVersion,
                (void *)NULL,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK
                );

            if (gpsTeamCtrl->hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
            {
                if (eErrorCode == DATASERVICE_ERROR_CODE_DATABASE_READONLY)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TEAM_OBJECT_NAME": database is read-only '%s'.",
                        &gpsTeamCtrl->pacFileName[0]);
                }
                else if (eErrorCode == DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TEAM_OBJECT_NAME": version mismatch with database '%s'.",
                        &gpsTeamCtrl->pacFileName[0]);
                }
                else if (eErrorCode == DATASERVICE_ERROR_CODE_DATABASE_CORRUPT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TEAM_OBJECT_NAME": corrupt database '%s'.",
                        &gpsTeamCtrl->pacFileName[0]);
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TEAM_OBJECT_NAME" Could not open connection to the "
                        "database '%s' used to identify sports leagues and teams.",
                        &gpsTeamCtrl->pacFileName[0]);
                }

                // continue init even if SQL connection fails. We can run without
                // db - we just dont get any persisted info.
            }
        }

        // Relinquish control of the object if we are the parent.
        if(hParent == SMS_INVALID_OBJECT)
        {
            SMSO_vUnlock((SMS_OBJECT)gpsTeamCtrl);
        }

        return TRUE;

    } while (FALSE);

    TEAM_vUnInitialize();

    return FALSE;
}

/*****************************************************************************
*
*   TEAM_vUnInitialize
*
* Uninitialize the TEAM list (upon SMS shutdown).
*
* NOTE: This function assumes it will be called only once after
* the object has been initialized, and by a single context.
*
*****************************************************************************/
void TEAM_vUnInitialize ( void )
{
    BOOLEAN bLocked;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsTeamCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error! Something went wrong or no object created in the first place.
        return;
    }

    // Save anything that needs updated
    TEAM_bSave(TEAM_INVALID_VERSION);

    if (gpsTeamCtrl->hSQLConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        // close the connection to the db
        SQL_INTERFACE.vDisconnect(gpsTeamCtrl->hSQLConnection);
    }

    // Check if team list exists.
    if(gpsTeamCtrl->hTeamList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy all team map entries first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            gpsTeamCtrl->hTeamList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyTeamObject
            );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy the list itself
            eReturnCode = OSAL.eLinkedListDelete(gpsTeamCtrl->hTeamList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gpsTeamCtrl->hTeamList = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Check if team trash bin exists.
    if(gpsTeamCtrl->hTeamTrashBin != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy all league entries first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            gpsTeamCtrl->hTeamTrashBin,
            (OSAL_LL_RELEASE_HANDLER)vDestroyTeamObject
            );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy the list itself
            eReturnCode = OSAL.eLinkedListDelete(gpsTeamCtrl->hTeamTrashBin);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gpsTeamCtrl->hTeamTrashBin = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    if (gpsTeamCtrl->pacFileName != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)gpsTeamCtrl->pacFileName);
        gpsTeamCtrl->pacFileName = NULL;
    }

    if(gpsTeamCtrl->hWorkingLeagueId != CID_INVALID_OBJECT)
    {
        // destroy working CID
        CID_vDestroy(gpsTeamCtrl->hWorkingLeagueId);
        gpsTeamCtrl->hWorkingLeagueId = CID_INVALID_OBJECT;
    }

    if(gpsTeamCtrl->hWorkingTeamId != CID_INVALID_OBJECT)
    {
        // destroy working CID
        CID_vDestroy(gpsTeamCtrl->hWorkingTeamId);
        gpsTeamCtrl->hWorkingTeamId = CID_INVALID_OBJECT;
    }

    // Unlock object
    SMSO_vUnlock((SMS_OBJECT)gpsTeamCtrl);

    // Destroy object itself
    SMSO_vDestroy((SMS_OBJECT)gpsTeamCtrl);

    gpsTeamCtrl = NULL;

    return;
}

/*****************************************************************************
*
*   TEAM_hAdd
*
*   This function assigns a CID to represent this team content. The CID
*   is used to uniquely describe this team and map it to some
*   textual information about the team (e.g. name, abbr, etc.).
*
*   Inputs
*       hTeamId - A CID representing this TEAM
*       n32Version -
*       pacName -
*       pacNickname -
*       pacAbbrev -
*
*       Outputs:
*       BOOLEAN - A value of TRUE if the CID was able to be added.
*       Otherwise returns FALSE if the request resulted if an error occurred.
*
*****************************************************************************/
TEAM_OBJECT TEAM_hAdd (
    CID_OBJECT hTeamId,
    N32 n32Version,
    char const *pacName,
    char const *pacNickname,
    char const *pacAbbrev,
    size_t tNumLeagues,
    UN8 const *paun8Leagues,
    size_t tNumTiers,
    UN8 const *paun8Tiers
        )
{
    BOOLEAN bLocked;
    TEAM_OBJECT_STRUCT *psNewTeamObject;
    BOOLEAN bUpdateLeagues = FALSE;

    // Check inputs
    if((hTeamId == CID_INVALID_OBJECT) || (pacName == NULL) ||
        (pacNickname == NULL) || (pacAbbrev == NULL))
    {
        // Error!
        return TEAM_INVALID_OBJECT;
    }

    // Invent a team entry
    psNewTeamObject = psCreateTeamObject(
        hTeamId, n32Version, pacName, 
        pacNickname, pacAbbrev, tNumLeagues, paun8Leagues);
    if (psNewTeamObject == NULL)
    {
        // Error! Unable to create
        return TEAM_INVALID_OBJECT;
    }

    // Finally, add this team to our list.
    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsTeamCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hExistingTeamObject;

        do
        {
            hExistingTeamObject = OSAL_INVALID_LINKED_LIST_ENTRY;

            // It is possible, that the current version contains
            // several entries with the same ID, but these teams
            // are in different leagues. The next table version may
            // contain only one team entry with the same ID, but
            // this team participates in all leagues, in which multiple
            // teams participates in the current table. This means,
            // that one new entry may replace multiple old entries.

            eReturnCode =
                OSAL.eLinkedListAdd(
                    gpsTeamCtrl->hTeamList,
                    &hExistingTeamObject,
                    psNewTeamObject);

            if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                TEAM_OBJECT_STRUCT *psExistingTeamObject;

                // We already found one in our list that looks just like
                // this one. We cannot enter duplicate items in this list.

                // So we need to remove this element, and add the new
                // one as a way to update it, but removed entry needs
                // to stay persistent so we put it into the trash bin.

                // This is the existing team entry we found
                psExistingTeamObject = (TEAM_OBJECT_STRUCT *)
                    OSAL.pvLinkedListThis(hExistingTeamObject);

                // Did the version change?
                if(psExistingTeamObject->psTeam->n32Version !=
                    psNewTeamObject->psTeam->n32Version)
                {
                    BOOLEAN bSuccess;

                    // Remove this entry
                    OSAL.eLinkedListRemove(hExistingTeamObject);

                    // Move it to the trash bin
                    OSAL.eLinkedListAdd(
                        gpsTeamCtrl->hTeamTrashBin, 
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, 
                        psExistingTeamObject
                            );

                    bSuccess = TEAM_bIterateLeagues(
                        (TEAM_OBJECT)psExistingTeamObject,
                        (LEAGUE_CONTENT_ITERATOR_CALLBACK) LEAGUE_bRemoveTeam,
                        psExistingTeamObject);

                    if (bSuccess == FALSE)
                    {
                        // Error!
                        vDestroyTeamObject(psNewTeamObject);
                        psNewTeamObject = NULL;
                        break;
                    }

                    // replaced successfully
                    SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
                        "Replaced team with %s %s (v=%d).\n", 
                        STRING.pacCStr(psNewTeamObject->psTeam->hName),
                        STRING.pacCStr(psNewTeamObject->psTeam->hNickname),
                        psExistingTeamObject->psTeam->n32Version
                            );

                    // We should update our league membership
                    bUpdateLeagues = TRUE;
                }
                else // Don't add it, just use it
                {
                    // Get rid of the new one we created, and just
                    // use the existing one.
                    vDestroyTeamObject(psNewTeamObject);
                    psNewTeamObject = psExistingTeamObject;
                    break;
                }
            }
            else if(eReturnCode != OSAL_SUCCESS)
            {
                // Something went wrong with adding this element. No point
                // in keeping the structure we just created.
                // Now we destroy the entry we created but leave
                // the provided CID alone.
                vDestroyTeamObject(psNewTeamObject);
                psNewTeamObject = NULL;
                break;
            }
            else
            {
                // added successfully
                SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
                    "Added team %s %s (v=%d).\n", 
                    STRING.pacCStr(psNewTeamObject->psTeam->hName),
                    STRING.pacCStr(psNewTeamObject->psTeam->hNickname),
                    n32Version
                        );

                // We should update our league membership
                bUpdateLeagues = TRUE;
            }

        } while (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsTeamCtrl);
    }
    else
    {
        // Now we destroy the entry we created but leave
        // the provided CID alone.
        vDestroyTeamObject(psNewTeamObject);
        psNewTeamObject = NULL;
    }

    // If we added or replaced an entry, update our league membership
    if ((TRUE == bUpdateLeagues) &&
        (NULL != psNewTeamObject))
    {
        UN32 un32LeagueId;
        LEAGUE_OBJECT hLeague;
        size_t tIndex;

        // Still need to see if we need to create some leagues.
        // It is possible, that the Team update with a new league
        // number will occur before the actual update of this league. 
        
        // The following code will create this league based on ID 
        // from the league membership array.
        // So when the actual league update will occur, it will
        // set the actual league properties.

        // For each league, try to add this team to it
        for (tIndex = 0; tIndex < MAX_NUM_MEMBER_LEAGUES; tIndex++)
        {
            if (tIndex < tNumLeagues)
            {
                un32LeagueId = paun8Leagues[tIndex];
                psNewTeamObject->asLeagueTiers[tIndex].un8LeagueId = 
                    paun8Leagues[tIndex];

                if (tIndex < tNumTiers)
                {
                    psNewTeamObject->asLeagueTiers[tIndex].un8Tier = 
                        paun8Tiers[tIndex];
                }
                else
                {
                    // by default tier value is 0
                    psNewTeamObject->asLeagueTiers[tIndex].un8Tier = 0;
                }

                // Find and add to league
                CID_bModify(&gpsTeamCtrl->hWorkingLeagueId, &un32LeagueId);

                // Find league
                hLeague = LEAGUE_hFind(gpsTeamCtrl->hWorkingLeagueId, TRUE);
                if (hLeague != LEAGUE_INVALID_OBJECT)
                {
                    // Add to league
                    LEAGUE_bAddTeam(hLeague, (TEAM_OBJECT)psNewTeamObject);
                }
            }
            else
            {
                psNewTeamObject->asLeagueTiers[tIndex].un8LeagueId = 
                    LEAGUE_INVALID_ID;
            }
        }
    }

    return (TEAM_OBJECT)psNewTeamObject;
}

/*****************************************************************************
*
*   TEAM_bAddLeague
*
*   This function assigns a league to represent this team content. The league
*   is used to uniquely describe this team and map it to some
*   textual information about the league (e.g. name, abbr, etc.).
*
*   Inputs
*       hTeam - A team object to add this league to
*       hLeague - A league object to add
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the league was able to be added.
*       Otherwise returns FALSE if the request resulted if an error occurred.
*
*****************************************************************************/
BOOLEAN TEAM_bAddLeague (
    TEAM_OBJECT hTeam,
    LEAGUE_OBJECT hLeague
    )
{
    BOOLEAN bSuccess = FALSE;
    CID_OBJECT hLeagueId;
    TEAM_OBJECT_STRUCT *psTeamObject =
        (TEAM_OBJECT_STRUCT *)hTeam;
    UN32 un32LeagueId;
    void *pvLeagueId = &un32LeagueId;

    // Verify a TEAM and LEAGUE was provided.
    // Without both these is no point in calling this.
    if((hLeague == LEAGUE_INVALID_OBJECT) || 
        (hTeam == TEAM_INVALID_OBJECT))
    {
        // Error!
        return FALSE;
    }

    // Extract league id
    hLeagueId = LEAGUE.hId(hLeague);

    // Find value for given league
    CID_n32GetValue(hLeagueId, &pvLeagueId);

    bSuccess = bSetLeagueMembershipBit(
        &psTeamObject->sLeagueMembership, un32LeagueId);
    if(bSuccess == TRUE)
    {
        // added successfully
        SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
            "Added league %s (%s) to the team %s %s.\n", 
            STRING.pacCStr(LEAGUE.hName(hLeague)),
            STRING.pacCStr(LEAGUE.hAbbreviation(hLeague)),
            STRING.pacCStr(TEAM.hName(hTeam)),
            STRING.pacCStr(TEAM.hNickname(hTeam)));
    }

    return bSuccess;
}

/*****************************************************************************
*
* TEAM_un32Count
*
*****************************************************************************/
UN32 TEAM_un32Count (
    N32 n32Version, 
    UN32 *pun32Total, 
    UN32 un32Expected
    )
{
    TEAM_VERSION_MATCH_ITERATOR_STRUCT sMatching;
    UN32 un32Total;

    sMatching.n32Version = n32Version;
    sMatching.un32Matching = 0;

    if(pun32Total == NULL)
    {
        // Use local
        pun32Total = &un32Total;
    }

    // Note: There is no need for exclusive access since the only
    // context able to update the list, is the same which wants the count.
    OSAL.eLinkedListIterate(
        gpsTeamCtrl->hTeamList,
        (OSAL_LL_ITERATOR_HANDLER)bCountMatchingVersion, &sMatching);

    *pun32Total = 0;
    OSAL.eLinkedListItems(gpsTeamCtrl->hTeamList, pun32Total);

    // Check if the number expected matches the number found.
    if(sMatching.un32Matching == un32Expected)
    {
        gpsTeamCtrl->bComplete = TRUE;
    }
    else
    {
        gpsTeamCtrl->bComplete = FALSE;
    }

    gpsTeamCtrl->n32Version = n32Version;

    return sMatching.un32Matching;
}

/*****************************************************************************
*
*   TEAM_bLoad
*
*  This is a function used to load saved teams from the db.
*
*****************************************************************************/
BOOLEAN TEAM_bLoad ( 
    CID_ENUM eType
    )
{
    BOOLEAN bReturn;
    size_t tTeamsLoaded = 0;
    UN32 un32Seconds;
    UN16 un16Msecs;
    UN32 un32LoadStartMsec, un32LoadEndMsec;

    // set the methods that will be used to create league cids and team cids
    gpsTeamCtrl->eType = eType;

    // Create a working league id
    if(gpsTeamCtrl->hWorkingLeagueId == CID_INVALID_OBJECT)
    {
        UN32 un32LeagueId = 0;

        gpsTeamCtrl->hWorkingLeagueId = 
            LEAGUE_hCreateCid(CID_POOL_INVALID_OBJECT, &un32LeagueId);
        if(gpsTeamCtrl->hWorkingLeagueId == CID_INVALID_OBJECT)
        {
            // Error!
            return FALSE;
        }
    }

    // Create a working team id
    if(gpsTeamCtrl->hWorkingTeamId == CID_INVALID_OBJECT)
    {
        UN32 un32TeamId = 0;

        gpsTeamCtrl->hWorkingTeamId = 
            TEAM_hCreateCid(CID_POOL_INVALID_OBJECT, &un32TeamId);
        if(gpsTeamCtrl->hWorkingTeamId == CID_INVALID_OBJECT)
        {
            // Error!
            return FALSE;
        }
    }

    OSAL.vTimeUp(&un32Seconds, &un16Msecs);
    un32LoadStartMsec = ((un32Seconds * 1000) + un16Msecs);

    bReturn = SQL_INTERFACE.bStartTransaction(gpsTeamCtrl->hSQLConnection);
    if (bReturn == TRUE)
    {
        // get all teams. TEAMs created in bTeamTableHandler
        bReturn = SQL_INTERFACE.bQuery (
            gpsTeamCtrl->hSQLConnection,
            QUERY_FOR_ALL_TEAMS,
            bTeamTableHandler,
            &tTeamsLoaded );

        SQL_INTERFACE.bEndTransaction(gpsTeamCtrl->hSQLConnection);
    }

    OSAL.vTimeUp(&un32Seconds, &un16Msecs);
    un32LoadEndMsec = ((un32Seconds * 1000) + un16Msecs);

    SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
        PC_BLUE": Loaded %u teams (%u msec).\n"PC_RESET,
        tTeamsLoaded, un32LoadEndMsec - un32LoadStartMsec);

    return bReturn;
}

/*****************************************************************************
*
*   TEAM_n16CompareIds
*
*       This is the function used to compare an existing sports TEAM objects
*       to each other. Specifically this function is used to perform a
*       relational comparison for sorting. This sorter keeps the TEAMs in order
*       based on each TEAM's id.
*
*       Outputs:
*               0   - TEAMs have the same value (equal)
*               > 0 - TEAM1 is greater than (after) TEAM2
*               < 0 - TEAM1 is less than (before) TEAM2 or error
*
*****************************************************************************/
N16 TEAM_n16CompareIds (
    TEAM_OBJECT hTeam1,
    TEAM_OBJECT hTeam2
        )
{
    N16 n16Result = N16_MIN;
    TEAM_OBJECT_STRUCT *psTeam1 = (TEAM_OBJECT_STRUCT *)hTeam1;
    TEAM_OBJECT_STRUCT *psTeam2 = (TEAM_OBJECT_STRUCT *)hTeam2;

    // Verify inputs
    if( (psTeam1 != NULL) && (psTeam2 != NULL) )
    {
        // Check CID relationship
        n16Result = CID.n16Compare(psTeam1->hId, psTeam2->hId);
    }

    return n16Result;
}

/*****************************************************************************
*
* TEAM_bSave
*
*****************************************************************************/
BOOLEAN TEAM_bSave ( N32 n32Version )
{
    BOOLEAN bResult;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    TEAM_SAVE_ITERATOR_STRUCT sTeamSaveIterator;

    if (gpsTeamCtrl->hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        // No error. Means, no file system.
        return TRUE;
    }

    bResult = SQL_INTERFACE.bStartTransaction(gpsTeamCtrl->hSQLConnection);
    if (bResult == TRUE)
    {
        // Get rid of any entries which are not this one.
        bResult = bPrune(n32Version);

        // Iterate leagues and save them
        sTeamSaveIterator.n32Version = n32Version;
        sTeamSaveIterator.tInsertCount = 0;
        sTeamSaveIterator.tReplacedCount = 0;
        sTeamSaveIterator.tAlreadyUpToDateCount = 0;

        // Iterate teams and save them

        // Note: There is no need for exclusive access since the only
        // context able to update the list, is the same which wants to save it.
        eReturnCode = OSAL.eLinkedListIterate(
            gpsTeamCtrl->hTeamList,
            (OSAL_LL_ITERATOR_HANDLER)bSaveTeam, &sTeamSaveIterator);
        if((eReturnCode == OSAL_SUCCESS) || (eReturnCode == OSAL_NO_OBJECTS))
        {
            BOOLEAN bSuccess;

            // All saved
            SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
                "%u teams saved with version v%d.\n"
                "%u inserted, %u replaced, %u already up-to-date.\n",
                sTeamSaveIterator.tInsertCount + sTeamSaveIterator.tReplacedCount + 
                sTeamSaveIterator.tAlreadyUpToDateCount,
                n32Version,
                sTeamSaveIterator.tInsertCount,
                sTeamSaveIterator.tReplacedCount,
                sTeamSaveIterator.tAlreadyUpToDateCount);

            // Write version info

            // Build the version string
            snprintf(&gpsTeamCtrl->acQueryBuffer[0],
                sizeof(gpsTeamCtrl->acQueryBuffer),
                UPDATE_VERSION,
                gpsTeamCtrl->n32Version, gpsTeamCtrl->bComplete
                );

            bSuccess = SQL_INTERFACE.bExecuteCommand(
                gpsTeamCtrl->hSQLConnection,
                &gpsTeamCtrl->acQueryBuffer[0]);

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TEAM_OBJECT_NAME
                    ": failed to set team table version");
            }
            else
            {
                bResult = TRUE;
            }
        }

        SQL_INTERFACE.bEndTransaction(gpsTeamCtrl->hSQLConnection);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TEAM_OBJECT_NAME
            ": failed to start transaction to recreate persistant storage");
    }

    return bResult;
}

/*****************************************************************************
*
* TEAM_hCreateCid
*
*****************************************************************************/
CID_OBJECT TEAM_hCreateCid(
    CID_POOL hCidPool,
    const void *pvSrcObjectData
    )
{
    CID_OBJECT hTeam;

    hTeam = 
        CID_hCreate(
        hCidPool, gpsTeamCtrl->eType, pvSrcObjectData);

    return hTeam;
}


/*****************************************************************************
*
*   TEAM_bCreateDBTables
*
*****************************************************************************/
BOOLEAN TEAM_bCreateDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
    )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TEAM_OBJECT_NAME
                ": failed to create teams version table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            INSERT_VERSION_ROW);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TEAM_OBJECT_NAME
                ": failed to set initial db team version");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            CREATE_TEAMS_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TEAM_OBJECT_NAME
                ": failed to create teams table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            CREATE_TEAMS_INDEX);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TEAM_OBJECT_NAME
                ": failed to create teams table index");
            break;
        }
    }
    while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   TEAM_hGetDummyObj
*
*****************************************************************************/
TEAM_OBJECT TEAM_hGetDummyObj (
    CID_OBJECT hTeamId
    )
{
    TEAM_OBJECT hTeam = TEAM_INVALID_OBJECT;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)gpsTeamCtrl);
    if(bOwner == TRUE)
    {
        // Load in object
        gpsTeamCtrl->sDummyTeam.hId = hTeamId;
        hTeam = (TEAM_OBJECT)&gpsTeamCtrl->sDummyTeam;
    }

    return hTeam;
}

/*****************************************************************************
*
*   TEAM_bIterateLeagues
*
*****************************************************************************/
BOOLEAN TEAM_bIterateLeagues(
    TEAM_OBJECT hTeam,
    LEAGUE_CONTENT_ITERATOR_CALLBACK bIteratorCallback,
    void *pvArg
        )
{
    BOOLEAN bLocked, bContinue;
    UN8 un8Element, un8League;

    LEAGUE_OBJECT hLeague;
    UN32 un32LeagueId;

    TEAM_OBJECT_STRUCT *psTeam;

    // Verify inputs
    if ((hTeam == TEAM_INVALID_OBJECT) ||
        (bIteratorCallback == NULL))
    {
        // Error!
        return FALSE;
    }

    // De-reference object
    psTeam = (TEAM_OBJECT_STRUCT *)hTeam;

    bLocked = SMSO_bLock((SMS_OBJECT)gpsTeamCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Iterate through all leagues the team participates in
    for (un8Element = 0; 
         un8Element < NUMBER_OF_LEAGUE_ELEMENTS; 
         un8Element++)
    {
        // Is there any league in this element?
        if (psTeam->sLeagueMembership.un8Mask[un8Element] != 0)
        {
            // If so, process this element leagues
            for (un8League = 0; un8League < UN8_BITLEN; un8League++)
            {
                // Check if this league number is present
                if (psTeam->sLeagueMembership.un8Mask[un8Element] & 
                    (1 << un8League))
                {
                    // Get an absolute League Id
                    un32LeagueId = 
                        (UN32)un8League + (UN8_BITLEN * un8Element);

                    // Modify a working CID
                    CID_bModify(
                        &gpsTeamCtrl->hWorkingLeagueId, &un32LeagueId);

                    // Find the league object by using its League Id
                    hLeague = LEAGUE_hFind(
                        gpsTeamCtrl->hWorkingLeagueId, TRUE);

                    if (hLeague != LEAGUE_INVALID_OBJECT)
                    {
                        // Now call the callback
                        bContinue = bIteratorCallback(hLeague, pvArg);
                        if (bContinue == FALSE)
                        {
                            // They want us to stop
                            SMSO_vUnlock((SMS_OBJECT)gpsTeamCtrl);
                            return TRUE;
                        }
                    }
                    else
                    {
                        // Error!
                        SMSAPI_DEBUG_vPrintErrorFull(
                            gpacThisFile, __LINE__,
                            TEAM_OBJECT_NAME
                            ": Failed to search League: %s",
                            STRING.pacCStr(LEAGUE.hAbbreviation(hLeague))
                                );

                        SMSO_vUnlock((SMS_OBJECT)gpsTeamCtrl);
                        return FALSE;
                    }
                }
            }
        }
    }

    SMSO_vUnlock((SMS_OBJECT)gpsTeamCtrl);
    return TRUE;
}

/*****************************************************************************
*
*   TEAM_bIterateContent
*
*****************************************************************************/
BOOLEAN TEAM_bIterateContent (
    LEAGUE_OBJECT hLeague,
    TEAM_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
        )
{
    BOOLEAN bSuccess;

    // Have the league object perform this action now
    bSuccess = LEAGUE_bIterateTeams(
        hLeague,
        FALSE,
        bContentIteratorCallback,
        pvContentIteratorCallbackArg);

    return bSuccess;
}

/*****************************************************************************
PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bCountMatchingVersion
*
*****************************************************************************/
static BOOLEAN bCountMatchingVersion (
    TEAM_OBJECT_STRUCT const *psTeamObject,
    TEAM_VERSION_MATCH_ITERATOR_STRUCT *psMatching
    )
{
    // Check inputs
    if((psTeamObject != NULL) && (psMatching != NULL))
    {
        if(psTeamObject->psTeam->n32Version == psMatching->n32Version)
        {
            psMatching->un32Matching++;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bVerifyDBVersion
*
*****************************************************************************/
static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
    )
{
    BOOLEAN bOk, bVersionMatched = FALSE;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        SPORTS_DB_SELECT_SCHEMA_VERSION,
        bProcessSelectDBVersionResult,
        &bVersionMatched
        );

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TEAM_OBJECT_NAME
            ": failed to get version row from table");
        return FALSE;
    }

    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        SELECT_VERSION,
        bProcessSelectTableVersionResult,
        NULL
        );

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TEAM_OBJECT_NAME
            ": failed to get table version row from table");
        return FALSE;
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   vProcessSelectDBVersionResult
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    BOOLEAN *pbVersionMatched = (BOOLEAN *)pvArg;
    UN8 un8Version = 0;

    if (pbVersionMatched == NULL)
    {
        return FALSE;
    }

    // Init our return value
    *pbVersionMatched = FALSE;

    if ( (n32NumberOfColumns == 1) &&
        (psColumns[0].eType == SQL_COLUMN_TYPE_INTEGER) )
    {
        // There is only one column, and it should be a UN32
        un8Version =
            (UN8)psColumns[0].uData.sUN32.un32Data;

        if (un8Version == SPORTS_DB_SCHEMA_VER)
        {
            *pbVersionMatched = TRUE;
        }
    }

    // Return false since we don't need to iterate through the table
    // should only be one row
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectTableVersionResult
*
*******************************************************************************/
static BOOLEAN bProcessSelectTableVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    gpsTeamCtrl->n32Version = TEAM_INVALID_VERSION;
    gpsTeamCtrl->bComplete = FALSE;

    if (n32NumberOfColumns == 2)
    {
        gpsTeamCtrl->n32Version = (N32)psColumns[0].uData.sUN32.un32Data;
        gpsTeamCtrl->bComplete = (BOOLEAN)psColumns[1].uData.sUN32.un32Data;
    }

    // Return false since we don't need to iterate through the table
    // should only be one row
    return FALSE;
}

/*****************************************************************************
*
* bQueryTeamVersion
*
*****************************************************************************/
static BOOLEAN bQueryTeamVersion (
    UN32 un32TeamId,
    TEAM_LEAGUE_TIER_STRUCT *psLeagueTier,
    N32 *pn32Version,
    N64 *pn64RowId
    )
{
    TEAM_VERSION_QUERY_STRUCT sQuery;

    // prepare the SQL statement
    snprintf(&gpsTeamCtrl->acQueryBuffer[0], TEAM_MAX_QUERY_LENGTH,
        QUERY_TEAM_VERSION, un32TeamId);

    // prepare the query struct
    OSAL.bMemCpy(sQuery.asLeagueTiers, psLeagueTier, 
        sizeof(sQuery.asLeagueTiers));
    sQuery.bFound = FALSE;

    // run the query
    SQL_INTERFACE.bQuery(
        gpsTeamCtrl->hSQLConnection,
        &gpsTeamCtrl->acQueryBuffer[0],
        bTeamVersionHandler,
        (void *)&sQuery);

    if (sQuery.bFound == TRUE)
    {
        OSAL.bMemCpy(psLeagueTier, sQuery.asLeagueTiers, 
            sizeof(sQuery.asLeagueTiers));
        *pn32Version = sQuery.n32Version;
        *pn64RowId = sQuery.n64RowId;
    }

    return sQuery.bFound;
}

/*****************************************************************************
*
* bTeamVersionHandler
*
*****************************************************************************/
static BOOLEAN bTeamVersionHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    int iLeagueCmp;

    TEAM_VERSION_QUERY_STRUCT *psQuery =
        (TEAM_VERSION_QUERY_STRUCT *)pvArg;

    iLeagueCmp = memcmp(psQuery->asLeagueTiers,
        psColumns[0].uData.sBLOB.pvData, 
        sizeof(psQuery->asLeagueTiers)
            );
    if (iLeagueCmp == 0)
    {
        OSAL.bMemCpy(&psQuery->asLeagueTiers, 
            psColumns[0].uData.sBLOB.pvData, 
            sizeof(psQuery->asLeagueTiers));
        psQuery->n32Version =
            (N32)psColumns[1].uData.sUN32.un32Data;
        psQuery->n64RowId =
            psColumns[2].n64NativeInteger;
        psQuery->bFound = TRUE;

        return FALSE;   // stop processing data
    }

    return TRUE;    // continue processing data
}

/*****************************************************************************
*
*   bPrepareTeamColumn
*
*****************************************************************************/
static BOOLEAN bPrepareTeamColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    TEAM_ROW_STRUCT *psTeamRow
    )
{
    BOOLEAN bSuccess = TRUE;

    switch (tIndex)
    {
    case DB_TEAM_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTeamRow->un32TeamId;
        }
        break;

    case DB_TEAM_VERSION:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTeamRow->n32Version;
        }
        break;

    case DB_TEAM_ABBREV:
        {
            *peType = SQL_BIND_TYPE_C_STRING;
            *ppvData = (void *)psTeamRow->pacAbbrev;
        }
        break;

    case DB_TEAM_NAME:
        {
            *peType = SQL_BIND_TYPE_C_STRING;
            *ppvData = (void *)psTeamRow->pacName;
        }
        break;

    case DB_TEAM_NICKNAME:
        {
            *peType = SQL_BIND_TYPE_C_STRING;
            *ppvData = (void *)psTeamRow->pacNickname;
        }
        break;

    case DB_TEAM_LEAGUES_TIERS:
        {
            *peType = SQL_BIND_TYPE_BLOB;
            *ppvData = (void *)(psTeamRow->asLeagueTiers);
            *ptDataSize = sizeof(psTeamRow->asLeagueTiers);
        }
        break;

    case DB_TEAM_MAX_FIELDS:
        {
            // Special case for virtual column ROWID
            *peType = SQL_BIND_TYPE_N64;
            *ppvData = (void *)&psTeamRow->n64RowId;
        }
        break;

    default:
        // we are done
        bSuccess = FALSE;
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
* bTeamTableHandler
*
*****************************************************************************/
static BOOLEAN bTeamTableHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    BOOLEAN bReturn = FALSE;
    size_t *ptNumTeams = (size_t *)pvArg;

    // Modify the working team CID
    CID_bModify(
        &gpsTeamCtrl->hWorkingTeamId,
        &psColumns[DB_TEAM_ID].uData.sUN32.un32Data);
    if( gpsTeamCtrl->hWorkingTeamId != CID_INVALID_OBJECT )
    {
        TEAM_OBJECT hTeam;
        size_t tLeagueNum = 0, tIndex;
        UN8 aun8Leagues[MAX_NUM_MEMBER_LEAGUES];
        UN8 aun8Tiers[MAX_NUM_MEMBER_LEAGUES];
        TEAM_LEAGUE_TIER_STRUCT asLeagueTier[MAX_NUM_MEMBER_LEAGUES];

        OSAL.bMemCpy(&asLeagueTier, 
            psColumns[DB_TEAM_LEAGUES_TIERS].uData.sBLOB.pvData, 
            sizeof(asLeagueTier));

        for (tIndex = 0; tIndex < MAX_NUM_MEMBER_LEAGUES; tIndex++)
        {
            if (asLeagueTier[tIndex].un8LeagueId != LEAGUE_INVALID_ID)
            {
                aun8Leagues[tLeagueNum] = asLeagueTier[tIndex].un8LeagueId;
                aun8Tiers[tLeagueNum++] = asLeagueTier[tIndex].un8Tier;
            }
        }

        // Add this team
        hTeam = TEAM_hAdd(gpsTeamCtrl->hWorkingTeamId, 
            (N32)psColumns[DB_TEAM_VERSION].
            uData.sUN32.un32Data,
            (const char *)psColumns[DB_TEAM_NAME].
            uData.sCString.pcData,
            (const char *)psColumns[DB_TEAM_NICKNAME].
            uData.sCString.pcData,
            (const char *)psColumns[DB_TEAM_ABBREV].
            uData.sCString.pcData,
            tLeagueNum,
            aun8Leagues,
            tLeagueNum,
            aun8Tiers
                );
        if(hTeam != TEAM_INVALID_OBJECT)
        {
            TEAM_OBJECT_STRUCT *psTeamObject = 
                (TEAM_OBJECT_STRUCT *)hTeam;

            // Any team loaded in is by definition 'saved'
            psTeamObject->bSaved = TRUE;

            (*ptNumTeams)++;

            // Success
            bReturn = TRUE;
        }
    }

    return bReturn;
}

/*****************************************************************************
*
* bPrune
*
*****************************************************************************/
static BOOLEAN bPrune(N32 n32Version)
{
    BOOLEAN bOk = FALSE;

    if (n32Version != TEAM_INVALID_VERSION)
    {
        BOOLEAN bSuccess;
        TEAM_OBJECT_STRUCT *psTeam, *psRemovedTeam;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hCurrent, 
            hRemove = OSAL_INVALID_LINKED_LIST_ENTRY;

        SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
            "Prune Teams with Table Version other than: %d\n",
            n32Version);

        hCurrent = OSAL.hLinkedListFirst(
            gpsTeamCtrl->hTeamList, (void **)&psTeam);
        
        while (hCurrent != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            if (psTeam->psTeam->n32Version != n32Version)
            {
                SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
                    "Removing Team: %s from the Global Teams List\n",
                    STRING.pacCStr(psTeam->psTeam->hNickname));

                hRemove = hCurrent;
            }

            hCurrent = OSAL.hLinkedListNext(hCurrent, (void **)&psTeam);

            if (hRemove != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                psRemovedTeam = (TEAM_OBJECT_STRUCT *)
                    OSAL.pvLinkedListThis(hRemove);

                eReturnCode = OSAL.eLinkedListRemove(hRemove);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TEAM_OBJECT_NAME
                        ": Failed to remove Team: %s, v%d "
                        "from Global Teams list",
                        STRING.pacCStr(psRemovedTeam->psTeam->hNickname), 
                        psRemovedTeam->psTeam->n32Version);
                }

                eReturnCode = OSAL.eLinkedListAdd(
                    gpsTeamCtrl->hTeamTrashBin,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    (void *)psRemovedTeam);

                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TEAM_OBJECT_NAME
                        ": Failed to add Team: %s, v%d into Trash Bin",
                        STRING.pacCStr(psRemovedTeam->psTeam->hNickname),
                        psRemovedTeam->psTeam->n32Version);
                }

                hRemove = OSAL_INVALID_LINKED_LIST_ENTRY;
            }
        }

        bSuccess = LEAGUE_bIterateContent(
            bPruneTeamsFromLeague, (void *)&n32Version);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TEAM_OBJECT_NAME
                ": Failed to prune teams from leagues");
        }

        snprintf(&gpsTeamCtrl->acQueryBuffer[0], TEAM_MAX_QUERY_LENGTH,
            PRUNE_TEAMS, n32Version);

        bOk = SQL_INTERFACE.bExecuteCommand (
            gpsTeamCtrl->hSQLConnection,
            &gpsTeamCtrl->acQueryBuffer[0]);
        if(bOk == TRUE)
        {
            gpsTeamCtrl->n32Version = n32Version;
        }
    }

    // Check purge results
    if (TRUE == bOk)
    {
        SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
            "Teams with version not equal to #%d"
            " have been purged.\n", n32Version);
    }
    else
    {
        SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
            "Teams, nothing has been purged.\n");
    }

    return bOk;
}

/*****************************************************************************
*
*   bSaveTeam
*
*****************************************************************************/
static BOOLEAN bSaveTeam (
    TEAM_OBJECT_STRUCT *psTeamObject,
    TEAM_SAVE_ITERATOR_STRUCT *psTeamSaveIterator
    )
{
    BOOLEAN bSuccess = TRUE;

    // Check inputs
    if((psTeamObject != NULL) && (psTeamSaveIterator != NULL))
    {
        N32 n32Version;

        // If the input version is 'invalid', this indicates
        // we want to store everything. However we will only
        // store teams which have a valid version.

        // Does this team have a valid version? Has it not
        // already been saved?
        if((psTeamObject->psTeam->n32Version != TEAM_INVALID_VERSION) &&
            (psTeamObject->bSaved == FALSE))
        {
            n32Version = psTeamSaveIterator->n32Version;

            if(psTeamSaveIterator->n32Version == TEAM_INVALID_VERSION)
            {
                // Force save
                n32Version = psTeamObject->psTeam->n32Version;
            }

            // Save this team if it is the same as the 
            // provided version.
            if(n32Version == psTeamObject->psTeam->n32Version)
            {
                BOOLEAN bExists;
                TEAM_ROW_STRUCT sTeamRow;
                N32 n32SavedVersion;
                void *pvTeamId = &sTeamRow.un32TeamId;

                // Get raw team id
                CID_n32GetValue(psTeamObject->hId, &pvTeamId);

                // Populate remaining part of the row
                sTeamRow.n32Version = 
                    psTeamObject->psTeam->n32Version;
                sTeamRow.pacName = 
                    STRING.pacCStr(psTeamObject->psTeam->hName);
                sTeamRow.pacAbbrev =
                    STRING.pacCStr(psTeamObject->psTeam->hAbbreviation);
                sTeamRow.pacNickname = 
                    STRING.pacCStr(psTeamObject->psTeam->hNickname);
                OSAL.bMemCpy(sTeamRow.asLeagueTiers,
                    psTeamObject->asLeagueTiers,
                    sizeof(sTeamRow.asLeagueTiers));

                bExists = bQueryTeamVersion(
                    sTeamRow.un32TeamId,
                    sTeamRow.asLeagueTiers,
                    &n32SavedVersion, &sTeamRow.n64RowId);
                if (FALSE == bExists)
                {
                    SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
                        "Inserting Team: %s %s (%s)\n", 
                        sTeamRow.pacName, sTeamRow.pacNickname,
                        sTeamRow.pacAbbrev);

                    bSuccess = psTeamObject->bSaved =
                        SQL_INTERFACE.bExecutePreparedCommand(
                        gpsTeamCtrl->hSQLConnection,
                        INSERT_TEAM,
                        DB_TEAM_MAX_FIELDS,
                        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareTeamColumn,
                        (void *)&sTeamRow);
                    if(bSuccess == TRUE)
                    {
                        psTeamSaveIterator->tInsertCount++;
                    }
                }
                else if (sTeamRow.n32Version != n32SavedVersion)
                {
                    SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
                        "Updating Team: %s %s (%s)\n", 
                        sTeamRow.pacName, sTeamRow.pacNickname,
                        sTeamRow.pacAbbrev);

                    bSuccess = psTeamObject->bSaved =
                        SQL_INTERFACE.bExecutePreparedCommand(
                        gpsTeamCtrl->hSQLConnection,
                        UPDATE_TEAM,
                        DB_TEAM_MAX_FIELDS + 1, // +1 for using extra virtual column ROWID
                        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareTeamColumn,
                        (void *)&sTeamRow);
                    if(bSuccess == TRUE)
                    {
                        psTeamSaveIterator->tReplacedCount++;
                    }
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(TEAM_OBJECT_NAME, 4,
                        "Team %s %s (%s) is already up to date\n",
                        sTeamRow.pacName, sTeamRow.pacNickname,
                        sTeamRow.pacAbbrev);
                    psTeamSaveIterator->tAlreadyUpToDateCount++;
                }
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   psCreateTeamObject
*
*****************************************************************************/
static TEAM_OBJECT_STRUCT *psCreateTeamObject (
    CID_OBJECT hId,
    N32 n32Version,
    char const *pacName,
    char const *pacNickname,
    char const *pacAbbrev,
    size_t tNumLeagues,
    UN8 const *paun8Leagues
        )
{
    size_t tSize = sizeof(TEAM_OBJECT_STRUCT) + sizeof(TEAM_STRUCT);
    TEAM_OBJECT_STRUCT *psTeamObject = NULL;
    TEAM_STRUCT *psLocalTeamStruct;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // check if we have a valid pointers
    if((pacName == NULL) || (pacNickname == NULL) || (pacAbbrev == NULL))
    {
        // bad input
        return NULL;
    }

    snprintf(&acName[0], sizeof(acName), TEAM_OBJECT_NAME":%s", pacAbbrev);

    // Now allocate a structure to hold the CID and market as one mapping.
    // This entry is available for all SMS objects and applications to use.
    psTeamObject = (TEAM_OBJECT_STRUCT *)
        OSAL.pvLinkedListMemoryAllocate(
        acName,
        tSize,
        FALSE
        );
    if(psTeamObject == NULL)
    {
        // Error! Unable to create
        return NULL;
    }

    // Object is not saved
    psTeamObject->bSaved = FALSE;

    // psTeamObject new TEAM object with provided inputs.
    psTeamObject->hId = CID.hDuplicate(hId);

    // Initialize league membership
    OSAL.bMemSet(
        &psTeamObject->sLeagueMembership, 0, 
        sizeof(psTeamObject->sLeagueMembership));

    // For each league provided, modify the league membership mask
    while(tNumLeagues-- > 0)
    {
        bSetLeagueMembershipBit(
            &psTeamObject->sLeagueMembership,
            paun8Leagues[tNumLeagues]);
    }

    // Initialize
    psLocalTeamStruct = (TEAM_STRUCT *)(psTeamObject + 1);

    psLocalTeamStruct->hName = 
        STRING.hCreate(pacName, 0);
    psLocalTeamStruct->hNickname = 
        STRING.hCreate(pacNickname, 0);
    psLocalTeamStruct->hAbbreviation =
        STRING.hCreate(pacAbbrev, 0);
    psLocalTeamStruct->n32Version = 
        n32Version;

    psTeamObject->psTeam = psLocalTeamStruct;

    return psTeamObject;
}

/*****************************************************************************
*
*   vDestroyTeamObject
*
*****************************************************************************/
static void vDestroyTeamObject (
    TEAM_OBJECT_STRUCT * psTeamObject
    )
{
    if(psTeamObject != NULL)
    {
        // Check if we have a CID
        if(psTeamObject->hId != CID_INVALID_OBJECT)
        {
            // Destroy CID
            CID_vDestroy(psTeamObject->hId);
            psTeamObject->hId = CID_INVALID_OBJECT;
        }

        // Destroy the name, abbrev and nickname strings if they exist

        if (psTeamObject->psTeam->hName != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psTeamObject->psTeam->hName);
            psTeamObject->psTeam->hName = STRING_INVALID_OBJECT;
        }

        if (psTeamObject->psTeam->hAbbreviation != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psTeamObject->psTeam->hAbbreviation);
            psTeamObject->psTeam->hAbbreviation = STRING_INVALID_OBJECT;
        }

        if (psTeamObject->psTeam->hNickname != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psTeamObject->psTeam->hNickname);
            psTeamObject->psTeam->hNickname = STRING_INVALID_OBJECT;
        }

        // Destroy object
        OSAL.vLinkedListMemoryFree((void *)psTeamObject);
    }

    return;
}

/*****************************************************************************
*
*   n16Compare
*
*       This is the function used to compare an existing sports TEAM objects
*       to each other. Specifically this function is used to perform a
*       relational comparison for sorting. This sorter keeps the TEAMs in order
*       based on each TEAM's id.
*
*       Outputs:
*               0   - TEAMs have the same value (equal)
*               > 0 - TEAM1 is greater than (after) TEAM2
*               < 0 - TEAM1 is less than (before) TEAM2 or error
*
*****************************************************************************/
static N16 n16Compare (
    TEAM_OBJECT_STRUCT const *psTeamObject1,
    TEAM_OBJECT_STRUCT const *psTeamObject2
    )
{
    N16 n16Result = N16_MIN;

    // Verify inputs
    if( (psTeamObject1 != NULL) && (psTeamObject2 != NULL) )
    {
        // Check CID relationship
        n16Result = CID.n16Compare(psTeamObject1->hId, psTeamObject2->hId);

        // If the CIDs are the same, we need to look deeper
        // We call these the same only if there is at least
        // one matching league member.
        if(n16Result == 0)
        {
            UN8 un8LeagueIndex = NUMBER_OF_LEAGUE_ELEMENTS -1,
                un8MatchingLeagues = 0;
            N16 n16Compare = 0;

            do
            {
                un8MatchingLeagues |= 
                    psTeamObject1->sLeagueMembership.un8Mask[un8LeagueIndex] &
                    psTeamObject2->sLeagueMembership.un8Mask[un8LeagueIndex];

                if(n16Compare == 0)
                {
                    n16Compare = COMPARE(
                        psTeamObject1->sLeagueMembership.un8Mask[un8LeagueIndex],
                        psTeamObject2->sLeagueMembership.un8Mask[un8LeagueIndex]
                            );
                }

                if(un8LeagueIndex == 0)
                {
                    break;
                }

                un8LeagueIndex--;

            } while(TRUE);

            if(un8MatchingLeagues == 0)
            {
                // Nothing matches, so just assign relation
                // according to league member magnitude.
                n16Result = n16Compare;
            }
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   bSetLeagueMembershipBit
*
*****************************************************************************/
static BOOLEAN bSetLeagueMembershipBit (
    LEAGUE_MEMBERSHIP_STRUCT *psLeagueMembership,
    UN8 un8LeagueId
        )
{
    BOOLEAN bSet = FALSE;

    // For this to work the un32LeagueId must always be
    // less than 32 and greater than 0.
    if((un8LeagueId < MAX_NUM_LEAGUES) && 
        (un8LeagueId != LEAGUE_INVALID_ID))
    {
        // Set league bit
        psLeagueMembership->un8Mask[un8LeagueId / 8]
            |= (1 << (un8LeagueId % 8));
        bSet = TRUE;
    }

    return bSet;
}

/*****************************************************************************
*
*   bPruneTeamsFromLeague
*
*****************************************************************************/
static BOOLEAN bPruneTeamsFromLeague(
    LEAGUE_OBJECT hLeague,
    void *pvArg
        )
{
    BOOLEAN bSuccess;
    N32 *pn32Version = (N32 *)pvArg;

    bSuccess = LEAGUE_bPruneTeams(hLeague, *pn32Version);

    return bSuccess;
}
