/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:TAG implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <time.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_obj.h"
#include "sms_update.h"
#include "string_obj.h"

#include "cm.h"

#include "tag_cm_interface.h"
#include "cm_tag_interface.h"
#include "tag_obj.h"
#include "_tag_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/******************************************************************************
*
*   TAG_eAdd
*
* This function adds a new tag instance to the configuration manager at
* the point indicated by hParentTag
*
* Inputs
* pacName  - 	A NULL terminated character array which contains the TagName
*               to be used for this new Tag. If NULL is provided, the function
*               will return SMSAPI_RETURN_CODE_INVALID_INPUT.
* hParentTag  -	A valid handle to the parent tag instance which is to act as
*               the newly added tag instance's immediate parent.
* phNewTag - 	A valid pointer to a TAG_OBJECT which is to be populated with
*               the registered tag's handle upon successful completion
* pacIdentifier - 	A pointer to a NULL terminated character array which is to
*                   be utilized for identifying this tag instance in the
*                   configuration file. The CM will include this identifier in
*                   the configuration file as the "name" attribute of the new
*                   tag instance.  If this argument is NULL, no identifier will
*                   be used for this tag instance. If this argument is provided,
*                   it must be unique for the Parent Tag to which it is being
*                   added.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eAdd(
    const char *pacName,
    TAG_OBJECT hParentTag,
    TAG_OBJECT *phNewTag,
    const char *pacIdentifier
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hParentTag;
    BOOLEAN bLocked;

    if (hParentTag == TAG_INVALID_OBJECT)
    {
        return SMSAPI_RETURN_CODE_CFG_NO_PARENT;
    }

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hParentTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // tag must have a name..
    if (pacName != NULL)
    {
        do
        {
            if (pacIdentifier != NULL)
            {
                // if the identifier isn't NULL we must make sure that
                // there isn't already a TAG with that instance name under the
                // parent tag
                eReturn = TAG_eGet(
                              pacName,
                              hParentTag,
                              phNewTag,
                              pacIdentifier,
                              FALSE
                                  );

                if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    // something went wrong
                    break;
                }

                if (*phNewTag != TAG_INVALID_OBJECT)
                {
                    // cannot create because there is already a tag under this
                    // parent with the specified instance name
                    *phNewTag = TAG_INVALID_OBJECT;
                    eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
                    break;
                }
            }

            // okay to add the tag
            eReturn = eAddTag(
                          pacName,
                          psObj,
                          phNewTag,
                          pacIdentifier
                              );
        } while (0);
    }

    SMSO_vUnlock((SMS_OBJECT)hParentTag);
    return eReturn;
}

/*****************************************************************************
*
*   TAG_eGet
*
* This function searches all tags under the Parent Tag for an instance with
* the requested name and identifier. The caller can optionally add a new Tag
* if one does not exist.

* Inputs
*  pacName  - A NULL terminated character array which contains the Tag Name for
*             this new TAG. If this input is NULL, the function will return
*             SMSAPI_RETURN_CODE_INVALID_INPUT.
*  hParentTag  - A valid handle to the parent tag instance which is to act
*                as the newly added tag instance's immediate parent.
*  phTag - 	A valid pointer to a TAG_OBJECT which is to be populated with the
*           requested tag's handle upon successful completion of this API.
* pacIdentifier - A pointer to a NULL-terminated character array which is to
*                 be utilized for identifying this tag instance in the
*                 configuration file. This may be NULL if no InstanceName is
*                 desired.
* bAddIfNotFound - A flag which, if TRUE, indicates that if the requested tag
*                  cannot be found, it will be created as a child of the
*                  Parent Tag. Note: Adding a Tag does not cause an immediate
*                  commit, nor is it considered a pending change. This is
*                  because the tag at this point has no value.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eGet(
    const char *pacName,
	TAG_OBJECT hParentTag,
	TAG_OBJECT *phTag,
	const char *pacIdentifier,
    BOOLEAN bAddIfNotFound
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hParentTag;

    // validate input
    if ( (pacName == NULL) ||
         (phTag == NULL) )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hParentTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if (psObj->bRemoved == TRUE)
        {
            // the parent tag has been removed
            eReturn = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
        }
        else
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;
            TAG_SEARCH_ITERATOR_STRUCT sSearchIteratorStruct;

            // Initialize structure
            OSAL.bMemSet(
                &sSearchIteratorStruct, 0, sizeof(sSearchIteratorStruct));
            sSearchIteratorStruct.hTag = TAG_INVALID_OBJECT;
            sSearchIteratorStruct.pacName = pacName;
            sSearchIteratorStruct.pacInstanceIdentifier = pacIdentifier;

            eOsalReturnCode = OSAL.eLinkedListIterate (
                psObj->hChildList,
                bChildListIterator,
                (void *)&sSearchIteratorStruct);

            if ((eOsalReturnCode != OSAL_SUCCESS) &&
                (eOsalReturnCode != OSAL_NO_OBJECTS))
            {
                eReturn = SMSAPI_RETURN_CODE_ERROR;
            }
            else
            {
                if ((sSearchIteratorStruct.hTag == TAG_INVALID_OBJECT) &&
                    (bAddIfNotFound == TRUE))
                {
                    // tag not found. create it
                    eReturn = eAddTag(
                       pacName,
                       psObj,
                       phTag,
                       pacIdentifier
                           );
                }
                else
                {
                    *phTag = sSearchIteratorStruct.hTag;
                    if(*phTag == TAG_INVALID_OBJECT)
                    {
                        // TODO:
                        // I think we need here an error code which says
                        // I don't have a tag, but I can continue.
                        eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                    }
                    else
                    {
                        eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                    }
                }
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hParentTag);
    }
    else
    {
        // There is no parent tag
        *phTag = TAG_INVALID_OBJECT;
        eReturn = SMSAPI_RETURN_CODE_CFG_NO_PARENT;
    }

    return eReturn;
}

/******************************************************************************
*
*   TAG_eRemove
*
* This function removes a tag instance (including any if its child tags)
* from the configuration file
*
* Inputs
*   hTag - 	  The handle of a valid tag instance.
*   bImmediateCommit - A flag indicating if the configuration file should be
*                      immediately updated upon invoking this API.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eRemove(
    TAG_OBJECT hTag,
    BOOLEAN bImmediateCommit
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)hTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SMSAPI_RETURN_CODE_ERROR;
    }

    if (psObj->bRemoved == TRUE)
    {
        // this tag has already been removed
        eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    }
    else
    {
        // remove the child from the parent's child list
        eOsalReturnCode = OSAL.eLinkedListRemove(psObj->hEntry);
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            BOOLEAN bError = FALSE;

            // mark this Tag's entry as invalid
            psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // now we'll mark all of this tag's children as removed
            vMarkRemoved(hTag, &bError);

            if (bError == FALSE)
            {
                BOOLEAN bOk;
                bOk = CMTAG_bTagRemoved(hTag, bImmediateCommit);

                if (bOk == TRUE)
                {
                    eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                }
            }
        }
    }
    SMSO_vUnlock((SMS_OBJECT)hTag);

    return eReturn;
}

/*****************************************************************************
*
*   TAG_eIterateChildren
*
* This API is used to iterate through the immediate child tags which are
* members of the provided tag
*
* Inputs
*   hParentTag - A handle to a tag instance in which to iterate.
*   bIterator -	This is a tag iteration function or handler defined by the
*               caller. This handler is called by the iteration API for
*               each child tag that is a member of the provided tag.
*   pvArg - A caller provided pointer which will be provided as an argument
*           when the bIterate call is made. This value is application
*           specific and must be cast to the appropriate type within the
*           caller provided iteration function.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eIterateChildren(
    TAG_OBJECT hParentTag,
    TAG_ITERATION_HANDLER bIterator,
    void *pvArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hParentTag;

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hParentTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if (psObj->bRemoved == TRUE)
        {
            // this tag has already been removed
            eReturnCode = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
        }
        else
        {
            // Iterate all child tags
            eOsalReturnCode =
                OSAL.eLinkedListIterate(
                    psObj->hChildList,
                    (OSAL_LL_ITERATOR_HANDLER)bIterator,
                    pvArg
                        );
            if( (eOsalReturnCode == OSAL_SUCCESS) ||
                (eOsalReturnCode == OSAL_NO_OBJECTS))
            {
                // List was iterated
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
        SMSO_vUnlock((SMS_OBJECT)hParentTag);
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   TAG_eNumberOfChildren
*
* This API is used to determine the number of child tags which are
* members of the provided tag
*
* Inputs
*   hParentTag - A handle to a tag instance in which to iterate.
*   pun32NumberOfChildren -	a valid pointer to UN32 memory block for which
*       the caller wishes to populate with
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eNumberOfChildren (
    TAG_OBJECT hParentTag,
    UN32 *pun32NumberOfChildren
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify inputs
    if (pun32NumberOfChildren == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hParentTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        UN32 un32Items = 0;
        OSAL_RETURN_CODE_ENUM eOsalCode;

        // De-reference object
        TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hParentTag;

        if (psObj->bRemoved == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
        }
        else
        {
            // Initialize return value
            *pun32NumberOfChildren = 0;

            eOsalCode = OSAL.eLinkedListItems(
                psObj->hChildList,
                &un32Items);

            if (eOsalCode == OSAL_SUCCESS)
            {
                *pun32NumberOfChildren = un32Items;

                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hParentTag);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   TAG_eFirstChild
*
* This function is used to get the first child tag found under the provided
* hTag argument
*
* Inputs
*  hParentTag - A handle to a tag instance in which to get the first child tag
*               instance.
*  phChildTag - A valid pointer to a tag instance handle which will store the
*               first child tag upon successful completion of this API. If no
*               child tags exist, this handle will be TAG_INVALID_OBJECT.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eFirstChild(
    TAG_OBJECT hParentTag,
    TAG_OBJECT *phFirstChildTag
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hParentTag;

    if (phFirstChildTag == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hParentTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if (psObj->bRemoved == TRUE)
        {
            // this tag has already been removed
            eReturnCode = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
        }
        else
        {
            OSAL_LINKED_LIST_ENTRY hEntry;

            hEntry =
                OSAL.hLinkedListFirst(
                    psObj->hChildList,
                    (void *)phFirstChildTag);

            if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                *phFirstChildTag = TAG_INVALID_OBJECT;
            }

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        SMSO_vUnlock((SMS_OBJECT)hParentTag);
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   TAG_eNextSibling
*
* This function is used to get the next sibling tag for the provided
* hTag argument
*
* Inputs
*   hTag  -  A handle to a tag instance in which to get the next sibling tag
*            instance.
*   phSiblingTag - 	A valid pointer to a tag instance handle which will store
*                   the next sibling tag upon successful completion of this
*                   API. If the provided tag is the last tag found within a
*                   group, the handle will be TAG_INVALID_OBJECT.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eNextSibling(
    TAG_OBJECT hTag,
    TAG_OBJECT *phNextSiblingTag
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    // validate input
    if (phNextSiblingTag == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if (psObj->bRemoved == TRUE)
        {
            // this tag has already been removed
            eReturnCode = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
        }
        else
        {
            OSAL_LINKED_LIST_ENTRY hEntry;

            hEntry = OSAL.hLinkedListNext(
                         psObj->hEntry,
                         (void *)phNextSiblingTag);

            if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                *phNextSiblingTag = TAG_INVALID_OBJECT;
            }

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        SMSO_vUnlock((SMS_OBJECT)hTag);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   TAG_ePreviousSibling
*
* This API is used to get the previous sibling tag found for the provided hTag
* argument
*
* Inputs
*    hTag - A handle to a tag instance in which to get the previous sibling tag
*           instance.
*    phSiblingTag - A valid pointer to a tag instance handle which will store
*                   the previous sibling tag upon successful completion of this
*                   API.  If the provided tag is the first tag found within a
*                   group, the handle will be TAG_INVALID_OBJECT.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_ePreviousSibling(
    TAG_OBJECT hTag,
    TAG_OBJECT *phPreviousSiblingTag
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    // validate input
    if (phPreviousSiblingTag == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if (psObj->bRemoved == TRUE)
        {
            // this tag has already been removed
            eReturnCode = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
        }
        else
        {
            OSAL_LINKED_LIST_ENTRY hEntry;

            hEntry = OSAL.hLinkedListPrev(
                         psObj->hEntry,
                         (void *)phPreviousSiblingTag);

            if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                *phPreviousSiblingTag = TAG_INVALID_OBJECT;
            }

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

        }
        SMSO_vUnlock((SMS_OBJECT)hTag);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   TAG_hTagName
*
*  This function is used to get the name of the tag indicated by the provided
*  hTag argument
*
* Inputs
*    hTag - A handle to a tag instance in which to get the associated tag
*           instance identifier.
*    phName - A valid pointer to a STRING_OBJECT which will store the text of
*             the Tag's Name.
*
* Outputs
*  A valid STRING_OBJECT which contains the Tag Name text
*  STRING_INVALID_OBJECT on error.
*
*****************************************************************************/
STRING_OBJECT TAG_hTagName(
    TAG_OBJECT hTag
        )
{
    BOOLEAN bLocked;
    STRING_OBJECT hTagName = STRING_INVALID_OBJECT;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        hTagName = psObj->hTagName;

        SMSO_vUnlock((SMS_OBJECT)hTag);
    }

    return hTagName;
}

/*****************************************************************************
*
*   TAG_hTagInstanceName
*
*
* Inputs
*    hTag - A handle to a valid tag in which to get the associated tag's
*           instance name.
*    phInstanceName - A valid pointer to a STRING_OBJECT which will store the
*                     text of the Tag's Identifier. STRING_INVALID_OBJECT if no
*                     Tag Identifier is present.
*
* Outputs:
*     A valid STRING_OBJECT which contains the Tag Instance Name text.
*     STRING_INVALID_OBJECT if the Tag doesn't have an Instance Name or on error
*
*****************************************************************************/
STRING_OBJECT TAG_hTagInstanceName(
    TAG_OBJECT hTag
        )
{
    BOOLEAN bLocked;
    STRING_OBJECT hTagInstanceIdentifier = STRING_INVALID_OBJECT;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    // lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // get the instance name
        hTagInstanceIdentifier = psObj->hTagInstanceIdentifier;

        SMSO_vUnlock((SMS_OBJECT)hTag);
    }

    return hTagInstanceIdentifier;
}

/******************************************************************************
*
*   TAG_eSetTagValue
*
* This function is used to set a TAG_OBJECT's value
*
* Inputs
*    hTag - The TAG_OBJECT whose value is being set
*    eTagType - the type of value being passed in
*    pvTagValue - pointer to the value
*    tNumBytes - number of bytes the value is
*    bImmediateCommit - TRUE if commit should occur as part of this call
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eSetTagValue(
    TAG_OBJECT hTag,
    TAG_TYPE_ENUM eTagType,
    void *pvTagValue,
    size_t tNumBytes,
    BOOLEAN bImmediateCommit
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)hTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SMSAPI_RETURN_CODE_ERROR;
    }

    if (psObj->bRemoved == TRUE)
    {
        // this tag has already been removed
        eReturn = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
    }
    else
    {
        // based on tag type
        switch(eTagType)
        {
            case TAG_TYPE_STRING:
            {
                // modify tag's STRING directly
                eReturn = eModifyValue(psObj, *(STRING_OBJECT *)pvTagValue);
            }
            break;

            case TAG_TYPE_INTEGER:
            {
                char *pacBuffer;

                pacBuffer =
                    CMTAG_pacBuffer(STRING_LENGTH_FOR_INTEGER, FALSE);
                if (pacBuffer != NULL)
                {
                    snprintf(
                        pacBuffer,
                        STRING_LENGTH_FOR_INTEGER,
                        "%i",
                        *(N32 *)pvTagValue);
                    eReturn = eModifyValueCStr(psObj, pacBuffer);
                }
                else
                {
                    eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                }
            }
            break;

            case TAG_TYPE_BINARY:
            {
                size_t tEncodedSize;
                char *pacBuffer;

                // determine how many bytes we'll need to store the
                // enocded string
                tEncodedSize = tGetEncodedStringLength(tNumBytes);

                // is there enough space in the buffer?
                pacBuffer = CMTAG_pacBuffer(tEncodedSize, FALSE);
                if (pacBuffer != NULL)
                {
                    tEncodedSize = tEncodeBase64(
                                       pvTagValue,
                                       tNumBytes,
                                       pacBuffer,
                                       tEncodedSize
                                           );

                    if (tEncodedSize == 0)
                    {
                        eReturn = SMSAPI_RETURN_CODE_CFG_BASE64_ERROR;
                    }
                    else
                    {
                        // modify tag's STRING directly
                        eReturn = eModifyValueCStr(psObj, pacBuffer);
                    }
                }
                else
                {
                    eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                }
            }
            break;

            case TAG_TYPE_INVALID:
            default:
            {
                // not allowed
                eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
            }
            break;
        } // switch

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // we successfully updated the tag

            // should we commit now, or later?
            if (bImmediateCommit == TRUE)
            {
                // commit now
                eReturn = CM_eCommitChangesToFile();
            }
            else
            {
                BOOLEAN bOk;

                bOk = CMTAG_bSetPendingChanges(TRUE);
                if (bOk == TRUE)
                {
                    eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                }
                else
                {
                    eReturn = SMSAPI_RETURN_CODE_ERROR;
                }
            }
        }
    }

    SMSO_vUnlock((SMS_OBJECT)hTag);

    return eReturn;
}

/******************************************************************************
*
*   TAG_eGetTagValue
*
* This function is used to get a TAG_OBJECT's value
*
* Inputs
*    hTag - The TAG_OBJECT whose value is being set
*    eTagType - the type of value that should be copied into the destination
*    ppvTagValue - pointer to the destination in which the value will be copied
*    ptNumAvailableBytes - size in bytes of the destination
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the request.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM TAG_eGetTagValue(
    TAG_OBJECT hTag,
    TAG_TYPE_ENUM eTagType,
    void **ppvTagValue,
    size_t *ptNumAvailableBytes
        )
{
    BOOLEAN bLocked;
    STRING_OBJECT hString;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    if (ppvTagValue == NULL)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)hTag, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SMSAPI_RETURN_CODE_ERROR;
    }

    hString = psObj->hTagValue;

    if (psObj->bRemoved == TRUE)
    {
        eReturn = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
    }
    else if (hString == STRING_INVALID_OBJECT)
    {

        eReturn =  SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET;
    }
    else
    {
        // based on tag type
        switch(eTagType)
        {
            case TAG_TYPE_STRING:
            {
                // copy STRING
                STRING_OBJECT hStringReturned;

                if (*((STRING_OBJECT*)*ppvTagValue) == STRING_INVALID_OBJECT)
                {
                    hStringReturned = STRING.hDuplicate(hString);
                    *((STRING_OBJECT *)(*ppvTagValue)) = hStringReturned;
                }
                else
                {
                    // create a temp buffer, then copy
                    size_t tLength;
                    char *pacStr;

                    tLength = STRING.tSize(hString);
                    tLength++; // add one for NULL

                    pacStr = (char *) SMSO_hCreate(
                                          TAG_OBJECT_NAME":TempBuff",
                                          tLength,
                                          SMS_INVALID_OBJECT,
                                          FALSE // No lock necessary
                                          );

                    if (pacStr != NULL)
                    {
                        size_t tNumWritten;
                        BOOLEAN bOk = FALSE;

                        tNumWritten = STRING.tCopyToCStr(
                                          hString,
                                          pacStr,
                                          tLength
                                              );
                        if (tNumWritten == tLength)
                        {
                            if (*((STRING_OBJECT *)(*ppvTagValue)) !=
                                STRING_INVALID_OBJECT)
                            {
                                STRING.bModifyCStr(
                                    *((STRING_OBJECT *)(*ppvTagValue)),
                                        pacStr
                                            );
                            }
                            else
                            {
                                hStringReturned = STRING.hCreate(pacStr, 0);
                                *((STRING_OBJECT *)(*ppvTagValue)) =
                                    hStringReturned;
                                bOk = TRUE;
                            }
                        }

                        if (bOk == FALSE)
                        {
                            eReturn = SMSAPI_RETURN_CODE_ERROR;
                        }
                        // we're done with the buffer
                        SMSO_vDestroy((SMS_OBJECT)pacStr);
                    }
                }
            }
            break;

            case TAG_TYPE_INTEGER:
            {
                char *pacBuffer;
                N32 n32Value;

                pacBuffer =
                    CMTAG_pacBuffer(STRING_LENGTH_FOR_INTEGER, FALSE);
                if (pacBuffer != NULL)
                {
                STRING.tCopyToCStr(
                    hString,
                    pacBuffer,
                    STRING_LENGTH_FOR_INTEGER);

                n32Value = atoi(pacBuffer);

                if (*ppvTagValue == NULL)
                {
                    N32 *pn32Value;
                    pn32Value = (N32 *)SMSO_hCreate(
                                           TAG_OBJECT_NAME":Ptr",
                                           sizeof(N32),
                                           SMS_INVALID_OBJECT,
                                           FALSE // No lock necessary
                                            );
                    *ppvTagValue = (void *)pn32Value;
                }

                if (*ppvTagValue != NULL)
                {
                    *((N32 *)*ppvTagValue) = n32Value;
                }
                else
                {
                    eReturn = SMSAPI_RETURN_CODE_ERROR;
                }
            }
                else
                {
                    eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                }
            }
            break;

            case TAG_TYPE_BINARY:
            {

                UN8 *paun8DecodedBinary;
                size_t tMemSizeReq, tDecodedSize, tNumEncodedChars;

                tNumEncodedChars = STRING.tSize(hString);
                tMemSizeReq = tGetDecodedLength(tNumEncodedChars);
                if ( tMemSizeReq == 0)
                {
                    // tag must not have a value
                    *ptNumAvailableBytes = 0;
                }
                else
                {
                    BOOLEAN bAllocatedMemory;
                    size_t tCopied;
                    char *pacBuffer;

                    do
                    {
                        if (*ppvTagValue != NULL)
                        {
                            if (*ptNumAvailableBytes < tMemSizeReq)
                            {
                                // caller didn't provide us with enough memory
                                eReturn = SMSAPI_RETURN_CODE_CFG_VALUE_TOO_LONG;
                                break;
                            }

                            paun8DecodedBinary = *ppvTagValue;
                            bAllocatedMemory = FALSE;
                        }
                        else
                        {
                            paun8DecodedBinary =
                                (UN8 *)SMSO_hCreate(
                                           TAG_OBJECT_NAME":Value",
                                           tMemSizeReq,
                                           SMS_INVALID_OBJECT,
                                           FALSE // No lock necessary
                                            );

                            if (paun8DecodedBinary == NULL)
                            {
                                eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                                break;
                            }

                            bAllocatedMemory = TRUE;
                        }

                        // add one for the NULL...
                        tNumEncodedChars++;

                        pacBuffer =
                            CMTAG_pacBuffer(tNumEncodedChars, FALSE);
                        if (pacBuffer == NULL)
                        {
                            eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                            break;
                        }

                        tCopied = STRING.tCopyToCStr(
                            hString,
                            pacBuffer,
                            tNumEncodedChars);
                        if (tCopied != tNumEncodedChars)
                        {
                            eReturn = SMSAPI_RETURN_CODE_ERROR;
                            break;
                        }

                        // run through the base64 decoder
                        tDecodedSize = tDecodeBase64(
                                           pacBuffer,
                                           paun8DecodedBinary,
                                           tMemSizeReq
                                               );
                        if (tMemSizeReq != tDecodedSize)
                        {
                            // Error
                            if (bAllocatedMemory == TRUE)
                            {
                                SMSO_vDestroy((SMS_OBJECT)paun8DecodedBinary);
                            }

                            eReturn = SMSAPI_RETURN_CODE_CFG_BASE64_ERROR;
                            break;
                        }
                        else
                        {
                            if (bAllocatedMemory == TRUE)
                            {
                                // update pointer
                                *ppvTagValue = (void *)paun8DecodedBinary;
                            }
                            *ptNumAvailableBytes = tDecodedSize;
                        }

                    } while (0);
                }
            }
            break;

            case TAG_TYPE_INVALID:
            default:
            {
                // not allowed
                eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
            }
            break;
        }
    }

    // unlock
    SMSO_vUnlock((SMS_OBJECT)hTag);

    return eReturn;
}

/*****************************************************************************
                         TAG-CM INTERFACE FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   TAGCM_hCreate
*
* This function will create a TAG_OBJECT
*
*****************************************************************************/
TAG_OBJECT TAGCM_hCreate (
    SMS_OBJECT hParent,
    const char *pacTagName,
    const char *pacTagInstanceName,
    STRING_OBJECT hTagValue
	    )
{
   TAG_OBJECT_STRUCT *psObj =
        (TAG_OBJECT_STRUCT *)TAG_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturn;
    BOOLEAN bOwner;
    size_t tSizeName = 0, tSizeInstanceName = 0, tSize = 0;
    char *pacPtr;

    // Verify input
    bOwner = SMSO_bOwner((SMS_OBJECT)hParent);

    if ((pacTagName == NULL) || (bOwner == FALSE))
    {
        return TAG_INVALID_OBJECT;
    }

    // compute the required size
    tSizeName = strlen(pacTagName);
    tSize = tSizeName+1; // add one for NULL
    if (pacTagInstanceName != NULL)
    {
        tSizeInstanceName = strlen(pacTagInstanceName);
        tSize = tSize + tSizeInstanceName +1; // add one for NULL
    }

     // Create an instance of this object and a buffer that will hold the
    // name and instance name
    psObj = (TAG_OBJECT_STRUCT *)
        SMSO_hCreate(
            TAG_OBJECT_NAME,
            sizeof(TAG_OBJECT_STRUCT)+tSize,
            (SMS_OBJECT)hParent,
            FALSE);
    if(psObj == NULL)
    {
        // Error!
        return TAG_INVALID_OBJECT;
    }

    do
    {
        // point to the location of the buffer
        psObj->pacNamesBuffer = (char*)((TAG_OBJECT_STRUCT *)psObj + 1);

        // copy in the name
        pacPtr = psObj->pacNamesBuffer;
        strncpy(pacPtr, pacTagName, tSizeName);
        pacPtr = pacPtr + tSizeName;
        *pacPtr = '\0';

        if (pacTagInstanceName != NULL)
        {
            // copy in the instance name
            pacPtr = pacPtr + 1;
            strncpy(pacPtr, pacTagInstanceName, tSizeInstanceName);
            pacPtr = pacPtr + tSizeInstanceName;
            *pacPtr = '\0';
        }

        // reset pointer to start of buffer
        pacPtr = psObj->pacNamesBuffer;

        // create a STRING containing the tag name
        psObj->hTagName = STRING_hCreateConst(pacPtr, tSizeName);

        if ( psObj->hTagName == STRING_INVALID_OBJECT)
        {
            // Error - failed to create obj
            break;
        }

        if (pacTagInstanceName != NULL)
        {
            // set ptr to beginning of the instance name
            pacPtr = pacPtr + tSizeName + 1;
            // create a STRING containing the tag instance name
            psObj->hTagInstanceIdentifier = STRING_hCreateConst(
                                                pacPtr,
                                                tSizeInstanceName
                                                    );
            if (psObj->hTagInstanceIdentifier == STRING_INVALID_OBJECT)
            {
                // Error - failed to create obj
                break;
            }
        }

        // Create the child list
        eReturn =
            OSAL.eLinkedListCreate(
                &psObj->hChildList,
                TAG_OBJECT_NAME":ChildList",
                NULL,
                OSAL_LL_OPTION_LINEAR
                    );
        if(eReturn != OSAL_SUCCESS)
        {
                break;
        }

        // Initialize tag object info
        psObj->hTagValue = hTagValue;
        psObj->bRemoved = FALSE;
        psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psObj->hParentTag = TAG_INVALID_OBJECT;

        return (TAG_OBJECT)psObj;

    } while(0);

    // Error!
    TAGCM_vDestroy((TAG_OBJECT)psObj);
    return TAG_INVALID_OBJECT;
}

/*****************************************************************************
*
*   TAGCM_vDestroy
*
* This function will destroy a TAG_OBJECT
*
*****************************************************************************/
void TAGCM_vDestroy (
    TAG_OBJECT hTag
        )
{
    BOOLEAN bOwner;
    TAG_OBJECT_STRUCT *psObj =
        (TAG_OBJECT_STRUCT *)hTag;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hTag);
    if(bOwner == TRUE)
    {
        // Un-Initialize

        // Destroy string objects (if they exist)...

        if(psObj->hTagName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hTagName);
            psObj->hTagName = STRING_INVALID_OBJECT;
        }

        if(psObj->hTagInstanceIdentifier != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hTagInstanceIdentifier);
            psObj->hTagInstanceIdentifier = STRING_INVALID_OBJECT;
        }

        if(psObj->hTagValue != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hTagValue);
            psObj->hTagValue = STRING_INVALID_OBJECT;
        }

        // Check if the child list exists.
        if(psObj->hChildList != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries from the list and destroy each tag
            OSAL.eLinkedListRemoveAll(
                psObj->hChildList,
                (OSAL_LL_RELEASE_HANDLER)TAGCM_vDestroy
                    );

            // Destroy the list itself
            OSAL.eLinkedListDelete(psObj->hChildList);
            psObj->hChildList = OSAL_INVALID_OBJECT_HDL;
        }

        psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psObj->hParentTag = TAG_INVALID_OBJECT;

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hTag);
    } // if(bValid == TRUE)

    return;
}

/*****************************************************************************
*
*   TAGCM_n32WriteTagToFile
*       hTag - the tag to write (including all of its children)
*       psFile - the file to write to
*       n16IndentLevel - the base indentation level
*                        (each child will be indeted one level from its parent)
*
*****************************************************************************/
N32 TAGCM_n32WriteTagToFile(
    TAG_OBJECT hTag,
    FILE *psFile,
    N16 n16IndentLevel
        )
{
    BOOLEAN bOwner;
    N32 n32NumWritten = EOF;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

    // validate object
    bOwner = SMSO_bOwner((SMS_OBJECT)hTag);
    if(bOwner == TRUE)
    {
        if (psFile != NULL)
        {
            N16 n16Indent = n16IndentLevel;
            n32NumWritten = 0;

            while (n16Indent-- > 0)
            {
                // print tabs to indent
                n32NumWritten += WRITE_LITERAL(psFile, "    ");
            }

            // write out the tag name
            n32NumWritten += n32WriteTagNameToFile(psObj, psFile, FALSE);
            n16IndentLevel++;

            if (psObj->hTagValue != STRING_INVALID_OBJECT)
            {
                // print the tag value
                n32NumWritten += WRITE_LITERAL(psFile, " ");
                n32NumWritten += STRING_n32FWrite(psObj->hTagValue, psFile, TRUE);
                n32NumWritten += WRITE_LITERAL(psFile, " ");
            }
            else
            {
                // this tag doesn't have a value set.

                UN32 un32Items = 0;

                OSAL.eLinkedListItems (
                    psObj->hChildList,
                    &un32Items);
                if (un32Items > 0)
                {
                    // this is a parent tag. print out all its children
                    TAG_WRITE_ITERATOR_STRUCT sTagWriteStruct;

                    n32NumWritten += WRITE_LITERAL(psFile, "\n");

                    sTagWriteStruct.psFile = psFile;
                    sTagWriteStruct.n16IndentLevel = n16IndentLevel;
                    sTagWriteStruct.n32NumWritten = n32NumWritten;

                    // iterate child list and print each one out.
                    OSAL.eLinkedListIterate(
                        psObj->hChildList,
                        bWriteTagIterator,
                        (void *)&sTagWriteStruct
                            );
                    n16IndentLevel = sTagWriteStruct.n16IndentLevel;
                    n32NumWritten = sTagWriteStruct.n32NumWritten;

                    n16Indent = n16IndentLevel-1;

                    while (n16Indent-- > 0)
                    {
                        // print tabs to indent
                        n32NumWritten += WRITE_LITERAL(psFile, "    ");
                    }
                }
                else
                {
                    // no value set. just print out a couple of blank spaces
                    n32NumWritten += WRITE_LITERAL(psFile, "  ");
                }
            }

            // print out the close tag

            n32NumWritten += n32WriteTagNameToFile(psObj, psFile, TRUE);

            // print newline
            n32NumWritten += WRITE_LITERAL(psFile, "\n");
        }
    }

    return n32NumWritten;
}

/*****************************************************************************
*
*   TAGCM_hGetParentTag
*
*  This function will return a TAG_OBJECT's Parent Tag
*
*****************************************************************************/
TAG_OBJECT TAGCM_hGetParentTag(
    TAG_OBJECT hChildTag
        )
{
    BOOLEAN bOwner;
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hChildTag;

    // validate object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChildTag);
    if(bOwner == TRUE)
    {
        return psObj->hParentTag;
    }

    // invalid object
    return TAG_INVALID_OBJECT;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eModifyValueCStr
*
* This function is used to modify the TAG_OBJECTs Value STRING's C-String.
* If the TAG doesn't currently have a valid value string it will be created.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyValueCStr(
    TAG_OBJECT_STRUCT *psObj,
    const char *pacValue
        )
{

    if (psObj->hTagValue == STRING_INVALID_OBJECT)
    {
        // this tag doesn't have a value yet, so we need to create a STRING to
        // hold the value
        psObj->hTagValue = STRING.hCreate(pacValue, 0);
    }
    else
    {
        // this TAG already has a valid string object, so just modify it
        // (the return from bModifyCStr will return FALSE if the string already
        //  contains the pacValue text so don't use it to return success or not)
        STRING.bModifyCStr(psObj->hTagValue, pacValue);
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eModifyValue
*
* This function is used to modify the value string of a tag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyValue(
    TAG_OBJECT_STRUCT *psObj,
    STRING_OBJECT hString
        )
{
    if (psObj->hTagValue != STRING_INVALID_OBJECT)
    {
        // destroy our old tag value string
        STRING_vDestroy(psObj->hTagValue);
    }

    // duplicate the provided strings
    psObj->hTagValue = STRING_hDuplicate(SMS_INVALID_OBJECT, hString);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eSetParentTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetParentTag(
    TAG_OBJECT hChildTag,
    TAG_OBJECT hParentTag,
    OSAL_LINKED_LIST_ENTRY hEntry
        )
{
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hChildTag;

    psObj->hParentTag = hParentTag;
    psObj->hEntry = hEntry;

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   n32WriteTagNameToFile
*
* This function writes the tag name, instance name and delimiters to a file
* It can be used for open or close tags
*
*****************************************************************************/
static N32 n32WriteTagNameToFile(
    TAG_OBJECT_STRUCT *psObj,
    FILE *psFile,
    BOOLEAN bCloseTag
        )
{
    N32 n32NumWritten = 0, n32Temp;

    // write the tag name and delimiters to the file
    if (bCloseTag == TRUE)
    {
        n32NumWritten += WRITE_LITERAL(psFile, TAG_BEGIN_CLOSE_DELIMITER);
    }
    else
    {
        n32NumWritten += WRITE_LITERAL(psFile, TAG_BEGIN_DELIMITER);
    }
    n32Temp = STRING.n32FWrite(psObj->hTagName, psFile);
    if (n32Temp > 0)
    {
        n32NumWritten += n32Temp;
    }
    if ((bCloseTag == FALSE) &&
        (psObj->hTagInstanceIdentifier != STRING_INVALID_OBJECT))
    {
        // only write the instance name if it exists and we're not printing
        // the close tag
        n32NumWritten += WRITE_LITERAL(psFile, TAG_INSTANCE_DELIMITER"\"");
        n32Temp = STRING.n32FWrite(psObj->hTagInstanceIdentifier, psFile);
        if (n32Temp > 0)
        {
            n32NumWritten += n32Temp;
        }
        n32NumWritten += WRITE_LITERAL(psFile, "\"");
    }
    n32NumWritten += WRITE_LITERAL(psFile, TAG_END_DELIMITER);

    return n32NumWritten;
}

/*****************************************************************************
*
*   vMarkRemoved
*
* This function is used to indicate that a TAG and all its children have been
* removed
*
*****************************************************************************/
static void vMarkRemoved(
    TAG_OBJECT hTag,
    BOOLEAN *pbError
        )
{
    TAG_OBJECT_STRUCT *psObj = (TAG_OBJECT_STRUCT *)hTag;

        OSAL_RETURN_CODE_ENUM eReturn;

        // iterate tree of children, marking each as removed.
        eReturn = OSAL.eLinkedListIterate(
            psObj->hChildList,
            bMarkRemovedIterator,
            (void *)pbError
                );

        if ( (eReturn != OSAL_SUCCESS) &&
             (eReturn != OSAL_NO_OBJECTS) )
        {
            *pbError |= TRUE;
        }

        // mark the tag removed
        psObj->bRemoved = TRUE;
    }

/*****************************************************************************
*
*   eAddTag
*
* This function creates a tag and adds it to the parent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddTag(
    const char *pacName,
    TAG_OBJECT_STRUCT *psObj,
    TAG_OBJECT *phNewTag,
    const char *pacIdentifier
        )
{
    TAG_OBJECT hNewTag = TAG_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        if (psObj->bRemoved == TRUE)
        {
            // the parent tag has been removed
            eReturn = SMSAPI_RETURN_CODE_CFG_TAG_REMOVED;
            break;
        }

        // create the new tag
        hNewTag = TAGCM_hCreate(
                      (SMS_OBJECT)psObj,
                      pacName,
                      pacIdentifier,
                      STRING_INVALID_OBJECT
                          );
        if (hNewTag == TAG_INVALID_OBJECT)
        {
            // Error - failed to create obj
            break;
        }

        // add the child to the end of the parent's child list
        eOsalReturnCode = OSAL.eLinkedListAdd(
                              psObj->hChildList,
                              &hEntry,
                              hNewTag
                                  );
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            eReturn = eSetParentTag(hNewTag, (TAG_OBJECT)psObj, hEntry);
        }

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // we successfully created a tag and added it to its parent
            // update the provided pointer with the new tag's handle
            *phNewTag = hNewTag;
        }
        else
        {
            // we failed to add the tag to its parent

            // remove the item from the ll
            OSAL.eLinkedListRemove(hEntry);

            // destroy the tag we just created
            // (STRING_OBJECTS are destroyed in the tag destroy function)
            TAGCM_vDestroy(hNewTag);
        }

    } while (0);

    return eReturn;
}

/*****************************************************************************
*
*   bWriteTagIterator
*
*****************************************************************************/
static BOOLEAN bWriteTagIterator(
    void *pvData,
    void *pvArg
        )
{
    TAG_OBJECT hTag = (TAG_OBJECT)pvData;
    TAG_WRITE_ITERATOR_STRUCT *psTagWriteStruct =
        (TAG_WRITE_ITERATOR_STRUCT *)pvArg;

    // write the tag to the file
    psTagWriteStruct->n32NumWritten += TAGCM_n32WriteTagToFile(
                                           hTag,
                                           psTagWriteStruct->psFile,
                                           psTagWriteStruct->n16IndentLevel
                                               );

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bChildListIterator
*
* This iterator is used iterate the child list of a TAG looking for a name
* (and instance if provided) match. It is used to retrieve a specific tag
* belonging to a parent tag.
*
*****************************************************************************/
static BOOLEAN bChildListIterator(
    void *pvData,
    void *pvArg
        )
{
    TAG_OBJECT hTag = (TAG_OBJECT)pvData;
    TAG_SEARCH_ITERATOR_STRUCT *psSearchStruct =
        (TAG_SEARCH_ITERATOR_STRUCT *)pvArg;

    STRING_OBJECT hName;

    hName = TAG_hTagName(hTag);

    // does tag name match?
    if (STRING.n16CompareCStr(psSearchStruct->pacName, 0, hName) == 0)
    {
        BOOLEAN bFound = FALSE;

        // did caller supply an instance name to look for?
        if (psSearchStruct->pacInstanceIdentifier != NULL)
        {
            // yes. we need to match instance name
            STRING_OBJECT hIdentifier;
            hIdentifier = TAG_hTagInstanceName(hTag);
            if (STRING.n16CompareCStr(psSearchStruct->pacInstanceIdentifier, 0,
                                      hIdentifier) == 0)
            {
                // both tag name AND instance name matched
                bFound = TRUE;
            }
        }
        else
        {
            // tag name matched and the caller doesn't care about instance name
            bFound = TRUE;
        }

        // did we find what we're looking for
        if (bFound == TRUE)
        {
            // yes
            psSearchStruct->hTag = hTag;

            // stop iterating - we found what we were looking for
            return FALSE;
        }
    }

    // still haven't found what we're looking for ...
    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bMarkRemovedIterator
*
* This iterator marks the tag and all its children as removed
*
*****************************************************************************/
static BOOLEAN bMarkRemovedIterator(
    void *pvData,
    void *pvArg
        )
{
    TAG_OBJECT hTag = (TAG_OBJECT)pvData;
    BOOLEAN *pbError = (BOOLEAN *)pvArg;

    // Mark this tag and all its children as removed
    vMarkRemoved(hTag, pbError);

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*       tGetDecodedLength
*
*       This function is used to determine the size in bytes of an
*       encoded string after decoding
*
*
*       Inputs:
*               tEncodedStringLength - the size in bytes of the encoded string
*
*       Outputs:
*               the decoded size in bytes
*
*****************************************************************************/
static size_t tGetDecodedLength(
    size_t tEncodedStringLength
        )
{
     size_t tDecodedLength;
     // Decoded length is 3/4 of the encoded length.
     // this is because base64 encoding takes 3 bytes each consisting of 8 bits
     // and represents them as 4 numbers of 6 bits. These 6 bits are mapped to
     // one of 64 ascii characters. The decoding reverses this process, so the
     // 4 bytes of 6 bits again become the original 3 bytes of 8 bits.
     tDecodedLength = (tEncodedStringLength / 4) * 3;

     return tDecodedLength;
}


/*****************************************************************************
*
*       tGetEncodedStringLength
*
*       This function is used to determine the size in bytes of an
*       encoded string given the number of binary bytes
*
*
*       Inputs:
*               tDecodedLength - the size in bytes of the binary to encode
*
*       Outputs:
*               the encoded string size in bytes
*
*****************************************************************************/
static size_t tGetEncodedStringLength(
    size_t tDecodedLength
        )
{
    size_t tEncodedStringLength;

    // encoded length is 1/3 more than the decoded length
     // Decoded length is 3/4 of the encoded length.
     // this is because base64 encoding takes 3 bytes each consisting of 8 bits
     // and represents them as 4 numbers of 6 bits. These 6 bits are mapped to
     // one of 64 ascii characters.
    tEncodedStringLength = tDecodedLength / 3;

    //  Check if we have a partially-filled block
    if (tEncodedStringLength * 3 != tDecodedLength)
    {
        tEncodedStringLength++;
    }

    tEncodedStringLength = tEncodedStringLength * 4;

    return tEncodedStringLength;
}


/*****************************************************************************
*
*       tEncodeBase64
*
*       This function converts a binary array of bytes to a string of chars.
*       Base64 encoding is one of the techniques used in MIME encoding to take
*       non-printable binary data and represent it as printable characters.
*       base64 encoding takes 3 bytes each consisting of 8 bits and represents
*       them as 4 numbers of 6 bits. These 6 bits are mapped to one of 64 ascii
*       characters. The decoding reverses this process, so the 4 bytes of 6 bits
*       again become the original 3 bytes of 8 bits.
*
*       Inputs:
*           paun8Source - array of bytes to encode
*           tSourceSize - number of bytes to encode
*           pacTarget - array to contain encoded characters
*           tTargetSize - the size of the target, in bytes
*
*       Outputs:
*               None
*
*****************************************************************************/
static size_t tEncodeBase64(
    const UN8 *paun8Source,
    size_t tSourceSize,
    char *pacTarget,
    size_t tTargetSize
        )
{
    size_t  tNumBlocks, tEncodedStringSize = 0;
    char *pacInTarget;
    UN8 *paun8InSource, un8Value;

    if ( (paun8Source == NULL) ||
         (tSourceSize == 0) ||
         (pacTarget == NULL) )
    {
        return (0);
    }

    //  Bit positions
    //              | byte 1 | byte 2 | byte 3 |
    // source block   87654321 87654321 87654321         -> 3 bytes of 8 bits
    //
    //              | byte 1 | byte 2 | byte 3 | byte 4 |
    // Encoded block  876543   218765   432187   654321  -> 4 bytes of 6 bits
    //

    tNumBlocks = tSourceSize / 3;

    //  Check if we have a partially-filled block
    if (tNumBlocks * 3 != tSourceSize)
    {
        tNumBlocks++;
    }

    tEncodedStringSize = (size_t) tNumBlocks * 4;
    if (tTargetSize < tEncodedStringSize)
    {
        return (0);
    }

    pacTarget[tEncodedStringSize] = '\0';

    paun8InSource = (UN8 *)paun8Source;      //  Point to start of buffers
    pacInTarget = pacTarget;

    while (tNumBlocks--)
    {
        //  Byte 1
        un8Value = *paun8InSource >> 2;
        *pacInTarget++ = gacBase64ToCharTbl[un8Value];

        //  Byte 2
        un8Value = (*paun8InSource++ & 0x03) << 4;
        if ((size_t) (paun8InSource - paun8Source) < tSourceSize)
        {
            un8Value |= (*paun8InSource & 0xF0) >> 4;
        }
        *pacInTarget++ = gacBase64ToCharTbl[un8Value];

        //  Byte 3 - pad the buffer with '=' if block not completed
        if ((size_t) (paun8InSource - paun8Source) < tSourceSize)
        {
            un8Value = (*paun8InSource++ & 0x0F) << 2;
            if ((size_t) (paun8InSource - paun8Source) < tSourceSize)
            {
                un8Value |= (*paun8InSource & 0xC0) >> 6;
            }
            *pacInTarget++ = gacBase64ToCharTbl[un8Value];
        }
        else
        {
            *pacInTarget++ = '=';
        }

        //  Byte 4 - pad the buffer with '=' if block not completed
        if ((size_t) (paun8InSource - paun8Source) < tSourceSize)
        {
            un8Value = *paun8InSource++ & 0x3F;
            *pacInTarget++ = gacBase64ToCharTbl[un8Value];
        }
        else
        {
            *pacInTarget++ = '=';
        }
   }
   return (tEncodedStringSize);
}


/*****************************************************************************
*
*       tDecodeBase64
*
*       This function converts a string of characters to a binary array
*       of bytes. The target buffer must be at least 3/4 the size of the
*       base64 data. This is because base64 encoding takes 3 bytes each
*       consisting of 8 bits and represents them as 4 numbers of 6 bits. These
*       6 bits are mapped to one of 64 ascii characters. The decoding reverses
*       this process, so the 4 bytes of 6 bits again become the original 3
*       bytes of 8 bits.
*
*       Inputs:
*           pacSource - array of characters to decode
*           paun8Target - byte array to place the binary data in
*           tTargetSize - number of bytes available for decoded binary
*
*       Outputs:
*           Returns the number of characters output into the target buffer.
*
*****************************************************************************/
static size_t tDecodeBase64(
    const char *pacSource,
    UN8 *paun8Target,
    size_t tTargetSize
        )
{
    size_t tSourceSize, tDecodedBinarySize = 0;
    int tNumBlocks;
    char *pacInSource;
    UN8 un8Value, *pun8InTarget;

    if ( (pacSource == NULL) ||
         (paun8Target == NULL) )
    {
        return (0);
    }

    /// Bit positions
    //              | byte 1 | byte 2 | byte 3 | byte 4 |
    //  Encoded block  654321   654321   654321   654321  -> 4 bytes of 6 bits
    //              | byte 1 | byte 2 | byte 3 |
    //  Decoded block  65432165 43216543 21654321         -> 3 bytes of 8 bits
    //
    tSourceSize = strlen(pacSource);
    tNumBlocks = tSourceSize / 4;
    tDecodedBinarySize = (size_t) tNumBlocks * 3;
    if (tTargetSize < tDecodedBinarySize)
    {
        return (0);
    }

    paun8Target[tDecodedBinarySize] = '\0';

    pacInSource = (char *)pacSource; //  Point to start of buffers
    pun8InTarget = paun8Target;

    while (tNumBlocks--)
    {
        //  Byte 1
        *pun8InTarget = gaun8CharToBase64Tbl [(UN8) *pacInSource++] << 2;
        un8Value = gaun8CharToBase64Tbl [(UN8) *pacInSource++];
        *pun8InTarget++ += ((un8Value & 0x30) >> 4);

        //  Byte 2
        *pun8InTarget = ((un8Value & 0x0F) << 4);
        un8Value = gaun8CharToBase64Tbl [(UN8) *pacInSource++];
        *pun8InTarget++ += ((un8Value & 0x3C) >> 2);

        //  Byte 3
        *pun8InTarget = (un8Value & 0x03) << 6;
        un8Value = gaun8CharToBase64Tbl [(UN8) *pacInSource++];
        *pun8InTarget++ += un8Value;
   }

   return (tDecodedBinarySize);
}

