/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SUBSCRIPTION implementation for the
 *  Sirius Module Services (SMS)
 *
******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"

#include "sms_event.h"
#include "sms_obj.h"
#include "cm.h"
#include "channel_obj.h"
#include "decoder_obj.h"

#include "sub_notification.h"
#include "_sub_notification.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eStart
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStart (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked, bOk;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SUB_NOTIFICATION_OBJECT hService;
        SUB_NOTIFICATION_OBJECT_STRUCT *psObj = NULL;

        // see if there is already a service handle for this type of
        // service
        hService = DECODER_hSubscriptionService(hDecoder);

        if (hService != SUB_NOTIFICATION_INVALID_OBJECT)
        {
            // this means one is already started
            SMSO_vUnlock((SMS_OBJECT)hDecoder);

            return SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
        }

        do
        {
            const char *pacName;
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // get the Decoder's name
            pacName = DECODER_pacName(hDecoder);
            if (pacName == NULL)
            {
                break;
            }

            // Create the name for this sub notification service object
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     SUB_NOTIFICATION_OBJECT_NAME":%s", pacName);
            psObj = (SUB_NOTIFICATION_OBJECT_STRUCT *)
                SMSO_hCreate(
                    &acName[0],
                    sizeof(SUB_NOTIFICATION_OBJECT_STRUCT),
                    (SMS_OBJECT)hDecoder,
                    FALSE
                        );

            if(psObj == NULL)
            {
                // Error!
                break;
            }

            psObj->eCurrentState = SUB_NOTIFICATION_INITIAL;
            psObj->bMetaNotificationDone = FALSE;
            psObj->bTuneNotificationDone = FALSE;
            psObj->un8Period = 0;
            psObj->hTunedChannel = CHANNEL_INVALID_OBJECT;
            psObj->hMetaWaitEvent = SMS_INVALID_EVENT_HDL;
            psObj->hTextReplacementEvent = SMS_INVALID_EVENT_HDL;

            // Create a Meta Data Wait Timer for this Module. It's a one shot
            // timer used to detect when metatdata changes on the tuned channel
            // after a period of time since the DECODER became ready
            eReturnCode = OSAL.eTimerCreate(&psObj->hMetadataWaitTimer,
                SUB_NOTIFICATION_OBJECT_NAME":MWTimer", vMetadataWaitTimerCallback,
                &psObj->hMetaWaitEvent);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SUB_NOTIFICATION_OBJECT_NAME": Can't Create MWTimer Timer.");
                break;
            }

            // Create a Text Replacement Timer for this Module. It's a one shot
            // timer used to control how long the text replacement will be
            // active
            eReturnCode = OSAL.eTimerCreate(&psObj->hTextReplaceTimer,
                SUB_NOTIFICATION_OBJECT_NAME":TRTimer",
                vTextReplacementTimerCallback, &psObj->hTextReplacementEvent);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SUB_NOTIFICATION_OBJECT_NAME": Can't Create TRTimer Timer.");
                break;
            }

            bOk = DECODER_bSetSubNotificationServiceHandle(hDecoder,
                      (SUB_NOTIFICATION_OBJECT)psObj);
            if(bOk == FALSE)
            {
                // Error!
                break;
            }

            SMSO_vUnlock((SMS_OBJECT)hDecoder);

            puts(SUB_NOTIFICATION_OBJECT_NAME": Created service");

            // success
            return SMSAPI_RETURN_CODE_SUCCESS;

        } while(0);

        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        // Something went wrong
        SUB_NOTIFICATION_vStop((SUB_NOTIFICATION_OBJECT)psObj);
    }

    return eResult;
}

/*****************************************************************************
*
*   vStop
*
*****************************************************************************/
static void vStop (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SUB_NOTIFICATION_OBJECT hService =
            SUB_NOTIFICATION_INVALID_OBJECT;

        // see if there is a service handle for this type of service
        hService = DECODER_hSubscriptionService(hDecoder);
        if (hService != SUB_NOTIFICATION_INVALID_OBJECT)
        {
            SUB_NOTIFICATION_vStop(hService);
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
*
*   eState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eState (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;
    SUB_NOTIFICATION_STATE_ENUM eState = SUB_NOTIFICATION_INVALID;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SUB_NOTIFICATION_OBJECT hService =
            SUB_NOTIFICATION_INVALID_OBJECT;

        // see if there is a service handle for this type of service
        hService = DECODER_hSubscriptionService(hDecoder);
        if (hService != SUB_NOTIFICATION_INVALID_OBJECT)
        {
            SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
               (SUB_NOTIFICATION_OBJECT_STRUCT *)hService;

            eState = psObj->eCurrentState;
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eState;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SUB_NOTIFICATION_vChannelTuned
*
*****************************************************************************/
void SUB_NOTIFICATION_vChannelTuned (
    SUB_NOTIFICATION_OBJECT hService,
    CHANNEL_OBJECT hChannel,
    BOOLEAN bInitialTune
        )
{
    BOOLEAN bOwner;
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
            (SUB_NOTIFICATION_OBJECT_STRUCT *)hService;

    // Verify ownership SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hService);
    if(bOwner == TRUE)
    {
        psObj->hTunedChannel = hChannel;

        if (bInitialTune == FALSE)
        {
            bManagerFSM(psObj, SUB_NOTIFICATION_TUNE);
        }
    }
    else
    {
        puts(SUB_NOTIFICATION_OBJECT_NAME": SUB_NOTIFICATION_vChannelTuned - "
            "Service not running");
    }
}

/*****************************************************************************
*
*   SUB_NOTIFICATION_vDecoderReady
*
*****************************************************************************/
void SUB_NOTIFICATION_vDecoderReady (
    SUB_NOTIFICATION_OBJECT hService
        )
{
    BOOLEAN bOwner;
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
            (SUB_NOTIFICATION_OBJECT_STRUCT *)hService;

    // Verify ownership SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hService);
    if(bOwner == TRUE)
    {
        bManagerFSM(psObj, SUB_NOTIFICATION_INITIAL);
    }
    else
    {
        puts(SUB_NOTIFICATION_OBJECT_NAME": SUB_NOTIFICATION_vDecoderReady - "
            "Service not running");
    }

    return;
}

/*****************************************************************************
*
*   SUB_NOTIFICATION_vMetadataTimeout
*
*****************************************************************************/
void SUB_NOTIFICATION_vMetadataTimeout (
    SUB_NOTIFICATION_OBJECT hService
        )
{
    BOOLEAN bOwner;

    puts(SUB_NOTIFICATION_OBJECT_NAME":METADATA TIMEOUT."
            " Wait for metadata change");

    // Verify ownership of SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hService);
    if(bOwner == TRUE)
    {
        SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
                (SUB_NOTIFICATION_OBJECT_STRUCT *)hService;

        // register for artist/title changes on the currently tuned channel
        CHANNEL_bRegisterNotification(
            psObj->hTunedChannel,
            (CHANNEL_OBJECT_EVENT_TITLE |
             CHANNEL_OBJECT_EVENT_ARTIST),
            vChannelMetadataEventCallback,
            psObj,
            TRUE);
    }
    else
    {
        puts(SUB_NOTIFICATION_OBJECT_NAME": Metadata Timeout - "
            "Service not running");
    }
}

/*****************************************************************************
*
*   SUB_NOTIFICATION_vTextReplacementTimeout
*
*****************************************************************************/
void SUB_NOTIFICATION_vTextReplacementTimeout (
    SUB_NOTIFICATION_OBJECT hService
        )
{
    BOOLEAN bOwner;
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
            (SUB_NOTIFICATION_OBJECT_STRUCT *)hService;

    puts(SUB_NOTIFICATION_OBJECT_NAME": TEXT REPLACEMENT TIMEOUT.");

    // Verify ownership of SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hService);
    if(bOwner == TRUE)
    {
        // go to the wait state since text replacement is done
        bManagerFSM(psObj, SUB_NOTIFICATION_WAIT);
    }
    else
    {
        puts(SUB_NOTIFICATION_OBJECT_NAME": Text Replacement Timeout - "
            "Service not running");
    }
}

/*****************************************************************************
*
*   SUB_NOTIFICATION_vStop
*
*****************************************************************************/
void SUB_NOTIFICATION_vStop (
    SUB_NOTIFICATION_OBJECT hService
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hService, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
            (SUB_NOTIFICATION_OBJECT_STRUCT *)hService;
        DECODER_OBJECT hDecoder;

        if (psObj->hTunedChannel != CHANNEL_INVALID_OBJECT)
        {
            CHANNEL_vUnregisterNotification (
                psObj->hTunedChannel,
                vChannelMetadataEventCallback,
                psObj
                    );

            psObj->hTunedChannel = CHANNEL_INVALID_OBJECT;
        }

        // Subscription service objs. belong to DECODER object (their parent)
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

        DECODER_bSetSubNotificationServiceHandle(hDecoder, SUB_NOTIFICATION_INVALID_OBJECT);

        // Destroy metadata wait timer if one exists
        if(psObj->hMetadataWaitTimer != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            OSAL.eTimerStop(psObj->hMetadataWaitTimer);

            // Delete timer
            eReturnCode = OSAL.eTimerDelete(psObj->hMetadataWaitTimer);
            if(eReturnCode == OSAL_SUCCESS)
            {
                psObj->hMetadataWaitTimer = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Destroy text replacement timer if one exists
        if(psObj->hTextReplaceTimer != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            OSAL.eTimerStop(psObj->hTextReplaceTimer);

            // Delete timer
            eReturnCode = OSAL.eTimerDelete(psObj->hTextReplaceTimer);
            if(eReturnCode == OSAL_SUCCESS)
            {
                psObj->hTextReplaceTimer = OSAL_INVALID_OBJECT_HDL;
            }
        }

        if (psObj->hMetaWaitEvent != SMS_INVALID_EVENT_HDL)
        {
            SMS_EVENT_HDL hEvent = psObj->hMetaWaitEvent;
            psObj->hMetaWaitEvent = SMS_INVALID_EVENT_HDL;

            // if this event was allocated and waiting to post we need to free
            // it. If we go ahead and post it
            // it will get handled by decoder (doing no action w/
            // sub service. then the event will be freed and can be used
            // by decoder
            SMSE_bPostEvent(hEvent);
        }

        if (psObj->hTextReplacementEvent != SMS_INVALID_EVENT_HDL)
        {
            SMS_EVENT_HDL hEvent = psObj->hTextReplacementEvent;
            psObj->hTextReplacementEvent = SMS_INVALID_EVENT_HDL;

            // if this event was allocated and waiting to post we need to free
            // it. If we go ahead and post it
            // it will get handled by decoder (doing no action w/
            // sub service. then the event will be freed and can be used
            // by decoder
            SMSE_bPostEvent(hEvent);
        }

        // Destroy the object itself
        SMSO_vDestroy((SMS_OBJECT)hService);

        puts(SUB_NOTIFICATION_OBJECT_NAME": Destroyed service");

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*******************************************************************************
*
*   SUB_NOTIFICATION_vMetaDataChanged
*
*******************************************************************************/
void SUB_NOTIFICATION_vMetaDataChanged(
    SUB_NOTIFICATION_OBJECT hService
        )
{
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
        (SUB_NOTIFICATION_OBJECT_STRUCT *)hService;

        BOOLEAN bOwner;

    // vfy SUB_NOTIFICATION object since this call is always made from
    // the context of the DECODER.
    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    if(bOwner == TRUE)
    {
        // we received a metadata change on the tuned channel. we
        // aren't interested in this anymore
        CHANNEL_vUnregisterNotification (
            psObj->hTunedChannel,
            vChannelMetadataEventCallback,
            psObj
                );

        bManagerFSM(psObj, SUB_NOTIFICATION_METADATA);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bManagerFSM
*
*****************************************************************************/
static BOOLEAN bManagerFSM (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj,
    SUB_NOTIFICATION_STATE_ENUM eRequestedState
        )
{
    BOOLEAN bStateChanged = TRUE;
    SUB_NOTIFICATION_STATE_ENUM eNextState;

    eNextState = psObj->eCurrentState;

    printf(SUB_NOTIFICATION_OBJECT_NAME": "
        "Current State = %s   Requested state=%s\n",
        pacSubStateText(psObj->eCurrentState),
        pacSubStateText(eRequestedState));

    switch (psObj->eCurrentState)
    {
        case SUB_NOTIFICATION_INITIAL:
        {
            switch (eRequestedState)
            {
                case SUB_NOTIFICATION_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case SUB_NOTIFICATION_WAIT:
                case SUB_NOTIFICATION_COMPLETE:
                case SUB_NOTIFICATION_NOT_REQD:
                case SUB_NOTIFICATION_TUNE:
                case SUB_NOTIFICATION_METADATA:

                default:
                    break;
            }
        }
        break;

        case SUB_NOTIFICATION_NOT_REQD:
        {
            switch (eRequestedState)
            {
                case SUB_NOTIFICATION_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case SUB_NOTIFICATION_NOT_REQD:
                case SUB_NOTIFICATION_WAIT:
                case SUB_NOTIFICATION_TUNE:
                case SUB_NOTIFICATION_METADATA:
                case SUB_NOTIFICATION_COMPLETE:

                default:
                    break;
            }
        }
        break;

        case SUB_NOTIFICATION_WAIT:
        {
            switch (eRequestedState)
            {
                case SUB_NOTIFICATION_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case SUB_NOTIFICATION_TUNE:
                    eNextState = eTransitionToTuneState(psObj);
                break;

                case SUB_NOTIFICATION_METADATA:
                    eNextState = eTransitionToMetadataState(psObj);
                break;

                case SUB_NOTIFICATION_WAIT:
                    eNextState = eTransitionToWaitState(psObj);
                break;

                case SUB_NOTIFICATION_NOT_REQD:
                case SUB_NOTIFICATION_COMPLETE:

                default:
                    break;
            }
        }
        break;

        case SUB_NOTIFICATION_TUNE:
        {
            switch (eRequestedState)
            {
                case SUB_NOTIFICATION_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case SUB_NOTIFICATION_WAIT:
                    eNextState = eTransitionToWaitState(psObj);
                break;

                case SUB_NOTIFICATION_TUNE:
                    eNextState = eRefreshTunedState(psObj);
                break;

                case SUB_NOTIFICATION_METADATA:
                case SUB_NOTIFICATION_NOT_REQD:
                case SUB_NOTIFICATION_COMPLETE:

                default:
                    break;
            }
        }
        break;

        case SUB_NOTIFICATION_METADATA:
        {
            switch (eRequestedState)
            {
                case SUB_NOTIFICATION_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case SUB_NOTIFICATION_WAIT:
                    eNextState = eTransitionToWaitState(psObj);
                break;

                case SUB_NOTIFICATION_TUNE:
                    eNextState = eTransitionToTuneState(psObj);
                break;

                case SUB_NOTIFICATION_COMPLETE:
                case SUB_NOTIFICATION_METADATA:
                case SUB_NOTIFICATION_NOT_REQD:
                default:
                    break;
            }
        }
        break;

        case SUB_NOTIFICATION_COMPLETE:
        {
            switch (eRequestedState)
            {
                case SUB_NOTIFICATION_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case SUB_NOTIFICATION_NOT_REQD:
                case SUB_NOTIFICATION_WAIT:
                case SUB_NOTIFICATION_TUNE:
                case SUB_NOTIFICATION_METADATA:
                case SUB_NOTIFICATION_COMPLETE:

                default:
                    break;
            }
        }
        break;

        default:
            break;
    }

    if (eNextState != psObj->eCurrentState)
    {
        DECODER_OBJECT hDecoder;

        bStateChanged = TRUE;
        psObj->eCurrentState = eNextState;

        // Subscription service objs. belong to DECODER object (their parent)
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);
        // have the decoder set the sub service state event
        DECODER_vGenericUpdateEventMask(hDecoder,
            DECODER_OBJECT_EVENT_SUB_NOTIFICATION_STATE);
    }

    printf(SUB_NOTIFICATION_OBJECT_NAME":Current State now = %s\n",
        pacSubStateText(psObj->eCurrentState));

    return bStateChanged;
}

/*****************************************************************************
*
*   eTransitionToInitialState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eTransitionToInitialState (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SUB_NOTIFICATION_STATE_ENUM eNextState;
    BOOLEAN bInPeriod = FALSE;

    // Period1 OR Period2 conditions exist?
    bInPeriod = bCheckPeriod(psObj);

    if (bInPeriod == TRUE)
    {
        // We are in Period1 or Period2
        BOOLEAN bStatus;

        // did we already provide this period's
        // subscription notification?
        bStatus = bAlreadyNotified(psObj, psObj->un8Period);
        if (bStatus == TRUE)
        {
            // already provided the notifcations
            eNextState = eTransitionToCompleteState(psObj);
        }
        else
        {
            // didn't provide notifications
            eNextState = eTransitionToWaitState(psObj);
        }
    }
    else
    {
        // We are not in Period1 or Period2
        eNextState = eTransitionToNotReqdState(psObj);
    }

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToWaitState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eTransitionToWaitState (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SUB_NOTIFICATION_STATE_ENUM eNextState;

    eNextState = SUB_NOTIFICATION_WAIT;

    if (psObj->eCurrentState == SUB_NOTIFICATION_INITIAL)
    {
        // If we don't already have event
        if(psObj->hMetaWaitEvent == SMS_INVALID_EVENT_HDL)
        {
            DECODER_OBJECT hDecoder =
                (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

            // Allocate an event from the DECODER, to use when this
            // timer expires.
            psObj->hMetaWaitEvent =
                DECODER_hAllocateEvent(hDecoder,
                    SMS_EVENT_METADATA_WAIT_TIMEOUT, SMS_EVENT_OPTION_NONE, NULL);
            if  (psObj->hMetaWaitEvent == SMS_INVALID_EVENT_HDL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SUB_NOTIFICATION_OBJECT_NAME
                        ": Unable to allocate meta wait event.");
                return SUB_NOTIFICATION_INVALID;
            }
        }

        // start timer. when it expires, we will start looking for metadata
        // changes on the tuned channel.
        OSAL.eTimerStartRelative(psObj->hMetadataWaitTimer,
            SUB_NOTIFICATION_META_WAIT_DURATION_MS, 0);
    }
    else if (psObj->eCurrentState == SUB_NOTIFICATION_METADATA)
    {
        // text replacement complete. Restore metadata
        CHANNEL_eClearAttribute(psObj->hTunedChannel,
            CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT);
        puts(SUB_NOTIFICATION_OBJECT_NAME": Meta Notification complete");
        psObj->bMetaNotificationDone = TRUE;
    }
    else if (psObj->eCurrentState == SUB_NOTIFICATION_TUNE)
    {
        // text replacement complete. Restore metadata
        CHANNEL_eClearAttribute(psObj->hTunedChannel,
            CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT);
        puts(SUB_NOTIFICATION_OBJECT_NAME": Tune Notification complete");
        psObj->bTuneNotificationDone = TRUE;
        eNextState = eTransitionToCompleteState(psObj);
    }
    else
    {
        // do nothing
    }

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToTuneState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eTransitionToTuneState (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SUB_NOTIFICATION_STATE_ENUM eNextState;

    if (psObj->eCurrentState == SUB_NOTIFICATION_METADATA)
    {
        eNextState = eRefreshTunedState(psObj);
    }
    else
    {
        bSubscriptionAlert(psObj);
        eNextState = SUB_NOTIFICATION_TUNE;
    }

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToMetadataState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eTransitionToMetadataState (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SUB_NOTIFICATION_STATE_ENUM eNextState = SUB_NOTIFICATION_METADATA;

    bSubscriptionAlert(psObj);

    return eNextState;
}

/*****************************************************************************
*
*   eTransitionToNotReqdState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eTransitionToNotReqdState (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    MODULE_SUBSTATUS_ENUM eStatus;
    DECODER_OBJECT hDecoder;
    BOOLEAN bOwner;

    // Grab the decoder handle
    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    // Ensure we have the right context
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        eStatus = RADIO_eSubStatus(MODULE_INVALID_OBJECT, hDecoder);

        if (eStatus == MODULE_SUBSTATUS_SUBSCRIBED)
        {
            TAG_OBJECT hParentTag, hTag;

            puts(SUB_NOTIFICATION_OBJECT_NAME": "
                 "Status MODULE_SUBSTATUS_SUBSCRIBED");

            // Get decoder tag
            hParentTag = DECODER_hGetTag(hDecoder);

            // Get Subscription Tag
            eReturn = TAG_eGet(SUB_NOTIFICATION_TAG_NAME,
                          hParentTag,
                          &hTag,
                          NULL,
                          FALSE
                              );

            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // remove tag (and all its children tags
                // (i.e. the Period Tags)
                TAG_eRemove(hTag, TRUE);
            }
        }
    }

    return SUB_NOTIFICATION_NOT_REQD;
}

/*****************************************************************************
*
*   eTransitionToCompleteState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eTransitionToCompleteState (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SUB_NOTIFICATION_STATE_ENUM eNextState;

    // need to write to config file
    if (psObj->eCurrentState != SUB_NOTIFICATION_INITIAL)
    {
        bNotified(psObj);
    }

    eNextState = SUB_NOTIFICATION_COMPLETE;

    return eNextState;
}

/*****************************************************************************
*
*   eRefreshTuneState
*
*****************************************************************************/
static SUB_NOTIFICATION_STATE_ENUM eRefreshTunedState (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SUB_NOTIFICATION_STATE_ENUM eNextState = SUB_NOTIFICATION_TUNE;
    DECODER_OBJECT hDecoder;
    CHANNEL_OBJECT hChannel;
    CHANNEL_ID tChannelId;
    CCACHE_OBJECT hCCache;

    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    tChannelId = DECODER.tLastTunedChannelId(hDecoder);
    hCCache = DECODER_hCCache(hDecoder);

    // get the channel obj for this requested chan id
    hChannel = CCACHE_hChannelFromIds (
        hCCache,
        SERVICE_INVALID_ID,
        tChannelId,
        FALSE
            );

    CHANNEL_eClearAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT);

    bMetadataReplacement(psObj);

    return eNextState;
}

/*****************************************************************************
*
*   bCheckPeriod
*
*****************************************************************************/
static BOOLEAN bCheckPeriod (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    BOOLEAN bInPeriod = FALSE, bOwner;
    DECODER_OBJECT hDecoder;
    UN8 un8Index = 0;

    // Grab the decoder handle
    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    // Verify we're in the right context
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        MODULE_SUBSTATUS_ENUM eStatus;

        // Grab the subscription status
        eStatus = RADIO_eSubStatus(MODULE_INVALID_OBJECT, hDecoder);

        printf(SUB_NOTIFICATION_OBJECT_NAME": Status: %u\n", eStatus);

        if (eStatus == MODULE_SUBSTATUS_SUSPEND_ALERT)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturn;
            UN32 un32CurrentTime;

            eOsalReturn = OSAL.eTimeGet(&un32CurrentTime);
            if (eOsalReturn == OSAL_SUCCESS)
            {
                UN32 un32UTCStartOfNotificationTime;

                // Grab the start of notification time
                un32UTCStartOfNotificationTime =
                    RADIO_un32SuspendDate(MODULE_INVALID_OBJECT, hDecoder);

                printf(SUB_NOTIFICATION_OBJECT_NAME": Start of Notification Time: %u\n",
                    un32UTCStartOfNotificationTime);

                // Check if the start of notification has occurred
                if (un32UTCStartOfNotificationTime < un32CurrentTime)
                {
                    MODULE_SUBSTATUS_REASON_CODE tReasonCode;
                    BOOLEAN bFound = FALSE;
                    UN32 un32DaysAfterNotificationStart;

                    // Start of notification has already occurred.

                    // Grab the reason code
                    tReasonCode = RADIO_tReasonCode(MODULE_INVALID_OBJECT, hDecoder);

                    printf(SUB_NOTIFICATION_OBJECT_NAME": Reason Code: %u\n",
                        tReasonCode);

                    un32DaysAfterNotificationStart =
                        (un32CurrentTime - un32UTCStartOfNotificationTime)/DAY_IN_SECONDS;
                    printf(SUB_NOTIFICATION_OBJECT_NAME": %u days from start of notification. "
                        "Reason Code: %u\n", un32DaysAfterNotificationStart,
                        tReasonCode);

                    // cycle through known reason codes to see if we get
                    // a match
                    for (un8Index=0;
                         un8Index < KNOWN_REASON_CODES;
                         un8Index++)
                    {
                        if (tReasonCode ==
                            (MODULE_SUBSTATUS_REASON_CODE)
                                gasReasonCode[un8Index].un8ReasonCode)
                        {
                            // found it. We know how to handle this code.
                            bFound = TRUE;

                            break;
                        }
                    }

                    // did we match the rxd reason code with one we know?
                    if (bFound == TRUE)
                    {
                        // Determine if we are within period 1
                        if (un32DaysAfterNotificationStart <
                            gasReasonCode[un8Index].un8Period1UpperLimit)
                        {
                            // Yes. Now we need to determine if we are in
                            // period 1 or in period 2.
                            if (gasReasonCode[un8Index].un8Period1UpperLimit >
                                   gasReasonCode[un8Index].un8Period2UpperLimit)
                            {
                                // Are we in period 1?
                                if (un32DaysAfterNotificationStart <=
                                      (UN32)(gasReasonCode[un8Index].un8Period1UpperLimit -
                                       gasReasonCode[un8Index].un8Period2UpperLimit))
                                {
                                    // We are in period 1.
                                    bInPeriod = TRUE;
                                    psObj->un8Period = 1;
                                }
                                else
                                {
                                    // Not in period 1, must be in period 2
                                    bInPeriod = TRUE;
                                    psObj->un8Period = 2;
                                }

                                printf(SUB_NOTIFICATION_OBJECT_NAME": "
                                    "Period %u.\n", psObj->un8Period);
                            }
                            else
                            {
                                // There is a problem in how the reason codes were
                                // mapped.
                                printf(SUB_NOTIFICATION_OBJECT_NAME": "
                                        "Error - Period 2 defined before period 1."
                                        "for reason code %u\n", un8Index);
                            }
                        }
                        else
                        {
                            // We are past the end of period 2.
                            // Notification no longer requried.
                            puts(SUB_NOTIFICATION_OBJECT_NAME": "
                                "Beyond sub notification period.\n");
                        }
                    }
                    else
                    {
                        printf(SUB_NOTIFICATION_OBJECT_NAME": "
                            "Don't have mapping for reason code %u.\n", tReasonCode);
                    }
                }
                else
                {
                    // start of notification takes place in the future
                    puts(SUB_NOTIFICATION_OBJECT_NAME": "
                        "Not yet in sub notification period.\n");
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SUB_NOTIFICATION_OBJECT_NAME": Couldn't get time (%u).",
                    eOsalReturn);
            }
        }
        else
        {
            puts(SUB_NOTIFICATION_OBJECT_NAME": "
                "No Subscription Alert Req'd.");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SUB_NOTIFICATION_OBJECT_NAME": "
            "Cannot get subscription status (not owner).");
    }

    if (bInPeriod == TRUE)
    {
        printf(SUB_NOTIFICATION_OBJECT_NAME": "
            "Setting Sub. Alert Text to: "
            "Artist: %s   Title: %s\n",
            gasReasonCode[un8Index].pacReasonText,
            gasReasonCode[un8Index].pacPhoneNumberText);

        // populate the CDO with the appropriate reason text and
        // phone number
        eReturn = DECODER_eSetSubscriptionAlertText(
            hDecoder,
            gasReasonCode[un8Index].pacReasonText,
            gasReasonCode[un8Index].pacPhoneNumberText);

        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                __LINE__,
                SUB_NOTIFICATION_OBJECT_NAME": "
                "Cannot set sub alert text (%u).",
                eReturn);

            // cannot do sub notiifcation if text could not be set
            bInPeriod = FALSE;
        }
    }

    return bInPeriod;
}

/*****************************************************************************
*
*   bAlreadyNotified
*
*****************************************************************************/
static BOOLEAN bAlreadyNotified (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj,
    UN8 un8Period
        )
{
    BOOLEAN bReturn = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT hParentTag = TAG_INVALID_OBJECT, hTag;
    char acName[2];
    DECODER_OBJECT hDecoder;

    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);
    hParentTag = DECODER_hGetTag(hDecoder);

    // Get Subscription Tag
    eReturn = TAG_eGet(SUB_NOTIFICATION_TAG_NAME,
                  hParentTag,
                  &hTag,
                  NULL,
                  FALSE
                      );

    if ((hTag != TAG_INVALID_OBJECT) &&
        (eReturn == SMSAPI_RETURN_CODE_SUCCESS))
    {
        // we want a child of the Subscription Tag
        // so we need to make the Subscription Tag the parent now.
        hParentTag = hTag;

        // Get Tag for this period
        snprintf(&acName[0], sizeof(acName),
                "%01d", un8Period);

        eReturn = TAG_eGet(PERIOD_TAG_NAME,
            hParentTag,
            &hTag,
            &acName[0],
            FALSE
                );

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // get tag value
            STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;
            size_t tSize;

            tSize = sizeof(STRING_OBJECT);
            phString = &hString;
            eReturn = TAG_eGetTagValue(
                hTag,
                TAG_TYPE_STRING,
                (void **)&phString,
                &tSize);

            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // convert the "TRUE" / "FALSE" text to a BOOLEAN
                eReturn = CM_eStringToBoolean(
                    hString,
                    &bReturn
                        );
                printf(SUB_NOTIFICATION_OBJECT_NAME": period tag found "
                    "with value set to: %s\n",
                    bReturn == TRUE ? "TRUE" : "FALSE");

                // we're done with this string
                STRING.vDestroy(hString);
            }
        }
        else
        {
            puts(SUB_NOTIFICATION_OBJECT_NAME": period tag not found");
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bNotified
*
*****************************************************************************/
static BOOLEAN bNotified (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bReturn = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT hParentTag = TAG_INVALID_OBJECT, hTag;
    char acName[2];
    DECODER_OBJECT hDecoder;

    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);
    hParentTag = DECODER_hGetTag(hDecoder);

    // Get Subscription Tag
    eReturn = TAG_eGet(SUB_NOTIFICATION_TAG_NAME,
                  hParentTag,
                  &hTag,
                  NULL,
                  TRUE // create if not found
                      );

    if (hTag != TAG_INVALID_OBJECT)
    {
        // we want a child of the Subscription Tag
        // so we need to make the Subscription Tag the parent now.
        hParentTag = hTag;

        // Get Tag for this period
        snprintf(&acName[0], sizeof(acName),
                "%01d", psObj->un8Period);

         eReturn = TAG_eGet(PERIOD_TAG_NAME,
                 hParentTag,
                 &hTag,
                 &acName[0],
                 TRUE // create if not found
                     );

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            STRING_OBJECT hString;

            // get the appropriate string that we'll use to set the
            // tag value
            hString = CM_hTrueString();

            // set the tag value
            eReturn = TAG_eSetTagValue(
                        hTag,
                        TAG_TYPE_STRING,
                        (void *)&hString,
                        sizeof(STRING_OBJECT),
                        TRUE); // commit the file
            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {
                bReturn = TRUE;
            }
        }
    }

    if (bReturn == FALSE)
    {
        printf(SUB_NOTIFICATION_OBJECT_NAME": Failed to set Tag for "
            "period %u!\n", psObj->un8Period);
    }

    return bReturn;
}

/*****************************************************************************
*
*   pacSubStateText
*
*****************************************************************************/
const char *pacSubStateText (
    SUB_NOTIFICATION_STATE_ENUM eState
        )
{
    const char *pacReturnString;

    switch (eState)
    {
        case SUB_NOTIFICATION_NOT_REQD:
            pacReturnString = MACRO_TO_STRING(SUB_NOTIFICATION_NOT_REQD);
        break;

        case SUB_NOTIFICATION_INITIAL:
            pacReturnString = MACRO_TO_STRING(SUB_NOTIFICATION_INITIAL);
        break;

        case SUB_NOTIFICATION_WAIT:
            pacReturnString = MACRO_TO_STRING(SUB_NOTIFICATION_WAIT);
        break;

        case SUB_NOTIFICATION_TUNE:
            pacReturnString = MACRO_TO_STRING(SUB_NOTIFICATION_TUNE);
        break;

        case SUB_NOTIFICATION_METADATA:
            pacReturnString = MACRO_TO_STRING(SUB_NOTIFICATION_METADATA);
        break;

        case SUB_NOTIFICATION_COMPLETE:
            pacReturnString = MACRO_TO_STRING(SUB_NOTIFICATION_COMPLETE);
        break;

        default:
            pacReturnString = "Unknown";
            break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   vTextReplacementTimerCallback
*
*****************************************************************************/
static void vTextReplacementTimerCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    BOOLEAN bPosted;
    SMS_EVENT_HDL hEvent;

    hEvent = *((SMS_EVENT_HDL *)pvArg);
    *((SMS_EVENT_HDL *)pvArg) = SMS_INVALID_EVENT_HDL;

    // post event to decoder.
    bPosted = SMSE_bPostEvent(hEvent);
    if(bPosted == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SUB_NOTIFICATION_OBJECT_NAME": Can't post txt replacement event.");
    }

    return;
}

/*****************************************************************************
*
*   vMetadataWaitTimerCallback
*
*****************************************************************************/
static void vMetadataWaitTimerCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    BOOLEAN bPosted;
    SMS_EVENT_HDL hEvent;

    hEvent = *((SMS_EVENT_HDL *)pvArg);
    *((SMS_EVENT_HDL *)pvArg) = SMS_INVALID_EVENT_HDL;

    // post event to decoder.
    bPosted = SMSE_bPostEvent(hEvent);
    if(bPosted == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SUB_NOTIFICATION_OBJECT_NAME": Cannot post meta wait event.");
    }

    return;
}

/*****************************************************************************
*
*   bSubscriptionAlert
*
*****************************************************************************/
static BOOLEAN bSubscriptionAlert (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    DECODER_OBJECT hDecoder;
    BOOLEAN bReturn;

    // DECODER is the parent object
    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    // If we don't already have event
    if(psObj->hTextReplacementEvent == SMS_INVALID_EVENT_HDL)
    {
        // Allocate an event from the DECODER, to use when this
        // timer expires.
        psObj->hTextReplacementEvent =
            DECODER_hAllocateEvent(hDecoder,
                SMS_EVENT_TEXT_REPLACEMENT_TIMEOUT, SMS_EVENT_OPTION_NONE, NULL);
        if  (psObj->hTextReplacementEvent == SMS_INVALID_EVENT_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SUB_NOTIFICATION_OBJECT_NAME
                    ": Unable to allocate text replacement event.");
            return FALSE;
        }
    }

    // start timer
    OSAL.eTimerStartRelative(psObj->hTextReplaceTimer,
        SUB_NOTIFICATION_TEXT_REPLACE_DURATION_MS, 0);

    puts(SUB_NOTIFICATION_OBJECT_NAME": Beep Beep Beep. "
        "Start Subscription Alert");

    DECODER_eAlertTone(hDecoder, 0);

    bReturn = bMetadataReplacement(psObj);

    return bReturn;
}

/*****************************************************************************
*
*   bMetadataReplacement
*
*****************************************************************************/
static BOOLEAN bMetadataReplacement (
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bReturn = TRUE;
    DECODER_OBJECT hDecoder;
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT hChannel;
    SERVICE_ID tServiceId;

    // start timer

    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    hCCache = DECODER_hCCache(hDecoder);

    tServiceId = DECODER.tCurrentServiceId(hDecoder);

    // get the channel obj for this requested chan id
    hChannel = CCACHE_hChannelFromIds (
        hCCache,
        tServiceId,
        CHANNEL_INVALID_ID,
        FALSE
            );

    // set attribute to the tuned channel to indicate its text needs to be
    // replaced with sub notification
    CHANNEL_eSetAttribute(hChannel, CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT);

    return bReturn;
}

/*******************************************************************************
*
*   vChannelMetaEventCallback
*
*   This callback function is used to register with all CHANNEL objects in
*   that are in the CHANNELLIST.  Since metadata doesn't affect the makeup of the
*   CHANNELLIST, this function simply records the event changes and notifies the
*   Application if any changes they care about were made.
*
*   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
*
*******************************************************************************/
static void vChannelMetadataEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    SUB_NOTIFICATION_OBJECT_STRUCT *psObj =
            (SUB_NOTIFICATION_OBJECT_STRUCT *)pvEventCallbackArg;
    DECODER_OBJECT hDecoder;
    SMS_EVENT_HDL hEvent;

    hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    puts(SUB_NOTIFICATION_OBJECT_NAME": "
        "Metadata changed on the tuned channel");

    // Allocate an event from the DECODER and post
    hEvent =
        DECODER_hAllocateEvent(hDecoder,
            SMS_EVENT_METADATA_CHANGED, SMS_EVENT_OPTION_NONE, NULL);
    if  (hEvent != SMS_INVALID_EVENT_HDL)
    {
        BOOLEAN bPosted;

        bPosted = SMSE_bPostEvent(hEvent);
        if(bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SUB_NOTIFICATION_OBJECT_NAME": Cannot post meta wait event.");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SUB_NOTIFICATION_OBJECT_NAME
                ": Unable to allocate meta data event.");
    }

    return;
}
