/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:STOCK_SYMBOL implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"

#include "_stock_symbol_obj.h"
#include "stock_symbol_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   hCreate
*
*****************************************************************************/
static STOCK_SYMBOL_OBJECT hCreate (
    STRING_OBJECT hSymbol
        )
{
    STOCK_SYMBOL_OBJECT hResult = STOCK_SYMBOL_INVALID_OBJECT;
    STOCK_SYMBOL_OBJECT_STRUCT *psObj = NULL;

    printf(STOCK_SYMBOL_OBJECT_NAME": creating object by Name\n");

    do
    {
        // Check input
        if (hSymbol == STRING_INVALID_OBJECT)
        {
            break;
        }

        psObj = psCreateObject(SMS_INVALID_OBJECT);
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_SYMBOL_OBJECT_NAME": failed to create obj");
            break;
        }

        // Copy name string
        psObj->hName = STRING_hDuplicate(SMS_INVALID_OBJECT, hSymbol);
        if (psObj->hName == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_SYMBOL_OBJECT_NAME": failed to duplicate STRING object");
            break;
        }

        hResult = (STOCK_SYMBOL_OBJECT) psObj;
    } while (FALSE);

    if ((hResult == STOCK_SYMBOL_INVALID_OBJECT) && (psObj != NULL))
    {
        vObjectDestroy(psObj);
    }

    return hResult;
}

/*****************************************************************************
*
*   n32FWrite
*
*****************************************************************************/
static N32 n32FWrite (
    STOCK_SYMBOL_OBJECT hTarget,
    FILE *psFile
        )
{
    STOCK_SYMBOL_OBJECT_STRUCT *psObj =
        (STOCK_SYMBOL_OBJECT_STRUCT *)hTarget;
    N32 n32Return = 0;
    BOOLEAN bOwner;

    // Verify inputs
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget);
    if((bOwner == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Write STOCK to file...
    n32Return +=
            STRING.n32FWrite(psObj->hName, psFile);

    return n32Return;
}

/*****************************************************************************
*
*   hFRead
*
*****************************************************************************/
static STOCK_SYMBOL_OBJECT hFRead (
    FILE *psFile
        )
{
    STOCK_SYMBOL_OBJECT_STRUCT *psObj = NULL;

    do
    {
        // Verify input
        if(psFile == NULL)
        {
            break;
        }

        // Create object to read
        psObj = psCreateObject(SMS_INVALID_OBJECT);
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_SYMBOL_OBJECT_NAME": failed to create object");
            break;
        }

        // Read STOCK from file...

        // Read string
        psObj->hName = STRING.hFRead(psFile);
        if (psObj->hName == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_SYMBOL_OBJECT_NAME": failed to read Stock Symbol");
            break;
        }

        return (STOCK_SYMBOL_OBJECT) psObj;
    } while (FALSE);

    if (psObj != NULL)
    {
        vObjectDestroy(psObj);
    }

    return STOCK_SYMBOL_INVALID_OBJECT;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    STOCK_SYMBOL_OBJECT hTarget,
    FILE *psFile
        )
{
    STOCK_SYMBOL_OBJECT_STRUCT *psObj =
        (STOCK_SYMBOL_OBJECT_STRUCT *)hTarget;
    N32 n32Return = 0;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget);
    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bOwner == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    n32Return += fprintf(psFile,"Stock:");
    n32Return += STRING.n32FPrintf(psObj->hName, psFile);

    return n32Return;
}

/*****************************************************************************
*
*   vDestroy
*
*****************************************************************************/
static void vDestroy (
    STOCK_SYMBOL_OBJECT hTarget
        )
{
    BOOLEAN bOwner;

    printf(STOCK_SYMBOL_OBJECT_NAME": destroying object 0x%p\n", hTarget);

    // Verify inputs. Object handle must be valid.
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    if (hTarget != STOCK_SYMBOL_INVALID_OBJECT)
    {
        vObjectDestroy((STOCK_SYMBOL_OBJECT_STRUCT*) hTarget);
    }

    return;
}

/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   STOCK_SYMBOL_hDuplicate
*
*****************************************************************************/
STOCK_SYMBOL_OBJECT STOCK_SYMBOL_hDuplicate (
    SMS_OBJECT hOwner,
    STOCK_SYMBOL_OBJECT hTarget
        )
{
    BOOLEAN bOwner;
    STOCK_SYMBOL_OBJECT_STRUCT *psObj = NULL;
    STOCK_SYMBOL_OBJECT_STRUCT *psArgObj = NULL;

    do
    {
        // Verify inputs
        bOwner = SMSO_bOwner(hOwner);
        if ((hOwner != SMS_INVALID_OBJECT) && (bOwner == FALSE ))
        {
            // Error!
            break;
        }

        bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget);
        if(bOwner == FALSE )
        {
            // Error!
            break;
        }

        psArgObj = (STOCK_SYMBOL_OBJECT_STRUCT*)hTarget;
        psObj = psCreateObject(hOwner);
        if (psObj == NULL)
        {
            break;
        }

        psObj->hName = STRING_hDuplicate(hOwner, psArgObj->hName);
        if (psObj->hName == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_SYMBOL_OBJECT_NAME": failed to duplicate string"
                    );
            break;
        }

        return (STOCK_SYMBOL_OBJECT) psObj;

    } while (FALSE);

    if (psObj != NULL)
    {
        vObjectDestroy(psObj);
    }

    return STOCK_SYMBOL_INVALID_OBJECT;
}

/*****************************************************************************
*
*   STOCK_SYMBOL_hName
*
*****************************************************************************/
STRING_OBJECT STOCK_SYMBOL_hName (
    STOCK_SYMBOL_OBJECT hTarget
        )
{
    BOOLEAN bOwner;
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;

    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget);
    if (bOwner == TRUE)
    {
        STOCK_SYMBOL_OBJECT_STRUCT *psObj =
                (STOCK_SYMBOL_OBJECT_STRUCT*) hTarget;

        hResult = psObj->hName;
    }
#if SMS_DEBUG == 1
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_SYMBOL_OBJECT_NAME": object not an owner");
    }
#endif

    return hResult;
}

/*****************************************************************************
*
*   STOCK_SYMBOL_n16Compare
*
*****************************************************************************/
N16 STOCK_SYMBOL_n16Compare (
    STOCK_SYMBOL_OBJECT hTarget1,
    STOCK_SYMBOL_OBJECT hTarget2
        )
{
    N16 n16Result = N16_MIN;
    BOOLEAN bOwner1, bOwner2;
	STOCK_SYMBOL_OBJECT_STRUCT *psObj1 = (STOCK_SYMBOL_OBJECT_STRUCT*)hTarget1;
	STOCK_SYMBOL_OBJECT_STRUCT *psObj2 = (STOCK_SYMBOL_OBJECT_STRUCT*)hTarget2;

    bOwner1 = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget1);
    bOwner2 = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget2);

    if ((bOwner1 == TRUE) && (bOwner2 == TRUE))
    {
        n16Result = STRING.n16Compare(psObj1->hName, psObj2->hName, TRUE);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_SYMBOL_OBJECT_NAME": object(s) not owned");
    }

    return n16Result;

}

/*****************************************************************************
*
*   STOCK_SYMBOL_hCreateTargetFromTag
*
*   Reads a tag and generates a DSRL_TARGET_OBJECT based on the
*   tags contents
*
*****************************************************************************/
DSRL_TARGET_OBJECT STOCK_SYMBOL_hCreateTargetFromTag (
    TAG_OBJECT hTag
        )
{
    DSRL_TARGET_OBJECT hTarget = DSRL_TARGET_INVALID_OBJECT;
    size_t tSize;
    STRING_OBJECT hTagName;

    // get this tag's name
    hTagName = TAG_hTagName(hTag);

    // is this a tag we're interested in?
    if (STRING.n16CompareCStr(STOCK_SYMBOL_TAG_NAME, 0, hTagName) == 0)
    {
        // yes, we're interested in this tag
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;

        tSize = sizeof(STRING_OBJECT);
        phString = &hString;
        eReturnCode = TAG_eGetTagValue(hTag, TAG_TYPE_STRING,
            (void **)&phString, &tSize);

        // We were able to get the LocId.  Create
        // the location object now
        if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            STOCK_SYMBOL_OBJECT hStock;

            // Create the Stock Ticker index based object
            hStock = STOCK_SYMBOL.hCreate(hString);

            // Provide it to the caller as a target object
            hTarget = (DSRL_TARGET_OBJECT)hStock;
        }

        // Destroy if exists
        if (hString != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(hString);
        }
    }

    return hTarget;
}

/*****************************************************************************
*
*   STOCK_SYMBOL_hGetTargetTag
*
*   Gets a TAG_OBJECT for the stock represented by the DSRL_TARGET_OBJECT
*   provided.  If the tag doesn't exist, this function creates that tag
*   and sets the value.
*
*****************************************************************************/
TAG_OBJECT STOCK_SYMBOL_hGetTargetTag (
    TAG_OBJECT hParentTag,
    DSRL_TARGET_OBJECT hTarget
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    STOCK_SYMBOL_OBJECT hStock;
    DSRL_TARGET_TYPE_ENUM eType;
    STOCK_SYMBOL_TAG_SEARCH_STRUCT sSearch;

    // Initialize the search structure
    sSearch.sInput.hSymbol = STRING_INVALID_OBJECT;
    sSearch.sOutput.hTag = TAG_INVALID_OBJECT;

    // Verify target type
    eType = DSRL_TARGET.eType(hTarget);
    if (eType != DSRL_TARGET_TYPE_STOCK_SYMBOL)
    {
        // We can't use this type of target
        return TAG_INVALID_OBJECT;
    }

    // Extract the target info
    hStock = DSRL_TARGET.hStockSymbol(hTarget);
    sSearch.sInput.hSymbol = STOCK_SYMBOL_hName(hStock);
    if (sSearch.sInput.hSymbol == STRING_INVALID_OBJECT)
    {
        // This isn't a target which can have a tag
        return TAG_INVALID_OBJECT;
    }

    // Iterate the child tags to locate the
    // tag for this location
    eReturnCode = TAG_eIterateChildren(
        hParentTag,
        (TAG_ITERATION_HANDLER)bFindStockTag,
        &sSearch);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error!
        return TAG_INVALID_OBJECT;
    }

    // Did our search find the appropriate tag?
    if (sSearch.sOutput.hTag == TAG_INVALID_OBJECT)
    {
        // No -- Attempt to add the tag now
        eReturnCode = TAG_eAdd(
            STOCK_SYMBOL_TAG_NAME,
            hParentTag, &sSearch.sOutput.hTag, NULL);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            TAG_eSetTagValue( sSearch.sOutput.hTag,
                TAG_TYPE_STRING,
                &sSearch.sInput.hSymbol,
                sizeof(STRING_OBJECT), TRUE);
        }
    }

    return sSearch.sOutput.hTag;
}

/*****************************************************************************
 *                           PRIVATE FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   psCreateObject
*
*****************************************************************************/
static STOCK_SYMBOL_OBJECT_STRUCT *psCreateObject(
    SMS_OBJECT hParent
        )
{
    STOCK_SYMBOL_OBJECT_STRUCT *psObj = NULL;

    psObj = (STOCK_SYMBOL_OBJECT_STRUCT*)
            DSRL_TARGET_hCreate(
                    STOCK_SYMBOL_OBJECT_NAME":Obj",
                    DSRL_TARGET_TYPE_STOCK_SYMBOL,
                    sizeof(STOCK_SYMBOL_OBJECT_STRUCT),
                    hParent,
                    FALSE
                        );
    if (psObj != NULL)
    {
        psObj->hName = STRING_INVALID_OBJECT;
    }

    return psObj;
}

/*****************************************************************************
*
*   bFindStockTag
*
*****************************************************************************/
static BOOLEAN bFindStockTag (
    TAG_OBJECT hTag,
    STOCK_SYMBOL_TAG_SEARCH_STRUCT *psSearch
        )
{
    STRING_OBJECT hTagName;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    size_t tSize;
    BOOLEAN bResult = TRUE;

    // get this tag's name
    hTagName = TAG_hTagName(hTag);

   if (STRING.n16CompareCStr(STOCK_SYMBOL_TAG_NAME, 0, hTagName) == 0)
   {
       STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;

       tSize = sizeof(STRING_OBJECT);
       phString = &hString;
       eReturnCode = TAG_eGetTagValue(hTag, TAG_TYPE_STRING,
           (void **)&phString, &tSize);
       if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
       {
           N16 n16Result = 0;

           n16Result = STRING.n16Compare(
               psSearch->sInput.hSymbol, hString, TRUE);

           if (n16Result == 0)
           {
               psSearch->sOutput.hTag = hTag;

               // Stop here
               bResult = FALSE;
           }
       }

       // Destroy string if exists
       if (hString != STRING_INVALID_OBJECT)
       {
           STRING_vDestroy(hString);
       }
   }

    // keep iterating
    return bResult;
}

/*****************************************************************************
*
*   vObjectDestroy
*
*****************************************************************************/
static void vObjectDestroy(
    STOCK_SYMBOL_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        if (psObj->hName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hName);
            psObj->hName = STRING_INVALID_OBJECT;
        }

        DSRL_TARGET_vDestroy((DSRL_TARGET_OBJECT) psObj);
    }
}

#ifdef SUPPORT_CUNIT
#include <stock_symbol_obj.cunit>
#endif
