/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:STOCK_MSG implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"

#include "stock_msg_obj.h"
#include "_stock_msg_obj.h"

#include "stock_ticker_interface.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   hSymbolName
*
*****************************************************************************/
static STRING_OBJECT hSymbolName (
    STOCK_MSG_OBJECT hMsg
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;
    STOCK_MSG_OBJECT_STRUCT *psObj = NULL;

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if (bOwner == TRUE)
    {
        psObj = (STOCK_MSG_OBJECT_STRUCT*) hMsg;
        hResult = psObj->hSymbolName;
    }
#if SMS_DEBUG == 1
    else
    {
        printf(STOCK_MSG_OBJECT_NAME": [%d] the object is not owner\n", __LINE__);
    }
#endif

    return hResult;
}

/*****************************************************************************
*
*   eValues
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eValues (
    STOCK_MSG_OBJECT hMsg,
    UN32*pun32Price,
    UN8 *pun8PriceFrac,
    UN32 *pun32PriceDelta,
    UN8 *pun8PriceDeltaFrac,
    STOCK_MSG_PRICE_DIRECTION_ENUM *peDirection
       )
{
    BOOLEAN bOwner;
    STOCK_MSG_OBJECT_STRUCT *psObj = NULL;

    if ((pun32Price == NULL) || (pun8PriceFrac == NULL) ||
        (pun32PriceDelta == NULL) || (pun8PriceDeltaFrac == NULL) ||
        (peDirection == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if (bOwner == FALSE)
    {
        printf(STOCK_MSG_OBJECT_NAME": [%d] the object is not owner\n", __LINE__);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    psObj = (STOCK_MSG_OBJECT_STRUCT*) hMsg;

    if (psObj->eState == STOCK_MSG_STATE_PRICE_AVAILABLE)
    {
        *pun32Price = psObj->sQuote.un32Price;
        *pun8PriceFrac = psObj->sQuote.un8PriceFrac;
        *pun32PriceDelta = psObj->sQuote.un32PriceDelta;
        *pun8PriceDeltaFrac = psObj->sQuote.un8PriceDeltaFrac;
        *peDirection = psObj->sQuote.eDir;
        return SMSAPI_RETURN_CODE_SUCCESS;
    }
    else if ((psObj->eState == STOCK_MSG_STATE_DELISTED) ||
             (psObj->eState == STOCK_MSG_STATE_PRICE_NOT_AVAILABLE))
    {
        *pun32Price = 0;
        *pun8PriceFrac = 0;
        *pun32PriceDelta = 0;
        *pun8PriceDeltaFrac = 0;
        *peDirection = STOCK_MSG_PRICE_DIRECTION_INVALID;
        return SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*   hStockSymbol
*
*****************************************************************************/
static STOCK_SYMBOL_OBJECT hStockSymbol (
    STOCK_MSG_OBJECT hMsg
       )
{
    STOCK_SYMBOL_OBJECT hResult = STOCK_SYMBOL_INVALID_OBJECT;
    BOOLEAN bOwner;
    STOCK_MSG_OBJECT_STRUCT *psObj;

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if (bOwner == TRUE)
    {
        psObj = (STOCK_MSG_OBJECT_STRUCT*) hMsg;
        if (psObj->hStockSymbol == STOCK_SYMBOL_INVALID_OBJECT)
        {
            // LAZY: If the target object is not here - create it.
            psObj->hStockSymbol = STOCK_SYMBOL.hCreate(psObj->hSymbolName);
            if (psObj->hStockSymbol == STOCK_SYMBOL_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_MSG_OBJECT_NAME": failed to create target object");
            }
        }

        hResult = psObj->hStockSymbol;
    }
#if SMS_DEBUG == 1
    else
    {
        printf(STOCK_MSG_OBJECT_NAME": [%d] the object is not owner\n", __LINE__);
    }
#endif

    return hResult;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    STOCK_MSG_OBJECT hMsg,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
       )
{
    N32 n32Result = 0;
    BOOLEAN bOwner;
    STOCK_MSG_OBJECT_STRUCT *psObj = (STOCK_MSG_OBJECT_STRUCT*) hMsg;

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // If owner is checked, then object is valid.
    // Checking for SYMBOL name
    if (psObj->hSymbolName == STRING_INVALID_OBJECT)
    {
        return EOF;
    }

    n32Result += fprintf(psFile, "Name: ");
    // We already checked STRING object and psFile, so do not expect
    // EOF returns here.
    n32Result += STRING.n32FPrintf(psObj->hSymbolName, psFile);

    n32Result += fprintf(psFile, "Index: %d\n", psObj->tIndex);
    n32Result += fprintf(psFile, "Price: %d.%u (%c%d.%u)\n",
                                psObj->sQuote.un32Price,
                                psObj->sQuote.un8PriceFrac,
                                ((psObj->sQuote.eDir == STOCK_MSG_PRICE_DIRECTION_DOWN) ? '-' : '+'),
                                psObj->sQuote.un32PriceDelta,
                                psObj->sQuote.un8PriceDeltaFrac
                                    );
    return n32Result;
}

/*****************************************************************************
 *                             FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   STOCK_MSG_eCreate
*
*****************************************************************************/
STOCK_TICKER_RETURN_CODE_ENUM STOCK_MSG_eCreate(
    STOCK_MSG_OBJECT *phMsg,
    SMS_OBJECT hParent,
    STOCK_MSG_DATA_STRUCT *psData,
    size_t tDescriptorSize
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult;
    STOCK_MSG_OBJECT_STRUCT *psObj = NULL;

    do
    {
        // Check input
        if ((hParent == SMS_INVALID_OBJECT) ||
            (phMsg == NULL) ||
            (psData == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Index must be set during creation
        if ((psData->tMask & STOCK_MSG_DATA_INDEX) != STOCK_MSG_DATA_INDEX)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init output object
        *phMsg = STOCK_MSG_INVALID_OBJECT;

        // Create object
        eResult = eCreateObject(&psObj, hParent, tDescriptorSize, psData->tIndex);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_MSG_OBJECT_NAME": failed to create object"
                        );
            break;
        }

        eResult = eUpdateData(psObj, psData);
        if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
            (eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_MSG_OBJECT_NAME": failed to perform object update during creation (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            break;
        }

        *phMsg = (STOCK_MSG_OBJECT) psObj;
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        vDestroyObject(psObj);
        psObj = NULL;
    }

    return eResult;
}

/*****************************************************************************
*
*   STOCK_MSG_eCreateDummy
*
*****************************************************************************/
STOCK_TICKER_RETURN_CODE_ENUM STOCK_MSG_eCreateDummy(
    STOCK_MSG_OBJECT *phMsg,
    SMS_OBJECT hParent
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_MSG_OBJECT_STRUCT* psObj = NULL;

    do
    {
        BOOLEAN bOwner;

        // Check input
        bOwner = SMSO_bOwner(hParent);
        if (bOwner == FALSE)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (phMsg == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init output object
        *phMsg = STOCK_MSG_INVALID_OBJECT;

        // Create object
        psObj = (STOCK_MSG_OBJECT_STRUCT *)
            DSRL_ENTRY_hCreate(STOCK_MSG_OBJECT_NAME":DummyObj",
            DSRL_ENTRY_TYPE_STOCK_MSG,
            sizeof(STOCK_MSG_OBJECT_STRUCT),
            0,
            hParent,
            FALSE
                );
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_MSG_OBJECT_NAME": failed to create object"
                        );
            break;
        }

        *phMsg = (STOCK_MSG_OBJECT)psObj;
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
    {
        vDestroyObject(psObj);
        psObj = NULL;
    }

    return eResult;
}

/*****************************************************************************
*
*   STOCK_MSG_eSetIndex
*
*****************************************************************************/
STOCK_TICKER_RETURN_CODE_ENUM STOCK_MSG_eSetIndex(
    STOCK_MSG_OBJECT hMsg,
    STOCK_ID tIndex
        )
{
    ((STOCK_MSG_OBJECT_STRUCT *)hMsg)->tIndex = tIndex;

    return STOCK_TICKER_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   STOCK_MSG_bHasName
*
*****************************************************************************/
BOOLEAN STOCK_MSG_bHasName(
    STOCK_MSG_OBJECT hMsg
        )
{
    BOOLEAN bResult;

    // Verify ownership of this object
    bResult = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if (bResult == TRUE)
    {
        STOCK_MSG_OBJECT_STRUCT *psObj = (STOCK_MSG_OBJECT_STRUCT*)hMsg;
        bResult = (psObj->hSymbolName != STRING_INVALID_OBJECT) ? TRUE : FALSE;
    }

    return bResult;
}

/*****************************************************************************
*
*   STOCK_MSG_eUpdate
*
*****************************************************************************/
STOCK_TICKER_RETURN_CODE_ENUM STOCK_MSG_eUpdate(
    STOCK_MSG_OBJECT hMsg,
    STOCK_MSG_DATA_STRUCT *psData
        )
{
    BOOLEAN bOwner;
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_MSG_OBJECT_STRUCT *psObj;

    do
    {
        // Check input
        if (psData == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Verify ownership of this object
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
        if (bOwner == FALSE)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Cast to get entity
        psObj = (STOCK_MSG_OBJECT_STRUCT*) hMsg;

        // Call unprotected implementation
        eResult = eUpdateData(psObj, psData);

    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   STOCK_MSG_tIndex
*
*   This function doesn't perform ownership check since it is assumed to
*   be called under parent lock.
*
*****************************************************************************/
STOCK_ID STOCK_MSG_tIndex (
    STOCK_MSG_OBJECT hMsg
       )
{
    STOCK_ID tResult;

    tResult = ((STOCK_MSG_OBJECT_STRUCT *)hMsg)->tIndex;

    return tResult;
}

/*****************************************************************************
*
*   STOCK_MSG_eState
*
*****************************************************************************/
STOCK_MSG_STATE_ENUM STOCK_MSG_eState (
    STOCK_MSG_OBJECT hMsg
       )
{
    STOCK_MSG_STATE_ENUM eResult = STOCK_MSG_STATE_INVALID;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if (bOwner == TRUE)
    {
        STOCK_MSG_OBJECT_STRUCT *psObj;
        psObj = (STOCK_MSG_OBJECT_STRUCT*) hMsg;
        eResult = psObj->eState;
    }
#if SMS_DEBUG == 1
    else
    {
        printf(STOCK_MSG_OBJECT_NAME": [%d] the object is not owner\n", __LINE__);
    }
#endif

    return eResult;
}

/*****************************************************************************
*
*   STOCK_MSG_bGoodToPopulate
*
*****************************************************************************/
BOOLEAN STOCK_MSG_bGoodToPopulate(
    STOCK_MSG_OBJECT hMsg
        )
{
    BOOLEAN bResult = FALSE;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if (bOwner == TRUE)
    {
        STOCK_MSG_OBJECT_STRUCT *psObj;

        psObj = (STOCK_MSG_OBJECT_STRUCT*) hMsg;

        if (psObj->eState != STOCK_MSG_STATE_INVALID)
        {
            bResult = TRUE;
        }
    }
#if SMS_DEBUG == 1
    else
    {
        printf(STOCK_MSG_OBJECT_NAME": [%d] the object is not owner\n", __LINE__);
    }
#endif

    return bResult;
}

/*****************************************************************************
*
*   STOCK_MSG_vDestroy
*
*****************************************************************************/
void STOCK_MSG_vDestroy (
    STOCK_MSG_OBJECT hMsg
        )
{
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hMsg);
    if (bOwner == TRUE)
    {
        STOCK_MSG_OBJECT_STRUCT *psObj =
            (STOCK_MSG_OBJECT_STRUCT*)hMsg;

        vDestroyObject(psObj);
    }
#if SMS_DEBUG == 1
    else
    {
        printf(STOCK_MSG_OBJECT_NAME": [%d] the object is not owner\n", __LINE__);
    }
#endif

    return;
}

/*****************************************************************************
 *                            PRIVATE FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   eCreateObject
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eCreateObject(
    STOCK_MSG_OBJECT_STRUCT **ppsObj,
    SMS_OBJECT hParent,
    size_t tDescriptorSize,
    STOCK_ID tIndex
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_MSG_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bValidParent = FALSE;

    do
    {
        // Check input
        if (ppsObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Init object
        *ppsObj = NULL;

        // Validate parent
        bValidParent = SMSO_bOwner(hParent);
        if (bValidParent == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_MSG_OBJECT_NAME": invalid parent provided");
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Create object
        psObj = (STOCK_MSG_OBJECT_STRUCT*)
                DSRL_ENTRY_hCreate(STOCK_MSG_OBJECT_NAME":Obj",
                        DSRL_ENTRY_TYPE_STOCK_MSG,
                        sizeof(STOCK_MSG_OBJECT_STRUCT),
                        tDescriptorSize,
                        hParent,
                        FALSE
                            );
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_MSG_OBJECT_NAME": failed to create object"
                        );
            eResult = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
            break;
        }

        // Init data
        OSAL.bMemSet(psObj, 0, sizeof(*psObj));
        psObj->tIndex = tIndex;
        psObj->hSymbolName = STRING_INVALID_OBJECT;
        psObj->sQuote.un32Price = 0;
        psObj->sQuote.un8PriceFrac = 0;
        psObj->sQuote.un32PriceDelta = 0;
        psObj->sQuote.un8PriceDeltaFrac = 0;
        psObj->sQuote.eDir = STOCK_MSG_PRICE_DIRECTION_INVALID;
        psObj->eState = STOCK_MSG_STATE_INVALID;
        psObj->hStockSymbol = STOCK_SYMBOL_INVALID_OBJECT;

        // Pre-create string object for name
        psObj->hSymbolName = STRING_hCreate(hParent, "",
                            GsStockTickerIntf.tSymbolNameMaxSize + 1, 0
                                );
        if (psObj->hSymbolName == STRING_INVALID_OBJECT)
        {
            eResult = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
            break;
        }

        *ppsObj = psObj;
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) && (psObj != NULL))
    {
        STOCK_MSG_vDestroy((STOCK_MSG_OBJECT) psObj);
    }

    return eResult;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(
    STOCK_MSG_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        printf(STOCK_MSG_OBJECT_NAME": destroying object 0x%p\n", psObj);

        if (psObj->hSymbolName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hSymbolName);
            psObj->hSymbolName = STRING_INVALID_OBJECT;
        }

        if (psObj->hStockSymbol != STOCK_SYMBOL_INVALID_OBJECT)
        {
            STOCK_SYMBOL.vDestroy(psObj->hStockSymbol);
            psObj->hStockSymbol = STOCK_SYMBOL_INVALID_OBJECT;
        }

        DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT) psObj);
    }

    return;
}

/*****************************************************************************
*
*   eUpdateData
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateData (
    STOCK_MSG_OBJECT_STRUCT *psObj,
    STOCK_MSG_DATA_STRUCT *psData
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bUpdated = FALSE;

    do
    {
        // Check input
        if ((psData == NULL) || (psObj == NULL) ||
            ((psData->tMask & STOCK_MSG_DATA_NAME_ALL) == STOCK_MSG_DATA_NAME_ALL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        if ((psData->tMask & STOCK_MSG_DATA_INDEX) == STOCK_MSG_DATA_INDEX)
        {
            eResult = eUpdateIndex(psObj, psData->tIndex);
            if ((eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE) &&
                (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_MSG_OBJECT_NAME": failed to perform c-str name update (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }

            if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                bUpdated = TRUE;
            }
        }

        if ((psData->tMask & STOCK_MSG_DATA_NAME_CSTR) == STOCK_MSG_DATA_NAME_CSTR)
        {
            eResult = eUpdateNameCStr(psObj, psData->uName.sCStr.pacValue,
                                psData->uName.sCStr.tValueLength);
            if ((eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE) &&
                (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_MSG_OBJECT_NAME": failed to perform c-str name update (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }

            if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                bUpdated = TRUE;
            }
        }

        if ((psData->tMask & STOCK_MSG_DATA_NAME) == STOCK_MSG_DATA_NAME)
        {
            eResult = eUpdateName(psObj, psData->uName.hValue);
            if ((eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE) &&
                (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_MSG_OBJECT_NAME": failed to perform STRING name update (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }

            if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                bUpdated = TRUE;
            }
        }

        if ((psData->tMask & STOCK_MSG_DATA_VALUES) == STOCK_MSG_DATA_VALUES)
        {
            eResult = eUpdateValues(psObj, psData->sValues.eState,
                                    &psData->sValues.sQuote);
            if ((eResult != STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE) &&
                (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_MSG_OBJECT_NAME": failed to perform values update (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }

            if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                bUpdated = TRUE;
            }
        }

        // Define return code
        eResult = (bUpdated == TRUE) ? STOCK_TICKER_RETURN_CODE_SUCCESS :
                                       STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eUpdateIndex
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateIndex(
    STOCK_MSG_OBJECT_STRUCT *psObj,
    STOCK_ID tIndex
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult;

    // Update state
    if (psObj->tIndex != tIndex)
    {
        psObj->tIndex = tIndex;
        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    }
    else
    {
        eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
    }

    return eResult;
}

/*****************************************************************************
*
*   eUpdateName
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateName(
    STOCK_MSG_OBJECT_STRUCT *psObj,
    STRING_OBJECT hNewName
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bOk;
    const char *pacName = NULL;

    do
    {
        // Check input
        if (hNewName == STRING_INVALID_OBJECT)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Try to update name by new one
        pacName = STRING.pacCStr(hNewName);
        bOk = STRING.bModifyCStr(psObj->hSymbolName, pacName);
        if (bOk == FALSE)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            break;
        }

        // Since at this point we know the name is changed the STOCK_SYMBOL
        // object became put of sync - remove it for later initialization
        if (psObj->hStockSymbol != STOCK_SYMBOL_INVALID_OBJECT)
        {
            STOCK_SYMBOL.vDestroy(psObj->hStockSymbol);
            psObj->hStockSymbol = STOCK_SYMBOL_INVALID_OBJECT;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eUpdateNameCStr
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateNameCStr(
    STOCK_MSG_OBJECT_STRUCT *psObj,
    const char *pacName,
    size_t tNameLength
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bOk = FALSE;

    do
    {
        // Check input
        if (pacName == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Try to update name by new one
        bOk = STRING.bModifyCStr(psObj->hSymbolName, pacName);
        if (bOk == FALSE)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
            break;
        }

        // Since at this point we know the name is changed the STOCK_SYMBOL
        // object became put of sync - remove it for later initialization
        if (psObj->hStockSymbol != STOCK_SYMBOL_INVALID_OBJECT)
        {
            STOCK_SYMBOL.vDestroy(psObj->hStockSymbol);
            psObj->hStockSymbol = STOCK_SYMBOL_INVALID_OBJECT;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
*
*   eUpdateValues
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateValues(
    STOCK_MSG_OBJECT_STRUCT *psObj,
    STOCK_MSG_STATE_ENUM eState,
    const STOCK_TICKER_QUOTE_STRUCT *psQuote
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bUpdated = FALSE;

    do
    {
        // Check input
        if ((psObj == NULL) || (psQuote == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Update state
        if (psObj->eState != eState)
        {
            psObj->eState = eState;
            bUpdated = TRUE;
        }

        // We have got only one valid quote related state
        if (psObj->eState == STOCK_MSG_STATE_PRICE_AVAILABLE)
        {
            // Update current value by new ones
            if ((psObj->sQuote.un32Price != psQuote->un32Price) ||
                (psObj->sQuote.un8PriceFrac != psQuote->un8PriceFrac) ||
                (psObj->sQuote.un32PriceDelta != psQuote->un32PriceDelta) ||
                (psObj->sQuote.un8PriceDeltaFrac != psQuote->un8PriceDeltaFrac) ||
                (psObj->sQuote.eDir != psQuote->eDir))
            {
                psObj->sQuote = *psQuote;
                bUpdated = TRUE;
            }
        }
        // State has been changed
        else if (bUpdated == TRUE)
        {
            // Invalidate values
            psObj->sQuote.un32Price = 0;
            psObj->sQuote.un32PriceDelta = 0;
            psObj->sQuote.un8PriceFrac = 0;
            psObj->sQuote.un8PriceDeltaFrac = 0;
            psObj->sQuote.eDir = STOCK_MSG_PRICE_DIRECTION_INVALID;
        }

        eResult = (bUpdated == TRUE) ? STOCK_TICKER_RETURN_CODE_SUCCESS :
                                       STOCK_TICKER_RETURN_CODE_NOTHING_TO_UPDATE;
    } while (FALSE);

    return eResult;
}
