/*****************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SPORTSERVICEMGR implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include "_sports_service_mgr_obj.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "dataservice_mgr_impl.h"
#include "sports_service_db_constants.h"
#include "sports_monitor_obj.h"
#include "sql_interface_obj.h"
#include "sms_api_debug.h"
#include "league_obj.h"
#include "team_obj.h"
#include "cid_obj.h"
#include "string_obj.h"

#include "sms_api_debug.h"

#if SPORTS_SERVICE_DEBUG == 0
/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_vPrintDummy
*
*****************************************************************************/
void SPORTS_SERVICE_MGR_vPrintDummy(
    UN32 un32DebugLevels,
    const char *pcFormat,
    ... )
{
    return;
}

#else
/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_vPrint
*
*****************************************************************************/
void SPORTS_SERVICE_MGR_vPrint(
    UN32 un32DebugLevels,
    const char *pcFormat,
    ... )
{
    if( un32DebugLevels )
    {
        va_list args;
        UN32 un32Sec;
        UN16 un16Msec;
        UN64 un64UpTimeMsec;
        OSAL.vTimeUp(&un32Sec, &un16Msec);
        un64UpTimeMsec = un32Sec *1000 + un16Msec;

        OSAL.vControlOutputThisTask(TRUE);

        printf("%llu SDS ", un64UpTimeMsec );

        va_start (args, pcFormat);
        vprintf (pcFormat, args);
        va_end (args);

        OSAL.vControlOutputThisTask(FALSE);
    }
    return;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_vPrintSilent
*
*****************************************************************************/
void SPORTS_SERVICE_MGR_vPrintSilent(
    UN32 un32DebugLevels,
    const char *pcFormat,
    ... )
{
    if( un32DebugLevels )
    {
        va_list args;

        OSAL.vControlOutputThisTask(TRUE);

        va_start (args, pcFormat);
        vprintf (pcFormat, args);
        va_end (args);

        OSAL.vControlOutputThisTask(FALSE);
    }
    return;
}
#endif

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static SPORTS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj;
    DATASERVICE_CREATE_STRUCT sCreate;
    BOOLEAN bOk;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
        "hStart: %s \n", pacSRHDriverName);

    bOk = DATASERVICE_IMPL_bProcessOptions(
        SS_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
    if (bOk == FALSE)
    {
        // Bad options!
        return SPORTS_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = SPORTS_SERVICE_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(SPORTS_SERVICE_MGR_OBJECT_STRUCT);
    sCreate.tDataID = (DATASERVICE_ID)GsSportsServiceIntf.tDSI;

    // Suggest an OTA buffer size
    sCreate.tSuggestedOTABufferByteSize =
            GsSportsServiceIntf.tOTABufferByteSize;

    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = DATASERVICE_EVENT_ALL;

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
        "hStart: starting on DSI %d\n", (DSI)sCreate.tDataID);
    psObj = (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hCreateNewService( &sCreate );
    if (psObj == NULL)
    {
        // Can't create the service, fail out!
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
            "hStart: failure to create\n");
        return SPORTS_SERVICE_INVALID_OBJECT;
    }

    psObj->un32DbConfigVersion = SPORTS_SERVICE_DATABASE_INVALID_VERSION;

    psObj->hInterface = OSAL_INVALID_OBJECT_HDL;

    // Initialize the DBMS connection handle
    psObj->hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;

    // Initialize the sports data service path attributes
    psObj->pacFullyQualifiedDatabaseFilePath = NULL;

    // Initialize the db read semaphore
    psObj->hDbReadSem = OSAL_INVALID_OBJECT_HDL;

    // Initialize the data set usage flag
    psObj->bDataSetInUse = FALSE;
    psObj->bMonitorProcessing = FALSE;
    psObj->bLockOutDueToConfigUpdate = FALSE;

    psObj->bHavePersistentDbPageSize = FALSE;

    // Initialize the supports table class, sports, and priorities.
    vIntitializeSupported( psObj );

    // Create the monitor list.
    psObj->hMonitorList = OSAL_INVALID_OBJECT_HDL;
    psObj->hMonitorListSem = OSAL_INVALID_OBJECT_HDL;
    bOk = bCreateMonitorList( psObj );
    if( bOk == FALSE )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
            "hStart: Failed to allocate monitor resources\n");

        vUninitObject( psObj );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

        return SPORTS_SERVICE_INVALID_OBJECT;
    }

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        DATASERVICE_EVENT_NONE,
        tEventRequestMask,
        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
        pvAppEventCallbackArg);

    // The service may now start
    bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);
    if (psObj->pacScratchPad == NULL)
    {
        psObj->pacScratchPad = OSAL.pvMemoryAllocate(
                    SPORTS_SERVICE_MGR_OBJECT_NAME":ScratchPad",
                    SCRATCH_PAD_INCREMENT_SIZE *
                    sizeof(char),
                    FALSE );
        if (psObj->pacScratchPad != NULL)
        {
            psObj->un32ScratchPadSize = SCRATCH_PAD_INCREMENT_SIZE * sizeof(char);
        }
        else
        {
            psObj->un32ScratchPadSize = 0;
            bOk = FALSE;
        }
    }

    if (bOk == FALSE)
    {
        // Error!
        vUninitObject( psObj );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
            "hStart: Failed to start\n");
        return SPORTS_SERVICE_INVALID_OBJECT;
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
        "hStart: started\n");
    return (SPORTS_SERVICE_OBJECT)psObj;
}

/*****************************************************************************
*
*   vStop
*
*****************************************************************************/
static void vStop(
    SPORTS_SERVICE_OBJECT hSportsDataService
        )
{
    // Start Removing the service from SMS now
    DATASERVICE_IMPL_vStop((DATASERVICE_IMPL_HDL)hSportsDataService);

    return;
}

/*****************************************************************************
*
*   eUseDataSet
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUseDataSet (
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_ACCESS_CALLBACK vCallback,
    void *pvCallbackArg )
{
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    OSAL_RETURN_CODE_ENUM eOsalCode;


    // Validate the object
    BOOLEAN bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hSportsService );

    if ((bOk != TRUE) || (vCallback == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // This is quick check to not block the app thread while we
    // performing a config update.  This could be removed.
    // If so the app thread will block while the dsm is
    // processing the config update.
    if( psObj->bLockOutDueToConfigUpdate == TRUE )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    // Only one access inside the user callback is permitted at a time
    if (psObj->bDataSetInUse == TRUE)
    {
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // Lock the database for reading.
    eOsalCode = OSAL.eSemTake(psObj->hDbReadSem, OSAL_OBJ_TIMEOUT_INFINITE);

    if (eOsalCode == OSAL_SUCCESS)
    {
        // See if the db is in a usable state.
        if( psObj->bLockOutDueToConfigUpdate == FALSE )
        {
            psObj->bDataSetInUse = TRUE;

            // Call the user function
            vCallback( hSportsService, pvCallbackArg);

            psObj->bDataSetInUse = FALSE;

            eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eSmsApiCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        OSAL.eSemGive(psObj->hDbReadSem);
    }

    return eSmsApiCode;
}

/*****************************************************************************
*
*   eGetSportList
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetSportList (
    SPORTS_SERVICE_OBJECT hSportsService,
    char *pcDest,
    UN32 *pun32DestSize
    )
{
    // Validate input and check conditions to use the data set.
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize
         );

    if( eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        BOOLEAN bOk;
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
        // Initialize the result building structure.
        SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;

        vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);

        bOk = bPerformQuery (
            psObj->hSQLConnection,
            SS_TAG_SPORT_LIST_OPEN,
            SS_TAG_SPORT_LIST_CLOSE,
            SS_SELECT_GET_SPORT_LIST,
            &sResult,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectSportList );

        if ( bOk == FALSE )
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        vEndProcessBuffer(&sResult);

        // Empty result set gets a special return code.
        if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (sResult.un32RowCount == 0) )
        {
            return SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        eSmsApiCode = sResult.eSmsApiCode;
    }

    return eSmsApiCode;
}

/*****************************************************************************
*
*   eGetRootAffiliateList
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetRootAffiliateList (
    SPORTS_SERVICE_OBJECT hSportsService,
    char *pcDest,
    UN32 *pun32DestSize
        )
{
    // Validate input and check conditions to use the data set.
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        BOOLEAN bOk;
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

        // Initialize the result building structure
        SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
        vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);

        bOk = bPerformQuery (
            psObj->hSQLConnection,
            SS_TAG_AFFILIATE_LIST_OPEN,
            SS_TAG_AFFILIATE_LIST_CLOSE,
            SS_SELECT_GET_ROOT_AFFILIATE_LIST,
            &sResult,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectAffiliateList );

        if ( bOk == FALSE )
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        vEndProcessBuffer(&sResult);

        // Empty result set gets a special return code.
        if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (sResult.un32RowCount == 0) )
        {
            return SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        eSmsApiCode = sResult.eSmsApiCode;
    }

    return eSmsApiCode;
}

/*****************************************************************************
*
*   eGetAffiliateChildren
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetAffiliateChildren (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        )
{
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    // Validate input and check conditions to use the data set.
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        BOOLEAN bOk;
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

        // Initialize the result building structure.
        SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
        vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);

        // Build the query string.
        bOk = bSnprintf( &psObj->acAppQueryBuf[0],
                         sizeof(psObj->acAppQueryBuf),
                         SS_SELECT_GET_AFFILIATE_CHILDREN,
                         tId );

        if( bOk == TRUE )
        {
            bOk = bPerformQuery (
                psObj->hSQLConnection,
                SS_TAG_AFFILIATE_LIST_OPEN,
                SS_TAG_AFFILIATE_LIST_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectAffiliateList );
        }

        if ( bOk == FALSE )
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        vEndProcessBuffer(&sResult);

        // Empty result set gets a special return code.
        if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (sResult.un32RowCount == 0) )
        {
            return SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        eSmsApiCode = sResult.eSmsApiCode;
    }

    return eSmsApiCode;
}

/*****************************************************************************
*
*   eGetAffiliateInfoClassList
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetAffiliateInfoClassList (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        )
{
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

    // Validate input and check conditions to use the data set.
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        BOOLEAN bOk;
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

        // Initialize the result building structure.
        SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
        vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);

        // Build the query string.
        bOk = bSnprintf( &psObj->acAppQueryBuf[0],
                         sizeof(psObj->acAppQueryBuf),
                         SS_SELECT_GET_AFFILIATE_INFO_CLASS_LIST,
                         tId );

        if( bOk == TRUE )
        {
            bOk = bPerformQuery (
                psObj->hSQLConnection,
                SS_TAG_INFORMATION_CLASS_LIST_OPEN,
                SS_TAG_INFORMATION_CLASS_LIST_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectAffiliateInfoClassList );
        }

        if ( bOk == FALSE )
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        vEndProcessBuffer(&sResult);

        // Empty result set gets a special return code.
        if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (sResult.un32RowCount == 0) )
        {
            return SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        eSmsApiCode = sResult.eSmsApiCode;
    }

    return eSmsApiCode;
}

/*****************************************************************************
*
*   eGetAffiliateTableList
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetAffiliateTableList (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    INFORMATION_CLASS tInfoClass,
    char *pcDest,
    UN32 *pun32DestSize
        )
{
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT sTableResult;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    BOOLEAN bOk;
    OSAL_OBJECT_HDL hTableList = OSAL_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    UN32 un32ListSize = 0;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Validate input and check conditions to use the data set.
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return eSmsApiCode;
    }

    // Set up the query struct to start writing payload.
    vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);

    // Make sure we support the table class.
    bOk = bIsTableClassSupported( psObj, tInfoClass );
    if( !bOk )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    do
    {
        // Build the query string.
        bOk = bSnprintf( &psObj->acAppQueryBuf[0],
                         sizeof(psObj->acAppQueryBuf),
                         SS_SELECT_GET_AFFILIATE_TABLE_LIST,
                         tAffiliateId,
                         tInfoClass );

        if( !bOk )
        {
            break;
        }

        eOsalCode = OSAL.eLinkedListCreate (
            &hTableList,
            SPORTS_SERVICE_MGR_OBJECT_NAME":TableList",
            NULL,
            OSAL_LL_OPTION_LINEAR );

        if( eOsalCode != OSAL_SUCCESS )
        {
           break;
        }

        // Populate the list.
        sTableResult.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
        sTableResult.un32RowCount = 0;
        sTableResult.hTableList = hTableList;

        bOk = SQL_INTERFACE.bQuery (
                    psObj->hSQLConnection,
                    psObj->acAppQueryBuf,
                    (SQL_QUERY_RESULT_HANDLER)bProcessGetTableList,
                    (void*)&sTableResult );

        if( !bOk || (sTableResult.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS) )
        {
            break;
        }

        // Get the size of the list.
        eOsalCode = OSAL.eLinkedListItems( hTableList, &un32ListSize );
        if( eOsalCode != OSAL_SUCCESS )
        {
           break;
        }

        // Empty result set get special return code.
        if( un32ListSize == 0 )
        {
            eSmsApiCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
            break;
        }

        //
        // Write the opening xml tag.
        //


        // Write the result.
        eSmsApiCode = eSports_Service_snprintf(
                                        &sResult,
                                        SS_TAG_TABLE_LIST_OPEN);

        if (eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            sResult.eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }


        //
        // Build the result set of table list entries.
        //

        eOsalCode = OSAL.eLinkedListIterate (
             hTableList,
             (OSAL_LL_ITERATOR_HANDLER) bUnsafeGetTableListHandler,
             (void*)&sResult );

        if( eOsalCode != OSAL_SUCCESS )
        {
            break;
        }

        if( sResult.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
            eSmsApiCode = sResult.eSmsApiCode;
            break;
        }

        //
        // Write the closing xml tag.
        //

        // Write the result.
        eSmsApiCode = eSports_Service_snprintf(
                                        &sResult,
                                        SS_TAG_TABLE_LIST_CLOSE);

        if (eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            sResult.eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        vEndProcessBuffer(&sResult);
        eSmsApiCode = sResult.eSmsApiCode;

    }
    while(0);

    if( hTableList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hTableList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        (void)OSAL.eLinkedListDelete( hTableList );
    }

    return eSmsApiCode;
}

/*****************************************************************************
*
*   eGetAffiliate
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetAffiliate (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        )
{
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

    // Validate input and check conditions to use the data set.
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS )
    {
        BOOLEAN bOk;
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

        // Initialize the result building structure.
        SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
        vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);

        // Build the query string.
        bOk = bSnprintf( &psObj->acAppQueryBuf[0],
                         sizeof(psObj->acAppQueryBuf),
                         SS_SELECT_GET_AFFILIATE,
                         tId );

        if( bOk == TRUE )
        {
            bOk = bPerformQuery (
                psObj->hSQLConnection,
                "",
                "",
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectAffiliateList );
        }

        if ( bOk == FALSE )
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        vEndProcessBuffer(&sResult);

        if( sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS )
        {
            if( sResult.un32RowCount != 1 )
            {
                // Empty result set gets a special return code.
                return SMSAPI_RETURN_CODE_NO_OBJECTS;
            }
        }
        eSmsApiCode = sResult.eSmsApiCode;
    }

    return eSmsApiCode;
}

/*****************************************************************************
*
*   eGetTableHeader
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetTableHeader (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    char *pcDest,
    UN32 *pun32DestSize
        )
{
    return eGetTableHeaderCommon (
        hSportsService,
        tAffiliateId,
        tTableId,
        pcDest,
        pun32DestSize,
        TRUE
        );
}

/*****************************************************************************
*
*   eGetTable
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetTable (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    BOOLEAN bOk;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    UN32 un32TdefKey;
    UN32 un32ColumnCount;
    UN8 aun8ColumnMap[SS_BYTES_FOR_BITS(SS_MAX_LABEL_COUNT)];

    // Validate input and check conditions to use the data set.
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return eSmsApiCode;
    }

    // Built the string to get the tdefkey of the table.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE_TDEF,
                        tTableId,
                        tAffiliateId );

    if( bOk )
    {
        bOk = bUnsafeGetTableQueryInfo(
                    psObj,
                    psObj->acAppQueryBuf,
                    &un32TdefKey,
                    &un32ColumnCount,
                    aun8ColumnMap,
                    SS_MAX_LABEL_COUNT );
    }

    if( !bOk )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    //
    // Get the table rows.
    //

    // Initialize the result building structure.
    vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);
    sResult.pun8ColumnMap = aun8ColumnMap;
    sResult.un32ColumnMapSize = SS_MAX_LABEL_COUNT;
    sResult.un32ExpectedColumnCount = un32ColumnCount;

    // Build the query string.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE,
                        un32TdefKey,
                        tTableId,
                        un32RowOffset );

    if( bOk == TRUE )
    {
        bOk = bPerformQuery (
                psObj->hSQLConnection,
                SS_TAG_TABLE_OPEN,
                SS_TAG_TABLE_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectTable );
    }

    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    vEndProcessBuffer(&sResult);

    // Empty result set gets a special return code.
    if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (sResult.un32RowCount == 0) )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    return sResult.eSmsApiCode;
}

/*****************************************************************************
*
*   eGetTableByColumnMatch
*
* TODO: Must limit the size of the match string in order to size the
* query string buffer.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetTableByColumnMatch (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    UN32 tColumnId,
    char *pValue,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    BOOLEAN bOk;
    UN32 un32TdefKey;
    UN32 un32ColumnCount;
    UN8 aun8ColumnMap[SS_BYTES_FOR_BITS(SS_MAX_LABEL_COUNT)];

    // Validate input and check conditions to use the data set.
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return eSmsApiCode;
    }

    // Built the string to get the tdefkey of the table.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE_TDEF,
                        tTableId,
                        tAffiliateId );

    if( bOk )
    {
        bOk = bUnsafeGetTableQueryInfo(
                    psObj,
                    psObj->acAppQueryBuf,
                    &un32TdefKey,
                    &un32ColumnCount,
                    aun8ColumnMap,
                    SS_MAX_LABEL_COUNT );
    }

    if( !bOk )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    //
    // Get the table rows.
    //

    // Initialize the result building structure.
    vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);
    sResult.pun8ColumnMap = aun8ColumnMap;
    sResult.un32ColumnMapSize = SS_MAX_LABEL_COUNT;
    sResult.un32ExpectedColumnCount = un32ColumnCount;

    // Build the query string.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE_BY_COLUMN_MATCH,
                        un32TdefKey,
                        tTableId,
                        tColumnId,
                        pValue,
                        un32RowOffset );

    if( bOk == TRUE )
    {
        bOk = bPerformQuery (
                psObj->hSQLConnection,
                SS_TAG_TABLE_OPEN,
                SS_TAG_TABLE_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectTable );
    }

    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    vEndProcessBuffer(&sResult);

    // Empty result set gets a special return code.
    if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (sResult.un32RowCount == 0) )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    return sResult.eSmsApiCode;
}

/*****************************************************************************
*
*   eGetTableByColumnRange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetTableByColumnRange (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    UN32 tColumnId,
    UN32 tValue1,
    UN32 tValue2,
    SORT_TYPE_ENUM eSortType,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    BOOLEAN bOk;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    UN32 un32TdefKey;
    UN32 un32ColumnCount;
    UN8 aun8ColumnMap[SS_BYTES_FOR_BITS(SS_MAX_LABEL_COUNT)];
    char acAsc[] = "ASC";
    char acDesc[] = "DESC";
    char *pcSort = acAsc;

    // Validate input and check conditions to use the data set.
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return eSmsApiCode;
    }

    // Built the string to get the tdefkey of the table.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE_TDEF,
                        tTableId,
                        tAffiliateId );

    if( bOk )
    {
        bOk = bUnsafeGetTableQueryInfo(
                    psObj,
                    psObj->acAppQueryBuf,
                    &un32TdefKey,
                    &un32ColumnCount,
                    aun8ColumnMap,
                    SS_MAX_LABEL_COUNT );
    }

    if( !bOk )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    //
    // Get the table rows.
    //

    // Initialize the result building structure.
    vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);
    sResult.pun8ColumnMap = aun8ColumnMap;
    sResult.un32ColumnMapSize = SS_MAX_LABEL_COUNT;
    sResult.un32ExpectedColumnCount = un32ColumnCount;

    // Set the sort to decend
    if( eSortType == SORT_TYPE_DECEND )
    {
        pcSort = acDesc;
    }

    // Build the query string.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE_BY_COLUMN_RANGE,
                        un32TdefKey,
                        tTableId,
                        tColumnId,
                        tValue1,
                        tValue2,
                        tColumnId,
                        pcSort,
                        un32RowOffset );

    if( bOk == TRUE )
    {
        bOk = bPerformQuery (
                psObj->hSQLConnection,
                SS_TAG_TABLE_OPEN,
                SS_TAG_TABLE_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectTable );
    }

    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    vEndProcessBuffer(&sResult);

    // Empty result set gets a special return code.
    if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (sResult.un32RowCount == 0) )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    return sResult.eSmsApiCode;
}

/*****************************************************************************
*
*   eGetTableByTeamPresence
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetTableByTeamPresence (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    UN32 un32ColumnOneId,
    UN32 un32ColumnTwoId,
    UN32 un32TeamId,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    BOOLEAN bOk;
    UN32 un32TdefKey;
    UN32 un32ColumnCount;
    UN8 aun8ColumnMap[SS_BYTES_FOR_BITS(SS_MAX_LABEL_COUNT)];

    // Validate input and check conditions to use the data set.
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return eSmsApiCode;
    }

    // Built the string to get the tdefkey of the table.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE_TDEF,
                        tTableId,
                        tAffiliateId );

    if( bOk )
    {
        bOk = bUnsafeGetTableQueryInfo(
                    psObj,
                    psObj->acAppQueryBuf,
                    &un32TdefKey,
                    &un32ColumnCount,
                    aun8ColumnMap,
                    SS_MAX_LABEL_COUNT );
    }

    if( !bOk )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    //
    // Get the table rows.
    //

    // Initialize the result building structure.
    vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);
    sResult.pun8ColumnMap = aun8ColumnMap;
    sResult.un32ColumnMapSize = SS_MAX_LABEL_COUNT;
    sResult.un32ExpectedColumnCount = un32ColumnCount;

    // Build the query string.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE_BY_TEAM_PRESENCE,
                        un32TdefKey,
                        tTableId,
                        un32ColumnOneId,
                        un32TeamId,
                        un32ColumnTwoId,
                        un32TeamId,
                        un32RowOffset );

    if( bOk == TRUE )
    {
        bOk = bPerformQuery (
                psObj->hSQLConnection,
                SS_TAG_TABLE_OPEN,
                SS_TAG_TABLE_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectTable );
    }

    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    vEndProcessBuffer(&sResult);

    // Empty result set gets a special return code.
    if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (sResult.un32RowCount == 0) )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    return sResult.eSmsApiCode;
}

/*****************************************************************************
*
*   eGetReferencedTableHeader
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetReferencedTableHeader (
    SPORTS_SERVICE_OBJECT hSportsService,
    REFERENCED_TABLE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        )
{
    return eGetTableHeaderCommon (
        hSportsService,
        0,
        (TABLE_ID) tId,
        pcDest,
        pun32DestSize,
        FALSE
        );
}

/*****************************************************************************
*
*   eGetTableHeaderCommon
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetTableHeaderCommon (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    char *pcDest,
    UN32 *pun32DestSize,
    BOOLEAN bIsPrimaryTable
        )
{
    BOOLEAN bOk;
    UN32 un32TdefKey;
    char *pcRewind;
    char acOpenTag[SS_TAG_TABLE_HEADER_OPEN_MAX_SIZE+1];
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Validate input and check conditions to use the data set.
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return eSmsApiCode;
    }

    // Initialize the result building structure
    vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);

    // Build the opening tag string.
    bOk = bSnprintf( acOpenTag,
                     SS_TAG_TABLE_HEADER_OPEN_MAX_SIZE,
                     SS_TAG_TABLE_HEADER_OPEN
                     SS_TAG_ID_OPEN"%llu"SS_TAG_ID_CLOSE,
                     tTableId );

    //
    // Get the non column header information.
    //

    // Build the query string.
    if( bOk )
    {
        if( bIsPrimaryTable )
        {
            bOk = bSnprintf( &psObj->acAppQueryBuf[0],
                         sizeof(psObj->acAppQueryBuf),
                         SS_SELECT_GET_TABLE_HEADER_FIXED,
                         tTableId,
                         tAffiliateId );
        }
        else
        {
            bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                                sizeof(psObj->acAppQueryBuf),
                                SS_SELECT_GET_REFERENCED_TABLE_HEADER_FIXED,
                                tTableId );
        }
    }

    if( bOk )
    {
        bOk = bPerformQuery (
            psObj->hSQLConnection,
            acOpenTag,
            "",
            psObj->acAppQueryBuf,
            &sResult,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectTableHeaderFixed );
    }

    if ( !bOk  )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Empty result set gets a special return code.
    if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (sResult.un32RowCount == 0) )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    // The carry over is the tdefkey.
    un32TdefKey = sResult.un32CarryOver;

    if (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        //
        // Get label overrides.
        //

        bOk = bSnprintf( &psObj->acAppQueryBuf[0],
                         sizeof(psObj->acAppQueryBuf),
                         SS_GET_LABEL_OVERRIDES,
                         tTableId,
                         un32TdefKey );

        if( bOk )
        {
            sResult.un32RowCount = 0;
            pcRewind = sResult.pacCursor;

            bOk = bPerformQuery (
                            psObj->hSQLConnection,
                            SS_TAG_OVERRIDE_LIST_OPEN,
                            SS_TAG_OVERRIDE_LIST_CLOSE,
                            psObj->acAppQueryBuf,
                            &sResult,
                            (SQL_QUERY_RESULT_HANDLER)bProcessGetLabelOverrides );
        }

        if((!bOk)
            ||  (sResult.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
          )
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        // Empty result, rewind the cursor.
        if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (sResult.un32RowCount == 0) )
        {
            sResult.pacCursor = pcRewind;
        }
    }

    eSmsApiCode = sResult.eSmsApiCode;
    if (eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        //
        // Get the column information.
        //

        // Build the query string.  We stored the tdefkey from the previous
        // query in the carryover field.  It eliminates a join.
        bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                            sizeof(psObj->acAppQueryBuf),
                            SS_SELECT_GET_TABLE_HEADER_VARIABLE,
                            un32TdefKey );

        if( bOk == TRUE )
        {
            bOk = bPerformQuery (
                psObj->hSQLConnection,
                "",
                SS_TAG_TABLE_HEADER_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectTableHeaderVariable );
        }

        if ( bOk == FALSE )
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        vEndProcessBuffer(&sResult);
        eSmsApiCode = sResult.eSmsApiCode;
    }


    return eSmsApiCode;
}


/*****************************************************************************
*
*   eGetReferencedTable
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetReferencedTable (
    SPORTS_SERVICE_OBJECT hSportsService,
    REFERENCED_TABLE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    BOOLEAN bOk;
    UN32 un32TdefKey;
    UN32 un32ColumnCount;
    UN8 aun8ColumnMap[SS_BYTES_FOR_BITS(SS_MAX_LABEL_COUNT)];

    // Validate input and check conditions to use the data set.
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    eSmsApiCode = eCheckDataSetUsage (
        hSportsService,
        pcDest,
        pun32DestSize );

    if( eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return eSmsApiCode;
    }

    // Built the string to get the tdefkey of the table.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_REFERENCED_TABLE_TDEF,
                        tId );

    if( bOk )
    {
        bOk = bUnsafeGetTableQueryInfo(
                    psObj,
                    psObj->acAppQueryBuf,
                    &un32TdefKey,
                    &un32ColumnCount,
                    aun8ColumnMap,
                    SS_MAX_LABEL_COUNT );
    }

    if( !bOk )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    //
    // Get the table rows.
    //

    // Initialize the result building structure.
    vInitializeQueryStruct(&sResult, psObj, pcDest, pun32DestSize);
    sResult.pun8ColumnMap = aun8ColumnMap;
    sResult.un32ColumnMapSize = SS_MAX_LABEL_COUNT;
    sResult.un32ExpectedColumnCount = un32ColumnCount;

    // Build the query string.
    bOk = bSnprintf(    &psObj->acAppQueryBuf[0],
                        sizeof(psObj->acAppQueryBuf),
                        SS_SELECT_TABLE,
                        un32TdefKey,
                        tId,
                        un32RowOffset );

    if( bOk == TRUE )
    {
        bOk = bPerformQuery (
                psObj->hSQLConnection,
                SS_TAG_TABLE_OPEN,
                SS_TAG_TABLE_CLOSE,
                psObj->acAppQueryBuf,
                &sResult,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectTable );
    }

    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    vEndProcessBuffer(&sResult);

    // Empty result set gets a special return code.
    if( (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (sResult.un32RowCount == 0) )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    return sResult.eSmsApiCode;
}

/*****************************************************************************
*
*   hCreateLeagueCid
*
*****************************************************************************/
static CID_OBJECT hCreateLeagueCid( UN32 un32LeagueId )
{
    CID_OBJECT hCid = CID_INVALID_OBJECT;
    CID_OBJECT hResult = CID_INVALID_OBJECT;
    hCid = LEAGUE_hCreateCid(CID_POOL_INVALID_OBJECT, &un32LeagueId);
    if( hCid )
    {
        // Make a duplicate.  The user is responsible for destroying the
        // CID with CID.vDestroy.  CID.vDestroy cannot destroy a CID
        // created with LEAGUE_hCreateLeagueCid.
        hResult = CID.hDuplicate( hCid );
        CID_vDestroy( hCid );
    }
    return hResult;
}

/*****************************************************************************
*
*   hCreateTeamCid
*
*****************************************************************************/
static CID_OBJECT hCreateTeamCid( UN32 un32LeagueId, UN32 un32TeamId )
{
    CID_OBJECT hCid = CID_INVALID_OBJECT;
    CID_OBJECT hResult = CID_INVALID_OBJECT;
    hCid = TEAM_hCreateCid(CID_POOL_INVALID_OBJECT, &un32TeamId);
    if( hCid )
    {
        // Make a duplicate.  The user is responsible for destroying the
        // CID with CID.vDestroy.  CID.vDestroy cannot destroy a CID
        // created with TEAM_hCreateCid.
        hResult = CID.hDuplicate( hCid );
        CID_vDestroy( hCid );
    }
    return hResult;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_hMonitorCreate
*
*****************************************************************************/
SPORTS_MONITOR_OBJECT SPORTS_SERVICE_MGR_hMonitorCreate (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_CALLBACK vCallback,
    void *pvCallbackArg    )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_MONITOR_OBJECT hMon;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;

    // Verify service object
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if ( bOk == FALSE )
    {
        return SPORTS_MONITOR_INVALID_OBJECT;
    }

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake(  psObj->hMonitorListSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode != OSAL_SUCCESS )
    {
        return SPORTS_MONITOR_INVALID_OBJECT;
    }

    hMon = SPORTS_MONITOR_hCreate(
                hService,
                vCallback,
                pvCallbackArg );

    if( hMon != SPORTS_MONITOR_INVALID_OBJECT )
    {
        // Add the monitor to the list.
        eOsalCode = OSAL.eLinkedListAdd(
            psObj->hMonitorList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hMon );

        if (eOsalCode != OSAL_SUCCESS)
        {
            SPORTS_MONITOR_Destroy( hMon );
            hMon = SPORTS_MONITOR_INVALID_OBJECT;
        }
    }

    OSAL.eSemGive( psObj->hMonitorListSem );
    return hMon;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_eMonitorGetEventMask
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_SERVICE_MGR_eMonitorGetEventMask (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORTS_MONITOR_EVENT_MASK *ptEventMask )
{
    BOOLEAN bOk;

    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Verify service object
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake(  psObj->hMonitorListSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode != OSAL_SUCCESS )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    eReturnCode = SPORTS_MONITOR_eGetEventMask ( hMonitor, ptEventMask );
    OSAL.eSemGive( psObj->hMonitorListSem );
    return eReturnCode;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_eMonitorSetEventMask
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_SERVICE_MGR_eMonitorSetEventMask (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORTS_MONITOR_EVENT_MASK tEventMask )
{
    BOOLEAN bOk;

    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Verify service object
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake(  psObj->hMonitorListSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode != OSAL_SUCCESS )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }
    eReturnCode = SPORTS_MONITOR_eSetEventMask ( hMonitor, tEventMask );
    OSAL.eSemGive( psObj->hMonitorListSem );
    return eReturnCode;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_vMonitorDestroy
*
*****************************************************************************/
void SPORTS_SERVICE_MGR_vMonitorDestroy (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor     )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;

    // Verify monitor object.
    bOk = SMSO_bValid( (SMS_OBJECT)hMonitor );
    if( bOk == FALSE )
    {
        return;
    }

    // If the sports service is valid we need to detach the monitor from it.
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if( bOk == TRUE )
    {
        eOsalCode = OSAL.eSemTake(  psObj->hMonitorListSem,
                                    OSAL_OBJ_TIMEOUT_INFINITE );
        if ( eOsalCode == OSAL_SUCCESS )
        {
            // Remove the monitor.
            eOsalCode = OSAL.eLinkedListSearch(
                psObj->hMonitorList,
                &hEntry,
                hMonitor );

            if (eOsalCode == OSAL_SUCCESS)
            {
                eOsalCode = OSAL.eLinkedListRemove( hEntry );
                if (eOsalCode != OSAL_SUCCESS)
                {
                    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
                        "vMonitorDestroy: Failure to remove "
                        "monitor list (%s)\n",
                        OSAL.pacGetReturnCodeName(eOsalCode));
                }
            }
            OSAL.eSemGive( psObj->hMonitorListSem );
        }
    }

    SPORTS_MONITOR_Destroy( hMonitor );
    return;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_eMonitorSetAffiliateMask
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_SERVICE_MGR_eMonitorSetAffiliateMask (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliate,
    SPORTS_MONITOR_EVENT_MASK tEventMask )
{
    BOOLEAN bOk;

    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Verify service object
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    bOk = bIsAffiliateValid( tAffiliate );
    if( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
    }

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake(  psObj->hMonitorListSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode != OSAL_SUCCESS )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    eReturnCode = SPORTS_MONITOR_eSetAffiliateMask (
            hMonitor,
            tAffiliate,
            tEventMask );

    OSAL.eSemGive( psObj->hMonitorListSem );
    return eReturnCode;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_eMonitorGetAffiliateMask
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_SERVICE_MGR_eMonitorGetAffiliateMask (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliate,
    SPORTS_MONITOR_EVENT_MASK *ptEventMask )
{
    BOOLEAN bOk;

    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Verify service object
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    bOk = bIsAffiliateValid( tAffiliate );
    if( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
    }

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake(  psObj->hMonitorListSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode != OSAL_SUCCESS )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    eReturnCode = SPORTS_MONITOR_eGetAffiliateMask (
            hMonitor,
            tAffiliate,
            ptEventMask );

    OSAL.eSemGive( psObj->hMonitorListSem );
    return eReturnCode;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_eMonitorRemoveAllAffiliates
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_SERVICE_MGR_eMonitorRemoveAllAffiliates (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Verify service object
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake(  psObj->hMonitorListSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode != OSAL_SUCCESS )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    eReturnCode = SPORTS_MONITOR_eRemoveAllAffiliates (    hMonitor );
    OSAL.eSemGive( psObj->hMonitorListSem );
    return eReturnCode;
}

/*****************************************************************************
*
*   SPORTS_SERVICE_MGR_eIterateChangedAffiliates
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM SPORTS_SERVICE_MGR_eIterateChangedAffiliates (
    SPORTS_SERVICE_OBJECT hService,
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORT_MONITOR_ITERATOR bIterator,
    void *pvIteratorArg )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Verify service object
    bOk = DATASERVICE_IMPL_bValid( (DATASERVICE_IMPL_HDL)hService );
    if ( bOk == FALSE )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // This can only be called if we are processing monitors.
    // No need to take the hMonitorListSem.  This ensures it is
    // already taken.  If you do take the sem it will lock up.
    if( psObj->bMonitorProcessing == FALSE )
    {
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    eReturnCode = SPORTS_MONITOR_eIterateChangedAffiliates (
        hMonitor,
        bIterator,
        pvIteratorArg );

    return eReturnCode;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid;
    DATASERVICE_ERROR_CODE_ENUM eResult;
    SMSAPI_EVENT_MASK tEventMask =
        DATASERVICE_EVENT_NONE;

    // Get our handle from the callback argument
    psObj = (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == TRUE)
    {
        switch( tCurrentEvent )
        {
            // State has changed
            case DATASERVICE_EVENT_STATE:
            {
                DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                    (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;
                BOOLEAN bStateChanged;

                // Process the state transition
                bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psObj,
                    psStateChange,
                    &GsSportsStateHandlers,
                    (void *)psObj);

                if (bStateChanged == TRUE)
                {
                    // The state has been updated
                    tEventMask |= DATASERVICE_EVENT_STATE;
                }
            }
            break;

            // This service has a message to process
            case DATASERVICE_EVENT_NEW_DATA:
            {
                // Event argument is a message payload
                OSAL_BUFFER_HDL hPayload =
                    (OSAL_BUFFER_HDL)pvEventArg;

                // Handle the message reception
                eResult = eHandleNewDataEvent( psObj, hPayload );

                if( eResult != DATASERVICE_ERROR_CODE_NONE )
                {
                    vSetError(psObj, eResult);
                }
            }
            break;

            case DATASERVICE_EVENT_TIMEOUT:
            {
                vHandleTimeoutEvent( psObj, pvEventArg );
            }
            break;

            default:
            break;
        }

        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psObj->sEvent, tEventMask);

        // Notify of any change via any registered callback which may
        // be present
        SMSU_bNotify(&psObj->sEvent);

        if (tCurrentEvent == DATASERVICE_EVENT_STATE)
        {
            DATASERVICE_STATE_ENUM eFinalState;

            // If we're stopped, don't allow any more events to be generated
            eFinalState = DATASERVICE_IMPL_eState((DATASERVICE_IMPL_HDL)psObj);
            if (eFinalState == DATASERVICE_STATE_STOPPED)
            {
                // Filter out all further updates
                SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

                // Clean up memory allocations
                vUninitObject( psObj );
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;

    if( psObj->hDbReadSem != OSAL_INVALID_OBJECT_HDL )
    {
        eOsalCode = OSAL.eSemTake(  psObj->hDbReadSem,
                                    OSAL_OBJ_TIMEOUT_INFINITE );
        if ( eOsalCode != OSAL_SUCCESS )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
                "vUninitObject: Failure to take db semaphore (%s)\n",
                OSAL.pacGetReturnCodeName(eOsalCode) );
        }
    }

    if( psObj->hMonitorListSem != OSAL_INVALID_OBJECT_HDL )
    {
        eOsalCode =
            OSAL.eSemTake( psObj->hMonitorListSem, OSAL_OBJ_TIMEOUT_INFINITE );
        if ( eOsalCode != OSAL_SUCCESS )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
                "vUninitObject: Failure to take monitor list semaphore (%s)\n",
                OSAL.pacGetReturnCodeName(eOsalCode) );
         }
    }

    // Clear the timed event handle, we're done with it
    psObj->hBackgroundEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

    // Destroy the interface object if it exists
    if (psObj->hInterface != SPORTS_SERVICE_INTERFACE_INVALID_OBJECT)
    {
        GsSportsServiceIntf.vUnInit( psObj->hInterface );
        psObj->hInterface = SPORTS_SERVICE_INTERFACE_INVALID_OBJECT;
    }

    // Disconnect from the database application
    if (psObj->hSQLConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect( psObj->hSQLConnection );
        psObj->hSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Destroy path allocations
    if (psObj->pacFullyQualifiedDatabaseFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)
            psObj->pacFullyQualifiedDatabaseFilePath);
        psObj->pacFullyQualifiedDatabaseFilePath = NULL;
    }

    // Destroy the monitor list.
    vDestroyMonitorList( psObj );

    // Destroy the db read semaphore
    if( psObj->hDbReadSem != OSAL_INVALID_OBJECT_HDL )
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL.eSemDelete(
                psObj->hDbReadSem );
        if( eReturnCode == OSAL_SUCCESS )
        {
            psObj->hDbReadSem = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy the SMS Update object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    BOOLEAN bError;
    UN32 un32UTCsec;
    UN32 un32Epoch;

    do
    {
        // Register for a timed event
        psObj->hBackgroundEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);

        if(psObj->hBackgroundEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the paths needed by the sports data service
        // (based on information gathered from the config manager)
        bOk = bBuildDBFilePath( psObj );

        if( !bOk )
        {
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        eErrorCode  = eLoadPersistentDb( psObj );
        if( eErrorCode != DATASERVICE_ERROR_CODE_NONE )
        {
            break;
        }

        // Not taken binary semaphore
        eOsalCode = OSAL.eSemCreate(
            &psObj->hDbReadSem,
            SPORTS_SERVICE_MGR_OBJECT_NAME":DbSem",
            1, 1, OSAL_SEM_OPTION_NONE);

        if( eOsalCode != OSAL_SUCCESS )
        {
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Retrieve the service config info from the database
        // (SC_Config and SA_Config tables)
        bOk = bUnsafeLoadSportsConfigDataFromDB( psObj, &bError );
        if( !bOk )
        {
            if (bError == TRUE)
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
                    "bHandleServiceReady: Failure to load config info from DB\n" );
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            }
            break;
        }

        //
        // Perform data pruning.
        //

        // Initialize the prune epoch to today.
        un32UTCsec = 0;
        eOsalCode = OSAL.eTimeGet( &un32UTCsec );
        if( eOsalCode == OSAL_SUCCESS )
        {
            // Convert it into a 2:00am starting epoch.
            un32Epoch = un32UTCsec;
            if( un32Epoch > SS_TWO_HOURS_IN_SECONDS )
            {
                un32Epoch -= SS_TWO_HOURS_IN_SECONDS;
            }
            un32Epoch /= SS_SECONDS_PER_DAY;
            psObj->sPruneCtrl.un32Epoch = un32Epoch;
        }
        else
        {
            printf("Result: %s\n",
                   OSAL.pacGetReturnCodeName(eOsalCode));
            psObj->sPruneCtrl.un32Epoch = 0;
        }

        psObj->sPruneCtrl.bRetryTdef = FALSE;
        psObj->sPruneCtrl.bRetryData = !bExecuteSaveRules(
                psObj,
                FALSE, // Do not notify using monitors.  Sports has not started.
                -1 );

        if( psObj->sPruneCtrl.bRetryData )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE | 1,
                            "bHandleServiceReady: "
                            "Age out rules rescheduled\n" );
        }

        // Initialize the Sports Service interface
        psObj->hInterface =
            GsSportsServiceIntf.hInit(
                (SPORTS_SERVICE_OBJECT) psObj,
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));

        if (psObj->hInterface == SPORTS_SERVICE_INTERFACE_INVALID_OBJECT)
        {
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Kick off the event, override any event already set
        bOk = DATASERVICE_IMPL_bSetTimedEvent(
            psObj->hBackgroundEvent,
            TRUE, TRUE, SS_BACKGROUND_EVENT_INTERVAL_SECONDS);
        if( bOk == FALSE)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
                            "bHandleServiceReady: "
                            "Failure to start background event\n" );
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
            "bHandleServiceReady: Success"
            " eTimeGet() osalcode %u time %u epoch(day) %u remain %u\n",
            eOsalCode,
            un32UTCsec,
            un32UTCsec / 86400,
            un32UTCsec % 86400 );

        return TRUE;

    } while (FALSE);

    vSetError(psObj, eErrorCode );

    return FALSE;
}

/*****************************************************************************
*
*   bHandleServiceStopped
*
*****************************************************************************/
static BOOLEAN bHandleServiceStopped (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bValid;

    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);
    if (bValid == TRUE)
    {
        vExecuteShutdownRules( psObj );
        vSaveMemoryDb( psObj );

        if (psObj->pacScratchPad != NULL)
        {
            OSAL.vMemoryFree((void *)psObj->pacScratchPad);
            psObj->pacScratchPad = NULL;
        }
        psObj->un32ScratchPadSize = 0;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bHandleServiceError
*
*****************************************************************************/
static BOOLEAN bHandleServiceError (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop the background event from occurring
    DATASERVICE_IMPL_bStopTimedEvent(psObj->hBackgroundEvent);

    return TRUE;
}

/*****************************************************************************
*
*   vSaveMemoryDb
*
*****************************************************************************/
static void vSaveMemoryDb (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
           DATASERVICE_ERROR_CODE_NONE;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SQL_INTERFACE_OBJECT hSQLPersistent;
    UN32 un32Sec;
    UN16 un16Msec;
    UN64 un64Start;
    UN64 un64Temp;
    UN64 un64Interval;
    size_t tStartSize = 0;
    size_t tEndSize = 0;
    FILE *psFile;

    eOsalCode = OSAL.eSemTake(  psObj->hDbReadSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );

    if( eOsalCode != OSAL_SUCCESS )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE | 1,
            "vSaveMemoryDb: Failure to take db semaphore (%s)\n",
            OSAL.pacGetReturnCodeName(eOsalCode) );
        return;
    }

    psFile = fopen (
            &psObj->pacFullyQualifiedDatabaseFilePath[0],
            "r" );

    if( psFile )
    {
        OSAL.bFileSystemGetFileSize( psFile, &tStartSize );
        fclose( psFile );
    }

    OSAL.vTimeUp(&un32Sec, &un16Msec);
    un64Start = un32Sec *1000 + un16Msec;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE | 1,
                    "Save db start: %llu\n",
                    un64Start );

    // Open the persistent database.
    hSQLPersistent = SQL_INTERFACE.hConnect(
       &psObj->pacFullyQualifiedDatabaseFilePath[0],
       SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND,
       &eErrorCode );

    if( hSQLPersistent == SQL_INTERFACE_INVALID_OBJECT )
    {
        if( eErrorCode == DATASERVICE_ERROR_CODE_DATABASE_CORRUPT )
        {
            int iReturnCode;
            iReturnCode = remove(&psObj->pacFullyQualifiedDatabaseFilePath[0]);
            
            if (0 == iReturnCode)
            {   
                // Open a new persistent database.
                hSQLPersistent = SQL_INTERFACE.hConnect(
                    &psObj->pacFullyQualifiedDatabaseFilePath[0],
                    SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND,
                    &eErrorCode );                                 
            }
        }
        if( hSQLPersistent == SQL_INTERFACE_INVALID_OBJECT )
        {         
            //nothing more we can do
            OSAL.eSemGive( psObj->hDbReadSem );
            return;
        }
    }

    OSAL.vTimeUp(&un32Sec, &un16Msec);
    un64Interval = un32Sec *1000 + un16Msec;

/*
    // Vacuum the db to compact it.
    bOk = SQL_INTERFACE.bExecuteCommand(
                                psObj->hSQLConnection,
                                "VACUUM;" );

    un64Temp = un64Interval;
    OSAL.vTimeUp(&un32Sec, &un16Msec);
    un64Interval = un32Sec *1000 + un16Msec;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE | 0,
                    "Vacuum db: Interval %llu Dur %llu\n",
                    un64Interval,
                    un64Interval - un64Temp );
*/

    // Copy the memory database to the persistent database.
    bOk = SQL_INTERFACE.bCopyMainDatabase(
            hSQLPersistent,
            psObj->hSQLConnection );

    SQL_INTERFACE.vDisconnect(hSQLPersistent);

    un64Temp = un64Interval;
    OSAL.vTimeUp(&un32Sec, &un16Msec);
    un64Interval = un32Sec *1000 + un16Msec;

    psFile = fopen (
                &psObj->pacFullyQualifiedDatabaseFilePath[0],
                "r" );

    if( psFile )
    {
        OSAL.bFileSystemGetFileSize( psFile, &tEndSize );
        fclose( psFile );
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE | 1,
            "Save db end: bOk %u Interval %llu Dur %llu Tot %llu"
            " StartSize %d EndSize %d\n",
            bOk,
            un64Interval,
            un64Interval - un64Temp,
            un64Interval - un64Start,
            tStartSize,
            tEndSize );

    OSAL.eSemGive( psObj->hDbReadSem );
    return;
}

/*****************************************************************************
*
*   eHandleNewDataEvent
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eHandleNewDataEvent (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    DATASERVICE_ERROR_CODE_ENUM eResult = DATASERVICE_ERROR_CODE_NONE;

    if( hPayload == OSAL_INVALID_BUFFER_HDL )
    {
        SPORTS_SERVICE_MGR_vPrint( 1,
            SPORTS_SERVICE_MGR_OBJECT_NAME": Empty payload\n");
        return eResult;
    }

    // Ask the interface to process this message
    eResult = GsSportsServiceIntf.eProcessMessage(
        psObj->hInterface, &hPayload );

    return eResult;
}

/*****************************************************************************
*
*   bBuildDBFilePath
*
*****************************************************************************/
static BOOLEAN bBuildDBFilePath (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    size_t tSMSPathLen;
    const char *pacSMSPath;
    size_t tSportsServiceDBPathLen;

    pacSMSPath = SMS_pacGetPath();
    if (pacSMSPath == NULL)
    {
        return FALSE;
    }

    tSMSPathLen = strlen(pacSMSPath);

    // The database file path len is the length of the SMS path
    // plus the name of the database file plus
    // the delimiter character.
    tSportsServiceDBPathLen = tSMSPathLen +
                     sizeof(SPORTS_SERVICE_DATABASE_FOLDER) +
                     sizeof(SPORTS_SERVICE_DATABASE_FILENAME)+ 2;
                     // + 2 for the '/' characters


    // Now we know the size of the path, so we can now
    // allocate the proper amount of memory
    psObj->pacFullyQualifiedDatabaseFilePath =
        (char *) SMSO_hCreate(
            SPORTS_SERVICE_MGR_OBJECT_NAME":DBFilePath",
            tSportsServiceDBPathLen,
            SMS_INVALID_OBJECT, FALSE );

    // Ensure allocation succeeded
    if (psObj->pacFullyQualifiedDatabaseFilePath == NULL)
    {
        return FALSE;
    }

    // Construct the full database filename
    return bSnprintf(   &psObj->pacFullyQualifiedDatabaseFilePath[0],
                        tSportsServiceDBPathLen,
                        "%s/%s/%s",
                        pacSMSPath,
                        SPORTS_SERVICE_DATABASE_FOLDER,
                        SPORTS_SERVICE_DATABASE_FILENAME );
}

/*****************************************************************************
*
*   bUnsafeLoadSportsConfigDataFromDB
*    returns true on successful retrieval of data
*
*****************************************************************************/
static BOOLEAN bUnsafeLoadSportsConfigDataFromDB(
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN *pbError
        )
{
    BOOLEAN bOk;

    // load the Config data in to my in-memory structs
    *pbError = TRUE;    // error

    //query database for SPORTS_SERVICE_MGR_CONFIG_STRUCT SC_Config;
    psObj->sSC_Config.bSuccess = TRUE;
    bOk = SQL_INTERFACE.bQuery (
            psObj->hSQLConnection,
            SS_TABLE_GET_SC_CONFIG_VER,
            (SQL_QUERY_RESULT_HANDLER)bProcessGetSCVerResult,
            &psObj->sSC_Config );

    if(!bOk || !psObj->sSC_Config.bSuccess)
    {
        return FALSE;
    }

    DATASERVICE_IMPL_vLog(
                SPORTS_SERVICE_MGR_OBJECT_NAME
                ": Loaded SC_Config: CfgVer = %d",
                psObj->sSC_Config.un8ConfigVersion);

    //query database for SPORTS_SERVICE_AF_CONFIG_STRUCT AF_Config;
    psObj->sAF_Config.bSuccess = TRUE;
    bOk = SQL_INTERFACE.bQuery (
            psObj->hSQLConnection,
            SS_TABLE_GET_AF_CONFIG_VER,
            (SQL_QUERY_RESULT_HANDLER)bProcessGetAFVerResult,
            &psObj->sAF_Config );

    if(!bOk || !psObj->sAF_Config.bSuccess)
    {
        return FALSE;
    }

    DATASERVICE_IMPL_vLog(
                SPORTS_SERVICE_MGR_OBJECT_NAME
                ": Loaded AF_Config: CfgVer = %d",
                psObj->sAF_Config.un8AfCfgVersion);

    return TRUE;
}


/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/

static void vSetError (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj,eErrorCode);

    return;
}

/*******************************************************************************
*
*   eCheckDataSetUsage
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCheckDataSetUsage (
        SPORTS_SERVICE_OBJECT hSportsService,
        char *pcDest,
        UN32 *pun32Size )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj =
                (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Validate the object
    BOOLEAN bOk = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hSportsService );

    if ((bOk != TRUE) || (pcDest == NULL) || (pun32Size == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // The call is only valid inside eUseDataSet
    if (psObj->bDataSetInUse == FALSE)
    {
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*******************************************************************************
*
*   bPerformQuery
*
*******************************************************************************/
static BOOLEAN bPerformQuery (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcOpenTag,
    char *pcCloseTag,
    char *pcQueryString,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult,
    SQL_QUERY_RESULT_HANDLER tRowHandler )
{
    BOOLEAN bOk;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    // Write the open tag

    // CSH - Add "%s" formatter string to avoid warning on linux
    // with gcc.
    eSmsApiCode = eSports_Service_snprintf(psResult,"%s",
                            pcOpenTag);

    if (eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }
    // Query the data set
    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        pcQueryString,
        tRowHandler,
        psResult );

    // SQL was successful
    if( bOk == TRUE )
    {
        // The row processing was OK or never called because of an empty result
        // set.
        if (psResult->eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Write the close tag. Re-calculate the size since the query may
            // have bumped the cursor
            eSmsApiCode = eSports_Service_snprintf(psResult,"%s",
                            pcCloseTag);
                            if (eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
                return FALSE;
            }
        }
    }
    return bOk;
}

/*******************************************************************************
*
*   bExecuteCommandWithString
*
*******************************************************************************/
static BOOLEAN bExecuteCommandWithString(
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacCommandString,
        char *pacTemplateString )
{
    return SQL_INTERFACE.bExecutePreparedCommand (
        hSQL,
        pacCommandString,
        1,
        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareCommandWithString,
        pacTemplateString );
}

/*****************************************************************************
*
*   bPrepareCommandWithString
*
*****************************************************************************/
static BOOLEAN bPrepareCommandWithString (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    char *pacTemplateString
        )
{
    /* Note: "column" in the naming is a little misleading.
     * Prepared statements use templates for filling in values.
     * The index indicates the templates position from start to end
     * in the SQL statement.  SQLite template indexing is one based.
     * We loop zero based and add 1 to the index when we call the
     * sqlite3_bind_xxx call.
     */
    // We only call this with one index.
    if(tIndex != 0)
    {
        return FALSE;
    }
    *peType = SQL_BIND_TYPE_C_STRING;
    *ppvData = (void *)pacTemplateString;
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectSportList
*
*******************************************************************************/
static BOOLEAN bProcessSelectSportList (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    SPORT_ID tSportId;
    char *pacName;
    BOOLEAN bIsSupported;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // All columns must be present
    if( n32NumberOfColumns != SS_TABLE_SPORTS_NAMES_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Get the sport id
    tSportId =
        (SPORT_ID)psColumn[SS_TABLE_SPORTS_NAMES_SPORT_ID].uData.sUN32.un32Data;

    // See if we support this sport.
    bIsSupported = bIsSportSupported( psResult->psSportsObj, tSportId );
    if( !bIsSupported )
    {
        // Unsupported sports are skipped.  The row count is not incremented so
        // we will properly return an api return code indicating an empty
        // result set.  Keep processing rows.
        return TRUE;
    }

    // Get the sport name
    pacName =
        (char*)psColumn[SS_TABLE_SPORTS_NAMES_SPORT_NAME].uData.sCString.pcData;

    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                    psResult,
                    SS_TAG_SPORT_LIST_ENTRY_OPEN
                    SS_TAG_ID_OPEN"%u"SS_TAG_ID_CLOSE
                    SS_TAG_NAME_OPEN"%s"SS_TAG_NAME_CLOSE
                    SS_TAG_SPORT_LIST_ENTRY_CLOSE,
                    tSportId,
                    pacName);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->un32RowCount++;
    }
    else
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }
    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectAffiliateList
*
*******************************************************************************/
static BOOLEAN bProcessSelectAffiliateList (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    AFFILIATE_ID tAffiliateId;
    char *pacName;
    SPORT_ID tSportId;
    BOOLEAN bIsSupported;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // All columns must be present
    if( n32NumberOfColumns != SS_SELECT_GET_AFFILIATE_LIST_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Get the sport id.
    tSportId =
        (SPORT_ID)
        psColumn[SS_SELECT_GET_AFFILIATE_LIST_SPORT_ID].uData.sUN32.un32Data;

    // See if we support this sport.
    bIsSupported = bIsSportSupported( psResult->psSportsObj, tSportId );
    if( !bIsSupported )
    {
        // Affiliates of unsupported sports are skipped.  The row count is not
        // incremented so we will properly return an api return code indicating
        // an empty result set.  Keep processing rows.
        return TRUE;
    }

    //
    // Each row is an affiliate element.
    //

    // Write the opening affiliate tag
    tAffiliateId =
        (AFFILIATE_ID)
        psColumn[SS_SELECT_GET_AFFILIATE_LIST_AFFILIATE_ID].uData.sUN32.un32Data;

    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                    psResult,
                    SS_TAG_AFFILIATE_OPEN
                    SS_TAG_ID_OPEN"%u"SS_TAG_ID_CLOSE,
                    tAffiliateId );

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Write the name
    pacName =
        (char*)
        psColumn[SS_SELECT_GET_AFFILIATE_LIST_AFFILIATE_NAME]
                                         .uData.sCString.pcData;

    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                    psResult,
                    SS_TAG_NAME_OPEN"%s"SS_TAG_NAME_CLOSE,
                    pacName );

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }


    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                    psResult,
                    SS_TAG_SPORT_ID_OPEN"%u"SS_TAG_SPORT_ID_CLOSE,
                    tSportId );

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Write the league id.  It may be NULL make sure it is present.
    if( psColumn[SS_SELECT_GET_AFFILIATE_LIST_LEAGUE_ID].eType ==
        SQL_COLUMN_TYPE_INTEGER )
    {
        LEAGUE_ID tLeagueId =
        (LEAGUE_ID)
        psColumn[SS_SELECT_GET_AFFILIATE_LIST_LEAGUE_ID].uData.sUN32.un32Data;

        // Write the result.
        eReturnCode = eSports_Service_snprintf(
                    psResult,
                    SS_TAG_LEAGUE_ID_OPEN"%u"SS_TAG_LEAGUE_ID_CLOSE,
                    tLeagueId );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
            return FALSE;
        }
    }

    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_AFFILIATE_CLOSE );

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    psResult->un32RowCount++;

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectAffiliateInfoClassList
*
*******************************************************************************/
static BOOLEAN bProcessSelectAffiliateInfoClassList (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    INFORMATION_CLASS tInformationClass;
    BOOLEAN bIsSupported;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // All columns must be present
    if( n32NumberOfColumns !=
        SS_SELECT_GET_AFFILIATE_INFO_CLASS_LIST_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Get the information class
    tInformationClass =
            (INFORMATION_CLASS)
            psColumn[SS_SELECT_GET_AFFILIATE_INFO_CLASS_LIST_TABLE_CLASS]
            .uData.sUN32.un32Data;

    bIsSupported = bIsTableClassSupported( psResult->psSportsObj,
                                           tInformationClass );

    if( !bIsSupported )
    {
        // Unsupported classes are skipped.  The row count is not
        // incremented so we will properly return an api return code indicating
        // an empty result set.  Keep processing rows.
        return TRUE;
    }

    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_INFORMATION_CLASS_OPEN"%u"SS_TAG_INFORMATION_CLASS_CLOSE,
                        tInformationClass );
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    psResult->un32RowCount++;

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectTableHeaderFixed
*
*******************************************************************************/
static BOOLEAN bProcessSelectTableHeaderFixed (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    UN32 un32SeasonStatus;
    INFORMATION_CLASS tInformationClass;
    UN32 un32Epoch;
    UN32 un32ReceiptTime;
    BOOLEAN bIsSupported;
    size_t tLength;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // All columns must be present.
    if( n32NumberOfColumns != SS_SELECT_GET_TABLE_HEADER_FIXED_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Record the tdef key.  It saves a second query and/or inner join.  Do it before
    // supported argument so it is always valid if the query is successful.
    psResult->un32CarryOver = psColumn[SS_SELECT_GET_TABLE_HEADER_FIXED_TDEFKEY]
                             .uData.sUN32.un32Data;

    tInformationClass =
        (INFORMATION_CLASS)
        psColumn[SS_SELECT_GET_TABLE_HEADER_FIXED_TABLE_CLASS]
        .uData.sUN32.un32Data;

    bIsSupported = bIsTableClassSupported( psResult->psSportsObj,
                                           tInformationClass );

    if( !bIsSupported )
    {
        // Unsupported classes are skipped.  The row count is not
        // incremented so we will properly return an api return code indicating
        // an empty result set.  Should only be one.
        return FALSE;
    }

    // Write the title if present.
    if( psColumn[SS_SELECT_GET_TABLE_HEADER_FIXED_ITITLE].eType ==
        SQL_COLUMN_TYPE_C_STRING )
    {
        char *pacTitle =
            (char*)psColumn[SS_SELECT_GET_TABLE_HEADER_FIXED_ITITLE]
            .uData.sCString.pcData;

        tLength = strlen( pacTitle );
        if(tLength > 0)
        {
            // Write the result.
            eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_NAME_OPEN"%s"SS_TAG_NAME_CLOSE,
                        pacTitle );
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
                return FALSE;
            }
        }
    }

    // Write the season status.
    un32SeasonStatus =
        psColumn[SS_SELECT_GET_TABLE_HEADER_FIXED_SEASON_STATUS]
        .uData.sUN32.un32Data;

    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_SEASON_STATUS_OPEN"%u"SS_TAG_SEASON_STATUS_CLOSE,
                        un32SeasonStatus );
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }
    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                        psResult,
                       SS_TAG_INFORMATION_CLASS_OPEN"%u"SS_TAG_INFORMATION_CLASS_CLOSE,
                        tInformationClass );
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Write the epoch.
    un32Epoch =
        psColumn[SS_SELECT_GET_TABLE_HEADER_FIXED_EPOCH].uData.sUN32.un32Data;

    // Write the result.
    eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_EPOCH_OPEN"%u"SS_TAG_EPOCH_CLOSE,
                        un32Epoch );

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    psResult->un32RowCount++;

    // Write the receipt time.
    un32ReceiptTime =
            psColumn[SS_SELECT_GET_TABLE_HEADER_FIXED_RECEIPT_TIME]
            .uData.sUN32.un32Data;

    eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_RECEIPT_TIME_OPEN"%u"SS_TAG_RECEIPT_TIME_CLOSE,
                        un32ReceiptTime );

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_HEADER_KEY_OPEN"%u"SS_TAG_HEADER_KEY_CLOSE,
                        psResult->un32CarryOver );

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Should only be one.
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectTableHeaderVariable
*
*******************************************************************************/
static BOOLEAN bProcessSelectTableHeaderVariable (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    UN32 un32Index;
    UN32 un32LabelType;
    UN32 un32Priority;
    UN32 un32LbSize;
    size_t tLength;
    BOOLEAN bIsSupported;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // All columns must be present.
    if( n32NumberOfColumns != SS_SELECT_GET_TABLE_HEADER_VARIABLE_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Get first since we cull based on priority.
    un32Priority =
        psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_LPRIORITY]
        .uData.sUN32.un32Data;

    un32LabelType =
            psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_LBTYPE]
            .uData.sUN32.un32Data;

    bIsSupported = bIsLabelPrioritySupported( psResult->psSportsObj,
                                              un32Priority );

    bIsSupported &= bIsLabelSupported( psResult->psSportsObj,
                                       un32LabelType );

    if( !bIsSupported )
    {
        // Unsupported priorities and labels are skipped.  The row count is not
        // incremented.
        return TRUE;
    }

    // Get the fields that are always present.
    un32Index =
        psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_ORDER_INDEX]
        .uData.sUN32.un32Data;

    un32LbSize =
        psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_LSIZE]
        .uData.sUN32.un32Data;

    eReturnCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_COLUMN_DESC_OPEN
                        SS_TAG_ID_OPEN"%u"SS_TAG_ID_CLOSE
                        SS_TAG_LABEL_OPEN"%u"SS_TAG_LABEL_CLOSE
                        SS_TAG_PRIORITY_OPEN"%u"SS_TAG_PRIORITY_CLOSE
                        SS_TAG_LBSIZE_OPEN"%u"SS_TAG_LBSIZE_CLOSE,
                        un32Index,
                        un32LabelType,
                        un32Priority,
                        un32LbSize );
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Add the title if present.
    if( psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_LBTXT].eType ==
        SQL_COLUMN_TYPE_C_STRING )
    {
        char *pacTitle =
            (char*)psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_LBTXT]
            .uData.sCString.pcData;

        tLength = strlen( pacTitle );
        if(tLength > 0)
        {
            eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_NAME_OPEN"%s"SS_TAG_NAME_CLOSE,
                            pacTitle );
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
                return FALSE;
            }
        }
    }

    // Add the dmod if present.
    if( psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_DMOD].eType ==
        SQL_COLUMN_TYPE_INTEGER )
    {
        UN32 un32Dmod =
            psColumn[SS_SELECT_GET_TABLE_HEADER_VARIABLE_DMOD]
            .uData.sUN32.un32Data;

        eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_DMOD_OPEN"%u"SS_TAG_DMOD_CLOSE,
                            un32Dmod );
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
            return FALSE;
        }
    }

    //
    // TODO: Handle extended binary data.  base64 encode HERE
    // SS_TAG_BINARY_DATA_OPENxxxSS_TAG_BINARY_DATA_CLOSE

    eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_COLUMN_DESC_CLOSE);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    psResult->un32RowCount++;

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectTable
*
*******************************************************************************/
static BOOLEAN bProcessSelectTable (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    N32 n32CurrentSize;
    char *pacRewind;
    UN32 un32Field;
    char *pacField;
    N32 n32Index;
    BOOLEAN bEmitColumn;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // At least the RowID and InstanceId must be present.
    if( n32NumberOfColumns < SS_SELECT_SELECT_TABLE_MIN_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // The number if columns with data must match what we specify.  We set expected
    // based on the tdef that describes the table.
    if( (unsigned)(n32NumberOfColumns - SS_SELECT_SELECT_TABLE_MIN_FIELDS) !=
        psResult->un32ExpectedColumnCount )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }


    // Keep a rewind point in case the row does not fit and we need to
    // truncate.
    pacRewind = psResult->pacCursor;

    eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_TABLE_ROW_OPEN);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    }

    // Loop each column skipping the first two which are the RowID and InstanceID
    for(n32Index=SS_SELECT_SELECT_TABLE_MIN_FIELDS; n32Index<n32NumberOfColumns; n32Index++)
    {
        bEmitColumn = bBitfieldGet(
                psResult->pun8ColumnMap,
                psResult->un32ColumnMapSize,
                n32Index - SS_SELECT_SELECT_TABLE_MIN_FIELDS );

        if( !bEmitColumn )
        {
            continue;
        }


        if(psColumn[n32Index].eType == SQL_COLUMN_TYPE_INTEGER)
        {
            un32Field = psColumn[n32Index].uData.sUN32.un32Data;

            eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_TABLE_ROW_DATA_OPEN"%u"SS_TAG_TABLE_ROW_DATA_CLOSE,
                            un32Field );
        }
        else if(psColumn[n32Index].eType == SQL_COLUMN_TYPE_C_STRING)
        {
            pacField = (char*)psColumn[n32Index].uData.sCString.pcData;

            eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_TABLE_ROW_DATA_OPEN"%s"SS_TAG_TABLE_ROW_DATA_CLOSE,
                            pacField );

        }
        else
        {
            eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_TABLE_ROW_DATA_OPEN""SS_TAG_TABLE_ROW_DATA_CLOSE );
        }
    }

    if( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Write the closing row tag.

    eReturnCode = eSports_Service_snprintf(
                            psResult,
                            SS_TAG_TABLE_ROW_CLOSE );

    if( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Calculate space for output if we are writing
    // to the output buffer
    n32CurrentSize =
            psResult->n32ResultBufferSize -
            (psResult->pacCursor - psResult->pacScratchPad);

    // Make sure there is room for the closing tag of the table or the row does
    // not count.
    if( n32CurrentSize < (N32)sizeof(SS_TAG_TABLE_CLOSE)+1 )
    {
        if (  (psResult->un32RowCount != 0)
            &&(psResult->un32LastRowStartPos == 0)
           )
        {
            psResult->un32LastRowStartPos = pacRewind - psResult->pacScratchPad;
        }
    }

    psResult->un32RowCount++;
    return TRUE;
}

/*******************************************************************************
*
*   bProcessGetInstanceIdResult
*
*     This function is provided to the SQL_INTERFACE_OBJECT in order to process
*     a "select InstanceId" query made on the database info relation.
*
*******************************************************************************/
static BOOLEAN bProcessGetInstanceIdResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT *psResult
        )
{
    // Query should return only two columns, one row (a single instance ID
    // and CRC)
    if ( n32NumberOfColumns != SS_SELECT_GET_INSTANCE_ID_AND_CRC_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        psResult->un32RowCount = 0;
        return FALSE;
    }

    // Get the instance id - should be the 1st column returned
    psResult->n64InstanceId =
            psColumn[SS_SELECT_GET_INSTANCE_ID_AND_CRC_ID].n64NativeInteger;
    // Get the Crc - should be the 2nd column returned
    psResult->un32Crc =
            psColumn[SS_SELECT_GET_INSTANCE_ID_AND_CRC_CRC].uData.sUN32.un32Data;

    psResult->un32DefKey =
            psColumn[SS_SELECT_GET_INSTANCE_ID_AND_CRC_TDEFKEY].uData.sUN32.un32Data;

    // Update the status
    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
    psResult->un32RowCount = 1;

    // We only want one row, one col of data
    return FALSE;
}

/*******************************************************************************
*
*   bProcessGetTdefKeyResult
*
*     This function is provided to the SQL_INTERFACE_OBJECT in order to process
*     a "select TdefKey" query made on the database info relation.
*
*******************************************************************************/
static BOOLEAN bProcessGetTdefKeyResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT *psResult
        )
{
    // Query should return only one column, one row (a unique TdefKey)
    if ( n32NumberOfColumns != SS_SELECT_GET_TABLE_KEY_AND_CLASS_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        psResult->un32RowCount = 0;
        return FALSE;
    }

    // Get the TdefKey - should be the 1st column returned
    psResult->un32DefKey = psColumn[SS_SELECT_GET_TABLE_KEY_AND_CLASS_KEY]
                                                        .uData.sUN32.un32Data;

    // Get the Table Class while we are here
    psResult->un8TableClass =
            (UN8) psColumn[SS_SELECT_GET_TABLE_KEY_AND_CLASS_CLASS]
                                                           .uData.sUN32.un32Data;

    // Update the status
    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
    psResult->un32RowCount = 1;

    // We only want one row, one col of data
    return FALSE;
}

/*******************************************************************************
*
*   bProcessGetLastRowIdResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select InstanceId" query made on the database info relation.
*
*******************************************************************************/
static BOOLEAN bProcessGetLastRowIdResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_ROW_ID_QUERY_RESULT_STRUCT *psResult
        )
{
    // Query should return only one column, one row (a single row ID)
    if ( n32NumberOfColumns != 1 )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        psResult->un32RowCount = 0;
        return FALSE;
    }

    // Get the row id - should be the only column returned
    psResult->n64LastRowId = psColumn[0].n64NativeInteger;

    // Update the status
    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
    psResult->un32RowCount = n32NumberOfColumns;

    // We only want one row, one col of data
    return FALSE;
}

/*******************************************************************************
*
*   bBeginTableUpdate
*
*******************************************************************************/
static BOOLEAN bBeginTableUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
    BOOLEAN bIsPrimaryTable,
    char *acSavePointName
        )
{
    SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    BOOLEAN bOk;
    BOOLEAN bUpdateDb = FALSE;
    BOOLEAN bCreateInstance = FALSE;
    BOOLEAN bShouldRollback = FALSE;
    N32 n32CurrentSize;
    char *pacCursor;
    N32 n32Written;
    UN32 un32UTCsec = 0;

    psTransaction->hStatement = SQL_PREPARED_STATEMENT_INVALID_HANDLE;

    psTableInstance->bTableComplete = FALSE;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
        "bBeginTableUpdate:"
        "Crc: %u AfId: %u Epoch: %u Title %s TabID: %u TabVer: %u"
        " Season: %u  RowCnt: %u ColCnt %u\n",
        psTableInstance->un32Crc,
        (UN32)psTableInstance->un16AffiliateId,
        (UN32)psTableInstance->un16Epoch,
        psTableInstance->InstanceTitle,
        (UN32)psTableInstance->un16TableId,
        (UN32)psTableInstance->un8TableVer,
        (UN32)psTableInstance->bInSeason,
        (UN32)psTableInstance->un8RowCount,
        psTableInstance->un32ColumnCount );

    if (bIsPrimaryTable)
    {
        // Lock the database so we can determine if we have to update

        // Take read semaphore.
        eOsalCode = OSAL.eSemTake(
                psSportsObj->hDbReadSem,
            OSAL_OBJ_TIMEOUT_INFINITE );
        if (eOsalCode != OSAL_SUCCESS)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate: End bOk: 0\n" );
            return FALSE;
        }
    }

    do{

        // Determine if we can handle an update
        // i.e. do we have a table def for this data?
        // if not, we have to reject the data
        // Initialize the result building structure
        vInitializeInstanceIdQueryStruct(&sResult);

        bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SS_SELECT_GET_TABLE_KEY_AND_CLASS,
                            psTableInstance->un16TableId,
                            psTableInstance->un8TableVer );

        if( bOk == FALSE )
        {
            break;
        }

        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf,
            (SQL_QUERY_RESULT_HANDLER)bProcessGetTdefKeyResult,
            &sResult );

        if( (bOk == FALSE) || (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_ERROR) ||
                (sResult.un32RowCount == 0) )
        {
            if (sResult.un32RowCount == 0)
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "No table def for TableID: %d  TabVer: %d\n",
                    psTableInstance->un16TableId,
                    psTableInstance->un8TableVer);
            }
            bOk = FALSE;
            break;

        } else {
            // now we have the key we can continue
            psTableInstance->un32TdefKey = sResult.un32DefKey;
            psTableInstance->un8Tclass = sResult.un8TableClass;
            vInitializeInstanceIdQueryStruct(&sResult);
        }

        //
        // Determine if an update is needed.
        //

        // Build the query string.
        bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SS_SELECT_GET_INSTANCE_ID_AND_CRC,
                            psTableInstance->un16TableId,
                            psTableInstance->un16AffiliateId,
                            psTableInstance->un16Epoch );

        if( bOk == TRUE )
        {
            bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bProcessGetInstanceIdResult,
                &sResult );
        }

        // SQL error nothing we can do for this table instance.
        if( (bOk == FALSE) || (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_ERROR) )
        {
            bOk = FALSE;
            break;
        }

        // Determine if we are going to update the db.
        if( sResult.un32RowCount == 0 )
        {
            // No entry exists so update.
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate:"
                "Need to Create new table inst\n");
            bUpdateDb = TRUE;
            bCreateInstance = TRUE;
        }
        else
        {
            // Set this now so debug statements can provide an instance id.
            psTableInstance->n64InstanceId = sResult.n64InstanceId;

            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate:"
                "Crc Check: InstId: %lld TdefKey: %u NewInstCrc: %u dbCrc: %u\n",
                sResult.n64InstanceId,
                sResult.un32DefKey,
                psTableInstance->un32Crc,
                sResult.un32Crc );

            // Have an entry only update if crc is different and requested.
            if ( psTableInstance->un32Crc != sResult.un32Crc)
            {
                bUpdateDb = TRUE;
            }
        }

        // Get the current time of day.
        eOsalCode = OSAL.eTimeGet( &un32UTCsec );
        if( eOsalCode != OSAL_SUCCESS )
        {
            printf("Result: %s\n",
                   OSAL.pacGetReturnCodeName(eOsalCode));
            un32UTCsec = 0;
        }
        psTableInstance->un32UTCsec = un32UTCsec;

        // No update is required.
        if( bUpdateDb == FALSE )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate:"
                "No table update required. instid %llu unixtime %u\n",
                psTableInstance->n64InstanceId,
                psTableInstance->un32UTCsec );

            vUnsafeUpdateTableReceiptTime(hSportsService,
                                    psTableInstance,
                                    psTransaction,
                                    bIsPrimaryTable );

            bOk = FALSE;
            break;
        }

        //
        // We are going to attempt to create a new table or update an existing
        // table.
        //

        // SET ROLLBACK POINT HERE!!
        if (bIsPrimaryTable)
        {
            // Set a rollback point
            if (!bUnsafeSetDbSavepoint(
                    hSportsService,
                     acSavePointName
                     ))
            {
                bOk = FALSE;
                break;
            }
            bShouldRollback = TRUE;
        }

        if ( bCreateInstance == TRUE )
        {
            // add a row to the DI_Info table
            bOk = bSnprintf(
                &psSportsObj->acInternalSqlBuf[0],
                sizeof(psSportsObj->acInternalSqlBuf),
                SS_INSERT_DATA_INFO_ROW,
                psTableInstance->un16Epoch,
                psTableInstance->un16AffiliateId,
                psTableInstance->bInSeason,
                ((psTableInstance->un16TableId<<16)+psTableInstance->un8TableVer),
                psTableInstance->un8Tclass,
                psTableInstance->un16TableId );

            if( bOk == TRUE )
            {
                bOk = bExecuteCommandWithString(
                        psSportsObj->hSQLConnection,
                        &psSportsObj->acInternalSqlBuf[0],
                        psTableInstance->InstanceTitle );
            }

            if( bOk == FALSE )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "Failed to bExecuteCommand: %s\n",
                    psSportsObj->acInternalSqlBuf );
                break;
            }

            // populate sResult from the new row entry
            bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                                sizeof(psSportsObj->acInternalSqlBuf),
                                SS_SELECT_GET_INSTANCE_ID_AND_CRC,
                                psTableInstance->un16TableId,
                                psTableInstance->un16AffiliateId,
                                psTableInstance->un16Epoch );

            if( bOk == TRUE )
            {
                // Initialize the result building structure
                vInitializeInstanceIdQueryStruct(&sResult);
                bOk = SQL_INTERFACE.bQuery (
                    psSportsObj->hSQLConnection,
                    psSportsObj->acInternalSqlBuf,
                    (SQL_QUERY_RESULT_HANDLER)bProcessGetInstanceIdResult,
                    &sResult );
            }

            // An internal error.
            if( (bOk == FALSE) ||
                (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_ERROR) )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "Failed to bQuery: %s return code: %u\n",
                    psSportsObj->acInternalSqlBuf,
                    sResult.eSmsApiCode);
                bOk = FALSE;
                break;
            }

            // We failed to find the instance we just added.
            if( sResult.un32RowCount == 0 )
            {
                DATASERVICE_IMPL_vLog(
                        SPORTS_SERVICE_MGR_OBJECT_NAME
                        ": Failed to get instance for table just added."
                        " TDef %u TDefVer %u\n",
                        psTableInstance->un16TableId,
                        psTableInstance->un8TableVer );

                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "Failed to get instance for table just added."
                    " TDef %u TDefVer %u\n",
                    psTableInstance->un16TableId,
                    psTableInstance->un8TableVer );

                bOk = FALSE;
                break;
            }
        }
        else
        {
            //
            // ASSERT(bIsPrimaryTable);
            //

            //
            // Remove any tref tables.
            //
            bOk = bUnsafeRemoveInstanceTables(
                        hSportsService,
                        SS_GET_TREF_REMOVE_INFO,
                        SS_DELETE_TREF_INSTANCES,
                        FALSE,
                        sResult.n64InstanceId );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "Failure to remove trefs\n" );
                break;
            }

            //
            // We are updating an instance so
            // remove all old instances of data for this InstanceId
            // from the DI_{defID}{defVer} table
            //
            bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                                sizeof(psSportsObj->acInternalSqlBuf),
                                SS_DELETE_OLD_INSTANCE_DATA_BY_TDEFKEY,
                                sResult.un32DefKey,
                                sResult.n64InstanceId );

            if( bOk )
            {
                // the new crc will be written to the table
                // after the data has been written
                bOk = SQL_INTERFACE.bExecuteCommand(
                        psSportsObj->hSQLConnection,
                        &psSportsObj->acInternalSqlBuf[0]);
            }

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "Failed bExecuteCommand:%s\n",
                    psSportsObj->acInternalSqlBuf );
                break;
            }

            //
            // Remove any label overrides for this table.
            //
            bOk = bSnprintf(
                            &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SS_DELETE_LABEL_OVERRIDES,
                            sResult.n64InstanceId );

            if( bOk)
            {
                bOk = SQL_INTERFACE.bExecuteCommand(
                                    psSportsObj->hSQLConnection,
                                    psSportsObj->acInternalSqlBuf );
            }

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "Failed bExecuteCommand:%s\n",
                    psSportsObj->acInternalSqlBuf );
                break;
            }
        }

        psTableInstance->n64InstanceId = sResult.n64InstanceId;

        //
        // Build a prepared statement to fill in the rows.
        //

        // Build the statement string.
        n32CurrentSize = sizeof(psSportsObj->acInternalSqlBuf);
        pacCursor = &psSportsObj->acInternalSqlBuf[0];
        bOk = FALSE;

        n32Written = snprintf(
            pacCursor,
            (size_t)n32CurrentSize,
            SS_INSERT_TABLE_ROW_BEGIN,
            psTableInstance->un16TableId,
            psTableInstance->un8TableVer,
            psTableInstance->n64InstanceId );

        if( n32Written < 0 )
        {
            // Keep bOk = FALSE if snprintf failed.
        }
        else if( n32Written < n32CurrentSize )
        {
            UN32 un32Count;
            pacCursor += n32Written;
            n32CurrentSize -= n32Written;

            for(un32Count=0; un32Count<psTableInstance->un32ColumnCount; un32Count++)
            {
                n32Written = snprintf(
                                    pacCursor,
                                    (size_t)n32CurrentSize,
                                    ",?" );

                if( !bSnprintfOk(n32Written, (size_t)n32CurrentSize) )
                {
                    break;
                }
                pacCursor += n32Written;
                n32CurrentSize -= n32Written;
            }

            // If we wrote all template parameters finish the syntax.
            if( un32Count == psTableInstance->un32ColumnCount )
            {
                n32Written = snprintf(
                                    pacCursor,
                                    (size_t)n32CurrentSize,
                                    " );" );

                if( (n32Written >= 0) && (n32Written < n32CurrentSize) )
                {
                    bOk = TRUE;
                }
            }
        }

        if( bOk == FALSE )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate:"
                "Failed to build prepared statement string\n");
            break;
        }

        // Create prepared statement parameters holder
        psTransaction->un32ColumnCount = psTableInstance->un32ColumnCount;
        psTransaction->psParams = (SQL_BIND_PARAMETER_STRUCT *)OSAL.pvMemoryAllocate(
            SPORTS_SERVICE_MGR_OBJECT_NAME":BindParams",
            psTableInstance->un32ColumnCount * 
                sizeof(SQL_BIND_PARAMETER_STRUCT),
            FALSE);
        if (NULL == psTransaction->psParams)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate:"
                "Failed to allocate memory for bind parameters\n");
            bOk = FALSE;
            break;
        }
        else
        {
            UN32 un32Idx;

            // reset bind parameters to invalid values
            // just in order to avoid possible logical errors
            // when not all the parameters are specified
            for (un32Idx = 0; un32Idx < psTableInstance->un32ColumnCount; 
                un32Idx++)
            {
                psTransaction->psParams[un32Idx].eType = 
                    SQL_BIND_TYPE_C_STRING;
                psTransaction->psParams[un32Idx].pvData = NULL;
            }
        }
        // Create the prepared statement to update columns.
        psTransaction->hStatement = SQL_INTERFACE.hCreatePreparedStatement (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf );

        if ( psTransaction->hStatement == SQL_PREPARED_STATEMENT_INVALID_HANDLE )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate:"
                "hCreatePreparedStatement failure\n");
            OSAL.vMemoryFree(psTransaction->psParams);
            psTransaction->psParams = NULL;
            bOk = FALSE;
            break;
        }

        // The read semaphore is locked, a transaction has been started,
        // and a prepared statement is ready to bind and execute.

    }while (0);

    if (!bOk)
    {
        if( bUpdateDb == TRUE )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bBeginTableUpdate:"
                "Failure to add data instance\n");
        }

        if (bShouldRollback)
        {
            if (!bUnsafeRollbackDb(
                hSportsService,
                 acSavePointName
                 ))
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "DB ROLLBACK FAILED:%s \n", acSavePointName);
            }
            // release save pt
            if( !bUnsafeReleaseDbSavepoint(
                    hSportsService,
                    acSavePointName) )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate:"
                    "Failure db savepoint release:%s\n",
                   acSavePointName );
            }
        }

        // Unlock the database for reading if we are the primary table
        if (bIsPrimaryTable)
        {
            OSAL.eSemGive(psSportsObj->hDbReadSem);
        }
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bBeginTableUpdate: End"
                    " bOk: %u\n",
                    (UN32)bOk );
    return bOk;
}

/*******************************************************************************
*
*   bEndTableUpdate
*
*******************************************************************************/
static BOOLEAN bEndTableUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    BOOLEAN bCrcUpdateOk = TRUE;
    UN32 un32ReceiptTime;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
        "bEndTableUpdate:"
        "Crc: %u AfId: %u Epoch: %u Title %s TabID: %u TabVer: %u "
        "TdefKey %x TabClass %u"
        " Season: %u  RowCnt: %u ColCnt %u InstId: %lld TabCmpl %u \n",
        psTableInstance->un32Crc,
        (UN32)psTableInstance->un16AffiliateId,
        (UN32)psTableInstance->un16Epoch,
        psTableInstance->InstanceTitle,
        (UN32)psTableInstance->un16TableId,
        (UN32)psTableInstance->un8TableVer,
        psTableInstance->un32TdefKey,
        (UN32)psTableInstance->un8Tclass,
        (UN32)psTableInstance->bInSeason,
        (UN32)psTableInstance->un8RowCount,
        psTableInstance->un32ColumnCount,
        psTableInstance->n64InstanceId,
        (UN32)psTableInstance->bTableComplete );

    if( psTableInstance->bTableComplete == TRUE )
    {
        BOOLEAN bOk;

        // If the unix time is zero we failed to get it.  Make the receipt time the
        // table epoch in this case it is the closest thing we have.  It is at most off by
        // 24 hours so receipt time arguments will still be reasonable?
        un32ReceiptTime = (psTableInstance->un32UTCsec == 0) ?
                    psTableInstance->un16Epoch * (UN32)SS_SECONDS_PER_DAY :
                    psTableInstance->un32UTCsec;

        // Update CRC,receipt time, tdefkey and class.  The tdef may have changed.
        bOk = bSnprintf(
                &psSportsObj->acInternalSqlBuf[0],
                sizeof(psSportsObj->acInternalSqlBuf),
                SS_UPDATE_TABLE_CRC_AND_RECEIPT_TIME,
                psTableInstance->un32Crc,
                un32ReceiptTime,
                psTableInstance->un32TdefKey,
                psTableInstance->un8Tclass,
                psTableInstance->n64InstanceId );

        if( !bOk )
        {
            bCrcUpdateOk = FALSE;
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bEndTableUpdate:"
                "Failed to build sql string to save "
                "crc and receipt time for table instance %lld\n",
                    psTableInstance->n64InstanceId );
        }
        else
        {
            bCrcUpdateOk = SQL_INTERFACE.bExecuteCommand(
                psSportsObj->hSQLConnection,
                &psSportsObj->acInternalSqlBuf[0] );

            if ( bCrcUpdateOk == FALSE )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                    "bEndTableUpdate:"
                    "Failed to update crc for table instance %lld SQL:%s\n",
                    psTableInstance->n64InstanceId,
                    psSportsObj->acInternalSqlBuf );
            }

        }
    } else
    {
        bCrcUpdateOk = FALSE;
    }

    // End transaction.

    // Free prepared statement.
    SQL_INTERFACE.vDestroyPreparedStatement(
            psSportsObj->hSQLConnection,
            psTransaction->hStatement );

    // Free bind parameters
    vReleaseBindings(psTransaction);
    OSAL.vMemoryFree(psTransaction->psParams);
    psTransaction->psParams = NULL;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
            "bEndTableUpdate: End"
            " bCrcUpdateOk: %u\n",
            (UN32)bCrcUpdateOk );

    return (bCrcUpdateOk);
}

/*******************************************************************************
*
*   bBeginRowUpdate
*
*******************************************************************************/
static BOOLEAN bBeginRowUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
        )
{
    vReleaseBindings(psTransaction);
    return TRUE;
}

/*******************************************************************************
*
*   bEndRowUpdate
*
*******************************************************************************/
static BOOLEAN bEndRowUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
        )
{
    // Execute statement.
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    BOOLEAN bOk = SQL_INTERFACE.bExecutePreparedStatement(
            psSportsObj->hSQLConnection, psTransaction->hStatement,
            NULL, NULL,
            psTransaction->un32ColumnCount, psTransaction->psParams);

    if ( bOk == FALSE )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
            "bEndRowUpdate:"
            "bExecutePreparedStatement failure\n");
    }

    return bOk;
}

/*******************************************************************************
*
*   bUnsafeGetLastRowId
*
*******************************************************************************/
static BOOLEAN bUnsafeGetLastRowId(
    SPORTS_SERVICE_OBJECT hSportsService,
    N64 *n64RowId
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_ROW_ID_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));  // no strings to return
    bOk = SQL_INTERFACE.bQuery (
        psSportsObj->hSQLConnection,
        SS_SELECT_LAST_INSERT_ROWID,
        (SQL_QUERY_RESULT_HANDLER)bProcessGetLastRowIdResult,
        &sResult
        );

    if ( bOk && sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        *n64RowId = sResult.n64LastRowId;
    }

    return bOk;
}

/*******************************************************************************
*
*   bUpdateColumnInt
*
*******************************************************************************/
static BOOLEAN bUpdateColumnInt(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
    int iColumn,
    UN8 un8LPriority,
    UN8 un8LType,
    int iValue
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                    (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    bOk = bIsLabelPrioritySupported( psSportsObj, un8LPriority );
    bOk &= bIsLabelSupported( psSportsObj, un8LType );

    if( !bOk )
    {
        return TRUE;
    }

    if ((0 <= iColumn) && ((UN32)iColumn < psTransaction->un32ColumnCount))
    {
        psTransaction->psParams[iColumn].eType = SQL_BIND_TYPE_UN32;
        psTransaction->psParams[iColumn].pvData = (void *)(size_t)iValue;
    }
    else
    {
        bOk = FALSE;
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
            "bUpdateColumnInt: invalid column\n");
    }

    return bOk;
}

/*******************************************************************************
*
*   bUpdateColumnString
*
*******************************************************************************/
static BOOLEAN bUpdateColumnString(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
    int iColumn,
    UN8 un8LPriority,
    UN8 un8LType,
    STRING_OBJECT hString
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    bOk = bIsLabelPrioritySupported( psSportsObj, un8LPriority );
    bOk &= bIsLabelSupported( psSportsObj, un8LType );
    if( !bOk )
    {
        return TRUE;
    }

    // Add one to the column offset
    if ((0 <= iColumn) && ((UN32)iColumn < psTransaction->un32ColumnCount))
    {
        psTransaction->psParams[iColumn].eType = SQL_BIND_TYPE_STRING_OBJECT;
        psTransaction->psParams[iColumn].pvData = STRING.hDuplicate(hString);
    }
    else
    {
        bOk = FALSE;
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
            "bUpdateColumnString: invalid column\n");
    }

    return bOk;
}

/*******************************************************************************
*
*   bDbgDumpTable
*
*******************************************************************************/
static BOOLEAN bDbgDumpTable(
    SPORTS_SERVICE_OBJECT hSportsService,
    char *acTableName
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Lock the database for reading.
    eOsalCode = OSAL.eSemTake(  psSportsObj->hDbReadSem,
                                OSAL_OBJ_TIMEOUT_INFINITE);

    if (eOsalCode == OSAL_SUCCESS)
    {
        // retrieve the table whose name is supplied in pcCmdLine
        // Dump to the console tabulated column headers first
        // then dump every row tabulated
        SPORTS_SERVICE_MGR_vPrint( 1,
            "Dumping Data for table:%s\n", acTableName);

        bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SELECT_ALL_FROM_TABLE,
                            acTableName    );

        if( bOk == TRUE )
        {
            // Initialize the result building structure
            vInitializeQueryStruct(&sResult, psSportsObj, NULL, 0);    // no strings to return
            bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bDbgProcessSelectAllAndDump,
                &sResult );
        }

        // An internal error.
        if(bOk == FALSE)
        {
            SPORTS_SERVICE_MGR_vPrint( 1,
                ": Failed to bQuery:%s\n", acTableName);
        } else
        {
            SPORTS_SERVICE_MGR_vPrintSilent( 1, "\n");
        }

        // Unlock the database for reading
        OSAL.eSemGive(psSportsObj->hDbReadSem);

        return bOk;
    }

    return FALSE;
}

/*******************************************************************************
*
*   bDbgExecuteSqlCmd
*
*******************************************************************************/
static BOOLEAN bDbgExecuteSqlCmd(
    SPORTS_SERVICE_OBJECT hSportsService,
    char *pacSqlStatement
        )
{
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    BOOLEAN bOk=FALSE;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Lock the database for reading.
    eOsalCode = OSAL.eSemTake(psSportsObj->hDbReadSem, OSAL_OBJ_TIMEOUT_INFINITE);

    if (eOsalCode == OSAL_SUCCESS)
    {
        // retrieve the table whose name is supplied in pcCmdLine
        // Dump to the console tabulated column headers first
        // then dump every row tabulated
        printf("Processing SQL Statement:%s\n", pacSqlStatement);

        vInitializeQueryStruct(
                &sResult,
                psSportsObj,
                NULL,
                0);

        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            pacSqlStatement,
            (SQL_QUERY_RESULT_HANDLER)bDbgProcessSelectAllAndDump,
            &sResult );

        // Unlock the database for reading
        OSAL.eSemGive(psSportsObj->hDbReadSem);

    }

    return bOk;
}

/*******************************************************************************
*
*   bDbgDumpAllTableNames
*
*    Dump the names of all of the tables currently in the database
*
*******************************************************************************/
static BOOLEAN bDbgDumpAllTableNames(
    SPORTS_SERVICE_OBJECT hSportsService
        )
{
    //SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Lock the database for reading.
    eOsalCode = OSAL.eSemTake(  psSportsObj->hDbReadSem,
                                OSAL_OBJ_TIMEOUT_INFINITE);

    if (eOsalCode == OSAL_SUCCESS)
    {
        // retrieve the table whose name is supplied in pcCmdLine
        // Dump to the console tabulated column headers first
        // then dump every row tabulated
        printf("Dumping all Table Names:\n");

        bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            GET_TABLE_NAMES_FROM_MASTER_TABLE );

        if( bOk == TRUE )
        {
            // Initialize the result building structure
            // No buffer required because we are not returning any strings
            vInitializeQueryStruct(
                    &sResult,
                    psSportsObj,
                    NULL,
                    0);

            bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bDbgProcessSelectAllAndDump,
                &sResult );
        }

        // An internal error.
        if(bOk == FALSE)
        {
            printf("\n");
            printf(SPORTS_SERVICE_MGR_OBJECT_NAME
                ": Failed to bQuery SQLITE_MASTER\n");
        } else
        {
            printf("\n");
        }

        // Unlock the database for reading
        OSAL.eSemGive(psSportsObj->hDbReadSem);

        return bOk;
    }

    return FALSE;
}

/*******************************************************************************
*
*   bStartServiceConfigUpdate
*
*    Returns TRUE if config info is up to date
*    Returns FALSE if a config update is required
*
*******************************************************************************/
static BOOLEAN bStartServiceConfigUpdate(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN8 un8CfgVer,
        UN32 un32Crc
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    UN32 un32Error = 0;
    SQL_INTERFACE_OBJECT hSQLMemory = SQL_INTERFACE_INVALID_OBJECT;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
        "bStartServiceConfigUpdate:"
        "hSportsService %u un8CfgVer %u un32Crc %u"
        " ObjCfgVer %u Obj32Crc %u\n",
        (UN32)hSportsService,
        (UN32)un8CfgVer,
        un32Crc,
        (UN32)psSportsObj->sSC_Config.un8ConfigVersion,
        psSportsObj->sSC_Config.un32ConfigCrc );

    if ( ( psSportsObj->sSC_Config.un8ConfigVersion == un8CfgVer )
        && ( psSportsObj->sSC_Config.un32ConfigCrc == un32Crc ) )
    {
        return TRUE;
    }

    // Our config is out of date.  Blow away the old database and
    // create a new one.
    psSportsObj->bLockOutDueToConfigUpdate = TRUE;

    DATASERVICE_IMPL_vLog(
            SPORTS_SERVICE_MGR_OBJECT_NAME
            ":Service config change (START)" );

    // Take read semaphore.
    eOsalCode = OSAL.eSemTake(
        psSportsObj->hDbReadSem,
        OSAL_OBJ_TIMEOUT_INFINITE );
    if (eOsalCode != OSAL_SUCCESS)
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
            "bStartServiceConfigUpdate:"
            "Failure to take sem: %u\n",
            eOsalCode );

        return TRUE;
    }
#if 1
    //
    // Create a new memory db.  This allows us to start completely fresh.  Sqlite
    // never lets go of pages.  A vacuum opperation is the only way.  A new config
    // is a chance to drop pages if we create a new db.  We will replace the old
    // with it.
    //
    hSQLMemory = hCreateMemoryDb( psSportsObj, TRUE );
    if( hSQLMemory != SQL_INTERFACE_INVALID_OBJECT )
    {
        // Drop the current memory db.
        SQL_INTERFACE.vDisconnect( psSportsObj->hSQLConnection );
        // Set clean state.
        psSportsObj->sAF_Config.un8AfCfgVersion = 0;
        psSportsObj->sAF_Config.un32AfConfigCrc = 0;
        psSportsObj->un32DbConfigVersion = SPORTS_SERVICE_DATABASE_VERSION;
        psSportsObj->hSQLConnection = hSQLMemory;
    }
#endif
    // Create a save point.
    bOk = bUnsafeSetDbSavepoint( hSportsService, SS_ROLLBACK_POINT_SC_UPDATE );
    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
            "bStartServiceConfigUpdate:"
            "Failure to set DB savepoint: %s\n",
            SS_ROLLBACK_POINT_SC_UPDATE );

        OSAL.eSemGive(psSportsObj->hDbReadSem);
        return TRUE;    // fail quietly
    }

    // We successfully created a new db and have our save point we are done.
    if( hSQLMemory != SQL_INTERFACE_INVALID_OBJECT )
    {
        OSAL.eSemGive(psSportsObj->hDbReadSem);
        return FALSE;
    }

    //
    // For some reason creating a new db failed.  Try to keep using the old
    // one.
    //

    // will change to FALSE if we are able to prep the database for
    // change of config
    bOk = TRUE;

    do{
        // Clear configuration tables
        if (!bUnsafeClearTable(hSportsService, SS_TABLE_SC_CONFIG))
            break;
        un32Error = 1;
        if (!bUnsafeClearTable(hSportsService, SS_TABLE_SC_SPORTS_NAMES))
            break;
        un32Error = 2;
        // Clear Affiliates
        if (!bUnsafeClearTable(hSportsService, SS_TABLE_SA_AFFILIATIONS))
            break;
        un32Error = 3;
        if (!bUnsafeClearTable(hSportsService, SS_TABLE_AF_CONFIG))
            break;
        un32Error = 4;
        // clear the in-memory AF info - we will collect new OTA
        psSportsObj->sAF_Config.un8AfCfgVersion = 0;
        psSportsObj->sAF_Config.un32AfConfigCrc = 0;

        // Clear data instances (must do this before TD_Info because we
        // get table names from TD_Info)
        if (!bUnsafeDropAllDataInstanceTables(hSportsService))
            break;
        un32Error = 5;
        if (!bUnsafeClearTable(hSportsService, SS_TABLE_TABLE_INSTANCES))
            break;
        un32Error = 6;

        // Clear table defs
        // Highband will clear the in-memory defs when we return FALSE
        // from this function
        if (!bUnsafeClearTable(hSportsService, SS_TABLE_TD_INSTANCES))
            break;
        un32Error = 7;

        if (!bUnsafeClearTable(hSportsService, SS_TABLE_TD_LABEL_INFO))
            break;
        un32Error = 8;

        if (!bUnsafeClearTable(hSportsService, SS_TABLE_SC_TABLEINFO))
            break;
        un32Error = 9;

        if (!bUnsafeClearTable(hSportsService, SS_TABLE_INSTANCE_OVERRIDES))
            break;
        un32Error = 10;

        bOk = FALSE;

    } while (0);

     if (bOk)
     {
         BOOLEAN bReleaseOk;

         DATASERVICE_IMPL_vLog(
                 SPORTS_SERVICE_MGR_OBJECT_NAME
                 ":Config change failed to clear db: %u", un32Error );

         // TODO: set error condition state - DB no good
         // hSportsService->UhOhYouAreInABadCondition
         SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
             "bStartServiceConfigUpdate:"
             "Failure to clear database for SC change. un32Error %u\n",
              un32Error );

         bOk = bUnsafeRollbackDb( hSportsService, SS_ROLLBACK_POINT_SC_UPDATE );
         if( !bOk )
         {
             SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
                 "bStartServiceConfigUpdate:"
                 "Rollback Failure:%s\n",
                 SS_ROLLBACK_POINT_SC_UPDATE );
         }

         // Give up the save point since we are never going to try and
         // process the config change.
         bReleaseOk = bUnsafeReleaseDbSavepoint( hSportsService,
                                           SS_ROLLBACK_POINT_SC_UPDATE );
         if( !bReleaseOk )
         {
             SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
                 "bStartServiceConfigUpdate:"
                 "Cfg ReleaseSavePoint Failure:%\n",
                 SS_ROLLBACK_POINT_SC_UPDATE );
         }
     }

    OSAL.eSemGive(psSportsObj->hDbReadSem);
    return bOk;
}

/*******************************************************************************
*
*   bEndServiceConfigUpdate
*
*
*******************************************************************************/
static BOOLEAN bEndServiceConfigUpdate(
        SPORTS_SERVICE_OBJECT hSportsService,
        BOOLEAN bUpdateOk,
        UN8 un8CfgVer,
        UN32 un32Crc
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    BOOLEAN bCommitOk = FALSE;
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
        "bEndServiceConfigUpdate:"
        "hSportsService %u un8CfgVer %u un32Crc %u bUpdateOk %u"
        " ObjCfgVer %u Obj32Crc %u\n",
        (UN32)hSportsService,
        (UN32)un8CfgVer,
        un32Crc,
        (UN32)bUpdateOk,
        psSportsObj->sSC_Config.un8ConfigVersion,
        psSportsObj->sSC_Config.un32ConfigCrc );

    // Lock the db.
    eOsalCode = OSAL.eSemTake(
            psSportsObj->hDbReadSem,
            OSAL_OBJ_TIMEOUT_INFINITE );
    if (eOsalCode != OSAL_SUCCESS)
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
            "bEndServiceConfigUpdate:"
            "Failure to take sem:%u\n",
            eOsalCode );
        return FALSE;
    }

    if( bUpdateOk )
    {
        // write the sys config to the DB
        bCommitOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        SS_TABLE_UPDATE_SC_CONFIG_VALUES,
                        un8CfgVer,
                        un32Crc    );

        if( bCommitOk == TRUE )
        {
            bCommitOk = SQL_INTERFACE.bExecuteCommand (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf);
        }
    }

    if( !bUpdateOk || (bUpdateOk && !bCommitOk) )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
            "bEndServiceConfigUpdate:"
            "Rollback bUpdateOk %u bCommitOk %u\n",
            (UN32)bUpdateOk,
            (UN32)bCommitOk );

        bOk = bUnsafeRollbackDb( hSportsService, SS_ROLLBACK_POINT_SC_UPDATE );
        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
                "bEndServiceConfigUpdate:"
                "Rollback Failure:%s\n",
                SS_ROLLBACK_POINT_SC_UPDATE );
        }
    }

    bOk = bUnsafeReleaseDbSavepoint( hSportsService, SS_ROLLBACK_POINT_SC_UPDATE );
    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
            "bEndServiceConfigUpdate:"
            "ReleaseSavePoint Failure:%s\n",
            SS_ROLLBACK_POINT_SC_UPDATE );
    }

    // BUG 5439 Only use the config if we wrote it to DB
    if( bCommitOk )
    {
        psSportsObj->sSC_Config.un8ConfigVersion = un8CfgVer;
        psSportsObj->sSC_Config.un32ConfigCrc = un32Crc;
        vNotifySportListChange( psSportsObj );
        psSportsObj->bLockOutDueToConfigUpdate = FALSE;
        OSAL.eSemGive(psSportsObj->hDbReadSem);
        vProcessMonitorChange( psSportsObj );
    }
    else
    {
        OSAL.eSemGive(psSportsObj->hDbReadSem);
    }

    DATASERVICE_IMPL_vLog(
        SPORTS_SERVICE_MGR_OBJECT_NAME
        ": SC Config change END: bUpdateOk %u bCommitOk %u ver %u crc %u",
        bUpdateOk,
        bCommitOk,
        (UN32)un8CfgVer,
        un32Crc );

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
        "bEndServiceConfigUpdate:"
        "End bUpdateOk %u bCommitOk %u ver %u crc %u"
        " ObjCfgVer %u Obj32Crc %u\n",
        (UN32)bUpdateOk,
        (UN32)bCommitOk,
        (UN32)un8CfgVer,
        un32Crc,
        psSportsObj->sSC_Config.un8ConfigVersion,
        psSportsObj->sSC_Config.un32ConfigCrc );

    return (bUpdateOk && bCommitOk);
}

/*******************************************************************************
*
*   un8GetServiceConfigVersion
*
*
*******************************************************************************/
static BOOLEAN bIsCorrectServiceConfigVersion(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN8 un8CfgVer
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    return (psSportsObj->sSC_Config.un8ConfigVersion == un8CfgVer);
}

/*******************************************************************************
*
*   vDebugSetServiceConfigVersion
*
*
*******************************************************************************/
static void vDebugSetServiceConfigVersion(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN8 un8CfgVer
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    psSportsObj->sSC_Config.un8ConfigVersion = un8CfgVer;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DBG_CMD,
        "vDebugSetServiceConfigVersion:"
        "DEBUG SC Config change to ver:%d\n", un8CfgVer);

    return;
}

/*******************************************************************************
*
*   bAddSportsNameToDB
*
*******************************************************************************/
static BOOLEAN bAddSportsNameToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        STRING_OBJECT hSportsName,
        UN8 un8SportId
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Add the provided sportsname string and sportID to the SC_Sportsnames
    // database
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Take read semaphore.
    eOsalCode = OSAL.eSemTake(
        psSportsObj->hDbReadSem,
        OSAL_OBJ_TIMEOUT_INFINITE );
    if (eOsalCode != OSAL_SUCCESS)
    {
        return FALSE;
    }


    bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        SS_TABLE_ADD_TO_SC_SPORTS_NAMES,
                        (char*)STRING.pacCStr(hSportsName),
                        un8SportId );

    if( bOk == TRUE )
    {
        bOk = SQL_INTERFACE.bExecuteCommand (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf);
    }

    OSAL.eSemGive(psSportsObj->hDbReadSem);
    return bOk;
}

/*******************************************************************************
*
*   bAddTdefSportMappingToDB
*
*******************************************************************************/
static BOOLEAN bAddTdefSportMappingToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8SportId
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Add the provided sportsname string and sportID to the SC_Sportsnames
    // database
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Take read semaphore.
    eOsalCode = OSAL.eSemTake(
        psSportsObj->hDbReadSem,
        OSAL_OBJ_TIMEOUT_INFINITE );
    if (eOsalCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        SS_TABLE_ADD_TDEF_SPORTID_MAPPING,
                        (UN32)un16TableID,
                        (UN32)un8SportId );

    if( bOk == TRUE )
    {
        bOk = SQL_INTERFACE.bExecuteCommand (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf);
    }

    OSAL.eSemGive(psSportsObj->hDbReadSem);
    return bOk;
}

/*******************************************************************************
*
*   bShouldKeepTdef
*
*******************************************************************************/
static BOOLEAN bShouldKeepTdef(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8TdefClass
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Check the class first it is easy.
    bOk = bIsTableClassSupported( psSportsObj, un8TdefClass );
    if( !bOk )
    {
        return FALSE;
    }

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));

    // We need to query the db.  No need for a savepoint. Take read semaphore.
    eOsalCode = OSAL.eSemTake(
        psSportsObj->hDbReadSem,
        OSAL_OBJ_TIMEOUT_INFINITE );

    if( eOsalCode != OSAL_SUCCESS )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bShouldKeepTdef:"
            "Failure to take sem:%u\n",
             eOsalCode );
        return FALSE;
    }

    // Now check the sport.
    bOk = bSnprintf(
                   &psSportsObj->acInternalSqlBuf[0],
                   sizeof(psSportsObj->acInternalSqlBuf),
                   SS_GET_SPORTID_FOR_TDEF,
                   (UN32)un16TableID );

    if( bOk )
    {
        // Reuse the un32pragma handler for existence.  It stops on the
        // first row.
        vInitializeUn32PragmaQueryStruct( &sResult );
        bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bProcessUn32Pragma,
                (void*)&sResult );
    }

    // We are done with the db.
    OSAL.eSemGive(psSportsObj->hDbReadSem);

    if( !bOk && !sResult.bSuccess )
    {
        return FALSE;
    }

    if( sResult.un32RowCount != 1 )
    {
        return FALSE;
    }

    return bIsSportSupported( psSportsObj, sResult.un32Value );
}

/*******************************************************************************
*
*   bAddTableDefToDB
*
*   Given
*   Returns TRUE on success
*
*******************************************************************************/
static BOOLEAN bAddTableDefToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *psTdefInfo,
        UN16 un16TableID
        )
{
    BOOLEAN bOk;
    BOOLEAN bRollbackOk;
    int n;
    const char *pacCreateTableSql;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    UN32 un32Error = 0;
    UN32 un32TdefKey = 0;
    UN32 un32StrLen;
    size_t tAppendLength;
    size_t tAppendCount;
    OSAL_LINKED_LIST_ENTRY hEntry;
    STRING_OBJECT hString;
    SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *pLabelInfo = NULL;

    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bAddTableDefToDB:"
            "hSportsService %u psTdefInfo %u un16TableID %u\n",
            (UN32)hSportsService,
            (UN32)psTdefInfo,
            (UN32)un16TableID );

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bAddTableDefToDB:"
            "psTdefInfo-> CRC %u IsValid %u hLabelInfoList %u"
            " un8LabelCount %u un8TableClass %u un8TableVer %u\n",
            (UN32)psTdefInfo->CRC,
            (UN32)psTdefInfo->IsValid,
            (UN32)psTdefInfo->hLabelInfoList,
            (UN32)psTdefInfo->un8LabelCount,
            (UN32)psTdefInfo->un8TableClass,
            (UN32)psTdefInfo->un8TableVer );

    hString = STRING.hCreate("", 1024);
    if( hString == STRING_INVALID_OBJECT )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "Failure to create string\n" );

        return FALSE;
    }

    // Take read semaphore.
    eOsalCode = OSAL.eSemTake(
        psSportsObj->hDbReadSem,
        OSAL_OBJ_TIMEOUT_INFINITE );
    if (eOsalCode != OSAL_SUCCESS)
    {
        STRING.vDestroy(hString);
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bAddTableDefToDB:"
            "Failure to take sem:%u\n",
             eOsalCode );
        return FALSE;
    }

    // Create a save point.
    bOk = bUnsafeSetDbSavepoint( hSportsService, SS_ROLLBACK_POINT_TD_UPDATE );
    if( !bOk  )
    {
        STRING.vDestroy(hString);
        OSAL.eSemGive(psSportsObj->hDbReadSem);

        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bAddTableDefToDB:"
            "Failure to set db savepoint:%s\n",
            SS_ROLLBACK_POINT_TD_UPDATE );

        return FALSE;
    }

    // Force a delete of the def just in case it is already in there.
    // (i.e. CRC failed)(will take semaphore)  Delete any underlying
    // data and labels
    bOk = bUnsafeDeleteTableDefFromDB(hSportsService,
            un16TableID,
            psTdefInfo->un8TableVer,
            TRUE,
            TRUE );

    /* <<<< TRANSACTION BEGIN >>>>>>
    1.  Write to the TD_Info table:
        insert or update to TD_Info: TDefID, TDefVer, TableName, TableClass,
    LabelCount, TdefKey
    2.  For each label in psTdefInfo->hLabelInfoList, add it as a column
        in DI_XXXXYYYY
    3.  Create table DI_XXXXYYYY.
    4.  Add the labels to TD_LabelInfo */

    if( bOk )
    {
        // two steps because we go out of range for un16TableID
        un32TdefKey = (UN32)un16TableID;
        un32TdefKey = un32TdefKey<<16;
        un32TdefKey |= psTdefInfo->un8TableVer;

        bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        SS_UPDATE_TABLE_DATA_INFO,
                        un16TableID,
                        psTdefInfo->un8TableVer,
                        un32TdefKey,
                        psTdefInfo->un8TableClass,
                        psTdefInfo->un8LabelCount,
                        un32TdefKey,
                        psTdefInfo->CRC );

        if( bOk )
        {
            bOk = SQL_INTERFACE.bExecuteCommand (
                    psSportsObj->hSQLConnection,
                    psSportsObj->acInternalSqlBuf );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                    "bAddTableDefToDB:"
                    "SQL Failure:SS_UPDATE_TABLE_DATA_INFO:%s\n",
                    psSportsObj->acInternalSqlBuf );
            }
        }
    }

    // Prep for the list walk to build the sql statement.
    if( bOk )
    {
        // Add each of the labels to TD_LabelInfo
        // Walk the label linked list to find the label info
        // Construct the quert string to create the DI_XXXXX table
        // at the same time.
        bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SS_CREATE_DATA_INSTANCE_TABLE_START,
                            un32TdefKey);
        if( bOk )
        {
            // convert to a STRING object to build the query
            tAppendLength = strlen(psSportsObj->acInternalSqlBuf);
            tAppendCount = STRING.tAppendCStr(
                    hString, psSportsObj->acInternalSqlBuf);

            if( tAppendCount != tAppendLength )
            {
                bOk = FALSE;
            }
        }
    }

    if( !bOk )
    {
        DATASERVICE_IMPL_vLog(
                SPORTS_SERVICE_MGR_OBJECT_NAME
                ": Failure to process table def for TableId %d",
                un16TableID );

        bOk = bUnsafeRollbackDb( hSportsService, SS_ROLLBACK_POINT_TD_UPDATE );
        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "Failure db rollback:%s\n",
                SS_ROLLBACK_POINT_TD_UPDATE );
        }

        bUnsafeReleaseDbSavepoint( hSportsService, SS_ROLLBACK_POINT_TD_UPDATE );
        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "Failure db savepoint release:%s\n",
                SS_ROLLBACK_POINT_TD_UPDATE );
        }

        STRING.vDestroy(hString);
        OSAL.eSemGive(psSportsObj->hDbReadSem);

        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "Failure first stage process of tdef\n" );

        return FALSE;
    }

    /* Grab the first entry in the list */
    hEntry = OSAL.hLinkedListFirst (
            psTdefInfo->hLabelInfoList,
            (void**)&pLabelInfo );

    for (n = 0; n < psTdefInfo->un8LabelCount; n++)
    {
        if( hEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "Read %d of %d labels in list\n",
                    n,
                    psTdefInfo->un8LabelCount );

            bOk = FALSE;
            un32Error = 1;
            break;
        }

        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                    "bAddTableDefToDB:"
                    "(%d)pLabelInfo->"
                    " un8LbType %u"
                    " un8LabelId %u"
                    " un16LabelSize %u"
                    " un8LPriority %u"
                    " un8Dtype %u"
                    " bIsDmodPresent %u"
                    " bIsExtDataPresent %u"
                    " bIsLabelTxtPresent %u"
                    " un8Dmod %u\n",
                    n,
                    (UN32)pLabelInfo->un8LbType,
                    (UN32)pLabelInfo->un8LabelId,
                    (UN32)pLabelInfo->un16LabelSize,
                    (UN32)pLabelInfo->un8LPriority,
                    (UN32)pLabelInfo->un8Dtype,
                    (UN32)pLabelInfo->bIsDmodPresent,
                    (UN32)pLabelInfo->bIsExtDataPresent,
                    (UN32)pLabelInfo->bIsLabelTxtPresent,
                    (UN32)pLabelInfo->un8Dmod );

        if( pLabelInfo->bIsLabelTxtPresent )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                    "bAddTableDefToDB:"
                    "LabelText %s\n", pLabelInfo->LabelText );
        }
        if( pLabelInfo->bIsExtDataPresent )
        {
//            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
//                  "bAddTableDefToDB:"
//                    "ExtData %s\n", pLabelInfo->ExtData );
        }

        do
        {
            /* Extract the object from the one we found */
            bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                                sizeof(psSportsObj->acInternalSqlBuf),
                                SS_CREATE_LABEL_INFO_ENTRY_START );

            // DMOD field may not be present
            if( bOk && pLabelInfo->bIsDmodPresent )
            {
                un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
                bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[un32StrLen],
                        sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                        SS_CREATE_LABEL_INFO_ENTRY_DMOD);
            }

            if( !bOk )
            {
                un32Error = 2;
                break;
            }

            un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
            bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[un32StrLen],
                    sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                    SS_CREATE_LABEL_INFO_ENTRY_LINFO );

            // EXTDATA field may not be present
            if( bOk && pLabelInfo->bIsExtDataPresent )
            {
                // TODO: We cannot handle extdata as it is.
                // We will BASE64 the data in to a string
                // using the tEncodeBase64 API?
                // Meanwhile throw it on the floor for now.
                /*un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
                 bOk = bSnprintf(
                         &psSportsObj->acInternalSqlBuf[un32StrLen],
                         sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                         SS_CREATE_LABEL_INFO_ENTRY_EXTDATA);
                 */
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                    "bAddTableDefToDB:"
                    "Encountered label extended data1: n %u LabelId %u\n",
                    (UN32)n,
                    (UN32)pLabelInfo->un8LabelId );

            }

            if( !bOk)
            {
                un32Error = 3;
                break;
            }

            un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
            bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[un32StrLen],
                    sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                    SS_CREATE_LABEL_INFO_ENTRY_END,
                    n,
                    pLabelInfo->un8LabelId, pLabelInfo->un8LbType );

            if( bOk && pLabelInfo->bIsDmodPresent )
            {
                un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
                bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[un32StrLen],
                        sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                        SS_CREATE_LABEL_INFO_VALUES_DMOD,
                        pLabelInfo->un8Dmod );
            }

            if( !bOk)
            {
                un32Error = 4;
                break;
            }

            un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
            bOk = bSnprintf(
                &psSportsObj->acInternalSqlBuf[un32StrLen],
                sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                SS_CREATE_LABEL_INFO_VALUES_LINFO,
                pLabelInfo->un8LPriority,
                pLabelInfo->un16LabelSize,
                (pLabelInfo->bIsLabelTxtPresent?pLabelInfo->LabelText:"") );

            if( bOk && pLabelInfo->bIsExtDataPresent )
            {
                // TODO: We cannot handle extdata as it is.
                // We will BASE64 the data in to a string
                // using the tEncodeBase64 API?
                // Meanwhile throw it on the floor for now.
/*                un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
                bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[un32StrLen],
                        sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                        SS_CREATE_LABEL_INFO_VALUES_EXTDATA,
                        (char*)pLabelInfo->ExtData );
*/
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                    "bAddTableDefToDB:"
                    ":Encountered label extended data2: n %u LabelId %u\n",
                    (UN32)n,
                    (UN32)pLabelInfo->un8LabelId );
            }

            if( !bOk)
            {
                un32Error = 5;
                break;
            }

            un32StrLen = strlen(psSportsObj->acInternalSqlBuf);
            bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[un32StrLen],
                    sizeof(psSportsObj->acInternalSqlBuf) - un32StrLen,
                    SS_CREATE_LABEL_INFO_VALUES_END,
                    pLabelInfo->un8Dtype,
                    un32TdefKey );

            if( bOk == TRUE )
            {
                bOk = SQL_INTERFACE.bExecuteCommand (
                    psSportsObj->hSQLConnection,
                    psSportsObj->acInternalSqlBuf
                    );
            }

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                    "bAddTableDefToDB:"
                    "SQL Failure:%s\n",
                    psSportsObj->acInternalSqlBuf );

                un32Error = 6;
                break;
            }

            // now add this label to the table def we are building
            bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    ", 'c%d' %s ",
                    n,
                    (pLabelInfo->un8Dtype?"TEXT":"INTEGER") );

            if( !bOk )
            {
                un32Error = 7;
                break;
            }

            tAppendLength = strlen(psSportsObj->acInternalSqlBuf);
            tAppendCount = STRING.tAppendCStr(
                    hString, psSportsObj->acInternalSqlBuf);

            if( tAppendCount != tAppendLength )
            {
                bOk = FALSE;
                un32Error = 8;
                break;
            }

            // Success with label move on to next label
            hEntry = OSAL.hLinkedListNext (
                        hEntry,
                        (void**)&pLabelInfo );

        } while( FALSE );

        if( !bOk )
        {
            //Exit the label processing loop.
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "Tdef label process Failure n %u error %u\n",
                (UN32)n,
                un32Error );

            break;
        }
    }

    if ( (n == psTdefInfo->un8LabelCount) && bOk )
    {
        const char *pacCloseChars = ");";
        tAppendLength = strlen(pacCloseChars);
        tAppendCount = STRING.tAppendCStr( hString, pacCloseChars );
        pacCreateTableSql = STRING.pacCStr(hString);

        if( tAppendCount == tAppendLength )
        {

            bOk = SQL_INTERFACE.bExecuteCommand (
                    psSportsObj->hSQLConnection,
                    pacCreateTableSql );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                    "bAddTableDefToDB:"
                    "SQL Failure:%s\n",
                    pacCreateTableSql );
            }
        }
        else
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "SQL Str Build Failure:%s\n",
                pacCreateTableSql );

            bOk = FALSE;
        }
    }
    else
    {
        bOk = FALSE;
    }

    // Index the instance table.
    if( bOk )
    {
        bOk = bSnprintf(    &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SS_CREATE_DATA_INSTANCE_TABLE_INDEX,
                            un32TdefKey,
                            un32TdefKey );

        if( bOk )
        {

            bOk = SQL_INTERFACE.bExecuteCommand (
                    psSportsObj->hSQLConnection,
                    psSportsObj->acInternalSqlBuf );
        }

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                                "bAddTableDefToDB:"
                                "SQL Failure:%s\n",
                                psSportsObj->acInternalSqlBuf );
        }
    }

    STRING.vDestroy(hString);

    // <<<< TRANSACTION END >>>>

    if (!bOk)
    {
        bRollbackOk = bUnsafeRollbackDb(
                hSportsService,
                SS_ROLLBACK_POINT_TD_UPDATE );
        if( !bRollbackOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bAddTableDefToDB:"
                "Failure db rollback:%s\n",
                SS_ROLLBACK_POINT_TD_UPDATE );
        }
    }

    // We did get the complete tdef.  Do not let a savepoint leak
    // affect the data usage.
    bRollbackOk = bUnsafeReleaseDbSavepoint(
            hSportsService,
            SS_ROLLBACK_POINT_TD_UPDATE );
    if( !bRollbackOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bAddTableDefToDB:"
            "Failure db savepoint release:%s\n",
            SS_ROLLBACK_POINT_TD_UPDATE );
    }

    DATASERVICE_IMPL_vLog(
            SPORTS_SERVICE_MGR_OBJECT_NAME
            ":New Table def for TableId %d",
            un16TableID
            );

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
        "bAddTableDefToDB:"
        "End bOk %u\n",
                (UN32)bOk );

    OSAL.eSemGive(psSportsObj->hDbReadSem);

    return bOk;
}

/*******************************************************************************
*
*   bDeleteTableDefFromDB
*
*   Currently only used in cli debug commands.
*
*******************************************************************************/
static BOOLEAN bDeleteTableDefFromDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8TabVer,
        BOOLEAN bEraseUnderlyingData
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    BOOLEAN bOk = FALSE;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    eOsalCode = OSAL.eSemTake(
        psSportsObj->hDbReadSem,
        OSAL_OBJ_TIMEOUT_INFINITE );

    if( eOsalCode == OSAL_SUCCESS )
    {
        bOk = bUnsafeDeleteTableDefFromDB(
                hSportsService,
                un16TableID,
                un8TabVer,
                bEraseUnderlyingData,
                bEraseUnderlyingData);

        OSAL.eSemGive(psSportsObj->hDbReadSem);
    }
    return bOk;
}

/*******************************************************************************
*
*   bUnsafeDeleteTableDefFromDB
*
*   Returns TRUE on success
*
*******************************************************************************/
static BOOLEAN bUnsafeDeleteTableDefFromDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8TabVer,
        BOOLEAN bDropTable,
        BOOLEAN bDeleteInstanceEntries
        )
{
    BOOLEAN bOk;
    UN32 un32Error = 0;
    UN32 un32TdefKey;

    // Add the provided sportsname string and sportID to the SC_Sportsnames db
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bUnsafeDeleteTableDefFromDB:"
            "hSportsService %u un16TableID %u un8TabVer"
            " %u bDropTable %u bDeleteEntries %u\n",
            (UN32)hSportsService,
            (UN32)un16TableID,
            (UN32)un8TabVer,
            (UN32)bDropTable,
            (UN32)bDeleteInstanceEntries );

    do
    {
        // Can we imply both of these fields based on TdefId and TabVer?
        // TdefKey = (TdefId<<8) | TabVer;
        // TableName = "DI_xxxxyyyy" where xxxx = TdefId, yyyy = TabVer.
        // two steps because we go out of range for un16TableID
        un32TdefKey = (UN32)un16TableID;
        un32TdefKey = un32TdefKey<<16;
        un32TdefKey |= un8TabVer;

        if(bDeleteInstanceEntries)
        {
#if 0
            char acTableName[SS_TABLE_MAX_TD_TABLE_NAME_SIZE];

            //
            // Optimization.  Drop all primary table data in one shot.
            // If this fails keep going.  It could fail if the table does not
            // exist which is OK.  This relies on the special behavior
            // of a WHERE free DELETE clause on a table with no triggers.
            // The rows are not examined and the table is truncated.
            // There is a compile option to turn off the special feature.
            //
            // However, this causes an sqlite print out indicating
            // the table does not exits.  We really do not care operationally.
            // To check for existence of the table you have to SELECT in the
            // sqlite master table.  Which is sqlite specific.  If triggers
            // are added to the table then this would silently revert to per row
            // processing.  Even per row processing here would speed up removal of
            // rows at the per table instance.  The per instance processing would
            // search a row less table.
            //
            bOk = bSnprintf(
                    acTableName,
                    sizeof(acTableName),
                    SS_MAKE_INSTANCE_TABLE_NAME,
                    un32TdefKey );

            if( bOk )
            {
                bOk = bUnsafeClearTable(
                                    hSportsService,
                                    acTableName );
            }
#endif
            bOk = bUnsafeRemoveTdefTables(
                    psSportsObj,
                    un32TdefKey );

            if (!bOk)
            {
                un32Error = 1;
                break;
            }
        }

        if (bDropTable)
        {
            bOk = bUnsafeDropTable( psSportsObj, un32TdefKey );

            if (!bOk)
            {
                un32Error = 2;
                break;
            }
        }

        //
        // Remove the Table def from its tables
        //

        bOk = bSnprintf(
                &psSportsObj->acInternalSqlBuf[0],
                sizeof(psSportsObj->acInternalSqlBuf),
                SS_DELETE_OLD_LABEL_INFO,
                un32TdefKey    );

        if( !bOk )
        {
            un32Error = 3;
            break;
        }

        bOk = SQL_INTERFACE.bExecuteCommand (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf );

        if (!bOk)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bUnsafeDeleteTableDefFromDB:"
                "SQL Failure:%s\n",
                psSportsObj->acInternalSqlBuf );
            un32Error = 4;
            break;
        }

        bOk = bSnprintf(
                &psSportsObj->acInternalSqlBuf[0],
                sizeof(psSportsObj->acInternalSqlBuf),
                SS_DELETE_OLD_TABLE_DEF_INFO,
                un32TdefKey    );

        if( !bOk )
        {
            un32Error = 5;
            break;
        }

        bOk = SQL_INTERFACE.bExecuteCommand (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf );

        if (!bOk)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
                "bUnsafeDeleteTableDefFromDB:"
                "SQL Failure:%s\n",
                psSportsObj->acInternalSqlBuf );
            un32Error = 6;
            break;
        }

        DATASERVICE_IMPL_vLog(
                    SPORTS_SERVICE_MGR_OBJECT_NAME
                    ":Deleted table def: Id %u Ver %u",
                    un16TableID,
                    un8TabVer );
    }
    while (0);

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_TD_UPDATE,
            "bUnsafeDeleteTableDefFromDB:"
            "End: bOk %u un32Error %u un32TdefKey 0x%x\n",
            (UN32)bOk,
            un32Error,
            un32TdefKey);

    return bOk;
}

/*******************************************************************************
*
*   bUnsafeRemoveTdefTables
*
*******************************************************************************/
static BOOLEAN bUnsafeRemoveTdefTables(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32TdefKey
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_OBJECT_HDL hRemoveList = OSAL_INVALID_OBJECT_HDL;
    OSAL_LINKED_LIST_ENTRY hEntry;
    UN32 un32ListSize;
    UN32 un32Cnt;
    SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT sQuery;
    SPORTS_SERVICE_TREF_REMOVE_STRUCT *psRemoveInfo;
    BOOLEAN bOk;

    //
    // Make a list of tables to remove.
    //
    eOsalCode = OSAL.eLinkedListCreate (
               &hRemoveList,
               SPORTS_SERVICE_MGR_OBJECT_NAME":RemoveTdefLL",
               NULL,
               OSAL_LL_OPTION_LINEAR );

    if( eOsalCode != OSAL_SUCCESS )
    {
       return FALSE;
    }

    do
    {
        bOk = bSnprintf(
                     &psSportsObj->acInternalSqlBuf[0],
                     sizeof(psSportsObj->acInternalSqlBuf),
                     SS_GET_TABLEDEF_INSTANCE_REMOVE_INFO,
                     un32TdefKey );

        if( !bOk )
        {
            break;
        }

        sQuery.hTableList = hRemoveList;
        sQuery.un32RowCount = 0;
        sQuery.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;

        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf,
            (SQL_QUERY_RESULT_HANDLER)bProcessMakeTableList,
            (void*)&sQuery );

        if( !bOk )
        {
            break;
        }

        if( sQuery.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        // Get the size of the list.
        eOsalCode = OSAL.eLinkedListItems( hRemoveList, &un32ListSize );
        if( eOsalCode != OSAL_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        // Empty result set means we are done.
        if( un32ListSize == 0 )
        {
            break;
        }

        // Go through the list.
        hEntry = OSAL.hLinkedListFirst ( hRemoveList, (void**)&psRemoveInfo );

        for(un32Cnt=0; un32Cnt<un32ListSize; un32Cnt++)
        {
            if( (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
             || (psRemoveInfo == NULL) )
            {
                break;
            }

            bOk = bUnsafeRemoveInstanceTables(
                       (SPORTS_SERVICE_OBJECT) psSportsObj,
                       SS_GET_TABLE_AND_TREF_REMOVE_INFO,
                       SS_DELETE_TABLE_AND_TREF_INSTANCES,
                       TRUE,
                       psRemoveInfo->n64IntanceId );

            if( !bOk )
            {
                break;
            }

            hEntry = OSAL.hLinkedListNext( hEntry, (void**)&psRemoveInfo  );
        }

        if( un32Cnt != un32ListSize )
        {
            bOk = FALSE;
            break;
        }
    }
    while(0);

    if( hRemoveList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hRemoveList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        (void)OSAL.eLinkedListDelete( hRemoveList );
    }

    return bOk;
}

/*******************************************************************************
*
*   bStartSportAffiliationUpdate
*
*******************************************************************************/
static BOOLEAN bStartSportAffiliationUpdate(
          SPORTS_SERVICE_OBJECT hSportsService,
         UN8 un8AdVer,
         UN32 un32CRC)
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_AF_UPDATE_STRUCT *pAfUpdate;
    SPORTS_SERVICE_ALL_AFID_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
            "bStartSportAffiliationUpdate:"
            "hSportsService %u un8AdVer %u un32Crc %u"
            " ObjAdVer %u Obj32Crc %u\n",
            (UN32)hSportsService,
            (UN32)un8AdVer,
            un32CRC,
            (UN32)psSportsObj->sAF_Config.un8AfCfgVersion,
            psSportsObj->sAF_Config.un32AfConfigCrc );

    // check version info and crc against memory copy
    if ( (psSportsObj->sAF_Config.un8AfCfgVersion == un8AdVer)
            && (psSportsObj->sAF_Config.un32AfConfigCrc == un32CRC) )
    {
        return TRUE;
    }

    DATASERVICE_IMPL_vLog( SPORTS_SERVICE_MGR_OBJECT_NAME
        ":Affiliation update START ver %u crc %u",
        (UN32)un8AdVer, un32CRC );

    // lock the semaphore for the entire duration of the update
    eOsalCode = OSAL.eSemTake(
        psSportsObj->hDbReadSem,
        OSAL_OBJ_TIMEOUT_INFINITE );

    if (eOsalCode != OSAL_SUCCESS)
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
            "bStartSportAffiliationUpdate:"
            "Failure to take sem: %u\n",
            eOsalCode );
        return TRUE;    // CANNOT OBTAIN LOCK - DO NOT UPDATE
    }

    do
    {
        bOk = bUnsafeSetDbSavepoint( hSportsService, SS_ROLLBACK_POINT_AF_UPDATE );
        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
                "bStartSportAffiliationUpdate:"
                "Failure to set DB savepoint:%s\n",
                SS_ROLLBACK_POINT_AF_UPDATE );
            break;
        }

        pAfUpdate = &psSportsObj->sAfUpdate;

        //
        // Initialize the update helper variables.
        //
        pAfUpdate->bNewAfAdded = FALSE;

        OSAL.bMemSet(
                pAfUpdate->aun8AfRemoveBitfield,
                0,
                sizeof(pAfUpdate->aun8AfRemoveBitfield) );

        OSAL.bMemSet(
                pAfUpdate->aun8AfNameChangeBitfield,
                0,
                sizeof(pAfUpdate->aun8AfNameChangeBitfield) );

        //
        // Populate the remove bitfield with all affiliates currently present.
        // As new affiliates are parsed we turn off the bits.  In the end we
        // have all affiliates that were removed.
        //
        sResult.pun8BitField = pAfUpdate->aun8AfRemoveBitfield;
        sResult.un32BitFieldSize = SS_MAX_AFFILIATE_COUNT;
        sResult.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
        sResult.un32RowCount = 0;

        bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                SS_SELECT_GET_ALL_AF_IDS,
                (SQL_QUERY_RESULT_HANDLER)bProcessGetAllAfIds,
                &sResult );

        if( !bOk || (sResult.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS) )
        {
            break;
        }

        //
        // Create table for new affiliates.
        //
        bOk = SQL_INTERFACE.bExecuteCommand(
                psSportsObj->hSQLConnection,
                SS_TABLE_CREATE_AFFILIATIONS_UPDATE );

        if( !bOk )
        {
            break;
        }

        //
        // Create index on the affiliate id.
        //
        bOk = bSnprintf(
                &psSportsObj->acInternalSqlBuf[0],
                sizeof(psSportsObj->acInternalSqlBuf),
                SS_CREATE_AF_ID_INDEX,
                un32CRC,
                (UN32)un8AdVer );

        if( !bOk )
        {
            break;
        }

        bOk = SQL_INTERFACE.bExecuteCommand(
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf );

        if( !bOk )
        {
            break;
        }

        //
        // Create index on the parent affiliate.
        //
        bOk = bSnprintf(
                &psSportsObj->acInternalSqlBuf[0],
                sizeof(psSportsObj->acInternalSqlBuf),
                SS_CREATE_AF_PARENT_INDEX,
                un32CRC,
                (UN32)un8AdVer );

        if( !bOk )
        {
            break;
        }

        bOk = SQL_INTERFACE.bExecuteCommand(
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf );

        if( !bOk )
        {
            break;
        }

        //
        // Remove the old config info.
        //
        bOk = SQL_INTERFACE.bExecuteCommand(
                        psSportsObj->hSQLConnection,
                        SS_CLEAR_AF_CONFIG_TABLE );

        if( !bOk )
        {
            break;
        }

        // We are doing an affiliate change.
        return FALSE;

    } while(0);

    OSAL.eSemGive(psSportsObj->hDbReadSem);
    return TRUE;
}

/*******************************************************************************
*
*   bUnsafeAddSportsAffiliationToDB
*
*******************************************************************************/
static BOOLEAN bUnsafeAddSportsAffiliationToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        STRING_OBJECT hSportAffiliation,
        UN16 un16AffiliateId,
        BOOLEAN bGDRefPresBit,
        UN8 un8GDRef,
        UN8 un8SportID
        )
{
    BOOLEAN bOk = FALSE;
    BOOLEAN bAlreadyHave;
    BOOLEAN bIsChildAffiliate = FALSE;
    size_t tmpStrlen;
    UN16 un16ParentAfId=0;
    SPORTS_SERVICE_AF_UPDATE_STRUCT *pAfUpdate;
    SPORTS_SERVICE_AF_NAME_CHANGE_QUERY_RESULT_STRUCT sResult;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    pAfUpdate = &psSportsObj->sAfUpdate;

    do
    {
        // See if this is an existing affiliate.  The bitfield was populated
        // with all existing affiliates at start of update.
        bAlreadyHave = bBitfieldGet(
                    pAfUpdate->aun8AfRemoveBitfield,
                    SS_MAX_AFFILIATE_COUNT,
                    un16AffiliateId );

        if( bAlreadyHave )
        {
            // It is not a candidate for removal.  Turn off its bit.
            vBitfieldSet(
                pAfUpdate->aun8AfRemoveBitfield,
                SS_MAX_AFFILIATE_COUNT,
                un16AffiliateId,
                FALSE );
        }
        else
        {
            // Save the presence of a new affiliate for monitor existance
            // indication.
            pAfUpdate->bNewAfAdded = TRUE;
        }

        // Create the SQL INSERT string to store AfId, AfName, etc
        if(!bUnsafeGetParentAffiliateIdFromString(
                psSportsObj,
                hSportAffiliation,
                un8SportID,
                &bIsChildAffiliate,
                &un16ParentAfId)
           )
        {
            break;
        }

        if (!bSnprintf(
                &psSportsObj->acInternalSqlBuf[0],
                sizeof(psSportsObj->acInternalSqlBuf),
                SS_TABLE_SET_SPORTS_AFFILIATE
                ))
        {
            break;
        }

        if (bIsChildAffiliate)
        {
            tmpStrlen = strlen(psSportsObj->acInternalSqlBuf);
            if (!bSnprintf(
                    &psSportsObj->acInternalSqlBuf[tmpStrlen],
                    sizeof(psSportsObj->acInternalSqlBuf)-tmpStrlen,
                    SS_TABLE_ADD_PARENT_AFFILIATE
                    )
                )
            {
                break;
            }
        }

        if (bGDRefPresBit)
        {
            tmpStrlen = strlen(psSportsObj->acInternalSqlBuf);
            if (!bSnprintf(
                    &psSportsObj->acInternalSqlBuf[tmpStrlen],
                    sizeof(psSportsObj->acInternalSqlBuf)-tmpStrlen,
                    SS_TABLE_ADD_AFFILIATE_LEAGUE_ID
                    )
                )
            {
                break;
            }
        }

        tmpStrlen = strlen(psSportsObj->acInternalSqlBuf);
        if (!bSnprintf(
                &psSportsObj->acInternalSqlBuf[tmpStrlen],
                 sizeof(psSportsObj->acInternalSqlBuf)-tmpStrlen,
                 SS_TABLE_SET_SPORTS_AFFILIATE_VALUES,
                 un16AffiliateId,
                 (char*)STRING.pacCStr(hSportAffiliation),
                 un8SportID
                 )
            )
        {
            break;
        }

        if (bIsChildAffiliate)
        {
            tmpStrlen = strlen(psSportsObj->acInternalSqlBuf);
            if (!bSnprintf(
                    &psSportsObj->acInternalSqlBuf[tmpStrlen],
                    sizeof(psSportsObj->acInternalSqlBuf)-tmpStrlen,
                    SS_TABLE_ADD_PARENT_AFFILIATE_VALUES,
                    un16ParentAfId
                    )
                )
            {
                    break;
            }
        }

        if (bGDRefPresBit)
        {
            tmpStrlen = strlen(psSportsObj->acInternalSqlBuf);
            if (!bSnprintf(
                    &psSportsObj->acInternalSqlBuf[tmpStrlen],
                    sizeof(psSportsObj->acInternalSqlBuf)-tmpStrlen,
                    SS_TABLE_ADD_AFFILIATE_LEAGUE_ID_VALUES,
                    un8GDRef
                    )
                )
            {
                break;
            }
        }

        tmpStrlen = strlen(psSportsObj->acInternalSqlBuf);
        if (!bSnprintf(
                &psSportsObj->acInternalSqlBuf[tmpStrlen],
                 sizeof(psSportsObj->acInternalSqlBuf)-tmpStrlen,
                 SS_TABLE_SET_SPORTS_AFFILIATE_END
                 )
            )
        {
            break;
        }

        // Now send the query to the database
        bOk = SQL_INTERFACE.bExecuteCommand (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf);

        if( !bOk )
        {
            break;
        }

        // See if there is a name change for this affiliate.
        if( bAlreadyHave )
        {
            bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        SS_TABLE_GET_NAME_CHANGE_AFFILIATE,
                        un16AffiliateId );

            if( !bOk )
            {
                break;
            }

            sResult.hSportAffiliation = hSportAffiliation;
            sResult.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
            sResult.un32RowCount = 0;

            bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bProcessAffiliateNameChange,
                &sResult );

            if( !bOk || (sResult.un32RowCount != 1) ||
                (sResult.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS) )
            {
                bOk = FALSE;
                break;
            }

            vBitfieldSet( pAfUpdate->aun8AfNameChangeBitfield,
                          SS_MAX_AFFILIATE_COUNT,
                          un16AffiliateId,
                          sResult.bNameChanged );
        }
    } while (0);

    return bOk;
}

/*******************************************************************************
*
*   bEndSportAffiliationUpdate
*
*******************************************************************************/
static BOOLEAN bEndSportAffiliationUpdate(
        SPORTS_SERVICE_OBJECT hSportsService,
        BOOLEAN bUpdateOk,
        UN8 un8AdVer,
        UN32 un32CRC )
{
    BOOLEAN bCommitOk = FALSE;
    BOOLEAN bOk;
    UN32 un32Cnt;
    SPORTS_SERVICE_AF_UPDATE_STRUCT *pAfUpdate;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
            "bEndSportAffiliationUpdate:"
            "hSportsService %u un8AdVer %u un32Crc %u bUpdateOk %u"
            " ObjAdVer %u Obj32Crc %u\n",
            (UN32)hSportsService,
            (UN32)un8AdVer,
            un32CRC,
            (UN32)bUpdateOk,
            (UN32)psSportsObj->sAF_Config.un8AfCfgVersion,
            psSportsObj->sAF_Config.un32AfConfigCrc );

    if( bUpdateOk )
    {
        bCommitOk = bDropRemovedAffiliates( hSportsService );

        // Drop the old affilaites table.
        if( bCommitOk )
        {
            bCommitOk = SQL_INTERFACE.bExecuteCommand (
                        psSportsObj->hSQLConnection,
                        SS_TABLE_DROP_AFFILIATIONS );
        }

        // Replace the old with the new.  Table rename.
        if( bCommitOk )
        {
            bCommitOk = SQL_INTERFACE.bExecuteCommand (
                            psSportsObj->hSQLConnection,
                            SS_TABLE_REPLACE_AFFILIATIONS );
        }

        // Update the config values.
        if( bCommitOk )
        {
            bCommitOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    SS_TABLE_UPDATE_AF_CONFIG_VALUES,
                    un8AdVer,
                    un32CRC );
        }

        if( bCommitOk )
        {
            bCommitOk = SQL_INTERFACE.bExecuteCommand (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf);
        }
    }

    if( !bUpdateOk || (bUpdateOk && !bCommitOk) )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
            "bEndSportAffiliationUpdate:"
            "Rollback bUpdateOk %u bCommitOk %u\n",
            (UN32)bUpdateOk,
            (UN32)bCommitOk );

        bOk = bUnsafeRollbackDb( hSportsService, SS_ROLLBACK_POINT_AF_UPDATE );
        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
                "bEndSportAffiliationUpdate:"
                "Rollback Failure:%s\n",
               SS_ROLLBACK_POINT_AF_UPDATE );
        }
    }

    bOk = bUnsafeReleaseDbSavepoint( hSportsService, SS_ROLLBACK_POINT_AF_UPDATE );
    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
            "bEndSportAffiliationUpdate:"
            "ReleaseSavePoint Failure:%s\n",
            SS_ROLLBACK_POINT_AF_UPDATE );
    }

    // fix for BUG 5439 -
    // only use af version if we successfully saved it in DB
    if( bCommitOk )
    {
        pAfUpdate = &psSportsObj->sAfUpdate;

        // Update monitors: existance.
        bOk = bBitfieldAnySet(
                pAfUpdate->aun8AfRemoveBitfield,
                SS_MAX_AFFILIATE_COUNT );

        if( bOk || pAfUpdate->bNewAfAdded )
        {
            vNotifyAffiliateExistanceChange( psSportsObj );
        }

        // Update monitors: body(name change)
        for( un32Cnt=0; un32Cnt<SS_MAX_AFFILIATE_COUNT; un32Cnt++ )
        {
            bOk = bBitfieldGet(
                     pAfUpdate->aun8AfNameChangeBitfield,
                     SS_MAX_AFFILIATE_COUNT,
                     un32Cnt );
            if( bOk )
            {
                vNotifyAffiliateChange(
                        psSportsObj,
                        un32Cnt,
                        (SPORTS_MONITOR_EVENT_MASK)
                            1<<AFFILIATE_EVENT_BODY_BIT_POSITION );
            }
        }

        psSportsObj->sAF_Config.un8AfCfgVersion = un8AdVer;
        psSportsObj->sAF_Config.un32AfConfigCrc = un32CRC;
        OSAL.eSemGive(psSportsObj->hDbReadSem);
        vProcessMonitorChange( psSportsObj );
    }
    else
    {
        OSAL.eSemGive(psSportsObj->hDbReadSem);
    }

    DATASERVICE_IMPL_vLog(
            SPORTS_SERVICE_MGR_OBJECT_NAME
            ": Affiliate update END: bUpdateOk %u bCommitOk %u ver %u crc %u",
            bUpdateOk,
            bCommitOk,
            un8AdVer,
            un32CRC );

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
            "bEndSportAffiliationUpdate:"
            "End bUpdateOk %u bCommitOk %u ver %u crc %u"
        " ObjAdVer %u Obj32Crc %u\n",
        (UN32)bUpdateOk,
        (UN32)bCommitOk,
        (UN32)un8AdVer,
        un32CRC,
        (UN32)psSportsObj->sAF_Config.un8AfCfgVersion,
        psSportsObj->sAF_Config.un32AfConfigCrc );

    return (bUpdateOk && bCommitOk);
}

/*******************************************************************************
*
*   bDropRemovedAffiliates
*
*******************************************************************************/
static BOOLEAN bDropRemovedAffiliates(
        SPORTS_SERVICE_OBJECT hSportsService
        )
{
    UN32 un32Cnt;
    BOOLEAN bOk;
    SPORTS_SERVICE_AF_UPDATE_STRUCT *pAfUpdate;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    pAfUpdate = &psSportsObj->sAfUpdate;

    for( un32Cnt=0; un32Cnt<SS_MAX_AFFILIATE_COUNT; un32Cnt++ )
    {
        bOk = bBitfieldGet(
                pAfUpdate->aun8AfRemoveBitfield,
                SS_MAX_AFFILIATE_COUNT,
                un32Cnt );

        if( !bOk )
        {
            continue;
        }

        // Remove the affiliate and its data instances.
        bOk = bUnsafeRemoveInstanceTables(
                    hSportsService,
                    SS_GET_AFFILIATE_REMOVE_INFO,
                    SS_DELETE_AFFILIATE_INSTANCES,
                    FALSE,
                    (N64)un32Cnt );

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
                            "bDropRemovedAffiliates:"
                            "Failure to remove: AfId %u\n",
                            un32Cnt );

            return FALSE;
        }
    }
    return TRUE;
}

/*******************************************************************************
*
*   vReleasePayload
*
*******************************************************************************/
static void vReleasePayload(
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOk;

    bOk = DATASERVICE_IMPL_bFreeDataPayload(hPayload);
    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( 1,
            "vReleasePayload:"
            "Failure hPayload %u\n",
            (UN32)hPayload );
    }

    return;
}

/*******************************************************************************
*
*   bUnsafeGetParentAffiliateIdFromString
*
*   Given an affiliate ID string, parse back up the hierarchy to first ':'
*   to find the parent affiliate name.  Then pass this string in to the
*   database to find the parents AfId along with the sport ID.
*   Returns TRUE if operation succeeded.
*
*******************************************************************************/
static BOOLEAN bUnsafeGetParentAffiliateIdFromString(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        STRING_OBJECT hSportAffiliation,
        UN8 un8SportID,
        BOOLEAN *bIsChildAffiliate,
        UN16 *pUn16AfId
        )
{
    BOOLEAN bSuccess = TRUE;
    UN32 un32DelimiterCount;
    char acParentAffiliateName
        [SS_MAX_AFFILIATION_NAME_SYMBOL_COUNT]="";
    size_t tCharIndex;
    BOOLEAN bOk;
    SPORTS_SERVICE_AFID_QUERY_RESULT_STRUCT sResult;

    *bIsChildAffiliate = FALSE;   // default

    (void) STRING.tCopyToCStr(  hSportAffiliation,
                                acParentAffiliateName,
                                sizeof(acParentAffiliateName) );

    // start at rightmost end of string - must be NULL terminated
    tCharIndex = strlen(acParentAffiliateName);

    // Empty string is invalid for PVN1
    if( tCharIndex == 0 )
    {
        return FALSE;
    }

    // The first or last cannot cannot be a ':'.  This the first check to
    // dis allow empty strings.  The code below catches :: in a hidden
    // way in its process of finding parents.  If there is a ::
    // sequence there must be a root with : in it which is impossible.
    // Somewhere along the processing of the entire table we will encounter
    // an affiliate with no parent which results in failure.
    if( (acParentAffiliateName[0]==':') ||
        (acParentAffiliateName[tCharIndex-1]==':') )
    {
        return FALSE;
    }

    un32DelimiterCount = 0;
    while (tCharIndex>0)
    {
        if (acParentAffiliateName[tCharIndex] == ':')
        {
            // Strip the first delimiter.
            if( un32DelimiterCount == 0 )
            {
                acParentAffiliateName[tCharIndex] = '\0'; // remove colon
            }
            un32DelimiterCount++;
        }

        // After the first delimiter stop srinking the string.  We only count
        // the remaining delimiters after that.
        if( un32DelimiterCount == 0 )
        {
            acParentAffiliateName[tCharIndex] = '\0';
        }
        tCharIndex--;
    }

    // No delimiters means root affiliate.
    if( un32DelimiterCount == 0 )
    {
        return TRUE;
    }

    // Too many delimiters is invalid.
    if( un32DelimiterCount > SS_MAX_AFFILIATE_DELIMITER_COUNT )
    {
        return FALSE;
    }

    // now use the acParentAffiliateName string as a lookup to find
    // the AfID
    *bIsChildAffiliate = TRUE;   // override default

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));

    bOk = bSnprintf(
        &psSportsObj->acInternalSqlBuf[0],
        sizeof(psSportsObj->acInternalSqlBuf),
        SS_TABLE_GET_FIRST_AFID_FROM_AFNAME,
        acParentAffiliateName,
        un8SportID
        );

    if( bOk )
    {
        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf,
            (SQL_QUERY_RESULT_HANDLER)bProcessGetParentAfIdResult,
            &sResult );
    }

    if( (bOk == FALSE) ||
        (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_ERROR) ||
        (sResult.un32RowCount == 0) )
    {
        bSuccess = FALSE;

    } else {
        *pUn16AfId = sResult.un16AfId;
        bSuccess = TRUE;
    }

    // bSuccess=TRUE and bIsChildAffiliate=FALSE by default
    return bSuccess;
}

/*******************************************************************************
*
*   bProcessGetParentAfIdResult
*
*******************************************************************************/
static BOOLEAN bProcessGetParentAfIdResult(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_AFID_QUERY_RESULT_STRUCT *psResult
            )
{
    // Only the AfId must be present.
    if( n32NumberOfColumns != 1 )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // only looking at the first column, ie n32Index=0
    if(psColumn[0].eType == SQL_COLUMN_TYPE_INTEGER)
    {
        // intentional truncation
        psResult->un16AfId = (UN16)(psColumn[0].uData.sUN32.un32Data & 0xFFFF);
        psResult->un32RowCount++;
        return TRUE;
    }

    return FALSE;

}


/*******************************************************************************
*
*   bDbgDumpDefs
*
*******************************************************************************/
static BOOLEAN bDbgDumpDefs(
    SPORTS_SERVICE_OBJECT hSportsService,
    char *DefInfo
        )
{
   /*psHighbandObj =
           (SPORTS_SERVICE_HIGHBAND_OBJECT_STRUCT*)psSportsObj->

    printf("Valid defs:\n");
    for (n = 0; n<1024; n++)
    {
        if (psHighbandObj->hTableDefs[n].IsValid)
            SPORTS_SERVICE_MGR_vPrint( 1,("\t%d\n", n);
    }
*/
    printf("No method available yet to dump the defs\n");

    return FALSE;
}

/*******************************************************************************
*
*   bUnsafeClearTable
*
*    NOTE: ASSUMES THAT THE SEMAPHORES ARE ALREADY LOCKED
*
*******************************************************************************/
static BOOLEAN bUnsafeClearTable(
        SPORTS_SERVICE_OBJECT hSportsService,
        char *pacName
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    bOk = bSnprintf(
            &psSportsObj->acInternalSqlBuf[0],
            sizeof(psSportsObj->acInternalSqlBuf),
            SS_DELETE_ALL_DATA_FROM_TABLE,
            pacName );

    if( bOk )
    {
        bOk = SQL_INTERFACE.bExecuteCommand (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf );
    }
    return bOk;
}

/*****************************************************************************
*
* bProcessGetSCVerResult
*
*****************************************************************************/
static BOOLEAN bProcessGetSCVerResult(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    BOOLEAN bReturn = FALSE;
    SPORTS_SERVICE_MGR_CONFIG_STRUCT *psSC_Config =
            (SPORTS_SERVICE_MGR_CONFIG_STRUCT *)pvArg;

    if( n32NumberOfColumns == SS_SELECT_GET_SC_CONFIG_VER_MAX_FIELDS )
    {
        // get the count & crc
        psSC_Config->un8ConfigVersion =
            (UN8)psColumns[SS_SELECT_GET_SC_CONFIG_VER_VER]
                                               .uData.sUN32.un32Data;
        psSC_Config->un32ConfigCrc =
            (UN32)psColumns[SS_SELECT_GET_SC_CONFIG_VER_CRC]
                                                .uData.sUN32.un32Data;

        bReturn = TRUE;
    }
    else
    {
        psSC_Config->bSuccess = FALSE;
    }


    return bReturn;
}

/*****************************************************************************
*
* bProcessGetAFVerResult
*
*****************************************************************************/
static BOOLEAN bProcessGetAFVerResult(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    BOOLEAN bReturn = FALSE;
    SPORTS_SERVICE_AF_CONFIG_STRUCT *psAF_Config =
            (SPORTS_SERVICE_AF_CONFIG_STRUCT *)pvArg;

    if( n32NumberOfColumns == SS_SELECT_GET_AF_CONFIG_VER_MAX_FIELDS )
    {
        // get the count & crc
        psAF_Config->un8AfCfgVersion =
            (UN8)psColumns[SS_SELECT_GET_AF_CONFIG_VER_VER]
                                               .uData.sUN32.un32Data;
        psAF_Config->un32AfConfigCrc =
            (UN32)psColumns[SS_SELECT_GET_AF_CONFIG_VER_CRC]
                                                .uData.sUN32.un32Data;

        bReturn = TRUE;
    }
    else
    {
        psAF_Config->bSuccess = FALSE;
    }

    return bReturn;
}

/*****************************************************************************
*
* bProcessRowCountHandler
*
*****************************************************************************/
static BOOLEAN bProcessRowCountHandler(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    BOOLEAN bReturn = FALSE;
    UN32 *pun32Count = (UN32 *)pvArg;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;

    // COUNT should only return 1 column
    if (n32NumberOfColumns == 1)
    {
        psCurrentCol = &psColumns[0];

        // get the count
        *pun32Count = (UN32)psCurrentCol->uData.sUN32.un32Data;

        bReturn = TRUE;
    }

    return bReturn;
}

/*******************************************************************************
*
*   bDbgProcessSelectAllAndDump
*
*******************************************************************************/
static BOOLEAN bDbgProcessSelectAllAndDump (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    UN32 un32Field;
    char *pacField;
    N32 n32Index;

    // At least the InstanceId must be present.
    if( n32NumberOfColumns < 1 )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    if (psResult->un32RowCount == 0)
    {
        // print out the Column headers first
        for (n32Index=0; n32Index<n32NumberOfColumns; n32Index++)
        {
            SPORTS_SERVICE_MGR_vPrintSilent( 1,
                "%s\t", psColumn[n32Index].pcColName);
        }
        SPORTS_SERVICE_MGR_vPrintSilent( 1,"\n");
    }

    // Loop each column
    for(n32Index=0; n32Index<n32NumberOfColumns; n32Index++)
    {
        if(psColumn[n32Index].eType == SQL_COLUMN_TYPE_INTEGER)
        {
            un32Field = psColumn[n32Index].uData.sUN32.un32Data;
            SPORTS_SERVICE_MGR_vPrintSilent( 1,"%u\t",un32Field);
        }
        else if(psColumn[n32Index].eType == SQL_COLUMN_TYPE_C_STRING)
        {
            pacField = (char*)psColumn[n32Index].uData.sCString.pcData;
            SPORTS_SERVICE_MGR_vPrintSilent( 1,"%s\t",pacField);
        }
        else
        {
            SPORTS_SERVICE_MGR_vPrintSilent( 1," \t");
        }
    }

    // Write the closing row tag.
    SPORTS_SERVICE_MGR_vPrintSilent( 1,"\n");

    psResult->un32RowCount++;
    return TRUE;
}

/*******************************************************************************
*
*   bUnsafeDropAllDataInstanceTables
*
*******************************************************************************/
static BOOLEAN bUnsafeDropAllDataInstanceTables(
        SPORTS_SERVICE_OBJECT hSportsService
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_OBJECT_HDL hRemoveList = OSAL_INVALID_OBJECT_HDL;
    OSAL_LINKED_LIST_ENTRY hEntry;
    UN32 un32ListSize;
    UN32 un32Cnt;
    SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT sQuery;
    UN32* pun32TdefKey;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    BOOLEAN bOk;

    //
    // Make a list of tables to remove.
    //
    eOsalCode = OSAL.eLinkedListCreate (
               &hRemoveList,
               SPORTS_SERVICE_MGR_OBJECT_NAME":TableRemoveLL",
               NULL,
               OSAL_LL_OPTION_LINEAR );

    if( eOsalCode != OSAL_SUCCESS )
    {
       return FALSE;
    }

    do
    {
        sQuery.hTableList = hRemoveList;
        sQuery.un32RowCount = 0;
        sQuery.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;

        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            SS_GET_ALL_TABLEDEF_KEYS,
            (SQL_QUERY_RESULT_HANDLER)bProcessMakeTableDropList,
            (void*)&sQuery );

        if( !bOk )
        {
            break;
        }

        if( sQuery.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        // Get the size of the list.
        eOsalCode = OSAL.eLinkedListItems( hRemoveList, &un32ListSize );
        if( eOsalCode != OSAL_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        // Empty result set means we are done.
        if( un32ListSize == 0 )
        {
            break;
        }

        // Go through the list.
        hEntry = OSAL.hLinkedListFirst ( hRemoveList, (void**)&pun32TdefKey );

        for(un32Cnt=0; un32Cnt<un32ListSize; un32Cnt++)
        {
            if( (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                || (pun32TdefKey == NULL) )
            {
                break;
            }

            bOk = bUnsafeDropTable( psSportsObj, (UN32)pun32TdefKey );

            if( !bOk )
            {
                break;
            }

            hEntry = OSAL.hLinkedListNext( hEntry, (void**)&pun32TdefKey );
        }

        if( un32Cnt != un32ListSize )
        {
            bOk = FALSE;
            break;
        }
    }
    while(0);

    if( hRemoveList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hRemoveList,
                NULL );

        (void)OSAL.eLinkedListDelete( hRemoveList );
    }

    return bOk;
}

/*******************************************************************************
*
*   bProcessMakeTableDropList
*
*******************************************************************************/
static BOOLEAN bProcessMakeTableDropList(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT *psResult
       )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;

    do
    {
        if( n32NumberOfColumns != SS_GET_ALL_TABLEDEF_KEYS_MAX_FIELDS )
        {
            break;
        }

        if( psColumn[SS_GET_ALL_TABLEDEF_KEYS_TDEFKEY].eType !=
                SQL_COLUMN_TYPE_INTEGER )
        {
            break;
        }

        // Add it to the list.
        eOsalCode = OSAL.eLinkedListAdd(
            psResult->hTableList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            (void*)psColumn[SS_GET_ALL_TABLEDEF_KEYS_TDEFKEY].uData.sUN32.un32Data );

        if( eOsalCode != OSAL_SUCCESS )
        {
            break;
        }

        psResult->un32RowCount++;
        return TRUE;
    }
    while(0);

    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    return FALSE;
}

/*******************************************************************************
*
*   bUnsafeDropTable
*
*******************************************************************************/
static BOOLEAN bUnsafeDropTable(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32TdefKey
        )
{
    BOOLEAN bOk;

    do
    {
        //
        // Drop the table.
        //

        bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    SS_DROP_TABLE_BY_TDEFKEY,
                    un32TdefKey );

        if( !bOk )
        {
            break;
        }

        bOk =  SQL_INTERFACE.bExecuteCommand(
                        psSportsObj->hSQLConnection,
                        &psSportsObj->acInternalSqlBuf[0] );

        if( !bOk )
        {
            break;
        }

        //
        // Drop the index.
        //
        bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    SS_DROP_DATA_INSTANCE_TABLE_INDEX,
                    un32TdefKey );

        if( !bOk )
        {
            break;
        }

        bOk =  SQL_INTERFACE.bExecuteCommand(
                    psSportsObj->hSQLConnection,
                    &psSportsObj->acInternalSqlBuf[0] );
    }
    while(0);

    return bOk;
}

/*******************************************************************************
*
*   bCreateMonitorList
*
*******************************************************************************/
static BOOLEAN bCreateMonitorList ( SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    N32 n32Written;

    // Keep each instance name unique.
    static UN32 un32Instance = 0;

    // Create object name.
    n32Written = snprintf(
            &acName[0],
            sizeof(acName),
            SPORTS_SERVICE_MGR_OBJECT_NAME":Monitor%u:list",
            un32Instance );

    if( !bSnprintfOk(n32Written, sizeof(acName)) )
    {
        return FALSE;
    }

    // Create the monitor list.
    eReturnCode = OSAL.eLinkedListCreate (
        &psObj->hMonitorList,
        &acName[0],
        NULL,
        OSAL_LL_OPTION_LINEAR );

    if ( eReturnCode != OSAL_SUCCESS )
    {
        return FALSE;
    }

    do
    {
        // Create the monitor list semaphore name.
        bOk = bSnprintf(
            &acName[n32Written],
            sizeof(acName) - n32Written,
            ":sem" );

        if( !bOk )
        {
            break;
        }

        // Create the semaphore for access to the list.  Not taken binary.
        psObj->hMonitorListSem = OSAL_INVALID_OBJECT_HDL;
        eReturnCode = OSAL.eSemCreate(
            &psObj->hMonitorListSem,
            &acName[0],
            1, 1, OSAL_SEM_OPTION_NONE);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        un32Instance++;
        return TRUE;
    }
    while( FALSE );

    vDestroyMonitorList( psObj );
    return FALSE;
}

/*******************************************************************************
*
*   vDestroyMonitorList
*
*******************************************************************************/
static void vDestroyMonitorList( SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj )
{
    if( psObj->hMonitorList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
            psObj->hMonitorList,
            NULL );

        (void)OSAL.eLinkedListDelete( psObj->hMonitorList );
        psObj->hMonitorList = OSAL_INVALID_OBJECT_HDL;
    }

    if( psObj->hMonitorListSem != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eSemDelete( psObj->hMonitorListSem );
        psObj->hMonitorListSem = OSAL_INVALID_OBJECT_HDL;
    }
}

/*******************************************************************************
*
*   vProcessMonitorChange
*
*******************************************************************************/
static void vProcessMonitorChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
    )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake( psSportsObj->hMonitorListSem,
                               OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode == OSAL_SUCCESS )
    {
        psSportsObj->bMonitorProcessing = TRUE;
        // We do not look at the return value.  It is only informational
        eOsalCode = OSAL.eLinkedListIterate (
            psSportsObj->hMonitorList,
            (OSAL_LL_ITERATOR_HANDLER) bProcessMonitorChangeHandler,
            NULL );
        if (eOsalCode != OSAL_SUCCESS)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_LIFE_CYCLE,
                "vProcessMonitorChange: Failure to "
                "iterate monitor list (%s)\n",
                OSAL.pacGetReturnCodeName(eOsalCode));
        }
        psSportsObj->bMonitorProcessing = FALSE;
        OSAL.eSemGive( psSportsObj->hMonitorListSem );
    }
    return;
}

/*******************************************************************************
*
*   bProcessMonitorChangeHandler
*
*******************************************************************************/
static BOOLEAN bProcessMonitorChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    void *pvArg
        )
{
    SPORTS_MONITOR_vProcessChange( hMonitor );
    return TRUE;
}

/*******************************************************************************
*
*   vNotifySportListChange
*
*******************************************************************************/
static void vNotifySportListChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake(  psSportsObj->hMonitorListSem,
                                OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode == OSAL_SUCCESS )
    {
        // We do not look at the return value.  It is only informational
        eOsalCode = OSAL.eLinkedListIterate (
            psSportsObj->hMonitorList,
            (OSAL_LL_ITERATOR_HANDLER) bNotifySportListChangeHandler,
            NULL );
        if (eOsalCode != OSAL_SUCCESS)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_SC_UPDATE,
                "vNotifySportListChange: Failure to "
                "iterate monitor list (%s)\n",
                OSAL.pacGetReturnCodeName(eOsalCode));
        }
        OSAL.eSemGive( psSportsObj->hMonitorListSem );
    }
    return;
}

/*******************************************************************************
*
*   bNotifySportListChangeHandler
*
*******************************************************************************/
static BOOLEAN bNotifySportListChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    void *pvArg
        )
{
    SPORTS_MONITOR_vNotifySportListChange( hMonitor );
    return TRUE;
}

/*******************************************************************************
*
*   vNotifyAffiliateExistanceChange
*
*******************************************************************************/
static void vNotifyAffiliateExistanceChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
    )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake( psSportsObj->hMonitorListSem,
                               OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode == OSAL_SUCCESS )
    {
        // We do not look at the return value.  It is only informational
        eOsalCode = OSAL.eLinkedListIterate (
            psSportsObj->hMonitorList,
            (OSAL_LL_ITERATOR_HANDLER) bNotifyAffiliateExistanceChangeHandler,
            NULL );
        if (eOsalCode != OSAL_SUCCESS)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
                "vNotifyAffiliateExistanceChange: Failure to "
                "iterate monitor list (%s)\n",
                OSAL.pacGetReturnCodeName(eOsalCode));
        }
        OSAL.eSemGive( psSportsObj->hMonitorListSem );
    }
    return;
}

/*******************************************************************************
*
*   bNotifyAffiliateExistanceChangeHandler
*
*******************************************************************************/
static BOOLEAN bNotifyAffiliateExistanceChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    void *pvArg
        )
{
    SPORTS_MONITOR_vNotifyAffiliateExistanceChange( hMonitor );
    return TRUE;
}

/*******************************************************************************
*
*   vNotifyAffiliateChange
*
*******************************************************************************/
static void vNotifyAffiliateChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
    AFFILIATE_ID tAffiliateId,
    SPORTS_MONITOR_EVENT_MASK tChangeBits
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    AFFILIATE_CHANGE_ITERATOR_STRUCT sHelper;

    // Lock out the monitor list.
    eOsalCode = OSAL.eSemTake( psSportsObj->hMonitorListSem,
                               OSAL_OBJ_TIMEOUT_INFINITE );
    if ( eOsalCode == OSAL_SUCCESS )
    {
        // Set up the args.
        sHelper.tAffiliateId = tAffiliateId;
        sHelper.tChangeBits = tChangeBits;

        // We do not look at the return value.  It is only informational.
        eOsalCode = OSAL.eLinkedListIterate (
            psSportsObj->hMonitorList,
            (OSAL_LL_ITERATOR_HANDLER) bNotifyAffiliateChangeHandler,
            (void*)&sHelper );
        if (eOsalCode != OSAL_SUCCESS)
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_AF_UPDATE,
                "vNotifyAffiliateChange: Failure to "
                "iterate monitor list (%s)\n",
                OSAL.pacGetReturnCodeName(eOsalCode));
        }
        OSAL.eSemGive( psSportsObj->hMonitorListSem );
    }
    return;
}

/*******************************************************************************
*
*   bNotifyAffiliateChangeHandler
*
*******************************************************************************/
static BOOLEAN bNotifyAffiliateChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_CHANGE_ITERATOR_STRUCT *psHelper
        )
{
    if( psHelper != NULL )
    {
        SPORTS_MONITOR_vNotifyAffiliateChange (
            hMonitor,
            psHelper->tAffiliateId,
            psHelper->tChangeBits );
    }
    return TRUE;
}

/*******************************************************************************
*
*   bSnprintf
*
*******************************************************************************/
static BOOLEAN bSnprintf( char *pcBuf, size_t tCount, const char *pcFormat, ... )
{
    N32 n32Written;
    va_list tArgList;

    va_start( tArgList, pcFormat );
    n32Written = vsnprintf( pcBuf, tCount, pcFormat, tArgList );
    va_end(tArgList);

    if( (n32Written < 0) || (n32Written >= (N32)tCount) )
    {
        return FALSE;
    }
    return TRUE;
}

/*******************************************************************************
*
*   bUnsafeSetDbJournalMode
*
*******************************************************************************/
static BOOLEAN bUnsafeSetDbJournalMode(
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pacBuffer,
    UN32 un32BufferSize,
    char *pcMode,
    char *pcDb )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_CSTRING_PRAGMA_RESULT_STRUCT sResult;
    char acPragmaMode[SS_JOURNAL_MODE_STRING_MAXSIZE];

    bOk = bSnprintf(    pacBuffer,
                        un32BufferSize,
                        SS_PRAGMA_SET_JOURNAL_MODE,
                        pcDb,
                        pcMode );

    if( !bOk )
    {
        return FALSE;
    }

    vInitializeCstringPragmaQueryStruct(
        &sResult,
        acPragmaMode,
        sizeof(acPragmaMode) );

    bOk = SQL_INTERFACE.bQuery(
            hSQLConnection,
            pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bProcessCStringPragma,
            (void*)&sResult );

    if( !bOk || !sResult.bSuccess )
    {
        return FALSE;
    }

    if( strcmp(pcMode, sResult.pacBuffer) != 0 )
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   bProcessCStringPragma
*
*******************************************************************************/
static BOOLEAN bProcessCStringPragma(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_CSTRING_PRAGMA_RESULT_STRUCT *psResult
            )
{
   if( (n32NumberOfColumns != 1) || (psResult->un32BufferSize == 0) )
   {
       psResult->bSuccess = FALSE;
       return FALSE;
   }

   if( psColumn[0].eType == SQL_COLUMN_TYPE_C_STRING )
   {
       psResult->pacBuffer[0] = '\0';
       if( psColumn[0].uData.sCString.pcData != NULL )
       {
           strncat( psResult->pacBuffer,
                    (const char*)psColumn[0].uData.sCString.pcData,
                    (size_t)(psResult->un32BufferSize - 1) );

           psResult->bSuccess = TRUE;
       }
   }
   else
   {
       psResult->bSuccess = FALSE;
   }

   psResult->un32RowCount++;
   return FALSE;
}

/*******************************************************************************
*
*   bProcessUn32Pragma
*
*******************************************************************************/
static BOOLEAN bProcessUn32Pragma(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT *psResult
            )
{
   if( n32NumberOfColumns != 1 )
   {
       psResult->bSuccess = FALSE;
       return FALSE;
   }

   if( psColumn[0].eType == SQL_COLUMN_TYPE_INTEGER )
   {
       psResult->un32Value = psColumn[0].uData.sUN32.un32Data;
       psResult->bSuccess = TRUE;
   }
   else
   {
       psResult->bSuccess = FALSE;
   }

   psResult->un32RowCount++;
   return FALSE;
}

/*******************************************************************************
*
*   bProcessGetAllAfIds
*
*******************************************************************************/
static BOOLEAN bProcessGetAllAfIds (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_ALL_AFID_QUERY_RESULT_STRUCT *psResult
        )
{
    do
    {
        if( n32NumberOfColumns != 1 )
        {
            break;
        }

        if( psColumn[0].eType != SQL_COLUMN_TYPE_INTEGER )
        {
            break;
        }

        if( psColumn[0].uData.sUN32.un32Data > SS_MAX_AFFILIATE_ID )
        {
            break;
        }

        vBitfieldSet(
                psResult->pun8BitField,
                psResult->un32BitFieldSize,
                psColumn[0].uData.sUN32.un32Data,
                TRUE );

        psResult->un32RowCount++;
        return TRUE;
    }
    while(0);

    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    return FALSE;
}

/*******************************************************************************
*
*   bProcessAffiliateNameChange
*
*******************************************************************************/
static BOOLEAN bProcessAffiliateNameChange (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_AF_NAME_CHANGE_QUERY_RESULT_STRUCT *psResult
        )
{
    N16 n16Result;

    // Only one affiliate can be present so alwasy return false.
    do
    {
        if( n32NumberOfColumns != 1 )
        {
            break;
        }

        if( psColumn[0].eType != SQL_COLUMN_TYPE_C_STRING )
        {
            break;
        }

        if( psColumn[0].uData.sCString.pcData == NULL )
        {
            break;
        }

        n16Result = STRING.n16CompareCStr (
                (const char*)psColumn[0].uData.sCString.pcData,
                0, // let function calculate length of input string
                psResult->hSportAffiliation );

        psResult->bNameChanged = (n16Result != 0) ? TRUE : FALSE;
        psResult->un32RowCount++;
        return FALSE;
    }
    while(0);

    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    return FALSE;
}

/*******************************************************************************
*
*   bUnsafeGetDbPageSize
*
*******************************************************************************/
static BOOLEAN bUnsafeGetDbPageSize(
        SQL_INTERFACE_OBJECT hSQLConnection,
        UN32 *pun32PageSize )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT sResult;

    vInitializeUn32PragmaQueryStruct( &sResult );
    bOk = SQL_INTERFACE.bQuery(
            hSQLConnection,
            SS_PRAGMA_GET_PAGE_SIZE,
            (SQL_QUERY_RESULT_HANDLER)bProcessUn32Pragma,
            (void*)&sResult );

    if( !bOk || !sResult.bSuccess )
    {
        return FALSE;
    }

    *pun32PageSize = sResult.un32Value;
    return TRUE;
}

/*******************************************************************************
*
*   bUnsafeSetDbPageSize
*
*******************************************************************************/
static BOOLEAN bUnsafeSetDbPageSize(
        SQL_INTERFACE_OBJECT hSQLConnection,
        char *pacBuffer,
        UN32 un32BufferSize,
        UN32 un32PageSize )
{
    BOOLEAN bOk;

    // Build the query string.
    bOk = bSnprintf(    pacBuffer,
                        un32BufferSize,
                        SS_PRAGMA_SET_PAGE_SIZE,
                        un32PageSize );

    if( bOk )
    {
        bOk =  SQL_INTERFACE.bExecuteCommand(
                    hSQLConnection,
                    pacBuffer );
    }

    return bOk;
}

/*******************************************************************************
*
*   bUnsafeUpdateTrefRelationships
*
*******************************************************************************/
static BOOLEAN bUnsafeUpdateTrefRelationships (
                SPORTS_SERVICE_OBJECT hSportsService,
                UN32 un32TdefKey,
                UN8 un8LabelId,
                TABLE_ID tChildInstanceId,
                N64 n64RowId,
                TABLE_ID tParentInstanceId,
                UN8 un8LPriority
        )
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    BOOLEAN bOk;

    // TODO: Fix gather to not save the referenced table in this case.  It is difficult
    // to parse and not save a table to the db with the current implementation.
    //
    // EASY WAY: Remove the table we just added if it is a priority we do not support.
    //
    bOk = bIsLabelPrioritySupported( psSportsObj, un8LPriority );

    if( !bOk )
    {
        // Remove the instance data in the DI_xxxxxxxx table.
        bOk = bSnprintf(
                       &psSportsObj->acInternalSqlBuf[0],
                       sizeof(psSportsObj->acInternalSqlBuf),
                       SS_DELETE_OLD_INSTANCE_DATA_BY_TDEFKEY,
                       un32TdefKey,
                       tChildInstanceId );

        if( bOk )
        {
            bOk = SQL_INTERFACE.bExecuteCommand(
                                       psSportsObj->hSQLConnection,
                                       psSportsObj->acInternalSqlBuf );
        }

        if( !bOk )
        {
            return FALSE;
        }

        // Remove the tref from the DI_Info table.
        bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        SS_DELETE_TREF_INSTANCES,
                        tChildInstanceId );

        if( bOk )
        {
            bOk = SQL_INTERFACE.bExecuteCommand(
                    psSportsObj->hSQLConnection,
                    psSportsObj->acInternalSqlBuf );
        }
        return bOk;
    }

    // First, update the parent data instance with the child's TREF
    bOk = bSnprintf(
            &psSportsObj->acInternalSqlBuf[0],
            sizeof(psSportsObj->acInternalSqlBuf),
            SS_UPDATE_PRIMARY_TABLE_TREF,
            un32TdefKey,
            un8LabelId,
            tChildInstanceId,
            n64RowId );

    if (bOk)
    {

        bOk =  SQL_INTERFACE.bExecuteCommand(
                psSportsObj->hSQLConnection,
                &psSportsObj->acInternalSqlBuf[0] );
    }

    // now update the child's DI_Info with the parent's instance ID
    if (bOk)
    {
        bOk = bSnprintf(
            &psSportsObj->acInternalSqlBuf[0],
            sizeof(psSportsObj->acInternalSqlBuf),
            SS_UPDATE_TREF_PARENT,
            tParentInstanceId,
            tChildInstanceId
        );

        if (bOk)
        {
            bOk =  SQL_INTERFACE.bExecuteCommand(
                    psSportsObj->hSQLConnection,
                    &psSportsObj->acInternalSqlBuf[0] );
        }

    }

    return bOk;
}

/*******************************************************************************
*
*   bUnsafeSetDbSavepoint
*
*   Set a savepoint in the database.  Must release it later
*
*******************************************************************************/
static BOOLEAN bUnsafeSetDbSavepoint(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *acSavePointName
        )
{
   BOOLEAN bOk;
   SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

   bOk = bSnprintf(
        &psSportsObj->acInternalSqlBuf[0],
        sizeof(psSportsObj->acInternalSqlBuf),
        SS_SET_ROLLBACK_POINT,
        acSavePointName
    );

    if (bOk)
    {

        bOk =  SQL_INTERFACE.bExecuteCommand(
                 psSportsObj->hSQLConnection,
                 &psSportsObj->acInternalSqlBuf[0]
            );
        //DBG_DB_PRINTF("SetDbSavepoint: %s = %d\n", acSavePointName, bOk);


    }
    return bOk;
}

/*******************************************************************************
*
*   bUnsafeReleaseDbSavepoint
*
*   Release a previously created savepoint.
*
*******************************************************************************/
static BOOLEAN bUnsafeReleaseDbSavepoint(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *acSavePointName
        )
{
   BOOLEAN bOk;
   SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

   bOk = bSnprintf(
        &psSportsObj->acInternalSqlBuf[0],
        sizeof(psSportsObj->acInternalSqlBuf),
        SS_CLEAR_ROLLBACK_POINT,
        acSavePointName
    );

    if (bOk)
    {

        bOk =  SQL_INTERFACE.bExecuteCommand(
                 psSportsObj->hSQLConnection,
                 &psSportsObj->acInternalSqlBuf[0]
            );
    }
    return bOk;
}

/*******************************************************************************
*
*   bUnsafeRollbackDb
*
*   Rollback the database to a previously defined save point.
*
*******************************************************************************/
static BOOLEAN bUnsafeRollbackDb(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *acSavePointName
        )
{
   BOOLEAN bOk;
   SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

   bOk = bSnprintf(
        &psSportsObj->acInternalSqlBuf[0],
        sizeof(psSportsObj->acInternalSqlBuf),
        SS_USE_ROLLBACK_POINT,
        acSavePointName
    );

    if (bOk)
    {

        bOk =  SQL_INTERFACE.bExecuteCommand(
                 psSportsObj->hSQLConnection,
                 &psSportsObj->acInternalSqlBuf[0]
            );
    }
    return bOk;
}

/*******************************************************************************
*
*   vEndDataTableInstance
*
*   Finalize the data instance and/or rollback the database to a previously
*   defined save point.
*
*******************************************************************************/
static void vEndDataTableInstance(
    SPORTS_SERVICE_OBJECT hSportsService,
    const char *acSavePointName,
    BOOLEAN bSuccess,
    SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                 (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // The primary table was successful.  Process replacement rules.
    if( bSuccess )
    {
        bSuccess = bUnsafeExecuteReplacementRules(
                        psSportsObj,
                        psTableInstance );
    }

    if ( bSuccess )
    {
        // only notify when we successfully processed a primary table
        // (and all of its TREFS if there were any)
        vNotifyAffiliateChange (
            psSportsObj,
            psTableInstance->un16AffiliateId,
            1 << psTableInstance->un8Tclass );

        DATASERVICE_IMPL_vLog(
            SPORTS_SERVICE_MGR_OBJECT_NAME
            ": Notify Host: AfId %d changed on ClassId:%d",
            psTableInstance->un16AffiliateId,
            psTableInstance->un8Tclass);

    } else {

        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
             "vEndDataTableInstance:"
             "Instance write failed: AfId=%d, ClassId:%d\n",
             psTableInstance->un16AffiliateId,
             psTableInstance->un8Tclass);

        // we must perform a rollback
        bOk = bUnsafeRollbackDb(
                hSportsService,
                acSavePointName);

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "vEndDataTableInstance:"
                "Rollback Failure:%s\n",
               acSavePointName );
        }
    }

    // release save pt
    bOk = bUnsafeReleaseDbSavepoint(
            hSportsService,
            acSavePointName);

    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
            "vEndDataTableInstance:"
            "Failure db savepoint release:%s\n",
            acSavePointName );
    }

    // Release read semaphore.
    OSAL.eSemGive(psSportsObj->hDbReadSem);
    vProcessMonitorChange ( psSportsObj );

    return;
}

/*******************************************************************************
*
*   bProcessRemoveTrefInstanceTables
*
*******************************************************************************/
static BOOLEAN bProcessRemoveTrefInstanceTables(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TREF_REMOVE_QUERY_STRUCT *psResult
            )
{
    if( n32NumberOfColumns != SS_GET_TREF_REMOVE_INFO_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    }

    if( psResult->un32RowCount >= psResult->un32MaxRowCount )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    }

    if( (psColumn[SS_GET_TREF_REMOVE_INFO_TDEFKEY].eType !=
            SQL_COLUMN_TYPE_INTEGER ) ||
        (psColumn[SS_GET_TREF_REMOVE_INFO_INSTID].eType !=
                        SQL_COLUMN_TYPE_INTEGER) ||
        (psColumn[SS_GET_TREF_REMOVE_INFO_AFID].eType !=
                        SQL_COLUMN_TYPE_INTEGER) )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    }

    if( psResult->eSmsApiCode != SMSAPI_RETURN_CODE_ERROR )
    {
        psResult->psRemoveInfo[psResult->un32RowCount].un32TdefKey =
                    psColumn[SS_GET_TREF_REMOVE_INFO_TDEFKEY].uData.sUN32.un32Data;

        psResult->psRemoveInfo[psResult->un32RowCount].n64IntanceId =
                    psColumn[SS_GET_TREF_REMOVE_INFO_INSTID].n64NativeInteger;

        psResult->psRemoveInfo[psResult->un32RowCount].un32AfId =
                    psColumn[SS_GET_TREF_REMOVE_INFO_AFID].uData.sUN32.un32Data;

        psResult->un32RowCount++;
        return TRUE;
    }

    return FALSE;
}

/*******************************************************************************
*
*   bUnsafeRemoveInstanceTables
*
*******************************************************************************/
static BOOLEAN bUnsafeRemoveInstanceTables(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *pcGetRemoveInfoSql,
        const char *pcRemoveSql,
        BOOLEAN bUseKeyTwice,
        N64 n64KeyValue
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_OBJECT_HDL hRemoveList = OSAL_INVALID_OBJECT_HDL;
    OSAL_LINKED_LIST_ENTRY hEntry;
    UN32 un32ListSize;
    UN32 un32Cnt;
    SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT sQuery;
    SPORTS_SERVICE_TREF_REMOVE_STRUCT *psRemoveInfo;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;
    BOOLEAN bOk;
    const UN32 un32PrintCont = 0;

    SPORTS_SERVICE_MGR_vPrint( un32PrintCont&(SSDL_MGR_DI_UPDATE | SSDL_MGR_AF_UPDATE),
                 "bUnsafeRemoveInstanceTables:"
                 "n64KeyValue: %lld\n",
                 n64KeyValue );

    //
    // Make a list of tables to remove.
    //
    eOsalCode = OSAL.eLinkedListCreate (
               &hRemoveList,
               SPORTS_SERVICE_MGR_OBJECT_NAME":TableRemoveLL",
               NULL,
               OSAL_LL_OPTION_LINEAR );

    if( eOsalCode != OSAL_SUCCESS )
    {
       return FALSE;
    }

    do
    {
        if( bUseKeyTwice )
        {
            bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    pcGetRemoveInfoSql,
                    n64KeyValue,
                    n64KeyValue );
        }
        else
        {
            bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    pcGetRemoveInfoSql,
                    n64KeyValue );
        }

        if( !bOk )
        {
            break;
        }

        sQuery.hTableList = hRemoveList;
        sQuery.un32RowCount = 0;
        sQuery.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;

        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf,
            (SQL_QUERY_RESULT_HANDLER)bProcessMakeTableList,
            (void*)&sQuery );

        if( !bOk )
        {
            break;
        }

        if( sQuery.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        // Get the size of the list.
        eOsalCode = OSAL.eLinkedListItems( hRemoveList, &un32ListSize );
        if( eOsalCode != OSAL_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        // Empty result set means we are done.
        if( un32ListSize == 0 )
        {
            break;
        }

        // Go through the list.
        hEntry = OSAL.hLinkedListFirst ( hRemoveList, (void**)&psRemoveInfo );

        for(un32Cnt=0; un32Cnt<un32ListSize; un32Cnt++)
        {
            if( (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                || (psRemoveInfo == NULL) )
            {
                break;
            }

            //
            // Drop row data.
            //

            bOk = bSnprintf(
                            &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SS_DELETE_OLD_INSTANCE_DATA_BY_TDEFKEY,
                            psRemoveInfo->un32TdefKey,
                            psRemoveInfo->n64IntanceId );

            SPORTS_SERVICE_MGR_vPrint( un32PrintCont&(SSDL_MGR_DI_UPDATE | SSDL_MGR_AF_UPDATE),
                                     "bUnsafeRemoveInstanceTables:"
                                     "Drop row data: %s\n",
                                     psSportsObj->acInternalSqlBuf );

            if( !bOk )
            {
                break;
            }

            bOk = SQL_INTERFACE.bExecuteCommand(
                                    psSportsObj->hSQLConnection,
                                    psSportsObj->acInternalSqlBuf );

            if( !bOk )
            {
                break;
            }

            //
            // Drop label overrides.
            //

            bOk = bSnprintf(
                            &psSportsObj->acInternalSqlBuf[0],
                            sizeof(psSportsObj->acInternalSqlBuf),
                            SS_DELETE_LABEL_OVERRIDES,
                            psRemoveInfo->n64IntanceId );

            SPORTS_SERVICE_MGR_vPrint( un32PrintCont&(SSDL_MGR_DI_UPDATE | SSDL_MGR_AF_UPDATE),
                                     "bUnsafeRemoveInstanceTables:"
                                     "Drop label overrides: %s\n",
                                     psSportsObj->acInternalSqlBuf );

            if( !bOk )
            {
                break;
            }

            bOk = SQL_INTERFACE.bExecuteCommand(
                                    psSportsObj->hSQLConnection,
                                    psSportsObj->acInternalSqlBuf );

            if( !bOk )
            {
                break;
            }

            hEntry = OSAL.hLinkedListNext( hEntry, (void**)&psRemoveInfo );
        }

        if( un32Cnt != un32ListSize )
        {
            bOk = FALSE;
            break;
        }

        // Remove the tables from the DI_info table.
        if( bUseKeyTwice )
        {
            bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        pcRemoveSql,
                        n64KeyValue,
                        n64KeyValue );

        }
        else
        {
            bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        pcRemoveSql,
                        n64KeyValue );
        }

        SPORTS_SERVICE_MGR_vPrint( un32PrintCont&(SSDL_MGR_DI_UPDATE | SSDL_MGR_AF_UPDATE),
                                     "bUnsafeRemoveInstanceTables:"
                                     "Drop instances: %s\n",
                                     psSportsObj->acInternalSqlBuf );

        if( !bOk )
        {
            break;
        }

        bOk = SQL_INTERFACE.bExecuteCommand(
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf );

    }
    while(0);

    if( hRemoveList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hRemoveList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        (void)OSAL.eLinkedListDelete( hRemoveList );
    }

    SPORTS_SERVICE_MGR_vPrint( un32PrintCont&(SSDL_MGR_DI_UPDATE | SSDL_MGR_AF_UPDATE),
                         "bUnsafeRemoveInstanceTables:"
                         "End: bOk %u listsize %u \n",
                         bOk,
                         un32ListSize );
    return bOk;
}

/*******************************************************************************
*
*   bProcessMakeTableList
*
*******************************************************************************/
static BOOLEAN bProcessMakeTableList(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT *psResult
       )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_TREF_REMOVE_STRUCT *psTableInfo = NULL;

    do
    {
        if( n32NumberOfColumns != SS_GET_TREF_REMOVE_INFO_MAX_FIELDS )
        {
            break;
        }

        if( (psColumn[SS_GET_TREF_REMOVE_INFO_TDEFKEY].eType !=
                SQL_COLUMN_TYPE_INTEGER ) ||
            (psColumn[SS_GET_TREF_REMOVE_INFO_INSTID].eType !=
                            SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_GET_TREF_REMOVE_INFO_AFID].eType !=
                            SQL_COLUMN_TYPE_INTEGER) )
        {
            break;
        }

        psTableInfo = (SPORTS_SERVICE_TREF_REMOVE_STRUCT *)SMSO_hCreate(
                                SPORTS_SERVICE_MGR_OBJECT_NAME":TableInfo",
                                sizeof(SPORTS_SERVICE_TREF_REMOVE_STRUCT),
                                SMS_INVALID_OBJECT,
                                FALSE );

        if( psTableInfo == NULL )
        {
           break;
        }

        psTableInfo->un32TdefKey =
                        psColumn[SS_GET_TREF_REMOVE_INFO_TDEFKEY].uData.sUN32.un32Data;

        psTableInfo->n64IntanceId =
                        psColumn[SS_GET_TREF_REMOVE_INFO_INSTID].n64NativeInteger;

        psTableInfo->un32AfId =
                        psColumn[SS_GET_TREF_REMOVE_INFO_AFID].uData.sUN32.un32Data;

        // Add it to the list.
        eOsalCode = OSAL.eLinkedListAdd(
            psResult->hTableList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            psTableInfo );

        if( eOsalCode != OSAL_SUCCESS )
        {
            break;
        }

        psResult->un32RowCount++;
        return TRUE;
    }
    while(0);

    if( psTableInfo != NULL )
    {
        SMSO_vDestroy((SMS_OBJECT)psTableInfo);
    }

    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    return FALSE;
}

/*******************************************************************************
*
*   bBitfieldGet
*
*******************************************************************************/
static BOOLEAN bBitfieldGet(
        UN8* pun8Bitfield,
        UN32 un32BitLength,
        UN32 un32Bit )
{

    N32 n32Entry = ((N32)SS_BYTES_FOR_BITS(un32BitLength) - 1) - (un32Bit>>3);
    if( n32Entry >= (N32)0 )
    {
        return ( pun8Bitfield[(UN32)n32Entry] >> (un32Bit&7) ) & 1;
    }
    return FALSE;
}

/*******************************************************************************
*
*   vBitfieldSet
*
*******************************************************************************/
static void vBitfieldSet(
        UN8* pun8Bitfield,
        UN32 un32BitLength,
        UN32 un32Bit,
        BOOLEAN bVal )
{
    UN8* pun8Byte;
    UN8 un8M;

    N32 n32Entry = ((N32)SS_BYTES_FOR_BITS(un32BitLength) - 1) - (un32Bit>>3);
    if( n32Entry >= (N32)0 )
    {
        pun8Byte = &pun8Bitfield[(UN32)n32Entry];
        un8M = 1<<(un32Bit&7);
        *pun8Byte = bVal ? (*pun8Byte|un8M) : (*pun8Byte&~un8M);
    }
}

/*******************************************************************************
*
*   bBitfieldAnySet
*
*******************************************************************************/
static BOOLEAN bBitfieldAnySet(
        UN8* pun8Bitfield,
        UN32 un32BitLength )
{
    UN32 un32Cnt;
    BOOLEAN bIsSet;

    for( un32Cnt=0; un32Cnt<un32BitLength; un32Cnt++ )
    {
        bIsSet = bBitfieldGet(
                    pun8Bitfield,
                    un32BitLength,
                    un32Cnt );
        if( bIsSet )
        {
            return TRUE;
        }
    }
    return FALSE;
}

/*******************************************************************************
*
*   bUnsafeLoadTdefs
*
*******************************************************************************/
static BOOLEAN bUnsafeLoadTdefs(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_INTERFACE_OBJECT hSportsHbInterface
        )
{
    BOOLEAN bSuccess;
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_OBJECT_HDL hAddList;
    OSAL_OBJECT_HDL hRemoveList = OSAL_INVALID_OBJECT_HDL;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // Record the interface.  We need to use it when we pass tdefs to
    // be loaded in RAM.
    psSportsObj->hInterface = hSportsHbInterface;

    // Lock the db.
    eOsalCode = OSAL.eSemTake(  psSportsObj->hDbReadSem,
                                OSAL_OBJ_TIMEOUT_INFINITE);
    if (eOsalCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    bSuccess = FALSE;
    do
    {
        eOsalCode = OSAL.eLinkedListCreate (
            &hAddList,
            SPORTS_SERVICE_MGR_OBJECT_NAME":TdefAddLL",
            NULL,
            OSAL_LL_OPTION_LINEAR );

        if( eOsalCode != OSAL_SUCCESS )
        {
           break;
        }

        eOsalCode = OSAL.eLinkedListCreate (
           &hRemoveList,
           SPORTS_SERVICE_MGR_OBJECT_NAME":TdefRemoveLL",
           NULL,
           OSAL_LL_OPTION_LINEAR );

        if( eOsalCode != OSAL_SUCCESS )
        {
           break;
        }

        bOk = bUnsafeGetTdefLists(
                psSportsObj,
                hAddList,
                hRemoveList );

        if( !bOk )
        {
            break;
        }

        // We do not keep a status of the load operation.  If it fails
        // and not all(any) tdefs are loaded in RAM they will be gathered
        // when they arrive since the RAM structure will not have a
        // valid entry for a particular tdef.
        (void) OSAL.eLinkedListIterate (
            hAddList,
            (OSAL_LL_ITERATOR_HANDLER) bUnsafeLoadTdefHandler,
            (void*)hSportsService );

        // Process the remove list.
// For evaluation sometimes we want to leave old table defs.
#if 1
        (void) OSAL.eLinkedListIterate (
            hRemoveList,
            (OSAL_LL_ITERATOR_HANDLER) bUnsafeRemoveUnusedTdefHandler,
            (void*)hSportsService );
#endif
        bSuccess = TRUE;
    }
    while(0);

    if( hAddList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hAddList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        (void)OSAL.eLinkedListDelete( hAddList );
    }

    if( hRemoveList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hRemoveList,
                NULL );

        (void)OSAL.eLinkedListDelete( hRemoveList );
    }

    OSAL.eSemGive(psSportsObj->hDbReadSem);

    if( !bSuccess )
    {
        psSportsObj->sPruneCtrl.bRetryTdef = TRUE;
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bUnsafeRemoveUnusedTdefHandler
*
*******************************************************************************/
static BOOLEAN bUnsafeRemoveUnusedTdefHandler(
        void *pvTdefKey,
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT sResult;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));

    bOk = bSnprintf(
                   &psSportsObj->acInternalSqlBuf[0],
                   sizeof(psSportsObj->acInternalSqlBuf),
                   SS_GET_DI_INFO_BY_TDEFKEY,
                   (UN32)pvTdefKey );

    if( bOk )
    {
        // Reuse the un32pragma handler for existence.  It stops on the
        // first row.
        vInitializeUn32PragmaQueryStruct( &sResult );
        bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bProcessUn32Pragma,
                (void*)&sResult );
    }

    if( bOk && sResult.bSuccess && (sResult.un32RowCount == 0) )
    {
        // Drop the DI_xxxxxxxx table,  No need to do the SQL to drop
        // instances in the DI_Info table.  No DI_Info instances was used
        // to decide to drop the tdef.
        bOk = bUnsafeDeleteTableDefFromDB(
                    (SPORTS_SERVICE_OBJECT) psSportsObj,
                    (UN16)((((UN32)pvTdefKey) & 0xffff0000)>>16),
                    (UN8)(((UN32)pvTdefKey) & 0xffff),
                    TRUE,       // Drop DI_xxxxxxxx
                    FALSE );    // Don't check DI_Info
        if ( bOk == FALSE )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                "bUnsafeRemoveUnusedTdefHandler: Failure to "
                "delete table def \n");
        }
    }

    // Try the next tdef.  The more we can drop the smaller the db.
    return TRUE;
}

/*******************************************************************************
*
*   bUnsafeLoadTdefHandler
*
*******************************************************************************/
static BOOLEAN bUnsafeLoadTdefHandler(
        SPORTS_SERVICE_TDEF_ADD_STRUCT *psTdefAdd,
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        )
{
    BOOLEAN bOk;
    BOOLEAN bSuccess;
    UN32 un32TdefKey;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_TDEF_QUERY_RESULT_STRUCT sResult;

    eOsalCode = OSAL.eLinkedListCreate (
                &sResult.hAddList,
                SPORTS_SERVICE_MGR_OBJECT_NAME":TdefAddLabelLL",
                NULL,
                OSAL_LL_OPTION_LINEAR );

    if( eOsalCode != OSAL_SUCCESS )
    {
        return TRUE;
    }

    sResult.un32RowCount = 0;
    sResult.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;

    un32TdefKey = psTdefAdd->un32TdefId << 16;
    un32TdefKey |= psTdefAdd->sTdefInfo.un8TableVer;
    bOk = bSnprintf(
                   &psSportsObj->acInternalSqlBuf[0],
                   sizeof(psSportsObj->acInternalSqlBuf),
                   SS_SELECT_GET_TDEF_LABELS,
                   un32TdefKey );

    if( bOk )
    {
        bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bProcessMakeLabelList,
                (void*)&sResult );
    }

    bSuccess = FALSE;
    if( bOk && (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (sResult.un32RowCount == psTdefAdd->sTdefInfo.un8LabelCount) )
    {
        psTdefAdd->sTdefInfo.hLabelInfoList = sResult.hAddList;

        // Send the tdef to the highband interface.
        bOk = GsSportsServiceIntf.bUnsafeAddTdef(
                   psSportsObj->hInterface,
                   psTdefAdd );

        if( bOk )
        {
            bSuccess = TRUE;
        }
    }

    // If there was success the list was consumed by the highband object.
    // Delete it if not successful.
    if( !bSuccess )
    {
        (void)OSAL.eLinkedListRemoveAll(
                sResult.hAddList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        (void)OSAL.eLinkedListDelete( sResult.hAddList );
    }

    // Try the next tdef.  The more we can get the quicker we can gather tables.
    return TRUE;
}

/*******************************************************************************
*
*   bProcessMakeLabelList
*
*******************************************************************************/
static BOOLEAN bProcessMakeLabelList(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TDEF_QUERY_RESULT_STRUCT *psResult
            )
{
    char *pacTxt;
    size_t tTxtLength;
    SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *psLabelInfo = NULL;
    OSAL_RETURN_CODE_ENUM eOsalCode;

    do
    {
        if( n32NumberOfColumns != SS_SELECT_GET_TDEF_LABELS_MAX_FIELDS )
        {
            break;
        }

        psLabelInfo = (SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT *)SMSO_hCreate(
            SPORTS_SERVICE_MGR_OBJECT_NAME":TDefAddLInfo",
            sizeof(SPORTS_SERVICE_HIGHBAND_LABEL_INFO_STRUCT),
            SMS_INVALID_OBJECT,
            FALSE );

        if( psLabelInfo == NULL )
        {
            break;
        }

        if( (psColumn[SS_SELECT_GET_TDEF_LABELS_LID].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_GET_TDEF_LABELS_LTYPE].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_GET_TDEF_LABELS_LPRIORITY].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_GET_TDEF_LABELS_LSIZE].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_GET_TDEF_LABELS_DATATYPE].eType != SQL_COLUMN_TYPE_INTEGER) )
        {
            break;
        }

        // When writing label text it is set to "" if it was not present.
        // This threw away the ability to distinguish between a ""
        // string in the broadcast and no sting once it hits the db.
        // The eType should always be xx_C_STRING.  We will repopulate the
        // memory tdef: "" ==> (bIsLabelTxtPresent = FALSE)
        psLabelInfo->bIsLabelTxtPresent = FALSE;
        if( psColumn[SS_SELECT_GET_TDEF_LABELS_LTXT].eType == SQL_COLUMN_TYPE_C_STRING )
        {
            pacTxt = (char*)psColumn[SS_SELECT_GET_TDEF_LABELS_LTXT].uData.sCString.pcData;
            tTxtLength = strlen(pacTxt);
            if( (tTxtLength > 0) && (tTxtLength < sizeof(psLabelInfo->LabelText)) )
            {
                strncpy( psLabelInfo->LabelText, pacTxt, sizeof(psLabelInfo->LabelText) );
                psLabelInfo->bIsLabelTxtPresent = TRUE;
            }
        }

        // Dmod will be SQL NULL if it was not present.
        if( psColumn[SS_SELECT_GET_TDEF_LABELS_DMOD].eType == SQL_COLUMN_TYPE_INTEGER )
        {
            psLabelInfo->bIsDmodPresent = TRUE;
            psLabelInfo->un8Dmod =
                BYTE0(psColumn[SS_SELECT_GET_TDEF_LABELS_DMOD].uData.sUN32.un32Data);
        }
        else
        {
            psLabelInfo->bIsDmodPresent = FALSE;
        }

        // Get the easy ones.
        psLabelInfo->un8LabelId =
                BYTE0(psColumn[SS_SELECT_GET_TDEF_LABELS_LID].uData.sUN32.un32Data);

        psLabelInfo->un8Dtype =
                BYTE0(psColumn[SS_SELECT_GET_TDEF_LABELS_DATATYPE].uData.sUN32.un32Data);

        psLabelInfo->un8LbType =
                BYTE0(psColumn[SS_SELECT_GET_TDEF_LABELS_LTYPE].uData.sUN32.un32Data);

        psLabelInfo->un8LPriority =
                BYTE0(psColumn[SS_SELECT_GET_TDEF_LABELS_LPRIORITY].uData.sUN32.un32Data);

        psLabelInfo->un16LabelSize =
                (UN16)psColumn[SS_SELECT_GET_TDEF_LABELS_LSIZE].uData.sUN32.un32Data;

        // TODO: Support extended label data.  Not needed until it is defined.
        // Maybe never.
        psLabelInfo->bIsExtDataPresent = FALSE;

        // Add it to the list.
        eOsalCode = OSAL.eLinkedListAdd(
            psResult->hAddList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            psLabelInfo );

        if( eOsalCode != OSAL_SUCCESS )
        {
            break;
        }

        psResult->un32RowCount++;
        return TRUE;
    }
    while(0);

    if( psLabelInfo != NULL )
    {
        SMSO_vDestroy((SMS_OBJECT)psLabelInfo);
    }

    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    return FALSE;
}

/*******************************************************************************
*
*   bUnsafeGetTdefLists
*
*******************************************************************************/
static BOOLEAN bUnsafeGetTdefLists(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        OSAL_OBJECT_HDL hAddList,
        OSAL_OBJECT_HDL hRemoveList
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_TDEF_QUERY_RESULT_STRUCT sResult;

    sResult.un32RowCount = 0;
    sResult.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
    sResult.n32CurrentTdefId = -1;
    sResult.hAddList = hAddList;
    sResult.hRemoveList = hRemoveList;

    bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            SS_SELECT_TDEF_DESC,
            (SQL_QUERY_RESULT_HANDLER)bProcessGetTdefLists,
            (void*)&sResult );

    if( bOk && (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) )
    {
        return TRUE;
    }

    return FALSE;
}

/*******************************************************************************
*
*   bProcessGetTdefLists
*
*******************************************************************************/
static BOOLEAN bProcessGetTdefLists(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TDEF_QUERY_RESULT_STRUCT *psResult
            )
{
    UN32 un32TdefId;
    UN32 un32TdefVer;
    UN32 un32TdefKey;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_TDEF_ADD_STRUCT *psTdefAdd = NULL;

    //
    // We are processing tdefs in order from lowest to highest id.
    // Tdefs of like id are further ordered most recent to oldest.
    // An auto inc integer is bound to tdefs as they are gathered.
    // This is used for the recent to oldest determination.
    // The most recent version of a tdef gets put in the add list.
    // All other tdefs get put in the remove list.  The most recent
    // tdefs are the closest to the broadcast.  Other tdefs are
    // candidates for removal if no existing table uses it.
    //

    do
    {
        if( n32NumberOfColumns != SS_SELECT_TDEF_DESC_MAX_FIELDS )
        {
            break;
        }

        if( (psColumn[SS_SELECT_TDEF_DESC_TDEFID].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_TDEF_DESC_TDEFVER].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_TDEF_DESC_TABLECLASS].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_TDEF_DESC_LABELCOUNT].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_SELECT_TDEF_DESC_CRC].eType != SQL_COLUMN_TYPE_INTEGER) )
        {
            break;
        }

        un32TdefId =
            psColumn[SS_SELECT_TDEF_DESC_TDEFID].uData.sUN32.un32Data;

        un32TdefVer =
            psColumn[SS_SELECT_TDEF_DESC_TDEFVER].uData.sUN32.un32Data;

        // This is the first time we have seen this tdef.
        // psResult->n32CurrentTdefId is initialized to -1.
        if( un32TdefId != (unsigned)psResult->n32CurrentTdefId )
        {
            psResult->n32CurrentTdefId = un32TdefId;

            // Put it in the add list if one was provided.
            if( psResult->hAddList != OSAL_INVALID_OBJECT_HDL )
            {
                psTdefAdd = (SPORTS_SERVICE_TDEF_ADD_STRUCT *)SMSO_hCreate(
                        SPORTS_SERVICE_MGR_OBJECT_NAME":TDefAddEntry",
                        sizeof(SPORTS_SERVICE_TDEF_ADD_STRUCT),
                        SMS_INVALID_OBJECT,
                        FALSE );

                if( psTdefAdd == NULL )
                {
                    break;
                }

                // Record the tdef info.
                psTdefAdd->un32TdefId = un32TdefId;

                psTdefAdd->sTdefInfo.IsValid = FALSE;
                psTdefAdd->sTdefInfo.hLabelInfoList = NULL;
                psTdefAdd->sTdefInfo.CRC =
                    psColumn[SS_SELECT_TDEF_DESC_CRC].uData.sUN32.un32Data;
                psTdefAdd->sTdefInfo.un8LabelCount =
                    BYTE0(psColumn[SS_SELECT_TDEF_DESC_LABELCOUNT].uData.sUN32.un32Data);
                psTdefAdd->sTdefInfo.un8TableClass =
                    BYTE0(psColumn[SS_SELECT_TDEF_DESC_TABLECLASS].uData.sUN32.un32Data);
                psTdefAdd->sTdefInfo.un8TableVer = BYTE0(un32TdefVer);

                // Add it to the list.
                eOsalCode = OSAL.eLinkedListAdd(
                    psResult->hAddList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    psTdefAdd );

                if( eOsalCode != OSAL_SUCCESS )
                {
                    SMSO_vDestroy((SMS_OBJECT)psTdefAdd);
                    break;
                }
            }
        }
        // This is a tdef which is not the most recent version.
        else
        {
            // Put it in the remove list if one was provided.
            if( psResult->hRemoveList != OSAL_INVALID_OBJECT_HDL )
            {
                un32TdefKey = un32TdefId << 16;
                un32TdefKey |= un32TdefVer;

                // Just use the list void pointer to hold the tdef key.
                // We will never dereference it.
                eOsalCode = OSAL.eLinkedListAdd(
                    psResult->hRemoveList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    (void*)un32TdefKey );

                if( eOsalCode != OSAL_SUCCESS )
                {
                    break;
                }
            }
        }

        psResult->un32RowCount++;
        return TRUE;
    }
    while(0);

    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    return FALSE;
}


/*******************************************************************************
*
*   bExecuteSaveRules
*
*******************************************************************************/
static BOOLEAN bExecuteSaveRules(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        BOOLEAN bNotify,
        N32 n32MaxDropCount
        )
{
    BOOLEAN bOk;
    BOOLEAN bWorkNeeded = FALSE;
    BOOLEAN bWorkLeft = FALSE;
    UN32 un32Cnt;
    UN32 un32Epoch;
    UN32 un32CheckEpoch;
    UN32 un32UTCsec;
    UN32 un32CheckReceiptTime;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_TABLE_CLASS_INFO_STRUCT *psInfo;

    // Get TOD
    un32UTCsec = 0;
    eOsalCode = OSAL.eTimeGet( &un32UTCsec );
    if( eOsalCode != OSAL_SUCCESS )
    {
        printf("Result: %s\n",
               OSAL.pacGetReturnCodeName(eOsalCode));
        return FALSE;
    }

    // Convert it into a 2:00am starting epoch.
    un32Epoch = un32UTCsec;
    if( un32Epoch > SS_TWO_HOURS_IN_SECONDS )
    {
        un32Epoch -= SS_TWO_HOURS_IN_SECONDS;
    }
    un32Epoch /= SS_SECONDS_PER_DAY;

    for(un32Cnt=0; un32Cnt<SS_MAX_TCLASS_COUNT; un32Cnt++)
    {
        psInfo = &psSportsObj->asClassInfo[un32Cnt];

        if( !psInfo->un8IsValid )
        {
            continue;
        }

        // This pruning only applies to these rules.
        if( (psInfo->un8SaveRule != SS_SAVE_RULE_KEEP_ALL) &&
            (psInfo->un8SaveRule != SS_SAVE_RULE_KEEP_ONE) )
        {
            continue;
        }

        // Determine the check epoch.
        if( un32Epoch < psInfo->un8SaveDur )
        {
            un32CheckEpoch = un32Epoch;
        }
        else
        {
            un32CheckEpoch = un32Epoch - psInfo->un8SaveDur;
        }

        // Determine the check receipt time.
        if( un32UTCsec < psInfo->un32ReceiptDur )
        {
            un32CheckReceiptTime = un32UTCsec;
        }
        else
        {
            un32CheckReceiptTime = un32UTCsec - psInfo->un32ReceiptDur;
        }

        // If we got here we are processing the db for age outs.
        bWorkNeeded = TRUE;

        bOk =  bExecuteAgeOut(
                psSportsObj,
                SS_SAVE_RULES_REMOVE_COUNT,
                SS_SAVE_RULES_GET_TABLES_TO_REMOVE,
                un32Cnt,
                un32CheckEpoch,
                un32CheckReceiptTime,
                TRUE,
                bNotify,
                n32MaxDropCount );

        if( !bOk )
        {
            bWorkLeft = TRUE;
        }
    }

    // We may have tried to do some work.
    if( bWorkNeeded )
    {
        // Return false if there is more work.
        return !bWorkLeft;
    }

    // No works was needed return true.
    return TRUE;
}

/*******************************************************************************
*
*   vExecuteShutdownRules
*
*******************************************************************************/
static void vExecuteShutdownRules(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj )
{
    UN32 un32Cnt;
    UN32 un32Epoch;
    UN32 un32UTCsec;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_TABLE_CLASS_INFO_STRUCT *psInfo;

    // Get TOD
    un32UTCsec = 0;
    eOsalCode = OSAL.eTimeGet( &un32UTCsec );
    if( eOsalCode != OSAL_SUCCESS )
    {
        printf("Result: %s\n",
               OSAL.pacGetReturnCodeName(eOsalCode));
        return;
    }

    // Convert it into a 2:00am starting epoch.
    un32Epoch = un32UTCsec;
    if( un32Epoch > SS_TWO_HOURS_IN_SECONDS )
    {
        un32Epoch -= SS_TWO_HOURS_IN_SECONDS;
    }
    un32Epoch /= SS_SECONDS_PER_DAY;

    for(un32Cnt=0; un32Cnt<SS_MAX_TCLASS_COUNT; un32Cnt++)
    {
        psInfo = &psSportsObj->asClassInfo[un32Cnt];

        if( !psInfo->un8IsValid )
        {
            continue;
        }

        if( psInfo->un8ShutdownRule == SS_SHUTDOWN_RULE_DROP_CURRENT_DAY )
        {
            // If we got here we are processing the db.
            bExecuteAgeOut(
                    psSportsObj,
                    SS_SHUTDOWN_RULES_REMOVE_COUNT,
                    SS_SHUTDOWN_RULES_GET_TABLES_TO_REMOVE,
                    un32Cnt,
                    un32Epoch,
                    0,
                    FALSE,
                    FALSE,
                    -1 );
        }
    }

    return;
}

/*******************************************************************************
*
*   bExecuteAgeOut
*
*******************************************************************************/
static BOOLEAN bExecuteAgeOut(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        const char *pcRowCountSql,
        const char *pcGetRemoveInfoSql,
        UN32 un32Class,
        UN32 un32Epoch,
        UN32 un32ReceiptTime,
        BOOLEAN bUseReceiptTime,
        BOOLEAN bNotify,
        N32 n32MaxDropCount
        )
{
    BOOLEAN bOk;
    BOOLEAN bReleaseOk;
    UN32 un32Row;
    UN32 un32DesiredDropCount = 0;
    UN32 un32ActualDropCount = 0;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    SPORTS_SERVICE_TREF_REMOVE_QUERY_STRUCT sResult;
    UN32 un32Sec;
    UN16 un16Msec;
    UN64 un64Start;
    UN64 un64End;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));

    // Lock the db.
    eOsalCode = OSAL.eSemTake(  psSportsObj->hDbReadSem,
                                OSAL_OBJ_TIMEOUT_INFINITE);
    if (eOsalCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    OSAL.vTimeUp(&un32Sec, &un16Msec);
    un64Start = (UN64)un32Sec * (UN64)1000 + (UN64)un16Msec;

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                 "bExecuteAgeOut:"
                 "un32Class: %u un32Epoch %u un32ReceiptTime %u bUseReceiptTime %u\n",
                 un32Class,
                 un32Epoch,
                 un32ReceiptTime,
                 bUseReceiptTime );

    if( bUseReceiptTime )
    {
        bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    pcRowCountSql,
                    un32Class,
                    un32Epoch,
                    un32ReceiptTime );
    }
    else
    {
        bOk = bSnprintf(
                    &psSportsObj->acInternalSqlBuf[0],
                    sizeof(psSportsObj->acInternalSqlBuf),
                    pcRowCountSql,
                    un32Class,
                    un32Epoch );
    }

    // Find out how many rows (tables) we will be receiving
    if( bOk )
    {
        sResult.un32MaxRowCount = 0;
        bOk = SQL_INTERFACE.bQuery (
                psSportsObj->hSQLConnection,
                psSportsObj->acInternalSqlBuf,
                (SQL_QUERY_RESULT_HANDLER)bProcessRowCountHandler,
                (void *)&sResult.un32MaxRowCount );
    }

    sResult.psRemoveInfo = NULL;
    if( bOk )
    {
        if( sResult.un32MaxRowCount != 0 )
        {
            sResult.psRemoveInfo = OSAL.pvMemoryAllocate(
                SPORTS_SERVICE_MGR_OBJECT_NAME":RULE_ONE_R_LIST",
                sResult.un32MaxRowCount *
                sizeof(SPORTS_SERVICE_TREF_REMOVE_STRUCT),
                FALSE );
        }
        else
        {
            // We are done.
            OSAL.eSemGive(psSportsObj->hDbReadSem);
            return TRUE;
        }
    }

    // The memory allocation failed.
    if( sResult.psRemoveInfo == NULL )
    {
        // We are done.
        OSAL.eSemGive(psSportsObj->hDbReadSem);
        return FALSE;
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                 "bExecuteAgeOut:"
                 "Removing %u tables\n",
                 sResult.un32MaxRowCount );

    if( bUseReceiptTime )
    {
        bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        pcGetRemoveInfoSql,
                        un32Class,
                        un32Epoch,
                        un32ReceiptTime );
    }
    else
    {
        bOk = bSnprintf(
                        &psSportsObj->acInternalSqlBuf[0],
                        sizeof(psSportsObj->acInternalSqlBuf),
                        pcGetRemoveInfoSql,
                        un32Class,
                        un32Epoch );
    }

    if( bOk )
    {
        sResult.un32RowCount = 0;
        sResult.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;

        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf,
            (SQL_QUERY_RESULT_HANDLER)bProcessRemoveTrefInstanceTables,
            (void*)&sResult );
    }

    // Free the db lock.  We let the user access the data in between removal.
    OSAL.eSemGive(psSportsObj->hDbReadSem);

    if (    (bOk) &&
            (sResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (sResult.un32RowCount == sResult.un32MaxRowCount) )
    {
        un32DesiredDropCount = sResult.un32MaxRowCount;

        // Limit the number of rows to keep the duration of the db lock short
        if( n32MaxDropCount >= 0 )
        {
            if( (UN32)n32MaxDropCount < un32DesiredDropCount )
            {
                un32DesiredDropCount = n32MaxDropCount;
            }
        }

        // We will remove as many tables as possible.
        for(un32Row = 0; un32Row<un32DesiredDropCount; un32Row++)
        {

            // Lock the db independently for each individual removal.
            eOsalCode = OSAL.eSemTake(  psSportsObj->hDbReadSem,
                                        OSAL_OBJ_TIMEOUT_INFINITE);

            if (eOsalCode != OSAL_SUCCESS)
            {
                OSAL.vMemoryFree(sResult.psRemoveInfo);
                return FALSE;
            }

            // Create a save point since we are going to modify the db.
            bOk = bUnsafeSetDbSavepoint(
                    (SPORTS_SERVICE_OBJECT) psSportsObj,
                    SS_ROLLBACK_POINT_AGE_OUT );

            if( !bOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                    "bExecuteAgeOut:"
                    "Failure to set DB savepoint:%s\n",
                    SS_ROLLBACK_POINT_AGE_OUT );

                OSAL.eSemGive(psSportsObj->hDbReadSem);
                continue;
            }

            bOk = bUnsafeRemoveInstanceTables(
                       (SPORTS_SERVICE_OBJECT) psSportsObj,
                       SS_GET_TABLE_AND_TREF_REMOVE_INFO,
                       SS_DELETE_TABLE_AND_TREF_INSTANCES,
                       TRUE,
                       sResult.psRemoveInfo[un32Row].n64IntanceId );

            // Record the successful drop.
            if( bOk )
            {
                un32ActualDropCount++;

                if( bNotify )
                {
                    vNotifyAffiliateChange (
                            psSportsObj,
                            sResult.psRemoveInfo[un32Row].un32AfId,
                            1 << un32Class );
                }
            }

            // Free the restore.
            bReleaseOk = bUnsafeReleaseDbSavepoint(
                    (SPORTS_SERVICE_OBJECT) psSportsObj,
                    SS_ROLLBACK_POINT_AGE_OUT);

            // Unlock the db.
            OSAL.eSemGive(psSportsObj->hDbReadSem);

            if( !bReleaseOk )
            {
                SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                    "bExecuteAgeOut:"
                    "Failure db savepoint release:%s\n",
                    SS_ROLLBACK_POINT_AGE_OUT );

                // SQlite will page fault if there are too many save points.
                // Let the db grow rather than grow the save point list.
                // At next power cycle hopefully the restore points will work.
                // and we can prune the db.  09_20_2011
                break;
            }

            if( !bOk )
            {
                break;
            }
        }
    }
    else
    {
        // Something went wrong.  We could case each of the three if
        // need be.
        bOk = FALSE;
    }

    OSAL.vMemoryFree(sResult.psRemoveInfo);

    OSAL.vTimeUp(&un32Sec, &un16Msec);
    un64End = un32Sec *1000 + un16Msec;

    if( bNotify )
    {
        vProcessMonitorChange( psSportsObj );
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
             "bExecuteAgeOut:"
             "End: bOk %u bNotify %u max %u actual %u desired %u dur %u\n",
             bOk,
             bNotify,
             sResult.un32MaxRowCount,
             un32ActualDropCount,
             un32DesiredDropCount,
             un64End - un64Start);

    // We only return TRUE if all tables in the db that meet the drop
    // requirement have been removed.  The caller can use the result
    // to manage scheduling the removal operation.
    return (un32ActualDropCount == sResult.un32MaxRowCount);
}

/*******************************************************************************
*
*   bUnsafeExecuteReplacementRules
*
*******************************************************************************/
static BOOLEAN bUnsafeExecuteReplacementRules(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_OBJECT_HDL hRemoveList = OSAL_INVALID_OBJECT_HDL;
    OSAL_LINKED_LIST_ENTRY hEntry;
    UN32 un32ListSize;
    UN32 un32Cnt;
    SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT sQuery;
    SPORTS_SERVICE_TREF_REMOVE_STRUCT *psRemoveInfo;
    BOOLEAN bOk;

    // No rule means success.
    if( psSportsObj->asClassInfo[psTableInstance->un8Tclass].un8IsValid &&
        (psSportsObj->asClassInfo[psTableInstance->un8Tclass].un8SaveRule !=
         SS_SAVE_RULE_KEEP_ONE) )
    {
        return TRUE;
    }

    //
    // Make a list of tables to remove.
    //
    eOsalCode = OSAL.eLinkedListCreate (
               &hRemoveList,
               SPORTS_SERVICE_MGR_OBJECT_NAME":RRulesLL",
               NULL,
               OSAL_LL_OPTION_LINEAR );

    if( eOsalCode != OSAL_SUCCESS )
    {
       return FALSE;
    }

    do
    {
        bOk = bSnprintf(
                     &psSportsObj->acInternalSqlBuf[0],
                     sizeof(psSportsObj->acInternalSqlBuf),
                     SS_GET_REPLACED_TABLES,
                     psTableInstance->n64InstanceId,
                     psTableInstance->un16TableId,
                     psTableInstance->un16AffiliateId );

        if( !bOk )
        {
            break;
        }

        sQuery.hTableList = hRemoveList;
        sQuery.un32RowCount = 0;
        sQuery.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;

        bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            psSportsObj->acInternalSqlBuf,
            (SQL_QUERY_RESULT_HANDLER)bProcessMakeTableList,
            (void*)&sQuery );

        if( !bOk )
        {
            break;
        }

        if( sQuery.eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        // Get the size of the list.
        eOsalCode = OSAL.eLinkedListItems( hRemoveList, &un32ListSize );
        if( eOsalCode != OSAL_SUCCESS )
        {
            bOk = FALSE;
            break;
        }

        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
            "bUnsafeExecuteReplacementRules"
            "REPLACE: listsize %u SQL %s\n",
            un32ListSize,
            psSportsObj->acInternalSqlBuf );

        // Empty result set means we are done.
        if( un32ListSize == 0 )
        {
            break;
        }

        // Go through the list.
        hEntry = OSAL.hLinkedListFirst ( hRemoveList, (void**)&psRemoveInfo );

         for(un32Cnt=0; un32Cnt<un32ListSize; un32Cnt++)
         {
             if( (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                 || (psRemoveInfo == NULL) )
             {
                 break;
             }

             SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                         "bUnsafeExecuteReplacementRules"
                         "REMOVE: inst %llu af %u tid %u tver %u\n",
                         psRemoveInfo->n64IntanceId,
                         psRemoveInfo->un32AfId,
                         (psRemoveInfo->un32TdefKey&0xffff0000)>>16,
                         psRemoveInfo->un32TdefKey&0xffff );

             bOk = bUnsafeRemoveInstanceTables(
                           (SPORTS_SERVICE_OBJECT) psSportsObj,
                           SS_GET_TABLE_AND_TREF_REMOVE_INFO,
                           SS_DELETE_TABLE_AND_TREF_INSTANCES,
                           TRUE,
                           psRemoveInfo->n64IntanceId );

            if( !bOk )
            {
                break;
            }

            hEntry = OSAL.hLinkedListNext( hEntry, (void**)&psRemoveInfo  );
        }

        if( un32Cnt != un32ListSize )
        {
            bOk = FALSE;
            break;
        }
    }
    while(0);

    if( hRemoveList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hRemoveList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        (void)OSAL.eLinkedListDelete( hRemoveList );
    }

    return bOk;
}

/*******************************************************************************
*
*   vHandleTimeoutEvent
*
*******************************************************************************/
static void vHandleTimeoutEvent (
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
        void *pvEventArg
        )
{
    UN32 un32Epoch;
    UN32 un32UTCsec;
    BOOLEAN bPerformDailyCheck = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    BOOLEAN bOk;
//    UN64 un64TimeUp;

    // Get TOD
    un32UTCsec = 0;
    eOsalCode = OSAL.eTimeGet( &un32UTCsec );
    if( eOsalCode == OSAL_SUCCESS )
    {
        // Convert it into a 2:00am starting epoch with a bias.
        un32Epoch = un32UTCsec;
        if( un32Epoch > (SS_TWO_HOURS_IN_SECONDS + SS_BACKGROUND_EXECUTE_PRUNE_BIAS) )
        {
            un32Epoch -= (SS_TWO_HOURS_IN_SECONDS + SS_BACKGROUND_EXECUTE_PRUNE_BIAS);
        }
        un32Epoch /= SS_SECONDS_PER_DAY;

        if( un32Epoch != psObj->sPruneCtrl.un32Epoch )
        {
            bPerformDailyCheck = TRUE;
            psObj->sPruneCtrl.un32Epoch = un32Epoch;
        }

        if( SSDL_MGR_PRUNE )
        {
            struct tm sTime;
            TIME_T tSecs;
            tSecs = un32UTCsec;
            OSAL.gmtime_r(&tSecs, &sTime);

            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                       "vHandleTimeoutEvent:"
                       "TIME:\n"
                       " tm_sec %u\n"
                       " tm_min %u\n"
                       " tm_hour %u\n"
                       " tm_mday %u\n"
                       " tm_mon %u\n"
                       " tm_year %u\n"
                       " tm_wday %u\n"
                       " tm_yday %u\n"
                       " tm_isdst %u\n",
                       sTime.tm_sec,
                       sTime.tm_min,
                       sTime.tm_hour,
                       sTime.tm_mday,
                       sTime.tm_mon,
                       sTime.tm_year,
                       sTime.tm_wday,
                       sTime.tm_yday,
                       sTime.tm_isdst );
        }
    }
    else
    {
        printf("Result: %s\n",
               OSAL.pacGetReturnCodeName(eOsalCode));
    }

    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                  "vHandleTimeoutEvent:"
                  "unixtime %u unixEpoch %u dailyEpoch %u \n",
                  un32UTCsec,
                  un32Epoch,
                  psObj->sPruneCtrl.un32Epoch
                  );

    // Handle data instance pruning.
    if( psObj->sPruneCtrl.bRetryData || bPerformDailyCheck )
    {
        // Perform data pruning.
        psObj->sPruneCtrl.bRetryData = !bExecuteSaveRules(
             psObj,
             TRUE, // Notify using monitors.
             -1 );

        if( psObj->sPruneCtrl.bRetryData )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                         "vHandleTimeoutEvent: "
                         "Data inst pruning rescheduled\n" );
        }
    }

    // Handle tdef pruning.
    if( psObj->sPruneCtrl.bRetryTdef || bPerformDailyCheck )
    {
        // Perform tdef pruning.
        psObj->sPruneCtrl.bRetryTdef = !bPruneTdefs( psObj );

        if( psObj->sPruneCtrl.bRetryTdef )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                         "vHandleTimeoutEvent: "
                         "Tdef pruning rescheduled\n" );
        }
    }

    //
    // Perform db optimizer maintenance.
    //
//    un64TimeUp = un64TimeUpMs();

    bOk = SQL_INTERFACE.bExecuteCommand(
            psObj->hSQLConnection,
            SS_DB_ANALYZE );

    if( !bOk )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                     "vHandleTimeoutEvent: "
                     "SQL Failure: %s\n", SS_DB_ANALYZE );
    }

//    SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
//             "vHandleTimeoutEvent: "
//             "DB analyze: %llu ms\n",  un64TimeUpMs() - un64TimeUp );
}

/*******************************************************************************
*
*   bPruneTdefs
*
*******************************************************************************/
static BOOLEAN bPruneTdefs(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        )
{
    BOOLEAN bSuccess;
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_OBJECT_HDL hRemoveList;

    // Lock the db.
    eOsalCode = OSAL.eSemTake(  psSportsObj->hDbReadSem,
                                OSAL_OBJ_TIMEOUT_INFINITE);
    if (eOsalCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    bSuccess = FALSE;
    do
    {
        eOsalCode = OSAL.eLinkedListCreate (
           &hRemoveList,
           SPORTS_SERVICE_MGR_OBJECT_NAME":TdefRemoveLL",
           NULL,
           OSAL_LL_OPTION_LINEAR );

        if( eOsalCode != OSAL_SUCCESS )
        {
           break;
        }

        bOk = bUnsafeGetTdefLists(
                psSportsObj,
                NULL,
                hRemoveList );

        if( !bOk )
        {
            break;
        }

        // Process the remove list.
        eOsalCode = OSAL.eLinkedListIterate (
            hRemoveList,
            (OSAL_LL_ITERATOR_HANDLER) bUnsafeRemoveUnusedTdefHandler,
            (void*)psSportsObj );

        if( eOsalCode == OSAL_NO_OBJECTS )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                                 "bPruneTdefs: "
                                 "Empty tdef prune list\n" );
        }
        else if( eOsalCode != OSAL_SUCCESS )
        {
            break;
        }
        else
        {
        }

        bSuccess = TRUE;
    }
    while(0);

    if( hRemoveList != OSAL_INVALID_OBJECT_HDL )
    {
        (void)OSAL.eLinkedListRemoveAll(
                hRemoveList,
                NULL );

        (void)OSAL.eLinkedListDelete( hRemoveList );
    }

    OSAL.eSemGive(psSportsObj->hDbReadSem);

    if( !bSuccess )
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_PRUNE,
                 "bPruneTdefs: "
                 "Retry tdef prune: eOsalCode %u bOk %u\n",
                 eOsalCode,
                 bOk );

        psSportsObj->sPruneCtrl.bRetryTdef = TRUE;
    }

    return bSuccess;
}

/*******************************************************************************
*
*   vIntitializeSupported
*
*******************************************************************************/
static void vIntitializeSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        )
{
    UN32 un32Cnt;

    // Set the supported sports.
    if( SS_SUPPORT_ALL_SPORTS )
    {
        for(un32Cnt=0; un32Cnt<SS_MAX_SPORT_COUNT; un32Cnt++)
        {
            vBitfieldSet(
                    psSportsObj->aun8SupportedSport,
                    SS_MAX_SPORT_COUNT,
                    un32Cnt,
                    TRUE );
        }
    }
    else
    {
        // Set the sports that are supported.
        for(un32Cnt=0; un32Cnt<sizeof(gaun8SupportedSport); un32Cnt++)
        {
            vBitfieldSet(
                    psSportsObj->aun8SupportedSport,
                    SS_MAX_SPORT_COUNT,
                    gaun8SupportedSport[un32Cnt],
                    TRUE );
        }
    }

    // Set the supported table classes
    psSportsObj->un32SupportedTableClass = gun32SupportedTableClass;

    // Populate the rules associates with supported
    for(un32Cnt=0; un32Cnt<SS_MAX_TCLASS_COUNT; un32Cnt++)
    {
        if( gun32SupportedTableClass & (1<<un32Cnt) )
        {
            psSportsObj->asClassInfo[un32Cnt] = gsClassInfo[un32Cnt];
        }
        else
        {
            psSportsObj->asClassInfo[un32Cnt].un8IsValid = 0;
        }
    }

    // Set the supported label priorities.
    psSportsObj->un32SupportedLabelPriority = gun32SupportedLabelPriority;

    // Set the supported labels
    if( SS_SUPPORT_ALL_LABELS )
    {
        for(un32Cnt=0; un32Cnt<=SS_PVN1_LAST_DEFINED_LABEL_TYPE; un32Cnt++)
        {
            vBitfieldSet(
                    psSportsObj->aun8SupportedLabel,
                    SS_MAX_LABEL_COUNT,
                    un32Cnt,
                    TRUE );
        }
    }
    else
    {
        // Set the sports that are supported.
        for(un32Cnt=0; un32Cnt<sizeof(gaun8SupportedLabel); un32Cnt++)
        {
            vBitfieldSet(
                    psSportsObj->aun8SupportedLabel,
                    SS_MAX_LABEL_COUNT,
                    gaun8SupportedLabel[un32Cnt],
                    TRUE );
        }
    }
}

/*******************************************************************************
*
*   bUnsafeGetTableQueryInfo
*
*******************************************************************************/
static BOOLEAN bUnsafeGetTableQueryInfo(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        char *pcTdefSql,
        UN32* pun32TdefKey,
        UN32* pun32ColumnCount,
        UN8* pun8ColumnMap,
        UN32 un32ColumnMapSize
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT sPragmaResult;
    SPORTS_SERVICE_COLUMN_CULL_QUERY_STRUCT sCullResult;
    OSAL.bMemSet(&sCullResult, 0, sizeof(sCullResult));

    // Get the tdefkey.  Reuse the un32pragma handler. It stops on the first row.
    vInitializeUn32PragmaQueryStruct( &sPragmaResult );
    bOk = SQL_INTERFACE.bQuery (
            psSportsObj->hSQLConnection,
            pcTdefSql,
            (SQL_QUERY_RESULT_HANDLER)bProcessUn32Pragma,
            (void*)&sPragmaResult );

    // Could not get a tdefkey bad news.
    if( !bOk && !sPragmaResult.bSuccess && (sPragmaResult.un32RowCount != 1) )
    {
        return FALSE;
    }

    // Build the label query string to get columns.
    bOk = bSnprintf(    &psSportsObj->acAppQueryBuf[0],
                        sizeof(psSportsObj->acAppQueryBuf),
                        SS_GET_COLUMN_PRIORITY,
                        sPragmaResult.un32Value );

    if( bOk )
    {
        sCullResult.psSportsObj = psSportsObj;
        sCullResult.pun8ColumnMap = pun8ColumnMap;
        sCullResult.un32ColumnMapSize = un32ColumnMapSize;
        sCullResult.un32RowCount = 0;
        sCullResult.eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;

        bOk = SQL_INTERFACE.bQuery (
                    psSportsObj->hSQLConnection,
                    psSportsObj->acAppQueryBuf,
                    (SQL_QUERY_RESULT_HANDLER)bProcessColumnPriorityCull,
                    (void*)&sCullResult );
    }

    if( bOk && (sCullResult.eSmsApiCode == SMSAPI_RETURN_CODE_SUCCESS) )
    {
        *pun32TdefKey = sPragmaResult.un32Value;
        *pun32ColumnCount = sCullResult.un32RowCount;
        return TRUE;
    }

    return FALSE;
}

/*******************************************************************************
*
*   bProcessColumnPriorityCull
*
*******************************************************************************/
static BOOLEAN bProcessColumnPriorityCull(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_COLUMN_CULL_QUERY_STRUCT *psResult
        )
{
    UN32 un32Priority;
    UN32 un32LabelType;
    BOOLEAN bIsSupported;

    do
    {
        if( n32NumberOfColumns != SS_GET_COLUMN_PRIORITY_MAX_FIELDS )
        {
            break;
        }

        if( (psColumn[SS_GET_COLUMN_PRIORITY_PRIORITY].eType != SQL_COLUMN_TYPE_INTEGER) ||
            (psColumn[SS_GET_COLUMN_PRIORITY_LTYPE].eType != SQL_COLUMN_TYPE_INTEGER) )
        {
            break;
        }

        un32Priority =
                psColumn[SS_GET_COLUMN_PRIORITY_PRIORITY].uData.sUN32.un32Data;

        un32LabelType =
                psColumn[SS_GET_COLUMN_PRIORITY_LTYPE].uData.sUN32.un32Data;

        bIsSupported = bIsLabelPrioritySupported(   psResult->psSportsObj,
                                                    un32Priority );

        bIsSupported &= bIsLabelSupported( psResult->psSportsObj,
                                           un32LabelType );

        if( psResult->un32RowCount >= psResult->un32ColumnMapSize )
        {
            break;
        }

        vBitfieldSet(
                psResult->pun8ColumnMap,
                psResult->un32ColumnMapSize,
                psResult->un32RowCount,
                bIsSupported );

        psResult->un32RowCount++;
        return TRUE;
    }
    while(0);

    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
    return FALSE;
}

/*******************************************************************************
*
*   bUnsafeBuildDb
*
*******************************************************************************/
static BOOLEAN bUnsafeBuildDb(
        SQL_INTERFACE_OBJECT hDb
        )
{
    BOOLEAN bOk = TRUE;
    UN32 un32Cnt;

    for(un32Cnt=0; un32Cnt<SS_BUILD_DB_STATEMENT_COUNT; un32Cnt++)
    {
        bOk = SQL_INTERFACE.bExecuteCommand(
                hDb,
                gapcDbBuildStrings[un32Cnt] );

        if( !bOk )
        {
            break;
        }
    }

    return bOk;
}

/*******************************************************************************
*
*   eLoadPersistentDb
*
*******************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eLoadPersistentDb (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;
    UN32 un32PageSize;
    SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT sResult;
    SQL_INTERFACE_OBJECT hSQLPersistent = SQL_INTERFACE_INVALID_OBJECT;
    SQL_INTERFACE_OBJECT hSQLMemory = SQL_INTERFACE_INVALID_OBJECT;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;

    do
    {
        // Open the persistent database.
        hSQLPersistent = SQL_INTERFACE.hConnect(
                &psObj->pacFullyQualifiedDatabaseFilePath[0],
                SQL_INTERFACE_OPTIONS_READONLY,
                &eErrorCode );

        if( hSQLPersistent == SQL_INTERFACE_INVALID_OBJECT )
        {
            break;
        }

        // Get the page size of the persistent db.
        bOk = bUnsafeGetDbPageSize( hSQLPersistent, &un32PageSize );
        if( !bOk )
        {
            break;
        }

        psObj->bHavePersistentDbPageSize = TRUE;
        psObj->un32PageSize = un32PageSize;

        // Get the version of the persistent db.
        sResult.bSuccess = TRUE;
        sResult.un32Value = SPORTS_SERVICE_DATABASE_INVALID_VERSION;
        bOk = SQL_INTERFACE.bQuery(
                hSQLPersistent,
                SS_SELECT_GET_DB_VERSION,
                (SQL_QUERY_RESULT_HANDLER)bProcessUn32Pragma,
                (void*)&sResult );

        if( !bOk || !sResult.bSuccess ||
            (sResult.un32Value != SPORTS_SERVICE_DATABASE_VERSION) )
        {
            break;
        }

        hSQLMemory = hCreateMemoryDb( psObj, FALSE );
        if( hSQLMemory == SQL_INTERFACE_INVALID_OBJECT )
        {
            break;
        }

        // Copy the persistent database to the memory database.
        bOk = SQL_INTERFACE.bCopyMainDatabase(  hSQLMemory,
                                                hSQLPersistent );

        if( !bOk )
        {
            break;
        }

        // Close the persistent database since we are done with it.
        SQL_INTERFACE.vDisconnect( hSQLPersistent );

        psObj->un32DbConfigVersion = sResult.un32Value;
        psObj->hSQLConnection = hSQLMemory;

        return DATASERVICE_ERROR_CODE_NONE;
    }
    while(0);

    // Something failed if we got here.  Clean up what we were doing.
    if( hSQLMemory != SQL_INTERFACE_INVALID_OBJECT )
    {
        SQL_INTERFACE.vDisconnect( hSQLMemory );
    }

    if( hSQLPersistent != SQL_INTERFACE_INVALID_OBJECT )
    {
        SQL_INTERFACE.vDisconnect( hSQLPersistent );
    }

    //
    // Build a new db.
    //

    hSQLMemory = hCreateMemoryDb( psObj, TRUE );
    if( hSQLMemory != SQL_INTERFACE_INVALID_OBJECT )
    {
        psObj->hSQLConnection = hSQLMemory;
        psObj->sAF_Config.un8AfCfgVersion = 0;
        psObj->sAF_Config.un32AfConfigCrc = 0;
        psObj->un32DbConfigVersion = SPORTS_SERVICE_DATABASE_VERSION;
        return DATASERVICE_ERROR_CODE_NONE;
    }

    return DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
}

/*******************************************************************************
*
*   vUnsafeUpdateTableReceiptTime
*
*******************************************************************************/
static void vUnsafeUpdateTableReceiptTime(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
        BOOLEAN bIsPrimaryTable
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
            (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    // This only applies to primary tables that we already have saved.  Below
    // we update the primary and any associated reference tables.  If the unix time
    // is zero we failed to get it.  Since this is an update do not modify the time
    // what we already have, it is the most recent time.
    if( !bIsPrimaryTable || (psTableInstance->un32UTCsec == 0) )
    {
        return;
    }

    // Update receipt time.
    bOk = bSnprintf(
            &psSportsObj->acInternalSqlBuf[0],
            sizeof(psSportsObj->acInternalSqlBuf),
            SS_UPDATE_TABLE_RECEIPT_TIME,
            psTableInstance->un32UTCsec,
            psTableInstance->n64InstanceId,
            psTableInstance->n64InstanceId );

    if( bOk )
    {
        bOk = SQL_INTERFACE.bExecuteCommand(
                    psSportsObj->hSQLConnection,
                    &psSportsObj->acInternalSqlBuf[0] );

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "vUnsafeUpdateTableReceiptTime:"
                "Failed to update table receipt time. instid %lld SQL:%s\n",
                psTableInstance->n64InstanceId,
                psSportsObj->acInternalSqlBuf );
        }
    }
    else
    {
        SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
            "vUnsafeUpdateTableReceiptTime:"
            "Failed to build table receipt time SQL string. instid %lld\n",
            psTableInstance->n64InstanceId );
    }
    return;
}

/*******************************************************************************
*
*   bUnsafeAddLabelOverride
*
*******************************************************************************/
static BOOLEAN bUnsafeAddLabelOverride(
        SPORTS_SERVICE_OBJECT hSportsService,
        TABLE_ID tTableId,
        UN32 un32LabelId,
        UN32 un32Override
        )
{
    BOOLEAN bOk;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj =
                        (SPORTS_SERVICE_MGR_OBJECT_STRUCT *)hSportsService;

    bOk = bSnprintf(
            &psSportsObj->acInternalSqlBuf[0],
            sizeof(psSportsObj->acInternalSqlBuf),
            SS_TABLE_ADD_LABEL_OVERRIDE,
            tTableId,
            un32LabelId,
            un32Override );

    if( bOk )
    {
        bOk = SQL_INTERFACE.bExecuteCommand(
                    psSportsObj->hSQLConnection,
                    &psSportsObj->acInternalSqlBuf[0] );

        if( !bOk )
        {
            SPORTS_SERVICE_MGR_vPrint( SSDL_MGR_DI_UPDATE,
                "bUnsafeAddLabelOverride:"
                "Failed to add override. TableId %lld LableId %u Override %u SQL:%s\n",
                tTableId,
                un32LabelId,
                un32Override,
                psSportsObj->acInternalSqlBuf );
        }
    }

    return bOk;
}

/*******************************************************************************
*
*   bProcessGetLabelOverrides
*
*******************************************************************************/
static BOOLEAN bProcessGetLabelOverrides (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    UN32 un32Index;
    UN32 un32LabelType;
    UN32 un32Priority;
    UN32 un32OverrideSize;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    BOOLEAN bIsSupported;

    // All columns must be present.
    if( n32NumberOfColumns != SS_GET_LABEL_OVERRRIDES_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    // Get first since we cull based on these.
    un32Priority =
        psColumn[SS_GET_LABEL_OVERRRIDES_LPRIORITY]
        .uData.sUN32.un32Data;

    un32LabelType =
            psColumn[SS_GET_LABEL_OVERRRIDES_LTYPE]
            .uData.sUN32.un32Data;

    bIsSupported = bIsLabelPrioritySupported( psResult->psSportsObj,
                                              un32Priority );

    bIsSupported &= bIsLabelSupported( psResult->psSportsObj,
                                       un32LabelType );

    if( !bIsSupported )
    {
        // Unsupported priorities and labels are skipped.  The row count is not
        // incremented.
        return TRUE;
    }

    // Get the fields that are always present.
    un32Index =
        psColumn[SS_GET_LABEL_OVERRRIDES_ORDERINDEX]
        .uData.sUN32.un32Data;

    un32OverrideSize =
        psColumn[SS_GET_LABEL_OVERRRIDES_LSIZE]
        .uData.sUN32.un32Data;

    //
    // Write the closing entry tag.
    //
    eSmsApiCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_OVERRIDE_LIST_ENTRY_OPEN
                        SS_TAG_ID_OPEN"%u"SS_TAG_ID_CLOSE
                        SS_TAG_LBSIZE_OPEN"%u"SS_TAG_LBSIZE_CLOSE
                        SS_TAG_OVERRIDE_LIST_ENTRY_CLOSE,
                        un32Index,
                        un32OverrideSize
                            );
    if (eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }
    psResult->un32RowCount++;

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessGetTableList
*
*******************************************************************************/
static BOOLEAN bProcessGetTableList (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT *psResult
        )
{
    SPORTS_SERVICE_TABLE_LIST_ENTRY_STRUCT *psEntry = NULL;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    char *pacTitle;
    size_t tLength;

    // All columns must be present.
    if( n32NumberOfColumns != SS_SELECT_GET_AFFILIATE_TABLE_LIST_MAX_FIELDS )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    psEntry = (SPORTS_SERVICE_TABLE_LIST_ENTRY_STRUCT *)SMSO_hCreate(
                            SPORTS_SERVICE_MGR_OBJECT_NAME":TListEntry",
                            sizeof(SPORTS_SERVICE_TABLE_LIST_ENTRY_STRUCT),
                            SMS_INVALID_OBJECT,
                            FALSE );

    if( psEntry == NULL )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    psEntry->tInstId =
        (TABLE_ID)
        psColumn[SS_SELECT_GET_AFFILIATE_TABLE_LIST_INSTANCE_ID]
                 .n64NativeInteger;

    psEntry->un32SeasonStatus =
        psColumn[SS_SELECT_GET_AFFILIATE_TABLE_LIST_SEASON_STATUS]
        .uData.sUN32.un32Data;

    psEntry->un32Epoch =
        psColumn[SS_SELECT_GET_AFFILIATE_TABLE_LIST_EPOCH]
        .uData.sUN32.un32Data;

    psEntry->un32ReceiptTime =
            psColumn[SS_SELECT_GET_AFFILIATE_TABLE_LIST_RECEIPT_TIME]
            .uData.sUN32.un32Data;

    psEntry->un32TdefKey =
            psColumn[SS_SELECT_GET_AFFILIATE_TABLE_LIST_TDEFKEY]
            .uData.sUN32.un32Data;

    psEntry->bTitlePresent = FALSE;
    if( psColumn[SS_SELECT_GET_AFFILIATE_TABLE_LIST_ITITLE].eType ==
        SQL_COLUMN_TYPE_C_STRING )
    {
        pacTitle =
            (char*)psColumn[SS_SELECT_GET_AFFILIATE_TABLE_LIST_ITITLE]
            .uData.sCString.pcData;

        tLength = strlen( pacTitle );
        if( (tLength > 0) && (tLength < (sizeof(psEntry->acTitle) - 1)) )
        {
            strncpy( psEntry->acTitle, pacTitle, sizeof(psEntry->acTitle) - 1 );
            psEntry->bTitlePresent = TRUE;
        }
    }

    eOsalCode = OSAL.eLinkedListAdd(
                        psResult->hTableList,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                        psEntry );

    if( eOsalCode != OSAL_SUCCESS )
    {
        SMSO_vDestroy((SMS_OBJECT)psEntry);
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    psResult->un32RowCount++;

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bUnsafeGetTableListHandler
*
*******************************************************************************/
static BOOLEAN bUnsafeGetTableListHandler(
        SPORTS_SERVICE_TABLE_LIST_ENTRY_STRUCT *psEntry,
        SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    BOOLEAN bOk;
    char *pcRewind = NULL;
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj = psResult->psSportsObj;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

    //
    // Write the table information.  Including the opening entry tag.
    //

    // Write the result.
    if( psEntry->bTitlePresent )
    {
        eSmsApiCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_TABLE_LIST_ENTRY_OPEN
                        SS_TAG_ID_OPEN"%llu"SS_TAG_ID_CLOSE
                        SS_TAG_NAME_OPEN"%s"SS_TAG_NAME_CLOSE
                        SS_TAG_SEASON_STATUS_OPEN"%u"SS_TAG_SEASON_STATUS_CLOSE
                        SS_TAG_EPOCH_OPEN"%u"SS_TAG_EPOCH_CLOSE
                        SS_TAG_RECEIPT_TIME_OPEN"%u"SS_TAG_RECEIPT_TIME_CLOSE
                        SS_TAG_HEADER_KEY_OPEN"%u"SS_TAG_HEADER_KEY_CLOSE,
                        psEntry->tInstId,
                        psEntry->acTitle,
                        psEntry->un32SeasonStatus,
                        psEntry->un32Epoch,
                        psEntry->un32ReceiptTime,
                        psEntry->un32TdefKey
                            );
    }
    else
    {
        eSmsApiCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_TABLE_LIST_ENTRY_OPEN
                        SS_TAG_ID_OPEN"%llu"SS_TAG_ID_CLOSE
                        SS_TAG_SEASON_STATUS_OPEN"%u"SS_TAG_SEASON_STATUS_CLOSE
                        SS_TAG_EPOCH_OPEN"%u"SS_TAG_EPOCH_CLOSE
                        SS_TAG_RECEIPT_TIME_OPEN"%u"SS_TAG_RECEIPT_TIME_CLOSE
                        SS_TAG_HEADER_KEY_OPEN"%u"SS_TAG_HEADER_KEY_CLOSE,
                        psEntry->tInstId,
                        psEntry->un32SeasonStatus,
                        psEntry->un32Epoch,
                        psEntry->un32ReceiptTime,
                        psEntry->un32TdefKey
                            );
    }

    if (eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    //
    // Write overrides
    //

    bOk = bSnprintf( &psObj->acAppQueryBuf[0],
                            sizeof(psObj->acAppQueryBuf),
                            SS_GET_LABEL_OVERRIDES,
                            psEntry->tInstId,
                            psEntry->un32TdefKey );

    if( bOk )
    {
       psResult->un32RowCount = 0;
       pcRewind = psResult->pacCursor;

       bOk = bPerformQuery (
                       psObj->hSQLConnection,
                       SS_TAG_OVERRIDE_LIST_OPEN,
                       SS_TAG_OVERRIDE_LIST_CLOSE,
                       psObj->acAppQueryBuf,
                       psResult,
                       (SQL_QUERY_RESULT_HANDLER)bProcessGetLabelOverrides );
    }

    if( !bOk )
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }

    if (psResult->eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    // Empty result, rewind the cursor to remove any override text.
    if( psResult->un32RowCount == 0 )
    {
        psResult->pacCursor = pcRewind;
    }

    //
    // Write the closing entry tag.
    //
    eSmsApiCode = eSports_Service_snprintf(
                        psResult,
                        SS_TAG_TABLE_LIST_ENTRY_CLOSE
                            );

    if (eSmsApiCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_ERROR;
        return FALSE;
    }
    return TRUE;
}

/*******************************************************************************
*
*   hCreateMemoryDb
*
*******************************************************************************/
static SQL_INTERFACE_OBJECT hCreateMemoryDb(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
        BOOLEAN bBuildDb
        )
{
    BOOLEAN bOk;
    SQL_INTERFACE_OBJECT hSQLMemory = SQL_INTERFACE_INVALID_OBJECT;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;

    do
    {
        // Open a new memory database
        hSQLMemory = SQL_INTERFACE.hConnect(
            ":memory:",
            SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK,
            &eErrorCode );

        if( hSQLMemory == SQL_INTERFACE_INVALID_OBJECT )
        {
            break;
        }

        if( psObj->bHavePersistentDbPageSize )
        {
            // Set the page size of the memory db.  It must be the same as the
            // persistent one or the copy is not possible.  Per SQLite
            // documentation this is valid to do since we have not yet
            // created any tables in the db.
            bOk = bUnsafeSetDbPageSize(   hSQLMemory,
                                    &psObj->acInternalSqlBuf[0],
                                    sizeof(psObj->acInternalSqlBuf),
                                    psObj->un32PageSize );

            if( !bOk )
            {
                break;
            }
        }

        // Set the journal mode.
        bOk = bUnsafeSetDbJournalMode(
                 hSQLMemory,
                 &psObj->acInternalSqlBuf[0],
                 sizeof(psObj->acInternalSqlBuf),
                 SS_JOURNAL_MODE_MEMORY,
                 SS_JOURNAL_MODE_ENTIRE_DB );

        if( !bOk )
        {
            break;
        }

        if( bBuildDb )
        {
            bOk = bUnsafeBuildDb( hSQLMemory );
        }

        if( !bOk )
        {
            break;
        }

        return hSQLMemory;
    }
    while(0);

    // Something failed if we got here.  Clean up what we were doing.
    if( hSQLMemory != SQL_INTERFACE_INVALID_OBJECT )
    {
        SQL_INTERFACE.vDisconnect( hSQLMemory );
    }

    return SQL_INTERFACE_INVALID_OBJECT;
}

/*******************************************************************************
*
*   vEndProcessBuffer
*
*******************************************************************************/
static void vEndProcessBuffer(
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        )
{
    N32 n32BufferSize;

    psResult->pacCursor[0] = '\0';
    psResult->pacCursor++;

    n32BufferSize = psResult->pacCursor - psResult->pacScratchPad;

    if (n32BufferSize <= psResult->n32ResultBufferSize)
    {
        memcpy(psResult->pacResultBuffer,psResult->pacScratchPad,
               n32BufferSize);
        psResult->eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
    }
    else
    {
        if (psResult->n32ResultBufferSize > 0)
        {
            psResult->pacResultBuffer[psResult->n32ResultBufferSize-1] = '\0';
        }
        if (  (psResult->un32LastRowStartPos != 0)
            &&(psResult->un32LastRowStartPos < (UN32)psResult->n32ResultBufferSize)
           )
        {
            memcpy(psResult->pacResultBuffer,psResult->pacScratchPad,
                   psResult->un32LastRowStartPos);
            psResult->pacResultBuffer[psResult->un32LastRowStartPos] = '\0';
            psResult->eSmsApiCode = SMSAPI_RETURN_CODE_PARTIAL_TRANSACTION;
        }
        else
        {
            psResult->eSmsApiCode = SMSAPI_RETURN_CODE_INSUFFICIENT_SIZE;
        }
    }

   // set the returned size value equal to how much was
   // written to the scratch pad
   *psResult->pun32RequiredSize = psResult->pacCursor -
            psResult->pacScratchPad;

    return;
}

/*******************************************************************************
*
*   sports_service_snprintf
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSports_Service_snprintf(
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult,
    const char *pcFormat,
    ...)
{
    va_list args;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    int iReturn;
    int iRemainingScratchPad;
    UN8 un8LoopCounter=0;

    iRemainingScratchPad = psResult->un32ScratchPadSize-(psResult->pacCursor-psResult->pacScratchPad)-1;

    va_start(args, pcFormat);
    iReturn = vsnprintf(psResult->pacCursor,
                          iRemainingScratchPad,
                          pcFormat,
                          args);
    va_end(args);

    // While we aren't able to write all the string, make the scratch pad
    // longer and try again.  Non standard implementation of vsnprintf
    // (specifically visual studio) return -1 in this case
    // other implementations return the actual size that would have been written
    // if the size_t value had been larger
    while ((iReturn < 0) || (iReturn >= iRemainingScratchPad))
    {
        char *pacTmpScratchPad;


        // arbitrary break point to make sure we don't get into some wicked infinite
        // loop
        if (un8LoopCounter++ > MAX_SCRATCH_PAD_INCREMENTS)
        {
            iReturn = -1;
            break;
        }

        //allocate a larger memory segment
        pacTmpScratchPad = OSAL.pvMemoryAllocate(
                        SPORTS_SERVICE_MGR_OBJECT_NAME":ScratchPad",
                        SCRATCH_PAD_INCREMENT_SIZE *
                        sizeof(char) + psResult->un32ScratchPadSize,
                        FALSE );
        if (pacTmpScratchPad != NULL)
        {
            // copy anything we previously had and make
            // sure our size is increased
            OSAL.bMemCpy(pacTmpScratchPad,
                         psResult->pacScratchPad,
                         psResult->un32ScratchPadSize);
            psResult->un32ScratchPadSize += SCRATCH_PAD_INCREMENT_SIZE *
                                  sizeof(char);

            psResult->pacCursor = pacTmpScratchPad+(psResult->pacCursor -
                                                    psResult->pacScratchPad);

            OSAL.vMemoryFree(psResult->pacScratchPad);
            psResult->pacScratchPad = pacTmpScratchPad;
            iRemainingScratchPad += psResult->un32ScratchPadSize -
                                     (psResult->pacCursor-psResult->pacScratchPad)-1;

            // try to write again
            va_start(args, pcFormat);
            iReturn = vsnprintf(psResult->pacCursor,
                          iRemainingScratchPad,
                          pcFormat,
                          args);
            va_end(args);
        }
        else
        {
            iReturn = -1;
            break;
        }
    }
    if (iReturn != -1)
    {
        psResult->pacCursor += iReturn;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }
    return eReturnCode;

}

/*******************************************************************************
*
*   vReleaseBindings
*
*******************************************************************************/
void vReleaseBindings(
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction )
{
    UN32 un32Idx;
    for (un32Idx = 0; un32Idx < psTransaction->un32ColumnCount; un32Idx++)
    {
        if (psTransaction->psParams[un32Idx].eType ==
            SQL_BIND_TYPE_STRING_OBJECT)
        {
            STRING_vDestroy(
                (STRING_OBJECT)psTransaction->psParams[un32Idx].pvData);
        }

        psTransaction->psParams[un32Idx].eType = SQL_BIND_TYPE_C_STRING;
        psTransaction->psParams[un32Idx].pvData = NULL;
    }
}
