/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
*  This module contains the Object:SPORTS implementation for the
*  Sirius Module Services (SMS)
*
******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "cdo_obj.h"
#include "cid_obj.h"
#include "cme.h"
#include "league_obj.h"
#include "team_obj.h"
#include "sports_obj.h"
#include "league_team_id.h"
#include "_sports_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eSubType
*
* Retrieves the enumerated value representing the SPORTS object type.
*
*****************************************************************************/
static SPORTS_ENUM eSubType (
    CD_OBJECT hCDO
    )
{
    CDO_TYPE_ENUM eType;
    SPORTS_OBJECT_STRUCT *psObj;
    SPORTS_ENUM eSubType = SPORTS_UNKNOWN;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_SPORTS)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            // Check if entry structure is valid
            if(psObj->psSport != NULL)
            {
                // Extract type
                eSubType = psObj->psSport->eSubType;
            }
        }
    }

    return eSubType;
}

/*****************************************************************************
*
*   hId
*
* Retrieves the CID associated with this CDO given that it is a
* SPORTS type. The returned CID is the uniuqe identifier for the
* sports this CDO describes.
*
*****************************************************************************/
static CID_OBJECT hId (
    CD_OBJECT hCDO
    )
{
    CDO_TYPE_ENUM eType;
    SPORTS_OBJECT_STRUCT *psObj;
    CID_OBJECT hCID = CID_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_SPORTS)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            // Check if entry structure is valid
            if(psObj->psSport != NULL)
            {
                // Extract CID
                hCID = psObj->hId;
            }
        }
    }

    return hCID;
}

/*****************************************************************************
*
*   hLeague
*
* Retrieves the LEAGUE_OBJECT handle which represents a known sports
* league or category this sports object represents.
*
*****************************************************************************/
static LEAGUE_OBJECT hLeague (
    CD_OBJECT hCDO
    )
{
    CDO_TYPE_ENUM eType;
    SPORTS_OBJECT_STRUCT *psObj;
    LEAGUE_OBJECT hLeague = LEAGUE_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_SPORTS)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            // Extract league
            hLeague = psObj->hLeague;
        }
    }

    return hLeague;
}

/*****************************************************************************
*
*   un8Teams
*
* Returns the number of teams (typically head-to-head) which are part
* of this SPORTS_CONTENT_OBJECT.
*
*****************************************************************************/
static UN8 un8Teams (
    CD_OBJECT hCDO
    )
{
    CDO_TYPE_ENUM eType;
    SPORTS_OBJECT_STRUCT *psObj;
    UN8 un8NumTeams = 0;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_SPORTS)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            un8NumTeams = un8TeamsLocal(psObj);
        }
    }

    return un8NumTeams;
}

/*****************************************************************************
*
*   hTeam
*
* Returns the handle to a TEAM_OBJECT which is part of this CDO. The team
* object handle provided to the caller is the one referenced by the provided
* index.
*
*****************************************************************************/
static TEAM_OBJECT hTeam (
    CD_OBJECT hCDO,
    UN8 un8Index
    )
{
    CDO_TYPE_ENUM eType;
    SPORTS_OBJECT_STRUCT *psObj;
    TEAM_OBJECT hTeam = TEAM_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_SPORTS)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            hTeam = hTeamLocal(psObj, un8Index);
        }
    }

    return hTeam;
}

/*****************************************************************************
*
*   un8Cars
*
* Returns the number of cars which are part of this SPORTS_CONTENT_OBJECT.
*
*****************************************************************************/
static UN8 un8Cars (
    CD_OBJECT hCDO
    )
{
    UN8 un8NumCars = 0;
    SPORTS_OBJECT_STRUCT *psObj;
    SPORTS_ENUM eThisSubType;

    // This API will only work for Auto-Racing
    eThisSubType = eSubType(hCDO);
    if(eThisSubType == SPORTS_AUTO_RACING)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            un8NumCars = un8TeamsLocal(psObj);
        }
    }

    return un8NumCars;
}

/*****************************************************************************
*
*   hCar
*
* Returns the handle to a CAR_OBJECT which is part of this CDO. The team
* object handle provided to the caller is the one referenced by the provided
* index.
*
*****************************************************************************/
static CAR_OBJECT hCar (
    CD_OBJECT hCDO,
    UN8 un8Index
    )
{
    CAR_OBJECT hCar = CAR_INVALID_OBJECT;
    SPORTS_OBJECT_STRUCT *psObj;
    SPORTS_ENUM eThisSubType;

    // This API will only work for Auto-Racing
    eThisSubType = eSubType(hCDO);
    if(eThisSubType == SPORTS_AUTO_RACING)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            hCar = (CAR_OBJECT)hTeamLocal(psObj, un8Index);
        }
    }

    return hCar;
}

/*****************************************************************************
*
*   eBroadcast
*
*****************************************************************************/
static SPORTS_BROADCAST_ENUM eBroadcast (
    CD_OBJECT hCDO,
    UN8 *pun8Index
    )
{
    CDO_TYPE_ENUM eType;
    SPORTS_OBJECT_STRUCT *psObj;
    SPORTS_BROADCAST_ENUM eBroadcastType = SPORTS_BROADCAST_UNKNOWN;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_SPORTS)
    {
        // Go get CDO structure from provided CDO
        psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            // Extract league
            eBroadcastType = psObj->eBCastType;

            if (eBroadcastType == SPORTS_BROADCAST_TEAM)
            {
                if (pun8Index != NULL)
                {
                    *pun8Index = psObj->un8BCastTeamIndex;
                }
            }
        }
    }

    return eBroadcastType;
}

/*****************************************************************************
*
*   bIterateContent
*
* Iterates the static SPORT content for all available sports known
* at the time this API is called. The caller provides a callback function
* which will be executed for each sport in our list. This list is static
* and available to any caller's context.
*
*****************************************************************************/
static BOOLEAN bIterateContent (
    SPORTS_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
    )
{
    BOOLEAN bRetval = FALSE, bLocked;
    SPORTS_ITERATOR_STRUCT sIterator;

    // Check inputs
    if(bContentIteratorCallback == NULL)
    {
        // Bad param
        return FALSE;
    }

    // Populate a local iterator structure with info needed to call
    // the provided callback to the caller along with their provided argument.
    sIterator.bContentIteratorCallback = bContentIteratorCallback;
    sIterator.pvContentIteratorCallbackArg = pvContentIteratorCallbackArg;

    // Iterate the list and execute caller's callback function for
    // each entry in the list.
    bLocked = SMSO_bLock((SMS_OBJECT)gpsSportsCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode =
            OSAL.eLinkedListIterate(
            gpsSportsCtrl->hSportsList,
            (OSAL_LL_ITERATOR_HANDLER)bIterator, &sIterator);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // List was iterated
            bRetval = TRUE;
        }

        SMSO_vUnlock((SMS_OBJECT)gpsSportsCtrl);
    }

    return bRetval;
}

/*****************************************************************************
FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SPORTS_bAdd
*
*   This function assigns a CID to represent this sport content. The CID
*   is used to uniquely describe this sport and map it to some
*   textual information about the sport (e.g. name and abbr).
*
*   Inputs
*       hId - A CID representing this SPORT
*       psSport - A structure defining a SPORT to associate with the CID
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the CID was able to be added.
*       Otherwise returns FALSE if the request resulted if an error occurred.
*
*****************************************************************************/
BOOLEAN SPORTS_bAdd (
    CID_OBJECT hId,
    const SPORT_STRUCT *psSport,
    BOOLEAN bCreate
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = FALSE, bLocked;
    SPORT_ENTRY_STRUCT const *psSportEntry;

    // Verify a CID and valid association was provided.
    // Without both these is no point in calling this.
    if((hId == CID_INVALID_OBJECT) || (psSport == NULL))
    {
        // Error!
        return FALSE;
    }

    // Now allocate a structure to hold the CID and sport as one mapping.
    // This entry is available for all SMS objects and applications to use.
    psSportEntry = psCreateSportEntry(hId, psSport, bCreate);
    if(psSportEntry == NULL)
    {
        // Error! Unable to create
        return FALSE;
    }

    // Finally, add this sport to our list.
    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsSportsCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        eReturnCode =
            OSAL.eLinkedListAdd(gpsSportsCtrl->hSportsList, 
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void *)psSportEntry);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // added successfully
            printf("Added sport (%s)\n", psSportEntry->psSport->pacName);
            bSuccess = TRUE;
        }
        else
        {
            // Something went wrong with adding this element. This could be 
            // because of list error or because we already found one in our 
            // list that looks just like this one. 
            vDestroySportEntry(psSportEntry);
        }

        SMSO_vUnlock((SMS_OBJECT)gpsSportsCtrl);
    }
    else
    {
        vDestroySportEntry(psSportEntry);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   SPORTS_bAssignId
*
*   This function assigns a CID to represent this sport content. The CID
*   is used to uniquely describe this sport and map it to some
*   textual information about the sport (e.g. name and abbr, etc).
*
*   Inputs
*       hCDO - The CDO in which to assign this CID.
*       hId - The CID representing this CDO's content.
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the CID was able to be assigned and
*       it caused a change in content. Otherwise returns FALSE if the
*       assignment resulted in no change or an error occurred preventing
*       the assignment, in which case it remained unchanged.
*
*****************************************************************************/
BOOLEAN SPORTS_bAssignId (
    CD_OBJECT hCDO,
    CID_OBJECT hId
    )
{
    BOOLEAN bChanged = FALSE;
    SPORTS_OBJECT_STRUCT *psObj;

    // Go get CDO structure from provided CDO
    psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
    if(psObj != NULL)
    {
        BOOLEAN bLocked;
        CID_OBJECT hOldId = psObj->hId;

        bLocked = SMSO_bLock((SMS_OBJECT)
            gpsSportsCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Find this CID in our SPORTs list (if we can)
            eReturnCode = OSAL.eLinkedListLinearSearch(
                gpsSportsCtrl->hSportsList,
                &hEntry,
                (OSAL_LL_COMPARE_HANDLER)n16IsSportEqualToId,
                hId
                );
            if(eReturnCode == OSAL_SUCCESS)
            {
                SPORT_ENTRY_STRUCT const *psSportEntry;

                // Found a match! Replace CDO info with it
                psSportEntry = ((SPORT_ENTRY_STRUCT const *)
                    OSAL.pvLinkedListThis(hEntry));
                psObj->psSport = psSportEntry->psSport;
                psObj->hId = psSportEntry->hId;
            }
            else if(eReturnCode == OSAL_OBJECT_NOT_FOUND)
            {
                // We couldn't find this CID in our list.
                // That's ok we'll just make a copy of the CID and
                // use it without a mapping (use defaults). Next time
                // we should be able to find it.

                // Use default sport mapping
                psObj->psSport = NULL;
                psObj->hLeague = LEAGUE_INVALID_OBJECT;
                psObj->hTeamList = OSAL_INVALID_OBJECT_HDL;

                // Make a copy of the CID which was provided to us
                psObj->hId = CID.hDuplicate(hId);
                if(psObj->hId != CID_INVALID_OBJECT)
                {
                    BOOLEAN bAdded;

                    // Add this CID with default mapping to our list
                    bAdded = SPORTS_bAdd(psObj->hId, &gsDefaultSport, TRUE);
                    if(bAdded == FALSE)
                    {
                        // Could not add new CID, so destroy the one
                        // we created.
                        CID.vDestroy(psObj->hId);
                        psObj->hId = CID_INVALID_OBJECT;
                    }
                }
            }

            SMSO_vUnlock((SMS_OBJECT)gpsSportsCtrl);
        }

        // Inform CME of the content change (sports)
        bChanged = CME_bChangeContent(hCDO, hOldId, hId);
    }

    return bChanged;
}

/*****************************************************************************
*
*   SPORTS_bAssignTeamId
*
*   This function assigns a CID to represent this sport's team associated
*   with this current sports broadcast. The CID
*   is used to uniquely describe this team and map it to some
*   textual information about the team (e.g. name and abbr, etc).
*
*   Inputs
*       hCDO - The CDO in which to assign this CID.
*       hId - The CID representing this CDO's content (a team).
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the CID was able to be assigned and
*       it caused a change in content. Otherwise returns FALSE if the
*       assignment resulted in no change or an error occurred preventing
*       the assignment, in which case it remained unchanged.
*
*****************************************************************************/
BOOLEAN SPORTS_bAssignTeamId (
    CD_OBJECT hCDO,
    CID_OBJECT hTeamId
    )
{
    BOOLEAN bChanged = FALSE;
    SPORTS_OBJECT_STRUCT *psObj;

    // Go get CDO structure from provided CDO
    psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
    if(psObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        LEAGUE_OBJECT hLeague;

        // Check if we do not have a team list
        if(psObj->hTeamList == OSAL_INVALID_OBJECT_HDL)
        {
            // We don't have a team list. So let's make one. Once made
            // this list sticks around as long as the CDO type remains
            // as sports.

            // Create list. Note that in this case the list generated has
            // no protect feature enabled since there is only one list
            // per CDO and that list is only accessed within a protected
            // context of that CDO. In other words the CDO access is already
            // guaranteed to be mutually exclusive.
            eReturnCode = OSAL.eLinkedListCreate(
                &psObj->hTeamList,
                SPORTS_CONTENT_OBJECT_NAME":TeamList",
                NULL,
                OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                );
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                return FALSE;
            }
        }

        // At this point, either because we already had one or we created
        // one, we have a team list. Now this list may already have teams
        // in it, or it may be empty.

        // To find the appropriate team, we need to use the league
        // this one belongs to.
        hLeague = SPORTS.hLeague(hCDO);
        if(hLeague != LEAGUE_INVALID_OBJECT)
        {
            TEAM_OBJECT hTeam;

            // Find a team which matches this incoming CID. If one 
            // cannot be found then it will get added. 
            // team list and returned.
            hTeam = LEAGUE_hFindTeam(hLeague, hTeamId, TRUE);
            if(hTeam != TEAM_INVALID_OBJECT)
            {
                // Attempt to add this team to the list, if it already
                // exists, then we don't need to do anything.
                eReturnCode = OSAL.eLinkedListAdd(
                    psObj->hTeamList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, 
                    (void *)hTeam);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    LEAGUE_TEAM_ID_STRUCT sLeagueTeamId;
                    CID_OBJECT hLeagueTeamId;

                    // Take league-id and team-id and make a CID
                    sLeagueTeamId.hLeagueId = LEAGUE.hId(hLeague);
                    sLeagueTeamId.hTeamId = hTeamId;

                    hLeagueTeamId = CID_hCreate(
                        CID_POOL_INVALID_OBJECT, 
                        CID_LEAGUE_TEAM_ID, &sLeagueTeamId);

                    // Inform CME of the content change, note if we
                    // successfully added this team then it did not
                    // exist previously. If we didn't then it must
                    // have already existed.
                    bChanged = CME_bChangeContent(
                        hCDO, CID_INVALID_OBJECT, hLeagueTeamId);

                    // Don't need CID
                    CID_vDestroy(hLeagueTeamId);
                }
            }
        }
    }

    return bChanged;
}

/*****************************************************************************
*
*   SPORTS_bAssignLeagueId
*
*   This function assigns a CID to represent this sport's league associated
*   with this current sports broadcast. The CID
*   is used to uniquely describe this league and map it to some
*   textual information about the league (e.g. name and abbr, etc).
*
*   Inputs
*       hCDO - The CDO in which to assign this CID.
*       hId - The CID representing this CDO's content (a league).
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the CID was able to be assigned and
*       it caused a change in content. Otherwise returns FALSE if the
*       assignment resulted in no change or an error occurred preventing
*       the assignment, in which case it remained unchanged.
*
*****************************************************************************/
BOOLEAN SPORTS_bAssignLeagueId (
    CD_OBJECT hCDO,
    CID_OBJECT hLeagueId,
    LEAGUE_ENUM eLeague
    )
{
    BOOLEAN bChanged = FALSE;
    SPORTS_OBJECT_STRUCT *psObj;

    // Go get CDO structure from provided CDO
    psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
    if(psObj != NULL)
    {
        CID_OBJECT hOldLeagueId;

        // Get the current(old) league-id
        hOldLeagueId = LEAGUE.hId(psObj->hLeague);

        if(hLeagueId == CID_INVALID_OBJECT)
        {
            // Invalidate league
            psObj->hLeague = LEAGUE_INVALID_OBJECT;
        }
        else
        {
            // Now try to find the new league based on new league-id
            psObj->hLeague = LEAGUE_hFind(hLeagueId, TRUE);
            if (psObj->hLeague == LEAGUE_INVALID_OBJECT)
            {
                // No league, so no id.
                hLeagueId = CID_INVALID_OBJECT;
            }
        }

        // Inform CME of the content change
        bChanged = CME_bChangeContent(hCDO, hOldLeagueId, hLeagueId);
        if (bChanged == TRUE)
        {
            // if the league changed, flush the teams
            SPORTS_bFlushTeams(hCDO);
        }
    }

    return bChanged;
}

/*****************************************************************************
*
*   SPORTS_bAssignBCastType
*
*****************************************************************************/
BOOLEAN SPORTS_bAssignBCastType (
    CD_OBJECT hCDO,
    SPORTS_BROADCAST_ENUM eBCastType
    )
{
    BOOLEAN bChanged = FALSE;
    SPORTS_OBJECT_STRUCT *psObj;

    // Go get CDO structure from provided CDO
    psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
    if(psObj != NULL)
    {
        psObj->eBCastType = eBCastType;

        bChanged = TRUE;
    }

    return bChanged;
}

/*****************************************************************************
*
*   SPORTS_bAssignBCastTeamIndex
*
*****************************************************************************/
BOOLEAN SPORTS_bAssignBCastTeamIndex (
    CD_OBJECT hCDO,
    UN8 un8TeamIndex
    )
{
    BOOLEAN bChanged = FALSE;
    SPORTS_OBJECT_STRUCT *psObj;

    // Go get CDO structure from provided CDO
    psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
    if(psObj != NULL)
    {
        psObj->un8BCastTeamIndex = un8TeamIndex;

        bChanged = TRUE;
    }

    return bChanged;
}

/*****************************************************************************
*
*   SPORTS_bFlushTeams
*
*****************************************************************************/
BOOLEAN SPORTS_bFlushTeams (
    CD_OBJECT hCDO
    )
{
    BOOLEAN bFlushedTeams = FALSE;
    SPORTS_OBJECT_STRUCT *psObj;

    // Go get CDO structure from provided CDO
    psObj = (SPORTS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
    if(psObj != NULL)
    {
        // Check if team list exists
        if(psObj->hTeamList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            UN32 un32Items = 0;

            // Check the size of the team list, if it is one or more
            // then it will be flushed (modified).
            eReturnCode = OSAL.eLinkedListItems(psObj->hTeamList, &un32Items);
            if((eReturnCode == OSAL_SUCCESS) && (un32Items > 0))
            {
                // Iterate list of teams which belong to this CDO. I need to
                // iterate the list since I cannot use the linked list
                // eLinkedListRemoveALL() API directly since I also need
                // it's CDO.
                OSAL.eLinkedListIterate(
                    psObj->hTeamList,
                    (OSAL_LL_ITERATOR_HANDLER)bTeamRemoveIterator,
                    hCDO
                    );

                // Flush existing team's which belong to this CDO's team
                // list (but don't delete team list itself).
                OSAL.eLinkedListRemoveAll(
                    psObj->hTeamList,
                    (OSAL_LL_RELEASE_HANDLER)vRemoveTeam
                    );

                // Indicate teams were flushed
                bFlushedTeams = TRUE;
            }
        }
    }

    return bFlushedTeams;
}

/*****************************************************************************
PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bInitialize
*
* Initialize the SPORTS CDO Object which creates a list to hold SPORTS
* and their CID mapping to sport information.
*
* This needs to be called once when SMS is started.
*
*****************************************************************************/
static BOOLEAN bInitialize ( void )
{
    BOOLEAN bRetval = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bExists = FALSE;
    SPORTS_CTRL_STRUCT *psTempSportsCtrl;

    // Initialize the known sports list.
    // Creates an empty sports list to be populated later.

    // Create an SMS object which can be locked to enforce exclusive
    // access to the elements within this structure.
    psTempSportsCtrl = (SPORTS_CTRL_STRUCT *)
        SMSO_hCreate(
        SPORTS_CONTENT_OBJECT_NAME":Ctrl",
        sizeof(SPORTS_CTRL_STRUCT),
        SMS_INVALID_OBJECT,
        TRUE // We want the lock feature
        );
    if(psTempSportsCtrl == NULL)
    {
        // Error! Something went wrong.
        return FALSE;
    }

    OSAL.eEnterTaskSafeSection();

    // Now, determine if the sports ctrl is already defined.
    // If so, do not attempt to re-define it. This structure is shared
    // amongst all DECODERS within SMS, so this only needs to be done
    // once when SMS is initialized.
    if(gpsSportsCtrl != NULL)
    {
        // Do nothing, except report an error since this should never be
        // done twice.
        bExists = TRUE;
    }
    else
    {
        // Assign global sports ctrl object
        gpsSportsCtrl = psTempSportsCtrl;
    }
    OSAL.eExitTaskSafeSection();

    // Check if we can continue
    if(bExists == TRUE)
    {
        // We're done
        SMSO_vDestroy((SMS_OBJECT)psTempSportsCtrl);
        return FALSE;
    }

    // Create the sports list of unique entries
    // At this point we have just created an empty list.
    eReturnCode =
        OSAL.eLinkedListCreate(
        &gpsSportsCtrl->hSportsList,
        SPORTS_CONTENT_OBJECT_NAME":List",
        (OSAL_LL_COMPARE_HANDLER)n16CompareSportNames,
        OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE |
        OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
        );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error! Something went wrong
        vUnInitialize();
        return FALSE;
    }

    // Initialize leagues.
    bRetval = LEAGUE_bInitialize((SMS_OBJECT)gpsSportsCtrl);
    if(bRetval == FALSE)
    {
        // League list cannot be created.
        vUnInitialize();
        return FALSE;
    }

    // Initialize teams.
    bRetval = TEAM_bInitialize((SMS_OBJECT)gpsSportsCtrl);
    if(bRetval == FALSE)
    {
        // Team list cannot be created.
        vUnInitialize();
        return FALSE;
    }

    // Relinquish control of the object
    SMSO_vUnlock((SMS_OBJECT)gpsSportsCtrl);

    return TRUE;
}

/*****************************************************************************
*
*   vUnInitialize
*
* Uninitialize the SPORTS CDO Object (upon SMS shutdown)
* Frees all resources associated with the sports list.
*
*****************************************************************************/
static void vUnInitialize ( void )
{
    BOOLEAN bLocked;

    // Lock and validate the object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsSportsCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        SPORTS_CTRL_STRUCT *psTempSportsCtrl = gpsSportsCtrl;

        // Decouple object
        gpsSportsCtrl = NULL;

        // Uninitialize the teams.
        TEAM_vUnInitialize();

        // Uninitialize the leagues.
        LEAGUE_vUnInitialize();

        // Destroy all sports map entries first
        eReturnCode =
            OSAL.eLinkedListRemoveAll(
            psTempSportsCtrl->hSportsList,
            (OSAL_LL_RELEASE_HANDLER)vDestroySportEntry
            );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy the list itself
            OSAL.eLinkedListDelete(psTempSportsCtrl->hSportsList);
            psTempSportsCtrl->hSportsList = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy the object
        SMSO_vDestroy((SMS_OBJECT)psTempSportsCtrl);
    }

    return;
}

/*****************************************************************************
*
*   vUnInit
*
* CDO Un-initialize method.
*
*****************************************************************************/
static void vUnInit (
    SPORTS_OBJECT_STRUCT *psObj
    )
{
    CD_OBJECT hCDO;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check input
    if(psObj == NULL)
    {
        return;
    }

    // Grab the CDO...
    hCDO = CDO_hCDO(psObj);

    // Inform CME of the content change. This sport is no
    // longer part of this CDO. In this case we only notify the
    // CME that the content has changed. We dont destroy anything since
    // these CIDs are constant (unlike other types).
    CME_bChangeContent(hCDO, psObj->hId, CID_INVALID_OBJECT);

    // Remove all teams from the list
    SPORTS_bFlushTeams(hCDO);

    SPORTS_bAssignLeagueId(hCDO, CID_INVALID_OBJECT, LEAGUE_UNKNOWN);

    // Delete the team list itself
    eReturnCode =
        OSAL.eLinkedListDelete(psObj->hTeamList);
    if(eReturnCode == OSAL_SUCCESS)
    {
        psObj->hTeamList = OSAL_INVALID_OBJECT_HDL;
    }

    // Default object now
    *psObj = gsDefaultSports;

    return;
}

/*****************************************************************************
*
*   n16Equal
*
*       This is the function used to compare TEAMS CDOs. Specifically
*       this function is used to perform a go, no-go comparison
*       or binary comparison of the two CDOs. CDOs are considered equal
*       if their CIDs match.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               -1  - CDOs are not equal
*               N16_MIN on error
*
*****************************************************************************/
static N16 n16Equal (
    const SPORTS_OBJECT_STRUCT *psObj1,
    const SPORTS_OBJECT_STRUCT *psObj2
    )
{
    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    return CID.n16Equal(psObj1->hId, psObj2->hId);
}

/*****************************************************************************
*
*   n16Compare
*
*       This is the function used to compare TEAMS CDOs. Specifically
*       this function is used to keep CDOs in order for
*       comparison/sorting and thus a relational compare is performed.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               > 0 - CDO1 is greater than (after) CDO2
*               < 0 - CDO1 is less than (before) CDO2
*               N16_MIN on error
*
*****************************************************************************/
static N16 n16Compare (
    const SPORTS_OBJECT_STRUCT *psObj1,
    const SPORTS_OBJECT_STRUCT *psObj2
    )
{
    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    return CID.n16Compare(psObj1->hId, psObj2->hId);
}

/*****************************************************************************
*
*   n32FPrintf
*
* This method is used by the caller to send formatted
* output of a CDO's contents to a specified file or device.
* This is mainly helpful during debugging of CDO's but could be used by
* a caller for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a CDO to a file for the
* purposes of later re-generating the CDO (for storage of the object).
* This API instead sends the CDO as a verbose formatted output version.
*
* Inputs:
*
*   psObj - The CDO structure the caller wishes to print.
*   psFile - The device to write the CDO contents to.
*
*****************************************************************************/
static N32 n32FPrintf (
    const SPORTS_OBJECT_STRUCT *psObj,
    FILE *psFile
    )
{
    N32 n32Return = 0;

#if SMS_DEBUG == 1
    N32 n32Temp   = 0;

    // Check inputs.
    if((psObj == NULL) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Print CDO information...

    if(psObj->psSport != NULL)
    {
        n32Return += fprintf(psFile,
            "eSubType: %s\n",
            SPORTS_pacEnumText(psObj->psSport->eSubType)
            );
        n32Return += fprintf(psFile, "hId[0x%p]:\n", psObj->hId);
        n32Temp = CID.n32FPrintf(psObj->hId, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }

        if (psObj->eBCastType == SPORTS_BROADCAST_TEAM)
        {
            n32Return += fprintf(psFile,
                "eBroadcastType: %s Team Index: %u\n",
                SPORTS_pacBCastEnumText(psObj->eBCastType),
                psObj->un8BCastTeamIndex
                );
        }
        else
        {
            n32Return += fprintf(psFile,
                "eBroadcastType: %s\n",
                SPORTS_pacBCastEnumText(psObj->eBCastType)
                );
        }

        // Check if a member of a known league
        n32Return += fprintf(psFile, "hLeague[0x%p]:\n",
            psObj->hLeague
            );
        if(psObj->hLeague != LEAGUE_INVALID_OBJECT)
        {
            // Print league info
            STRING_OBJECT hString;

            n32Return += fprintf(psFile, "\n\t");
            hString = LEAGUE.hName(psObj->hLeague);
            n32Temp = STRING.n32FWrite(hString, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
            hString = LEAGUE.hAbbreviation(psObj->hLeague);
            n32Return += fprintf(psFile, " (");
            n32Temp = STRING.n32FWrite(hString, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
            n32Return += fprintf(psFile, ")\n\n");
        }
    }
    else
    {
        n32Return += fprintf(psFile, "No Sports Type info!\n");
    }

    // Print team information

    // Is there a team list?
    n32Return += fprintf(psFile, "hTeamList[0x%p]: ", psObj->hTeamList);
    if(psObj->hTeamList != OSAL_INVALID_OBJECT_HDL)
    {
        UN32 un32NumTeams = 0;
        SPORTS_TEAM_ITERATOR_STRUCT sTeamIterator;

        // Print number of teams to expect
        OSAL.eLinkedListItems(psObj->hTeamList, &un32NumTeams);
        n32Return += fprintf(psFile, "%u teams\n", un32NumTeams);

        // Iterate team list, printing each team's info

        // Configure iterator structure
        sTeamIterator.un8CurrentIndex = 0;
        sTeamIterator.n32Return = 0;
        sTeamIterator.psFile = psFile;

        OSAL.eLinkedListIterate(
            psObj->hTeamList,
            (OSAL_LL_ITERATOR_HANDLER)bPrintTeamIterator,
            &sTeamIterator
            );

        // Update return value from iteration(s)
        n32Return += sTeamIterator.n32Return;
    }
    else
    {
        n32Return += fprintf(psFile, "No teams.\n");
    }
#endif /* SMS_DEBUG == 1 */

    return n32Return;
}

/*****************************************************************************
*
*   bHasId
*
* This method is used to examine a CDO sub-type for a specific Content-ID (CID)
* provided by the caller. If this sub-type has this id, then TRUE is returned
* otherwise, FALSE is returned if it does not contain it. Keep in mind
* this functions checks the entire sub-type for any matches.
*
* Inputs:
*   psObj - A pointer to an object for which to examine and determine if
*       the provided CID is contained within it.
*   hId - A valid CID to look for within the object.
*
* Returns:
*   TRUE of the CID was found anywhere within the object, otherwise FALSE
*   is returned if it could not be found.
*
*****************************************************************************/
static BOOLEAN bHasId (
    const SPORTS_OBJECT_STRUCT *psObj,
    CID_OBJECT hId
    )
{
    BOOLEAN bHasId = FALSE;

    // We can look to see if this CID matches one we have in this object.
    // Any hit is enough to cause us to fall out and return TRUE.
    do
    {
        UN16 n16Equal;
        CID_OBJECT hLeagueId;
        SPORTS_TEAM_CID_ITERATOR_STRUCT sTeamIterator;

        // hId == hId? (sports id)
        n16Equal = CID.n16Equal(hId, psObj->hId);
        if(n16Equal == 0)
        {
            // Equal, so they match
            bHasId = TRUE;
            break;
        }

        hLeagueId = LEAGUE.hId(psObj->hLeague);
        n16Equal = CID.n16Equal(hId, hLeagueId);
        if(n16Equal == 0)
        {
            // Equal, so they match
            bHasId = TRUE;
            break;
        }

        // Set-up team iterator
        sTeamIterator.pbHasId = &bHasId;
        sTeamIterator.hId = hId;
        sTeamIterator.hLeague = psObj->hLeague;

        // Iterate list of teams which belong to this CDO
        OSAL.eLinkedListIterate(
            psObj->hTeamList,
            (OSAL_LL_ITERATOR_HANDLER)bTeamIdIterator,
            &sTeamIterator
            );

    } while(FALSE);

    return bHasId;
}

/*****************************************************************************
*
*   vDestroySportEntry
*
* Destroy a sport entry
*
*****************************************************************************/
static void vDestroySportEntry ( SPORT_ENTRY_STRUCT const *psSportEntry )
{
    // Check input
    if(psSportEntry == NULL)
    {
        return;
    }

    // Check if a CID exists
    if(psSportEntry->hId != CID_INVALID_OBJECT)
    {
        // Destroy CID
        CID_vDestroy(psSportEntry->hId);
    }

    // Destroy object
    OSAL.vLinkedListMemoryFree((void *)psSportEntry);

    return;
}

/*****************************************************************************
*
*   vRemoveTeam
*
* Remove a team entry (from team list)
*
*****************************************************************************/
static void vRemoveTeam ( TEAM_OBJECT hTeam )
{
    // Nothing to do.
    return;
}

/*****************************************************************************
*
*   n16CompareSportNames
*
*       This is the function used to compare existing SPORTS objects
*       to each other. Specifically this function is used to perform a
*       relational comparison for sorting. This sorter keeps the SPORTs
*       in order based on each SPORTS's Name.
*
*       Outputs:
*               0   - SPORTs have the same value (equal)
*               > 0 - SPORT1 is greater than (after) SPORT2
*               < 0 - SPORT1 is less than (before) SPORT2
*               N16_MIN on error
*
*****************************************************************************/
static N16 n16CompareSportNames (
    SPORT_ENTRY_STRUCT const *psSportEntry1,
    SPORT_ENTRY_STRUCT const *psSportEntry2
    )
{
    N16 n16Result;
    int iStrCmpResult;

    // Verify inputs
    if( (psSportEntry1 == NULL) || (psSportEntry2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    // Verify inputs cont'd...
    if ( (psSportEntry1->psSport->pacName == NULL) ||
        (psSportEntry2->psSport->pacName == NULL) )
    {
        // Error
        return N16_MIN;
    }

    // Determine relationship
    iStrCmpResult = strcmp(psSportEntry1->psSport->pacName,
        psSportEntry2->psSport->pacName);

    if (iStrCmpResult < 0)
    {
        n16Result = -1;
    }
    else if (iStrCmpResult > 0)
    {
        n16Result = 1;
    }
    else
    {
        // Check further for CID relationship
        n16Result = CID.n16Compare(psSportEntry1->hId,
            psSportEntry2->hId);
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16IsSportEqualToId
*
*       This is the function used to compare an existing SPORTS CID
*       to a provided CID. Specifically this function is used to perform a go,
*       no-go comparison or binary comparison for searching or looking up
*       a sport given a CID.
*
*       Outputs:
*               0   - Sport's CID and provided CID have the same value (equal)
*               < 0 - Sport's CID and provided CID are not equal
*               N16_MIN on error
*
*****************************************************************************/
static N16 n16IsSportEqualToId (
    SPORT_ENTRY_STRUCT const *psSportEntry,
    CID_OBJECT hCID
    )
{
    // Verify inputs
    if( (psSportEntry == NULL) || (hCID == CID_INVALID_OBJECT) )
    {
        // Error
        return N16_MIN;
    }

    return CID.n16Equal(psSportEntry->hId, hCID);
}

/*****************************************************************************
*
*   bTeamIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* team iterator structure to call the caller's own iterator function along
* with an argument for each team in the list.
*
* In this case it is used to iterate the available teams in a CDO
* and stop when the requested team has been reached.
*
*****************************************************************************/
static BOOLEAN bTeamIterator (
    TEAM_OBJECT hTeam,
    SPORTS_TEAM_ITERATOR_STRUCT *psTeamIterator
    )
{
    // Check inputs
    if((hTeam == TEAM_INVALID_OBJECT) || (psTeamIterator == NULL))
    {
        // Cease iterating
        return FALSE;
    }

    // Check if we have iterated to the index we were looking for
    if(psTeamIterator->un8CurrentIndex == psTeamIterator->un8StopIndex)
    {
        psTeamIterator->hTeam = hTeam;
        return FALSE; // Stop iterating
    }

    psTeamIterator->un8CurrentIndex++;

    return TRUE;
}

/*****************************************************************************
*
*   bTeamRemoveIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* team iterator structure to call the caller's own iterator function along
* with an argument for each team in the list.
*
* In this case it is used to iterate the available teams in a CDO
* and remove the CID which causes the CME to be updated.
*
*****************************************************************************/
static BOOLEAN bTeamRemoveIterator (
    TEAM_OBJECT hTeam,
    CD_OBJECT hCDO
    )
{
    CID_OBJECT hTeamId;
    LEAGUE_OBJECT hLeague;

    hLeague = SPORTS.hLeague(hCDO);
    hTeamId = LEAGUE.hTeamId(hLeague, hTeam);

    // Inform CME of the content change. This team is no
    // longer part of this CDO.
    CME_bChangeContent(hCDO, hTeamId, CID_INVALID_OBJECT);

    return TRUE;
}

#if SMS_DEBUG == 1
/*****************************************************************************
*
*   bPrintTeamIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* team iterator structure to call the caller's own iterator function along
* with an argument for each team in the list.
*
* In this case this iterator is used with the n32FPrintf() method to
* print each of the teams which belong to a TEAM object.
*
*****************************************************************************/
static BOOLEAN bPrintTeamIterator (
    TEAM_OBJECT hTeam,
    SPORTS_TEAM_ITERATOR_STRUCT *psTeamIterator
    )
{
    N32 n32Temp   = 0;

    // Check inputs
    if((hTeam == TEAM_INVALID_OBJECT) || (psTeamIterator == NULL))
    {
        // Cease iterating
        return FALSE;
    }

    psTeamIterator->n32Return += fprintf(
        psTeamIterator->psFile, "\nhTeam[%u]: 0x%p\n",
        psTeamIterator->un8CurrentIndex++, hTeam);

    if(hTeam != TEAM_INVALID_OBJECT)
    {
        STRING_OBJECT hString;

        psTeamIterator->n32Return +=
            fprintf(psTeamIterator->psFile, "\n\t");
        hString = TEAM.hName(hTeam);
        n32Temp = STRING.n32FWrite(hString, psTeamIterator->psFile);
        if (n32Temp > 0)
        {
            psTeamIterator->n32Return += n32Temp;
        }
        psTeamIterator->n32Return +=
            fprintf(psTeamIterator->psFile, " ");
        hString = TEAM.hNickname(hTeam);
        n32Temp = STRING.n32FWrite(hString, psTeamIterator->psFile);
        if (n32Temp > 0)
        {
            psTeamIterator->n32Return += n32Temp;
        }
        psTeamIterator->n32Return += fprintf(psTeamIterator->psFile, " (");
        hString = TEAM.hAbbreviation(hTeam);
        n32Temp = STRING.n32FWrite(hString, psTeamIterator->psFile);
        if (n32Temp > 0)
        {
            psTeamIterator->n32Return += n32Temp;
        }
        psTeamIterator->n32Return += fprintf(psTeamIterator->psFile, ")\n\n");
        n32Temp = CID.n32FPrintf(TEAM.hId(hTeam), psTeamIterator->psFile);
        if (n32Temp > 0)
        {
            psTeamIterator->n32Return += n32Temp;
        }
    }

    return TRUE;
}
#endif /* SMS_DEBUG == 1 */

/*****************************************************************************
*
*   bIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* sport iterator structure to call the caller's own iterator function along
* with an argument for each sport in the list.
*
*****************************************************************************/
static BOOLEAN bIterator (
    SPORT_ENTRY_STRUCT const *psSportEntry,
    SPORTS_ITERATOR_STRUCT *psIterator
    )
{
    BOOLEAN bReturn = FALSE;

    // Check inputs
    if((psSportEntry == NULL) || (psIterator == NULL))
    {
        // Cease iterating
        return FALSE;
    }

    // Call provided iterator function if one exists
    if(psIterator->bContentIteratorCallback != NULL)
    {
        // Don't include unknown (discovered) sports
        if(psSportEntry->psSport->eSubType != SPORTS_UNKNOWN)
        {
            // Return whatever their callback returns back to the iterator
            bReturn = psIterator->bContentIteratorCallback(
                psSportEntry->psSport->pacAbbreviation,
                psSportEntry->psSport->pacName,
                psSportEntry->psSport->eSubType,
                psSportEntry->hId,
                psIterator->pvContentIteratorCallbackArg
                );
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bTeamIdIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* team id iterator structure to iterate the team list and check if an id
* matches.
*
*****************************************************************************/
static BOOLEAN bTeamIdIterator (
    TEAM_OBJECT hTeam,
    SPORTS_TEAM_CID_ITERATOR_STRUCT *psTeamIterator
    )
{
    UN16 n16Equal;
    CID_OBJECT hTeamId;

    // Extract team id
    hTeamId = LEAGUE.hTeamId(psTeamIterator->hLeague, hTeam);

    // Does the team match?
    n16Equal = CID.n16Equal(psTeamIterator->hId, hTeamId);
    if(n16Equal == 0)
    {
        // Equal, so they match
        *(psTeamIterator->pbHasId) = TRUE;
        return FALSE; // cease iterating
    }

    return TRUE; // keep iterating
}

/*****************************************************************************
*
*   hTeamLocal
*
* Returns the number of teams (of the requested Team Subtypes) which are part
* of this SPORTS_CONTENT_OBJECT.
*
*****************************************************************************/
static UN8 un8TeamsLocal (
    SPORTS_OBJECT_STRUCT *psObj
    )
{
    UN8 un8NumTeams = 0;

    if(psObj != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32NumTeams = 0;

        // Get number of teams
        eReturnCode = OSAL.eLinkedListItems(
            psObj->hTeamList, &un32NumTeams);
        if(eReturnCode == OSAL_SUCCESS)
        {
            un8NumTeams = (UN8)un32NumTeams;
        }
    }

    return un8NumTeams;
}

/*****************************************************************************
*
*   hTeamLocal
*
* Returns the handle to a TEAM_OBJECT (of the requested Team Subtypes) which
* is art of this CDO. The team object handle provided to the caller is the
* one referenced by the provided index.
*
*****************************************************************************/
static TEAM_OBJECT hTeamLocal (
    SPORTS_OBJECT_STRUCT *psObj,
    UN8 un8Index
    )
{
    TEAM_OBJECT hTeam = TEAM_INVALID_OBJECT;

    if(psObj != NULL)
    {
        UN8 un8NumTeams = 0;

        // Get number of teams
        un8NumTeams = un8TeamsLocal(psObj);

        // Verify object has teams
        if(un8NumTeams > 0)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Verify input index
            if((un8Index < un8NumTeams) && (un8NumTeams != 0))
            {
                SPORTS_TEAM_ITERATOR_STRUCT sTeamIterator;

                // Configure iterator structure
                sTeamIterator.hTeam = TEAM_INVALID_OBJECT;
                sTeamIterator.un8CurrentIndex = 0;
                sTeamIterator.un8StopIndex = un8Index;

                // Iterate list of teams which belong to this CDO
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->hTeamList,
                    (OSAL_LL_ITERATOR_HANDLER)bTeamIterator,
                    &sTeamIterator
                    );
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Extract team (if one found)
                    hTeam = sTeamIterator.hTeam;
                }
            }
        }
    }

    return hTeam;
}

/*****************************************************************************
*
*   psCreateSportEntry
*
*   This function creates a sport entry for our linked list of sports.
*
*****************************************************************************/
static SPORT_ENTRY_STRUCT const *psCreateSportEntry (
    CID_OBJECT hId,
    const SPORT_STRUCT *psSport,
    BOOLEAN bCopySport
    )
{
    size_t tSize = sizeof(SPORT_ENTRY_STRUCT), tAbbrevSize = 0, tNameSize = 0;
    SPORT_ENTRY_STRUCT *psSportEntry = NULL;

    // Determine if we also need to allocate memory
    if(bCopySport == TRUE)
    {
        // Yes we need to add some to our memory requirements
        tSize += sizeof(SPORT_STRUCT);

        // Name
        if(psSport->pacName != NULL)
        {
            tNameSize += strlen(psSport->pacName);
        }
        else
        {
            tNameSize += strlen(gsDefaultSport.pacName);
        }

        // Add null terminator for name
        tNameSize++;

        // Name
        if(psSport->pacAbbreviation != NULL)
        {
            tAbbrevSize += strlen(psSport->pacAbbreviation);
        }
        else
        {
            tAbbrevSize += strlen(gsDefaultSport.pacAbbreviation);
        }

        // Add null terminator for name
        tAbbrevSize++;

        // Compute required size
        tSize += tAbbrevSize + tNameSize;
    }

    // Now allocate a structure to hold the CID and sport as one mapping.
    // This entry is available for all SMS objects and applications to use.
    psSportEntry = (SPORT_ENTRY_STRUCT *)
        OSAL.pvLinkedListMemoryAllocate(
        SPORTS_CONTENT_OBJECT_NAME":Obj",
        tSize,
        FALSE
        );
    if(psSportEntry == NULL)
    {
        // Error! Unable to create
        return NULL;
    }

    // Populate new SPORT object with provided inputs.
    psSportEntry->psSport = &gsDefaultSport;

    // Check if we need to re-populate with a copy of the provided sport
    if(bCopySport == TRUE)
    {
        SPORT_STRUCT *psLocalSport =
            (SPORT_STRUCT *)(psSportEntry + 1);
        char *pacName, *pacAbbreviation;

        // Assign and populate sport

        // ID
        psLocalSport->eSubType = psSport->eSubType;

        psLocalSport->pacAbbreviation = pacAbbreviation = (char *)(psLocalSport + 1);
        if(psSport->pacAbbreviation == NULL)
        {
            // Use default
            strncpy(pacAbbreviation, gsDefaultSport.pacAbbreviation, tAbbrevSize);
        }
        else
        {
            // Use provided
            strncpy(pacAbbreviation, psSport->pacAbbreviation, tAbbrevSize);
        }

        // Name
        psLocalSport->pacName = pacName = pacAbbreviation + tAbbrevSize;
        if(psSport->pacName == NULL)
        {
            // Use default
            strncpy(pacName, gsDefaultSport.pacName, tNameSize);
        }
        else
        {
            // Use provided
            strncpy(pacName, psSport->pacName, tNameSize);
        }

        // Now transfer assigned pointers
        psSportEntry->psSport = psLocalSport;
    }
    else
    {
        psSportEntry->psSport = psSport;
    }

    psSportEntry->hId = hId;

    return psSportEntry;
}
