/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SPORTS_FLASH implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "sms_event.h"
#include "decoder_obj.h"
#include "league_obj.h"
#include "team_obj.h"
#include "module_obj.h"
#include "module_version_obj.h"
#include "cm.h"
#include "string_obj.h"

#include "sports_flash_obj.h"
#include "_sports_flash_obj.h"

#include "sms_api_debug.h"

static const char *gpacThisFile = __FILE__;
/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eStart
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStart (
    DECODER_OBJECT hDecoder,
    UN32 *pun32MaxSportsFlash,
    SPORTS_FLASH_GAME_EVENT_CALLBACK vGameEventCallback,
    void *pvGameEventCallbackArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    do
    {
        MODULE_OBJECT hModule;
        BOOLEAN bSuccess;
        UN32 un32ProtocolVer;
        SPORTS_FLASH_OBJECT hSportsFlash;
        UN8 un8MaxSportsFlash;
        SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;

        if (vGameEventCallback == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Get the module this decoder is attached to
        hModule = DECODER_hModule(hDecoder);

        // Determine MaxSportsFlash value (in case if eStart has been
        // used to get the max sports flash value only, we need to do it
        // before anything)
        bSuccess = MODULE_VERSION_bMaxSportsFlash(hModule, 
            &un8MaxSportsFlash);
        if (bSuccess == FALSE)
        {
            puts(SPORTS_FLASH_OBJECT_NAME
                ": MaxSportsFlash value is not available");
            break;
        }

        if (pun32MaxSportsFlash != NULL)
        {
            *pun32MaxSportsFlash = un8MaxSportsFlash;
        }

        if (un8MaxSportsFlash == 0)
        {
            // nothing to do if no channels could be monitored
            break;
        }

        // Get Sports Flash service handle from DECODER
        hSportsFlash = DECODER_hSportsFlash(hDecoder);
        if (hSportsFlash != SPORTS_FLASH_INVALID_OBJECT)
        {
            // Service is already started
            eReturnCode = SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
            break;
        }

        // Verify IR is supported
        bSuccess = MODULE_bIsIRSupported(hModule);
        if (bSuccess == FALSE)
        {
            puts(SPORTS_FLASH_OBJECT_NAME
                ": IR is not supported by the Module");
            break;
        }

        // Verify SXI3.0 features are supported
        bSuccess = MODULE_VERSION_bGetProtocolVersion(hModule, 
            &un32ProtocolVer);
        if (bSuccess == FALSE)
        {
            puts(SPORTS_FLASH_OBJECT_NAME
                ": Module version info is not available");
            break;
        }

        if (un32ProtocolVer < SPORTS_FLASH_MIN_SXI_VERSION_REQUIRED)
        {
            puts(SPORTS_FLASH_OBJECT_NAME
                ": This SXI version doesn't support Sports Flash");
            break;
        }

        // Service is not started yet
        eReturnCode = eCreateSportsFlash(
            hDecoder,
            un8MaxSportsFlash,
            vGameEventCallback,
            pvGameEventCallbackArg
                );

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
        bLocked = FALSE;

        sEvent.eAction = SPORTS_FLASH_ACTION_LOAD_CONFIG_EVENT;
        bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
        if (bSuccess == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vStop
*
*****************************************************************************/
static void vStop (
    DECODER_OBJECT hDecoder
        )
{
    SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;

    sEvent.eAction = SPORTS_FLASH_ACTION_STOP_EVENT;
    DECODER_bSportsFlash(hDecoder, &sEvent);

    return;
}

/*****************************************************************************
*
*   eAddFavoriteTeam
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddFavoriteTeam (
    DECODER_OBJECT hDecoder,
    CID_OBJECT hLeagueId,
    CID_OBJECT hTeamId
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    do 
    {
        LEAGUE_OBJECT hLeague;
        TEAM_OBJECT hTeam;
        SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;
        BOOLEAN bEligible, bSuccess;
        CID_ENUM eCIDType;

        eCIDType = CID_eType(hLeagueId);
        if (eCIDType != CID_SXI_SPORTS_ID)
        {
            // invalid type
            break;
        }

        hLeague = LEAGUE_hFind(hLeagueId, FALSE);
        if (hLeague == LEAGUE_INVALID_OBJECT)
        {
            // Unknown League Id
            break;
        }

        eCIDType = CID_eType(hTeamId);
        if (eCIDType != CID_SXI_SPORTS_ID)
        {
            // invalid type
            break;
        }

        hTeam = LEAGUE_hFindTeam(hLeague, hTeamId, FALSE);
        if (hTeam == TEAM_INVALID_OBJECT)
        {
            // Team does not belong to the League
            break;
        }

        bEligible = TEAM.bIsSportsFlashEligible(hLeague, hTeam);
        if (bEligible == FALSE)
        {
            // The team is not eligible for Sports Flash
            break;
        }

        sEvent.uEvent.sTeam.un16TeamID = TEAM.un16Id(hTeam);
        sEvent.uEvent.sTeam.un8LeagueID = LEAGUE.un8Id(hLeague);

        sEvent.eAction = SPORTS_FLASH_ACTION_ADD_TEAM;
        bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
        if (bSuccess != FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eRemoveFavoriteTeam
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveFavoriteTeam (
    DECODER_OBJECT hDecoder,
    CID_OBJECT hLeagueId,
    CID_OBJECT hTeamId
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    do 
    {
        SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;
        BOOLEAN bSuccess;
        UN32 un32Id, *pun32Id = &un32Id;
        N32 n32Result;
        CID_ENUM eCIDType;

        eCIDType = CID_eType(hLeagueId);
        if (eCIDType != CID_SXI_SPORTS_ID)
        {
            // invalid type
            break;
        }

        n32Result = CID_n32GetValue(hLeagueId, (void **)&pun32Id);
        if (n32Result <= 0)
        {
            break;
        }
        sEvent.uEvent.sTeam.un8LeagueID = (UN8)un32Id;

        eCIDType = CID_eType(hTeamId);
        if (eCIDType != CID_SXI_SPORTS_ID)
        {
            // invalid type
            break;
        }

        n32Result = CID_n32GetValue(hTeamId, (void **)&pun32Id);
        if (n32Result <= 0)
        {
            break;
        }
        sEvent.uEvent.sTeam.un16TeamID = (UN16)un32Id;

        sEvent.eAction = SPORTS_FLASH_ACTION_REMOVE_TEAM;
        bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
        if (bSuccess != FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eRemoveAllFavoriteTeams
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveAllFavoriteTeams (
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;
    BOOLEAN bSuccess;

    sEvent.eAction = SPORTS_FLASH_ACTION_REMOVE_ALL_TEAMS;
    bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
    if (bSuccess != FALSE)
    {
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eIterateFavoriteTeams
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateFavoriteTeams (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_TEAMS_ITERATOR_CALLBACK bTeamsIteratorCallback,
    void *pvTeamsIteratorCallbackArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (bTeamsIteratorCallback == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        SPORTS_FLASH_OBJECT hSportsFlash;

        hSportsFlash = DECODER_hSportsFlash(hDecoder);

        if (hSportsFlash != SPORTS_FLASH_INVALID_OBJECT)
        {
            SPORTS_FLASH_OBJECT_STRUCT *psObj =
                (SPORTS_FLASH_OBJECT_STRUCT *)hSportsFlash;
            SPORTS_FLASH_TEAMS_ITERATOR_STRUCT sTeamsIterator;

            sTeamsIterator.bIteratorCallback = bTeamsIteratorCallback;
            sTeamsIterator.pvTeamsIteratorCallbackArg = 
                pvTeamsIteratorCallbackArg;

            eOsalReturnCode =
                OSAL.eLinkedListIterate(
                    psObj->hTeamsList,
                    (OSAL_LL_ITERATOR_HANDLER)bTeamsIterator,
                    (void*)&sTeamsIterator);
            if ((eOsalReturnCode == OSAL_SUCCESS) ||
                (eOsalReturnCode == OSAL_NO_OBJECTS))
            {
                // List was iterated
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to iterate the list "
                    "(%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
            }
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED;
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eIterateCurrentGames
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateCurrentGames (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_GAMES_ITERATOR_CALLBACK bGamesIteratorCallback,
    void *pvGamesIteratorCallbackArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (bGamesIteratorCallback == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        SPORTS_FLASH_OBJECT hSportsFlash;

        hSportsFlash = DECODER_hSportsFlash(hDecoder);

        if (hSportsFlash != SPORTS_FLASH_INVALID_OBJECT)
        {
            SPORTS_FLASH_OBJECT_STRUCT *psObj =
                (SPORTS_FLASH_OBJECT_STRUCT *)hSportsFlash;

            eOsalReturnCode =
                OSAL.eLinkedListIterate(
                    psObj->hGamesList,
                    (OSAL_LL_ITERATOR_HANDLER)bGamesIteratorCallback,
                    pvGamesIteratorCallbackArg
                        );
            if ((eOsalReturnCode == OSAL_SUCCESS) ||
                (eOsalReturnCode == OSAL_NO_OBJECTS))
            {
                // List was iterated
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to iterate the list "
                    "(%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
            }
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED;
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eSetGamesMonitor
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetGamesMonitor (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID *ptChannelIDList,
    UN16 un16Count,
    SPORTS_FLASH_FLASH_EVENT_CALLBACK vFlashEventCallback,
    void *pvFlashEventCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        BOOLEAN bSuccess;
        SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;

        if (((vFlashEventCallback == NULL) || (ptChannelIDList == NULL)) &&
            (un16Count != 0))
        {
            // error, we need the channels
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        sEvent.eAction = SPORTS_FLASH_ACTION_SET_GAMES_MONITOR;
        sEvent.uEvent.sSetGamesMonitor.un16Count = 0;
        sEvent.uEvent.sSetGamesMonitor.ptChannelIDList = NULL;

        if ((ptChannelIDList != NULL) && (un16Count != 0))
        {
            sEvent.uEvent.sSetGamesMonitor.ptChannelIDList =
                (CHANNEL_ID *)OSAL.pvMemoryAllocate(
                    SPORTS_FLASH_OBJECT_NAME":ChannelIDList",
                    sizeof(CHANNEL_ID) * un16Count, TRUE);
            if (sEvent.uEvent.sSetGamesMonitor.ptChannelIDList == NULL)
            {
                // Out of memory. Very bad...
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": No memory for ChannelIDs "
                    "array");
                break;
            }

            OSAL.bMemCpy(sEvent.uEvent.sSetGamesMonitor.ptChannelIDList, 
                ptChannelIDList, sizeof(CHANNEL_ID) * un16Count);

            sEvent.uEvent.sSetGamesMonitor.un16Count = un16Count;
            sEvent.uEvent.sSetGamesMonitor.vEventCallback = 
                vFlashEventCallback;
            sEvent.uEvent.sSetGamesMonitor.pvArg = pvFlashEventCallbackArg;
        }

        // Post the Event
        bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
        if (bSuccess != FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            // looks like we need to cleanup the created event struct
            vCleanupEvent(&sEvent);
        }
    }
    while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eIterateCurrentFlashEvents
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateCurrentFlashEvents (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENTS_ITERATOR_CALLBACK bEventsIteratorCallback,
    void *pvEventsIteratorCallbackArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (bEventsIteratorCallback == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        SPORTS_FLASH_OBJECT_STRUCT *psObj;

        psObj = (SPORTS_FLASH_OBJECT_STRUCT *)
            DECODER_hSportsFlash(hDecoder);

        if (psObj != NULL)
        {
            SPORTS_FLASH_EVENTS_ITERATOR_STRUCT sFlashEventsIterator;

            sFlashEventsIterator.bIteratorCallback = 
                bEventsIteratorCallback;
            sFlashEventsIterator.pvEventsIteratorCallbackArg = 
                pvEventsIteratorCallbackArg;

            eOsalReturnCode =
                OSAL.eLinkedListIterate(
                    psObj->hEventsList,
                    (OSAL_LL_ITERATOR_HANDLER)bFlashEventsIterator,
                    (void*)&sFlashEventsIterator
                        );
            if ((eOsalReturnCode == OSAL_SUCCESS) ||
                (eOsalReturnCode == OSAL_NO_OBJECTS))
            {
                // List was iterated
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to iterate the list "
                    "(%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
            }
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED;
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   ePlayFlashEvent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePlayFlashEvent (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tFlashEventID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bSuccess;
    SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;

    if (tFlashEventID == SPORTS_FLASH_INVALID_EVENT_ID)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    sEvent.eAction = SPORTS_FLASH_ACTION_PLAY_FLASH_EVENT;
    sEvent.uEvent.sFlashAction.tFlashEventID = tFlashEventID;

    bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
    if (bSuccess != FALSE)
    {
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eAbortFlashEvent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAbortFlashEvent (
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bSuccess;
    SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;

    sEvent.eAction = SPORTS_FLASH_ACTION_ABORT_FLASH_EVENT;
    bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
    if (bSuccess != FALSE)
    {
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   tCurrentFlashEventID
*
*****************************************************************************/
static SPORTS_FLASH_EVENT_ID tCurrentFlashEventID (
    DECODER_OBJECT hDecoder
        )
{
    SPORTS_FLASH_EVENT_ID tFlashEventId = SPORTS_FLASH_INVALID_EVENT_ID;
    SPORTS_FLASH_OBJECT hSportsFlash;

    hSportsFlash = DECODER_hSportsFlash(hDecoder);
    if (hSportsFlash != SPORTS_FLASH_INVALID_OBJECT)
    {
        SPORTS_FLASH_OBJECT_STRUCT *psObj =
            (SPORTS_FLASH_OBJECT_STRUCT *)hSportsFlash;

        tFlashEventId = psObj->tCurrentFlashEventID;
    }

    return tFlashEventId;
}

/*****************************************************************************
*
*   eEnableNotification
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnableNotification (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bSuccess;
    SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;

    sEvent.eAction = SPORTS_FLASH_ACTION_ENABLE_NOTIFICATION;
    sEvent.uEvent.sEnableNotification.bEnable = bEnable;

    bSuccess = DECODER_bSportsFlash(hDecoder, &sEvent);
    if (bSuccess != FALSE)
    {
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eIsNotificationEnabled
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsNotificationEnabled (
    DECODER_OBJECT hDecoder,
    BOOLEAN *pbEnable
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bSuccess;

    if (pbEnable != NULL)
    {
        // Verify and lock decoder
        bSuccess = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bSuccess == TRUE)
        {
            SPORTS_FLASH_OBJECT hSportsFlash;

            hSportsFlash = DECODER_hSportsFlash(hDecoder);
            if (hSportsFlash != SPORTS_FLASH_INVALID_OBJECT)
            {
                SPORTS_FLASH_OBJECT_STRUCT *psObj =
                    (SPORTS_FLASH_OBJECT_STRUCT *)hSportsFlash;

                *pbEnable = psObj->bNotificationEnabled;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }

            // Relinquish ownership
            SMSO_vUnlock((SMS_OBJECT)hDecoder);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   SPORTS_FLASH_bEventHandler
 *
 *****************************************************************************/
BOOLEAN SPORTS_FLASH_bEventHandler (
    SPORTS_FLASH_OBJECT hSportsFlash,
    const SMS_EVENT_SPORTS_FLASH_STRUCT *psEventData
        )
{
    SPORTS_FLASH_OBJECT_STRUCT *psObj = 
        (SPORTS_FLASH_OBJECT_STRUCT*)hSportsFlash;
    BOOLEAN bReturn = FALSE, bOwner;

    // Verify ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hSportsFlash);
    if (bOwner == TRUE)
    {
        // Process event...
        switch (psEventData->eAction)
        {
            case SPORTS_FLASH_ACTION_LOAD_CONFIG_EVENT:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Loading config");

                bReturn = bRestoreSportsFlash(psObj);
            }
            break;

            case SPORTS_FLASH_ACTION_ADD_TEAM:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Add team");

                bReturn = bHandleAddTeam(psObj, &psEventData->uEvent.sTeam);
            }
            break;

            case SPORTS_FLASH_ACTION_REMOVE_TEAM:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Remove team");

                bReturn = bHandleRemoveTeam(psObj, &psEventData->uEvent.sTeam);
            }
            break;

            case SPORTS_FLASH_ACTION_REMOVE_ALL_TEAMS:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Remove all teams");

                bReturn = bHandleRemoveAllTeams(psObj);
            }
            break;

            case SPORTS_FLASH_ACTION_SET_GAMES_MONITOR:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Set games monitor");

                bReturn = bHandleSetGamesMonitor(psObj, &psEventData->uEvent.sSetGamesMonitor);
            }
            break;

            case SPORTS_FLASH_ACTION_PLAY_FLASH_EVENT:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Play flash event");

                bReturn = bHandlePlayFlashEvent(psObj, &psEventData->uEvent.sFlashAction);
            }
            break;

            case SPORTS_FLASH_ACTION_ABORT_FLASH_EVENT:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Abort flash event");

                bReturn = bHandleAbortFlashEvent(psObj);
            }
            break;

            case SPORTS_FLASH_ACTION_REMAIN_FLASH_EVENT_CHANNEL:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Remain flash event channel");

                bReturn = bHandleRemainFlashEvent(psObj, &psEventData->uEvent.sFlashAction);
            }
            break;

            case SPORTS_FLASH_ACTION_STOP_EVENT:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Stop service");

                bReturn = bHandleStopEvent(psObj);
            }
            break;

            case SPORTS_FLASH_ACTION_ENABLE_NOTIFICATION:
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Enable/disable notifications");

                bReturn = bHandleEnableNotificationEvent(
                    psObj, &psEventData->uEvent.sEnableNotification);
            }
            break;

            case SPORTS_FLASH_EVENT_GAME_EVENT:
            case SPORTS_FLASH_EVENT_FLASH_EVENT:
            {
                printf(SPORTS_FLASH_OBJECT_NAME": Unexpected event (%d)\n",
                    psEventData->eAction);
                bReturn = FALSE;
            }
            break;
        }
    }
    else if (hSportsFlash == SPORTS_FLASH_INVALID_OBJECT)
    {
        // its possible that event can be received after
        // the sports flash has been stopped
        // then we need to cleanup the event content
        vCleanupEvent(psEventData);
    }

    return bReturn;
}

/*****************************************************************************
 *
 *   SPORTS_FLASH_bHandleGameEvent
 *
 *****************************************************************************/
BOOLEAN SPORTS_FLASH_bHandleGameEvent (
    DECODER_OBJECT hDecoder,
    const SPORTS_FLASH_EVENT_GAME_EVENT_STRUCT *psGameEvent
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        SPORTS_FLASH_OBJECT_STRUCT *psObj;

        if (psGameEvent == NULL)
        {
            break;
        }

        psObj = (SPORTS_FLASH_OBJECT_STRUCT *)
            DECODER_hSportsFlash(hDecoder);
        if (psObj == NULL)
        {
            // probably, we just stopped so lets just ignore it.
            bResult = TRUE;
            break;
        }

        if (psGameEvent->bEnded == FALSE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_LINKED_LIST_ENTRY hLLEntry = 
                OSAL_INVALID_LINKED_LIST_ENTRY;
            BOOLEAN bSuccess;
            SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psEventInfo;
            N16 n16CompareResult;

            // we probably need to add new event to list,
            // lets create new item
            psEventInfo = (SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                SPORTS_FLASH_OBJECT_NAME":GameInfo",
                sizeof(SPORTS_FLASH_GAME_EVENT_INFO_STRUCT), TRUE);
            if (psEventInfo == NULL)
            {
                // No memory for Game Event Info struct
                break;
            }

            bSuccess = bInitGameInfoEvent(psGameEvent, psEventInfo);
            if (bSuccess == FALSE)
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Failed to init game "
                    "info event");
                OSAL.vLinkedListMemoryFree(psEventInfo);
                break;
            }

            // we process game events only for favorite teams
            bSuccess = bCheckGameEvent(psObj, psEventInfo);
            if (bSuccess == FALSE)
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Game event is not related "
                    "to our favorite teams");
                vDestroyGame(psEventInfo);
                bResult = TRUE;
                break;
            }

            // Adding newly created event
            eReturnCode = OSAL.eLinkedListAdd(psObj->hGamesList,
                &hLLEntry, psEventInfo);
            if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                // Duplicate game found
                SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psSavedInfo;

                psSavedInfo = (SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *)
                    OSAL.pvLinkedListThis(hLLEntry);

                n16CompareResult = n16ComparePriority(psObj, 
                    psSavedInfo, psEventInfo);
                if (n16CompareResult > 0)
                {
                    // an old game has priority lower than new
                    // lets update bias and channel id
                    psSavedInfo->tChannelID = psEventInfo->tChannelID;
                    psSavedInfo->un16Bias = psEventInfo->un16Bias;

                    // now removing psEventInfo - its not needed anymore
                    vDestroyGame(psEventInfo);
                    psEventInfo = psSavedInfo;
                }
                else
                {
                    // priority is the same or lower, lets 
                    // skip notification
                    printf(SPORTS_FLASH_OBJECT_NAME": Game event priority "
                        "is lower or the same (channel %u)\n", 
                        psEventInfo->tChannelID);
                    vDestroyGame(psEventInfo);
                    bResult = TRUE;
                    break;
                }
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                // something wrong happened, lets just remove this
                // game event, report error and keep silence
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to add game to "
                    "list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                vDestroyGame(psEventInfo);
                break;
            }

            // Notify the application about new or updated game
            if (psObj->bNotificationEnabled == TRUE)
            {
                psObj->vGameEventCallback(
                    hDecoder,
                    SPORTS_FLASH_PROGRAM_AVAILABLE,
                    psEventInfo,
                    psObj->pvGameEventCallbackArg);
            }
        }
        else
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_LINKED_LIST_ENTRY hLLEntry = 
                OSAL_INVALID_LINKED_LIST_ENTRY;
            SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psEventInfo;

            // we don't need to create new event,
            // lets use the dummy object
            SPORTS_FLASH_GAME_EVENT_INFO_STRUCT sSearchCriteria;

            sSearchCriteria.tChannelID = psGameEvent->tChannelID;
            sSearchCriteria.bEnded = TRUE;

            // Try to find this game in a list
            eReturnCode = OSAL.eLinkedListSearch(psObj->hGamesList,
                &hLLEntry, &sSearchCriteria);

            if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
            {
                // just ignore it
                printf(SPORTS_FLASH_OBJECT_NAME": Game for channel %u was "
                    "not found\n", psGameEvent->tChannelID);
                bResult = TRUE;
                break;
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                // if item is not found, its ok, but in other cases...
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to search in games "
                    "list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            psEventInfo = (SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *)
                OSAL.pvLinkedListThis(hLLEntry);

            // Remove the found entry
            eReturnCode = OSAL.eLinkedListRemove(hLLEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // cannot delete, just exit without notification
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to remove game "
                    "from list (%s)", OSAL.pacGetReturnCodeName(
                    eReturnCode));
                break;
            }

            // Notify the application about ended game
            if (psObj->bNotificationEnabled == TRUE)
            {
                psObj->vGameEventCallback(
                    hDecoder,
                    SPORTS_FLASH_PROGRAM_ENDED,
                    psEventInfo,
                    psObj->pvGameEventCallbackArg);
            }

            vDestroyGame(psEventInfo);
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   SPORTS_FLASH_bHandleFlashEvent
 *
 *****************************************************************************/
BOOLEAN SPORTS_FLASH_bHandleFlashEvent (
    DECODER_OBJECT hDecoder,
    const SPORTS_FLASH_EVENT_FLASH_EVENT_STRUCT *psFlashEvent
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        SPORTS_FLASH_OBJECT_STRUCT *psObj;

        if (psFlashEvent == NULL)
        {
            break;
        }

        psObj = (SPORTS_FLASH_OBJECT_STRUCT *)
            DECODER_hSportsFlash(hDecoder);
        if (psObj == NULL)
        {
            bResult = TRUE;
            break;
        }

        if (psFlashEvent->eEventStatus == SPORTS_FLASH_EVENT_OCCURRED)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            SPORTS_FLASH_FLASH_EVENT_STRUCT *psEventInfo;

            psEventInfo = (SPORTS_FLASH_FLASH_EVENT_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                SPORTS_FLASH_OBJECT_NAME":FlashEventInfo",
                sizeof(SPORTS_FLASH_FLASH_EVENT_STRUCT), TRUE);
            if (psEventInfo == NULL)
            {
                // No memory for Flash Event Info struct
                break;
            }

            psEventInfo->tChannelID = psFlashEvent->tChannelID;
            psEventInfo->tFlashEventID = psFlashEvent->tFlashEventID;

            eReturnCode = OSAL.eLinkedListAdd(psObj->hEventsList, 
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psEventInfo);
            if (eReturnCode != OSAL_SUCCESS)
            {
                OSAL.vLinkedListMemoryFree(psEventInfo);
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to add flash event "
                    "to list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            // Notify the application
            if ((psObj->vFlashEventCallback != NULL) &&
                (psObj->bNotificationEnabled == TRUE))
            {
                psObj->vFlashEventCallback(
                    hDecoder,
                    psEventInfo->tChannelID,
                    psEventInfo->tFlashEventID,
                    psFlashEvent->eEventStatus,
                    psObj->pvFlashEventCallbackArg);
            }
        }
        else
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_LINKED_LIST_ENTRY hLLEntry = 
                OSAL_INVALID_LINKED_LIST_ENTRY;
            SPORTS_FLASH_FLASH_EVENT_STRUCT sSearchCriteria;

            sSearchCriteria.tFlashEventID = psFlashEvent->tFlashEventID;

            // Try to find this event in the list
            eReturnCode = OSAL.eLinkedListSearch(psObj->hEventsList,
                &hLLEntry, &sSearchCriteria);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Remove the found entry
                SPORTS_FLASH_FLASH_EVENT_STRUCT *psSavedEvent;

                psSavedEvent = (SPORTS_FLASH_FLASH_EVENT_STRUCT *)
                    OSAL.pvLinkedListThis(hLLEntry);
                eReturnCode = OSAL.eLinkedListRemove(hLLEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SPORTS_FLASH_OBJECT_NAME": Failed to remove the "
                        "flash event from the list (%s)", 
                        OSAL.pacGetReturnCodeName(eReturnCode));
                    break;
                }

                OSAL.vLinkedListMemoryFree(psSavedEvent);

                // Notify the application
                if ((psObj->vFlashEventCallback != NULL) &&
                    (psObj->bNotificationEnabled == TRUE))
                {
                    psObj->vFlashEventCallback(
                        hDecoder,
                        psFlashEvent->tChannelID,
                        psFlashEvent->tFlashEventID,
                        psFlashEvent->eEventStatus,
                        psObj->pvFlashEventCallbackArg);
                }
            }
            else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
            {
                 // if any error occurred...
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to search in the "
                    "flash events list (%s)", OSAL.pacGetReturnCodeName(
                    eReturnCode));
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   SPORTS_FLASH_vUpdateFlashPlaybackStatus
 *
 *****************************************************************************/
void SPORTS_FLASH_vUpdateFlashPlaybackStatus (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tFlashEventId
        )
{
    do 
    {
        SPORTS_FLASH_OBJECT_STRUCT *psObj;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        psObj = (SPORTS_FLASH_OBJECT_STRUCT *)
            DECODER_hSportsFlash(hDecoder);
        if (psObj == NULL)
        {
            // looks like indication received after sports flash stop
            break;
        }

        psObj->tCurrentFlashEventID = tFlashEventId;

        if (tFlashEventId != SPORTS_FLASH_INVALID_EVENT_ID)
        {
            // New flash event playback has been started. So start timer.
            // When it expires, we will remain on the flash event channel.
            eReturnCode = OSAL.eTimerStartRelative(psObj->hEventTimer,
                SPORTS_FLASH_EVENT_TIMER_DURATION_MS, 0);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Can't start EventTimer Timer(%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            // The flash event playback has been stopped. So stop the timer.
            if (psObj->hEventTimer != OSAL_INVALID_OBJECT_HDL)
            {
                OSAL.eTimerStop(psObj->hEventTimer);
            }
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
 *
 *   SPORTS_FLASH_vStop
 *
 *****************************************************************************/
void SPORTS_FLASH_vStop (
    SPORTS_FLASH_OBJECT hSportsFlash
        )
{
    if (hSportsFlash != SPORTS_FLASH_INVALID_OBJECT)
    {
        bHandleStopEvent((SPORTS_FLASH_OBJECT_STRUCT *)hSportsFlash);
    }

    return;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   bHandleUpdateTeams
 *
 *****************************************************************************/
static BOOLEAN bHandleUpdateTeams (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;
    UN32 *paun32Teams = NULL;

    do 
    {
        UN32 un32Count = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        SPORTS_FLASH_TEAM_CONVERT_ITERATOR_STRUCT sIterator;

        eReturnCode = OSAL.eLinkedListItems(psObj->hTeamsList, &un32Count);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Failed to get number of teams "
                "(%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        if (un32Count == 0)
        {
            // list is empty, checking the monitor status
            if (psObj->bSeekMonitorEnabled == TRUE)
            {
                bResult = RADIO_bSeekMonitorDisable(psObj->hDecoder,
                    SEEK_MONITOR_TYPE_SPORTS_FLASH);
                if (bResult == TRUE)
                {
                    psObj->bSeekMonitorEnabled = FALSE;
                }
            }
            else
            {
                bResult = TRUE;
            }
            break;
        }

        paun32Teams = (UN32 *)OSAL.pvMemoryAllocate(
            SPORTS_FLASH_OBJECT_NAME":TeamsArray", sizeof(UN32) * un32Count,
            TRUE);
        if (paun32Teams == NULL)
        {
            // Out of memory. Very bad...
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": No memory for Teams array");
            break;
        }

        sIterator.un32Idx = 0;
        sIterator.paun32Teams = paun32Teams;
        eReturnCode = OSAL.eLinkedListIterate(psObj->hTeamsList, 
            (OSAL_LL_ITERATOR_HANDLER)bTeamConvertIterator, &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Team iterate failed (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Check the count
        if (sIterator.un32Idx != un32Count)
        {
            // Error!
            printf(SPORTS_FLASH_OBJECT_NAME": Failed to prepare teams for "
                "seek (only %u teams processed from %u)", sIterator.un32Idx,
                un32Count);
            break;
        }

        if (psObj->bSeekMonitorEnabled == TRUE)
        {
            // need to disable it before applying next config
            bResult = RADIO_bSeekMonitorDisable(psObj->hDecoder,
                SEEK_MONITOR_TYPE_SPORTS_FLASH);
            if (bResult == FALSE)
            {
                puts(SPORTS_FLASH_OBJECT_NAME": Failed to disable seek "
                    "monitor");
                break;
            }
        }

        // Set the Seek Monitor
        psObj->bSeekMonitorEnabled = bResult = RADIO_bSeekMonitorEnable(
            psObj->hDecoder, SEEK_MONITOR_TYPE_SPORTS_FLASH, (UN8)un32Count,
            sizeof(UN32), paun32Teams);

    } while (FALSE);

    if (paun32Teams != NULL)
    {
        OSAL.vMemoryFree(paun32Teams);
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bHandleSetGamesMonitor
 *
 *****************************************************************************/
static BOOLEAN bHandleSetGamesMonitor (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_SET_GAMES_MONITOR_STRUCT *psSetGamesMonitor
        )
{
    BOOLEAN bResult = FALSE;
    SERVICE_ID *ptServiceIDList = NULL;

    do 
    {
        CCACHE_OBJECT hCCache;
        UN8 un8ChannelsCount, un8Idx, un8Count = 0;
        CHANNEL_OBJECT hChannel;

        // are we going to disable monitoring?
        if (psSetGamesMonitor->un16Count == 0)
        {
            if (psObj->bGamesMonitorEnabled == TRUE)
            {
                bResult = RADIO_bSetGamesMonitor(psObj->hDecoder, NULL, 0);
                psObj->bGamesMonitorEnabled = FALSE;

                psObj->pvFlashEventCallbackArg = NULL;
                psObj->vFlashEventCallback = NULL;
            }

            // ok, nothing to do here
            break;
        }

        psObj->pvFlashEventCallbackArg = psSetGamesMonitor->pvArg;
        psObj->vFlashEventCallback = psSetGamesMonitor->vEventCallback;

        hCCache = DECODER_hCCache(psObj->hDecoder);
        if (hCCache == CCACHE_INVALID_OBJECT)
        {
            break;
        }

        if (psSetGamesMonitor->un16Count > psObj->un8MaxSportsFlash)
        {
            un8ChannelsCount = psObj->un8MaxSportsFlash;
        }
        else
        {
            un8ChannelsCount = (UN8)(psSetGamesMonitor->un16Count);
        }

        // Allocating the array for SERVICE_IDs
        ptServiceIDList = (SERVICE_ID *)OSAL.pvMemoryAllocate(
            SPORTS_FLASH_OBJECT_NAME":ServiceIdList",
            sizeof(SERVICE_ID) * un8ChannelsCount, FALSE);
        if (ptServiceIDList == NULL)
        {
            break;
        }

        // Convert the CHANNEL_IDs to SERVICE_IDs
        for (un8Idx = 0; un8Idx < un8ChannelsCount; ++un8Idx)
        {
            hChannel = CCACHE_hChannel(hCCache,
                &(psSetGamesMonitor->ptChannelIDList[un8Idx]), NULL);
            if (hChannel != CHANNEL_INVALID_OBJECT)
            {
                ptServiceIDList[un8Count++] =
                    CHANNEL.tServiceId(hChannel);
            }
        }

        psObj->bGamesMonitorEnabled = bResult = RADIO_bSetGamesMonitor(
            psObj->hDecoder, ptServiceIDList, un8Count);

    } while (FALSE);

    if (psSetGamesMonitor->ptChannelIDList != NULL)
    {
        OSAL.vMemoryFree(psSetGamesMonitor->ptChannelIDList);
    }

    if (ptServiceIDList != NULL)
    {
        OSAL.vMemoryFree(ptServiceIDList);
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bHandlePlayFlashEvent
 *
 *****************************************************************************/
static BOOLEAN bHandlePlayFlashEvent (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_FLASH_EVENT_STRUCT *psFlashEvent
        )
{
    BOOLEAN bSuccess;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SPORTS_FLASH_FLASH_EVENT_STRUCT sSeachCriteria;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    sSeachCriteria.tFlashEventID = psFlashEvent->tFlashEventID;
    // looking for event in our list of events
    eReturnCode = OSAL.eLinkedListSearch(psObj->hEventsList, &hEntry,
        &sSeachCriteria);
    if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        // looks like this event expired or incorrect
        // lets ignore it
        printf(SPORTS_FLASH_OBJECT_NAME" Failed to find flash event %u, "
            "ignoring request\n", sSeachCriteria.tFlashEventID);
        return TRUE;
    }
    else if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SPORTS_FLASH_OBJECT_NAME": Failed to search for event (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
        return FALSE;
    }

    bSuccess = DECODER_bPlayFlashEvent(psObj->hDecoder,
        psFlashEvent->tFlashEventID);
    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SPORTS_FLASH_OBJECT_NAME": Failed to start flash event playback");
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bHandleAbortFlashEvent
 *
 *****************************************************************************/
static BOOLEAN bHandleAbortFlashEvent (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = TRUE;

    if (psObj->tCurrentFlashEventID != SPORTS_FLASH_INVALID_EVENT_ID)
    {
        bSuccess = DECODER_bAbortFlashEvent(psObj->hDecoder, 
            psObj->tCurrentFlashEventID);
        if (bSuccess != FALSE)
        {
            psObj->tCurrentFlashEventID = SPORTS_FLASH_INVALID_EVENT_ID;
            if (psObj->hEventTimer != OSAL_INVALID_OBJECT_HDL)
            {
                OSAL.eTimerStop(psObj->hEventTimer);
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Failed to abort flash event playback");
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bHandleRemainFlashEvent
 *
 *****************************************************************************/
static BOOLEAN bHandleRemainFlashEvent (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_FLASH_EVENT_STRUCT *psFlashEvent
        )
{
    BOOLEAN bSuccess = FALSE;

    if (psFlashEvent->tFlashEventID == psObj->tCurrentFlashEventID)
    {
        psObj->tCurrentFlashEventID = SPORTS_FLASH_INVALID_EVENT_ID;
        bSuccess = DECODER_bRemainFlashEvent(psObj->hDecoder,
            psFlashEvent->tFlashEventID);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Failed to remain flash event channel");
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   n16ComparePriority
 *
 *   This function returns TRUE if the new game has higher priority than old.
 *
 *****************************************************************************/
static N16 n16ComparePriority(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame1,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame2
        )
{
    N16 n16Result;
    UN8 un8Priority1, un8Priority2;

    // Determine game priorities
    un8Priority1 = un8FindPriority(psObj, psGame1);
    un8Priority2 = un8FindPriority(psObj, psGame2);

    n16Result = COMPARE(un8Priority1, un8Priority2);
    if (n16Result == 0)
    {
        // Unlikely corner case - equal priorities
        // Try to find the channel with lowest SERVICE_ID
        CCACHE_OBJECT hCCache;

        hCCache = DECODER_hCCache(psObj->hDecoder);
        if (hCCache != CCACHE_INVALID_OBJECT)
        {
            CHANNEL_OBJECT hChannel;
            SERVICE_ID tSID1 = SERVICE_INVALID_ID;
            SERVICE_ID tSID2 = SERVICE_INVALID_ID;
            CHANNEL_ID tChannelID;

            tChannelID = psGame1->tChannelID;
            hChannel = CCACHE_hChannel(hCCache, &tChannelID, NULL);
            if (hChannel != CHANNEL_INVALID_OBJECT)
            {
                tSID1 = CHANNEL.tServiceId(hChannel);
            }

            tChannelID = psGame2->tChannelID;
            hChannel = CCACHE_hChannel(hCCache, &tChannelID, NULL);
            if (hChannel != CHANNEL_INVALID_OBJECT)
            {
                tSID2 = CHANNEL.tServiceId(hChannel);
            }

            n16Result = COMPARE(tSID1, tSID2);
        }
    }

    return n16Result;
}

/*****************************************************************************
 *
 *   bFindPriority
 *
 *****************************************************************************/
static UN8 un8FindPriority(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psInfo
        )
{
    // initializing result to lowest priority value
    UN8 un8Result = UN8_MAX;
    OSAL_LINKED_LIST_ENTRY hLLAwayEntry, hLLHomeEntry;

    // Check parameters
    if (psInfo->un16Bias >= sizeof(aun8GamePriorityHomeTeam))
    {
        return un8Result;
    }

    // Search favorite teams list
    hLLAwayEntry = hFindInTeamsList(psObj, psInfo->hAwayLeagueId, 
        psInfo->hAwayTeamId);
    hLLHomeEntry = hFindInTeamsList(psObj, psInfo->hHomeLeagueId, 
        psInfo->hHomeTeamId);

    if (hLLAwayEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        if (hLLHomeEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            un8Result = aun8GamePriorityBothTeams[psInfo->un16Bias];
        }
        else
        {
            un8Result = aun8GamePriorityAwayTeam[psInfo->un16Bias];
        }
    }
    else if (hLLHomeEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        un8Result = aun8GamePriorityHomeTeam[psInfo->un16Bias];
    }

    return un8Result;
}

/*****************************************************************************
 *
 *   eCreateSportsFlash
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCreateSportsFlash (
    DECODER_OBJECT hDecoder,
    UN8 un8MaxSportsFlash,
    SPORTS_FLASH_GAME_EVENT_CALLBACK vGameEventCallback,
    void *pvGameEventCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SPORTS_FLASH_OBJECT_STRUCT *psObj = NULL;

    do
    {
        BOOLEAN bOk;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Create new object which is child of decoder
        psObj = (SPORTS_FLASH_OBJECT_STRUCT *)SMSO_hCreate(
            SPORTS_FLASH_OBJECT_NAME,
            sizeof(SPORTS_FLASH_OBJECT_STRUCT), (SMS_OBJECT)hDecoder,
            FALSE);

        // Not enough memory
        if (psObj == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME
                ": Not enough memory to create Sports Flash object");
            break;
        }

        // Initialize object with defaults
        *psObj = gsObjectDefaults;

        // Set decoder object handle into Sports Flash object struct
        psObj->hDecoder = hDecoder;

        psObj->un8MaxSportsFlash = un8MaxSportsFlash;

        // Create the lists
        eOsalReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hTeamsList,
                SPORTS_FLASH_OBJECT_NAME":TeamsList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareFavoriteTeam,
                OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
                OSAL_LL_OPTION_UNIQUE);
        if(eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME
                ": Could not create favorite teams list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        eOsalReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hGamesList,
                SPORTS_FLASH_OBJECT_NAME":GamesList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareGameInfo,
                OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
                OSAL_LL_OPTION_UNIQUE | OSAL_LL_OPTION_LINEAR
                    );
        if(eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME
                ": Could not create games list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        eOsalReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hEventsList,
                SPORTS_FLASH_OBJECT_NAME":EventsList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareFlashEvent,
                OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
                OSAL_LL_OPTION_UNIQUE
                    );
        if(eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME
                ": Could not create flash events list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        // Create a Flash Event Timer for this Service. It's a one shot
        // timer used to remain on a Flash Event channel
        eOsalReturnCode = OSAL.eTimerCreate(&psObj->hEventTimer,
            SPORTS_FLASH_OBJECT_NAME":EventTimer",
            (OSAL_TIMER_HANDLER)vFlashEventTimerCallback,
            psObj
                );
        if(eOsalReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Can't Create EventTimer Timer "
                "(%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        // Get Sports Flash Tag
        bOk = bCreateTag(psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME
                ": Failed to get Sports Flash tag object");
            break;
        }

        psObj->vGameEventCallback = vGameEventCallback;
        psObj->pvGameEventCallbackArg = pvGameEventCallbackArg;

        // Set Sports Flash handle into Decoder
        bOk = DECODER_bSetSportsFlashHandle(hDecoder, 
            (SPORTS_FLASH_OBJECT)psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME
                ": Failed to set Sports Flash handle");
            break;
        }

        puts(SPORTS_FLASH_OBJECT_NAME": Created service");

        // Seems to be OK, if we can go so far ...
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (psObj != NULL)
        {
            // Dissociate from decoder
            DECODER_bSetSportsFlashHandle(
                hDecoder,
                SPORTS_FLASH_INVALID_OBJECT
                    );

            // Destroy created objects
            vDestroySportsFlash(psObj);

            // and destroy the main object
            SMSO_vDestroy((SMS_OBJECT)psObj);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   vDestroySportsFlash
 *
 *****************************************************************************/
static void vDestroySportsFlash (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    // Zero the events handlers
    psObj->vGameEventCallback = NULL;
    psObj->pvGameEventCallbackArg = NULL;
    psObj->vFlashEventCallback = NULL;
    psObj->pvFlashEventCallbackArg = NULL;

    // Destroy Flash Event wait timer if one exists
    if(psObj->hEventTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eTimerStop(psObj->hEventTimer);
        // Delete timer
        OSAL.eTimerDelete(psObj->hEventTimer);
        psObj->hEventTimer = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the created lists
    if (psObj->hTeamsList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psObj->hTeamsList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyFavoriteTeam);
        OSAL.eLinkedListDelete(psObj->hTeamsList);
        psObj->hTeamsList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->hGamesList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psObj->hGamesList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyGame);
        OSAL.eLinkedListDelete(psObj->hGamesList);
        psObj->hGamesList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->hEventsList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psObj->hEventsList,
           (OSAL_LL_RELEASE_HANDLER)OSAL.vLinkedListMemoryFree);
        OSAL.eLinkedListDelete(psObj->hEventsList);
        psObj->hEventsList = OSAL_INVALID_OBJECT_HDL;
    }
}

/*******************************************************************************
 *
 *   bCreateTag
 *
 *******************************************************************************/
static BOOLEAN bCreateTag (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    TAG_OBJECT hParentTag;
    BOOLEAN bSuccess = FALSE;

    // Get the decoder tag
    hParentTag = DECODER_hGetTag(psObj->hDecoder);
    if (hParentTag != TAG_INVALID_OBJECT)
    {
        TAG_OBJECT hTag;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Get Sports Flash Tag
        eReturnCode = TAG_eGet(SPORTS_FLASH_OBJECT_NAME, hParentTag, &hTag, NULL, TRUE);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Save Sports Flash Tag Locally
            psObj->hTag = hTag;
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   bRestoreSportsFlash
 *
 *******************************************************************************/
static BOOLEAN bRestoreSportsFlash (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    TAG_OBJECT hTeamsTag;
    TAG_OBJECT hEnableNotifTag;
    BOOLEAN bSuccess = FALSE;

    // Get the Favorite Teams Tag
    eReturnCode = TAG_eGet(
        SPORTS_FLASH_TAG_NAME_FAVORITE_TEAMS,
        psObj->hTag,
        &hTeamsTag,
        NULL,
        FALSE);

    // We have a Favorite Teams tag handle
    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hTeamsTag != TAG_INVALID_OBJECT))
    {
        SPORTS_FLASH_TEAM_LOAD_ITERATOR_STRUCT sIterator;

        // we don't need the parent tag, so we'll just set it to invalid
        sIterator.hParentTag = hTeamsTag;

        // Initially the buffer will be NULL. The memory will be allocated
        // the first time it is used, all tags iterated can then share it.
        sIterator.tBuffSize = 0;
        sIterator.pacBuff = NULL;
        sIterator.bNeedUpdate = FALSE;

        sIterator.psObj = psObj;

        eReturnCode = TAG_eIterateChildren(
            hTeamsTag,
            (TAG_ITERATION_HANDLER)bTeamsLoadIterator,
            (void *)&sIterator);
        // Done iterating. Free the buffer if it was created.
        if (sIterator.pacBuff != NULL)
        {
            OSAL.vMemoryFree((void *)sIterator.pacBuff);
        }

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        if (sIterator.bNeedUpdate == TRUE)
        {
            bSuccess = bHandleUpdateTeams(psObj);
        }
    }

    // Enable notifications by default
    psObj->bNotificationEnabled = TRUE;

    // Get Enable Notification Tag
    eReturnCode = TAG_eGet(
        SPORTS_FLASH_TAG_NAME_NOTIFICATIONS_ENABLED,
        psObj->hTag,
        &hEnableNotifTag,
        NULL,
        FALSE);

    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hEnableNotifTag != TAG_INVALID_OBJECT))
    {
        N32 n32TagValue, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        // get the tag value
        eReturnCode = TAG_eGetTagValue(hEnableNotifTag, TAG_TYPE_INTEGER,
            (void **)&pn32TagValue, &tSize);

        if((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
           (n32TagValue == 0))
        {
            // Disable notifications
            psObj->bNotificationEnabled = FALSE;
        }
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   bTeamsLoadIterator
 *
 *******************************************************************************/
static BOOLEAN bTeamsLoadIterator(
    TAG_OBJECT hTag,
    SPORTS_FLASH_TEAM_LOAD_ITERATOR_STRUCT *psIterator
        )
{
    STRING_OBJECT hTagName;
    OSAL_RETURN_CODE_ENUM eOSALResult = OSAL_ERROR;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psFavoriteTeam = NULL;
    CID_OBJECT hCid;

    if (hTag == TAG_INVALID_OBJECT || psIterator == NULL)
    {
        return FALSE;
    }

    do
    {
        N16 n16Compare;

        hTagName = TAG_hTagName(hTag);
        if (hTagName == STRING_INVALID_OBJECT)
        {
            break;
        }

        n16Compare = STRING.n16CompareCStr(SPORTS_FLASH_TAG_NAME_TEAM, 0, 
            hTagName);
        if (n16Compare != 0)
        {
            break;
        }

        psFavoriteTeam =
            (SPORTS_FLASH_FAVORITE_TEAM_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
                SPORTS_FLASH_OBJECT_NAME":FavoriteTeam",
                sizeof(SPORTS_FLASH_FAVORITE_TEAM_STRUCT),
                TRUE
                    );
        if (psFavoriteTeam == NULL)
        {
            break;
        }

        // get the League CID
        TAG_eGet(SPORTS_FLASH_TAG_NAME_LEAGUE_CID, hTag, &hChildTag, NULL, FALSE);
        // check if we got a valid tag handle. a valid tag implies that the
        // TAG_eGet returned success so we don't need to actually check the
        // result in this case
        if (hChildTag == TAG_INVALID_OBJECT)
        {
            break;
        }

        hCid = hLoadCID(hChildTag, psIterator);
        if (hCid == CID_INVALID_OBJECT)
        {
            break;
        }
        psFavoriteTeam->hLeagueId = CID.hDuplicate(hCid);
        CID_vDestroy(hCid);

        // get the Team CID
        TAG_eGet(SPORTS_FLASH_TAG_NAME_TEAM_CID, hTag, &hChildTag, NULL, FALSE);
        if (hChildTag == TAG_INVALID_OBJECT)
        {
            break;
        }

        hCid = hLoadCID(hChildTag, psIterator);
        if (hCid == CID_INVALID_OBJECT)
        {
            break;
        }
        psFavoriteTeam->hTeamId = CID.hDuplicate(hCid);
        CID_vDestroy(hCid);

        // Remember the Tag handle
        psFavoriteTeam->hTag = hTag;

        // add this to our list.
        eOSALResult = OSAL.eLinkedListAdd(psIterator->psObj->hTeamsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psFavoriteTeam);
        if (eOSALResult != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Failed to add team to list (%s)",
                OSAL.pacGetReturnCodeName(eOSALResult));
        }

        // if we successfully added any team, need to set update flag
        psIterator->bNeedUpdate = TRUE;
    } while (FALSE);

    if ((eOSALResult != OSAL_SUCCESS) && (psFavoriteTeam != NULL))
    {
        // Unable to add team to list
        vDestroyFavoriteTeam(psFavoriteTeam);
    }

    // Will iterate through all the teams available
    return TRUE;
}

/*******************************************************************************
 *
 *   hLoadCID
 *
 *******************************************************************************/
static CID_OBJECT hLoadCID(
    TAG_OBJECT hTag,
    SPORTS_FLASH_TEAM_LOAD_ITERATOR_STRUCT *psIterator
        )
{
    CID_OBJECT hId = CID_INVALID_OBJECT;
    size_t tSize;
    STRING_OBJECT hCIDString = STRING_INVALID_OBJECT, *phString;

    phString = &hCIDString;
    tSize = sizeof(STRING_OBJECT);
    TAG_eGetTagValue(
        hTag,
        TAG_TYPE_STRING,
        (void**)&phString,
        &tSize);
    // check if we got a valid string handle. a valid string implies
    // that the TAG_eGetValue returned success and that the tag has a
    // value so we don't need to actually check the result in this case
    if (hCIDString == STRING_INVALID_OBJECT)
    {
        return hId;
    }

    tSize = STRING.tSize(hCIDString) + 1; // add one for NULL
    if (tSize > psIterator->tBuffSize) // Need a bigger buffer?
    {
        // Do we already have a buffer allocated?
        if (psIterator->pacBuff != NULL)
        {
            // Free previously allocated buffer
            OSAL.vMemoryFree((void *)psIterator->pacBuff);
            psIterator->tBuffSize = 0;
        }

        // allocate memory for the read buffer
        psIterator->pacBuff =
            (char *) OSAL.pvMemoryAllocate(
                SPORTS_FLASH_OBJECT_NAME":CSTR",
                tSize,
                TRUE
                    );
        if (psIterator->pacBuff != NULL)
        {
            // Record the size of our new buffer
            psIterator->tBuffSize = tSize;
        }
    }

    // If we have a buffer, do our thing
    if (psIterator->pacBuff != NULL)
    {
        char *pacBuff = psIterator->pacBuff;

        // Now use that buffer
        STRING.tCopyToCStr(hCIDString, pacBuff, tSize);
        // Read from memory assumes memory provided as an input is
        // NULL terminated. It is in fact since we just did a
        // string copy above into a buffer we know is at least one
        // larger than the source (thus it will be NULL terminated).
        hId = CID_hReadFromMemory((const void **)&pacBuff);
    }

    // removing the created string
    STRING_vDestroy(hCIDString);

    return hId;
}

/*******************************************************************************
 *
 *   vSaveTeam
 *
 *******************************************************************************/
static void vSaveTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psFavoriteTeam
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    TAG_OBJECT hTeamsTag;
    TAG_OBJECT hChildTag;
    TAG_OBJECT hCIDTag;
    size_t tBuffSize = 0;
    char *pacBuff = NULL;

    // Receive favorite teams tag value
    TAG_eGet(SPORTS_FLASH_TAG_NAME_FAVORITE_TEAMS, psObj->hTag,
        &hTeamsTag, NULL, TRUE);

    // Do we have a Favorite Teams tag handle?
    if (hTeamsTag == TAG_INVALID_OBJECT)
    {
        // No, nowhere to save new team
        return;
    }

    eReturnCode = TAG_eAdd(
        SPORTS_FLASH_TAG_NAME_TEAM,
        hTeamsTag,
        &hChildTag,
        NULL);
    if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
        (hChildTag == TAG_INVALID_OBJECT))
    {
        // Could not add new team tag
        return;
    }

    psFavoriteTeam->hTag = hChildTag;

    if (psFavoriteTeam->hLeagueId != CID_INVALID_OBJECT)
    {
        size_t tSize;

        // get the size of this cid
        tSize = CID_tSize(psFavoriteTeam->hLeagueId) + 1; // add one for NULL
        // allocate memory for the buffer
        pacBuff =
            (char *) OSAL.pvMemoryAllocate(
                SPORTS_FLASH_OBJECT_NAME":CSTR",
                tSize,
                TRUE
                    );
        if (pacBuff != NULL)
        {
            char *pacBuffl = pacBuff;
            STRING_OBJECT hCIDString;

            // record the size of the buffer we just created
            tBuffSize = tSize;
            // write the CID to memory
            CID_n32FWriteToMemory(psFavoriteTeam->hLeagueId, (void **)&pacBuffl);
            // create a STRING from the CID info
            hCIDString = STRING.hCreate(pacBuff, 0);

            // get the CID tag
            eReturnCode = TAG_eGet(
                SPORTS_FLASH_TAG_NAME_LEAGUE_CID,
                hChildTag,
                &hCIDTag,
                NULL,
                TRUE
                    );
            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                eReturnCode = TAG_eSetTagValue(
                    hCIDTag,
                    TAG_TYPE_STRING,
                    &hCIDString,
                    sizeof(STRING_OBJECT),
                    FALSE
                        );
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                        SPORTS_FLASH_OBJECT_NAME": Failed to set tag value "
                        "(%d)", eReturnCode);
                }
            }

            // we're done with these
            STRING.vDestroy(hCIDString);
        }
    }

    if (psFavoriteTeam->hTeamId != CID_INVALID_OBJECT)
    {
        size_t tSize;

        // get the size of this cid
        tSize = CID_tSize(psFavoriteTeam->hTeamId) + 1; // add one for NULL
        // is it bigger than the current size of our buffer
        if (tSize > tBuffSize)
        {
            // do we even have a buffer?
            if (pacBuff != NULL)
            {
                // yes destroy this one since we need a bigger one
                OSAL.vMemoryFree((void *)pacBuff);
                pacBuff = NULL;
            }

            // allocate memory for the buffer
            pacBuff =
                (char *) OSAL.pvMemoryAllocate(
                    SPORTS_FLASH_OBJECT_NAME":CSTR",
                    tSize,
                    TRUE
                        );
        }

        // make sure we have a buffer
        if (pacBuff != NULL)
        {
            char *pacBuffl;
            STRING_OBJECT hCIDString;

            pacBuffl = pacBuff;
            // write the CID to memory
            CID_n32FWriteToMemory(psFavoriteTeam->hTeamId, (void **)&pacBuffl);
            // create a STRING from the CID info
            hCIDString = STRING.hCreate(pacBuff, 0);

            // get the CID tag
            eReturnCode = TAG_eGet(
                          SPORTS_FLASH_TAG_NAME_TEAM_CID,
                          hChildTag,
                          &hCIDTag,
                          NULL,
                          TRUE);
            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                eReturnCode = TAG_eSetTagValue(
                              hCIDTag,
                              TAG_TYPE_STRING,
                              &hCIDString,
                              sizeof(STRING_OBJECT),
                              TRUE);
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                        SPORTS_FLASH_OBJECT_NAME": Failed to set tag value "
                        "(%d)", eReturnCode);
                }
            }

            // we're done with these
            STRING.vDestroy(hCIDString);
        }
    }

    // Free the buffer
    if (pacBuff != NULL)
    {
        OSAL.vMemoryFree(pacBuff);
        pacBuff = NULL;
    }

    return;
}

/*****************************************************************************
*
*   vDestroyFavoriteTeam
*
*****************************************************************************/
static void vDestroyFavoriteTeam(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psFavoriteTeam
        )
{
    if(psFavoriteTeam != NULL)
    {
        // Destroy the object fields
        if (psFavoriteTeam->hLeagueId != CID_INVALID_OBJECT)
        {
            CID_vDestroy(psFavoriteTeam->hLeagueId);
            psFavoriteTeam->hLeagueId = CID_INVALID_OBJECT;
        }
        if (psFavoriteTeam->hTeamId != CID_INVALID_OBJECT)
        {
            CID_vDestroy(psFavoriteTeam->hTeamId);
            psFavoriteTeam->hTeamId = CID_INVALID_OBJECT;
        }

        // Destroy the object itself
        OSAL.vLinkedListMemoryFree(psFavoriteTeam);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyGame
*
*****************************************************************************/
static void vDestroyGame(
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame
        )
{
    if(psGame != NULL)
    {
        // Destroy the object fields
        vUninitGameEventInfo(psGame);

        // Destroy the object itself
        OSAL.vLinkedListMemoryFree((void *)psGame);
    }

    return;
}


/*****************************************************************************
*
*   vUninitGameEventInfo
*
*****************************************************************************/
static void vUninitGameEventInfo(
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame
        )
{
    if (psGame->hAwayLeagueId != CID_INVALID_OBJECT)
    {
        CID_vDestroy(psGame->hAwayLeagueId);
        psGame->hAwayLeagueId = CID_INVALID_OBJECT;
    }

    if (psGame->hAwayTeamId != CID_INVALID_OBJECT)
    {
        CID_vDestroy(psGame->hAwayTeamId);
        psGame->hAwayTeamId = CID_INVALID_OBJECT;
    }

    if (psGame->hHomeLeagueId != CID_INVALID_OBJECT)
    {
        CID_vDestroy(psGame->hHomeLeagueId);
        psGame->hHomeLeagueId = CID_INVALID_OBJECT;
    }

    if (psGame->hHomeTeamId != CID_INVALID_OBJECT)
    {
        CID_vDestroy(psGame->hHomeTeamId);
        psGame->hHomeTeamId = CID_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   hFindInTeamsList
*
*****************************************************************************/
static OSAL_LINKED_LIST_ENTRY hFindInTeamsList (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    CID_OBJECT hLeagueId,
    CID_OBJECT hTeamId
        )
{
    OSAL_LINKED_LIST_ENTRY hLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    SPORTS_FLASH_FAVORITE_TEAM_STRUCT sFavoriteTeam;
    sFavoriteTeam.hLeagueId = hLeagueId;
    sFavoriteTeam.hTeamId = hTeamId;

    // LL search
    eReturnCode = OSAL.eLinkedListSearch(psObj->hTeamsList, &hLLEntry,
        (void*)&sFavoriteTeam);
    if ((eReturnCode != OSAL_SUCCESS) && 
        (eReturnCode != OSAL_OBJECT_NOT_FOUND))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
            SPORTS_FLASH_OBJECT_NAME": Failed to search in list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return hLLEntry;
}

/*****************************************************************************
*
*   n16FindFavoriteTeam
*
*****************************************************************************/
static N16 n16CompareFavoriteTeam(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam1,
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam2
        )
{
    N16 n16Result = N16_MIN;

    if ((psTeam1 != NULL) && (psTeam2 != NULL))
    {
        n16Result = CID.n16Compare(psTeam1->hLeagueId, psTeam2->hLeagueId);
        if (n16Result == 0)
        {
            n16Result = CID.n16Compare(psTeam1->hTeamId, psTeam2->hTeamId);
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareGameInfo
*
*****************************************************************************/
static N16 n16CompareGameInfo (
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psObj1,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psObj2
        )
{
    N16 n16Result = N16_MIN;

    do 
    {
        if ((psObj1 == NULL) || (psObj2 == NULL))
        {
            break;
        }

        if (psObj1->bEnded == TRUE || psObj2->bEnded == TRUE)
        {
            if (psObj1->tChannelID == psObj2->tChannelID)
            {
                n16Result = 0;
            }
        }
        else
        {
            n16Result = CID.n16Compare(psObj1->hAwayLeagueId,
                psObj2->hAwayLeagueId);
            if (n16Result != 0)
            {
                break;
            }

            n16Result = CID.n16Compare(psObj1->hAwayTeamId,
                psObj2->hAwayTeamId);
            if (n16Result != 0)
            {
                break;
            }

            n16Result = CID.n16Compare(psObj1->hHomeLeagueId,
                psObj2->hHomeLeagueId);
            if (n16Result != 0)
            {
                break;
            }

            n16Result = CID.n16Compare(psObj1->hHomeTeamId,
                psObj2->hHomeTeamId);
        }
    } while (FALSE);

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareFlashEvent
*
*****************************************************************************/
static N16 n16CompareFlashEvent(
    SPORTS_FLASH_FLASH_EVENT_STRUCT *psObj1,
    SPORTS_FLASH_FLASH_EVENT_STRUCT *psObj2
        )
{
    if ((psObj1 == NULL) || (psObj2 == NULL))
    {
        return N16_MIN;
    }

    return COMPARE(psObj1->tFlashEventID, psObj2->tFlashEventID);
}

/*****************************************************************************
*
*   bTeamsIterator
*
*****************************************************************************/
static BOOLEAN bTeamsIterator(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT const *psObj,
    SPORTS_FLASH_TEAMS_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bSuccess = FALSE;

    if (psObj == NULL || psIterator == NULL)
    {
        return FALSE;
    }

    if (psIterator->bIteratorCallback != NULL)
    {
        bSuccess = psIterator->bIteratorCallback(
            psObj->hLeagueId,
            psObj->hTeamId,
            psIterator->pvTeamsIteratorCallbackArg
                );
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bTeamConvertIterator
*
*****************************************************************************/
static BOOLEAN bTeamConvertIterator(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT const *psObj,
    SPORTS_FLASH_TEAM_CONVERT_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bSuccess = FALSE;

    if (psObj == NULL || psIterator == NULL)
    {
        return FALSE;
    }

    do
    {
        UN32 un32League, un32Team, *pun32Id = &un32League;
        N32 n32Result;

        n32Result = CID_n32GetValue(psObj->hLeagueId, (void **)&pun32Id);
        if (n32Result <= 0)
        {
            // Error!
            break;
        }

        pun32Id = &un32Team;
        n32Result = CID_n32GetValue(psObj->hTeamId, (void **)&pun32Id);
        if (n32Result <= 0)
        {
            // Error!
            break;
        }

        psIterator->paun32Teams[psIterator->un32Idx++] = 
            (un32League << 16) | un32Team;

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bFlashEventsIterator
*
*****************************************************************************/
static BOOLEAN bFlashEventsIterator(
    SPORTS_FLASH_FLASH_EVENT_STRUCT const *psObj,
    SPORTS_FLASH_EVENTS_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bSuccess = FALSE;

    if (psObj == NULL || psIterator == NULL)
    {
        return FALSE;
    }

    if (psIterator->bIteratorCallback != NULL)
    {
        bSuccess = psIterator->bIteratorCallback(
            psObj->tChannelID,
            psObj->tFlashEventID,
            psIterator->pvEventsIteratorCallbackArg
                );
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vFlashEventTimerCallback
*
*****************************************************************************/
static void vFlashEventTimerCallback(
    OSAL_OBJECT_HDL hTimer,
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    if (psObj->tCurrentFlashEventID != SPORTS_FLASH_INVALID_EVENT_ID)
    {
        BOOLEAN bSuccess;
        SMS_EVENT_SPORTS_FLASH_STRUCT sEvent;

        sEvent.eAction = SPORTS_FLASH_ACTION_REMAIN_FLASH_EVENT_CHANNEL;
        sEvent.uEvent.sFlashAction.tFlashEventID = psObj->tCurrentFlashEventID;

        bSuccess = DECODER_bSportsFlash(psObj->hDecoder, &sEvent);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME
                ": Failed to send Remain Flash Event Channel event");
        }
    }
}

/*****************************************************************************
*
*   bInitGameInfoEvent
*
*****************************************************************************/
static BOOLEAN bInitGameInfoEvent(
    const SPORTS_FLASH_EVENT_GAME_EVENT_STRUCT *psGameEvent,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psEventInfo
        )
{
    do 
    {
        UN32 un32ID;

        psEventInfo->tChannelID = psGameEvent->tChannelID;
        psEventInfo->un16Bias = psGameEvent->un16Bias;
        psEventInfo->bEnded = psGameEvent->bEnded;

        un32ID = psGameEvent->un8AwayLeagueID;
        psEventInfo->hAwayLeagueId = 
            LEAGUE_hCreateCid(CID_POOL_INVALID_OBJECT, &un32ID);
        if (psEventInfo->hAwayLeagueId == CID_INVALID_OBJECT)
        {
            break;
        }

        un32ID = psGameEvent->un16AwayTeamID;
        psEventInfo->hAwayTeamId = 
            TEAM_hCreateCid(CID_POOL_INVALID_OBJECT, &un32ID);
        if (psEventInfo->hAwayTeamId == CID_INVALID_OBJECT)
        {
            break;
        }

        un32ID = psGameEvent->un8HomeLeagueID;
        psEventInfo->hHomeLeagueId = 
            LEAGUE_hCreateCid(CID_POOL_INVALID_OBJECT, &un32ID);
        if (psEventInfo->hHomeLeagueId == CID_INVALID_OBJECT)
        {
            break;
        }

        un32ID = psGameEvent->un16HomeTeamID;
        psEventInfo->hHomeTeamId = 
            TEAM_hCreateCid(CID_POOL_INVALID_OBJECT, &un32ID);
        if (psEventInfo->hHomeTeamId == CID_INVALID_OBJECT)
        {
            break;
        }

        return TRUE;
    } while (FALSE);

    vUninitGameEventInfo(psEventInfo);

    return FALSE;
}

/*****************************************************************************
*
*   vReleaseFavoriteTeam
*
*****************************************************************************/
static void vReleaseFavoriteTeam(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam
        )
{
    if (psTeam != NULL)
    {
        if (psTeam->hTag != TAG_INVALID_OBJECT)
        {
            TAG_eRemove(psTeam->hTag, FALSE);
        }

        vDestroyFavoriteTeam(psTeam);
    }

    return;
}

/*****************************************************************************
*
*   bHandleStopEvent
*
*****************************************************************************/
static BOOLEAN bHandleStopEvent(
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;

    // Clear the Sports Flash service handle for this decoder
    DECODER_bSetSportsFlashHandle(psObj->hDecoder,
        SPORTS_FLASH_INVALID_OBJECT);

    // aborting the current flash event if exists
    bSuccess = bHandleAbortFlashEvent(psObj);
    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SPORTS_FLASH_OBJECT_NAME": Failed to abort flash event");
    }

    if (psObj->bSeekMonitorEnabled == TRUE)
    {
        bSuccess = RADIO_bSeekMonitorDisable(psObj->hDecoder,
            SEEK_MONITOR_TYPE_SPORTS_FLASH);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Failed to disable seek monitor");
        }
    }

    if (psObj->bGamesMonitorEnabled == TRUE)
    {
        bSuccess = RADIO_bSetGamesMonitor(psObj->hDecoder, NULL, 0);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Failed to disable games monitor");
        }
    }

    // Destroy sub-objects
    vDestroySportsFlash(psObj);

    // Destroy main object
    SMSO_vDestroy((SMS_OBJECT)psObj);

    return TRUE;
}

/*****************************************************************************
*
*   bHandleAddTeam
*
*****************************************************************************/
static BOOLEAN bHandleAddTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_TEAM_STRUCT *psTeam
        )
{
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psFavoriteTeam = NULL;
    BOOLEAN bResult = FALSE;

    do 
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        UN32 un32Value;

        eOsalReturnCode = OSAL.eLinkedListItems(psObj->hTeamsList, 
            &un32Value);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Failed to get number of "
                "favorite teams (%s)", 
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        if (un32Value >= SPORTS_FLASH_TEAMS_COUNT_MAX)
        {
            printf(SPORTS_FLASH_OBJECT_NAME": Number of teams %u is equal "
                "or more than allowed (%u)\n", un32Value, 
                SPORTS_FLASH_TEAMS_COUNT_MAX);
            break;
        }

        psFavoriteTeam = (SPORTS_FLASH_FAVORITE_TEAM_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
            SPORTS_FLASH_OBJECT_NAME":FavoriteTeam", 
            sizeof(*psFavoriteTeam), TRUE);
        if (psFavoriteTeam == NULL)
        {
            // Could not create Favorite Team object
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Cannot allocate memory for new "
                " team");
            break;
        }

        un32Value = psTeam->un8LeagueID;
        psFavoriteTeam->hLeagueId = LEAGUE_hCreateCid(
            CID_POOL_INVALID_OBJECT, &un32Value);
        if (psFavoriteTeam->hLeagueId == CID_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Cannot allocate memory for new "
                " CID");
            break;
        }

        un32Value = psTeam->un16TeamID;
        psFavoriteTeam->hTeamId = TEAM_hCreateCid(CID_POOL_INVALID_OBJECT, 
            &un32Value);
        if (psFavoriteTeam->hTeamId == CID_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Cannot allocate memory for new "
                " CID");
            break;
        }

        // add this to our list.
        eOsalReturnCode = OSAL.eLinkedListAdd(psObj->hTeamsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psFavoriteTeam);
        if (eOsalReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
        {
            // valid case
            bResult = TRUE;
            break;
        }
        else if (eOsalReturnCode != OSAL_SUCCESS)
        {
            // Unable to add team to list
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Failed to add team to "
                "list (%s)", OSAL.pacGetReturnCodeName(
                eOsalReturnCode));
            break;
        }

        vSaveTeam(psObj, psFavoriteTeam);
        psFavoriteTeam = NULL;

        bResult = bHandleUpdateTeams(psObj);
    } while (FALSE);

    vDestroyFavoriteTeam(psFavoriteTeam);

    return bResult;
}

/*****************************************************************************
*
*   bHandleRemoveTeam
*
*****************************************************************************/
static BOOLEAN bHandleRemoveTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_TEAM_STRUCT *psTeam
        )
{
    BOOLEAN bResult = FALSE;
    CID_OBJECT hTeamId = CID_INVALID_OBJECT, 
        hLeagueId = CID_INVALID_OBJECT;

    do 
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        UN32 un32Value;
        OSAL_LINKED_LIST_ENTRY hLLEntry, hNextLLEntry;
        SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psFavoriteTeam;
        SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame, *psNextGame;
        BOOLEAN bSuccess = TRUE;

        un32Value = psTeam->un8LeagueID;
        hLeagueId = LEAGUE_hCreateCid(CID_POOL_INVALID_OBJECT, &un32Value);
        if (hLeagueId == CID_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Cannot allocate memory for new "
                " CID");
            break;
        }

        un32Value = psTeam->un16TeamID;
        hTeamId = TEAM_hCreateCid(CID_POOL_INVALID_OBJECT, &un32Value);
        if (hTeamId == CID_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Cannot allocate memory for new "
                " CID");
            break;
        }

        hLLEntry = hFindInTeamsList(psObj, hLeagueId, hTeamId);
        if (hLLEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            bResult = TRUE;
            break;
        }

        psFavoriteTeam = (SPORTS_FLASH_FAVORITE_TEAM_STRUCT *)
            OSAL.pvLinkedListThis(hLLEntry);
        eOsalReturnCode = OSAL.eLinkedListRemove(hLLEntry);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Failed to remove team from "
                "the list (%s)", 
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            bResult = FALSE;
            break;
        }

        // now removing games for this team
        hLLEntry = OSAL.hLinkedListFirst(psObj->hGamesList,
            (void **)&psGame);
        while (hLLEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            hNextLLEntry = OSAL.hLinkedListNext(hLLEntry, 
                (void **)&psNextGame);

            bSuccess = bRemoveGamesByTeam(psObj, hLLEntry, psGame, 
                psFavoriteTeam);
            if (bSuccess == FALSE)
            {
                break;
            }

            hLLEntry = hNextLLEntry;
            psGame = psNextGame;
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        if (psFavoriteTeam->hTag != TAG_INVALID_OBJECT)
        {
            TAG_eRemove(psFavoriteTeam->hTag, TRUE);
        }

        vDestroyFavoriteTeam(psFavoriteTeam);

        bResult = bHandleUpdateTeams(psObj);
    } while (FALSE);

    // removing our CIDs
    if (hLeagueId != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hLeagueId);
    }

    if (hTeamId != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hTeamId);
    }

    return bResult;
}

/*****************************************************************************
*
*   bRemoveGamesByTeam
*
*****************************************************************************/
static BOOLEAN bRemoveGamesByTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY hGameLLEntry,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame,
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam
        )
{
    BOOLEAN bResult = TRUE;

    do 
    {
        N16 n16Result;
        SPORTS_FLASH_FAVORITE_TEAM_STRUCT sSearchCriteria;
        BOOLEAN bHomeTeam = FALSE, bAwayTeam = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        n16Result = CID.n16Equal(psGame->hHomeLeagueId, psTeam->hLeagueId);
        if (n16Result == 0)
        {
            // maybe we have this team in home teams?
            n16Result = CID.n16Equal(psGame->hHomeTeamId,
                psTeam->hTeamId);
            if (n16Result == 0)
            {
                bHomeTeam = TRUE;
            }
        }

        n16Result = CID.n16Equal(psGame->hAwayLeagueId, psTeam->hLeagueId);
        if (n16Result == 0)
        {
            // maybe we have this team in home teams?
            n16Result = CID.n16Equal(psGame->hAwayTeamId,
                psTeam->hTeamId);
            if (n16Result == 0)
            {
                bAwayTeam = TRUE;
            }
        }

        if ((bHomeTeam != TRUE) && (bAwayTeam != TRUE))
        {
            // just skip to next
            break;
        }

        if ((bHomeTeam == TRUE) && (bAwayTeam == TRUE))
        {
            // removing the game from the list
            eReturnCode = OSAL.eLinkedListRemove(hGameLLEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                    SPORTS_FLASH_OBJECT_NAME": Failed to remove game from "
                    "list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                bResult = FALSE;
                break;
            }

            // Notify the application about ended game
            if (psObj->bNotificationEnabled == TRUE)
            {
                psObj->vGameEventCallback(
                    psObj->hDecoder,
                    SPORTS_FLASH_PROGRAM_ENDED,
                    psGame,
                    psObj->pvGameEventCallbackArg);
            }

            // destroy game
            vDestroyGame(psGame);
            break;
        }

        if (bHomeTeam == TRUE)
        {
            // check if away team in the list of teams
            sSearchCriteria.hLeagueId = psGame->hAwayLeagueId;
            sSearchCriteria.hTeamId = psGame->hAwayTeamId;
        }
        else
        {
            // check if home team in the list of teams
            sSearchCriteria.hLeagueId = psGame->hHomeLeagueId;
            sSearchCriteria.hTeamId = psGame->hHomeTeamId;
        }

        eReturnCode = OSAL.eLinkedListSearch(psObj->hTeamsList, &hEntry,
            &sSearchCriteria);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // ok, the team was found, means that entry cannot be removed
            break;
        }
        else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Failed to search in teams list "
                "(%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            bResult = FALSE;
            break;
        }

        // removing the game from the list
        eReturnCode = OSAL.eLinkedListRemove(hGameLLEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Failed to remove game from list"
                "(%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            bResult = FALSE;
            break;
        }

        // Notify the application about ended game
        if (psObj->bNotificationEnabled == TRUE)
        {
            psObj->vGameEventCallback(
                psObj->hDecoder,
                SPORTS_FLASH_PROGRAM_ENDED,
                psGame,
                psObj->pvGameEventCallbackArg);
        }

        // destroy game
        vDestroyGame(psGame);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bHandleRemoveAllTeams
*
*****************************************************************************/
static BOOLEAN bHandleRemoveAllTeams(
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;

    do 
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        UN32 un32Count;
        OSAL_LINKED_LIST_ENTRY hEntry, hNextEntry;
        SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame, *psNextGame;
        BOOLEAN bSuccess = TRUE;

        eOsalReturnCode = OSAL.eLinkedListItems(psObj->hTeamsList, 
            &un32Count);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SPORTS_FLASH_OBJECT_NAME": Failed to get number of "
                "teams (%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        if (un32Count > 0)
        {
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(psObj->hTeamsList,
                (OSAL_LL_RELEASE_HANDLER)vReleaseFavoriteTeam);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SPORTS_FLASH_OBJECT_NAME": Failed to cleanup teams "
                    "(%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
                break;
            }

            // committing the changes
            CM_eCommitChangesToFile();
        }

        // now removing games for all the teams
        hEntry = OSAL.hLinkedListFirst(psObj->hGamesList,
            (void **)&psGame);
        while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            hNextEntry = OSAL.hLinkedListNext(hEntry, 
                (void **)&psNextGame);

            // removing the game from the list
            eOsalReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                    SPORTS_FLASH_OBJECT_NAME": Failed to remove game from "
                    "list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode));
                bSuccess = FALSE;
                break;
            }

            // Notify the application about ended game
            if (psObj->bNotificationEnabled == TRUE)
            {
                psObj->vGameEventCallback(
                    psObj->hDecoder,
                    SPORTS_FLASH_PROGRAM_ENDED,
                    psGame,
                    psObj->pvGameEventCallbackArg);
            }

            // destroy game
            vDestroyGame(psGame);

            hEntry = hNextEntry;
            psGame = psNextGame;
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        bResult = bHandleUpdateTeams(psObj);
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bHandleEnableNotificationEvent
*
*****************************************************************************/
static BOOLEAN bHandleEnableNotificationEvent(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_ENABLE_NOTIFICATION_STRUCT *psEnableNotification
        )
{
    TAG_OBJECT hDecoderTag;
    BOOLEAN bSuccess = FALSE;

    // Get the decoder tag
    hDecoderTag = DECODER_hGetTag(psObj->hDecoder);
    if (hDecoderTag != TAG_INVALID_OBJECT)
    {
        TAG_OBJECT hSportsFlashTag;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Get Sports Flash Tag
        eReturnCode = TAG_eGet(SPORTS_FLASH_OBJECT_NAME,
            hDecoderTag, &hSportsFlashTag, NULL, TRUE);
        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) && 
            (hSportsFlashTag != TAG_INVALID_OBJECT))
        {
            TAG_OBJECT hTag;

            // Get Notifications Enabled Tag
            eReturnCode = TAG_eGet(
                SPORTS_FLASH_TAG_NAME_NOTIFICATIONS_ENABLED,
                hSportsFlashTag, &hTag, NULL, TRUE);

            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                N32 n32TagValue = (N32)psEnableNotification->bEnable;

                // Set new Tag value
                TAG_eSetTagValue(hTag, TAG_TYPE_INTEGER, 
                    (void *)&n32TagValue, sizeof(N32), TRUE);

                // Keep new value in SF object
                psObj->bNotificationEnabled = psEnableNotification->bEnable;
                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bCheckGameEvent
*
*****************************************************************************/
static BOOLEAN bCheckGameEvent(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame
        )
{
    BOOLEAN bResult = FALSE;

    do 
    {
        SPORTS_FLASH_FAVORITE_TEAM_STRUCT sSearchCriteria;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        sSearchCriteria.hLeagueId = psGame->hAwayLeagueId;
        sSearchCriteria.hTeamId = psGame->hAwayTeamId;
        eOsalReturnCode = OSAL.eLinkedListSearch(psObj->hTeamsList,
            &hEntry, &sSearchCriteria);
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            bResult = TRUE;
            break;
        }
        else if (eOsalReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
             SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                 SPORTS_FLASH_OBJECT_NAME": Failed to search in teams list "
                 "(%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
             break;
        }

        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        sSearchCriteria.hLeagueId = psGame->hHomeLeagueId;
        sSearchCriteria.hTeamId = psGame->hHomeTeamId;
        eOsalReturnCode = OSAL.eLinkedListSearch(psObj->hTeamsList,
            &hEntry, &sSearchCriteria);
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            bResult = TRUE;
        }
        else if (eOsalReturnCode != OSAL_OBJECT_NOT_FOUND)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                SPORTS_FLASH_OBJECT_NAME": Failed to search in teams list "
                "(%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   vCleanupEvent
*
*****************************************************************************/
static void vCleanupEvent(
    const SMS_EVENT_SPORTS_FLASH_STRUCT *psEventData
        )
{
    if ((psEventData->eAction == SPORTS_FLASH_ACTION_SET_GAMES_MONITOR) &&
        (psEventData->uEvent.sSetGamesMonitor.ptChannelIDList != NULL))
    {
        OSAL.vMemoryFree(
            psEventData->uEvent.sSetGamesMonitor.ptChannelIDList);
    }

    return;
}
