/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SPORT_ZONE implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "decoder_obj.h"
#include "cas.h"
#include "league_obj.h"

#include "sport_zone.h"
#include "_sport_zone.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eStart
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStart (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SPORT_ZONE_SERVICE_OBJECT hService;
        SPORT_ZONE_SERVICE_OBJECT_STRUCT *psObj = NULL;

        // see if there is already a service handle for this type of
        // seek service
        hService = DECODER_hSportZoneService(hDecoder);

        if (hService != SPORT_ZONE_SERVICE_INVALID_OBJECT)
        {
            // this means one is already started
            SMSO_vUnlock((SMS_OBJECT)hDecoder);

            return SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
        }

        do
        {
            const char *pacName;
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
            OSAL_RETURN_CODE_ENUM eReturn;
            BOOLEAN bOk;

            // get the Decoder's name
            pacName = DECODER_pacName(hDecoder);
            if (pacName == NULL)
            {
                break;
            }

            // Create the name for this seek service object
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     SPORT_ZONE_OBJECT_NAME":%s", pacName);
            psObj = (SPORT_ZONE_SERVICE_OBJECT_STRUCT *)
                SMSO_hCreate(
                    &acName[0],
                    sizeof(SPORT_ZONE_SERVICE_OBJECT_STRUCT),
                    (SMS_OBJECT)hDecoder,
                    FALSE
                        );

            if(psObj == NULL)
            {
                // Error!
                break;
            }

            // Create the list
            eReturn =
                OSAL.eLinkedListCreate(
                    & psObj->hZoneList,
                    SPORT_ZONE_OBJECT_NAME":ZoneList",
                    NULL,
                    OSAL_LL_OPTION_LINEAR
                        );
            if(eReturn != OSAL_SUCCESS)
            {
                // Error!
                break;
            }

            bOk = DECODER_bSetSportZoneServiceHandle(hDecoder,
                      (SPORT_ZONE_SERVICE_OBJECT)psObj);
            if(bOk == FALSE)
            {
                // Error!
                break;
            }

            SMSO_vUnlock((SMS_OBJECT)hDecoder);

            puts(SPORT_ZONE_OBJECT_NAME": Created service\n");

            // success
            return SMSAPI_RETURN_CODE_SUCCESS;

        } while(0);

        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        // Something went wrong
        SPORT_ZONE_vStop((SPORT_ZONE_SERVICE_OBJECT)psObj);
    }

    return eResult;
}

/*****************************************************************************
*
*   vStop
*
*****************************************************************************/
static void vStop (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SPORT_ZONE_SERVICE_OBJECT hService;

        // see if there is a service handle for this type of seek service
        hService = DECODER_hSportZoneService(hDecoder);
        if (hService != SPORT_ZONE_SERVICE_INVALID_OBJECT)
        {
            SPORT_ZONE_vStop(hService);
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
}

/*****************************************************************************
*
*   eIterate
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate (
    DECODER_OBJECT hDecoder,
    SPORT_ZONE_ITERATION_HANDLER bIterator,
    void *pvArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        SPORT_ZONE_SERVICE_OBJECT hSportZoneService;

        hSportZoneService = DECODER_hSportZoneService(hDecoder);
        if (hSportZoneService != SPORT_ZONE_SERVICE_INVALID_OBJECT)
        {
            SPORT_ZONE_SERVICE_OBJECT_STRUCT *psObj =
                (SPORT_ZONE_SERVICE_OBJECT_STRUCT *)hSportZoneService;

            // Iterate all zones
            eOsalReturnCode =
                OSAL.eLinkedListIterate(
                    psObj->hZoneList,
                    (OSAL_LL_ITERATOR_HANDLER)bIterator,
                    pvArg
                        );
            if( (eOsalReturnCode == OSAL_SUCCESS) ||
                (eOsalReturnCode == OSAL_NO_OBJECTS))
            {
                // List was iterated
                eResult = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eResult;
}

/*****************************************************************************
*
*   hCreate
*
*****************************************************************************/
static SPORT_ZONE_OBJECT hCreate(
    DECODER_OBJECT hDecoder,
    LEAGUE_OBJECT hLeague
      )
{
    SPORT_ZONE_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eResult;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    STRING_OBJECT hLongCategoryName = STRING_INVALID_OBJECT,
                  hLeagueAbbrev = STRING_INVALID_OBJECT;
    static UN32 un32Instance = 0;
    size_t tLeagueAbbrevSize, tLongCatNameSize;
    char *pacLongCatName, *pacShortCatName;
    CID_OBJECT hLeagueId = CID_INVALID_OBJECT;

    // Verify inputs are correct
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);

    if (bValid == FALSE) // Valid Decoder?
    {
        // Error!
        return SPORT_ZONE_INVALID_OBJECT;
    }
    else
    {
        OSAL_LINKED_LIST_ENTRY hLLEntry;

        hLLEntry = hFindInZoneList(hDecoder, hLeague);
        if (hLLEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // exists, extract the data from this entry
            return (SPORT_ZONE_OBJECT)OSAL.pvLinkedListThis(hLLEntry);
        }
    }

    do
    {
        // Construct a unique name for this list.
        snprintf(&acName[0],
            sizeof(acName),
            SPORT_ZONE_OBJECT_NAME":Obj:%u",
            un32Instance++);

        // Create a CAL object
        psObj = (SPORT_ZONE_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(SPORT_ZONE_OBJECT_STRUCT),
                (SMS_OBJECT)hDecoder,
                FALSE
                    );
        if(psObj == NULL)
        {
            // Error!
            break;
        }

        psObj->hLeague = hLeague;

        // Create the CAL for this zone
        psObj->hCAL = CAS_hCreateList(
                          hDecoder,
                          vCALCallback,
                          (void*)(SPORT_ZONE_OBJECT)psObj,
                          TRUE // auto ack
                              );
        if (psObj->hCAL == CAL_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        hLeagueAbbrev = LEAGUE.hAbbreviation(hLeague);
        if (hLeagueAbbrev == STRING_INVALID_OBJECT)
        {
            // Error
            break;
        }

        tLeagueAbbrevSize = STRING.tSize(hLeagueAbbrev) +1; // add 1 for NULL
        // create a string. we know we are going to be appending this string,
        // so we'll make the min size large enough to hold it all.
        hLongCategoryName = STRING.hCreate(SPORT_ZONE_LONG_CAT_NAME_PREFIX,
                                tLeagueAbbrevSize+
                                strlen(SPORT_ZONE_LONG_CAT_NAME_PREFIX)+
                                strlen(SPORT_ZONE_LONG_CAT_NAME_SUFFIX));
        if (hLongCategoryName == STRING_INVALID_OBJECT)
        {
            // Error
            break;
        }

        STRING.tAppendString(hLeagueAbbrev, hLongCategoryName);
        STRING.tAppendCStr(hLongCategoryName, " Zone");

        tLongCatNameSize = STRING.tSize(hLongCategoryName) +1; // +1 for NULL

        // allocate memory for the buffer
        pacLongCatName =
            (char *) SMSO_hCreate(
                SPORT_ZONE_OBJECT_NAME":CSTR",
                tLongCatNameSize+tLeagueAbbrevSize,
                SMS_INVALID_OBJECT, FALSE // No lock necessary
                );
        if (pacLongCatName != NULL)
        {
            size_t tCopy;
            // copy to our buffer
            tCopy = STRING.tCopyToCStr(hLongCategoryName, pacLongCatName,
                                       tLongCatNameSize);
            pacShortCatName = pacLongCatName + tCopy;
            (void)STRING.tCopyToCStr(hLeagueAbbrev, pacShortCatName,
                                       tLeagueAbbrevSize);
        }
        else
        {
            // Error
            break;
        }

        // attach category to CAL
        eResult = CAL.eAttachCategory (
            psObj->hCAL,
            CATEGORY_INVALID_ID,
            SPORT_ZONE_CAL_AUTO_ADD_OPTIONS,
            pacLongCatName,
            pacShortCatName
                );

        // destroy memory allocated for names
        SMSO_vDestroy((SMS_OBJECT)pacLongCatName);

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error
            break;
        }

        hLeagueId = LEAGUE.hId(hLeague);

        eResult = CAL.eRegister(psObj->hCAL, hLeagueId, NULL,
                      ( SMS_EVENT_REGISTRATION_OPTION_INITIAL |
                        SMS_EVENT_REGISTRATION_OPTION_CURRENT |
                        SMS_EVENT_REGISTRATION_OPTION_END |
                        SMS_EVENT_REGISTRATION_OPTION_RETRIGGER ),
                      hLongCategoryName,
                      hLeagueAbbrev,
                      NULL);
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error
            break;
        }

        // done with this string we created
        STRING.vDestroy(hLongCategoryName);
        hLongCategoryName = STRING_INVALID_OBJECT;

        eResult = eAddToZoneList(hDecoder, (SPORT_ZONE_OBJECT)psObj);
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error
            break;
        }

        puts(SPORT_ZONE_OBJECT_NAME": Created SportZone");
        // Return CAL object to caller
        return (SPORT_ZONE_OBJECT)psObj;

    } while(0);

    // Something went wrong
    if (hLongCategoryName != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hLongCategoryName);
    }

    vDestroy((SPORT_ZONE_OBJECT)psObj);

    return SPORT_ZONE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDestroy
*
*****************************************************************************/
static void vDestroy(
    SPORT_ZONE_OBJECT hSportZone
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSportZone, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        LEAGUE_OBJECT hLeague;
        DECODER_OBJECT hDecoder;

        // decode is the zone's parent
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)hSportZone);
        hLeague = SPORT_ZONE.hLeague(hSportZone);

        // remove the zone from the service's list of zones
        vRemove(hDecoder, hLeague);

        // destroy the zone
        vDestroyZone(hSportZone);

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
*
*   tCategoryId
*
*****************************************************************************/
static CATEGORY_ID tCategoryId(
    SPORT_ZONE_OBJECT hSportZone
        )
{
    SPORT_ZONE_OBJECT_STRUCT *psObj =
        (SPORT_ZONE_OBJECT_STRUCT *)hSportZone;
    BOOLEAN bLocked;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Verify and lock SMS Object (CAL)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSportZone, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        // At this point we have exclusive access to the CAL
        tCategoryId = CAL.tCategoryId(psObj->hCAL);

        SMSO_vUnlock((SMS_OBJECT)hSportZone);
    }

    return tCategoryId;
}

/*****************************************************************************
*
*   hLeague
*
*****************************************************************************/
static LEAGUE_OBJECT hLeague(
    SPORT_ZONE_OBJECT hSportZone
        )
{
    SPORT_ZONE_OBJECT_STRUCT *psObj =
        (SPORT_ZONE_OBJECT_STRUCT *)hSportZone;
    BOOLEAN bLocked;
    LEAGUE_OBJECT hLeague = LEAGUE_INVALID_OBJECT;

    // Verify and lock SMS Object (CAL)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSportZone, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        // At this point we have exclusive access to the CAL
        hLeague = psObj->hLeague;

        SMSO_vUnlock((SMS_OBJECT)hSportZone);
    }

    return hLeague;
}

/*****************************************************************************
*
*   SPORT_ZONE_vStop
*
*****************************************************************************/
void SPORT_ZONE_vStop(
    SPORT_ZONE_SERVICE_OBJECT hService
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hService, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SPORT_ZONE_SERVICE_OBJECT_STRUCT *psObj =
            (SPORT_ZONE_SERVICE_OBJECT_STRUCT *)hService;
        DECODER_OBJECT hDecoder;

        // All SPORT_ZONE_SERVICE objs. belong to DECODER object (their parent)
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

        // Destroy LL
        if (psObj->hZoneList != OSAL_INVALID_OBJECT_HDL)
        {
            // Remove all entries from the list and destroy each
            OSAL.eLinkedListRemoveAll(
                psObj->hZoneList,
                (OSAL_LL_RELEASE_HANDLER)vDestroyZone
                    );

            // Destroy the list itself
            OSAL.eLinkedListDelete(psObj->hZoneList);
            psObj->hZoneList = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy the object itself
        SMSO_vDestroy((SMS_OBJECT)hService);

        hService = SPORT_ZONE_SERVICE_INVALID_OBJECT;
        DECODER_bSetSportZoneServiceHandle(hDecoder, hService);

        puts(SPORT_ZONE_OBJECT_NAME": Destroyed service\n");

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
*
*   eAddToZoneList
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM eAddToZoneList(
    DECODER_OBJECT hDecoder,
    SPORT_ZONE_OBJECT hSportZone)
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SPORT_ZONE_SERVICE_OBJECT hSportZoneService;
        OSAL_RETURN_CODE_ENUM eOSALResult;

        hSportZoneService = DECODER_hSportZoneService(hDecoder);

        if (hSportZoneService != SPORT_ZONE_SERVICE_INVALID_OBJECT)
        {
            SPORT_ZONE_SERVICE_OBJECT_STRUCT *psObj =
                (SPORT_ZONE_SERVICE_OBJECT_STRUCT *)hSportZoneService;

            // add this to our list.
            eOSALResult =
                OSAL.eLinkedListAdd(psObj->hZoneList, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hSportZone);

            if (eOSALResult == OSAL_SUCCESS)
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vRemove
*
*****************************************************************************/
void vRemove(
    DECODER_OBJECT hDecoder,
    LEAGUE_OBJECT hLeague
        )
{
    BOOLEAN bLocked;

    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_LINKED_LIST_ENTRY hLLEntry;

        hLLEntry = hFindInZoneList(hDecoder, hLeague);

        if (hLLEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL.eLinkedListRemove(hLLEntry);
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
}

/*****************************************************************************
*
*   hFindInZoneList
*
*****************************************************************************/
OSAL_LINKED_LIST_ENTRY hFindInZoneList(
    DECODER_OBJECT hDecoder,
    LEAGUE_OBJECT hLeague
        )
{
    BOOLEAN bLocked;
    OSAL_LINKED_LIST_ENTRY hLLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SPORT_ZONE_SERVICE_OBJECT hSportZoneService;

        hSportZoneService = DECODER_hSportZoneService(hDecoder);

        if (hSportZoneService != SPORT_ZONE_SERVICE_INVALID_OBJECT)
        {
            SPORT_ZONE_SERVICE_OBJECT_STRUCT *psObj =
                (SPORT_ZONE_SERVICE_OBJECT_STRUCT *)hSportZoneService;

            CID_OBJECT hLeagueId;
            hLeagueId = LEAGUE.hId(hLeague);

            // LL search
            OSAL.eLinkedListLinearSearch(
                psObj->hZoneList,
                &hLLEntry,
                (OSAL_LL_COMPARE_HANDLER)n16FindZoneByLeague,
                (void*)hLeagueId
                  );
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return hLLEntry;
}

/*****************************************************************************
*
*   n16FindZoneByLeague
*
*****************************************************************************/
static N16 n16FindZoneByLeague( void *pvArg1, void *pvArg2 )
{
    CID_OBJECT hId1, hId2;
    SPORT_ZONE_OBJECT_STRUCT *psObj = (SPORT_ZONE_OBJECT_STRUCT *)pvArg1;

	hId2 = (CID_OBJECT)pvArg2;
    hId1 = LEAGUE.hId(psObj->hLeague);

    return CID.n16Equal(hId1, hId2);
}

/*****************************************************************************
*
*   vCALCallback
*
*****************************************************************************/
static void vCALCallback(
    CAL_OBJECT hCAL,
    CAL_ALERT_OBJECT hAlert,
    UN32 un32Flags,
    void *pvCALCallbackArg
        )
{
    CAL_eAcknowledge(hCAL);
}

/*****************************************************************************
*
*   vDestroyZone
*
*****************************************************************************/
static void vDestroyZone(
    SPORT_ZONE_OBJECT hSportZone
        )
{
    BOOLEAN bValid;
    SPORT_ZONE_OBJECT_STRUCT *psObj = (SPORT_ZONE_OBJECT_STRUCT *)hSportZone;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT)hSportZone);
    if(bValid == TRUE)
    {
        // Destroy CAL
        if (psObj->hCAL != CAL_INVALID_OBJECT)
        {
            CAS.vDestroyList(psObj->hCAL);
            psObj->hCAL = CAL_INVALID_OBJECT;
        }

        // uninit league info
        psObj->hLeague = LEAGUE_INVALID_OBJECT;

        // Destroy the object itself
        SMSO_vDestroy((SMS_OBJECT)psObj);

        puts(SPORT_ZONE_OBJECT_NAME": Destroyed Sport Zone");
    }

    return;
}
