/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SP_SEEK_CONTENT implementation for the
 *  Satellite Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"

#include "decoder_obj.h"
#include "string_obj.h"

#include "seek.h"
#include "sp_seek.h"
#include "sp_seek_content.h"
#include "_sp_seek_content.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hRegisteredName
*
*****************************************************************************/
static STRING_OBJECT hRegisteredName (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    STRING_OBJECT hNameText = STRING_INVALID_OBJECT;

    // verify the content is associated with the Sports Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_SPORTS)
    {
        // get the text. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hNameText = CAL_CONTENT.hArtistText((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hNameText;
}

/*****************************************************************************
*
*   hRegisteredNickname
*
*****************************************************************************/
static STRING_OBJECT hRegisteredNickname (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    STRING_OBJECT hText = STRING_INVALID_OBJECT;

    // verify the content is associated with the Sports Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_SPORTS)
    {
        // get the text. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hText = CAL_CONTENT.hTitleText((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hText;
}

/*****************************************************************************
*
*   hRegisteredLeagueId
*
*****************************************************************************/
static CID_OBJECT hRegisteredLeagueId (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    CID_OBJECT hLeagueId = CID_INVALID_OBJECT;

    // verify the content is associated with the Sports Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_SPORTS)
    {
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
        SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psSPSeekContentItem;

        psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
        psSPSeekContentItem = (SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                                  &psRegisteredItem->uServiceSpecific;

        // verify that the pointer to the data isn't NULL
        if ( psSPSeekContentItem != NULL )
        {
            hLeagueId = psSPSeekContentItem->hLeagueCID;
        }
    }

    return hLeagueId;
}

/*****************************************************************************
*
*   hRegisteredTeamId
*
*****************************************************************************/
static CID_OBJECT hRegisteredTeamId (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    CID_OBJECT hTeamId = CID_INVALID_OBJECT;

    // verify the content is associated with the Sports Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_SPORTS)
    {
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
        SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psSPSeekContentItem;

        psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
        psSPSeekContentItem = (SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                                  &psRegisteredItem->uServiceSpecific;

        // verify that the pointer to the data isn't NULL
        if ( psSPSeekContentItem != NULL )
        {
            hTeamId = psSPSeekContentItem->hTeamCID;
        }
    }

    return hTeamId;
}


/*****************************************************************************
*
*   eType
*
*****************************************************************************/
static SP_SEEK_ENUM eType (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{

    SEEK_SERVICE_ENUM eService;
    SP_SEEK_ENUM eType = SP_SEEK_UNKNOWN;

    // verify that this content is from an sports seek service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_SPORTS)
    {
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
        SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psSPSeekContentItem;

        psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
        psSPSeekContentItem = (SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                                  &psRegisteredItem->uServiceSpecific;

        // verify that the pointer to the data isn't NULL
        if ( psSPSeekContentItem != NULL )
        {
            // extract the type
            eType = psSPSeekContentItem->eType;
        }
    }
    return eType;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SP_SEEK_CONTENT_vSetServiceSpecificInfo
*
* This function is used to populate the SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT
*
*****************************************************************************/
void SP_SEEK_CONTENT_vSetServiceSpecificInfo(
    void *pvItem,
    CID_OBJECT hLeagueCID,
    CID_OBJECT hTeamCID,
    SP_SEEK_ENUM eType,
    SEEK_SERVICE_OBJECT hSeekService
        )
{
    SEEK_SERVICE_ENUM eService;

    eService = SEEK_eService(hSeekService);
    if (eService == SEEK_SERVICE_SPORTS)
    {
        SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psSPSpecificInfo;
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psGenericItem;

        psGenericItem = (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)pvItem;
        psSPSpecificInfo = (SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                               &psGenericItem->uServiceSpecific;

        // populate the service specific portion
        psSPSpecificInfo->hLeagueCID = CID.hDuplicate(hLeagueCID);
        psSPSpecificInfo->hTeamCID = CID.hDuplicate(hTeamCID);
        psSPSpecificInfo->eType = eType;
    }

    return;
}

/*****************************************************************************
*
*   SP_SEEK_CONTENT_hCID
*
*****************************************************************************/
CID_OBJECT SP_SEEK_CONTENT_hCID (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    CID_OBJECT hCID = CID_INVALID_OBJECT;

    // verify the content is associated with the Sports Seek Service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_SPORTS)
    {
        // get the id. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hCID = CAL_CONTENT.hCID((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hCID;
}

/*****************************************************************************
*
*    SP_SEEK_CONTENT_pacSPSeekTypeText
*
*    This is a local function which simply maps an enumerated type to
*    a textual representation for formatting the enumerated type.
*
*****************************************************************************/
const char *SP_SEEK_CONTENT_pacSPSeekTypeText( SP_SEEK_ENUM eType )
{
    const char *pacReturnString;

    // based on the seek state, get the text that describes it
    switch (eType)
    {
        case SP_SEEK_TEAM:
            pacReturnString = MACRO_TO_STRING(SP_SEEK_TEAM);
        break;

        case SP_SEEK_LEAGUE:
            pacReturnString = MACRO_TO_STRING(SP_SEEK_LEAGUE);
        break;

        case SP_SEEK_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(SP_SEEK_UNKNOWN);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n32FPrintf
*
*    This is used to print out to a file human readable service specific
*    information about the items registered to the seek service
*
*****************************************************************************/
static N32 n32FPrintf (
    void *pvServiceSpecificData,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32NumWritten = 0;
    SEEK_CONTENT_UNION *puServiceSpecificData =
            (SEEK_CONTENT_UNION *)pvServiceSpecificData;

    SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegItem =
            (SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT*)puServiceSpecificData;

    // check input
    if ((pvServiceSpecificData == NULL) || (psFile == NULL))
    {
        return EOF;
    }

    switch(eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            // print out what type of sports seek this is
            n32NumWritten += fprintf(psFile, "%s",
                        SP_SEEK_CONTENT_pacSPSeekTypeText(psRegItem->eType));
        }
        break;

        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        default:
        {
            // print out what type of sports seek this is
            n32NumWritten +=
                fprintf(psFile, "%s\n",
                SP_SEEK_CONTENT_pacSPSeekTypeText(psRegItem->eType));

            // print out the League CID
            n32NumWritten +=
                fprintf(psFile, "League Id ");
            n32NumWritten +=
                CID.n32FPrintf(psRegItem->hLeagueCID, psFile);

            if (psRegItem->eType == SP_SEEK_TEAM)
            {
                // print out the Team CID
                n32NumWritten +=
                    fprintf(psFile, "Team Id ");
                n32NumWritten +=
                    CID.n32FPrintf(psRegItem->hTeamCID, psFile);
            }
        }
        break;
    }
    return n32NumWritten;
}

/*****************************************************************************
*
*   vUnInit
*
*****************************************************************************/
static void vUnInit(
    void *pvServiceSpecific
        )
{
    // check input
    if (pvServiceSpecific != NULL)
    {
        SEEK_CONTENT_UNION *puServiceSpecific =
            (SEEK_CONTENT_UNION *)pvServiceSpecific;

        SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegItem =
            (SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT*)puServiceSpecific;

        // destroy and uninitialize the CIDs
        if (psRegItem->hLeagueCID != CID_INVALID_OBJECT)
        {
            CID.vDestroy(psRegItem->hLeagueCID);
            psRegItem->hLeagueCID = CID_INVALID_OBJECT;
        }
        if (psRegItem->hTeamCID != CID_INVALID_OBJECT)
        {
            CID.vDestroy(psRegItem->hTeamCID);
            psRegItem->hTeamCID = CID_INVALID_OBJECT;
        }

        psRegItem->eType = SP_SEEK_UNKNOWN;
    }
}

/*****************************************************************************
*
*   bSearchForCurrent
*
*****************************************************************************/
static BOOLEAN bSearchForCurrentAfterEnable (
    void *pvServiceSpecific
        )
{
    // we do want to search for current
    return TRUE;
}
