/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SP_SEEK implementation for the
 *  Satellite Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "ccache.h"
#include "cas.h"
#include "cal_alert.h"
#include "decoder_obj.h"
#include "league_obj.h"
#include "team_obj.h"
#include "cm.h"

#include "seek.h"
#include "seek_content.h"
#include "sp_seek_content.h"
#include "league_team_id.h"
#include "sp_seek.h"
#include "_sp_seek.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
* eAvailableToRegister
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAvailableToRegister (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    SEEK_AVAILABILTY_ENUM *peAvailablility,
    SP_SEEK_ENUM eType,
    ...
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    UN8 un8Index = 0;

    if (eType == SP_SEEK_TEAM)
    {
      // the user is registering for team seek so we need to index into
      // the CDO's Team list
      va_list tTeamArg;
      va_start( tTeamArg, eType );
      un8Index = (UN8)va_arg( tTeamArg, int );
      va_end( tTeamArg );
    }
    else
    {
        // if not seeking team, must be seeking league
        if (eType != SP_SEEK_LEAGUE)
        {
            // can't register this
            return SMSAPI_RETURN_CODE_INVALID_INPUT;
        }
    }

    // did the caller provide us with a valid pointer?
    if ( peAvailablility == NULL )
    {
        // Error!
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock the object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // Get this seek service's handle from the provided DECODER
        eReturnCode = DECODER_eSeekService(hDecoder,
                          SEEK_SERVICE_SPORTS, &hSeekService);

        // did we successfully get the service handle?
        if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
        {
            CD_OBJECT hCDO;
            CID_OBJECT hCID;
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // get the CDO for this cahnnel
            hCDO = hGetCDO(hDecoder, tChannelId);

            if (hCDO == CD_INVALID_OBJECT)
            {
                // we got an invalid CDO so we cannot identify content
                // so seeks are currently unavailable for this channel.
                *peAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;

            }
            else
            {
                // we got a valid sports CDO.

                // now dig into it and get a CID
                hCID = hRetrieveCidForSeeks(hCDO, eType, un8Index);

                do
                {
                    BOOLEAN bRegistered = FALSE;

                    // did we retrieve a valid CID?
                    if (hCID == CID_INVALID_OBJECT)
                    {
                        // no, so we cannot identify the team/league
                        *peAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;
                        break;
                    }

                    // we got a valid CID by which we can identify the
                    // team/league.

                    // now see if the team/league is registered already
                    bRegistered = CAL.bExists(
                                      psObj->hSeekList,
                                      hCID);

                    if (bRegistered == TRUE)
                    {
                        // already registered
                        *peAvailablility =
                            SEEK_AVAILIBILITY_ALREADY_REGISTERED;
                    }
                    else
                    {
                        // not already registered
                        *peAvailablility = SEEK_AVAILABILTY_AVAILABLE;
                    }

                    // In case of SP Seek Type: 'Team', a synthetic CID will 
                    // be created (a CID_LEAGUE_TEAM_ID) based on Team and League CIDs.
                    // So once it has been checked for existence, we don't need it anymore.
                    if (eType == SP_SEEK_TEAM)
                    {
                        // We dont' need the CID
                        CID_vDestroy(hCID);
                    }

                } while(FALSE);
            }
        }
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eRegister
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegister (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    SP_SEEK_ENUM eType,
    SEEK_CONTENT_OBJECT *phSeekContent,
    ...
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    UN8 un8Index = 0;

    if (eType == SP_SEEK_TEAM)
    {
      // the user is registering for team seek so we need to index into
      // the CDO's Team list
      va_list tTeamArg;
      va_start( tTeamArg, phSeekContent );
      un8Index = (UN8)va_arg( tTeamArg, int );
      va_end( tTeamArg );
    }
    else
    {
        // if not seeking team, must be seeking league
        if (eType != SP_SEEK_LEAGUE)
        {
            // can't register this
            return SMSAPI_RETURN_CODE_INVALID_INPUT;
        }
    }

    // Lock the service
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        do
        {
            CD_OBJECT hCDO;
            SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
            SEEK_SERVICE_OBJECT_STRUCT *psObj;
            LEAGUE_OBJECT hLeague;

            eReturnCode =
                DECODER_eSeekService(hDecoder,
                    SEEK_SERVICE_SPORTS, &hSeekService);

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            psObj = (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // get the channel's CDO
            hCDO = hGetCDO(hDecoder, tChannelId);

            if (hCDO == CD_INVALID_OBJECT)
            {
                // failure to get valid sports CDO
                eReturnCode =  SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
                break;
            }

            // get the LEAGUE
            hLeague = SPORTS.hLeague(hCDO);

            if (eType == SP_SEEK_TEAM)
            {
                // caller wants team seek, so get the TEAM at the
                // requested index
                TEAM_OBJECT hTeam;
                CID_OBJECT hLeagueId;

                hTeam = SPORTS.hTeam(hCDO, un8Index);
                hLeagueId = LEAGUE.hId(hLeague);
                // register the team
                eReturnCode = eRegisterTeamLocal(psObj, hLeagueId, hTeam, phSeekContent);
            }
            else
            {
                // register the league
                eReturnCode = eRegisterLeagueLocal(psObj, hLeague, phSeekContent);
            }

        } while (0);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eRegisterTeam
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegisterTeam (
    DECODER_OBJECT hDecoder,
    LEAGUE_OBJECT hLeague,
    TEAM_OBJECT hTeam,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Lock the service
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // get the seek service handle
        eReturnCode =
            DECODER_eSeekService(hDecoder, SEEK_SERVICE_SPORTS, &hSeekService);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;
            CID_OBJECT hLeagueId;

            hLeagueId = LEAGUE.hId(hLeague);

            // register this team
            eReturnCode = eRegisterTeamLocal(psObj, hLeagueId, hTeam, phSeekContent);
        }
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eRegisterLeague
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegisterLeague (
    DECODER_OBJECT hDecoder,
    LEAGUE_OBJECT hLeague,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Lock the service
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // get the seek service handle
        eReturnCode =
            DECODER_eSeekService(hDecoder, SEEK_SERVICE_SPORTS, &hSeekService);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // register this league
            eReturnCode = eRegisterLeagueLocal(psObj, hLeague, phSeekContent);
        }
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*   bHandleSeekAlert
*
* This is the specific seek service interface function used to
* handle seek alerts
*
* Returns TRUE if a Seek Alert Event should be generated. FALSE if not
*****************************************************************************/
static BOOLEAN bHandleSeekAlert(
    SEEK_SERVICE_OBJECT hSeek,
    CAL_ALERT_OBJECT hAlert
        )
{
    BOOLEAN bSetDecoderSeekAlertEvent = FALSE;
    CAL_CONTENT_OBJECT hContent;
    CHANNEL_OBJECT hChannel;

    // get the content and channel that caused the alert
    hContent = CAL_ALERT.hContent(hAlert);
    hChannel = CAL_ALERT.hChannel(hAlert);

    // should we allow the seek alert
    bSetDecoderSeekAlertEvent = bSeekAlertAllowed(
                                 hSeek,
                                 hContent,
                                 hChannel
                                     );

    return bSetDecoderSeekAlertEvent;
}

/*****************************************************************************
*   bSeekAlertAllowed
*
* This function implements the rules to determine if the application should be
* informed of a content match via a seek alert
*
* Returns TRUE is a Seek Alert is allowed, FALSE if it is not
*****************************************************************************/
static BOOLEAN bSeekAlertAllowed(
    SEEK_SERVICE_OBJECT hSeek,
    CAL_CONTENT_OBJECT hContent,
    CHANNEL_OBJECT hChannel
        )
{
    // we start out assuming the alert is allowed, then we'll check the rules
    // to see if it is not allowed
    BOOLEAN bValid, bEnabled, bAlertAllowed = TRUE;
    SP_SEEK_ENUM eType;
    CHANNEL_ID tCurrentlyTunedChannelId, tChannelId;
    SEEK_SERVICE_OBJECT_STRUCT *psObj;
    DECODER_OBJECT hDecoder;

    bValid = SMSO_bValid((SMS_OBJECT)hSeek);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    psObj = (SEEK_SERVICE_OBJECT_STRUCT*)hSeek;

    do
    {
        // 1. The service must be enabled for a seek alert to be allowed
        printf(SP_SEEK_OBJECT_NAME": Service is :%s\n",
               psObj->eState == SEEK_STATE_ENABLED ? "ENABLED" : "NOT_ENABLED");
        // is service enabled?
        if (psObj->eState != SEEK_STATE_ENABLED)
        {
            // service is disabled, so seek alert is not allowed
            bAlertAllowed = FALSE;
            break;
        }

        // 2. The registered item must be enabled for a seek alert to be allowed
        bEnabled = SEEK_CONTENT.bEnabled(hContent);
        printf(SP_SEEK_OBJECT_NAME":Item is :%s\n",
               bEnabled == TRUE ? "ENABLED" : "DISABLED");
        // is item enabled?
        if (bEnabled == FALSE)
        {
            // item is disabled, so seek alert is not allowed
            bAlertAllowed = FALSE;
            break;
        }

        // 3. If it's not indicated otherwise the matched content should not 
        //    be occurring on the tuned channel for a seek alert to be allowed

        // All SEEK_SERVICE objects belong to a DECODER object (their parent)
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

        // Are alerts allowed on the tuned channel?
        bEnabled = SEEK.bGetAlertsTunedStatus(hDecoder, SEEK_SERVICE_SPORTS);
        if (bEnabled == FALSE)
        {
            tCurrentlyTunedChannelId = DECODER.tCurrentChannelId(hDecoder);
            tChannelId = CHANNEL.tChannelId(hChannel);
            printf(SP_SEEK_OBJECT_NAME": Content %s currently tuned channel\n",
                tCurrentlyTunedChannelId == tChannelId ? "ON" : "NOT ON");
            // is the content on the currently tuned channel?
            if (tCurrentlyTunedChannelId == tChannelId)
            {
                // content occurred on the currently tuned channel, so seek alert
                // is not allowed
                bAlertAllowed = FALSE;
                break;
            }
        }

        // is this a league seek?
        eType = SP_SEEK_CONTENT.eType(hContent);
        if (eType == SP_SEEK_LEAGUE)
        {
        // 4. If this is a LEAGUE ALERT, make sure no TEAM's on this channel
        // are also registered AND enabled
            TEAM_OBJECT hTeam;
            CD_OBJECT hCDO;
            UN8 un8Index, un8NumTeams;
            SP_SEEK_ITERATOR_STRUCT sIteratorStruct;
            sIteratorStruct.bMatch = FALSE;

            hCDO = CHANNEL.hCDO(hChannel);
            un8NumTeams = SPORTS.un8Teams(hCDO);

            // for each team in this broadcast, seek if it is
            // registered AND enabled
            for (un8Index = 0; un8Index < un8NumTeams; un8Index++)
            {
                hTeam = SPORTS.hTeam(hCDO, un8Index);
                sIteratorStruct.hId = TEAM.hId(hTeam);

                // iterate this service searching for a match
                SEEK.eIterate(hDecoder,
                    SEEK_SERVICE_SPORTS,
                    bSportSeekContentIterator,
                    &sIteratorStruct);

                if (sIteratorStruct.bMatch == TRUE)
                {
                    puts(SP_SEEK_OBJECT_NAME
                         ": Suppressing league seek - team is also registered "
                         "and enabled.");
                    // yes one of the teams is registered AND enabled, so we
                    // will not allow this alert to be passed on to the app.
                    bAlertAllowed = FALSE;
                    break;
                }
            }
        }

    } while (0);

    printf(SP_SEEK_OBJECT_NAME": Seek Alert %s \n",
           bAlertAllowed == TRUE ? "ALLOWED" : "NOT ALLOWED");

    return bAlertAllowed;
}

/*****************************************************************************
*   bInit
*
* This function implements the service specific initialization
*
* Returns TRUE if successful, FALSE if not
*****************************************************************************/
static BOOLEAN bInit(
    SEEK_SERVICE_OBJECT hSeek
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    BOOLEAN bValid, bReturn = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hSeek);
    if (bValid == TRUE)
    {
        SEEK_SERVICE_OBJECT_STRUCT *psObj;

        psObj = (SEEK_SERVICE_OBJECT_STRUCT*)hSeek;

        // Attach a category to the CAL
        eResult = CAL.eAttachCategory (
            psObj->hSeekList,
            CATEGORY_INVALID_ID,
            SP_SEEK_CAL_AUTO_ADD_OPTIONS,
            "My Games",
            "Games"
                );

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            bReturn = FALSE;
        }
        else
        {
            // re-register content that is in the config file
            bReturn = bRegisterContentFromConfigFile(psObj);
        }
    }

    return bReturn;
}

/*****************************************************************************
*   eRegisterTeamLocal
*
* This local function registers the team with the service
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegisterTeamLocal(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    CID_OBJECT hLeagueId,
    TEAM_OBJECT hTeam,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bOk;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem = NULL;
    STRING_OBJECT hTeamNameString = STRING_INVALID_OBJECT,
                  hTeamNicknameString = STRING_INVALID_OBJECT;

    // You must have a valid league-id and team object
    if((hLeagueId == CID_INVALID_OBJECT) || (hTeam == TEAM_INVALID_OBJECT))
    {
        // Error!
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        CID_OBJECT hTeamId, hLeagueTeamId;
        LEAGUE_TEAM_ID_STRUCT sLeagueTeamId;

        hTeamId = TEAM.hId(hTeam);
        if (hTeamId == CID_INVALID_OBJECT)
        {
            // can't register without a valid CID
            break;
        }

        // Take league-id and team-id and make a CID
        sLeagueTeamId.hLeagueId = hLeagueId;
        sLeagueTeamId.hTeamId = hTeamId;

        hLeagueTeamId = CID_hCreate(
            CID_POOL_INVALID_OBJECT, CID_LEAGUE_TEAM_ID, &sLeagueTeamId);

        // get the strings used to describe the team being registered.
        hTeamNameString = TEAM.hName(hTeam);
        hTeamNicknameString = TEAM.hNickname(hTeam);

        // register the team
        psItem = psRegisterContent(
            psObj,
            hLeagueTeamId,
            hLeagueId,
            hTeamId,
            hTeamNameString,
            hTeamNicknameString,
            SP_SEEK_TEAM,
            &eReturnCode,
            TRUE, // when newly registering, the content is enabled
            phSeekContent
                );

        // We dont' need the CID
        CID_vDestroy(hLeagueTeamId);

        if (psItem == NULL)
        {
            // couldn't register
            break;
        }

        // add registered item to config file
        bOk = bAddToConfigFile(
                  psObj,
                  psItem,
                  hTeamNameString,
                  hTeamNicknameString
                      );
        if (bOk == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

    } while (0);

    return eReturnCode;
}

/*****************************************************************************
*   eRegisterLeagueLocal
*
* This local function registers the team with the service
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegisterLeagueLocal(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    LEAGUE_OBJECT hLeague,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bOk;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem = NULL;
    STRING_OBJECT hLeagueNameString = STRING_INVALID_OBJECT,
                  hLeagueAbbrevString = STRING_INVALID_OBJECT;

    do
    {
        CID_OBJECT hLeagueId;

        hLeagueId = LEAGUE.hId(hLeague);
        if (hLeagueId == CID_INVALID_OBJECT)
        {
            // can't register without a valid CID
            break;
        }

        // get the strings used to describe the league being registered.
        hLeagueNameString = LEAGUE.hName(hLeague);
        hLeagueAbbrevString = LEAGUE.hAbbreviation(hLeague);

        // register the league
        psItem = psRegisterContent(
            psObj,
            hLeagueId,
            hLeagueId,
            CID_INVALID_OBJECT,
            hLeagueNameString,
            hLeagueAbbrevString,
            SP_SEEK_LEAGUE,
            &eReturnCode,
            TRUE, // when newly registering, the content is enabled
            phSeekContent
                );

        if (psItem == NULL)
        {
            // couldn't register
            break;
        }

        // add registered item to config file
        bOk = bAddToConfigFile(
                  psObj,
                  psItem,
                  hLeagueNameString,
                  hLeagueAbbrevString
                      );
        if (bOk == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

    } while (0);

    return eReturnCode;
}


/*****************************************************************************
*
*   hGetCDO
*
*****************************************************************************/
static CD_OBJECT hGetCDO(
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId
        )
{
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT hChannel;
    CD_OBJECT hCDO;
    CDO_TYPE_ENUM eCDOType;

    // did caller give us a valid id?
    if (tChannelId == CHANNEL_INVALID_ID)
    {
        // no, so instead we use the currently tuned channel id
        tChannelId = DECODER.tCurrentChannelId(hDecoder);
    }

    // From the DECODER handle, determine the channel cache handle
    hCCache = DECODER_hCCache(hDecoder);

    // get the channel obj for this requested chan id
    hChannel = CCACHE_hChannelFromIds (
        hCCache,
        SERVICE_INVALID_ID,
        tChannelId,
        FALSE
            );

    // get the channel's cdo
    hCDO = CHANNEL.hCDO(hChannel);

    // get the type of cdo
    eCDOType = CDO.eType(hCDO);
    if (eCDOType != CDO_SPORTS)
    {
        // sports aren't being broadcast on this channel!
        return CD_INVALID_OBJECT;
    }

    return hCDO;
}


/*****************************************************************************
*
*    bRetrieveCidForSeeks
*
*    This is a local function which looks at a CDO and extracts a
*    CID that can be used for registering.
*    (Note: CDO type has already been confirmed to be of type SPORTS)
*
*****************************************************************************/
static CID_OBJECT hRetrieveCidForSeeks(
    CD_OBJECT hCDO,
    SP_SEEK_ENUM eType,
    UN8 un8Index
        )
{
    CID_OBJECT hCID;

    if (eType == SP_SEEK_LEAGUE)
    {
        LEAGUE_OBJECT hLeague;
        // get the league
        hLeague = SPORTS.hLeague(hCDO);
        // get the league CID
        hCID = LEAGUE.hId(hLeague);
    }
    else if (eType == SP_SEEK_TEAM)
    {
        TEAM_OBJECT hTeam;
        LEAGUE_OBJECT hLeague;
        LEAGUE_TEAM_ID_STRUCT sLeagueTeamId;

        // get the team
        hTeam = SPORTS.hTeam(hCDO, un8Index);
        // get the LEAGUE
        hLeague = SPORTS.hLeague(hCDO);

        if ((hTeam != TEAM_INVALID_OBJECT) &&
            (hLeague != LEAGUE_INVALID_OBJECT))
        {
            // Take league-id and team-id and make a CID
            sLeagueTeamId.hLeagueId = LEAGUE.hId(hLeague);
            sLeagueTeamId.hTeamId = TEAM.hId(hTeam);

            // Create a CID based on Team and League CIDs.
            hCID = CID_hCreate(
                CID_POOL_INVALID_OBJECT, CID_LEAGUE_TEAM_ID, &sLeagueTeamId);
        }
        else
        {
            // Error! We need both CIDs
            hCID = CID_INVALID_OBJECT;
        }
    }
    else
    {
        // error
        hCID = CID_INVALID_OBJECT;
    }

    return hCID;
}

/*****************************************************************************
*
*    bAddToConfigFile
*
*****************************************************************************/
static BOOLEAN bAddToConfigFile(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem,
    STRING_OBJECT hNameString,
    STRING_OBJECT hAbbrevString
        )
{
    TAG_OBJECT hTag;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    if (psObj->hTag == TAG_INVALID_OBJECT)
    {
        // must not be a config file
        return TRUE;
    }

    // add a new Registered Content Tag
    // this will be the parent tag of all tags needed to persist an
    // instance of content registered with this service
    eReturn = TAG_eAdd(
          SEEK_REGISTERED_CONTENT,
          psObj->hTag,
          &hTag,
          NULL
              );
    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    // add all the tags needed to persist an
    // instance of at seek content
    eReturn = eAddToConfigFile (
                  hTag,
                  psItem,
                  hNameString,
                  hAbbrevString
                      );

    // successful?
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // yes
        return TRUE;
    }

    // no
    return FALSE;
}

/*****************************************************************************
*
*    psRegisterContent
*
*****************************************************************************/
static SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisterContent(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    CID_OBJECT hCID,
    CID_OBJECT hLeagueCID,
    CID_OBJECT hTeamCID,
    STRING_OBJECT hNameString,
    STRING_OBJECT hAbbrevString,
    SP_SEEK_ENUM eType,
    SMSAPI_RETURN_CODE_ENUM *peReturn,
    BOOLEAN bEnabled,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem = NULL;
    *peReturn = SMSAPI_RETURN_CODE_ERROR;

    // create a seek content item of type that is applicable to the
    // traffic/weather service
    psItem = SEEK_CONTENT_psCreateRegisteredItem((SEEK_SERVICE_OBJECT)psObj,
                 bEnabled);
    if (psItem != NULL)
    {
        // initialize the service specific content portion
        SP_SEEK_CONTENT_vSetServiceSpecificInfo(
            (void *)psItem,
            hLeagueCID,
            hTeamCID,
            eType,
            (SEEK_SERVICE_OBJECT)psObj
                );

        // we have all the information we need, now we can
        // register with the CAL
        *peReturn = CAL.eRegister(
            psObj->hSeekList,
            hCID,
            (void *)psItem,
            SP_SEEK_CAL_REGISTRATION_OPTIONS,
            hNameString,
            hAbbrevString,
            (CAL_CONTENT_OBJECT *)phSeekContent
                );

        if (*peReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // we couldn't successfully register for some reason, destroy the
            // item we created
            SEEK_CONTENT_vDestroyRegisteredItem(psItem);
            psItem = NULL;
        }
    }

    return psItem;
}

/*****************************************************************************
*
*   bRegisterContentFromConfigFile
*
*****************************************************************************/
static BOOLEAN bRegisterContentFromConfigFile(
    SEEK_SERVICE_OBJECT_STRUCT *psObj
        )
{
    REGISTERED_CONTENT_TAG_ITERATOR_STRUCT sIteratorStruct;
    SMSAPI_RETURN_CODE_ENUM eResult;

    if (psObj->hTag == TAG_INVALID_OBJECT)
    {
        // must not be a config file
        return TRUE;
    }

    sIteratorStruct.psObj = psObj;

    // the seek service object holds the parent tag of all registered content
    // tags. Iterate through all the child tags, and for each registered
    // content tag extract the information and register and configure.
    eResult = TAG_eIterateChildren(
        psObj->hTag,
        bRegisteredContentTagIterator,
        &sIteratorStruct
            );

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bRegisteredContentTagIterator
*
*****************************************************************************/
static BOOLEAN bRegisteredContentTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    REGISTERED_CONTENT_TAG_ITERATOR_STRUCT *psIteratorStruct =
        (REGISTERED_CONTENT_TAG_ITERATOR_STRUCT *)pvArg;
    STRING_OBJECT hTagName;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    CID_OBJECT hCID = CID_INVALID_OBJECT, hLeagueCID = CID_INVALID_OBJECT,
               hTeamCID = CID_INVALID_OBJECT;
    STRING_OBJECT hNameString = STRING_INVALID_OBJECT,
                  hAbbrevString = STRING_INVALID_OBJECT,
                  hNameStringTag = STRING_INVALID_OBJECT,
                  hAbbrevStringTag = STRING_INVALID_OBJECT;
    SP_SEEK_LEAGUE_ITERATOR_STRUCT sIterator;

    // clean up string pointers for correct memory deallocation at the end
    sIterator.hAbbreviation = STRING_INVALID_OBJECT;
    sIterator.hName = STRING_INVALID_OBJECT;

    // get the tag name
    hTagName = TAG_hTagName(hTag);
    // see if the tag name is one that we're looking for
    if (STRING.n16CompareCStr(SEEK_REGISTERED_CONTENT, 0, hTagName) == 0)
    {
        SP_SEEK_ENUM eType = SP_SEEK_UNKNOWN;
        BOOLEAN bEnabled = FALSE;

        do
        {
            BOOLEAN bOk;

            // get type
            eType = eGetSPSeekTypeFromTags(hTag);

            if (eType == SP_SEEK_UNKNOWN)
            {
                // can't do anything more with this entry
                break;
            }

            eReturn =
                eGetEnabledFromTag(psIteratorStruct->psObj, hTag, &bEnabled);

            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            // get the CID
            eReturn = eGetCidsFromTags(
                          hTag,
                          eType,
                          &hCID,
                          &hLeagueCID,
                          &hTeamCID
                              );

            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            eReturn = eGetNameAbbrevTextFromTags(
                hTag,
                &hNameStringTag,
                &hAbbrevStringTag
                    );
            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            hNameString = hNameStringTag;
            hAbbrevString = hAbbrevStringTag;

            // Prepare iterator struct for searching
            sIterator.bFound = FALSE;
            sIterator.eType = eType;
            sIterator.hLeagueId = hLeagueCID;
            sIterator.hTeamId = hTeamCID;

            // Search for a corresponding league / team by content id
            bOk = LEAGUE_bIterateContent(
                bLeagueIteratorCallback,
                &sIterator);

            if (bOk == FALSE)
            {
                // Couldn't iterate leagues
                break;
            }

            // At this point, hNameStringTag and hAbbrevStringTag shall
            // be a valid strings. Otherwise we would have never got here.

            if (sIterator.bFound == TRUE)
            {
                N16 n16Equal;

                n16Equal = STRING.n16Compare(hNameStringTag, sIterator.hName, TRUE);
                if (n16Equal != 0)
                {
                    hNameString = sIterator.hName;
                    hAbbrevString = sIterator.hAbbreviation;
                }
                else
                {
                    n16Equal = STRING.n16Compare(hAbbrevStringTag, sIterator.hAbbreviation, TRUE);
                    if (n16Equal != 0)
                    {
                        hNameString = sIterator.hName;
                        hAbbrevString = sIterator.hAbbreviation;
                    }
                }

                if (n16Equal != 0)
                {
                    eReturn = eSetNameAbbrevTextToTags(
                        hTag,
                        hNameString,
                        hAbbrevString
                            );

                    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        break;
                    }
                }
            }

            psItem = psRegisterContent(
                         psIteratorStruct->psObj,
                         hCID,
                         hLeagueCID,
                         hTeamCID,
                         hNameString,
                         hAbbrevString,
                         eType,
                         &eReturn,
                         bEnabled,
                         NULL
                             );

            if (psItem != NULL)
            {
                psItem->hTag = hTag;
            }

        } while (FALSE);
    }


    // destroy anything allocated
    if (hLeagueCID != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hLeagueCID);

        // When the type is SP_SEEK_LEAGUE, these CIDs will
        // be the same. But if the type is SP_SEEK_TEAM, then all 
        // three CIDs will point into different memory blocks.
        if (hLeagueCID == hCID)
        {
            hCID = CID_INVALID_OBJECT;
        }
    }

    if (hTeamCID != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hTeamCID);
    }

    if (hCID != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hCID);
    }

    if (hNameStringTag != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hNameStringTag);
    }
    if (hAbbrevStringTag != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hAbbrevStringTag);
    }

    if (sIterator.hName != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(sIterator.hName);
    }
    if (sIterator.hAbbreviation != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(sIterator.hAbbreviation);
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   eGetSPSeekTypeFromTags
*
*****************************************************************************/
static SP_SEEK_ENUM eGetSPSeekTypeFromTags(
    TAG_OBJECT hTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;
    SP_SEEK_ENUM eType = SP_SEEK_UNKNOWN;

    // get the type tag
    eResult = TAG_eGet(
        SP_SEEK_TYPE,
        hTag,
        &hChildTag,
        NULL,
        FALSE // don't create if it isn't found
            );
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;
        size_t tSize;

        tSize = sizeof(STRING_OBJECT);
        phString = &hString;
        eResult = TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // got a value, now compare it to the type enums to see which one
            // it is
            do
            {
                // try SP_SEEK_LEAGUE
                eType = SP_SEEK_LEAGUE;
                // compare value to the type text
                if (STRING.n16CompareCStr(
                        SP_SEEK_CONTENT_pacSPSeekTypeText(eType),
                        0,
                        hString) == 0)
                {
                    // matched, so we now know it is SP_SEEK_LEAGUE
                    break;
                }

                // try SP_SEEK_TEAM
                eType = SP_SEEK_TEAM;
                // Initialize CID entry as empty
                if (STRING.n16CompareCStr(
                        SP_SEEK_CONTENT_pacSPSeekTypeText(eType),
                        0,
                        hString) == 0)
                {
                    // matched, so we now know it is SP_SEEK_TEAM
                    break;
                }

                // couldn't find a match, so the type will
                // remain SP_SEEK_UNKNOWN
            } while (0);

            STRING.vDestroy(hString);
        }
    }

    return eType;
}

/*****************************************************************************
*
*   eGetEnabledFromTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetEnabledFromTag(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    TAG_OBJECT hTag,
    BOOLEAN *pbEnabled
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;

    if (pbEnabled != NULL)
    {
        // get enabled tag
        eReturn = TAG_eGet(
            SP_SEEK_ENABLED,
            hTag,
            &hChildTag,
            NULL,
            FALSE
                );
        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // get enabled tag value
            STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;
            size_t tSize;

            tSize = sizeof(STRING_OBJECT);
            phString = &hString;
            eReturn = TAG_eGetTagValue(
                hChildTag,
                TAG_TYPE_STRING,
                (void **)&phString,
                &tSize);

            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {

                eReturn = CM_eStringToBoolean(
                    hString,
                    pbEnabled
                        );

                // we're done with this
                STRING.vDestroy(hString);
            }
        }
    }
    return eReturn;
}

/*****************************************************************************
*
*   eGetCidsFromTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetCidsFromTags(
    TAG_OBJECT hTag,
    SP_SEEK_ENUM eType,
    CID_OBJECT *phCID,
    CID_OBJECT *phLeagueCID,
    CID_OBJECT *phTeamCID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hChildTag;
    STRING_OBJECT hLeagueCIDString = STRING_INVALID_OBJECT,
                  hTeamCIDString = STRING_INVALID_OBJECT,
                  *phString;
    size_t tSize, tLeagueStringSize = 0, tTeamStringSize = 0;
    char *pacCStrBuffer = NULL;
    void *pvVoid;

    *phCID = CID_INVALID_OBJECT;
    *phLeagueCID = CID_INVALID_OBJECT;
    *phTeamCID = CID_INVALID_OBJECT;

    // get League cid tag
    eReturn = TAG_eGet(
        SP_SEEK_LEAGUE_CID,
        hTag,
        &hChildTag,
        NULL,
        FALSE
            );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // get the League cid
        phString = &hLeagueCIDString;
        eReturn = TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize);
        tLeagueStringSize = STRING.tSize(hLeagueCIDString);
    }

    if (eType == SP_SEEK_TEAM)
    {
        // get team cid tag
        eReturn = TAG_eGet(
            SP_SEEK_TEAM_CID,
            hTag,
            &hChildTag,
            NULL,
            FALSE
                );
        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // get the title cid tag
            phString = &hTeamCIDString;
            eReturn = TAG_eGetTagValue(
                hChildTag,
                TAG_TYPE_STRING,
                (void **)&phString,
                &tSize);
            tTeamStringSize = STRING.tSize(hTeamCIDString);
        }
    }

    // size for the bigger of the two
    tSize = (tLeagueStringSize > tTeamStringSize ?
            tLeagueStringSize : tTeamStringSize);

    tSize++; // add one for NULL termination

    // allocate memory for the read buffer
    pacCStrBuffer =
        (char *) SMSO_hCreate(
                    SP_SEEK_OBJECT_NAME":CSTR",
                    tSize,
                    SMS_INVALID_OBJECT, FALSE // No lock necessary
                    );
    if (pacCStrBuffer != NULL)
    {
        if (hLeagueCIDString != STRING_INVALID_OBJECT)
        {
            UN32 un32LeagueId;
            void *pvLeagueId = &un32LeagueId;

            // copy to a c-string so that we can create a CID from it
            STRING.tCopyToCStr(hLeagueCIDString, pacCStrBuffer, tSize);
            pvVoid = (void *)pacCStrBuffer;
            // Read from memory assumes memory provided as an input is
            // NULL terminated. It is in fact since we just did a
            // string copy above into a buffer we know is at least one
            // larger than the source (thus it will be NULL terminated).
            *phLeagueCID = CID_hReadFromMemory((void const**)&pvVoid);
            STRING.vDestroy(hLeagueCIDString);

            // Normalize league-id
            CID_n32GetValue(*phLeagueCID, &pvLeagueId);
            if(un32LeagueId > 0xFFFF)
            {
                un32LeagueId >>= 16;
                un32LeagueId &= 0xFFFF;

                // Modify it
                CID_bModify(phLeagueCID, pvLeagueId);
            }

        }

        if (hTeamCIDString != STRING_INVALID_OBJECT)
        {
            UN32 un32TeamId;
            void *pvTeamId = &un32TeamId;

            // copy to a c-string so that we can create a CID from it
            STRING.tCopyToCStr(hTeamCIDString, pacCStrBuffer, tSize);
            pvVoid = (void *)pacCStrBuffer;
            // Read from memory assumes memory provided as an input is
            // NULL terminated. It is in fact since we just did a
            // string copy above into a buffer we know is at least one
            // larger than the source (thus it will be NULL terminated).
            *phTeamCID = CID_hReadFromMemory((void const**)&pvVoid);
            STRING.vDestroy(hTeamCIDString);

            // Normalize team-id
            CID_n32GetValue(*phTeamCID, &pvTeamId);
            if(un32TeamId > 0xFFFF)
            {
                un32TeamId &= 0xFFFF;

                // Modify it
                CID_bModify(phTeamCID, pvTeamId);
            }
        }

        // we're done with the buffer
        SMSO_vDestroy((SMS_OBJECT)pacCStrBuffer);

        // based on the at seek type, set the cid to register for
        switch (eType)
        {
            case SP_SEEK_LEAGUE:
                *phCID = *phLeagueCID;
            break;
            case SP_SEEK_TEAM:
            {
                LEAGUE_TEAM_ID_STRUCT sLeagueTeamId;

                // Take league-id and team-id and make a CID
                sLeagueTeamId.hLeagueId = *phLeagueCID;
                sLeagueTeamId.hTeamId = *phTeamCID;

                *phCID = CID_hCreate(
                    CID_POOL_INVALID_OBJECT, 
                    CID_LEAGUE_TEAM_ID, &sLeagueTeamId);
            }
            break;
            default:
                eReturn = SMSAPI_RETURN_CODE_ERROR;
            break;
        }
    }
    else
    {
        // couldn't allocate memory
        eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
    }

    return eReturn;
}

/*****************************************************************************
*
*   eGetNameAbbrevTextFromTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetNameAbbrevTextFromTags(
    TAG_OBJECT hTag,
    STRING_OBJECT *phNameText,
    STRING_OBJECT *phAbbrevText
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;

    if ((phNameText == NULL) || (phAbbrevText == NULL))
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // get name text
    eReturn = TAG_eGet(
        SP_SEEK_NAME_TEXT,
        hTag,
        &hChildTag,
        NULL,
        TRUE
            );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        size_t tSize;
        tSize = sizeof(STRING_OBJECT);

        // get name text
        eReturn = TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phNameText,
            &tSize);

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // get abbrev text tag
            eReturn = TAG_eGet(
                          SP_SEEK_ABBREV_TEXT,
                          hTag,
                          &hChildTag,
                          NULL,
                          TRUE
                              );
            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // get abbrev text
                eReturn = TAG_eGetTagValue(
                    hChildTag,
                    TAG_TYPE_STRING,
                    (void **)&phAbbrevText,
                    &tSize);
            }
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eSetNameAbbrevTextToTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetNameAbbrevTextToTags(
    TAG_OBJECT hTag,
    STRING_OBJECT hNameText,
    STRING_OBJECT hAbbrevText
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    do
    {
        TAG_OBJECT hNameTag;
        TAG_OBJECT hAbbrevTag;

        // Verify inputs
        if ((hTag == TAG_INVALID_OBJECT) ||
            (hNameText == STRING_INVALID_OBJECT) ||
            (hAbbrevText == STRING_INVALID_OBJECT))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        eReturnCode = TAG_eGet(
            SP_SEEK_NAME_TEXT,
            hTag,
            &hNameTag,
            NULL,
            FALSE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eReturnCode = TAG_eSetTagValue(
            hNameTag,
            TAG_TYPE_STRING,
            &hNameText,
            sizeof(STRING_OBJECT),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eReturnCode = TAG_eGet(
            SP_SEEK_ABBREV_TEXT,
            hTag,
            &hAbbrevTag,
            NULL,
            FALSE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eReturnCode = TAG_eSetTagValue(
            hAbbrevTag,
            TAG_TYPE_STRING,
            &hAbbrevText,
            sizeof(STRING_OBJECT),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eWriteTypeToTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteTypeToTag(
    TAG_OBJECT hParentTag,
    SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psSPSpecificInfo
        )
{
    STRING_OBJECT hValueString;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hTypeTag = TAG_INVALID_OBJECT;

    // add the type tag
    eReturn = TAG_eAdd(
                  SP_SEEK_TYPE,
                  hParentTag,
                  &hTypeTag,
                  NULL
                      );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // extract the type
        SP_SEEK_ENUM eType;

        // create a string containing the type text
        eType = psSPSpecificInfo->eType;
        hValueString =
            STRING.hCreate(SP_SEEK_CONTENT_pacSPSeekTypeText(eType), 0);

        eReturn = TAG_eSetTagValue(
                      hTypeTag,
                      TAG_TYPE_STRING,
                      &hValueString,
                      sizeof(STRING_OBJECT),
                      FALSE
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // couldn't set the value, so remove the tag we just added
            TAG_eRemove(hTypeTag, FALSE);
        }

        // done with this string
        STRING.vDestroy(hValueString);
    }

    return eReturn;
}

/*****************************************************************************
*
*   eWriteCIDToTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteCIDsToTag(
    TAG_OBJECT hParentTag,
    SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psSPSpecificInfo
        )
{
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hTeamCidTag = TAG_INVALID_OBJECT,
               hLeagueCidTag = TAG_INVALID_OBJECT;
    size_t tTeamCIDSize, tLeagueCIDSize, tCIDSize;
    char *pacCIDBuffer;
    void *pvVoid;

    do
    {
        // get the cid sizes
        tTeamCIDSize = CID_tSize(psSPSpecificInfo->hTeamCID);
        tLeagueCIDSize = CID_tSize(psSPSpecificInfo->hLeagueCID);
        // get the largest cid size
        tCIDSize = (tLeagueCIDSize > tTeamCIDSize ?
                    tLeagueCIDSize : tTeamCIDSize);

        // allocate memory for the read buffer
        pacCIDBuffer =
            (char *) SMSO_hCreate(
                         SP_SEEK_OBJECT_NAME":CB",
                         tCIDSize,
                         SMS_INVALID_OBJECT, FALSE // No lock necessary
                         );
        if (pacCIDBuffer == NULL)
        {
            eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
            break;
        }

        pvVoid = pacCIDBuffer;

        // add a tag for the league cid
        eReturn = TAG_eAdd(
                      SP_SEEK_LEAGUE_CID,
                      hParentTag,
                      &hLeagueCidTag,
                      NULL
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // create a string containing the League cid
        CID_n32FWriteToMemory(psSPSpecificInfo->hLeagueCID, (void **)&pvVoid);
        hValueString = STRING.hCreate(pacCIDBuffer, tCIDSize);
        // set the tag value
        eReturn = TAG_eSetTagValue(
                      hLeagueCidTag,
                      TAG_TYPE_STRING,
                      &hValueString,
                      sizeof(STRING_OBJECT),
                      FALSE
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        if (psSPSpecificInfo->hTeamCID != CID_INVALID_OBJECT)
        {

            // add tag for the team cid
            eReturn = TAG_eAdd(
                          SP_SEEK_TEAM_CID,
                          hParentTag,
                          &hTeamCidTag,
                          NULL
                              );
            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            // create a string containing the team cid
            pvVoid = pacCIDBuffer;
            CID_n32FWriteToMemory(psSPSpecificInfo->hTeamCID, (void **)&pvVoid);
            STRING.bModifyCStr(hValueString, pacCIDBuffer);

            // write the tag vale
            eReturn = TAG_eSetTagValue(
                          hTeamCidTag,
                          TAG_TYPE_STRING,
                          &hValueString,
                          sizeof(STRING_OBJECT),
                          FALSE
                              );
            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

    } while(0);

    if(pacCIDBuffer != NULL)
    {
        // if we allocated this buffer, delete it
        SMSO_vDestroy((SMS_OBJECT)pacCIDBuffer);
    }

    if (hValueString != STRING_INVALID_OBJECT)
    {
        // if we created the string, delete it
        STRING.vDestroy(hValueString);
    }

    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // something went wrong, so remove the tags we added
        if (TAG_INVALID_OBJECT != hLeagueCidTag)
        {
            TAG_eRemove(hLeagueCidTag, FALSE);
        }
        if (TAG_INVALID_OBJECT != hTeamCidTag)
        {
            TAG_eRemove(hTeamCidTag, FALSE);
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eWriteTextToTag
*
* common fxn that can be used for NameText, AbbrevText
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteTextToTag(
    TAG_OBJECT hParentTag,
    const char *pacTagName,
    STRING_OBJECT hText
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hTextTag = TAG_INVALID_OBJECT;

    // add the text tag
    eReturn = TAG_eAdd(
          pacTagName,
          hParentTag,
          &hTextTag,
          NULL
              );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the text tag value
        eReturn = TAG_eSetTagValue(
            hTextTag,
            TAG_TYPE_STRING,
            &hText,
            sizeof(STRING_OBJECT),
            FALSE
                );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // something went wrong, so remove the tag we added
            TAG_eRemove(hTextTag, FALSE);
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eAddToConfigFile
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddToConfigFile (
    TAG_OBJECT hParentTag,
    void *pvItem,
    STRING_OBJECT hNameString,
    STRING_OBJECT hAbbrevString
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psSPSpecificInfo;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psGenericItem;

    psGenericItem = (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)pvItem;
    psSPSpecificInfo = (SP_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                           &psGenericItem->uServiceSpecific;

    // verify that the pointer to the data isn't NULL

    if ( psSPSpecificInfo == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        // set the parent tag
        psGenericItem->hTag = hParentTag;

        // update the enable tag (the tag gets created in this case since
        // it doesn't already exist)
        eReturn = SEEK_CONTENT_eUpdateEnabledTag(psGenericItem, FALSE);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write the type tag
        eReturn = eWriteTypeToTag(hParentTag, psSPSpecificInfo);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write the CID tags
        eReturn = eWriteCIDsToTag(hParentTag, psSPSpecificInfo);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write name text
        eReturn = eWriteTextToTag(
                      hParentTag,
                      SP_SEEK_NAME_TEXT,
                      hNameString);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write abbrev text
        eReturn = eWriteTextToTag(
                      hParentTag,
                      SP_SEEK_ABBREV_TEXT,
                      hAbbrevString);

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            eReturn = CM_eCommitChangesToFile();
        }

    } while (0);

    return eReturn;
}


/*****************************************************************************
*
*   bSportSeekContentIterator
*
*****************************************************************************/
static BOOLEAN bSportSeekContentIterator(
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService,
    SEEK_CONTENT_OBJECT hContent,
    void *pvSeekContentIteratorArg)
{

    SP_SEEK_ENUM eType;
    SP_SEEK_ITERATOR_STRUCT *psStruct =
      (SP_SEEK_ITERATOR_STRUCT *)pvSeekContentIteratorArg;

    if (psStruct == NULL)
    {
        // stop iterating
        return FALSE;
    }

    eType = SP_SEEK_CONTENT.eType(hContent);
    if (eType == SP_SEEK_TEAM)
    {
        CID_OBJECT hId;
        hId = SP_SEEK_CONTENT_hCID(hContent);
        if (CID.n16Equal(hId, psStruct->hId) == 0)
        {
            BOOLEAN bEnabled;
            bEnabled = SEEK_CONTENT.bEnabled(hContent);
            if (bEnabled == TRUE)
            {
                psStruct->bMatch = TRUE;
                // stop iterating
                return FALSE;
            }
        }
    }
    // continue iterating
    return TRUE;
}

/*****************************************************************************
*
*   bLeagueIteratorCallback
*
*****************************************************************************/
static BOOLEAN bLeagueIteratorCallback(
    LEAGUE_OBJECT hLeague,
    void *pvContentIteratorCallbackArg
        )
{
    N16 n16Equal;
    CID_OBJECT hId;
    BOOLEAN bReturn = TRUE;

    SP_SEEK_LEAGUE_ITERATOR_STRUCT *psIterator =
        (SP_SEEK_LEAGUE_ITERATOR_STRUCT *)pvContentIteratorCallbackArg;

    hId = LEAGUE.hId(hLeague);
    n16Equal = CID.n16Equal(psIterator->hLeagueId, hId);

    if (n16Equal == 0)
    {
        if (psIterator->eType == SP_SEEK_LEAGUE)
        {
            STRING_OBJECT hName;
            STRING_OBJECT hAbbrev;

            hName = LEAGUE.hName(hLeague);
            hAbbrev = LEAGUE.hAbbreviation(hLeague);

            // Need to duplicate these strings
            // to eliminate possible errors on Win7 (x64) + VS2012

            // We are looking for a league
            psIterator->hName = STRING.hDuplicate(hName);
            psIterator->hAbbreviation = STRING.hDuplicate(hAbbrev);

            psIterator->bFound = TRUE;
        }
        else if (psIterator->eType == SP_SEEK_TEAM)
        {
            BOOLEAN bSuccess;

            // We are looking for a team
            bSuccess = TEAM_bIterateContent(
                hLeague,
                bTeamContentIteratorCallback,
                psIterator);

            if (bSuccess == FALSE)
            {
                // Make sure that we did not find anything in this case.
                psIterator->bFound = FALSE;
            }
        }

        bReturn = FALSE;
    }

    return bReturn;
}

/*****************************************************************************
*
*   bTeamContentIteratorCallback
*
*****************************************************************************/
static BOOLEAN bTeamContentIteratorCallback(
    TEAM_OBJECT hTeam,
    void *pvContentIteratorCallbackArg
        )
{
    N16 n16Equal;
    CID_OBJECT hId;

    SP_SEEK_LEAGUE_ITERATOR_STRUCT *psIterator =
        (SP_SEEK_LEAGUE_ITERATOR_STRUCT *)pvContentIteratorCallbackArg;

    hId = TEAM.hId(hTeam);
    n16Equal = CID.n16Equal(psIterator->hTeamId, hId);

    if (n16Equal == 0)
    {
        STRING_OBJECT hName;
        STRING_OBJECT hAbbrev;

        hName = TEAM.hName(hTeam);
        hAbbrev = TEAM.hNickname(hTeam);

        // Need to duplicate these strings
        // to eliminate possible errors on Win7 (x64) + VS2012

        psIterator->hName = STRING.hDuplicate(hName);
        psIterator->hAbbreviation = STRING.hDuplicate(hAbbrev);

        psIterator->bFound = TRUE;

        return FALSE;
    }

    return TRUE;
}
