/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SNOW_CONDITIONS implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_snow_conditions_obj.h"
#include "snow_conditions_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eMinBaseDepth
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eMinBaseDepth (
    SNOW_CONDITIONS_OBJECT hSnowCond,
    UN8 *pun8MinBaseDepth
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SNOW_CONDITIONS_OBJECT_STRUCT *psObj = (SNOW_CONDITIONS_OBJECT_STRUCT *)hSnowCond;

    do
    {
        if (pun8MinBaseDepth == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSnowCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bBMinAvailable == TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            *pun8MinBaseDepth = psObj->un8BMin;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eMaxBaseDepth
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eMaxBaseDepth (
    SNOW_CONDITIONS_OBJECT hSnowCond,
    UN8 *pun8MaxBaseDepth
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SNOW_CONDITIONS_OBJECT_STRUCT *psObj = (SNOW_CONDITIONS_OBJECT_STRUCT *)hSnowCond;

    do
    {
        if (pun8MaxBaseDepth == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSnowCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bBMaxAvailable == TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            *pun8MaxBaseDepth = psObj->un8BMax;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}



/*****************************************************************************
*
*   eNewSnowRange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eNewSnowRange (
    SNOW_CONDITIONS_OBJECT hSnowCond,
    UN8 *pun8NewSnowRangeLow,
    UN8 *pun8NewSnowRangeHigh
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SNOW_CONDITIONS_OBJECT_STRUCT *psObj = (SNOW_CONDITIONS_OBJECT_STRUCT *)hSnowCond;

    do
    {
        if ((pun8NewSnowRangeHigh == NULL) || (pun8NewSnowRangeLow == NULL))
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSnowCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bNewAvailable == TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            *pun8NewSnowRangeHigh = psObj->sNewData.un8High;
            *pun8NewSnowRangeLow = psObj->sNewData.un8Low;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eConditions
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eConditions (
    SNOW_CONDITIONS_OBJECT hSnowCond,
    SKI_SNOW_CONDITIONS_ENUM *peConditions
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SNOW_CONDITIONS_OBJECT_STRUCT *psObj = (SNOW_CONDITIONS_OBJECT_STRUCT *)hSnowCond;

    do
    {
        if (peConditions == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSnowCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bSnowAvailable == TRUE)
        {
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            *peConditions = (SKI_SNOW_CONDITIONS_ENUM)psObj->un8Snow;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    SNOW_CONDITIONS_OBJECT hSnowCond,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    SNOW_CONDITIONS_OBJECT_STRUCT *psObj = (SNOW_CONDITIONS_OBJECT_STRUCT *)hSnowCond;
    BOOLEAN bValid;

    // Determine if the handle is valid
    bValid = SMSO_bValid((SMS_OBJECT)hSnowCond);
    if ((bValid == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print SNOW_CONDITIONS information header
    n32Return += fprintf(psFile, "SNOW_CONDITIONS:\n\thSnowCond = %p\n",
                         psObj);

    if (psObj->bBMinAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tMinimum Base Snow Depth = %d\n",
                psObj->un8BMin);
    }

    if (psObj->bBMaxAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tMaximum Base Snow Depth = %d\n",
                psObj->un8BMax);
    }

    if (psObj->bNewAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tNew Snow Depth: %d - %d\n",
                psObj->sNewData.un8Low, psObj->sNewData.un8High);
    }

    if (psObj->bSnowAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tSnow Conditions = %d\n",
                psObj->un8Snow);
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SNOW_CONDITIONS_hCreate
*
*****************************************************************************/
SNOW_CONDITIONS_OBJECT SNOW_CONDITIONS_hCreate(
        SMS_OBJECT hParent,
        BOOLEAN bBMinAvailable,
        UN8 un8BMin,
        BOOLEAN bBMaxAvailable,
        UN8 un8BMax,
        BOOLEAN bNewAvailable,
        NEW_SNOW_UNPARSED_DATA tNewData,
        BOOLEAN bSnowAvailable,
        UN8 un8Snow
            )
{
    SNOW_CONDITIONS_OBJECT_STRUCT *psObj =
        (SNOW_CONDITIONS_OBJECT_STRUCT *)SNOW_CONDITIONS_INVALID_OBJECT;

    // Create an instance of the HUMIDITY object
    psObj = (SNOW_CONDITIONS_OBJECT_STRUCT *)
        SMSO_hCreate(
            SNOW_CONDITIONS_OBJECT_NAME,
            sizeof(SNOW_CONDITIONS_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return SNOW_CONDITIONS_INVALID_OBJECT;
    }

    // Initialize object per inputs
    psObj->bBMinAvailable = bBMinAvailable;
    psObj->bBMaxAvailable = bBMaxAvailable;
    psObj->bNewAvailable = bNewAvailable;
    psObj->bSnowAvailable = bSnowAvailable;
    psObj->un8BMin = un8BMin;
    psObj->un8BMax = un8BMax;
    psObj->un8Snow = un8Snow;

    if (bNewAvailable == TRUE)
    {
        psObj->bNewAvailable = GsWeatherIntf.bProcessNewSnowData(tNewData, &psObj->sNewData);
    }

    return (SNOW_CONDITIONS_OBJECT)psObj;
}

/*****************************************************************************
*
*   SNOW_CONDITIONS_vDestroy
*
*****************************************************************************/
void SNOW_CONDITIONS_vDestroy (
    SNOW_CONDITIONS_OBJECT hSnowCond
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSnowCond);
    if(bOwner == TRUE)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hSnowCond);
    }
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
