/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains various APIs and functions which implement the
 *  Sirius Module Services Object Update.
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "sms_update.h"
#include "_sms_update.h"

/*****************************************************************************
*
*   SMSU_vInitialize
*
*   This function initializes an SMS-Update object for first use. An
*   SMS-Update object is an object and set of APIs which handle to common
*   and recurring tasks of maintaining and handling notifications when
*   elements of an owning object change. The interface allows for the
*   registration of an initial bitmask representing attributes or elements
*   of the owning object which are marked as "changed" or "updated" at the
*   time of initialization.
*   Also registered is the request mask which identifies or masks
*   the specific bits the caller is interested in as the caller may not
*   initially desire notifications of all possible bits specified in the
*   bitmask.
*   Lastly the caller provides a callback function which will be called
*   when any one or more of the request mask bits are set (this happens
*   when an update occurred).
*
*   psEvent - A pointer to the SMS-Update event structure which is
*       held by the object which needs to provide update notifications.
*       Callers of this function are required to allocate their own
*       memory resources to contain this structure.
*   pvObject - A pointer to the object update notifications are being
*       tracked for.
*   tInitialMask - A bitmask which is used to set the initial event
*       state for an SMS-update object. The initial bitmask not only
*       represents the events which are set initially, but will also
*       be loaded as the initial bitmask for all subsequent update
*       notification registrations when the SMSU_bRegisterCallback()
*       function is called.
*   tRequestMask - A bitmask which is used to indicate which of the
*       bits in the event mask are of interest to the caller and will
*       trigger update notifications when set.
*   vCallback - A callback function pointer to a function which will
*       be called when an update has occurred to a bit matching the request
*       mask.
*   pvCallbackArg - An argument which is provided to the callback
*       function when it is called.
*
*****************************************************************************/
void SMSU_vInitialize (
    SMSU_EVENT_STRUCT *psEvent,
    void *pvObject,
    SMSAPI_EVENT_MASK tInitialMask,
    SMSAPI_EVENT_MASK tRequestMask,
    SMSAPI_OBJECT_EVENT_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    static UN32 un32Instance = 0;

    // Initialize structure elements
    psEvent->tMask = tInitialMask;
    psEvent->tPolledMask = tInitialMask;
    psEvent->tInitialMask = tInitialMask;
    psEvent->tFilterMask = SMS_OBJECT_EVENT_NONE;
    psEvent->pvObject = pvObject;
    psEvent->bInActiveCallbacks = FALSE;
    psEvent->hCurCallback = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Create unique name for callback list
    snprintf(&acName[0], sizeof(acName),
             "SMSUCallbackList:%u", un32Instance++);

    // Create callback list
    eReturnCode = OSAL.eLinkedListCreate(
        &psEvent->hCallbackList,
        &acName[0],
        NULL,
        OSAL_LL_OPTION_CIRCULAR|OSAL_LL_OPTION_UNIQUE
            );
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Add the RequestMask-Callback pair
        SMSU_bRegisterCallback(psEvent,
                               tRequestMask,
                               vCallback,
                               pvCallbackArg,
                               TRUE);
    }

    return;
}

/*****************************************************************************
*
*   SMSU_bNotify
*
*****************************************************************************/
BOOLEAN SMSU_bNotify (
    SMSU_EVENT_STRUCT *psEvent
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bChanged = FALSE;
    SMSU_EVENT_NOTIFICATION_ARG_STRUCT sNotifyArg;

    sNotifyArg.psEvent = psEvent;
    sNotifyArg.tCumulativeMask = SMS_OBJECT_EVENT_NONE;

    eReturnCode = OSAL.eLinkedListIterate(
        psEvent->hCallbackList,
        bNotifyIterator,
        &sNotifyArg);

    if(eReturnCode == OSAL_SUCCESS)
    {
         // Clear update mask, with those we provided notification for
         psEvent->tMask &= ~sNotifyArg.tCumulativeMask;
         bChanged = TRUE;

         // Clean up any notifications which have gone in-active
         vRemoveInactiveCallbacks(psEvent);
    }
    else if (eReturnCode == OSAL_NO_OBJECTS)
    {
        SMSAPI_EVENT_MASK tEventMask;

        // Mask what they requested and apply any required filters
        tEventMask = psEvent->tMask & ~(psEvent->tFilterMask);

        // Has data changed?
        if(tEventMask != SMS_OBJECT_EVENT_NONE)
        {
            // Indicate event mask has changed and notification will
            // be sent.
            bChanged =  TRUE;
        }
    }

    return bChanged;
}

/*****************************************************************************
*
*   SMSU_bNotifyCallback
*
*****************************************************************************/
BOOLEAN SMSU_bNotifyCallback (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_OBJECT_EVENT_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bChanged = FALSE;

    if (vCallback != NULL &&
        psEvent->hCallbackList != OSAL_INVALID_OBJECT_HDL)
    {
        SMSU_EVENT_CALLBACK_STRUCT sCallbackToFind;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Construct a SMSU_EVENT_CALLBACK_STRUCT to use to find
        // see if this callback exists
        sCallbackToFind.vCallback = vCallback;
        sCallbackToFind.pvCallbackArg = pvCallbackArg;

        // Look to see if this callback exists
        eReturnCode = OSAL.eLinkedListLinearSearch(
                psEvent->hCallbackList,
                &hEntry,
                n16FindActiveCallbacks,
                (void *)&sCallbackToFind );

        if(eReturnCode == OSAL_SUCCESS)
        {
            SMSU_EVENT_NOTIFICATION_ARG_STRUCT sNotifyArg;
            SMSU_EVENT_CALLBACK_STRUCT *psCallback;

            // We found the callback we want to notify
            psCallback =
                 (SMSU_EVENT_CALLBACK_STRUCT *)OSAL.pvLinkedListThis(hEntry);

            // Notify this callback
            sNotifyArg.psEvent = psEvent;
            sNotifyArg.tCumulativeMask = SMS_OBJECT_EVENT_NONE;

            bChanged = bNotifyIterator(psCallback, &sNotifyArg);

            // Clean up any notifications which have gone in-active
            vRemoveInactiveCallbacks(psEvent);
        }
    }

    return bChanged;
}

/*****************************************************************************
*
*   SMSU_tMask
*
*****************************************************************************/
__INLINE__ SMSAPI_EVENT_MASK SMSU_tMask (
    SMSU_EVENT_STRUCT *psEvent
        )
{
    SMSAPI_EVENT_MASK tUpdatePolledMask;

    // Capture update mask since last poll
    tUpdatePolledMask = psEvent->tPolledMask;

    // Clear polled update mask, with those we are providing now
    psEvent->tPolledMask &= ~tUpdatePolledMask;

    return tUpdatePolledMask;
}

/*****************************************************************************
*
*   SMSU_tUpdate
*
*****************************************************************************/
SMSAPI_EVENT_MASK SMSU_tUpdate (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_EVENT_MASK tEventMask
        )
{
    if (tEventMask != SMS_OBJECT_EVENT_NONE)
    {
        // Iterate and set the mask for each callback.
        // We don't worry about the error code since we can't
        // do anything about it anyway.
        OSAL.eLinkedListIterate(
            psEvent->hCallbackList,
            bUpdateCallbackStructIterator,
            (void*)tEventMask);
    }

    psEvent->tPolledMask |= tEventMask;

    // Set & return main mask
    return psEvent->tMask |= tEventMask;
}

/*****************************************************************************
*
*   SMSU_tUpdateAll
*
*****************************************************************************/
SMSAPI_EVENT_MASK SMSU_tUpdateAll (
    SMSU_EVENT_STRUCT *psEvent
        )
{
    return SMSU_tUpdate(psEvent, SMS_OBJECT_EVENT_ALL);
}

/*****************************************************************************
*
*   SMSU_tFilter
*
* This function allows the caller to filter out update events by providing
* a filter mask. This function returns the filter mask prior to setting it
* to allow callers to retain the original filter mask so they can restore
* it later if necessary.
*
*****************************************************************************/
__INLINE__ SMSAPI_EVENT_MASK SMSU_tFilter (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_EVENT_MASK tFilterMask
        )
{
    SMSAPI_EVENT_MASK tPriorMask = psEvent->tFilterMask;

    // Set new filter mask with events provided.
    psEvent->tFilterMask |= tFilterMask;

    // Return previous mask (prior to filtering)
    return tPriorMask;
}

/*****************************************************************************
*
*   SMSU_tUnFilter
*
* This function allows the caller to unfilter update events by providing
* a filter mask. This function returns the filter mask prior to setting it
* to allow callers to retain the original filter mask so they can restore
* it later if necessary.
*
*****************************************************************************/
__INLINE__ SMSAPI_EVENT_MASK SMSU_tUnFilter (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_EVENT_MASK tUnFilterMask
        )
{
    SMSAPI_EVENT_MASK tPriorMask = psEvent->tFilterMask;

    // Clear filter mask of events provided.
    psEvent->tFilterMask &= ~tUnFilterMask;

    // Return previous mask (prior to filtering)
    return tPriorMask;
}

/*****************************************************************************
*
*   SMSU_bModifyRequestMask
*
*****************************************************************************/
BOOLEAN SMSU_bModifyRequestMask (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_OBJECT_EVENT_CALLBACK vCallback,
    void *pvCallbackArg,
    SMSAPI_EVENT_MASK tNewRequestMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        )
{
    BOOLEAN bChanged = FALSE;

    if (psEvent->hCallbackList != OSAL_INVALID_OBJECT_HDL)
    {
        SMSU_EVENT_REQUEST_MASK_MOD_STRUCT sModificationDesc;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        sModificationDesc.tMask = tNewRequestMask;
        sModificationDesc.eModification = eModification;

        if (vCallback == NULL)
        {
            // If the callback is null, then that means the user
            // wants to modify the request mask for all callback
            // functions
            eReturnCode = OSAL.eLinkedListIterate(
                psEvent->hCallbackList,
                bModifyEventMaskIterator,
                &sModificationDesc);

            if((eReturnCode == OSAL_SUCCESS) ||
               (eReturnCode == OSAL_NO_OBJECTS))
            {
                 bChanged = TRUE;
            }
        }
        else
        {
            OSAL_LINKED_LIST_ENTRY hEntry =
                OSAL_INVALID_LINKED_LIST_ENTRY;
            SMSU_EVENT_CALLBACK_STRUCT sCallbackToFind;

            // Construct a SMSU_EVENT_CALLBACK_STRUCT to use to find
            // the structure to modify
            sCallbackToFind.vCallback = vCallback;
            sCallbackToFind.pvCallbackArg = pvCallbackArg;

            // Look to see if this callback exists
            eReturnCode = OSAL.eLinkedListLinearSearch(
                    psEvent->hCallbackList,
                    &hEntry,
                    n16CompareCallbacks,
                    (void *)&sCallbackToFind );

            if(eReturnCode == OSAL_SUCCESS)
            {
                SMSU_EVENT_CALLBACK_STRUCT *psCallbackToModify;

                // We found the callback we want to modify
                psCallbackToModify =
                     (SMSU_EVENT_CALLBACK_STRUCT *)OSAL.pvLinkedListThis(hEntry);

                // Since we are only modifying a single element in the list
                // call the iterator function directly
                bChanged = bModifyEventMaskIterator(
                    (void*)psCallbackToModify, (void*)&sModificationDesc);
            }
        }
    }

    return bChanged;
}

/*****************************************************************************
*
*   SMSU_bRegisterCallback
*
*****************************************************************************/
BOOLEAN SMSU_bRegisterCallback (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_EVENT_MASK tRequestMask,
    SMSAPI_OBJECT_EVENT_CALLBACK vCallback,
    void *pvCallbackArg,
    BOOLEAN bSetInitialEvents
        )
{
    BOOLEAN bAdded = FALSE;

    if (vCallback != NULL &&
        psEvent->hCallbackList != OSAL_INVALID_OBJECT_HDL)
    {
        SMSU_EVENT_CALLBACK_STRUCT *psNewCallback;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntryFound =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        SMSU_EVENT_CALLBACK_STRUCT sCallbackToFind;

        // Construct a SMSU_EVENT_CALLBACK_STRUCT to use to find
        // the structure to modify
        sCallbackToFind.vCallback = vCallback;
        sCallbackToFind.pvCallbackArg = pvCallbackArg;

        eReturnCode = OSAL.eLinkedListLinearSearch(
                psEvent->hCallbackList,
                &hEntryFound,
                n16CompareCallbacks,
                (void *)&sCallbackToFind );

        if(eReturnCode == OSAL_SUCCESS)
        {
            psNewCallback =
                 (SMSU_EVENT_CALLBACK_STRUCT *)
                    OSAL.pvLinkedListThis(hEntryFound);

            // Already there, so just silently succeed
            psNewCallback->bActive = TRUE;
            bAdded = TRUE;
        }
        else
        {
            // Create an entry for it
            psNewCallback = (SMSU_EVENT_CALLBACK_STRUCT *)
                            SMSO_hCreate(
                                  "SMSUCallbackEntry",
                                  sizeof(SMSU_EVENT_CALLBACK_STRUCT),
                                  SMS_INVALID_OBJECT,
                                  FALSE);

            // Set the member values of the callback
            // if we were able to create it
            if (psNewCallback != NULL)
            {
                psNewCallback->bActive = TRUE;
                psNewCallback->tRequestMask = tRequestMask;

                if (bSetInitialEvents == TRUE)
                {
                    psNewCallback->tMask = psEvent->tInitialMask;
                }
                else
                {
                    psNewCallback->tMask = ((SMSAPI_EVENT_MASK)0);
                }

                psNewCallback->vCallback = vCallback;
                psNewCallback->pvCallbackArg = pvCallbackArg;

                // Add the callback to our linked list
                eReturnCode = OSAL.eLinkedListAdd(
                    psEvent->hCallbackList,
                    &psNewCallback->hCallbackEntry,
                    psNewCallback);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    bAdded = TRUE;
                }
                else
                {
                    // Error!
                    SMSO_vDestroy((SMS_OBJECT)psNewCallback);
                }
            }
        }
    }

    return bAdded;
}

/*****************************************************************************
*
*   SMSU_bIsCallbackPending
*
*****************************************************************************/
BOOLEAN SMSU_bIsCallbackPending (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_OBJECT_EVENT_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bPending = FALSE;

    if (vCallback != NULL &&
        psEvent->hCallbackList != OSAL_INVALID_OBJECT_HDL)
    {
        SMSU_EVENT_CALLBACK_STRUCT sCallbackToFind;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Construct a SMSU_EVENT_CALLBACK_STRUCT to use to find
        // see if this callback exists
        sCallbackToFind.vCallback = vCallback;
        sCallbackToFind.pvCallbackArg = pvCallbackArg;

        // Look to see if this callback exists
        eReturnCode = OSAL.eLinkedListLinearSearch(
                psEvent->hCallbackList,
                &hEntry,
                n16FindActiveCallbacks,
                (void *)&sCallbackToFind );

        if(eReturnCode == OSAL_SUCCESS)
        {
            SMSAPI_EVENT_MASK tCurMask;
            SMSU_EVENT_CALLBACK_STRUCT *psCallback;

            // We found the callback we want to check the status of
            psCallback =
                 (SMSU_EVENT_CALLBACK_STRUCT *)OSAL.pvLinkedListThis(hEntry);

            // See what the status of the mask is
            tCurMask = psCallback->tMask &
                            psCallback->tRequestMask;

            // Is there currently bits set waiting?
            if(tCurMask != SMS_OBJECT_EVENT_NONE)
            {
                bPending = TRUE;
            }
        }
    }

    return bPending;
}

/*****************************************************************************
*
*   SMSU_bUnregisterCallback
*
*****************************************************************************/
BOOLEAN SMSU_bUnregisterCallback (
    SMSU_EVENT_STRUCT *psEvent,
    SMSAPI_OBJECT_EVENT_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bRemoved = FALSE;

    if (vCallback != NULL &&
        psEvent->hCallbackList != OSAL_INVALID_OBJECT_HDL)
    {
        SMSU_EVENT_CALLBACK_STRUCT sCallbackToFind;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Construct a SMSU_EVENT_CALLBACK_STRUCT to use to find
        // see if this callback exists
        sCallbackToFind.vCallback = vCallback;
        sCallbackToFind.pvCallbackArg = pvCallbackArg;

        // Look to see if this callback exists
        eReturnCode = OSAL.eLinkedListLinearSearch(
                psEvent->hCallbackList,
                &hEntry,
                n16CompareCallbacks,
                (void *)&sCallbackToFind );

        if(eReturnCode == OSAL_SUCCESS)
        {
            SMSU_EVENT_CALLBACK_STRUCT *psCallbackToRemove;

            // We found the callback we wanted to remove.
            psCallbackToRemove =
                 (SMSU_EVENT_CALLBACK_STRUCT *)OSAL.pvLinkedListThis(hEntry);

            // If we are currently notifying, just flag this as inactive
            // so it can be removed later.
            // Might cause a LL issue if a notification is removed futher up
            // the list while the curent notification is being removed.

            if (psEvent->hCurCallback == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                // We are good to remove the LL entry and callback node.
                bRemoved = bRemoveCallback(psCallbackToRemove);
            }
            else
            {
                // Mark it as not active, but remove later
                psCallbackToRemove->bActive = FALSE;
                psEvent->bInActiveCallbacks = TRUE;
                bRemoved = TRUE;
            }
        }
    }

    return bRemoved;
}

/*****************************************************************************
*
*   SMSU_vDestroy
*
*****************************************************************************/
void SMSU_vDestroy (
    SMSU_EVENT_STRUCT *psEvent
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Clear all the Callback functions from the linked list.
    OSAL.eLinkedListRemoveAll (
                    psEvent->hCallbackList,
                    vReleaseCallbackNode );

    // Delete linked list itself
    eReturnCode = OSAL.eLinkedListDelete (
                    psEvent->hCallbackList );

    if(eReturnCode == OSAL_SUCCESS)
    {
        psEvent->hCallbackList = OSAL_INVALID_OBJECT_HDL;
    }

    return;
}

/*****************************************************************************
*
*   n16CompareCallbacks
*
*****************************************************************************/
static N16 n16CompareCallbacks (
    void *pvObj1,
    void *pvObj2
        )
{
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc1 =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvObj1;
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc2 =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvObj2;
    N16 n16Result = -1; // Forward

    if ((psCallbackDesc1->vCallback == psCallbackDesc2->vCallback) &&
        (psCallbackDesc1->pvCallbackArg == psCallbackDesc2->pvCallbackArg))
    {
        // Found ya!
        n16Result = 0;
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16FindActiveCallbacks
*
*****************************************************************************/
static N16 n16FindActiveCallbacks (
    void *pvObj1,
    void *pvObj2
        )
{
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc1 =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvObj1;
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc2 =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvObj2;
    N16 n16Result = -1; // Forward

    if ((psCallbackDesc1->vCallback == psCallbackDesc2->vCallback) &&
        (psCallbackDesc1->pvCallbackArg == psCallbackDesc2->pvCallbackArg) &&
        (psCallbackDesc1->bActive == TRUE))
    {
        // Found ya!
        n16Result = 0;
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16FindInactiveCallbacks
*
*****************************************************************************/
static N16 n16FindInactiveCallbacks (
    void *pvObj1,
    void *pvObj2
        )
{
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvObj1;
    N16 n16Result = -1; // Forward

    if (psCallbackDesc->bActive == FALSE)
    {
        // Found ya!
        n16Result = 0;
    }

    return n16Result;
}

/*****************************************************************************
*
*   bRemoveCallback
*
*****************************************************************************/
static BOOLEAN bRemoveCallback (
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackToRemove
        )
{
    BOOLEAN bRemoved = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Remove the linked-list item
    eReturnCode = OSAL.eLinkedListRemove(psCallbackToRemove->hCallbackEntry);
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Free the callback entry
        vReleaseCallbackNode(psCallbackToRemove);
        bRemoved = TRUE;
    }

    return bRemoved;
}

/*****************************************************************************
*
*   vReleaseCallbackNode
*
*****************************************************************************/
static void vReleaseCallbackNode (
    void *pvData
        )
{
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvData;

    if (psCallbackDesc != NULL)
    {
        // Clear variables
        psCallbackDesc->bActive = FALSE;
        psCallbackDesc->hCallbackEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psCallbackDesc->tRequestMask = SMS_OBJECT_EVENT_NONE;
        psCallbackDesc->tMask = SMS_OBJECT_EVENT_NONE;
        psCallbackDesc->vCallback = NULL;
        psCallbackDesc->pvCallbackArg = NULL;

        // Free the callback entry
        SMSO_vDestroy((SMS_OBJECT)psCallbackDesc);
    }

    return;
}

/*****************************************************************************
*
*   bModifyEventMaskIterator
*
*****************************************************************************/
static BOOLEAN bModifyEventMaskIterator (
    void *pvData,
    void *pvArg
        )
{
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvData;
    SMSU_EVENT_REQUEST_MASK_MOD_STRUCT *psModification =
        (SMSU_EVENT_REQUEST_MASK_MOD_STRUCT *)pvArg;

    if (psCallbackDesc == NULL || psModification == NULL)
    {
        return FALSE;
    }

    switch (psModification->eModification)
    {
            case SMSAPI_MODIFY_EVENT_MASK_REPLACE:
                psCallbackDesc->tRequestMask = psModification->tMask;
            break;
            case SMSAPI_MODIFY_EVENT_MASK_ENABLE:
                psCallbackDesc->tRequestMask |= psModification->tMask;
            break;
            case SMSAPI_MODIFY_EVENT_MASK_DISABLE:
                psCallbackDesc->tRequestMask &= ~psModification->tMask;
            break;
            default:
                printf("SMSU: eModification is out of bounds (%u) \n",psModification->tMask );
            break;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateCallbackStructIterator
*
*****************************************************************************/
static BOOLEAN bUpdateCallbackStructIterator (
    void *pvData,
    void *pvArg
        )
{
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvData;
    SMSAPI_EVENT_MASK tEventMask =
        (SMSAPI_EVENT_MASK)pvArg;

    if (psCallbackDesc == NULL)
    {
        return FALSE;
    }

    psCallbackDesc->tMask |= tEventMask;

    return TRUE;
}

/*****************************************************************************
*
*   bNotifyIterator
*
*****************************************************************************/
static BOOLEAN bNotifyIterator (
    void *pvData,
    void *pvArg
        )
{
    SMSU_EVENT_CALLBACK_STRUCT *psCallbackDesc =
        (SMSU_EVENT_CALLBACK_STRUCT *)pvData;
    SMSU_EVENT_NOTIFICATION_ARG_STRUCT *psNotifyArg =
        (SMSU_EVENT_NOTIFICATION_ARG_STRUCT *)pvArg;

    SMSAPI_EVENT_MASK tEventMask;

    if (psCallbackDesc == NULL || psNotifyArg->psEvent == NULL)
    {
        return FALSE;
    }

    // Only process notification if it is active
    if (psCallbackDesc->bActive == TRUE)
    {
        // Mask what they requested and apply any required filters
        tEventMask = psCallbackDesc->tMask &
                        psCallbackDesc->tRequestMask &
                            ~psNotifyArg->psEvent->tFilterMask;

        // Add new updates to last polled update mask.
        psNotifyArg->psEvent->tPolledMask |= tEventMask;

        // Has data changed?
        if(tEventMask != SMS_OBJECT_EVENT_NONE)
        {
            // Has a notification callback been registered?
            if(psCallbackDesc->vCallback != (SMSAPI_OBJECT_EVENT_CALLBACK)NULL)
            {
                // Mark the current callback
                psNotifyArg->psEvent->hCurCallback = psCallbackDesc->hCallbackEntry;

                // Notify of change...
                psCallbackDesc->vCallback(
                    psNotifyArg->psEvent->pvObject,
                    tEventMask,
                    psCallbackDesc->pvCallbackArg
                        );

                // Clear our mask value now that the notification has
                // happened
                psCallbackDesc->tMask &= ~tEventMask;

                // Update our Cumulative mask with the events this
                // callback was notified about.
                // This cumulative value will be used later in bNotify to
                // clear the SMSU_EVENT tMask value.
                psNotifyArg->tCumulativeMask |= tEventMask;

                // Mark that we aren't notifying this callback any longer
                psNotifyArg->psEvent->hCurCallback = OSAL_INVALID_LINKED_LIST_ENTRY;
            }
        }
    }

    return TRUE;
}


/*****************************************************************************
*
*   vRemoveInactiveCallbacks
*
*****************************************************************************/
static void vRemoveInactiveCallbacks (
    SMSU_EVENT_STRUCT *psEvent
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Do we have any inactive callbacks?
    if(TRUE == psEvent->bInActiveCallbacks)
    {
        do
        {
            OSAL_LINKED_LIST_ENTRY hEntry =
                OSAL_INVALID_LINKED_LIST_ENTRY;

            // Look to see if there are any in-active callbacks
            eReturnCode = OSAL.eLinkedListLinearSearch(
                    psEvent->hCallbackList,
                    &hEntry,
                    n16FindInactiveCallbacks,
                    NULL );
            if(eReturnCode == OSAL_SUCCESS)
            {
                SMSU_EVENT_CALLBACK_STRUCT *psCallback;

                // We found an inactive entry
                psCallback =
                     (SMSU_EVENT_CALLBACK_STRUCT *)OSAL.pvLinkedListThis(hEntry);

                // Remove this entry
                bRemoveCallback(psCallback);
            }

        } while (eReturnCode == OSAL_SUCCESS);

        // Check if we have no more
        if(eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // Apparently we have no more
            psEvent->bInActiveCallbacks = FALSE;
        }
    }

    return;
}
