/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains various APIs and functions which implement the
 *  Sirius Module Services Task.
 *
 ******************************************************************************/

#include <time.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_event.h"
#include "sms_obj.h"
#include "sms_update.h"

#include "sms_task.h"
#include "_sms_task.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
*
*   SMST_hInstall
*
*****************************************************************************/
SMS_TASK_HANDLE SMST_hInstall (
    SMS_OBJECT hSMS,
    const SMS_TASK_CONFIGURATION_STRUCT *psParameters,
    SMS_OBJECT_EVENT_HANDLER_PROTOTYPE vEventHandler,
    SMS_EVENT_HANDLER *phEventHdlr
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    SMS_TASK_CONTROL_STRUCT *psCtrl;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // For this function to work, the caller must be the owner of the
    // SMS Object being provided to us. They also must provide a valid
    // event control pointer.
    bOwner = SMSO_bOwner((SMS_OBJECT)hSMS);
    if((bOwner == FALSE) || (phEventHdlr == NULL))
    {
        // Error!
        return SMS_INVALID_TASK_HANDLE;
    }

    // Allocate SMS object for control task
    psCtrl = (SMS_TASK_CONTROL_STRUCT *)
        SMSO_hCreate(
            psParameters->pacName,
            sizeof(SMS_TASK_CONTROL_STRUCT),
            hSMS, // Parent
            FALSE); // Ignored (parent's lock is inherited)
    if(psCtrl == NULL)
    {
        // Error!
        return SMS_INVALID_TASK_HANDLE;
    }

    // Initialize control structure...

    // Copy task/resource name
    snprintf(&psCtrl->acName[0], sizeof(psCtrl->acName), "%s",
        &psParameters->pacName[0]);

    // Initialize module
    psCtrl->hSMS = hSMS;

    // Extract parameters
    psCtrl->un32EventQueueSize = psParameters->un32EventQueueSize;
    psCtrl->un32EventHandlerOptions = psParameters->un32EventHandlerOptions;

    // Definition for the maximum duration to allow a task to run without
    // checking in with the task monitor. This duration must be less
    // than 1/2 the check-in interval. The 'less-than' part is important because
    // assume the interval is 10 seconds, and "tick" time is 10 msec.
    // After 4990 msec a task runs but since 5000 msec have not elapsed, the
    // task does not check-in. Instead it pends for another 5000 msec, making a
    // total of 9990 msec before it checks in again. Due to the acuracy of the
    // internal clock ticker which is +/- 10 msec the task may just miss the
    // next check-in interval and cause the task monitor to fail. Thus the rule
    // of thumb here should be the task should check-in no less than 2 times per
    // check-in interval (not once, which is what happens when 1/2 the interval
    // is used.
    psCtrl->un16ReportingInterval = psParameters->un16ReportingInterval;

    // Any number less than 3 seconds is just too low
    if(psCtrl->un16ReportingInterval >= 3)
    {
        // Calculate timeout
        psCtrl->un16Timeout =
            ((psParameters->un16ReportingInterval + 1) / 2) - 1;
    }
    else
    {
        // Instead select a timeout of the reporting interval
        psCtrl->un16Timeout = psCtrl->un16ReportingInterval;
        psCtrl->un16ReportingInterval = 0; // Disable reporting requirement
    }

    // Provided event handler
    psCtrl->vEventHandler = vEventHandler;
    if(psCtrl->vEventHandler == NULL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_TASK_NAME
            ": Cannot install with no event handler.");
        SMST_vUninstall((SMS_TASK_HANDLE)psCtrl);
        return SMS_INVALID_TASK_HANDLE;
    }

    // Initialize install complete semaphore
    psCtrl->hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;

    // Initialize task handle
    psCtrl->hTask = OSAL_INVALID_OBJECT_HDL;

    // Initialize event control handle
    psCtrl->phEventHdlr = phEventHdlr;
    *psCtrl->phEventHdlr = SMS_INVALID_EVENT_HANDLER;

    // Copy task/resource name
    snprintf(&acName[0], sizeof(acName), "%s:Install",
        &psParameters->pacName[0]);

    // Create install complete semaphore
    eReturnCode =
        OSAL.eSemCreate(
            &psCtrl->hInstallSemaphore, &acName[0],
            0, 1, OSAL_SEM_OPTION_NONE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Give up ownership, new task will take it from here...
        SMSO_vUnlock(hSMS);

        // Create the task
        eReturnCode = OSAL.eTaskCreate(
                &psCtrl->hTask,
                &psCtrl->acName[0],
                SMST_n32Task,
                (void*)psCtrl,
                psParameters->un32StackSize,
                psParameters->ePriority,
                psParameters->un32Options
                    );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Wait for task to install
            eReturnCode = OSAL.eSemTake(
                          psCtrl->hInstallSemaphore,
                          OSAL_OBJ_TIMEOUT_INFINITE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Task has been installed and initialized.
                return (SMS_TASK_HANDLE)psCtrl;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_TASK_NAME": Cannot take install semaphore.");
            }
        }
        else
        {
            BOOLEAN bLocked;

            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_TASK_NAME": Cannot create Task: %d (%s)",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));

            // Lock the object
            bLocked = SMSO_bLock(hSMS, OSAL_OBJ_TIMEOUT_INFINITE);
            if (TRUE == bLocked)
            {
                SMST_vUninstall((SMS_TASK_HANDLE)psCtrl);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_TASK_NAME": Unable to lock control object.");
            }
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_TASK_NAME": Cannot create install semaphore.");
        SMST_vUninstall((SMS_TASK_HANDLE)psCtrl);
    }

    return SMS_INVALID_TASK_HANDLE;
}

/*****************************************************************************
*
*   SMST_vUninstall
*
*****************************************************************************/
void SMST_vUninstall(
    SMS_TASK_HANDLE hTask
        )
{
    SMS_TASK_CONTROL_STRUCT *psCtrl =
        (SMS_TASK_CONTROL_STRUCT *)hTask;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bValid;

    // Just check validity
    bValid = SMSO_bValid((SMS_OBJECT)hTask);
    if(bValid == FALSE)
    {
        // Error!
        return;
    }

    // Request kill of SMS Task, if it was created
    if(psCtrl->hTask != OSAL_INVALID_OBJECT_HDL)
    {
        // Saving task handle into local variable
        OSAL_OBJECT_HDL hTask = psCtrl->hTask;

        // Cleaning up handle before the call to OSAL.eTaskDelete(). If it is
        // not done here, psCtrl could be already destroyed after return from
        // the function.
        psCtrl->hTask = OSAL_INVALID_OBJECT_HDL;
        eReturnCode = OSAL.eTaskDelete(hTask);
        if(eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(__FILE__, __LINE__,
                SMS_TASK_NAME": Error: Cannot delete task %p",
                hTask);
        }
    }
    else
    {
        // Task does not exist but some resources still need to be released

        // Destroy the semaphore (no longer needed)
        if(psCtrl->hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eSemDelete(psCtrl->hInstallSemaphore);
            psCtrl->hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }

        // Destroy object (no need to unlock, cuz ya can't anyway)
        SMSO_vDestroy((SMS_OBJECT)psCtrl);
    }

    return;
}

/*******************************************************************************
*
*   SMST_n32Task
*
*****************************************************************************/
static N32 SMST_n32Task ( void *pvArg )
{
    SMS_TASK_CONTROL_STRUCT *psCtrl =
        (SMS_TASK_CONTROL_STRUCT *)pvArg;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bTaskInitialized, bLocked;
    UN32 un32TimeSec = 0, un32LastReportTimeSec = 0;
    N32 n32Timeout;
    SMS_EVENT_RESULT_ENUM eEventResult = SMS_EVENT_RESULT_OK;
    BOOLEAN bInitComplete = FALSE;

    // Check input, by locking access to it. If we cannot get access
    // immediately then something is wrong.
    bLocked = SMSO_bLock((SMS_OBJECT)psCtrl, OSAL_OBJ_TIMEOUT_NONE);
    if(bLocked == FALSE)
    {
        // Error!
        return (N32)SMS_EVENT_RESULT_ERROR;
    }

    // Initialize this task!
    bTaskInitialized = SMST_bInitialize(psCtrl);
    if( bTaskInitialized == FALSE )
    {
        // Error!
        SMST_vUninitialize(psCtrl);
        return (N32)SMS_EVENT_RESULT_ERROR;
    }

    // Calculate the queue timeout
    if (psCtrl->un16Timeout == 0)
    {
        // Don't worry about a timeout here
        n32Timeout = OSAL_OBJ_TIMEOUT_INFINITE;
    }
    else
    {
        n32Timeout = (N32)(psCtrl->un16Timeout * 1000);
    }

    // Unlock access to control structure
    SMSO_vUnlock((SMS_OBJECT)psCtrl);

    // Forever
    for(;;)
    {
        // Dispatch events as needed.
        // Get a message from the Event Queue, and process it
        eEventResult = SMSE_eDispatchEvent(
            *psCtrl->phEventHdlr, n32Timeout);

        // Check if initialization is complete and needs to run
        // (one time only). We do this only after the first
        // event is completed.
        if( (bInitComplete == FALSE) &&
            (eEventResult == SMS_EVENT_RESULT_OK))
        {
            // Indicate task is installed by releasing the semaphore
            eReturnCode = OSAL.eSemGive(psCtrl->hInstallSemaphore);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Change flag to indicate init is complete
                // and not to do this again.
                bInitComplete = TRUE;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_TASK_NAME": %s is unable to release semaphore (%s)\n",
                    &psCtrl->acName[0], OSAL.pacGetReturnCodeName(eReturnCode));

                eEventResult = SMS_EVENT_RESULT_ERROR;
            }
        }

        // Check if this task needs to shutdown
        if(eEventResult != SMS_EVENT_RESULT_OK)
        {
            // Check result code.
            if(eEventResult == SMS_EVENT_RESULT_SHUTDOWN)
            {
                // Recondition result, if it's shutdown it is still
                // intentional and not an error.
                eEventResult = SMS_EVENT_RESULT_OK;
            }

            // Exit task an any error, or shutdown request
            break;
        }

        /***************************************/
        // Track report time and report if needed
        OSAL.vTimeUp( &un32TimeSec, (UN16 *)NULL );

        if ( ( ( un32TimeSec - un32LastReportTimeSec )
           >= psCtrl->un16Timeout ) )
        {
            // Timeout interval has passed since the
            // last time we reported to the task monitor, so it's
            // time to report again...
            eReturnCode = OSAL.eTaskReport( (N32)eEventResult );
            if (eReturnCode == OSAL_SUCCESS)
            {
                // record time so we can check for the next report time
                un32LastReportTimeSec = un32TimeSec;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_TASK_NAME": %s is unable to check in (%s).",
                   &psCtrl->acName[0], OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        /***************************************/

    } // Forever

    // Lock access to the control structure.
    bLocked = SMSO_bLock((SMS_OBJECT)psCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        return (N32)SMS_EVENT_RESULT_ERROR;
    }

    // Unallocate task resources (psCtrl will be destroyed)
    SMST_vUninitialize(psCtrl);

    return (N32)eEventResult;
}

/*****************************************************************************
*
*   SMST_bInitialize
*
*****************************************************************************/
static BOOLEAN SMST_bInitialize( void *pvArg )
{
    SMS_TASK_CONTROL_STRUCT *psCtrl =
        (SMS_TASK_CONTROL_STRUCT *)pvArg;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess;

    // Call Event Handler Initialization
    bSuccess =
        SMSE_bEventHdlrInit(
            &psCtrl->acName[0],
            psCtrl->hSMS,
            psCtrl->phEventHdlr,
            psCtrl->un32EventQueueSize,
            psCtrl->vEventHandler,
            psCtrl->un32EventHandlerOptions
                );
    if(bSuccess == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Register Task with OSAL
    eReturnCode =
        OSAL.eTaskRegister(
            psCtrl->un16ReportingInterval,
            SMST_vSleepHandler,
            SMST_vWakeupHandler,
            SMST_vShutdownHandler,
            psCtrl );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        return FALSE;
    }

    // All done
    return TRUE;
}

/*****************************************************************************
*
*   SMST_vUninitialize
*
*****************************************************************************/
static void SMST_vUninitialize( void *pvArg )
{
    SMS_TASK_CONTROL_STRUCT *psCtrl =
        (SMS_TASK_CONTROL_STRUCT *)pvArg;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SMS_OBJECT hSMS;

    // Unregister Task with OSAL
    OSAL.eTaskUnregister();

    // Uninitialize event handler
    if(*psCtrl->phEventHdlr != SMS_INVALID_EVENT_HANDLER)
    {
        SMSE_vEventHdlrUninit(*psCtrl->phEventHdlr);
        *psCtrl->phEventHdlr = SMS_INVALID_EVENT_HANDLER;
    }

    // Delete the semaphore (no longer needed) - force post
    if(psCtrl->hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(psCtrl->hInstallSemaphore);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psCtrl->hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Invalidate task handle, to indicate task no longer exists
    psCtrl->hTask = OSAL_INVALID_OBJECT_HDL;

    // Save copy of SMS handle
    hSMS = psCtrl->hSMS;

    // Destroy object
    SMSO_vDestroy((SMS_OBJECT)psCtrl);

    // Destroy object handle provided
    SMSO_vDestroy(hSMS);

    return;
}

/*****************************************************************************
*
*   SMST_vSleepHandler
*
*   This function handles putting a task to sleep
*
*   Inputs:
*       pvArg - A pointer to the main task control structure.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void SMST_vSleepHandler( void *pvArg )
{
    SMS_TASK_CONTROL_STRUCT *psCtrl =
        (SMS_TASK_CONTROL_STRUCT *)pvArg;

    // Post sleep-signal. We do not check the return value because it
    // would have no consequence anyway.
    SMSE_bPostSignal(*psCtrl->phEventHdlr,
        SMS_EVENT_SLEEP, SMS_EVENT_OPTION_NONE);

    return;
}

/*****************************************************************************
*
*   SMST_vWakeupHandler
*
*   This function handles waking a task
*
*   Inputs:
*       pvArg - A pointer to the main task control structure.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void SMST_vWakeupHandler( void *pvArg )
{
    SMS_TASK_CONTROL_STRUCT *psCtrl =
        (SMS_TASK_CONTROL_STRUCT *)pvArg;

    // Post wakeup-signal. We do not check the return value because it
    // would have no consequence anyway.
    SMSE_bPostSignal(*psCtrl->phEventHdlr,
        SMS_EVENT_WAKEUP, SMS_EVENT_OPTION_NONE);

    return;
}

/*****************************************************************************
*
*   SMST_vShutdownHandler
*
*   This function handles shutting a task down
*
*   Inputs:
*       pvArg - A pointer to the main task control structure.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void SMST_vShutdownHandler( void *pvArg )
{
    SMS_TASK_CONTROL_STRUCT *psCtrl =
        (SMS_TASK_CONTROL_STRUCT *)pvArg;

    // Post shutdown-signal. We do not check the return value because it
    // would have no consequence anyway.
    SMSE_bPostSignal(*psCtrl->phEventHdlr,
        SMS_EVENT_SHUTDOWN, SMS_EVENT_OPTION_NONE); // put on queue

    return;
}
