/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SMS implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_obj.h"
#include "_sms_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
*
*   SMSO_hCreate
*
*   This API allows the caller to create an SMS object owned by the caller.
*   The caller must provide an identifying name, the size of the object
*   required (user data, payload, etc.) and the handle of another SMS object
*   which is its parent. If this object is the parent then SMS_INVALID_OBJECT
*   should be provided.
*
*   Inputs:
*       pacName     A pointer to a NULL terminated string which identifies
*                   the use of this object.
*       tSize       The number of bytes required by the caller.
*       hParent     An SMS handle of an associated parent object or
*                   SMS_INVALID_OBJECT if this object is the parent.
*       bLock       If the value of hParent is SMS_INVALID_OBJECT this
*                   parameter is used to indicate if the object
*                   should have the lock feature enabled. If hParent is not
*                   equal to SMS_INVALID_OBJECT this parameter is ignored.
*                   Furthermore if the hParent provided is SMS_INVALID_OBJECT and
*                   bLock is TRUE then the caller of this API also initially owns
*                   the created object and much unlock it when it is done with it.
*
*   Outputs:
*       SMS_OBJECT  A handle to the new SMS object on success. Otherwise
*                   a value of SMS_INVALID_OBJECT is returned.
*
******************************************************************************/
SMS_OBJECT SMSO_hCreate (
    const char *pacName,
    size_t tSize,
    SMS_OBJECT hParent,
    BOOLEAN bLock
        )
{
    SMS_OBJECT_STRUCT *psObj;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    BOOLEAN bValid;

    // Verify inputs
    if(pacName == NULL)
    {
        // Error!
        return SMS_INVALID_OBJECT;
    }

    // Construct an appropriate name for memory to be allocated
    snprintf(&acName[0], sizeof(acName),
        SMS_OBJECT_NAME":%s", pacName);

    // Allocate an SMS object plus the requested memory
    psObj = (SMS_OBJECT_STRUCT *)OSAL.pvMemoryAllocate(
        &acName[0], sizeof(SMS_OBJECT_STRUCT) + tSize, TRUE);
    if(psObj == NULL)
    {
        return SMS_INVALID_OBJECT;
    }

    // Initialize this SMS object...
    *psObj = gsInitObj;
    psObj->ptMutexOwner = &psObj->tMutexOwner;
    psObj->pbLocked = &psObj->bLocked;
    psObj->ptOwnerLockCount = &psObj->tOwnerLockCount;
    psObj->hParent = hParent;
    psObj->tMagicNumber = SMS_OBJECT_MAGIC_NUMBER(psObj);

    // Check if object is linked to a parent object
    bValid = SMSO_bIsValid((SMS_OBJECT)psObj->hParent);
    if(bValid == TRUE)
    {
        BOOLEAN bLocked;

        // Lock the parent object, forcing anyone using this object
        // to actually finish using it before we do anything.
        bLocked = SMSO_bLock(hParent, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // De-reference object
            SMS_OBJECT_STRUCT *psParentObj =
                (SMS_OBJECT_STRUCT *)psObj->hParent - 1;

            // They want this object linked to the parent, so we must
            // use the parent mutex (if any).
            psObj->hMutex = psParentObj->hMutex;

            // The parent object holds the mutex owner
            psObj->ptMutexOwner = psParentObj->ptMutexOwner;
            psObj->pbLocked = psParentObj->pbLocked;

            // The parent object holds the owner lock counter
            psObj->ptOwnerLockCount = psParentObj->ptOwnerLockCount;

            // Increment parent's child count
            psParentObj->tChildCount++;

            // We're done, unlock parent now.
            SMSO_vUnlock(hParent);
        }
        else
        {
            // Error!
            SMSO_vDestroy((SMS_OBJECT)(psObj + 1));

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_OBJECT_NAME": Failed to create object - "
                "parent could not be locked.");
            vHandleFatalError();
            return SMS_INVALID_OBJECT;
        }

    } // if(bValid == TRUE)
    else
    {
        // Check if the locking feature is desired.
        if(bLock == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_TASK_ID tThisTask;
            BOOLEAN bOk;

            // Create mutex...
            // This mutex is initially consumed, as the creator owns it
            // upon success of this API.
            eReturnCode =
                OSAL.eSemCreate(
                    &psObj->hMutex, &acName[0], 0, 1,
                    OSAL_SEM_OPTION_MUTEX
                        );
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSO_vDestroy((SMS_OBJECT)(psObj + 1));

                // FATAL ERROR
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_OBJECT_NAME": Failed to create object lock. (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                vHandleFatalError();
                return SMS_INVALID_OBJECT;
            }

            // Initialize object as owned by the caller.
            bOk = OSAL.bTaskGetId(&tThisTask);
            if (bOk == TRUE)
            {
                psObj->tMutexOwner = tThisTask;
                psObj->bLocked = TRUE;
            }
            else
            {
                // Error!
                SMSO_vDestroy((SMS_OBJECT)(psObj + 1));

                // FATAL ERROR
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_OBJECT_NAME": Failed to create object. "
                    "Could not get task id.");
                vHandleFatalError();
                return SMS_INVALID_OBJECT;
            }
        }
    }

    // Return object to caller
    return (SMS_OBJECT)(psObj + 1);
}

/*****************************************************************************
*
*   SMSO_vDestroy
*
*   This API destroys a previously created SMS Object. The caller must own
*   the object to destroy it. Additionally any locks applied to the object
*   will be removed if the caller provides a parent object. All child objects
*   must be destroyed before the parent object can be destroyed.
*
*   Inputs:
*       hSMS        A handle to the SMS object to destroy
*
*   Outputs:
*       NONE
*
*****************************************************************************/
void SMSO_vDestroy (
    SMS_OBJECT hSMS
        )
{
    // De-reference object
    SMS_OBJECT_STRUCT *psObj =
        (SMS_OBJECT_STRUCT *)hSMS - 1;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;
    size_t tChildCount = !0;

    // Verify input and check object validity. Additionally, make sure
    // the caller actually owns the object. If not they cannot destroy it.
    bOwner = SMSO_bOwner(hSMS);
    if(bOwner == FALSE)
    {
        // FATAL ERROR!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": Failed to destroy obj - caller is not owner.");
        vHandleFatalError();
        return;
    }

    // Check if this object is the top-parent (not a child of anything)
    if(psObj->hParent == SMS_INVALID_OBJECT) // Top-most parent object
    {
        // Check if a mutex is associated with this object
        if(psObj->hMutex != OSAL_INVALID_OBJECT_HDL)
        {
            // Delete resource
            eReturnCode = OSAL.eSemDelete(psObj->hMutex);
            if(eReturnCode == OSAL_SUCCESS)
            {
                psObj->hMutex = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }
    else // This is a child object, dis-associate from parent
    {
        BOOLEAN bLocked;

        // Lock the parent object, forcing anyone using this object
        // to actually finish using it before we do anything.
        bLocked = SMSO_bLock(psObj->hParent, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // Dereference parent object
            SMS_OBJECT_STRUCT *psParentObj =
                (SMS_OBJECT_STRUCT *)psObj->hParent - 1;

            // Decrement parent's child count
            psParentObj->tChildCount--;

            // We're done, unlock parent now.
            SMSO_vUnlock(psObj->hParent);
        }
    }

    // Record locally the child count, we'll need it later
    tChildCount = psObj->tChildCount;

    // Re-initialize internal object data, to keep everything clean
    *psObj = gsInitObj;
    psObj->ptMutexOwner = &psObj->tMutexOwner;
    psObj->pbLocked = &psObj->bLocked;
    psObj->ptOwnerLockCount = &psObj->tOwnerLockCount;
    psObj->tMagicNumber = ~SMS_OBJECT_MAGIC_NUMBER(psObj);

    // Determine if all children have been destroyed, if not something
    // has gone terribly wrong.
    if(tChildCount != 0)
    {
        // FATAL ERROR!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": Not all children have been "
            "destroyed (%u remain).", tChildCount);
        OSALM_vPrintMemoryBlock(psObj);
        vHandleFatalError();
    }

    // Free memory
    OSAL.vMemoryFree(psObj);
    return;
}

/*****************************************************************************
*
*   SMSO_hParent
*
*   This function is used to return an SMS object's parent.
*
*   Inputs:
*       hSMS        A handle to the SMS object to query
*
*   Outputs:
*       SMS_OBJECT  A handle to the SMS object's parent.
*
*****************************************************************************/
SMS_OBJECT SMSO_hParent (
    SMS_OBJECT hSMS
        )
{
    BOOLEAN bOwner;
    SMS_OBJECT hParent = SMS_INVALID_OBJECT;

    // Lock object
    bOwner = SMSO_bOwner(hSMS);
    if(bOwner == TRUE)
    {
        // De-reference object
        SMS_OBJECT_STRUCT *psObj =
            (SMS_OBJECT_STRUCT *)hSMS - 1;

        // Extract and return parent handle
        hParent = psObj->hParent;
    }

    return hParent;
}

/*****************************************************************************
*
*   SMSO_bLock
*
*   This function allows the caller to lock a provided SMS object that has the
*   lock feature enabled. The lock may be part of the object itself or its
*   parent('s). The result is the entire object, including all parent/children
*   are locked. If the object does not have the lock feature or a lock
*   failure occurs a value of FALSE is returned.
*
*   Inputs:
*       hSMS        A handle to the SMS object to lock
*       n32Timeout  A timeout value (msec) to wait for the lock
*
*   Outputs:
*       BOOLEAN     A value of TRUE if the object is locked, otherwise FALSE
*
*****************************************************************************/
BOOLEAN SMSO_bLock (
    SMS_OBJECT hSMS,
    N32 n32Timeout
        )
{
    // De-reference object
    SMS_OBJECT_STRUCT *psObj =
        (SMS_OBJECT_STRUCT *)hSMS - 1;
    BOOLEAN bOk, bValid, bLocked = FALSE;
    OSAL_TASK_ID tThisTaskId;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify input and check object validity
    bValid = SMSO_bIsValid(hSMS);
    if(bValid == FALSE)
    {
        // Error!
        return FALSE;
    }

    // if this object has a lock (a valid mutex) then the caller trying to lock
    // it must be a valid OSAL task
    bOk = OSAL.bTaskGetId(&tThisTaskId);
    if (bOk == FALSE)
    {
        // this obj has the lock attribute but caller is not a valid OSAL task
        // FATAL ERROR!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": Object could not be locked. Could not get task id.");
        vHandleFatalError();
        return FALSE;
    }

    // Lock resource (try - no timeout first)
    eReturnCode = OSAL.eSemTake(psObj->hMutex, OSAL_OBJ_TIMEOUT_NONE);
    if(eReturnCode == OSAL_SEM_NOT_AVAILABLE)
    {
        /********************************************************************/
        eReturnCode = OSAL.eEnterTaskSafeSection();
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Resource is not available, check if it is because we own it
            if( (*psObj->ptMutexOwner == tThisTaskId) &&
                (*psObj->pbLocked == TRUE) )
            {
                // We already own the resource, so nothing else to do
                // Just increment our lock count so we know when to unlock
                // it later.
                (*psObj->ptOwnerLockCount)++;
                bLocked = TRUE;
            }

            // Exit
            OSAL.eExitTaskSafeSection();
        }
        /********************************************************************/
        else
        {
            // Error! Something is wrong.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_OBJECT_NAME
                ": Cannot enter task safe section.");
            vHandleFatalError();
        }
    }
    else if(eReturnCode == OSAL_SUCCESS)
    {
        // We now have the resource, so nothing else to do
        *psObj->ptMutexOwner = tThisTaskId;
        *psObj->pbLocked = TRUE;
        bLocked = TRUE;
    }
    else if(eReturnCode == OSAL_ERROR_INVALID_HANDLE)
    {
        // This object does not have the lock attribute so we
        // consider it "locked" or in other words ok to use.
        bLocked = TRUE;
    }

    // Check if the resource is still unlocked.
    // If so we now have to wait for it (if we wanted to)
    if((bLocked == FALSE) && (n32Timeout != OSAL_OBJ_TIMEOUT_NONE))
    {

#if SMS_OBJECT_TRAP_VIOLATIONS == 1
        n32Timeout = 5000;  // 5 seconds. Force timeout for any task
                            // which may block otherwise indefinitely.
#endif

        // If we got this far, this means the resource is tied up by
        // someone else and it is not us, so we just wait like good
        // little boys...
        // try again (for real) with provided timeout.
        eReturnCode = OSAL.eSemTake(psObj->hMutex, n32Timeout);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // We now are the owner
            *psObj->ptMutexOwner = tThisTaskId;
            *psObj->pbLocked = TRUE;
            bLocked = TRUE;
        }
        else if(eReturnCode == OSAL_TIMEOUT)
        {
            // Timeout
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_OBJECT_NAME
                ": FATAL ERROR! Timeout waiting for resource 'SMSO_bLock'.");
        }
    }

    // Check if FATAL ERROR handler should be called. We call it if the
    // object could not be locked.
    if(bLocked == FALSE)
    {
        // We get here only in the case of an error. So handle the error
        // FATAL ERROR!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": Object could not be locked.");
        vHandleFatalError();
    }

    // Return lock state
    return bLocked;
}

/*****************************************************************************
*
*   SMSO_vUnlock
*
*   This function allows the caller to unlock a provided SMS object that has the
*   lock feature enabled. The lock may be part of the object itself or its
*   parent('s). The result is the entire object, including all parent/children
*   are unlocked.
*
*   Inputs:
*       hSMS        A handle to the SMS object to unlock
*
*   Outputs:
*       None
*
*****************************************************************************/
void SMSO_vUnlock (
    SMS_OBJECT hSMS
        )
{
    // De-reference object
    SMS_OBJECT_STRUCT *psObj =
        (SMS_OBJECT_STRUCT *)hSMS - 1;
    BOOLEAN bValid, bUnlock = FALSE, bOwner = FALSE;

    // Verify input and check object validity
    bValid = SMSO_bValid(hSMS);
    if(bValid == FALSE)
    {
        // Error!
        return;
    }

    // Check if mutex was created
    if(psObj->hMutex != OSAL_INVALID_OBJECT_HDL)
    {
        BOOLEAN bOk;
        OSAL_TASK_ID tThisTaskId;

        bOk = OSAL.bTaskGetId(&tThisTaskId);
        if (bOk == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            /******************************************************************/
            eReturnCode = OSAL.eEnterTaskSafeSection();
            if(eReturnCode == OSAL_SUCCESS)
            {
                // If we are the owner, we can do something with the
                // resource. Otherwise, don't even bother.
                if( (*psObj->ptMutexOwner == tThisTaskId) &&
                    (*psObj->pbLocked == TRUE) )
                {
                    // We are the owner
                    bOwner = TRUE;

                    // Check if the lock count is something other than
                    // zero. If so, this means we should not unlock
                    // the resource, otherwise we may do so.
                    if(*psObj->ptOwnerLockCount == 0)
                    {
                        // We are no longer the owner of this resource
                        *psObj->ptMutexOwner = (OSAL_TASK_ID)0;
                        *psObj->pbLocked = FALSE;

                        // Unlock resource
                        bUnlock = TRUE;
                    }
                    else
                    {
                        // We are the owner so we may decrement
                        // the lock count
                        (*psObj->ptOwnerLockCount)--;
                    }
                }

                // Exit
                OSAL.eExitTaskSafeSection();
            }
            /********************************************************************/
            else
            {
                // Error! Something is wrong.
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_OBJECT_NAME
                    ": Cannot enter task safe section.");
                vHandleFatalError();
            }
        }
        else
        {
            // we couldn't get the task id, so we can't
            // verify ownership.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_OBJECT_NAME": Could not get task id.");
            vHandleFatalError();
        }
    }
    else
    {
        // This object does not have the lock attribute so we
        // consider it "unlocked" and the caller is the owner.
        bOwner = TRUE;
    }

    // Check if we can/should unlock the resource
    if(bUnlock == TRUE)
    {
        // Unlock resource...
        OSAL.eSemGive(psObj->hMutex);
    }
    else if(bOwner == FALSE)
    {
        // Someone tried to unlock a resource they do not own.

        // FATAL ERROR!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": Caller cannot unlock resouce it doesn't own.");
        vHandleFatalError();
    }

    return;
}

/*****************************************************************************
*
*   SMSO_bValid
*
*   This function is used to check an SMS object's validity. If the object
*   is found to be invalid, a FATAL error is recorded and FALSE is returned.
*   Otherwise TRUE is returned if the object is valid. This API is used
*   more oftent than the bIsValid variety. This is because the caller
*   does not expect the result to be FALSE as it is usually a precursor
*   to any further operations with will operate on the object being tested.
*
*   Inputs:
*       hSMS        A handle to the SMS object to check.
*
*   Outputs:
*       BOOLEAN     A value of TRUE if the object is valid, otherwise FALSE
*
*****************************************************************************/
// NOTE: See header file for MACRO

#if SMSO_INCLUDE_OBJECT_CHECKING == 1

BOOLEAN SMSO_bValid (
    SMS_OBJECT hSMS
        )
{
    // Return validity result(do not enforce)
    return bValid(hSMS, FALSE);
}

#endif

/*****************************************************************************
*
*   SMSO_bIsValid
*
*   This function is used to check an SMS object's validity. If the object
*   is found to be invalid, FALSE is returned. Otherwise TRUE is returned if
*   the object is valid. This API does not record any error as a result.
*   This is to be used more rarely than the bValid call. This is because
*   we are calling this to make a decision on an object's validity and in
*   some cases it is possible for the object to be invalid, and that is ok.
*
*   Inputs:
*       hSMS        A handle to the SMS object to check.
*
*   Outputs:
*       BOOLEAN     A value of TRUE if the object is valid, otherwise FALSE
*
*****************************************************************************/
BOOLEAN SMSO_bIsValid (
    SMS_OBJECT hSMS
        )
{
    // Return validity result(do not enforce)
    return bValid(hSMS, FALSE);
}

/*****************************************************************************
*
*   SMSO_bOwner
*
*   This API allows the caller to test if it is the owner of an object
*   Callers may use this to determine if the context from which this API
*   is called matches the owner of the object. This API is used
*   more oftent than the bIsOwner variety. This is because the caller
*   does not expect the result to be FALSE as it is usually a precursor
*   to any further operations with will operate on the object being tested,
*   and that object must be owned by the caller.
*
*   Inputs:
*       hSMS        A handle to the SMS object to test ownership of
*
*   Outputs:
*       BOOLEAN     TRUE if the object and caller context match.
*                   FALSE otherwise.
*
*****************************************************************************/
BOOLEAN SMSO_bOwner (
    SMS_OBJECT hSMS
        )
{

#if SMSO_INCLUDE_OBJECT_CHECKING == 1

    // Return ownership result(enforce)
    return bOwner(hSMS, TRUE);

#else

    // Reduce owner checks to simply checks
    // for validity without enforcement (cheaper!).
    return bValid(hSMS, FALSE);

#endif

}

/*****************************************************************************
*
*   SMSO_bTaskOwner
*
*   This API allows the caller to check if it is the owner of an object and
*   also verify operational context by providing a reference task ID. If
*   the caller isn't the owner, this returns FALSE but no fatal error is
*   fired.
*
*   However, the point of this API is to ensure correct task context when
*   object ownership is verified.  If the task context is incorrect AND
*   the object is owned, a fatal error is fired.
*
*   Callers may use this to determine if the context from which this API
*   is called matches the owner of the object.
*   This is to be used more rarely than the bOwner call. This is because
*   we are calling this to make a decision on an object's validity and in
*   some cases it is possible for the object to not be owned, and that is ok.
*
*   Inputs:
*       hSMS        A handle to the SMS object to test ownership of
*       tTask       The task of the caller.  This task must match the
*                   object owner task.
*
*   Outputs:
*       BOOLEAN     TRUE if the object, caller context, and task ids match.
*                   FALSE otherwise.
*
*****************************************************************************/
BOOLEAN SMSO_bTaskOwner (
    SMS_OBJECT hSMS,
    OSAL_TASK_ID tTask
        )
{
    BOOLEAN bTaskOwner = FALSE,
            bObjectOwner;

    // First, do we own the object?
    bObjectOwner = SMSO_bIsOwner(hSMS);
    if (bObjectOwner == TRUE)
    {
        // De-reference object
        SMS_OBJECT_STRUCT *psObj =
            (SMS_OBJECT_STRUCT *)hSMS - 1;

        // Compare the provided task id with the lock task id
        if (*(psObj->ptMutexOwner) == tTask)
        {
            bTaskOwner = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_OBJECT_NAME": SMSO_bTaskOwner: object owned but incorrect context.");
            vHandleFatalError();
        }
    }

    return bTaskOwner;
}

/*****************************************************************************
*
*   SMSO_bIsOwner
*
*   This API allows the caller to check if it is the owner of an object
*   Callers may use this to determine if the context from which this API
*   is called matches the owner of the object.
*   This is to be used more rarely than the bOwner call. This is because
*   we are calling this to make a decision on an object's validity and in
*   some cases it is possible for the object to not be owned, and that is ok.
*
*   Inputs:
*       hSMS        A handle to the SMS object to test ownership of
*
*   Outputs:
*       BOOLEAN     TRUE if the object and caller context match.
*                   FALSE otherwise.
*
*****************************************************************************/
BOOLEAN SMSO_bIsOwner (
    SMS_OBJECT hSMS
        )
{
    // Return ownership result(do not enforce)
    return bOwner(hSMS, FALSE);
}

/*****************************************************************************
*
*   bValid
*
*   This API allows the caller to check if it is a valid object.
*
*   Inputs:
*       hSMS        A handle to the SMS object to check validity of
*       bEnforce    TRUE if the check if performed and should always be TRUE
*                   or FALSE if we are just checking (no error).
*
*   Outputs:
*       BOOLEAN     TRUE if the object is valid.
*                   FALSE otherwise.
*
*****************************************************************************/
static BOOLEAN bValid (
    SMS_OBJECT hSMS,
    BOOLEAN bEnforce
        )
{
    BOOLEAN bValid = FALSE;

    // Check for NULL
    if(hSMS != SMS_INVALID_OBJECT)
    {
        // De-reference object
        SMS_OBJECT_STRUCT *psObj =
            (SMS_OBJECT_STRUCT *)hSMS - 1;

        // Check for object validity, and set
        bValid = SMS_OBJECT_MAGIC_NUMBER_OK(psObj);
    }

    // Handle FATAL ERRORs
    if(bValid == FALSE)
    {
        if (bEnforce == TRUE)
        {
            // FATAL ERROR!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_OBJECT_NAME": Object is not valid.");
            vHandleFatalError();
        }
        else if (hSMS != SMS_INVALID_OBJECT)
        {
            // SIMPLE ERROR
            SMSAPI_DEBUG_vPrint(SMS_OBJECT_NAME, 2,
                "Object is not valid (%p)", hSMS);
        }
    }

    return bValid;
}

/*****************************************************************************
*
*   bOwner
*
*   This API allows the caller to check if it is the owner of an object
*   Callers may use this to determine if the context from which this API
*   is called matches the owner of the object.
*
*   Inputs:
*       hSMS        A handle to the SMS object to check ownership of
*       bEnforce    TRUE if the check if performed and should always be TRUE
*                   or FALSE if we are just checking (no error).
*
*   Outputs:
*       BOOLEAN     TRUE if the object and caller context match.
*                   FALSE otherwise.
*
*****************************************************************************/
static BOOLEAN bOwner (
    SMS_OBJECT hSMS,
    BOOLEAN bEnforce
        )
{
    // De-reference object
    SMS_OBJECT_STRUCT *psObj =
        (SMS_OBJECT_STRUCT *)hSMS - 1;
    BOOLEAN bOwner = FALSE, bValid;

    // Verify input and check object validity
    bValid = SMSO_bIsValid(hSMS);
    if(bValid == TRUE)
    {
        BOOLEAN bOk;
        OSAL_TASK_ID tThisTaskId;

        bOk = OSAL.bTaskGetId(&tThisTaskId);
        if (bOk == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            /******************************************************************/
            eReturnCode = OSAL.eEnterTaskSafeSection();
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Check if the caller is the owner of this object. Ownership
                // is determined by the following criteria...
                if(
                    // There is no mutex (nobody can own it, so nobody cares)
                    (psObj->hMutex == OSAL_INVALID_OBJECT_HDL) ||
                    // The mutex owner is this task
                    ( (*psObj->ptMutexOwner == tThisTaskId) &&
                      (*psObj->pbLocked == TRUE) )
                  )
                {
                    // Yes, they are the owner
                    bOwner = TRUE;
                }

                // Exit
                OSAL.eExitTaskSafeSection();
            }
            /******************************************************************/
            else
            {
                // Error! Something is wrong.
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_OBJECT_NAME
                    ": Cannot enter task safe section.");
            }
        }
        else
        {
            // This obj has the lock attribute but caller is not a
            // valid OSAL task.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_OBJECT_NAME": Object ownership cannot be determined.");
        }
    }
    else if(hSMS != SMS_INVALID_OBJECT)
    {
        // Not only is it invalid but it is also not the NULL
        // or specifically 'invalid' value so some garbage was given to us.
        // This condition is always bad.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": Invalid SMS Object provided.");
    }
    else
    {
        // This is the case where a NULL object was provided,
        // and thus the result cannot be that the object being
        // tested is owned. However, this may just be a procedureal
        // check before use, so we just need to indicate we are
        // not the owner, but this is never fatal.
        bEnforce = FALSE;
#if 0
        // NULL SMS object
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": SMS_INVALID_OBJECT provided.");
#endif
    }

    // Handle FATAL ERRORs
    if((bOwner == FALSE) && (bEnforce == TRUE))
    {
        // FATAL ERROR!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_OBJECT_NAME": Ownership check failed.");
        vHandleFatalError();
    }

    // Return ownership result
    return bOwner;
}

/*****************************************************************************
*
*   vHandleFatalError
*
*   This function provides a catch-all function to trap lock and ownership
*   violations.
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void vHandleFatalError ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // FATAL ERROR!
    // Set breakpoint here as this should NEVER happen!

    /********************************************************************/
    eReturnCode = OSAL.eEnterTaskSafeSection();
    if(eReturnCode == OSAL_SUCCESS)
    {
        // increment global fatal error counter
        gtSMSObjectFatalErrorCounter++;

        // Exit
        OSAL.eExitTaskSafeSection();
    }
    /********************************************************************/

    // TODO: BREAKPOINT
    return;
}
