/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

#ifndef SMS_EVENT_H_
#define SMS_EVENT_H_

#include "standard.h"
#include "osal.h"
#include "sms.h"
#include "sms_event_types.h"
#include "dataservice_base.h"
#include "dataservice_mgr_obj.h"
#include "rfd_interface_obj.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "scache.h"
#include "ccache.h"
#include "radio.h"
#include "presets_obj.h"

#if SMS_DEBUG == 1
// Only include this header if SMS_DEBUG is
// defined and equal to 1
#include "sms_api_debug.h"
#endif /* SMS_DEBUG == 1 */

// If logging hasn't been configured already
// then only enable when we're in debug
#ifndef SMS_LOGGING
#if SMS_DEBUG == 1
#define SMS_LOGGING 1
#else
#define SMS_LOGGING 0
#endif
#endif

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

// Event handler options
#define SMS_EVENT_HANDLER_OPTION_NONE   (0x00000000)
#define SMS_EVENT_HANDLER_OPTION_DUAL_Q (0x00000001)

#define SMS_EVENT_HANDLER_OPTION_ALL ( \
    SMS_EVENT_HANDLER_OPTION_DUAL_Q \
        )

// Event allocation option flags
#define SMS_EVENT_OPTION_NONE           (0x00000000)
#define SMS_EVENT_OPTION_URGENT         (0x00000001) // Always post to front
#define SMS_EVENT_OPTION_STATIC         (0x00000002) // never freed
#define SMS_EVENT_OPTION_SYNCHRONOUS    (0x00000004) // Waits for response
#define SMS_EVENT_OPTION_NONBLOCK       (0x00000008) // Event allocation never blocks
#define SMS_EVENT_OPTION_DEFERRED       (0x00000010) // Always dispatched last

// Array size used for data stream config events
// Kept small in order to ensure it stays witin
// the existing size limits of the SMS_EVENT_UNION,
// and because the common case (that we've encountered
// so far) is less than 6 DMIs configured at the same time
#define SMS_EVENT_DATASERVICE_DMI_ARRAY_SIZE (6)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

/** EVENT DATA **/

/** GENERAL Events **/

typedef struct sms_event_shutdown_struct
{
    void *pvData;

} SMS_EVENT_SHUTDOWN_STRUCT;

typedef struct sms_event_sleep_struct
{
    void *pvData;

} SMS_EVENT_SLEEP_STRUCT;

typedef struct sms_event_wakeup_struct
{
    void *pvData;

} SMS_EVENT_WAKEUP_STRUCT;

typedef union sms_event_general_union
{
    SMS_EVENT_SHUTDOWN_STRUCT sShutdown;
    SMS_EVENT_SLEEP_STRUCT sSleep;
    SMS_EVENT_WAKEUP_STRUCT sWakeup;

} SMS_EVENT_GENERAL_UNION;

typedef struct sms_event_update_struct
{
    SMSAPI_EVENT_MASK tEventMask;

} SMS_EVENT_UPDATE_STRUCT;

typedef struct sms_event_object_release_struct
{
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator;

} SMS_EVENT_OBJECT_RELEASE_STRUCT;

/** Data Service Events **/

// Standard argument for data services
typedef struct sms_event_dataservice_std_arg_struct
{
    void *pvArg;

} SMS_EVENT_DATASERVICE_STD_ARG_STRUCT;

// Argument for position updates via the device object
typedef struct sms_event_dataservice_pos_arg_struct
{
    // Lat / Lon fixed values
    OSAL_FIXED_OBJECT_DATA atLatData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT_DATA atLonData[OSAL_FIXED_OBJECT_SIZE];

    // Timestamp for this event
    DATASERVICE_EVENT_TIMESTAMP_STRUCT sTimestamp;

} SMS_EVENT_DATASERVICE_POS_ARG_STRUCT;

typedef struct sms_event_dataservice_stream_config_arg_struct
{
    // Array is declared here to save users of this event
    // from allocating memory for event data in the case
    // where the event data is small (which is more common)
    // Note: This is used as an optimization when there are
    // SMS_EVENT_DATASERVICE_DMI_ARRAY_SIZE or less DMIs
    // provided in this event.  If the poster of this event
    // needs to specify more than SMS_EVENT_DATASERVICE_DMI_ARRAY_SIZE
    // DMIs then we'll allocate memory at that time to support it.
    DATASERVICE_DMI_CONFIG_STRUCT asDMIs[SMS_EVENT_DATASERVICE_DMI_ARRAY_SIZE];

    // Pointer used as the location in which the DMIs are
    // read from.  Allows for simple processing of this event
    DATASERVICE_DMI_CONFIG_STRUCT *pasDMIs;

    // Number of DMIs specified in pasDMIs
    size_t tNumDMIs;

    // Flag indicating DMI list need to be de-allocated
    // after processing
    BOOLEAN bFreeDMIs;

} SMS_EVENT_DATASERVICE_STREAM_CONFIG_ARG_STRUCT;

typedef struct sms_event_data_module_assoc_arg_struct
{
    STRING_OBJECT hDriverName;
    MODULE_OBJECT hModule;
    STI_HDL hSTI;

} SMS_EVENT_DATA_MODULE_ASSOC_ARG_STRUCT;

typedef struct sms_event_dataservice_product_arg_struct
{
    DATA_PRODUCT_TYPE_ENUM eProductType;
    DATA_PRODUCT_MASK tMask;

} SMS_EVENT_DATASERVICE_PRODUCT_ARG_STRUCT;

typedef struct sms_event_dataservice_timed_event_struct
{
    BOOLEAN bRepeat;
    UN32 un32IntervalInSeconds;
} SMS_EVENT_DATASERVICE_TIMED_EVENT_STRUCT;

typedef struct sxm_event_dataservice_sxi_content_buffered_struct
{
    CHANNEL_ID tChannleId;
    SERVICE_ID tServiceId;
    UN8 un8PIDCount;
    PROGRAM_ID *atPIDList;
} SXM_EVENT_DATASERVICE_SXI_CONTENT_BUFFERED_STRUCT;

typedef union sms_event_dataservice_sxi_message_event_union
{
    SXM_EVENT_DATASERVICE_SXI_CONTENT_BUFFERED_STRUCT sContentBuffered;

} SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION;

typedef union sms_event_dataservice_union
{
    SMS_EVENT_DATASERVICE_STD_ARG_STRUCT sStandard;
    SMS_EVENT_DATASERVICE_POS_ARG_STRUCT sPosition;
    SMS_EVENT_DATA_MODULE_ASSOC_ARG_STRUCT sModule;
    SMS_EVENT_DATASERVICE_STREAM_CONFIG_ARG_STRUCT sStreamConfig;
    SMS_EVENT_DATASERVICE_PRODUCT_ARG_STRUCT sProduct;
    SMS_EVENT_DATASERVICE_TIMED_EVENT_STRUCT sTimed;
    SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION uSxiMessage;

} SMS_EVENT_DATASERVICE_UNION;

typedef struct sms_event_dataservice_impl_arg_struct
{
    // Create enough space for impl events so that
    // they may make use of the entire block of memory
    // available to a data event for their specific purposes.
    // This creates a space of pointers for anything the impl
    // want to use them for
    void *apvData[sizeof(SMS_EVENT_DATASERVICE_UNION) / sizeof(void *)];

} SMS_EVENT_DATASERVICE_IMPL_ARG_STRUCT;

typedef union sms_event_dataservice_arg_union
{
    SMS_EVENT_DATASERVICE_STD_ARG_STRUCT sStandard;
    SMS_EVENT_DATASERVICE_POS_ARG_STRUCT sPosition;
    SMS_EVENT_DATA_MODULE_ASSOC_ARG_STRUCT sModule;
    SMS_EVENT_DATASERVICE_STREAM_CONFIG_ARG_STRUCT sStreamConfig;
    SMS_EVENT_DATASERVICE_PRODUCT_ARG_STRUCT sProduct;
    SMS_EVENT_DATASERVICE_TIMED_EVENT_STRUCT sTimed;
    SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION uSxiMessage;
    SMS_EVENT_DATASERVICE_IMPL_ARG_STRUCT sImpl;

} SMS_EVENT_DATASERVICE_ARG_UNION;

typedef struct sms_event_dataservice_struct
{
    DATASERVICE_FW_EVENT_ENUM eDataServiceEvent;
    DATASERVICE_MGR_OBJECT hManager;
    SMS_EVENT_DATASERVICE_ARG_UNION uArg;

} SMS_EVENT_DATASERVICE_STRUCT;

typedef struct sms_event_rfd_interface_struct
{
    RFD_INTERFACE_EVENT_ENUM eRFDEvent;
    OSAL_BUFFER_HDL hPayload;

} SMS_EVENT_RFD_INTERFACE_STRUCT;

/** SRM Events **/

typedef struct sms_event_srm_error_struct
{
    SRM_ERROR_CODE_ENUM eErrorCode;

} SMS_EVENT_SRM_ERROR_STRUCT;

typedef struct sms_event_srm_event_struct
{
    SRM_EVENT_MASK tRequestMask;
    SRM_OBJECT_EVENT_CALLBACK vCallback;
    void *pvCallbackArg;

} SMS_EVENT_SRM_EVENT_STRUCT;

typedef struct sms_event_remove_module_struct
{
    MODULE_OBJECT hModule;

} SMS_EVENT_REMOVE_MODULE_STRUCT;

typedef struct sms_event_module_reset_struct
{
    MODULE_OBJECT hModule;

} SMS_EVENT_MODULE_RESET_STRUCT;

typedef struct sms_event_srm_radio_ready_struct
{
    FILE *psDevice;

} SMS_EVENT_SRM_RADIO_READY_STRUCT;

typedef struct sms_event_srm_event_mask_struct
{
    SRM_EVENT_MASK tEventMask;
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification;

} SMS_EVENT_SRM_EVENT_MASK_STRUCT;

typedef union sms_event_srm_union
{
    SMS_EVENT_MODULE_RESET_STRUCT sReset;
    SMS_EVENT_UPDATE_STRUCT sUpdate;
    SMS_EVENT_SRM_ERROR_STRUCT sError;
    SMS_EVENT_REMOVE_MODULE_STRUCT sRemoveModule;
    SMS_EVENT_OBJECT_RELEASE_STRUCT sRelease;
    SMS_EVENT_SRM_RADIO_READY_STRUCT sRadioReady;
    SMS_EVENT_SRM_EVENT_MASK_STRUCT sSrmEventMask;

} SMS_EVENT_SRM_UNION;

/** MODULE Events **/

typedef struct sms_event_module_error_struct
{
    MODULE_ERROR_CODE_ENUM eErrorCode;

} SMS_EVENT_MODULE_ERROR_STRUCT;

typedef struct sms_event_module_event_struct
{
    MODULE_EVENT_MASK tRequestMask;
    MODULE_OBJECT_EVENT_CALLBACK vCallback;
    void *pvCallbackArg;

} SMS_EVENT_MODULE_EVENT_STRUCT;

typedef struct sms_event_remove_decoder_struct
{
    DECODER_OBJECT hDecoder;

} SMS_EVENT_REMOVE_DECODER_STRUCT;

typedef struct sms_event_save_time_params_struct
{
    N16 n16GmtOffset; // minutes

} SMS_EVENT_SAVE_TIME_PARAMS_STRUCT;

typedef struct sms_event_module_associate_struct
{
    SRM_OBJECT hSRM;
    const char *pacName;
    FILE *psDevice;

} SMS_EVENT_MODULE_ASSOCIATE_STRUCT;

typedef struct sms_event_decoder_reset_struct
{
    DECODER_OBJECT hDecoder;

} SMS_EVENT_DECODER_RESET_STRUCT;

typedef struct sms_event_module_radio_ready_struct
{
    STI_HDL hSTI;

} SMS_EVENT_MODULE_RADIO_READY_STRUCT;

typedef struct sms_event_module_event_mask_struct
{
    MODULE_EVENT_MASK tEventMask;
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification;

} SMS_EVENT_MODULE_EVENT_MASK_STRUCT;

typedef struct sms_event_module_engdata_struct
{
    DECODER_OBJECT hDecoder;
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification;

} SMS_EVENT_MODULE_ENGDATA_STRUCT;

typedef struct sms_event_module_set_time_struct
{
    // New time
    TIME_T tTod;

    // Time provided at
    UN32 un32AtSeconds;
    UN16 un16AtMsecs;

} SMS_EVENT_MODULE_SET_TIME_STRUCT;

typedef union sms_event_module_union
{
    SMS_EVENT_DECODER_RESET_STRUCT sReset;
    SMS_EVENT_UPDATE_STRUCT sUpdate;
    SMS_EVENT_SAVE_TIME_PARAMS_STRUCT sSaveTimeParams;
    SMS_EVENT_MODULE_ERROR_STRUCT sError;
    SMS_EVENT_OBJECT_RELEASE_STRUCT sRelease;
    SMS_EVENT_REMOVE_DECODER_STRUCT sRemoveDecoder;
    SMS_EVENT_MODULE_ASSOCIATE_STRUCT sAssociate;
    SMS_EVENT_MODULE_RADIO_READY_STRUCT sRadioReady;
    SMS_EVENT_MODULE_EVENT_MASK_STRUCT sModuleEventMask;
    SMS_EVENT_MODULE_ENGDATA_STRUCT sModModifyEngData;
    SMS_EVENT_MODULE_SET_TIME_STRUCT sSetTime;
} SMS_EVENT_MODULE_UNION;

/** DECODER Events **/

typedef struct sms_event_decoder_error_struct
{
    DECODER_ERROR_CODE_ENUM eErrorCode;

} SMS_EVENT_DECODER_ERROR_STRUCT;

typedef struct sms_event_add_channel_struct
{
    CHANNEL_OBJECT hChannel;
    BOOLEAN bInitialize;

} SMS_EVENT_ADD_CHANNEL_STRUCT;

typedef SMS_EVENT_ADD_CHANNEL_STRUCT SMS_EVENT_REMOVE_CHANNEL_STRUCT;

typedef struct sms_event_seek_time_struct
{
    N32 n32Offset;
    BOOLEAN bPauseAfterSeek;
} SMS_EVENT_SEEK_TIME_STRUCT;

typedef struct sms_event_seek_song_struct
{
    N16 n16Offset;
    BOOLEAN bPauseAfterSeek;
} SMS_EVENT_SEEK_SONG_STRUCT;

typedef struct sms_event_seek_proximal_struct
{
    BOOLEAN bPauseAfterSeek;
} SMS_EVENT_SEEK_PROXIMAL_STRUCT;

typedef struct sms_event_scache_print_struct
{
    BOOLEAN bVerbose;

} SMS_EVENT_SCACHE_PRINT_STRUCT;

typedef union sms_event_scache_union
{
    SMS_EVENT_SCACHE_PRINT_STRUCT sPrint;

} SMS_EVENT_SCACHE_UNION;

typedef struct sms_event_print_scache_struct
{
    PLAYBACK_OBJECT hPlayback;
    SMS_EVENT_SCACHE_UNION uSCache;

} SMS_EVENT_PRINT_SCACHE_STRUCT;

typedef struct sms_event_songlist_struct
{
    SONGLIST_OBJECT hSonglist;

} SMS_EVENT_SONGLIST_STRUCT;

typedef struct sms_event_decoder_getsonginfo_struct
{
    N16 n16Offset;

} SMS_EVENT_DECODER_GETSONGINFO_STRUCT;

typedef struct sms_event_decoder_update_struct
{
    DECODER_EVENT_MASK tEventMask;

} SMS_EVENT_DECODER_UPDATE_STRUCT;

typedef struct sms_event_decoder_associate_struct
{
    MODULE_OBJECT hModule;
    MODULE_SUBSTATUS_ENUM eSubStatus;
    MODULE_SUBSTATUS_REASON_CODE tSubReasonCode;
    UN32 un32SubSuspendDate;
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities;
    STI_HDL hSTI;

} SMS_EVENT_DECODER_ASSOCIATE_STRUCT;

typedef struct sms_event_all_chan_cat_notification
{
    BOOLEAN bNotifyChannels;
} SMS_EVENT_ALL_CHAN_CAT_NOTIFICATION;

typedef struct sms_event_decoder_event_mask_struct
{
    DECODER_EVENT_MASK tDecoderEventMask;
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification;

} SMS_EVENT_DECODER_EVENT_MASK_STRUCT;

typedef struct sms_event_chanlist_event_mask_struct
{
    CHANNEL_EVENT_MASK tChannelEventMask;
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification;
    CHANNELLIST_OBJECT hChannelList;

} SMS_EVENT_CHANLIST_EVENT_MASK_STRUCT;

// tone gen types
typedef enum decoder_tone_gen_type_enum
{
    DECODER_TONE_GEN_START,
    DECODER_TONE_GEN_STOP,
    DECODER_TONE_GEN_ALERT

} DECODER_TONE_GEN_ACTION_ENUM;

// Tone Generation events
typedef struct sms_event_tone_generation_struct
{
    DECODER_TONE_GEN_ACTION_ENUM eAction;
    UN32 un32ToneFreqHz;
    DECODER_TONE_GENERATION_BALANCE_ENUM eBalance;
    N8 n8Volume;

} SMS_EVENT_TONE_GENERATION_STRUCT;

// Smart favorites action enum
typedef enum smart_favorites_action_enum
{
    SMART_FAVORITES_ACTION_START,
    SMART_FAVORITES_ACTION_STOP,
    SMART_FAVORITES_ACTION_ENABLE,
    SMART_FAVORITES_ACTION_DISABLE,
    SMART_FAVORITES_ACTION_UPDATE

} SMART_FAVORITES_ACTION_ENUM;

// Smart Favorites events
typedef struct sms_event_smart_favorites_struct
{
    SMART_FAVORITES_ACTION_ENUM eAction;
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint;
    size_t tCategorySize;

} SMS_EVENT_SMART_FAVORITES_STRUCT;

// Tune Scan Event Types
typedef enum tune_scan_event_type_enum
{
    TUNE_SCAN_EVENT_TYPE_CFG,
    TUNE_SCAN_EVENT_TYPE_START,
    TUNE_SCAN_EVENT_TYPE_STOP,
    TUNE_SCAN_EVENT_TYPE_ABORT,
    TUNE_SCAN_EVENT_TYPE_FORWARD,
    TUNE_SCAN_EVENT_TYPE_BACKWARD,
    TUNE_SCAN_EVENT_TYPE_PRINT

} TUNE_SCAN_EVENT_TYPE_ENUM;

// Tune Scan events
typedef struct sms_event_tune_scan_struct
{
    UN8 un8PlaySeconds;
    BOOLEAN bScanLockedMature;
    BOOLEAN bScanSkipped;
    DECODER_TUNE_SCAN_STYLE_ENUM eScanStyle;
    TUNE_SCAN_EVENT_TYPE_ENUM eEventType;

} SMS_EVENT_TUNE_SCAN_STRUCT;

// audio controls
typedef enum decoder_audio_control_enum
{
    DECODER_AUDIO_CONTROL_VOLUME,
    DECODER_AUDIO_CONTROL_TONE, // Bass /Treble
    DECODER_AUDIO_CONTROL_EXCITER

} DECODER_AUDIO_CONTROL_ENUM;

// Audio Control events
typedef struct sms_event_audio_control_struct
{
    DECODER_AUDIO_CONTROL_ENUM eAction;
    N16 n16Level;
    // additional info for futute audio controls
    // can be added here if needed

} SMS_EVENT_AUDIO_CONTROL_STRUCT;

typedef enum sms_event_art_type_enum
{
    SMS_EVENT_ART_TYPE_CHANNEL,
    SMS_EVENT_ART_TYPE_CATEGORY,
    SMS_EVENT_ART_TYPE_ALBUM

} SMS_EVENT_ART_TYPE_ENUM;

typedef enum sms_event_art_status_enum
{
    SMS_EVENT_ART_STATUS_START,
    SMS_EVENT_ART_STATUS_STOP,
    SMS_EVENT_ART_STATUS_ALBUM_START,
    SMS_EVENT_ART_STATUS_ALBUM_STOP,
    SMS_EVENT_ART_STATUS_CHAN_START,
    SMS_EVENT_ART_STATUS_CHAN_STOP,
    SMS_EVENT_ART_STATUS_UPDATE

} SMS_EVENT_ART_STATUS_ENUM;

typedef struct sms_event_art_struct
{
    SMS_EVENT_ART_STATUS_ENUM eStatus;
    CHANNEL_ART_SERVICE_OBJECT hArtService;
    SMS_EVENT_ART_TYPE_ENUM eType;
    UN32 un32ActiveProductMask;

    OSAL_OBJECT_HDL hArtList;

} SMS_EVENT_ART_STRUCT;

typedef struct sms_event_channel_struct
{
    BOOLEAN bLocked;
    BOOLEAN bSkipped;

} SMS_EVENT_CHANNEL_STRUCT;

typedef enum sms_event_epg_status_enum
{
    SMS_EVENT_EPG_STATUS_START,
    SMS_EVENT_EPG_STATUS_STOP,
    SMS_EVENT_EPG_STATUS_UPDATE

} SMS_EVENT_EPG_STATUS_ENUM;

typedef struct sms_event_epg_struct
{
    SMS_EVENT_EPG_STATUS_ENUM eStatus;
    EPG_SERVICE_OBJECT hEpgService;

} SMS_EVENT_EPG_STRUCT;

typedef struct sms_event_browse_struct
{
    BROWSE_TYPE_ENUM eBrowseType;

    SMSAPI_DIRECTION_ENUM eDirection;

    union sms_browse_id_union
    {
        CHANNEL_ID tChannel;
        CATEGORY_ID tCategory;

    } uId;

} SMS_EVENT_BROWSE_STRUCT;

typedef struct sms_event_tune_struct
{
    CHANNEL_ID tChannelId;
    SERVICE_ID tServiceId;
    BOOLEAN bMatureOverride;
    BOOLEAN bLockedOverride;
    BOOLEAN bSkippedOverride;
    BOOLEAN bPlayUnrestricted;

} SMS_EVENT_TUNE_STRUCT;

typedef struct sms_event_tunemix_struct
{
    TUNEMIX_OBJECT hTuneMix;
    UN8 un8ChIndex;

} SMS_EVENT_TUNEMIX_STRUCT;

/* SPORTS FLASH ACTIONS */
typedef enum sports_flash_action_enum
{
    SPORTS_FLASH_ACTION_LOAD_CONFIG_EVENT,
    SPORTS_FLASH_ACTION_ADD_TEAM,
    SPORTS_FLASH_ACTION_REMOVE_TEAM,
    SPORTS_FLASH_ACTION_REMOVE_ALL_TEAMS,
    SPORTS_FLASH_ACTION_SET_GAMES_MONITOR,
    SPORTS_FLASH_ACTION_PLAY_FLASH_EVENT,
    SPORTS_FLASH_ACTION_REMAIN_FLASH_EVENT_CHANNEL,
    SPORTS_FLASH_ACTION_ABORT_FLASH_EVENT,
    SPORTS_FLASH_ACTION_STOP_EVENT,
    SPORTS_FLASH_ACTION_ENABLE_NOTIFICATION,
    SPORTS_FLASH_EVENT_GAME_EVENT,
    SPORTS_FLASH_EVENT_FLASH_EVENT

} SPORTS_FLASH_ACTION_ENUM;

typedef struct sports_flash_action_enable_notification_struct
{
    BOOLEAN bEnable;

} SPORTS_FLASH_ACTION_ENABLE_NOTIFICATION_STRUCT;

typedef struct sports_flash_event_game_event_struct
{
    CHANNEL_ID tChannelID;
    BOOLEAN bEnded;
    UN8 un8AwayLeagueID;
    UN8 un8HomeLeagueID;
    UN16 un16AwayTeamID;
    UN16 un16HomeTeamID;
    UN16 un16Bias;

} SPORTS_FLASH_EVENT_GAME_EVENT_STRUCT;

typedef struct sports_flash_action_set_games_monitor_struct
{
    CHANNEL_ID *ptChannelIDList;
    UN16 un16Count;
    SPORTS_FLASH_FLASH_EVENT_CALLBACK vEventCallback;
    void *pvArg;
} SPORTS_FLASH_ACTION_SET_GAMES_MONITOR_STRUCT;

typedef struct sports_flash_action_flash_event_struct
{
    SPORTS_FLASH_EVENT_ID tFlashEventID;

} SPORTS_FLASH_ACTION_FLASH_EVENT_STRUCT;

typedef struct sports_flash_event_flash_event_struct
{
    SPORTS_FLASH_EVENT_ID tFlashEventID;
    CHANNEL_ID tChannelID;
    PROGRAM_ID tProgramID;
    SPORTS_FLASH_EVENT_STATUS_ENUM eEventStatus;

} SPORTS_FLASH_EVENT_FLASH_EVENT_STRUCT;

typedef struct sports_flash_event_team_struct
{
    UN16 un16TeamID;
    UN8 un8LeagueID;
} SPORTS_FLASH_ACTION_TEAM_STRUCT;

typedef union sms_event_sports_flash_action_union
{
    SPORTS_FLASH_ACTION_SET_GAMES_MONITOR_STRUCT sSetGamesMonitor;
    SPORTS_FLASH_ACTION_FLASH_EVENT_STRUCT sFlashAction;
    SPORTS_FLASH_ACTION_TEAM_STRUCT sTeam;
    SPORTS_FLASH_ACTION_ENABLE_NOTIFICATION_STRUCT sEnableNotification;
} SMS_EVENT_SPORTS_FLASH_ACTION_UNION;

typedef struct sms_event_sports_flash_struct
{
    SPORTS_FLASH_ACTION_ENUM eAction;
    SMS_EVENT_SPORTS_FLASH_ACTION_UNION uEvent;

} SMS_EVENT_SPORTS_FLASH_STRUCT;

/* TW Now ACTIONS */
typedef enum tw_now_action_enum
{
    TW_NOW_ACTION_INITIALIZE_MARKET_ID,
    TW_NOW_ACTION_SET_MARKET_ID,
    TW_NOW_ACTION_PLAY_BULLETIN,
    TW_NOW_ACTION_ABORT_BULLETIN,
    TW_NOW_ACTION_STOP_EVENT

} TW_NOW_ACTION_ENUM;

typedef struct sms_event_tw_now_struct
{
    TW_NOW_ACTION_ENUM eAction;
    union
    {
        TRAFFIC_MARKET tMarketID;
        TW_NOW_BULLETIN_ID tBulletinID;
    } uData;

} SMS_EVENT_TW_NOW_STRUCT;

typedef union sms_event_decoder_union
{
    SMS_EVENT_ADD_CHANNEL_STRUCT sAddChannel;
    SMS_EVENT_REMOVE_CHANNEL_STRUCT sRemoveChannel;
    SMS_EVENT_SEEK_TIME_STRUCT sSeekTime;
    SMS_EVENT_SEEK_SONG_STRUCT sSeekSong;
    SMS_EVENT_SEEK_PROXIMAL_STRUCT sSeekProximal;
    SMS_EVENT_PRINT_SCACHE_STRUCT sPrintSCache;
    SMS_EVENT_SONGLIST_STRUCT sSonglist;
    SMS_EVENT_CCACHE_STRUCT sCache;
    SMS_EVENT_CHANNEL_STRUCT sChannel;
    SMS_EVENT_DECODER_GETSONGINFO_STRUCT sGetSongInfo;
    SMS_EVENT_UPDATE_STRUCT sUpdate;
    SMS_EVENT_DECODER_ERROR_STRUCT sError;
    SMS_EVENT_OBJECT_RELEASE_STRUCT sRelease;
    SMS_EVENT_DECODER_EVENT_MASK_STRUCT sDecoderEventStruct;
    SMS_EVENT_CHANLIST_EVENT_MASK_STRUCT sChanlistEventStruct;
    SMS_EVENT_DECODER_ASSOCIATE_STRUCT sAssociate;
    SMS_EVENT_ALL_CHAN_CAT_NOTIFICATION sAllChanCatNotify;
    SMS_EVENT_TONE_GENERATION_STRUCT sToneGeneration;
    SMS_EVENT_AUDIO_CONTROL_STRUCT sAudioControl;
    SMS_EVENT_FEATURED_FAVORITES_STRUCT sFeaturedFavorites;
    SMS_EVENT_SMART_FAVORITES_STRUCT sSmartFavorites;
    SMS_EVENT_ART_STRUCT sArt;
    SMS_EVENT_TUNE_SCAN_STRUCT sTuneScan;
    SMS_EVENT_EPG_STRUCT sEpg;
    SMS_EVENT_BROWSE_STRUCT sBrowse;
    SMS_EVENT_TUNE_STRUCT sTune;
    SMS_EVENT_TUNEMIX_STRUCT sTuneMix;
    SMS_EVENT_SPORTS_FLASH_STRUCT sSportsFlash;
    SMS_EVENT_TW_NOW_STRUCT sTWNow;

} SMS_EVENT_DECODER_UNION;

typedef union sms_event_union
{
    SMS_EVENT_GENERAL_UNION uGeneral;
    SMS_EVENT_SRM_UNION uSRM;
    SMS_EVENT_MODULE_UNION uModule;
    SMS_EVENT_DECODER_UNION uDecoder;
    SMS_EVENT_DATASERVICE_STRUCT sData;
    SMS_EVENT_RFD_INTERFACE_STRUCT sRFD;

} SMS_EVENT_UNION;

typedef struct radio_event_radio_struct
{
    // Create enough space for radio events so that
    // they may make use of the entire block of memory
    // available to an event for their specific purposes.
    // This creates a space of pointers which may be used
    // for anything the radio wants to.
    void *apvData[sizeof(SMS_EVENT_UNION) / sizeof(void *)];

} SMS_EVENT_RADIO_STRUCT;

/** EVENT DATA UNION **/
typedef union sms_event_data_union
{
    SMS_EVENT_GENERAL_UNION uGeneral;
    SMS_EVENT_SRM_UNION uSRM;
    SMS_EVENT_MODULE_UNION uModule;
    SMS_EVENT_DECODER_UNION uDecoder;
    SMS_EVENT_DATASERVICE_STRUCT sData;
    SMS_EVENT_RFD_INTERFACE_STRUCT sRFD;
    SMS_EVENT_RADIO_STRUCT sRadio;

} SMS_EVENT_DATA_UNION;

/*
    SMS Event Handler Results
*/

typedef enum sms_event_result_enum
{
    SMS_EVENT_RESULT_OK = OSAL_TASK_REPORT_NO_ERROR,
    SMS_EVENT_RESULT_SHUTDOWN,
    SMS_EVENT_RESULT_ERROR

} SMS_EVENT_RESULT_ENUM;

#define SMS_OBJECT_EVENT_HANDLER(x)                 \
    void (x)                                        \
    (                                               \
        SMS_OBJECT hSMS,                            \
        SMS_EVENT_HDL hEvent                        \
    )
typedef SMS_OBJECT_EVENT_HANDLER \
    (*SMS_OBJECT_EVENT_HANDLER_PROTOTYPE);

typedef SMS_EVENT_HANDLER \
    (*SMS_EVENT_HANDLER_PROTOTYPE);

#define SMS_EVENT_HANDLER_INIT(x)                   \
    BOOLEAN (x)                                     \
    (                                               \
        const char *pacName,                        \
        SMS_OBJECT hSMS,                            \
        SMS_EVENT_HANDLER *phEventHdlr,             \
        UN32 un32EventQueueSize,                    \
        SMS_OBJECT_EVENT_HANDLER_PROTOTYPE vEventHandler,  \
        UN32 un32Options    \
    )
typedef SMS_EVENT_HANDLER_INIT \
    (*SMS_EVENT_HANDLER_INIT_PROTOTYPE);

#define SMS_EVENT_HANDLER_UNINIT(x)                 \
    void (x)                                        \
    (                                               \
        SMS_EVENT_HANDLER hEventHdlr                \
    )
typedef SMS_EVENT_HANDLER_UNINIT \
    (*SMS_EVENT_HANDLER_UNINIT_PROTOTYPE);


  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

SMS_EVENT_HANDLER_INIT(SMSE_bEventHdlrInit);
SMS_EVENT_HANDLER_UNINIT(SMSE_vEventHdlrUninit);

SMS_EVENT_HDL SMSE_hAllocateEvent (
    SMS_EVENT_HANDLER hEventHdlr,
    SMS_EVENT_TYPE_ENUM eEvent,
    SMS_EVENT_DATA_UNION **ppuData,
    EVENT_OPTIONS_TYPE tOptions
        );

void SMSE_vReleaseStaticEvent (
    SMS_EVENT_HDL hEvent
        );

BOOLEAN SMSE_bPostEvent (
    SMS_EVENT_HDL hEvent
        );

BOOLEAN SMSE_bPostSignal (
    SMS_EVENT_HANDLER hEventHdlr,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions
        );

SMS_EVENT_RESULT_ENUM SMSE_eDispatchEvent (
    SMS_EVENT_HANDLER hEventHdlr,
    N32 n32Timeout
        );

SMS_EVENT_TYPE_ENUM SMSE_eGetEvent (
    SMS_EVENT_HDL hEvent,
    SMS_EVENT_DATA_UNION const **ppuData
        );

#if SMS_LOGGING == 1
void SMSE_vLog (
    SMS_EVENT_HANDLER hEventHdlr,
    const char *pcFormat,
    ...
        );

void SMSE_vLogUsingList (
    SMS_EVENT_HANDLER hEventHdlr,
    const char *pcFormat,
    va_list *ptArgList
        );
#else
#if __STDC_VERSION__ >= 199901L
#define SMSE_vLog(...)
#define SMSE_vLogUsingList(...)
#else
void SMSE_vLogNothing(
    SMS_EVENT_HANDLER hEventHdlr,
    const char *pcFormat,
    ...
        );
void SMSE_vLogUsingListNothing (
    SMS_EVENT_HANDLER hEventHdlr,
    const char *pcFormat,
    va_list *ptArgList
        );
#define SMSE_vLog if(1);else SMSE_vLogNothing
#define SMSE_vLogUsingList if(1);else SMSE_vLogUsingListNothing
#endif
#endif


#endif /* SMS_EVENT_H */
