/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains various APIs and functions which implement the
 *  Sirius Module Services.
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"
#include "cm.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "srm_obj.h"
#include "cdo_obj.h"
#include "cid_obj.h"
#include "report_obj.h"
#include "dataservice_mgr_obj.h"
#include "device_obj.h"
#include "localization_obj.h"
#include "song_tag_service.h"
#include "sms_event.h"
#include "sti_api.h"
#include "sql_interface_obj.h"

#include "radio.h"

#include "sms.h"
#include "_sms.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*******************************************************************************
*
*   pacGetVersion
*
* This function is used to return a string which contains the version
* information of the SMS currently in use. Additionally each individual
* component of the version can be returned by providing the appropriate
* pointer to retrieve the version numbers.
*
*    Where:
*       pun8MajorVersion - A pointer to a UN8 which will return the major
*       version number. If this return value is not required, NULL should be
*       provided.
*       pun8MinorVersion - A pointer to a UN8 which will return the minor
*       version number. If this return value is not required, NULL should be
*       provided.
*       pun8SubVersion - A pointer to a UN8 which will return the sub version
*       number. If this return value is not required, NULL should be provided.
*
*    Return:
*       A pointer to a constant character array which contains an ASCII Null
*       terminated string with the text representing the SMS version
*       information (e.g. "Satellite Module Services (SMS) v02.36.03").
*
*******************************************************************************/
static const char *pacGetVersion (
    UN8 *pun8MajorVersion,
    UN8 *pun8MinorVersion,
    UN8 *pun8SubVersion
        )
{
    const char *pacVersion = PROJECT_NAME" (v" MAJOR_VERSION "."
        MINOR_VERSION "." SUBVERSION ") " DATE;

    if(pun8MajorVersion != NULL)
    {
        *pun8MajorVersion = (UN8)atoi(MAJOR_VERSION);
    }
    if(pun8MinorVersion != NULL)
    {
        *pun8MinorVersion = (UN8)atoi(MINOR_VERSION);
    }
    if(pun8SubVersion != NULL)
    {
        *pun8SubVersion = (UN8)atoi(SUBVERSION);
    }

    return pacVersion;
}

/*******************************************************************************
*
*   eInitialize
*
* This API is used to inform the SMS library that it must prepare for operation
* with the Application. In addition, an optional path may be provided which
* specifies a location within the file system designated (by the Application)
* for use solely by SMS. This path indicates the location at which the
* top-level SMS directory is to be found. If this path does not exist, SMS will
* attempt to create it if possible. If the provided path utilizes volatile
* storage, internal SMS configuration will not persist across power cycles.
* If no path is given, SMS will not make use of device storage (which will
* result in a reduction of available features).
*
* This API must be exercised before any other SMS APIs are invoked.
*
*   Where:
*       pacConfigPath - A pointer to a const char array which holds the
*       location within on-board storage indicating where the SMS library
*       may place configuration files.  This pointer may be NULL if no
*       configuration path is available.
*
*       pacConfigInitializerPath - A pointer to a const char array which holds
*       the location where sms-related configuration initializer files is
*       stored.
*
*       pacConfigInitializerFile - A pointer to a const char array which holds
*       the filename for the config initializer.
*
*       pbConfigFileReset - A pointer to a boolean which will be set to
*       TRUE or FALSE, depending on whether the sms.cfg file was reset
*       upon startup. Can be NULL.
*
*   Return:
*       SMSAPI_RETURN_CODE_SUCCESS on success or an error code as a value
*       of SMSAPI_RETURN_CODE type.
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eInitialize (
    const char *pacConfigPath,
    const char *pacConfigInitializerPath,
    const char *pacConfigInitializerFile,
    BOOLEAN *pbConfigFileReset
        )
{
    BOOLEAN bInitOk, bOwner;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SMS_CONTROL_OBJECT_STRUCT *psTempObj;
    size_t tConfigPathSize = 0,
           tInitializerPathSize = 0,
           tInitializerFileSize = 0,
           tAdditionalSpace = 0;

#if SMS_DEBUG == 1
    // Initializing debug printout utility before doing anything else
    SMSAPI_DEBUG_vInitPrintingParameters();
#endif

    SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
        "SMS.%s(pacConfigPath: %s, pacConfigInitializerPath: %s, "
        "pacConfigInitializerFile: %s, pbConfigFileReset: %p(%d))\n",
        __FUNCTION__, pacConfigPath, pacConfigInitializerPath,
        pacConfigInitializerFile, pbConfigFileReset,
        pbConfigFileReset != NULL ? *pbConfigFileReset : 0);

    // We are going to first allocate an object (which also locks it)
    // before we test for it. Why? Well this helps us avoid a potential
    // race condition where two entities call this same API. Rare and unlikely
    // for sure, but just to be extra paranoid we do it like this. This logic
    // is potentially wasteful (allocating when not necessary, only to free
    // it. But I favor the side that this is unlikely if ever to be a problem
    // and typically this API will only be called when needed.

    // Check configuration path provided for non-null. If it is,
    // then make sure we have space for it.
    if(pacConfigPath != NULL)
    {
        // Determine the path length and use it to compute the additional
        // space we need to store it.
        tConfigPathSize = strlen(pacConfigPath) + 1;
    }

    // Check if the initializer path is non-null. If it is,
    // then make sure we have space for it as well.
    if(pacConfigInitializerPath != NULL)
    {
        // Determine the path length and use it to compute the additional
        // space we need to store it (including an extra byte for the
        // null terminator)
        tInitializerPathSize = strlen(pacConfigInitializerPath) + 1;
    }

    // Check if the initializer file is non-null. If it is,
    // then make sure we have space for it.
    if(pacConfigInitializerFile != NULL)
    {
        // Determine the path length and use it to compute the additional
        // space we need to store it (including an extra byte for the
        // null terminator)
        tInitializerFileSize = strlen(pacConfigInitializerFile) + 1;
    }

    // The additional space we need is the sum of the config path
    // and default path string sizes.
    tAdditionalSpace = tConfigPathSize + tInitializerPathSize +
                        tInitializerFileSize;

    // Create an instance of this object
    psTempObj = (SMS_CONTROL_OBJECT_STRUCT *)
        SMSO_hCreate(
            SMS_NAME,
            sizeof(SMS_CONTROL_OBJECT_STRUCT) + tAdditionalSpace,
            SMS_INVALID_OBJECT, // Parent
            TRUE ); // Lock
    if(psTempObj == NULL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_NAME
            ": Unable to allocate memory or object already exists.");
        return SMSAPI_RETURN_CODE_ALREADY_INITIALIZED;
    }

    /*************************************************************************/
    // We simply want to make sure we cannot enter this section of code
    // multiple times. We protect this section to update the global
    // variable which holds the master object.
    // Note this on happens during first-time start and/or initialization.
    eReturnCode = OSAL.eEnterTaskSafeSection();
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Check if global pointer is NULL. If so, populate it with
        // locked object.
        if(gpsSms == NULL)
        {
            // This is the new object, and it is locked already.
            gpsSms = psTempObj;
        }
        else
        {
            OSAL.eExitTaskSafeSection();

            // Error! Global object already exists somehow.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_NAME
                ": Object already exists.");
            SMSO_vDestroy((SMS_OBJECT)psTempObj);
            return SMSAPI_RETURN_CODE_ALREADY_INITIALIZED;
        }

        // Exit
        OSAL.eExitTaskSafeSection();
    }
    /*************************************************************************/
    else
    {
        // Error! Something is wrong.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_NAME
            ": Cannot enter task safe section.");
        SMSO_vDestroy((SMS_OBJECT)psTempObj);
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Is this object locked by us?
    bOwner = SMSO_bIsOwner((SMS_OBJECT)gpsSms);
    if(bOwner == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_NAME
            ": Library has already been initialized.");
        SMSO_vDestroy((SMS_OBJECT)psTempObj);
        return SMSAPI_RETURN_CODE_ALREADY_INITIALIZED;
    }

    SMSAPI_DEBUG_vPrint(SMS_NAME, 4, "Initializing.\n");

    bInitOk = SQL_INTERFACE.bInitialize();
    if(bInitOk == FALSE)
    {
        // We were not able to initialize the framework
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Unable to initialize SQL.");
        eUninitialize();
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // We need the STI-Framework
    bInitOk = STI_bInit();
    if(bInitOk == FALSE)
    {
        // We were not able to initialize the framework
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Unable to initialize STI-Framework.");
        eUninitialize();
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Initialize structure (just for completeness)
    gpsSms->bInitialized = FALSE;
    gpsSms->hTag = TAG_INVALID_OBJECT;
    gpsSms->hSRMList = OSAL_INVALID_OBJECT_HDL;
    gpsSms->pacConfigPath = (const char *)(gpsSms + 1);

    // The initializers can be NULL; if we weren't given any values for these items
    // then make the pointers NULL
    if ( pacConfigInitializerPath != NULL )
    {
        gpsSms->pacInitializerPath = ((const char *)(gpsSms + 1))
            + tConfigPathSize;
    }
    else
    {
        gpsSms->pacInitializerPath = NULL;
    }

    if ( pacConfigInitializerFile != NULL )
    {
        gpsSms->pacInitializerFile = ((const char *)(gpsSms + 1))
                + tConfigPathSize + tInitializerPathSize;
    }
    else
    {
        gpsSms->pacInitializerFile = NULL;
    }

    gpsSms->sBehaviors = gsBehaviorDefault;

    //////////////////////////////////////////////////////
    /// Build the SRM linked list                      ///
    //////////////////////////////////////////////////////

    // Create SRM list which contains unique elements.
    eReturnCode =
        OSAL.eLinkedListCreate(
            &gpsSms->hSRMList,
            SMS_NAME":SRMs",
            n16CompareHandles,
            OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_UNIQUE
                );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_NAME
            ": Unable to create list.");
        eUninitialize();
        return SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
    }

    // Initialize the Data Service Manager
    gpsSms->sDSM.hDSMCtrl = DATASERVICE_MGR_hInitialize(
        &gpsSms->sDSM.hDSMEventHandler,
        &gpsSms->sDSM.hDevice);
    if(gpsSms->sDSM.hDSMCtrl == SMS_INVALID_OBJECT)
    {
        // We were not able to Data Services initialization
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_NAME
            ": Unable to initialize the Data Service Manager.");
        eUninitialize();
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Set the initializer path if we were given one. Note that
    // SMS objects are already zeroed out, so no need to set
    // the pacInitializerPath to NULL otherwise.
    if ( pacConfigInitializerPath != NULL )
    {
        // Copy the data into SMS's configuration path.
        strncpy((char *)gpsSms->pacInitializerPath,
            pacConfigInitializerPath, tInitializerPathSize);
    }

    // Set the initializer file if we were given one. Note that
    // SMS objects are already zeroed out, so no need to set
    // the pacInitializerFile to NULL otherwise.
    if ( pacConfigInitializerFile != NULL )
    {
        // Copy the data into SMS's configuration path.
        strncpy((char *)gpsSms->pacInitializerFile,
            pacConfigInitializerFile, tInitializerFileSize);
    }

    // Only initialize SMS configuration if
    // we were provided a valid configuration path
    if (pacConfigPath != NULL)
    {
        // Copy the data into SMS's configuration path.
        strncpy((char *)gpsSms->pacConfigPath,
            pacConfigPath, tConfigPathSize);

        // Set up SMS configuration; the config manager
        // checks to see if the boolean pointer for
        // config reset is NULL before write, so there's no
        // need to check it at this level.
        bInitOk = bInitConfig( pbConfigFileReset );
        if (bInitOk == FALSE)
        {
            // Cfg failed to start up
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SMS_NAME
                ": Unable to initialize configuration.");
            eUninitialize();
            return SMSAPI_RETURN_CODE_INVALID_INPUT;
        }
    }
    else // Don't use config path
    {
        gpsSms->pacConfigPath = NULL;
    }

    // Indicate we are initialized. this was move up here because during CDO
    // init an API is used that checks for SMS initialization
    // (basically it needs the path we we just got done handling)
    gpsSms->bInitialized = TRUE;

    // Initialize Content Description Objects...
    bInitOk = CDO_bInitialize();
    if(bInitOk == FALSE)
    {
        // We were not able to initialize the object
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Unable to initialize CDOs.");
        eUninitialize();
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Install a CID type registry into SMS. The CID type registry is global
    // to all of SMS and maintains a registered list of all supported CID types.
    // CID types need to be registered as required. Artist, Title and Composer
    // text CID types are automatically registered via this call.
    bInitOk = CID_bInstall((SMS_OBJECT)gpsSms);
    if(bInitOk == FALSE)
    {
        // We were not able to install the object
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Unable to install CIDs.");
        eUninitialize();
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Perform RADIO specific SMS initialization
    bInitOk = RADIO_bInitializeSMS();
    if(bInitOk == FALSE)
    {
        // We were not able to perform RADIO specific initialization.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Unable to to perform RADIO specific initialization.");
        eUninitialize();
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Load in any markets which have been stored.
    REPORT_vLoadMarkets();

    gpsSms->hSongTag = SONGTAG_hCreate(gpsSms->hTag);

    // Relinquish control of this object
    SMSO_vUnlock((SMS_OBJECT)gpsSms);

    SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
        "SMS is successfully initialized.\n");

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*******************************************************************************
*
*   eUninitialize
*
*   This API is used to inform the SMS library that the Application is no
*   longer making use of it. When this API successfully returns, the SMS
*   library has performed all final operations needed to free all remaining
*   memory associated with it.  This API should only be invoked after all
*   SRM objects have been released.
*
*   Once this API has been exercised, no other SMS APIs may be invoked.
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUninitialize (void)
{
    UN8 un8AttemptsRemaining = SMS_UNINITIALIZE_ATTEMPTS;

    SMSAPI_DEBUG_vPrint(SMS_NAME, 4, "SMS.%s()\n", __FUNCTION__);

    // Request SRM(s) release
    vReleaseSRMs();

    do
    {
        UN32 un32Items = 0;
        BOOLEAN bLocked;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

        SMSAPI_DEBUG_vPrint(SMS_NAME, 4, "Uninitialize (attempt %u of %u)\n",
            (SMS_UNINITIALIZE_ATTEMPTS - un8AttemptsRemaining) + 1,
            SMS_UNINITIALIZE_ATTEMPTS);

        // Verify and lock the object
        bLocked = SMS_bLock();
        if(bLocked == FALSE)
        {
            // Error!  Object hasn't been initialized yet!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_NAME": SMS has not been initialized.");
            return SMSAPI_RETURN_CODE_NOT_INITIALIZED;
        }

        // Check if there are any installed SRMs
        if (gpsSms->hSRMList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListItems(
                gpsSms->hSRMList,
                &un32Items);
        }

        // Only un-init if there are no SRMs currently in operation
        // or if SRM List is not initialized
        if((eReturnCode == OSAL_SUCCESS) && (un32Items == 0))
        {
            SMS_CONTROL_OBJECT_STRUCT *psSmsToDestroy;

            // Uninitialize RADIO specific stuff
            RADIO_vUninitializeSMS();

            // Uninitialize Content Description Objects (CDOs)
            CDO_vUnInitialize();

            // Uninstall the CID object, all CIDs in the CID pool and
            // all registered types.
            CID_vUninstall();

            // Stop and free the contents of the control object
            vUninitConfig();

            // Delete the SRM object linked list
            if (gpsSms->hSRMList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListDelete(
                    gpsSms->hSRMList );

                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Invalidate linked list
                    gpsSms->hSRMList = OSAL_INVALID_OBJECT_HDL;
                }
            }

            // Destroy Song Tag Service
            if (gpsSms->hSongTag != SONG_TAG_INVALID_OBJECT)
            {
                SONGTAG_vDestroy(gpsSms->hSongTag);
                gpsSms->hSongTag = SONG_TAG_INVALID_OBJECT;
            }

            // Uninstall Data Service Manager
            DATASERVICE_MGR_vUnInitialize(gpsSms->sDSM.hDSMCtrl);
            gpsSms->sDSM.hDSMCtrl = SMS_INVALID_OBJECT;
            gpsSms->sDSM.hDSMEventHandler = SMS_INVALID_EVENT_HANDLER;
            gpsSms->sDSM.hDevice = DEVICE_INVALID_OBJECT;

            // Uninitialize Localization object.
            LOCALIZATION_vUnInitialize();

            // Uninitialize the STI-framework
            STI_vUninit();

            // Free the control object
            psSmsToDestroy = gpsSms;
            gpsSms = NULL;
            SMSO_vDestroy( (SMS_OBJECT)psSmsToDestroy );
            psSmsToDestroy = NULL;

            OSAL.eTaskDelay(SMS_UNINITIALIZE_TIMEOUT_MSEC);

            SQL_INTERFACE.vUninitialize();

            SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
                "SMS.%s(): SMS is un-initialized.\n\n", __FUNCTION__);

            return SMSAPI_RETURN_CODE_SUCCESS;
        }

        // Unlock SMS and just be patient.
        SMS_vUnLock();

        SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
            "Waiting %u seconds for Uninitialization to complete...\n",
             SMS_UNINITIALIZE_TIMEOUT_SEC);

        OSAL.eTaskDelay(SMS_UNINITIALIZE_TIMEOUT_MSEC);

    } while(--un8AttemptsRemaining);

    SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
        "Exhausted timeout to wait for all uninitialization "
        "attempts(%u). Try again.\n", SMS_UNINITIALIZE_ATTEMPTS);

    // We still have some active SRM's -- we cannot
    // complete this request to uninitialize
    return SMSAPI_RETURN_CODE_STILL_ACTIVE;
}

/*******************************************************************************
*
*   eBehavior
*
*******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBehavior (
    SMS_BEHAVIOR_CODE eBehaviorCode,
    ...
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_NOT_INITIALIZED;

    // Verify and lock the object
    bLocked = SMS_bLock();
    if(bLocked == TRUE)
    {
        va_list tList; // variable arguments list
        int iArg = 0;

        // grab variable arguments (pop)
        va_start(tList, eBehaviorCode);

        // Get initialized flag
        if (gpsSms->bInitialized)
        {
            switch (eBehaviorCode)
            {
                case SMS_BEHAVIOR_RESTORE_FACTORY_DEFAULTS:
                case SMS_BEHAVIOR_DECODER_SELF_TUNE_MATURE:
                {
                    BOOLEAN *pbValue;

                    pbValue =
                        (eBehaviorCode == SMS_BEHAVIOR_DECODER_SELF_TUNE_MATURE) ?
                            &gpsSms->sBehaviors.bAllowSelfTuneMature :
                            &gpsSms->sBehaviors.bRestoreFactoryDefaults;

                    // take an argument from stack
                    iArg = va_arg(tList, int);

                    switch( iArg )
                    {
                        case TRUE:
                        {
                            *pbValue = TRUE;
                            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                        }
                        break;

                        case FALSE:
                        {
                            *pbValue = FALSE;
                            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                        }
                        break;

                        default:
                        {
                            eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
                        }
                        break;
                    }
                }
                break;

                default:
                {
                    eReturn = SMSAPI_RETURN_CODE_INVALID_OPTIONS;

                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SMS_NAME": Unrecognized Behavior: %d.", eBehaviorCode);
                }
                break;
            }

        }

        // restore stack (push)
        va_end(tList);

        SMS_vUnLock();
    }

    return eReturn;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/******************************************************************************
*
*   SMS_bIsInitialized
*
******************************************************************************/
BOOLEAN SMS_bIsInitialized ( void )
{
    BOOLEAN bLocked, bInitialized = FALSE;

    // Verify and lock the object
    bLocked = SMS_bLock();
    if(bLocked == TRUE)
    {
        // Get initialized flag
        bInitialized = gpsSms->bInitialized;
        SMS_vUnLock();
    }

    // If the object is valid and flag set, it is initialized...
    return bInitialized;
}

/******************************************************************************
*
*   SMS_bGetBehavior
*
******************************************************************************/
BOOLEAN SMS_bGetBehavior(
    int iBehaviorCode,
    SMS_BEHAVIOR_VALUE_UNION *puValue
        )
{
    BOOLEAN bLocked, bOk = TRUE;

    // Verify and lock the object
    bLocked = SMS_bLock();
    if(bLocked == TRUE)
    {
        switch (iBehaviorCode)
        {
            case SMS_BEHAVIOR_DECODER_SELF_TUNE_MATURE:
                puValue->bValue = gpsSms->sBehaviors.bAllowSelfTuneMature;
                break;
            case SMS_BEHAVIOR_RESTORE_FACTORY_DEFAULTS:
                puValue->bValue = gpsSms->sBehaviors.bRestoreFactoryDefaults;
                break;
            default:
                bOk = FALSE;
                break;
        }

        SMS_vUnLock();
    }
    else
    {
        bOk = FALSE;
    }

    return bOk;
}

/******************************************************************************
*
*   SMS_bLock
*
******************************************************************************/
BOOLEAN SMS_bLock ( void )
{
    BOOLEAN bLocked;

    // Verify and lock the object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsSms, OSAL_OBJ_TIMEOUT_INFINITE );

    return bLocked;
}

/******************************************************************************
*
*   SMS_vUnLock
*
******************************************************************************/
void SMS_vUnLock ( void )
{
    // Verify and unlock the object
    SMSO_vUnlock( (SMS_OBJECT)gpsSms );

    return;
}

/******************************************************************************
*
*   SMS_bAddSRM
*
******************************************************************************/
BOOLEAN SMS_bAddSRM (
    SRM_OBJECT hSRM
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
        "%s(hSRM: %p)\n", __FUNCTION__, hSRM);

    // Verify the SMS control object is owned by caller
    bOwner =
        SMSO_bOwner((SMS_OBJECT)gpsSms);
    if (bOwner == TRUE)
    {
        // Check validity of the object being added
        BOOLEAN bValid;

        bValid = SMSO_bValid((SMS_OBJECT)hSRM);
        if(bValid == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

            // Add this SRM to the list. This list allows only unique
            // entries. If one exists this means this SRM already exists.
            eReturnCode = OSAL.eLinkedListAdd(
                gpsSms->hSRMList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)hSRM
                    );
            if(eReturnCode == OSAL_SUCCESS) // Added
            {
                SMSAPI_DEBUG_vPrint(SMS_NAME, 4, "Added SRM entry.\n");

                // SRM has added to SMS.
                bSuccess = TRUE;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_NAME": Could not add SRM entry: %s.",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_NAME": Invalid SRM object.");
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Not the owner of the SMS object.");
    }

    return bSuccess;
}

/******************************************************************************
*
*   SMS_bRemoveSRM
*
******************************************************************************/
BOOLEAN SMS_bRemoveSRM (
    SRM_OBJECT hSRM
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
        "%s(hSRM: %p)\n", __FUNCTION__, hSRM);

    // Verify the SMS control object is owned by caller
    bOwner =
        SMSO_bOwner((SMS_OBJECT)gpsSms);
    if (bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // Search from start(top) of list
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Find this SRM entry
        eReturnCode = OSAL.eLinkedListSearch(
            gpsSms->hSRMList,
            &hEntry,
            (void *)hSRM
                );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Remove this SRM from the list
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMS_NAME": Could not remove SRM entry: %s.",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
                    "Removed SRM entry from list.\n");
                bSuccess = TRUE;
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMS_NAME": Unable to find SRM entry: %s\n",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Not the owner of the SMS object.");
    }

    return bSuccess;
}

/******************************************************************************
*
*   SMS_hSRM
*
*   This function is used to determine if a requested SRM (by name) has
*   already been instantiated. If so, the existing SRM handle is returned to
*   the caller in the locked state. Meaning the caller must unlock this
*   object when they are through with it (the hSRM).
**
*   Note: When an SRM handle is found, the handle is returned locked. This
*   means the caller must check the return value and unlock the handle
*   if it received a valid one when it is done with it.
*
******************************************************************************/
SRM_OBJECT SMS_hSRM (
    const char *pacDriverName,
    const char *pacSRMName
        )
{
    BOOLEAN bOwner;
    SRM_OBJECT hSRM = SRM_INVALID_OBJECT;

    // Validate inputs
    if ((pacDriverName == NULL) ||
        (pacSRMName == NULL))
    {
        return SRM_INVALID_OBJECT;
    }

    // Verify the SMS control object is owned by caller
    bOwner =
        SMSO_bOwner((SMS_OBJECT)gpsSms);
    if (bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        SMS_CONTROL_FIND_SRM_STRUCT sFind;

        // Populate the structure
        sFind.pacDriverName = pacDriverName;
        sFind.pacSRMName = pacSRMName;

        // Locate the SRM descriptor with this name.
        eReturnCode = OSAL.eLinkedListLinearSearch(
            gpsSms->hSRMList,
            &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16CompareNames,
            (void *)&sFind );
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Extract SRM handle
            hSRM = (SRM_OBJECT)OSAL.pvLinkedListThis( hEntry );
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Not the owner of the SMS object.");
    }

    return hSRM;
}

/******************************************************************************
*
*   SMS_bLoadDeviceGroup
*
******************************************************************************/
BOOLEAN SMS_bLoadDeviceGroup (
    FILE *psDevice
        )
{
    N32 n32Err;
    DEVICE_GROUP tDeviceGroup;
    BOOLEAN bSuccess = FALSE,
            bLocked = FALSE;

    do
    {
        // Find the device group
        n32Err = ioctl(
            psDevice,
            SRH_IOCTL_DEVICE_GROUP,
            &tDeviceGroup);

        if ( n32Err != DEV_OK )
        {
            // Could not find device group data
            break;
        }

        bLocked = SMS_bLock();
        if (bLocked == FALSE)
        {
            break;
        }

        // No errors
        bSuccess = TRUE;

        // We only set this once
        if (gpsSms->bDeviceGroupSet == FALSE)
        {
            gpsSms->tDeviceGroup = tDeviceGroup;
            gpsSms->bDeviceGroupSet = TRUE;
        }

        SMS_vUnLock();
    } while (FALSE);

    return bSuccess;
}

/******************************************************************************
*
*   SMS_bDeviceGroup
*
******************************************************************************/
BOOLEAN SMS_bDeviceGroup (
    DEVICE_GROUP *ptDeviceGroup
        )
{
    BOOLEAN bLocked, bHaveDeviceGroup = FALSE;

    if (ptDeviceGroup == NULL)
    {
        return FALSE;
    }

    // Lock SMS
    bLocked = SMS_bLock();

    // Extract the device group only if  we're locked
    if (bLocked == TRUE)
    {
        // Do we have a device group?
        bHaveDeviceGroup = gpsSms->bDeviceGroupSet;

        if (bHaveDeviceGroup == TRUE)
        {
            // Yes, extract the group value
            *ptDeviceGroup = gpsSms->tDeviceGroup;
        }

        SMS_vUnLock();
    }

    return bHaveDeviceGroup;
}

/******************************************************************************
*
*   SMS_hCreateDSMChildObject
*
*   Creates a DSM child object under the lock of the DSM
*
******************************************************************************/
SMS_OBJECT SMS_hCreateDSMChildObject (
    SMS_CREATE_DSM_CHILD_CALLBACK hCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bIsInitialized;
    SMS_OBJECT hDSMChild = SMS_INVALID_OBJECT;

    // Are we initialized?
    bIsInitialized = SMS_bIsInitialized();
    if (bIsInitialized == TRUE)
    {
        BOOLEAN bLocked;

        // Lock it down
        bLocked = SMSO_bLock(gpsSms->sDSM.hDSMCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            // Create the object
            hDSMChild = hCallback(gpsSms->sDSM.hDSMCtrl, pvCallbackArg);

            // All done!
            SMSO_vUnlock(gpsSms->sDSM.hDSMCtrl);
        }
    }

    return hDSMChild;
}

/******************************************************************************
*
*   SMS_hAllocateDSMEvent
*
******************************************************************************/
SMS_EVENT_HDL SMS_hAllocateDSMEvent (
    SMS_EVENT_TYPE_ENUM eEvent,
    void **ppvData,
    EVENT_OPTIONS_TYPE tOptions
        )
{
    SMS_EVENT_DATA_UNION **ppuEventData =
        (SMS_EVENT_DATA_UNION **)ppvData;
    SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;
    BOOLEAN bValid;

    bValid = SMSO_bIsValid((SMS_OBJECT)gpsSms);
    if (bValid == TRUE)
    {
        // Allocate the event from the DSM's event handler
        hEvent = SMSE_hAllocateEvent(
            gpsSms->sDSM.hDSMEventHandler,
            eEvent, ppuEventData, tOptions);
    }

    return hEvent;
}

/******************************************************************************
*
*   SMS_hDevice
*
******************************************************************************/
DEVICE_OBJECT SMS_hDevice ( void )
{
    BOOLEAN bValid;
    DEVICE_OBJECT hDevice = DEVICE_INVALID_OBJECT;

    bValid = SMSO_bIsValid((SMS_OBJECT)gpsSms);
    if (bValid == TRUE)
    {
        hDevice = gpsSms->sDSM.hDevice;
    }

    return hDevice;
}

/******************************************************************************
*
*   SMS_hUseDSMCtrl
*
******************************************************************************/
SMS_OBJECT SMS_hUseDSMCtrl ( void )
{
    BOOLEAN bValid;
    SMS_OBJECT hDSMCtrl = SMS_INVALID_OBJECT;

    bValid = SMSO_bIsValid((SMS_OBJECT)gpsSms);
    if (bValid == TRUE)
    {
        BOOLEAN bOwner;

        bOwner = SMSO_bOwner((SMS_OBJECT)gpsSms->sDSM.hDSMCtrl);
        if (bOwner == TRUE)
        {
            hDSMCtrl = gpsSms->sDSM.hDSMCtrl;
        }
    }

    return hDSMCtrl;
}

/******************************************************************************
*
*   SMS_pacGetPath
*
******************************************************************************/
const char *SMS_pacGetPath ( void )
{
    BOOLEAN bInitialized;
    const char *pacPath = NULL;

    // Verify
    bInitialized = SMS_bIsInitialized();
    if (bInitialized == TRUE)
    {
        pacPath = gpsSms->pacConfigPath;
    }

    return pacPath;
}

/******************************************************************************
*
*   SMS_hGetTag
*
******************************************************************************/
TAG_OBJECT SMS_hGetTag ( void )
{
    BOOLEAN bInitialized;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;

    // Verify
    bInitialized = SMS_bIsInitialized();
    if (bInitialized == TRUE)
    {
        hTag = gpsSms->hTag;
    }

    return hTag;
}

/******************************************************************************
*
*   SMS_hSongTagService
*
******************************************************************************/
SONG_TAG_OBJECT SMS_hSongTagService ( void )
{
    BOOLEAN bIsInitialized;
    SONG_TAG_OBJECT hSongTag = SONG_TAG_INVALID_OBJECT;

    // Verify the SMS control object is initialized
    bIsInitialized = SMS_bIsInitialized();
    if (bIsInitialized == TRUE)
    {
        hSongTag = gpsSms->hSongTag;
    }

    return hSongTag;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/******************************************************************************
*
*   bInitConfig
*
******************************************************************************/
static BOOLEAN bInitConfig ( BOOLEAN *pbConfigFileReset )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bDirectoryExists;
    UN8 un8DirAttrs;
    TAG_OBJECT hParentTag, hTag = TAG_INVALID_OBJECT;

    //////////////////////////////////////////////////////
    /// Validate and create path if necessary          ///
    //////////////////////////////////////////////////////

    bDirectoryExists =
        OSAL.bFileSystemGetFileAttributes(gpsSms->pacConfigPath, &un8DirAttrs);

    // We may need to create this directory...
    if (bDirectoryExists == FALSE)
    {
        BOOLEAN bMakeDir;

        // Create the requested directory
        bMakeDir = OSAL.bFileSystemMakeDir( gpsSms->pacConfigPath );
        if (bMakeDir == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "SMS: Failed to create configuration dir. Exiting...");
            return FALSE;
        }

    }
    else if ((un8DirAttrs & OSAL_FILE_ATTR_DIRECTORY) == FALSE)
    {
        // Pointed us to a file -- why would anybody do that?
        return FALSE;
    }

    //////////////////////////////////////////////////////
    /// Start the configuration manager, set path tag  ///
    //////////////////////////////////////////////////////

    // CM_eInitialize NULL checks on the boolean pointer
    // for config file reset, so need to check it this level.
    eReturnCode = CM_eInitialize(
                                  SMS_CONF_FILE,
                                  gpsSms->pacConfigPath,
                                  gpsSms->pacInitializerFile,
                                  gpsSms->pacInitializerPath,
                                  pbConfigFileReset
                                 );

    do
    {
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        hParentTag = CM_hGetTopTag();
        eReturnCode = TAG_eGet(SMS_NAME, hParentTag, &hTag, NULL, TRUE);
            // create if it doesn't exist
        if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
             // couldn't get or create this SRM's tag
             break;
        }

        gpsSms->hTag = hTag;

        return TRUE;

    } while (0);

    return FALSE;
}

/******************************************************************************
*
*   vUninitSMSConfig
*
******************************************************************************/
static void vUninitConfig ( void )
{
    // Stop the configuration manager
    CM_vUninitialize();

    // Invalidate the pointer
    gpsSms->pacConfigPath = NULL;

    gpsSms->hTag = TAG_INVALID_OBJECT;

    return;
}

/******************************************************************************
*
*   n16CompareHandles
*
*   An OSAL linked list compare handler that compares a descriptor to
*   a provided handle.
*
******************************************************************************/
static N16 n16CompareHandles (
    void *pvObj1,
    void *pvObj2
        )
{
    SRM_OBJECT hSRM1 = (SRM_OBJECT)pvObj1,
        hSRM2 = (SRM_OBJECT)pvObj2;
    N16 n16Result = N16_MIN;
    BOOLEAN bValid;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)hSRM2);
    if(bValid == TRUE)
    {
        if (hSRM1 < hSRM2)
        {
            n16Result = -1;
        }
        else if (hSRM1 > hSRM2)
        {
            n16Result = 1;
        }
        else
        {
            n16Result = 0;
        }
    }

    return n16Result;
}

/******************************************************************************
*
*   n16CompareNames
*
*   An OSAL linked list compare handler that always traverses the linked
*   list in the same direction and invokes the SRM OBJECT's compare name
*   function in order to determine when a match is found.
*
******************************************************************************/
static N16 n16CompareNames (
    SRM_OBJECT hSRM,
    SMS_CONTROL_FIND_SRM_STRUCT *psFind
        )
{
    N16 n16Result;

    n16Result = SRM_n16CompareName(
        hSRM, psFind->pacDriverName, psFind->pacSRMName);
    if(n16Result != 0)
    {
        n16Result = -1;
    }

    return n16Result;
}


/*******************************************************************************
*
*   SMS_eCompareVersions
*
*   This function compares two version numbers and returns an ordering.
*   This function will also determine if a possible rollover has occurred.
*   For the purposes of this function, a rollover is defined as a high-order
*   bit (HOB) change (otherwise known as the 'Deerfield Park' algorithm.)
*   This function is commutative, so the order of versions fed in as "A"
*   and "B" will not change the rollover detection result.
*
*   Inputs:
*       tVersionBitWidth     - the number of bits required to represent the
*                               maximum version number
*       un32VersionA         - version number A
*       un32VersionB         - version number B
*       pbRolloverIndication - a pointer to a BOOLEAN that will be set to TRUE
*                              on a roll-over indication. Can be NULL.
*
*   Outputs:
*      SMS_VERSION_EQUAL              - version numbers have the same value
*      SMS_VERSION_GREATER            - version B is numerically
*                                       greater than version A
*      SMS_VERSION_LESSER             - version B is numerically
*                                       lesser than version A
*      SMS_VERSION_COMPARISON_INVALID - version numbers can not be compared
*                                       due to invalid inputs
*
*******************************************************************************/

SMS_VERSION_COMPARE_ENUM SMS_eCompareVersions (
        size_t   tVersionBitWidth,
        UN32     un32VersionA,
        UN32     un32VersionB,
        BOOLEAN *pbRolloverIndication
    )
{
    SMS_VERSION_COMPARE_ENUM eResult    = SMS_VERSION_EQUAL;
    BOOLEAN                  bHOBChange = FALSE;

    //
    // First, ensure that tVersionBitWidth is reasonable (less than
    // or equal to 32 bits)
    //

    if ( tVersionBitWidth > SMS_MAX_BIT_WIDTH )
    {
        return SMS_VERSION_COMPARISON_INVALID;
    }

    //
    // First, mask out the HOB from both the old and new versions.
    // If the high-order bit changed, then mark it.
    //

    {
        UN32 un32HOBMask;
        UN32 un32HOBA;
        UN32 un32HOBB;

        un32HOBMask = (1 << (tVersionBitWidth - 1) );
        un32HOBA    = un32VersionA & un32HOBMask;
        un32HOBB    = un32VersionB & un32HOBMask;

        if ( (un32HOBA ^ un32HOBB) > 0 )
        {
            bHOBChange = TRUE;
        }
    }

    //
    // We now do our numeric comparison on the version
    // numbers to determine which is bigger.
    //

    if ( un32VersionB < un32VersionA )
    {
        eResult = SMS_VERSION_LESSER;
    }
    else if ( un32VersionB > un32VersionA )
    {
        eResult = SMS_VERSION_GREATER;
    }
    else
    {
        eResult = SMS_VERSION_EQUAL;
    }

    //
    // If the caller is interested (the rollover pointer is not NULL),
    // then we'll inform them of possible rollover via pbRollover. We
    // will set this to true upon detecting a HOB change.
    //

    if ( pbRolloverIndication != NULL )
    {
        if ( bHOBChange == TRUE )
        {
            *pbRolloverIndication = TRUE;
        }
        else
        {
            *pbRolloverIndication = FALSE;
        }
    }

    return eResult;
}

/******************************************************************************
*
*   vReleaseSRMs
*
******************************************************************************/
static void vReleaseSRMs( void )
{
    BOOLEAN bResult;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify and lock the object
    bResult = SMS_bLock();
    if (FALSE == bResult)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Unable to lock SMS.");
        return;
    }

    if (OSAL_INVALID_OBJECT_HDL == gpsSms->hSRMList)
    {
        // Nothing to do
        SMS_vUnLock();
        return;
    }

    // Tell all SRMs to release...
    eReturnCode =
        OSAL.eLinkedListIterate(
            gpsSms->hSRMList,
            bReleaseSRMIterator,
            (void *)NULL);
    if (OSAL_SUCCESS == eReturnCode)
    {
        SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
            "SRM(s) release is requested.\n");
    }
    else if (OSAL_NO_OBJECTS == eReturnCode)
    {
        SMSAPI_DEBUG_vPrint(SMS_NAME, 4,
            "There are no SRMs to release.\n");
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SMS_NAME": Unable to initiate SRM(s) release: %d (%s)",
            eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
    }

    SMS_vUnLock();

    return;
}

/******************************************************************************
*
*   bReleaseSRMIterator
*
******************************************************************************/
static BOOLEAN bReleaseSRMIterator (
    void *pvData,
    void *pvArg
        )
{
    SRM_OBJECT hSRM = (SRM_OBJECT)pvData;

    SRM_bRelease(hSRM, SMS_OBJECT_RELEASE_BY_PARENT);

    // Keep iteration going
    return TRUE;
}

#ifdef SUPPORT_CUNIT
#include <sms.cunit>
#endif
