/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SMART_FAVORITES implementation for the
 *  Satellite Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "module_obj.h"
#include "decoder_obj.h"
#include "category_obj.h"
#include "channel_obj.h"
#include "cm.h"

#include "sms_obj.h"

#include "smart_favorites_obj.h"
#include "_smart_favorites_obj.h"

#include "sms_api_debug.h"

static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   eStart
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStart (
    DECODER_OBJECT hDecoder,
    BOOLEAN bCurrentBand
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        MODULE_OBJECT hModule;
        BOOLEAN bEnabled = FALSE, bSupported;
        SMART_FAVORITES_OBJECT hSmartFavorites;
        SMS_EVENT_SMART_FAVORITES_STRUCT sEventData;
        SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint = SMART_FAVORITES_PLAY_POINT_CTRL_START;

        // Get the module this decoder is attached to
        hModule = DECODER_hModule(hDecoder);

        // Verify Advanced IR is supported
        bSupported = MODULE_bIsAdvancedIrSupported(hModule);
        if (bSupported == FALSE)
        {
            puts(SMART_FAVORITES_OBJECT_NAME
                ": Advanced IR is not supported by the Module");

            // Relinquish ownership
            SMSO_vUnlock((SMS_OBJECT)hDecoder);

            return SMSAPI_RETURN_CODE_UNSUPPORTED_API;
        }

        // Get Smart Favorites service handle from DECODER
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);
        if (hSmartFavorites != SMART_FAVORITES_INVALID_OBJECT)
        {
            // Service is already started
            eReturnCode = SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
        }
        else
        {
            // Service is not started yet
            eReturnCode = eCreateSmartFavorites(
                hDecoder, &ePlayPoint, &bEnabled, bCurrentBand);
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        // Only if the service was just started
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            BOOLEAN bOk;

            // Populate Smart Favorites event data struct
            sEventData.eAction = SMART_FAVORITES_ACTION_START;

            // Send Smart Favorites start event
            bOk = DECODER_bSmartFavorites(hDecoder, &sEventData);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME
                    ": Cannot send START action");
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            }

            // Smart Favorites service state was saved as 'enabled'
            if (bEnabled == TRUE)
            {
                // Populate Smart favorites event data struct
                sEventData.eAction = SMART_FAVORITES_ACTION_ENABLE;
                sEventData.ePlayPoint = ePlayPoint;

                // Send Smart Favorites enable event
                bOk = DECODER_bSmartFavorites(hDecoder, &sEventData);
                if (bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SMART_FAVORITES_OBJECT_NAME
                        ": Cannot send ENABLE action");
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                }
            }
            else
            {
                // Smart Favorites service is not enabled yet
                // Populate Smart favorites event data struct
                sEventData.eAction = SMART_FAVORITES_ACTION_DISABLE;

                // Send Smart Favorites enable event
                bOk = DECODER_bSmartFavorites(hDecoder, &sEventData);
                if (bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SMART_FAVORITES_OBJECT_NAME
                        ": Cannot send ENABLE action");
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                }
            }
        }
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   vStop
 *
 *****************************************************************************/
static void vStop (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        size_t tCategorySize = 0;
        SMART_FAVORITES_OBJECT hSmartFavorites;
        SMS_EVENT_SMART_FAVORITES_STRUCT sEventData;

        // Get Smart Favorites handle
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);
        if (hSmartFavorites != SMART_FAVORITES_INVALID_OBJECT)
        {
            CCACHE_OBJECT hCCache;

            // De-reference object
            SMART_FAVORITES_OBJECT_STRUCT *psObj =
                (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;

            // Clear the Smart Favorites service handle for this decoder
            DECODER_bSetSmartFavoritesHandle(
                hDecoder,
                SMART_FAVORITES_INVALID_OBJECT,
                CATEGORY_INVALID_ID
                    );

            // get channel cache
            hCCache = DECODER_hCCache(psObj->hDecoder);

            tCategorySize = CATEGORY.tSize(hDecoder, psObj->tCategoryId);

            // Unregister channel notification
            CCACHE_bUnRegisterAllChannelsNotification(
                 hCCache,
                 vChannelEventCallback,
                 (void *)psObj);

            // Remove the category
            vRemoveCategory(psObj);

            // Destroy object
            SMSO_vDestroy((SMS_OBJECT)hSmartFavorites);
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        // Send out STOP event
        if (tCategorySize > 0)
        {
            BOOLEAN bSuccess;

            // Populate Smart Favorites event struct
            sEventData.eAction = SMART_FAVORITES_ACTION_STOP;

            bSuccess = DECODER_bSmartFavorites(hDecoder, &sEventData);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME
                    ": Cannot send STOP action");
            }
        }
    }

    return;
}

/*****************************************************************************
 *
 *   eEnable
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnable (
    DECODER_OBJECT hDecoder,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint
        )
{
    BOOLEAN bLocked = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    SMS_EVENT_SMART_FAVORITES_STRUCT sEventData;
    SMART_FAVORITES_OBJECT_STRUCT *psObj;
    SMART_FAVORITES_OBJECT hSmartFavorites;
    BOOLEAN bOk;

    do
    {
        // Validate the input
        if (ePlayPoint < SMART_FAVORITES_PLAY_POINT_CTRL_LIVE ||
            ePlayPoint > SMART_FAVORITES_PLAY_POINT_CTRL_AUTOMATIC)
        {
            break;
        }

        // Lock the DECODER
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Get Smart Favorites handle
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);
        if (hSmartFavorites == SMART_FAVORITES_INVALID_OBJECT)
        {
            // Service is not started started yet
            eReturnCode = SMSAPI_RETURN_CODE_NOT_INITIALIZED;
            break;
        }

        // Service is running
        psObj = (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;

        // Populate Smart favorites event data struct
        sEventData.eAction = SMART_FAVORITES_ACTION_ENABLE;
        sEventData.ePlayPoint = ePlayPoint;

        // Set current state
        psObj->bEnabled = TRUE;
        psObj->ePlayPoint = ePlayPoint;

        // Save state in tag
        bOk = bSaveState(psObj->hTag, TRUE);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Cannot save Smart Favorites enabled state");
        }

        // Save Play Point in tag
        bOk = bSavePlayPoint(psObj->hTag, ePlayPoint);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Cannot save Smart Favorites Play Point position");
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    // Send Smart Favorites enable event if everything else is good.
    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        bOk = DECODER_bSmartFavorites(hDecoder, &sEventData);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Cannot send ENABLE action");
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eDisable
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisable (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        SMART_FAVORITES_OBJECT hSmartFavorites;
        SMS_EVENT_SMART_FAVORITES_STRUCT sEventData;

        // Get Smart Favorites object handle
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);
        if (hSmartFavorites == SMART_FAVORITES_INVALID_OBJECT)
        {
            // Service is not started started yet
            eReturnCode = SMSAPI_RETURN_CODE_NOT_INITIALIZED;
        }
        else
        {
            BOOLEAN bOk;

            // Service is running
            SMART_FAVORITES_OBJECT_STRUCT *psObj =
                (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;

            // Populate Smart favorites event data struct
            sEventData.eAction = SMART_FAVORITES_ACTION_DISABLE;

            // Set current state
            psObj->bEnabled = FALSE;

            // Save state in tag
            bOk = bSaveState(psObj->hTag, FALSE);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME
                    ": Cannot save Smart Favorites disabled state");
            }

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        // Send Smart Favorites disable event
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            BOOLEAN bOk = DECODER_bSmartFavorites(hDecoder, &sEventData);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME
                    ": Cannot send ENABLE action");
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            }
        }
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eGetCurrentConfig
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetCurrentConfig (
    DECODER_OBJECT hDecoder,
    BOOLEAN *pbEnabled,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM *pePlayPoint
        )
{
    BOOLEAN bLocked = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = 
        SMSAPI_RETURN_CODE_ERROR;

    do
    {
        SMART_FAVORITES_OBJECT hSmartFavorites;
        SMART_FAVORITES_OBJECT_STRUCT *psObj;

        // Verify inputs
        if ((pbEnabled == NULL) && (pePlayPoint == NULL))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break; 
        }

        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Get Smart Favorites object handle
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);
        if (hSmartFavorites == SMART_FAVORITES_INVALID_OBJECT)
        {
            // Service is not started yet
            eReturnCode = SMSAPI_RETURN_CODE_NOT_INITIALIZED;
            break;
        }

        // Service is running
        psObj = (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;

        // Pass throungh the service status
        if (pbEnabled != NULL)
        {
            *pbEnabled = psObj->bEnabled;
        }

        if (pePlayPoint != NULL)
        {
            *pePlayPoint = psObj->ePlayPoint;
        }

        // Everything went well
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   tCategoryId
 *
 *****************************************************************************/
static CATEGORY_ID tCategoryId (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        SMART_FAVORITES_OBJECT hSmartFavorites;

        // Get Smart Favorites object handle
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);
        if (hSmartFavorites != SMART_FAVORITES_INVALID_OBJECT)
        {
            // De-reference object
            SMART_FAVORITES_OBJECT_STRUCT *psObj =
                (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;

            // Pass thru category id
            tCategoryId = psObj->tCategoryId;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tCategoryId;
}

/*****************************************************************************
 *
 *   n32FPrintf
 *
 *****************************************************************************/
static N32 n32FPrintf (
    DECODER_OBJECT hDecoder,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOptions
        )
{
    BOOLEAN blocked;
    N32 n32Chars = 0;

    // Verify and lock object
    blocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if ( blocked == TRUE && psFile != NULL )
    {
        SMART_FAVORITES_OBJECT hSmartFavorites;

        // Get Smart Favorites service handle
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);
        if (hSmartFavorites != SMART_FAVORITES_INVALID_OBJECT)
        {
            SMART_FAVORITES_PRINT_STRUCT sPrint;

            // De-reference object
            SMART_FAVORITES_OBJECT_STRUCT *psObj =
                (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;

            n32Chars += fprintf(psFile, "Smart Favorites Service");

            if (eOutputOptions == SMS_OUTPUT_OPTION_GROSS)
            {
                n32Chars += fprintf(psFile, " (object: %p)",
                    hSmartFavorites);
            }

            n32Chars += fprintf(psFile, "\n");

            sPrint.tSize = CATEGORY.tSize(hDecoder, psObj->tCategoryId);

            if (psObj->bCurrentBand == FALSE)
            {
                n32Chars += fprintf(psFile, " Using Virtual Category\n");
            }
            else
            {
                n32Chars += fprintf(psFile, " Using Current Preset Band\n");
            }

            n32Chars += fprintf(psFile, " Enabled: %s\n", ((psObj->bEnabled == TRUE) ? "Yes" : "No"));
            n32Chars += fprintf(psFile, " Play Point: %s\n", pacPlayPoint(psObj->ePlayPoint));

            if (psObj->tCategoryId == CATEGORY_INVALID_ID)
            {
                n32Chars += fprintf(psFile, " Category Id: N/A\n");
            }
            else
            {
                n32Chars += fprintf(psFile, " Category Id: %d\n", psObj->tCategoryId);
            }

            n32Chars += fprintf(psFile, " Channels total: %d\n", sPrint.tSize);

            if (eOutputOptions == SMS_OUTPUT_OPTION_TERSE)
            {
                n32Chars += fprintf(psFile,
                    " Channel IDs:");

                // Populate print struct
                sPrint.pn32Chars = &n32Chars;
                sPrint.psFile = psFile;

                // Print channels
                CATEGORY.eIterate(
                    hDecoder,
                    psObj->tCategoryId,
                    n16PrintChannelsTerse,
                    &sPrint);

                n32Chars += fprintf(psFile, "\n");
            }
            else if (eOutputOptions == SMS_OUTPUT_OPTION_VERBOSE)
            {
                n32Chars += fprintf(psFile,
                    "*************************************\n");

                n32Chars += fprintf(psFile,
                    " %*.*s  %*.*s  %*.*s  %-*.*s\n",
                    SMART_FAVORITES_MAX_ID_LENGTH, SMART_FAVORITES_MAX_ID_LENGTH,
                    " # ",
                    SMART_FAVORITES_MAX_ID_LENGTH, SMART_FAVORITES_MAX_ID_LENGTH,
                    " ID",
                    SMART_FAVORITES_MAX_ID_LENGTH, SMART_FAVORITES_MAX_ID_LENGTH,
                    "SID",
                    SMART_FAVORITES_MAX_NAME_LENGTH, SMART_FAVORITES_MAX_NAME_LENGTH,
                    "Channel Name"
                        );

                n32Chars += fprintf(psFile,
                    "*************************************\n");

                // Populate print struct
                sPrint.pn32Chars = &n32Chars;
                sPrint.psFile = psFile;

                // Print channels
                CATEGORY.eIterate(
                    hDecoder,
                    psObj->tCategoryId,
                    n16PrintChannelsVerbose,
                    &sPrint);

                n32Chars += fprintf(psFile,
                    "*************************************\n");
            }
            else if (eOutputOptions == SMS_OUTPUT_OPTION_GROSS)
            {
                n32Chars += fprintf(psFile,
                    "**********************************************************************\n");

                n32Chars += fprintf(psFile,
                    " %*.*s  %*.*s  %*.*s  %-*.*s  %-*.*s  %-*.*s\n",
                    SMART_FAVORITES_MAX_ID_LENGTH, SMART_FAVORITES_MAX_ID_LENGTH,
                    " # ",
                    SMART_FAVORITES_MAX_ID_LENGTH, SMART_FAVORITES_MAX_ID_LENGTH,
                    " ID",
                    SMART_FAVORITES_MAX_ID_LENGTH, SMART_FAVORITES_MAX_ID_LENGTH,
                    "SID",
                    SMART_FAVORITES_MAX_NAME_LENGTH, SMART_FAVORITES_MAX_NAME_LENGTH,
                    "Channel Name",
                    SMART_FAVORITES_MAX_POS_LENGTH, SMART_FAVORITES_MAX_POS_LENGTH,
                    "Play on Select",
                    SMART_FAVORITES_MAX_CCT_LENGTH, SMART_FAVORITES_MAX_CCT_LENGTH,
                    "Content Type"
                        );

                n32Chars += fprintf(psFile,
                    "**********************************************************************\n");

                // Populate print struct
                sPrint.pn32Chars = &n32Chars;
                sPrint.psFile = psFile;

                // Print channels
                CATEGORY.eIterate(
                    hDecoder,
                    psObj->tCategoryId,
                    n16PrintChannelsGross,
                    &sPrint);

                n32Chars += fprintf(psFile,
                    "**********************************************************************\n");
            }
        }
    }

    if (blocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    if (n32Chars == 0)
    {
        n32Chars = EOF;
    }

    return n32Chars;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   SMART_FAVORITES_bEventHandler
 *
 *****************************************************************************/
BOOLEAN SMART_FAVORITES_bEventHandler (
    DECODER_OBJECT hDecoder,
    SMART_FAVORITES_OBJECT hSmartFavorites,
    const SMS_EVENT_SMART_FAVORITES_STRUCT *psEventData
        )
{
    SMART_FAVORITES_OBJECT_STRUCT *psObj = (SMART_FAVORITES_OBJECT_STRUCT*)NULL;
    BOOLEAN bReturn = FALSE, bOwner = FALSE, bValid = FALSE;

    if (psEventData->eAction != SMART_FAVORITES_ACTION_STOP)
    {
        if (hSmartFavorites == SMART_FAVORITES_INVALID_OBJECT)
        {
            puts(SMART_FAVORITES_OBJECT_NAME": Service is not started yet");
            return TRUE;
        }

        // For STOP action we do not need a valid Smart Favorites object
        bValid = SMSO_bValid((SMS_OBJECT)hSmartFavorites);
        if (bValid == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME": Object is not valid");
            return FALSE;
        }

        // De-reference object
        psObj = (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;
    }

    // Verify ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // Process event...
        switch (psEventData->eAction)
        {
            case SMART_FAVORITES_ACTION_START:
            {
                puts(SMART_FAVORITES_OBJECT_NAME": Start");

                bReturn = bStartSmartFavorites(psObj);
            }
            break;

            case SMART_FAVORITES_ACTION_STOP:
            {
                puts(SMART_FAVORITES_OBJECT_NAME": Stop");

                bReturn = RADIO_bSmartSelect(hDecoder, NULL, 0);
            }
            break;

            case SMART_FAVORITES_ACTION_DISABLE:
            {
                puts(SMART_FAVORITES_OBJECT_NAME": Disable");

                bReturn = bDisableSmartFavorites(hDecoder);
            }
            break;

            case SMART_FAVORITES_ACTION_ENABLE:
            {
                puts(SMART_FAVORITES_OBJECT_NAME": Enable");

                bReturn = bEnableSmartFavorites(
                    hDecoder, 
                    psEventData->ePlayPoint);
            }
            break;

            case SMART_FAVORITES_ACTION_UPDATE:
            {
                puts(SMART_FAVORITES_OBJECT_NAME": Update");

                bReturn = bUpdateSmartFavorites(psObj);
            }
            break;
        }
    }

    return bReturn;
}

/*****************************************************************************
 *
 *   SMART_FAVORITES_bIsPresetsSync
 *
 *****************************************************************************/
BOOLEAN SMART_FAVORITES_bIsPresetsSync (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bLocked, bSyncronized = FALSE;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        SMART_FAVORITES_OBJECT_STRUCT *psObj;
        SMART_FAVORITES_OBJECT hSmartFavorites;

        // Get Smart Favorites object handle
        hSmartFavorites = DECODER_hGetSmartFavoritesHandle(hDecoder);

        if (hSmartFavorites != SMART_FAVORITES_INVALID_OBJECT)
        {
            // De-reference object
            psObj = (SMART_FAVORITES_OBJECT_STRUCT *)hSmartFavorites;

            bSyncronized = psObj->bCurrentBand;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return bSyncronized;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   vCategoryEventCallback
 *
 *****************************************************************************/
static void vCategoryEventCallback (
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    void * pvEventCallbackArg
        )
{
    SMART_FAVORITES_OBJECT_STRUCT *psObj =
        (SMART_FAVORITES_OBJECT_STRUCT *)pvEventCallbackArg;

    puts(SMART_FAVORITES_OBJECT_NAME": Internal category event call-back");

    // Process events based on event mask
    if ((tEventMask & CATEGORY_OBJECT_EVENT_CONTENTS) ==
        CATEGORY_OBJECT_EVENT_CONTENTS)
    {
        BOOLEAN bOwner;

        puts(SMART_FAVORITES_OBJECT_NAME": Category event CONTENTS");

        bOwner = SMSO_bIsOwner((SMS_OBJECT)hCategory);
        if (bOwner == TRUE)
        {
            puts(SMART_FAVORITES_OBJECT_NAME": Perform synchronous update");

            bUpdateSmartFavorites(psObj);
        }
        else
        {
            BOOLEAN bPosted;
            SMS_EVENT_SMART_FAVORITES_STRUCT sEventData;

            puts(SMART_FAVORITES_OBJECT_NAME": Perform asynchronous update");

            sEventData.eAction = SMART_FAVORITES_ACTION_UPDATE;

            bPosted = DECODER_bSmartFavorites(psObj->hDecoder, &sEventData);
            if (bPosted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME": Failed to post Update event");
            }
        }
    }
#if DEBUG_OBJECT==1
    else if ((tEventMask & CATEGORY_OBJECT_EVENT_ART) ==
        CATEGORY_OBJECT_EVENT_ART)
    {
        puts(SMART_FAVORITES_OBJECT_NAME": category event ART");
    }
    else if ((tEventMask & CATEGORY_OBJECT_EVENT_NAME) ==
        CATEGORY_OBJECT_EVENT_NAME)
    {
        puts(SMART_FAVORITES_OBJECT_NAME": Category event NAME");
    }
    else if ((tEventMask & CATEGORY_OBJECT_EVENT_REMOVED) ==
        CATEGORY_OBJECT_EVENT_REMOVED)
    {
        puts(SMART_FAVORITES_OBJECT_NAME": Category event REMOVED");
    }
#endif

    return;
}

/*****************************************************************************
 *
 *   eCreateSmartFavorites
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCreateSmartFavorites (
    DECODER_OBJECT hDecoder,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM *pePlayPoint,
    BOOLEAN *pbEnabled,
    BOOLEAN bCurrentBand
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SMART_FAVORITES_OBJECT_STRUCT *psObj = (SMART_FAVORITES_OBJECT_STRUCT*)NULL;

    do
    {
        BOOLEAN bOk, bRegistered;
        CCACHE_OBJECT hCCache;
        CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

        // Verify ownership
        bOk = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if (bOk == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Create new object
        psObj = (SMART_FAVORITES_OBJECT_STRUCT *)SMSO_hCreate(
            SMART_FAVORITES_OBJECT_NAME,
            sizeof(SMART_FAVORITES_OBJECT_STRUCT),
            SMS_INVALID_OBJECT, // No parent
            FALSE // Not lockable
                );

        // Not enough memory
        if (psObj == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Not enough memory to create Smart Favorites object");
            break;
        }

        // Initialize object with defaults
        *psObj = gsObjectDefaults;

        // Set decoder object handle into Smart Favorites object struct
        psObj->hDecoder = hDecoder;

        // Set Current Band Tracking Status
        psObj->bCurrentBand = bCurrentBand;

        // get channel cache
        hCCache = DECODER_hCCache(hDecoder);

        // Register all channel notification
        bRegistered = CCACHE_bRegisterAllChannelsNotification(
                hCCache,
                CHANNEL_OBJECT_EVENT_REMOVED,
                FALSE,
                vChannelEventCallback,
                (void *)psObj);

        if (bRegistered == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME": Failed to register channel notification");
        }

        // Get Smart Favorites Tag
        bOk = bCreateTag(psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Failed to get Smart Favorites tag object");
            break;
        }

        // Restore Play Point Ctrl and Smart Favorites Channels from SMS cfg
        bOk = bRestoreSmartFavorites(psObj, pePlayPoint, pbEnabled);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Failed to restore Smart Favorites");
            break;
        }

        if (psObj->tCategoryId == CATEGORY_INVALID_ID)
        {
            // Create virtual category
            psObj->tCategoryId = CATEGORY_tCreateVirtualCategory(
                hDecoder,
                SMART_FAVORITES_CATEGORY_LONG_NAME,
                SMART_FAVORITES_CATEGORY_SHORT_NAME,
                0, TRUE, TRUE);

            // Failed to create virtual category for some reason
            if (psObj->tCategoryId == CATEGORY_INVALID_ID)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME
                    ": Failed to create Smart Favorites virtual category");
                break;
            }
        }

        psObj->ePlayPoint = *pePlayPoint;
        psObj->bEnabled = *pbEnabled;

        // Register category update notifications
        bOk = bRegisterCategory(psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Failed to register Smart Favorites category notifications");
            break;
        }

        if (bCurrentBand == TRUE)
        {
            tCategoryId = psObj->tCategoryId;
        }

        // Set Smart Favorites handle into Decoder
        bOk = DECODER_bSetSmartFavoritesHandle(hDecoder, (SMART_FAVORITES_OBJECT)psObj, tCategoryId);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME
                ": Failed to set Smart Favorites handle");
            break;
        }

        // Seems to be OK, if we can go so far ...
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (psObj != NULL)
        {
            // Dissociate from decoder
            DECODER_bSetSmartFavoritesHandle(
                hDecoder,
                SMART_FAVORITES_INVALID_OBJECT,
                CATEGORY_INVALID_ID
                    );

            // Remove the category
            vRemoveCategory(psObj);

            //and destroy the object
            SMSO_vDestroy((SMS_OBJECT)psObj);
        }
    }

    return eReturnCode;
}

/*******************************************************************************
 *
 *   bCreateTag
 *
 *******************************************************************************/
static BOOLEAN bCreateTag (
    SMART_FAVORITES_OBJECT_STRUCT *psObj
        )
{
    TAG_OBJECT hParentTag;
    BOOLEAN bReturn = FALSE;

    // Get the decoder tag
    hParentTag = DECODER_hGetTag(psObj->hDecoder);
    if (hParentTag != TAG_INVALID_OBJECT)
    {
        TAG_OBJECT hTag;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Get Smart Favorites Tag
        eReturnCode = TAG_eGet(SMART_FAVORITES_OBJECT_NAME, hParentTag, &hTag, NULL, TRUE);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Save Smart Favorites Tag Locally
            psObj->hTag = hTag;
            bReturn = TRUE;
        }
    }

    return bReturn;
}

/*******************************************************************************
 *
 *   bDisableSmartFavorites
 *
 *******************************************************************************/
static BOOLEAN bDisableSmartFavorites (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bReturn;

    // Set Play Point at Live
    bReturn = RADIO_bSetSmartFavsCfg(
        hDecoder, SMART_FAVORITES_PLAY_POINT_CTRL_LIVE);

    return bReturn;
}

/*******************************************************************************
 *
 *   bEnableSmartFavorites
 *
 *******************************************************************************/
static BOOLEAN bEnableSmartFavorites (
    DECODER_OBJECT hDecoder,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint
        )
{
    BOOLEAN bReturn;

    bReturn = RADIO_bSetSmartFavsCfg(hDecoder, ePlayPoint);

    return bReturn;
}

/*******************************************************************************
 *
 *   n16PrintChannelsTerse
 *
 *******************************************************************************/
static N16 n16PrintChannelsTerse (
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    SMART_FAVORITES_PRINT_STRUCT *psPrint =
        (SMART_FAVORITES_PRINT_STRUCT *)pvIterateArg;

    *psPrint->pn32Chars += fprintf( psPrint->psFile, " %d",
        CHANNEL.tChannelId(hChannel) );

    // Check if we are done
    if (--psPrint->tSize == 0)
    {
        // Stop iterating
        return 0;
    }

    return 1;
}

/*******************************************************************************
 *
 *   n16PrintChannelsVerbose
 *
 *******************************************************************************/
static N16 n16PrintChannelsVerbose (
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    SMART_FAVORITES_PRINT_STRUCT *psPrint =
        (SMART_FAVORITES_PRINT_STRUCT *)pvIterateArg;

    *psPrint->pn32Chars += fprintf(
        psPrint->psFile,
        " %*d  %*d  %*d  %-*.*s\n",
        SMART_FAVORITES_MAX_ID_LENGTH,
        tCurrentIndex,
        SMART_FAVORITES_MAX_ID_LENGTH,
        (N16)CHANNEL.tChannelId(hChannel),
        SMART_FAVORITES_MAX_ID_LENGTH,
        (N16)CHANNEL.tServiceId(hChannel),
        SMART_FAVORITES_MAX_NAME_LENGTH, SMART_FAVORITES_MAX_NAME_LENGTH,
        STRING.pacCStr(CHANNEL.hLongName(hChannel))
                );

    // Check if we are done
    if (--psPrint->tSize == 0)
    {
       // Stop iterating
       return 0;
    }

    return 1;
}

/*******************************************************************************
 *
 *   n16PrintChannelsGross
 *
 *******************************************************************************/
static N16 n16PrintChannelsGross (
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    SMART_FAVORITES_PRINT_STRUCT *psPrint =
        (SMART_FAVORITES_PRINT_STRUCT *)pvIterateArg;

    *psPrint->pn32Chars += fprintf(
        psPrint->psFile,
        " %*d  %*d  %*d  %-*.*s  %-*.*s  %-*.*s\n",
        SMART_FAVORITES_MAX_ID_LENGTH,
        tCurrentIndex,
        SMART_FAVORITES_MAX_ID_LENGTH,
        (N16)CHANNEL.tChannelId(hChannel),
        SMART_FAVORITES_MAX_ID_LENGTH,
        (N16)CHANNEL.tServiceId(hChannel),
        SMART_FAVORITES_MAX_NAME_LENGTH, SMART_FAVORITES_MAX_NAME_LENGTH,
        STRING.pacCStr(CHANNEL.hLongName(hChannel)),
        SMART_FAVORITES_MAX_POS_LENGTH, SMART_FAVORITES_MAX_POS_LENGTH,
        CHANNEL_pacPlayOnSelectMethod(hChannel),
        SMART_FAVORITES_MAX_CCT_LENGTH, SMART_FAVORITES_MAX_CCT_LENGTH,
        CHANNEL_pacContentType(hChannel)
                );

    // Check if we are done
    if (--psPrint->tSize == 0)
    {
       // Stop iterating
       return 0;
    }

    return 1;
}

/*******************************************************************************
 *
 *   bRegisterCategory
 *
 *   This function performs all the necessary steps needed to register
 *   use of the Smart Favorites category.
 *
 *******************************************************************************/
static BOOLEAN bRegisterCategory (
    SMART_FAVORITES_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bRegistered;
    CCACHE_OBJECT hCCache;
    CATEGORY_OBJECT hCategory;

    // Get Channel Cache from Decoder
    hCCache = DECODER_hCCache(psObj->hDecoder);

    // Get Smart Favorites category handle
    hCategory = CCACHE_hCategory(hCCache, &psObj->tCategoryId, 0);

    // Register our local call-back
    bRegistered = CATEGORY_bRegisterNotification(
        hCategory,
        CATEGORY_OBJECT_EVENT_CONTENTS,
        vCategoryEventCallback,
        (void *)psObj);

    return bRegistered;
}

/*******************************************************************************
 *
 *   vRemoveCategory
 *
 *   This function performs all the necessary steps to remove the Smart Favorites
 *   category from SMS.
 *
 *******************************************************************************/
static void vRemoveCategory (
    SMART_FAVORITES_OBJECT_STRUCT *psObj
        )
{
    // Remove the category if necessary
    if (psObj->tCategoryId != CATEGORY_INVALID_ID)
    {
        CCACHE_OBJECT hCCache;
        CATEGORY_OBJECT hCategory;

        hCCache = DECODER_hCCache(psObj->hDecoder);
        hCategory = CCACHE_hCategory(hCCache, &psObj->tCategoryId, 0);

        if (hCategory != CATEGORY_INVALID_OBJECT)
        {
            CATEGORY_vUnregisterNotification(
                hCategory,
                vCategoryEventCallback,
                (void *)psObj);
        }

        // First remove this category from the ccache
        CCACHE_vRemoveCategory(hCCache, psObj->tCategoryId);

        // Complete the cleanup
        CATEGORY_vDestroy(hCategory);

        // Clear the category id
        psObj->tCategoryId = CATEGORY_INVALID_ID;
    }

    return;
}

/*******************************************************************************
 *
 *   bRestoreSmartFavorites
 *
 *******************************************************************************/
static BOOLEAN bRestoreSmartFavorites (
    SMART_FAVORITES_OBJECT_STRUCT *psObj,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM *pePlayPoint,
    BOOLEAN *pbEnabled
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    TAG_OBJECT hPlayPointTag, hEnabledTag;

    // Restore Play Point Ctrl value
    eReturnCode = TAG_eGet(
        SMART_FAVORITES_PLAY_POINT_CTRL_TAG_NAME,
        psObj->hTag,
        &hPlayPointTag,
        NULL,
        FALSE);

    // We have a Play Point tag handle
    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hPlayPointTag != TAG_INVALID_OBJECT))
    {
        size_t tSize;
        N32 n32Value = (N32)SMART_FAVORITES_PLAY_POINT_CTRL_LIVE,
            *pn32Value;

        pn32Value = &n32Value;

        // Get tag value
        (void)TAG_eGetTagValue(
            hPlayPointTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32Value,
            &tSize);

        *pePlayPoint = (SMART_FAVORITES_PLAY_POINT_CTRL_ENUM)n32Value;
    }

    // Restore service state
    eReturnCode = TAG_eGet(
        SMART_FAVORITES_ENABLED_TAG_NAME,
        psObj->hTag,
        &hEnabledTag,
        NULL,
        FALSE);

    // We have service state tag handle
    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hEnabledTag != TAG_INVALID_OBJECT))
    {
        size_t tSize;
        N32 n32Value = (N32)FALSE,
            *pn32Value;

        pn32Value = &n32Value;

        // Get tag value
        (void)TAG_eGetTagValue(
            hEnabledTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32Value,
            &tSize);

        *pbEnabled = (BOOLEAN)n32Value;
    }

    // There is no need to restore Smart Favorites
    // category when the service is linked to presets
    if (psObj->bCurrentBand == FALSE)
    {
        psObj->tCategoryId =
            CATEGORY_tCreateFromTags(
                psObj->hDecoder, psObj->hTag);
    }

    return TRUE;
}

/*******************************************************************************
 *
 *   bSaveState
 *
 *******************************************************************************/
static BOOLEAN bSaveState (
    TAG_OBJECT hParentTag,
    BOOLEAN bEnabled
        )
{
    TAG_OBJECT hTag;
    BOOLEAN bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Save service state

    // Get Smart Favorites Enabled Tag
    eReturnCode = TAG_eGet(
        SMART_FAVORITES_ENABLED_TAG_NAME,
        hParentTag,
        &hTag,
        NULL,
        TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32Value = (N32)bEnabled;

        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            TRUE);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   bSaveState
 *
 *******************************************************************************/
static BOOLEAN bSavePlayPoint (
    TAG_OBJECT hParentTag,
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint
        )
{
    TAG_OBJECT hTag;
    BOOLEAN bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = TAG_eGet(
        SMART_FAVORITES_PLAY_POINT_CTRL_TAG_NAME,
        hParentTag,
        &hTag,
        NULL,
        TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32Value = (N32)ePlayPoint;

        eReturnCode = TAG_eSetTagValue(
            hTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            TRUE);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   bStartSmartFavorites
 *
 *******************************************************************************/
static BOOLEAN bStartSmartFavorites (
    SMART_FAVORITES_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bReturn = TRUE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    SMART_FAVORITES_ITERATOR_STRUCT sIterate;

    if (psObj->bCurrentBand == TRUE)
    {
        return bReturn;
    }

    // Smart Favorites configured to use internal category
    OSAL.bMemSet(&sIterate, 0, sizeof(SMART_FAVORITES_ITERATOR_STRUCT));

    // Populate iterate struct
    sIterate.un16Size = (UN16)CATEGORY.tSize(psObj->hDecoder, psObj->tCategoryId);

    if (sIterate.un16Size > 0)
    {
        sIterate.ptServiceId = (SERVICE_ID *)
            SMSO_hCreate(
                "SF ARR", sizeof(SERVICE_ID) * sIterate.un16Size, SMS_INVALID_OBJECT, FALSE);

        if (sIterate.ptServiceId == NULL)
        {
            // Out of memory. Very bad...
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SMART_FAVORITES_OBJECT_NAME": No memory for SID array");

            return FALSE;
        }

        eReturnCode = CATEGORY.eIterate(
            psObj->hDecoder,
            psObj->tCategoryId,
            n16StartIterator,
            (void *)&sIterate
                );

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bReturn = RADIO_bSmartSelect(
                psObj->hDecoder,
                &sIterate.ptServiceId[0],
                sIterate.un16ArraySize
                    );
        }
        else
        {
            bReturn = FALSE;
        }

        SMSO_vDestroy((SMS_OBJECT)sIterate.ptServiceId);
    }

    return bReturn;
}

/*******************************************************************************
 *
 *   n16StartIterator
 *
 *******************************************************************************/
static N16 n16StartIterator (
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    SERVICE_ID tServiceId;
    CHANNEL_ID tChannelId;
    SMART_FAVORITES_ITERATOR_STRUCT *psIterate =
        (SMART_FAVORITES_ITERATOR_STRUCT *)pvIterateArg;

    // Get channel Id
    tServiceId = CHANNEL.tServiceId(hChannel);
    tChannelId = CHANNEL.tChannelId(hChannel);

    // Verify that channel is valid
    if ((tServiceId != SERVICE_INVALID_ID) && (tChannelId != CHANNEL_INVALID_ID))
    {
        // Add this channel to array
        psIterate->ptServiceId[psIterate->un16ArraySize++] = tServiceId;
    }

    if (--psIterate->un16Size == 0)
    {
        // Stop iterating
        return 0;
    }

    return 1;
}

/*******************************************************************************
 *
 *   n16UpdateIterator
 *
 *******************************************************************************/
static N16 n16UpdateIterator (
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        )
{
    SERVICE_ID tServiceId;
    CHANNEL_ID tChannelId;

    SMART_FAVORITES_ITERATOR_STRUCT *psIterate =
        (SMART_FAVORITES_ITERATOR_STRUCT *)pvIterateArg;

    printf(SMART_FAVORITES_OBJECT_NAME
        ": Currently iterated channel Idx: %d, Id: %d, sId: %d\n",
        tCurrentIndex, (N16)CHANNEL.tChannelId(hChannel), (N16)CHANNEL.tServiceId(hChannel));

    // Get channel Id
    tServiceId = CHANNEL.tServiceId(hChannel);
    tChannelId = CHANNEL.tChannelId(hChannel);

    do
    {
        // Verify that channel is valid. SID was already checked above.
        if ((tServiceId != SERVICE_INVALID_ID) && (tChannelId != CHANNEL_INVALID_ID))
        {
            // Add this channel to array
            psIterate->ptServiceId[psIterate->un16ArraySize++] = tServiceId;
        }

    } while (FALSE);

    // We have reach the end of category
    if (--psIterate->un16Size == 0)
    {
        // Stop iterating
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
 *
 *   bUpdateSmartFavorites
 *
 *******************************************************************************/
static BOOLEAN bUpdateSmartFavorites (
    SMART_FAVORITES_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = FALSE;
    SMART_FAVORITES_ITERATOR_STRUCT sIterate;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;

    OSAL.bMemSet(&sIterate, 0, sizeof(SMART_FAVORITES_ITERATOR_STRUCT));

    do
    {
        /* Serialize category data */

        // Make sure that the service is not running in the 'Presets' mode
        if (psObj->bCurrentBand == FALSE)
        {
            eReturnCode = CATEGORY_eSerialize(
                psObj->hDecoder,
                psObj->tCategoryId,
                psObj->hTag);

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

        // Populate iterate struct
        sIterate.un16Size = (UN16)CATEGORY.tSize(psObj->hDecoder, psObj->tCategoryId);

        if (sIterate.un16Size > 0)
        {
            sIterate.ptServiceId = (SERVICE_ID * )SMSO_hCreate(
                SMART_FAVORITES_OBJECT_NAME" ARR",
                    sizeof(SERVICE_ID) * sIterate.un16Size, SMS_INVALID_OBJECT, FALSE);

            if (sIterate.ptServiceId == NULL)
            {
                // Out of memory. Very bad...
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME": No memory for SID array");
                break;
            }

            // iterate the category, adding a tag for every favorite
            eReturnCode = CATEGORY.eIterate(
                psObj->hDecoder,
                psObj->tCategoryId,
                n16UpdateIterator,
                (void *)&sIterate
                    );
        }

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ||
            (eReturnCode == SMSAPI_RETURN_CODE_NO_OBJECTS))
        {
            bSuccess = RADIO_bSmartSelect(
                psObj->hDecoder,
                &sIterate.ptServiceId[0],
                sIterate.un16ArraySize
                    );

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SMART_FAVORITES_OBJECT_NAME": Failed to send Smart Select request");
                break;
            }
        }

    } while (FALSE);

    if (sIterate.ptServiceId != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)sIterate.ptServiceId);
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   pacPlayPoint
 *
 *******************************************************************************/
static char *pacPlayPoint (
    SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPointCtrl
        )
{
    char *pacReturnString = "Unknown";

    if (ePlayPointCtrl == SMART_FAVORITES_PLAY_POINT_CTRL_AUTOMATIC)
    {
        pacReturnString = "Auto";
    }
    else if (ePlayPointCtrl == SMART_FAVORITES_PLAY_POINT_CTRL_LIVE)
    {
        pacReturnString = "Live";
    }
    else if (ePlayPointCtrl == SMART_FAVORITES_PLAY_POINT_CTRL_START)
    {
        pacReturnString = "Start";
    }

    return pacReturnString;
}

/*******************************************************************************
 *
 *   vChannelEventCallback
 *
 *******************************************************************************/
static void vChannelEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void * pvEventCallbackArg
        )
{
    SMART_FAVORITES_OBJECT_STRUCT *psObj =
        (SMART_FAVORITES_OBJECT_STRUCT *)pvEventCallbackArg;

    if ((tEventMask & CHANNEL_OBJECT_EVENT_REMOVED) &&
        (psObj->bCurrentBand == FALSE))
    {
        CCACHE_OBJECT hCCache;
        CATEGORY_OBJECT hCategory;

        hCCache = DECODER_hCCache(psObj->hDecoder);
        hCategory = CCACHE_hCategory(hCCache, &psObj->tCategoryId, 0);

        CATEGORY_vRemoveChannel(hCategory, hChannel, FALSE);
    }

    return;
}
