/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SKI_CONDITIONS implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_ski_conditions_obj.h"
#include "ski_conditions_obj.h"
#include "weather_event_obj.h"
#include "temperature_obj.h"
#include "ski_attributes_obj.h"
#include "snow_conditions_obj.h"

#if SMS_DEBUG == 1
#include "sms_api_debug.h"
//static const char *gpacThisFile = __FILE__;
#endif /* SMS_DEBUG == 1 */

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       un32TimeStamp
*
*****************************************************************************/
static UN32 un32TimeStamp (
    SKI_CONDITIONS_OBJECT hSkiCond
        )
{
    SKI_CONDITIONS_OBJECT_STRUCT *psObj =
        (SKI_CONDITIONS_OBJECT_STRUCT *)hSkiCond;
    BOOLEAN bOwner = FALSE;
    UN32 un32TimeStamp = 0;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
    if (bOwner == TRUE)
    {
        un32TimeStamp = psObj->un32TimeStamp;
    }

    return un32TimeStamp;
}


/*****************************************************************************
*
*    eOperationalStatus
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eOperationalStatus (
    SKI_CONDITIONS_OBJECT hSkiCond,
    BOOLEAN *pbOperStatus
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *) hSkiCond;

    do
    {
        if (pbOperStatus == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->sData.eOpStat == WEATHER_FLAG_TRUE)
        {
            *pbOperStatus = TRUE;
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else if (psObj->sData.eOpStat == WEATHER_FLAG_FALSE)
        {
            *pbOperStatus = FALSE;
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*    eWeatherEvent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWeatherEvent (
    SKI_CONDITIONS_OBJECT hSkiCond,
    WEATHER_EVENT_OBJECT *phWeatherEvent
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *) hSkiCond;

    do
    {
        if (phWeatherEvent == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hWeatherEvent == WEATHER_EVENT_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << SKI_EVENT_CODE_FLAG))
            {
                psObj->hWeatherEvent = WEATHER_EVENT_hCreate((SMS_OBJECT)hSkiCond, TRUE, psObj->sData.tEvent);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }
        if (psObj->hWeatherEvent == WEATHER_EVENT_INVALID_OBJECT)
        {
            break;
        }

        *phWeatherEvent = psObj->hWeatherEvent;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*    eTemperature
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTemperature (
    SKI_CONDITIONS_OBJECT hSkiCond,
    TEMPERATURE_OBJECT *phTemperature
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *) hSkiCond;

    do
    {
        if (phTemperature == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hTemperature == TEMPERATURE_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << SKI_CURRENT_TEMP_FLAG))
            {
                psObj->hTemperature = TEMPERATURE_hCreate(
                        (SMS_OBJECT)hSkiCond,
                        FALSE,
                        0,
                        FALSE,
                        0,
                        TRUE,
                        psObj->sData.n16Temp);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }

        if (psObj->hTemperature == TEMPERATURE_INVALID_OBJECT)
        {
            break;
        }

        *phTemperature = psObj->hTemperature;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*    eWindCondition
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWindCondition (
    SKI_CONDITIONS_OBJECT hSkiCond,
    SKI_WIND_CONDITION_ENUM *peWindCond
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *) hSkiCond;

    do
    {
        if (peWindCond == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->sData.un16Flags & (1 << SKI_WIND_COND_FLAG))
        {
            *peWindCond = (SKI_WIND_CONDITION_ENUM)psObj->sData.eWind;
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*    eSnowConditions
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSnowConditions (
    SKI_CONDITIONS_OBJECT hSkiCond,
    SNOW_CONDITIONS_OBJECT *phSnowCond
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *) hSkiCond;
    BOOLEAN bMinFlag = FALSE, bMaxFlag = FALSE, bNewFlag = FALSE, bSnowFlag = FALSE;

    do
    {
        if (phSnowCond == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hSnow == SNOW_CONDITIONS_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << SKI_SNOW_COND_FLAG))
            {
                bSnowFlag = TRUE;
            }

            if (psObj->sData.un16Flags & (1 << SKI_MIN_BASE_DEPTH_FLAG))
            {
                bMinFlag = TRUE;
            }

            if (psObj->sData.un16Flags & (1 << SKI_MAX_BASE_DEPTH_FLAG))
            {
                bMaxFlag = TRUE;
            }

            if (psObj->sData.un16Flags & (1 << SKI_NEW_SNOW_FLAG))
            {
                bNewFlag = TRUE;
            }


            if ((bMinFlag == TRUE) || (bMaxFlag == TRUE) ||
                    (bSnowFlag == TRUE) || (bNewFlag == TRUE))
            {

                psObj->hSnow = SNOW_CONDITIONS_hCreate(
                        (SMS_OBJECT)hSkiCond,
                        bMinFlag,
                        psObj->sData.un8BMin,
                        bMaxFlag,
                        psObj->sData.un8BMax,
                        bNewFlag,
                        psObj->sData.tNew,
                        bSnowFlag,
                        (UN8) psObj->sData.eSnow);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }
        if (psObj->hSnow == SNOW_CONDITIONS_INVALID_OBJECT)
        {
            break;
        }

        *phSnowCond = psObj->hSnow;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*    eAttributes
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAttributes (
    SKI_CONDITIONS_OBJECT hSkiCond,
    SKI_ATTRIBUTES_OBJECT *phSkiAttributes
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *)hSkiCond;
    BOOLEAN bTrailFlag = FALSE, bLiftFlag = FALSE;

    do
    {
        if (phSkiAttributes == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Clear pointer
        *phSkiAttributes = SKI_ATTRIBUTES_INVALID_OBJECT;

        bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->hAttributes == SKI_ATTRIBUTES_INVALID_OBJECT)
        {
            if (psObj->sData.un16Flags & (1 << SKI_NUM_OF_TRAILS_FLAG))
            {
                bTrailFlag = TRUE;
            }

            if (psObj->sData.un16Flags & (1 << SKI_NUM_OF_LIFTS_FLAG))
            {
                bLiftFlag = TRUE;
            }


            if ((psObj->sData.eBoard != WEATHER_FLAG_NOT_AVAILABLE) ||
                    (psObj->sData.eGroom != WEATHER_FLAG_NOT_AVAILABLE) ||
                    (psObj->sData.eMake != WEATHER_FLAG_NOT_AVAILABLE) ||
                    (psObj->sData.eNight != WEATHER_FLAG_NOT_AVAILABLE) ||
                    (bTrailFlag == TRUE) || (bLiftFlag == TRUE))
            {

                psObj->hAttributes =
                    SKI_ATTRIBUTES_hCreate(
                        (SMS_OBJECT)hSkiCond,
                        psObj->sData.eBoard,
                        psObj->sData.eGroom,
                        psObj->sData.eMake,
                        psObj->sData.eNight,
                        bLiftFlag,
                        psObj->sData.un8Lift,
                        bTrailFlag,
                        psObj->sData.tTrail);
            }
            else
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }
        }

        if (psObj->hAttributes == SKI_ATTRIBUTES_INVALID_OBJECT)
        {
            break;
        }

        *phSkiAttributes = psObj->hAttributes;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*    n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    SKI_CONDITIONS_OBJECT hSkiCond,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0, n32Temp;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *) hSkiCond;
    BOOLEAN bOwner;
    WEATHER_EVENT_OBJECT hWeatherEvent = WEATHER_EVENT_INVALID_OBJECT;
    TEMPERATURE_OBJECT hTemperature = TEMPERATURE_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult;
    BOOLEAN bOpStat;

    // Determine if we own this object
    bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print SKI_CONDITIONS information header
    n32Return = fprintf(psFile, "\nSki Conditions");
    if (eOutputOption == SMS_OUTPUT_OPTION_GROSS)
    {
        n32Return += fprintf(psFile, " (Object: 0x%X)", hSkiCond);
    }
    n32Return += fputs(":", psFile);

    // Always print these
    eResult = SKI_CONDITIONS.eOperationalStatus(hSkiCond, &bOpStat);
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        n32Temp = fprintf(psFile, "\tOperational: %s\n",
            (bOpStat == TRUE)?"Yes":"No");
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    eResult = SKI_CONDITIONS.eWeatherEvent(hSkiCond, &hWeatherEvent);
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        n32Temp = WEATHER_EVENT.n32FPrintf(hWeatherEvent, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    eResult = SKI_CONDITIONS.eTemperature(hSkiCond, &hTemperature);
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        n32Temp = TEMPERATURE.n32FPrintf(hTemperature, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    if ((eOutputOption == SMS_OUTPUT_OPTION_VERBOSE) ||
        (eOutputOption == SMS_OUTPUT_OPTION_GROSS))
    {
        SKI_WIND_CONDITION_ENUM eWindCond;
        SKI_ATTRIBUTES_OBJECT hAttrs;
        SNOW_CONDITIONS_OBJECT hSnowCond;

        eResult = SKI_CONDITIONS.eSnowConditions(hSkiCond, &hSnowCond);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = SNOW_CONDITIONS.n32FPrintf(hSnowCond, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }

        eResult = SKI_CONDITIONS.eAttributes(hSkiCond, &hAttrs);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = SKI_ATTRIBUTES.n32FPrintf(hAttrs, psFile);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }

        eResult = SKI_CONDITIONS.eWindCondition(hSkiCond, &eWindCond);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            n32Temp = fprintf(psFile, "\tWind Condition = %x\n",
                eWindCond);
            if (n32Temp > 0)
            {
                n32Return += n32Temp;
            }
        }
    }
    if (eOutputOption == SMS_OUTPUT_OPTION_GROSS)
    {
        UN32 un32TimeStamp;
        struct tm tTimeStruct;
        char acBuffer[OSAL_ASCBUFSIZE];
        TIME_T tTime;

        n32Temp = fprintf(psFile, "\tHash Entry = %x\n", psObj->sData.hHashEntry);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }

        // Get timestamp of the object (in UTC)
        un32TimeStamp = SKI_CONDITIONS.un32TimeStamp(hSkiCond);
        // Convert into local time
        tTime = (TIME_T)(un32TimeStamp);
        OSAL.localtime_r( &tTime, &tTimeStruct );
        // Now convert into text for printout
        OSAL.asctime_r( &tTimeStruct, acBuffer );

        n32Temp = fprintf(psFile, "\tTimeStamp = %s\n", acBuffer);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   SKI_CONDITIONS_hCreate
*
*****************************************************************************/
SKI_CONDITIONS_OBJECT SKI_CONDITIONS_hCreate(
    SMS_OBJECT hParent,
    SKI_DATA_STRUCT *psData,
    UN32 un32TimeStamp
        )
{
    SKI_CONDITIONS_OBJECT_STRUCT *psObj;

    // Create an instance of the object
    psObj = (SKI_CONDITIONS_OBJECT_STRUCT *)
        SMSO_hCreate(
            SKI_CONDITIONS_OBJECT_NAME,
            sizeof(SKI_CONDITIONS_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return SKI_CONDITIONS_INVALID_OBJECT;
    }

    psObj->hAttributes = SKI_ATTRIBUTES_INVALID_OBJECT;
    psObj->hSnow = SNOW_CONDITIONS_INVALID_OBJECT;
    psObj->hTemperature = TEMPERATURE_INVALID_OBJECT;
    psObj->hWeatherEvent = WEATHER_EVENT_INVALID_OBJECT;
    OSAL.bMemCpy(&psObj->sData, psData, sizeof(psObj->sData));

    if (un32TimeStamp == 0)
    {
        // Set the timestamp to record the creation time
        OSAL.eTimeGet(&psObj->un32TimeStamp);
    }
    else
    {
        psObj->un32TimeStamp = un32TimeStamp;
    }

    gun32SkiCondCounter++;

    return (SKI_CONDITIONS_OBJECT)psObj;
}


/*****************************************************************************
*
*   SKI_CONDITIONS_vDestroy
*
*****************************************************************************/
void SKI_CONDITIONS_vDestroy (
    SKI_CONDITIONS_OBJECT hSkiCond
        )
{
    BOOLEAN bOwner;
    SKI_CONDITIONS_OBJECT_STRUCT *psObj = (SKI_CONDITIONS_OBJECT_STRUCT *)hSkiCond;

    bOwner = SMSO_bOwner((SMS_OBJECT) hSkiCond);
    if(bOwner == TRUE)
    {
        vDestroy(psObj);
    }
    return;
}

/*****************************************************************************
*
*   SKI_CONDITION_un32Count
*
*****************************************************************************/
UN32 SKI_CONDITIONS_un32Count()
{
    return gun32SkiCondCounter;
}

/*****************************************************************************
*
*       SKI_CONDITIONS_vUpdateTimeStamp
*
*****************************************************************************/
void SKI_CONDITIONS_vUpdateTimeStamp (
    SKI_CONDITIONS_OBJECT hSkiCond
        )
{
    SKI_CONDITIONS_OBJECT_STRUCT *psObj =
        (SKI_CONDITIONS_OBJECT_STRUCT *)hSkiCond;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT) hSkiCond);
    if(bOwner == TRUE)
    {
        OSAL.eTimeGet(&(psObj->un32TimeStamp));
    }
    return;
}


/*****************************************************************************
*
*   SKI_CONDITIONS_hHashEntry
*
*****************************************************************************/
WEATHER_HASH_OBJECT SKI_CONDITIONS_hHashEntry (
    SKI_CONDITIONS_OBJECT hSkiCond
        )
{
    SKI_CONDITIONS_OBJECT_STRUCT *psObj =
        (SKI_CONDITIONS_OBJECT_STRUCT *)hSkiCond;
    WEATHER_HASH_OBJECT hHashEntry = NULL;
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hSkiCond);

    // Verify inputs.
    if(bValid == TRUE)
    {
        hHashEntry = psObj->sData.hHashEntry;
    }

    return hHashEntry;
}


/*****************************************************************************
*
*   SKI_CONDITIONS_bUpdateHashEntry
*
*****************************************************************************/
BOOLEAN SKI_CONDITIONS_bUpdateHashEntry(
    SKI_CONDITIONS_OBJECT hSkiCond,
    WEATHER_HASH_OBJECT hNewHashEntry
        )
{
    SKI_CONDITIONS_OBJECT_STRUCT *psObj =
        (SKI_CONDITIONS_OBJECT_STRUCT *)hSkiCond;
    BOOLEAN bResult = FALSE;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hSkiCond);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        psObj->sData.hHashEntry = hNewHashEntry;
        bResult = TRUE;
    }

    return bResult;
}


/*****************************************************************************
*
*       SKI_CONDITIONS_bCompareData
*
*****************************************************************************/
BOOLEAN SKI_CONDITIONS_bCompareData (
    SKI_CONDITIONS_OBJECT hSkiCond,
    SKI_DATA_STRUCT *psSkiData
        )
{
    SKI_CONDITIONS_OBJECT_STRUCT *psObj =
        (SKI_CONDITIONS_OBJECT_STRUCT *)hSkiCond;
    BOOLEAN bValid, bResult = FALSE;

    do {
        bValid = SMSO_bValid((SMS_OBJECT)hSkiCond);
        if (bValid == FALSE)
        {
            break;
        }

        bValid = SMSO_bValid((SMS_OBJECT)psSkiData);
        if (bValid == FALSE)
        {
            break;
        }

        if (psObj->sData.un16Flags != psSkiData->un16Flags)
        {
            break;
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_EVENT_CODE_FLAG))
        {
            if (psObj->sData.tEvent != psSkiData->tEvent)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_CURRENT_TEMP_FLAG))
        {
            if (psObj->sData.n16Temp != psSkiData->n16Temp)
            {
                break;
            }
        }
        if (psObj->sData.eGroom != psSkiData->eGroom)
        {
            break;
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_MAX_BASE_DEPTH_FLAG))
        {
            if (psObj->sData.un8BMax != psSkiData->un8BMax)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_MIN_BASE_DEPTH_FLAG))
        {
            if (psObj->sData.un8BMin != psSkiData->un8BMin)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_NEW_SNOW_FLAG))
        {
            if (psObj->sData.tNew != psSkiData->tNew)
            {
                break;
            }
        }

        if (psObj->sData.eNight != psSkiData->eNight)
        {
            break;
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_NUM_OF_LIFTS_FLAG))
        {
            if (psObj->sData.un8Lift != psSkiData->un8Lift)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_NUM_OF_TRAILS_FLAG))
        {
            if (psObj->sData.tTrail != psSkiData->tTrail)
            {
                break;
            }
        }

        if (psObj->sData.eOpStat != psSkiData->eOpStat)
        {
            break;
        }

        if (psObj->sData.eBoard != psSkiData->eBoard)
        {
            break;
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_SNOW_COND_FLAG))
        {
            if (psObj->sData.eSnow != psSkiData->eSnow)
            {
                break;
            }
        }

        if ((psObj->sData.un16Flags) & (1 << SKI_WIND_COND_FLAG))
        {
            if (psObj->sData.eWind != psSkiData->eWind)
            {
                break;
            }
        }

        if (psObj->sData.eMake != psSkiData->eMake)
        {
            break;
        }

        //we do not compare ext data because it is not used for now

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroy
*
*****************************************************************************/
static void vDestroy(SKI_CONDITIONS_OBJECT_STRUCT *psObj)
{
    if (psObj->hAttributes != SKI_ATTRIBUTES_INVALID_OBJECT)
    {
        SKI_ATTRIBUTES_vDestroy(psObj->hAttributes);
        psObj->hAttributes = SKI_ATTRIBUTES_INVALID_OBJECT;
    }

    if (psObj->hSnow != SNOW_CONDITIONS_INVALID_OBJECT)
    {
        SNOW_CONDITIONS_vDestroy(psObj->hSnow);
        psObj->hSnow = SNOW_CONDITIONS_INVALID_OBJECT;
    }

    if (psObj->hTemperature != TEMPERATURE_INVALID_OBJECT)
    {
        TEMPERATURE_vDestroy(psObj->hTemperature);
        psObj->hTemperature = TEMPERATURE_INVALID_OBJECT;
    }

    if (psObj->hWeatherEvent != WEATHER_EVENT_INVALID_OBJECT)
    {
        WEATHER_EVENT_vDestroy(psObj->hWeatherEvent);
        psObj->hWeatherEvent = WEATHER_EVENT_INVALID_OBJECT;
    }

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)psObj);

    gun32SkiCondCounter--;

    return;
}
