/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SEEK implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "cas.h"
#include "ccache.h"
#include "cal_alert.h"
#include "decoder_obj.h"
#include "league_obj.h"
#include "report_obj.h"
#include "string_obj.h"
#include "tag_obj.h"
#include "cm.h"

#include "seek_content.h"
#include "seek.h"
#include "_seek.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eStart
*
* Creates a seek service (SEEK_SERVICE_OBJECT).
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               will be associated with.
*   eService    The specific service that is to be started
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM
*       SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStart (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hService;
        SEEK_SERVICE_OBJECT_STRUCT *psObj = NULL;

        do
        {
            BOOLEAN bOk = TRUE;
            UN32 un32Index;
            const char *pacName;
            const char *pacEnabledText, *pacDisabledText;
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

            // see if there is already a service handle for this type of
            // seek service
            eResult = DECODER_eSeekService(hDecoder, eService, &hService);

            if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // this means one is already started
                eResult = SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
                break;
            }

            // if we got anything other than service not started,
            // we can't start the service
            if (eResult != SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED)
            {
                break;
            }

            // set to error. If all goes well, this will get changed to SUCCESS
            eResult = SMSAPI_RETURN_CODE_ERROR;

            // get the Decoder's name
            pacName = DECODER_pacName(hDecoder);
            if (pacName == NULL)
            {
                break;
            }

            // Create the name for this seek service object
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     SEEK_OBJECT_NAME":%s:%s", pacName, pacSeekServiceText(eService) );
            psObj = (SEEK_SERVICE_OBJECT_STRUCT *)
                SMSO_hCreate(
                    &acName[0],
                    sizeof(SEEK_SERVICE_OBJECT_STRUCT),
                    (SMS_OBJECT)hDecoder,
                    FALSE
                        );

            if(psObj == NULL)
            {
                // Error!
                break;
            }

            // Create the CAL for this alert list
            // use the seek service handle as the argument.
            psObj->hSeekList =
                CAS_hCreateList(
                    hDecoder,
                    vCalAlertCallback,
                   (void*)(SEEK_SERVICE_OBJECT)psObj,
                   FALSE // don't auto ack
                        );
            if (psObj->hSeekList == CAL_INVALID_OBJECT)
            {
                // Error!
                break;
            }

            psObj->psInfo = NULL;

            // set the specific seek service information
            for(un32Index = 0; un32Index < SEEK_NUM_SERVICES; un32Index++)
            {
                // Check if content matches this entry in the map table
                if(gasSeekServiceMap[un32Index].psInfo->eService == eService)
                {
                    // this is the info we need
                    psObj->psInfo = (SEEK_SERVICE_SPECIFIC_INFO_STRUCT *)
                                        gasSeekServiceMap[un32Index].psInfo;
                }
            }

            if (psObj->psInfo == NULL)
            {
                // Error - we were unable to set the specific seek information
                break;
            }

            // initially configured to be enabled
            psObj->eState = SEEK_STATE_ENABLED;

            // not iterating right now, so set to invalid obj
            psObj->hCurrentlyIteratedContent = CAL_CONTENT_INVALID_OBJECT;

            // set the handle of this service in the decoder
            bOk = DECODER_bSetSeekServiceHandle(
                      hDecoder,
                      eService,
                      (SEEK_SERVICE_OBJECT)psObj
                          );

            if (bOk == FALSE)
            {
                break;
            }

            // create some constant strings used for persistence
            pacEnabledText = pacSeekStateText(SEEK_STATE_ENABLED);
            psObj->hEnabledString = STRING_hCreateConst(
                                       pacEnabledText,
                                       strlen(pacEnabledText));
            pacDisabledText = pacSeekStateText(SEEK_STATE_DISABLED);
            psObj->hDisabledString = STRING_hCreateConst(
                                         pacDisabledText,
                                         strlen(pacDisabledText));

            if ( (psObj->hEnabledString == STRING_INVALID_OBJECT) ||
                 (psObj->hDisabledString == STRING_INVALID_OBJECT)
                     )
            {
                break;
            }

            bOk = bRestoreServiceFromConfigFile(
                      psObj,
                      hDecoder
                          );
            if (bOk == FALSE)
            {
                break;
            }

            // is there a service specific init fxn?
            if (psObj->psInfo->psInterface->bInit != NULL)
            {
                // yes. call it
                bOk = psObj->psInfo->psInterface->bInit((SEEK_SERVICE_OBJECT)psObj);

                // successful service specific init?
                if (bOk == FALSE)
                {
                    // unsuccessful
                    break;
                }
            }

            printf(SEEK_OBJECT_NAME": Created %s\n",
                   pacSeekServiceText(eService));

            // Moved the unlock of the decoder object to before
            // the call to modify the registered event mask as
            // modifying the registered event mask is now handled
            // through a synchronous call to the decoder object itself,
            // having the hDecoder locked before making this call
            // would deadlock.
            SMSO_vUnlock((SMS_OBJECT)hDecoder);
            bLocked = FALSE;

            // Update the decoder's event mask
            eResult = DECODER.eModifyRegisteredEventMask(
                          hDecoder,
                          DECODER_OBJECT_EVENT_SEEK_ALERT,
                          SMSAPI_MODIFY_EVENT_MASK_ENABLE
                              );

            if ( eResult != SMSAPI_RETURN_CODE_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SEEK_OBJECT_NAME": could not update registered "
                    "event mask");
                break;
            }

            // success
            return SMSAPI_RETURN_CODE_SUCCESS;

        } while(0);

        if (bLocked == TRUE)
        {
            SMSO_vUnlock((SMS_OBJECT)hDecoder);
        }

        // Something went wrong
        SEEK_vStop((SEEK_SERVICE_OBJECT)psObj);
    }

    return eResult;
}

/*****************************************************************************
*
*   vStop
*
* Stops and destroys a Seek Service.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service that will be stopped
*
* Outputs:
*
*   None.
*
*****************************************************************************/
static void vStop (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SMSAPI_RETURN_CODE_ENUM eResult;
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_vStop(hSeekService);
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
}

/*****************************************************************************
*
*   eState
*
* This API is used to retrieve the Seek Service's state.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service that will be queried
*
* Outputs:
*
*   SEEK_STATE_ENUM indicating the service's state
*
*****************************************************************************/
static SEEK_STATE_ENUM eState (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    BOOLEAN bLocked;
    SEEK_STATE_ENUM eState = SEEK_STATE_INVALID;

    // lock the service
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // retrieve the information
            eState = psObj->eState;
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }


    return eState;
}

/*****************************************************************************
*
*   eEnable
*
* This API is used to globally enable seek alerts for the entire
* Seek Service.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service which the
*                   caller wishes to globally enable seek alerts.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnable(
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    // lock the service
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            if (psObj->eState != SEEK_STATE_ENABLED)
            {
                // globally enable
                psObj->eState = SEEK_STATE_ENABLED;

                bUpdateStateTag(psObj);

                eResult = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                // already enabled. no need to do anything further.
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    // couldn't lock it
    return eResult;
}

/*****************************************************************************
*
*   eDisable
*
* This API is used to globally disable seek alerts for the entire
* Seek Service.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service which the
*                   caller wishes to globally disable seek alerts.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisable (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    // lock the service
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;
            if (psObj->eState != SEEK_STATE_DISABLED)
            {
            // globally disable
            psObj->eState = SEEK_STATE_DISABLED;

            bUpdateStateTag(psObj);

            eResult = SMSAPI_RETURN_CODE_SUCCESS;
        }
            else
            {
                // already disabled. no need to do anything further.
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    // couldn't lock it
    return eResult;
}

/*****************************************************************************
*
*   tCategoryId
*
* This API is used to retrieve the id of the CATEGORY associated with this
* Seek Service.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service which the caller wishes to query.
*
* Outputs:
*
*   A valid CATEGORY_ID on success - CATEGORY_INVALID_ID on error.
*
*****************************************************************************/
static CATEGORY_ID tCategoryId (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    BOOLEAN bLocked;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
                SEEK_SERVICE_OBJECT_STRUCT *psObj =
                    (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // retrieve info
            tCategoryId = CAL.tCategoryId(psObj->hSeekList);
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tCategoryId;
}

/*****************************************************************************
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a seek service's contents to a specified file or device.
* This is mainly helpful during debugging of seek service's but could be used
* by an application for any reason. This API instead sends the seek service
* as a verbose formatted output version.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service which the caller wishes to print.
*   psFile      The device to write the seek service contents to.
*   eOutputOption   The option indicating what level of output should be
*                   printed to psFile
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
N32 n32FPrintf (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = EOF;
    BOOLEAN bLocked;
    SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult;

    if (psFile == NULL)
    {
        return EOF;
    }

    // lock the service
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return EOF;
    }

    // see if there is a service handle for this type of seek service
    eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        SEEK_ITERATOR_STRUCT sIterator;
        size_t tItems;
        CATEGORY_ID tCatId;
        SEEK_SERVICE_OBJECT_STRUCT *psObj =
        (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

        // initialize our iterator struct
        sIterator.un32Count = 0;
        sIterator.psFile = psFile;
        sIterator.n32NumWritten = 0;
        sIterator.eOutputOption = eOutputOption;

        // print
        switch(eOutputOption)
        {
            case SMS_OUTPUT_OPTION_TERSE:
            {
                // just print out the type of service
                sIterator.n32NumWritten +=
                    fprintf(psFile, "%s:\n",
                            pacSeekServiceText(psObj->psInfo->eService));
            }
            break;

            case SMS_OUTPUT_OPTION_VERBOSE:
            case SMS_OUTPUT_OPTION_GROSS:
            default:
            {
                // print out the type of service and its handle
                sIterator.n32NumWritten +=
                        fprintf(psFile, "%s: hSeekService = 0x%p\n",
                                pacSeekServiceText(psObj->psInfo->eService),
                                hSeekService);
            }
            break;
        }

        tCatId = SEEK.tCategoryId(hDecoder, eService);

        if (tCatId != CATEGORY_INVALID_ID)
        {
            sIterator.n32NumWritten += fprintf(psFile, "Category Id: %u\n", tCatId);
        }
        else
        {
            sIterator.n32NumWritten += fprintf(psFile, "Category Id: N/A\n");
        }

        // get the enabled /disabled status and print it
        sIterator.n32NumWritten += fprintf(psFile, "State: %s \n",
                                               pacSeekStateText(psObj->eState));

        // get the number of items that are registered and print it out
        tItems = (UN32)SEEK.tNumItems(hDecoder, eService);
        sIterator.n32NumWritten += fprintf(psFile,
                "There are %u items currently registered\n\n", tItems);

        // based on the type, print out service specific information....
        if (psObj->psInfo->psInterface->n32FPrintf != NULL)
        {
            sIterator.n32NumWritten +=
                            psObj->psInfo->psInterface->n32FPrintf(
                                     hSeekService,
                                     psFile,
                                     eOutputOption
                                         );
        }

        // print out the information on each registered content
        CAL.eIterate( psObj->hSeekList, bPrintContent, &sIterator );

        n32Return = sIterator.n32NumWritten;
    }
    // Unlock object
    SMSO_vUnlock((SMS_OBJECT)hDecoder);

    return n32Return;
}

/*****************************************************************************
*
*   tNumItems
*
*   This API is used to report to the caller the number of registered items
*    to this Seek Service.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service which the caller wishes to query.
*
*   Outputs:
*
*   The number of registered items in this seek service object.
*
*****************************************************************************/
static 	size_t tNumItems (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{

    SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult;
    BOOLEAN bLocked;
    size_t tItems = 0;

    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // ask our CAL how many items are registered...
            tItems = CAL.tNumItems(psObj->hSeekList);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    return tItems;
}

/*****************************************************************************
*
*   eRemoveAll
*
*   This API is used to remove all registered content from the seek service
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service from which the caller wishes to remove
*               all registered content.
*
*   Outputs:
*
*  SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveAll (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{

    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // tell our CAL to remove all registered items...
            eResult = CAL.eRemoveAll(psObj->hSeekList);

            if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // remove from config file
                STRING_OBJECT hTagName;

                hTagName = STRING.hCreate("RegisteredContent", 0);

                // iterate children tags (to remove all registered content tags)
                eResult = TAG_eIterateChildren(psObj->hTag,
                    bRemoveContentTagIterator, (void *)hTagName);

                if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    CM_eCommitChangesToFile();
                }

                // done with this
                STRING.vDestroy(hTagName);
            }
        }
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eResult;
}

/*****************************************************************************
*
*   eIterate
*
* This method allows iteration of the seek service.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    The specific service to iterate over.
*   bContentIterator
*           A function to call for each content entry in the list.
*   pvContentIteratorArg
*           An anonymous pointer which is caller specific and provided
*           each time bContentIterator is called.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService,
    SEEK_CONTENT_ITERATOR_CALLBACK bSeekContentIterator,
    void *pvContentIteratorArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Verify inputs.
    if (bSeekContentIterator == NULL)
    {
        // need to have a non-null callback, otherwise
        // what's the point of iterating?
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock the decoder
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
        SEEK_TO_CAL_ITERATOR_STRUCT sCALIteratorStruct;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // populate the iterator structure
            sCALIteratorStruct.hSeekService = hSeekService;
            sCALIteratorStruct.pvSeekIteratorArg = pvContentIteratorArg;
            sCALIteratorStruct.bSeekIteratorCallback =
                bSeekContentIterator;

            // iterate list
            eResult = CAL.eIterate (
                            psObj->hSeekList,
                            bSeekCalIterator,
                            &sCALIteratorStruct
                                );
        }
        // unlock decoder
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eResult;
}

/*****************************************************************************
*
*   hGetNextEvent
*
* This method is used to get the next active SEEK_EVENT for a given Decoder
*
* Inputs:
*
*   hDecoder - A handle to a valid decoder object for which to get the
*              next active seek event
*
* Outputs:
*
*   A valid SEEK_EVENT_OBJECT if one exists. SEEK_EVENT_INVALID_OBJECT if
*   there are no active events or on error
*****************************************************************************/
static SEEK_EVENT_OBJECT hGetNextEvent(
    DECODER_OBJECT hDecoder
        )
{
    SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
    SEEK_EVENT_OBJECT hEvent = SEEK_EVENT_INVALID_OBJECT;
    UN32 un32Index;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if (bOwner == TRUE)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // loop through the possible seek services
        for(un32Index = 0; un32Index < SEEK_NUM_SERVICES; un32Index++)
        {
            // get the handle for this decoder's instance of this specific service
            eReturnCode = DECODER_eSeekService(
                               hDecoder,
                               gasSeekServiceMap[un32Index].psInfo->eService,
                               &hSeekService
                                   );

            // if we got a handle from the decoder, get that service's next event
            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                SEEK_SERVICE_OBJECT_STRUCT *psObj =
                    (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

                // Looking for first available non-null event
                hEvent = CAL_hGetNextActiveEvent(psObj->hSeekList);
            }
            else
            {
                SMSAPI_DEBUG_vPrint(SEEK_OBJECT_NAME, 2, "Cannot get SEEK service for this DECODER");
            }

            // did we get an event?
            if (hEvent != SEEK_EVENT_INVALID_OBJECT)
            {
                // yes, we got an event. break out of this loop and return it.
                break;
            }
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrint(SEEK_OBJECT_NAME, 2, "Wrong owner for DECODER");
    }
    return hEvent;
}

/*****************************************************************************
*
*   eGetNextEndedEventAttributes
*
* This method is used to get the next ended event's attributes
* for a given Decoder
*
* Inputs:
*
*   hDecoder - A handle to a valid decoder object for which to get the
*              next event's attributes
*
*   ptChannelId   Pointer to the ChannelId to which the event's ChannelId
*                 will be copied.
*   phArtist   Pointer to the Artist to which the event's Artist name
*                 will be copied.
*   ptChannelId   Pointer to the Title to which the event's Title
*                 will be copied.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS if ended event exists.
*   SMSAPI_RETURN_CODE_NO_OBJECTS if list of ended events is empty
*   SMSAPI_RETURN_CODE_ERROR or SMSAPI_RETURN_CODE_INVALID_INPUT on error
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetNextEndedEventAttributes (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID *ptChannelId,
    STRING_OBJECT *phArtist,
    STRING_OBJECT *phTitle
        )
{
    SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    UN32 un32Index;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if (bOwner == TRUE)
    {
        // loop through the possible seek services
        for(un32Index = 0; un32Index < SEEK_NUM_SERVICES; un32Index++)
        {
            // get the handle for this decoder's instance of this specific service
            eReturnCode = DECODER_eSeekService(
                               hDecoder,
                               gasSeekServiceMap[un32Index].psInfo->eService,
                               &hSeekService
                                   );

            // if we got a handle from the decoder, get that service's next event
            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                SEEK_SERVICE_OBJECT_STRUCT *psObj =
                    (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

                // get the service's next active event
                eReturnCode = CAL_hGetNextEndedEventAttributes(
                    psObj->hSeekList,
                    ptChannelId,
                    phArtist,
                    phTitle
                        );

                // did we get an event?
                if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    // yes, we got an event. break out of this loop.
                    break;
                }
            }
        }
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   bGetAlertsTunedStatus
*
* This method is used to get enabled status of alerts
* on the currently tuned channel.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    Seek service enum.
*
* Outputs:
*
*   TRUE if Enabled. FALSE if Disabled
*
*****************************************************************************/
static BOOLEAN bGetAlertsTunedStatus (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked, bEnabled = FALSE;

    // Lock the decoder
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            bEnabled =  CAL_bGetAlertsTunedStatus(psObj->hSeekList);
        }
        // unlock decoder
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return bEnabled;
}

/*****************************************************************************
*
*   eEnableAlertsTuned
*
* This method enables/disables alerts on the currently tuned channel.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    Seek service enum.
*   bEnable     TRUE to enable, FALSE to disable alerts
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnableAlertsTuned (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService,
    BOOLEAN bEnable
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Lock the decoder
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            eResult =  CAL_eEnableAlertsTuned(psObj->hSeekList, bEnable);
        }
        // unlock decoder
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eResult;
}

/*****************************************************************************
*
*   bGetAlertsFinishedStatus
*
* This method is used to get enabled status of alerts at content finish.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    Seek service enum.
*
* Outputs:
*
*   TRUE if Enabled. FALSE if Disabled
*
*****************************************************************************/
static BOOLEAN bGetAlertsFinishedStatus (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked, bEnabled = FALSE;

    // Lock the decoder
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            bEnabled =  CAL_bGetAlertsFinishedStatus(psObj->hSeekList);
        }
        // unlock decoder
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return bEnabled;
}

/*****************************************************************************
*
*   eEnableAlertsFinished
*
* This method enables/disables alerts at content finish.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object to which this seek service
*               is associated with.
*   eService    Seek service enum.
*   bEnable     TRUE to enable, FALSE to disable alerts
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnableAlertsFinished (
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService,
    BOOLEAN bEnable
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Lock the decoder
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

        // see if there is a service handle for this type of seek service
        eResult = DECODER_eSeekService(hDecoder, eService, &hSeekService);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            eResult =  CAL_eEnableAlertsFinished(psObj->hSeekList, bEnable);
        }
        // unlock decoder
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eResult;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   SEEK_psServiceSpecificContentInterface
*
* The function extracts the specific seek service's content interface
*
*****************************************************************************/
const SEEK_CONTENT_OBJ_INTERFACE_STRUCT *SEEK_psServiceSpecificContentInterface(
    SEEK_SERVICE_OBJECT hSeekService
        )
{
    SEEK_SERVICE_OBJECT_STRUCT *psObj =
        (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

    return psObj->psInfo->psContentInterface;
}

/*****************************************************************************
*
*   SEEK_eService
*
*****************************************************************************/
SEEK_SERVICE_ENUM SEEK_eService (
    SEEK_SERVICE_OBJECT hSeekService
        )
{
    BOOLEAN bLocked;
    SEEK_SERVICE_ENUM eService;

    SEEK_SERVICE_OBJECT_STRUCT *psObj =
        (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

    // lock the service
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSeekService, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SEEK_SERVICE_INVALID;
    }

    // retrieve the information
    eService = psObj->psInfo->eService;

    SMSO_vUnlock((SMS_OBJECT)hSeekService);

    return eService;
}

/*****************************************************************************
*
*   SEEK_vStop
*
*****************************************************************************/
void SEEK_vStop(
    SEEK_SERVICE_OBJECT hSeekService
        )
{
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hSeekService, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        SEEK_SERVICE_OBJECT_STRUCT *psObj =
            (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;
        DECODER_OBJECT hDecoder;
    #if DEBUG_OBJECT == 1
        SEEK_SERVICE_ENUM eService;
    #endif

        // All SEEK_SERVICE objects belong to a DECODER object (their parent)
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

        // was the service specific info set?
        if (psObj->psInfo != NULL)
        {
            // yes the service specific info was set

            // uninitialize
            DECODER_bSetSeekServiceHandle(
                hDecoder,
                psObj->psInfo->eService,
                SEEK_SERVICE_INVALID_OBJECT
                    );

            // is there any service specific uninit fxn?
            if (psObj->psInfo->psInterface->vUnInit != NULL)
            {
                // yes there is a service specific uninit fxn
                psObj->psInfo->psInterface->vUnInit(hSeekService);
            }
        }

        // destroy strings
        if (psObj->hEnabledString != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hEnabledString);
            psObj->hEnabledString = STRING_INVALID_OBJECT;
        }
        if (psObj->hDisabledString != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hDisabledString);
            psObj->hDisabledString = STRING_INVALID_OBJECT;
        }

        // Destroy CAL
        if (psObj->hSeekList != CAL_INVALID_OBJECT)
        {
            CAS.vDestroyList(psObj->hSeekList);
            psObj->hSeekList = CAL_INVALID_OBJECT;
        }

#if DEBUG_OBJECT == 1
        // only need if debugging, so conditionally compile to avoid compiler
        // warning about setting but not using a variable
        eService = psObj->psInfo->eService;
#endif

        // uninit
        psObj->hCurrentlyIteratedContent = CAL_CONTENT_INVALID_OBJECT;
        psObj->eState = SEEK_STATE_INVALID;
        psObj->psInfo = NULL;

        // Destroy the object itself
        SMSO_vDestroy((SMS_OBJECT)hSeekService);

#if DEBUG_OBJECT == 1
		printf(SEEK_OBJECT_NAME": Destroyed %s\n", pacSeekServiceText(eService));
#endif

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   bSeekCalIterator
*
* The SEEK iterator maps to the CAL list. This CAL callback translates to a
* SEEK callback
*
*****************************************************************************/
static BOOLEAN bSeekCalIterator (
    CAL_CONTENT_OBJECT hContent,
    void *pvContentIteratorArg
        )
{
    BOOLEAN bReturn = FALSE;

    // non-null arg?
    if (pvContentIteratorArg != NULL)
    {
        BOOLEAN bValid;
        SEEK_TO_CAL_ITERATOR_STRUCT *psIterator =
            (SEEK_TO_CAL_ITERATOR_STRUCT *)pvContentIteratorArg;

        // valid service?
        bValid = SMSO_bValid((SMS_OBJECT)psIterator->hSeekService);
        if (bValid == TRUE)
        {
            DECODER_OBJECT hDecoder;
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)psIterator->hSeekService;

            psObj->hCurrentlyIteratedContent = hContent;
            hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

            // call the seek iterator callback
            bReturn = psIterator->bSeekIteratorCallback(
                          hDecoder,
                          psObj->psInfo->eService,
                          psObj->hCurrentlyIteratedContent,
                          psIterator->pvSeekIteratorArg
                              );

            // this is no longer the currently iterated content
            psObj->hCurrentlyIteratedContent = CAL_CONTENT_INVALID_OBJECT;

        } // if (bValid == TRUE)
    } // if (pvContentIteratorArg != NULL)

    return bReturn;
}

/*****************************************************************************
*
*   vCalAlertCallback
*
*   This is the callback the CAL calls when an Alert occurs.
*   In this fxn the Seek Service decides whether or not
*   it should inform the app of a Seek Alert. If the decision is to inform the
*   app then DECODER_OBJECT_EVENT_SEEK_ALERT is set for the Decoder on which
*   this seek service is running.
*
*****************************************************************************/
static void vCalAlertCallback (
    CAL_OBJECT hCAL,
    CAL_ALERT_OBJECT hAlert,
    UN32 un32Flags,
    void *pvCALCallbackArg
        )
{
    CAL_CONTENT_OBJECT hContent;
    BOOLEAN bValid;

    SEEK_SERVICE_OBJECT hSeekService =
        (SEEK_SERVICE_OBJECT)pvCALCallbackArg;

    // get the content which caused the alert
    hContent = CAL_ALERT.hContent(hAlert);

    // double check that the seek service handle is valid before we use it
    bValid = SMSO_bValid((SMS_OBJECT)hSeekService);
    if (bValid == TRUE)
    {
        SEEK_SERVICE_OBJECT_STRUCT *psObj =
            (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;
        BOOLEAN bEnabled =
            CAL_bGetAlertsFinishedStatus(psObj->hSeekList);

        // Notify the app in two cases:
        // 1. content is beginning
        // 2. content ended and alerts are allowed
        if (((un32Flags & CAL_INITIAL) == CAL_INITIAL) ||
            (((un32Flags & CAL_END) == CAL_END) && 
             ((bEnabled == TRUE))))
        {
            // check if we should inform the app via a seek alert

            BOOLEAN bSetDecoderSeekAlertEvent = FALSE;

            // is there a service specific function to determine if
            // a seek alert event should be set?
            if (psObj->psInfo->psInterface->bHandleSeekAlert != NULL)
            {
                // yes, so call it to see if we should set the seek alert event
                bSetDecoderSeekAlertEvent =
                    psObj->psInfo->psInterface->bHandleSeekAlert(
                        hSeekService,
                        hAlert
                            );
            }

            // should we set the event?
            if (bSetDecoderSeekAlertEvent == TRUE)
            {
                // yes
                DECODER_OBJECT hDecoder;

                puts(SEEK_OBJECT_NAME": Setting event.");
                hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);
                DECODER_vSeekAlertEvent(hDecoder);
            }
            else
            {
                // no
                puts(SEEK_OBJECT_NAME": Not setting event.");

                // tell cal to ack so that this won't show up in the active
                // event list
                CAL_eAcknowledge(hCAL);
            }
        }
    }

    // If the FINAL flag is set, destroy any objects we supplied
    // during registration
    if((un32Flags & ACTIVE_EVENT_FLAG_FINAL) == ACTIVE_EVENT_FLAG_FINAL)
    {
        // we need to destroy the information contained in the cal content obj
        // that is for the seek service
        SEEK_CONTENT_vDestroySeekContent((SEEK_CONTENT_OBJECT)hContent);
        CAL_eAcknowledge(hCAL);
    }

    return;
}

/*****************************************************************************
*
*   bPrintContent
*
*    This is a CAL iterator function. It is used to print out to a file human
*    readable information about the items registered to the seek service
*
*****************************************************************************/
static BOOLEAN bPrintContent (
    CAL_CONTENT_OBJECT hContent,
    void *pvContentIteratorArg
        )
{
    SEEK_ITERATOR_STRUCT *psIterator =
        (SEEK_ITERATOR_STRUCT*)pvContentIteratorArg;

    // Check inputs
    if(pvContentIteratorArg == NULL)
    {
        // error. stop iterating
        return FALSE;
    }

    // increment count number
    psIterator->un32Count++;

    // print out the count number
    if (psIterator->eOutputOption == SMS_OUTPUT_OPTION_TERSE)
    {
        psIterator->n32NumWritten += fprintf(psIterator->psFile,
                                         "#%u: ",
                                         psIterator->un32Count);
    }
    else
    {
        psIterator->n32NumWritten += fprintf(psIterator->psFile,
                                     "Content: %u\n",
                                     psIterator->un32Count);
    }

    // print out information for this seek content obj
    psIterator->n32NumWritten +=
        SEEK_CONTENT.n32FPrintf(
            (SEEK_CONTENT_OBJECT)hContent,
            psIterator->psFile,
            psIterator->eOutputOption
                );

    // Keep looking
    return TRUE;
}

/*****************************************************************************
*
*    pacSeekTypeText
*
*    This is a local function which simply maps an enumerated type to
*    a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacSeekServiceText( SEEK_SERVICE_ENUM eType )
{
    const char *pacReturnString;

    // based on the seek type, get the text that describes it
    switch (eType)
    {
        case SEEK_SERVICE_ARTIST_TITLE:
            pacReturnString = MACRO_TO_STRING(SEEK_SERVICE_ARTIST_TITLE);
        break;

        case SEEK_SERVICE_TRAFFIC_WEATHER:
            pacReturnString = MACRO_TO_STRING(SEEK_SERVICE_TRAFFIC_WEATHER);
        break;

        case SEEK_SERVICE_SPORTS:
            pacReturnString = MACRO_TO_STRING(SEEK_SERVICE_SPORTS);
        break;

        case SEEK_SERVICE_INVALID:
        default:
            pacReturnString = MACRO_TO_STRING(SEEK_SERVICE_INVALID);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*    pacSeekStateText
*
*    This is a local function which simply maps an enumerated type to
*    a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacSeekStateText( SEEK_STATE_ENUM eState )
{
    const char *pacReturnString;

    // based on the seek state, get the text that describes it
    switch (eState)
    {
        case SEEK_STATE_ENABLED:
            pacReturnString = MACRO_TO_STRING(SEEK_STATE_ENABLED);
        break;

        case SEEK_STATE_DISABLED:
            pacReturnString = MACRO_TO_STRING(SEEK_STATE_DISABLED);
        break;

        case SEEK_STATE_INVALID:
        default:
            pacReturnString = MACRO_TO_STRING(SEEK_STATE_INVALID);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*    bRestoreServiceFromConfigFile
*
*****************************************************************************/
static BOOLEAN bRestoreServiceFromConfigFile(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        )
{
    TAG_OBJECT hParentTag, hTag;
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT;
    STRING_OBJECT *phString;
    SMSAPI_RETURN_CODE_ENUM eResult;
    size_t tSize;

    tSize = sizeof(STRING_OBJECT);
    hParentTag = DECODER_hGetTag(hDecoder);

    do
    {
        eResult = TAG_eGet(
            SEEK_OBJECT_NAME,
            hParentTag,
            &hTag,
            pacSeekServiceText(psObj->psInfo->eService),
            TRUE
                );
        if (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
        {
            // must mean there is no file system
            return TRUE;
        }
        else if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SEEK_OBJECT_NAME": Could not get cm tag.");
            break;
        }

        // save this tag
        psObj->hTag = hTag;

        eResult = TAG_eGet(
            SEEK_STATE,
            psObj->hTag,
            &hTag,
            NULL,
            TRUE
                );
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SEEK_OBJECT_NAME": Could not get cm tag.");
            break;
        }

        phString = &hValueString;
        TAG_eGetTagValue(
            hTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize
                );

        if (hValueString != STRING_INVALID_OBJECT)
        {
            if (STRING.n16Compare(
                    psObj->hEnabledString,
                    hValueString,
                    TRUE) == 0)
            {
                psObj->eState = SEEK_STATE_ENABLED;
            }
            else if (STRING.n16Compare(
                         psObj->hDisabledString,
                         hValueString,
                         TRUE) == 0)
            {
                psObj->eState = SEEK_STATE_DISABLED;
            }
            else
            {
                psObj->eState = SEEK_STATE_INVALID;
                eResult = SMSAPI_RETURN_CODE_ERROR;
            }

            // done with this string
            STRING.vDestroy(hValueString);
        }
        else
        {
            // initially configured to be enabled
            psObj->eState = SEEK_STATE_ENABLED;

            hValueString = psObj->hEnabledString;

            eResult = TAG_eSetTagValue(
                hTag,
                TAG_TYPE_STRING,
                &hValueString,
                sizeof(STRING_OBJECT),
                FALSE
                    );
        }

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SEEK_OBJECT_NAME": Couldn't restore from config file.");
            break;
        }

        return TRUE;

    } while (0);

    return FALSE;
}

/*****************************************************************************
*
*    bUpdateStateTag
*
*****************************************************************************/
static BOOLEAN bUpdateStateTag(
    SEEK_SERVICE_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hTag;
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT;

    eResult = TAG_eGet(
        SEEK_STATE,
        psObj->hTag,
        &hTag,
        NULL,
        FALSE
            );

    if (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
    {
        // must mean there is no file system
        return TRUE;
    }
    else if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SEEK_OBJECT_NAME": Could not get cm tag.");
        return FALSE;
    }

    if (psObj->eState == SEEK_STATE_ENABLED)
    {
        hValueString = psObj->hEnabledString;
    }
    else if (psObj->eState == SEEK_STATE_DISABLED)
    {
        hValueString = psObj->hDisabledString;
    }

    if (hValueString != STRING_INVALID_OBJECT)
    {
        TAG_eSetTagValue(
            hTag,
            TAG_TYPE_STRING,
            (void *)&hValueString,
            sizeof(STRING_OBJECT),
            TRUE
                );

        return TRUE;
    }

    return FALSE;

}

/*****************************************************************************
*
*    bRemoveContentTagIterator
*
*****************************************************************************/
static BOOLEAN bRemoveContentTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    STRING_OBJECT hSoughtTagName, hThisTagName;
    hSoughtTagName = (STRING_OBJECT)pvArg;
    hThisTagName = TAG_hTagName(hTag);

    if (STRING.n16Compare(hSoughtTagName, hThisTagName, TRUE)== 0)
    {
        // remove the tag, but don't commit now.
        // we'll commit after iteration is done
        TAG_eRemove(hTag, FALSE);
    }

    // keep iterating
    return TRUE;
}
