/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*             Licensed Materials - Property of Sirius XM Radio, Inc.         */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SAFECAM_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"
#include "sql_interface_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "location_obj.h"
#include "dsrl_obj.h"
#include "db_util.h"
#include "dsrl_entry_obj.h"
#include "dsrl_target_obj.h"

#include "safecam_mgr_obj.h"
#include "_safecam_mgr_obj.h"
#include "safecam_db_constants.h"
#include "safecam_interface.h"
#include "safecam_location_obj.h"
#include "safecam_sublocation_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static SAFETY_CAMERAS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    SAFECAM_MGR_OBJECT_STRUCT *psObj =
        (SAFECAM_MGR_OBJECT_STRUCT*)SAFETY_CAMERAS_SERVICE_INVALID_OBJECT;
    BOOLEAN bOk = FALSE;
    DATASERVICE_CREATE_STRUCT sCreate;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    OSAL.bMemSet(&sOptionValues, 0, sizeof(sOptionValues));

    do
    {
        bOk = DATASERVICE_IMPL_bProcessOptions(
            SAFECAM_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
        if (bOk == FALSE)
        {
            // Bad options!
            break;
        }

        // Populate our data service creation structure
        DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
        sCreate.pacSRHDriverName = pacSRHDriverName;
        sCreate.pacServiceObjectName = SAFECAM_MGR_OBJECT_NAME;
        sCreate.tServiceObjectSize = sizeof(SAFECAM_MGR_OBJECT_STRUCT);
        sCreate.tDataID = (DATASERVICE_ID)GsSafeCamIntf.tDSI;

        // Get the interface to tell us how much memory
        // this service needs at a minimum for the given
        // startup options
        sCreate.tSuggestedOTABufferByteSize =
            GsSafeCamIntf.tMinimumOTABufferByteSize(sOptionValues.bUpdateRefDB);

        // Configure the data service's static event attributes
        sCreate.vEventCallback = vEventHandler;
        sCreate.tEventRequestMask = (
            DATASERVICE_EVENT_STATE |
            DATASERVICE_EVENT_NEW_DATA |
            DATASERVICE_INTERNAL_EVENT_DSRL );

        // Ask the data service manager controller to
        // create our manager object and do everything
        // necessary to create the underlying objects required
        // in order to support this service
        psObj = (SAFECAM_MGR_OBJECT_STRUCT*)
            DATASERVICE_IMPL_hCreateNewService(&sCreate);
        if (psObj == NULL)
        {
            // Can't create the service, fail out!

            // Free options memory
            DATASERVICE_IMPL_vFreeOptions(&sOptionValues);
            break;
        }

        // Update the path and the flag for the reference db
        psObj->pacRefDatabaseDirPath = sOptionValues.pcRefDBPath;
        psObj->bRefDBUpdatesEnabled = sOptionValues.bUpdateRefDB;

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_ALL,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvAppEventCallbackArg
                );

        // Initialize facing objects
        bOk = bInitAppFacingObject(psObj);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SAFECAM_MGR_OBJECT_NAME
                ": failed to initialize facing objects"
                    );
            break;
        }

        // The service may now start
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, SAFECAM_MGR_OBJECT_NAME
                ": unable to start service"
                    );
            break;
        }

        return (SAFETY_CAMERAS_SERVICE_OBJECT)psObj;
    }
    while (FALSE);

    // Error!
    if (psObj != NULL)
    {
        vUninitObject(psObj, TRUE);
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);
    }

    return SAFETY_CAMERAS_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eIterateAlertTypes
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateAlertTypes (
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService,
    SAFETY_CAMERAS_ALERT_TYPES_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj = NULL;
    BOOLEAN bValid;

    // Validate service object
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hSafeCamService);
    if ((FALSE == bValid) ||
        (NULL == bCallback))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        psAppObj = psGetAppFacingObject(hSafeCamService);
        if (psAppObj == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListIterate(psAppObj->hAlertTypesList,
            (OSAL_LL_ITERATOR_HANDLER)bCallback, pvCallbackArg
                );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to iterate alert types list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        psAppObj->bAppAlertTypesUpdateNotify = FALSE;

        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    vReleaseAppFacingObject(psAppObj);

    return eResult;
}

/*****************************************************************************
*
*   vTooOldUpdate
*
*****************************************************************************/
static void vTooOldUpdate (
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService
        )
{
    if (hSafeCamService != SAFETY_CAMERAS_SERVICE_INVALID_OBJECT)
    {
        BOOLEAN bLocked = FALSE;

        bLocked = DATASERVICE_IMPL_bLock((DATASERVICE_IMPL_HDL)hSafeCamService);

        if (bLocked == TRUE)
        {
            vSetError((SAFECAM_MGR_OBJECT_STRUCT*)hSafeCamService,
                DATASERVICE_ERROR_CODE_DATABASE_TOO_OLD
                    );

            DATASERVICE_IMPL_vUnlock((DATASERVICE_IMPL_HDL)hSafeCamService);
        }
    }

    return;
}

/*****************************************************************************
*
*   eGetReferenceDataVersion
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eReturnCode = DATASERVICE_ERROR_CODE_GENERAL;
    char *pacDatabaseFilePathA = NULL,
        *pacDatabaseFilePathB = NULL;

    if (ptCurrentRefDataVer == NULL)
    {
        return DATASERVICE_ERROR_CODE_GENERAL;
    }

    do
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            SAFECAM_DATABASE_FOLDER,
            SAFECAM_REF_DATABASE_FILENAMEA,
            &pacDatabaseFilePathA
                );
        if (bSuccess != TRUE)
        {
            break;
        }

        bSuccess = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            SAFECAM_DATABASE_FOLDER,
            SAFECAM_REF_DATABASE_FILENAMEB,
            &pacDatabaseFilePathB
                );

        if (bSuccess != TRUE)
        {
            break;
        }

        // Check ref databases
        eReturnCode = DB_UTIL_eCheckReferenceBanks(
            &pacDatabaseFilePathA[0],
            &pacDatabaseFilePathB[0],
            n32ExtractDataVersion, NULL,
            GsSafeCamIntf.tMaxVersionBitlen,
            ptCurrentRefDataVer, ptNextRefDataVer );

    } while (FALSE);

    // Remove database path resources
    if (pacDatabaseFilePathA != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathA);
    }

    if (pacDatabaseFilePathB != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathB);
    }

    return eReturnCode;
}

/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n16CompareAlertTypeById
*
*****************************************************************************/
static N16 n16CompareAlertTypeById (
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType1,
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType2
        )
{
    N16 n16Result = N16_MIN;

    if ((psAlertType1 != (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)NULL) &&
        (psAlertType2 != (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)NULL))
    {
        n16Result = COMPARE(psAlertType1->tID, psAlertType2->tID);
    }

    return n16Result;
}


/*****************************************************************************
*
*   bDBFillAlertTypeRowData
*
*****************************************************************************/
static BOOLEAN bDBFillAlertTypeRowData(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_RESULT_STRUCT *psResult
        )
{
    SAFECAM_ALERT_TYPES_FIELDS_ENUM eCurrentField = 
                                        SAFECAM_ALERT_TYPES_FIELD_TYPE;

    psResult->bResult = FALSE;

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns != SAFECAM_ALERT_TYPES_MAX_FIELDS)
    {
        return psResult->bResult;
    }

    do
    {
        switch (eCurrentField)
        {
            case SAFECAM_ALERT_TYPES_FIELD_TYPE:
            {
                psResult->uDbRow.psAlertTypes->un8Type = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_ALERT_TYPES_FIELD_DESC:
            {
                size_t tLength = 0;

                tLength =
                    strlen((const char*)
                    psColumn[eCurrentField].uData.sCString.pcData
                        );

                psResult->uDbRow.psAlertTypes->hDesc = STRING.hCreate(
                    (const char*)psColumn[eCurrentField].uData.sCString.pcData,
                    tLength );
            }
            break;

            default: // Shouldn't happen
            break;
        }

    } while (++eCurrentField < SAFECAM_ALERT_TYPES_MAX_FIELDS);

    psResult->bResult = TRUE;

    return psResult->bResult;
}


/*****************************************************************************
*
*   vAlertTypeDescRelease
*
*****************************************************************************/
static void vAlertTypeRelease (
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType
        )
{
    if (psAlertType != (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)NULL)
    {
        if (psAlertType->hDescription != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psAlertType->hDescription);
            psAlertType->hDescription = STRING_INVALID_OBJECT;
        }

        SMSO_vDestroy((SMS_OBJECT)psAlertType);
    }

    return;
}


/*****************************************************************************
*
*   psCreateAlertType
*
*****************************************************************************/
static SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psCreateAlertType(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertTypeRow
        )
{
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psResult;

    psResult = (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)SMSO_hCreate(
        SAFECAM_MGR_OBJECT_NAME":AlertTypeData", sizeof(*psResult),
        (SMS_OBJECT)psAppObj, FALSE
            );
    if (psResult == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Failed to allocare alert type data mem"
                );
    }
    else
    {
        psResult->tID = psAlertTypeRow->un8Type;
        psResult->hDescription = psAlertTypeRow->hDesc;
        psAlertTypeRow->hDesc = STRING_INVALID_OBJECT;
    }

    return psResult;
}


/*****************************************************************************
*
*   psGetAlertTypeByID
*
*****************************************************************************/
static SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psGetAlertTypeByID (
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    SAFETY_CAMERAS_ALERT_ID tId
        )
{
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psResult =
        (SAFETY_CAMERAS_ALERT_TYPE_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT sDummyAlertType;

    sDummyAlertType.tID = tId;

    eReturnCode = OSAL.eLinkedListSearch(psAppObj->hAlertTypesList,
        &hEntry, &sDummyAlertType
            );
    if ((eReturnCode == OSAL_SUCCESS) && (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
    {
        psResult = (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)
            OSAL.pvLinkedListThis(hEntry);
    }
    else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": failed search for alert type object "
            "(%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }

    return psResult;
}


/*****************************************************************************
*
*   bDBCreateNewAlertTypes
*
*****************************************************************************/
static BOOLEAN bDBCreateNewAlertTypes (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_QUERY_OBJECTS_STRUCT *psResult
        )
{
    SAFECAM_ALERT_TYPES_ROW_STRUCT sAlertTypeRow;
    BOOLEAN bContinue = TRUE;
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType =
        (SAFETY_CAMERAS_ALERT_TYPE_STRUCT *)NULL;

    OSAL.bMemSet(&sAlertTypeRow, 0, sizeof(sAlertTypeRow));

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;


        // Verify input
        if (psResult == NULL)
        {
            bContinue = FALSE;
            break;
        }

        psResult->bResult = FALSE;

        if (n32NumberOfColumns != SAFECAM_ALERT_TYPES_MAX_FIELDS)
        {
            break;
        }

        // Get the Alert type id
        sAlertTypeRow.un8Type = (UN8)
            psColumn[SAFECAM_ALERT_TYPES_FIELD_TYPE].uData.sUN32.un32Data;

        // See if we already have this location available and its not updated
        psAlertType = psGetAlertTypeByID(psResult->psAppObj,
            sAlertTypeRow.un8Type
                );

        // if alert type object doesn't exist, lets read data and create new
        if (psAlertType == (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)NULL)
        {
            SAFECAM_DB_RESULT_STRUCT sResult;
            BOOLEAN bSuccess = FALSE;

            OSAL.bMemSet(&sResult, 0, sizeof(sResult));
            sResult.uDbRow.psAlertTypes = &sAlertTypeRow;

            bSuccess = bDBFillAlertTypeRowData(psColumn, n32NumberOfColumns,
                &sResult
                    );
            if ((bSuccess == FALSE) || (sResult.bResult == FALSE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to fill alert type %x",
                    sAlertTypeRow.un8Type
                        );
                break;
            }

            psAlertType =
                psCreateAlertType(psResult->psAppObj, &sAlertTypeRow);
            if (psAlertType == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to create alert type"
                        );
                break;
            }
            eReturnCode = OSAL.eLinkedListAdd(psResult->psAppObj->hAlertTypesList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psAlertType
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to add alert type struct to list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        // Set struct to null since it was stored in linked list
        psAlertType = (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)NULL;

        psResult->bResult = TRUE;
    } while (FALSE);

    if (psAlertType != (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)NULL)
    {
        vAlertTypeRelease(psAlertType);
    }

    if (sAlertTypeRow.hDesc != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sAlertTypeRow.hDesc);
        sAlertTypeRow.hDesc = STRING_INVALID_OBJECT;
    }

    // FALSE will stop iteration; TRUE will keep going
    return bContinue;
}


/*****************************************************************************
*
*   bCreateNewAlertTypes
*
*****************************************************************************/
static BOOLEAN bCreateNewAlertTypes(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    BOOLEAN bSkipNew
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_DB_QUERY_OBJECTS_STRUCT sResult;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));
    sResult.bSkipNew = bSkipNew;
    sResult.psAppObj = psAppObj;
    bResult = SQL_INTERFACE.bQuery(psAppObj->hSQLRefConnection,
        SAFECAM_SELECT_ALL_ALERT_TYPES, (SQL_QUERY_RESULT_HANDLER)bDBCreateNewAlertTypes, &sResult
            );
    if (sResult.bResult == FALSE)
    {
        bResult = FALSE;
    }

    return bResult;
}


/*****************************************************************************
*
*   vDSRLRelease
*
*****************************************************************************/
static void vDSRLRelease (
    DSRL_OBJECT hDSRL
        )
{
    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        psDSRLDesc = (SAFECAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc != NULL)
        {
            if (psDSRLDesc->hTargetList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hTargetList,
                    (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eReturnCode = OSAL.eLinkedListDelete(
                    psDSRLDesc->hTargetList
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
                psDSRLDesc->hTargetList = OSAL_INVALID_OBJECT_HDL;
            }

            if (psDSRLDesc->hEntryList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psDSRLDesc->hEntryList, NULL
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }

                eReturnCode = OSAL.eLinkedListDelete(psDSRLDesc->hEntryList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
                psDSRLDesc->hEntryList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        DSRL_vDestroy(hDSRL);
    }

    return;
}


/*****************************************************************************
*
*   vEntryRelease
*
*****************************************************************************/
static void vEntryRelease(
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        )
{
    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        SAFECAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        if (hSafeCamLoc == SAFETY_CAMERAS_LOCATION_INVALID_OBJECT)
        {
            //nothing to do
            break;
        }

        psEntryDesc = (SAFECAM_DSRL_ENTRY_DESC_STRUCT *)
            DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hSafeCamLoc);
        if (psEntryDesc != NULL)
        {
            if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListRemoveAll(
                    psEntryDesc->hDSRLList, NULL
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to cleanup"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                }

                eReturnCode = OSAL.eLinkedListDelete(psEntryDesc->hDSRLList);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                }
                psEntryDesc->hDSRLList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        SAFECAM_LOCATION_vDestroy(hSafeCamLoc);
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vUninitAppFacingObject
*
*****************************************************************************/
static void vUninitAppFacingObject(
    SAFECAM_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Lock the app object
    bLocked = SMSO_bLock((SMS_OBJECT)psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Unable to lock app object"
                );
        return;
    }

    // Disconnect from the database application
    if (psAppObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDestroyPreparedStatement(
            psAppObj->hSQLRefConnection,
            psAppObj->hSelectSubLocationsByUIDStmt);

        psAppObj->hSelectSubLocationsByUIDStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDestroyPreparedStatement(
            psAppObj->hSQLRefConnection,
            psAppObj->hSelectAlertLocationsByLocStmt);

        psAppObj->hSelectAlertLocationsByLocStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDisconnect(psAppObj->hSQLRefConnection);
        psAppObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    printf(SAFECAM_MGR_OBJECT_NAME": Destroying DSRL nodes.\n");

    if (psAppObj->hDSRLList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psAppObj->hDSRLList,
            (OSAL_LL_RELEASE_HANDLER)vDSRLRelease
                );
        OSAL.eLinkedListDelete(psAppObj->hDSRLList);
    }

    printf(SAFECAM_MGR_OBJECT_NAME": Destroying Entry nodes.\n");

    if (psAppObj->hDSRLEntryPool != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psAppObj->hDSRLEntryPool,
            (OSAL_LL_RELEASE_HANDLER)vEntryRelease
                );
        OSAL.eLinkedListDelete(psAppObj->hDSRLEntryPool);
    }

    printf(SAFECAM_MGR_OBJECT_NAME" Entry nodes destroyed.\n");

    if (psAppObj->hAlertTypesList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(psAppObj->hAlertTypesList,
            (OSAL_LL_RELEASE_HANDLER)vAlertTypeRelease
                );
        OSAL.eLinkedListDelete(psAppObj->hAlertTypesList);
    }

    if (psAppObj->hDummySafeCamLoc != SAFETY_CAMERAS_LOCATION_INVALID_OBJECT)
    {
        SAFECAM_LOCATION_vDestroy(psAppObj->hDummySafeCamLoc);
    }

    if (psAppObj->hRemovedEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(psAppObj->hRemovedEntriesList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to clean removed entries"
                " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
        }

        eReturnCode =
            OSAL.eLinkedListDelete(psAppObj->hRemovedEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to delete removed"
                " entries list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    if (psAppObj->hNewDSRLEntriesList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hNewDSRLEntriesList,
            NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot cleanup linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hNewDSRLEntriesList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot delete linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        psAppObj->hNewDSRLEntriesList = OSAL_INVALID_OBJECT_HDL;
    }


    // Destroy the app object
    DATASERVICE_IMPL_vDestroyDSRLParent((DATASERVICE_IMPL_HDL)psObj, (SMS_OBJECT)psAppObj);
    psObj->psAppObj = NULL;

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    printf(SAFECAM_MGR_OBJECT_NAME": %s start\n", __FUNCTION__);

    if (psObj->psAppObj != NULL)
    {
        // Releasing App interface object
        vUninitAppFacingObject(psObj);
    }

    // Releasing plugin object
    if (psObj->hInterface != SAFECAM_INTERFACE_INVALID_OBJECT)
    {
        GsSafeCamIntf.vUnInit(psObj->hInterface);
        psObj->hInterface = SAFECAM_INTERFACE_INVALID_OBJECT;
    }

    if (psObj->pacCurRefDatabaseFilePath != NULL)
    {
        // Remove database path resources
        SMSO_vDestroy((SMS_OBJECT)psObj->pacCurRefDatabaseFilePath);
        psObj->pacCurRefDatabaseFilePath = NULL;
    }

    if (psObj->pacNextRefDatabaseFilePath != NULL)
    {
        // Remove database path resources
        SMSO_vDestroy((SMS_OBJECT)psObj->pacNextRefDatabaseFilePath);
        psObj->pacNextRefDatabaseFilePath = NULL;
    }

    if (bFullDelete == TRUE)
    {
        // Destroy the SMS Update Event object
        SMSU_vDestroy(&psObj->sEvent);
    }

    printf(SAFECAM_MGR_OBJECT_NAME": %s end\n", __FUNCTION__);

    return;
}

/*****************************************************************************
*
*   bInitAppFacingObject
*
*****************************************************************************/
static BOOLEAN bInitAppFacingObject(
    SAFECAM_MGR_OBJECT_STRUCT *psObj
        )
{

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;

        // Begin DSRL config
        DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

        // Set service type
        sDSRLConfig.eServiceType = DATASERVICE_TYPE_SAFETY_CAMERAS;

        // Set the parent object size
        sDSRLConfig.tParentObjectSize = sizeof(SAFECAM_APP_OBJECT_STRUCT);

        // Set the DSRL descriptor size
        sDSRLConfig.tServiceDataSize = sizeof(SAFECAM_DSRL_DESC_STRUCT);

        // Configure the device feature
        sDSRLConfig.bDeviceEnabled = TRUE;
        sDSRLConfig.un32DeviceNotifyDistance = SAFECAM_DEVICE_DISTANCE_THRESHOLD;
        sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

        // Create it
        psObj->psAppObj = (SAFECAM_APP_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hConfigureDSRL(
                (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);
        if (psObj->psAppObj == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hDSRLList),
                SAFECAM_MGR_OBJECT_NAME":DSRLList",
                n16CompareHandles,
                OSAL_LL_OPTION_UNIQUE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Cannot create DSRLList object."
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &(psObj->psAppObj->hDSRLEntryPool),
                SAFECAM_MGR_OBJECT_NAME":DSRLEntriesList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareEntriesByUID,
                OSAL_LL_OPTION_NONE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Cannot create DSRLEntriesList object."
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &(psObj->psAppObj->hAlertTypesList),
            SAFECAM_MGR_OBJECT_NAME":AlertTypesList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareAlertTypeById,
            OSAL_LL_OPTION_LINEAR|OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": Cannot create AlertTypesList object."
                    );
                break;
        }

        psObj->psAppObj->hDummySafeCamLoc = 
            SAFECAM_LOCATION_hCreateDummy((SMS_OBJECT)psObj->psAppObj);
        if (psObj->psAppObj->hDummySafeCamLoc == 
            SAFETY_CAMERAS_LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to create dummy location");
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &(psObj->psAppObj->hRemovedEntriesList),
            SAFECAM_MGR_OBJECT_NAME":RemovedEntriesList",
            n16CompareHandles,
            OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &(psObj->psAppObj->hNewDSRLEntriesList),
            SAFECAM_MGR_OBJECT_NAME":hNewDSRLEntriesList",
            n16CompareHandles,
            OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bResult = TRUE;

    } while (FALSE);

    if (NULL != psObj->psAppObj)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)psObj->psAppObj);
    }

    return bResult;
}

/*****************************************************************************
*
*   psGetAppFacingObject
*
*****************************************************************************/
static SAFECAM_APP_OBJECT_STRUCT *psGetAppFacingObject(
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService
        )
{
    SAFECAM_MGR_OBJECT_STRUCT *psObj =
        (SAFECAM_MGR_OBJECT_STRUCT *)hSafeCamService;
    BOOLEAN bValid, bLocked;

    do
    {
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hSafeCamService);

        if (bValid == FALSE)
        {
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}


/*****************************************************************************
*
*   vDestroyUnusedEntries
*
*****************************************************************************/
static void vDestroyUnusedEntries(
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    do
    {
        SAFECAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        UN32 un32Items = 0;

        if (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            // list corruption
            break;
        }

        psEntryDesc = (SAFECAM_DSRL_ENTRY_DESC_STRUCT *)
            DSRL_ENTRY_pvServiceData(hDSRLEntry);
        if (psEntryDesc != NULL)
        {
            // in some cases DSRL list could even be not created yet
            // in this case we suggest that no assigned DSRLs exist
            if (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL)
            {
                eReturnCode = OSAL.eLinkedListItems(psEntryDesc->hDSRLList,
                    &un32Items
                        );
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to get number of DSRL"
                        " desc items in list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                    break;
                }
            }
            else
            {
                un32Items = 0;
            }

            if (un32Items == 0)
            {
                eReturnCode = OSAL.eLinkedListRemove(psEntryDesc->hPoolEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to remove DSRL"
                        " entry from pool (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
                vEntryRelease((SAFETY_CAMERAS_LOCATION_OBJECT)hDSRLEntry);
            }
        }
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bRemoveUnusedSafecamLocs
*
*****************************************************************************/
static BOOLEAN bRemoveUnusedSafecamLocs(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hRemovedEntriesList,
        (OSAL_LL_RELEASE_HANDLER)vDestroyUnusedEntries
            );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": failed to remove unused DSRL"
            " entries (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   bDBProcessSelectDBVersion
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select version" query made on the database version relation.  It populates
*   a pointer to a SAFECAM_DB_VERSION_ROW_STRUCT with the information found
*   in the database as well as performs version verification.
*
*******************************************************************************/
static BOOLEAN bDBProcessSelectDBVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_RESULT_STRUCT *psResult
        )
{
    // Verify input
    if ((psResult == NULL) || (psResult->uDbRow.psDBVersion == NULL))
    {
        return FALSE;
    }

    psResult->uDbRow.psDBVersion->un32Version =
        psColumn[0].uData.sUN32.un32Data;
    psResult->bResult = TRUE;

    // FALSE will stop iteration; TRUE will keep going
    return psResult->bResult;
}

/*****************************************************************************
*
*   bDBPrepareAlertLocationRow
*
*****************************************************************************/
static BOOLEAN bDBPrepareAlertLocationRow(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    SAFECAM_DB_RESULT_STRUCT *psResult
        )
{
    psResult->bResult = TRUE;

    switch (tIndex)
    {
        case SAFECAM_ALERT_LOCATIONS_FIELD_INFO:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psResult->uDbRow.psAlertLocations->hInfo;
        }
        break;

        case SAFECAM_ALERT_LOCATIONS_FIELD_LATITUDE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData =
                (void *)(size_t)(UN32)psResult->uDbRow.psAlertLocations->n32Lat;
        }
        break;

        case SAFECAM_ALERT_LOCATIONS_FIELD_LONGITUDE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData =
                (void *)(size_t)(UN32)psResult->uDbRow.psAlertLocations->n32Lon;
        }
        break;

        case SAFECAM_ALERT_LOCATIONS_FIELD_NOSUB:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData =
                (void *)(size_t)psResult->uDbRow.psAlertLocations->un8NoSub;
        }
        break;

        case SAFECAM_ALERT_LOCATIONS_FIELD_UID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData =
                (void *)(size_t)psResult->uDbRow.psAlertLocations->un32UID;
        }
        break;

        case SAFECAM_ALERT_LOCATIONS_FIELD_UNIT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData =
                (void *)(size_t)psResult->uDbRow.psAlertLocations->bMilesUnits;
        }
        break;

        default:
        {
            psResult->bResult = FALSE;
        }
        break;
    }

    return psResult->bResult;
}

/*****************************************************************************
*
*   bDBPrepareAlertTypeRow
*
*****************************************************************************/
static BOOLEAN bDBPrepareAlertTypeRow(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    SAFECAM_DB_RESULT_STRUCT *psResult
        )
{
    psResult->bResult = TRUE;

    switch (tIndex)
    {
        case SAFECAM_ALERT_TYPES_FIELD_DESC:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psResult->uDbRow.psAlertTypes->hDesc;
        }
        break;

        case SAFECAM_ALERT_TYPES_FIELD_TYPE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData =
                (void *)(size_t)psResult->uDbRow.psAlertTypes->un8Type;
        }
        break;

        default:
        {
            psResult->bResult = FALSE;
        }
        break;
    }

    return psResult->bResult;
}

/*****************************************************************************
*
*   bDBPrepareSubLocationRow
*
*****************************************************************************/
static BOOLEAN bDBPrepareSubLocationRow(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    SAFECAM_DB_RESULT_STRUCT *psResult
        )
{
    psResult->bResult = TRUE;

    switch (tIndex)
    {
        case SAFECAM_SUB_LOCATIONS_FIELD_DIR:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psResult->uDbRow.psSubLocations->un16Dir;
        }
        break;

        case SAFECAM_SUB_LOCATIONS_FIELD_SPEED1:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psResult->uDbRow.psSubLocations->un16Speed1;
        }
        break;

        case SAFECAM_SUB_LOCATIONS_FIELD_SPEED2:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psResult->uDbRow.psSubLocations->un16Speed2;
        }
        break;

        case SAFECAM_SUB_LOCATIONS_FIELD_SUB_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psResult->uDbRow.psSubLocations->un8SubID;
        }
        break;

        case SAFECAM_SUB_LOCATIONS_FIELD_UID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psResult->uDbRow.psSubLocations->un32UID;
        }
        break;

        case SAFECAM_SUB_LOCATIONS_FIELD_TYPE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psResult->uDbRow.psSubLocations->un8Type;
        }
        break;

        case SAFECAM_SUB_LOCATIONS_FIELD_LATITUDE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psResult->uDbRow.psSubLocations->n32Lat;
        }
        break;

        case SAFECAM_SUB_LOCATIONS_FIELD_LONGITUDE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psResult->uDbRow.psSubLocations->n32Lon;
        }
        break;

        default:
        {
            psResult->bResult = FALSE;
        }
        break;
    }

    return psResult->bResult;
}

/*****************************************************************************
*
*   bDBFillAlertLocationData
*
*****************************************************************************/
static BOOLEAN bDBFillAlertLocationData(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_RESULT_STRUCT *psResult
        )
{
    SAFECAM_ALERT_LOCATIONS_FIELDS_ENUM eCurrentField =
        (SAFECAM_ALERT_LOCATIONS_FIELDS_ENUM)0;

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns != SAFECAM_ALERT_LOCATIONS_MAX_FIELDS)
    {
        psResult->bResult = FALSE;
        return FALSE;
    }

    do
    {
        switch (eCurrentField)
        {
            case SAFECAM_ALERT_LOCATIONS_FIELD_UID:
            {
                psResult->uDbRow.psAlertLocations->un32UID =
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_ALERT_LOCATIONS_FIELD_LATITUDE:
            {
                psResult->uDbRow.psAlertLocations->n32Lat = (N32)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_ALERT_LOCATIONS_FIELD_LONGITUDE:
            {
                psResult->uDbRow.psAlertLocations->n32Lon = (N32)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_ALERT_LOCATIONS_FIELD_INFO:
            {
                size_t tLength = 0;

                tLength =
                    strlen((const char*)
                        psColumn[eCurrentField].uData.sCString.pcData
                            );

                psResult->uDbRow.psAlertLocations->hInfo = STRING.hCreate(
                    (const char*)psColumn[eCurrentField].uData.sCString.pcData,
                    tLength);
            }
            break;

            case SAFECAM_ALERT_LOCATIONS_FIELD_NOSUB:
            {
                psResult->uDbRow.psAlertLocations->un8NoSub = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_ALERT_LOCATIONS_FIELD_UNIT:
            {
                psResult->uDbRow.psAlertLocations->bMilesUnits = (BOOLEAN)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }

    } while (++eCurrentField < SAFECAM_ALERT_LOCATIONS_MAX_FIELDS);

    psResult->bResult = TRUE;

    return psResult->bResult;
}

/*****************************************************************************
*
*   bDBFillSubLocationData
*
*****************************************************************************/
static BOOLEAN bDBFillSubLocationData(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_RESULT_STRUCT *psResult
        )
{
    SAFECAM_SUB_LOCATIONS_FIELDS_ENUM eCurrentField =
        (SAFECAM_SUB_LOCATIONS_FIELDS_ENUM)0;

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns != SAFECAM_SUB_LOCATIONS_MAX_FIELDS)
    {
        psResult->bResult = FALSE;
        return FALSE;
    }

    do
    {
        switch (eCurrentField)
        {
            case SAFECAM_SUB_LOCATIONS_FIELD_UID:
            {
                psResult->uDbRow.psSubLocations->un32UID =
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_SUB_LOCATIONS_FIELD_LATITUDE:
            {
                psResult->uDbRow.psSubLocations->n32Lat = (N32)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_SUB_LOCATIONS_FIELD_LONGITUDE:
            {
                psResult->uDbRow.psSubLocations->n32Lon = (N32)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_SUB_LOCATIONS_FIELD_SUB_ID:
            {
                psResult->uDbRow.psSubLocations->un8SubID = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_SUB_LOCATIONS_FIELD_TYPE:
            {
                psResult->uDbRow.psSubLocations->un8Type = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_SUB_LOCATIONS_FIELD_DIR:
            {
                psResult->uDbRow.psSubLocations->un16Dir = (UN16)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_SUB_LOCATIONS_FIELD_SPEED1:
            {
                psResult->uDbRow.psSubLocations->un16Speed1 = (UN16)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case SAFECAM_SUB_LOCATIONS_FIELD_SPEED2:
            {
                psResult->uDbRow.psSubLocations->un16Speed2 = (UN16)
                            psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }

    } while (++eCurrentField < SAFECAM_SUB_LOCATIONS_MAX_FIELDS);

    psResult->bResult = TRUE;
    return TRUE;
}

/*****************************************************************************
*
*   bDBDeleteAllSubLocations
*
*****************************************************************************/
static BOOLEAN bDBDeleteAllSubLocations(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;
    int iReturn;

    // Build our update string for this new version using our
    // DB's schema version to find the version row
    iReturn = snprintf(pacBuffer, tBufferSize,
        SAFECAM_DELETE_ALL_SUB_LOCATIONS, psAlertLocation->un32UID
            );

    if (iReturn > 0)
    {
        // update in database
        bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Unable to delete "
                "sub location records with uid %u",
                psAlertLocation->un32UID
                    );
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bDBDeleteAlertLocation
*
*****************************************************************************/
static BOOLEAN bDBDeleteAlertLocation(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;
    int iReturn;

    iReturn = snprintf(pacBuffer, tBufferSize,
        SAFECAM_DELETE_ALERT_LOCATION, psAlertLocation->un32UID
            );

    if (iReturn > 0)
    {
        // update in database
        bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Unable to delete "
                "alert location record with uid %u",
                psAlertLocation->un32UID
                    );
        }
        else
        {
            // Build our string to remove the
            // alert location rtree row from the database
            iReturn = snprintf(pacBuffer, tBufferSize,
                SAFECAM_DELETE_ALERT_LOCATION_RTREE,
                psAlertLocation->un32UID
                    );
            if (iReturn > 0)
            {
                bResult = SQL_INTERFACE.bExecuteCommand(
                    hConnection,
                    pacBuffer
                        );
                if (bResult == FALSE)
                {
                    printf(SAFECAM_MGR_OBJECT_NAME": "
                        "Unable to delete alert location rtree\n"
                            );
                }
            }
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bDBRowCountHandler
*
*****************************************************************************/
static BOOLEAN bDBRowCountHandler(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    BOOLEAN bResult = FALSE;
    UN32* pun32Count = (UN32*)pvArg;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;

    // COUNT should only return 1 column
    if (n32NumberOfColumns == 1)
    {
        psCurrentCol = &psColumns[0];

        // get the count
        *pun32Count = (UN32)psCurrentCol->uData.sUN32.un32Data;

        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bDBSubLocationPointsHandler
*
*****************************************************************************/
static BOOLEAN bDBSubLocationPointsHandler(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    SAFECAM_DB_QUERY_OBJECTS_STRUCT *psResult
       )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_GEO_POINT_FIXED_STRUCT *psGeoPoint = NULL;

    psResult->bResult = TRUE;

    do
    {
        N32 n32FixedValue = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // Request should return 2 columns
        if (n32NumberOfColumns != 2)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Invalid number of columns %u",
                n32NumberOfColumns
                    );
            break;
        }

        psGeoPoint = (SAFECAM_GEO_POINT_FIXED_STRUCT*)
            SMSO_hCreate(SAFECAM_MGR_OBJECT_NAME":GeoPoint",
            sizeof(*psGeoPoint), SMS_INVALID_OBJECT, FALSE
                );
        if (psGeoPoint == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to allocate memory"
                    );
            break;
        }

        n32FixedValue = (N32)(psColumns[0].uData.sUN32.un32Data);
        psGeoPoint->hLat = OSAL_FIXED.hCreateInMemory(n32FixedValue,
            LOCATION_BINPOINT,
            &(psGeoPoint->atFixedData[OSAL_FIXED_OBJECT_SIZE * 0])
                );
        n32FixedValue = (N32)(psColumns[1].uData.sUN32.un32Data);
        psGeoPoint->hLon = OSAL_FIXED.hCreateInMemory(n32FixedValue,
            LOCATION_BINPOINT,
            &(psGeoPoint->atFixedData[OSAL_FIXED_OBJECT_SIZE * 1])
                );

        if ((psGeoPoint->hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psGeoPoint->hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to create fixed object"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListAdd(psResult->hObjectsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psGeoPoint
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to add geo point to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }
        psGeoPoint = NULL;

        bResult = TRUE;
    } while (FALSE);

    if (psGeoPoint != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psGeoPoint);
    }

    if (bResult == FALSE)
    {
        psResult->bResult = FALSE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bDBGetSubLocationPoints
*
*****************************************************************************/
static BOOLEAN bDBGetSubLocationPoints(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation,
    SAFECAM_DB_QUERY_OBJECTS_STRUCT *psResult
        )
{
    BOOLEAN bResult = FALSE;
    int iReturn;

    iReturn = snprintf(pacBuffer, tBufferSize,
        SAFECAM_SELECT_SUB_LOCATIONS_COORDS_BY_UID, psAlertLocation->un32UID
            );
    if (iReturn > 0)
    {
        psResult->bResult = TRUE;
        // Find out how many rows (tables) I will be receiving
        bResult = SQL_INTERFACE.bQuery(hConnection, pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bDBSubLocationPointsHandler,
            (void*)psResult
                );
        if (psResult->bResult != TRUE)
        {
            bResult = FALSE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   bDBGetAlertLocationData
*
*****************************************************************************/
static BOOLEAN bDBGetAlertLocationData(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME" %s start\n", __FUNCTION__);

    do
    {
        BOOLEAN bSuccess = FALSE;
        int iReturn = 0;
        SAFECAM_DB_RESULT_STRUCT sResult;

        OSAL.bMemSet(psAlertLocation, 0, sizeof(*psAlertLocation));
        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.uDbRow.psAlertLocations = psAlertLocation;

        // Build string for alert location request
        iReturn = snprintf(
            pacBuffer,
            tBufferSize,
            SAFECAM_SELECT_ALERT_LOCATION_BY_UID,
            psAlertLocation->un32UID
                );

        printf(SAFECAM_MGR_OBJECT_NAME" SQL "
            "request is %s, size is %d\n", pacBuffer, iReturn
                );

        if (iReturn <= 0)
        {
            break;
        }

        // look in database
        bSuccess = SQL_INTERFACE.bQuery(hConnection, pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bDBFillAlertLocationData,
            (void*)&sResult
                );
        if ((sResult.bResult == FALSE) || (bSuccess == FALSE))
        {
            if (psAlertLocation->hInfo != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(psAlertLocation->hInfo);
                psAlertLocation->hInfo = STRING_INVALID_OBJECT;
            }

            printf(SAFECAM_MGR_OBJECT_NAME": Failed"
                " to get alert location\n"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    printf(SAFECAM_MGR_OBJECT_NAME" %s end, result is %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   bDBGetSubLocationData
*
*****************************************************************************/
static BOOLEAN bDBGetSubLocationData (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME" %s start\n", __FUNCTION__);

    do
    {
        BOOLEAN bSuccess = FALSE;
        int iReturn = 0;
        SAFECAM_DB_RESULT_STRUCT sResult;

        // Clear the location structure
        OSAL.bMemSet(psSubLocation, 0, sizeof(*psSubLocation));
        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.uDbRow.psSubLocations = psSubLocation;

        // Build string for sub location request
        iReturn = snprintf(
            pacBuffer,
            tBufferSize,
            SAFECAM_SELECT_SUB_LOCATION_BY_UID_SUB_ID,
            psSubLocation->un32UID,
            psSubLocation->un8SubID
                );

        printf(SAFECAM_MGR_OBJECT_NAME" SQL "
            "request is %s, size is %d\n", pacBuffer, iReturn
                );

        if (iReturn <= 0)
        {
            break;
        }

        // look in database
        bSuccess = SQL_INTERFACE.bQuery(hConnection, pacBuffer,
            (SQL_QUERY_RESULT_HANDLER)bDBFillSubLocationData,
            &sResult
                );
        if ((sResult.bResult == FALSE) || (bSuccess == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to get sub location"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    printf(SAFECAM_MGR_OBJECT_NAME" %s end, result is %u\n", __FUNCTION__, bResult);

    return bResult;
}

/*****************************************************************************
*
*   n16SearchEntryByUID
*
*****************************************************************************/
static N16 n16CompareEntriesByUID (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc1,
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc2
        )
{
    N16 n16Result = N16_MIN;

    if ((hSafeCamLoc1 != SAFETY_CAMERAS_LOCATION_INVALID_OBJECT) &&
        (hSafeCamLoc2 != SAFETY_CAMERAS_LOCATION_INVALID_OBJECT))
    {
        UN32 un32UID1, un32UID2;

        un32UID1 = SAFECAM_LOCATION_un32UID(hSafeCamLoc1);
        un32UID2 = SAFECAM_LOCATION_un32UID(hSafeCamLoc2);
        n16Result = COMPARE(un32UID1, un32UID2);
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareHandles
*
*****************************************************************************/
static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        )
{
    return COMPARE(pvArg1, pvArg2);
}

/*****************************************************************************
*
*   bDSRLDescInit
*
*****************************************************************************/
static BOOLEAN bDSRLDescInit(
    SAFECAM_APP_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bResult = FALSE;

    do
    {
        if (psDSRLArg->hDSRL == DSRL_INVALID_OBJECT)
        {
            break;
        }

        // get the DSRL desc pointer from DSRL
        psDSRLDesc = 
            (SAFECAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot get DSRL desc from DSRL"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hTargetList,
            SAFECAM_MGR_OBJECT_NAME":DSRLDesc:TgtList",
            NULL,
            OSAL_LL_OPTION_NONE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot create targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hEntryList,
            SAFECAM_MGR_OBJECT_NAME":DSRLDesc:EntryList",
            n16CompareHandles,
            OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot create entry list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Add this to our list of tracked DSRL lists of targets for this service
        eReturnCode =
            OSAL.eLinkedListAdd(psObj->hDSRLList,
                                &(psDSRLDesc->hDSRLListEntry),
                                psDSRLArg->hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot add DSRL to pool (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if ((bResult == FALSE) && (psDSRLDesc != NULL))
    {
        if (psDSRLDesc->hDSRLListEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": cannot remove DSRL from pool (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            psDSRLDesc->hDSRLListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        vDSRLRelease(psDSRLArg->hDSRL);
    }

    return bResult;
}


/*****************************************************************************
*
*   bDSRLRemoveEntryLink
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntryLink(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        )
{
    BOOLEAN bResult = FALSE;

    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        psDSRLDesc = DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc != NULL)
        {
            eReturnCode = OSAL.eLinkedListSearch(psDSRLDesc->hEntryList,
                &hEntry, hDSRLEntry
                    );
            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListRemove(hEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to remove entry"
                        " from list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to find entry"
                    " in list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }

        bResult = TRUE;
    }

    return bResult;
}


/*****************************************************************************
*
*   bEntryRemoveDSRLLink
*
*****************************************************************************/
static BOOLEAN bEntryRemoveDSRLLink(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL,
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    SAFECAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

    psEntryDesc = (SAFECAM_DSRL_ENTRY_DESC_STRUCT*)
        DSRL_ENTRY_pvServiceData(hDSRLEntry);

    if ((psEntryDesc != NULL) &&
        (psEntryDesc->hDSRLList != OSAL_INVALID_OBJECT_HDL))
    {
        eReturnCode = OSAL.eLinkedListSearch(psEntryDesc->hDSRLList,
            &hEntry, hDSRL
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to remove DSRL desc"
                    " from list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }

            // adding entry to list of removed entries
            eReturnCode = OSAL.eLinkedListAdd(
                psAppObj->hRemovedEntriesList, NULL, hDSRLEntry);
            if ((eReturnCode != OSAL_SUCCESS) &&
                (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to add DSRL entry to list"
                    " of removed entries (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                bResult = TRUE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to find DSRL desc"
                " in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return bResult;
}


/*****************************************************************************
*
*   vDSRLFinalizeEntry
*
*****************************************************************************/
static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    if ((hDSRL != DSRL_INVALID_OBJECT) &&
        (hDSRLEntry != DSRL_ENTRY_INVALID_OBJECT) &&
        (psAppObj != NULL))
    {
        BOOLEAN bSuccess = FALSE;

        bSuccess = bEntryRemoveDSRLLink(hDSRLEntry, hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to remove entry"
                " to DSRL link"
                    );
        }

        bSuccess = bDSRLRemoveEntryLink(hDSRL, hDSRLEntry);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to remove DSRL"
                " to entry link"
                    );
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": failed to finalize entry: hDSRL %p,"
            " hDSRLEntry %p, callback arg %p", hDSRL, hDSRLEntry, psAppObj
                );
    }
    return;
}


/*****************************************************************************
*
*   bDSRLAddTarget
*
*****************************************************************************/
static BOOLEAN bDSRLAddTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj,
    SMS_OBJECT hParent
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        DSRL_TARGET_TYPE_ENUM eTargetType = DSRL_TARGET_TYPE_UNKNOWN;
        SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        psDSRLDesc = DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": unknown DSRL 0x%p", hDSRL);
            break;
        }

        // Make sure the data target type is a location
        eTargetType = DSRL_TARGET.eType(hTargetObj);
        printf(SAFECAM_MGR_OBJECT_NAME" eTargetType is %d\n", eTargetType);
        if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
        {
            // Error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": target type is not location (%d)",
                eTargetType
                    );
            break;
        }

        // Add this target to our DSRL description
        eReturnCode = OSAL.eLinkedListAdd(
            psDSRLDesc->hTargetList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hTargetObj
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to add target to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   bDSRLAddTargets
*
*****************************************************************************/
static BOOLEAN bDSRLAddTargets(
    DSRL_ARG_STRUCT *psDSRLArg,
    SMS_OBJECT hParent
        )
{
    size_t tDSRLTargetIndex;
    BOOLEAN bSuccess = TRUE;

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         tDSRLTargetIndex++)
    {
        bSuccess = bDSRLAddTarget(psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex], hParent
                );
        if (bSuccess == FALSE)
        {
            break;
        }
        else
        {
            // if target has been added successfully
            // need to clean it in array to avoid target destroy
            // in general DATASERVICE event processor
            psDSRLArg->ahTargetList[tDSRLTargetIndex] =
                DSRL_TARGET_INVALID_OBJECT;
        }
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bLocationExists
*
*******************************************************************************/
static BOOLEAN bLocationExists(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32UID,
    SAFETY_CAMERAS_LOCATION_OBJECT *phSafeCamLoc
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc =
            SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;
        BOOLEAN bSuccess;

        bSuccess = 
            SAFECAM_LOCATION_bSetUID(psAppObj->hDummySafeCamLoc, un32UID);
        if (bSuccess == FALSE)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListSearch(psAppObj->hDSRLEntryPool,
            &hEntry, psAppObj->hDummySafeCamLoc
                );

        //  if entry was not found, lets break this
        if ((eReturnCode != OSAL_SUCCESS) ||
            (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            break;
        }

        hSafeCamLoc = 
            (SAFETY_CAMERAS_LOCATION_OBJECT)OSAL.pvLinkedListThis(hEntry);
        if (hSafeCamLoc == SAFETY_CAMERAS_LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to get safecam location handle "
                "from entry %p", hEntry
                    );
            break;
        }

        *phSafeCamLoc = hSafeCamLoc;

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*******************************************************************************
*
*   bDBProcessSelectLocationsAndBuildObjects
*
*******************************************************************************/
static BOOLEAN bDBProcessSelectLocationsAndBuildObjects (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT sAlertLocationRow;
    BOOLEAN bContinue = TRUE;
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc =
        SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;

    OSAL.bMemSet(&sAlertLocationRow, 0, sizeof(sAlertLocationRow));

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        BOOLEAN bLocationExist = FALSE;
        SAFECAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        // Verify input
        if (psAppObj == NULL)
        {
            bContinue = FALSE;
            break;
        }

        if (n32NumberOfColumns != SAFECAM_ALERT_LOCATIONS_MAX_FIELDS)
        {
            break;
        }

        // Get the Alert Location UID
        sAlertLocationRow.un32UID =
            psColumn[SAFECAM_ALERT_LOCATIONS_FIELD_UID].uData.sUN32.un32Data;

        // See if we already have this location available and its not updated
        bLocationExist = bLocationExists(psAppObj,
            sAlertLocationRow.un32UID, &hSafeCamLoc
                );

        // if location object doesn't exist, lets read data and create new entry desc
        if (bLocationExist == FALSE)
        {
            SAFECAM_DB_RESULT_STRUCT sResult;
            BOOLEAN bSuccess = FALSE;

            OSAL.bMemSet(&sResult, 0, sizeof(sResult));
            sResult.uDbRow.psAlertLocations = &sAlertLocationRow;

            bSuccess = bDBFillAlertLocationData(psColumn, n32NumberOfColumns,
                &sResult
                    );
            if ((bSuccess == FALSE) || (sResult.bResult == FALSE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to fill alert location %x",
                    sAlertLocationRow.un32UID
                        );
                break;
            }

            hSafeCamLoc = SAFECAM_LOCATION_hCreate((SMS_OBJECT)psAppObj,
                &sAlertLocationRow, sizeof(SAFECAM_DSRL_ENTRY_DESC_STRUCT)
                    );
            if (hSafeCamLoc == SAFETY_CAMERAS_LOCATION_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to create safecam location object"
                        );
                break;
            }

            psEntryDesc = (SAFECAM_DSRL_ENTRY_DESC_STRUCT *)
                DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hSafeCamLoc);
            if (psEntryDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to get DSRL entry descriptor"
                        );
                break;
            }

            eReturnCode = OSAL.eLinkedListAdd(psAppObj->hDSRLEntryPool,
                &psEntryDesc->hPoolEntry, hSafeCamLoc
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to add entry to pool (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }
        else
        {
            psEntryDesc = 
                (SAFECAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
                (DSRL_ENTRY_OBJECT)hSafeCamLoc);

            if (psEntryDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": failed to get DSRL entry descriptor"
                    );
                break;
            }
        }

        psEntryDesc->un32TransactionID = psAppObj->un32TransactionID;

        eReturnCode = OSAL.eLinkedListAdd(psAppObj->hNewDSRLEntriesList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hSafeCamLoc
                );

        // Set object handle to null since it was stored in pull
        hSafeCamLoc = SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;

        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to add entry desc to list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            // even if error happens, entry is already in pool and not linked
            // to any DSRL. It means that this entry will be destroyed at
            // the end of DSRL operation
            break;
        }

    } while (FALSE);

    // if something went wrong, need to destroy entry
    if (hSafeCamLoc != SAFETY_CAMERAS_LOCATION_INVALID_OBJECT)
    {
        vEntryRelease(hSafeCamLoc);
    }

    if (sAlertLocationRow.hInfo != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sAlertLocationRow.hInfo);
        sAlertLocationRow.hInfo = STRING_INVALID_OBJECT;
    }

    // FALSE will stop iteration; TRUE will keep going
    return bContinue;
}

/*****************************************************************************
*
*   bDBProcessSubLocationsAndBuildObjects
*
*****************************************************************************/
static BOOLEAN bDBProcessSubLocationsAndBuildObjects(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_QUERY_SUBLOCATIONS_STRUCT *psResult
        )
{
    BOOLEAN bContinue = TRUE;
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT sSubLocationRow;
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc = SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT;

    OSAL.bMemSet(&sSubLocationRow, 0, sizeof(sSubLocationRow));

    do
    {
        BOOLEAN bSuccess = FALSE;
        SAFECAM_DB_RESULT_STRUCT sResult;
        SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType =
            (SAFETY_CAMERAS_ALERT_TYPE_STRUCT*)NULL;
        SAFETY_CAMERAS_ALERT_ID tID = SAFETY_CAMERAS_ALERT_INVALID_ID;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.uDbRow.psSubLocations = &sSubLocationRow;

        // Verify input
        if (psResult == NULL)
        {
            break;
        }

        psResult->bResult = FALSE;

        // Put sublocation data to struct
        bSuccess = bDBFillSubLocationData(psColumn, n32NumberOfColumns, &sResult);
        if ((bSuccess == FALSE) || (sResult.bResult == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to fill sublocation data"
                    );
            break;
        }

        // Get alert type object handle for sublocation
        psAlertType = psGetAlertTypeByID(psResult->psAppObj,
            sSubLocationRow.un8Type
                );
        if (psAlertType != NULL)
        {
            tID = psAlertType->tID;
        }

        // Create sublocation using sublocation data struct and alert type
        hSafeCamSubLoc = SAFECAM_SUBLOCATION_hCreate((SMS_OBJECT)psResult->psAppObj,
            &sSubLocationRow, psResult->bSpeedInMiles, tID
                );
        if (hSafeCamSubLoc == SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to create sublocation object"
                    );
            psResult->bResult = FALSE;
            bContinue = FALSE;
            break;
        }

        bSuccess = SAFECAM_LOCATION_bSetSubLocation(psResult->hSafeCamLoc,
            hSafeCamSubLoc
                );
        if (bSuccess == FALSE)
        {
            printf(SAFECAM_MGR_OBJECT_NAME": failed to assign sublocation object"
                " %u to location object\n",
                SAFECAM_SUBLOCATION_un8ID(hSafeCamSubLoc)
                    );
            SAFECAM_SUBLOCATION_vDestroy(hSafeCamSubLoc);
        }

        psResult->bResult = TRUE;
    } while (FALSE);

    // FALSE will stop iteration; TRUE will keep going
    return bContinue;
}


/*****************************************************************************
*
*   bIterateNewLocationsToFill
*
*****************************************************************************/
static BOOLEAN bIterateNewLocationsToFill(
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc,
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bContinue = TRUE;

    do
    {
        BOOLEAN bInitialized = FALSE;
        UN32 un32UID = 0;
        BOOLEAN bSuccess = FALSE;
        SAFECAM_DB_QUERY_SUBLOCATIONS_STRUCT sResult;

        if ((hSafeCamLoc == SAFETY_CAMERAS_LOCATION_INVALID_OBJECT) ||
            (psAppObj == NULL))
        {
            bContinue = FALSE;
            break;
        }

        // checking state of location object
        // if it was filled with data already, then exit
        bInitialized =
            SAFECAM_LOCATION_bInitCompleted(hSafeCamLoc);
        if (bInitialized == TRUE)
        {
            break;
        }

        un32UID = SAFECAM_LOCATION_un32UID(hSafeCamLoc);

        // initialize structure for sublocations request
        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.psAppObj = psAppObj;
        sResult.hSafeCamLoc = hSafeCamLoc;
        sResult.bSpeedInMiles =
            SAFECAM_LOCATION_bSpeedInMiles(hSafeCamLoc);

        psAppObj->asBindParams[SAFECAM_SELECT_SUBLOC_STMT_UID_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psAppObj->asBindParams[SAFECAM_SELECT_SUBLOC_STMT_UID_PARAM].pvData =
            (void *)(size_t)un32UID;

        bSuccess = SQL_INTERFACE.bExecutePreparedStatement(
            psAppObj->hSQLRefConnection,
            psAppObj->hSelectSubLocationsByUIDStmt,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSubLocationsAndBuildObjects,
            &sResult,
            SAFECAM_SELECT_SUBLOC_STMT_PARAMS_COUNT,
            psAppObj->asBindParams);

        if ((bSuccess == FALSE) || (sResult.bResult == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to initialize location"
                "with sublocations"
                    );
        }

        SAFECAM_LOCATION_vSetInitCompleted(hSafeCamLoc);
    } while (FALSE);

    // TRUE to continue iteration, FALSE to stop
    return bContinue;
}


/*****************************************************************************
*
*   bGetSafecamLocsList
*
*****************************************************************************/
static BOOLEAN bGetSafecamLocsList(
    SAFECAM_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        OSAL_FIXED_OBJECT hTopRightLat,
            hTopRightLon,
            hBottomLeftLat,
            hBottomLeftLon,
            hCenterLat,
            hCenterLon;

        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
        UN8 un8NumFixed = 0;    // Keep track of the fixed objects

        N32 n32FixedTopRightLat,
            n32FixedTopRightLon,
            n32FixedBottomLeftLat,
            n32FixedhBottomLeftLon;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

        // Check the lat/lon provided
        hCenterLat = LOCATION.hLat(hLocation);
        hCenterLon = LOCATION.hLon(hLocation);

        // Is the target location populated?
        if ((hCenterLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hCenterLon == OSAL_FIXED_INVALID_OBJECT))
        {
            // No -- but that's okay...just hold off for now
            bResult = TRUE;
            break;
        }

        // Create the objects.  If this fails, LOCATION_bTopRight
        // will result in an error
        hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the top right lat, lon
        bResult = LOCATION_bTopRight(hLocation,
            hTopRightLat, hTopRightLon);
        if (bResult == FALSE)
        {
            break;
        }

        // Create the objects.  If this fails, LOCATION_bBottomLeft
        // will result in an error
        hBottomLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hBottomLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the bottom left lat, lon
        bResult = LOCATION_bBottomLeft(hLocation,
            hBottomLeftLat, hBottomLeftLon);
        if (bResult == FALSE)
        {
            break;
        }

        n32FixedTopRightLat =
            OSAL_FIXED.n32ScaledValue(hTopRightLat, LOCATION_BINPOINT);
        n32FixedTopRightLon =
            OSAL_FIXED.n32ScaledValue(hTopRightLon, LOCATION_BINPOINT);

        n32FixedBottomLeftLat =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLat, LOCATION_BINPOINT);
        n32FixedhBottomLeftLon =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLon, LOCATION_BINPOINT);

        // Perform the SQL query and process the result
        // Please note that all locations found by select
        // will be added to special list for newly found locations
        // New locations cannot be added to DSRL right now because
        // sublocations are not added yet

        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MIN_LAT_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MIN_LAT_PARAM].pvData =
            (void *)(size_t)n32FixedBottomLeftLat;

        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MAX_LAT_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MAX_LAT_PARAM].pvData =
            (void *)(size_t)n32FixedTopRightLat;

        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MIN_LON_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MIN_LON_PARAM].pvData =
            (void *)(size_t)n32FixedhBottomLeftLon;

        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MAX_LON_PARAM].eType =
            SQL_BIND_TYPE_UN32;
        psObj->asBindParams[SAFECAM_SELECT_ALERT_LOC_STMT_MAX_LON_PARAM].pvData =
            (void *)(size_t)n32FixedTopRightLon;

        bResult = SQL_INTERFACE.bExecutePreparedStatement(
            psObj->hSQLRefConnection,
            psObj->hSelectAlertLocationsByLocStmt,
            (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectLocationsAndBuildObjects,
            psObj,
            SAFECAM_SELECT_ALERT_LOC_STMT_PARAMS_COUNT,
            psObj->asBindParams);

        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": SQL request failed"
                    );
            break;
        }

        // Need to iterate list of entry desc structs and fill alert locations objects
        // with sub locations
        eReturnCode = OSAL.eLinkedListIterate(psObj->hNewDSRLEntriesList,
            (OSAL_LL_ITERATOR_HANDLER)bIterateNewLocationsToFill, psObj
                );
        if (eReturnCode != OSAL_SUCCESS &&
            eReturnCode != OSAL_NO_OBJECTS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                 SAFECAM_MGR_OBJECT_NAME": Error iterating objects list");
        }
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   bTargetEntriesGenerator
*
*****************************************************************************/
static BOOLEAN bTargetEntriesGenerator(
    DSRL_TARGET_OBJECT hTarget,
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bContinue = TRUE;
    LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;

    if ((hTarget == DSRL_TARGET_INVALID_OBJECT) || (psAppObj == NULL))
    {
        return FALSE;
    }

    hLocation = DSRL_TARGET.hLocation(hTarget);
    if (hLocation == LOCATION_INVALID_OBJECT)
    {
        return FALSE;
    }

    bContinue = bGetSafecamLocsList(psAppObj, hLocation);

    return bContinue;
}


/*****************************************************************************
*
*   bGenerateDSRLEntryList
*
*****************************************************************************/
static BOOLEAN bGenerateDSRLEntryList(
    SAFECAM_APP_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hTargetList
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListIterate(hTargetList,
        (OSAL_LL_ITERATOR_HANDLER)bTargetEntriesGenerator, psObj
            );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Error iterating targets list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   vChangeDSRLState
*
*****************************************************************************/
static void vChangeDSRLState(DSRL_OBJECT hDSRL, DSRL_STATE_ENUM eState)
{
    DSRL_STATE_ENUM eCurState = DSRL_STATE_ERROR;

    if (hDSRL != DSRL_INVALID_OBJECT)
    {
        eCurState = DSRL.eState(hDSRL);
        if (eCurState != DSRL_STATE_ERROR)
        {
            DSRL_vSetState(hDSRL, eState);
        }
    }

    return;
}

/*****************************************************************************
*
*   bEntryAddToDSRL
*
*****************************************************************************/
static BOOLEAN bEntryAddToDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bContinue = FALSE, bAdded = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        DSRL_ADD_REPLACE_RESULT_ENUM eDSRLResult = DSRL_ADD_REPLACE_ERROR;
        SAFECAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;
        SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        if ((hDSRL == DSRL_INVALID_OBJECT) ||
            (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT))
        {
            break;
        }

        psDSRLDesc = (SAFECAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListAdd(psDSRLDesc->hEntryList,
            &hEntry, hDSRLEntry);
        if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
        {
            // nothing to do here, item in list already
            bAdded = TRUE;
            bContinue = TRUE;
            break;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        psEntryDesc = 
            (SAFECAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        // now changing DSRL state to UPDATING
        vChangeDSRLState(hDSRL, DSRL_STATE_UPDATING);

        eDSRLResult = DSRL_eAddEntry(hDSRL, hDSRLEntry);
        if (eDSRLResult != DSRL_ADD_REPLACE_OK)
        {
            printf(SAFECAM_MGR_OBJECT_NAME": entry cannot be "
                "added for some reason.\n");

            // removing entry from DSRL desc entries list
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": cannot remove entry"
                    " from list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
            bContinue = TRUE;
            break;
        }
        bAdded = TRUE;

        if (psEntryDesc->hDSRLList == OSAL_INVALID_OBJECT_HDL)
        {
            // in case if DSRLs list in EntryDesc struct is not created
            // lets create it
            eReturnCode = OSAL.eLinkedListCreate(&(psEntryDesc->hDSRLList),
                SAFECAM_MGR_OBJECT_NAME":DSRLEntry:DSRLList",
                NULL, OSAL_LL_OPTION_NONE);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": cannot create link list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }

        eReturnCode = OSAL.eLinkedListAdd(psEntryDesc->hDSRLList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hDSRL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot add entry"
                " to list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        bContinue = TRUE;
    } while (FALSE);

    if (((bContinue == FALSE) && (bAdded == TRUE)) || 
        ((bContinue == TRUE) && (bAdded == FALSE)))
    {
        //lets remove entry from DSRL to avoid any issues
        DSRL_vRemoveEntry(hDSRL, hDSRLEntry);
    }

    return bContinue;
}


/*****************************************************************************
*
*   bDSRLAddEntries
*
*****************************************************************************/
static BOOLEAN bDSRLAddEntries(
    DSRL_OBJECT hDSRL,
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    eReturnCode = OSAL.eLinkedListIterate(psAppObj->hNewDSRLEntriesList,
        (OSAL_LL_ITERATOR_HANDLER)bEntryAddToDSRL, hDSRL
            );
    if (eReturnCode == OSAL_NO_OBJECTS)
    {
        printf(SAFECAM_MGR_OBJECT_NAME": nothing to add, "
            "list is empty\n");
    }
    else if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": cannot iterate add entry"
            " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
        bResult = FALSE;
    }

    return bResult;
}


/*****************************************************************************
*
*   bEntryRemoveFromDSRL
*
*****************************************************************************/
static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    SAFECAM_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    do 
    {
        SAFECAM_DSRL_ENTRY_DESC_STRUCT *psEntryDesc = NULL;

        if ((hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT) ||
            (psIterator == NULL))
        {
            break;
        }

        psEntryDesc = 
            (SAFECAM_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
            hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            break;
        }

        if (psEntryDesc->un32TransactionID == 
            psIterator->psAppObj->un32TransactionID)
        {
            // this entry has valid transaction ID and it means that it should
            // stay here.
            bContinue = TRUE;
            break;
        }

        // now changing DSRL state to UPDATING
        vChangeDSRLState(psIterator->hDSRL, DSRL_STATE_UPDATING);

        // if we are here, it means that transaction ID is old and entry should
        // be removed
        DSRL_vRemoveEntry(psIterator->hDSRL, hDSRLEntry);

        bContinue = TRUE;
    } while (FALSE);
    
    return bContinue;
}


/*****************************************************************************
*
*   bDSRLRemoveEntries
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveEntries(
    DSRL_OBJECT hDSRL,
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc = (SAFECAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
    if (psDSRLDesc != NULL)
    {
        SAFECAM_ENTRIES_LIST_ITERATOR_STRUCT sIterator;

        sIterator.hDSRL = hDSRL;
        sIterator.psAppObj = psAppObj;

        eReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hEntryList,
            (OSAL_LL_ITERATOR_HANDLER)bEntryRemoveFromDSRL, &sIterator
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            bResult = TRUE;
        }
        else if (eReturnCode == OSAL_NO_OBJECTS)
        {
            printf(SAFECAM_MGR_OBJECT_NAME": nothing to remove, "
                "list is empty\n");
            bResult = TRUE;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot iterate remove entry"
                " list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return bResult;
}


/*****************************************************************************
*
*   bDSRLUpdateContent
*
*****************************************************************************/
static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    SAFECAM_APP_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        BOOLEAN bSuccess = FALSE;
        SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc;

        psDSRLDesc = (SAFECAM_DSRL_DESC_STRUCT *)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        // Now incrementing the transaction ID to use it for newly
        // generated DSRL entries
        // we suppose that transaction ID will uniquely identify these entries
        psObj->un32TransactionID++;

        bSuccess = bGenerateDSRLEntryList(psObj, psDSRLDesc->hTargetList);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot generate new list "
                "of entries");
            break;
        }

        // adding entries from generated list
        bSuccess = bDSRLAddEntries(hDSRL, psObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot add entries to DSRL from"
                " list of new entries"
                    );
            break;
        }

        //removing entries from generated list
        bSuccess = bDSRLRemoveEntries(hDSRL, psObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot remove entries from DSRL");
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    eReturnCode = OSAL.eLinkedListRemoveAll(psObj->hNewDSRLEntriesList, NULL);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": cannot cleanup new entries list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return bResult;
}


/*****************************************************************************
*
*   bHandleCreateList
*
*****************************************************************************/
static BOOLEAN bHandleCreateList(
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        BOOLEAN bSuccess = FALSE;

        psAppObj = psGetAppFacingObject((SAFETY_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        bSuccess = bDSRLDescInit(psAppObj, psDSRLArg);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to init DSRL desc for"
                    " DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        // Set the finalizer
        bSuccess = DSRL_bSetFinalizeFunction(psDSRLArg->hDSRL,
            (DSRL_FINALIZE_FUNCTION)vDSRLFinalizeEntry, psAppObj
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to set finalizer for DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        bSuccess = bDSRLAddTargets(psDSRLArg, (SMS_OBJECT)psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to add targets to DSRL %p",
                psDSRLArg->hDSRL
                    );
            break;
        }

        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        // Looks good
        bResult = TRUE;
    } while (FALSE);

    if (psAppObj != NULL)
    {
        bRemoveUnusedSafecamLocs(psAppObj);
    }

    vReleaseAppFacingObject(psAppObj);

    return bResult;
}


/*****************************************************************************
*
*   bDSRLReplaceTargets
*
*****************************************************************************/
static BOOLEAN bDSRLReplaceTargets(
    DSRL_ARG_STRUCT *psDSRLArg,
    SMS_OBJECT hParent
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc = DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        return FALSE;
    }

    eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetList,
        (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
            );
    if (eReturnCode !=  OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": failed to release target desc"
            " items in DSRL desc (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode)
                );
    }
    else
    {
        //Add new targets
        bResult = bDSRLAddTargets(psDSRLArg, hParent);
    }

    return bResult;
}


/*****************************************************************************
*
*   n16FindTargetEntry
*
*****************************************************************************/
static N16 n16FindTargetEntry(
    DSRL_TARGET_OBJECT hTarget1,
    DSRL_TARGET_OBJECT hTarget2
        )
{
    N16 n16Result = N16_MIN;

    if ((hTarget1 != DSRL_TARGET_INVALID_OBJECT) &&
        (hTarget2 != DSRL_TARGET_INVALID_OBJECT))
    {
        LOCATION_OBJECT hLocation1 = LOCATION_INVALID_OBJECT,
            hLocation2 = LOCATION_INVALID_OBJECT;
        BOOLEAN bCompare = FALSE;

        hLocation1 = DSRL_TARGET.hLocation(hTarget1);
        hLocation2 = DSRL_TARGET.hLocation(hTarget2);
        bCompare = LOCATION_bCompare(hLocation1, hLocation2);
        if (bCompare == TRUE)
        {
            n16Result = 0;
        }
        else
        {
            n16Result = 1;
        }
    }

    return n16Result;
}


/*****************************************************************************
*
*   bRemoveTarget
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME" %s start\n", __FUNCTION__);

    do
    {
        OSAL_LINKED_LIST_ENTRY hTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
        DSRL_TARGET_OBJECT hTargetToRemove = DSRL_TARGET_INVALID_OBJECT;

        psDSRLDesc = DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        if (hTargetObj == DSRL_TARGET_INVALID_OBJECT)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLDesc->hTargetList,
            &hTargetEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindTargetEntry,
            hTargetObj
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to find target entry (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        hTargetToRemove = OSAL.pvLinkedListThis(hTargetEntry);
        if (hTargetToRemove == DSRL_TARGET_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to get target handle"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(hTargetEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to remove target entry"
                " from list of DSRL targets (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
        DSRL_TARGET_vDestroyByType(hTargetToRemove);

        bResult = TRUE;
    } while (FALSE);

    printf(SAFECAM_MGR_OBJECT_NAME" %s end, result is %d\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   bDSRLRemoveTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tDSRLTargetIndex;

    printf(SAFECAM_MGR_OBJECT_NAME" %s start\n", __FUNCTION__);

    for (tDSRLTargetIndex = 0;
         tDSRLTargetIndex < psDSRLArg->tNumTargets;
         ++tDSRLTargetIndex)
    {
        bDSRLRemoveTarget(
            psDSRLArg->hDSRL,
            psDSRLArg->ahTargetList[tDSRLTargetIndex]
                );
    }

    printf(SAFECAM_MGR_OBJECT_NAME" %s end \n", __FUNCTION__);

    return TRUE;
}


/*****************************************************************************
*
*   bDSRLRemoveAllTargets
*
*****************************************************************************/
static BOOLEAN bDSRLRemoveAllTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    printf(SAFECAM_MGR_OBJECT_NAME" %s start\n", __FUNCTION__);

    psDSRLDesc = DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc == NULL)
    {
        bResult = FALSE;
    }
    else
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLDesc->hTargetList,
            (OSAL_LL_RELEASE_HANDLER)DSRL_TARGET_vDestroyByType
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to remove DSRL targets"
                " (%s)", OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            bResult = FALSE;
        }
    }

    printf(SAFECAM_MGR_OBJECT_NAME" %s end with result %d\n", __FUNCTION__, bResult);

    return bResult;
}




/*****************************************************************************
*
*   bHandleModifyList
*
*****************************************************************************/
static BOOLEAN bHandleModifyList(
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        BOOLEAN bSuccess = FALSE;

        psAppObj = psGetAppFacingObject((SAFETY_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        vChangeDSRLState(psDSRLArg->hDSRL,
            DSRL_STATE_UPDATING
                );

        printf(SAFECAM_MGR_OBJECT_NAME" Modify type is %d\n",
            psDSRLArg->uAction.sModify.eModifyType
                );

        // Modify the DSRL as the user specified
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                bSuccess =
                    bDSRLAddTargets(psDSRLArg, (SMS_OBJECT)psAppObj);
            }
            break;

            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                bSuccess =
                    bDSRLReplaceTargets(psDSRLArg, (SMS_OBJECT)psAppObj);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                bSuccess =
                    bDSRLRemoveTargets(psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVEALL:
                bSuccess =
                    bDSRLRemoveAllTargets(psDSRLArg);
            break;

            case DSRL_MODIFY_OPERATION_INVALID:
            default:
                break;

        }

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": DSRL modify targets operation failed"
                    );
            break;
        }

        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": DSRL modify entries operation failed"
                    );
            break;
        }

        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    if (psAppObj != NULL)
    {
        bRemoveUnusedSafecamLocs(psAppObj);
    }

    vReleaseAppFacingObject(psAppObj);

    return bResult;
}




/*****************************************************************************
*
*   bHandleRefreshList
*
*****************************************************************************/
static BOOLEAN bHandleRefreshList(
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        BOOLEAN bSuccess = FALSE;

        psAppObj = psGetAppFacingObject((SAFETY_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        bSuccess = bDSRLUpdateContent(psDSRLArg->hDSRL, psAppObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": failed to update DSRL content"
                    );
            break;
        }

        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_READY);

        bResult = TRUE;
    } while (FALSE);

    if (bResult == FALSE)
    {
        vChangeDSRLState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
    }

    if (psAppObj != NULL)
    {
        bRemoveUnusedSafecamLocs(psAppObj);
    }

    vReleaseAppFacingObject(psAppObj);

    return bResult;
}




/*****************************************************************************
*
*   vHandleDeleteList
*
*****************************************************************************/
static void vHandleDeleteList(
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    SAFECAM_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        SAFECAM_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

        psAppObj = psGetAppFacingObject((SAFETY_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            break;
        }

        // Remove / free all entries in the DSRL
        DSRL_vRemoveAllEntries(psDSRLArg->hDSRL);

        psDSRLDesc = DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hDSRLListEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": cannot remove DSRL from pool"
                " (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        vDSRLRelease(psDSRLArg->hDSRL);

        psDSRLArg->hDSRL = DSRL_INVALID_OBJECT;
    } while (FALSE);

    if (psAppObj != NULL)
    {
        bRemoveUnusedSafecamLocs(psAppObj);
    }

    vReleaseAppFacingObject(psAppObj);

    return;
}


/*****************************************************************************
*
*   bDBGetContentVersion
*
*****************************************************************************/
static BOOLEAN bDBGetContentVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N32 *pn32DBVer
        )
{
    SAFECAM_DB_RESULT_STRUCT sResult;
    BOOLEAN bResult = FALSE;
    SAFECAM_DB_VERSION_ROW_STRUCT sVersion;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));
    OSAL.bMemSet(&sVersion, 0, sizeof(sVersion));

    sResult.uDbRow.psDBVersion = &sVersion;

    if ((hSQLConnection == SQL_INTERFACE_INVALID_OBJECT) ||
        (pn32DBVer == NULL))
    {
        return FALSE;
    }

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bResult = SQL_INTERFACE.bQuery(
        hSQLConnection,
        SAFECAM_SELECT_DB_DATA_VERSION,
        (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectDBVersion,
        &sResult
            );

    if ((bResult == TRUE) &&
        (sResult.bResult == TRUE))
    {
        *pn32DBVer = (N32)sResult.uDbRow.psDBVersion->un32Version;
    }

    return sResult.bResult;
}


/*****************************************************************************
*
*   n32ExtractDataVersion
*
*****************************************************************************/
static N32 n32ExtractDataVersion(
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        )
{
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    BOOLEAN bLoaded;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    // Get the Content Version (table!)
    bLoaded = bDBGetContentVersion(
        hConnection, &n32Result
            );
    if (bLoaded != TRUE)
    {
        n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    return n32Result;
}


/*****************************************************************************
*
*   bDBGetUpdateNumber
*
*****************************************************************************/
static BOOLEAN bDBGetUpdateNumber (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N32 *pn32UpdateNumber
        )
{
    SAFECAM_DB_RESULT_STRUCT sResult;
    BOOLEAN bResult = FALSE;
    SAFECAM_DB_VERSION_ROW_STRUCT sVersion;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));
    OSAL.bMemSet(&sVersion, 0, sizeof(sVersion));

    sResult.uDbRow.psDBVersion = &sVersion;

    if ((hSQLConnection == SQL_INTERFACE_INVALID_OBJECT) ||
        (pn32UpdateNumber == NULL))
    {
        return FALSE;
    }

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bResult = SQL_INTERFACE.bQuery(
        hSQLConnection,
        SAFECAM_SELECT_UPDATE_NUMBER,
        (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectDBVersion,
        &sResult
            );

    if ((bResult == TRUE) &&
        (sResult.bResult == TRUE))
    {
        *pn32UpdateNumber = (N32)sResult.uDbRow.psDBVersion->un32Version;
    }

    return sResult.bResult;
}


/*****************************************************************************
*
*   n32ExtractUpdateNumber
*
*****************************************************************************/
static N32 n32ExtractUpdateNumber(
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        )
{
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    BOOLEAN bLoaded;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    // Get the Update Number from DB
    bLoaded = bDBGetUpdateNumber(
        hConnection, &n32Result
            );
    if (bLoaded != TRUE)
    {
        n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    return n32Result;
}

/*****************************************************************************
*
*   bVerifyDBSchemaVersion
*
*****************************************************************************/
static BOOLEAN bVerifyDBSchemaVersion(
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN8 un8SchemaVer
        )
{
    SAFECAM_DB_RESULT_STRUCT sResult;
    BOOLEAN bSuccess = FALSE, bVersionMatched = FALSE;
    SAFECAM_DB_VERSION_ROW_STRUCT sVersion;

    if (hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return FALSE;
    }

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));
    OSAL.bMemSet(&sVersion, 0, sizeof(sVersion));
    sResult.uDbRow.psDBVersion = &sVersion;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bSuccess = SQL_INTERFACE.bQuery(
        hSQLConnection,
        SAFECAM_SELECT_DB_SCHEMA_VERSION,
        (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectDBVersion,
        &sResult
            );

    if ((bSuccess == TRUE) && (sResult.bResult == TRUE))
    {
        if ((sResult.uDbRow.psDBVersion != NULL) &&
            (sResult.uDbRow.psDBVersion->un32Version == un8SchemaVer))
        {
            // Version match!
            bVersionMatched = TRUE;
        }
    }

    return bVersionMatched;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*   Although we (currently) only have a timestamp entry for the last database
*   update, this infrastructure should allow for easy expansion wrt/any new
*   _TIMESTAMP_TYPES_ENUM values.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    SAFECAM_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj = NULL;
    char *pacDatabaseFilePath = NULL;
    char *pacDatabaseFilePathB = NULL;

    do
    {
        BOOLEAN bSuccess = FALSE;
        RFD_UPDATE_VERSION tCurrentVersion = 0;

        psAppObj = psGetAppFacingObject((SAFETY_CAMERAS_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            // Error, not sure why this would happen
            eErrorCode =  DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        if (psAppObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            // Construct the paths needed by the databases
            bSuccess = DB_UTIL_bCreateFilePath(
                psObj->pacRefDatabaseDirPath,
                SAFECAM_DATABASE_FOLDER,
                SAFECAM_REF_DATABASE_FILENAMEA,
                &pacDatabaseFilePath
                    );
            if (bSuccess == FALSE)
            {
                // Error!  Couldn't build database path
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }

            // Construct the path needed for the reference database
            bSuccess = DB_UTIL_bCreateFilePath(
                psObj->pacRefDatabaseDirPath,
                SAFECAM_DATABASE_FOLDER,
                SAFECAM_REF_DATABASE_FILENAMEB,
                &pacDatabaseFilePathB
                    );
            if (bSuccess == FALSE)
            {
                // Error!  Couldn't build database path
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }

            // Connect to the ref database bank, but
            // never perform an integrity check
            psAppObj->hSQLRefConnection =
                DB_UTIL_hConnectToReferenceBank(
                pacDatabaseFilePath,
                pacDatabaseFilePathB,
                &psObj->pacCurRefDatabaseFilePath,
                n32ExtractUpdateNumber, NULL,
                TRUE, NULL,
                GsSafeCamIntf.tMaxVersionBitlen,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyDBSchemaVersion,
                (void*)(size_t)SAFECAM_DATABASE_FILE_VERSION,
                &eErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK
                    );

            // checking connection state
            if (psAppObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND;
                break;
            }

            // create prepared statements
            psAppObj->hSelectSubLocationsByUIDStmt =
                SQL_INTERFACE.hCreatePreparedStatement(
                    psAppObj->hSQLRefConnection,
                    SAFECAM_SELECT_SUB_LOCATIONS_BY_UID);

            if (psAppObj->hSelectSubLocationsByUIDStmt ==
                SQL_PREPARED_STATEMENT_INVALID_HANDLE)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }

            psAppObj->hSelectAlertLocationsByLocStmt =
                SQL_INTERFACE.hCreatePreparedStatement(
                    psAppObj->hSQLRefConnection,
                    SAFECAM_SELECT_ALERT_LOCATIONS_BY_LOCATION);

            if (psAppObj->hSelectAlertLocationsByLocStmt ==
                SQL_PREPARED_STATEMENT_INVALID_HANDLE)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }

            if (psObj->pacCurRefDatabaseFilePath == pacDatabaseFilePath)
            {
                psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePathB;
            }
            else
            {
                psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePath;
            }

            // These are not needed any more.
            // Cleaning up pointers to avoid deallocation.
            pacDatabaseFilePath = NULL;
            pacDatabaseFilePathB = NULL;
        }

        // Now, we have to actually use the data version from
        // the selected connection
        tCurrentVersion = (RFD_UPDATE_VERSION)
                n32ExtractDataVersion(psAppObj->hSQLRefConnection, NULL);

        if ((psObj->hInterface == SAFECAM_INTERFACE_INVALID_OBJECT) &&
            (psObj->bRefDBUpdatesEnabled == TRUE))
        {
            // Start the interface
            psObj->hInterface = GsSafeCamIntf.hInit(
                (SAFETY_CAMERAS_SERVICE_OBJECT)psObj,
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
                tCurrentVersion
                    );
            if (psObj->hInterface == SAFECAM_INTERFACE_INVALID_OBJECT)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }
        }

        bSuccess = bCreateNewAlertTypes(psAppObj, TRUE);
        if (bSuccess == FALSE)
        {
            eErrorCode =  DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    vReleaseAppFacingObject(psAppObj);

    if (bResult != TRUE)
    {
        // Set the error we experienced
        vSetError(psObj, eErrorCode);
    }

    // Remove database path resources
    if (pacDatabaseFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);
    }

    if (pacDatabaseFilePathB != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePathB);
    }

    return bResult;
}

/*****************************************************************************
*
*   bProcessMessage
*
*   This function is called to provide the safecam interface with
*   a newly received message
*
*****************************************************************************/
static BOOLEAN bProcessMessage(
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bProcessed = TRUE;

    // Only process this message if
    // the handle is valid
    if ((*phPayload != OSAL_INVALID_BUFFER_HDL) &&
        (psObj->bRefDBUpdatesEnabled == TRUE))
    {
        // Tell the interface it's time to
        // process another message
        bProcessed = GsSafeCamIntf.bProcessMessage(psObj->hInterface,
            phPayload
                );
    }

    return bProcessed;
}


/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    SAFECAM_MGR_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bValid = FALSE, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our safecam handle from the callback argument
    psObj = (SAFECAM_MGR_OBJECT_STRUCT*)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == FALSE)
    {
        return;
    }

    switch (tCurrentEvent)
    {
        // Handle SafeCam Service events here...

        // State has changed
        case DATASERVICE_EVENT_STATE:
        {
            BOOLEAN bStateChanged;
            DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

            // Process the state transition
            bStateChanged = DATASERVICE_IMPL_bStateFSM(
                (DATASERVICE_IMPL_HDL)psObj,
                psStateChange,
                &GsSafeCamStateHandlers,
                (void *)psObj);

            if (bStateChanged == TRUE)
            {
                // The state has been updated
                tEventMask |= DATASERVICE_EVENT_STATE;
                // Is the service stopped now?
                if (psStateChange->eCurrentState ==
                        DATASERVICE_STATE_STOPPED)
                {
                    bStopEvent = TRUE;
                }
            }
        }
        break;

        // This service has a message to process
        case DATASERVICE_EVENT_NEW_DATA:
        {
            OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
            BOOLEAN bSuccess = TRUE;

            // Ensure the payload handle is valid
            // If it isn't, there's a problem with
            // SMS
            if (hPayload == OSAL_INVALID_BUFFER_HDL)
            {
                // Set the error
                vSetError(psObj,
                    DATASERVICE_ERROR_CODE_GENERAL
                        );
            }
            // Only process this message if we are not stopping
            else
            {
                printf(SAFECAM_MGR_OBJECT_NAME": payload received (%u)\n",
                    OSAL.tBufferGetSize(hPayload)
                        );

                // Process this message now
                bSuccess = bProcessMessage(psObj, &hPayload);

                if (bSuccess == TRUE)
                {
                    puts(SAFECAM_MGR_OBJECT_NAME": Message Payload Processed Ok");
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        SAFECAM_MGR_OBJECT_NAME": failed to process payload"
                            );
                    DATASERVICE_IMPL_vLog(SAFECAM_MGR_OBJECT_NAME
                        ": failed to process payload\n"
                            );
                }
            }

            // We're all done with this payload
            DATASERVICE_IMPL_bFreeDataPayload(hPayload);
        }
        break;

        // We need to do some work with our DSRLs
        case DATASERVICE_INTERNAL_EVENT_DSRL:
        {
            BOOLEAN bSuccess = TRUE;
            DSRL_ARG_STRUCT *psDSRLArg =
                (DSRL_ARG_STRUCT*)pvEventArg;
            if (psDSRLArg == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": DSRL_EVENT with no targets"
                        );
                break;
            }

            switch (psDSRLArg->eAction)
            {
                case DSRL_ACTION_ADD:
                {
                    // Create a new list
                    bSuccess = bHandleCreateList(
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_MODIFY:
                {
                    // Modify a pre-existing list
                    bSuccess = bHandleModifyList(
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_REFRESH:
                {
                    // Refresh a pre-existing list
                    bSuccess = bHandleRefreshList(
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_REMOVE:
                {
                    // Handle the deletion
                    vHandleDeleteList(psObj, psDSRLArg);
                    bSuccess = TRUE;
                }
                break;

                case DSRL_ACTION_INVALID:
                default:
                {
                    bSuccess = FALSE;
                }
                break;
            }

            if (bSuccess == FALSE)
            {
                // If an error occurred, indicate a state change
                // to the application
                tEventMask |= DATASERVICE_EVENT_STATE;

                // Indicate an error
                vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

                DATASERVICE_IMPL_vLog(SAFECAM_MGR_OBJECT_NAME
                    ": failed to process DSRL event\n"
                        );
            }
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Got unknown event (%u)",
                tCurrentEvent
                    );
        }
        break;
    }


    if (bStopEvent == TRUE)
    {
        // Uninitialize the object, but leave
        // enough to be useful to the callback
        vUninitObject(psObj, FALSE);
    }

    // Update event mask with any relevant events which have occurred
    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    if (bStopEvent == TRUE)
    {
        // Filter out all further weather manager updates
        SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

        vUninitObject(psObj, TRUE);
    }

    return;
}


/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}


/*******************************************************************************
*
*        bRefDBBank
*
*******************************************************************************/
static BOOLEAN bRefDBBank (
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    if ((phInUseDB == NULL) || (phNextDB == NULL))
    {
        return FALSE;
    }

    // Initialize inputs
    *phInUseDB = *phNextDB = STRING_INVALID_OBJECT;

    bLocked = DATASERVICE_IMPL_bLock((DATASERVICE_IMPL_HDL)hSafeCamService);
    if (bLocked == TRUE)
    {
        SAFECAM_MGR_OBJECT_STRUCT *psObj =
            (SAFECAM_MGR_OBJECT_STRUCT *)hSafeCamService;

        do
        {
            // Provide the caller with the path info
            *phInUseDB = STRING.hCreate(
                psObj->pacCurRefDatabaseFilePath,
                strlen(psObj->pacCurRefDatabaseFilePath));

            if (*phInUseDB == STRING_INVALID_OBJECT)
            {
                // Error! Stop here
                break;
            }

            *phNextDB = STRING.hCreate(
                psObj->pacNextRefDatabaseFilePath,
                strlen(psObj->pacNextRefDatabaseFilePath));

            if (*phNextDB == STRING_INVALID_OBJECT)
            {
                // Clear the other string now
                STRING_vDestroy(*phInUseDB);
                *phInUseDB = STRING_INVALID_OBJECT;

                break;
            }

            // All went well
            bSuccess = TRUE;
        } while (FALSE);

        DATASERVICE_IMPL_vUnlock((DATASERVICE_IMPL_HDL)hSafeCamService);
    }

    return bSuccess;
}


/*******************************************************************************
*
*        bDBUpdateBegin
*
*******************************************************************************/
static BOOLEAN bDBUpdateBegin (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    BOOLEAN bResult = FALSE;
    int iReturn = 0;
    N32 n32UpdateNumber = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;

    // Generate the SQL command to update the version row
    // in the database
    iReturn = snprintf(pacBuffer, tBufferSize,
        SAFECAM_SET_UPDATE_NUMBER, DB_UTIL_DB_UNDER_CONSTRUCTION_VER
            );

    if (iReturn <= 0)
    {
        return FALSE;
    }

    n32UpdateNumber = n32ExtractUpdateNumber(hConnection, NULL);
    if (n32UpdateNumber == DB_UTIL_DB_UNDER_CONSTRUCTION_VER)
    {
        return FALSE;
    }

    // update in database
    bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Unable to set update number"
                );
    }

    // Generate the SQL command to set next update number
    // in the database
    iReturn = snprintf(pacBuffer, tBufferSize,
        SAFECAM_SET_NEXT_UPDATE_NUMBER, ++n32UpdateNumber
            );

    if (iReturn <= 0)
    {
        return FALSE;
    }

    // update in database
    bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Unable to set next update number" );
    }

    return bResult;
}


/*******************************************************************************
*
*        bDBUpdateEnd
*
*******************************************************************************/
static BOOLEAN bDBUpdateEnd (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    N32 n32NextUpdateNumber = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    N32 n32Return = 0;
    SAFECAM_DB_RESULT_STRUCT sResult;
    SAFECAM_DB_VERSION_ROW_STRUCT sUpdateNumber;
    BOOLEAN bResult = FALSE;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));
    OSAL.bMemSet(&sUpdateNumber, 0, sizeof(sUpdateNumber));
    sResult.uDbRow.psDBVersion = &sUpdateNumber;

    bResult = SQL_INTERFACE.bQuery(hConnection,
        SAFECAM_SELECT_NEXT_UPDATE_NUMBER,
        (SQL_QUERY_RESULT_HANDLER)bDBProcessSelectDBVersion, &sResult
            );
    if ((bResult == FALSE) || (sResult.bResult == FALSE))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Unable to get next update number" );
    }

    n32NextUpdateNumber = (N32)sResult.uDbRow.psDBVersion->un32Version;
    if (n32NextUpdateNumber == DB_UTIL_DB_UNDER_CONSTRUCTION_VER)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Incorrect next update number" );

        return FALSE;
    }

    n32Return = snprintf(pacBuffer, tBufferSize, SAFECAM_SET_UPDATE_NUMBER,
        n32NextUpdateNumber
            );
    if (n32Return <= 0)
    {
        return FALSE;
    }

    bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Failed to set update number"
                );
    }

    // While the DB_UTIL function will print its own error, we also print
    // one here to make it easier to trace the DB failure back to a
    // particular service.
    bResult = DB_UTIL_bUpdateTimestamp( hConnection, pacBuffer, tBufferSize );
    if ( bResult == FALSE )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Unable to set timestamp.");
    }

    n32Return = snprintf(pacBuffer, tBufferSize, SAFECAM_UPDATE_DB_DATA_VERSION,
        tUpdateVersion
            );
    if (n32Return <= 0)
    {
        return FALSE;
    }

    // Updating the baseline version comes last, as this effectively stamps
    // the DB as "ready."
    bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            SAFECAM_MGR_OBJECT_NAME": Failed to update DB data version"
                );

        return FALSE;
    }

    return bResult;
}


/*****************************************************************************
*
*   bAddAlertLocation
*
*****************************************************************************/
static BOOLEAN bAddAlertLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((psAlertLocation != NULL) && (pacBuffer != NULL) &&
        (hConnection != SQL_INTERFACE_INVALID_OBJECT))
    {
        //adding location
        SAFECAM_DB_RESULT_STRUCT sResult;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));

        sResult.uDbRow.psAlertLocations = psAlertLocation;
        bResult = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection, SAFECAM_INSERT_ALERT_LOCATION,
            SAFECAM_ALERT_LOCATIONS_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bDBPrepareAlertLocationRow,
            (void*)&sResult
                );
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   bAddSubLocation
*
*****************************************************************************/
static BOOLEAN bAddSubLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((psSubLocation != NULL) && (pacBuffer != NULL) &&
        (hConnection != SQL_INTERFACE_INVALID_OBJECT))
    {
        SAFECAM_DB_RESULT_STRUCT sResult;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));

        // remove sublocation but do not check result
        // to avoid duplicated sublocations in table
        bDeleteSubLocation(hConnection, pacBuffer, tBufferSize, psSubLocation);

        //adding sublocation
        sResult.uDbRow.psSubLocations = psSubLocation;
        bResult = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection, SAFECAM_INSERT_SUB_LOCATION,
            SAFECAM_SUB_LOCATIONS_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bDBPrepareSubLocationRow,
            (void*)&sResult
                );
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   bModifyAlertLocation
*
*****************************************************************************/
static BOOLEAN bModifyAlertLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((psAlertLocation != NULL) && (pacBuffer != NULL) &&
        (hConnection != SQL_INTERFACE_INVALID_OBJECT))
    {
        //adding location
        SAFECAM_DB_RESULT_STRUCT sResult;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));

        sResult.uDbRow.psAlertLocations = psAlertLocation;
        bResult = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection, SAFECAM_INSERT_ALERT_LOCATION,
            SAFECAM_ALERT_LOCATIONS_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bDBPrepareAlertLocationRow,
            (void*)&sResult
                );
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   bModifySubLocation
*
*****************************************************************************/
static BOOLEAN bModifySubLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((psSubLocation != NULL) && (pacBuffer != NULL) &&
        (hConnection != SQL_INTERFACE_INVALID_OBJECT))
    {
        SAFECAM_DB_RESULT_STRUCT sResult;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));

        // remove sublocation but do not check result
        // to avoid duplicated sublocations in table
        bDeleteSubLocation(hConnection, pacBuffer, tBufferSize, psSubLocation);

        //adding sublocation
        sResult.uDbRow.psSubLocations = psSubLocation;
        bResult = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection, SAFECAM_INSERT_SUB_LOCATION,
            SAFECAM_SUB_LOCATIONS_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bDBPrepareSubLocationRow,
            (void*)&sResult
                );
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n",
        __FUNCTION__, bResult
            );

    return bResult;
}


/*****************************************************************************
*
*   bDeleteAlertLocation
*
*****************************************************************************/
static BOOLEAN bDeleteAlertLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((psAlertLocation != NULL) && (pacBuffer != NULL) &&
        (hConnection != SQL_INTERFACE_INVALID_OBJECT))
    {
        //removing sub locations
        bResult = bDBDeleteAllSubLocations(hConnection, pacBuffer,
            tBufferSize, psAlertLocation
                );
        if (bResult == TRUE)
        {
            bResult = bDBDeleteAlertLocation(hConnection, pacBuffer,
                tBufferSize, psAlertLocation
                    );
        }
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n",
        __FUNCTION__, bResult
            );

    return bResult;
}


/*****************************************************************************
*
*   bGetAlertLocationData
*
*****************************************************************************/
static BOOLEAN bGetAlertLocationData (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((psAlertLocation != NULL) && (pacBuffer != NULL) &&
        (hConnection != SQL_INTERFACE_INVALID_OBJECT))
    {
        bResult = bDBGetAlertLocationData(hConnection, pacBuffer, tBufferSize,
            psAlertLocation
                );
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   bGetSubLocationData
*
*****************************************************************************/
static BOOLEAN bGetSubLocationData (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((psSubLocation != NULL) && (pacBuffer != NULL) &&
        (hConnection != SQL_INTERFACE_INVALID_OBJECT))
    {
        bResult = bDBGetSubLocationData(hConnection, pacBuffer,
            tBufferSize, psSubLocation
                );
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   bGeoPointsIterator
*
*****************************************************************************/
static BOOLEAN bGeoPointsIterator(
    SAFECAM_GEO_POINT_FIXED_STRUCT *psGeoPoint,
    SAFECAM_LOCATION_BOX_STRUCT *psLocationBox
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        N16 n16Result = 0;
        BOOLEAN bSuccess = TRUE;

        if ((psGeoPoint == NULL) || (psLocationBox == NULL))
        {
            break;
        }

        n16Result = OSAL_FIXED.n16Compare(psLocationBox->hLatMin,
            psGeoPoint->hLat
                );
        if (n16Result == 1)
        {
            bSuccess = OSAL_FIXED.bCopyToMemory(psGeoPoint->hLat,
                psLocationBox->hLatMin
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": Failed to copy fixed variable"
                        );
                break;
            }
        }

        n16Result = OSAL_FIXED.n16Compare(psLocationBox->hLatMax,
            psGeoPoint->hLat
                );
        if (n16Result == -1)
        {
            bSuccess = OSAL_FIXED.bCopyToMemory(psGeoPoint->hLat,
                psLocationBox->hLatMax
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": Failed to copy fixed variable"
                        );
                break;
            }
        }


        n16Result = OSAL_FIXED.n16Compare(psLocationBox->hLonMin,
            psGeoPoint->hLon
                );
        if (n16Result == 1)
        {
            bSuccess = OSAL_FIXED.bCopyToMemory(psGeoPoint->hLon,
                psLocationBox->hLonMin
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": Failed to copy fixed variable"
                        );
                break;
            }
        }

        n16Result = OSAL_FIXED.n16Compare(psLocationBox->hLonMax,
            psGeoPoint->hLon
                );
        if (n16Result == -1)
        {
            bSuccess = OSAL_FIXED.bCopyToMemory(psGeoPoint->hLon,
                psLocationBox->hLonMax
                    );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    SAFECAM_MGR_OBJECT_NAME": Failed to copy fixed variable"
                        );
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bUpdateAlertLocationRtree
*
*****************************************************************************/
static BOOLEAN bUpdateAlertLocationRtree (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        )
{
    BOOLEAN bResult = FALSE;
    SAFECAM_DB_QUERY_OBJECTS_STRUCT sResult;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    OSAL.bMemSet(&sResult, 0, sizeof(sResult));

    do
    {
        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 6];
        SAFECAM_LOCATION_BOX_STRUCT sLocationBox;
        BOOLEAN bSuccess = TRUE;
        N32 n32LatMin = 0, n32LonMin = 0, n32LonMax = 0, n32LatMax = 0;
        int iReturn = 0;

        if ((hConnection == SQL_INTERFACE_INVALID_OBJECT) ||
            (pacBuffer == NULL) || (psAlertLocation == NULL))
        {
            break;
        }

        OSAL.bMemSet(&sLocationBox, 0, sizeof(sLocationBox));
        sLocationBox.hLatMin = OSAL_FIXED.hCreateInMemory(
            (psAlertLocation->n32Lat - SAFECAM_RTREE_RANGE),
            LOCATION_BINPOINT, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]
                );
        sLocationBox.hLatMax = OSAL_FIXED.hCreateInMemory(
            (psAlertLocation->n32Lat + SAFECAM_RTREE_RANGE),
            LOCATION_BINPOINT, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]
                );

        sLocationBox.hLonMin = OSAL_FIXED.hCreateInMemory(
            (psAlertLocation->n32Lon - SAFECAM_RTREE_RANGE),
            LOCATION_BINPOINT, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 2]
                );
        sLocationBox.hLonMax = OSAL_FIXED.hCreateInMemory(
            (psAlertLocation->n32Lon + SAFECAM_RTREE_RANGE),
            LOCATION_BINPOINT, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 3]
                );

        if ((sLocationBox.hLatMin == OSAL_FIXED_INVALID_OBJECT) ||
            (sLocationBox.hLatMax == OSAL_FIXED_INVALID_OBJECT) ||
            (sLocationBox.hLonMin == OSAL_FIXED_INVALID_OBJECT) ||
            (sLocationBox.hLonMax == OSAL_FIXED_INVALID_OBJECT))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to create fixed variables"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&(sResult.hObjectsList),
            SAFECAM_MGR_OBJECT_NAME":SubLocPointsList",
            NULL, OSAL_LL_OPTION_LINEAR|OSAL_LL_OPTION_UNIQUE
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to create linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }
        bSuccess = bDBGetSubLocationPoints(hConnection, pacBuffer, tBufferSize,
            psAlertLocation, &sResult
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed get sub locations points"
                    );
            break;
        }

        eReturnCode = OSAL.eLinkedListIterate(sResult.hObjectsList,
            (OSAL_LL_ITERATOR_HANDLER)bGeoPointsIterator, &sLocationBox
                );
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to iterate linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            break;
        }

        n32LatMin = OSAL_FIXED.n32ScaledValue(sLocationBox.hLatMin,
            LOCATION_BINPOINT
                );
        n32LatMax = OSAL_FIXED.n32ScaledValue(sLocationBox.hLatMax,
            LOCATION_BINPOINT
                );
        n32LonMin = OSAL_FIXED.n32ScaledValue(sLocationBox.hLonMin,
            LOCATION_BINPOINT
                );
        n32LonMax = OSAL_FIXED.n32ScaledValue(sLocationBox.hLonMax,
            LOCATION_BINPOINT
                );

        // Build our string to remove the locations_rtree row from the database
        iReturn = snprintf(pacBuffer, tBufferSize,
            SAFECAM_DELETE_ALERT_LOCATION_RTREE, psAlertLocation->un32UID
                );
        if (iReturn > 0)
        {
            bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
            if (bSuccess == FALSE)
            {
                printf(SAFECAM_MGR_OBJECT_NAME": Unable to delete location rtree"
                    " for uid u%\n", psAlertLocation->un32UID
                        );
            }
        }

        iReturn = snprintf(pacBuffer, tBufferSize,
            SAFECAM_INSERT_LOCATION_RTREE, psAlertLocation->un32UID,
            n32LatMin, n32LatMax, n32LonMin, n32LonMax
                );
        if (iReturn <= 0)
        {
            break;
        }

        printf(SAFECAM_MGR_OBJECT_NAME": SQL string %s\n", pacBuffer);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to insert location rtree"
                    );
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (sResult.hObjectsList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(sResult.hObjectsList,
            (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy
                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to cleanup linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        eReturnCode = OSAL.eLinkedListDelete(sResult.hObjectsList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                SAFECAM_MGR_OBJECT_NAME": Failed to delete linked list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        sResult.hObjectsList = OSAL_INVALID_OBJECT_HDL;
    }

    return bResult;
}


/*****************************************************************************
*
*   bDeleteSubLocation
*
*****************************************************************************/
static BOOLEAN bDeleteSubLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((hConnection != SQL_INTERFACE_INVALID_OBJECT) && (pacBuffer != NULL) &&
        (psSubLocation != NULL))
    {
        //deleting sublocation info
        int iReturn = 0;

        // Build string for deletion
        iReturn = snprintf(
            pacBuffer,
            tBufferSize,
            SAFECAM_DELETE_SUBLOCATION_BY_ID_UID,
            psSubLocation->un8SubID,
            psSubLocation->un32UID
                );

        printf(SAFECAM_MGR_OBJECT_NAME" SQL "
            "request is %s, size is %d\n", pacBuffer, iReturn
                );

        if (iReturn <= 0)
        {
            return FALSE;
        }

        bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   vReleaseAppFacingObject
*
*****************************************************************************/
static void vReleaseAppFacingObject(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        )
{
    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return;
}

/*****************************************************************************
*
*   bAddAlertType
*
*****************************************************************************/
static BOOLEAN bAddAlertType (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertType
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((hConnection != SQL_INTERFACE_INVALID_OBJECT) && (pacBuffer != NULL) &&
        (psAlertType != NULL))
    {
        SAFECAM_DB_RESULT_STRUCT sResult;

        OSAL.bMemSet(&sResult, 0, sizeof(sResult));
        sResult.uDbRow.psAlertTypes = psAlertType;
        bResult = SQL_INTERFACE.bExecutePreparedCommand(
            hConnection, SAFECAM_INSERT_ALERT_TYPE,
            SAFECAM_ALERT_TYPES_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bDBPrepareAlertTypeRow,
            (void*)&sResult
                );
    }

    return bResult;
}


/*****************************************************************************
*
*   bDeleteAlertType
*
*****************************************************************************/
static BOOLEAN bDeleteAlertType (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertType
        )
{
    BOOLEAN bResult = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((hConnection != SQL_INTERFACE_INVALID_OBJECT) && (pacBuffer != NULL) &&
        (psAlertType != NULL))
    {
        int iReturn;

        iReturn = snprintf(pacBuffer, tBufferSize,
            SAFECAM_DELETE_ALERT_TYPE, psAlertType->un8Type
                );
        if (iReturn > 0)
        {
            // update in database
            bResult = SQL_INTERFACE.bExecuteCommand(hConnection, pacBuffer);
        }
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__, bResult);

    return bResult;
}


/*****************************************************************************
*
*   bIsAlertTypeExist
*
*****************************************************************************/
static BOOLEAN bIsAlertTypeExist (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertType
        )
{
    BOOLEAN bAlertTypeExist = FALSE;

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] start\n", __FUNCTION__);

    if ((hConnection != SQL_INTERFACE_INVALID_OBJECT) && (pacBuffer != NULL) &&
        (psAlertType != NULL))
    {
        int iReturn;
        UN32 un32RowCount = 0;

        iReturn = snprintf(pacBuffer, tBufferSize,
            SAFECAM_GET_COUNT_FOR_ALERT_TYPE, psAlertType->un8Type
                );
        if (iReturn > 0)
        {
            // Find out how many rows (tables) I will be receiving
            SQL_INTERFACE.bQuery(hConnection, pacBuffer,
                (SQL_QUERY_RESULT_HANDLER)bDBRowCountHandler, (void*)&un32RowCount
                    );
            if (un32RowCount > 0)
            {
                bAlertTypeExist = TRUE;
            }
            else
            {
                bAlertTypeExist = FALSE;
            }
        }
    }

    printf(SAFECAM_MGR_OBJECT_NAME": [%s] end with result %u\n", __FUNCTION__,
        bAlertTypeExist
            );

    return bAlertTypeExist;
}
