/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:RFD_INTERFACE implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms.h"
#include "sms_obj.h"

#include "rfd_receiver.h"
#include "rfd_file_consumer.h"
#include "rfd_msg_collector.h"

#include "rfd_interface_obj.h"
#include "_rfd_interface_obj.h"

#include "string_obj.h"
#include "db_util.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
 *
 *       RFD_INTERFACE_hConnect
 *
 *       This API is used to create a new RFD connection for a data service.
 *       If this is the first connection created for RFD, then the shared object
 *       will be created first and the RFD SDK will be started.
 *
 *       Inputs:
 *           un8ClientId - The ID of this client as defined in
 *              rfd_receiver_config.h
 *           tCurrentVersion - The current version of the client's DB
 *           eProcessFile - A callback that is used to process an RFD file and
 *               convert it to the client's preferred format.
 *           pvProcessFileArg - The pointer to an optional argument
 *               provided to the process file callback
 *
 *       Outputs:
 *           A valid RFD_INTERFACE_OBJECT on success.
 *           RFD_INTERFACE_INVALID_OBJECT if error.
 *
 *****************************************************************************/
RFD_INTERFACE_OBJECT RFD_INTERFACE_hConnect(
    UN8 un8ClientId,
    RFD_UPDATE_VERSION tCurrentVersion,
    size_t tVersionBitWidth,
    RFD_FILE_PROCESSOR_CALLBACK eProcessFile,
    RFD_OPTIONAL_CALLBACKS_STRUCT *psCallbacks,
    void *pvProcessFileArg
        )
{
    RFD_LIB_STATUS_ENUM eRFDLibStatus;
    RFD_INTERFACE_OBJECT_STRUCT *psObj =
        (RFD_INTERFACE_OBJECT_STRUCT *)NULL;

    // Verify inputs
    if (eProcessFile == NULL)
    {
        // Error!
        return RFD_INTERFACE_INVALID_OBJECT;
    }

    // Initialize our RFD support
    eRFDLibStatus = eInitRFD();
    if (eRFDLibStatus == RFD_LIB_STATUS_ERROR)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RFD_INTERFACE_OBJECT_NAME": Can't init the RFD Lib");
        return RFD_INTERFACE_INVALID_OBJECT;
    }

    do
    {
        const char *pacClientName;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Grab the name for this client
        pacClientName = pacRFDClientName(un8ClientId);
        if (pacClientName == (const char *)NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": rfd_receiver_config.h not properly configured");
            break;
        }

        // Construct a unique name for the RFD client interface object
        snprintf( &acName[0], sizeof(acName),
                RFD_INTERFACE_OBJECT_NAME":%s", pacClientName);

        // Create connection object (with lock)
        psObj = (RFD_INTERFACE_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(RFD_INTERFACE_OBJECT_STRUCT),
                SMS_INVALID_OBJECT, TRUE);

        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Failed to create RFD interface object");
            break;
        }

        // Save the client id
        psObj->un8ClientId = un8ClientId;

        // We are not yet shutting down
        psObj->bShuttingDown = FALSE;

        // Let the client tell us when they don't want
        // an update.
        psObj->bClientWantsThisRFDUpdate = TRUE;

        // Init optional callbacks
        vInitOptionalCallbacks(psObj, psCallbacks);

        if (eRFDLibStatus == RFD_LIB_STATUS_DISABLED)
        {
            // Remember there's no RFD Lib right now
            psObj->bRFDLibPresent = FALSE;

            // We are not processing messages now
            psObj->bMsgProcessingActive = FALSE;
        }
        else // RFD_LIB_STATUS_ENABLED
        {
            BOOLEAN bOk;
            OSAL_RETURN_CODE_ENUM eReturnCode;
            RFD_STATUS eStatusCode;

            // Mark that this is an active connection
            psObj->bMsgProcessingActive = TRUE;

            // RFD Lib is alive & well
            psObj->bRFDLibPresent = TRUE;

            // Load any status for this client that
            // we may have in the database
            bOk = bLoadClientStatus(psObj);
            if (bOk == FALSE)
            {
                break;
            }

            // Initialize the attributes we use to handle/track
            // file processing progress
            bOk = bInitFileProcessingAttrs(
                psObj, un8ClientId, eProcessFile,
                pvProcessFileArg, tVersionBitWidth);
            if (bOk == FALSE)
            {
                break;
            }

            // Initialize timer event
            psObj->hPollEvent = SMS_INVALID_EVENT_HDL;

            // Construct a unique name for the RFD client timer
            snprintf( &acName[0], sizeof(acName),
                RFD_INTERFACE_OBJECT_NAME":PollTimer:%s", pacClientName);

            // Create the poll timer now
            eReturnCode = OSAL.eTimerCreate(
                &psObj->hPollTimer, &acName[0], vPollTimerExpire,
                &psObj->hPollEvent);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            // Init our client structure with the client index & version
            psObj->tCurVersion = tCurrentVersion;

            // Get the client index handles from the SDK
            // No specification on how to check for errors if these calls
            // fail, so assume that they never fail
            psObj->hCollector = RFD_GetMessageCollectorHandle(
                gpsRFDCtrlObj->hRFDReceiver, un8ClientId);
            psObj->hConsumer = RFD_GetFileConsumerHandle (
                gpsRFDCtrlObj->hRFDReceiver, un8ClientId);

            // Initilize the consumer storage
            eStatusCode = RFD_ConsumerInitializeStorage (psObj->hConsumer);

            if (eStatusCode != RFD_STATUS_OK)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME
                    ": Error calling RFD_ConsumerInitializeStorage for client: %d -- result: %s",
                    un8ClientId, pacRFDStatusText(eStatusCode));
                break;
            }

            // Get the collector's default message config info.
            // We need it to parse the name field from metadata messages
            eStatusCode = RFD_CollectorGetDefaultMessageConfig(
                psObj->hCollector, &psObj->psMsgCfgInfo);
            if (eStatusCode != RFD_STATUS_OK)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME
                    ": Error calling RFD_ConsumerInitializeStorage for client: %s -- result: %s",
                    pacClientName, pacRFDStatusText(eStatusCode));
                break;
            }

            // Lock the control object
            bOk = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj,OSAL_OBJ_TIMEOUT_INFINITE);
            if (bOk == FALSE)
            {
                break;
            }

            // Update the active connection count
            gpsRFDCtrlObj->un32NumActiveConnections++;

            // Done with that
            SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);

            // Create the consumer task now
            if (psObj->hConsumerTask == SMS_INVALID_TASK_HANDLE)
            {
                SMS_TASK_CONFIGURATION_STRUCT sConfig =
                    sRFDIntTaskConfiguration;

                // Construct a unique name for the RFD client interface task
                snprintf( &acName[0], sizeof(acName),
                    RFD_INTERFACE_OBJECT_NAME":%s", pacClientName);

                // Construct the name to be displayed by the SMST
                sConfig.pacName = &acName[0];

                psObj->hConsumerTask = SMST_hInstall(
                    (SMS_OBJECT)psObj,
                    &sConfig,
                    (SMS_OBJECT_EVENT_HANDLER_PROTOTYPE)vEventHandler,
                    &psObj->hEventHdlr);
                if (psObj->hConsumerTask == SMS_INVALID_TASK_HANDLE)
                {
                    break;
                }
            }
        }

        // Memory allocated and handles acquired, we are all set
        return (RFD_INTERFACE_OBJECT)psObj;

    } while (FALSE);

    // Free any memory that may exist
    if (psObj != NULL)
    {
        vUninitObject(psObj);
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return RFD_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
 *
 *   RFD_INTERFACE_vDisconnect
 *
 *   This API is used to close an existing RFD connection for a data service.
 *   If this is the last connection existing for RFD, then the ctrl object
 *   will be destroyed and the RFD SDK will be shut down.
 *
 *   Inputs:
 *       hConnection - A handle to a valid RFD connection that the caller
 *          wishes to shutdown.
 *
 *   Outputs:
 *       Nothing
 *
 *****************************************************************************/
void RFD_INTERFACE_vDisconnect (
    RFD_INTERFACE_OBJECT hConnection
        )
{
    BOOLEAN bValid;

    // Validate the object
    bValid = SMSO_bValid((SMS_OBJECT)hConnection);

    if (bValid == TRUE)
    {
        RFD_INTERFACE_OBJECT_STRUCT *psObj =
            (RFD_INTERFACE_OBJECT_STRUCT *)hConnection;

        if (psObj->bRFDLibPresent == TRUE)
        {
            SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;
            SMS_EVENT_DATA_UNION *puEventData;
            BOOLEAN bPostedAndProcessed;

            // Allocate the shutdown event -- have the RFD
            // consumer task process this event in this
            // context, so we know it's done when this returns
            hEvent = SMSE_hAllocateEvent(
                psObj->hEventHdlr,
                SMS_EVENT_RFD_INTERFACE, &puEventData,
                SMS_EVENT_OPTION_SYNCHRONOUS);
            if (hEvent != SMS_INVALID_EVENT_HDL)
            {
                puEventData->sRFD.eRFDEvent = RFD_INTERFACE_SHUTDOWN;
            }

            // Post the data service event
            bPostedAndProcessed = SMSE_bPostEvent(hEvent);
            if(bPostedAndProcessed == TRUE)
            {
                if (psObj->hConsumerTask != SMS_INVALID_TASK_HANDLE)
                {
                    SMS_TASK_HANDLE hConsumerTask = psObj->hConsumerTask;

                    // Task shutdown itself will destroy the object (psObj)
                    // so there is no need to handle that here. This function
                    // is called from the DSM task context, so it will
                    // request the SMS-Task to be deleted (shutdown). Once the
                    // task is shutdown the last thing it does is destroy the
                    // SMS-Object provided at the time of SMS-task creation.
                    // This means you must not use psObj anymore after this call!
                    psObj->hConsumerTask = SMS_INVALID_TASK_HANDLE;
                    SMST_vUninstall(hConsumerTask);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME": Cannot post shutdown event.");
            }
        }
        else
        {
            vUninitObject(psObj);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   RFD_INTERFACE_bAcquireRFDConnectionsForMFM
 *
 *   This API is used by callers which need to utilize the MFM.  This function
 *   does all of the RFD lib management that RFD_INTERFACE_hConnect performs,
 *   but instead of providing the caller with an RFD_INTERFACE_OBJECT it
 *   gives the caller the raw RFD handles needed for the MFM connection.
 *
 *   Inputs:
 *       un8BaseClientId - The first RFD client ID the caller uses
 *       pasConnections - The array of RFD client structures this
 *                          call will populate.
 *       tNumConnections - The size of the array, and the number of
 *                          RFD clients used.
 *
 *   Outputs:
 *       TRUE on success, FALSE otherwise
 *
 *****************************************************************************/
BOOLEAN RFD_INTERFACE_bAcquireRFDConnectionsForMFM (
    UN8 un8BaseClientId,
    MFM_RFD_CLIENT_STRUCT *pasConnections,
    size_t tNumConnections
        )
{
    BOOLEAN bSuccess = FALSE;
    RFD_LIB_STATUS_ENUM eRFDLibStatus;
    BOOLEAN bLocked;

    // Validate inputs
    if (((MFM_RFD_CLIENT_STRUCT *)NULL == pasConnections) ||
        (0 == tNumConnections))
    {
        return FALSE;
    }

    // Initialize our RFD support
    eRFDLibStatus = eInitRFD();
    if (RFD_LIB_STATUS_ERROR == eRFDLibStatus)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RFD_INTERFACE_OBJECT_NAME": Can't init the RFD Lib");
        return FALSE;
    }

    // Lock the control object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (TRUE == bLocked)
    {
        RFD_STATUS eStatusCode = RFD_STATUS_OK;
        UN8 un8Index;

        // Update the active connection count
        gpsRFDCtrlObj->un32NumActiveConnections++;

        // Make all connections to RFD now
        for (un8Index = 0; un8Index < tNumConnections; un8Index++)
        {
            // Get the client index handles from the SDK
            // No specification on how to check for errors if these calls
            // fail, so assume that they never fail

            // Collector handle
            pasConnections[un8Index].rfdMessageCollectorHandle =
                RFD_GetMessageCollectorHandle(
                gpsRFDCtrlObj->hRFDReceiver,
                un8BaseClientId + un8Index);

            // Consumer handle
            pasConnections[un8Index].rfdFileConsumerHandle =
                RFD_GetFileConsumerHandle (
                gpsRFDCtrlObj->hRFDReceiver,
                un8BaseClientId + un8Index);

            // Initialize the consumer storage
            eStatusCode = RFD_ConsumerInitializeStorage(
                pasConnections[un8Index].rfdFileConsumerHandle);

            if (eStatusCode != RFD_STATUS_OK)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME
                    ": Error calling RFD_ConsumerInitializeStorage for client: %d -- result: %s",
                    un8BaseClientId + un8Index,
                    pacRFDStatusText(eStatusCode));
                break;
            }
        }

        // Did that work?
        if (eStatusCode == RFD_STATUS_OK)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Yes, now we add this to our MFM connection
            // tracking list -- the connection is identified
            // by the first client ID in the array
            eReturnCode = OSAL.eLinkedListAdd(
                gpsRFDCtrlObj->hMFMConnections,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)(size_t)un8BaseClientId);
            if (OSAL_SUCCESS == eReturnCode)
            {
                bSuccess = TRUE;
            }
        }

        // Done with that
        SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);
    }

    if (FALSE == bSuccess)
    {
        // Clean up this mess now
        vUninitRFD();
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   RFD_INTERFACE_vReleaseRFDConnectionsForMFM
 *
 *   This API is used by callers which utilize the MFM when they are ready to
 *   drop their RFD connections (typically this happens during shutdown).
 *   This function does all of the RFD lib management that
 *   RFD_INTERFACE_vDisconnect performs.
 *
 *   Inputs:
 *       un8BaseClientId - The first RFD client ID the caller uses
 *
 *   Outputs:
 *       None
 *
 *****************************************************************************/
void RFD_INTERFACE_vReleaseRFDConnectionsForMFM (
    UN8 un8BaseClientId
        )
{
    BOOLEAN bLocked, bUninitRFD = FALSE;

    // Lock the control object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj,OSAL_OBJ_TIMEOUT_INFINITE);
    if (FALSE == bLocked)
    {
        return;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Search the list for this connection (identified by the first client id)
        eReturnCode = OSAL.eLinkedListSearch(
            gpsRFDCtrlObj->hMFMConnections,
            &hEntry,
            (void *)(size_t)un8BaseClientId);

        if (OSAL_SUCCESS != eReturnCode)
        {
            // Can't find this connection!  Stop here
            break;
        }

        // Remove this connection from the list
        eReturnCode = OSAL.eLinkedListRemove(hEntry);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Can't remove this connection!  Stop here
            break;
        }

        // We may now uninit RFD
        bUninitRFD = TRUE;

    } while (FALSE);

    // All done
    SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);

    if (TRUE == bUninitRFD)
    {
        // We can now bring down RFD
        vUninitRFD();
    }

    return;
}

/*****************************************************************************
 *
 *   RFD_INTERFACE_bProcessPayload
 *
 *   This API is used to provide an RFD payload to the RFD SDK for processing
 *   by the specified client connection.  The OSAL buffer is always freed
 *   by this API.
 *
 *   Inputs:
 *       hConnection - A handle to a valid RFD connection that the payload
 *          is meant for
 *       hPayload -- An OSAL Buffer containing the complete Access Unit payload
 *           (including PVN, CarID and CRC) that should be processed by the
 *           RFD SDK.
 *
 *   Outputs:
 *       TRUE: An attempt was made to provide the payload to RFD
 *       FALSE: Attempt failed due to invalid input
 *
 *****************************************************************************/
BOOLEAN RFD_INTERFACE_bProcessPayload (
    RFD_INTERFACE_OBJECT hConnection,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    if (hPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return FALSE;
    }

    bValid = SMSO_bValid((SMS_OBJECT)hConnection);
    if (bValid == TRUE)
    {
        RFD_INTERFACE_OBJECT_STRUCT *psObj =
            (RFD_INTERFACE_OBJECT_STRUCT *)hConnection;

        // Does this connection want to process messages?
        if (psObj->bRFDLibPresent == TRUE)
        {
            SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;
            SMS_EVENT_DATA_UNION *puEventData;

            // Send a payload event
            hEvent = SMSE_hAllocateEvent(
                psObj->hEventHdlr,
                SMS_EVENT_RFD_INTERFACE, &puEventData,
                SMS_EVENT_OPTION_NONBLOCK);
            if (hEvent != SMS_INVALID_EVENT_HDL)
            {
                puEventData->sRFD.eRFDEvent = RFD_INTERFACE_EVENT_PROCESS_PAYLOAD;
                puEventData->sRFD.hPayload = hPayload;

                // Post the data service event
                bPosted = SMSE_bPostEvent(hEvent);
                if(bPosted == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        "DATASERVICE_MGR_bPostEvent() Cannot post event.");
                }
            }
        }
    }
    // Don't return an error if the connection is simply
    // invalid.  Just trash the payload gracefully.  If
    // SMSO_bValid failed because the object is actually
    // invalid that'll still be an error and isn't something
    // we want to mask
    else if (hConnection == RFD_INTERFACE_INVALID_OBJECT)
    {
        bValid = TRUE;
    }

    if (bPosted == FALSE)
    {
        // Free the payload since we are done with it
        OSAL.eBufferFree(hPayload);
    }

    return bValid;
}

/*****************************************************************************
 *
 *   RFD_INTERFACE_bReportProgress
 *
 *   This API is used to indicate to RFD the current status of RFD
 *   update file processing for a particular client / file pir.
 *
 *   Inputs:
 *       hConnection - The RFD connection to reporting progress
 *       tCurrentProgress - How far the client has gotten
 *
 *   Outputs:
 *       Returns TRUE if report was successfully processed, FALSE
 *       on error.
 *
 *****************************************************************************/
BOOLEAN RFD_INTERFACE_bReportProgress (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROGRESS_INDEX tCurrentProgress
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hConnection);
    if (bOwner == TRUE)
    {
        RFD_INTERFACE_OBJECT_STRUCT *psObj =
            (RFD_INTERFACE_OBJECT_STRUCT *)hConnection;
        char acCommand[RFD_INTERFACE_UPDATE_PROGRESS_LEN];
        BOOLEAN bCtrlLocked;

        SMSE_vLog(
            psObj->hEventHdlr,
            RFD_INTERFACE_OBJECT_NAME
            ": Client Progress Update: Last: %u, Cur: %u\n",
            psObj->psProcess->sProgressRow.tProgressIndex, tCurrentProgress);

        // Save the progress in memory
        psObj->psProcess->sProgressRow.tProgressIndex = tCurrentProgress;

        do
        {
            bCtrlLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
            if (bCtrlLocked == FALSE)
            {
                break;
            }

            // Start the transaction to update our tracking
            bSuccess = SQL_INTERFACE.bStartTransaction(gpsRFDCtrlObj->hSQL);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Generate the command for this update
            snprintf( &acCommand[0], sizeof(acCommand),
                RFD_INTERFACE_UPDATE_IND_PROGRESS_ROW,
                tCurrentProgress,
                psObj->psProcess->sProgressRow.un8ClientId);

            // Update the progress for this client/file
            bSuccess = SQL_INTERFACE.bExecuteCommand(gpsRFDCtrlObj->hSQL, &acCommand[0]);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Close the transaction
            bSuccess = SQL_INTERFACE.bEndTransaction(gpsRFDCtrlObj->hSQL);
            if (bSuccess == FALSE)
            {
                break;
            }

        } while (FALSE);

        SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);
    }

    return bSuccess;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
 *
 *       vInitOptionalCallbacks
 *
 *****************************************************************************/
static void vInitOptionalCallbacks (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    RFD_OPTIONAL_CALLBACKS_STRUCT *psCallbacks
        )
{
    // Populate the callbacks with the defaults
    psObj->sOptCallbacks.vReportExpiredDB = vReportExpiredDBDefault;
    psObj->sOptCallbacks.eUpdateNeeded = eRFDUpdateNeededDefault;

    // Load in the provided callbacks if they were provided
    if (psCallbacks != NULL)
    {
        if (psCallbacks->vReportExpiredDB != NULL)
        {
            psObj->sOptCallbacks.vReportExpiredDB = psCallbacks->vReportExpiredDB;
        }

        if (psCallbacks->eUpdateNeeded != NULL)
        {
            psObj->sOptCallbacks.eUpdateNeeded = psCallbacks->eUpdateNeeded;
        }
    }

    return;
}


/*****************************************************************************
 *
 *       bInitRFD
 *
 *****************************************************************************/
static RFD_LIB_STATUS_ENUM eInitRFD(void)
{
    RFD_LIB_STATUS_ENUM eReturn = RFD_LIB_STATUS_ERROR;
    BOOLEAN bSMSLocked = FALSE;

    if (FALSE) // For future enhancement
    {
        return RFD_LIB_STATUS_DISABLED;
    }

    /////////////////////////////////////////////////////
    bSMSLocked = SMS_bLock();
    if (bSMSLocked == TRUE)
    {
        if (gpsRFDCtrlObj == NULL)
        {
            // Create the shared object, which will be locked upon return
            gpsRFDCtrlObj = psCreateCtrlObject();
            if (gpsRFDCtrlObj != NULL)
            {
                BOOLEAN bOk;

                // No need to lock SMS anymore, we have ctrl lock now
                SMS_vUnLock();

                // Shared object is created and still in the locked state.
                // We can now safely start the RFD Library
                bOk = bStartRFD(gpsRFDCtrlObj);
                if (bOk == TRUE)
                {
                    // Unlock the ctrl object now
                    SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);
                    eReturn = RFD_LIB_STATUS_ENABLED;
                }
                else
                {
                    // Lock SMS to avoid conflict, then uninit the
                    // control object.
                    bSMSLocked = SMS_bLock();
                    if (bSMSLocked == TRUE)
                    {
                        vUninitCtrlObject(gpsRFDCtrlObj);
                        gpsRFDCtrlObj = NULL;
                        SMS_vUnLock();
                    }
                }
            }
            else
            {
                // Unlock SMS
                SMS_vUnLock();

                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME": Can't create ctrl object");
            }
        }
        else
        {
            // Object already created. Nothing to do.

            // Unlock SMS
            SMS_vUnLock();

            eReturn = RFD_LIB_STATUS_ENABLED;
        }
    }
    /////////////////////////////////////////////////////
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RFD_INTERFACE_OBJECT_NAME": Can't lock SMS object");
    }

    return eReturn;
}

/*****************************************************************************
*
*   bStartRFD
*
*   This private function starts the RFD SDK.
*
*   This function assumes that it is being called from a context where the
*   shared object has already been locked.  This is typically called
*   after calling bCreateSharedObject and SMS_vUnLock.
*
*****************************************************************************/
static BOOLEAN bStartRFD (
    RFD_INTERFACE_CTRL_OBJECT_STRUCT *psCtrl
        )
{
    RFD_STATUS eStatusCode;
    BOOLEAN bDirExists, bOk;
    const char *pacSMSPath;
    char *pcDBPath = NULL;
    UN8 un8FileAttributes = 0;
    char acRFDRootPath[RFD_MAX_PATH_LEN];

    do
    {
        // Build the base paths for RFD
        pacSMSPath = SMS_pacGetPath();
        if (pacSMSPath == NULL)
        {
            return FALSE;
        }

        // Set the base paths for RFD.  If snprintf fails, then
        // later RFD SDK code will fail as well, so we are ignoring
        // the return values
        snprintf( &acRFDRootPath[0], sizeof(acRFDRootPath),
            "%s/"RFD_BASE_DIR, pacSMSPath);
        snprintf( &GacBasePathName[0], sizeof(GacBasePathName),
            "%s/"RFD_WORKING_DIR, &acRFDRootPath[0]);
        snprintf( &GacConsumerBasePathName[0], sizeof(GacConsumerBasePathName),
            "%s/"RFD_CONSUMER_DIR, &acRFDRootPath[0]);

        // Check to see if our directories exist
        bDirExists = OSAL.bFileSystemGetFileAttributes(
            &acRFDRootPath[0], &un8FileAttributes);

        if (bDirExists == FALSE)
        {
            bDirExists = OSAL.bFileSystemMakeDir(&acRFDRootPath[0]);

            if (bDirExists == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME
                    ": Failed to make RFD Root directory: %s",
                    &acRFDRootPath[0]);
            }
        }

        bDirExists = OSAL.bFileSystemGetFileAttributes(
            &GacBasePathName[0], &un8FileAttributes);

        if (bDirExists == FALSE)
        {
            bDirExists = OSAL.bFileSystemMakeDir(&GacBasePathName[0]);

            if (bDirExists == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME
                    ": Failed to make RFD Base directory: %s",
                    &GacBasePathName[0]);
            }
        }

        bDirExists = OSAL.bFileSystemGetFileAttributes(
            &GacConsumerBasePathName[0], &un8FileAttributes);

        if (bDirExists == FALSE)
        {
            bDirExists = OSAL.bFileSystemMakeDir(&GacConsumerBasePathName[0]);

            if (bDirExists == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME
                    ": Failed to make RFD Consumer directory: %s",
                    &GacConsumerBasePathName[0]);
            }
        }

        // Open the receiver
        eStatusCode = RFD_OpenReceiver(&psCtrl->hRFDReceiver);

        if (eStatusCode != RFD_STATUS_OK)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Error calling RFD_OpenReceiver: %s",
                pacRFDStatusText(eStatusCode));
            break;
        }

        // Start the receiver
        eStatusCode = RFD_StartReceiver(psCtrl->hRFDReceiver);

        if (eStatusCode != RFD_STATUS_OK)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Error calling RFD_StartReceiver: %s",
                pacRFDStatusText(eStatusCode));
            break;
        }

        // Create the file path to our database
        bOk = DB_UTIL_bCreateFilePath(
            (const char *)NULL, // We don't know the base path
            RFD_BASE_DIR,
            RFD_INTERFACE_DB_NAME, &pcDBPath);
        if (bOk == FALSE)
        {
            break;
        }

        // Connect to the progress tracking DB
        psCtrl->hSQL =
            DB_UTIL_hConnectToPersistent(
                &pcDBPath[0],
                bCreateDBTables, psCtrl,
                bVerifyDBVersion, NULL,
                (DATASERVICE_ERROR_CODE_ENUM *)NULL);

        // Always destroy the path
        SMSO_vDestroy((SMS_OBJECT)pcDBPath);
        pcDBPath = (char *)NULL;

        if (psCtrl->hSQL == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

       // Leave the ctrl object locked so that the connection creation
       // will unlock it

       return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vUninitRFD
*
*   This private API handles the teardown of the RFD library.
*   This must be called after the last connection has been closed.
*
*****************************************************************************/
static void vUninitRFD(void)
{
    BOOLEAN bLocked;

    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // Decrement our connections count if possible
        if (gpsRFDCtrlObj->un32NumActiveConnections > 0)
        {
            gpsRFDCtrlObj->un32NumActiveConnections--;
        }

        // Get rid of the ctrl object if it has been
        // created and we have no active connections
        if (gpsRFDCtrlObj->un32NumActiveConnections == 0)
        {
            BOOLEAN bSMSLocked;

            vUninitCtrlObject(gpsRFDCtrlObj);

            /////////////////////////////////////////////////////
            bSMSLocked = SMS_bLock();
            if (bSMSLocked == TRUE)
            {
                gpsRFDCtrlObj = NULL;
                SMS_vUnLock();
            }
            /////////////////////////////////////////////////////
        }
        else
        {
            SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   psCreateCtrlObject
*
*   This private API handles the creation of the RFD control object that
*   manages the RFD library itself.  This must be called before any
*   connection has been created.
*
*   This function assumes that it is being called from a context where the main
*   SMS object has been locked using SMS_bLock() to prevent any conflicts.
*****************************************************************************/
static RFD_INTERFACE_CTRL_OBJECT_STRUCT *psCreateCtrlObject( void )
{
    RFD_INTERFACE_CTRL_OBJECT_STRUCT *psCtrl =
        (RFD_INTERFACE_CTRL_OBJECT_STRUCT *)NULL;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Create shared memory
        psCtrl = (RFD_INTERFACE_CTRL_OBJECT_STRUCT *)
           SMSO_hCreate(
               RFD_INTERFACE_OBJECT_NAME,
               sizeof(RFD_INTERFACE_CTRL_OBJECT_STRUCT),
               SMS_INVALID_OBJECT,
               TRUE);

       if ((RFD_INTERFACE_CTRL_OBJECT_STRUCT *)NULL == psCtrl)
       {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
               RFD_INTERFACE_OBJECT_NAME": Error creating RFD control object");
           break;
       }

       // Create the MFM connection tracking list --
       // all entries in this list must be unique
       eReturnCode = OSAL.eLinkedListCreate(
           &psCtrl->hMFMConnections,
           RFD_INTERFACE_OBJECT_NAME": MFM Connections",
           (OSAL_LL_COMPARE_HANDLER)NULL,
           OSAL_LL_OPTION_UNIQUE);
       if (OSAL_SUCCESS != eReturnCode)
       {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
               RFD_INTERFACE_OBJECT_NAME": Error creating MFM list");
           break;
       }

       // All is well so we can provide
       // the caller with the (locked) control object
       return psCtrl;

    } while (FALSE);

    if (NULL != psCtrl)
    {
        // Free the control object if possible
        SMSO_vDestroy((SMS_OBJECT)psCtrl);
    }

    // Return NULL to the caller when we fail
    return (RFD_INTERFACE_CTRL_OBJECT_STRUCT *)NULL;
}

/*****************************************************************************
 *
 *       vUninitCtrlObject
 *
 *****************************************************************************/
static void vUninitCtrlObject (
    RFD_INTERFACE_CTRL_OBJECT_STRUCT *psCtrl
        )
{
    RFD_STATUS eStatusCode;

    // Stop the receiver
    eStatusCode = RFD_StopReceiver(psCtrl->hRFDReceiver);

    if (eStatusCode != RFD_STATUS_OK)
    {
       SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
           RFD_INTERFACE_OBJECT_NAME": Error calling RFD_StopReceiver: %s",
           pacRFDStatusText(eStatusCode));
    }

    // Close the receiver
    eStatusCode = RFD_CloseReceiver(psCtrl->hRFDReceiver);

    if (eStatusCode != RFD_STATUS_OK)
    {
       SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
           RFD_INTERFACE_OBJECT_NAME": Error calling RFD_CloseReceiver: %s",
           pacRFDStatusText(eStatusCode));
    }

    // Disconnect from SQL
    if (psCtrl->hSQL != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psCtrl->hSQL);
        psCtrl->hSQL = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Destroy the MFM connection list (we don't
    // do a remove all here because we are
    // guaranteed that the list is empty at this point)
    OSAL.eLinkedListDelete(psCtrl->hMFMConnections);

    // Destroy the object
    SMSO_vDestroy((SMS_OBJECT)psCtrl);

    return;
}

/*****************************************************************************
 *
 *       vUninitObject
 *
 *****************************************************************************/
static void vUninitObject (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    vUnInitFileProcessingAttrs(psObj);

    if (psObj->hPollTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eTimerDelete(psObj->hPollTimer);
        psObj->hPollTimer = OSAL_INVALID_OBJECT_HDL;
    }

    // We are no longer processing messages
    psObj->bMsgProcessingActive = FALSE;

    // And we're shutting down now
    psObj->bShuttingDown = TRUE;

    // Now, pull down RFD
    vUninitRFD();

    return;
}

/*****************************************************************************
*
*   vEventHandler
*
*****************************************************************************/
static void vEventHandler (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    SMS_EVENT_HDL hEvent
        )
{
    BOOLEAN bLocked;
    SMS_EVENT_TYPE_ENUM eEventType;
    SMS_EVENT_DATA_UNION const *puEventData;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        return;
    }

    // Extract event information
    // Caution! Do not pass puEventData by reference to any other function
    // unless you are absolutely sure you do not end up re-allocating the
    // current event, modify the event data and continue using data from the
    // original event. If you need to do this, first copy what is needed
    // to a local variable and use that instead. This will prevent any
    // issues with potential re-allocation/overwriting of the current event.
    eEventType = SMSE_eGetEvent(hEvent, &puEventData);

    // Process event...
    switch(eEventType)
    {
        case SMS_EVENT_INITIALIZE:
        {
            BOOLEAN bProcessingFile;

            // PRE-ALLOCATE EVENT (SMS_EVENT_RFD_INTERFACE)
            if(psObj->hPollEvent == SMS_INVALID_EVENT_HDL)
            {
                SMS_EVENT_DATA_UNION *puNewEventData;

                // Pre-allocate poll event
                psObj->hPollEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
                    SMS_EVENT_RFD_INTERFACE, &puNewEventData,
                    SMS_EVENT_OPTION_STATIC);
                if (psObj->hPollEvent != SMS_INVALID_EVENT_HDL)
                {
                    // Populate event
                    puNewEventData->sRFD.eRFDEvent =
                        RFD_INTERFACE_EVENT_POLL_FILE;
                }
            }

            // Load any current progress we may
            // already be tracking
            bProcessingFile = bLoadCurrentProgress(psObj);

            // Do we already have an RFD file to process?
            if (bProcessingFile == TRUE)
            {
                BOOLEAN bStarted;

                // Attempt to get file processing underway
                bStarted = bStartFileProcessing(psObj->psProcess, psObj->hEventHdlr);

                if (bStarted == TRUE)
                {
                    // We have better things to do now
                    psObj->bMsgProcessingActive = FALSE;
                }
                else
                {
                    SMSE_vLog(
                        psObj->hEventHdlr,
                        RFD_INTERFACE_OBJECT_NAME": Failed to start file processing");

                    if (psObj->psProcess->eStatus == RFD_PROCESS_STATUS_BEGIN)
                    {
                        // We failed to start file processing, but that's not the fault
                        // of the file, so here we just stop file processing and leave
                        // the FS alone
                        vEndFileProcessing(psObj->psProcess);
                    }
                    else
                    {
                        // File handle is not valid. Need to recollect 
                        // the file anew
                        vResetAll(psObj, TRUE);
                    }
                }
            }
            else
            {
                // Start the poll timer
                vStartPollTimer(psObj);
            }
        }
        break;

        case SMS_EVENT_RFD_INTERFACE:
        {
            switch (puEventData->sRFD.eRFDEvent)
            {
                case RFD_INTERFACE_SHUTDOWN:
                {
                    vUninitObject(psObj);
                }
                break;

                case RFD_INTERFACE_EVENT_PROCESS_PAYLOAD:
                {
                    const OSAL_BUFFER_HDL hPayload =
                        puEventData->sRFD.hPayload;

                    vHandlePayloadEvent(psObj, hPayload);
                }
                break;

                case RFD_INTERFACE_EVENT_POLL_FILE:
                {
                    vHandlePollEvent(psObj);
                }
                break;

                case RFD_INTERFACE_EVENT_PROCESS_FILE:
                {
                    vHandleProcessEvent(psObj);
                }
                break;
            }
        }
        break;

        default:
        {
        }
        break;
    }

    // Unlock object, any modifications to be made to this object
    // have already been made.
    SMSO_vUnlock((SMS_OBJECT)psObj);

    return;
}

/*****************************************************************************
*
*   bHandleNewFileEvent
*
*****************************************************************************/
static BOOLEAN bHandleNewFileEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    RFD_STATUS eStatusCode;
    BOOLEAN bResetRFD = FALSE;

    do
    {
        RFD_INTERFACE_FILE_TRANSFER_STRUCT sTransfer;

        sTransfer.bFileReady = FALSE;
        sTransfer.bSuccess = FALSE;
        sTransfer.psObj = psObj;

        // Transfer the file now in the context of
        // an RFD transaction
        eStatusCode = RFD_ConsumerTransferNewFileExt (
            psObj->hConsumer,
            (RFD_CONSUMER_FILE_TRANSFER_CALLBACK)vRFDFileTransferCallback,
            &sTransfer);
        if (eStatusCode != RFD_STATUS_OK)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": File transfer from RFD failed: %s",
                pacRFDStatusText(eStatusCode));

            if (eStatusCode != RFD_STATUS_ERROR_MEM_ALLOC)
            {
                // Don't reset RFD if this failed because of
                // a lack of memory
                bResetRFD = TRUE;
            }

            break;
        }

        if (sTransfer.bSuccess == FALSE)
        {
            // Well, the software or FS is broken, so we won't
            // get to process this file
            break;
        }

        // Tell the caller if the file was ready
        return sTransfer.bFileReady;

    } while (FALSE);

    if (TRUE == bResetRFD)
    {
        // Finalize file processing for this file, and
        // indicate a retry is needed
        vFinalizeFileProcessing(psObj, TRUE);
    }

    // Something went wrong in accessing this file
    // from the RFD lib -- oh well
    vClearDirectory(psObj->psProcess);

    return FALSE;
}

/*****************************************************************************
*
*   vRFDFileTransferCallback
*
*****************************************************************************/
static void vRFDFileTransferCallback (
    const TCHAR acFileName[],
    RFD_CONSUMER_FILE_INFO_STRUCT *psFileInfo,
    RFD_INTERFACE_FILE_TRANSFER_STRUCT *psTransfer
        )
{
    // If this callback is invoked it means the file is ready
    psTransfer->bFileReady = TRUE;

    // Log the transfer
    SMSE_vLog(psTransfer->psObj->hEventHdlr,
        RFD_INTERFACE_OBJECT_NAME
        ": Transfer file \"%s\" from RFD\n",
        psFileInfo->name);

    // Initialize progress tracking for this client/file
    psTransfer->bSuccess = bStartProgressTracking(
        psTransfer->psObj, psFileInfo,
        psTransfer->psObj->tFileVersion, &acFileName[0]);

    if (psTransfer->bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Failed to start progress tracking");
    }

    return;
}

/*****************************************************************************
*
*   vHandlePayloadEvent
*
*****************************************************************************/
static void vHandlePayloadEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    if (hPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return;
    }

    do
    {
        size_t tPayloadSize, tBytesPeeked;
        RFD_STATUS eStatusCode;
        BOOLEAN bMessageNeeded;

        if ((psObj->bMsgProcessingActive == FALSE) ||
            (psObj->bShuttingDown == TRUE))
        {
            // Just trash this payload
            break;
        }

        // Copy payload to our shared buffer
        tPayloadSize = OSAL.tBufferGetSize(hPayload);
        tBytesPeeked = OSAL.tBufferPeek(hPayload,
            &psObj->aun8PayloadBuffer[0],tPayloadSize, 0);

        // Did we copy the right amount?
        if (tBytesPeeked != tPayloadSize)
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Didn't copy the whole payload during eProcessPayload");
            break;
        }

        // Determine if this message is needed by determining
        // if the update associated with it is needed
        bMessageNeeded = bThisUpdateNeeded(psObj, tPayloadSize);
        if (bMessageNeeded == FALSE)
        {
            // Nope - just trash this payload
            break;
        }

        // Check to see if we need this payload
        eStatusCode = RFD_CollectorIsMessageCollectable(
            psObj->hCollector,
            &psObj->aun8PayloadBuffer[0],
            tPayloadSize);

        if (eStatusCode == RFD_STATUS_OK)
        {
            eStatusCode = RFD_CollectorPutMessage(
                psObj->hCollector,
                &psObj->aun8PayloadBuffer[0],
                tPayloadSize);

            if (eStatusCode != RFD_STATUS_OK)
            {
                // This is worth adding to the log
                SMSE_vLog(psObj->hEventHdlr,
                    RFD_INTERFACE_OBJECT_NAME": Error copying payload to RFD Lib: %s\n",
                    pacRFDStatusText(eStatusCode));

                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME": Error copying payload to RFD Lib: %s\n",
                    pacRFDStatusText(eStatusCode));
                break;
            }
        }
        else if (eStatusCode != RFD_STATUS_FILE_SET_COMPLETE)
        {
            SMSE_vLog(psObj->hEventHdlr,
                RFD_INTERFACE_OBJECT_NAME": RFD Lib rejected payload: %s\n",
                pacRFDStatusText(eStatusCode));
        }

    } while (FALSE);

    // Free the payload since we are done with it
    OSAL.eBufferFree(hPayload);

    return;
}

/*****************************************************************************
*
*   bThisUpdateNeeded
*
*   This function determines if we need the RFD update file associated with
*   this message. We say this update isn't needed if:
*   1) We can't parse this message using the RFD SDK APIs
*   2) The client's callback indicates this is an update that we don't want
*       based on information in the metadata message
*   3) Scenario 2 has occurred and this is a block message (the block
*       message carousel and the metadata carousel are tightly coupled)
*
*****************************************************************************/
static BOOLEAN bThisUpdateNeeded (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    size_t tPayloadSize
        )
{
    BOOLEAN bUpdateNeeded = FALSE;

    do
    {
        RFD_STATUS eStatusCode;
        BOOL bMessageIsFileMetadata = (BOOL)FALSE;
        RFD_FSET_MDATA_STRUCT sFileSetMetadata;
        TCHAR *ptName;
        RFD_FILE_STATUS_ENUM eFileStatus = RFD_FILE_STATUS_INVALID;

        // Get the RFD SDK to determine if this is
        // an update file metadata message
        eStatusCode = RFD_CollectorIsUpdateFileMetadataMessage(
            psObj->psMsgCfgInfo,
            &psObj->aun8PayloadBuffer[0],
            tPayloadSize, &bMessageIsFileMetadata);
        if (eStatusCode != RFD_STATUS_OK)
        {
            // RFD SDK can't parse this message!
            break;
        }

        // Is this a metadata message?
        if (bMessageIsFileMetadata != (BOOL)TRUE)
        {
            // No, this is a block message.  Use the
            // client's current decision to decide if we
            // want this update
            bUpdateNeeded = psObj->bClientWantsThisRFDUpdate;
            break;
        }

        // Parse the RFD Update File Metadata Message into an easy
        // to use structure
        eStatusCode = RFD_CollectorParseUpdateFileMetadataMsg(
            psObj->psMsgCfgInfo,
            &psObj->aun8PayloadBuffer[0],
            tPayloadSize,
            &sFileSetMetadata);
        if (eStatusCode != RFD_STATUS_OK)
        {
            break;
        }

        // Extract the name field
        eStatusCode = RFD_CollectorGetUpdateFileMetadataName(
            &sFileSetMetadata, &ptName);
        if (eStatusCode != RFD_STATUS_OK)
        {
            break;
        }

        eFileStatus = psObj->sOptCallbacks.eUpdateNeeded(
            (RFD_INTERFACE_OBJECT)psObj, (const char *)ptName,
            &psObj->tFileVersion,
            psObj->psProcess->tVersionBitWidth,
            psObj->psProcess->pvFileProcessorCallbackArg);
        if (eFileStatus == RFD_FILE_STATUS_APPLICABLE)
        {
            // Processing is possible
            bUpdateNeeded = TRUE;
        }
        else if (eFileStatus == RFD_FILE_STATUS_EXPIRED_DB)
        {
            // Report expired DB state to service
            psObj->sOptCallbacks.vReportExpiredDB((RFD_INTERFACE_OBJECT)psObj,
                psObj->psProcess->pvFileProcessorCallbackArg
                    );
        }

        // The client has made their decision for this
        // RFD update, so remember what it was
        psObj->bClientWantsThisRFDUpdate = bUpdateNeeded;

    } while (FALSE);

    return bUpdateNeeded;
}

/*****************************************************************************
*
*   vHandlePollEvent
*
*****************************************************************************/
static void vHandlePollEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    RFD_STATUS eStatusCode;
    BOOLEAN bNewFileReady = FALSE;

    if (psObj->bShuttingDown == TRUE)
    {
        // Stop here
        return;
    }

    // See if the file is there (no waiting)
    eStatusCode =
        RFD_ConsumerWaitForNewFileEvent(psObj->hConsumer, 0);

    if (eStatusCode == RFD_STATUS_OK)
    {
        // Handle the event and see if we
        // have a file ready for processing
        bNewFileReady = bHandleNewFileEvent(psObj);

        if (bNewFileReady == TRUE)
        {
            BOOLEAN bStarted;

            // Attempt to get file processing underway
            bStarted = bStartFileProcessing(psObj->psProcess, psObj->hEventHdlr);
            if (bStarted == TRUE)
            {
                // We have better things to do now
                psObj->bMsgProcessingActive = FALSE;
            }
            else
            {
                SMSE_vLog(
                    psObj->hEventHdlr,
                    RFD_INTERFACE_OBJECT_NAME": Failed to start file processing");

                if (psObj->psProcess->eStatus == RFD_PROCESS_STATUS_BEGIN)
                {
                    // Update file is valid so we will try to repeat 
                    // processing after restart
                    vEndFileProcessing(psObj->psProcess);
                }
                else
                {
                    // File handle is not valid. Need to recollect 
                    // the file anew
                    vResetAll(psObj, TRUE);
                }
            }
        }
    }
    else if (eStatusCode == RFD_STATUS_OBJECT_TIMEOUT)
    {
        // Try again later
        vStartPollTimer(psObj);
    }
    else
    {
        SMSE_vLog(
            psObj->hEventHdlr,
            RFD_INTERFACE_OBJECT_NAME
            ": RFD_ConsumerWaitForNewFileEvent error: %s",
            pacRFDStatusText(eStatusCode));
    }

    return;
}

/*****************************************************************************
*
*   vHandleProcessEvent
*
*****************************************************************************/
static void vHandleProcessEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    RFD_PROCESS_RESULT_ENUM eResult;
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess =
        psObj->psProcess;
    BOOLEAN bResetRFD = FALSE;

    if (psObj->bShuttingDown == TRUE)
    {
        return;
    }

    // Process the next chunk of the file
    eResult = psProcess->eFileProcessorCallback(
        psProcess->hRFDConnection,
        psProcess->eStatus,
        psProcess->psRFDFile,
        psProcess->sProgressRow.tFileVersion,
        psProcess->sProgressRow.tProgressIndex,
        psProcess->pvFileProcessorCallbackArg);
    if (eResult >= RFD_PROCESS_RESULT_ERROR)
    {
        // How many errors have we experienced?
        psProcess->sProgressRow.un8NumIntAttempts++;

        SMSE_vLog(
            psObj->hEventHdlr,
            RFD_INTERFACE_OBJECT_NAME": File Processing Error (%u attempts)\n",
            psProcess->sProgressRow.un8NumIntAttempts);

        // Did we hit our limit for errors while processing a file?
        if (psProcess->sProgressRow.un8NumIntAttempts >= RFD_INTERFACE_MAX_FILE_ATTEMPTS)
        {
            // Yeah, just call this complete
            eResult = RFD_PROCESS_RESULT_COMPLETE;

            // Determine if we want to reset RFD to retry this file
            if (psObj->sClientRow.bRFDRetry == FALSE)
            {
                // Yeah, have RFD retry now
                bResetRFD = TRUE;
            }
        }
        else
        {
            BOOLEAN bOk;

            // Rewind the process in memory & in the DB
            // so we can try again from the beginning
            bOk = bRewindProgress(psProcess);
            if (bOk == FALSE)
            {
                SMSE_vLog(
                    psObj->hEventHdlr,
                    RFD_INTERFACE_OBJECT_NAME": Unable to update DB");
            }

            // We're done with this for now
            vEndFileProcessing(psProcess);
        }
    }

    if (eResult == RFD_PROCESS_RESULT_COMPLETE)
    {
        SMSE_vLog(
            psObj->hEventHdlr,
            RFD_INTERFACE_OBJECT_NAME
            ": File Processing Complete\n");

        // Clear all record of this happening
        vResetAll(psObj, bResetRFD);
    }
    else if (eResult == RFD_PROCESS_RESULT_INCOMPLETE)
    {
        // Re-post the event to continue processing
        vResumeFileProcessing(psProcess, psObj->hEventHdlr);
    }

    return;
}

/*****************************************************************************
*
*   pacRFDClientName
*
*   The "working directory" for any RFD client is known as the "consumer
*   directory" in RFD-land.  So, find the appropriate consumer directory
*   given a client ID.
*
*****************************************************************************/
static const char *pacRFDClientName (
    UN8 un8ClientId
        )
{
    const char *pcClientName = (const char *)NULL;

    // Grab the directory name for the client
    switch(un8ClientId)
    {
        case RFDR_CLIENT_0_ID:
        {
            pcClientName = RFD_CLIENT_0_NAME;
        }
        break;

        case RFDR_CLIENT_1_ID:
        {
            pcClientName = RFD_CLIENT_1_NAME;
        }
        break;

        case RFDR_CLIENT_2_ID:
        {
            pcClientName = RFD_CLIENT_2_NAME;
        }
        break;

        case RFDR_CLIENT_3_ID:
        {
            pcClientName = RFD_CLIENT_3_NAME;
        }
        break;

        case RFDR_CLIENT_4_ID:
        {
            pcClientName = RFD_CLIENT_4_NAME;
        }
        break;

        case RFDR_CLIENT_5_ID:
        {
            pcClientName = RFD_CLIENT_5_NAME;
        }
        break;

        case RFDR_CLIENT_6_ID:
        {
            pcClientName = RFD_CLIENT_6_NAME;
        }
        break;

        case RFDR_CLIENT_7_ID:
        {
            pcClientName = RFD_CLIENT_7_NAME;
        }
        break;

        case RFDR_CLIENT_8_ID:
        {
            pcClientName = RFD_CLIENT_8_NAME;
        }
        break;

        case RFDR_CLIENT_9_ID:
        {
            pcClientName = RFD_CLIENT_9_NAME;
        }
        break;
    }

    return pcClientName;
}

/*****************************************************************************
*
*   bLoadClientStatus
*
*****************************************************************************/
static BOOLEAN bLoadClientStatus (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    RFD_INTERFACE_CLIENT_RESULT_STRUCT sClientResult;

    // Initialize result structure
    sClientResult.bDataPresent = FALSE;
    sClientResult.bSuccess = TRUE;
    sClientResult.psObj = psObj;

    do
    {
        char acSQL[RFD_INTERFACE_INSERT_CLIENT_LEN];
        BOOLEAN bLocked, bOk;

        // Lock the control object
        bLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj,OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Build client query
        snprintf( &acSQL[0], sizeof(acSQL),
            RFD_INTERFACE_SELECT_CLIENT_ROW,
            psObj->un8ClientId);

        // Query DB to see if it's in there
        // if not add it
        //bProcessSelectClient
        bOk = SQL_INTERFACE.bQuery(
            gpsRFDCtrlObj->hSQL, &acSQL[0],
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectClient, &sClientResult);

        // Merge these results
        sClientResult.bSuccess &= bOk;

        if (sClientResult.bSuccess == TRUE)
        {
            if (sClientResult.bDataPresent == FALSE)
            {
                // Build client insert command
                snprintf( &acSQL[0], sizeof(acSQL),
                    RFD_INTERFACE_INSERT_CLIENT_ROW,
                    psObj->un8ClientId);

                // Need to insert this client's data now
                sClientResult.bSuccess =
                    SQL_INTERFACE.bExecuteCommand(gpsRFDCtrlObj->hSQL, &acSQL[0]);
            }
        }

        // Done with this object
        SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);
    } while (FALSE);

    return sClientResult.bSuccess;
}

/*****************************************************************************
*
*   bSetWorkingDir
*
*   The "working directory" for any RFD client is known as the "consumer
*   directory" in RFD-land.  So, find the appropriate consumer directory
*   given a client ID.
*
*****************************************************************************/
static BOOLEAN bSetWorkingDir (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        )
{
    const char *pcClientDirName = (const char *)NULL;
    size_t tPathLen;

    // Grab the directory name for the client
    switch(psProcess->sProgressRow.un8ClientId)
    {
        case RFDR_CLIENT_0_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_0_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_1_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_1_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_2_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_2_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_3_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_3_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_4_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_4_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_5_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_5_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_6_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_6_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_7_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_7_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_8_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_8_DIR_NAME;
        }
        break;

        case RFDR_CLIENT_9_ID:
        {
            pcClientDirName = RFD_FILE_CONSUMER_9_DIR_NAME;
        }
        break;

        default:
        {
            // We don't know what this is
            return FALSE;
        }
    }

    // Calculate how long the path is (plus one for
    // separator and one for NULL terminator
    tPathLen = strlen(GacConsumerBasePathName)
        + strlen(pcClientDirName) + 2;

    psProcess->pcWorkingDirectory = (char *)SMSO_hCreate(
        RFD_INTERFACE_OBJECT_NAME": WorkingDir",
        tPathLen, SMS_INVALID_OBJECT, FALSE);
    if (psProcess->pcWorkingDirectory != NULL)
    {
        snprintf( &psProcess->pcWorkingDirectory[0], tPathLen,
                "%s/%s", &GacConsumerBasePathName[0],
                pcClientDirName);

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
 *
 *  bExtractVersionDefault
 *
 *  This API is used to parse a file name from an RFD file to determine
 *  base version of file and file version.  Filenames parsed
 *  by this function must be in the form UXXYY where XX is the version
 *  the file is being upgraded from and YY is the version it will be
 *  upgraded to.
 *
 *  This function is a default which may be overwritten by the client
 *
 *****************************************************************************/
static BOOLEAN bExtractVersionDefault(
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptBaseVersion,
    RFD_UPDATE_VERSION *ptNewVersion
        )
{
    char acBuffer[RFD_UPDATE_VER_STRING_LEN + 1];
    UN8 un8Idx = 0;
    size_t tFileNameStrLen = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        // Verify that the length of pcFileName is 5
        tFileNameStrLen = strlen(pcFileName);

        if (tFileNameStrLen != RFD_UPDATE_FILE_NAME_LEN)
        {
            break;
        }

        // Verify that the first character is 'U'
        if (pcFileName[0] != RFD_UPDATE_FILE_START_CHAR)
        {
            break;
        }

        // Parse the first two characters into a number
        for (un8Idx = 0; un8Idx < RFD_UPDATE_VER_STRING_LEN; un8Idx++)
        {
            // Offsetting un8Idx by one to account for the first char
            acBuffer[un8Idx] = pcFileName[1 + un8Idx];
        }

        acBuffer[RFD_UPDATE_VER_STRING_LEN] = '\0';

        *ptBaseVersion = (RFD_UPDATE_VERSION)
            atoi((const char *)acBuffer);

        // Parse the last two characters into a number
        for (un8Idx = 0; un8Idx < RFD_UPDATE_VER_STRING_LEN; un8Idx++)
        {
            // Offsetting un8Idx by one to account for the first char
            // and the first version number
            acBuffer[un8Idx] =
                pcFileName[1 + RFD_UPDATE_VER_STRING_LEN + un8Idx];
        }

        acBuffer[RFD_UPDATE_VER_STRING_LEN] = '\0';

        *ptNewVersion = (RFD_UPDATE_VERSION)
            atoi((const char *)acBuffer);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  eCompareFileVersionDefault
 *
 *  This function compares the reported file version with the current
 *  version of the connection to determine if the file is needed.
 *
 *  This function is a default which may be overwritten by the client
 *
 *****************************************************************************/
RFD_FILE_STATUS_ENUM RFD_INTERFACE_eCompareFileVersions (
    RFD_UPDATE_VERSION tBaseVersion,
    RFD_UPDATE_VERSION tCurrentVersion,
    RFD_UPDATE_VERSION tNewVersion,
    size_t tVersionBitWidth,
    RFD_INTERFACE_BASELINE_CHECK eCheckBaseline
        )
{
    RFD_FILE_STATUS_ENUM eResult = RFD_FILE_STATUS_INVALID;

    do
    {
        SMS_VERSION_COMPARE_ENUM eCompareResult      = SMS_VERSION_EQUAL;
        BOOLEAN                  bRolloverIndication = FALSE;

        // We'll accept the update if our versions can be
        // ordered as follows: final > current >= base

        // First, we'll compare the old version and the new version

        eCompareResult = SMS_eCompareVersions(
                            tVersionBitWidth,
                            tCurrentVersion,
                            tNewVersion,
                            &bRolloverIndication);

        // So is our current version equal to (or greater)
        // than the new version? Is there also no indication of
        // rollover? If both these conditions are true, exit out
        // since we already processed this update or something like it.

        if ( ( eCompareResult != SMS_VERSION_GREATER ) &&
             ( bRolloverIndication == FALSE ) )
        {
            eResult = RFD_FILE_STATUS_NOT_NEEDED;
            break;
        }

        // If the user in interested in checking against the baseline
        // then do so now.

        if ( eCheckBaseline == RFD_INTERFACE_BASELINE_CHECK_PERFORM )
        {
            // We'll compare the base against the current version
            // to make sure we're not too old for an update.

            eCompareResult = SMS_eCompareVersions(
                                tVersionBitWidth,
                                tBaseVersion,
                                tCurrentVersion,
                                &bRolloverIndication);

            // Is our current version less than the base version?
            // If so, we are a really old receiver that hasn't seen
            // OTA signal in over five years.  Lots has probably changed.
            // There may not be roads anymore or fuel stations.  All the
            // cars could be flying cars.  Either way, we are too out of date
            // to accept this update.  We must sadly let it go and live on
            // in a stale state.

            if ( ( eCompareResult == SMS_VERSION_LESSER ) &&
                 ( bRolloverIndication == FALSE ) )
            {
                eResult = RFD_FILE_STATUS_EXPIRED_DB;
                break;
            }
        }

        // If we've failed to break out of our while loop thus far, then
        // our RFD update is applicable.

        eResult = RFD_FILE_STATUS_APPLICABLE;

    } while (FALSE);

    return eResult;
}

/*****************************************************************************
 *
 *   RFD_INTERFACE_bMFMReportTime
 *
 *   This API is a central implementation of the "report time" MFM callback
 *   that all MFM-using clients need.
 *
 *****************************************************************************/
BOOL RFD_INTERFACE_bMFMReportTime (
    RFD_SXI_TIME_INFO_STRUCT *psSxiTimeInfo,
    void *pvUnused
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    struct tm sUTCTime;
    TIME_T tSecs;
    UN32 un32OSALTimeInSecs;
    BOOLEAN bHaveTime = FALSE;

    do
    {
        // Did the caller give us a non-null pointer?
        if ((RFD_SXI_TIME_INFO_STRUCT *)NULL == psSxiTimeInfo)
        {
            // Nope!
            break;
        }

        // Get current time from OSAL
        eReturnCode = OSAL.eTimeGet(&un32OSALTimeInSecs);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Can't get the time now
            break;
        }

        // Get our time as a tm struct(broken down time)
        tSecs = un32OSALTimeInSecs;
        OSAL.gmtime_r(&tSecs, &sUTCTime);

        // Provide the caller with their data
        psSxiTimeInfo->minute = sUTCTime.tm_min;
        psSxiTimeInfo->hour = sUTCTime.tm_hour;
        psSxiTimeInfo->day = sUTCTime.tm_mday;
        psSxiTimeInfo->month = sUTCTime.tm_mon + 1;

        // Validate the year before we subtract from it
        // (it must be reporting a year beyond 2000, and
        // the year is reported with a base of 1900,
        // so this better be >= 100)
        if (sUTCTime.tm_year < 100)
        {
            // Invalid time!
            break;
        }

        // Substract the appropriate number of years to report
        // in the range requested
        psSxiTimeInfo->yearSince2k = (sUTCTime.tm_year - 100);

        // We have time data for the caller!
        bHaveTime = TRUE;

    }  while (FALSE);

    return bHaveTime;
}

/*****************************************************************************
 *
 *  vReportExpiredDBDefault
 *
 *  This function is just a placeholder no action is taken in this scenario
 *  by default.
 *
 *  This function is a default which may be overwritten by the client
 *
 *****************************************************************************/
static void vReportExpiredDBDefault(
    RFD_INTERFACE_OBJECT hConnection,
    void *pvCallbackArg
        )
{
    return;
}

/*****************************************************************************
 *
 *  eRFDUpdateNeededDefault
 *
 *  This function is the default implementation of the "update needed"
 *  callback and uses the default version extractor function
 *
 *****************************************************************************/
static RFD_FILE_STATUS_ENUM eRFDUpdateNeededDefault (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        )
{
    RFD_FILE_STATUS_ENUM eFileStatus = RFD_FILE_STATUS_INVALID;

    if ( hConnection != RFD_INTERFACE_INVALID_OBJECT )
    {
        RFD_INTERFACE_OBJECT_STRUCT *psObj;
        RFD_UPDATE_VERSION tBaseVersion = 0;
        BOOLEAN bOk = FALSE;

        psObj = (RFD_INTERFACE_OBJECT_STRUCT *)hConnection;

        // Get the version info from the file name using the default
        // version extractor function
        bOk = bExtractVersionDefault(
            pcFileName, &tBaseVersion, ptFileVersion);

        // Did we extract that information without error?
        if ( bOk == TRUE )
        {
            eFileStatus = RFD_INTERFACE_eCompareFileVersions(
                tBaseVersion, psObj->tCurVersion,
                *ptFileVersion, tVersionBitWidth,
                RFD_INTERFACE_BASELINE_CHECK_PERFORM );
        }
    }

    return eFileStatus;
}

/*****************************************************************************
*
*   vResetAll
*
*****************************************************************************/
static void vResetAll (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    BOOLEAN bRetryRFD
        )
{
    // Finalize file processing
    vFinalizeFileProcessing(psObj, bRetryRFD);

    // Don't need this DB row anymore
    vDeleteDBProgressRow(psObj->psProcess->sProgressRow.un8ClientId);

    // Clean up stuff in memory associated with this file
    vEndFileProcessing(psObj->psProcess);

    // Clear all files associated with this client
    vClearDirectory(psObj->psProcess);

    return;
}

/*****************************************************************************
*
*   vFinalizeFileProcessing
*
*****************************************************************************/
static void vFinalizeFileProcessing (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    BOOLEAN bRetryNeeded
        )
{
    char acSQL[RFD_INTERFACE_CLIENT_ROW_LEN];
    BOOLEAN bSuccess, bLocked;

    // Generate the command for this update
    snprintf( &acSQL[0], sizeof(acSQL),
        RFD_INTERFACE_SET_RETRY_CLIENT_ROW,
        bRetryNeeded, psObj->un8ClientId);

    // Write this into the database now
    bLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        // Well, that's that I guess
        return;
    }

    // Update this client's data
    bSuccess = SQL_INTERFACE.bExecuteCommand(gpsRFDCtrlObj->hSQL, &acSQL[0]);

    // All done with this
    SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);

    if ((TRUE == bSuccess) && (TRUE == bRetryNeeded))
    {
        // Only reset RFD if we were able to update our status,
        // this is because a failure to update status would mean
        // that we weren't able to mark the failure and we'd
        // keep retrying the same file forever.
        SMSE_vLog(
            psObj->hEventHdlr,
            RFD_INTERFACE_OBJECT_NAME
            ": Resetting RFD Lib to recapture file\n");

        // Don't worry about the return code here
        // since there's nothing we can do about it
        RFD_ConsumerResetFileCollectionRequest(psObj->hConsumer);

        // Update the client row data now
        psObj->sClientRow.bRFDRetry = TRUE;
    }

    return;
}

/*****************************************************************************
*
*   vClearDirectory
*
*****************************************************************************/
static void vClearDirectory (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hDirItems = OSAL_INVALID_OBJECT_HDL;
    char acFile[RFD_MAX_PATH_LEN];

    // Get the list of all the items in the directory
    eReturnCode = OSAL.eFileSystemGetDirItems(
        &hDirItems, psProcess->pcWorkingDirectory,
        (OSAL_FS_FILE_QUALIFIER)bFileQualifier, NULL);
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        const char *pcEntry;

        // Now, iterate all items to remove them
        hEntry = OSAL.hLinkedListFirst(hDirItems, (void **)&pcEntry);

        while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
               (pcEntry != NULL))
        {
            // Prepend the working directory to the file name
            snprintf( &acFile[0], RFD_MAX_PATH_LEN,
                "%s/%s", &psProcess->pcWorkingDirectory[0],
                &pcEntry[0] );

            // Remove this file
            remove(&acFile[0]);

            // Move to the next entry
            hEntry = OSAL.hLinkedListNext(hEntry, (void **)&pcEntry);
        }

        // Release our dir items list
        OSAL.eFileSystemReleaseDirItems(hDirItems);
    }

    return;
}

/*****************************************************************************
*
*   bFileQualifier
*
*****************************************************************************/
static BOOLEAN bFileQualifier (
    const char *pcItemName,
    const void *pvArg
        )
{
    // Just say yes to all files
    return TRUE;
}

/*****************************************************************************
*
*   vDeleteDBProgressRow
*
*****************************************************************************/
static void vDeleteDBProgressRow (
    UN8 un8ClientId
        )
{
    BOOLEAN bSuccess, bLocked;
    char acDelete[RFD_INTERFACE_CLEAR_PROGRESS_LEN];

    bLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return;
    }

    do
    {
        // Start the transaction to update our tracking
        bSuccess = SQL_INTERFACE.bStartTransaction(gpsRFDCtrlObj->hSQL);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Generate the command for this update
        snprintf( &acDelete[0], sizeof(acDelete),
            RFD_INTERFACE_CLEAR_PROGRESS_ROW,
            un8ClientId);

        // Delete the progress for this client/file
        bSuccess = SQL_INTERFACE.bExecuteCommand(gpsRFDCtrlObj->hSQL, &acDelete[0]);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Close the transaction
        bSuccess = SQL_INTERFACE.bEndTransaction(gpsRFDCtrlObj->hSQL);
        if (bSuccess == FALSE)
        {
            break;
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);

    return;
}

/*****************************************************************************
*
*   vClearMemoryProgressRow
*
*****************************************************************************/
static void vClearMemoryProgressRow (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        )
{
    // Leave the client index alone!

    // Clear these guys...
    if (psProcess->sProgressRow.hFilePath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psProcess->sProgressRow.hFilePath);
        psProcess->sProgressRow.hFilePath = STRING_INVALID_OBJECT;
    }

    psProcess->sProgressRow.tFileVersion = 0;
    psProcess->sProgressRow.tProgressIndex = RFD_INVALID_PROGRESS_INDEX;
    psProcess->sProgressRow.un8NumIntAttempts = 0;

    return;
}

/*****************************************************************************
*
*   vSetMemoryProgressRow
*
*****************************************************************************/
static void vSetMemoryProgressRow (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess,
    RFD_INTERFACE_PROGRESS_ROW_STRUCT *psProgressRow
        )
{
    vClearMemoryProgressRow(psProcess);

    psProcess->sProgressRow.hFilePath = psProgressRow->hFilePath;
    psProcess->sProgressRow.tFileVersion = psProgressRow->tFileVersion;
    psProcess->sProgressRow.tProgressIndex = psProgressRow->tProgressIndex;
    psProcess->sProgressRow.un8NumIntAttempts = psProgressRow->un8NumIntAttempts;

    return;
}

/*****************************************************************************
*
*   bStartProgressTracking
*
*****************************************************************************/
static BOOLEAN bStartProgressTracking (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    RFD_CONSUMER_FILE_INFO_STRUCT *psInfo,
    RFD_UPDATE_VERSION tUpdateFileVersion,
    const char *pcRFDFilePath
        )
{
    BOOLEAN bSuccess = FALSE, bLocked = FALSE;
    RFD_INTERFACE_PROGRESS_ROW_STRUCT sProgressRow;

    // Initialize this attribute
    sProgressRow.hFilePath = STRING_INVALID_OBJECT;

    do
    {
        RFD_INTERFACE_PROCESS_FILE_DESC *psProcess = psObj->psProcess;

        // Create a string from the provided path argument
        sProgressRow.hFilePath = STRING.hCreate(pcRFDFilePath, strlen(pcRFDFilePath));

        if (sProgressRow.hFilePath == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Populate the rest of the progress row
        sProgressRow.tFileVersion = tUpdateFileVersion;
        sProgressRow.tProgressIndex = 0;
        sProgressRow.un8NumIntAttempts = 0;

        // Save this state in memory
        vSetMemoryProgressRow(psProcess, &sProgressRow);

        // This object has been handed over -- if we
        // experience an error in the code below
        // make sure we don't free this in response
        sProgressRow.hFilePath = STRING_INVALID_OBJECT;

        // Lock the CTRL object to update the DB
        bLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Start the transaction to update the DB
        bSuccess = SQL_INTERFACE.bStartTransaction(gpsRFDCtrlObj->hSQL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME
                ": failed to start transaction");
            break;
        }

        // Update the progress table to include this file
        bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
            gpsRFDCtrlObj->hSQL,
            RFD_INTERFACE_INSERT_PROGRESS_ROW,
            RFD_INTERFACE_DB_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareProgressColumn,
            (void *)&psProcess->sProgressRow);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME
                ": failed to update progress table");
            break;
        }

        // Is this actually a retry of a previously
        // failed RFD file?  Check the version to find out
        if (psObj->sClientRow.tRetryVersion != psProcess->sProgressRow.tFileVersion)
        {
            char acSQL[RFD_INTERFACE_CLIENT_ROW_LEN];

            // No, this is a different file.  Reset the Client row now

            // Generate the command for this update
            snprintf( &acSQL[0], sizeof(acSQL),
                RFD_INTERFACE_SET_CLIENT_ROW,
                psProcess->sProgressRow.tFileVersion,
                psProcess->sProgressRow.un8ClientId);

            // Update the client table as well now
            bSuccess = SQL_INTERFACE.bExecuteCommand(
                gpsRFDCtrlObj->hSQL, &acSQL[0]);

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME
                    ": failed to update client table");
                break;
            }
        }
        else
        {
            SMSE_vLog(
                psObj->hEventHdlr,
                RFD_INTERFACE_OBJECT_NAME
                ": This file is being retried from an earlier attempt\n");
        }

        // Always end the transaction
        bSuccess = SQL_INTERFACE.bEndTransaction(gpsRFDCtrlObj->hSQL);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME
                ": failed to end transaction");
            break;
        }

    } while (FALSE);

    if (sProgressRow.hFilePath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sProgressRow.hFilePath);
    }

    if (bLocked == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bLoadCurrentProgress
*
*****************************************************************************/
static BOOLEAN bLoadCurrentProgress (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    RFD_INTERFACE_PROGRESS_RESULT_STRUCT sProgressResult;

    // Initialize progress structure row
    sProgressResult.bSuccess = TRUE;
    sProgressResult.bDataPresent = FALSE;

    // Lock the Ctrl object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        char acQuery[RFD_INTERFACE_SELECT_PROGRESS_LEN];
        BOOLEAN bSuccess;

        // Initialize progress row
        OSAL.bMemSet(&sProgressResult.sProgressRow, 0,
            sizeof(RFD_INTERFACE_PROGRESS_ROW_STRUCT));

        snprintf( &acQuery[0], sizeof(acQuery),
            RFD_INTERFACE_SELECT_PROGRESS_ROW,
            psObj->un8ClientId);

        // Extract the progress data with a query for this client
        bSuccess = SQL_INTERFACE.bQuery(
            gpsRFDCtrlObj->hSQL,
            &acQuery[0],
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectProgress,
            &sProgressResult);

        // Don't need the lock any longer
        SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);

        if (bSuccess == FALSE)
        {
            sProgressResult.bSuccess = FALSE;
        }

        // If we didn't experience an error and we have data available
        if ((sProgressResult.bSuccess == TRUE) &&
            (sProgressResult.bDataPresent == TRUE))
        {
            // Restore our state from the DB
            vSetMemoryProgressRow(psObj->psProcess, &sProgressResult.sProgressRow);
        }
        else if (sProgressResult.bSuccess == FALSE)
        {
            // Clear the DB progress row & directory because we
            // have no idea what's going on here and we'll never
            // recover wherever we may have been in our processing
            // status for this file (if any exists).  So, clear this
            // directory in order to prevent unchecked growth of files here
            // In addition, reset collection for this client
            vDeleteDBProgressRow(psObj->psProcess->sProgressRow.un8ClientId);
            vClearDirectory(psObj->psProcess);
        }
    }

    return sProgressResult.bDataPresent;
}

/*****************************************************************************
*
*   bRewindProgress
*
*****************************************************************************/
static BOOLEAN bRewindProgress(
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        )
{
    BOOLEAN bSuccess = FALSE;
    char acCommand[RFD_INTERFACE_UPDATE_PROGRESS_LEN];
    BOOLEAN bCtrlLocked;

    // "Rewind" progress in memory
    psProcess->sProgressRow.tProgressIndex = 0;

    // Then, in the DB
    do
    {
        bCtrlLocked = SMSO_bLock((SMS_OBJECT)gpsRFDCtrlObj, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bCtrlLocked == FALSE)
        {
            break;
        }

        // Start the transaction to update our tracking
        bSuccess = SQL_INTERFACE.bStartTransaction(gpsRFDCtrlObj->hSQL);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Generate the command for this update
        snprintf( &acCommand[0], sizeof(acCommand),
            RFD_INTERFACE_UPDATE_ATTEMPTS_PROGRESS_ROW,
            psProcess->sProgressRow.un8NumIntAttempts,
            psProcess->sProgressRow.un8ClientId);

        // Update the progress for this client/file
        bSuccess = SQL_INTERFACE.bExecuteCommand(gpsRFDCtrlObj->hSQL, &acCommand[0]);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Close the transaction
        bSuccess = SQL_INTERFACE.bEndTransaction(gpsRFDCtrlObj->hSQL);
        if (bSuccess == FALSE)
        {
            break;
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT)gpsRFDCtrlObj);

    return bSuccess;
}

/*****************************************************************************
*
*   bCreateDBTables
*
*   Creates tables for our database
*
*****************************************************************************/
static BOOLEAN bCreateDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    BOOLEAN bSuccess = FALSE;
    RFD_INTERFACE_CTRL_OBJECT_STRUCT *psObj =
        (RFD_INTERFACE_CTRL_OBJECT_STRUCT *)pvArg;
    char acVersionText[RFD_INTERFACE_INSERT_VERSION_LEN];

    if (psObj == NULL)
    {
        return FALSE;
    }

    // Start a transaction this connection
    bSuccess = SQL_INTERFACE.bStartTransaction(hSQLConnection);

    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RFD_INTERFACE_OBJECT_NAME
            ": failed to start transaction");
        return FALSE;
    }

    do
    {
        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            RFD_INTERFACE_CREATE_CLIENT_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME
                ": failed to create client table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            RFD_INTERFACE_CREATE_PROGRESS_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME
                ": failed to create progress table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            RFD_INTERFACE_CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME
                ": failed to create version table");
            break;
        }

        // Build the version string
        snprintf(&acVersionText[0],
            RFD_INTERFACE_INSERT_VERSION_LEN,
            RFD_INTERFACE_INSERT_VERSION_ROW,
            RFD_INTERFACE_DB_VERSION);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection, &acVersionText[0]);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME
                ": failed to insert version row");
            break;
        }
    }
    while (FALSE);

    bSuccess &= SQL_INTERFACE.bEndTransaction(hSQLConnection);

    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            RFD_INTERFACE_OBJECT_NAME
            ": failed to end transaction");
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bVerifyDBVersion
*
*****************************************************************************/
static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    BOOLEAN bVersionMatched = FALSE, bOk;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
             hSQLConnection, RFD_INTERFACE_SELECT_ALL_DB_VERSION,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectVersion,
            &bVersionMatched ) ;

    if (bOk == FALSE)
    {
        bVersionMatched = FALSE;
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   bProcessSelectVersion
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.
*
*******************************************************************************/
static BOOLEAN bProcessSelectVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbVersionMatched
        )
{
    // Verify input
    if (pbVersionMatched == NULL)
    {
        return FALSE;
    }

    // Initialize input
    *pbVersionMatched = FALSE;

    // If we have the correct number of columns, then we have good results
    if ( (n32NumberOfColumns == 1) &&
         (psColumn->eType == SQL_COLUMN_TYPE_INTEGER) )
    {
        UN8 un8Version;

        // Grab the first row's data
        un8Version = (UN8)psColumn->uData.sUN32.un32Data;
        if (un8Version == RFD_INTERFACE_DB_VERSION)
        {
            // If the versions match, we're god
            *pbVersionMatched = TRUE;
        }
    }

    // Just want a single row
    return FALSE;
}

/*****************************************************************************
*
*   bProcessSelectClient
*
*****************************************************************************/
static BOOLEAN bProcessSelectClient (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    RFD_INTERFACE_CLIENT_RESULT_STRUCT *psResult
        )
{
    // Verify input
    if (psResult == NULL)
    {
        // Stop now
        return FALSE;
    }

    // If we have the correct number of columns, then we have good results
    if ( n32NumberOfColumns != RFD_INTERFACE_DB_CLIENT_MAX_FIELDS )
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;
    }
    else
    {
        RFD_INTERFACE_CLIENT_ROW_STRUCT *psClientRow = &psResult->psObj->sClientRow;

        // Start off by marking this a success
        psResult->bSuccess = TRUE;

        // We were able to get a row of data
        psResult->bDataPresent = TRUE;

        // Grab the retry flag
        psClientRow->bRFDRetry =
            (BOOLEAN)psColumn[RFD_INTERFACE_DB_CLIENT_RFD_RETRY].uData.sUN32.un32Data;

        // And then the retry version
        psClientRow->tRetryVersion =
            (RFD_UPDATE_VERSION)psColumn[RFD_INTERFACE_DB_CLIENT_RETRY_VER].uData.sUN32.un32Data;
    }

    // Just want a single row
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectProgress
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database progress relation.
*
*******************************************************************************/
static BOOLEAN bProcessSelectProgress (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    RFD_INTERFACE_PROGRESS_RESULT_STRUCT *psResult
        )
{
    RFD_INTERFACE_DB_FIELDS_ENUM eCurrentField =
        (RFD_INTERFACE_DB_FIELDS_ENUM)0;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If we have the correct number of columns, then we have good results
    if ( n32NumberOfColumns == RFD_INTERFACE_DB_MAX_FIELDS )
    {
        // Start off by marking this a success
        psResult->bSuccess = TRUE;
    }
    else
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case RFD_INTERFACE_DB_CLIENT_ID:
            {
                psResult->sProgressRow.un8ClientId =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case RFD_INTERFACE_DB_FILENAME:
            {
                psResult->sProgressRow.hFilePath =
                    STRING.hCreate(
                        (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                        strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (psResult->sProgressRow.hFilePath == STRING_INVALID_OBJECT)
                {
                    // Error!
                    return FALSE;
                }
            }
            break;

            case RFD_INTERFACE_DB_FILEVER:
            {
                psResult->sProgressRow.tFileVersion =
                    (RFD_UPDATE_VERSION)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case RFD_INTERFACE_DB_PROGRESS_INDEX:
            {
                psResult->sProgressRow.tProgressIndex =
                    (RFD_PROGRESS_INDEX)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case RFD_INTERFACE_DB_NUM_INT_ATTEMPTS:
            {
                psResult->sProgressRow.un8NumIntAttempts =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    RFD_INTERFACE_OBJECT_NAME": bad field in client table");
            }
            break;
        }

    } while ( ++eCurrentField < RFD_INTERFACE_DB_MAX_FIELDS);

    // We were able to get a row of data
    psResult->bDataPresent = TRUE;

    // Just want a single row
    return FALSE;
}

/*******************************************************************************
*
*   bPrepareProgressColumn
*
*******************************************************************************/
static BOOLEAN bPrepareProgressColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    RFD_INTERFACE_PROGRESS_ROW_STRUCT *psProgressRow
        )
{
    BOOLEAN bSuccess = TRUE;

    switch (tIndex)
    {
        case RFD_INTERFACE_DB_CLIENT_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psProgressRow->un8ClientId;
        }
        break;

        case RFD_INTERFACE_DB_FILENAME:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psProgressRow->hFilePath;
        }
        break;

        case RFD_INTERFACE_DB_FILEVER:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psProgressRow->tFileVersion;
        }
        break;

        case RFD_INTERFACE_DB_PROGRESS_INDEX:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psProgressRow->tProgressIndex;
        }
        break;

        case RFD_INTERFACE_DB_NUM_INT_ATTEMPTS:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psProgressRow->un8NumIntAttempts;
        }
        break;

        case RFD_INTERFACE_DB_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*******************************************************************************
*
*   bStartFileProcessing
*
*******************************************************************************/
static BOOLEAN bStartFileProcessing (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess,
    SMS_EVENT_HANDLER hEventHdlr
        )
{
    BOOLEAN bPosted;
    const char *pcRFDFile;
    SMS_EVENT_DATA_UNION *puNewEventData;
    SMS_EVENT_HDL hProcessEvent;

    do
    {
        if (psProcess->sProgressRow.hFilePath == STRING_INVALID_OBJECT)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Invalid file path");
            break;
        }

        // Extract the character array
        pcRFDFile = STRING.pacCStr(psProcess->sProgressRow.hFilePath);
        if (pcRFDFile == NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Unable to extract data from STRING");
            break;
        }

        // Open the file
        psProcess->psRFDFile = fopen(pcRFDFile, RFD_FILE_MODE);
        if (psProcess->psRFDFile == (FILE *)NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Unable to open RFD file");
            break;
        }

        // Get an event ready
        hProcessEvent = SMSE_hAllocateEvent(
            hEventHdlr, SMS_EVENT_RFD_INTERFACE, &puNewEventData,
            SMS_EVENT_OPTION_NONE
                );

        if (hProcessEvent == SMS_INVALID_EVENT_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                RFD_INTERFACE_OBJECT_NAME": Unable to alloc event");
            break;
        }

        // We are going to begin processing
        psProcess->eStatus = RFD_PROCESS_STATUS_BEGIN;

        // Setup the event now
        puNewEventData->sRFD.eRFDEvent = RFD_INTERFACE_EVENT_PROCESS_FILE;
        puNewEventData->sRFD.hPayload = OSAL_INVALID_BUFFER_HDL;// this is not used

        // Send the event now
        bPosted = SMSE_bPostEvent(hProcessEvent);
        if (bPosted == FALSE)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    if (psProcess->psRFDFile != NULL)
    {
        fclose(psProcess->psRFDFile);
        psProcess->psRFDFile = (OSAL_FILE_STRUCT*)NULL;
    }

    return FALSE;
}

/*******************************************************************************
*
*   vResumeFileProcessing
*
*******************************************************************************/
static void vResumeFileProcessing (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess,
    SMS_EVENT_HANDLER hEventHdlr
        )
{
    SMS_EVENT_DATA_UNION *puNewEventData;
    SMS_EVENT_HDL hProcessEvent;

    hProcessEvent = SMSE_hAllocateEvent(
        hEventHdlr, SMS_EVENT_RFD_INTERFACE, &puNewEventData,
        SMS_EVENT_OPTION_NONE);
    if (hProcessEvent != SMS_INVALID_EVENT_HDL)
    {
        BOOLEAN bPosted;

        // Tell the client we're continuing to process a file
        psProcess->eStatus = RFD_PROCESS_STATUS_CONTINUE;

        puNewEventData->sRFD.eRFDEvent = RFD_INTERFACE_EVENT_PROCESS_FILE;
        puNewEventData->sRFD.hPayload = OSAL_INVALID_BUFFER_HDL;//this is not used

        // Send the event now
        bPosted = SMSE_bPostEvent(hProcessEvent);
        if (bPosted == FALSE)
        {
            printf(RFD_INTERFACE_OBJECT_NAME
                ": Cannot post RFD_INTERFACE_EVENT_PROCESS_FILE event\n");
        }
    }

    return;
}

/*******************************************************************************
*
*   vEndFileProcessing
*
*******************************************************************************/
static void vEndFileProcessing (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        )
{
    // Tell the client we've stopped now
    if ((psProcess->eFileProcessorCallback != NULL) &&
        (psProcess->pvFileProcessorCallbackArg != NULL))
    {
        // Tell the client we're stopping
        psProcess->eFileProcessorCallback(
            psProcess->hRFDConnection,
            RFD_PROCESS_STATUS_STOP,
            (FILE *)NULL, 0,0,
            psProcess->pvFileProcessorCallbackArg);

        // We no longer need these
        psProcess->eFileProcessorCallback = NULL;
        psProcess->pvFileProcessorCallbackArg = NULL;
    }

    // Close the file
    if (psProcess->psRFDFile != NULL)
    {
        fclose(psProcess->psRFDFile);
        psProcess->psRFDFile = NULL;
    }

    // Clear the progress memory
    vClearMemoryProgressRow(psProcess);

    // Reset status
    psProcess->eStatus = RFD_PROCESS_STATUS_STOP;

    return;
}

/*******************************************************************************
*
*   bInitFileProcessingAttrs
*
*******************************************************************************/
static BOOLEAN bInitFileProcessingAttrs (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    UN8 un8ClientId,
    RFD_FILE_PROCESSOR_CALLBACK eProcessFile,
    void *pvProcessFileArg,
    size_t tVersionBitWidth
        )
{
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess;
    BOOLEAN bOk;

    do
    {
        // Create the process argument
        psProcess = (RFD_INTERFACE_PROCESS_FILE_DESC *)
                SMSO_hCreate(
                    RFD_INTERFACE_OBJECT_NAME":EventArg",
                    sizeof(RFD_INTERFACE_PROCESS_FILE_DESC),
                    (SMS_OBJECT)psObj,
                    FALSE);
        if (psProcess == NULL)
        {
            break;
        }

        // Store all necessary attributes
        psProcess->hRFDConnection = (RFD_INTERFACE_OBJECT)psObj;
        psProcess->eFileProcessorCallback = eProcessFile;
        psProcess->pvFileProcessorCallbackArg = pvProcessFileArg;
        psProcess->eStatus = RFD_PROCESS_STATUS_STOP;
        psProcess->psRFDFile = (FILE *)NULL;
        psProcess->tVersionBitWidth = tVersionBitWidth;

        // Initialize the in-memory row
        vClearMemoryProgressRow(psProcess);

        // Then set the client index
        psProcess->sProgressRow.un8ClientId = un8ClientId;

        // Set the working directory for this connection
        bOk = bSetWorkingDir(psProcess);
        if (bOk == FALSE)
        {
            break;
        }

        // Track the object now
        psObj->psProcess = psProcess;

        return TRUE;
    } while (FALSE);

    return FALSE;
}

/*******************************************************************************
*
*   vUnInitFileProcessingAttrs
*
*******************************************************************************/
static void vUnInitFileProcessingAttrs (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    if (psObj->psProcess != (RFD_INTERFACE_PROCESS_FILE_DESC *)NULL)
    {
        // Clean up whatever is necessary
        vEndFileProcessing(psObj->psProcess);

        // Clear the working directory
        if (psObj->psProcess->pcWorkingDirectory != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->psProcess->pcWorkingDirectory);
            psObj->psProcess->pcWorkingDirectory = NULL;
        }

        // Remove RFD interface handle
        psObj->psProcess->hRFDConnection = RFD_INTERFACE_INVALID_OBJECT;

        SMSO_vDestroy((SMS_OBJECT)psObj->psProcess);

        // Clear handle from object
        psObj->psProcess = (RFD_INTERFACE_PROCESS_FILE_DESC *)NULL;
    }

    return;
}

/*******************************************************************************
*
*   vStartPollTimer
*
*******************************************************************************/
static void vStartPollTimer (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Start the timer now
    eReturnCode = OSAL.eTimerStartRelative(
        psObj->hPollTimer,
        RFD_INTERFACE_FILE_POLL_RATE, 0);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSE_vLog(
            psObj->hEventHdlr,
            RFD_INTERFACE_OBJECT_NAME
            ": Failed to start timer: %s",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return;
}

/*******************************************************************************
*
*   vPollTimerExpire
*
*******************************************************************************/
static void vPollTimerExpire (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    SMS_EVENT_HDL *phEvent = (SMS_EVENT_HDL *)pvArg;

    // Post this event now
    SMSE_bPostEvent(*phEvent);

    return;
}

/*******************************************************************************
*
*   pacRFDStatusText
*
*******************************************************************************/
static const char *pacRFDStatusText (
    RFD_STATUS eStatus
        )
{
    const char *pacReturnString;

    switch (eStatus)
    {
        case RFD_STATUS_OK:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OK);
        break;

        case RFD_STATUS_OK_DECODE_COMPLETE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OK_DECODE_COMPLETE);
        break;

        case RFD_STATUS_OK_MESSAGE_RECEIVED:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OK_MESSAGE_RECEIVED);
        break;

        case RFD_STATUS_OK_MESSAGE_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OK_MESSAGE_TIMEOUT);
        break;

        case RFD_STATUS_OK_MAX_SEMAPHORE_COUNT:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OK_MAX_SEMAPHORE_COUNT);
        break;

        case RFD_STATUS_OK_BUF_FULL:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OK_BUF_FULL);
        break;

        case RFD_STATUS_OK_MSG_OVERSIZED:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OK_MSG_OVERSIZED);
        break;

        case RFD_STATUS_ERROR_MESSAGE_QUEUE_EMPTY:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_MESSAGE_QUEUE_EMPTY);
        break;

        case RFD_STATUS_ERROR_MESSAGE_QUEUE_CREATE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_MESSAGE_QUEUE_CREATE);
        break;

        case RFD_STATUS_EARLY_EXIT_ON_REQUEST:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_EARLY_EXIT_ON_REQUEST);
        break;

        case RFD_STATUS_UNSUPPORTED_BLOCK_TYPE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_UNSUPPORTED_BLOCK_TYPE);
        break;

        case RFD_STATUS_UNSUPPORTED_CODE_TYPE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_UNSUPPORTED_CODE_TYPE);
        break;

        case RFD_STATUS_UNSUPPORTED_COMPRESSION_TYPE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_UNSUPPORTED_COMPRESSION_TYPE);
        break;

        case RFD_STATUS_UNSUPPORTED_FEATURE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_UNSUPPORTED_FEATURE);
        break;

        case RFD_STATUS_ERROR_GENERAL:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_GENERAL);
        break;

        case RFD_STATUS_ERROR_BUFFER_SIZE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_BUFFER_SIZE);
        break;

        case RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_PARAM_OUT_OF_RANGE);
        break;

        case RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_FSET_NUM_FILES);
        break;

        case RFD_STATUS_ERROR_INVALID_CODE_TYPE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_CODE_TYPE);
        break;

        case RFD_STATUS_ERROR_INVALID_COMP_TYPE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_COMP_TYPE);
        break;

        case RFD_STATUS_ERROR_INVALID_BLOCK_TYPE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_BLOCK_TYPE);
        break;

        case RFD_STATUS_ERROR_INVALID_BLOCK_INDEX:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_BLOCK_INDEX);
        break;

        case RFD_STATUS_ERROR_INVALID_BLOCK_LENGTH:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_BLOCK_LENGTH);
        break;

        case RFD_STATUS_ERROR_INVALID_FSET_BAUDOT_NAME:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_FSET_BAUDOT_NAME);
        break;

        case RFD_STATUS_ERROR_INVALID_FILE_SIZE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_FILE_SIZE);
        break;

        case RFD_STATUS_ERROR_INVALID_MESSAGE_LENGTH:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_MESSAGE_LENGTH);
        break;

        case RFD_STATUS_ERROR_INVALID_MESSAGE_TYPE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_MESSAGE_TYPE);
        break;

        case RFD_STATUS_ERROR_MEM_ALLOC:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_MEM_ALLOC);
        break;

        case RFD_STATUS_ERROR_FILE_OPEN:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_OPEN);
        break;

        case RFD_STATUS_ERROR_FILE_FIND:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_FIND);
        break;

        case RFD_STATUS_ERROR_FILE_READ:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_READ);
        break;

        case RFD_STATUS_ERROR_FILE_WRITE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_WRITE);
        break;

        case RFD_STATUS_ERROR_FILE_SEEK:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_SEEK);
        break;

        case RFD_STATUS_ERROR_FILE_SIZE_CHECK:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_SIZE_CHECK);
        break;

        case RFD_STATUS_ERROR_FILE_RENAME:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_RENAME);
        break;

        case RFD_STATUS_ERROR_FILE_DELETE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_DELETE);
        break;

        case RFD_STATUS_ERROR_FILE_LEN:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_LEN);
        break;

        case RFD_STATUS_ERROR_TRANSACTION_IN_PROGRESS:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_TRANSACTION_IN_PROGRESS);
        break;

        case RFD_STATUS_ERROR_INVALID_STORAGE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_STORAGE);
        break;

        case RFD_STATUS_ERROR_DIRECTORY_CREATE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_DIRECTORY_CREATE);
        break;

        case RFD_STATUS_ERROR_DECODER_STATE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_DECODER_STATE);
        break;

        case RFD_STATUS_ERROR_DECODER_BLK_LEN:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_DECODER_BLK_LEN);
        break;

        case RFD_STATUS_ERROR_DECODER_EQU_LEN:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_DECODER_EQU_LEN);
        break;

        case RFD_STATUS_ERROR_DECODER_NUM_INPUT_BLKS:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_DECODER_NUM_INPUT_BLKS);
        break;

        case RFD_STATUS_ERROR_DECODER_INSUFFICIENT_MEMORY:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_DECODER_INSUFFICIENT_MEMORY);
        break;

        case RFD_STATUS_ERROR_DECODER_GENERAL:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_DECODER_GENERAL);
        break;

        case RFD_STATUS_ERROR_NUM_COLLECTED_BLKS:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_NUM_COLLECTED_BLKS);
        break;

        case RFD_STATUS_ERROR_PIVOT_NUM:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_PIVOT_NUM);
        break;

        case RFD_STATUS_ERROR_INVALID_VALUE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_VALUE);
        break;

        case RFD_STATUS_ERROR_OBJECT_ABANDONED:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_OBJECT_ABANDONED);
        break;

        case RFD_STATUS_OBJECT_TIMEOUT:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_OBJECT_TIMEOUT);
        break;

        case RFD_STATUS_ERROR_MUTEX_RELEASE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_MUTEX_RELEASE);
        break;

        case RFD_STATUS_ERROR_CREATE_THREAD:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_CREATE_THREAD);
        break;

        case RFD_STATUS_ERROR_SET_THREAD_PRIORITY:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_SET_THREAD_PRIORITY);
        break;

        case RFD_STATUS_ERROR_CREATE_MUTEX:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_CREATE_MUTEX);
        break;

        case RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_BITSTREAM_INIT_ERROR);
        break;

        case RFD_STATUS_ERROR_BITSTREAM_READ_ERROR:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_BITSTREAM_READ_ERROR);
        break;

        case RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_BITSTREAM_WRITE_ERROR);
        break;

        case RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_BITSTREAM_SEEK_ERROR);
        break;

        case RFD_STATUS_ERROR_GZ_OPEN_INPUT:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_GZ_OPEN_INPUT);
        break;

        case RFD_STATUS_ERROR_GZ_OPEN_OUTPUT:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_GZ_OPEN_OUTPUT);
        break;

        case RFD_STATUS_ERROR_GZ_CLOSE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_GZ_CLOSE);
        break;

        case RFD_STATUS_ERROR_GZ_READ:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_GZ_READ);
        break;

        case RFD_STATUS_ERROR_GZ_WRITE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_GZ_WRITE);
        break;

        case RFD_STATUS_ERROR_GZ_FILE_SIZE_ERROR:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_GZ_FILE_SIZE_ERROR);
        break;

        case RFD_STATUS_ERROR_CRC:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_CRC);
        break;

        case RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_OUTPUT_BUFFER_OVERFLOW);
        break;

        case RFD_STATUS_ERROR_INPUT_BUFFER_OVERFLOW:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INPUT_BUFFER_OVERFLOW);
        break;

        case RFD_STATUS_ERROR_BAUDOT_PARSE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_BAUDOT_PARSE);
        break;

        case RFD_STATUS_ERROR_BAUDOT_UNSUPPORTED_CHAR:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_BAUDOT_UNSUPPORTED_CHAR);
        break;

        case RFD_STATUS_MSG_PROTOCOL_VERSION_NUM_OUT_OF_RANGE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_MSG_PROTOCOL_VERSION_NUM_OUT_OF_RANGE);
        break;

        case RFD_STATUS_ERROR_THREAD_EXIT:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_THREAD_EXIT);
        break;

        case RFD_STATUS_FILE_SET_COMPLETE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_FILE_SET_COMPLETE);
        break;

        case RFD_STATUS_BLK_MSG_FILE_ID_NOT_MATCHED:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_STATE);
        break;

        case RFD_STATUS_BLK_MSG_DMI_SOURCE_NOT_MATCHED:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_STATE);
        break;

        case RFD_STATUS_CAROUSEL_ID_NOT_MATCHED:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_STATE);
        break;

        case RFD_STATUS_ERROR_INVALID_STATE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_STATE);
        break;

        case RFD_STATUS_ERROR_INVALID_MODE:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_INVALID_MODE);
        break;

        case RFD_STATUS_ERROR_FILE_SYNC:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_FILE_SYNC);
        break;

        case RFD_STATUS_ERROR_LLIST_FIND:
            pacReturnString = MACRO_TO_STRING(RFD_STATUS_ERROR_LLIST_FIND);
        break;

        default:
        {
            pacReturnString = "Unknown";
        }
    }

    return pacReturnString;
}
