/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:REPORT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <stddef.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "cid_obj.h"
#include "cdo_obj.h"
#include "cme.h"
#include "decoder_obj.h"
#include "ccache.h"
#include "string_obj.h"
#include "cm.h"
#include "sms.h"
#include "radio.h"

#include "report_obj.h"
#include "_report_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pacId
*
* Retries the Market Id as a null terminated string associated with this
* CDO given that it is a REPORT type. The returned string is constant
* and static throughout the life of SMS.
*
*****************************************************************************/
static const char *pacId (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    REPORT_OBJECT_STRUCT *psObj;
    const char *pacId = NULL;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_REPORT)
    {
        // Go get CDO structure from provided CDO
        psObj = (REPORT_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            // return the pacId
            pacId = psObj->psMarket->pacId;
        }
    }

    return pacId;
}

/*****************************************************************************
*
*   pacName
*
* Retrieves the Market Name as a null terminated string associated with this
* CDO given that it is a REPORT type. The returned string is constant
* and static throughout the life of SMS.
*
*****************************************************************************/
static const char *pacName (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    REPORT_OBJECT_STRUCT *psObj;
    const char *pacName = NULL;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_REPORT)
    {
        // Go get CDO structure from provided CDO
        psObj = (REPORT_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            // return the pacName
            pacName = psObj->psMarket->pacName;
        }
    }

    return pacName;
}

/*****************************************************************************
*
*   hId
*
* Retrieves the CID associated with this CDO given that it is a REPORT type.
* The returned CID is the unique identifier for the market this CDO describes.
*
*****************************************************************************/
static CID_OBJECT hId (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    REPORT_OBJECT_STRUCT *psObj;
    CID_OBJECT hId = CID_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_REPORT)
    {
        // Go get CDO structure from provided CDO
        psObj = (REPORT_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            // return the Id
            hId = psObj->hId;
        }
    }

    return hId;
}

/*****************************************************************************
*
*   bIterateContent
*
* Iterates the static REPORT content for all available markets known
* at the time this API is called. The caller provides a callback function
* which will be executed for each market in our list. This list is static
* and available to any caller's context. NOTE: Only VALID markets are
* iterated. Non-valid markets will not be iterated. they will be skipped over.
*
*****************************************************************************/
static BOOLEAN bIterateContent (
    REPORT_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
        )
{
    BOOLEAN bRetval = FALSE, bLocked;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    REPORT_ITERATOR_STRUCT sIterator;

    // Check inputs
    if(bContentIteratorCallback == NULL)
    {
        // Bad param
        return FALSE;
    }
    // Populate a local iterator structure with info needed to call
    // the provided callback to the caller along with their provided argument.
    sIterator.bContentIteratorCallback = bContentIteratorCallback;
    sIterator.pvContentIteratorCallbackArg = pvContentIteratorCallbackArg;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error! Something went wrong.
        return FALSE;
    }

    // Iterate the list and execute caller's callback function for
    // each entry in the list.
    eReturnCode =
        OSAL.eLinkedListIterate(
            gpsReportCtrl->hMarketList,
            (OSAL_LL_ITERATOR_HANDLER)bIterator, &sIterator);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // List was iterated
        bRetval = TRUE;
    }

    // Unlock object
    SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);

    return bRetval;
}

/*****************************************************************************
*
*   bContentComplete
*
*   Returns the flag indicating if all markets have been confirmed.
*
*   Inputs: None
*
*   Outputs: TRUE if all markets confirmed, otherwise FALSE
*
*****************************************************************************/
static BOOLEAN bContentComplete ( void )
{
    BOOLEAN bRetval = FALSE, bLocked;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error! Something went wrong.
        return FALSE;
    }

    bRetval = gpsReportCtrl->bMarketListComplete;

    // Unlock object
    SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);

    return bRetval;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   REPORT_bAdd
*
*   This function assigns a CID to represent this report content. The CID
*   is used to uniquely describe this report and map it to some
*   textual information about the report (e.g. name and abbr).
*
*   Inputs
*       hId - A CID representing this REPORT
*       psMarket - A structure defining a market to associate with the CID
*       bCreate - Flag to indicate if memory needs to be created to hold the
*                 market structure data
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the CID was able to be added.
*       Otherwise returns FALSE if the request resulted if an error occurred.
*
*****************************************************************************/
BOOLEAN REPORT_bAdd (
    CID_OBJECT hId,
    const MARKET_STRUCT *psMarket,
    BOOLEAN bCreate
        )
{
    BOOLEAN bSuccess = FALSE, bLocked;
    REPORT_ENTRY_STRUCT *psReportEntry;

    // Verify a CID and valid association was provided.
    // Without both these is no point in calling this.
    if((hId == CID_INVALID_OBJECT) || (psMarket == NULL))
    {
        // Error!
        return FALSE;
    }

    // Now allocate a structure to hold the CID and market as one mapping.
    // This entry is available for all SMS objects and applications to use.
    psReportEntry = psCreateReportEntry(hId, psMarket, bCreate);
    if(psReportEntry == NULL)
    {
        // Error! Unable to create
        return FALSE;
    }

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Find this CID in our MARKET list (if we can)
        eReturnCode = OSAL.eLinkedListLinearSearch(
            gpsReportCtrl->hMarketList,
            &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16MarketEqualToId,
            hId
                );
        if(eReturnCode == OSAL_SUCCESS)
        {
            REPORT_ENTRY_STRUCT *psOldReportEntry;
            CID_OBJECT hCid;

            psOldReportEntry = ((REPORT_ENTRY_STRUCT *)
                                   OSAL.pvLinkedListThis(hEntry));

            // we already have an entry for this market.

            // is it confirmed?
            if (psOldReportEntry->hDecoderHdl != MARKET_CONFIRMED)
            {
                // no. let's update it.

                hCid = psReportEntry->sReport.hId;
                psReportEntry->sReport.hId = psOldReportEntry->sReport.hId;
                // swap so the unneeded cid gets destroyed when destroying the
                // old report entry
                psOldReportEntry->sReport.hId = hCid;

                // Replace this existing report in our list.
                eReturnCode =
                    OSAL.eLinkedListReplaceEntry(
                        gpsReportCtrl->hMarketList, hEntry,
                        psReportEntry
                            );
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Report that replacing was successful
                    printf(
                        REPORT_OBJECT_NAME
                        " Replaced existing market"
                        " (%s,%s) Updated market list.\n",
                        psReportEntry->sReport.psMarket->pacId,
                        psReportEntry->sReport.psMarket->pacName
                            );

                    // Replacement success
                    if (MARKET_CONFIRMED == psReportEntry->hDecoderHdl)
                    {
                        gpsReportCtrl->un16ConfirmedMarkets++;
                    }

                    vDestroyReportEntry(psOldReportEntry);
                }

                bSuccess = TRUE;
            }
            else
            {
                printf(REPORT_OBJECT_NAME": Market already seen %s\n",
                        psReportEntry->sReport.psMarket->pacId);
                // Replacement unnecessary
                vDestroyReportEntry(psReportEntry);
                bSuccess = TRUE;
            }
        }
        else if(eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // Finally, add this report to our list.
            eReturnCode =
                OSAL.eLinkedListAdd(
                    gpsReportCtrl->hMarketList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psReportEntry);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Success!
                bSuccess = TRUE;
                if (MARKET_CONFIRMED == psReportEntry->hDecoderHdl)
                {
                    gpsReportCtrl->un16ConfirmedMarkets++;
                }
                printf(REPORT_OBJECT_NAME": Market added %s\n",
                    psReportEntry->sReport.psMarket->pacId);
            }
            else if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                // We already found one in our list that looks just like this
                // one. That's ok, we just need to consume the CID we were
                // provided and clean-up resources. But still return ok!
                vDestroyReportEntry(psReportEntry);
                bSuccess = TRUE;
            }
            else
            {
                // Something went wrong with adding this element. No point
                // in keeping the structure we just created.
                psReportEntry->sReport.hId = CID_INVALID_OBJECT;
                vDestroyReportEntry(psReportEntry);
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);

    }

    return bSuccess;
}

/*****************************************************************************
*
*   REPORT_bAssignId
*
*   This function assigns a CID to represent this report content. The CID
*   is used to uniquely describe this report market and map it to some
*   textual information about the market (e.g. name and abbr).
*
*   Inputs
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the CID was able to be assigned and
*       it caused a change in content. Otherwise returns FALSE if the
*       assignment resulted in no change or an error occurred preventing
*       the assignment, in which case it remained unchanged.
*
*****************************************************************************/
BOOLEAN REPORT_bAssignId (
    CD_OBJECT hCDO,
    CID_OBJECT hId,
    BOOLEAN *pbFound
        )
{
    BOOLEAN bChanged = FALSE, bFound = FALSE;
    REPORT_OBJECT_STRUCT *psObj;

    // Go get CDO structure from provided CDO
    psObj = (REPORT_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
    if(psObj != NULL)
    {
        BOOLEAN bLocked;

        // Lock this object
        bLocked = SMSO_bLock(
            (SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            REPORT_ENTRY_STRUCT *psReportEntry = NULL;
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_LINKED_LIST_ENTRY hEntry =
                OSAL_INVALID_LINKED_LIST_ENTRY;
            CID_OBJECT hOldId = psObj->hId;

            // Find this CID in our MARKET list (if we can)
            eReturnCode = OSAL.eLinkedListLinearSearch(
                gpsReportCtrl->hMarketList,
                &hEntry,
                (OSAL_LL_COMPARE_HANDLER)n16MarketEqualToId,
                hId
                    );
            if(eReturnCode == OSAL_SUCCESS)
            {
                BOOLEAN bChanged;

                // Found a match! Replace CDO info with it
                psReportEntry = ((REPORT_ENTRY_STRUCT *)
                    OSAL.pvLinkedListThis(hEntry));
                *psObj = psReportEntry->sReport;
                bFound = TRUE;

                // Inform CME of the content change (report)
                bChanged = CME_bChangeContent(hCDO, hOldId, hId);
                if(bChanged == FALSE)
                {
                    printf( REPORT_OBJECT_NAME": Error updating CID." );
                }
            }

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);
        }
    }

    if (pbFound != NULL)
    {
        *pbFound = bFound;
    }

    return bChanged;
}

/*****************************************************************************
*
*   REPORT_bLoadMarketCidCreator
*
*  This is a function used to load the market cid creator from the radio
*
*****************************************************************************/
BOOLEAN REPORT_bLoadMarketCidCreator (
    MARKET_CID_CREATOR hCreateMarketCid
        )
{
    BOOLEAN bLocked;

    if (hCreateMarketCid == NULL)
    {
        return FALSE;
    }

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        gpsReportCtrl->hCreateMarketCid = hCreateMarketCid;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);
    }

    return bLocked;
}

/*****************************************************************************
*
*   REPORT_hCreateMarketCid
*
*****************************************************************************/
MARKET_CID_CREATOR REPORT_hCreateMarketCid( void )
{
    BOOLEAN bLocked;
    MARKET_CID_CREATOR hMarketCreator = NULL;

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        hMarketCreator = gpsReportCtrl->hCreateMarketCid;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);
    }

    return hMarketCreator;
}

/*****************************************************************************
*
*   REPORT_vLoadMarkets
*
*  This is a function used to load from the config file any
*  previously saved confirmed or seen markets.
*
*****************************************************************************/
void REPORT_vLoadMarkets (
    void
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    size_t tSize;
    N32 n32MktListCompleteTagValue, *pn32TagValue;
    BOOLEAN bLocked;
    TAG_OBJECT hParentTag, hTag = TAG_INVALID_OBJECT;

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error! Something went wrong.
        return;
    }

    do
    {
        // Get the SMS tag. The Market tags are children of the SMS tag
        hParentTag = SMS_hGetTag();

        eResult = TAG_eGet(MARKETS_TAG, hParentTag, &hTag, NULL, TRUE);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            REPORT_MKT_TAG_ITERATOR_STRUCT sIterator;

            // we don't need the parent tag, so we'll just set it to invalid
            sIterator.hParentTag = TAG_INVALID_OBJECT;

            // Initially the buffer will be NULL. The memory will be allocated
            // the first time it is used, all tags iterated can then share it.
            sIterator.tBuffSize = 0;
            sIterator.pacBuff = NULL;

            gpsReportCtrl->hMarketsTag = hTag;
            TAG_eIterateChildren(
                gpsReportCtrl->hMarketsTag,
                bReadMktTag,
                (void *)&sIterator);
            // Done iterating. Free the buffer if it was created.
            if (sIterator.pacBuff != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)sIterator.pacBuff);
            }
        }
        else
        {
            // Couldn't retrieve or create the Markets tag
            break;
        }

        eResult =
            TAG_eGet(MARKET_LIST_COMPLETE_TAG, hParentTag, &hTag, NULL, TRUE);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            gpsReportCtrl->hMarketListCompleteTag = hTag;
        }
        else
        {
            // Couldn't retrieve or create the MarketListComplete tag
            break;
        }

        // Retrieve the market complete value
        pn32TagValue = &n32MktListCompleteTagValue;
        tSize = sizeof(N32);
        eResult = TAG_eGetTagValue(
                      hTag,
                      TAG_TYPE_INTEGER,
                      (void **)&pn32TagValue,
                      &tSize);
        if (eResult == SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET)
        {
            // Set default value
            n32MktListCompleteTagValue = 0;
        }
        else if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error! Can't read tag
            break;
        }

        // assign value
        gpsReportCtrl->bMarketListComplete =
            n32MktListCompleteTagValue == 0 ? FALSE : TRUE;

        eResult = TAG_eGet(MARKETS_RADIO_SPECIFIC,
                           gpsReportCtrl->hMarketsTag,
                           &hTag, NULL, TRUE);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            gpsReportCtrl->hRadioSpecificData = hTag;
        }
        else
        {
            // Couldn't retrieve or create the radio specific tag
            break;
        }

    } while (0);

    // Unlock object
    SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);

    // Dump generated report list (DEBUG only)
    vPrintMarkets(stdout);

    return;
}

/*****************************************************************************
*
*   REPORT_vClear
*
*   This function clears the known markets. Basically this means we
*   no longer have any confirmed markets and must reconfirm them and
*   rebuild our list.
*
*   This function should be called by a DECODER following a Chan Map Update
*
*****************************************************************************/
void REPORT_vClear ( void )
{
    BOOLEAN bLocked;
    N32 n32MarketListComplete;

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error! Something went wrong.
        return;
    }

    // Iterate the list and invalidate each report entry in the list.
    OSAL.eLinkedListIterate(
        gpsReportCtrl->hMarketList,
        (OSAL_LL_ITERATOR_HANDLER)bInvalidate, NULL);

    // since we're clearing out the markets, set the list complete flag to false
    gpsReportCtrl->bMarketListComplete = FALSE;
    gpsReportCtrl->un16ConfirmedMarkets = 0;

    // Set the Market List Complete value
    n32MarketListComplete =
        REPORT_MARKET_LIST_COMPLETE_TO_N32(gpsReportCtrl->bMarketListComplete);

    TAG_eSetTagValue(
        gpsReportCtrl->hMarketListCompleteTag,
        TAG_TYPE_INTEGER,
        (void *)&n32MarketListComplete,
        sizeof(N32),
        FALSE);

    // Iterate the children of the Markets tag, removing each one found.
    TAG_eIterateChildren(
        gpsReportCtrl->hMarketsTag,
        bRemovedMktTag,
        NULL
            );

    // Unlock object
    SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);

    // Dump generated report list (DEBUG only)
    vPrintMarkets(stdout);

    return;
}

/*****************************************************************************
 *
 *   REPORT_hGetRadioSpecificTag
 *
 *****************************************************************************/
TAG_OBJECT REPORT_hGetRadioSpecificTag(void)
{
    BOOLEAN bLocked;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        hTag = gpsReportCtrl->hRadioSpecificData;
        SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);
    }

    return hTag;
}

/*****************************************************************************
 *
 *   REPORT_vAllMarketsFound
 *
 *****************************************************************************/
void REPORT_vAllMarketsFound(void)
{
    BOOLEAN bLocked;

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        gpsReportCtrl->bMarketListComplete = TRUE;
        SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);
    }

    // Save all confirmed/seen markets
    vSaveMarkets();
}

/*****************************************************************************
 *
 *   REPORT_un16CountConfirmedMarkets
 *
 *****************************************************************************/
UN16 REPORT_un16CountConfirmedMarkets( void )
{
    BOOLEAN bLocked;
    UN16 un16NumConfirmed = 0;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        un16NumConfirmed = gpsReportCtrl->un16ConfirmedMarkets;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);
    }

    return un16NumConfirmed;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bInitialize
*
* Initialize the REPORT CDO Object. This needs to be called once
* when SMS is started. The REPORT objects being initialized here can be
* used by multiple SMS objects and thus are static for the life of SMS.
*
* This function is used to actually generate the known CID to Market map.
* This map is a linked list of CIDs to Market Ids and Names. The list is
* initially created with all known markets but does allow newly discovered
* markets to be added as well during run time.
*
* This effectively maps incoming CIDs to textual market names and ids.
* These are useful for UI purposes to display available markets and reference
* them with their associated CID. The associated CID may be used to monitor
* content and trigger an event when that CID is received in the broadcast.
*
*****************************************************************************/
static BOOLEAN bInitialize ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // First, determine if the control structure is already defined.
    // If so, do not attempt to re-define it. This structure is shared
    // amongst all DECODERS within SMS, so this only needs to be done
    // once when SMS is initialized.
    if(gpsReportCtrl != NULL)
    {
        // Do nothing, except report an error since this should never be
        // done twice.
        return FALSE;
    }

    // Create an SMS object which can be locked to enforce exclusive
    // access to the elements within this structure.
    gpsReportCtrl = (REPORT_CTRL_STRUCT *)
        SMSO_hCreate(
            REPORT_OBJECT_NAME":Ctrl",
            sizeof(REPORT_CTRL_STRUCT),
            SMS_INVALID_OBJECT,
            TRUE // We want the lock feature
                );
    if(gpsReportCtrl == NULL)
    {
        // Error! Something went wrong.
        return FALSE;
    }

    // We need to create the list. Note the list does not use the "protect"
    // feature as the list of markets is already protected by the SMS Object
    // mechanisms built in for this control structure.
    eReturnCode =
        OSAL.eLinkedListCreate(
            &gpsReportCtrl->hMarketList,
            REPORT_OBJECT_NAME":MarketList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareReportNames,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error! Something went wrong.
        vUnInitialize();
        return FALSE;
    }

    gpsReportCtrl->hMarketListCompleteTag = TAG_INVALID_OBJECT;
    gpsReportCtrl->hMarketsTag = TAG_INVALID_OBJECT;
    gpsReportCtrl->hRadioSpecificData = TAG_INVALID_OBJECT;

    gpsReportCtrl->bMarketListComplete = FALSE;

    gpsReportCtrl->un16ConfirmedMarkets = 0;

    // Relinquish control of the object
    SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);

    return TRUE;
}

/*****************************************************************************
*
*   vUnInitialize
*
* Uninitialize the REPORT CDO Object (upon SMS shutdown).
* Frees all resources associated with the static market list.
*
*****************************************************************************/
static void vUnInitialize ( void )
{
    BOOLEAN bLocked;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error! Something went wrong or no object created in the first place.
        return;
    }

    // Check if market list exists
    if(gpsReportCtrl->hMarketList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy all market entries first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            gpsReportCtrl->hMarketList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyReportEntry
                );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy list
            eReturnCode = OSAL.eLinkedListDelete(gpsReportCtrl->hMarketList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gpsReportCtrl->hMarketList = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    gpsReportCtrl->hMarketListCompleteTag = TAG_INVALID_OBJECT;
    gpsReportCtrl->hMarketsTag = TAG_INVALID_OBJECT;

    // Destroy object itself
    SMSO_vDestroy((SMS_OBJECT)gpsReportCtrl);
    gpsReportCtrl = NULL;

    return;
}

/*****************************************************************************
*
*   vUnInit
*
* CDO Un-initialize method
*
*****************************************************************************/
static void vUnInit (
    REPORT_OBJECT_STRUCT *psObj
        )
{
    CD_OBJECT hCDO;

    // Check input
    if(psObj == NULL)
    {
        return;
    }

    // Grab the CDO...
    hCDO = CDO_hCDO(psObj);

    // Inform CME of the content change. This report is no
    // longer part of this CDO. In this case we only notify the
    // CME that the content has changed. We don't destroy anything since
    // these CIDs are constant (unlike other types).
    CME_bChangeContent(hCDO, psObj->hId, CID_INVALID_OBJECT);

    // Default object now
    *psObj = gsDefaultReport;

    return;
}

/*****************************************************************************
*
*   n16Equal
*
*       This is the function used to compare REPORT CDOs. Specifically
*       this function is used to perform a go, no-go comparison
*       or binary comparison of the two CDOs. CDOs are considered equal
*       if their CIDs match.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               -1  - CDOs are not equal (or error)
*
*****************************************************************************/
static N16 n16Equal (
    const REPORT_OBJECT_STRUCT *psObj1,
    const REPORT_OBJECT_STRUCT *psObj2
        )
{
    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    return CID.n16Equal(psObj1->hId, psObj2->hId);
}

/*****************************************************************************
*
*   n16Compare
*
*       This is the function used to compare REPORT CDOs. Specifically
*       this function is used to keep CDOs in order for
*       comparison/sorting and thus a relational compare is performed.
*       CDOs are compared by comparing their CIDs with each other.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               > 0 - CDO1 is greater than (after) CDO2
*               < 0 - CDO1 is less than (before) CDO2 or error
*
*****************************************************************************/
static N16 n16Compare (
    const REPORT_OBJECT_STRUCT *psObj1,
    const REPORT_OBJECT_STRUCT *psObj2
        )
{
    N16 n16Result;

    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    if ((psObj1->hId == CID_INVALID_OBJECT) &&
        (psObj2->hId == CID_INVALID_OBJECT))
    {
        return 0;
    }

    n16Result = CID.n16Compare(
        psObj1->hId, psObj2->hId);

    // Call internal function which sorts by CID given a REPORT structure
    return n16Result;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This method is used by the caller to send formatted
* output of a CDO's contents to a specified file or device.
* This is mainly helpful during debugging of CDO's but could be used by
* a caller for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a CDO to a file for the
* purposes of later re-generating the CDO (for storage of the object).
* This API instead sends the CDO as a verbose formatted output version.
*
* Inputs:
*
*   psObj - The CDO structure the caller wishes to print.
*   psFile - The device to write the CDO contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    const REPORT_OBJECT_STRUCT *psObj,
    FILE *psFile
        )
{
    N32 n32Return = EOF;
    N32 n32Temp   = 0;

    // Check inputs.
    if((psObj != NULL) && (psFile != NULL))
    {
        n32Return = 0;

        n32Return += fprintf(psFile, "hId:\n");
        n32Temp = CID.n32FPrintf(psObj->hId, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
        n32Return += fprintf(psFile, "\n");

        n32Return += fprintf(psFile, "Market: %s (%s)\n",
            psObj->psMarket->pacName, psObj->psMarket->pacId);
    }

    return n32Return;
}

/*****************************************************************************
*
*   bHasId
*
* This method is used to examine a CDO sub-type for a specific Content-ID (CID)
* provided by the caller. If this sub-type has this id, then TRUE is returned
* otherwise, FALSE is returned if it does not contain it. Keep in mind
* this functions checks the entire sub-type for any matches.
*
* Inputs:
*   psObj - A pointer to an object for which to examine and determine if
*       the provided CID is contained within it.
*   hId - A valid CID to look for within the object.
*
* Returns:
*   TRUE of the CID was found anywhere within the object, otherwise FALSE
*   is returned if it could not be found.
*
*****************************************************************************/
static BOOLEAN bHasId (
    const REPORT_OBJECT_STRUCT *psObj,
    CID_OBJECT hId
        )
{
    BOOLEAN bHasId = FALSE;

    // We can look to see if this CID matches one we have in this object.
    // Any hit is enough to cause us to fall out and return TRUE.
    do
    {
        UN16 n16Equal;

        // hId == hId?
        n16Equal = CID.n16Equal(hId, psObj->hId);
        if(n16Equal == 0)
        {
            // Equal, so they match
            bHasId = TRUE;
            break;
        }

    } while(FALSE);

    return bHasId;
}

/*****************************************************************************
*
*   psCreateReportEntry
*
*   This function creates a report entry for our linked list of markets.
*
*****************************************************************************/
REPORT_ENTRY_STRUCT *psCreateReportEntry (
    CID_OBJECT hId,
    const MARKET_STRUCT *psMarket,
    BOOLEAN bCopyMarket
        )
{
    size_t tSize = sizeof(REPORT_ENTRY_STRUCT), tIdSize = 0, tNameSize = 0;
    REPORT_ENTRY_STRUCT *psReportEntry = NULL;

    // Determine if we also need to allocate memory for a provided market
    if(bCopyMarket == TRUE)
    {
        // Yes we need to add some to our memory requirements
        tSize += sizeof(MARKET_STRUCT);

        // Id
        if(psMarket->pacId != NULL)
        {
            tIdSize += strlen(psMarket->pacId);
        }
        else
        {
            tIdSize += strlen(gsDefaultMarket.pacId);
        }

        // Add null terminator for id
        tIdSize++;

        // Name
        if(psMarket->pacName != NULL)
        {
            tNameSize += strlen(psMarket->pacName);
        }
        else
        {
            tNameSize += strlen(gsDefaultMarket.pacName);
        }

        // Add null terminator for name
        tNameSize++;

        // Compute required size
        tSize += tIdSize + tNameSize;
    }

    // Now allocate a structure to hold the CID and market as one mapping.
    // This entry is available for all SMS objects and applications to use.
    psReportEntry = (REPORT_ENTRY_STRUCT *)
        SMSO_hCreate(
            REPORT_OBJECT_NAME":Obj",
            tSize,
            (SMS_OBJECT)gpsReportCtrl,
            FALSE
                );
    if(psReportEntry == NULL)
    {
        // Error! Unable to create
        return NULL;
    }

    // Populate new REPORT object with provided inputs.
    psReportEntry->sReport.hId = hId;
    psReportEntry->hDecoderHdl = MARKET_NOT_SEEN;
    psReportEntry->sReport.psMarket = psMarket;

    // Check if we need to re-populate with a copy of the provided market
    if(bCopyMarket == TRUE)
    {
        MARKET_STRUCT *psLocalMarket =
            (MARKET_STRUCT *)(psReportEntry + 1);
        char *pacName, *pacId;

        // Assign and populate market

        // If we are creating an entry with a provided market then that
        // means it has been loaded in from a known source (cfg file). Thus
        // it is a confirmed market now.
        psReportEntry->hDecoderHdl = MARKET_CONFIRMED;

        // ID
        psLocalMarket->pacId = pacId = (char *)(psLocalMarket + 1);
        if(psMarket->pacId == NULL)
        {
            // Use default
            strncpy(pacId, gsDefaultMarket.pacId, tIdSize);
        }
        else
        {
            // Use provided
            strncpy(pacId, psMarket->pacId, tIdSize);
        }

        // Name
        psLocalMarket->pacName = pacName = pacId + tIdSize;
        if(psMarket->pacName == NULL)
        {
            // Use default
            strncpy(pacName, gsDefaultMarket.pacName, tNameSize);
        }
        else
        {
            // Use provided
            strncpy(pacName, psMarket->pacName, tNameSize);
        }

        // Now transfer assigned pointers
        psReportEntry->sReport.psMarket = psLocalMarket;
    }

    return psReportEntry;
}

/*****************************************************************************
*
*   psCreateReportEntryFromTagValue
*
*   This function creates a report entry for our linked list of markets.
*
*****************************************************************************/
static REPORT_ENTRY_STRUCT *psCreateReportEntryFromTagValue (
    CID_OBJECT hId,
    STRING_OBJECT hName,
    STRING_OBJECT hMktId
        )
{
    size_t tSize = sizeof(REPORT_ENTRY_STRUCT),
           tIdSize = 0, tMktIdSize = 0, tNameSize = 0;
    REPORT_ENTRY_STRUCT *psReportEntry = NULL;
    MARKET_STRUCT *psLocalMarket;
    char *pacId, *pacName;

    // Determine if we also need to allocate memory for a provided market

    // Yes we need to add some to our memory requirements
    tSize += sizeof(MARKET_STRUCT);

    tNameSize += STRING.tSize(hName);
    // Add null terminator for Name
    tNameSize++;

    tMktIdSize += STRING.tSize(hMktId);
    // Add null terminator for Mkt Id
    tMktIdSize++;

    tIdSize = sizeof(CID_OBJECT);

    // Compute required size
    tSize += tNameSize + tMktIdSize + tIdSize;

    // Now allocate a structure to hold the CID and market as one mapping.
    // This entry is available for all SMS objects and applications to use.
    psReportEntry = (REPORT_ENTRY_STRUCT *)
        SMSO_hCreate(
            REPORT_OBJECT_NAME":Obj",
            tSize,
            (SMS_OBJECT)gpsReportCtrl, // Parent
            FALSE
                );
    if(psReportEntry == NULL)
    {
        // Error! Unable to create
        return NULL;
    }

    // Populate new REPORT object with provided inputs.
    psReportEntry->sReport.hId = hId;
    // If we are creating an entry with a provided market then that
    // means it has been loaded in from a known source (cfg file). Thus
    // it is a confirmed market now.
    psReportEntry->hDecoderHdl = MARKET_CONFIRMED;

    psLocalMarket = (MARKET_STRUCT *)(psReportEntry + 1);

    // Assign and populate market

    // ID
    psLocalMarket->pacId = pacId = (char *)(psLocalMarket + 1);
    STRING.tCopyToCStr( hMktId, pacId, tMktIdSize);

    // Name
    psLocalMarket->pacName = pacName = pacId + tMktIdSize;
    STRING.tCopyToCStr( hName, pacName, tNameSize);

    // Now transfer assigned pointers
    psReportEntry->sReport.psMarket = psLocalMarket;

    return psReportEntry;
}


/*****************************************************************************
*
*   vDestroyReportEntry
*
* Destroy a Report Entry
*
*****************************************************************************/
static void vDestroyReportEntry (
    REPORT_ENTRY_STRUCT *psReportEntry
        )
{
    // Invalidate entry info
    psReportEntry->sReport.psMarket = NULL;
    psReportEntry->hDecoderHdl = MARKET_NOT_SEEN;

    // Check if ID exists
    if(psReportEntry->sReport.hId != CID_INVALID_OBJECT)
    {
        // Destroy CID
        CID_vDestroy(psReportEntry->sReport.hId);
        psReportEntry->sReport.hId = CID_INVALID_OBJECT;
    }

    // Destroy object
    SMSO_vDestroy((SMS_OBJECT)psReportEntry);

    return;
}

/*****************************************************************************
*
*   n16MarketEqualToId
*
*       This is the function used to compare an existing REPORT CDO's CID
*       to a provided CID. Specifically this function is used to perform a go,
*       no-go comparison or binary comparison for searching or looking up
*       a market given a CID.
*
*       Outputs:
*               0   - REPORT CDO and CID have the same value (equal)
*               -1  - REPORT CDO and CID are not equal (or error)
*
*****************************************************************************/
static N16 n16MarketEqualToId (
    const REPORT_ENTRY_STRUCT *psObj,
    CID_OBJECT hCID
        )
{
    // Verify inputs
    if( (psObj == NULL) || (hCID == CID_INVALID_OBJECT) )
    {
        // Error
        return N16_MIN;
    }

    return CID.n16Equal(psObj->sReport.hId, hCID);
}

/*****************************************************************************
*
*   n16CompareReportNames
*
*       This is the function used to compare existing REPORT objects
*       to each other. Specifically this function is used to perform a
*       relational comparison for sorting. This sorter keeps the REPORT's
*       in order based on each REPORT's Name.
*
*       Outputs:
*               0   - REPORTs have the same value (equal)
*               > 0 - REPORT1 is greater than (after) REPORT2
*               < 0 - REPORT1 is less than (before) REPORT2 or error
*
*****************************************************************************/
static N16 n16CompareReportNames (
    const REPORT_ENTRY_STRUCT *psReport1,
    const REPORT_ENTRY_STRUCT *psReport2
        )
{
    N16 n16Result;
    int iStrCmpResult;

    // Verify inputs
    if( (psReport1 == NULL) || (psReport2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    // Verify inputs cont'd...
    if ( (psReport1->sReport.psMarket->pacName == NULL) ||
                    (psReport2->sReport.psMarket->pacName == NULL) )
    {
        // Error
        return N16_MIN;
    }

    // Determine relationship
    iStrCmpResult = strcmp(psReport1->sReport.psMarket->pacName,
        psReport2->sReport.psMarket->pacName);

    if (iStrCmpResult < 0)
    {
        n16Result = -1;
    }
    else if (iStrCmpResult > 0)
    {
        n16Result = 1;
    }
    else
    {
        // Check further for CID relationship
        n16Result = CID.n16Compare(psReport1->sReport.hId,
            psReport2->sReport.hId);
    }

    return n16Result;
}

/*****************************************************************************
*
*   bIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* report iterator structure to call the caller's own iterator function along
* with an argument for each market in the list.
*
*****************************************************************************/
static BOOLEAN bIterator (
    REPORT_ENTRY_STRUCT *psObj,
    REPORT_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bReturn = FALSE;

    // Check inputs
    if((psObj == NULL) || (psIterator == NULL))
    {
        // Cease iterating
        return FALSE;
    }

    // Call provided iterator function if one exists
    if(psIterator->bContentIteratorCallback != NULL)
    {
        if( (psObj->hDecoderHdl == MARKET_CONFIRMED) ||
            (psObj->hDecoderHdl != MARKET_NOT_SEEN))
        {
            // this report is valid, so we can call the caller's callback
            // Return whatever their callback returns back to the iterator
            bReturn = psIterator->bContentIteratorCallback(
                psObj->sReport.psMarket->pacId,
                psObj->sReport.psMarket->pacName,
                psObj->sReport.hId,
                psIterator->pvContentIteratorCallbackArg
                    );
        }
        else
        {
            // this report isn't valid, keep iterating to check the next one.
            bReturn = TRUE;
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bInvalidate
*
*  This is a function used as an object iterator to make each iterated
*  report object invalid
*
*****************************************************************************/
static BOOLEAN bInvalidate (
    REPORT_ENTRY_STRUCT *psObj,
    REPORT_ITERATOR_STRUCT *psIterator
        )
{
    // Check inputs
    if (psObj == NULL)
    {
        // Cease iterating
        return FALSE;
    }

    // make this report invalid
    psObj->hDecoderHdl = MARKET_NOT_SEEN;

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bCopyMkt
*
*  This is a function used as an object iterator to copy the CID to a
*  memory location
*
*****************************************************************************/
static BOOLEAN bWriteMktTag (
    const char *pacId,
    const char *pacName,
    CID_OBJECT hId,
    void *pvContentIteratorCallbackArg
        )
{
    TAG_OBJECT hCIDTag = TAG_INVALID_OBJECT, hNameTag = TAG_INVALID_OBJECT, hMktTag = TAG_INVALID_OBJECT;
    REPORT_MKT_TAG_ITERATOR_STRUCT *psIterator =
        (REPORT_MKT_TAG_ITERATOR_STRUCT *)pvContentIteratorCallbackArg;

    // make sure we were given a id
    if (pacId != NULL)
    {
        SMSAPI_RETURN_CODE_ENUM eReturn;

        if (strcmp(pacId, gsDefaultMarket.pacId) == 0)
        {
            // don't save if the market has a default id
            // just skip it and keep iterating
            return TRUE;
        }

        // try to get the tag for this Id, create it if it doesn't exist.
        eReturn =
            TAG_eGet(MARKET_TAG, psIterator->hParentTag, &hMktTag, pacId, TRUE);
        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // the call was successful, see if we found the tag
            if (hMktTag != TAG_INVALID_OBJECT)
            {
                size_t tSize;

                // get the size of this cid
                tSize = CID_tSize(hId)+1; // add one for NULL
                // is it bigger than the current size of our buffer
                if (tSize > psIterator->tBuffSize)
                {
                    // do we even have a buffer?
                    if (psIterator->pacBuff != NULL)
                    {
                        // yes destroy this one since we need a bigger one
                        SMSO_vDestroy((SMS_OBJECT)psIterator->pacBuff);
                        psIterator->pacBuff = NULL;
                    }

                    // allocate memory for the buffer
                    psIterator->pacBuff =
                        (char *) SMSO_hCreate(
                            REPORT_OBJECT_NAME":CSTR",
                            tSize,
                            SMS_INVALID_OBJECT, FALSE // No lock necessary
                            );
                    if (psIterator->pacBuff != NULL)
                    {
                        // record the size of the buffer we just created
                        psIterator->tBuffSize = tSize;
                    }
                }

                // make sure we have a buffer
                if (psIterator->pacBuff != NULL)
                {
                    char *pacBuff;
                    STRING_OBJECT hCIDString, hNameString;

                    pacBuff = psIterator->pacBuff;
                    // write the CID to memory
                    CID_n32FWriteToMemory(hId, (void **)&pacBuff);
                    // create a STRING from the CID info
                    hCIDString = STRING.hCreate(
                                     psIterator->pacBuff,
                                     0);

                    // create a STRING from the Name
                    hNameString =
                        STRING.hCreate(pacName, 0);

                    // get the name tag
                    eReturn =
                        TAG_eGet(MARKET_NAME_TAG, hMktTag, &hNameTag, NULL, TRUE);
                    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        eReturn = TAG_eSetTagValue(
                                      hNameTag,
                                      TAG_TYPE_STRING,
                                      &hNameString,
                                      sizeof(STRING_OBJECT),
                                      FALSE);
                        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                        {
                            // get the CID tag
                            eReturn = TAG_eGet(
                                          MARKET_CID_TAG,
                                          hMktTag,
                                          &hCIDTag,
                                          NULL,
                                          TRUE);
                            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                            {
                                eReturn = TAG_eSetTagValue(
                                              hCIDTag,
                                              TAG_TYPE_STRING,
                                              &hCIDString,
                                              sizeof(STRING_OBJECT),
                                              FALSE);
                            }
                        }
                    }

                    // we're done with these
                    STRING.vDestroy(hCIDString);
                    STRING.vDestroy(hNameString);
                }
            }
        }

        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // something went wrong

            // remove the Name and CID TAGs if we added them
            // we don't need to commit, since the additions weren't committed

            if (hNameTag != TAG_INVALID_OBJECT)
            {
                TAG_eRemove(hNameTag, FALSE);
            }
            if (hCIDTag != TAG_INVALID_OBJECT)
            {
                TAG_eRemove(hCIDTag, FALSE);
            }
        }
    }

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bRemovedMktTag
*
*  This is an iterator which will remove each iterated tag from its parent
*
*****************************************************************************/
static BOOLEAN bRemovedMktTag (
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    STRING_OBJECT hTagName;

    hTagName = TAG_hTagName(hTag);

    if (STRING.n16CompareCStr(MARKET_TAG, 0, hTagName )== 0)
    {
    // remove this tag from its parent
    TAG_eRemove(hTag, FALSE);
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bReadMktTag
*
*  This is an iterator which process each tag iterated and create a
*  report out of it
*
*****************************************************************************/
static BOOLEAN bReadMktTag (
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    STRING_OBJECT hIdString,
                  hCIDString = STRING_INVALID_OBJECT,
                  hNameString = STRING_INVALID_OBJECT, *phString;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;
    CID_OBJECT hId = CID_INVALID_OBJECT;
    REPORT_MKT_TAG_ITERATOR_STRUCT *psIterator =
        (REPORT_MKT_TAG_ITERATOR_STRUCT *)pvArg;

    // get the Id (the Tag Instance Identifier)
    hIdString = TAG_hTagInstanceName(hTag);

    if (hIdString != STRING_INVALID_OBJECT)
    {
        do
        {
            size_t tSize;

            // get the CID
            TAG_eGet(MARKET_CID_TAG, hTag, &hChildTag, NULL, FALSE);
            // check if we got a valid tag handle. a valid tag implies that the
            // TAG_eGet returned success so we don't need to actually check the
            // result in this case
            if (hChildTag == TAG_INVALID_OBJECT)
            {
                break;
            }
            phString = &hCIDString;
            tSize = sizeof(STRING_OBJECT);
            TAG_eGetTagValue(
                hChildTag,
                TAG_TYPE_STRING,
                (void**)&phString,
                &tSize);
            // check if we got a valid string handle. a valid string implies
            // that the TAG_eGetValue returned success and that the tag has a
            // value so we don't need to actually check the result in this case
            if (hCIDString == STRING_INVALID_OBJECT)
            {
                break;
            }

            tSize = STRING.tSize(hCIDString) + 1; // add one for NULL
            if (tSize > psIterator->tBuffSize) // Need a bigger buffer?
            {
                // Do we already have a buffer allocated?
                if (psIterator->pacBuff != NULL)
                {
                    // Free previously allocated buffer
                    SMSO_vDestroy((SMS_OBJECT)psIterator->pacBuff);
                    psIterator->tBuffSize = 0;
                }

                // allocate memory for the read buffer
                psIterator->pacBuff =
                    (char *) SMSO_hCreate(
                        REPORT_OBJECT_NAME":CSTR",
                        tSize,
                        SMS_INVALID_OBJECT, FALSE // No lock necessary
                        );
                if (psIterator->pacBuff != NULL)
                {
                    // Record the size of our new buffer
                    psIterator->tBuffSize = tSize;
                }
            }

            // If we have a buffer, do our thing
            if (psIterator->pacBuff != NULL)
            {
                char *pacBuff = psIterator->pacBuff;

                // Now use that buffer
                STRING.tCopyToCStr(hCIDString, pacBuff, tSize);
                // Read from memory assumes memory provided as an input is
                // NULL terminated. It is in fact since we just did a
                // string copy above into a buffer we know is at least one
                // larger than the source (thus it will be NULL terminated).
                hId = CID_hReadFromMemory((const void **)&pacBuff);
                if (hId == CID_INVALID_OBJECT)
                {
                    break;
                }

                // Get the Name tag
                hChildTag = TAG_INVALID_OBJECT;
                tSize = sizeof(STRING_OBJECT);
                TAG_eGet(MARKET_NAME_TAG, hTag, &hChildTag, NULL, FALSE);
                if (hChildTag == TAG_INVALID_OBJECT)
                {
                    break;
                }
                phString = &hNameString;
                TAG_eGetTagValue(
                    hChildTag,
                    TAG_TYPE_STRING,
                    (void**)&phString,
                    &tSize);
                if (hNameString == STRING_INVALID_OBJECT)
                {
                    break;
                }

                vLoadMarket(hId, hNameString, hIdString);
            }
        } while (0);

        // done with these strings
        if (hNameString != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(hNameString);
        }
        if (hCIDString != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(hCIDString);
        }
    }

    // We will always keep iterating, even if we couldn't make sense of this
    // tag. We'll just get as many markets as we can
    return TRUE;
}

/*****************************************************************************
*
*   vLoadMarket
*
*  This is function will take the information read from the config file and
*  create a report out of it
*
*****************************************************************************/
static void vLoadMarket (
    CID_OBJECT hId,
    STRING_OBJECT hName,
    STRING_OBJECT hMktId
        )
{
    do
    {
        // Read CID from memory
        if (hId != CID_INVALID_OBJECT)
        {
            REPORT_ENTRY_STRUCT *psReportEntry;

            // Now allocate a structure to hold the CID and market as
            // one mapping. This entry is available for all SMS objects
            // and applications to use.
            psReportEntry =
                psCreateReportEntryFromTagValue( hId, hName, hMktId);
            if(psReportEntry != NULL)
            {
                OSAL_LINKED_LIST_ENTRY hEntry =
                    OSAL_INVALID_LINKED_LIST_ENTRY;
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Check if this one already exists in our list
                eReturnCode = OSAL.eLinkedListLinearSearch(
                    gpsReportCtrl->hMarketList,
                    &hEntry,
                    (OSAL_LL_COMPARE_HANDLER)n16MarketEqualToId,
                    psReportEntry->sReport.hId
                        );
                if(eReturnCode == OSAL_SUCCESS) // Exists
                {
                    REPORT_ENTRY_STRUCT *psExistingReportEntry =
                        (REPORT_ENTRY_STRUCT *)
                            OSAL.pvLinkedListThis(hEntry);
                    DECODER_OBJECT hExistingHdl =
                        psExistingReportEntry->hDecoderHdl;

                    // Replace this existing report in our list.
                    eReturnCode =
                        OSAL.eLinkedListReplaceEntry(
                            gpsReportCtrl->hMarketList, hEntry,
                            psReportEntry
                                );
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // Report that replacing was successful
                        printf(
                            REPORT_OBJECT_NAME
                            " Replaced existing market"
                            " (%s,%s) Updated market list.\n",
                            psReportEntry->sReport.psMarket->pacId,
                            psReportEntry->sReport.psMarket->pacName
                                );

                        // Replacement success
                        if (MARKET_CONFIRMED != hExistingHdl)
                        {
                            gpsReportCtrl->un16ConfirmedMarkets++;
                        }

                        vDestroyReportEntry(psExistingReportEntry);
                    }
                    else
                    {
                        // Replacement unsuccessful. Try just replacing
                        // with new confirmation field

                        // Destroy what we created (can't use it)
                        vDestroyReportEntry(psReportEntry);

                        // Assign current to existing entry
                        psReportEntry = psExistingReportEntry;

                        // Mark existing as confirmed
                        psReportEntry->hDecoderHdl = MARKET_CONFIRMED;

                        // Try again...
                        eReturnCode =
                            OSAL.eLinkedListReplaceEntry(
                                gpsReportCtrl->hMarketList, hEntry,
                                psReportEntry
                                    );
                        if(eReturnCode == OSAL_SUCCESS)
                        {
                            // Report that replacing was successful
                            if (MARKET_CONFIRMED != hExistingHdl)
                            {
                                gpsReportCtrl->un16ConfirmedMarkets++;
                            }

                            printf(
                                REPORT_OBJECT_NAME
                                " Updated existing market"
                                " (%s,%s) Updated market list.\n",
                                psReportEntry->sReport.psMarket->pacId,
                                psReportEntry->sReport.psMarket->pacName
                                    );
                        }
                        else
                        {
                            // Wow! There is just nothing else I can do.
                            break; // Error
                        }
                    }
                }
                else // Does not exist, add it
                {
                    // Finally, add this report to our list.
                    eReturnCode =
                        OSAL.eLinkedListAdd(
                            gpsReportCtrl->hMarketList,
                            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                            psReportEntry
                                );
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // Report that adding was successful
                        gpsReportCtrl->un16ConfirmedMarkets++;

                        printf(
                            REPORT_OBJECT_NAME
                            " Read and added new market"
                            " (%s,%s) Updated market list.\n",
                            psReportEntry->sReport.psMarket->pacId,
                            psReportEntry->sReport.psMarket->pacName
                                );
                    }
                    else
                    {
                        // Report that adding was unsuccessful
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            REPORT_OBJECT_NAME
                            ": Unable to add market (%s.%s).",
                            psReportEntry->sReport.psMarket->pacId,
                            psReportEntry->sReport.psMarket->pacName
                                );

                        // Something went wrong with adding this
                        // element. No point in keeping the structure
                        // we just created.
                        psReportEntry->sReport.hId = CID_INVALID_OBJECT;
                        vDestroyReportEntry(psReportEntry);
                        break; // Error!
                    }

                } // Doesn't exist

            } // Can't create entry (skip to next one)
        }
        else
        {
            // Error!
            break;
        }
    } while (0);
}

/*****************************************************************************
*
*   vSaveMarkets
*
*   This function saves all confirmed or seen markets to the configuration
*   file along with the confirmed markets flag.
*
*****************************************************************************/
static void vSaveMarkets ( void )
{
    N32 n32MarketListComplete;
    REPORT_MKT_TAG_ITERATOR_STRUCT sIterator;

    // Debug
    puts(REPORT_OBJECT_NAME": Saving markets to config file ...\n\n");
    printf(REPORT_OBJECT_NAME": Markets%sconfirmed\n",
           gpsReportCtrl->bMarketListComplete == TRUE ? " " : " not ");

    // Record market list complete flag
    n32MarketListComplete =
        REPORT_MARKET_LIST_COMPLETE_TO_N32(gpsReportCtrl->bMarketListComplete);

    // Now set the tags in the config file
    TAG_eSetTagValue(
        gpsReportCtrl->hMarketListCompleteTag,
        TAG_TYPE_INTEGER,
        (void *)&n32MarketListComplete,
        sizeof(N32),
        FALSE);

    // iterate reports, writing markets tag

    sIterator.hParentTag = gpsReportCtrl->hMarketsTag;

    // Initially the buffer will be NULL. The memory will be allocated
    // the first time it is used, all tags iterated can then share it.
    sIterator.tBuffSize = 0;
    sIterator.pacBuff = NULL;

    REPORT.bIterateContent(bWriteMktTag, (void *)&sIterator);

    // Done iterating. Free the buffer if it was created.
    if (sIterator.pacBuff != NULL)
    {
        char *pacBuff = sIterator.pacBuff;
        SMSO_vDestroy((SMS_OBJECT)pacBuff);
    }

    // commit the changes to the file
    CM_eCommitChangesToFile();

    puts(REPORT_OBJECT_NAME": Finished saving market list.\n");

    return;
}

#if DEBUG_OBJECT == 1
/*****************************************************************************
*
*   bPrint
*
*  This is a DEBUG function used as an object iterator to simply format print
*  the REPORT market list.
*
*****************************************************************************/
static BOOLEAN bPrint (
    REPORT_ENTRY_STRUCT *psReportEntry,
    REPORT_PRINT_ITERATOR_STRUCT *psIterator
        )
{
    // Check inputs
    if((psIterator == NULL) || (psReportEntry == NULL))
    {
        // Stop iterating
        return FALSE;
    }

    // Increment number of markets known
    psIterator->un32NumMarkets++;

    // Determine if the market is confirmed
    if(psReportEntry->hDecoderHdl == MARKET_CONFIRMED)
    {
        // Increment confirmed markets
        psIterator->un32NumMarketsConfirmed++;
    }

    // Determine if the market is seen
    if(psReportEntry->hDecoderHdl != MARKET_NOT_SEEN)
    {
        // Increment seen markets
        psIterator->un32NumMarketsSeen++;
    }

    // Print market info
    fprintf(psIterator->psFile, "\t%s (%s) hId = 0x%p - MARKET %s\n",
        psReportEntry->sReport.psMarket->pacName,
        psReportEntry->sReport.psMarket->pacId,
        psReportEntry->sReport.hId,
        (psReportEntry->hDecoderHdl == MARKET_CONFIRMED) ? "CONFIRMED" :
        (psReportEntry->hDecoderHdl != MARKET_NOT_SEEN) ? "SEEN" :
        "NOT_VALID"
    );

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   vPrintMarkets
*
* This is a DEBUG function used to format print all known markets
* to an output file specified by psFile.
*
*****************************************************************************/
static void vPrintMarkets (
    FILE *psFile
        )
{
    REPORT_PRINT_ITERATOR_STRUCT sIterator;
    BOOLEAN bLocked;

    // Check inputs
    if(psFile == NULL)
    {
        return;
    }

    // Configure iterator
    sIterator.psFile = psFile;
    sIterator.un32NumMarkets = 0;
    sIterator.un32NumMarketsSeen = 0;
    sIterator.un32NumMarketsConfirmed = 0;

    // Dump Markets for this object
    fprintf(psFile, "\n\nREPORT: Markets...\n\n");

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsReportCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // Iterate the list and execute printer for each entry in the list.
        OSAL.eLinkedListIterate(
            gpsReportCtrl->hMarketList,
            (OSAL_LL_ITERATOR_HANDLER)bPrint, &sIterator);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsReportCtrl);
    }

    fprintf(psFile,
        "\n<--- End of market list (%u confirmed of %u seen, %u total"
        "). --->\n\n",
        sIterator.un32NumMarketsConfirmed,
        sIterator.un32NumMarketsSeen,
        sIterator.un32NumMarkets);

    return;
}

#endif
