
/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 *
 * DESCRIPTION
 *
 *  This module contains the Object:PRESETS implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "cm.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "decoder_obj.h"
#include "ccache.h"
#include "string_obj.h"
#include "category_obj.h"
#include "channel_obj.h"
#include "ccache.h"
#include "tag_obj.h"
#include "preset_band_obj.h"
#include "presets_obj.h"
#include "_presets_obj.h"

#if SMS_DEBUG == 1
// Only include this header if SMS_DEBUG is
// defined and equal to 1
#include "presets_debug.h"
#endif

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *       eStart
 *
 *       This API is used to start the presets service on a particular decoder.
 *       Additionally, this API will automatically determine if the SMS config
 *       file contains any preset data.  If so, the presets service started
 *       will be based on what is present in the file.  If not, the presets
 *       service will be started "fresh".  Also, the presets service will
 *       automatically save to the config file if one is present.  This allows
 *       the caller of this API to completely disregard the need to save/load
 *       this service in order to maintain state across power cycles.
 *
 *       Inputs:
 *           *phPresets - A pointer to handle which will store the PRESETS
 *               object handle upon successfull completion.
 *           hDecoder - A handle to a valid DECODER object for which the caller
 *               wishes to start the presets service.
 *           *pacLongName - A character string which is to be used as the
 *               "long name" for this instance of the presets service.  The
 *               presets virtual category will also utilize this as its
 *               "long name".
 *           *pacShortName - A character string which is to be used as the
 *               "short name" for this instance of the presets service. The
 *               presets virtual category will also utilize this as its
 *               "short name"
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_LOAD_SERVICE or SMSAPI_RETURN_CODE_NEW_SERVICE
 *           on success or an appropriate error code of type
 *           SMSAPI_RETURN_CODE_ENUM on error.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStart(
    PRESETS_OBJECT *phPresets,
    DECODER_OBJECT hDecoder,
    const char *pacLongName,
    const char *pacShortName
        )
{
    SMSAPI_RETURN_CODE_ENUM eSMSReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    PRESETS_OBJECT_STRUCT *psObj = NULL;

    do
    {
        BOOLEAN bValid;

        // Ensure that the two names provided are not both NULL
        if ((pacLongName == NULL) && (pacLongName == pacShortName))
        {
            break;
        }

        // Verify the rest of the inputs
        if (phPresets == NULL)
        {
            break;
        }

        // Initialize pointer value
        *phPresets = PRESETS_INVALID_OBJECT;

        // Validate the decoder handle
        bValid = SMSO_bValid((SMS_OBJECT) hDecoder);
        if (bValid == TRUE)
        {
            PRESETS_OBJECT hDecoderPresets;

            // See if the Presets service is already
            // running on the provided DECODER
            hDecoderPresets = DECODER_hPresets(hDecoder);

            if (hDecoderPresets != PRESETS_INVALID_OBJECT)
            {
                // Error! Presets service already running!
                eSMSReturnCode = SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
                break;
            }
        }
        else
        {
            break;
        }

        // Create the Presets service
        eSMSReturnCode = PRESETS_eCreate(hDecoder, (PRESETS_OBJECT *)&psObj, pacLongName, pacShortName);

        if ((eSMSReturnCode != SMSAPI_RETURN_CODE_ERROR) &&
            (psObj != NULL))
        {
            // Give the caller their handle
            *phPresets = (PRESETS_OBJECT) psObj;

            return eSMSReturnCode;
        }

    } while (FALSE);

    eStop((PRESETS_OBJECT) psObj);

    return eSMSReturnCode;
}

/*****************************************************************************
 *
 *       eStop
 *
 *       This API is used to stop the presets service for a particular DECODER.
 *       Current status of the service will be stored to the config file
 *       if this service is making use of it and the file is present.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object which represets
 *               the presets service the caller wishes to stop.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success or SMSAPI_RETURN_CODE_ERROR
 *               on error.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStop(
    PRESETS_OBJECT hPresets
        )
{
    DECODER_OBJECT hDecoder;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked = FALSE, bDecoderLocked;

    // Get DECODER object handle
    hDecoder = PRESETS_hDecoder(hPresets);

    // Verify and lock DECODER
    bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bDecoderLocked == TRUE)
    {
        // Verify and lock the Presets service
        bLocked = SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            BOOLEAN bOk;
            PRESETS_OBJECT_STRUCT *psObj =
                (PRESETS_OBJECT_STRUCT *) hPresets;

            do
            {
                BOOLEAN bSuccess;

                // Turn off Featured Favorites receive
                if (psObj->bFeaturedFavoritesEnabled == TRUE)
                {
                    RADIO_bFeaturedFavoritesEnable(psObj->hDecoder, FALSE);
                }

                // Clear the presets service handle for
                // this decoder
                bOk = DECODER_bSetPresetsHandle(
                    psObj->hDecoder, PRESETS_INVALID_OBJECT);

                if (bOk != TRUE)
                {
                    // Error!
                    break;
                }

                // There is no return after this point. Object will be
                // destroyed partially if an error occurred anyway. So,
                // let's try to free as much resources as possible.
                // We do not check return values of OSAL functions down
                // this function, just becasue in case of failure there
                // is nothing we can do anyway.

                // Destroy asynchronous update configuration
                SMSU_vDestroy(&psObj->sEvent);

                // Destroy bands first, to remove channels
                // from the category

                // Remove the band list and clear it if necessary
                if (psObj->hBands != OSAL_INVALID_OBJECT_HDL)
                {
                    BOOLEAN bBlocked;

                    // Block category notifications
                    bBlocked = CATEGORY_bBlockNotifications(
                        hDecoder, psObj->tCategoryId);

                    // Remove all bands from the list
                    // and free their descriptors
                    // (includes memory for presets)
                    OSAL.eLinkedListRemoveAll(
                        psObj->hBands,
                        vDestroyBandDescriptor
                            );

                    // Destroy the list itself
                    OSAL.eLinkedListDelete(psObj->hBands);
                    psObj->hBands = OSAL_INVALID_OBJECT_HDL;

                    // Current band has been removed
                    psObj->hCurrentBand = OSAL_INVALID_LINKED_LIST_ENTRY;

                    // Releasing PRESETS Category notifications
                    if (bBlocked == TRUE)
                    {
                        CATEGORY_vReleaseNotifications(
                            hDecoder, psObj->tCategoryId);
                    }
                }

                // Need to synchronize Smart Favorites with 
                // empty channel set.
                bSuccess = PRESETS_bCurrentBandSync(
                    hPresets, PRESET_BAND_INVALID_OBJECT);
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        PRESETS_OBJECT_NAME
                        ": Failed to sync current preset band");
                }

                // Remove the category
                vRemoveCategory( psObj );

                // Relinquish ownership
                SMSO_vUnlock((SMS_OBJECT)hDecoder);
                bDecoderLocked = FALSE;

                // Destroy the names if necessary
                if (psObj->hLongName != STRING_INVALID_OBJECT)
                {
                    STRING_vDestroy(psObj->hLongName);
                    psObj->hLongName = STRING_INVALID_OBJECT;
                }

                if (psObj->hShortName != STRING_INVALID_OBJECT)
                {
                    STRING_vDestroy(psObj->hShortName);
                    psObj->hShortName = STRING_INVALID_OBJECT;
                }

                if (psObj->hID != OSAL_INVALID_OBJECT_HDL)
                {
                    OSAL.eLinkedListRemoveAll(
                        psObj->hID,
                        (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy
                            );

                    OSAL.eLinkedListDelete(psObj->hID);
                    psObj->hID = OSAL_INVALID_OBJECT_HDL;
                }

                // Remove the master preset list
                if (psObj->hPresets != OSAL_INVALID_OBJECT_HDL)
                {
                    // Remove all presets from the list
                    OSAL.eLinkedListRemoveAll(
                        psObj->hPresets,
                        NULL);

                    // Destroy the list itself
                    OSAL.eLinkedListDelete(psObj->hPresets);
                    psObj->hPresets = OSAL_INVALID_OBJECT_HDL;
                }

                // Uninitialize callback shim
                psObj->sCallbackShim.hBand = PRESET_BAND_INVALID_OBJECT;
                psObj->sCallbackShim.pvEventCallbackArg = NULL;
                psObj->sCallbackShim.vEventCallback = NULL;

                // Uninitialize receiver targeting information
                psObj->sCapabilities.tPurpose = FAVORITES_BAND_PURPOSE_NONE;
                psObj->sCapabilities.tCapacity = FAVORITES_BAND_CAPACITY_MIN;
                psObj->sCapabilities.tBandsNumber = FAVORITES_BANDS_NUMBER_MIN;

                // Clear the decoder handle
                psObj->hDecoder = DECODER_INVALID_OBJECT;

                // Destroy object
                SMSO_vDestroy((SMS_OBJECT) hPresets);

                // Un-initialize object
                hPresets = PRESETS_INVALID_OBJECT;

                // All done!
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

            } while (FALSE);
        }
    }

    // If Presets Service Locked
    // (in case of some error we will 'break' out from while w/o unlock)
    if ((bLocked == TRUE) && (hPresets != PRESETS_INVALID_OBJECT))
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    // If Decoder locked
    if (bDecoderLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *       tCategoryId
 *
 *       This API is used get the Id of the category object which is currently
 *       used to represent the current presets.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object for which the caller
 *               wishes to get the current category Id.
 *
 *       Outputs:
 *           A valid CATETGORY_ID on success or CATEGORY_INVALID_ID
 *               on error.
 *
 *****************************************************************************/
static CATEGORY_ID tCategoryId(
    PRESETS_OBJECT hPresets
    )
{
    BOOLEAN bLocked;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;

        tCategoryId = psObj->tCategoryId;

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return tCategoryId;
}

/*****************************************************************************
 *
 *       tNumBands
 *
 *       This API is used get the current number of bands available in the
 *       presets service.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object for which the caller
 *               wishes to get the current number of bands.
 *
 *       Outputs:
 *           A size_t indicating the current number of bands.
 *
 *****************************************************************************/
static size_t tNumBands(
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked;
    UN32 un32NumBands = 0;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;

        // Get the size of the bands LL
        OSAL.eLinkedListItems(
            psObj->hBands,
            &un32NumBands );

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return (size_t) un32NumBands;
}

/*****************************************************************************
 *
 *       hShortName
 *
 *       This API is used get the short name assigned to the presets service
 *       (and to the associated category for the presets service).
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object which stores the
 *               presets service short name.
 *
 *       Outputs:
 *           A valid STRING object on success; otherwise STRING_INVALID_OBJECT
 *           on error.
 *
 *****************************************************************************/
static STRING_OBJECT hShortName(
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked;
    STRING_OBJECT hShortName = STRING_INVALID_OBJECT;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;

        hShortName = psObj->hShortName;

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return hShortName;
}

/*****************************************************************************
 *
 *       hLongName
 *
 *       This API is used get the long name assigned to the presets service
 *       (and to the associated category for the presets service).
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object which stores the
 *               presets service long name.
 *
 *       Outputs:
 *           A valid STRING object on success; otherwise STRING_INVALID_OBJECT
 *           on error.
 *
 *****************************************************************************/
static STRING_OBJECT hLongName(
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked;
    STRING_OBJECT hLongName = STRING_INVALID_OBJECT;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;

        hLongName = psObj->hLongName;

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return hLongName;
}

/*****************************************************************************
 *
 *       eIterate
 *
 *       This API is used to iterate all bands/presets found within the
 *       presets service.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object for which the caller
 *               wishes to iterate
 *           n16Iterator - The iteration handler that will be used during the
 *               iteration of bands/channels.
 *           pvIterateArg - The pointer to an optional argument
 *               provided to the iteration handler.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success or an appropriate error code
 *               of type SMSAPI_RETURN_CODE_ENUM on error.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate(
    PRESETS_OBJECT hPresets,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        )
{
    BOOLEAN bLocked, bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    // ee@ NOTE: In this function need to lock
    // a DECODER object first, since inside eIterateCommon it
    // uses to get channels info from ccache under DECODER's lock
    // An interactive deadlock condition is possible otherwise

    // Check object
    bValid = SMSO_bValid((SMS_OBJECT)hPresets);
    if (bValid == TRUE)
    {
        DECODER_OBJECT hDecoder;
        // de-reference object
        PRESETS_OBJECT_STRUCT *psObj =
                (PRESETS_OBJECT_STRUCT *) hPresets;

        // Get DECODER instance
        hDecoder = psObj->hDecoder;
        // Get locked instance
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            // Verify and lock the Presets service
            bLocked =
                SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
            if (bLocked == TRUE)
            {
                // Pass iteration request to
                // the common iterator function
                eReturnCode = eIterateCommon(
                    psObj,
                    PRESET_BAND_INVALID_OBJECT,
                    n16Iterator,
                    pvIterateArg
                        );

                SMSO_vUnlock((SMS_OBJECT) hPresets);
            }
            SMSO_vUnlock((SMS_OBJECT)hDecoder);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *       eSetCurrentBand
 *
 *       This API is used to set the provided band as the "current" band for
 *       the presets service.
 *
 *       Inputs:
 *           hBand - A handle to a valid PRESET_BAND object which the caller
 *               wishes to set as the current band.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success or an appropriate error code
 *               of type SMSAPI_RETURN_CODE_ENUM on error.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetCurrentBand(
    PRESET_BAND_OBJECT hBand
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLock;

    do
    {
        PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
        if (hBand == PRESET_BAND_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Invalid Preset Band object provided");

            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Lock BAND object to extract its parent
        bLock = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLock == TRUE)
        {
            // Extract the parent PRESETS object
            hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
            SMSO_vUnlock((SMS_OBJECT)hBand);
        }

        if (hPresets == PRESETS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Failed to extract the "
                "Presets service handle");
            break;
        }

        hBand = hBrowseBands(hPresets, hBand, SMSAPI_DIRECTION_DIRECT);
        if (hBand == PRESET_BAND_INVALID_OBJECT)
        {
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *
 *       hCurrentBand
 *
 *       This API is used get the current band for the presets service
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object for which the caller
 *               wishes to get the current band.
 *
 *       Outputs:
 *           A valid PRESETS_BAND_OBJECT on success or
 *               PRESETS_INVALID_BAND_OBJECT on error.
 *
 *****************************************************************************/
static PRESET_BAND_OBJECT hCurrentBand(
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked;
    PRESET_BAND_OBJECT hCurrentBand =
        PRESET_BAND_INVALID_OBJECT;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;
        PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;

        // Extract the current band
        psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
            OSAL.pvLinkedListThis( psObj->hCurrentBand );

        if (psBandDesc != NULL)
        {
            // Get the band handle
            hCurrentBand = psBandDesc->hBand;

            // update the band tags to indicate this band is the current one
            eUpdateCurrentBandTags(psObj, hCurrentBand);
        }

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return hCurrentBand;
}

/*****************************************************************************
 *
 *       hPreviousBand
 *
 *       This API is used get the band previous to the current band for the
 *       presets service. Additionally, this API also sets the previous
 *       as the current.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object for which the caller
 *               wishes to get the previous band.
 *
 *       Outputs:
 *           A valid PRESETS_BAND_OBJECT on success or
 *               PRESETS_INVALID_BAND_OBJECT on error.
 *
 *****************************************************************************/
static PRESET_BAND_OBJECT hPreviousBand(
    PRESETS_OBJECT hPresets
        )
{
    PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;

    if (hPresets != PRESETS_INVALID_OBJECT)
    {
        hBand = hBrowseBands(
            hPresets, 
            PRESET_BAND_INVALID_OBJECT, 
            SMSAPI_DIRECTION_PREVIOUS
                );
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            PRESETS_OBJECT_NAME": Invalid Presets object handle provided");
    }

    return hBand;
}

/*****************************************************************************
 *
 *       hNextBand
 *
 *       This API is used get the band after to the current band for the
 *       presets service. Additionally, this API also sets the next
 *       as the current.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object for which the caller
 *               wishes to get the previous band.
 *
 *       Outputs:
 *           A valid PRESETS_BAND_OBJECT on success or
 *               PRESETS_INVALID_BAND_OBJECT on error.
 *
 *****************************************************************************/
static PRESET_BAND_OBJECT hNextBand(
    PRESETS_OBJECT hPresets
        )
{
    PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;

    if (hPresets != PRESETS_INVALID_OBJECT)
    {
        hBand = hBrowseBands(
            hPresets,
            PRESET_BAND_INVALID_OBJECT,
            SMSAPI_DIRECTION_NEXT
                );
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            PRESETS_OBJECT_NAME": Invalid Presets object handle provided");
    }

    return hBand;
}

/*****************************************************************************
 *
 *       n32FPrintf
 *
 *       This API is used to print the presets service to a file.  This
 *       function is meant to aid in debugging and is not meant to assist
 *       the presets service to persist across power cycles.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object which the caller
 *               wishes to print.
 *           *psFile - A pointer to a device which is to receive formatted
 *               text output.
 *
 *       Outputs:
 *           The number of characters written.
 *
 *****************************************************************************/
static N32 n32FPrintf(
    PRESETS_OBJECT hPresets,
    FILE *psFile
        )
{
    N32 n32CharsWritten = 0;
#if SMS_DEBUG == 1
    BOOLEAN bLocked, bDecoderLocked, bValid;
    PRESETS_OBJECT_STRUCT *psObj =
        (PRESETS_OBJECT_STRUCT *) hPresets;
    PRESETS_DEBUG_ATTRS_STRUCT sDebugAttrs;
    PRESETS_PRINT_STRUCT sPrintBands;
    DECODER_OBJECT hDecoder;

    // Verify input
    if (psFile == NULL)
    {
        return EOF;
    }

    bValid = SMSO_bValid((SMS_OBJECT)hPresets);
    if (bValid == FALSE)
    {
        return EOF;
    }

    // Extract decoder handle
    hDecoder = psObj->hDecoder;

    bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bDecoderLocked == FALSE)
    {
        return EOF;
    }

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
        return EOF;
    }

    // Populate the debug struct
    sDebugAttrs.hPresets = hPresets;
    sDebugAttrs.hLongName = psObj->hLongName;
    sDebugAttrs.hShortName = psObj->hShortName;
    sDebugAttrs.tNumberOfBands = PRESETS.tNumBands(hPresets);
    sDebugAttrs.tCategoryId = psObj->tCategoryId;
    sDebugAttrs.tCategorySyncId = psObj->tCategorySyncId;
    sDebugAttrs.hBandList = psObj->hBands;
    sDebugAttrs.bFeaturedFavoritesEnabled = psObj->bFeaturedFavoritesEnabled;

    // Print the presets service
    n32CharsWritten =
            PRESETS_DEBUG_n32PrintPresetsService(
                &sDebugAttrs,
                psFile );

    if (n32CharsWritten >= 0)
    {
        // Initialize print iterator structure
        sPrintBands.n32CharsWritten = n32CharsWritten;
        sPrintBands.psFile = psFile;
        sPrintBands.pacFormat = "\t";
        sPrintBands.hPresets = hPresets;

        // Now, print out all the band information
        OSAL.eLinkedListIterate(
                psObj->hBands,
                bPrintBands,
                (void *) (size_t) &sPrintBands);

        // Get the total number of characters written
        n32CharsWritten = sPrintBands.n32CharsWritten;
    }
    SMSO_vUnlock((SMS_OBJECT) hPresets);
    SMSO_vUnlock((SMS_OBJECT)hDecoder);

#endif
    return n32CharsWritten;
}

/*****************************************************************************
 *
 *       eFeaturedFavoritesStart
 *
 *       This API is used to start Featured Favorites table processing and
 *       to configure receiver target information
 *
 *       Inputs:
 *          hPresets - A handle to a valid PRESETS object which the caller
 *               wishes to use for Featured Favorites
 *          tBandCapacity - A maximum number of presets which Featured Band
 *               shall consist of.
 *          tBandsNumber - A maximum number of bands which the service
 *               shall contain
 *          tPurpose - Featured Band purpose allowed on device
 *
 *       Outputs:
 *          SMSAPI_RETURN_CODE_SUCCESS on success or an appropriate error code
 *               of type SMSAPI_RETURN_CODE_ENUM on error
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFeaturedFavoritesStart (
    PRESETS_OBJECT hPresets,
    FAVORITES_BAND_CAPACITY tBandCapacity,
    FAVORITES_BAND_NUMBER tBandsNumber,
    FAVORITES_BAND_PURPOSE_MASK tPurpose
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        BOOLEAN bEnabled, bValid;
        SMS_EVENT_FEATURED_FAVORITES_STRUCT sEvent;
        DECODER_OBJECT hDecoder;
        PRESETS_OBJECT_STRUCT *psObj = (PRESETS_OBJECT_STRUCT *)hPresets;

        bValid = SMSO_bValid((SMS_OBJECT)hPresets);

        // Checking input parameters
        if ((bValid == FALSE) ||
            (tBandCapacity < FAVORITES_BAND_CAPACITY_MIN) ||
            (tBandsNumber < FAVORITES_BANDS_NUMBER_MIN))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        if (psObj->bFeaturedFavoritesEnabled == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED;
            break;
        }

        hDecoder = PRESETS_hDecoder(hPresets);

        // Ask DECODER to handle the request.
        sEvent.bEnable = TRUE;
        sEvent.uEvent.sEnable.sCapabilities.tBandsNumber =
            tBandsNumber;
        sEvent.uEvent.sEnable.sCapabilities.tCapacity =
            tBandCapacity;
        sEvent.uEvent.sEnable.sCapabilities.tPurpose =
            tPurpose;

        bEnabled = DECODER_bFeaturedFavorites(
            hDecoder, &sEvent);
        if (bEnabled == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *
 *       eFeaturedFavoritesStop
 *
 *       This API is used to disable Featured Favorites processing after
 *       the current table would be completely received
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object which the caller
 *               wishes to use for Featured Favorites
 *           bRemoveBands - A boolean value. If TRUE – all featured bands would
 *               be eliminated from PRESETS service. Otherwise –
 *               kept in service.
 *
 *       Outputs:
 *          SMSAPI_RETURN_CODE_SUCCESS on success or an appropriate error code
 *               of type SMSAPI_RETURN_CODE_ENUM on error
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFeaturedFavoritesStop (
    PRESETS_OBJECT hPresets,
    BOOLEAN bRemoveBands
        )
{
    BOOLEAN bDisabled, bValid;
    DECODER_OBJECT hDecoder;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SMS_EVENT_FEATURED_FAVORITES_STRUCT sEvent;

    bValid = SMSO_bValid((SMS_OBJECT)hPresets);
    if (bValid == FALSE)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Ask DECODER to handle the request.
    hDecoder = PRESETS_hDecoder(hPresets);

    sEvent.bEnable = FALSE;
    sEvent.uEvent.sDisable.bRemoveBands = bRemoveBands;

    bDisabled = DECODER_bFeaturedFavorites(
        hDecoder, &sEvent);
    if (bDisabled == TRUE)
    {
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *       eNotifyOnChange
 *
 *       This API is used to register a user provided callback function which
 *       would be used for Featured Favorites notifications
 *
 *       Inputs:
 *          hPresets - A handle to a valid PRESETS object which the caller
 *               wishes to use for Featured Favorites
 *          vCallback - A pointer to the user-specified callback function for
 *               which the caller wishes to use notifications. A NULL value
 *               is possible to use to disable notifications
 *          vCallbackArg - A void pointer to the user-specified argument
 *               for notification callback
 *          bNotify - A boolean value. If true and the callback is not NULL,
 *               it would be used to notify the caller using provided callback
 *               about all available featured bands in the service
 *
 *       Outputs:
 *          SMSAPI_RETURN_CODE_SUCCESS on success or an appropriate error code
 *               of type SMSAPI_RETURN_CODE_ENUM on error
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eNotifyOnChange (
    PRESETS_OBJECT hPresets,
    PRESETS_OBJECT_EVENT_CALLBACK vCallback,
    void * pvCallbackArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock object instance
    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        // Save new callback into shim
        psObj->sCallbackShim.vEventCallback = vCallback;
        psObj->sCallbackShim.pvEventCallbackArg = pvCallbackArg;

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);

        // Notify all the caller about all available featured bands
        if (vCallback != NULL)
        {
            OSAL_RETURN_CODE_ENUM eOsalCode;

            eOsalCode = OSAL.eLinkedListIterate(
                psObj->hBands,
                (OSAL_LL_ITERATOR_HANDLER)bPresetsNotifyIterator,
                (void *)psObj);
            if (OSAL_SUCCESS != eOsalCode && OSAL_NO_OBJECTS != eOsalCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Cannot iterate bands list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalCode));
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *       PRESETS_eCreate
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESETS_eCreate (
    DECODER_OBJECT hDecoder,
    PRESETS_OBJECT *phPresets,
    const char *pacLongName,
    const char *pacShortName
        )
{
    BOOLEAN bLocked;
    static UN32 un32Instance = 0;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    PRESETS_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bOk, bRestored = FALSE;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    SMSAPI_RETURN_CODE_ENUM eSMSReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        // Lock decoder object first to prevent deadlock in category events handler
        // Presets gives itself as argument for this callback
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Create the name for this presets object
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            PRESETS_OBJECT_NAME":Obj:%u", un32Instance++);

        // Create an instance of this object
        psObj = (PRESETS_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(PRESETS_OBJECT_STRUCT),
                SMS_INVALID_OBJECT,
                TRUE); // Enable locking

        if (psObj == NULL)
        {
            break;
        }

        // Create the virtual "presets" category
        // with no initial channels (we'll use this
        // Id as an identifier for this object)
        psObj->tCategoryId =
            CATEGORY_tCreateVirtualCategory(
                hDecoder,
                pacLongName,
                pacShortName,
                0, FALSE, FALSE);

        if (psObj->tCategoryId == CATEGORY_INVALID_ID)
        {
            // Error!
            break;
        }

        // Save the decoder handle
        psObj->hDecoder = hDecoder;

        // We are not currently updating the category
        psObj->bUpdatingCategory = FALSE;

        // We do not need to rebuild the category
        psObj->bRebuildCategory = FALSE;

        // Initialize FF service state
        psObj->bFeaturedFavoritesEnabled = FALSE;

        // Initialize current band category id
        psObj->tCategorySyncId = CATEGORY_INVALID_ID;

        // initialize callback shim
        psObj->sCallbackShim.hBand = PRESET_BAND_INVALID_OBJECT;
        psObj->sCallbackShim.pvEventCallbackArg = NULL;
        psObj->sCallbackShim.vEventCallback = NULL;

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            PRESETS_OBJECT_EVENT_NONE,
            PRESETS_OBJECT_EVENT_ALL,
           (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallbackShim,
            &psObj->sCallbackShim);

        // Initialize receiver targeting information
        psObj->sCapabilities.tPurpose = FAVORITES_BAND_PURPOSE_NONE;
        psObj->sCapabilities.tCapacity = FAVORITES_BAND_CAPACITY_MIN;
        psObj->sCapabilities.tBandsNumber = FAVORITES_BANDS_NUMBER_MIN;

        // Register use of this category
        bOk = bRegisterCategory( psObj );

        if (bOk != TRUE)
        {
            // Error!
            break;
        }

        // Construct an appropriate name for band list to be created
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                PRESETS_OBJECT_NAME":%u Bands",
                psObj->tCategoryId);

        // Create the band list
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hBands,
                &acName[0],
                n16SortUserAndFeaturedFavorites,
                OSAL_LL_OPTION_CIRCULAR
                    );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Construct an appropriate name for unique SIDs list to be created
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                PRESETS_OBJECT_NAME":%u SIDs",
                psObj->tCategoryId);

        // Create the IDs list
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hID,
                &acName[0],
                (OSAL_LL_COMPARE_HANDLER)n16CompareFFBankIDs,
                OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                    );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Construct an appropriate name for master preset list
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                PRESETS_OBJECT_NAME":%u Presets",
                psObj->tCategoryId);

        // Create the master preset list
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hPresets,
                &acName[0],
                n16SortPresetsByServiceId,
                OSAL_LL_OPTION_CIRCULAR
                    );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Start band Ids at zero no matter how we start
        psObj->tNextBandId = 0;

        // Create the string name attributes
        if (pacLongName != NULL)
        {
            psObj->hLongName = STRING.hCreate(
                pacLongName,
                strlen(pacLongName));

            if (psObj->hLongName == STRING_INVALID_OBJECT)
            {
                // Error!
                break;
            }
        }

        if (pacShortName != NULL)
        {
            psObj->hShortName = STRING.hCreate(
                pacShortName,
                strlen(pacShortName));

            if (psObj->hShortName == STRING_INVALID_OBJECT)
            {
                // Error!
                break;
            }
        }

        // check if there is information about the preset service for this
        // decoder in the config file
        eSMSReturnCode = eRestoreFromConfigFile(psObj, &bRestored);

        if (eSMSReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            if (bRestored == TRUE)
            {
                // We used the presets stored in the config file
                eSMSReturnCode = SMSAPI_RETURN_CODE_LOAD_SERVICE;
            }
            else
            {
                // Indicate this is a new service
                eSMSReturnCode = SMSAPI_RETURN_CODE_NEW_SERVICE;

                // There is no current band
                psObj->hCurrentBand = OSAL_INVALID_LINKED_LIST_ENTRY;
            }
        }
        else
        {
            // something went wrong when trying to restore ...
            break;
        }

        // Attempt to set up this instance of the
        // presets service as the service for our decoder
        bOk = DECODER_bSetPresetsHandle(
            hDecoder, (PRESETS_OBJECT) psObj);

        if (bOk != TRUE)
        {
            // Error!  What happened here?
            // We've already established the
            // fact that presets is not
            // running on this decoder...
            break;
        }

        // Return create instance
        *phPresets = (PRESETS_OBJECT)psObj;

    } while (FALSE);

    if (psObj != NULL)
    {
        if ((eSMSReturnCode == SMSAPI_RETURN_CODE_LOAD_SERVICE) ||
            (eSMSReturnCode == SMSAPI_RETURN_CODE_NEW_SERVICE))
        {
            // Relinquish ownership
            SMSO_vUnlock((SMS_OBJECT)(PRESETS_OBJECT)psObj);
        }
        else
        {
            // Object created, but service initiation wasn't success
            eStop((PRESETS_OBJECT)psObj);
        }
    }

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eSMSReturnCode;
}

/*****************************************************************************
 *
 *       PRESETS_hAddBand
 *
 *       This function adds a band to the presets service.
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
OSAL_LINKED_LIST_ENTRY PRESETS_hAddBand(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    size_t tBandCapacity
        )
{
    BOOLEAN bLocked;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;

    // Verify input
    if (tBandCapacity == 0)
    {
        return OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;
        OSAL_LINKED_LIST_ENTRY hNewBandEntry;

        // Get this band created and add it to our list
        // using the next available band Id
        hNewBandEntry = hCreateBandDescriptor(
            psObj, hBand, tBandCapacity, psObj->tNextBandId);

        // Continue if we now have a valid entry for this band
        if (hNewBandEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            BOOLEAN bSuccess;

            // Attempt to add this band's presets
            // to the master list
            bSuccess = bAddPresetsForBand(
                psObj, hNewBandEntry);

            if (bSuccess == FALSE)
            {
                PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;

                // Extract the band descriptor
                psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                    OSAL.pvLinkedListThis(hNewBandEntry);

                // If that failed, delete & remove this band
                vDestroyBandDescriptor((void *) (size_t) psBandDesc);
                OSAL.eLinkedListRemove(hNewBandEntry);
            }
            else
            {
                // Determine if this is the only band (indicated
                // by the fact that we do not have a current valid
                // band entry)
                if (psObj->hCurrentBand == OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    // This is our only band,
                    // set it as the current
                    psObj->hCurrentBand = hNewBandEntry;
                }

                // We may now increment to the next band Id
                psObj->tNextBandId++;

                // Give the caller the entry since
                // we didn't experience any issues
                hEntry = hNewBandEntry;
            }
        }

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return hEntry;
}

/*****************************************************************************
 *
 *       PRESETS_bRemoveBand
 *
 *       This function removes a previously added band from the presets
 *       service.
 *
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
BOOLEAN PRESETS_bRemoveBand(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    BOOLEAN bUpdate
        )
{
    BOOLEAN bValid, bSuccess = FALSE;

    // Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT) hPresets);
    if (bValid == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;

        // Extract the band descriptor from the
        // provided band entry
        psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
            OSAL.pvLinkedListThis(hBandEntry);

        // Is the resulting pointer valid?
        if (psBandDesc == (PRESETS_BAND_DESCRIPTOR_STRUCT *) NULL)
        {
            // Bad!
            return FALSE;
        }

        // Does this entry agree on the band?
        if (psBandDesc->hBand != hBand)
        {
            // Nope!
            return FALSE;
        }

        // Determine if we are removing the current band
        if (hBandEntry == psObj->hCurrentBand)
        {
            do
            {
                PRESETS_BAND_DESCRIPTOR_STRUCT *psCurrentDesc = NULL;
                OSAL_LINKED_LIST_ENTRY hEntry = psObj->hCurrentBand;

                if (bUpdate == FALSE)
                {
                    break;
                }

                // Set the current band to the next band
                // in line.
                psObj->hCurrentBand =
                    OSAL.hLinkedListNext(
                        psObj->hCurrentBand, (void **)&psCurrentDesc);

                // Presets are configured to use circular LL. So, when trying
                // to remove the last preset band, the current band will wrap
                // around to itself. In that case, update is unnecessary
                if (hEntry == psObj->hCurrentBand)
                {
                    break;
                }

                if (psCurrentDesc == NULL)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        PRESETS_OBJECT_NAME
                        ":SF: Current Band descriptor is NULL");
                    break;
                }

                bSuccess = PRESETS_bCurrentBandSync(
                    hPresets, psCurrentDesc->hBand);
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        PRESETS_OBJECT_NAME
                        ":SF: Failed to sync current Band: %s (%#d)",
                        STRING.pacCStr(BAND.hName(psCurrentDesc->hBand)), 
                        psCurrentDesc->tBandId);
                    break;
                }

            } while (FALSE);
        }

        if (bUpdate == TRUE)
        {
            // Update presets
            bSuccess = PRESETS_bUpdate(
                hPresets, hBand, PRESETS_OBJECT_EVENT_BAND_REMOVED);
            if(bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME
                    ": Failed to notify about the Band: %s (%#d) removing",
                        STRING.pacCStr(BAND.hName(psBandDesc->hBand)), 
                        psBandDesc->tBandId);
            }
        }

        // Remove this band from this service
        bSuccess = bRemoveBandFromService(psObj, psBandDesc);

        if (bSuccess == TRUE)
        {
            // Remove this entry from the list
            eReturnCode = OSAL.eLinkedListRemove(hBandEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                UN32 un32NumBands = 0;

                // Destroy the band descriptor and
                // its associated presets memory
                SMSO_vDestroy((SMS_OBJECT) psBandDesc->pasPresets);
                psBandDesc->pasPresets = NULL;
                SMSO_vDestroy((SMS_OBJECT) psBandDesc);

                // Find out how many bands are left
                OSAL.eLinkedListItems(
                    psObj->hBands,
                    &un32NumBands);

                // If there are no bands left,
                // clear out the current band attribute
                if (un32NumBands == 0)
                {
                    BOOLEAN bOK;

                    psObj->hCurrentBand = OSAL_INVALID_LINKED_LIST_ENTRY;

                    if (bUpdate == TRUE)
                    {
                        bOK = PRESETS_bCurrentBandSync(
                            hPresets, PRESET_BAND_INVALID_OBJECT);
                        if (bOK == FALSE)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(
                                gpacThisFile, __LINE__,
                                PRESETS_OBJECT_NAME":SF: Failed to syncronize"
                                " the current Band: %s (%#d)",
                                STRING.pacCStr(BAND.hName(psBandDesc->hBand)),
                                psBandDesc->tBandId);
                        }
                    }
                }
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *       PRESETS_eSetPreset
 *
 *       This function is called by a preset band in order to set an
 *       attribute of a specific preset.
 *
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESETS_eSetPreset(
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex,
    PRESET_UPDATE_TYPE_ENUM eUpdateType,
    SERVICE_ID tServiceId,
    const char *pacName,
    BOOLEAN bUpdateSmartFavorites
        )
{
    PRESETS_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    // Get the presets object from the band
    psObj = (PRESETS_OBJECT_STRUCT *)
        SMSO_hParent((SMS_OBJECT) hBand);

    if (psObj != NULL)
    {
        PRESET_STRUCT *psPreset;

        // Get this preset
        psPreset = psGetPreset(hBand, hBandEntry, tPresetIndex);

        if (psPreset == NULL)
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        // If we were instructed to update the name
        // for the preset do so
        if (eUpdateType == PRESET_UPDATE_TYPE_NAME)
        {
            // If the caller provided a valid character
            // pointer, then attempt to modify the STRING
            // for this preset's name.  If the caller provided
            // a null pointer, it means we are to clear this
            // preset's name
            if (pacName != NULL)
            {
                // We already have a string for this
                // preset's name.  Modify it
                if (psPreset->hName != STRING_INVALID_OBJECT)
                {
                    BOOLEAN bModified;

                    // Attempt to modify the preset name
                    // with the given string
                    bModified = STRING.bModifyCStr(
                        psPreset->hName,
                        pacName);

                    if (bModified != TRUE)
                    {
                        // Error!
                        return SMSAPI_RETURN_CODE_ERROR;
                    }
                }
                else // We gotta create a string for the preset name
                {
                    // Create a name for the preset
                    psPreset->hName =
                            STRING.hCreate(pacName, strlen(pacName));

                    if (psPreset->hName == STRING_INVALID_OBJECT)
                    {
                        // Error!
                        return SMSAPI_RETURN_CODE_ERROR;
                    }
                }
            }
            else // Clear the name
            {
                // Determine if an old name is present
                if (psPreset->hName != STRING_INVALID_OBJECT)
                {
                    // Destroy the old name
                    STRING_vDestroy(psPreset->hName);

                    // Clear the new name
                    psPreset->hName = STRING_INVALID_OBJECT;
                }
            }

            // All worked out okay
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        // If we were instructed to update the service id
        // for the preset do so
        else if (eUpdateType == PRESET_UPDATE_TYPE_CHAN)
        {
            BOOLEAN bBlocked = FALSE;

            // Grab the old service id
            SERVICE_ID tPrevServiceId = psPreset->tServiceId;

            // Initialize our return code
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

            // Block category notifications
            bBlocked = CATEGORY_bBlockNotifications(
                psObj->hDecoder, psObj->tCategoryId);

            // Do we need to rebuild the category?
            if (psObj->bRebuildCategory == TRUE)
            {
                // This only happens if somebody
                // outside of presets alters
                // the presets category
                BOOLEAN bRebuilt;

                bRebuilt = bRebuildCategory(psObj);

                if (bRebuilt != TRUE)
                {
                    // Releasing PRESETS Category notifications
                    if (bBlocked == TRUE)
                    {
                        CATEGORY_vReleaseNotifications(
                            psObj->hDecoder, psObj->tCategoryId);
                    }

                    // Error!
                    return SMSAPI_RETURN_CODE_ERROR;
                }
            }

            // Is the new service ID SERVICE_INVALID_ID?
            // If so, we're clearing this preset and we don't
            // need to be bothering the DECODER
            if (tServiceId != SERVICE_INVALID_ID)
            {
                BOOLEAN bDecoderLocked;

                // Verify and lock the Decoder so
                // we can test this channel id out
                bDecoderLocked =
                    SMSO_bLock((SMS_OBJECT) psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
                if (bDecoderLocked == TRUE)
                {
                    CCACHE_OBJECT hCCache;

                    // Get the ccache handle
                    hCCache = DECODER_hCCache(psObj->hDecoder);
                    if (hCCache == CCACHE_INVALID_OBJECT)
                    {
                        // We need the cache!
                        eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                    }
                    else
                    {
                        CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

                        // Get the channel, based on service id. Create a channel if
                        // one doesn't currently exist
                        hChannel = CCACHE_hChannelFromIds(
                            hCCache, tServiceId, CHANNEL_INVALID_ID, TRUE);

                        if (hChannel == CHANNEL_INVALID_OBJECT)
                        {
                            // This doesn't seem to be a valid service id...
                            eReturnCode = SMSAPI_RETURN_CODE_INVALID_CHANNEL;
                        }
                    }

                    SMSO_vUnlock((SMS_OBJECT) psObj->hDecoder);
                }
                else
                {
                    // Unable to get a hold of the decoder!
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                }
            }

            // Ensure everything is okay so far
            // and that a change occurred
            if (   (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
                && (tServiceId != tPrevServiceId)
               )
            {
                BOOLEAN bSuccess;

                if (tPrevServiceId == SERVICE_INVALID_ID)
                {
                    // This preset is new and has
                    // just been added
                    psPreset->eState = PRESET_STATE_ADDED;
                }
                else if (tServiceId == SERVICE_INVALID_ID)
                {
                    // This preset had a value before
                    // but has just been removed
                    psPreset->eState = PRESET_STATE_REMOVED;
                }
                else
                {
                    // Otherwise, this preset was updated
                    // from a previously valid channel to
                    // a new valid channel
                    psPreset->eState = PRESET_STATE_UPDATED;
                }

                // Set the new service Id
                psPreset->tServiceId = tServiceId;

#if SMS_DEBUG==1
                // This section is used to insert a channel object
                // handle used in debug prints
                if (tServiceId != SERVICE_INVALID_ID)
                {
                    BOOLEAN bLocked;

                    // Verify and lock object
                    bLocked = SMSO_bLock((SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
                    if (bLocked == TRUE)
                    {
                        CCACHE_OBJECT hCCache;
                        CHANNEL_OBJECT hChannel;

                        hCCache = DECODER_hCCache(psObj->hDecoder);

                        hChannel = CCACHE_hChannelFromIds(hCCache, tServiceId, CHANNEL_INVALID_ID, FALSE);
                        if (hChannel != CHANNEL_INVALID_OBJECT)
                        {
                            psPreset->hChannel = hChannel;
                        }

                        // Relinquish ownership
                        SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
                    }
                }
#endif // SMS_DEBUG==1

                // Update the channel preset handles
                bSuccess = bUpdateChannelPresetHdls(
                    psObj, psPreset, tPrevServiceId);

                if (bSuccess == TRUE)
                {
                    // To avoid duplication of notifications
                    // block one triggered from channel callback
                    PRESETS_vBlockNotifications((PRESETS_OBJECT)psObj, TRUE);

                    // Only need to update the category if
                    // all else has gone well
                    bSuccess = bUpdateCategory(psObj);
                    if (bSuccess == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            PRESETS_OBJECT_NAME": Failed to update category");

                        // Unblock notifications from channel callback
                        PRESETS_vBlockNotifications((PRESETS_OBJECT)psObj, FALSE);

                        // Releasing PRESETS Category notifications
                        if (bBlocked == TRUE)
                        {
                            CATEGORY_vReleaseNotifications(
                                psObj->hDecoder, psObj->tCategoryId);
                        }

                        return SMSAPI_RETURN_CODE_ERROR;
                    }

                    // Unblock notifications from channel callback
                    PRESETS_vBlockNotifications((PRESETS_OBJECT)psObj, FALSE);
                }

                if (bUpdateSmartFavorites == TRUE)
                {
                    bSuccess = PRESETS_bCurrentBandSync(
                        (PRESETS_OBJECT)psObj, hBand);
                    if (bSuccess == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            PRESETS_OBJECT_NAME
                            ": Failed to sync current preset band");
                    }
                }

                // Condition return value
                if (bSuccess == FALSE)
                {
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                }
                else
                {
                    eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
                }
            }

            // Releasing PRESETS Category notifications
            if (bBlocked == TRUE)
            {
                CATEGORY_vReleaseNotifications(
                    psObj->hDecoder, psObj->tCategoryId);
            }
        }

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            PRESET_BAND_TYPE_ENUM eType = BAND.eType(hBand);

            if (eType == PRESET_BAND_TYPE_USER)
            {
                // Notify only user band, since featured band cannot be modified
                // this way
                PRESETS_bUpdate((PRESETS_OBJECT)psObj,
                    hBand,
                    PRESETS_OBJECT_EVENT_BAND_MODIFIED);
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *       PRESETS_bGetPreset
 *
 *       This function is called by a preset band in order to get the
 *       attribute(s) of a specific preset.
 *
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
BOOLEAN PRESETS_bGetPreset(
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex,
    SERVICE_ID *ptServiceId,
    STRING_OBJECT *phName
        )
{
    PRESETS_OBJECT_STRUCT *psObj;
    BOOLEAN bSuccess = FALSE;

    if ((ptServiceId == NULL) && (phName == NULL))
    {
        return FALSE;
    }

    // Get the presets object from the band
    psObj = (PRESETS_OBJECT_STRUCT *)
        SMSO_hParent((SMS_OBJECT) hBand);

    if (psObj != NULL)
    {
        PRESET_STRUCT *psPreset;

        // Get this preset
        psPreset = psGetPreset(hBand, hBandEntry, tPresetIndex);

        // Did we find a valid preset?
        if (psPreset != NULL)
        {
            bSuccess = TRUE;

            // Copy out the attributes of this preset
            if (ptServiceId != NULL)
            {
                *ptServiceId = psPreset->tServiceId;
            }

            if (phName != NULL)
            {
                *phName = psPreset->hName;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *       PRESETS_bClearAllPresets
 *
 *       This function is called by a preset band in order to clear
 *       all of its presets.
 *
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
BOOLEAN PRESETS_bClearAllPresets(
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bClearNames,
    BOOLEAN bUpdateSmartFavorites
        )
{
    BOOLEAN bSuccess = FALSE, bLocked = FALSE;

    do
    {
        BOOLEAN bValid;

        PRESETS_OBJECT_STRUCT *psObj;
        PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;

        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Lock band (locks PRESETS)
        bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Get the presets object from the band
        psObj = (PRESETS_OBJECT_STRUCT *)SMSO_hParent((SMS_OBJECT)hBand);

        // Verify the presets service
        bValid = SMSO_bValid((SMS_OBJECT)psObj);
        if (bValid == FALSE)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psObj->hBands,
            &hEntry,
            n16CompareBands,
            hBand);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // All bands are belongs to the PRESETS
            // common list. If we cannot find some
            // recently added band, then we are in trouble
            break;
        }

        psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

        // Clear all of this band's presets but don't
        // remove them from the master preset list
        vClearBandPresets(psObj, psBandDesc, bClearNames, FALSE);

        // Update the Presets category
        bSuccess = bUpdateCategory(psObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Failed to update category");
            break;
        }

        if (bUpdateSmartFavorites == TRUE)
        {
            bSuccess = PRESETS_bCurrentBandSync((PRESETS_OBJECT)psObj,
                hBand);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Failed to sync current preset band");
                break;
            }
        }

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *       PRESETS_bSetAllFeaturedBanksRemovedFlag
 *
 *       This API is used to destroy all featured bands provided by the
 *       presets service.
 *
 *       Inputs:
 *           hPresets - A handle to a valid PRESETS object for which the caller
 *               wishes to get the previous band.
 *
 *       Outputs:
 *           A boolean value:
 *              TRUE - if bands are destroyed,
 *              FALSE - if an error occurred
 *
 *****************************************************************************/
BOOLEAN PRESETS_bSetAllFeaturedBanksRemovedFlag(
    PRESETS_OBJECT hPresets,
    BOOLEAN bRemoved
        )
{
    PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;
    BOOLEAN bSuccess;

    bSuccess = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bSuccess == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalCode = OSAL_SUCCESS;
        SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

        printf( "Setting all banks into removed state\n");
        // Iterate the list and remove only featured bands
        eOsalCode = OSAL.eLinkedListIterate(
            psObj->hBands,
            (OSAL_LL_ITERATOR_HANDLER)bSetAllFFBanksRemovedIterator,
            (void*)(size_t)bRemoved);

        if (eOsalCode == OSAL_SUCCESS)
        {
            // Change config only if we had something to iterate
            eReturnCode = CM_eCommitChangesToFile();
        }
        else if (eOsalCode == OSAL_NO_OBJECTS)
        {
            // Empty list. No issues.
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        // we done with this linked list entries
        SMSO_vUnlock((SMS_OBJECT)hPresets);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *       PRESETS_tGetCurrentBandId
 *
 *       This API is used to return Band ID of the current band.
 *
 *****************************************************************************/
size_t PRESETS_tGetCurrentBandId (
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked;
    size_t tBandId = FAVORITES_BAND_ID_INVALID;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;
        PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;

        psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
            OSAL.pvLinkedListThis( psObj->hCurrentBand );

        if (psBandDesc != NULL)
        {
            tBandId = psBandDesc->tBandId;
        }
        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }
    return tBandId;
}

/*****************************************************************************
 *
 *   PRESETS_bRemoveAllSelectedFFBanks
 *
 *****************************************************************************/
BOOLEAN PRESETS_bRemoveAllSelectedFFBanks (
    PRESETS_OBJECT hPresets
        )
{
    PRESETS_OBJECT_STRUCT *psObj =
        (PRESETS_OBJECT_STRUCT *)hPresets;
    BOOLEAN bLocked, bReturn = FALSE;
    PRESETS_FAVORITES_REMOVE_ITERATOR_STRUCT sIterator;

    BOOLEAN bComplete;
    DECODER_OBJECT hDecoder;

    hDecoder = PRESETS_hDecoder(hPresets);
    bComplete = RADIO_bGetFFTableStatus(hDecoder);

    if (bComplete == FALSE)
    {
        return TRUE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        /* Store the Current Band ID and Presets object handle */
        sIterator.tBandID = PRESETS_tGetCurrentBandId(hPresets);
        sIterator.hPresets = hPresets;

        // Iterate the list and remove only featured bands
        OSAL.eLinkedListIterate(
            psObj->hBands,
            (OSAL_LL_ITERATOR_HANDLER)bRemoveAllSelectedFFBanksIterator,
            &sIterator);

        // we done with this linked list entries
        SMSO_vUnlock((SMS_OBJECT)hPresets);
        bReturn = TRUE;
    }
    return bReturn;
}

/*****************************************************************************
 *
 *   PRESETS_bRemoveFirstSelectedFFBank
 *
 *****************************************************************************/
BOOLEAN PRESETS_bRemoveFirstSelectedFFBank (
    PRESETS_OBJECT hPresets
        )
{
    PRESETS_OBJECT_STRUCT *psObj =
        (PRESETS_OBJECT_STRUCT *)hPresets;
    BOOLEAN bLocked, bReturn = FALSE;
    PRESETS_FAVORITES_REMOVE_ITERATOR_STRUCT sIterator;

    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        /* Store the Current Band ID and Presets object handle */
        sIterator.tBandID = PRESETS_tGetCurrentBandId(hPresets);
        sIterator.hPresets = hPresets;

        // Iterate the list and remove only featured bands
        OSAL.eLinkedListIterate(
            psObj->hBands,
            (OSAL_LL_ITERATOR_HANDLER)bRemoveFirstSelectedFFBankIterator,
            &sIterator);

        // we done with this linked list entries
        SMSO_vUnlock((SMS_OBJECT)hPresets);
        bReturn = TRUE;
    }
    return bReturn;
}

/*****************************************************************************
 *
 *   PRESETS_bResetFeaturedFavoritesId
 *
 *****************************************************************************/
BOOLEAN PRESETS_bResetFeaturedFavoritesId (
    PRESETS_OBJECT hPresets
        )
{
    PRESETS_OBJECT_STRUCT *psObj = (PRESETS_OBJECT_STRUCT *)hPresets;
    BOOLEAN bSuccess;
    OSAL_RETURN_CODE_ENUM eReturn = OSAL_ERROR;

    bSuccess = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bSuccess == TRUE)
    {
        printf( "Clearing Favorites ID Pool\n");

        eReturn = OSAL.eLinkedListRemoveAll (
                psObj->hID,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy
                    );

        if (eReturn != OSAL_SUCCESS)
        {
            bSuccess = FALSE;
        }
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }
    return bSuccess;
}

/*****************************************************************************
 *
 *   PRESETS_bSetSelectedFavoritesBankRemoved
 *
 *****************************************************************************/
BOOLEAN PRESETS_bSetSelectedFavoritesBankRemoved (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_MARKER_STRUCT sMarkerStruct
        )
{
    PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;
    BOOLEAN bLocked, bReturn = TRUE;

    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        // Iterate the list and remove only featured bands
        OSAL.eLinkedListIterate(
            psObj->hBands,
            (OSAL_LL_ITERATOR_HANDLER)bSetSelectedFFBankRemovedIterator,
            &sMarkerStruct);

        CM_eCommitChangesToFile();

        // we done with this linked list entries
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }
    else
    {
        bReturn = FALSE;
    }

    return bReturn;
}

/*****************************************************************************
 *
 *   PRESETS_bSortBands
 *
 *****************************************************************************/
BOOLEAN PRESETS_bSortBands (
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        OSAL.eLinkedListSort(
            psObj->hBands,
            n16SortUserAndFeaturedFavorites
                );

        bSuccess = TRUE;

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *       PRESETS_eTune
 *
 *       This function is called by a preset band in order to tune to
 *       a specific preset.
 *
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESETS_eTune(
    DECODER_OBJECT hDecoder,
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex,
    BOOLEAN bLockOverride
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bDecoderLocked = FALSE;
    BOOLEAN bBandLocked = FALSE;
    BOOLEAN bOk = FALSE;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;

    do
    {
        PRESET_STRUCT *psPreset;

        bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bDecoderLocked != TRUE)
        {
            break;
        }

        // Verify and lock the band (locks the Presets service)
        bBandLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE );
        if (bBandLocked != TRUE)
        {
            break;
        }

        // Get this preset
        psPreset = psGetPreset(hBand, hBandEntry, tPresetIndex);
        if (psPreset == NULL)
        {
            // No preset - incorrect index
            break;
        }

        if (psPreset->tServiceId == SERVICE_INVALID_ID)
        {
            // Don't attempt to tune an empty preset
            eReturnCode = SMSAPI_RETURN_CODE_EMPTY_PRESET;
            break;
        }

        // Get channel ID.
        tChannelId = DECODER.tGetChannelId(
            hDecoder, 
            psPreset->tServiceId );

        // We're good since now
        bOk = TRUE;
    } while (FALSE);

    if (bBandLocked == TRUE)
    {
        // Release BAND object
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    if (bDecoderLocked == TRUE)
    {
        // Release DECODER object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    if (bOk == TRUE)
    {
        // Perform the tune
        eReturnCode = DECODER.eTuneDirect(
            hDecoder, 
            tChannelId,
            bLockOverride);
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *       PRESETS_eIterateBand
 *
 *       This function is called by a preset band in order to iterate
 *       through its presets.
 *
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESETS_eIterateBand(
    PRESET_BAND_OBJECT hBand,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        )
{
    PRESETS_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    // Verify the band and extract the parent PRESETS object
    psObj = (PRESETS_OBJECT_STRUCT *)
        SMSO_hParent((SMS_OBJECT) hBand);

    // Ensure we got something useful
    if (psObj != NULL)
    {
        // Pass iteration request to
        // the common iterator function
        eReturnCode = eIterateCommon(
            psObj,
            hBand,
            n16Iterator,
            pvIterateArg);
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *       PRESETS_n32PrintBand
 *
 *       This function is called by a preset band in order to print
 *       it's current status & presets.
 *
 *       Note: Only the PRESET_BAND_OBJECT should call this!
 *
 *****************************************************************************/
N32 PRESETS_n32PrintBand(
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    FILE *psFile
        )
{
    N32 n32Return = 0;
#if SMS_DEBUG == 1
    PRESETS_OBJECT_STRUCT *psObj;

    // Verify the band and extract the parent PRESETS object
    psObj = (PRESETS_OBJECT_STRUCT *)
        SMSO_hParent((SMS_OBJECT) hBand);

    if (psObj != NULL)
    {
        PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;
        PRESETS_PRINT_STRUCT sPrintBand;

        // Extract the band descriptor from the
        // provided band entry
        psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
            OSAL.pvLinkedListThis(hBandEntry);

        // Is the resulting pointer valid?
        if (psBandDesc == (PRESETS_BAND_DESCRIPTOR_STRUCT *) NULL)
        {
            // Bad!
            return 0;
        }

        // Does this entry agree on the band?
        if (psBandDesc->hBand == hBand)
        {
            // Initialize the print structure
            sPrintBand.psFile = psFile;
            sPrintBand.n32CharsWritten = 0;
            sPrintBand.pacFormat = "";
            sPrintBand.hPresets = (PRESETS_OBJECT)psObj;

            // Use the band print iterator
            bPrintBands(
                (void *) (size_t) psBandDesc,
                (void *) (size_t) &sPrintBand);

            // Copy back the number of characters
            // written by the band printer
            n32Return = sPrintBand.n32CharsWritten;
        }
    }
#endif
    return n32Return;
}

/*****************************************************************************
 *
 *       PRESETS_bThisPreset
 *
 *       This function is called by a channel object in order to extract
 *       the details of its current preset (provided via a CHANNEL_PRESET_HDL).
 *
 *       Note: Only the CHANNEL_OBJECT should call this!
 *
 *****************************************************************************/
BOOLEAN PRESETS_bThisPreset(
    CHANNEL_OBJECT hChannel,
    CHANNEL_PRESET_HDL hPresetHdl,
    PRESET_BAND_OBJECT *phBand,
    STRING_OBJECT *phPresetName,
    size_t *ptPresetIndex
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    CCACHE_OBJECT hCCache;
    DECODER_OBJECT hDecoder;
    PRESETS_OBJECT hPresets;

    // Verify we have a real handle here
    if (hPresetHdl == CHANNEL_PRESET_INVALID_HDL)
    {
        // Not an error
        return TRUE;
    }

    // All CHANNEL objects belong to a CCACHE object (their parent). So
    // here we extract the CCACHE handle.
    hCCache = (CCACHE_OBJECT) SMSO_hParent((SMS_OBJECT) hChannel);
    if (hCCache == CCACHE_INVALID_OBJECT)
    {
        // Error!
        return FALSE;
    }

    // All CCACHE objects belong to a DECODER object (their parent). So
    // here we extract the DECODER handle.
    hDecoder = (DECODER_OBJECT) SMSO_hParent((SMS_OBJECT) hCCache);
    if (hDecoder == DECODER_INVALID_OBJECT)
    {
        return FALSE;
    }

    // Now, get the presets object handle
    hPresets = DECODER_hPresets(hDecoder);

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // Now, extract the data for the caller
        PRESET_STRUCT *psPreset;

        // Extract this preset
        psPreset = (PRESET_STRUCT *) OSAL.pvLinkedListThis(
            (OSAL_LINKED_LIST_ENTRY) hPresetHdl);

        if (psPreset != NULL)
        {
            // Copy out only what the caller wants

            if (phBand != NULL)
            {
                *phBand = psPreset->hBand;
            }

            if (phPresetName != NULL)
            {
                *phPresetName = psPreset->hName;
            }

            if (ptPresetIndex != NULL)
            {
                *ptPresetIndex = psPreset->tPresetIndex;
            }

            // All worked out
            bSuccess = TRUE;
        }

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *       PRESETS_vUpdateHdl
 *
 *       This function is called by a category object in order to update
 *       which preset handle a channel should refer to as its "current".
 *       Used by CATEGORY.eIterate and indirectly by CHANNELLIST browsing.
 *
 *       Note: Only the CATEGORY_OBJECT should call this!
 *
 *****************************************************************************/
void PRESETS_vUpdateHdl(
    PRESETS_OBJECT hPresets,
    CHANNEL_OBJECT hChannel,
    size_t tHdlIndex
        )
{
    BOOLEAN bLocked;
    SERVICE_ID tServiceId;

    // Get and verify the service id
    tServiceId = CHANNEL.tServiceId(hChannel);
    if (tServiceId == SERVICE_INVALID_ID)
    {
        return;
    }

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *) hPresets;

        // Are we currently updating the category?
        if (psObj->bUpdatingCategory == FALSE)
        {
            CHANNEL_PRESET_HDL hBasePresetHdl;
            CHANNEL_PRESET_HDL hCurrentPresetHdl;

            // Get this channel's base preset handle
            // and start off the current as well
            hBasePresetHdl = CHANNEL_hGetBasePresetHdl(hChannel);
            hCurrentPresetHdl = hBasePresetHdl;

            // Ensure we have a valid starting point
            if (hBasePresetHdl != CHANNEL_PRESET_INVALID_HDL)
            {
                OSAL_LINKED_LIST_ENTRY hLastEntry =
                    (OSAL_LINKED_LIST_ENTRY) hBasePresetHdl;
                PRESET_STRUCT *psPreset;
                // We have a valid base preset handle.  So,
                // that means we have one entry already
                size_t tNumEntries = 1;

                // Find the last entry for this channel
                do
                {
                    // Get the next entry
                    hLastEntry = OSAL.hLinkedListNext(
                        hLastEntry, (void **) (void *) &psPreset);

                    // Ensure we have a valid entry & pointer
                    if ((hLastEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                                    || (psPreset == NULL))
                    {
                        // Make it look like we
                        // found nothing
                        tNumEntries = 0;

                        break;
                    }

                    // Did we wrap around?
                    if (hLastEntry == (OSAL_LINKED_LIST_ENTRY) hBasePresetHdl)
                    {
                        // All done!
                        break;
                    }

                    // If this preset's service id matches
                    // the service id we've been given,
                    // then we haven't reached the end yet
                    if (psPreset->tServiceId == tServiceId)
                    {
                        // Compute how many entries
                        // this service id has
                        tNumEntries++;
                    }

                    // Stop iterating if the channel ids don't match
                } while (psPreset->tServiceId == tServiceId);

                if (tNumEntries > 0)
                {
                    // Compute the requested preset hdl index
                    tHdlIndex = tHdlIndex % tNumEntries;

                    // Navigate to the correct preset entry
                    while (tHdlIndex > 0)
                    {
                        hCurrentPresetHdl = (CHANNEL_PRESET_HDL)
                            OSAL.hLinkedListNext(
                                (OSAL_LINKED_LIST_ENTRY) hCurrentPresetHdl,
                                NULL);

                        tHdlIndex--;
                    }
                }

                // Update the channel's current preset handle now
                CHANNEL_vSetCurrentPresetHdl(hChannel, hCurrentPresetHdl);
            }
        }

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return;
}

/*****************************************************************************
*
* PRESETS_hGetTag
*
*****************************************************************************/
TAG_OBJECT PRESETS_hGetTag(
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;
    PRESETS_OBJECT_STRUCT *psObj = (PRESETS_OBJECT_STRUCT *)hPresets;

    // Verify the object
    bLocked = SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        hTag = psObj->hTag;

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return hTag;
}

/*****************************************************************************
*
* PRESETS_hDecoder
*
*****************************************************************************/
DECODER_OBJECT PRESETS_hDecoder(
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bLocked;
    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
    PRESETS_OBJECT_STRUCT *psObj = (PRESETS_OBJECT_STRUCT *)hPresets;

    bLocked = SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        hDecoder = psObj->hDecoder;

        SMSO_vUnlock((SMS_OBJECT) hPresets);
    }

    return hDecoder;
}

/*****************************************************************************
*
* PRESETS_eGetPresetBandAndState
*
*****************************************************************************/
PRESET_BAND_STATE_ENUM PRESETS_eGetPresetBandAndState (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIteratorStruct,
    FAVORITES_BAND_CAPACITY tBandsLimit
        )
{
    BOOLEAN bLocked;
    PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;
    PRESET_BAND_STATE_ENUM eReturn = PRESET_BAND_STATE_NEW;

    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalCode;

        // Initialize with invalid object handle
        psIteratorStruct->hBand = PRESET_BAND_INVALID_OBJECT;
        psIteratorStruct->tCurrentBandId = PRESETS_tGetCurrentBandId(hPresets);

        // Search for band
        eOsalCode = OSAL.eLinkedListIterate(
                psObj->hBands,
                bSearchForFavoritesBandIterator,
                (void *)psIteratorStruct
                    );

        if (eOsalCode == OSAL_SUCCESS || eOsalCode == OSAL_NO_OBJECTS)
        {
            if (psIteratorStruct->hBand != PRESET_BAND_INVALID_OBJECT)
            {
                FAVORITES_BAND_SEQUENCE tSequence;
                eReturn = PRESET_BAND_STATE_EXISTING;

                tSequence = PRESET_BAND_tGetSequence(psIteratorStruct->hBand);
                if (tSequence != psIteratorStruct->tSeq)
                {
                    eReturn = PRESET_BAND_STATE_MODIFIED;
                }
            }
            else
            {
                // Here is a new band, but we're already have much enough
                if (psIteratorStruct->un32BandsCount >= tBandsLimit)
                {
                    if (psIteratorStruct->un32Removed == 0)
                    {
                        eReturn = PRESET_BAND_STATE_REJECTED;
                    }
                    else
                    {
                        BOOLEAN bOk;

                        bOk = PRESETS_bRemoveFirstSelectedFFBank(hPresets);
                        if (bOk == FALSE)
                        {
                            // Can remove no band... Sorry.
                            eReturn = PRESET_BAND_STATE_REJECTED;
                        }
                    }
                }
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Cannot iterate bands list (%s)",
                OSAL.pacGetReturnCodeName(eOsalCode));
            eReturn = PRESET_BAND_STATE_REJECTED;
        }

        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }
    return eReturn;
}

/*****************************************************************************
*
* PRESETS_bAddToID
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM PRESETS_bAddToID (
    PRESETS_OBJECT hPresets,
    FAVORITES_BAND_ID tId
        )
{
    BOOLEAN bLocked;
    OSAL_RETURN_CODE_ENUM eReturn = OSAL_ERROR;

    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        OSAL_LINKED_LIST_ENTRY hEntry =
                OSAL_INVALID_LINKED_LIST_ENTRY;

        FAVORITES_BAND_ID *ptTemp = (FAVORITES_BAND_ID *)SMSO_hCreate(
                PRESETS_OBJECT_NAME":FFBankId",
                sizeof(FAVORITES_BAND_ID),
                SMS_INVALID_OBJECT,
                FALSE);

        if (ptTemp != NULL)
        {
            *ptTemp = tId;

            eReturn = OSAL.eLinkedListAdd (
                 psObj->hID,
                 &hEntry,
                 ptTemp);

            if(eReturn != OSAL_SUCCESS)
            {
                SMSO_vDestroy((SMS_OBJECT)ptTemp);
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    return eReturn;
}

/*****************************************************************************
*
* PRESETS_un32BankIDCount
*
*****************************************************************************/
UN32 PRESETS_un32BankIDCount (
    PRESETS_OBJECT hPresets
        )
{
    UN32 un32Items = 0;
    BOOLEAN bLocked;

    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        OSAL.eLinkedListItems (
                psObj->hID,
                &un32Items);

        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    return un32Items;
}

/*****************************************************************************
*
* PRESETS_vUpdateBankDescriptorOrder
*
*****************************************************************************/
void PRESETS_vUpdateBankDescriptorOrder (
    OSAL_LINKED_LIST_ENTRY hEntry,
    FAVORITES_BAND_ORDER tOrder
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;
    // Get This Preset Band descriptor
    if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        psBandDesc =
                (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);
        // Modify Preset Band descriptor
        psBandDesc->tOrder = tOrder;
    }
    return;
}

/*****************************************************************************
 *
 * PRESETS_bRemoveAllFeaturedBands
 *
 *****************************************************************************/
BOOLEAN PRESETS_bRemoveAllFeaturedBands (
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bOwner, bReturn = FALSE;

    // Verify objcet ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hPresets);
    if (bOwner == TRUE)
    {
        // De-reference object
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        OSAL.eLinkedListIterate(
            psObj->hBands,
            bRemoveAllFavorites,
            (void *)NULL);

        bReturn = TRUE;
    }

    return bReturn;
}

/*******************************************************************************
 *
 *   PRESETS_bGetCapabilities
 *
 *******************************************************************************/
BOOLEAN PRESETS_bGetCapabilities (
    PRESETS_OBJECT hPresets,
    FAVORITES_RECEIVER_INFO_STRUCT *psInfo
        )
{
    BOOLEAN bLocked;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESETS_OBJECT_STRUCT *psObj = (PRESETS_OBJECT_STRUCT *)hPresets;

        // Pass thru target info
        *psInfo = psObj->sCapabilities;

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    return bLocked;
}

/*******************************************************************************
 *
 *   PRESETS_bUpdate
 *
 *******************************************************************************/
BOOLEAN PRESETS_bUpdate (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    PRESETS_EVENT_MASK tUpdateMask
        )
{
    BOOLEAN bReturn = FALSE, bValid;

    // Verify object
    bValid = SMSO_bValid((SMS_OBJECT)hPresets);
    if (bValid == TRUE)
    {
        // De-reference object
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        // Update event mask
        SMSU_tUpdate(&psObj->sEvent, tUpdateMask);

        // Save band to notify to in the callback shim
        psObj->sCallbackShim.hBand = hBand;

        // Notify all interested parties
        bReturn = SMSU_bNotify(&psObj->sEvent);
    }

    return bReturn;
}

/*******************************************************************************
 *
 *   PRESETS_eSetBandSyncCategory
 *
 *******************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESETS_eSetBandSyncCategory (
    PRESETS_OBJECT hPresets,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        // Set category id
        psObj->tCategorySyncId = tCategoryId;

        if (tCategoryId != CATEGORY_INVALID_ID)
        {
            BOOLEAN bSuccess;
            PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;

            if (psObj->hCurrentBand != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;

                psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                    OSAL.pvLinkedListThis( psObj->hCurrentBand );

                hBand = psBandDesc->hBand;
            }

            // If there is no bands, hBand will be INVALID and nothing 
            // will happen, i.e., there is nothing to synchronize.
            bSuccess = PRESETS_bCurrentBandSync(hPresets, hBand);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Failed to sync current preset band");
            }
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    return eReturnCode;
}

/*******************************************************************************
 *
 *   PRESETS_bCurrentBandSync
 *
 *******************************************************************************/
BOOLEAN PRESETS_bCurrentBandSync (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand
        )
{
    DECODER_OBJECT hDecoder;
    BOOLEAN bLocked = FALSE, bDecoderLocked = FALSE, bSuccess = FALSE;
    CHANNEL_OBJECT *phChannels = NULL;

    do
    {
        size_t tIndex = 0, tArraySize = 0;
        PRESETS_OBJECT_STRUCT *psObj;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;

        // Get decoder instance
        hDecoder = PRESETS_hDecoder(hPresets);

        // Lock decoder instance
        bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bDecoderLocked == FALSE)
        {
            break;
        }

        // Verify and lock object
        bLocked = SMSO_bLock((SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // De-reference object
        psObj = (PRESETS_OBJECT_STRUCT *)hPresets;

        // Check that presets are in 'Smart' mode
        if (psObj->tCategorySyncId == CATEGORY_INVALID_ID)
        {
            bSuccess = TRUE;
            break;
        }

        // Make sure, that the current band was updated
        if ((hBand != PRESET_BAND_INVALID_OBJECT) &&
            (psObj->hCurrentBand != OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                OSAL.pvLinkedListThis(psObj->hCurrentBand);

            // There are many conditions when current band need 
            // to be syncronized, e.g. removal or update. However,
            // in the same conditions could be updated not current
            // bands. To eliminate unnecessary operations, we are 
            // going to receive a band handle which is currently 
            // updated and then perform it's comparison to determine
            // necessity of syncronization.

            // Compare bands
            if (hBand != psBandDesc->hBand)
            {
                bSuccess = TRUE;
                break;
            }
        }

        puts(PRESETS_OBJECT_NAME
            ":SF: Updating Current Band with Smart Favorites");

        // Do we have a current band?
        if (psObj->hCurrentBand != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Make sure, that we have it!
            if (psBandDesc == NULL)
            {
                psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                    OSAL.pvLinkedListThis(psObj->hCurrentBand);
            }

            // Allocate channel array
            phChannels = (CHANNEL_OBJECT *)
                SMSO_hCreate(
                    PRESETS_OBJECT_NAME": SyncChanArr",
                    psBandDesc->tCapacity * sizeof(CHANNEL_OBJECT),
                    SMS_INVALID_OBJECT,
                    FALSE);

            if (phChannels == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME
                    ":SF: Failed to allocate Channels Array");
                break;
            }

            // Initialize array
            for (tIndex = 0; tIndex < psBandDesc->tCapacity; tIndex++)
            {
                phChannels[tIndex] = CHANNEL_INVALID_OBJECT;
            }

            // Iterate thru band presets and populate them
            for (tIndex = 0; tIndex < psBandDesc->tCapacity; tIndex++)
            {
                SERVICE_ID tServiceId;
                CHANNEL_ID tChannelId;

                CHANNEL_OBJECT hChannel;

                // Get Channel Cache
                CCACHE_OBJECT hCCache = DECODER_hCCache(psObj->hDecoder);

                // Get Service Id
                tServiceId = psBandDesc->pasPresets[tIndex].tServiceId;

                // Seems like this preset is empty
                if (tServiceId == SERVICE_INVALID_ID)
                {
                    continue;
                }

                // Get channel handle
                hChannel = CCACHE_hChannelFromIds(hCCache, tServiceId, CHANNEL_INVALID_ID, FALSE);

                // Get Channel Id
                tChannelId = CHANNEL.tChannelId(hChannel);

                // Populate this channel only if it is valid (both SID and CID exist)
                if (tChannelId != CHANNEL_INVALID_ID)
                {
                    phChannels[tArraySize++] = hChannel;
                }
            }
        }

        // Perform sync category update
        eReturnCode = CATEGORY_eFill(psObj->hDecoder, psObj->tCategorySyncId, phChannels, tArraySize);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

    } while(FALSE);

    if (phChannels != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)phChannels);
    }

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    if (bDecoderLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   PRESETS_hCurrentBandEntry
 *
 *******************************************************************************/
OSAL_LINKED_LIST_ENTRY PRESETS_hCurrentBandEntry(
    PRESETS_OBJECT hPresets
        )
{
    BOOLEAN bValid;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    bValid = SMSO_bValid((SMS_OBJECT)hPresets);
    if (bValid == TRUE)
    {
        // De-reference object
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        hEntry = psObj->hCurrentBand;
    }

    return hEntry;
}

/*****************************************************************************
*
*   PRESETS_vBlockNotifications
*
*   This API blocks/unblocks notifications from channel updates
*
*****************************************************************************/
void PRESETS_vBlockNotifications (
    PRESETS_OBJECT hPresets,
    BOOLEAN bBlock
        )
{
    PRESETS_OBJECT_STRUCT *psPresets =
        (PRESETS_OBJECT_STRUCT *)hPresets;

    if (psPresets != NULL)
    {
        psPresets->bBlockNotifications = bBlock;
    }
}

/*****************************************************************************
*
*   PRESETS_vFavoritesStart
*
*   This API must be called only from the context of DECODER task
*
*****************************************************************************/
void PRESETS_vFavoritesStart (
    PRESETS_OBJECT hPresets,
    SMS_EVENT_FEATURED_FAVORITIES_ENABLE_STRUCT const * psEnable
        )
{
    BOOLEAN bPresetsLocked = FALSE;

    do
    {
        BOOLEAN bResult;
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        // Verify input parameters
        if ( hPresets == PRESETS_INVALID_OBJECT )
        {
            break;
        }

        // Verify and lock object
        bPresetsLocked = SMSO_bLock(
            (SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bPresetsLocked == FALSE)
        {
            break;
        }

        // Check if already enabled, if so bail
        if(psObj->bFeaturedFavoritesEnabled == TRUE)
        {
            // Already enabled!
            break;
        }

        bResult = RADIO_bFeaturedFavoritesEnable(psObj->hDecoder, TRUE);
        if(bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Cannot enable Featured Favorites"
                    );
            break;
        }

        if ((psObj->sCapabilities.tCapacity !=
                psEnable->sCapabilities.tCapacity) ||
            (psObj->sCapabilities.tPurpose !=
                psEnable->sCapabilities.tPurpose) ||
            (psObj->sCapabilities.tBandsNumber !=
                psEnable->sCapabilities.tBandsNumber))
        {
            BOOLEAN bSuccess;
            PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT sIterator;

            puts(PRESETS_OBJECT_NAME
                ": Featured Favorites Re-Configuration required");

            // Need to re-configure bands number
            if ((psObj->sCapabilities.tBandsNumber >
                    psEnable->sCapabilities.tBandsNumber) &&
                (psObj->hCurrentBand != OSAL_INVALID_LINKED_LIST_ENTRY))
            {
                PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc =
                    (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                        OSAL.pvLinkedListThis(psObj->hCurrentBand);

                // This block is used to eliminate extra bands.
                // In case when the greater number of bands required,
                // it will be enough to request update

                sIterator.tBandsNumber =
                    psEnable->sCapabilities.tBandsNumber;
                sIterator.un32BandsCount = 0;
                sIterator.tCurrentBandId = psBandDesc->tBandId;
                sIterator.hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                // Remove limited bands
                vRemoveExtraBands(psObj, &sIterator);

                // Current band is out of scope
                if (sIterator.hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                {
                    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                    // Get the previous entry
                    hEntry = OSAL.hLinkedListPrev(sIterator.hEntry, (void **)&psBandDesc);
                    if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
                    {
                        // Remove the previous band to fit into
                        // the requested capacity
                        PRESET_BAND_vDestroy(psBandDesc->hBand);
                    }
                }
            }

            // Need to re-configure bands capacity
            if (psObj->sCapabilities.tCapacity !=
                    psEnable->sCapabilities.tCapacity)
            {
                sIterator.hPresets = hPresets;
                sIterator.tCapacity =
                    psEnable->sCapabilities.tCapacity;

                // Adjust presets number
                vUpdateBandsCapacity(psObj, &sIterator);
            }

            // Mark as 'in-progress'
            bSuccess = RADIO_bUpdateFeaturedFavoritesState(
                psObj->hDecoder, FALSE);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Failed to update Featured "
                    "Favorites Table state");
                break;
            }

            // Force update of bands
            vClearBandSequenceNumbers(psObj);
        }

        // Assign receiver targeting info
        psObj->sCapabilities = psEnable->sCapabilities;

        // Save new target info
        vSaveReceiverCapabilities(psObj);

        // Add featured mode
        psObj->bFeaturedFavoritesEnabled = TRUE;

    } while (FALSE);

    if (TRUE == bPresetsLocked)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    return;
}

/*****************************************************************************
*
*   PRESETS_vFavoritesStop
*
*   This API must be called only from the context of DECODER task
*
*****************************************************************************/
void PRESETS_vFavoritesStop (
    PRESETS_OBJECT hPresets,
    SMS_EVENT_FEATURED_FAVORITIES_DISABLE_STRUCT const * psDisable
        )
{
    do
    {
        BOOLEAN bPresetsLocked = FALSE, bResult;
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        // Verify input parameters
        if ( hPresets == PRESETS_INVALID_OBJECT )
        {
            break;
        }

        // Verify and lock object
        bPresetsLocked = SMSO_bLock(
            (SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bPresetsLocked == FALSE)
        {
            break;
        }

        // Check if already disabled, if so bail
        if(psObj->bFeaturedFavoritesEnabled == FALSE)
        {
            // Already disabled!

            // Relinquish ownership
            SMSO_vUnlock((SMS_OBJECT)hPresets);
            break;
        }

        bResult = PRESETS_bResetFeaturedFavoritesId(hPresets);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Cannot reset Featured Favorites"
                " IDs");

            // Relinquish ownership
            SMSO_vUnlock((SMS_OBJECT)hPresets);
            break;
        }

        // Remove all featured bands if required
        if (psDisable->bRemoveBands == TRUE)
        {
            // Remove all featured bands
            bResult = PRESETS_bRemoveAllFeaturedBands(hPresets);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Cannot cleanup obsolete "
                    "Featured Bands");

                // Relinquish ownership
                SMSO_vUnlock((SMS_OBJECT)hPresets);
                break;
            }

            // Mark this table as 'in-progress'
            bResult = RADIO_bUpdateFeaturedFavoritesState(
                psObj->hDecoder, FALSE);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Failed to update Featured "
                    "Favorites Table State");

                // Relinquish ownership
                SMSO_vUnlock((SMS_OBJECT)hPresets);
                break;
            }
        }
        else
        {
            bResult = PRESETS_bRemoveAllSelectedFFBanks(hPresets);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Cannot cleanup obsolete "
                    "Featured Bands");

                // Relinquish ownership
                SMSO_vUnlock((SMS_OBJECT)hPresets);
                break;
            }
        }

        bResult = RADIO_bFeaturedFavoritesEnable(psObj->hDecoder, FALSE);
        if(bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Cannot disable Featured Favorites"
                    );

            // This should not block the whole service from moving into
            // correct status.
        }

        // Update preset object

        // Remove featured mode
        psObj->bFeaturedFavoritesEnabled = FALSE;

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);

    } while (FALSE);

    return;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*******************************************************************************
 *
 *   bRegisterCategory
 *
 *   This function performs all the necessary steps needed to register
 *   use of the presets category.
 *
 *******************************************************************************/
static BOOLEAN bRegisterCategory (
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    BOOLEAN bSuccess = FALSE;

    // Lock the decoder
    bLocked =
        SMSO_bLock((SMS_OBJECT) psObj->hDecoder,OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;

        // Get the ccache handle
        hCCache = DECODER_hCCache( psObj->hDecoder );

        if (hCCache != CCACHE_INVALID_OBJECT)
        {
            CATEGORY_OBJECT hPresetsCategory;

            // Get the category handle
            hPresetsCategory = CCACHE_hCategory(
                hCCache, &psObj->tCategoryId, 0);

            if (hPresetsCategory != CATEGORY_INVALID_OBJECT)
            {
                // Register use of this category
                bSuccess = CATEGORY_bRegisterNotification(
                    hPresetsCategory,
                    CATEGORY_OBJECT_EVENT_ALL,
                    vCategoryEventCallback,
                    psObj);
            }
        }

        SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   vRemoveCategory
 *
 *   This function performs all the necessary steps to remove the presets
 *   category from SMS.
 *
 *******************************************************************************/
static void vRemoveCategory (
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    // Remove the category if necessary
    if (psObj->tCategoryId != CATEGORY_INVALID_ID)
    {
        CCACHE_OBJECT hCCache;
        CATEGORY_OBJECT hCategory;

        hCCache = DECODER_hCCache(psObj->hDecoder);
        hCategory = CCACHE_hCategory(hCCache, &psObj->tCategoryId, 0);

        if (hCategory != CATEGORY_INVALID_OBJECT)
        {
            CATEGORY_vUnregisterNotification(
                hCategory,
                (CATEGORY_OBJECT_EVENT_CALLBACK)vCategoryEventCallback,
                (void *)psObj);
        }

        // Flag that we're updating the category
        // so we ignore update callbacks
        psObj->bUpdatingCategory = TRUE;

        // Destroy the category

        // First remove this category from the ccache
        CCACHE_vRemoveCategory(hCCache, psObj->tCategoryId);

        // Complete the cleanup
        CATEGORY_vDestroy( hCategory );

        // Clear the category id
        psObj->tCategoryId = CATEGORY_INVALID_ID;

        // We're done updating the category
        psObj->bUpdatingCategory = FALSE;

    }

    return;
}

/*******************************************************************************
 *
 *   bRebuildCategory
 *
 *******************************************************************************/
static BOOLEAN bRebuildCategory (
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    PRESET_STRUCT *psPreset;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_SUCCESS;
    size_t tPresetIndex = 0, tSize = 0;
    BOOLEAN bUpdateComplete = FALSE;
    OSAL_LINKED_LIST_ENTRY hFirstBand;
    OSAL_LINKED_LIST_ENTRY hCurrentBand;
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;

    // Get the first band & its descriptor
    hFirstBand = OSAL.hLinkedListFirst(
        psObj->hBands,
        (void **)(void *)&psBandDesc);

    // Use this as our current
    hCurrentBand = hFirstBand;

    // Verify that worked
    if (   (hCurrentBand == OSAL_INVALID_LINKED_LIST_ENTRY)
        || (psBandDesc == NULL)
        )
    {
        return FALSE;
    }

    // Get category size
    tSize = CATEGORY.tSize(psObj->hDecoder, psObj->tCategoryId);

    if (tSize > 0)
    {
        // Iterate category and remove all channels
        eReturnCode = CATEGORY.eIterate(
            psObj->hDecoder, 
            psObj->tCategoryId, 
            n16RemoveAllChannels, 
            NULL
                );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Failed to remove all channels "
                "from Category Id: %d, %s (#%d)",
                psObj->tCategoryId,
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);

            return FALSE;
        }
    }

    do {
        // Get the next preset
        psPreset = &psBandDesc->pasPresets[tPresetIndex++];

        // Only insert valid channels
        if (psPreset->tServiceId != SERVICE_INVALID_ID)
        {
            eReturnCode = CATEGORY_eInsertNewChannel(
                psObj->hDecoder,
                psObj->tCategoryId,
                psPreset->hChannel);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

        // Did we reach the end of this band?
        if (tPresetIndex == psBandDesc->tCapacity)
        {
            // We're now at the first index
            // of the next band
            tPresetIndex = 0;

            // Move on to the next band
            hCurrentBand = OSAL.hLinkedListNext(
                hCurrentBand,
                (void **) (void*) &psBandDesc);

            // Verify we didn't encounter any errors
            if (   (hCurrentBand == OSAL_INVALID_LINKED_LIST_ENTRY)
                || (psBandDesc == NULL)
               )
            {
                // Mark the error condition
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }

            // Determine if we've wrapped back around
            // to the first band
            if (hCurrentBand == hFirstBand)
            {
                // All done!
                bUpdateComplete = TRUE;
            }
        }

        // Stop if we've iterated through all the presets
    } while (bUpdateComplete == FALSE);

    // Condition return value
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    // We don't need to rebuild the category anymore!
    psObj->bRebuildCategory = FALSE;

    return TRUE;
}

/*******************************************************************************
 *
 *   bUpdateCategory
 *
 *   This function updates the Presets virtual category to match the current
 *   state of the Presets service.  Only Presets with valid channel Ids will
 *   be a part of the category.
 *
 *******************************************************************************/
static BOOLEAN bUpdateCategory(
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess, bBlocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    PRESETS_CATEGORY_UPDATE_STRUCT sUpdate;

    // Initialize Update Struct
    sUpdate.bError = FALSE;
    sUpdate.bUpdateComplete = FALSE;
    sUpdate.tLastChannelIndex = CATEGORY_CHANNEL_INVALID_INDEX;
    sUpdate.bChanRemoved = FALSE;

    // Block category notifications
    bBlocked = CATEGORY_bBlockNotifications(
        psObj->hDecoder, psObj->tCategoryId);

    // Get the first band and band descriptor
    sUpdate.hFirstBand = OSAL.hLinkedListFirst(
        psObj->hBands,
        (void **) (void *) &sUpdate.psBandDesc);

    // Initialize the current band & preset
    sUpdate.hCurrentBand = sUpdate.hFirstBand;
    sUpdate.tPresetIndex = 0;

    // Indicate that we are currently updating the category.
    // This stops us from reacting to category update callback
    // indications as well as tells us to ignore any calls
    // to PRESETS_vUpdateHdl
    psObj->bUpdatingCategory = TRUE;

    // Call the category iterator
    eReturnCode = CATEGORY.eIterate(
        psObj->hDecoder,
        psObj->tCategoryId,
        n16IterateCategoryForUpdate,
        (void *) &sUpdate);

    // If the iterator encountered an error,
    // indicate failure to the caller
    if (   (eReturnCode == SMSAPI_RETURN_CODE_ERROR)
        || (sUpdate.bError == TRUE)
       )
    {
        bSuccess = FALSE;
    }
    // Determine if we need to finish up
    // the category update
    else if (sUpdate.bUpdateComplete == FALSE)
    {
        // Complete the category update process
        bSuccess = bCompleteCategoryUpdate(
            psObj, &sUpdate);
    }
    else
    {
        bSuccess = TRUE;
    }

    // We are no longer updating the category
    psObj->bUpdatingCategory = FALSE;

    if (bBlocked == TRUE)
    {
        // Release category notifications
        CATEGORY_vReleaseNotifications(
            psObj->hDecoder, psObj->tCategoryId);
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   bCompleteCategoryUpdate
 *
 *   This function finalizes the updates to the category if the category
 *   iterator callback is unable to.  For example, the category iterator
 *   callback won't be invoked if the cateogry is empty and we want to
 *   add a channel.  So, this function only performs channel inserts which
 *   occur at the "end" of the category.
 *
 *******************************************************************************/
static BOOLEAN bCompleteCategoryUpdate(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESETS_CATEGORY_UPDATE_STRUCT *psUpdate
        )
{
    PRESET_STRUCT *psPreset;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_SUCCESS;

    do {

        if (psUpdate->psBandDesc == NULL)
        {
            // Nothing to update
            break;
        }

        // Get the next preset
        psPreset = &psUpdate->psBandDesc->pasPresets[psUpdate->tPresetIndex++];

        // Only update the new presets
        if (psPreset->eState == PRESET_STATE_ADDED)
        {
            eReturnCode = CATEGORY_eInsertNewChannel(
                psObj->hDecoder,
                psObj->tCategoryId,
                psPreset->hChannel
                    );

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

        // This preset has been handled
        psPreset->eState = PRESET_STATE_UNCHANGED;

        // Did we reach the end of this band?
        if (psUpdate->tPresetIndex == psUpdate->psBandDesc->tCapacity)
        {
            // We're now at the first index
            // of the next band
            psUpdate->tPresetIndex = 0;

            // Move on to the next band
            psUpdate->hCurrentBand = OSAL.hLinkedListNext(
                psUpdate->hCurrentBand,
                (void **) (void*) &psUpdate->psBandDesc);

            // Verify we didn't encounter any errors
            if (   (psUpdate->hCurrentBand == OSAL_INVALID_LINKED_LIST_ENTRY)
                || (psUpdate->psBandDesc == NULL)
               )
            {
                // Mark the error condition
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }

            // Determine if we've wrapped back around
            // to the first band
            if (psUpdate->hCurrentBand == psUpdate->hFirstBand)
            {
                // All done!
                psUpdate->bUpdateComplete = TRUE;
            }
        }

        // Stop if we've iterated through all the presets
    } while (psUpdate->bUpdateComplete == FALSE);

    // Condition return value
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
 *
 *   n16CompareBands
 *
 *   Just looks for a matching band handle.
 *
 *******************************************************************************/
static N16 n16CompareBands(
    void *pvObj1,
    void *pvObj2
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc =
        (PRESETS_BAND_DESCRIPTOR_STRUCT *) pvObj1;
    PRESET_BAND_OBJECT hBand =
        (PRESET_BAND_OBJECT) pvObj2;

    if ((psBandDesc != NULL) && (psBandDesc->hBand == hBand))
    {
        // Match!
        return 0;
    }

    return 1;
}

/*******************************************************************************
 *
 *   n16SortPresetsByServiceId
 *
 *   This function compares two PRESET_STRUCT * and orders them such
 *   that presets are ordered by channel, then by band, then by preset index.
 *   This ensures that the list is grouped at the top level by service id, then
 *   at the next level by band, then at the next level by preset index.
 *
 *   Example ( 3 bands ):
 *
 *                  +---+---+---+---+---+---+---+
 *                  | 0 | 1 | 2 | 3 | 4 | 5 | 6 |
 *   +--------------+---+---+---+---+---+---+---+
 *   | Service Id   | 1 | 1 | 1 | 2 | 2 | 3 | 3 |
 *   +--------------+---+---+---+---+---+---+---+
 *   | Band Id      | 0 | 0 | 1 | 2 | 2 | 2 | 2 |
 *   +--------------+---+---+---+---+---+---+---+
 *   | Preset Index | 0 | 1 | 9 | 2 | 4 | 0 | 1 |
 *   +--------------+---+---+---+---+---+---+---+
 *
 *   Inputs:
 *       pvArg1 (in list), pvArg2 (compare against)
 *
 *   Outputs:
 *       0   - Objects have the same value (equal, error)
 *       > 0 - Object1(in list) is greater than (after) Object2
 *       < 0 - Object1(in list) is less than (before) Object2
 *
 *******************************************************************************/
static N16 n16SortPresetsByServiceId(
    void *pvObj1,
    void *pvObj2
        )
{
    PRESET_STRUCT *psPreset1 =
        (PRESET_STRUCT *) pvObj1;
    PRESET_STRUCT *psPreset2 =
        (PRESET_STRUCT *) pvObj2;

    // First, order by channel
    if (psPreset1->tServiceId < psPreset2->tServiceId)
    {
        return -1;
    }
    else if (psPreset1->tServiceId > psPreset2->tServiceId)
    {
        return 1;
    }

    // Service Ids are equal -- sort by band
    if (psPreset1->tBandId < psPreset2->tBandId)
    {
        return -1;
    }
    else if (psPreset1->tBandId > psPreset2->tBandId)
    {
        return 1;
    }

    // Channel Ids and Band Ids are equal --
    // sory by preset index within the band
    if (psPreset1->tPresetIndex < psPreset2->tPresetIndex)
    {
        return -1;
    }
    else if (psPreset1->tPresetIndex > psPreset2->tPresetIndex)
    {
        return 1;
    }

    // Everything is equal
    return 0;
}


/*******************************************************************************
 *
 *   n16SortUserAndFeaturedFavorites
 *
 *******************************************************************************/
static N16 n16SortUserAndFeaturedFavorites (
    void *pvObj1,
    void *pvObj2
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT
            *psBandDesc1 = (PRESETS_BAND_DESCRIPTOR_STRUCT *)pvObj1,
            *psBandDesc2 = (PRESETS_BAND_DESCRIPTOR_STRUCT *)pvObj2;

    if ((psBandDesc1->eType == PRESET_BAND_TYPE_USER) &&
        (psBandDesc2->eType == PRESET_BAND_TYPE_USER))
    {
        if (psBandDesc1->tBandId > psBandDesc2->tBandId)
        {
            return 1;
        }
        else
        {
            return -1;
        }
    }
    else if ((psBandDesc1->eType == PRESET_BAND_TYPE_USER) &&
             (psBandDesc2->eType == PRESET_BAND_TYPE_FEATURED))
    {
        return -1;
    }
    else if ((psBandDesc1->eType == PRESET_BAND_TYPE_FEATURED) &&
             (psBandDesc2->eType == PRESET_BAND_TYPE_USER))
    {
        return 1;
    }
    else if ((psBandDesc1->eType == PRESET_BAND_TYPE_FEATURED) &&
             (psBandDesc2->eType == PRESET_BAND_TYPE_FEATURED))
    {
        if (psBandDesc1->tOrder > psBandDesc2->tOrder)
        {
            return 1;
        }
        else
        {
            return -1;
        }
    }

    return 0;
}

/*******************************************************************************
 *
 *   hCreateBandDescriptor
 *
 *   This function creates a PRESETS_BAND_DESCRIPTOR_STRUCT, populates it with
 *   the provided parameters, and adds the band descriptor to the band list,
 *   providing the band's linked list entry back to the caller.
 *
 *******************************************************************************/
static OSAL_LINKED_LIST_ENTRY hCreateBandDescriptor(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_OBJECT hBand,
    size_t tBandCapacity,
    size_t tBandId
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Name the new band descriptor
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                PRESETS_OBJECT_NAME"%u: Band %u",
                psObj->tCategoryId,
                tBandId);

        // Allocate memory for the new band descriptor
        psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(PRESETS_BAND_DESCRIPTOR_STRUCT),
                (SMS_OBJECT) psObj,
                FALSE);

        if (psBandDesc == NULL)
        {
            // Stop & return failure
            return OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        // Name the band's preset memory
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                PRESETS_OBJECT_NAME"%u: Band %u Presets",
                psObj->tCategoryId,
                tBandId);

        // Allocate space for this band's presets
        psBandDesc->pasPresets = (PRESET_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(PRESET_STRUCT) * tBandCapacity,
                (SMS_OBJECT) psObj,
                FALSE);

        if (psBandDesc->pasPresets == NULL)
        {
            // Stop & return failure (and destroy)
            SMSO_vDestroy((SMS_OBJECT) psBandDesc);
            return OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        // Initialize the band descriptor's attributes
        psBandDesc->hBand = hBand;
        psBandDesc->tBandId = tBandId;
        psBandDesc->tCapacity = tBandCapacity;
        psBandDesc->eType = BAND.eType(hBand);

        psBandDesc->tId = PRESET_BAND_eGetId(hBand);
        psBandDesc->tOrder = PRESET_BAND_eGetOrder(hBand);
        psBandDesc->bRemoved = PRESET_BAND_bIsRemoved(hBand);

        // Add the band to the list of bands.
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hBands,
            &hEntry,
            psBandDesc);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Return band entry
        return hEntry;

    } while (FALSE);

    // Destroy this band
    vDestroyBandDescriptor((void *) (size_t) psBandDesc);

    return OSAL_INVALID_LINKED_LIST_ENTRY;
}

/*******************************************************************************
 *
 *   bAddPresetsForBand
 *
 *   This function initializes all the band's presets and add them to the
 *   master preset list.
 *
 *******************************************************************************/
static BOOLEAN bAddPresetsForBand(
    PRESETS_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY hBandEntry
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;
    PRESET_STRUCT *psCurPreset;
    BOOLEAN bSuccess = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    size_t tPresetIndex;

    // Extract the band descriptor
    psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
        OSAL.pvLinkedListThis(hBandEntry);

    // Populate the presets for this band
    for (tPresetIndex = 0;
         tPresetIndex < psBandDesc->tCapacity;
         tPresetIndex++)
    {
        // Get the current preset pointer from the
        // band descriptor struct
        psCurPreset = &psBandDesc->pasPresets[tPresetIndex];

        // Initialize each preset with some attributes
        // from the band, start each preset with an
        // invalid channel
        psCurPreset->hBand = psBandDesc->hBand;
        psCurPreset->tBandId = psBandDesc->tBandId;
        psCurPreset->tPresetIndex = tPresetIndex;
        psCurPreset->tServiceId = SERVICE_INVALID_ID;
        psCurPreset->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Add it to the master list of presets
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hPresets,
            &psCurPreset->hEntry,
            psCurPreset);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error! Stop processing
            bSuccess = FALSE;

            break;
        }
    }

    // Verify creation succeeded
    if (bSuccess == FALSE)
    {
        // If this failed at any point,
        // clean up this band's presets
        vClearBandPresets(psObj, psBandDesc, TRUE, TRUE);
    }

    return bSuccess;
}

/*******************************************************************************
 *
 *   vClearBandPresets
 *
 *   This function clears the contents of all presets with belong to a given
 *   band.  If bRemovePresets is set to TRUE, those presets are also
 *   removed from the master preset list completely.
 *
 *   This function also ensures channel presets handles are updated as well.
 *
 *******************************************************************************/
static void vClearBandPresets(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc,
    BOOLEAN bClearNames,
    BOOLEAN bRemovePresets
        )
{
    PRESET_STRUCT *psPreset;
    size_t tCurrentPreset;

    // Clear this band's presets
    for (tCurrentPreset = 0;
            tCurrentPreset < psBandDesc->tCapacity;
            tCurrentPreset++)
    {
        // Get the current preset
        psPreset = &psBandDesc->pasPresets[tCurrentPreset];

        if (bClearNames == TRUE)
        {
            if (psPreset->hName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(psPreset->hName);
                psPreset->hName = STRING_INVALID_OBJECT;
            }
        }

        // Update this preset if necessary
        if (psPreset->tServiceId != SERVICE_INVALID_ID)
        {
            SERVICE_ID tPrevServiceId = psPreset->tServiceId;

            // Clear the service Id & handle
            psPreset->tServiceId = SERVICE_INVALID_ID;

            psPreset->hChannel = CHANNEL_INVALID_OBJECT;

            // Mark this preset as removed
            psPreset->eState = PRESET_STATE_REMOVED;

            // Update any affected channel object
            bUpdateChannelPresetHdls(
                psObj, psPreset, tPrevServiceId);
        }

        // Were we asked to remove this band's presets?
        if (bRemovePresets == TRUE)
        {
            OSAL.eLinkedListRemove(psPreset->hEntry);
            psPreset->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
    }

    return;
}

/*******************************************************************************
 *
 *   bRemoveBandFromService
 *
 *   This function prepares the presets service for deletion of a particular
 *   band by ensuring the presets service completely removes this band and
 *   its state from the service (by updating channel preset handles and the
 *   presets category).
 *
 *******************************************************************************/
static BOOLEAN bRemoveBandFromService(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc
        )
{
    BOOLEAN bSuccess;

    // Clear all of this band's presets
    // and remove them from the master preset list
    vClearBandPresets(psObj, psBandDesc, TRUE, TRUE);

    // Update the category with our changes
    bSuccess = bUpdateCategory(psObj);

    return bSuccess;
}

/*******************************************************************************
 *
 *   n16IterateCategoryForUpdate
 *
 *   This function is provided to CATEGORY.eIterate in order to manipulate
 *   the category so that it matches the current state of the presets service.
 *
 *   This function only navigates forward in the category, until all presets
 *   have been sychronized to this category, at which point iteration stops.
 *
 *******************************************************************************/
static N16 n16IterateCategoryForUpdate(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    PRESETS_CATEGORY_UPDATE_STRUCT *psUpdate =
        (PRESETS_CATEGORY_UPDATE_STRUCT *) pvIterateArg;
    PRESET_STRUCT *psPreset;
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;
    BOOLEAN bIncrement, bNextChannel;
    N16 n16ReturnValue = 0;

    // Verify inputs
    if ((psUpdate == NULL) || (psUpdate->psBandDesc == NULL))
    {
        // Error!
        return 0;
    }

    // Does our current index indicate we wrapped back
    // around during our iteration?
    if (tCurrentIndex <= psUpdate->tLastChannelIndex)
    {
        // Yeah, but verify we didn't remove a channel
        if (psUpdate->bChanRemoved == FALSE)
        {
            // We've iterated through the entire
            // category.  Stop now
            return 0;
        }
        else
        {
            // The index didn't change because
            // we just removed a channel

            // Unflag this event
            psUpdate->bChanRemoved = FALSE;
        }
    }

    // Track the last channel index
    psUpdate->tLastChannelIndex = tCurrentIndex;

    // Get the current channel's Service Id
    tServiceId = CHANNEL.tServiceId(hChannel);

    do
    {
        // Initialize this iteration's state
        bIncrement = bNextChannel = FALSE;

        // Get the current preset
        psPreset = &psUpdate->psBandDesc->
            pasPresets[psUpdate->tPresetIndex];

        // Did this preset get updated at all?
        if (psPreset->eState == PRESET_STATE_UNCHANGED)
        {
            // Move on to the next preset
            bIncrement = TRUE;

            if (tServiceId == psPreset->tServiceId)
            {
                // Move on to next channel in this category
                // if the Service Id's match
                bNextChannel = TRUE;
            }

        }
        // Is this preset newly populated with a valid
        // channel Id?
        else if (psPreset->eState == PRESET_STATE_ADDED)
        {
            // Insert this new preset in this category
            eReturnCode = CATEGORY_eInsertBeforeChannel(
                hCategory, psPreset->hChannel);
            // Category manipulation failed
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                psUpdate->bError = TRUE;
                return 0;
            }

            // Increment the preset index and
            // move on to the next channel
            bIncrement = bNextChannel = TRUE;
        }
        // Is this preset a simple swap of valid
        // channel Ids?
        else if (psPreset->eState == PRESET_STATE_UPDATED)
        {
            // Replace this channel with the new preset
            // channel
            eReturnCode = CATEGORY_eReplaceChannel(
                hCategory, psPreset->hChannel);
            // Category manipulation failed
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                psUpdate->bError = TRUE;
                return 0;
            }

            // Move on to the next channel
            // and increment the preset index
            bIncrement = bNextChannel = TRUE;
        }
        // Did this preset previously contain a valid
        // channel Id and now it is empty?
        else if (psPreset->eState == PRESET_STATE_REMOVED)
        {
            // Remove this channel from the category
            eReturnCode = CATEGORY.eRemoveChannel(hCategory);

            // Category manipulation failed
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                psUpdate->bError = TRUE;
                return 0;
            }

            // Flag that we just removed a channel
            // so we don't stop iterating early
            // (Removing a channel affects the
            // tCurrentIndex value we are provided)
            psUpdate->bChanRemoved = TRUE;

            // Move on to the next channel
            // and increment the preset index
            bIncrement = bNextChannel = TRUE;
        }

        // This preset has been handled
        psPreset->eState = PRESET_STATE_UNCHANGED;

        // Check flag to see if we are going
        // to increment to the next preset
        if (bIncrement == TRUE)
        {
            // Increment the index
            psUpdate->tPresetIndex++;

            // See if we're over-stepped our bounds
            if (psUpdate->tPresetIndex == psUpdate->psBandDesc->tCapacity)
            {
                // We're now at the first index
                // of the next band
                psUpdate->tPresetIndex = 0;

                // Move on to the next band
                psUpdate->hCurrentBand = OSAL.hLinkedListNext(
                    psUpdate->hCurrentBand,
                    (void **) (void*) &psUpdate->psBandDesc);

                // Did we encounter any errors?
                if ((psUpdate->hCurrentBand == OSAL_INVALID_LINKED_LIST_ENTRY)
                    || (psUpdate->psBandDesc == NULL)
                   )
                {
                    // We have, so we're done
                    psUpdate->bError = TRUE;
                    n16ReturnValue = 0;

                    break;
                }

                // Verify we have not wrapped back around
                // to the first band.
                if (psUpdate->hCurrentBand == psUpdate->hFirstBand)
                {
                    // We're all done!
                    psUpdate->bUpdateComplete = TRUE;
                    n16ReturnValue = 0;

                    break;
                }
            }
        }

        // Do we need to move on to the next
        // channel in this category?
        if (bNextChannel == TRUE)
        {
            n16ReturnValue = 1;
        }

    } while (bNextChannel == FALSE);

    return n16ReturnValue;
}

/*******************************************************************************
 *
 *   n16RemoveAllChannels
 *
 *   This function is provided to CATEGORY.eIterate in order to manipulate
 *   the category so that it matches the current state of the presets service.
 *
 *   This function only navigates forward in the category, until all presets
 *   have been sychronized to this category, at which point iteration stops.
 *
 *******************************************************************************/
static N16 n16RemoveAllChannels(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        )
{
    CATEGORY.eRemoveChannel(hCategory);

    return TRUE;
}

/*******************************************************************************
 *
 *   psGetPreset
 *
 *   This function finds and returns the PRESET_STRUCT * associated with a
 *   specific band and preset index
 *
 *******************************************************************************/
static PRESET_STRUCT *psGetPreset(
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex
        )
{
    PRESET_STRUCT *psPreset = NULL;
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;

    // Extract the band descriptor from the
    // provided band entry
    psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
        OSAL.pvLinkedListThis(hBandEntry);

    // Is the resulting pointer valid?
    if (psBandDesc == (PRESETS_BAND_DESCRIPTOR_STRUCT *) NULL)
    {
        // Bad!
        return NULL;
    }

    // Does this entry agree on the band?
    if (psBandDesc->hBand != hBand)
    {
        // Nope!
        return NULL;
    }

    // Check bounds!
    if (tPresetIndex < psBandDesc->tCapacity)
    {
        // Get the requested preset
        psPreset = &psBandDesc->pasPresets[tPresetIndex];
    }

    return psPreset;
}

/*******************************************************************************
 *
 *   eIterateCommon
 *
 *   This function performs all the common actions required when iterating
 *   a single band or the entire presets service.
 *
 *******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateCommon(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_OBJECT hBand,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;
    BOOLEAN bIterateAllBands = FALSE;
    OSAL_LINKED_LIST_ENTRY hBandEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Verify an iterator is present
    if (n16Iterator == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get the band entry handle and the band
    // descriptor pointer
    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        // We were not given a band handle, so
        // we are going to iterate all bands
        bIterateAllBands = TRUE;

        // Get the first band entry
        // and extract the descriptor
        hBandEntry = OSAL.hLinkedListFirst(
            psObj->hBands,
            (void **) (void *) &psBandDesc);
    }
    else
    {
        // We were given a specific band to iterate,
        // so get its entry handle and descriptor
        OSAL.eLinkedListLinearSearch(
            psObj->hBands,
            &hBandEntry,
            n16CompareBands,
            hBand);

        // Verify we found the band entry
        if (hBandEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Extract the band pointer
            psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                OSAL.pvLinkedListThis( hBandEntry);
        }
        else
        {
            // No bands were found, determine if we have
            // any bands at all
            size_t tNumberOfBands;

            // Determine how many bands we have
            tNumberOfBands = PRESETS.tNumBands(
                (PRESETS_OBJECT) psObj);

            if (tNumberOfBands == 0)
            {
                // Inform the caller now that we have no bands
                return SMSAPI_RETURN_CODE_NO_OBJECTS;
            }

            // Some other error occurred
            return SMSAPI_RETURN_CODE_ERROR;
        }
    }

    // We're going to be utilizing the decoder's ccache
    // in order to get access to CHANNEL objects, so we
    // must lock our decoder
    bLocked =
        SMSO_bLock((SMS_OBJECT) psObj->hDecoder,OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;
        CHANNEL_OBJECT hChannel;
        PRESET_STRUCT *psPreset = NULL;
        N16 n16ReturnValue;
        size_t tCurrentPreset = 0;

        // Get the decoder's ccache
        hCCache = DECODER_hCCache(psObj->hDecoder);

        // Start iterating presets now...
        do
        {
            if (   (hBandEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                || (psBandDesc == NULL)
                )
            {
                // If we don't have a valid band entry or a valid band
                // descriptor pointer then we can go no further
                // We've already had the band verified, if one was
                // provided so this is some major issue
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }

            // Get the current preset
            psPreset = &psBandDesc->pasPresets[tCurrentPreset];

            // Get the channel handle, but don't create if it
            // doesn't exist
            hChannel = CCACHE_hChannelFromIds(
                hCCache, psPreset->tServiceId, CHANNEL_INVALID_ID, FALSE);

            // Invoke the iterator callback
            n16ReturnValue = n16Iterator(
                (PRESETS_OBJECT) psObj,
                psBandDesc->hBand,
                psBandDesc->tBandId,
                hChannel,
                psPreset->hName,
                psPreset->tPresetIndex,
                pvIterateArg
                    );

            // Determine how to proceed..
            if (n16ReturnValue < 0)
            {
                // The iterator wants us to move backward through
                // the presets now...

                // Are we currently at the first preset in this band?
                if (tCurrentPreset == 0)
                {
                    // Yes, this is the first preset.  Find
                    // out if we're supposed to iterate all
                    // bands or just this one
                    if (bIterateAllBands == TRUE)
                    {
                        // We may iterate all the bands in the preset
                        // service. Make the previous band the currently
                        // iterated one
                        hBandEntry = OSAL.hLinkedListPrev(
                            hBandEntry,
                            (void **) (void *) &psBandDesc);
                    }

                    // Verify the band pointer is still good
                    if (psBandDesc != NULL)
                    {
                        // Move to the last preset in this band
                        tCurrentPreset = psBandDesc->tCapacity - 1;
                    }
                }
                else
                {
                    // Just decrement the preset index as we
                    // are staying in the same band
                    tCurrentPreset--;
                }
            }
            else if (n16ReturnValue > 0)
            {
                // The iterator wants us to move forward through
                // the presets now...

                // Are we currently at the last preset in this band?
                if (tCurrentPreset == (psBandDesc->tCapacity - 1))
                {
                    // Yes, this is the last preset.  Find
                    // out if we're supposed to iterate all
                    // bands or just this one
                    if (bIterateAllBands == TRUE)
                    {
                        // We may iterate all the bands in the preset
                        // service. Make the previous band the currently
                        // iterated one
                        hBandEntry = OSAL.hLinkedListNext(
                            hBandEntry,
                            (void **) (void *) &psBandDesc);
                    }

                    // Move to the first preset in this band
                    tCurrentPreset = 0;
                }
                else
                {
                    // Just increment the preset index as we
                    // are staying in the same band
                    tCurrentPreset++;
                }
            }

            if (n16ReturnValue == 0)
            {
                // We're done -- no problems
                eReturnCode =
                    SMSAPI_RETURN_CODE_SUCCESS;
            }

        } while (n16ReturnValue != 0);

        SMSO_vUnlock((SMS_OBJECT) psObj->hDecoder);
    }

    return eReturnCode;
}

/*******************************************************************************
 *
 *   bUpdateChannelPresetHdls
 *
 *   This function is called whenever a preset is updated.  Any affected
 *   channels are provided an updated "base" preset handle as a result.
 *
 *******************************************************************************/
static BOOLEAN bUpdateChannelPresetHdls(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_STRUCT *psPreset,
    SERVICE_ID tPrevServiceId
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bLocked;
    BOOLEAN bOk = FALSE;

    // We have just updated our presets - resort
    // them to ensure they're ordered correctly
    eReturnCode = OSAL.eLinkedListSort(
        psObj->hPresets,
        n16SortPresetsByServiceId);
    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Verify and lock the decoder
    bLocked =
        SMSO_bLock((SMS_OBJECT) psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;
        BOOLEAN bValid;

        // Get & verify the decoder's ccache
        hCCache = DECODER_hCCache(psObj->hDecoder);
        bValid = SMSO_bValid((SMS_OBJECT) hCCache);
        if (bValid == TRUE)
        {
            CHANNEL_PRESET_HDL hChannelPresetHdl;
            PRESETS_CHANNEL_OCCURRENCE_STRUCT sOccurrence;

            // Flag this as a success for now
            bOk = TRUE;

            // If the preset was newly added or updated,
            // then we must inform the new preset channel
            // of its new handle
            if (   (psPreset->eState == PRESET_STATE_ADDED)
                || (psPreset->eState == PRESET_STATE_UPDATED)
                )
            {
                hChannelPresetHdl = CHANNEL_PRESET_INVALID_HDL;

                // Initialize the occurrence struct
                sOccurrence.psFirstOccurrence = (PRESET_STRUCT *) NULL;
                sOccurrence.tNumOccurrences = 0;

                // We're looking for the provided preset's
                // service id
                sOccurrence.tServiceId = psPreset->tServiceId;

                // Get the occurrences for this service id:
                //   First preset occurrence
                //   Number of times this channel appears in list
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->hPresets,
                    bDiscoverChannelOccurrence,
                    (void *) (size_t) &sOccurrence
                        );

                if (eReturnCode == OSAL_SUCCESS)
                {
                    if (sOccurrence.tNumOccurrences > 0)
                    {
                        // Set the preset handle for the new channel
                        if (sOccurrence.psFirstOccurrence != NULL)
                        {
                            hChannelPresetHdl = (CHANNEL_PRESET_HDL)
                                sOccurrence.psFirstOccurrence->hEntry;
                        }
                    }
                    else
                    {
                        // This means we've added a preset but
                        // can't find it.  Weird stuff!
                        bOk = FALSE;
                    }
                }
                else
                {
                    // This shouldn't happen
                    bOk = FALSE;
                }

                if (bOk == TRUE)
                {
                    CHANNEL_OBJECT hNewPresetChannel;

                    // Get the channel handle for the
                    // new preset channel
                    hNewPresetChannel = CCACHE_hChannelFromIds(
                        hCCache, psPreset->tServiceId, CHANNEL_INVALID_ID, TRUE);

                    if (hNewPresetChannel != CHANNEL_INVALID_OBJECT)
                    {
                        // Only register use of this channel
                        // when this is the only occurrence
                        // of this channel in the preset list.
                        if (sOccurrence.tNumOccurrences == 1)
                        {
                            // We are "using" this channel now.  Register
                            // for updates regarding the service id
                            // and the channel id
                            bOk = CHANNEL_bRegisterNotification(
                                hNewPresetChannel,
                                CHANNEL_OBJECT_EVENT_SERVICE_ID |
                                CHANNEL_OBJECT_EVENT_CHANNEL_ID,
                                vChannelEventCallback,
                                (void *)psObj,
                                TRUE);
                        }

                        // Set the new preset handle
                        CHANNEL_vSetBasePresetHdl(
                            hNewPresetChannel,
                            hChannelPresetHdl);

                        // Save the channel handle
                        psPreset->hChannel = hNewPresetChannel;
                    }
                    else
                    {
                        // Error!
                        bOk = FALSE;
                    }
                }
            }

            // If the preset was removed or updated (and different),
            // then we must inform the old preset channel
            // of the change
            if ( (psPreset->eState == PRESET_STATE_REMOVED) ||
                 ( (psPreset->eState == PRESET_STATE_UPDATED) &&
                    (tPrevServiceId != psPreset->tServiceId) )
                )
            {
                hChannelPresetHdl = CHANNEL_PRESET_INVALID_HDL;

                // Initialize the occurrence struct
                sOccurrence.psFirstOccurrence = (PRESET_STRUCT *) NULL;
                sOccurrence.tNumOccurrences = 0;

                // We're looking for the previously stored
                // service id
                sOccurrence.tServiceId = tPrevServiceId;

                // Get the occurrences for this service id:
                //   First preset occurrence
                //   Number of times this channel appears in list
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->hPresets,
                    &bDiscoverChannelOccurrence,
                    (void *) (size_t) &sOccurrence);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    // Set the preset handle for the new channel
                    // this may be valid or invalid, and that's
                    // okay since it may no longer be in the
                    // list at all
                    if (sOccurrence.psFirstOccurrence != NULL)
                    {
                        hChannelPresetHdl = (CHANNEL_PRESET_HDL)
                            sOccurrence.psFirstOccurrence->hEntry;
                    }
                }
                else
                {
                    // This shouldn't happen
                    bOk = FALSE;
                }

                if (bOk == TRUE)
                {
                    CHANNEL_OBJECT hOldPresetChannel;

                    // Get the channel handle for the previous
                    // preset channel
                    hOldPresetChannel = CCACHE_hChannelFromIds(
                        hCCache, tPrevServiceId, CHANNEL_INVALID_ID, FALSE);

                    if (hOldPresetChannel != CHANNEL_INVALID_OBJECT)
                    {
                        if (sOccurrence.tNumOccurrences == 0)
                        {
                            // We are done using this channel
                            CHANNEL_vUnregisterNotification(
                                hOldPresetChannel,
                                vChannelEventCallback,
                                (void *) psObj);
                        }

                        // Clear this channel's preset handle
                        CHANNEL_vSetBasePresetHdl(
                            hOldPresetChannel,
                            hChannelPresetHdl);

                        if (psPreset->eState == PRESET_STATE_REMOVED)
                        {
                            // Remove the channel handle
                            psPreset->hChannel = CHANNEL_INVALID_OBJECT;
                        }
                    }
                    else
                    {
                        // Error!
                        bOk = FALSE;
                    }
                }
            }
        }

        SMSO_vUnlock((SMS_OBJECT) psObj->hDecoder);
    }

    return bOk;
}

/*******************************************************************************
 *
 *   vChannelEventCallback
 *
 *   This function is called whenever a channel used as a preset has a
 *   channel id or service id update.
 *
 *******************************************************************************/
static void vChannelEventCallback(
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bLocked;
    PRESETS_OBJECT hPresets =
        (PRESETS_OBJECT)pvEventCallbackArg;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;
        PRESETS_UPDATE_FROM_CHANNEL_EVENT_ITERATOR_STRUCT sIterator;

        sIterator.hPresets = hPresets;
        sIterator.hChannel = hChannel;

        // Iterate through the entire set of presets
        // in order to update all of them based on this event
        OSAL.eLinkedListIterate(
            psObj->hPresets,
            bUpdatePresetsFromChannelEvent,
            (void *)&sIterator );

        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    return;
}

/*******************************************************************************
 *
 *   vCategoryEventCallback
 *
 *   This function is called whenever a channel used as a preset has a
 *   channel id or service id update.
 *
 *******************************************************************************/
static void vCategoryEventCallback (
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bLocked;
    PRESETS_OBJECT hPresets =
        (PRESETS_OBJECT)pvEventCallbackArg;

    // Verify and lock the Presets service
    bLocked =
        SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESETS_OBJECT_STRUCT *psObj =
            (PRESETS_OBJECT_STRUCT *)hPresets;

        // Are we currently updating this category?
        if (psObj->bUpdatingCategory == TRUE)
        {
            // Don't do nuttin'
        }
        // Has this category been destroyed?
        else if ((tEventMask & CATEGORY_OBJECT_EVENT_REMOVED)
                        == CATEGORY_OBJECT_EVENT_REMOVED)
        {
            // Clear our category id attribute
            psObj->tCategoryId = CATEGORY_INVALID_ID;

            // Indicate that we need to rebuild the category
            psObj->bRebuildCategory = TRUE;
        }
        else if ((tEventMask & CATEGORY_OBJECT_EVENT_CONTENTS)
                        == CATEGORY_OBJECT_EVENT_CONTENTS)
        {
            // Indicate that we need to rebuild the category
            psObj->bRebuildCategory = TRUE;
        }

        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }
}

/*****************************************************************************
 *
 *   vEventCallbackShim
 *
 *****************************************************************************/
static void vEventCallbackShim(
    PRESETS_OBJECT hPresets,
    SMSAPI_EVENT_MASK tEventMask,
    void * pvEventCallbackArg
        )
{
    PRESETS_CALLBACK_SHIM_STRUCT *psShim =
        (PRESETS_CALLBACK_SHIM_STRUCT *)pvEventCallbackArg;

    if (psShim->vEventCallback != NULL)
    {
        // Call to the original callback
        psShim->vEventCallback(
            hPresets,
            psShim->hBand,
            tEventMask,
            psShim->pvEventCallbackArg
                );
    }

    return;
}

/*****************************************************************************
 *
 *       vDestroyBandDescriptor
 *
 *       An OSAL LL Remove callback which removes a band from
 *       service and destroys any associated memory
 *
 *****************************************************************************/
static void vDestroyBandDescriptor(
    void *pvData
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc =
        (PRESETS_BAND_DESCRIPTOR_STRUCT *) pvData;

    if (psBandDesc != NULL)
    {
        PRESETS_OBJECT hPresets;

        // Get the presets service handle
        hPresets = (PRESETS_OBJECT)
            SMSO_hParent((SMS_OBJECT) psBandDesc);

        if (PRESETS_INVALID_OBJECT != hPresets)
        {
            // Remove this band from the service
            bRemoveBandFromService(
                (PRESETS_OBJECT_STRUCT *) hPresets,
                psBandDesc);
        }

        // Delete the presets memory
        if (psBandDesc->pasPresets != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT) psBandDesc->pasPresets);
            psBandDesc->pasPresets = NULL;
        }

        // Tell the band to go away
        PRESET_BAND_vDelete(psBandDesc->hBand);

        // Clear the band handle
        psBandDesc->hBand = PRESET_BAND_INVALID_OBJECT;

        // Delete the band descriptor
        SMSO_vDestroy((SMS_OBJECT) psBandDesc);
    }

    return;
}

/*****************************************************************************
 *
 *       bDiscoverChannelOccurrence
 *
 *       An OSAL LL iteration callback which determines both the first
 *       occurrence of a channel in the presets list as well as the
 *       number of times the channel appears in the preset list.
 *
 *****************************************************************************/
static BOOLEAN bDiscoverChannelOccurrence(
    void *pvData,
    void *pvArg
        )
{
    PRESET_STRUCT *psPreset = (PRESET_STRUCT *) pvData;
    PRESETS_CHANNEL_OCCURRENCE_STRUCT *psOccurrence =
        (PRESETS_CHANNEL_OCCURRENCE_STRUCT *) pvArg;

    if ((pvData == NULL) || (pvArg == NULL))
    {
        // Stop iterating
        return FALSE;
    }

    // Does this preset match our service Id?
    if (psOccurrence->tServiceId == psPreset->tServiceId)
    {
        // Yes -- if this is the first occurrence, save
        // the preset handle
        if (psOccurrence->tNumOccurrences == 0)
        {
            psOccurrence->psFirstOccurrence = psPreset;
        }

        psOccurrence->tNumOccurrences++;
    }

    // Go through the entire list
    return TRUE;
}

/*****************************************************************************
 *
 *       bUpdatePresetsFromChannelEvent
 *
 *       An OSAL LL iteration callback which ensures that all presets
 *       utilizing a specific channel are updated when that channel reports
 *       update events.
 *
 *****************************************************************************/
static BOOLEAN bUpdatePresetsFromChannelEvent (
    void *pvData,
    void *pvArg
        )
{
    PRESET_STRUCT *psPreset = (PRESET_STRUCT *)pvData;
    PRESETS_UPDATE_FROM_CHANNEL_EVENT_ITERATOR_STRUCT *psIterator =
        (PRESETS_UPDATE_FROM_CHANNEL_EVENT_ITERATOR_STRUCT *)pvArg;

    // Verify inputs
    if (   (psPreset == NULL)
        || (psIterator == NULL)
        || (psIterator->hChannel == CHANNEL_INVALID_OBJECT)
       )
    {
        // Error -- stop!
        return FALSE;
    }

    // Is this preset using this channel?
    if (psPreset->hChannel == psIterator->hChannel)
    {
        PRESETS_OBJECT_STRUCT *psPresetsObj =
            (PRESETS_OBJECT_STRUCT *)psIterator->hPresets;

        // Update this preset's service id
        psPreset->tServiceId = CHANNEL.tServiceId(psIterator->hChannel);

        if (psPresetsObj->bBlockNotifications != TRUE)
        {
            // Notifying subscribed client
            PRESETS_bUpdate(
                psIterator->hPresets,
                psPreset->hBand,
                PRESETS_OBJECT_EVENT_BAND_MODIFIED);
        }
    }

    // Go through the entire list
    return TRUE;
}

/*****************************************************************************
 *
 *       eRestoreFromConfigFile
 *
 *       This function interacts with the config manager and tag objects to
 *       see if there is stored information on the preset service being
 *       started. If there is, the stored information will be used to
 *       recreate bands, restore preset channels etc.
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM  eRestoreFromConfigFile(
    PRESETS_OBJECT_STRUCT *psObj,
    BOOLEAN *pbAlreadyInConfig
        )
{
    TAG_OBJECT hParentTag, hTag;
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    if (pbAlreadyInConfig == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    do
    {
        BOOLEAN bRestored, bUpdate = FALSE, bComplete;

        // Verify and lock the decoder
        bLocked = SMSO_bLock((SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // get the Decoder's tag. this is the parent tag of the preset tag
        hParentTag = DECODER_hGetTag(psObj->hDecoder);

        // get this decoder's preset service tag
        // (don't create if it doesn't exist)
        eResult = TAG_eGet(PRESETS_OBJECT_NAME, hParentTag, &hTag, NULL, FALSE);
        if ( (eResult != SMSAPI_RETURN_CODE_SUCCESS) &&
             (eResult != SMSAPI_RETURN_CODE_CFG_NO_PARENT) )
        {
            break;
        }

        // did we find a preset service tag?
        if (hTag == TAG_INVALID_OBJECT)
        {
            // no presets tag was found in the file
            *pbAlreadyInConfig = FALSE;

            if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // add the tag
                eResult = TAG_eAdd(PRESETS_OBJECT_NAME, hParentTag, &hTag, NULL);
            }
            else
            {
                // no parent, must not be a file system
                eResult = SMSAPI_RETURN_CODE_SUCCESS;
                break;
            }
        }
        else
        {
            // we found a presets tag in the file
            *pbAlreadyInConfig = TRUE;
        }

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // we have a preset service tag
        // (either one that was existing or one that we created)
        psObj->hTag = hTag;

        if (*pbAlreadyInConfig == FALSE) 
        {
            // Nothing to load from config
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            break;
        }

        bComplete = RADIO_bGetFFTableStatus(psObj->hDecoder);
        if (bComplete == TRUE)
        {
            // Remove selected Featured Bands from tag before restoration
            eResult = TAG_eIterateChildren(
                psObj->hTag, 
                PRESET_BAND_bRemoveBandTagIterator, 
                &bUpdate);

            if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // Failed to remove bands
                break;
            }
        }

        // now restore the bands
        bRestored = PRESET_BAND_bRestorePresetBands((PRESETS_OBJECT)psObj);
        if (bRestored == FALSE)
        {
            // the attempt to restore bands failed
            eResult = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // the attempt to restore succeeded, but there may
        // not have been any bands saved, or the saved bands might
        // not have the current band information saved
        // (like if edited by hand)
        if (psObj->hCurrentBand == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            PRESETS_BAND_DESCRIPTOR_STRUCT *psBand;

            // nothing from the config file indicated which band
            // was the current band, so let's just make the first
            // one the current band
            psObj->hCurrentBand =
                OSAL.hLinkedListFirst(psObj->hBands, (void**)&psBand);

            // since we selected a band to be the current band,
            // update the config file to reflect this
            if (psObj->hCurrentBand != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                eResult = PRESET_BAND_eSaveCurrentBand(psBand->hBand);
                if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        PRESETS_OBJECT_NAME": Failed to save current preset band (%s)",
                        SMSAPI_DEBUG_pacReturnCodeText(eResult)
                            );
                    break;
                }
                else
                {
                    BOOLEAN bSuccess; 

                    bSuccess = PRESETS_bCurrentBandSync((PRESETS_OBJECT)psObj, psBand->hBand);
                    if (bSuccess == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            PRESETS_OBJECT_NAME": Failed to sync current preset band");
                    }
                }
            }
        }

        // Load target info
        vLoadReceiverCapabilities(psObj);

        if (bUpdate == TRUE)
        {
            // Should not affect the whole restoration procedure
            vInitiateUpdate(psObj);
        }

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
    }

    return eResult;
}

/*****************************************************************************
 *
 *       bClearCurrentBandTags
 *
 *       An OSAL LL iteration callback which ensures that all preset
 *       bands have their current band tag set to FALSE
 *****************************************************************************/
static BOOLEAN bClearCurrentBandTags(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    // Clear (set to false) the current band tag of the band represented by
    // this tag. This function will verify that tag we're passing in is a band.
    printf("Cleaning out current band tags. Presets Hdl: 0x%p\n", pvArg);
    PRESET_BAND_vClearCurrentBand(hTag);

    // keep iterating
    return TRUE;
}

/*****************************************************************************
 *
 *       eUpdateCurrentBandTags
 *
 *       A function that will update the current band information for all
 *       bands
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM eUpdateCurrentBandTags(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_OBJECT hBand
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // iterate the bands and set the current tags to false
    eReturnCode = TAG_eIterateChildren(
       psObj->hTag,
       bClearCurrentBandTags,
       (PRESETS_OBJECT)psObj
           );

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the current band to true for this band
        eReturnCode = PRESET_BAND_eSaveCurrentBand(hBand);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vLoadReceiverCapabilities
*
*****************************************************************************/
static void vLoadReceiverCapabilities (
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    size_t tSize = 0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    TAG_OBJECT hCapacityTag = TAG_INVALID_OBJECT,
               hPurposeTag = TAG_INVALID_OBJECT,
               hLimitTag = TAG_INVALID_OBJECT;

    // Get capacity tag
    eReturnCode = TAG_eGet(
        PRESETS_FEATURED_BAND_CAPACITY,
        psObj->hTag,
        &hCapacityTag,
        NULL, FALSE);

    // Get capacity value
    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hCapacityTag != TAG_INVALID_OBJECT))
    {
        N32 n32TagValue, *pn32TagValueRef = &n32TagValue;

        eReturnCode = TAG_eGetTagValue(
            hCapacityTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32TagValueRef,
            &tSize);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            psObj->sCapabilities.tCapacity = (FAVORITES_BAND_CAPACITY)n32TagValue;
        }
    }

    // Get purpose tag
    eReturnCode = TAG_eGet(
        PRESETS_FEATURED_BAND_PURPOSE,
        psObj->hTag,
        &hPurposeTag,
        NULL, FALSE);

    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hPurposeTag != TAG_INVALID_OBJECT))
    {
        N32 n32TagValue, *pn32TagValueRef = &n32TagValue;

        eReturnCode = TAG_eGetTagValue(
            hPurposeTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32TagValueRef,
            &tSize);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            psObj->sCapabilities.tPurpose = (FAVORITES_BAND_PURPOSE_MASK)n32TagValue;
        }
    }

    // Get bank limit tag
    eReturnCode = TAG_eGet(
        PRESETS_FEATURED_BAND_LIMIT,
        psObj->hTag,
        &hLimitTag,
        NULL, FALSE);

    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
        (hLimitTag != TAG_INVALID_OBJECT))
    {
        N32 n32TagValue, *pn32TagValueRef = &n32TagValue;

        eReturnCode = TAG_eGetTagValue(
            hLimitTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32TagValueRef,
            &tSize);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            psObj->sCapabilities.tBandsNumber = (FAVORITES_BAND_NUMBER)n32TagValue;
        }
    }

    return;
}

/*****************************************************************************
*
*   vSaveReceiverCapabilities
*
*****************************************************************************/
static void vSaveReceiverCapabilities (
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    TAG_OBJECT hCapacityTag = TAG_INVALID_OBJECT,
               hPurposeTag = TAG_INVALID_OBJECT,
               hLimitTag = TAG_INVALID_OBJECT;

    // Get capacity tag
    eReturnCode = TAG_eGet(
        PRESETS_FEATURED_BAND_CAPACITY,
        psObj->hTag,
        &hCapacityTag,
        NULL, TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32TagValue = (N32)psObj->sCapabilities.tCapacity;

        eReturnCode = TAG_eSetTagValue(
            hCapacityTag,
            TAG_TYPE_INTEGER,
            (void *)&n32TagValue,
            sizeof(N32),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            printf(PRESETS_OBJECT_NAME": Failed to save target capacity value %s (%d)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);
        }
    }
    else
    {
        printf(PRESETS_OBJECT_NAME": Failed to get target capacity tag %s (%d)",
            SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);
    }

    // Get purpose tag
    eReturnCode = TAG_eGet(
        PRESETS_FEATURED_BAND_PURPOSE,
        psObj->hTag,
        &hPurposeTag,
        NULL, TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32TagValue = (N32)psObj->sCapabilities.tPurpose;

        eReturnCode = TAG_eSetTagValue(
            hPurposeTag,
            TAG_TYPE_INTEGER,
            (void *)&n32TagValue,
            sizeof(N32),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            printf(PRESETS_OBJECT_NAME": Failed to save target purpose value %s (%d)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);
        }
    }
    else
    {
         printf(PRESETS_OBJECT_NAME": Failed to get target purpose tag %s (%d)",
            SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);
    }

    // Get bank limit tag
    eReturnCode = TAG_eGet(
        PRESETS_FEATURED_BAND_LIMIT,
        psObj->hTag,
        &hLimitTag,
        NULL, TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32TagValue = (N32)psObj->sCapabilities.tBandsNumber;

        eReturnCode = TAG_eSetTagValue(
            hLimitTag,
            TAG_TYPE_INTEGER,
            (void *)&n32TagValue,
            sizeof(N32),
            TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            printf(PRESETS_OBJECT_NAME": Failed to save target bank limit value %s (%d)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);
        }
    }
    else
    {
         printf(PRESETS_OBJECT_NAME": Failed to get target bank limit tag %s (%d)",
            SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);
    }

    return;
}

/*****************************************************************************
*
*   vClearBandSequenceNumbers
*
*****************************************************************************/
static void vClearBandSequenceNumbers (
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    OSAL.eLinkedListIterate(
        psObj->hBands,
        bClearBandSeqNumberIterator,
        (void *)NULL);

    return;
}

/*****************************************************************************
*
*   vRemoveExtraBands
*
*****************************************************************************/
static void vRemoveExtraBands (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIterator
        )
{
    OSAL_LINKED_LIST_ENTRY hEntry, hFirstEntry;
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;

    // Get the first band
    hFirstEntry = hEntry =
        OSAL.hLinkedListFirst(psObj->hBands, (void **)&psBandDesc);

    if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return;
    }

    do
    {
        if (psBandDesc->eType == PRESET_BAND_TYPE_FEATURED)
        {
            if (psIterator->tBandsNumber <= psIterator->un32BandsCount)
            {
                if (psBandDesc->tBandId != psIterator->tCurrentBandId)
                {
                    printf(PRESETS_OBJECT_NAME": Attempt to destroy Band: %s\n",
                        STRING.pacCStr(BAND.hName(psBandDesc->hBand)));

                    // Save the next entry because the current one will be deleted
                    hEntry = OSAL.hLinkedListNext(hEntry, (void **)NULL);

                    // Increment count
                    psIterator->un32BandsCount++;

                    // Destroy band
                    PRESET_BAND_vDestroy(psBandDesc->hBand);

                    // Now populate band descriptor with new the next band entry dataS
                    psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)OSAL.pvLinkedListThis(hEntry);

                    // Skip next
                    continue;
                }
                else
                {
                    // If the current band entry is out of scope
                    // need to save it since current band cannot be removed
                    // further we will use this entry to remove the previous
                    // available band to fit into the requested capacity
                    psIterator->hEntry = hEntry;

                    printf(PRESETS_OBJECT_NAME": Current Band Detected: %s\n",
                        STRING.pacCStr(BAND.hName(psBandDesc->hBand)));
                }
            }

            psIterator->un32BandsCount++;
        }

        // Get the next entry
        hEntry = OSAL.hLinkedListNext(hEntry, (void **)&psBandDesc);
    } while (hEntry != hFirstEntry);

    return;
}

/*****************************************************************************
*
*   vUpdateBandsCapacity
*
*****************************************************************************/
static void vUpdateBandsCapacity (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIterator
        )
{
    OSAL.eLinkedListIterate(
        psObj->hBands,
        bAddRemovePresetsIterator,
        (void *)psIterator);

    return;
}

/*****************************************************************************
*
*   bAddRemovePresetsIterator
*
*****************************************************************************/
static BOOLEAN bAddRemovePresetsIterator (
    void * pvData,
    void * pvArg
        )
{
    SERVICE_ID *ptServiceId = NULL;
    PRESET_STRUCT *pasPresets = NULL;

    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc =
        (PRESETS_BAND_DESCRIPTOR_STRUCT *)pvData;

    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIterator =
        (PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *)pvArg;

    do
    {
        size_t tIndex, tCapacity;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        PRESETS_OBJECT_STRUCT *psObj;
        BOOLEAN bSuccess;
        OSAL_RETURN_CODE_ENUM eOSALCode;
        SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        OSAL_LINKED_LIST_ENTRY hBandEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        if (pvData == NULL || pvArg == NULL)
        {
            break;
        }

        if (psBandDesc->eType != PRESET_BAND_TYPE_FEATURED)
        {
            break;
        }

        // De-reference object
        psObj = (PRESETS_OBJECT_STRUCT *)psIterator->hPresets;

        // Name the band's preset memory
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                PRESETS_OBJECT_NAME"%u:Band%uPresets",
                psObj->tCategoryId,
                psBandDesc->tBandId
                    );

        // Allocate space for this band's presets
        pasPresets = (PRESET_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(PRESET_STRUCT) * (psIterator->tCapacity),
                (SMS_OBJECT)psIterator->hPresets,
                FALSE
                    );

        if (pasPresets == NULL)
        {
            break;
        }

        // Calculate capacity to copy
        tCapacity = ((psBandDesc->tCapacity < psIterator->tCapacity) ? 
            psBandDesc->tCapacity : psIterator->tCapacity);

        // Allocate space for service IDs storage
        ptServiceId = (SERVICE_ID *)SMSO_hCreate(
            PRESETS_OBJECT_NAME":RECFG_SID_ARR",
            sizeof(SERVICE_ID) * tCapacity,
            SMS_INVALID_OBJECT, // Parent is not required
            FALSE // Locking is not required for local temporary array
                );

        if (ptServiceId == NULL)
        {
            break;
        }

        // Copy Service IDs and names of presets
        for (tIndex = 0; tIndex < tCapacity; tIndex++)
        {
            // Service IDs shall be copied into the temporary array
            // to be properly set by PRESETS_eSetPreset later
            ptServiceId[tIndex] = psBandDesc->pasPresets[tIndex].tServiceId;

            // Put Invalid SID so PRESETS_eSetPreset will correctly
            // report updated status
            pasPresets[tIndex].tServiceId = SERVICE_INVALID_ID;

            // Name can be assigned directly
            pasPresets[tIndex].hName = psBandDesc->pasPresets[tIndex].hName;

            // and cleared up from source structure so it will be correctly
            // processed by vClearBandPresets() below
            psBandDesc->pasPresets[tIndex].hName = STRING_INVALID_OBJECT;
        }

        // Clear all of this band's presets and remove them from the master 
        // preset list.
        vClearBandPresets(psObj, psBandDesc, TRUE, TRUE);

        // Update the Presets category
        bSuccess = bUpdateCategory(psObj);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME":FF: Failed to update category");
            break;
        }

        // Destroy old presets array
        SMSO_vDestroy((SMS_OBJECT)psBandDesc->pasPresets);

        // Assign new presets array
        psBandDesc->pasPresets = pasPresets;

        // New presets array has been set as current.
        // So now, this is not our problem ...
        pasPresets = NULL;

        // Update band descriptor capacity
        psBandDesc->tCapacity = psIterator->tCapacity;

        // Update band capacity
        bSuccess = PRESET_BAND_bUpdateCapacity(
            psBandDesc->hBand,
            psIterator->tCapacity
                );

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME":FF: Failed to update Band capacity");
            break;
        }

        // Search for band
        eOSALCode = OSAL.eLinkedListLinearSearch(
            psObj->hBands,
            &hBandEntry,
            n16CompareBands,
            (void *)psBandDesc->hBand
                );

        if (eOSALCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME
                ":FF: Failed to search for a Band: %s (#%d)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), 
                eReturnCode);
            break;
        }

        // Initialize presets and add them to the master presets list
        bSuccess = bAddPresetsForBand(psObj, hBandEntry);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME":FF: Failed to add presets for a Band");
            break;
        }

        // Assign values for presets
        eReturnCode = PRESET_BAND_eSetPresets(
            psBandDesc->hBand, ptServiceId, tCapacity);

        // Presets were populated successfully?
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Shall be synchronized from here since some 
            // channels may go away
            bSuccess = PRESETS_bCurrentBandSync(
                psIterator->hPresets, psBandDesc->hBand);
            if (bSuccess == FALSE)
            {
                printf(PRESETS_OBJECT_NAME
                    ":SF: Failed to synchronize the current Band: %s (#%d)\n",
                    STRING.pacCStr(BAND.hName(psBandDesc->hBand)), 
                    psBandDesc->tBandId);
                break;
            }
        }
        else
        {
            printf(PRESETS_OBJECT_NAME
                ":FF: Failed to set the preset for Band: %s (#%d), "
                "Error: %s (#%d)\n",
                STRING.pacCStr(BAND.hName(psBandDesc->hBand)), 
                psBandDesc->tId, 
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), 
                eReturnCode);
            break;
        }

    } while (FALSE);

    if (pasPresets != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pasPresets);
    }

    if (ptServiceId != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)ptServiceId);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bClearBandSeqNumberIterator
*
*****************************************************************************/
static BOOLEAN bClearBandSeqNumberIterator (
    void * pvData,
    void * pvArg
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc =
        (PRESETS_BAND_DESCRIPTOR_STRUCT *)pvData;

    if (pvData == NULL)
    {
        return FALSE;
    }

    if (psBandDesc->eType == PRESET_BAND_TYPE_FEATURED)
    {
        PRESET_BAND_bUpdateSequence(
            psBandDesc->hBand, FAVORITES_BAND_SEQUENCE_INVALID);
    }

    return TRUE;
}

/*****************************************************************************
*
* bSetAllFFBanksRemovedIterator
*
*****************************************************************************/
static BOOLEAN bSetAllFFBanksRemovedIterator (
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        )
{
    BOOLEAN bRemoved = (BOOLEAN)(size_t)pvArg;

    // Check inputs
    if(psEntry == NULL)
    {
        // Stop iterating
        return FALSE;
    }

    if (psEntry->eType == PRESET_BAND_TYPE_FEATURED)
    {
        psEntry->bRemoved = bRemoved;
        PRESET_BAND_eSetRemoved(psEntry->hBand, bRemoved);
        printf( "Setting Up in the Removed State Bank ID: %u\n", psEntry->tId);
    }

    return TRUE;
}

/*****************************************************************************
*
* bRemoveAllSelectedFFBanksIterator
*
*****************************************************************************/
static BOOLEAN bRemoveAllSelectedFFBanksIterator(
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        )
{
    PRESETS_FAVORITES_REMOVE_ITERATOR_STRUCT *psIterator =
            (PRESETS_FAVORITES_REMOVE_ITERATOR_STRUCT *)pvArg;

    // Check inputs
    if(psEntry == NULL)
    {
        // Stop iterating
        return FALSE;
    }

    if ((psEntry->eType == PRESET_BAND_TYPE_FEATURED) &&
        (psEntry->bRemoved == TRUE) &&
        (psEntry->tBandId != psIterator->tBandID))
    {
        printf( "We Destroy Now Bank ID: %u\n", psEntry->tId);
        PRESET_BAND_vDestroy(psEntry->hBand);
    }

    return TRUE;
}

/*****************************************************************************
*
* bRemoveFirstSelectedFFBanksIterator
*
*****************************************************************************/
static BOOLEAN bRemoveFirstSelectedFFBankIterator(
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        )
{
    PRESETS_FAVORITES_REMOVE_ITERATOR_STRUCT *psIterator =
            (PRESETS_FAVORITES_REMOVE_ITERATOR_STRUCT *)pvArg;

    // Check inputs
    if(psEntry == NULL)
    {
        // Stop iterating
        return FALSE;
    }

    if ((psEntry->eType == PRESET_BAND_TYPE_FEATURED) &&
        (psEntry->bRemoved == TRUE) &&
        (psEntry->tBandId != psIterator->tBandID))
    {
        printf( "We Destroy Now Bank ID: %u\n", psEntry->tId);
        PRESET_BAND_vDestroy(psEntry->hBand);

        // Stop iterating
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveAllFavorites
*
*****************************************************************************/
static BOOLEAN bRemoveAllFavorites (
    void * pvData,
    void * pvArg
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc =
        (PRESETS_BAND_DESCRIPTOR_STRUCT *)pvData;

    // Check inputs
    if (pvData == NULL)
    {
        // Stop iterating
        return FALSE;
    }

    if (psBandDesc->eType == PRESET_BAND_TYPE_FEATURED)
    {
        printf( "We Destroy Now Bank ID: %u\n", psBandDesc->tId);
        PRESET_BAND_vDestroy(psBandDesc->hBand);
    }

    return TRUE;
}

/*****************************************************************************
*
* bSetSelectedFFBankRemovedIterator
*
*****************************************************************************/
static BOOLEAN bSetSelectedFFBankRemovedIterator (
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        )
{
    PRESET_BAND_MARKER_STRUCT *psMarkerStruct;

    // Check inputs
    if(psEntry == NULL)
    {
        printf( "The band entry is null\n");
        // Stop iterating
        return FALSE;
    }

    psMarkerStruct = (PRESET_BAND_MARKER_STRUCT *)pvArg;

    if ((psEntry->eType == PRESET_BAND_TYPE_FEATURED) &&
            (psEntry->tId == psMarkerStruct->tBankId))
    {
        psEntry->bRemoved = psMarkerStruct->bRemoved;
        PRESET_BAND_eSetRemoved(psEntry->hBand, psMarkerStruct->bRemoved);

        printf( "\t  Set Removed state to %s for bank #%u\n",
            (psMarkerStruct->bRemoved) ? "TRUE" : "FALSE",
            psEntry->tId);
    }

    return TRUE;
}

/*****************************************************************************
*
* bSearchForFavoritesBandIterator
*
*****************************************************************************/
static BOOLEAN bSearchForFavoritesBandIterator (
    void * pvEntry,
    void * pvArg
            )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry =
        (PRESETS_BAND_DESCRIPTOR_STRUCT *)pvEntry;
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIteratorStruct =
            (PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *)pvArg;

    if (psEntry == NULL)
    {
        // Error!
        return FALSE;
    }

    if (psEntry->eType == PRESET_BAND_TYPE_FEATURED)
    {
        if (psEntry->tId == psIteratorStruct->tId)
        {
            // Found!
            psIteratorStruct->hBand = psEntry->hBand;
        }

        if ((psEntry->bRemoved == TRUE) &&
            (psEntry->tBandId != psIteratorStruct->tCurrentBandId))
        {
            psIteratorStruct->un32Removed++;
        }

        psIteratorStruct->un32BandsCount++;
    }
    // Keep iteratin'
    return TRUE;
}

/*****************************************************************************
*
* bPresetsNotifyIterator
*
*****************************************************************************/
static BOOLEAN bPresetsNotifyIterator (
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc,
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    if ((psBandDesc == NULL) || (psObj == NULL))
    {
        return FALSE;
    }

    PRESETS_bUpdate( (PRESETS_OBJECT)psObj,
        psBandDesc->hBand,
        PRESETS_OBJECT_EVENT_BAND_ADDED );

    return TRUE;
}

/*****************************************************************************
*
* n16CompareFFBankIDs
*
*****************************************************************************/
static N16 n16CompareFFBankIDs(void *pvData1, void *pvData2)
{
    FAVORITES_BAND_ID tData1, tData2;
    N16 n16Return = 1;

    if(pvData1 != NULL && pvData2 != NULL)
    {
        tData1 = *(FAVORITES_BAND_ID *)pvData1;
        tData2 = *(FAVORITES_BAND_ID *)pvData2;

        if(tData1 > tData2)
        {
            n16Return = 1;
        }
        else if(tData1 < tData2)
        {
            n16Return = -1;
        }
        else
        {
            n16Return = 0;
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   bIsUpdateNeeded
*
*****************************************************************************/
static BOOLEAN bIsUpdateNeeded(
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;

    psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
        OSAL.pvLinkedListThis(psObj->hCurrentBand);

    return psBandDesc->bRemoved;
}

/*****************************************************************************
*
*   vInitiateUpdate
*
*****************************************************************************/
static void vInitiateUpdate(
    PRESETS_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOwner, bSuccess;

    // Should own the DECODER object
    bOwner = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
    if (bOwner == FALSE)
    {
        return;
    }

    // Set "in-progress" state for FF table
    bSuccess = RADIO_bUpdateFeaturedFavoritesState(psObj->hDecoder, FALSE);
    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            PRESETS_OBJECT_NAME
            ": Failed to set FF table into 'in-progress' state");
        return;
    }

    // Turn on the FF. If it fails, an error message will be
    // printed inside.
    RADIO_bFeaturedFavoritesEnable(psObj->hDecoder, TRUE);

    return;
}

/*****************************************************************************
*
*   hBrowseBands
*
*****************************************************************************/
static PRESET_BAND_OBJECT hBrowseBands(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    SMSAPI_DIRECTION_ENUM eDirection
        )
{
    DECODER_OBJECT hDecoder;
    BOOLEAN bPresetsLocked = FALSE, bDecoderLocked = FALSE;
    PRESET_BAND_OBJECT hReturnBand = PRESET_BAND_INVALID_OBJECT;

    do
    {
        UN32 un32Bands;
        BOOLEAN bSuccess, bUpdate;
        PRESETS_OBJECT_STRUCT *psObj;
        SMSAPI_RETURN_CODE_ENUM eSuccess;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc = NULL;

        // Get decoder intsance
        hDecoder = PRESETS_hDecoder(hPresets);
        if (hDecoder == DECODER_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Failed to get a Decoder handle from Presets");
            break;
        }

        // Lock decoder instance
        bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bDecoderLocked == FALSE)
        {
            break;
        }

        // Verify and lock the Presets service
        bPresetsLocked = SMSO_bLock((SMS_OBJECT) hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bPresetsLocked == FALSE)
        {
            break;
        }

        // De-reference object
        psObj = (PRESETS_OBJECT_STRUCT *) hPresets;

        // Determine, how many preset bands we have
        eReturnCode = OSAL.eLinkedListItems(psObj->hBands, &un32Bands);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Failed to get a number of Preset Bands: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            break;
        }

        if (un32Bands == 0)
        {
            break;
        }
        else if (eDirection == SMSAPI_DIRECTION_DIRECT)
        {
            OSAL_LINKED_LIST_ENTRY hEntry = 
                OSAL_INVALID_LINKED_LIST_ENTRY;

            psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                OSAL.pvLinkedListThis(psObj->hCurrentBand);

            // Already current?
            if (psBandDesc->hBand == hBand)
            {
                // Nothing to do
                hReturnBand = hBand;
                break;
            }

            // Find the band specified
            eReturnCode = OSAL.eLinkedListLinearSearch(
                psObj->hBands,
                &hEntry,
                n16CompareBands,
                hBand );

            if (eReturnCode != OSAL_SUCCESS)
            {
                // All Preset Bands are belongs to the Presets service
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESETS_OBJECT_NAME": Preset Band not found: %s (#%d)",
                        OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
                break;
            }

            psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

            // Set the current band
            psObj->hCurrentBand = hEntry;
        }
        else
        {
            if (un32Bands == 1)
            {
                // Presets service configured to use a circular LL of preset bands.
                // If there is only one band, this API will wrap into the same band.
                // So, just return the current band
                psBandDesc = (PRESETS_BAND_DESCRIPTOR_STRUCT *)
                    OSAL.pvLinkedListThis(psObj->hCurrentBand);

                // Preset Band to return
                hReturnBand = psBandDesc->hBand;
                break;
            }
            else if (eDirection == SMSAPI_DIRECTION_NEXT)
            {
                // Get the next band and set it as
                // the current
                psObj->hCurrentBand = OSAL.hLinkedListNext(
                    psObj->hCurrentBand,
                    (void **)((void *) &psBandDesc)
                        );
            }
            else if (eDirection == SMSAPI_DIRECTION_PREVIOUS)
            {
                 // Get the previous band and set it as
                // the current
                psObj->hCurrentBand = OSAL.hLinkedListPrev(
                    psObj->hCurrentBand,
                    (void **)((void *) &psBandDesc)
                        );
            }
        }

        // Get the band handle
        hReturnBand = psBandDesc->hBand;

        // update the band tags to indicate this band is the current one
        eSuccess = eUpdateCurrentBandTags(psObj, hReturnBand);
        if (eSuccess != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Failed to update the current band tags: %s (#%d)",
                    SMSAPI_DEBUG_pacReturnCodeText(eSuccess), eSuccess);

            // Invalidate the return
            hReturnBand = PRESET_BAND_INVALID_OBJECT;
            break;
        }

        bSuccess = PRESETS_bCurrentBandSync(hPresets, hBand);
        if (bSuccess == FALSE)
        {
            puts(PRESETS_OBJECT_NAME": Failed to sync current preset band");

            // This fail consider as non-critical for the Presets service;
        }

        bSuccess = PRESETS_bRemoveAllSelectedFFBanks(hPresets);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESETS_OBJECT_NAME": Failed to remove all selected featured bands");

            // Should not go ahead without this
            break;
        }

        // Determine, if the we need to initiate the Featured Favorites update
        bUpdate = bIsUpdateNeeded(psObj);
        if (bUpdate == TRUE)
        {
            // Should not affect the current band switch
            vInitiateUpdate((PRESETS_OBJECT_STRUCT *)hPresets);
        }

    } while (FALSE);

    if (bPresetsLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    if (bDecoderLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return hReturnBand;
}

#if SMS_DEBUG == 1
/*****************************************************************************
 *
 *       bPrintBands
 *
 *       An OSAL LL iteration callback which aids in the printing
 *       of each band currently available in the Presets service.
 *
 *****************************************************************************/
static BOOLEAN bPrintBands(
    void *pvData,
    void *pvArg
        )
{
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc =
        (PRESETS_BAND_DESCRIPTOR_STRUCT *) pvData;
    PRESETS_PRINT_STRUCT *psPrint =
        (PRESETS_PRINT_STRUCT *) pvArg;
    PRESETS_DEBUG_BAND_ATTR_STRUCT sBandAttrs;
    PRESETS_DEBUG_PRESET_ATTR_STRUCT sPresetAttrs;
    size_t tCurrentPreset, tCurrentBandId;

    if ((pvData == NULL) || (pvArg == NULL))
    {
        // Stop iterating
        return FALSE;
    }

    // if this is a band
    sBandAttrs.hBand = psBandDesc->hBand;
    sBandAttrs.hDescription = BAND.hDescription(psBandDesc->hBand);
    sBandAttrs.hLongName = BAND.hLongName(psBandDesc->hBand);
    sBandAttrs.hName = BAND.hName(psBandDesc->hBand);
    sBandAttrs.hVerbose = BAND.hVerboseName(psBandDesc->hBand);
    sBandAttrs.tCapacity = psBandDesc->tCapacity;
    sBandAttrs.tId = psBandDesc->tId;
    sBandAttrs.tBandId = psBandDesc->tBandId;
    sBandAttrs.tOrder = psBandDesc->tOrder;

    tCurrentBandId = PRESETS_tGetCurrentBandId(psPrint->hPresets);
    if (tCurrentBandId == psBandDesc->tBandId)
    {
        sBandAttrs.bCurrent = TRUE;
    }
    else
    {
        sBandAttrs.bCurrent = FALSE;
    }

    if (psBandDesc->eType == PRESET_BAND_TYPE_USER)
    {
        sBandAttrs.eType = PRESET_BAND_TYPE_USER;
        sBandAttrs.tPurpose = FAVORITES_BAND_PURPOSE_NONE;
        sBandAttrs.tSequence = FAVORITES_BAND_SEQUENCE_MIN;
        sBandAttrs.bRemoved = FALSE;
    }
    else
    {
        sBandAttrs.eType = PRESET_BAND_TYPE_FEATURED;
        sBandAttrs.tPurpose = PRESET_BAND_tGetPurpose(psBandDesc->hBand);
        sBandAttrs.tSequence = PRESET_BAND_tGetSequence(psBandDesc->hBand);
        sBandAttrs.bRemoved = psBandDesc->bRemoved;
    }

    // Print the band
    psPrint->n32CharsWritten +=
        PRESETS_DEBUG_n32PrintBand(
            &sBandAttrs,
            psPrint->psFile,
            psPrint->pacFormat );

    // Print this band's presets
    for (tCurrentPreset = 0;
         tCurrentPreset < psBandDesc->tCapacity;
         tCurrentPreset++)
    {
        sPresetAttrs.tPresetIndex = tCurrentPreset;
        sPresetAttrs.hName = psBandDesc->pasPresets[tCurrentPreset].hName;
        sPresetAttrs.tServiceId = psBandDesc->pasPresets[tCurrentPreset].tServiceId;
        sPresetAttrs.tChannelId = CHANNEL.tChannelId(
            psBandDesc->pasPresets[tCurrentPreset].hChannel);
        sPresetAttrs.hChannelName = CHANNEL.hShortName(
            psBandDesc->pasPresets[tCurrentPreset].hChannel);

        psPrint->n32CharsWritten +=
            PRESETS_DEBUG_n32PrintPreset(
                &sPresetAttrs,
                psPrint->psFile,
                psPrint->pacFormat );
    }
    return TRUE;
}
#endif
