/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:PRESET_BAND implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "string_obj.h"
#include "cm.h"
#include "tag_obj.h"
#include "ccache.h"
#include "decoder_obj.h"
#include "category_obj.h"

#include "presets_obj.h"
#include "presets_debug.h"

#include "preset_band_obj.h"
#include "_preset_band_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       hCreate
*
*       This API is used to create a new preset band and add it to the given
*       presets service.
*
*       Inputs:
*           hPresets - A valid PRESETS object handle to which this band is
*               to be added.
*           tCapacity - An integer indicating the number of presets this
*               band is to manage.
*           pacName - A valid pointer to an array of characters indicating
*               the name for this band.
*
*       Outputs:
*           A valid PRESET_BAND_OBJECT handle on success or
*           PRESET_BAND_INVALID_OBJECT on error.
*
*****************************************************************************/
static PRESET_BAND_OBJECT hCreate (
    PRESETS_OBJECT hPresets,
    size_t tCapacity,
    const char *pacName
        )
{
    BOOLEAN bLocked;
    DECODER_OBJECT hDecoder;
    PRESET_BAND_PRIVATE_ATTRS_STRUCT sAttrs;
    PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;

    // Get decoder instance
    hDecoder = PRESETS_hDecoder(hPresets);

    // Lock decoder instance
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        PRESET_BAND_OBJECT hCurrentBand;

        // Clear the attributes structure
        OSAL.bMemSet(&sAttrs, 0, sizeof(sAttrs));

        //Set band capacity
        sAttrs.tCapacity = tCapacity;

        //Set band name
        sAttrs.hName = STRING.hCreate(pacName, 0);

        // Set band type
        sAttrs.eType = PRESET_BAND_TYPE_USER;

        // create the band
        // pass in TAG_INVALID_OBJECT to indicate that this is a brand new band,
        // and not one being re-created from the config file
        hBand = hCreateBandLocal(
            hPresets,
            tCapacity,
            &sAttrs,
            TAG_INVALID_OBJECT
                );

        STRING_vDestroy(sAttrs.hName);

        hCurrentBand = PRESETS.hCurrentBand(hPresets);
        if (hCurrentBand == PRESET_BAND_INVALID_OBJECT)
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            // No current band? Means that this one is the first one.
            eReturnCode = PRESETS.eSetCurrentBand(hBand);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME": Failed to set the current Band: %s, %s (#%d)",
                    STRING.pacCStr(BAND.hName(hBand)),
                    SMSAPI_DEBUG_pacReturnCodeText(eReturnCode),
                    eReturnCode);
            }
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return hBand;
}

/*****************************************************************************
*
*       eDestroyPublic
*
*       This API is used to destroy a previously created band and remove it
*       from the presets service.
*
*       Inputs:
*           hBand - A valid BAND object handle which the user wishes to
*               destroy.
*
*       Outputs:
*           None
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDestroyPublic (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    PRESET_BAND_TYPE_ENUM eType;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        eType = BAND.eType(hBand);

        if (eType == PRESET_BAND_TYPE_USER)
        {
            PRESET_BAND_vDestroy(hBand);
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       hName
*
*       This API is used to get this band's name.
*
*       Inputs:
*           hBand - A valid band object handle from which the user wishes to
*               query for its name.
*
*       Outputs:
*           A valid STRING_OBJECT on success, STRING_INVALID_OBJECT on error.
*
*****************************************************************************/
static STRING_OBJECT hName (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    STRING_OBJECT hName = STRING_INVALID_OBJECT;

    // Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT)hBand);

    if (bValid == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        hName = psObj->hName;
    }

    return hName;
}

/*****************************************************************************
*
*       hLongName
*
*       This API is used to get this band's long name.
*
*       Inputs:
*           hBand - A valid band object handle from which the user wishes to
*               query for its long name.
*
*       Outputs:
*           A valid STRING_OBJECT on success, STRING_INVALID_OBJECT on error.
*
*****************************************************************************/
static STRING_OBJECT hLongName (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    STRING_OBJECT hLongName = STRING_INVALID_OBJECT;

    // Verify object
    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        hLongName = psObj->hLongName;
    }

    return hLongName;
}

/*****************************************************************************
*
*       eSetLongName
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetLongName (
    PRESET_BAND_OBJECT hBand,
    char *pacLongName
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        if (psObj->eType == PRESET_BAND_TYPE_USER)
        {
            BOOLEAN bModified = FALSE;

            if (pacLongName == NULL)
            {
                pacLongName = "";
            }

            if (psObj->hLongName == STRING_INVALID_OBJECT)
            {
                psObj->hLongName = STRING.hCreate(pacLongName, 0);
                if (psObj->hLongName != STRING_INVALID_OBJECT)
                {
                    bModified = TRUE;
                }
                else
                {
                    // Cannot create STRING object. Very bad.
                    eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                }
            }
            else
            {
                bModified = STRING.bModifyCStr(psObj->hLongName, pacLongName);
                // Even if STRING is not modified, it is not the error, since
                // the text could be just the same as it was before
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }

            if (bModified == TRUE)
            {
                eReturnCode = eSetLongNameTag(psObj->hTag, psObj->hLongName);
            }
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Update PRESETS object data
            PRESETS_OBJECT hPresets =
                (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);

            PRESETS_bUpdate(hPresets, hBand, PRESETS_OBJECT_EVENT_BAND_MODIFIED);
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       hVerboseName
*
*       This API is used to get this band's verbose name.
*
*       Inputs:
*           hBand - A valid band object handle from which the user wishes to
*               query for its verbose name.
*
*       Outputs:
*           A valid STRING_OBJECT on success, STRING_INVALID_OBJECT on error.
*
*****************************************************************************/
static STRING_OBJECT hVerboseName (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    STRING_OBJECT hVerboseName = STRING_INVALID_OBJECT;

    // Verify object
    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        hVerboseName = psObj->hVerboseName;
    }

    return hVerboseName;
}

/*****************************************************************************
*
*       eSetVerboseName
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetVerboseName (
    PRESET_BAND_OBJECT hBand,
    char *pacVerboseName
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        if (psObj->eType == PRESET_BAND_TYPE_USER)
        {
            BOOLEAN bModified = FALSE;

            if (pacVerboseName == NULL)
            {
                pacVerboseName = "";
            }

            if (psObj->hVerboseName == STRING_INVALID_OBJECT)
            {
                psObj->hVerboseName = STRING.hCreate(pacVerboseName, 0);
                if (psObj->hVerboseName != STRING_INVALID_OBJECT)
                {
                    bModified = TRUE;
                }
                else
                {
                    // Cannot create STRING object. Very bad.
                    eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                }
            }
            else
            {
                bModified = STRING.bModifyCStr(psObj->hVerboseName, pacVerboseName);
                // Even if STRING is not modified, it is not the error, since
                // the text could be just the same as it was before
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }

            if (bModified == TRUE)
            {
                eReturnCode = eSetVerboseNameTag(psObj->hTag, psObj->hVerboseName);
            }
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Update PRESETS object data
            PRESETS_OBJECT hPresets =
                (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);

            PRESETS_bUpdate(hPresets, hBand, PRESETS_OBJECT_EVENT_BAND_MODIFIED);
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       hDescription
*
*       This API is used to get this band's description.
*
*       Inputs:
*           hBand - A valid band object handle from which the user wishes to
*               query for its verbose name.
*
*       Outputs:
*           A valid STRING_OBJECT on success, STRING_INVALID_OBJECT on error.
*
*****************************************************************************/
static STRING_OBJECT hDescription (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    STRING_OBJECT hDescription = STRING_INVALID_OBJECT;

    // Verify object
    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        hDescription = psObj->hDescription;
    }

    return hDescription;
}

/*****************************************************************************
*
*       eSetDescription
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetDescription (
    PRESET_BAND_OBJECT hBand,
    char *pacDescription
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and lock object
    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        if (psObj->eType == PRESET_BAND_TYPE_USER)
        {
            BOOLEAN bModified = FALSE;

            if (pacDescription == NULL)
            {
                pacDescription = "";
            }

            if (psObj->hDescription == STRING_INVALID_OBJECT)
            {
                psObj->hDescription = STRING.hCreate(pacDescription, 0);
                if (psObj->hDescription != STRING_INVALID_OBJECT)
                {
                    bModified = TRUE;
                }
                else
                {
                    // Cannot create STRING object. Very bad.
                    eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
                }
            }
            else
            {
                bModified = STRING.bModifyCStr(psObj->hDescription, pacDescription);
                // Even if STRING is not modified, it is not the error, since
                // the text could be just the same as it was before
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }

            if (bModified == TRUE)
            {
                eReturnCode = eSetDescriptionTag(psObj->hTag, psObj->hDescription);
            }
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Update PRESETS object data
            PRESETS_OBJECT hPresets =
                (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);

            PRESETS_bUpdate(hPresets, hBand, PRESETS_OBJECT_EVENT_BAND_MODIFIED);
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       tCapacity
*
*       This API is used to get this band's preset capacity.
*
*       Inputs:
*           hBand - A valid band object handle from which the user wishes to
*               query for its capacity.
*
*       Outputs:
*           A size_t indicating the capacity of this band
*
*****************************************************************************/
static size_t tCapacity (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    size_t tCapacity = 0;

    // Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Extract the band capacity
        tCapacity = psObj->tPresetCapacity;
    }

    return tCapacity;
}

/*****************************************************************************
*
*       eType
*
*       This API is used to get this band's loaded flag.
*
*       Inputs:
*           hBand - A valid band object handle from which the user wishes to
*               query for its status.
*
*       Outputs:
*           A PRESET_BAND_TYPE_ENUM value that indicates a status of this band
*
*****************************************************************************/
static PRESET_BAND_TYPE_ENUM eType (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    PRESET_BAND_TYPE_ENUM eType = PRESET_BAND_TYPE_INVALID;

    // Verify SMS Objects
    bValid = SMSO_bValid((SMS_OBJECT)hBand);

    if (bValid == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        eType = psObj->eType;
    }

    return eType;
}

/*****************************************************************************
*
*       eSetPreset
*
*       This API is used to set the preset channel in this band at a specific
*       preset index.
*
*       Inputs:
*           hBand - A valid band object handle in which the user wishes to
*               set a preset channel.
*           tPresetIndex - A value indicating the index at which to store
*               the preset
*           tChannelId - The channel's identifier for this preset
*
*       Outputs:
*           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
*           of type SMSAPI_RETURN_CODE_ENUM
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetPreset (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    CHANNEL_ID tChannelId
        )
{
    BOOLEAN bLocked, bDecoderLocked;
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
    DECODER_OBJECT hDecoder;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // Get the presets
        hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    // now get the decoder
    hDecoder = PRESETS_hDecoder(hPresets);

    bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bDecoderLocked == TRUE)
    {
        // Verify and lock the band (locks the Presets service)
        bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            CCACHE_OBJECT hCCache;

            // Get the ccache handle
            hCCache = DECODER_hCCache(hDecoder);
            if(CCACHE_INVALID_OBJECT != hCCache)
            {
                CHANNEL_OBJECT hChannel;
                PRESET_BAND_OBJECT_STRUCT *psObj = 
                    (PRESET_BAND_OBJECT_STRUCT *)hBand;
                SERVICE_ID tServiceId = SERVICE_INVALID_ID;

                // Attempt find an existing channel with this given channel id.
                hChannel = CCACHE_hChannelFromIds(
                    hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);

                // This is an existing channel, extract the service id
                tServiceId = CHANNEL.tServiceId(hChannel);

                // Save preset or clear it up if we have Invalid Channel ID.
                if ((tServiceId != SERVICE_INVALID_ID) ||
                    (tChannelId == CHANNEL_INVALID_ID))
                {
                    // Get the presets service to update
                    // this preset
                    eReturnCode = eSetPresetLocal(
                                        psObj,
                                        tPresetIndex,
                                        PRESET_UPDATE_TYPE_CHAN,
                                        tServiceId,
                                        NULL);

                    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        // Commit now, because it is user's request
                        eReturnCode = eSavePresetToConfigFile(
                                      psObj,
                                      tPresetIndex,
                                      PRESET_UPDATE_TYPE_CHAN,
                                      tServiceId,
                                      NULL,
                                      TRUE);
                    }
                }
                else
                {
                    // Requested ChannelID cannot be resolved into SID
                    eReturnCode = SMSAPI_RETURN_CODE_INVALID_CHANNEL;
                }
            }  // hCCache

            else
            {
                // Error!
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            }

            SMSO_vUnlock((SMS_OBJECT)hBand);

        } // hBand Locked

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       tGetPreset
*
*       This API is used to get the preset channel in this band from a
*       specific preset index.
*
*       Inputs:
*           hBand - A valid band object handle in which the user wishes to
*               get a preset channel.
*           tPresetIndex - A value indicating the index at which to locate
*               the preset
*
*       Outputs:
*           A valid CHANNEL_ID on success; otherwise CHANNEL_INVALID_ID
*
*****************************************************************************/
static CHANNEL_ID tGetPreset (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex
        )
{
    BOOLEAN bLocked;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;

    // Verify and lock the band (locks the Presets service)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;
        PRESETS_OBJECT hPresets;
        DECODER_OBJECT hDecoder;

        // Verify the provided index
        if (tPresetIndex < psObj->tPresetCapacity)
        {
            // Get this from the presets service
            PRESETS_bGetPreset(
                hBand,
                psObj->hEntry,
                tPresetIndex,
                &tServiceId,
                NULL );
        }

        hPresets = (PRESETS_OBJECT)
            SMSO_hParent((SMS_OBJECT)hBand);

        hDecoder = PRESETS_hDecoder(hPresets);

        SMSO_vUnlock((SMS_OBJECT)hBand);

        // Do the channel id look up outside of locking the band to help avoid
        // deadlock conditions between PRESETS and DECODER
        tChannelId =
            DECODER.tGetChannelId(hDecoder, tServiceId);
    }

    return tChannelId;
}

/*****************************************************************************
*
*       eSetPresetName
*
*       This API is used to set the preset name in this band at a specific
*       preset index.
*
*       Inputs:
*           hBand - A valid band object handle in which the user wishes to
*               set a preset channel.
*           tPresetIndex - A value indicating the index at which to store
*               the preset name
*           pacName - A valid pointer to an array of characters which is
*               to be used as the preset's name
*
*       Outputs:
*           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
*           of type SMSAPI_RETURN_CODE_ENUM
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetPresetName (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    const char *pacName
        )
{
    BOOLEAN bLocked, bDecoderLocked;
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
    DECODER_OBJECT hDecoder;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    // Get presets instance
    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    // Get the decoder object
    hDecoder = PRESETS_hDecoder(hPresets);

    // Lock decoder instance
    bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bDecoderLocked == TRUE)
    {
        // Verify and lock the band (locks the Presets service)
        bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PRESET_BAND_OBJECT_STRUCT *psObj =
                (PRESET_BAND_OBJECT_STRUCT *)hBand;

            // Get the presets service to update
            // this preset
            eReturnCode = eSetPresetLocal(
                  psObj,
                  tPresetIndex,
                  PRESET_UPDATE_TYPE_NAME,
                  SERVICE_INVALID_ID,
                  pacName
                      );

            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // need to save to the config file....
                eReturnCode = eSavePresetToConfigFile(
                    psObj, 
                    tPresetIndex, 
                    PRESET_UPDATE_TYPE_NAME, 
                    SERVICE_INVALID_ID, 
                    pacName, 
                    TRUE);
            }

            SMSO_vUnlock((SMS_OBJECT)hBand);
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       hGetPresetName
*
*       This API is used to get the preset name in this band from a specific
*       preset index.
*
*       Inputs:
*           hBand - A valid band object handle in which the user wishes to
*               get a preset name.
*           tPresetIndex - A value indicating the index at which to locate
*               the preset
*
*       Outputs:
*           A valid STRING_OBJECT on success; otherwise STRING_INVALID_OBJECT
*
*****************************************************************************/
static STRING_OBJECT hGetPresetName (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex
        )
{
    BOOLEAN bLocked;
    STRING_OBJECT hName = STRING_INVALID_OBJECT;

    // Verify and lock the band (locks the Presets service)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Verify the provided index
        if (tPresetIndex < psObj->tPresetCapacity)
        {
            // Get this from the presets service
            PRESETS_bGetPreset(
                hBand,
                psObj->hEntry,
                tPresetIndex,
                NULL,
                &hName );
        }

        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return hName;
}

/*****************************************************************************
*
*       eClearAllPresets
*
*       This API is used to clear all presets of their channels and optionaly
*       their names as well.
*
*       Inputs:
*           hBand - A valid band object handle from which the user wishes to
*               clear all presets.
*           bClearNames - A flag indicating if all preset names should be
*               cleared as well
*
*       Outputs:
*           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
*           of type SMSAPI_RETURN_CODE_ENUM
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eClearAllPresets (
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bClearNames
        )
{
    BOOLEAN bLocked, bDecoderLocked;
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
    DECODER_OBJECT hDecoder;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    // Get Presets object
    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    // Get Decoder object
    hDecoder = PRESETS_hDecoder(hPresets);

    bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bDecoderLocked == TRUE)
    {
        // Verify and lock the band (locks the Presets service)
        bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            BOOLEAN bOk;

            // Ask the Presets service to clear our presets
            bOk = PRESETS_bClearAllPresets(hBand, bClearNames, TRUE);

            // Request update of the attached category (if any) in that
            // case, because this is the user's decision.

            // Verify operation result and condition return value
            if (bOk == TRUE)
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                // Error - we were unable to successfully
                // update the presets service
                eReturnCode =  SMSAPI_RETURN_CODE_ERROR;
            }

            SMSO_vUnlock((SMS_OBJECT)hBand);

            // Update presets
            PRESETS_bUpdate(hPresets, hBand, PRESETS_OBJECT_EVENT_BAND_MODIFIED);
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*    eClearAllPresetsPublic
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eClearAllPresetsPublic (
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bClearNames
        )
{
    BOOLEAN bValid;
    PRESET_BAND_TYPE_ENUM eType;
    SMSAPI_RETURN_CODE_ENUM eReturn =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        eType = BAND.eType(hBand);

        if (eType == PRESET_BAND_TYPE_USER)
        {
            eReturn = eClearAllPresets(
                hBand,
                bClearNames
                    );
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*    eSetPresetPublic
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetPresetPublic (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    CHANNEL_ID tChannelId
        )
{
    BOOLEAN bValid;
    PRESET_BAND_TYPE_ENUM eType;
    SMSAPI_RETURN_CODE_ENUM eReturn =
        SMSAPI_RETURN_CODE_INVALID_INPUT;

    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        eType = BAND.eType(hBand);

        if (eType == PRESET_BAND_TYPE_USER)
        {
            eReturn = eSetPreset(
                hBand,
                tPresetIndex,
                tChannelId
                    );
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*       eIterate
*
*       This API is used to iterate all presets found within the provided
*       band object.
*
*       Inputs:
*           hBand - A handle to a valid BAND object in which the caller
*               wishes to iterate
*           n16Iterator - The iteration handler that will be used during the
*               iteration of presets.
*           pvIterateArg - The pointer to an optional argument
*               provided to the iteration handler.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_SUCCESS on success or an appropriate error code
*               of type SMSAPI_RETURN_CODE_ENUM on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate (
    PRESET_BAND_OBJECT hBand,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        )
{
    BOOLEAN bLocked, bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;

    // Verify input
    if (n16Iterator == NULL)
    {
        return eReturnCode;
    }

    // ee@ NOTE: In this function need to lock
    // a DECODER object first, since inside eIterateCommon it
    // uses to get channels info from ccache under DECODER's lock
    // An interactive deadlock condition is possible otherwise

    //  object
    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    // Verify presets object instance
    bValid = SMSO_bValid((SMS_OBJECT)hPresets);
    if (bValid == TRUE)
    {
        DECODER_OBJECT hDecoder;

        // Get decoder object instance
        hDecoder = PRESETS_hDecoder(hPresets);

        // Verify and lock decoder object
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            // Verify and lock the band (locks the Presets service)
            bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
            if(bLocked == TRUE)
            {
                // Get the Presets service to iterate this band only
                eReturnCode = PRESETS_eIterateBand (
                    hBand,
                    n16Iterator,
                    pvIterateArg );

                SMSO_vUnlock((SMS_OBJECT)hBand);
            }

            // Relinquish ownership
            SMSO_vUnlock((SMS_OBJECT)hDecoder);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       eTune
*
*       This API is used to tune the decoder to a specific preset within
*       a band.
*
*       Inputs:
*           hBand - A valid band object handle in which to locate the
*               preset to be tuned.
*           tPresetIndex - A value indicating the index at which to locate
*               the preset to tune
*           bLockOverride - A flag used to signify if the caller wishes to
*               override the lock attribute of the channel associated with
*               this preset.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error
*           code of type SMSAPI_RETURN_CODE_ENUM
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTune (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    BOOLEAN bLockOverride
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bLocked;
    PRESETS_OBJECT hPresets;
    DECODER_OBJECT hDecoder;
    PRESET_BAND_OBJECT_STRUCT *psObj =
        (PRESET_BAND_OBJECT_STRUCT *)hBand;

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }
    else
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    hDecoder = PRESETS_hDecoder(hPresets);

    // Verify the provided index
    if (tPresetIndex < psObj->tPresetCapacity)
    {
        // Have the presets service perform the tune
        eReturnCode = PRESETS_eTune (
            hDecoder,
            hBand,
            psObj->hEntry,
            tPresetIndex,
            bLockOverride );
    }
    else
    {
        // We were given a bad preset index
        eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       n32FPrintf
*
*       This API is used to print the band object to a file.  This
*       function is meant to aid in debugging and is not meant to assist
*       the presets service to persist across power cycles.
*
*       Inputs:
*           hBand - A handle to a valid PRESET_BAND object which the caller
*               wishes to print.
*           *psFile - A pointer to a device which is to receive formatted
*               text output.
*
*       Outputs:
*           The number of characters written.
*
*****************************************************************************/
static N32 n32FPrintf (
    PRESET_BAND_OBJECT hBand,
    FILE *psFile
        )
{
    N32 n32Return = EOF;
    BOOLEAN bLocked, bDecoderLocked = FALSE;
    DECODER_OBJECT hDecoder;
    PRESETS_OBJECT hPresets;
    PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

    // Verify input
    if (psFile == NULL)
    {
        return EOF;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // Get the presets
        hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }
    else
    {
        return EOF;
    }

    // now get the decoder
    hDecoder = PRESETS_hDecoder(hPresets);

    bDecoderLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bDecoderLocked == TRUE)
    {
        // Verify and lock the band (locks the Presets service)
        bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // Get the presets service to print the
            // information about this band
            n32Return = PRESETS_n32PrintBand( hBand, psObj->hEntry, psFile );

            SMSO_vUnlock((SMS_OBJECT)hBand);
        }

        SMSO_vUnlock((SMS_OBJECT) hDecoder);
    }

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       PRESET_BAND_vDelete
*
*       This function is called by the presets service in order to
*       tell band objects it's time to go away.  Don't worry about calling
*       PRESETS_bRemoveBand -- just delete ourselves.
*
*****************************************************************************/
void PRESET_BAND_vDelete (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Delete the name
        if (psObj->hName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hName);
            psObj->hName = STRING_INVALID_OBJECT;
        }

        if (psObj->hLongName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hLongName);
            psObj->hLongName = STRING_INVALID_OBJECT;
        }

        if (psObj->hVerboseName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hVerboseName);
            psObj->hVerboseName = STRING_INVALID_OBJECT;
        }

        if (psObj->hDescription != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hDescription);
            psObj->hDescription = STRING_INVALID_OBJECT;
        }

        // Destroy this band
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   PRESET_BAND_bRestorePresetBands
*
* This function will process information found in the config file
* and create bands and populate them to the state they were in.
*
*****************************************************************************/
BOOLEAN PRESET_BAND_bRestorePresetBands(
    PRESETS_OBJECT hPresets
        )
{
    PRESET_BAND_TAG_ITERATOR_STRUCT sIteratorStruct;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bReturn = FALSE;

    // Construct an appropriate name for restore preset list
    snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            PRESET_BAND_OBJECT_NAME":RestoreLL");

    sIteratorStruct.hPresets = hPresets;

    eReturnCode = OSAL.eLinkedListCreate(
        &sIteratorStruct.hLL,
        &acName[0],
        NULL,
        OSAL_LL_OPTION_LINEAR);

    sIteratorStruct.bPresets = FALSE;

    if (eReturnCode == OSAL_SUCCESS)
    {
        hTag = PRESETS_hGetTag(hPresets);

        // the preset service object holds the parent tag of all preset band
        // tags. Iterate through all the child tags, and for each preset band
        // tag extract the information and store it in our LL.
        eResult = TAG_eIterateChildren(
            hTag,
            bPresetBandTagIterator,
            &sIteratorStruct
                );

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            DECODER_OBJECT hDecoder;
            CATEGORY_ID tCategoryId;
            BOOLEAN bBlocked;

            hDecoder = PRESETS_hDecoder(hPresets);
            tCategoryId = PRESETS.tCategoryId(hPresets);

            // Block category notifications
            bBlocked = CATEGORY_bBlockNotifications(hDecoder, tCategoryId);

            // now we iterate the LL and restore each preset
            OSAL.eLinkedListIterate(sIteratorStruct.hLL, bSetThePresets,
                                    (void *)hDecoder);

            // Releasing PRESETS Category notifications
            if (bBlocked == TRUE)
            {
                CATEGORY_vReleaseNotifications(hDecoder, tCategoryId);
            }

            bReturn = TRUE;
        }

        // remove entries and destroy the list since it isn't needed any more
        OSAL.eLinkedListRemoveAll(sIteratorStruct.hLL,
                                  vRemovePresetRestoreStruct);
        OSAL.eLinkedListDelete(sIteratorStruct.hLL);

    }

    return bReturn;
}

/*****************************************************************************
*
*   PRESET_BAND_vClearCurrentBand
*
*   This function takes a band tag, finds the child tag "current" and sets
*   its value to FALSE, indicating that it isn't the current band.
*
*****************************************************************************/
void PRESET_BAND_vClearCurrentBand(
    TAG_OBJECT hTag
        )
{
    STRING_OBJECT hTagName;

    // get this tag's name
    hTagName = TAG_hTagName(hTag);
    // is this a tag we're interested in?
    if (STRING.n16CompareCStr(PRESET_BAND_OBJECT_NAME, 0, hTagName) == 0)
    {
        TAG_OBJECT hCurrentTag = TAG_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eResult;

        // get the "Current" tag
        eResult = TAG_eGet(
            PRESET_BAND_CURRENT,
            hTag,
            &hCurrentTag,
            NULL,
            TRUE
                );

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            STRING_OBJECT hValueString;

            // Set the appropriate tag value (FALSE)
            hValueString = CM_hFalseString();

            // set the tag value
            eResult = TAG_eSetTagValue(
                          hCurrentTag,
                          TAG_TYPE_STRING,
                          &hValueString,
                          sizeof(STRING_OBJECT),
                          FALSE // don't commit yet
                              );

            if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME": Failed to clear Current Band attribute in the TAG");
            }
        }
    }
}

/*****************************************************************************
*
*   PRESET_BAND_eSaveCurrentBand
*
*   This function takes a band, finds its "Current" tag and sets it value to
*   TRUE
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESET_BAND_eSaveCurrentBand(
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    bValid = SMSO_bValid((SMS_OBJECT)hBand);
    if (bValid == TRUE)
    {
        TAG_OBJECT hCurrentTag = TAG_INVALID_OBJECT;
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // get the band's "current" tag
        eResult = TAG_eGet(
            PRESET_BAND_CURRENT,
            psObj->hTag,
            &hCurrentTag,
            NULL,
            TRUE
                );

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            STRING_OBJECT hValueString;

            // get a string with the appropriate text
            hValueString = CM_hTrueString();

            // set the tag value
            eResult = TAG_eSetTagValue(
                          hCurrentTag,
                          TAG_TYPE_STRING,
                          &hValueString,
                          sizeof(STRING_OBJECT),
                          TRUE // commit
                              );
        }
        else if (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
        {
            // must mean there is no file system
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*       PRESET_BAND_hCreateFeaturedBand
*
*       This API is used to create a new featured favorites band and add it
*       to the given presets service.
*
*       Inputs:
*           hPresets - A valid PRESETS object handle to which this band is
*               to be added.
*           tCapacity - An integer indicating the number of presets this
*               band is to manage.
*           sBank - A valid Featured Favorites object which contains a bank
*               information
*
*       Outputs:
*           A valid PRESET_BAND_OBJECT handle on success or
*           PRESET_BAND_INVALID_OBJECT on error.
*
*****************************************************************************/
PRESET_BAND_OBJECT PRESET_BAND_hCreateFeaturedBand (
    PRESETS_OBJECT hPresets,
    size_t tCapacity,
    PRESET_BAND_PRIVATE_ATTRS_STRUCT *psAttrs,
    SERVICE_ID const *ptPresets,
    size_t tSIDCount
        )
{
    BOOLEAN bSuccess = FALSE;
    PRESET_BAND_OBJECT hBand;
    SMSAPI_RETURN_CODE_ENUM eSuccess;
    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
    BOOLEAN bBlocked;
    CATEGORY_ID tCategoryId;

    if ((psAttrs != NULL) && (psAttrs->eType != PRESET_BAND_TYPE_FEATURED))
    {
        printf(PRESET_BAND_OBJECT_NAME
            ":FF: This API: %s should be called only for Type: Featured\n",
            __FUNCTION__);

        return PRESET_BAND_INVALID_OBJECT;
    }

    // create the band
    // pass in TAG_INVALID_OBJECT to indicate that this is a brand new band,
    // and not one being re-created from the config file
    hBand = hCreateBandLocal(
        hPresets,
        tCapacity,
        psAttrs,
        TAG_INVALID_OBJECT
            );

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        printf(PRESET_BAND_OBJECT_NAME":FF: Failed to create Bank ID: %d\n",
            psAttrs->sFeatured.tId);

        return PRESET_BAND_INVALID_OBJECT;
    }

    printf(PRESET_BAND_OBJECT_NAME":FF: Bank ID: %u created\n",
        psAttrs->sFeatured.tId);

    // Get the DECODER from PRESETS
    hDecoder = PRESETS_hDecoder(hPresets);
    // Get PRESETS' category ID
    tCategoryId = PRESETS.tCategoryId(hPresets);

    // Blocking notifications for PRESETS Category
    bBlocked = CATEGORY_bBlockNotifications(hDecoder, tCategoryId);

    // Blocking notifications from channel updates
    PRESETS_vBlockNotifications(hPresets, TRUE);

    do
    {
        // if a supported Featured Favorites Bank contains more assigned
        // channels than are supported per bank by the product, the product
        // displays only the first channels in the list when that bank is
        // selected by the user, truncating the list, see
        // TRS_Featured_Favorites_v1.1_2011-04-22

        eSuccess = PRESET_BAND_eSetPresets(hBand, ptPresets, tSIDCount);

        // Featured band was created. So, let's check error conditions
        // and perform some post-processing if we are OK.

        if (eSuccess != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Having some troubles with presets.
            break;
        }

        printf(PRESET_BAND_OBJECT_NAME
            ":FF: The presets of Bank ID: %u were successfully assigned\n", 
            psAttrs->sFeatured.tId);

        bSuccess = PRESETS_bCurrentBandSync(hPresets, hBand);
        if (bSuccess == FALSE)
        {
            // Failed to synchronize the current band with attached category.
            // Of course, it is bad. However, we already have successfully 
            // created band and it's presets. This will not affect the basic 
            // operation of the service. So, let's proceed and let them to 
            // synchronize it later ...
            puts(PRESET_BAND_OBJECT_NAME
                ":FF: Failed to synchronize the current band");
        }

        // Let's notify application when we have finished
        bSuccess = PRESETS_bUpdate(
            hPresets, 
            hBand, 
            PRESETS_OBJECT_EVENT_BAND_ADDED
                );

        if (bSuccess == FALSE)
        {
            // Failed to notify application. What's happened here?
            printf(PRESET_BAND_OBJECT_NAME
                ":FF: Bank ID: %d, Name: %s notification: Added failed\n",
                psAttrs->sFeatured.tId, STRING.pacCStr(psAttrs->hName));

            // Let's be optimistic and allow the application to get this 
            // band by another way, e.g., while iterating thru all bands 
            // on the next update
        }

        bSuccess = TRUE;

    } while (FALSE);

    if (bSuccess == FALSE)
    {
        vDestroyBandLocal(hPresets, hBand, FALSE);
        hBand = PRESET_BAND_INVALID_OBJECT;
    }

    // Unblocking notifications from channel updates
    PRESETS_vBlockNotifications(hPresets, FALSE);

    // Releasing PRESETS Category notifications
    if (bBlocked == TRUE)
    {
        CATEGORY_vReleaseNotifications(hDecoder, tCategoryId);
    }

    return hBand;
}

/*****************************************************************************
*
*   PRESET_BAND_eGetId
*
*****************************************************************************/
FAVORITES_BAND_ID PRESET_BAND_eGetId (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bOwner;
    FAVORITES_BAND_ID tId = FAVORITES_BAND_ID_INVALID;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBand);
    if (bOwner == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Pass thru Id
        tId = psObj->tId;
    }

    return tId;
}

/*****************************************************************************
*
*   PRESET_BAND_tGetSequence
*
*****************************************************************************/
FAVORITES_BAND_SEQUENCE PRESET_BAND_tGetSequence (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bOwner;
    FAVORITES_BAND_SEQUENCE tSequence = FAVORITES_BAND_SEQUENCE_INVALID;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBand);
    if (bOwner == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Pass thru sequence
        tSequence = psObj->tSequence;
    }

    return tSequence;
}

/*****************************************************************************
*
*   PRESET_BAND_tGetPurpose
*
*****************************************************************************/
FAVORITES_BAND_PURPOSE_MASK PRESET_BAND_tGetPurpose (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bOwner;
    FAVORITES_BAND_PURPOSE_MASK tPurpose = FAVORITES_BAND_PURPOSE_NONE;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBand);
    if (bOwner == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        tPurpose = psObj->tPurpose;
    }

    return tPurpose;
}

/*****************************************************************************
*
*   PRESET_BAND_eGetOrder
*
*****************************************************************************/
FAVORITES_BAND_ORDER PRESET_BAND_eGetOrder (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bOwner;
    FAVORITES_BAND_ORDER tOrder = FAVORITES_BAND_ORDER_INVALID;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBand);
    if (bOwner == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        tOrder = psObj->tOrder;
    }

    return tOrder;
}

/*****************************************************************************
*
*   PRESET_BAND_bIsRemoved
*
*****************************************************************************/
BOOLEAN PRESET_BAND_bIsRemoved (
    PRESET_BAND_OBJECT hBand
        )
{
    BOOLEAN bOwner, bRemoved = FALSE;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBand);
    if (bOwner == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        bRemoved = psObj->bRemoved;
    }

    return bRemoved;
}

/*****************************************************************************
*
*   PRESET_BAND_eSetRemoved
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESET_BAND_eSetRemoved(
    PRESET_BAND_OBJECT hBand,
    size_t tRemoved
        )
{
    TAG_OBJECT hBandTag = ((PRESET_BAND_OBJECT_STRUCT *)hBand)->hTag;
    return eSetRemoved(hBandTag, (BOOLEAN)tRemoved);
}

/*****************************************************************************
*
*       PRESET_BAND_vDestroy
*
*       This API is used to destroy a previously created band and remove it
*       from the presets service.
*
*       Inputs:
*           hBand - A valid BAND object handle which the user wishes to
*               destroy.
*
*       Outputs:
*           None
*
*****************************************************************************/
void PRESET_BAND_vDestroy (
    PRESET_BAND_OBJECT hBand
        )
{
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
    BOOLEAN bDecoderLocked = FALSE, bPresetsLocked = FALSE;

    do
    {
        BOOLEAN bLocked;

        if (hBand == PRESET_BAND_INVALID_OBJECT)
        {
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if (TRUE == bLocked)
        {
            // Get our parent PRESETS_OBJECT (this
            // will also validate the band handle)
            hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
            SMSO_vUnlock((SMS_OBJECT)hBand);
        }

        // Determine if we got a parent
        if (hPresets == PRESETS_INVALID_OBJECT)
        {
            break;
        }
        
        hDecoder = PRESETS_hDecoder(hPresets);

        // Need to lock decoder object first, 
        // cause an interactive deadlock is possible
        bDecoderLocked = SMSO_bLock(
            (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bDecoderLocked == FALSE)
        {
            break;
        }

        // Lock the presets service
        bPresetsLocked = SMSO_bLock(
            (SMS_OBJECT)hPresets, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bPresetsLocked == FALSE)
        {
            break;
        }

        // There is nothing we can do here in case of failure ... 
        vDestroyBandLocal(hPresets, hBand, TRUE);

    } while (FALSE);

    if (bPresetsLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hPresets);
    }

    if (bDecoderLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
*
*   bUpdatePresets
*
*****************************************************************************/
static BOOLEAN bUpdatePresets (
    PRESET_BAND_OBJECT hBand,
    SERVICE_ID const *ptArrangement,
    size_t tSIDCount,
    FAVORITES_BAND_CAPACITY tCapacity
        )
{
    BOOLEAN bBlocked = FALSE, bReturnCode = FALSE;
    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    do
    {
        BOOLEAN bCleared, bSuccess, bLocked = FALSE;
        SMSAPI_RETURN_CODE_ENUM eSuccess;
        PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;

        bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            // Get BAND's object parent (assuming the PRESETS)
            hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
            SMSO_vUnlock((SMS_OBJECT)hBand);
        }

        if(PRESETS_INVALID_OBJECT == hPresets)
        {
            break;
        }

        // Do not request update of the attached category, since this
        // could be populated with the same channels below
        bCleared = PRESETS_bClearAllPresets(hBand, TRUE, FALSE);
        if (bCleared == FALSE)
        {
            break;
        }
    
        eSuccess = PRESET_BAND_eSetPresets(hBand, ptArrangement, tSIDCount);
        if (eSuccess != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Having some troubles with presets.
            break;
        }

        printf(PRESET_BAND_OBJECT_NAME
            ":FF: The presets of Featured Band: %s successfully assigned\n", 
            STRING.pacCStr(BAND.hName(hBand)));


        // Get the DECODER from PRESETS
        hDecoder = PRESETS_hDecoder(hPresets);
        // Get PRESETS' category ID
        tCategoryId = PRESETS.tCategoryId(hPresets);

        // Blocking notifications for PRESETS Category
        bBlocked = CATEGORY_bBlockNotifications(hDecoder, tCategoryId);

        bSuccess = PRESETS_bCurrentBandSync(hPresets, hBand);
        if (bSuccess == FALSE)
        {
            // Failed to synchronize the current band with attached category.
            // Of course, it is bad. However, this will not affect the basic 
            // operation of the service. So, let's proceed and let them to 
            // synchronize it later ...
            puts(PRESET_BAND_OBJECT_NAME
                ":FF: Failed to synchronize the current band");
        }

        bReturnCode = TRUE;
        
    } while (FALSE);

    // Releasing PRESETS Category notifications
    if (bBlocked == TRUE)
    {
        CATEGORY_vReleaseNotifications(hDecoder, tCategoryId);
    }

    return bReturnCode;
}

/*****************************************************************************
*
*   bUpdateOrder
*
*****************************************************************************/
static BOOLEAN bUpdateOrder (
    PRESET_BAND_OBJECT hBand,
    FAVORITES_BAND_ORDER tOrder
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Modify Preset Band Object Field
        psObj->tOrder = tOrder;

        PRESETS_vUpdateBankDescriptorOrder (
                psObj->hEntry,
                tOrder);

        // Modify This Filed in the Tag
        eReturnCode = eSetOrder(psObj->hTag, tOrder);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   PRESET_BAND_bUpdateSequence
*
*****************************************************************************/
BOOLEAN PRESET_BAND_bUpdateSequence (
    PRESET_BAND_OBJECT hBand,
    FAVORITES_BAND_SEQUENCE tSequence
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Modify Preset Band Object Field
        psObj->tSequence = tSequence;

        // Modify This Filed in the Tag
        eReturnCode = eSetSequence(psObj->hTag, tSequence);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdatePurpose
*
*****************************************************************************/
static BOOLEAN bUpdatePurpose (
    PRESET_BAND_OBJECT hBand,
    FAVORITES_BAND_PURPOSE_MASK tPurpose
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Modify Preset Band Object Field
        psObj->tPurpose = tPurpose;

        // Modify This Filed in the Tag
        eReturnCode = eSetPurpose(psObj->hTag, tPurpose);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateShortName
*
*****************************************************************************/
static BOOLEAN bUpdateShortName (
    PRESET_BAND_OBJECT hBand,
    char const * pacShortName
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Modify Preset Band Object Field
        STRING.bModifyCStr(psObj->hName, pacShortName);

        // Modify This Filed in the Tag
        eReturnCode = eSetNameTag(psObj->hTag, psObj->hName);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateLongName
*
*****************************************************************************/
static BOOLEAN bUpdateLongName (
    PRESET_BAND_OBJECT hBand,
    char const * pacLongName
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // Relinquish ownership
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Modify Preset Band Object Field
        STRING.bModifyCStr(psObj->hLongName, pacLongName);

        // Modify This Filed in the Tag
        eReturnCode = eSetLongNameTag(psObj->hTag, psObj->hLongName);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateVerboseName
*
*****************************************************************************/
static BOOLEAN bUpdateVerboseName (
    PRESET_BAND_OBJECT hBand,
    char const * pacVerboseName
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Modify Preset Band Object Field
        STRING.bModifyCStr(psObj->hVerboseName, pacVerboseName);

        // Modify This Filed in the Tag
        eReturnCode = eSetVerboseNameTag(psObj->hTag, psObj->hVerboseName);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateDescription
*
*****************************************************************************/
static BOOLEAN bUpdateDescription (
    PRESET_BAND_OBJECT hBand,
    char const * pacDescription
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Modify Preset Band Object Field
        STRING.bModifyCStr(psObj->hDescription, pacDescription);

        // Modify This Filed in the Tag
        eReturnCode = eSetDescriptionTag(psObj->hTag, psObj->hDescription);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   PRESET_BAND_bUpdateBandCapacity
*
*****************************************************************************/
BOOLEAN PRESET_BAND_bUpdateCapacity (
    PRESET_BAND_OBJECT hBand,
    size_t tCapacity
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (hBand == PRESET_BAND_INVALID_OBJECT)
    {
        return FALSE;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        PRESET_BAND_OBJECT_STRUCT *psObj =
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Set new capacity
        psObj->tPresetCapacity = tCapacity;

        // Save new capacity
        eReturnCode = eSetBandCapacity(psObj->hTag, tCapacity);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = TRUE;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   PRESET_BAND_bUpdate
*
*****************************************************************************/
BOOLEAN PRESET_BAND_bUpdate (
    PRESET_BAND_OBJECT hBand,
    FAVORITES_BAND_ID tId,
    FAVORITES_BAND_ORDER tOrder,
    FAVORITES_BAND_SEQUENCE tSequence,
    FAVORITES_BAND_PURPOSE_MASK tPurpose,
    char const * pacTitleShort,
    char const * pacTitleLong,
    char const * pacTitleVerbose,
    char const * pacDescription,
    SERVICE_ID const * ptArrangement,
    size_t tSIDCount
        )
{
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
    BOOLEAN bSuccess = FALSE, bLocked;
    FAVORITES_BAND_CAPACITY tCapacity;
    PRESET_BAND_MARKER_STRUCT sMarkerStruct;

    if ((ptArrangement == NULL) ||
        (pacTitleShort == NULL) ||
        (pacTitleLong == NULL) ||
        (pacTitleVerbose == NULL) ||
        (pacDescription == NULL))
    {
        // These pointers should be not null
        return FALSE;
    }

    // Lock BAND object to get its parent (PRESETS)
    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // Get PRESETS handle
        hPresets = (PRESETS_OBJECT)SMSO_hParent((SMS_OBJECT)hBand);
        SMSO_vUnlock((SMS_OBJECT)hBand);
    }

    if (hPresets == PRESETS_INVALID_OBJECT)
    {
        return FALSE;
    }

    // per protocol, FF band capacity is defined as UN8
    tCapacity = (FAVORITES_BAND_CAPACITY)BAND.tCapacity(hBand);

    do
    {
        bSuccess = bUpdatePresets(
            hBand, ptArrangement, tSIDCount, tCapacity);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = bUpdateOrder(hBand, tOrder);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = PRESET_BAND_bUpdateSequence(hBand, tSequence);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = bUpdatePurpose(hBand, tPurpose);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = bUpdateShortName(hBand, pacTitleShort);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = bUpdateLongName(hBand, pacTitleLong);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = bUpdateVerboseName(hBand, pacTitleVerbose);
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = bUpdateDescription(hBand, pacDescription);
        if (bSuccess == FALSE)
        {
            break;
        }

    } while (FALSE);

    if (bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            PRESET_BAND_OBJECT_NAME":FF: Failed to update Bank ID: %d", tId);
    }

    sMarkerStruct.tBankId = tId;
    sMarkerStruct.bRemoved = FALSE;

    bSuccess = PRESETS_bSetSelectedFavoritesBankRemoved(
        hPresets,
        sMarkerStruct
            );

    // Sort Bands cuz the FF order may be changed due to update
    bSuccess &= PRESETS_bSortBands(hPresets);

    return bSuccess;
}

/*****************************************************************************
*
*   PRESET_BAND_eSetPresets
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM PRESET_BAND_eSetPresets(
    PRESET_BAND_OBJECT hBand,
    SERVICE_ID const *ptArrangement,
    size_t tSIDCount
        )
{
    BOOLEAN bLocked = FALSE;
    size_t tIndex = 0, tChannels;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    PRESET_BAND_OBJECT_STRUCT *psObj;

    if (ptArrangement == NULL)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)hBand, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // de-reference object
    psObj = (PRESET_BAND_OBJECT_STRUCT *)hBand;

    do
    {
        // calculate the minimum value
        tChannels =
            ((psObj->tPresetCapacity > tSIDCount) ? 
                tSIDCount : psObj->tPresetCapacity);

        eReturnCode = PRESETS_eSetPreset(
            hBand, 
            psObj->hEntry, 
            tIndex, 
            PRESET_UPDATE_TYPE_CHAN, 
            ptArrangement[tIndex],
            NULL, 
            FALSE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESET_BAND_OBJECT_NAME
                ":FF: Failed to set Service ID: %d to Band: %d, "
                "at Index: %d, %s (#%d)",
                ptArrangement[tIndex],
                psObj->tId, 
                tIndex, 
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), 
                eReturnCode);
            break;
        }

        eReturnCode = eSavePresetToConfigFile(
            psObj, 
            tIndex, 
            PRESET_UPDATE_TYPE_CHAN, 
            ptArrangement[tIndex],
            NULL, 
            FALSE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESET_BAND_OBJECT_NAME
                ":FF: Failed to save Service ID: %d to SMS config: "
                "Band: %d, at Index: %d, %s (#%d)",
                ptArrangement[tIndex],
                psObj->tId, 
                tIndex, 
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), 
                eReturnCode);
            break;
        }

        tIndex++;

    } while (tIndex < tChannels);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Commit all the stuff now
        eReturnCode = CM_eCommitChangesToFile();
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Considered as a major fail
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESET_BAND_OBJECT_NAME": Failed to commit changes "
                "to SMS config: %s (#%d)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), 
                eReturnCode );
        }
    }
    
    // Relinquish ownership
    SMSO_vUnlock((SMS_OBJECT)hBand);

    return eReturnCode;
}

/*****************************************************************************
*
*   PRESET_BAND_bRemoveBandTagIterator
*
*****************************************************************************/
BOOLEAN PRESET_BAND_bRemoveBandTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    BOOLEAN bRemoved, *pbUpdate = (BOOLEAN *)pvArg;
    
    if ((hTag == TAG_INVALID_OBJECT) ||
        (pbUpdate == NULL))
    {
        return FALSE;
    }

    bRemoved = bGetRemoved(hTag);
    if (bRemoved == TRUE)
    {
        SMSAPI_RETURN_CODE_ENUM eReturn;

        if (*pbUpdate == FALSE)
        {
            // The current band was removed from OTA and now we are
            // going to remove it from the config
            *pbUpdate = bGetCurrent(hTag);
        }
        
        eReturn = TAG_eRemove(hTag, TRUE);
        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            FAVORITES_BAND_ID tId;

            tId = tGetId(hTag);

            printf(PRESET_BAND_OBJECT_NAME
                ":FF: Marked Band: %d has been removed from the SMS cfg\n",
                tId);
        }
    }

    return TRUE;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreateBandLocal
*
*   This function creates a Band object, either from information passed into
*   the .hCreate API (hBandTag == TAG_INVALID_OBJECT) or from a tag found in
*   the cfg file (hBandTag != TAG_INVALID_OBJECT).
*
*****************************************************************************/
static PRESET_BAND_OBJECT hCreateBandLocal(
    PRESETS_OBJECT hPresets,
    size_t tCapacity,
    PRESET_BAND_PRIVATE_ATTRS_STRUCT *psAttrs,
    TAG_OBJECT hBandTag
        )
{
    PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;
    PRESET_BAND_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bValid;
    TAG_OBJECT hPresetsTag;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Ensure we've been given a proper presets object
    bValid = SMSO_bValid((SMS_OBJECT)hPresets);
    if (bValid == FALSE)
    {
        return PRESET_BAND_INVALID_OBJECT;
    }

    do
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        const char *pacName;

        // Verify inputs
        if ((tCapacity == 0) || (psAttrs == NULL) ||
            (psAttrs != NULL && psAttrs->hName == STRING_INVALID_OBJECT))
        {
            break;
        }

        pacName = STRING.pacCStr(psAttrs->hName);

        // Create the name for the band
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 PRESET_BAND_OBJECT_NAME":%s",
                 pacName);

        // Create an instance of this object
        psObj = (PRESET_BAND_OBJECT_STRUCT *)SMSO_hCreate(
                &acName[0],
                sizeof(PRESET_BAND_OBJECT_STRUCT),
                (SMS_OBJECT)hPresets,
                FALSE);

        if(psObj == NULL)
        {
            break;
        }

        // Store the band Type
        psObj->eType = psAttrs->eType;

        // Store the band capacity
        psObj->tPresetCapacity = tCapacity;

        // Initialize band names w/ null
        psObj->hName = STRING_INVALID_OBJECT;
        psObj->hLongName = STRING_INVALID_OBJECT;
        psObj->hVerboseName = STRING_INVALID_OBJECT;
        psObj->hDescription = STRING_INVALID_OBJECT;

        if (psObj->eType == PRESET_BAND_TYPE_FEATURED)
        {
            psObj->tOrder = psAttrs->sFeatured.tOrder;
            psObj->tId = psAttrs->sFeatured.tId;
            psObj->tSequence = psAttrs->sFeatured.tSequence;
            psObj->tPurpose = psAttrs->sFeatured.tPurpose;
            psObj->bRemoved = psAttrs->sFeatured.bRemoved;
        }

        psObj->hName = STRING.hDuplicate(psAttrs->hName);
        if (psObj->hName == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PRESET_BAND_OBJECT_NAME": Failed to create Band's name!");
            break;
        }

        // Try to create long name
        psObj->hLongName = STRING.hDuplicate(psAttrs->hLongName);
        if (psObj->hLongName == STRING_INVALID_OBJECT)
        {
            psObj->hLongName = STRING.hCreate("", 0);
            if (psObj->hLongName == STRING_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME
                    ": Failed to create Band's Long name");
                break;
            }
        }

        // Try to create verbose name
        psObj->hVerboseName = STRING.hDuplicate(psAttrs->hVerboseName);
        if (psObj->hVerboseName == STRING_INVALID_OBJECT)
        {
            psObj->hVerboseName = STRING.hCreate("", 0);

            if (psObj->hVerboseName == STRING_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME
                    ": Failed to create Band's Verbose name");
                break;
            }
        }

        // Try to create description name
        psObj->hDescription = STRING.hDuplicate(psAttrs->hDescription);
        if (psObj->hDescription == STRING_INVALID_OBJECT)
        {
            psObj->hDescription = STRING.hCreate("", 0);

            if (psObj->hDescription == STRING_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME
                    ": Failed to create Band's Description");
                break;
            }
        }

        // Add this band object to our parent PRESET object
        // and get our entry handle
        psObj->hEntry = PRESETS_hAddBand(
                hPresets,
                (PRESET_BAND_OBJECT)psObj,
                psObj->tPresetCapacity);

        if (psObj->hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            break;
        }

        // get the preset service tag
        // (this is the parent tag of the preset_band tag)
        hPresetsTag = PRESETS_hGetTag(hPresets);

        // get the preset band tag
        if (hBandTag == TAG_INVALID_OBJECT)
        {
            eReturnCode = TAG_eAdd(
                    PRESET_BAND_OBJECT_NAME,
                    hPresetsTag,
                    &hBandTag,
                    NULL);

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            // set information in the config file
            eReturnCode = eSetNameTag(hBandTag, psObj->hName);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            eReturnCode = eSetLongNameTag(hBandTag, psObj->hLongName);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            eReturnCode = eSetVerboseNameTag(hBandTag, psObj->hVerboseName);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            eReturnCode = eSetDescriptionTag(hBandTag, psObj->hDescription);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            eReturnCode = eSetLoaded(hBandTag, (BOOLEAN)
                ((psObj->eType == PRESET_BAND_TYPE_USER) ? FALSE : TRUE));

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            eReturnCode = eSetBandCapacity(hBandTag, tCapacity);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            if (psObj->eType == PRESET_BAND_TYPE_FEATURED)
            {
                eReturnCode = eSetRemoved(hBandTag, psAttrs->sFeatured.bRemoved);
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }

                eReturnCode = eSetOrder(hBandTag, psAttrs->sFeatured.tOrder);
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }

                eReturnCode = eSetId(hBandTag, psAttrs->sFeatured.tId);
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }

                eReturnCode = eSetSequence(hBandTag, psAttrs->sFeatured.tSequence);
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }

                eReturnCode = eSetPurpose(hBandTag, psAttrs->sFeatured.tPurpose);
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }
            }

            // write the changes to the cm file
            CM_eCommitChangesToFile();

        }

        // we got the band tag, so save this information for future use
        psObj->hTag = hBandTag;

        hBand = (PRESET_BAND_OBJECT)psObj;

        // Perform notification only for user favorites
        // For Featured Favorites notification is invoked only
        // when the bank is completely created and populated with presets
        if (psObj->eType == PRESET_BAND_TYPE_USER)
        {
            PRESETS_bUpdate(hPresets, hBand, PRESETS_OBJECT_EVENT_BAND_ADDED);
        }

        return hBand;

    } while (FALSE);

    vDestroyBandLocal(hPresets, (PRESET_BAND_OBJECT)psObj, FALSE);
    return PRESET_BAND_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eSetPresetLocal
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetPresetLocal(
    PRESET_BAND_OBJECT_STRUCT *psObj,
    size_t tPresetIndex,
    PRESET_UPDATE_TYPE_ENUM eUpdateType,
    SERVICE_ID tServiceId,
    const char *pacName
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    // Verify the provided index
    if (tPresetIndex >= psObj->tPresetCapacity)
    {
        eReturnCode =
            SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
    }
    else
    {
        BOOLEAN bUpdateSmartFavorites;

        bUpdateSmartFavorites =
            ((eUpdateType == PRESET_UPDATE_TYPE_CHAN) ? TRUE : FALSE);

        // Get the presets service to update
        // this preset's name
        eReturnCode = PRESETS_eSetPreset (
            (PRESET_BAND_OBJECT)psObj,
            psObj->hEntry,
            tPresetIndex,
            eUpdateType,
            tServiceId,
            pacName,
            bUpdateSmartFavorites);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eSavePresetToConfigFile
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSavePresetToConfigFile(
    PRESET_BAND_OBJECT_STRUCT *psObj,
    size_t tPresetIndex,
    PRESET_UPDATE_TYPE_ENUM eUpdateType,
    SERVICE_ID tServiceId,
    const char *pacName,
    BOOLEAN bCommit
        )
{
    // need to save to the config file....
    TAG_OBJECT hPresetTag;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    hPresetTag = hGetPresetTagByIndex(psObj, tPresetIndex);
    if (hPresetTag != TAG_INVALID_OBJECT)
    {
        if (eUpdateType == PRESET_UPDATE_TYPE_CHAN)
        {
            TAG_OBJECT hServiceIdTag;

            // get the service id tag
            eReturnCode = TAG_eGet(
                              PRESET_SERVICE_ID,
                              hPresetTag,
                              &hServiceIdTag,
                              NULL,
                              TRUE); // create if it isn't found

            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // got the tag, now set its value
                N32 n32Value = (N32)tServiceId;

                eReturnCode = TAG_eSetTagValue(
                                  hServiceIdTag,
                                  TAG_TYPE_INTEGER,
                                  (void *)&n32Value,
                                  sizeof(N32),
                                  bCommit);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME
                    ":FF: Failed to get Service ID: %d Tag: %s (#%d)",
                    tServiceId, 
                    SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), 
                    eReturnCode);
            }
        }
        else if (eUpdateType == PRESET_UPDATE_TYPE_NAME)
        {
            TAG_OBJECT hNameTag;

            eReturnCode = TAG_eGet(
                PRESET_NAME, 
                hPresetTag, 
                &hNameTag, 
                NULL, 
                TRUE); // create if it isn't found

            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                STRING_OBJECT hName;
                hName = STRING.hCreate(pacName, 0);

                // set the tag value
                eReturnCode = TAG_eSetTagValue(
                                  hNameTag,
                                  TAG_TYPE_STRING,
                                  &hName,
                                  sizeof(STRING_OBJECT),
                                  bCommit);

                STRING_vDestroy(hName);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME
                    ":FF: Failed to get preset Name: %s Tag: %s (#%d)",
                    pacName, 
                    SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), 
                    eReturnCode);
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hGetBandName
*
*   This will retrieve the preset band's name from the tag
*
*****************************************************************************/
static STRING_OBJECT hGetBandName(
    TAG_OBJECT hBandTag
        )
{
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT, *phString;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hNameTag;

    // get the band name tag
    eResult = TAG_eGet(
        PRESET_NAME,
        hBandTag,
        &hNameTag,
        NULL,
        FALSE // don't create if it isn't found
            );

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (hNameTag != TAG_INVALID_OBJECT)
        {
            size_t tSize;
            tSize = sizeof(STRING_OBJECT);

            // get the tag value
            phString = &hValueString;
            eResult = TAG_eGetTagValue(
                          hNameTag,
                          TAG_TYPE_STRING,
                          (void **)&phString,
                          &tSize
                              );

            if ((eResult != SMSAPI_RETURN_CODE_SUCCESS) &&
                (eResult != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME": Failed to get Shot Name value from TAG");
            }
        }
    }

    return hValueString;
}

/*****************************************************************************
*
*   tGetCapacity
*
*****************************************************************************/
static size_t tGetCapacity(
    TAG_OBJECT hBandTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hCapacityTag = TAG_INVALID_OBJECT;
    size_t tCapacity = 0;

    // get the band capacity tag
    eResult = TAG_eGet(
        PRESET_BAND_CAPACITY,
        hBandTag,
        &hCapacityTag,
        NULL,
        FALSE // don't create if it isn't found
            );
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        eResult = TAG_eGetTagValue(
                      hCapacityTag,
                      TAG_TYPE_INTEGER,
                      (void **)&pn32TagValue,
                      &tSize);

        if ((eResult == SMSAPI_RETURN_CODE_SUCCESS) && (n32TagValue >= 0))
        {
            tCapacity = (size_t)n32TagValue;
        }
    }

    return tCapacity;
}

/*****************************************************************************
*
*   eSetBandCapacity
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetBandCapacity(
    TAG_OBJECT hBandTag,
    size_t tCapacity
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hCapacityTag = TAG_INVALID_OBJECT;

    // get the band capacity tag
    eResult = TAG_eGet(
        PRESET_BAND_CAPACITY,
        hBandTag,
        &hCapacityTag,
        NULL,
        TRUE // create if it isn't found
            );
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue;
        n32TagValue = (N32)tCapacity;

        eResult = TAG_eSetTagValue(
                      hCapacityTag,
                      TAG_TYPE_INTEGER,
                      (void *)&n32TagValue,
                      sizeof(N32),
                      FALSE);
    }

    return eResult;
}

/*****************************************************************************
*
*   bGetLoaded
*
*****************************************************************************/
static BOOLEAN bGetLoaded(
    TAG_OBJECT hParentTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hLoadedTag = TAG_INVALID_OBJECT;
    BOOLEAN bLoaded = FALSE;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_BAND_LOADED,
              hParentTag,
              &hLoadedTag,
              NULL,
              FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        eResult = TAG_eGetTagValue(
                  hLoadedTag,
                  TAG_TYPE_INTEGER,
                  (void **)&pn32TagValue,
                  &tSize);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bLoaded = ( n32TagValue == 0 )? FALSE : TRUE;
        }
    }

    return bLoaded;
}

/*****************************************************************************
*
*   tGetRemoved
*
*****************************************************************************/
static BOOLEAN bGetRemoved(
    TAG_OBJECT hParentTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hRemovedTag = TAG_INVALID_OBJECT;
    BOOLEAN bRemoved = FALSE;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_BAND_REMOVED,
              hParentTag,
              &hRemovedTag,
              NULL,
              FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        eResult = TAG_eGetTagValue(
                  hRemovedTag,
                  TAG_TYPE_INTEGER,
                  (void **)&pn32TagValue,
                  &tSize);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bRemoved = (n32TagValue == 0) ? FALSE : TRUE;
        }
    }

    return bRemoved;
}

/*****************************************************************************
*
*   eSetRemoved
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetRemoved(
    TAG_OBJECT hParentTag,
    BOOLEAN bRemoved
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hRemovedTag = TAG_INVALID_OBJECT;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_BAND_REMOVED,
              hParentTag,
              &hRemovedTag,
              NULL,
              TRUE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue;
        n32TagValue = (N32)bRemoved;

        eResult = TAG_eSetTagValue(
                  hRemovedTag,
                  TAG_TYPE_INTEGER,
                  (void *)&n32TagValue,
                  sizeof(N32),
                  FALSE);
    }

    return eResult;
}

/*****************************************************************************
*
*   eSetLoaded
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetLoaded(
    TAG_OBJECT hParentTag,
    BOOLEAN bLoaded
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hLoadedTag = TAG_INVALID_OBJECT;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_BAND_LOADED,
              hParentTag,
              &hLoadedTag,
              NULL,
              TRUE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue;
        n32TagValue = (N32)bLoaded;

        eResult = TAG_eSetTagValue(
                  hLoadedTag,
                  TAG_TYPE_INTEGER,
                  (void *)&n32TagValue,
                  sizeof(N32),
                  FALSE);
    }

    return eResult;
}

/*****************************************************************************
*
*   eSetOrder
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetOrder(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_ORDER tOrder
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hOrderTag = TAG_INVALID_OBJECT;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_ORDER,
              hParentTag,
              &hOrderTag,
              NULL,
              TRUE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue;
        n32TagValue = (N32)tOrder;

        eResult = TAG_eSetTagValue(
                  hOrderTag,
                  TAG_TYPE_INTEGER,
                  (void *)&n32TagValue,
                  sizeof(N32),
                  FALSE);
    }

    return eResult;
}

/*****************************************************************************
*
*   tGetOrder
*
*****************************************************************************/
static FAVORITES_BAND_ORDER tGetOrder(
    TAG_OBJECT hParentTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hOrderTag = TAG_INVALID_OBJECT;
    FAVORITES_BAND_ORDER tOrder = 0;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_ORDER,
              hParentTag,
              &hOrderTag,
              NULL,
              FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        eResult = TAG_eGetTagValue(
                  hOrderTag,
                  TAG_TYPE_INTEGER,
                  (void **)&pn32TagValue,
                  &tSize);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            tOrder = (FAVORITES_BAND_ORDER)n32TagValue;
        }
    }

    return tOrder;
}

/*****************************************************************************
*
*   eSetId
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetId(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_ID tId
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hIdTag = TAG_INVALID_OBJECT;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_ID,
              hParentTag,
              &hIdTag,
              NULL,
              TRUE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue;
        n32TagValue = (N32)tId;

        eResult = TAG_eSetTagValue(
                  hIdTag,
                  TAG_TYPE_INTEGER,
                  (void *)&n32TagValue,
                  sizeof(N32),
                  FALSE);
    }

    return eResult;
}

/*****************************************************************************
*
*   tGetId
*
*****************************************************************************/
static FAVORITES_BAND_ID tGetId(
    TAG_OBJECT hParentTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hIdTag = TAG_INVALID_OBJECT;
    FAVORITES_BAND_ID tId = 0;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_ID,
              hParentTag,
              &hIdTag,
              NULL,
              FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        eResult = TAG_eGetTagValue(
                  hIdTag,
                  TAG_TYPE_INTEGER,
                  (void **)&pn32TagValue,
                  &tSize);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            tId = (FAVORITES_BAND_ID)n32TagValue;
        }
    }

    return tId;
}

/*****************************************************************************
*
*   eSetSequence
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetSequence(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_SEQUENCE tSequence
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hSequenceTag = TAG_INVALID_OBJECT;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_SEQ,
              hParentTag,
              &hSequenceTag,
              NULL,
              TRUE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue;
        n32TagValue = (N32)tSequence;

        eResult = TAG_eSetTagValue(
                  hSequenceTag,
                  TAG_TYPE_INTEGER,
                  (void *)&n32TagValue,
                  sizeof(N32),
                  FALSE);
    }

    return eResult;
}

/*****************************************************************************
*
*   tGetSequence
*
*****************************************************************************/
static FAVORITES_BAND_SEQUENCE tGetSequence(
    TAG_OBJECT hParentTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hSequenceTag = TAG_INVALID_OBJECT;
    FAVORITES_BAND_SEQUENCE tSequence = FAVORITES_BAND_SEQUENCE_MIN;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_SEQ,
              hParentTag,
              &hSequenceTag,
              NULL,
              FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        eResult = TAG_eGetTagValue(
                  hSequenceTag,
                  TAG_TYPE_INTEGER,
                  (void **)&pn32TagValue,
                  &tSize);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            tSequence = (FAVORITES_BAND_SEQUENCE)n32TagValue;
        }
    }

    return tSequence;
}

/*****************************************************************************
*
*   eSetPurpose
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetPurpose(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_PURPOSE_MASK tPurpose
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hPurposeTag = TAG_INVALID_OBJECT;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_PURPOSE,
              hParentTag,
              &hPurposeTag,
              NULL,
              TRUE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue;
        n32TagValue = (N32)tPurpose;

        eResult = TAG_eSetTagValue(
                  hPurposeTag,
                  TAG_TYPE_INTEGER,
                  (void *)&n32TagValue,
                  sizeof(N32),
                  FALSE);
    }

    return eResult;
}

/*****************************************************************************
*
*   tGetPurpose
*
*****************************************************************************/
static FAVORITES_BAND_PURPOSE_MASK tGetPurpose(
    TAG_OBJECT hParentTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hPurposeTag = TAG_INVALID_OBJECT;
    FAVORITES_BAND_PURPOSE_MASK tPurpose = 0;

    // get the band capacity tag
    eResult = TAG_eGet(
              PRESET_PURPOSE,
              hParentTag,
              &hPurposeTag,
              NULL,
              FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        eResult = TAG_eGetTagValue(
                  hPurposeTag,
                  TAG_TYPE_INTEGER,
                  (void **)&pn32TagValue,
                  &tSize);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            tPurpose = (FAVORITES_BAND_PURPOSE_MASK)n32TagValue;
        }
    }

    return tPurpose;
}

/*****************************************************************************
*
*   eSetLongName
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetLongNameTag (
    TAG_OBJECT hParentTag,
    STRING_OBJECT hString
        )
{
    TAG_OBJECT hLongNameTag = TAG_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    eReturnCode = TAG_eGet(
            PRESET_LONG_NAME,
            hParentTag,
            &hLongNameTag,
            NULL,
            TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the tag value
        eReturnCode = TAG_eSetTagValue(
                      hLongNameTag,
                      TAG_TYPE_STRING,
                      &hString,
                      sizeof(STRING_OBJECT),
                      FALSE); // not committing at this time
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hGetLongName
*
*   This will retrieve the preset band's long name from the tag
*
*****************************************************************************/
static STRING_OBJECT hGetLongName(
    TAG_OBJECT hParentTag
        )
{
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT, *phString;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hLongNameTag;

    // get the band long name tag
    eResult = TAG_eGet(
        PRESET_LONG_NAME,
        hParentTag,
        &hLongNameTag,
        NULL,
        FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (hLongNameTag != TAG_INVALID_OBJECT)
        {
            size_t tSize;
            tSize = sizeof(STRING_OBJECT);

            // get the tag value
            phString = &hValueString;
            eResult = TAG_eGetTagValue(
                      hLongNameTag,
                      TAG_TYPE_STRING,
                      (void **)&phString,
                      &tSize);

            if ((eResult != SMSAPI_RETURN_CODE_SUCCESS) &&
                (eResult != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME": Failed to get Long Name from TAG");
            }
        }
    }

    return hValueString;
}

/*****************************************************************************
*
*   eSetVerboseName
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetVerboseNameTag (
    TAG_OBJECT hParentTag,
    STRING_OBJECT hVerboseNameString
        )
{
    TAG_OBJECT hVerboseNameTag = TAG_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    eReturnCode = TAG_eGet(
            PRESET_VERBOSE_NAME,
            hParentTag,
            &hVerboseNameTag,
            NULL,
            TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the tag value
        eReturnCode = TAG_eSetTagValue(
                      hVerboseNameTag,
                      TAG_TYPE_STRING,
                      &hVerboseNameString,
                      sizeof(STRING_OBJECT),
                      FALSE); // not committing at this time
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hGetVerboseName
*
*   This will retrieve the preset band's verbose name from the tag
*
*****************************************************************************/
static STRING_OBJECT hGetVerboseName(
    TAG_OBJECT hParentTag
        )
{
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT, *phString;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hVerboseNameTag;

    // get the band verbose name tag
    eResult = TAG_eGet(
        PRESET_VERBOSE_NAME,
        hParentTag,
        &hVerboseNameTag,
        NULL,
        FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (hVerboseNameTag != TAG_INVALID_OBJECT)
        {
            size_t tSize;
            tSize = sizeof(STRING_OBJECT);

            // get the tag value
            phString = &hValueString;
            eResult = TAG_eGetTagValue(
                      hVerboseNameTag,
                      TAG_TYPE_STRING,
                      (void **)&phString,
                      &tSize);

            if ((eResult != SMSAPI_RETURN_CODE_SUCCESS) &&
                (eResult != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME": Failed to get Verbose Name from TAG");
            }
        }
    }

    return hValueString;
}

/*****************************************************************************
*
*   eSetDescription
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetDescriptionTag (
    TAG_OBJECT hParentTag,
    STRING_OBJECT hString
        )
{
    TAG_OBJECT hDescriptionTag = TAG_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    eReturnCode = TAG_eGet(
            PRESET_DESCRIPTION,
            hParentTag,
            &hDescriptionTag,
            NULL,
            TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the tag value
        eReturnCode = TAG_eSetTagValue(
                      hDescriptionTag,
                      TAG_TYPE_STRING,
                      &hString,
                      sizeof(STRING_OBJECT),
                      FALSE); // not committing at this time
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hGetDescription
*
*   This will retrieve the preset band's description name from the tag
*
*****************************************************************************/
static STRING_OBJECT hGetDescription(
    TAG_OBJECT hParentTag
        )
{
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT, *phString;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hDescriptionTag;

    // get the band description tag
    eResult = TAG_eGet(
        PRESET_DESCRIPTION,
        hParentTag,
        &hDescriptionTag,
        NULL,
        FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (hDescriptionTag != TAG_INVALID_OBJECT)
        {
            size_t tSize;
            tSize = sizeof(STRING_OBJECT);

            // get the tag value
            phString = &hValueString;
            eResult = TAG_eGetTagValue(
                      hDescriptionTag,
                      TAG_TYPE_STRING,
                      (void **)&phString,
                      &tSize);

            if ((eResult != SMSAPI_RETURN_CODE_SUCCESS) &&
                (eResult != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    PRESET_BAND_OBJECT_NAME": Failed to get Description from TAG");
            }
        }
    }

    return hValueString;
}

/*****************************************************************************
*
*   pacGetPresetName
*
*   This will retrieve the preset's name from the tag and then create a
*   c-string containing that name
*
*****************************************************************************/
static char *pacGetPresetName(
    TAG_OBJECT hPresetTag
        )
{
    char *pacPresetName = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    TAG_OBJECT hNameTag;

    // get the preset name tag
    eReturnCode = TAG_eGet(PRESET_NAME, hPresetTag, &hNameTag, NULL, FALSE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // we got the tag, now get its value as a STRING

        STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;
        size_t tSize;

        phString = &hString;
        tSize = sizeof(STRING_OBJECT);
        TAG_eGetTagValue(
            hNameTag,
            TAG_TYPE_STRING,
            (void**)&phString,
            &tSize);

        if (hString != STRING_INVALID_OBJECT)
        {
            // got a value, now allocate a c-string and copy in the name

            tSize = STRING.tSize(hString) +1; // add 1 for NULL

            // allocate memory for the buffer
            pacPresetName =
                (char *) SMSO_hCreate(
                    PRESET_BAND_OBJECT_NAME":CSTR",
                    tSize,
                    SMS_INVALID_OBJECT, FALSE // No lock necessary
                    );
            if (pacPresetName != NULL)
            {
                // copy to our c-string
                STRING.tCopyToCStr(hString, pacPresetName, tSize);
            }

            // done with this
            STRING.vDestroy(hString);
        }
    }
    return pacPresetName;
}

/*****************************************************************************
*
*   bGetPresetIndex
*
*   This will retrieve the preset's index from a preset tag
*
*****************************************************************************/
static BOOLEAN bGetPresetIndex(
    TAG_OBJECT hPresetTag,
    size_t *tIndex
        )
{
    BOOLEAN bOk = FALSE;
    TAG_OBJECT hIndexTag = TAG_INVALID_OBJECT;

    if (tIndex != NULL)
    {
        SMSAPI_RETURN_CODE_ENUM eResult;

        // get the preset index tag
        eResult = TAG_eGet(
            PRESET_INDEX,
            hPresetTag,
            &hIndexTag,
            NULL,
            FALSE // don't create if it isn't found
                );
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // got the tag, now get its value as an integer
            N32 n32TagValue = -1, *pn32TagValue;
            size_t tSize;

            pn32TagValue = &n32TagValue;
            tSize = sizeof(N32);
            eResult = TAG_eGetTagValue(
                          hIndexTag,
                          TAG_TYPE_INTEGER,
                          (void **)&pn32TagValue,
                          &tSize);

            // did we get a value, and if so is it positive?
            // (someone could have put a negative number into the config file
            // and that wouldn't make sense)
            if ((eResult == SMSAPI_RETURN_CODE_SUCCESS) && (n32TagValue >= 0))
            {
                // we successfully got a non-negative number
                *tIndex = (size_t)n32TagValue;
                // indicate that we got a value
                bOk = TRUE;
            }
        }
    }
    return bOk;
}

/*****************************************************************************
*
*   tGetPresetServiceId
*
*   This will retrieve the preset's service id from a preset tag
*
*****************************************************************************/
static SERVICE_ID tGetPresetServiceId(
    TAG_OBJECT hPresetTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hServiceIdTag = TAG_INVALID_OBJECT;
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;

    // get the service id tag
    eResult = TAG_eGet(
                  PRESET_SERVICE_ID,
                  hPresetTag,
                  &hServiceIdTag,
                  NULL,
                  FALSE // don't create if it isn't found
                      );
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value as an integer
        N32 n32TagValue = 0, *pn32TagValue;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);
        eResult = TAG_eGetTagValue(
                      hServiceIdTag,
                      TAG_TYPE_INTEGER,
                      (void **)&pn32TagValue,
                      &tSize);

        // did we get a value, and if so is it positive?
        // (someone could have put a negative number into the config file
        // and that wouldn't make sense)
        if ((eResult == SMSAPI_RETURN_CODE_SUCCESS) && (n32TagValue >= 0))
        {
            tServiceId = (SERVICE_ID)n32TagValue;
        }
    }

    return tServiceId;
}

/*****************************************************************************
*
*   eSetNameTag
*
*   This will get a name tag that is a child of the provided tag.
*   If it doesn't exist it'll be created.
*   The name tag will have it's value set to the provided name
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetNameTag(
    TAG_OBJECT hParentTag,
    STRING_OBJECT hString
        )
{
    TAG_OBJECT hNameTag = TAG_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    eReturnCode = TAG_eGet(PRESET_NAME, hParentTag, &hNameTag, NULL, TRUE);

    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the tag value
        eReturnCode = TAG_eSetTagValue(
                          hNameTag,
                          TAG_TYPE_STRING,
                          &hString,
                          sizeof(STRING_OBJECT),
                          FALSE); // not committing at this time
    }
    else if (eReturnCode == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
    {
        // must mean there is no file system
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}
/*****************************************************************************
*
*   bGetCurrent
*
*   This will retrieve the value of the current tag of the given band tag.
*   The return indicate if this is the current band (TRUE) or not (FALSE)
*
*****************************************************************************/
static BOOLEAN bGetCurrent(
    TAG_OBJECT hBandTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hCurrentTag;

    BOOLEAN bCurrent = FALSE;

    eResult = TAG_eGet(
                PRESET_BAND_CURRENT,
                hBandTag,
                &hCurrentTag,
                NULL,
                FALSE);

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        size_t tSize;
        STRING_OBJECT hValueString = STRING_INVALID_OBJECT,
                      *phString = &hValueString;

        tSize = sizeof(STRING_OBJECT);

        eResult = TAG_eGetTagValue(
                      hCurrentTag,
                      TAG_TYPE_STRING,
                      (void **)&phString,
                      &tSize);

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // we got the value.
            CM_eStringToBoolean(hValueString, &bCurrent);

            // done with this string
            STRING.vDestroy(hValueString);
        }
    }

    return bCurrent;
}

/*****************************************************************************
*
*   bBandTagPresetIterator
*
*****************************************************************************/
static BOOLEAN bBandTagPresetIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    PRESET_BAND_TAG_ITERATOR_STRUCT *psStruct =
        (PRESET_BAND_TAG_ITERATOR_STRUCT *)pvArg;
    STRING_OBJECT hTagName;

    // get this tag's tag name
    hTagName = TAG_hTagName(hTag);
    // see if the tag name is one that we're looking for
    if (STRING.n16CompareCStr(PRESET_TAG_NAME, 0, hTagName) == 0)
    {
        // we found a preset tag
        BOOLEAN bOk;
        size_t tPresetIndex;

        // get the preset tag's index
        bOk = bGetPresetIndex(hTag, &tPresetIndex);
        if (bOk == TRUE)
        {
            // we successfully got the preset index.

            PRESETS_RESTORE_STRUCT *psRestoreStruct;

            // create a struct that we can store the preset info so that it can
            // be restored after we are done iterating the tags.
            // We don't want to do the restoration during the iteration because
            // it requires mapping service id to channel id. The mapping
            // requires a decoder lock. But if the decoder is wating for the
            // config manager lock, we'll get into a deadlock. (Bug 4440)
            psRestoreStruct = (PRESETS_RESTORE_STRUCT *)
                SMSO_hCreate(
                    PRESET_BAND_OBJECT_NAME":PRestoreStruct",
                    sizeof(PRESETS_RESTORE_STRUCT),
                    SMS_INVALID_OBJECT,
                    FALSE
                       );

            if (psRestoreStruct != NULL)
            {
                OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

                // set this preset's band
                psRestoreStruct->hBand = psStruct->hBand;

                // set this preset's index
                psRestoreStruct->tPresetIndex = tPresetIndex;

                // get this preset's service id, from the cm
                psRestoreStruct->tServiceId = tGetPresetServiceId(hTag);

                // get this preset's name from the cm
                psRestoreStruct->pacPresetName = pacGetPresetName(hTag);

                // initialize a list entry handle
                psRestoreStruct->hEntry = NULL;

                eOsalReturnCode = OSAL.eLinkedListAdd(psStruct->hLL,
                                            &psRestoreStruct->hEntry,
                                            (void *)psRestoreStruct
                                                );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    printf("BANK: failed to add entry %s\n",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                }

                psStruct->bPresets = TRUE;
            }
        }
    }
    else
    {
        // ignore this tag
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   hGetPresetTagByIndex
*
*****************************************************************************/
static TAG_OBJECT hGetPresetTagByIndex(
    PRESET_BAND_OBJECT_STRUCT *psObj,
    size_t tPresetIndex
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    PRESET_SEARCH_FOR_TAG_ITERATOR_STRUCT sIterator;
    TAG_OBJECT hIndexTag = TAG_INVALID_OBJECT, hPresetTag = TAG_INVALID_OBJECT;

    sIterator.hPresetTag = TAG_INVALID_OBJECT;
    sIterator.tIndex = tPresetIndex;

    eReturnCode = TAG_eIterateChildren(
        psObj->hTag,
        bSearchForPresetTagIterator,
        &sIterator);
    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        hPresetTag = sIterator.hPresetTag;
        // see if we found the preset tag we were looking for
        if ( hPresetTag == TAG_INVALID_OBJECT)
        {
            // we couldn't find the tag we were looking for, so we'll create it
            // don't bother with the return, if it fails hPreseTag will remain
            // invalid and we'll catch it when we try to add the index id tag.
            TAG_eAdd(
                PRESET_TAG_NAME,
                psObj->hTag,
                &hPresetTag,
                NULL);

            // now add the the index id tag
            eReturnCode = TAG_eAdd(
                              PRESET_INDEX,
                              hPresetTag,
                              &hIndexTag,
                              NULL);
            if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // got the tag, now set its value
                N32 n32Value = (N32)tPresetIndex;

                eReturnCode = TAG_eSetTagValue(
                              hIndexTag,
                              TAG_TYPE_INTEGER,
                              (void *)&n32Value,
                              sizeof(N32),
                              FALSE); // don't immediately commit
            }

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // something went wrong, so let's remove these tags if
                // they were created
                TAG_eRemove(hPresetTag, FALSE);
                TAG_eRemove(hIndexTag, FALSE);
                hPresetTag = TAG_INVALID_OBJECT;
            }
        }
    }

    return hPresetTag;
}

/*****************************************************************************
*
*   bSearchForPresetTagIterator
*
*****************************************************************************/
static BOOLEAN bSearchForPresetTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    PRESET_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIterator =
        (PRESET_SEARCH_FOR_TAG_ITERATOR_STRUCT *)pvArg;

    STRING_OBJECT hTagName;

    // get the tag name
    hTagName = TAG_hTagName(hTag);
    // see if the tag name is one that we're looking for
    if (STRING.n16CompareCStr(PRESET_TAG_NAME, 0, hTagName) == 0)
    {
        // we found a preset tag
        BOOLEAN bOk;
        size_t tPresetIndex;

        bOk = bGetPresetIndex(hTag, &tPresetIndex);
        if (bOk == TRUE)
        {
            if (psIterator->tIndex == tPresetIndex)
            {
                // found what we're looking for
                psIterator->hPresetTag = hTag;
                return FALSE;
            }
        }
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bPresetBandTagIterator
*
*****************************************************************************/
static BOOLEAN bPresetBandTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    PRESET_BAND_TAG_ITERATOR_STRUCT *psIteratorStruct;
    STRING_OBJECT hTagName;
    PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;
    PRESET_BAND_PRIVATE_ATTRS_STRUCT sAttrs;

    psIteratorStruct = (PRESET_BAND_TAG_ITERATOR_STRUCT *)pvArg;

    // get the tag name
    hTagName = TAG_hTagName(hTag);
    // see if the tag name is one that we're looking for
    if (STRING.n16CompareCStr(PRESET_BAND_OBJECT_NAME, 0, hTagName) == 0)
    {
        do
        {
            BOOLEAN bValue, bCurrent;

            // Clean up
            OSAL.bMemSet(&sAttrs, 0, sizeof(PRESET_BAND_PRIVATE_ATTRS_STRUCT));

            // Read string fields from cfg
            sAttrs.hName = hGetBandName(hTag);
            if (sAttrs.hName == STRING_INVALID_OBJECT)
            {
                // Cannot create band with empty name
                break;
            }

            sAttrs.hLongName = hGetLongName(hTag);
            sAttrs.hVerboseName = hGetVerboseName(hTag);
            sAttrs.hDescription = hGetDescription(hTag);

            // Read band type
            bValue = bGetLoaded(hTag);

            sAttrs.eType = (bValue == TRUE) ?
                            PRESET_BAND_TYPE_FEATURED :
                            PRESET_BAND_TYPE_USER;

            sAttrs.tCapacity = tGetCapacity(hTag);
            sAttrs.sFeatured.tId = tGetId(hTag);
            sAttrs.sFeatured.tOrder = tGetOrder(hTag);
            sAttrs.sFeatured.tPurpose = tGetPurpose(hTag);
            sAttrs.sFeatured.tSequence = tGetSequence(hTag);
            sAttrs.sFeatured.bRemoved = bGetRemoved(hTag);

            // OK. Now we have all bank fields w/o channel presets
            hBand = hCreateBandLocal (
                    psIteratorStruct->hPresets,
                    sAttrs.tCapacity,
                    &sAttrs,
                    hTag);

            if (hBand == PRESET_BAND_INVALID_OBJECT)
            {
                // can't do anything more with this entry
                break;
            }

            psIteratorStruct->hBand = hBand;

            // now we need to get the presets
            TAG_eIterateChildren(
                hTag,
                bBandTagPresetIterator,
                (void *)psIteratorStruct
                    );

            // If this band was current before shutdown,
            // we need to restore this as well.
            bCurrent = bGetCurrent(hTag);
            if (bCurrent == TRUE)
            {
                SMSAPI_RETURN_CODE_ENUM eReturnCode;

                eReturnCode = PRESETS.eSetCurrentBand(hBand);
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        PRESET_BAND_OBJECT_NAME": Failed to set the current Band: %s, %s (#%d)",
                            STRING.pacCStr(BAND.hName(hBand)),
                            SMSAPI_DEBUG_pacReturnCodeText(eReturnCode),
                            eReturnCode);
                }
            }

        } while (FALSE);

        // Free string fields memory
        if (sAttrs.hName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sAttrs.hName);
            sAttrs.hName = STRING_INVALID_OBJECT;
        }

        if (sAttrs.hLongName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sAttrs.hLongName);
            sAttrs.hLongName = STRING_INVALID_OBJECT;
        }

        if (sAttrs.hVerboseName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sAttrs.hVerboseName);
            sAttrs.hVerboseName = STRING_INVALID_OBJECT;
        }

        if (sAttrs.hDescription != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sAttrs.hDescription);
            sAttrs.hDescription = STRING_INVALID_OBJECT;
        }
    } // if (STRING.n16CompareCStr(PRESET_BAND_OBJECT_NAME, 0, hTagName) == 0)

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bSetThePresets
*
*****************************************************************************/
static BOOLEAN bSetThePresets(
    void *pvData,
    void *pvArg
        )
{
    PRESETS_RESTORE_STRUCT *psStruct = (PRESETS_RESTORE_STRUCT *)pvData;
    PRESET_BAND_OBJECT_STRUCT *psObj =
        (PRESET_BAND_OBJECT_STRUCT *)psStruct->hBand;

    // set the preset's channel id
    eSetPresetLocal(
        psObj,
        psStruct->tPresetIndex,
        PRESET_UPDATE_TYPE_CHAN,
        psStruct->tServiceId,
        NULL);

    if (psStruct->pacPresetName != NULL)
    {
        // we got a non-NULL name, so set the preset's name
        eSetPresetLocal(
            psObj,
            psStruct->tPresetIndex,
            PRESET_UPDATE_TYPE_NAME,
            SERVICE_INVALID_ID,
            psStruct->pacPresetName);
    }

    // keep on iteratin'
    return TRUE;
}

/*****************************************************************************
*
*   vRemovePresetRestoreStruct
*
*****************************************************************************/
static void vRemovePresetRestoreStruct(void *pvData)
{
    PRESETS_RESTORE_STRUCT *psStruct = (PRESETS_RESTORE_STRUCT *)pvData;

    if (psStruct != NULL)
    {
        if (psStruct->pacPresetName != NULL)
        {
            // free memory allocated for us when getting the tag value
            SMSO_vDestroy((SMS_OBJECT)psStruct->pacPresetName);
        }

        // destroy the object we created as an enrty in the LL
        SMSO_vDestroy((SMS_OBJECT)psStruct);
    }
}

/*****************************************************************************
*
*   vDestroyBandLocal
*
*****************************************************************************/
static void vDestroyBandLocal(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bNotify
        )
{
    if ((hBand != PRESET_BAND_INVALID_OBJECT) && 
        (hPresets != PRESETS_INVALID_OBJECT))
    {
        BOOLEAN bSuccess;

        PRESET_BAND_OBJECT_STRUCT *psObj = 
            (PRESET_BAND_OBJECT_STRUCT *)hBand;

        // Attempt to remove the band and
        // all its related presets
        bSuccess = PRESETS_bRemoveBand(
            hPresets, hBand, psObj->hEntry, bNotify);

        // Only delete the band if it has been removed
        // from the presets service.
        if (bSuccess == TRUE)
        {
            // remove the band tag (and all it's child tags)
            TAG_eRemove(psObj->hTag, TRUE);
            psObj->hTag = TAG_INVALID_OBJECT;

            // Clear our linked list entry
            psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            
            // Delete the object
            PRESET_BAND_vDelete(hBand);
        }
    }

    return;
}

