/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:PRECIPITATION implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_precipitation_obj.h"
#include "precipitation_obj.h"

#include "sms_api_debug.h"


/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eType
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eType (
    PRECIPITATION_OBJECT hPrecipitation,
    PRECIPITATION_TYPE_ENUM *ePrecipitationType
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    PRECIPITATION_OBJECT_STRUCT *psObj = (PRECIPITATION_OBJECT_STRUCT *)hPrecipitation;

    do
    {
        if (ePrecipitationType == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hPrecipitation);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bPrecipAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        *ePrecipitationType = psObj->sData.eType;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eAmountRange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAmountRange (
    PRECIPITATION_OBJECT hPrecipitation,
    OSAL_FIXED_OBJECT hRangeLow,
    OSAL_FIXED_OBJECT hRangeHigh
        )
{
    BOOLEAN bOwner = FALSE, bTemp;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    PRECIPITATION_OBJECT_STRUCT *psObj = (PRECIPITATION_OBJECT_STRUCT*)hPrecipitation;

    do
    {
        if ((hRangeLow == OSAL_FIXED_INVALID_OBJECT) ||
            (hRangeHigh == OSAL_FIXED_INVALID_OBJECT))
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hPrecipitation);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bPrecipAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }

        bTemp = OSAL_FIXED.bCopyToMemory(psObj->sData.hHigh, hRangeHigh);
        if (bTemp == FALSE)
        {
            break;
        }
        bTemp = OSAL_FIXED.bCopyToMemory(psObj->sData.hLow, hRangeLow);
        if (bTemp == FALSE)
        {
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    }
    while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   eChance
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eChance (
    PRECIPITATION_OBJECT hPrecipitation,
    UN8 *pun8PrecipitationChance
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    PRECIPITATION_OBJECT_STRUCT *psObj = (PRECIPITATION_OBJECT_STRUCT *)hPrecipitation;

    do {
        if (pun8PrecipitationChance == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hPrecipitation);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bPopAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        *pun8PrecipitationChance = psObj->un8Chance;
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    PRECIPITATION_OBJECT hPrecipitation,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    PRECIPITATION_OBJECT_STRUCT *psObj = (PRECIPITATION_OBJECT_STRUCT *)hPrecipitation;
    BOOLEAN bOwner;

    // Determine if we are the owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hPrecipitation);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print PRECIPITATION information header
    n32Return += fprintf(psFile, "\nPrecipitation (Object: 0x%X):\n", hPrecipitation);

#if (DEBUG_OBJECT == 1)
    if (psObj->bPrecipAvailable == TRUE)
    {
        float fValueLow = 0;
        float fValueHigh = 0;
        N32 n32FixedValue;
        UN8 un8NumFractionalBits;

        if (psObj->sData.hHigh != OSAL_FIXED_INVALID_OBJECT)
        {
            n32FixedValue = OSAL_FIXED.n32Value(psObj->sData.hHigh);
            un8NumFractionalBits = OSAL_FIXED.un8NumFractionalBits(psObj->sData.hHigh);
            fValueHigh = ((float) n32FixedValue) / (1 << un8NumFractionalBits);
        }

        if (psObj->sData.hLow != OSAL_FIXED_INVALID_OBJECT)
        {
            n32FixedValue = OSAL_FIXED.n32Value(psObj->sData.hLow);
            un8NumFractionalBits = OSAL_FIXED.un8NumFractionalBits(psObj->sData.hLow);
            fValueLow = ((float) n32FixedValue) / (1 << un8NumFractionalBits);
        }

        n32Return += fprintf(psFile, "\tAmount range: %f - %f inches, type is %d\n",
                fValueLow, fValueHigh, psObj->sData.eType);

    }
#endif

    if (psObj->bPrecipAvailable == TRUE)
    {
        const char *pacType = "Unknown";

        switch (psObj->sData.eType)
        {
            case PRECIPITATION_TYPE_RAIN:
            {
                pacType = "Rain";
            }
            break;

            case PRECIPITATION_TYPE_SNOW:
            {
                pacType = "Snow";
            }
            break;

            case PRECIPITATION_TYPE_MIXED:
            {
                pacType = "Mixed";
            }
            break;

            case PRECIPITATION_TYPE_UNKNOWN:
            {
                // Noting to do. Use default value
            }
            break;
        }

        n32Return += fprintf(psFile, "\tType: %s\n",
            pacType);
    }

    if (psObj->bPopAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tChance: %d %%\n",
                                psObj->un8Chance);
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   PRECIPITATION_hCreate
*
*****************************************************************************/
PRECIPITATION_OBJECT PRECIPITATION_hCreate(
        SMS_OBJECT hParent,
        BOOLEAN bPopAvailable,
        BOOLEAN bPrecipAvailable,
        UN8 un8Pop,
        PRECIP_UNPARSED_DATA tData
            )
{
    PRECIPITATION_OBJECT_STRUCT *psObj =
        (PRECIPITATION_OBJECT_STRUCT *)PRECIPITATION_INVALID_OBJECT;

    // Create an instance of the PRECIPITATION object
    psObj = (PRECIPITATION_OBJECT_STRUCT *)
        SMSO_hCreate(
            PRECIPITATION_OBJECT_NAME,
            sizeof(PRECIPITATION_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return PRECIPITATION_INVALID_OBJECT;
    }

    // Initialize object per inputs
    psObj->bPopAvailable = bPopAvailable;
    psObj->un8Chance = un8Pop;
    psObj->bPrecipAvailable = bPrecipAvailable;

    if (bPrecipAvailable == TRUE)
    {
        psObj->bPrecipAvailable = GsWeatherIntf.bProcessPrecipitationData(tData, &psObj->sData);
    }

    return (PRECIPITATION_OBJECT)psObj;
}

/*****************************************************************************
*
*   PRECIPITATION_vDestroy
*
*****************************************************************************/
void PRECIPITATION_vDestroy (
    PRECIPITATION_OBJECT hPrecipitation
        )
{
    BOOLEAN bOwner;
    PRECIPITATION_OBJECT_STRUCT *psObj =
        (PRECIPITATION_OBJECT_STRUCT *)hPrecipitation;

    bOwner = SMSO_bOwner((SMS_OBJECT)hPrecipitation);
    if(bOwner == TRUE)
    {
        if (psObj->sData.hHigh != OSAL_FIXED_INVALID_OBJECT)
        {
            OSAL_FIXED.vDestroy(psObj->sData.hHigh);
        }

        if (psObj->sData.hLow != OSAL_FIXED_INVALID_OBJECT)
        {
            OSAL_FIXED.vDestroy(psObj->sData.hLow);
        }

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hPrecipitation);
    }
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
