/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:PLAYBACK implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "radio.h"
#include "sms_event.h"
#include "sms_obj.h"
#include "sms_update.h"
#include "decoder_obj.h"
#include "song_obj.h"
#include "songlist_obj.h"
#include "channel_obj.h"
#include "playback_obj.h"
#include "_playback_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
*
*   ePause
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePause (
    PLAYBACK_OBJECT hPlayback
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Validate obj
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);
    if(bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // ask the decoder to pause
        BOOLEAN bOk = DECODER_bPlayPause(psObj->hDecoder, TRUE);
        if (bOk == TRUE)
        {
            // Pause request was accepted by the decoder
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   ePlay
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePlay (
    PLAYBACK_OBJECT hPlayback
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Validate obj
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);
    if(bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // ask the decoder to play
        BOOLEAN bOk = DECODER_bPlayPause(psObj->hDecoder, FALSE);
        if (bOk == TRUE)
        {
            // Play request was accepted by the decoder
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } // if(bValid == TRUE)

    return eReturnCode;
}

/*****************************************************************************
*
*   eSeekTime
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSeekTime (
    PLAYBACK_OBJECT hPlayback,
    N32 n32Seconds,
    BOOLEAN bPauseAfterSeek
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Validate obj
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);
    if(bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj = (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // ask the decoder to seek by time
        BOOLEAN bOk = DECODER_bSeekTime (
            psObj->hDecoder,
            n32Seconds,
            bPauseAfterSeek
                );
        if (bOk == TRUE)
        {
              // seek request was accepted by the decoder
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } // if(bValid == TRUE)

    return eReturnCode;
}

/*****************************************************************************
*
*   eSeekSong
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSeekSong (
    PLAYBACK_OBJECT hPlayback,
    N16 n16RelativeOffset,
    BOOLEAN bPauseAfterSeek
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Validate obj
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);
    if(bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // ask the decoder to seek by song
        BOOLEAN bOk = DECODER_bSeekSong (
            psObj->hDecoder,
            n16RelativeOffset,
            bPauseAfterSeek
                );

        if (bOk == TRUE)
        {
            // seek request was accepted by the decoder
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } // if(bValid == TRUE)

    return eReturnCode;
}

/*****************************************************************************
*
*   eSeekDirect
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSeekDirect (
    PLAYBACK_OBJECT hPlayback,
    SONGLIST_OBJECT hSonglist,
    N16 n16Offset,
    BOOLEAN bPauseAfterSeek
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Validate obj
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);
    if(bValid == TRUE)
    {
        BOOLEAN bOk;
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        N16 n16AbsoluteOffset, n16CurrentSongOffset,
            n16LiveSongOffset, n16RelativeOffset;
        SONG_OBJECT hSong, hLive;
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == TRUE)
        {
            hSong = SONGLIST.hSong(hSonglist, n16Offset);
            if (hSong != SONG_INVALID_OBJECT)
            {
                // get the live song entry
                hLive = SCACHE_hGetTail(psObj->hSCache);
                // get the absolute offset of the live song
                n16LiveSongOffset = SONG_n16Offset(hLive);
                // get the absolute offset of the requested song
                n16AbsoluteOffset = SONG_n16Offset(hSong);

                SMSO_vUnlock((SMS_OBJECT)hPlayback);

                // get the current song offset
                n16CurrentSongOffset =
                    SCACHE_n16CurrentSongOffset(psObj->hSCache);

                n16RelativeOffset = n16AbsoluteOffset-n16CurrentSongOffset;

                if (n16CurrentSongOffset == n16LiveSongOffset)
                {
                    // we are currently live.
                    if (n16RelativeOffset == 0)
                    {
                        // we cannot pass in a relative offset of 0 'cause that
                        // will be inerpreted as going to the beginning of the
                        // current song and that is not what we want to do. we
                        // want to remain live
                        n16RelativeOffset = 1;
                    }
                    else
                    {
                        // subtract an extra one to account for live entry...
                        n16RelativeOffset = n16RelativeOffset + 1;
                    }
                }

                // ask the decoder to seek by song
                bOk = DECODER_bSeekSong (
                    psObj->hDecoder,
                    n16RelativeOffset,
                    bPauseAfterSeek
                        );

                if (bOk == TRUE)
                {
                    // seek request was accepted by the decoder
                    eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
                }
            } //if (hSong != SONG_INVALID_OBJECT)
            else
            {
                SMSO_vUnlock((SMS_OBJECT)hPlayback);
            }
        } // if (bLocked == TRUE)
    } // if(bValid == TRUE)
    return eReturnCode;
}

/*****************************************************************************
*
*   eSeekPrevious
*
*   Jump (seek) to the previous track boundary in the stored playback content
*   (i.e., the start of the current content block) and resume playback
*   if bPauseAfterSeek is false, or pause if bPauseAfterSeek is true.
*   If this command is received less than 3 seconds into playing the current
*   content block then the Module should jump to the track boundary prior to
*   the current content block. If there is no previous track boundary then
*   jump to the start of the stored content and resume playback. If the
*   previous track duration is less than 3 seconds the Module should again
*   jump to the previous track and iterate this check (i.e., in case there are
*   consecutive short tracks) until it gets to a track with duration greater
*   than 3 seconds or the start of the stored content and resume playback.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSeekPrevious (
    PLAYBACK_OBJECT hPlayback,
    BOOLEAN bPauseAfterSeek
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Validate obj
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);

    if(bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // ask the decoder to seek by song
        BOOLEAN bOk = DECODER_bSeekPrevious (
            psObj->hDecoder,
            bPauseAfterSeek
                );

        if (bOk == TRUE)
        {
            // seek request was accepted by the decoder
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } // if(bValid == TRUE)

    return eReturnCode;
}

/*****************************************************************************
*
*   eSeekNext
*
*   Jump (seek) to the next track boundary in the stored playback content and
*   resume playback if bPauseAfterSeek is false, or pause if bPauseAfterSeek
*   is true. If there is no next track boundary then treat this as a
*   jump to live play.
*
*   While this operation can be accomplished using eSeekSong with
*   n16RelativeOffset = +1, eSeekNext invokes a module command to perform the
*   jump-to-next-track directly (no intermidiate offset calculations in SMS).
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSeekNext (
    PLAYBACK_OBJECT hPlayback,
    BOOLEAN bPauseAfterSeek
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Validate obj
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);

    if(bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // ask the decoder to seek by song
        BOOLEAN bOk = DECODER_bSeekNext (
            psObj->hDecoder,
            bPauseAfterSeek
                );

        if (bOk == TRUE)
        {
            // seek request was accepted by the decoder
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } // if(bValid == TRUE)

    return eReturnCode;
}

/*****************************************************************************
*
*   ePlayPercentage
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePlayPercentage (
    PLAYBACK_OBJECT hPlayback,
    UN8 *pun8PlayPercentage
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pun8PlayPercentage !=NULL)
    {
          // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            *pun8PlayPercentage = psObj->sStats.un8PlayPercentage;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eFillPercentage
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFillPercentage (
    PLAYBACK_OBJECT hPlayback,
    UN8 *pun8FillPercentage
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
       SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pun8FillPercentage !=NULL)
    {
        // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            *pun8FillPercentage = psObj->sStats.un8FillPercentage;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eTimeOffset
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTimeOffset (
    PLAYBACK_OBJECT hPlayback,
    N32 *pn32TimeOffset
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
       SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pn32TimeOffset !=NULL)
    {
        // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            *pn32TimeOffset = psObj->sStats.n32TimeOffset;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eTimeFromTrackStart
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTimeFromTrackStart (
    PLAYBACK_OBJECT hPlayback,
    UN32 *pun32TimeFromTrackStart
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
       SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pun32TimeFromTrackStart !=NULL)
    {
        // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;
            if (psObj->sStats.un32TimeFromTrackStart != PLAYBACK_INVALID_TIME_FROM_START)
            {
                *pun32TimeFromTrackStart = psObj->sStats.un32TimeFromTrackStart;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            }

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eDurationOfTrack
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTrackDuration (
    PLAYBACK_OBJECT hPlayback,
    UN32 *pun32Duration
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
       SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pun32Duration != NULL)
    {
        // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);

        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            if (psObj->sStats.un32DurationOfTrack != PLAYBACK_INVALID_DURATION_OF_TRACK)
            {
                *pun32Duration = psObj->sStats.un32DurationOfTrack;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            }

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eTracksBefore
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTracksBefore (
    PLAYBACK_OBJECT hPlayback,
    UN16 *pun16TracksBefore
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
       SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pun16TracksBefore != NULL)
    {
        // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);

        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            *pun16TracksBefore = psObj->sStats.un16TracksBefore;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eTracksRemaining
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTracksRemaining (
    PLAYBACK_OBJECT hPlayback,
    UN16 *pun16TracksRemaining
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
       SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pun16TracksRemaining != NULL)
    {
        // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);

        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            *pun16TracksRemaining = psObj->sStats.un16TracksRemaining;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eIsPaused
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsPaused (
    PLAYBACK_OBJECT hPlayback,
    BOOLEAN *pbPaused
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
       SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (pbPaused !=NULL)
    {
        // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            *pbPaused = psObj->bPaused;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   tEventMask
*
*****************************************************************************/
static PLAYBACK_OBJECT_EVENT_MASK tEventMask (
    PLAYBACK_OBJECT hPlayback
        )
{
    PLAYBACK_OBJECT_EVENT_MASK tMask = PLAYBACK_OBJECT_EVENT_NONE;

    // Lock object for exclusive access
    BOOLEAN bLocked =
        SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // copy the pending events
        tMask = psObj->tPlaybackEventMask;

        // clear the pending events
        psObj->tPlaybackEventMask = PLAYBACK_OBJECT_EVENT_NONE;

         // Unlock object
         SMSO_vUnlock((SMS_OBJECT)hPlayback);
    }
    return tMask;
}

/*****************************************************************************
*
*   eErrorCode
*
*****************************************************************************/
static PLAYBACK_ERROR_CODE_ENUM eErrorCode (
    PLAYBACK_OBJECT hPlayback
        )
{
    PLAYBACK_ERROR_CODE_ENUM eErrorCode =
        PLAYBACK_ERROR_CODE_INVALID_OBJECT;

    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // Extract object error code
        eErrorCode = psObj->eErrorCode;
        // clear the error after it was read, unless the error is indicating
        // that the hwardware does not support playback
        if (eErrorCode != PLAYBACK_ERROR_CODE_PLAYBACK_NOT_SUPPORTED)
        {
            psObj->eErrorCode = PLAYBACK_ERROR_CODE_NONE;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hPlayback);
    }

    return eErrorCode;
}

/*****************************************************************************
*
*   eWarningOffsetGet
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWarningOffsetGet (
    PLAYBACK_OBJECT hPlayback,
    UN32 *pun32WarningOffset
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    if (pun32WarningOffset != NULL)
    {
          // non-NULL pointer passed in. just copy the last known value

        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            *pun32WarningOffset = psObj->un32WarningOffset;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);
        } // if(bLocked == TRUE)
    } // if (pun32WarningOffset != NULL)

    return eReturnCode;
}

/*****************************************************************************
*
*   eWarningOffsetSet
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWarningOffsetSet (
    PLAYBACK_OBJECT hPlayback,
    UN32 un32WarningOffset
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Lock object for exclusive access
    BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;
        PLAYBACK_ERROR_CODE_ENUM eErrorCode;

        // set the warning offset
        eErrorCode = RADIO_eSetPlaybackParams (
            psObj->hDecoder,
            un32WarningOffset
                );
        if( eErrorCode == PLAYBACK_ERROR_CODE_NONE )
        {
            // update with the new value.
            psObj->un32WarningOffset = un32WarningOffset;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else if (eErrorCode == PLAYBACK_ERROR_CODE_OUT_OF_RANGE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
        }


        SMSO_vUnlock((SMS_OBJECT)hPlayback);

    } // if(bLocked == TRUE)

    return eReturnCode;
}

/*****************************************************************************
*
*   eRecordedContentTimeInfo
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRecordedContentTimeInfo (
    PLAYBACK_OBJECT hPlayback,
    UN32 *pun32Duration,
    UN32 *pun32ElapsedTime,
    UN32 *pun32RemainingTime
        )
{
   SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    if ( ( pun32Duration == NULL )
      && ( pun32ElapsedTime == NULL )
      && ( pun32RemainingTime == NULL ) )
    {
        // all are NULL, so go get the info

        // but currently there is no implementation for playback of
        // recorded content.....
    }
    else
    {
        // Lock object for exclusive access
        BOOLEAN bLocked =
            SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            PLAYBACK_OBJECT_STRUCT *psObj =
                (PLAYBACK_OBJECT_STRUCT *)hPlayback;

            if ( pun32Duration != NULL )
            {
                *pun32Duration = psObj->un32Duration;
            }
            if ( pun32ElapsedTime != NULL )
            {
                *pun32ElapsedTime = psObj->un32ElapsedTime;
            }
            if ( pun32RemainingTime != NULL )
            {
                *pun32RemainingTime = psObj->un32RemainingTime;
            }
            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hPlayback);

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        } // if(bLocked == TRUE)
    }
    return eReturnCode;
}

/*****************************************************************************
 *
 *   un32IRBufferDuration
 *
 *****************************************************************************/
static UN32 un32IRBufferDuration (
        PLAYBACK_OBJECT hPlayback
            )
{
    UN32 un32IRBufferDuration=0;
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hPlayback);
    if (bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj;
        BOOLEAN bLocked;
        DECODER_OBJECT hDecoder;

        psObj = (PLAYBACK_OBJECT_STRUCT *)hPlayback;
        hDecoder = psObj->hDecoder;

        // Verify Object is valid
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;


            eReturnCode = RADIO_eGetIRDurationOfBuffer(hDecoder,
                                               &un32IRBufferDuration);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
               // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        PLAYBACK_OBJECT_NAME": Unable to determine IR Buffer.");
                        un32IRBufferDuration = 0;
            }
            SMSO_vUnlock((SMS_OBJECT)hDecoder);
        }
    }
    return un32IRBufferDuration;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   PLAYBACK_hCreate
*
*          This function is called to create an instance of a Playback object
*
*       Inputs:
*                DECODER_OBJECT hDecoder - the decoder to which this will belong
*
*       Outputs:
*            a handle to a valid PLAYBACK object if successful
*           or PLAYBACK_INVALID_OBJECT if failed
*
*****************************************************************************/
PLAYBACK_OBJECT PLAYBACK_hCreate (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bValid;
    static UN32 un32Instance = 0;
    PLAYBACK_OBJECT_STRUCT *psObj;
    SCACHE_OBJECT hSCache;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid != TRUE)
    {
         return PLAYBACK_INVALID_OBJECT;
    }

    // Construct a unique name for this channel.
    snprintf(&acName[0],
            sizeof(acName),
            PLAYBACK_OBJECT_NAME":%u",
            un32Instance++);

    // Create an instance of this object
    psObj = (PLAYBACK_OBJECT_STRUCT *)
        SMSO_hCreate(
            PLAYBACK_OBJECT_NAME,
            sizeof(PLAYBACK_OBJECT_STRUCT),
            (SMS_OBJECT)hDecoder, // Child
            FALSE); // No Lock (ignored)
    if(psObj == NULL)
    {
        // Error!
        return PLAYBACK_INVALID_OBJECT;
    }

    hSCache = SCACHE_hCreate((PLAYBACK_OBJECT)psObj);
    if(hSCache == SCACHE_INVALID_OBJECT)
    {
        // Error!
        PLAYBACK_vDestroy((PLAYBACK_OBJECT)psObj);
        return PLAYBACK_INVALID_OBJECT;
    }
    psObj->hSCache = hSCache;

    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->hSonglistLists,
        PLAYBACK_OBJECT_NAME":SLLL",
        (OSAL_LL_COMPARE_HANDLER)NULL,
        OSAL_LL_OPTION_NONE);

    if (eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        PLAYBACK_vDestroy((PLAYBACK_OBJECT)psObj);
        return PLAYBACK_INVALID_OBJECT;
    }

    // Initialize object...

    // Extract & Initialize decoder information
    psObj->hDecoder = hDecoder;

    // Initialize playback object info
    psObj->eErrorCode = PLAYBACK_ERROR_CODE_NONE;
    psObj->bPaused = FALSE; // default is Play
    psObj->sStats.un8FillPercentage = 0; // %
    psObj->sStats.un8PlayPercentage = 0; // %
    psObj->sStats.n32TimeOffset = 0;    // seconds
    psObj->sStats.un32TimeFromTrackStart = PLAYBACK_INVALID_TIME_FROM_START;
    psObj->sStats.un32DurationOfTrack = PLAYBACK_INVALID_DURATION_OF_TRACK;
    psObj->sStats.un16TracksBefore = 0;    // number of tracks before play point
    psObj->sStats.un16TracksRemaining = 0;    // number of tracks after play point
    psObj->un32WarningOffset = 0;

    psObj->tPlaybackEventMask = PLAYBACK_OBJECT_EVENT_NONE;

    return (PLAYBACK_OBJECT)psObj;
}

/*****************************************************************************
*
*   PLAYBACK_vDestroy
*
*          This function is called to destroy an instance of a Playback object
*
*       Inputs:
*                PLAYBACK_OBJECT hPlayback - the playback obj to destroy
*
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vDestroy (
    PLAYBACK_OBJECT hPlayback
        )
{
     // Lock object for exclusive access
    BOOLEAN bLocked =
                SMSO_bLock((SMS_OBJECT)hPlayback, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // Un-Initialize parameters...
        psObj->hDecoder = DECODER_INVALID_OBJECT;
        psObj->eErrorCode = PLAYBACK_ERROR_CODE_NONE;
        psObj->bPaused = FALSE;
        psObj->un32WarningOffset = 0;
        psObj->sStats.un8FillPercentage = 0;
        psObj->sStats.un8PlayPercentage = 0;
        psObj->sStats.n32TimeOffset = 0;
        psObj->sStats.un32TimeFromTrackStart = PLAYBACK_INVALID_TIME_FROM_START;
        psObj->sStats.un32DurationOfTrack = PLAYBACK_INVALID_DURATION_OF_TRACK;
        psObj->sStats.un16TracksBefore = 0;
        psObj->sStats.un16TracksRemaining = 0;
        psObj->un32Duration = 0;
        psObj->un32ElapsedTime = 0;
        psObj->un32RemainingTime = 0;
        psObj->tPlaybackEventMask = PLAYBACK_OBJECT_EVENT_NONE;

        // Destroy all remaining SongList objects.
        OSAL.eLinkedListRemoveAll(
            psObj->hSonglistLists, (OSAL_LL_RELEASE_HANDLER)SONGLIST_vDestroy);

        OSAL.eLinkedListDelete(psObj->hSonglistLists);

        if (psObj->hSCache != SCACHE_INVALID_OBJECT)
        {
            // destroy the song cache
            SCACHE_vDestroy(psObj->hSCache);
            psObj->hSCache = SCACHE_INVALID_OBJECT;
        }

        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hPlayback);
    } // if(bLocked == TRUE)

    return;
}

/*****************************************************************************
*
*   PLAYBACK_bUpdatePause
*
*          This friend function allows the decoder to update play / pause
*
*       Inputs:
*           hPlayback - the playback obj whose info is being modified
*           bPaused - TRUE if Pause, FALSE if Play
*
*       Outputs:
*            TRUE if ok
*           FALSE if not ok
*
*****************************************************************************/
void PLAYBACK_vUpdatePause (
    PLAYBACK_OBJECT hPlayback,
    BOOLEAN bPaused
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);

    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // call the local function
        vUpdatePause( psObj, bPaused);
    } // if (bOwner == TRUE)
    return;
}

/*****************************************************************************
*
*   PLAYBACK_vUpdatePlaybackParams
*
*          This friend function allows the decoder to update play / pause
*
*       Inputs:
*           hPlayback - the playback obj whose info is being modified
*           un8WarningOffset - the new warning offset
*           un8StatUpdateInterval - the new stat update interval
*
*       Outputs:
*            TRUE if ok
*           FALSE if not ok
*
*****************************************************************************/
void PLAYBACK_vUpdatePlaybackParams(
    PLAYBACK_OBJECT hPlayback,
    UN32 un32WarningOffset
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);
    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        psObj->un32WarningOffset = (UN32)un32WarningOffset;

    } // if (bOwner == TRUE)

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vWarningOccurred
*
*          This friend function is used to process the occurence of a Warning event
*
*       Inputs:
*           hPlayback - the playback obj whose is being modified
*
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vWarningOccurred (
    PLAYBACK_OBJECT hPlayback
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);

    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // Set event mask to PLAY
        psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_WARNING;

        // have the decoder set the decoder level playback event
        DECODER_vPlaybackEvent(psObj->hDecoder, DECODER_OBJECT_EVENT_PLAYBACK);
    } // if (bOwner == TRUE)
    return;
}

/*****************************************************************************
*
*   PLAYBACK_vLimitOccurred
*
*          This friend function is used to process the occurence of a Warning event
*
*       Inputs:
*           hPlayback - the playback obj whose is being modified
*
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vLimitOccurred (
    PLAYBACK_OBJECT hPlayback
        )
{
      BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);

    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // Set event mask to PLAY
        psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_LIMIT;

        // have the decoder set the decoder level playback event
        DECODER_vPlaybackEvent(psObj->hDecoder, DECODER_OBJECT_EVENT_PLAYBACK);
    } // if (bOwner == TRUE)
    return;
}

/*****************************************************************************
*
*   PLAYBACK_hSCache
*
*          This friend function is used to ...
*
*       Inputs:
*           hPlayback - the playback obj whose is being modified
*
*       Outputs:
*            ....
*
*****************************************************************************/
SCACHE_OBJECT PLAYBACK_hSCache (
    PLAYBACK_OBJECT hPlayback
        )
{
      SCACHE_OBJECT hSCache = SCACHE_INVALID_OBJECT;
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);

    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;
        if(psObj != NULL)
        {
            hSCache = psObj->hSCache;
        }

    } // if (bOwner == TRUE)
    return hSCache;
}

/*****************************************************************************
*
*   PLAYBACK_vUpdateSCache
*
*          This friend function is used to tell PLAYBACK to call any registerd
*       callback for this SCACHE
*
*       Inputs:
*           hPlayback - the playback obj whose is being modified
*           hSCache - the updated scache
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vUpdateSCache (
    PLAYBACK_OBJECT hPlayback
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);

    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // tell decoder that songlist was updated
        SCACHE_vUpdate(psObj->hSCache);
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vPostCreateSonglist
*
*          This friend function is used to perform a portion of the
*       creation of a Songlist
*
*       Inputs:
*           hPlayback - the playback obj
*           hSonglist - the songlist being created
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vPostCreateSongList(PLAYBACK_OBJECT hPlayback,
                                  SONGLIST_OBJECT hSonglist)
{
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);

    if(bValid == TRUE)
    {
          PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        DECODER_vCreateSongList(psObj->hDecoder, hSonglist);
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vPostDestroySongList
*
*          This friend function is used to perform a portion of the
*       destruction of a Songlist
*
*       Inputs:
*           hPlayback - the playback obj
*           hSonglist - the songlist being destroyed
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vPostDestroySongList(PLAYBACK_OBJECT hPlayback,
                                  SONGLIST_OBJECT hSonglist)
{
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);

    if(bValid == TRUE)
    {
          PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        DECODER_vDestroySongList(psObj->hDecoder, hSonglist);
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vProcessCreateSonglist
*
*          This friend function is used to perform a portion of the
*       creation of a Songlist
*
*       Inputs:
*           hPlayback - the playback obj
*           hSonglist - the songlist being created
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vProcessCreateSongList(PLAYBACK_OBJECT hPlayback,
                                     SONGLIST_OBJECT hSonglist)
{
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);

    if(bValid == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // Add songlist object to common LL
        eReturnCode = OSAL.eLinkedListAdd(psObj->hSonglistLists,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hSonglist);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // All SongList objects shall belong to Playback's common LL
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PLAYBACK_OBJECT_NAME": Failed to add Songlist: %p into common LL",
                hSonglist);
        }

        SONGLIST_vCreate(hSonglist, hPlayback);
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vProcessDestroySongList
*
*          This friend function is used to perform a portion of the
*       destruction of a Songlist
*
*       Inputs:
*           hPlayback - the playback obj
*           hSonglist - the songlist being destroyed
*       Outputs:
*            none
*
*****************************************************************************/
void PLAYBACK_vProcessDestroySongList(PLAYBACK_OBJECT hPlayback,
                                     SONGLIST_OBJECT hSonglist)
{
    BOOLEAN bValid =
                SMSO_bValid((SMS_OBJECT)hPlayback);

    if(bValid == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Search for SongList handle entry
        eReturnCode = OSAL.eLinkedListSearch(
            psObj->hSonglistLists,
            &hEntry,
            (void *)hSonglist);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListRemove(hEntry);
        }
        else
        {
            // All SongList objects shall belong to Playback's common LL
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                PLAYBACK_OBJECT_NAME": Failed to remove Songlist: %p from common LL",
                hSonglist);
        }

        SONGLIST_vDestroy(hSonglist);
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vSetError
*
*       This friend function is used to set the PLAYBACK ERROR EVENT
*
*       Inputs:
*           hPlayback
*           eError - error code
*
*       Outputs:
*           none
*
*****************************************************************************/
void PLAYBACK_vSetError (
    PLAYBACK_OBJECT hPlayback,
    PLAYBACK_ERROR_CODE_ENUM eError
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);
    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        // save the new value
        psObj->eErrorCode = eError;

        // Set event mask to ERROR
        psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_ERROR;

        // have the decoder set the decoder level playback event
        DECODER_vPlaybackEvent(psObj->hDecoder, DECODER_OBJECT_EVENT_PLAYBACK);
   }

   return;
}

/*****************************************************************************
*
*   PLAYBACK_vUpdatePlaybackInfo
*
*       This friend function updates playback info
*
*       Inputs:
*           hPlayback - the playback obj whose info is being modified
*           psStats - the structure that contains the updated info
*
*       Outputs:
*           none
*****************************************************************************/
void PLAYBACK_vUpdatePlaybackInfo (
    PLAYBACK_OBJECT hPlayback,
    const PLAYBACK_STATS_STRUCT *psStats
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);
    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;
        BOOLEAN bUpdate = FALSE;

        // did the fill percentage change?
        if(psObj->sStats.un8FillPercentage != psStats->un8FillPercentage)
        {
            // save the new value
            psObj->sStats.un8FillPercentage = psStats->un8FillPercentage;
            // indicate that the fill percentage changed
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_FILL_PERCENTAGE;
            // one of the pieces of information has changed
            bUpdate = TRUE;
        }

        // did the play percentage change?
        if(psObj->sStats.un8PlayPercentage != psStats->un8PlayPercentage)
        {
            // save the new value
            psObj->sStats.un8PlayPercentage = psStats->un8PlayPercentage;
            // indicate that the fill percentage changed
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_PLAY_PERCENTAGE;
            // one of the pieces of information has changed
            bUpdate = TRUE;
        }

        // did the time offset change?
        if(psObj->sStats.n32TimeOffset != psStats->n32TimeOffset)
        {
            // save the new value
            psObj->sStats.n32TimeOffset = psStats->n32TimeOffset;
            // indicate that the time offset changed
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_TIME_OFFSET;

            psObj->un32RemainingTime = (UN32)(-1 * psObj->sStats.n32TimeOffset);
            if (psObj->un32Duration < psObj->un32RemainingTime)
            {
                // computed duration cannot be less than remaining time
                psObj->un32Duration = psObj->un32RemainingTime;
            }

            psObj->un32ElapsedTime = psObj->un32Duration - psObj->un32RemainingTime;
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO;

            // one of the pieces of information has changed
            bUpdate = TRUE;
        }

        // did the time from start of track change?
        if(psObj->sStats.un32TimeFromTrackStart != psStats->un32TimeFromTrackStart)
        {
            // save the new value
            psObj->sStats.un32TimeFromTrackStart = psStats->un32TimeFromTrackStart;
            // indicate that the time offset changed
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_TIME_FROM_TRACK_START;
            // one of the pieces of information has changed
            bUpdate = TRUE;
        }

        // did the duration of the track change?
        if(psObj->sStats.un32DurationOfTrack != psStats->un32DurationOfTrack)
        {
            // save the new value
            psObj->sStats.un32DurationOfTrack = psStats->un32DurationOfTrack;
            // indicate that the time offset changed
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_DURATION_OF_TRACK;
            // one of the pieces of information has changed
            bUpdate = TRUE;
        }

        // did the number of tracks before play point change?
        if(psObj->sStats.un16TracksBefore != psStats->un16TracksBefore)
        {
            // save the new value
            psObj->sStats.un16TracksBefore = psStats->un16TracksBefore;
            // indicate that the time offset changed
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_TRACKS_BEFORE;
            // one of the pieces of information has changed
            bUpdate = TRUE;
        }

        // did the number of tracks after play point change?
        if(psObj->sStats.un16TracksRemaining != psStats->un16TracksRemaining)
        {
            // save the new value
            psObj->sStats.un16TracksRemaining = psStats->un16TracksRemaining;
            // indicate that the time offset changed
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_TRACKS_REMAINING;
            // one of the pieces of information has changed
            bUpdate = TRUE;
        }

        if (bUpdate == TRUE)
        {
            puts( PLAYBACK_OBJECT_NAME": Playback Time Info Updated" );

            //have the decoder set the playback event
            DECODER_vPlaybackEvent(
                psObj->hDecoder, DECODER_OBJECT_EVENT_PLAYBACK);
        }
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vUpdateTimeInfo
*
*****************************************************************************/
void PLAYBACK_vUpdateTotalDuration (
    PLAYBACK_OBJECT hPlayback,
    UN32 un32Duration
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);
    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        if (un32Duration < psObj->un32RemainingTime)
        {
            // computed duration can't be less than remaining time
            un32Duration = psObj->un32RemainingTime;
        }

        if (psObj->un32Duration != un32Duration)
        {
            psObj->un32Duration = un32Duration;

            // elapsed time = duration - remaining time
            // duration changed and/or remaining time changed
            // so recompute the elapsed time
            psObj->un32ElapsedTime = psObj->un32Duration - psObj->un32RemainingTime;
            psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO;

            //have the decoder set the playback event
            DECODER_vPlaybackEvent(
                psObj->hDecoder, DECODER_OBJECT_EVENT_PLAYBACK);
        }
    }

    return;
}

/*****************************************************************************
*
*   PLAYBACK_vFlushTimeInfo
*
*****************************************************************************/
void PLAYBACK_vFlushTimeInfo (
    PLAYBACK_OBJECT hPlayback
        )
{
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hPlayback);
    if (bOwner == TRUE)
    {
        PLAYBACK_OBJECT_STRUCT *psObj =
            (PLAYBACK_OBJECT_STRUCT *)hPlayback;

        puts( PLAYBACK_OBJECT_NAME": Flushing Time Info" );

        psObj->un32Duration = 0;
        psObj->un32ElapsedTime = 0;
        psObj->un32RemainingTime = 0;

        psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO;

        //have the decoder set the playback event
        DECODER_vPlaybackEvent(psObj->hDecoder, DECODER_OBJECT_EVENT_PLAYBACK);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vUpdatePause
*
*          This private function is used to update the play/paused status
*       of a playback obj
*
*       Inputs:
*           hPlayback - the playback obj whose is being modified
*           bPaused - the new pause status
*
*       Outputs:
*            none
*
*****************************************************************************/
static void vUpdatePause (
    PLAYBACK_OBJECT_STRUCT *psObj,
    BOOLEAN bPaused
        )
{
    // save the new value
    psObj->bPaused = bPaused;

    if (psObj->bPaused == FALSE)
    {
        // Set event mask to PLAY
        psObj->tPlaybackEventMask &= ~(PLAYBACK_OBJECT_EVENT_PAUSE);
        psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_PLAY;
    }
    else
    {
        // set the mask to PAUSE
        psObj->tPlaybackEventMask &= ~(PLAYBACK_OBJECT_EVENT_PLAY);
        psObj->tPlaybackEventMask |= PLAYBACK_OBJECT_EVENT_PAUSE;
    }

    // have the decoder set the decoder level playback event
    DECODER_vPlaybackEvent(psObj->hDecoder, DECODER_OBJECT_EVENT_PLAYBACK);

    return;
}
