/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:NEWS implementation for the
 *  Sirius Module Services (SMS)
 *
******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "cdo_obj.h"
#include "cid_obj.h"
#include "cme.h"
#include "news_obj.h"
#include "_news_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eSubType
*
* Retrievs the enumerated value representing the NEWS object type.
*
*****************************************************************************/
static NEWS_ENUM eSubType (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    NEWS_OBJECT_STRUCT *psObj;
    NEWS_ENUM eSubType = NEWS_UNKNOWN;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_NEWS)
    {
        // Go get CDO structure from provided CDO
        psObj = (NEWS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            eSubType = psObj->eSubType;
        }
    }

    return eSubType;
}

/*****************************************************************************
*
*   hId
*
* Retrieves the CID associated with this CDO given that it is a NEWS type.
* The returned CID is the uniuqe identifier for the news this CDO describes.
*
*****************************************************************************/
static CID_OBJECT hId (
    CD_OBJECT hCDO
        )
{
    CDO_TYPE_ENUM eType;
    NEWS_OBJECT_STRUCT *psObj;
    CID_OBJECT hId = CID_INVALID_OBJECT;

    // Verify CDO type is correct for this API
    eType = CDO.eType(hCDO);
    if(eType == CDO_NEWS)
    {
        // Go get CDO structure from provided CDO
        psObj = (NEWS_OBJECT_STRUCT *)CDO_pvContentData(hCDO);
        if(psObj != NULL)
        {
            hId = psObj->hId;
        }
    }

    return hId;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   NEWS_bValidSubId
*
* Local function which checks an enumerated type to make sure it is
* one of the valid NEWS_ENUM types.
*
*****************************************************************************/
BOOLEAN NEWS_bValidSubId (
    NEWS_ENUM eType
        )
{
    switch (eType)
    {
        case NEWS_GLOBAL:
        case NEWS_WEATHER:
        case NEWS_FINANCIAL:
        case NEWS_US:
        case NEWS_PUBLIC_RADIO:
        case NEWS_OTHER:
            return TRUE;

        default:
            return FALSE;
    }
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vUnInit
*
* CDO Un-initialize method. Called when a NEWS CDO is replaced by
* another type, requiring any resources to be released.
*
*****************************************************************************/
static void vUnInit (
    NEWS_OBJECT_STRUCT *psObj
        )
{

    // Check inputs.
    if(psObj == NULL)
    {
        // Error!
        return;
    }

    // Un-init NEWS CDO structure
    psObj->eSubType = NEWS_UNKNOWN;

    // Release CID
    if(psObj->hId != CID_INVALID_OBJECT)
    {
        CD_OBJECT hCDO = CDO_hCDO(psObj);
        CME_eDestroyContent(hCDO, &psObj->hId);
    }

    // Default object now
    *psObj = gsDefaultNews;

    return;
}

/*****************************************************************************
*
*   n16Equal
*
*       This is the function used to compare NEWS CDOs. Specifically
*       this function is used to perform a go, no-go comparison
*       or binary comparison of the two CDOs. Two NEWS CDOs are considered
*       equal if both the CIDs are the same.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               -1  - CDOs are not equal (or error)
*
*****************************************************************************/
static N16 n16Equal (
    NEWS_OBJECT_STRUCT *psObj1,
    NEWS_OBJECT_STRUCT *psObj2
        )
{
    N16 n16Result;

    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return -1;
    }

    // Id's must match
    n16Result = CID.n16Equal(
        psObj1->hId, psObj2->hId);

   return n16Result;
}

/*****************************************************************************
*
*   n16Compare
*
*       This is the function used to compare NEWS CDOs. Specifically
*       this function is used to keep CDOs in order for
*       comparison/sorting and thus a relational compare is performed.
*
*       Outputs:
*               0   - CDOs have the same value (equal)
*               > 0 - CDO1 is greater than (after) CDO2
*               < 0 - CDO1 is less than (before) CDO2 or error
*
*****************************************************************************/
static N16 n16Compare (
    NEWS_OBJECT_STRUCT *psObj1,
    NEWS_OBJECT_STRUCT *psObj2
        )
{
    N16 n16Result;

    // Verify inputs
    if( (psObj1 == NULL) || (psObj2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    if ((psObj1->hId == CID_INVALID_OBJECT) &&
        (psObj2->hId == CID_INVALID_OBJECT))
    {
        return 0;
    }

    // Sort by Id's
    n16Result = CID.n16Compare(
        psObj1->hId, psObj2->hId);

    return n16Result;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This method is used by the caller to send formatted
* output of a CDO's contents to a specified file or device.
* This is mainly helpful during debugging of CDO's but could be used by
* a caller for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a CDO to a file for the
* purposes of later re-generating the CDO (for storage of the object).
* This API instead sends the CDO as a verbose formatted output version.
*
* Inputs:
*
*   psObj - The CDO structure the caller wishes to print.
*   psFile - The device to write the CDO contents to.
*
*****************************************************************************/
static N32 n32FPrintf (
    const NEWS_OBJECT_STRUCT *psObj,
    FILE *psFile
        )
{
    N32 n32Return = EOF;
    N32 n32Temp   = 0;

    // Check inputs.
    if((psObj != NULL) && (psFile != NULL))
    {
        n32Return = 0;

        // Print CDO information...
        n32Return += fprintf(psFile,
            "eSubType: %s\n\n", pacTypeText(psObj->eSubType));

        n32Return += fprintf(psFile, "hId:\n");
        n32Temp = CID.n32FPrintf(psObj->hId, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
        n32Return += fprintf(psFile, "\n");
    }

    return n32Return;
}

/*****************************************************************************
*
*   bHasId
*
* This method is used to examine a CDO sub-type for a specific Content-ID (CID)
* provided by the caller. If this sub-type has this id, then TRUE is returned
* otherwise, FALSE is returned if it does not contain it. Keep in mind
* this functions checks the entire sub-type for any matches.
*
* Inputs:
*   psObj - A pointer to an object for which to examine and determine if
*       the provided CID is contained within it.
*   hId - A valid CID to look for within the object.
*
* Returns:
*   TRUE of the CID was found anywhere within the object, otherwise FALSE
*   is returned if it could not be found.
*
*****************************************************************************/
static BOOLEAN bHasId (
    const NEWS_OBJECT_STRUCT *psObj,
    CID_OBJECT hId
        )
{
    BOOLEAN bHasId = FALSE;

    // We can look to see if this CID matches one we have in this object.
    // Any hit is enough to cause us to fall out and return TRUE.
    do
    {
        UN16 n16Equal;

        // hId == hId?
        n16Equal = CID.n16Equal(hId, psObj->hId);
        if(n16Equal == 0)
        {
            // Equal, so they match
            bHasId = TRUE;
            break;
        }

    } while(FALSE);

    return bHasId;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pacTypeText
*
* This local function translates a provided NEWS_ENUM into a text
* string representation.
*
*****************************************************************************/
static const char *pacTypeText(
    NEWS_ENUM eType
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case NEWS_GLOBAL:
            pacReturnString = MACRO_TO_STRING(NEWS_GLOBAL);
        break;

        case NEWS_WEATHER:
            pacReturnString = MACRO_TO_STRING(NEWS_WEATHER);
        break;

        case NEWS_FINANCIAL:
            pacReturnString = MACRO_TO_STRING(NEWS_FINANCIAL);
        break;

        case NEWS_US:
            pacReturnString = MACRO_TO_STRING(NEWS_US);
        break;

        case NEWS_PUBLIC_RADIO:
            pacReturnString = MACRO_TO_STRING(NEWS_PUBLIC_RADIO);
        break;

        case NEWS_OTHER:
            pacReturnString = MACRO_TO_STRING(NEWS_OTHER);
        break;

        case NEWS_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(NEWS_UNKNOWN);
        break;
    }

    return pacReturnString;
}
