/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the movies
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef MOVIES_DB_CONSTANTS_H_
#define MOVIES_DB_CONSTANTS_H_

  /**************/
 /** INCLUDES **/
/**************/
#include "sms_api.h"
#include "location_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* Database File Name */
#define MOVIES_DATABASE_FOLDER "movies"
#define MOVIES_REF_DATABASE_FILENAMEA "movies_ref.db"
#define MOVIES_REF_DATABASE_FILENAMEB "movies_ref.b.db"
#define MOVIES_PERSIST_DATABASE_FILENAME "movies.db"

/* Database Table Names */
#define MOVIES_VERSION_TABLE_NAME "Version"
#define MOVIES_THEATER_TABLE_NAME "Theaters"
#define MOVIES_TABLE_NAME "Movies"
#define MOVIES_DESC_TABLE_NAME "Times"
#define MOVIES_RATINGS_TABLE_NAME "Ratings"
#define MOVIES_CONTROL_TABLE_NAME "Control"
#define MOVIES_THEATER_RTREE_TABLE_NAME "Theaters_rtree"

/* Database Table Attributes */
#define MOVIES_DATABASE_FILE_VERSION (8)

/*  set fixed_constant [expr 1 << $binpoint]
    set rtree_range [expr .000015 * $fixed_constant] */
#define MOVIES_RTREE_RANGE      ((N32)62)

#define MOVIES_CONTROL_TABLE_TIMEOUT_SECS  (36 * 60 * 60)

/* Database Queries */

// Select all from version
#define MOVIES_SELECT_DB_VERSION \
	"select * from "MOVIES_VERSION_TABLE_NAME";"

#define MOVIES_UPDATE_BASELINE_VERSION \
    "update "MOVIES_VERSION_TABLE_NAME" set Baseline = %d where SchemaVer = %u"

// Select all from ratings
#define MOVIES_SELECT_DB_RATINGS \
    "select * from "MOVIES_RATINGS_TABLE_NAME";"

#define MOVIES_CLEAR_DB_RATINGS \
    "delete from "MOVIES_RATINGS_TABLE_NAME";"

#define MOVIES_INSERT_DB_RATINGS \
    "insert into "MOVIES_RATINGS_TABLE_NAME" " \
    "values(?, ?, ?, ?);"

#define MOVIES_SELECT_THEATERS_BY_LOCATION \
    "select * from "MOVIES_THEATER_TABLE_NAME" where ID in " \
    "(select ID from "MOVIES_THEATER_RTREE_TABLE_NAME" where " \
    MOVIES_THEATER_RTREE_TABLE_NAME".minLat >= %d and " \
    MOVIES_THEATER_RTREE_TABLE_NAME".maxLat <= %d and " \
    MOVIES_THEATER_RTREE_TABLE_NAME".minLon >= %d and " \
    MOVIES_THEATER_RTREE_TABLE_NAME".maxLon <= %d) order by ID;"

#define MOVIES_SELECT_THEATERS_BY_ID \
    "select * from "MOVIES_THEATER_TABLE_NAME" where ID = %u;"

#define MOVIES_DELETE_THEATER \
    "DELETE from "MOVIES_THEATER_TABLE_NAME" where ID = %u;"

#define MOVIES_DELETE_THEATER_RTREE \
    "DELETE from "MOVIES_THEATER_RTREE_TABLE_NAME" where ID = %u;"

#define MOVIES_UPDATE_THEATER \
    "Update "MOVIES_THEATER_TABLE_NAME" set " \
    "ID = ?, " \
    "Name = ?, " \
    "State = ?, " \
    "Address = ?, " \
    "City = ?, " \
    "Zip = ?, " \
    "Phone = ?, " \
    "Amenities = ?, " \
    "Lat = ?, " \
    "Lon = ? " \
    "where ID = ?1;"

#define MOVIES_UPDATE_THEATER_RTREE \
    "Update "MOVIES_THEATER_RTREE_TABLE_NAME" set " \
    "minLat = %i, " \
    "maxLat = %i, " \
    "minLon = %i, " \
    "maxLon = %i " \
    "where ID = %u;"

#define MOVIES_INSERT_THEATER \
    "insert or replace into "MOVIES_THEATER_TABLE_NAME" " \
    "values(?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"

#define MOVIES_INSERT_THEATER_RTREE \
    "insert into "MOVIES_THEATER_RTREE_TABLE_NAME" values(%u, %i, %i, %i, %i);"

#define MOVIES_INSERT_DESC_STMT_ID_PARAM            0
#define MOVIES_INSERT_DESC_STMT_ENG_NAME_PARAM      1
#define MOVIES_INSERT_DESC_STMT_SPN_NAME_PARAM      2
#define MOVIES_INSERT_DESC_STMT_FR_NAME_PARAM       3
#define MOVIES_INSERT_DESC_STMT_RUN_TIME_PARAM      4
#define MOVIES_INSERT_DESC_STMT_ACTORS_PARAM        5
#define MOVIES_INSERT_DESC_STMT_ENG_SYNOPSIS_PARAM  6
#define MOVIES_INSERT_DESC_STMT_SPN_SYNOPSIS_PARAM  7
#define MOVIES_INSERT_DESC_STMT_FR_SYNOPSIS_PARAM   8
#define MOVIES_INSERT_DESC_STMT_NUMRATING_PARAM     9
#define MOVIES_INSERT_DESC_STMT_RATING_PARAM        10
#define MOVIES_INSERT_DESC_STMT_PARAMS_COUNT        11

#define MOVIES_INSERT_DESCRIPTION \
    "insert into "MOVIES_TABLE_NAME" (" \
     "MVID, EnglishName, SpanishName, FrenchName, " \
     "RunTime, Actors, "\
     "EnglishSynopsis, SpanishSynopsis, FrenchSynopsis, "\
     "NumRatings, Ratings) " \
    "VALUES(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"

#define MOVIES_FETCH_DESCRIPTIONS \
    "select * from "MOVIES_TABLE_NAME";"

#define MOVIES_CLEAR_DESCRIPTIONS \
    "DELETE FROM "MOVIES_TABLE_NAME";"

#define MOVIES_INSERT_TIMES_STMT_ID_PARAM           0
#define MOVIES_INSERT_TIMES_STMT_INIT_TIME_PARAM    1
#define MOVIES_INSERT_TIMES_STMT_NUM_MOVIES_PARAM   2
#define MOVIES_INSERT_TIMES_STMT_NUM_UN16S_PARAM    3
#define MOVIES_INSERT_TIMES_STMT_SHOW_TIMES_PARAM   4
#define MOVIES_INSERT_TIMES_STMT_PARAMS_COUNT       5

#define MOVIES_INSERT_TIMES_FOR_THEATER \
    "insert into "MOVIES_DESC_TABLE_NAME" (ID, InitialTime, NumberMovies, NumUN16s, ShowTimes) " \
    "VALUES(?, ?, ?, ?, ?)"

#define MOVIES_FETCH_TIMES \
    "select * from "MOVIES_DESC_TABLE_NAME";"

#define MOVIES_CLEAR_TIMES \
    "DELETE FROM "MOVIES_DESC_TABLE_NAME";"

#define MOVIES_INSERT_CONTROL_ROW \
    "insert into "MOVIES_CONTROL_TABLE_NAME" (TimesVer, TimesTimeStamp, DescVer, DescTimeStamp) " \
    "VALUES(%u, %u, %u, %u);"

#define MOVIES_FETCH_CONTROL_ROW \
    "select * from "MOVIES_CONTROL_TABLE_NAME" where " \
    "TimesTimeStamp > %u and DescTimeStamp > %u;"

#define MOVIES_CLEAR_CONTROL_TABLE \
    "DELETE FROM "MOVIES_CONTROL_TABLE_NAME";"

#define MOVIES_CREATE_CONTROL_TABLE \
     "create table "MOVIES_CONTROL_TABLE_NAME" (  " \
        "TimesVer int,  " \
        "TimesTimeStamp int, " \
        "DescVer int, " \
        "DescTimeStamp int " \
        "); "

#define MOVIES_INSERT_DEFAULT_CONTROL_ROW \
    "insert into "MOVIES_CONTROL_TABLE_NAME" " \
        "(TimesVer, TimesTimeStamp, DescVer, DescTimeStamp) " \
        "VALUES(0, 0, 0, 0);"

#define MOVIES_CREATE_MOVIES_TABLE \
     "create table "MOVIES_TABLE_NAME" (  " \
        "MVID int, " \
        "EnglishName text, " \
        "SpanishName text, " \
        "FrenchName text, " \
        "RunTime int, " \
        "Actors text, " \
        "EnglishSynopsis text, " \
        "SpanishSynopsis text, " \
        "FrenchSynopsis text, " \
        "NumRatings int, " \
        "Ratings blob, " \
        "primary key (MVID) " \
        " ); "

#define MOVIES_CREATE_TIMES_TABLE \
     "CREATE TABLE "MOVIES_DESC_TABLE_NAME" ( " \
        "ID int, " \
        "InitialTime int, " \
        "NumberMovies int, " \
        "NumUN16s int, " \
        "ShowTimes blob, " \
        "primary key (ID) " \
        "); "

#define MOVIES_CREATE_VERSION_TABLE \
    "create table "MOVIES_VERSION_TABLE_NAME" (  " \
        "SchemaVer int,  " \
        "DSI int  " \
        " ); "

#define MOVIES_INSERT_VERSION_ROW \
    "INSERT INTO "MOVIES_VERSION_TABLE_NAME" VALUES(%u, %u);"

// Constant used to size buffers used to perform
// DB queries.  Size is based on largest string
// size (all defines above specify field lengths)
#define MOVIES_MAX_SQL_STRING_LENGTH (600)

#define MOVIES_AMENITIES_ARRAY_SIZE (4)
#define MOVIES_MAX_MOVIE_RATINGS (16)

// Number of possible start times for a movie at a theater
#define MOVIES_THEATER_TIMES_MAX_NUM_TIME_OFFSETS (31)
#define MOVIES_THEATER_TIMES_MAX_NUM_MOVIES (32)

// Number of UN16s is equal to the number of movies times the sum
// of the max number of times plus overhead (defined privately in 
// the theater times object header)
#define THEATER_TIMES_BLOB_NUM_OVERHEAD_UN16 (5)
#define THEATER_TIMES_BLOB_MAX_NUM_UN16   \
    (MOVIES_THEATER_TIMES_MAX_NUM_MOVIES * \
      (MOVIES_THEATER_TIMES_MAX_NUM_TIME_OFFSETS + \
      THEATER_TIMES_BLOB_NUM_OVERHEAD_UN16))

// Max number of parameters among all used prepared statements
#define MOVIES_MAX_SQL_BIND_PARAMS (MOVIES_INSERT_DESC_STMT_PARAMS_COUNT)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// A version field used by each type of movie message
typedef N16 MOVIE_VERSION;
#define MOVIE_INVALID_VERSION ((MOVIE_VERSION)-1)

typedef N8 MOVIE_RATING_SYS_CODE;
#define MOVIE_RATING_INVALID_SYS_CODE ((MOVIE_RATING_SYS_CODE)-1)

typedef N8 MOVIE_RATING_CODE;
#define MOVIE_RATING_INVALID_CODE ((MOVIE_RATING_CODE)-1)

// The ID used for the state the movie is being shown in
typedef UN8 MOVIE_STATE_ID;
#define MOVIE_INVALID_STATE_ID ((MOVIE_STATE_ID)0)
#define MOVIE_STATE_ID_MAX     (96)
#define MOVIE_STATE_ID_MIN     (1)

// Enumeration specifying all the available fields
// in the db_version database relation
typedef enum db_version_fields_enum
{
    DB_VERSION_FIELD_DB_VER = 0,
    DB_VERSION_FIELD_DSI,
    DB_VERSION_BASELINE,
    DB_VERSION_MAX_FIELDS

} DB_VERSION_FIELDS_ENUM;

typedef enum db_theater_fields_enum 
{
    DB_THEATER_ID = 0,
    DB_THEATER_NAME,
    DB_THEATER_STATE,
    DB_THEATER_ADDRESS,
    DB_THEATER_CITY,
    DB_THEATER_ZIP,
    DB_THEATER_PHONE,
    DB_THEATER_AMENITIES,
    DB_THEATER_LAT,
    DB_THEATER_LON,
    DB_THEATER_MAX_FIELDS

} DB_THEATER_FIELDS_ENUM;

typedef enum db_ratings_fields_enum
{
    DB_RATINGS_SCODE = 0,
    DB_RATINGS_STEXT,
    DB_RATINGS_RCODE,
    DB_RATINGS_RTEXT,
    DB_RATINGS_MAX_FIELDS

} DB_RATINGS_FIELDS_ENUM;

typedef enum db_control_fields_enum
{
    DB_CONTROL_TIMES_VER = 0,
    DB_CONTROL_TIMES_TIMESTAMP,
    DB_CONTROL_DESC_VER,
    DB_CONTROL_DESC_TIMESTAMP,
    DB_CONTROL_MAX_FIELDS
} DB_CONTROL_FIELDS_ENUM;

typedef enum db_movie_desc_fields_enum
{
    DB_MOVIE_DESC_MVID = 0,
    DB_MOVIE_DESC_ENG_NAME,
    DB_MOVIE_DESC_SPN_NAME,
    DB_MOVIE_DESC_FR_NAME,
    DB_MOVIE_DESC_RUNTIME,
    DB_MOVIE_DESC_ACTORS,
    DB_MOVIE_DESC_ENG_SYNOPSIS,
    DB_MOVIE_DESC_SPN_SYNOPSIS,
    DB_MOVIE_DESC_FR_SYNOPSIS,
    DB_MOVIE_DESC_NUM_RATINGS,
    DB_MOVIE_DESC_RATINGS,
    DB_MOVIE_DESC_MAX_FIELDS

} DB_MOVIE_DESC_FIELDS_ENUM;

typedef enum db_movie_times_fields_enum
{
    DB_MOVIE_TIMES_ID = 0,
    DB_MOVIE_TIMES_INITIAL_TIME,
    DB_MOVIE_TIMES_NUM_MOVIES,
    DB_MOVIE_TIMES_NUM_UN16S,
    DB_MOVIE_TIMES_SHOWTIMES,
    DB_MOVIE_TIMES_MAX_FIELDS

} DB_MOVIE_TIMES_FIELDS_ENUM;

typedef enum db_movies_insert_description_columns_enum {

    DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_MVID = 0,
    DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_NAME,
    DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_RUNTIME,
    DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_ACTORS,
    DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_SYNOPSIS,
    DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_RATING,
    DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_COUNT

} DB_MOVIES_INSERT_DESCRIPTION_COLUMNS_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

typedef struct movie_multi_lang_field_struct
{
    STRING_OBJECT hEnglish;
    STRING_OBJECT hSpanish;
    STRING_OBJECT hFrench;

} MOVIE_MULTI_LANG_FIELD_STRUCT;

typedef struct movie_rating_struct
{
    MOVIE_RATING_SYSTEM_ENUM eRatingSystem;
    STRING_OBJECT hRatingSystemText;
    MOVIE_RATING_ENUM eRating;
    STRING_OBJECT hRatingText;
    MOVIE_RATING_SYS_CODE tRatingSys;
    MOVIE_RATING_CODE tRating;

} MOVIE_RATING_STRUCT;

// Structure which specifies all attributes
// of a single database version row that is
// available within the movies database
// These are the raw values that are retrieved from the database
typedef struct movies_version_row_struct
{
    UN8 un8DBVer;           // Database schema version
    DSI tDSI;               // USI
    MOVIE_VERSION tBaselineVersion;

} MOVIES_VERSION_ROW_STRUCT;

typedef struct theater_row_struct
{
    THEATER_ID tID;
    STRING_OBJECT hName;
    STRING_OBJECT hAddr;
    STRING_OBJECT hCity;
    STATE_ID tStateID;
    STRING_OBJECT hZIP;
    STRING_OBJECT hPhone;
    OSAL_FIXED_OBJECT_DATA atLatFixedData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT hLat;
    OSAL_FIXED_OBJECT_DATA atLonFixedData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT hLon;
    UN16 un16Amenities;

} THEATER_ROW_STRUCT;

typedef struct movie_ratings_row_struct
{
    MOVIE_RATING_SYS_CODE tSysCode;
    STRING_OBJECT hSysName;
    MOVIE_RATING_CODE tRatingCode;
    STRING_OBJECT hRatingText;

} MOVIE_RATINGS_ROW_STRUCT;

typedef struct theater_times_row_struct
{
    THEATER_ID tID;
    UN32 un32InitialTime;
    UN8 un8NumMovies;
    size_t tNumUN16s;
    UN16 *pun16BlobData;

} THEATER_TIMES_ROW_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/
// Union to group all the row definitions
// in a convenient manner
typedef union movies_db_row_union
{
    MOVIES_VERSION_ROW_STRUCT sVersion;
    THEATER_ROW_STRUCT sTheater;

} MOVIES_DB_ROW_UNION;

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif // MOVIES_DB_CONSTANTS_H_
