/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:MOVIE implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_api_debug.h"
#include "sms_obj.h"
#include "string_obj.h"

#include "movie_obj.h"
#include "_movie_obj.h"

static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tID
*
* Returns the MOVIE_ID for this object
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid MOVIE_ID on success
*   A MOVIE_INVALID_ID on failrue
*
*****************************************************************************/
static MOVIE_ID tID (
    MOVIE_OBJECT hMovie
        )
{
    BOOLEAN bValid;
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;

    do
    {
        // Determine if this is a valid object
        bValid = SMSO_bValid((SMS_OBJECT)hMovie);

        if (bValid == FALSE)
        {
            break;
        }

        return psObj->tID;

    } while (FALSE);

    return MOVIE_INVALID_ID;
}

/*****************************************************************************
*
*   hName
*
* Returns the Name for the MOVIE_OBJECT as a STRING_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid STRING_OBJECT on success
*   A STRING_INVALID_OBJECT on failrue
*
*****************************************************************************/
static STRING_OBJECT hName (
    MOVIE_OBJECT hMovie
        )
{
    SMS_LANGUAGE_ENUM eLanguage = SMS_LANGUAGE_ENGLISH;

    // Get the name in English
    return hNameInLanguage(hMovie, &eLanguage);
}

/*****************************************************************************
*
*   hNameInLanguage
*
* Returns the Name (in the requested language) for the MOVIE_OBJECT as a 
* STRING_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*   *peLanguage - A valid SMS_LANGUAGE_ENUM used to indicate which language
*       was requested.  Populated with the actual language found
*
* Outputs:
*
*   A valid STRING_OBJECT on success
*   A STRING_INVALID_OBJECT on failure
*
*****************************************************************************/
static STRING_OBJECT hNameInLanguage (
    MOVIE_OBJECT hMovie,
    SMS_LANGUAGE_ENUM *peLanguage
        )
{
    BOOLEAN bValid;
    STRING_OBJECT hName = STRING_INVALID_OBJECT;

    // Validate input
    if ((SMS_LANGUAGE_ENUM *)NULL == peLanguage)
    {
        return STRING_INVALID_OBJECT;
    }

    // Determine if this is a valid object
    bValid = SMSO_bValid((SMS_OBJECT)hMovie);

    if (TRUE == bValid)
    {
        MOVIE_OBJECT_STRUCT *psObj =
            (MOVIE_OBJECT_STRUCT *)hMovie;
        
        // If they asked for english give 'em english
        if (SMS_LANGUAGE_ENGLISH == *peLanguage)
        {
            hName = psObj->sName.hEnglish;
        }
        // Did they ask for French?
        else if (SMS_LANGUAGE_FRENCH == *peLanguage)
        {
            // Do we have anything here?
            if (STRING_INVALID_OBJECT != psObj->sName.hFrench)
            {
                // Yep, give it to them
                hName = psObj->sName.hFrench;
            }
            else
            {
                // No, only have english
                hName = psObj->sName.hEnglish;
                *peLanguage = SMS_LANGUAGE_ENGLISH;
            }
        }
        // Did they ask for Spanish?
        else if (SMS_LANGUAGE_SPANISH == *peLanguage)
        {
            // Do we have anything here?
            if (STRING_INVALID_OBJECT != psObj->sName.hSpanish)
            {
                // Yep, give it to them
                hName = psObj->sName.hSpanish;
            }
            else
            {
                // No, only have english
                hName = psObj->sName.hEnglish;
                *peLanguage = SMS_LANGUAGE_ENGLISH;
            }
        }
        else
        {
            // We don't know what the caller wants
            hName = STRING_INVALID_OBJECT;
            *peLanguage = SMS_INVALID_LANGUAGE;
        }
    }

    return hName;
}

/*****************************************************************************
*
*   un8RunTime
*
* Returns the running time as minutes for the MOVIE_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid UN8 on success
*   A UN8_MAX on failure
*
*****************************************************************************/
static UN8 un8RunTime (
    MOVIE_OBJECT hMovie
        )
{
    BOOLEAN bValid;
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;

    do
    {
        // Determine if this is a valid object
        bValid = SMSO_bValid((SMS_OBJECT)hMovie);

        if (bValid == FALSE)
        {
            break;
        }

        return psObj->un8RunTime;

    } while (FALSE);

    return UN8_MAX;
}

/*****************************************************************************
*
*   hActors
*
* Returns the actors for the MOVIE_OBJECT as a STRING_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid STRING_OBJECT on success
*   A STRING_INVALID_OBJECT on failrue
*
*****************************************************************************/
static STRING_OBJECT hActors (
    MOVIE_OBJECT hMovie
        )
{
    BOOLEAN bValid;
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;

    do
    {
        // Determine if this is a valid object
        bValid = SMSO_bValid((SMS_OBJECT)hMovie);

        if (bValid == FALSE)
        {
            break;
        }

        return psObj->hActors;

    } while (FALSE);

    return STRING_INVALID_OBJECT;
}

/*****************************************************************************
*
*   hSynopsis
*
* Returns the synopsis for the MOVIE_OBJECT as a STRING_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid STRING_OBJECT on success
*   A STRING_INVALID_OBJECT on failure
*
*****************************************************************************/
static STRING_OBJECT hSynopsis (
    MOVIE_OBJECT hMovie
        )
{
    SMS_LANGUAGE_ENUM eLanguage = SMS_LANGUAGE_ENGLISH;

    // Get the synopsis in English
    return hSynopsisInLanguage(hMovie, &eLanguage);
}

/*****************************************************************************
*
*   hSynopsisInLanguage
*
* Returns the synopsis (in the requested language) for the MOVIE_OBJECT as a 
* STRING_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*   *peLanguage - A valid SMS_LANGUAGE_ENUM used to indicate which language
*       was requested.  Populated with the actual language found
*
* Outputs:
*
*   A valid STRING_OBJECT on success
*   A STRING_INVALID_OBJECT on failure
*
*****************************************************************************/
static STRING_OBJECT hSynopsisInLanguage (
    MOVIE_OBJECT hMovie,
    SMS_LANGUAGE_ENUM *peLanguage
        )
{
    BOOLEAN bValid;
    STRING_OBJECT hSynopsis = STRING_INVALID_OBJECT;

    // Validate input
    if ((SMS_LANGUAGE_ENUM *)NULL == peLanguage)
    {
        return STRING_INVALID_OBJECT;
    }

    // Determine if this is a valid object
    bValid = SMSO_bValid((SMS_OBJECT)hMovie);

    if (TRUE == bValid)
    {
        MOVIE_OBJECT_STRUCT *psObj =
            (MOVIE_OBJECT_STRUCT *)hMovie;
        
        // If they asked for english give 'em english
        if (SMS_LANGUAGE_ENGLISH == *peLanguage)
        {
            hSynopsis = psObj->sSynopsis.hEnglish;
        }
        // Did they ask for French?
        else if (SMS_LANGUAGE_FRENCH == *peLanguage)
        {
            // Do we have anything here?
            if (STRING_INVALID_OBJECT != psObj->sSynopsis.hFrench)
            {
                // Yep, give it to them
                hSynopsis = psObj->sSynopsis.hFrench;
            }
            else
            {
                // No, only have english
                hSynopsis = psObj->sSynopsis.hEnglish;
                *peLanguage = SMS_LANGUAGE_ENGLISH;
            }
        }
        // Did they ask for Spanish?
        else if (SMS_LANGUAGE_SPANISH == *peLanguage)
        {
            // Do we have anything here?
            if (STRING_INVALID_OBJECT != psObj->sSynopsis.hSpanish)
            {
                // Yep, give it to them
                hSynopsis = psObj->sSynopsis.hSpanish;
            }
            else
            {
                // No, only have english
                hSynopsis = psObj->sSynopsis.hEnglish;
                *peLanguage = SMS_LANGUAGE_ENGLISH;
            }
        }
        else
        {
            // We don't know what the caller wants
            hSynopsis = STRING_INVALID_OBJECT;
            *peLanguage = SMS_INVALID_LANGUAGE;
        }
    }

    return hSynopsis;
}

/*****************************************************************************
*
*   eRating
*
* Returns the rating for the MOVIE_OBJECT as a MOVIE_RATING_ENUM.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid MOVIE_RATING_ENUM on success
*   A MOVIE_RATING_ERROR on failrue
*
*****************************************************************************/
static MOVIE_RATING_ENUM eRating (
    MOVIE_OBJECT hMovie
        )
{
    BOOLEAN bValid;
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;

    do
    {
        // Determine if this is a valid object
        bValid = SMSO_bValid((SMS_OBJECT)hMovie);

        if (bValid == FALSE)
        {
            break;
        }

        // Do we have a rating exception to report?
        if ((psObj->bExceptionActive == TRUE) && 
            (psObj->eRatingExceptionSystem == MOVIE_RATING_SYSTEM_US_MPAA))
        {
            return psObj->eRatingException;
        }

        // Provide the US-MPAA rating
        return (psObj->psUSMPAA != NULL) ?
                    psObj->psUSMPAA->eRating : MOVIE_RATING_UNKNOWN;

    } while (FALSE);

    return MOVIE_RATING_ERROR;
}

/*****************************************************************************
*
*   hRatingText
*
* Returns the rating text for the MOVIE_OBJECT as a STRING_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid STRING_OBJECT on success
*   A STRING_INVALID_OBJECT on failrue
*
*****************************************************************************/
static STRING_OBJECT hRatingText (
    MOVIE_OBJECT hMovie
        )
{
    BOOLEAN bValid;
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;

    do
    {
        // Determine if this is a valid object
        bValid = SMSO_bValid((SMS_OBJECT)hMovie);

        if (bValid == FALSE)
        {
            break;
        }

        // Do we have a rating exception to report?
        if ((psObj->bExceptionActive == TRUE) && 
            (psObj->eRatingExceptionSystem == MOVIE_RATING_SYSTEM_US_MPAA))
        {
            return psObj->hRatingExceptionText;
        }

        // Provide the US-MPAA rating text
        if (psObj->psUSMPAA != NULL)
        {
            return psObj->psUSMPAA->hRatingText;
        }

    } while (FALSE);

    return STRING_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eIterateRatings
*
* Returns the rating text for the MOVIE_OBJECT as a STRING_OBJECT.
*
* Inputs:
*
*   hMovie - A handle to a valid MOVIE_OBJECT
*
* Outputs:
*
*   A valid STRING_OBJECT on success
*   A STRING_INVALID_OBJECT on failrue
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateRatings (
    MOVIE_OBJECT hMovie,
    MOVIE_RATINGS_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    BOOLEAN bValid;
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        MOVIE_RATING_ITERATOR_STRUCT sIterator;

        // Populate the iterator now
        sIterator.bIterator = bIterator;
        sIterator.pvIteratorArg = pvIteratorArg;
        sIterator.psObj = psObj;

        // Check iterate
        if ((MOVIE_RATINGS_ITERATOR)NULL == bIterator)
        {
            // Bad input
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Determine if this is a valid object
        bValid = SMSO_bValid((SMS_OBJECT)hMovie);
        if (FALSE == bValid)
        {
            // Bad input
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Iterate the list of ratings using 
        // and call the provided callback
        eOsalReturnCode = OSAL.eLinkedListIterate(
            psObj->hRatings,
            (OSAL_LL_ITERATOR_HANDLER)bIterateRatings,
            (void *)&sIterator);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            // General error code already set
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a MOVIE's contents to a specified file or device.
* This is mainly helpful during debugging of MOVIE's but could be used by
* an application for any reason.
*
* Inputs:
*
*   hMovie - The MOVIE handle the caller wishes to write.
*   psFile - The device to write the MOVIE contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    MOVIE_OBJECT hMovie,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;
    N32 n32Return = 0, n32Temp;
    BOOLEAN bValid;

    // Determine if the handle is valid
    bValid = SMSO_bValid((SMS_OBJECT)hMovie);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bValid == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }
    switch (eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            n32Return += fprintf(psFile, "MOVIE: ");
            n32Temp = STRING.n32FWrite(psObj->sName.hEnglish, psFile);
            if(n32Temp > 0)
            {
                n32Return += n32Temp;
            }

            n32Return += fprintf(psFile, "(MVID:%d, 0x%p)\n",
                psObj->tID, psObj);
        }
        break;
        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        {
            MOVIE_RATING_PRINT_STRUCT sPrint;
            SMS_LANGUAGE_ENUM eLanguage;
            STRING_OBJECT hString;

            // Print MOVIE information header
            n32Return += fprintf(psFile, "MOVIE:\n\thMovie = 0x%p\n",
                psObj);

            n32Return += fprintf(psFile, "\ttID = %d\n",
                psObj->tID);

            // English name
            eLanguage = SMS_LANGUAGE_ENGLISH;
            hString = MOVIE.hNameInLanguage(hMovie, &eLanguage);

            if (hString == STRING_INVALID_OBJECT)
            {
                n32Return += fprintf(psFile, "\tNo English Name\n");
            }
            else
            {
                n32Return += fprintf(psFile, "\tEnglish Name = ");
                n32Temp = STRING.n32FWrite(hString, psFile);
                if(n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
                n32Return += fprintf(psFile, "\n");
            }

            // Spanish name
            eLanguage = SMS_LANGUAGE_SPANISH;
            hString = MOVIE.hNameInLanguage(hMovie, &eLanguage);

            if (eLanguage != SMS_LANGUAGE_SPANISH)
            {
                n32Return += fprintf(psFile, "\tNo Spanish Name\n");
            }
            else
            {
                n32Return += fprintf(psFile, "\tSpanish Name = ");
                n32Temp = STRING.n32FWrite(hString, psFile);
                if(n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
                n32Return += fprintf(psFile, "\n");
            }

            // French name
            eLanguage = SMS_LANGUAGE_FRENCH;
            hString = MOVIE.hNameInLanguage(hMovie, &eLanguage);

            if (eLanguage != SMS_LANGUAGE_FRENCH)
            {
                n32Return += fprintf(psFile, "\tNo French Name\n");
            }
            else
            {
                n32Return += fprintf(psFile, "\tFrench Name = ");
                n32Temp = STRING.n32FWrite(hString, psFile);
                if(n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
                n32Return += fprintf(psFile, "\n");
            }

            n32Return += fprintf(psFile, "\tun8RunTime = %d\n",
                psObj->un8RunTime);

            n32Return += fprintf(psFile, "\thActors = ");
            n32Temp = STRING.n32FWrite(psObj->hActors, psFile);
            if(n32Temp > 0)
            {
                n32Return += n32Temp;
            }
            n32Return += fprintf(psFile, "\n");

            // English synopsis
            eLanguage = SMS_LANGUAGE_ENGLISH;
            hString = MOVIE.hSynopsisInLanguage(hMovie, &eLanguage);

            if (hString == STRING_INVALID_OBJECT)
            {
                n32Return += fprintf(psFile, "\tNo English Name\n");
            }
            else
            {
                n32Return += fprintf(psFile, "\tEnglish Synopsis = ");
                n32Temp = STRING.n32FWrite(hString, psFile);
                if(n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
                n32Return += fprintf(psFile, "\n");
            }

            // Spanish synopsis
            eLanguage = SMS_LANGUAGE_SPANISH;
            hString = MOVIE.hSynopsisInLanguage(hMovie, &eLanguage);

            if (eLanguage != SMS_LANGUAGE_SPANISH)
            {
                n32Return += fprintf(psFile, "\tNo Spanish Synopsis\n");
            }
            else
            {
                n32Return += fprintf(psFile, "\tSpanish Synopsis = ");
                n32Temp = STRING.n32FWrite(hString, psFile);
                if(n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
                n32Return += fprintf(psFile, "\n");
            }

            // French synopsis
            eLanguage = SMS_LANGUAGE_FRENCH;
            hString = MOVIE.hSynopsisInLanguage(hMovie, &eLanguage);

            if (eLanguage != SMS_LANGUAGE_FRENCH)
            {
                n32Return += fprintf(psFile, "\tNo French Synopsis\n");
            }
            else
            {
                n32Return += fprintf(psFile, "\tFrench Synopsis = ");
                n32Temp = STRING.n32FWrite(hString, psFile);
                if(n32Temp > 0)
                {
                    n32Return += n32Temp;
                }
                n32Return += fprintf(psFile, "\n");
            }

            // Initialize ratings print struct
            sPrint.n32NumCharsWritten = 0;
            sPrint.psFile = psFile;

            // Print ratings now
            eIterateRatings(
                hMovie,
                (MOVIE_RATINGS_ITERATOR)bPrintRatingEntry,
                &sPrint);

            n32Return += sPrint.n32NumCharsWritten;
            n32Return += fprintf(psFile, "\n");
        }
        break;
        default:
        {
            n32Return = EOF;
        }
        break;
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   MOVIE_hCreate
*
* This object interface method is used by the caller to create a MOVIE
* object with data received from the movie times protocl.
*
* This function assumes that the STRING_OBJECTs were created for our use
* by hMoviesService, so there is no need to duplicate
*
* Inputs:
*
*   hMoviesService - The data service object that is the parent to this object
*   tID - The service-defined ID for this movie
*   hName - The name of the movie
*   un8RunTime - The nubmer of minutes this movie runs for
*   hActors - Text containing the actors in this movie
*   hSynopis - The text description for the movie
*   eRating - The rating as an enum, if available
*   hRatingText - The rating as text if available
*
* Outputs:
*
*   A handle to a MOVIE object on success
*   A MOVIE_INVALID_OBJECT on failrue
*
*****************************************************************************/
MOVIE_OBJECT MOVIE_hCreate (
    SMS_OBJECT hParent,
    MOVIE_ID tID,
    MOVIE_MULTI_LANG_FIELD_STRUCT *psName,
    UN8 un8RunTime,
    STRING_OBJECT hActors,
    MOVIE_MULTI_LANG_FIELD_STRUCT *psSynopsis,
    size_t tNumRatings,
    MOVIE_RATING_STRUCT *psRatings
        )
{
    MOVIE_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hRatings = OSAL_INVALID_OBJECT_HDL;

    // Create an instance of the MOVIE object
    psObj = (MOVIE_OBJECT_STRUCT *)
        SMSO_hCreate(
            MOVIE_OBJECT_NAME,
            sizeof(MOVIE_OBJECT_STRUCT),
            hParent,
            FALSE);

    if(psObj == NULL)
    {
        // Error!
        return MOVIE_INVALID_OBJECT;
    }

    do
    {
        
        OSAL_RETURN_CODE_ENUM eReturnCode;
        size_t tIndex;
        MOVIE_RATING_STRUCT *psNewEntry, *psCurEntry;
        BOOLEAN bSuccess = TRUE;

        // This is not yet known
        psObj->psUSMPAA = (MOVIE_RATING_STRUCT *)NULL;

        // Create the linked list of ratings assigned
        // to this movie
        eReturnCode = OSAL.eLinkedListCreate(
            &hRatings,
            MOVIE_OBJECT_NAME":Ratings",
            (OSAL_LL_COMPARE_HANDLER)n16CompareRatingEntries,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Iterate through all the ratings provided
        // and store them
        for (tIndex = 0; tIndex < tNumRatings; tIndex++)
        {
            // Get the current entry
            psCurEntry = &psRatings[tIndex];

            // Allocate memory for the new entry
            psNewEntry = (MOVIE_RATING_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    MOVIE_OBJECT_NAME":RatingEntry",
                    sizeof(MOVIE_RATING_STRUCT), 
                    FALSE);

            // Did we get anything?
            if (psNewEntry == (MOVIE_RATING_STRUCT *)NULL)
            {
                // No, stop here
                bSuccess = FALSE;
                break;
            }

            // Copy the contents of this entry
            bSuccess = OSAL.bMemCpy(
                psNewEntry, psCurEntry, sizeof(MOVIE_RATING_STRUCT));
            if (bSuccess == FALSE)
            {
                // Free this memory
                OSAL.vLinkedListMemoryFree(psNewEntry);

                // Stop here
                break;
            }

            // Add this entry to the list
            eReturnCode = OSAL.eLinkedListAdd(
                hRatings, 
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void *)psNewEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // Free this memory
                OSAL.vLinkedListMemoryFree(psNewEntry);

                // Get outta here
                bSuccess = FALSE;
                break;
            }

            // Is this the US-MPAA entry?
            if (psNewEntry->eRatingSystem == MOVIE_RATING_SYSTEM_US_MPAA)
            {
                // Save this since it's used the most often
                psObj->psUSMPAA = psNewEntry;
            }
        }

        // Copy the multi-language fields now
        bSuccess = OSAL.bMemCpy(
            &psObj->sName, 
            (const void *)psName, 
            sizeof(MOVIE_MULTI_LANG_FIELD_STRUCT));
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = OSAL.bMemCpy(
            &psObj->sSynopsis, 
            (const void *)psSynopsis, 
            sizeof(MOVIE_MULTI_LANG_FIELD_STRUCT));
        if (bSuccess == FALSE)
        {
            break;
        }

        // Initialize object per inputs
        // We are holding onto the strings since they
        // should be created by our owner for our use
        psObj->tID = tID;
        psObj->un8RunTime = un8RunTime;
        psObj->hActors = hActors;
        psObj->hRatings = hRatings;

        return (MOVIE_OBJECT)psObj;

    } while (FALSE);

    // Free all of the entries
    if (hRatings != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        
        // Clear the list of any entries
        eReturnCode = OSAL.eLinkedListRemoveAll(
            hRatings, OSAL.vLinkedListMemoryFree);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    MOVIE_OBJECT_NAME ": unable to remove all ratings!" );
        }

        eReturnCode = OSAL.eLinkedListDelete(hRatings);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    MOVIE_OBJECT_NAME ": unable to delete ratings list!" );
        }

    }

    return MOVIE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   MOVIE_hCreateDummy
*
*****************************************************************************/
MOVIE_OBJECT MOVIE_hCreateDummy (
    SMS_OBJECT hParent
        )
{
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)MOVIE_INVALID_OBJECT;

    // Create an instance of the MOVIE object
    psObj = (MOVIE_OBJECT_STRUCT *)
        SMSO_hCreate(
            MOVIE_OBJECT_NAME,
            sizeof(MOVIE_OBJECT_STRUCT),
            hParent,
            FALSE);

    return (MOVIE_OBJECT)psObj;
}

/*****************************************************************************
*
*   MOVIE_bUpdateDummyID
*
*****************************************************************************/
BOOLEAN MOVIE_bUpdateDummyID (
    MOVIE_OBJECT hDummy,
    MOVIE_ID tID
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDummy);
    if (TRUE == bOwner)
    {
        MOVIE_OBJECT_STRUCT *psObj = 
            (MOVIE_OBJECT_STRUCT *)hDummy;

        // Update the ID 
        psObj->tID = tID;
    }

    return bOwner;
}

/*****************************************************************************
*
*   MOVIE_bGetMultiValueFields
*
*****************************************************************************/
BOOLEAN MOVIE_bGetMultiValueFields (
    MOVIE_OBJECT hMovie,
    MOVIE_MULTI_LANG_FIELD_STRUCT **ppsName,
    MOVIE_MULTI_LANG_FIELD_STRUCT **ppsSynopsis,
    size_t *ptNumRatings,
    MOVIE_RATING_STRUCT *pasRatings
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if ((ppsName == NULL) || 
        (ppsSynopsis == NULL) || 
        (ptNumRatings == NULL) ||
        (pasRatings == NULL))
    {
        return FALSE;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hMovie);
    if (TRUE == bOwner)
    {
        MOVIE_OBJECT_STRUCT *psObj = 
            (MOVIE_OBJECT_STRUCT *)hMovie;
        OSAL_LINKED_LIST_ENTRY hThisRatingEntry;
        MOVIE_RATING_STRUCT *psThisRating;
        size_t tMaxRatings = *ptNumRatings, tIndex = 0;

        // Give some direct pointers for these
        *ppsName = &psObj->sName;
        *ppsSynopsis = &psObj->sSynopsis;

        // Extract the ratings now
        // Grab the first rating entry
        hThisRatingEntry = OSAL.hLinkedListFirst(
            psObj->hRatings, (void **)&psThisRating);

        while ((hThisRatingEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
               (tIndex < tMaxRatings))
        {
            // Copy the rating data for this entry
            OSAL.bMemCpy(
                &pasRatings[tIndex++], psThisRating, 
                sizeof(MOVIE_RATING_STRUCT));

            hThisRatingEntry = OSAL.hLinkedListNext(
                hThisRatingEntry, (void **)&psThisRating);
        }

        // Save the number of ratings 
        *ptNumRatings = tIndex;
    }

    return bOwner;
}

/*****************************************************************************
*
*   MOVIE_vSetException
*
*****************************************************************************/
void MOVIE_vSetException (
    MOVIE_OBJECT hMovie,
    MOVIE_RATING_SYSTEM_ENUM eRatingSytem,
    MOVIE_RATING_ENUM eRating,
    STRING_OBJECT hRatingText
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hMovie);
    if (TRUE == bOwner)
    {
        MOVIE_OBJECT_STRUCT *psObj = 
            (MOVIE_OBJECT_STRUCT *)hMovie;

        // Set the exception
        psObj->bExceptionActive = TRUE;
        psObj->eRatingExceptionSystem = eRatingSytem;
        psObj->eRatingException = eRating;
        psObj->hRatingExceptionText = hRatingText;
    }

    return;
}

/*****************************************************************************
*
*   MOVIE_vClearException
*
*****************************************************************************/
void MOVIE_vClearException (
    MOVIE_OBJECT hMovie
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hMovie);
    if (TRUE == bOwner)
    {
        MOVIE_OBJECT_STRUCT *psObj = 
            (MOVIE_OBJECT_STRUCT *)hMovie;

        // Clear the exception
        psObj->bExceptionActive = FALSE;
    }

    return;
}

/*****************************************************************************
*
*   MOVIE_n16CompareID
*
*****************************************************************************/
N16 MOVIE_n16CompareID (
    MOVIE_OBJECT hMovie1,
    MOVIE_OBJECT hMovie2
        )
{
    MOVIE_OBJECT_STRUCT *psObj1 =
        (MOVIE_OBJECT_STRUCT *)hMovie1;
    MOVIE_OBJECT_STRUCT *psObj2 =
        (MOVIE_OBJECT_STRUCT *)hMovie2;
    BOOLEAN bValid;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hMovie1);
    if(bValid == FALSE)
    {
        // Error!
        return N16_MIN;
    }

    bValid = SMSO_bValid((SMS_OBJECT)hMovie2);
    if(bValid == FALSE)
    {
        // Error!
        return N16_MIN;
    }

    // Compare the IDs
    return psObj1->tID - psObj2->tID;
}

/*****************************************************************************
*
*   MOVIE_vDestroy
*
* This object interface method is used by the caller to destroy the specified
* MOVIE object and all members of the object
*

* Inputs:
*
*   hMovie - A handle to a valid MOVIE object that the caller wants
*                 to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
void MOVIE_vDestroy (
    MOVIE_OBJECT hMovie
        )
{
    BOOLEAN bValid;
    MOVIE_OBJECT_STRUCT *psObj =
        (MOVIE_OBJECT_STRUCT *)hMovie;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hMovie);
    if(bValid == FALSE)
    {
        // Error!
        return;
    }

    STRING_vDestroy(psObj->hActors);

    // Multi-language name
    STRING_vDestroy(psObj->sName.hEnglish);
    STRING_vDestroy(psObj->sName.hFrench);
    STRING_vDestroy(psObj->sName.hSpanish);
    
    // Multi-language synopsis
    STRING_vDestroy(psObj->sSynopsis.hEnglish);
    STRING_vDestroy(psObj->sSynopsis.hFrench);
    STRING_vDestroy(psObj->sSynopsis.hSpanish);

    // Clear the US-MPAA rating
    psObj->psUSMPAA = (MOVIE_RATING_STRUCT *)NULL;

    // Only attempt to remove all ratings entries if we actually
    // have a ratings list.

    if ( OSAL_INVALID_OBJECT_HDL != psObj->hRatings )
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->hRatings, OSAL.vLinkedListMemoryFree);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    MOVIE_OBJECT_NAME ": unable to remove all ratings!" );
        }

        eReturnCode = OSAL.eLinkedListDelete(psObj->hRatings);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    MOVIE_OBJECT_NAME ": unable to delete ratings list!" );
        }
    }

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hMovie);

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pacRatingSysEnumToString
*
*****************************************************************************/
static const char *pacRatingSysEnumToString (
    MOVIE_RATING_SYSTEM_ENUM eRatingSys
        )
{
    const char *pacReturnString = 
        MACRO_TO_STRING(MOVIE_RATING_SYSTEM_UNKNOWN);

    switch (eRatingSys)
    {
        case MOVIE_RATING_SYSTEM_US_MPAA:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_SYSTEM_US_MPAA);
        }
        break;

        case MOVIE_RATING_SYSTEM_CANADA1:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_SYSTEM_CANADA1);
        }
        break;

        case MOVIE_RATING_SYSTEM_CANADA2:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_SYSTEM_CANADA2);
        }
        break;

        case MOVIE_RATING_SYSTEM_QUEBEC:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_SYSTEM_QUEBEC);
        }
        break;

        case MOVIE_RATING_SYSTEM_MEXICO:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_SYSTEM_MEXICO);
        }
        break;

        case MOVIE_RATING_SYSTEM_UNKNOWN:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_SYSTEM_UNKNOWN);
        }
        break;

    }

    return pacReturnString;
}

/*****************************************************************************
*
*   pacRatingEnumToString
*
*****************************************************************************/
static const char *pacRatingEnumToString (
    MOVIE_RATING_ENUM eRating
        )
{
    const char *pacReturnString;

    switch (eRating)
    {
        /**** US-MPAA ****/
        case MOVIE_RATING_MPAA_G:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MPAA_G);
        }
        break;

        case MOVIE_RATING_MPAA_PG:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MPAA_PG);
        }
        break;

        case MOVIE_RATING_MPAA_PG13:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MPAA_PG13);
        }
        break;

        case MOVIE_RATING_MPAA_R:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MPAA_R);
        }
        break;

        case MOVIE_RATING_MPAA_NC17:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MPAA_NC17);
        }
        break;

        case MOVIE_RATING_MPAA_UNRATED:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MPAA_UNRATED);
        }
        break;

        /**** Canada 1 ****/
        case MOVIE_RATING_CANADA1_G:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA1_G);
        }
        break;

        case MOVIE_RATING_CANADA1_PG:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA1_PG);
        }
        break;

        case MOVIE_RATING_CANADA1_14A:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA1_14A);
        }
        break;

        case MOVIE_RATING_CANADA1_18A:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA1_18A);
        }
        break;

        case MOVIE_RATING_CANADA1_R:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA1_R);
        }
        break;

        case MOVIE_RATING_CANADA1_UNRATED:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA1_UNRATED);
        }
        break;

        /**** Canada 2 ****/
        case MOVIE_RATING_CANADA2_G:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA2_G);
        }
        break;

        case MOVIE_RATING_CANADA2_PG:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA2_PG);
        }
        break;

        case MOVIE_RATING_CANADA2_14:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA2_14);
        }
        break;

        case MOVIE_RATING_CANADA2_18:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA2_18);
        }
        break;

        case MOVIE_RATING_CANADA2_E:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA2_E);
        }
        break;

        case MOVIE_RATING_CANADA2_R:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA2_R);
        }
        break;

        case MOVIE_RATING_CANADA2_UNRATED:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_CANADA2_UNRATED);
        }
        break;

        /**** Quebecois ****/
        case MOVIE_RATING_QUEBEC_G:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_QUEBEC_G);
        }
        break;

        case MOVIE_RATING_QUEBEC_13PLUS:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_QUEBEC_13PLUS);
        }
        break;

        case MOVIE_RATING_QUEBEC_16PLUS:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_QUEBEC_16PLUS);
        }
        break;

        case MOVIE_RATING_QUEBEC_18PLUS:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_QUEBEC_18PLUS);
        }
        break;

        case MOVIE_RATING_QUEBEC_NONCLASS:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_QUEBEC_NONCLASS);
        }
        break;

        /**** Mexico ****/
        case MOVIE_RATING_MEXICO_SC:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MEXICO_SC);
        }
        break;

        case MOVIE_RATING_MEXICO_AA:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MEXICO_AA);
        }
        break;

        case MOVIE_RATING_MEXICO_A:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MEXICO_A);
        }
        break;

        case MOVIE_RATING_MEXICO_B:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MEXICO_B);
        }
        break;

        case MOVIE_RATING_MEXICO_B15:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MEXICO_B15);
        }
        break;

        case MOVIE_RATING_MEXICO_C:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MEXICO_C);
        }
        break;

        case MOVIE_RATING_MEXICO_D:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_MEXICO_D);
        }
        break;

        case MOVIE_RATING_UNKNOWN:
        default:
        {
            pacReturnString =
                MACRO_TO_STRING(MOVIE_RATING_UNKNOWN);
        }
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   bIterateRatings
*
*****************************************************************************/
static BOOLEAN bIterateRatings (
    MOVIE_RATING_STRUCT *psRating,
    MOVIE_RATING_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    // Validate inputs
    if (((MOVIE_RATING_STRUCT *)NULL != psRating) &&
        ((MOVIE_RATING_ITERATOR_STRUCT *)NULL != psIterator))
    {
        // Initialize to the rating value stored
        MOVIE_RATING_ENUM eRating = psRating->eRating;
        STRING_OBJECT hRatingText = psRating->hRatingText;

        // Do we have a rating exception to report?
        if ((psIterator->psObj->bExceptionActive == TRUE) && 
            (psIterator->psObj->eRatingExceptionSystem == 
             psRating->eRatingSystem))
        {
            // Use this rating
            eRating = psIterator->psObj->eRatingException;
            hRatingText = psIterator->psObj->hRatingExceptionText;
        }

        // Call the user iterator now
        bContinue = psIterator->bIterator(
            psRating->eRatingSystem,
            psRating->hRatingSystemText,
            eRating,
            hRatingText,
            psIterator->pvIteratorArg);
    }

    return bContinue;
}

/*****************************************************************************
*
*   n16CompareRatingEntries
*
*****************************************************************************/
static N16 n16CompareRatingEntries (
    MOVIE_RATING_STRUCT *psRating1,
    MOVIE_RATING_STRUCT *psRating2
        )
{
    N16 n16Return = N16_MIN;

    // Validate inputs
    if (((MOVIE_RATING_STRUCT *)NULL != psRating1) &&
        ((MOVIE_RATING_STRUCT *)NULL != psRating2))
    {
        // Compare by rating system
        n16Return = COMPARE(
            psRating1->eRatingSystem, 
            psRating2->eRatingSystem);

        if (0 == n16Return)
        {
            // Compare rating
            n16Return = COMPARE(
                psRating1->eRating,
                psRating2->eRating);
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   bPrintRatingEntry
*
*****************************************************************************/
static BOOLEAN bPrintRatingEntry (
    MOVIE_RATING_SYSTEM_ENUM eRatingSystem,
    STRING_OBJECT hRatingSystemName,
    MOVIE_RATING_ENUM eRating,
    STRING_OBJECT hRatingText,
    MOVIE_RATING_PRINT_STRUCT *psPrint
        )
{
    N32 n32Written;

    // Validate inputs
    if ((MOVIE_RATING_PRINT_STRUCT *)NULL == psPrint)
    {
        return FALSE;
    }

    // Write the rating system values
    psPrint->n32NumCharsWritten += fprintf(psPrint->psFile, "\teRatingSystem = %s\n",
        pacRatingSysEnumToString(eRatingSystem));

    psPrint->n32NumCharsWritten += fprintf(psPrint->psFile, "\thRatingSystemText = ");
    n32Written = STRING.n32FWrite(hRatingSystemName, psPrint->psFile);
    if(n32Written > 0)
    {
        psPrint->n32NumCharsWritten += n32Written;
    }

    // Write the rating values
    psPrint->n32NumCharsWritten += fprintf(psPrint->psFile, "\n\teRating = %s\n",
        pacRatingEnumToString(eRating));

    psPrint->n32NumCharsWritten += fprintf(psPrint->psFile, "\thRatingText = ");
    n32Written = STRING.n32FWrite(hRatingText, psPrint->psFile);
    if(n32Written > 0)
    {
        psPrint->n32NumCharsWritten += n32Written;
    }
    psPrint->n32NumCharsWritten += fprintf(psPrint->psFile, "\n");

    // Keep going
    return TRUE;
}
