/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*             Licensed Materials - Property of Sirius XM Radio, Inc.         */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object: MAPS_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <ctype.h>
#include "standard.h"
#include "osal.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sql_interface_obj.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "location_obj.h"
#include "db_util.h"

#include "maps_interface.h"
#include "_maps_mgr_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   hStart
 *
 *   This function will create all the basics needed for this service to
 *   operate.  However, all initial processing to actually get this service
 *   running is done at a later time.
 *
 *   Inputs:
 *       pacSRHDriverName - A constant string which represents the exact name of
 *           the SRH driver used by the application.
 *       pacOutputDir - Path to the output folder. 
 *       tEventRequestMask - A bit mask indicating which events the caller
 *           wishes to receive.
 *       vEventCallback - A callback function provided by the caller which will
 *           be called whenever one or more of the selected events occurs.
 *       pvAppEventCallbackArg - The caller specific data that will be provided
 *          to vEventCallback.
 *       tOptions - The options the caller wishes to utilize in order to
 *          configure the behavior of the service.
 *
 *   Outputs:
 *       MAPS_SERVICE_OBJECT type object.
 *
 *****************************************************************************/
static MAPS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    const char *pacOutputDir,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    MAPS_MGR_OBJECT_STRUCT *psObj =
            (MAPS_MGR_OBJECT_STRUCT*)MAPS_SERVICE_INVALID_OBJECT;
    BOOLEAN bOk;
    DATASERVICE_CREATE_STRUCT sCreate;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    do
    {
        bOk = DATASERVICE_IMPL_bProcessOptions(
            MAPS_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
        // restore stack (push)
        if (bOk == FALSE)
        {
            // Bad options!
            break;
        }

        // Populate our data service creation structure
        DATASERVICE_IMPL_vInitCreateStruct( &sCreate );
        sCreate.pacSRHDriverName = pacSRHDriverName;
        sCreate.pacServiceObjectName = MAPS_MGR_OBJECT_NAME;
        sCreate.tServiceObjectSize = sizeof(MAPS_MGR_OBJECT_STRUCT);
        sCreate.tDataID = (DATASERVICE_ID)GsMapsPluginIntf.tDSI;
        // Suggest an OTA buffer size
        sCreate.tSuggestedOTABufferByteSize =
            GsMapsPluginIntf.tOTABufferByteSize;
        // Configure the data service's static event attributes
        sCreate.vEventCallback = vEventHandler;
        sCreate.tEventRequestMask = (
                DATASERVICE_EVENT_STATE |
                DATASERVICE_EVENT_NEW_DATA );

        // Ask the data service manager controller to
        // create our manager object and do everything
        // necessary to create the underlying objects required
        // in order to support this service
        psObj = (MAPS_MGR_OBJECT_STRUCT*)
            DATASERVICE_IMPL_hCreateNewService( &sCreate );
        if (psObj == NULL)
        {
            // Can't create the service, fail out!
            break;
        }

        // Create path to the reference db
        bOk = DB_UTIL_bCreateFilePath(
            sOptionValues.pcRefDBPath,
            MAPS_REF_DB_FOLDER_DEFAULT,
            NULL, // We need only directory.
            &psObj->pcReferenceDBDirPath);

        if (sOptionValues.pcRefDBPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)sOptionValues.pcRefDBPath);
            sOptionValues.pcRefDBPath = NULL;
        }

        if (bOk == FALSE)
        {
            // Error! Couldn't build path to reference DB folder
            break;
        }

        // Create path to the persistent db
        bOk = DB_UTIL_bCreateFilePath(
            pacOutputDir,
            MAPS_PERSIST_DB_FOLDER_DEFAULT,
            NULL, // We need only directory.
            &psObj->pcPersistentDBDirPath);

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_ALL,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvAppEventCallbackArg
                );

        bOk = bInitAppFacingObject(psObj);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": unable to initialize AppFacing object");
            break;
        }

        // The service may now start
        bOk = DATASERVICE_IMPL_bStart( (DATASERVICE_IMPL_HDL)psObj );
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": unable to start service" );
            break;
        }

        return (MAPS_SERVICE_OBJECT)psObj;
    } while (FALSE);

    // Error!
    if (psObj != NULL)
    {
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);
        vUninitObject(psObj, TRUE);
    }

    return MAPS_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
 *
 *   ePersistentDataCleanup
 *
 *   This API is used to restore the initial service state.
 *
 *   Inputs:
 *       hService - A handle to a valid MAPS_SERVICE object.
 *       pcReferenceDBPath - Path to the reference database.
 *           If NULL then the default path will be used.
 *       pcPesistentDBPath - Path to the persistent database.
 *           If NULL then the default path will be used.
 *
 *   Outputs:
 *       SMSAPI_RETURN_CODE_ENUM code on success, or error code otherwise.
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM ePersistentDataCleanup (
    MAPS_SERVICE_OBJECT hService,
    const char *pcReferenceDBPath,
    const char *pcPesistentDBPath
        )
{
    MAPS_MGR_OBJECT_STRUCT *psObj = (MAPS_MGR_OBJECT_STRUCT *)hService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    BOOLEAN bOk;

    if (hService == MAPS_SERVICE_INVALID_OBJECT)
    {
        return SMSAPI_RETURN_CODE_BAD_ARGUMENT;
    }

    psAppObj = psGetAppFacingObject(hService);
    if (psAppObj == NULL)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        DATASERVICE_ERROR_CODE_ENUM eErrorCode;

        // Release connections
        vReleaseDatabaseConnectionsLocked(psAppObj);

        // Clear persistent data
        bOk = bClearPersistentDirectory(psObj);
        if (bOk == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Re-create pathes
        if (psObj->pcReferenceDBDirPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pcReferenceDBDirPath);
            psObj->pcReferenceDBDirPath = NULL;
        }

        bOk = DB_UTIL_bCreateFilePath(
            pcReferenceDBPath,
            MAPS_REF_DB_FOLDER_DEFAULT,
            NULL, // We need only directory.
            &psObj->pcReferenceDBDirPath);

        if (bOk == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        if (psObj->pcPersistentDBDirPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pcPersistentDBDirPath);
            psObj->pcPersistentDBDirPath = (char*)NULL;
        }

        bOk = DB_UTIL_bCreateFilePath(
            pcPesistentDBPath,
            MAPS_PERSIST_DB_FOLDER_DEFAULT,
            NULL, // We need only directory.
            &psObj->pcPersistentDBDirPath);

        if (bOk == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Re-create connections
        bOk = bCreateDatabaseConnectionsLocked(psObj, &eErrorCode);
        if (bOk == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Re-build descriptors array
        bOk = bBuildDescriptorsArrayLocked(psObj);
        if (bOk == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    } while (FALSE);
    SMSO_vUnlock((SMS_OBJECT)psAppObj);

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eFindNearestMap
 *
 *   This API is used to query the nearest maps for the provided coordinates.
 *   Then nearest map is chosen based on distance between specified point and
 *   each map center point (centroid). If there are any maps that includes
 *   mentioned coordinates then the nearest one will be chosen from them, else
 *   it will be chosen from all maps.
 *
 *   Inputs:
 *       hService - A handle to a valid MAPS_SERVICE object.
 *       hLat - The latitude of the point of interest.
 *       hLon - The longitude of the point of interests.
 *       eDetalization - Map zoom level.
 *       tImageFilePathSize - Size of the passed buffer to store path to the
 *            map image.
 *
 *   Outputs:
 *       pacImageFilePath - Path to the map image file.
 *       SMSAPI_RETURN_CODE_SUCCESS on success, or error code otherwise.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFindNearestMap (
    MAPS_SERVICE_OBJECT hService,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon,
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
    char *pacImageFilePath,
    size_t tImageFilePathSize
        )
{

    MAPS_MGR_OBJECT_STRUCT *psObj = (MAPS_MGR_OBJECT_STRUCT *)hService;
    MAPS_DB_QUERY_NEAREST_STRUCT sNearestData;
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    N64 n64Latitude = 0, n64Longitude = 0;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    const char *pcImagesDir = (const char *)NULL;
    BOOLEAN bOk;

    do
    {
        // Init structure
        sNearestData.bSuccess = TRUE;
        sNearestData.hUserLocation = LOCATION_INVALID_OBJECT;
        sNearestData.un32MinDistanceAreaID = UN32_MAX;
        sNearestData.hMinDistance = DISTANCE_INVALID_OBJECT;
        sNearestData.hMinDistanceLocation = LOCATION_INVALID_OBJECT;
        sNearestData.pcPath = (char *)NULL;
        sNearestData.bBaseline = FALSE;

        // Check inputs
        if ((pacImageFilePath == NULL) ||
            ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
             (hLon == OSAL_FIXED_INVALID_OBJECT)) ||
            ((eDetailLevel < MAPS_SERVICE_DETAIL_LEVEL_LOW) ||
             (eDetailLevel > MAPS_SERVICE_DETAIL_LEVEL_HIGHEST)))
        {
            puts( MAPS_MGR_OBJECT_NAME": eFindNearestMap(): Bad parameters" );
            eReturn =  SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // In order to protect the caller in case if the service
        // is shutting down.
        psAppObj = psGetAppFacingObject(hService);
        if (psAppObj == NULL)
        {
            eReturn =  SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        sNearestData.hUserLocation = LOCATION.hCreateForRadius(
            hLat, hLon,
            DISTANCE_INVALID_OBJECT);
        if (LOCATION_INVALID_OBJECT == sNearestData.hUserLocation)
        {
            // looks like we cannot create the location object 
            // for provided coordinates, most likely these coordinates
            // out of valid area
            eReturn =  SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }

        // Extract scaled coordinates
        n64Latitude = OSAL_FIXED.n32ScaledValue(hLat, MAPS_FRACTIONAL_BITS);
        n64Longitude = OSAL_FIXED.n32ScaledValue(hLon, MAPS_FRACTIONAL_BITS);

        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MIN_LAT].eType =
            SQL_BIND_TYPE_N64;
        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MIN_LAT].pvData =
            (void *)&n64Latitude;

        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MAX_LAT].eType =
            SQL_BIND_TYPE_N64;
        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MAX_LAT].pvData =
            (void *)&n64Latitude;

        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MAX_LON].eType =
            SQL_BIND_TYPE_N64;
        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MAX_LON].pvData =
            (void *)&n64Longitude;

        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MIN_LON].eType =
            SQL_BIND_TYPE_N64;
        psAppObj->asBindParams[MAPS_SELECT_NEAREST_STMT_ATTR_MIN_LON].pvData =
            (void *)&n64Longitude;

        bOk = SQL_INTERFACE.bExecutePreparedStatement(
            psAppObj->hSQLMemoryConnection,
            psAppObj->hFindNearestStmt,
            (SQL_QUERY_RESULT_HANDLER)bProcessNearestAreaQuery,
            &sNearestData,
            MAPS_SELECT_NEAREST_STMT_ATTR_COUNT,
            psAppObj->asBindParams);

        if ((bOk == FALSE) || (sNearestData.bSuccess == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            MAPS_MGR_OBJECT_NAME": unable to get nearest map."
                                );
            break;
        }

        if (sNearestData.un32MinDistanceAreaID == UN32_MAX)
        {
            // There are no maps in db that contains user coordinates.
            // Try to find the nearest one from all maps.
            bOk = SQL_INTERFACE.bQuery(
                          psAppObj->hSQLMemoryConnection,
                          MAPS_MEMORY_SELECT_AREA_DESCRIPTION,
                          (SQL_QUERY_RESULT_HANDLER)bProcessNearestAreaQuery,
                          (void*) &sNearestData);

            if ((bOk == FALSE) || (sNearestData.bSuccess == FALSE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MAPS_MGR_OBJECT_NAME": unable to get nearest map.");
                break;
            }
        }

        if (sNearestData.un32MinDistanceAreaID != UN32_MAX)
        {

            psAppObj->asBindParams[MAPS_SELECT_MAP_DETAILS_STMT_ATTR_AREA_ID].eType =
                SQL_BIND_TYPE_UN32;
            psAppObj->asBindParams[MAPS_SELECT_MAP_DETAILS_STMT_ATTR_AREA_ID].pvData =
                (void *)(size_t)sNearestData.un32MinDistanceAreaID;

            psAppObj->asBindParams[MAPS_SELECT_MAP_DETAILS_STMT_ATTR_LEVEL].eType =
                SQL_BIND_TYPE_UN32;
            psAppObj->asBindParams[MAPS_SELECT_MAP_DETAILS_STMT_ATTR_LEVEL].pvData =
                (void *)(size_t)eDetailLevel;

            bOk = SQL_INTERFACE.bExecutePreparedStatement(
                psAppObj->hSQLMemoryConnection,
                psAppObj->hFindMapDetailsStmt,
                (SQL_QUERY_RESULT_HANDLER)bProcessNearestMapQuery,
                &sNearestData,
                MAPS_SELECT_MAP_DETAILS_STMT_ATTR_COUNT,
                psAppObj->asBindParams);

            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MAPS_MGR_OBJECT_NAME": unable to get nearest map.");
                break;
            }

            pcImagesDir =
                sNearestData.bBaseline ?
                psObj->pcReferenceDBDirPath : psObj->pcPersistentDBDirPath;
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }

        if (sNearestData.pcPath != NULL)
        {
            // Create full path
            eReturn = eBuildFullMapPath(psObj,
                sNearestData.pcPath,
                pcImagesDir,
                pacImageFilePath,
                tImageFilePathSize );
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
    } while (FALSE);

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    if (sNearestData.pcPath != NULL)
    {
        OSAL.vMemoryFree(sNearestData.pcPath);
    }

    if (sNearestData.hUserLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(sNearestData.hUserLocation);
    }

    if (sNearestData.hMinDistance != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(sNearestData.hMinDistance);
    }

    if (sNearestData.hMinDistanceLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(sNearestData.hMinDistanceLocation);
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   eLookupMapByName
 *
 *   This API is used to match map file path in accordance with the map name
 *   declared inside the XML file.
 *
 *   Inputs:
 *       hService - A handle to a valid MAPS_SERVICE object.
 *       pacMapName - The name of the map name extracted from XML file.
 *       eDetalization - Map zoom level.
 *       tImageFilePathSize - Size of the passed buffer to store path to
 *                            the map image.
 *
 *   Outputs:
 *       pacImageFilePath - Path to the map image file.
 *       SMSAPI_RETURN_CODE_SUCCESS on success, or error code otherwise.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eLookupMapByName (
    MAPS_SERVICE_OBJECT hService,
    const char *pacMapName,
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
    char *pacImageFilePath,
    size_t tImageFilePathSize
        )
{
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        MAPS_MGR_OBJECT_STRUCT *psObj = (MAPS_MGR_OBJECT_STRUCT *) hService;
        MAPS_DB_QUERY_MAP_BY_NAME_STRUCT sByNameQuery;
        BOOLEAN bOk;

        // Check inputs
        if ((pacMapName == NULL) ||
            (pacImageFilePath == NULL) ||
            ((eDetailLevel < MAPS_SERVICE_DETAIL_LEVEL_LOW) ||
             (eDetailLevel > MAPS_SERVICE_DETAIL_LEVEL_HIGHEST)))
        {
            puts( MAPS_MGR_OBJECT_NAME": eLookupMapByName(): Bad params" );
            eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // In order to protect the caller in case if the service
        // is shutting down.
        psAppObj = psGetAppFacingObject(hService);
        if (psAppObj == NULL)
        {
            puts( MAPS_MGR_OBJECT_NAME": eLookupMapByName(): cannot lock" );
            eReturn =  SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Initialize
        OSAL.bMemSet(&sByNameQuery, 0, sizeof(MAPS_DB_QUERY_MAP_BY_NAME_STRUCT));

        // Make a query
        bOk = bQueryMapByName(psObj, pacMapName, eDetailLevel, &sByNameQuery);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": unable to get map by name.");
            break;
        }

        if (sByNameQuery.pcPath != NULL)
        {
            // Create full path
            char *pcImagesPath =
                sByNameQuery.bBaseline ?
                psObj->pcReferenceDBDirPath : psObj->pcPersistentDBDirPath;

            eReturn = eBuildFullMapPath( psObj,
                sByNameQuery.pcPath,
                pcImagesPath,
                pacImageFilePath,
                tImageFilePathSize );
            OSAL.vMemoryFree(sByNameQuery.pcPath);
        }
        else
        {
            eReturn = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    } while (FALSE);

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   ePrioritizeMarket
 *
 *   This API is used to specify prioritized market for OTA maps update.
 *
 *   Inputs:
 *       hService - A handle to a valid MAPS_SERVICE object.
 *       tMarketId - The market identifier to be prioritized one for
 *                   OTA updates.
 *
 *   Outputs:
 *       SMSAPI_RETURN_CODE_SUCCESS on success, or error code otherwise.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePrioritizeMarket (
    MAPS_SERVICE_OBJECT hService,
    TRAFFIC_MARKET tMarketId
        )
{
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturn =
        SMSAPI_RETURN_CODE_ERROR;

    psAppObj = psGetAppFacingObject(hService);
    if (psAppObj != NULL)
    {
        psAppObj->tPrioritizedMarketId = tMarketId;
        eReturn = SMSAPI_RETURN_CODE_SUCCESS;
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return eReturn;
}


/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   bAddUpdate
 *
 *   Add or update record in Persistent DB.
 *
 *****************************************************************************/
static BOOLEAN bAddUpdate (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacSubname,
    UN32 un32NewVersion,
    UN16 un16HorResolution,
    FILE *psRFDFile,
    size_t tFileSize
        )
{
    BOOLEAN bSuccess = FALSE;
    MAPS_MGR_OBJECT_STRUCT *psObj =
            (MAPS_MGR_OBJECT_STRUCT *)hMapsService;
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    MAPS_UPDATES_ROW_STRUCT sUpdateRow;
    MAPS_MARKET_DESCRIPTOR_STRUCT sDescriptor;
    N16 n16DescriptionIdx;
    MAPS_DB_QUERY_MAP_BY_NAME_STRUCT sQueryResult;
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eLevel;
    SMSAPI_RETURN_CODE_ENUM eErrorCode;

    if ((pacSubname == NULL) ||
        (psRFDFile == NULL) ||
        (psObj == NULL))
    {
        return FALSE;
    }

    psAppObj = psGetAppFacingObject(hMapsService);

    if (psAppObj != NULL)
    {
        do
        {
            // Find market in descriptors array
            n16DescriptionIdx = n16GetDescriptor(
                hMapsService,
                pacSubname,
                &sDescriptor);

            if (n16DescriptionIdx < 0)
            {
                // There is no such subname
                bSuccess = FALSE;
                break;
            }

            // Create Image File
            eErrorCode = eCreateUpdateImageFile(hMapsService,
                psRFDFile,
                tFileSize,
                &sDescriptor.acMarketName[0],
                un16HorResolution,
                un32NewVersion);

            if (eErrorCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                bSuccess = FALSE;
                break;
            }

            eLevel = eResolutionToEnum(un16HorResolution);

            // Save the old file path
            sQueryResult.pcPath = NULL;
            snprintf(&psObj->acBuffer[0], sizeof(psObj->acBuffer),
                MAPS_PERSISTENT_SELECT_MAP_BY_NAME,
                &sDescriptor.acMarketName[0],
                (int)eLevel);

            bSuccess = SQL_INTERFACE.bQuery(
                psAppObj->hSQLPersistConnection,
                &psObj->acBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bSelectMapByNamePersistCallback,
                &sQueryResult);

            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MAPS_MGR_OBJECT_NAME": unable to get data from Persistent DB!");
                break;
            }

            // Fill row data
            snprintf(&psObj->acBuffer[0],
                    sizeof(psObj->acBuffer),
                    MAPS_IMAGE_FILE_NAME_FMT,
                    &sDescriptor.acMarketName[0],
                    un16HorResolution,
                    un32NewVersion);

            sUpdateRow.pcMarketName = &sDescriptor.acMarketName[0];
            sUpdateRow.pcData = (char *)&psObj->acBuffer[0];
            sUpdateRow.eLevel = eLevel;
            sUpdateRow.un32ToVersion = un32NewVersion;

            bSuccess =
                SQL_INTERFACE.bExecutePreparedCommand(
                    psAppObj->hSQLPersistConnection,
                    MAPS_INSERT_UPDATED_MAPS,
                    DB_MAPS_UPDATED_MAPS_COUNT,
                    (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareUpdatedMapInsert,
                    (void *)&sUpdateRow);

            if (bSuccess == FALSE)
            {
                break;
            }

            // Now, update file version in descriptors array
            bSuccess = bSetVersion(hMapsService,
                n16DescriptionIdx,
                un32NewVersion);

            if (bSuccess == FALSE)
            {
                break;
            }

            // And add record in "updated_images" table
            if (sQueryResult.pcPath != NULL)
            {
                bSuccess =
                    SQL_INTERFACE.bExecutePreparedCommand(
                        psAppObj->hSQLPersistConnection,
                        MAPS_INSERT_UPDATED_IMAGES,
                        DB_MAPS_UPDATED_IMAGES_COUNT,
                        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareUpdatedImageInsert,
                        (void *)sQueryResult.pcPath);

                  OSAL.vMemoryFree((void *)sQueryResult.pcPath);
            }
        }while (FALSE);
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }
    return bSuccess;
}

/*****************************************************************************
 *
 *   un16GetVersionForSubname
 *
 *   Returns current file version based on provided market subname.
 *
 *****************************************************************************/
static UN32 un32GetVersionForSubname (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacSubname
        )
{
    N16 n16DescriptionIdx;
    MAPS_MARKET_DESCRIPTOR_STRUCT sDescriptor;
    UN32 un32Version = MAPS_INVALID_VERSION;

    n16DescriptionIdx = n16GetDescriptor(hMapsService, 
        pacSubname,
        &sDescriptor);

    if (n16DescriptionIdx >= 0)
    {
        un32Version = sDescriptor.un32Version;
    }

    return un32Version;
}

/*****************************************************************************
 *
 *   n8ComparePriority
 *
 *   Check if one of provided markets is prioritized.
 *
 *****************************************************************************/
static N8 n8ComparePriority (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacSubnameA,
    const char *pacSubnameB
        )
{
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    N8 n8Result = 0;

    if ((pacSubnameA == NULL) ||
        (pacSubnameB == NULL))
    {
        return 0;
    }

    psAppObj = psGetAppFacingObject(hMapsService);
    if (psAppObj != NULL)
    {
        MAPS_MARKET_DESCRIPTOR_STRUCT sDescriptorA, sDescriptorB;
        N16 n16DescriptionIdxA, n16DescriptionIdxB;

        n16DescriptionIdxA = n16GetDescriptor(
            hMapsService,
            pacSubnameA,
            &sDescriptorA);

        n16DescriptionIdxB = n16GetDescriptor(
            hMapsService,
            pacSubnameB,
            &sDescriptorB);

        if ((n16DescriptionIdxA >= 0) &&
            (n16DescriptionIdxB >= 0))
        {
            if (sDescriptorA.tMarketID == psAppObj->tPrioritizedMarketId)
            {
                n8Result = 1;
            }
            else if (sDescriptorB.tMarketID ==
                psAppObj->tPrioritizedMarketId)
            {
                n8Result = -1;
            }
            else
            {
                n8Result = 0;
            }
        }
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return n8Result;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   bHandleServiceReady
 *
 *   This function is called when SMS is ready for the service to startup.
 *   At this time, the service has a context in which to operate (SMS
 *   assigned a resource to us). When this call is made, this service will
 *   perform all of its time-intensive startup procedures.
 *
 *****************************************************************************/
static BOOLEAN bHandleServiceReady (
    MAPS_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = FALSE,
            bOk = TRUE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;

    do
    {
        psAppObj = psGetAppFacingObject((MAPS_SERVICE_OBJECT)psObj);

        if (psAppObj == NULL)
        {
            // Cannot get App Facing object
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the path to the Reference DB
        bOk = bCreateDatabaseConnectionsLocked(psObj, &eErrorCode);
        if (bOk == FALSE)
        {
            break;
        }

        // Delete all images referenced in "updated_images" table
        bOk = SQL_INTERFACE.bQuery(
            psAppObj->hSQLPersistConnection,
            MAPS_SELECT_UPDATED_IMAGE_PATH,
            (SQL_QUERY_RESULT_HANDLER)bRemoveUnusedImageFiles,
            (void *)psObj);

        if (bOk == FALSE)
        {
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Now, clear the table
        bOk = SQL_INTERFACE.bExecuteCommand(
            psAppObj->hSQLPersistConnection,
            MAPS_DROP_UPDATED_IMAGES_TABLE);

        if (bOk == FALSE)
        {
            // Cannot clear table.
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        psAppObj->tPrioritizedMarketId = TRAFFIC_INVALID_MARKET;

        // Build descriptors array
        bOk = bBuildDescriptorsArrayLocked(psObj);
        if (bOk == FALSE)
        {
            // Something goes wrong.
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Start the interface
        psObj->hParser = GsMapsPluginIntf.hInit(
            (MAPS_SERVICE_OBJECT)psObj,
            DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj)
                );

        if (psObj->hParser == MAPS_INTERFACE_INVALID_OBJECT)
        {
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    if (bResult != TRUE)
    {
        // Set the error we experienced
        vSetError(psObj, eErrorCode);
    }

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return bResult;
}

/*******************************************************************************
*
*   hCreateMemoryDb
*
*******************************************************************************/
static BOOLEAN bCreateMemoryDB(
    MAPS_MGR_OBJECT_STRUCT *psObj,
    SQL_INTERFACE_OBJECT hReferenceConection,
    SQL_INTERFACE_OBJECT hPersistentConection,
    SQL_INTERFACE_OBJECT *phMemoryConection
        )
{
    BOOLEAN bOk = FALSE,
        bTransactionStarted = FALSE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;

    do
    {
        if ((hReferenceConection == NULL) ||
            (hPersistentConection == NULL) ||
            (phMemoryConection == NULL))
        {
            break;
        }

        // Open a new memory database
        *phMemoryConection = SQL_INTERFACE.hConnect(
            ":memory:",
            SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND |
            SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK,
            &eErrorCode);

        if(*phMemoryConection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        bTransactionStarted =
            SQL_INTERFACE.bStartTransaction(*phMemoryConection);

        // Copy the Reference DB into the Memory DB.
        bOk = SQL_INTERFACE.bCopyMainDatabase(*phMemoryConection,
            hReferenceConection);

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": Cannot create Memory DB!");
            break;
        }

        // Add column to track baseline
        bOk = SQL_INTERFACE.bExecuteCommand(
            *phMemoryConection,
            MAPS_MEMORY_ADD_COLUMN);

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": Cannot add column to Memory DB!");
            break;
        }

        // Update records with data from Persistent DB
        bOk = SQL_INTERFACE.bQuery(
            hPersistentConection,
            MAPS_SELECT_UPDATED_MAPS,
            (SQL_QUERY_RESULT_HANDLER)bUpdateMemoryDB,
            (void *)psObj);

        if (bOk == FALSE)
        {
            //Cannot update Memory DB.
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }
    }while(FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        // End transaction
        bTransactionEnded =
            SQL_INTERFACE.bEndTransaction(*phMemoryConection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": failed to end transaction");
            bOk = FALSE;
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bProcessMessage
 *
 *   This function is called to provide the maps interface with
 *   a newly received message
 *
 *****************************************************************************/
static BOOLEAN bProcessMessage(
    MAPS_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bProcessed = TRUE;

    // Only process this message if
    // the handle is valid
    if (*phPayload != OSAL_INVALID_BUFFER_HDL)
    {
        // Tell the interface it's time to
        // process another message
        bProcessed =
            GsMapsPluginIntf.bProcessMessage(psObj->hParser, phPayload);
    }

    return bProcessed;
}

/*****************************************************************************
 *
 *   vEventHandler
 *
 *   This function runs in the context of an SMS resource which has been
 *   assigned to this service.
 *
 *****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    MAPS_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our Non-Nav Maps handle from the callback argument
    psObj = (MAPS_MGR_OBJECT_STRUCT*)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": Invalid MAPS MGR object.");
        return;
    }

    switch (tCurrentEvent)
    {
        // Handle Maps Service events here...
        // State has changed
        case DATASERVICE_EVENT_STATE:
        {
            BOOLEAN bStateChanged;
            DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

            // Process the state transition
            bStateChanged = DATASERVICE_IMPL_bStateFSM(
                (DATASERVICE_IMPL_HDL)psObj,
                psStateChange,
                &GsMapsStateHandlers,
                (void *)psObj);

            if (bStateChanged == TRUE)
            {
                // The state has been updated
                tEventMask |= DATASERVICE_EVENT_STATE;
                // Is the service stopped now?
                if (psStateChange->eCurrentState ==
                        DATASERVICE_STATE_STOPPED)
                {
                    bStopEvent = TRUE;
                }
            }
        }
        break;

        // This service has a message to process
        case DATASERVICE_EVENT_NEW_DATA:
        {
            OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
            BOOLEAN bSuccess = TRUE;

            // Ensure the payload handle is valid
            // If it isn't, there's a problem with
            // SMS
            if (hPayload == OSAL_INVALID_BUFFER_HDL)
            {
                // Set the error
                vSetError(psObj,
                    DATASERVICE_ERROR_CODE_GENERAL
                        );
            }
            // Only process this message if we are not stopping
            else
            {
                printf(MAPS_MGR_OBJECT_NAME": payload received (%u)\n",
                    OSAL.tBufferGetSize(hPayload)
                        );

                // Process this message now
                bSuccess = bProcessMessage(psObj, &hPayload);

                if (bSuccess == TRUE)
                {
                    puts(MAPS_MGR_OBJECT_NAME": Message Payload Processed Ok");
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        MAPS_MGR_OBJECT_NAME": failed to process payload"
                            );
                    DATASERVICE_IMPL_vLog(MAPS_MGR_OBJECT_NAME
                        ": failed to process payload\n"
                            );
                }
            }

            // We're all done with this payload
            DATASERVICE_IMPL_bFreeDataPayload(hPayload);
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": Got unknown event (%u)",
                tCurrentEvent
                    );
        }
        break;
    }


    if (bStopEvent == TRUE)
    {
        // Uninitialize the object, but leave
        // enough to be useful to the callback
        vUninitObject(psObj, FALSE);
    }

    // Update event mask with any relevant events which have occurred
    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    if (bStopEvent == TRUE)
    {
        // Filter out all further maps updates
        SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

        vUninitObject(psObj, TRUE);
    }

    return;
}

/*****************************************************************************
 *
 *   vSetError
 *
 *****************************************************************************/
static void vSetError (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}

/*****************************************************************************
 *
 *   vUninitObject
 *
 *****************************************************************************/
static void vUninitObject (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    // Uninit App Facing object first to avoid sync issues
    if (psObj->psAppObj != NULL)
    {
        vUninitAppFacingObject(psObj->psAppObj);
        psObj->psAppObj = NULL;
    }

    // Releasing plug-in object
    if (psObj->hParser != MAPS_INTERFACE_INVALID_OBJECT)
    {
        GsMapsPluginIntf.vUnInit(psObj->hParser);
        psObj->hParser = MAPS_INTERFACE_INVALID_OBJECT;
    }

    if (psObj->pcReferenceDBDirPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pcReferenceDBDirPath);
        psObj->pcReferenceDBDirPath = NULL;
    }

    if (psObj->pcPersistentDBDirPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pcPersistentDBDirPath);
        psObj->pcPersistentDBDirPath = NULL;
    }

    if (bFullDelete == TRUE)
    {
        // Clear the data service manager handle
        psObj->hDataServiceManager = DATASERVICE_MGR_INVALID_OBJECT;

        // Destroy the SMS Update Event object
        SMSU_vDestroy(&psObj->sEvent);
    }

    return;
}

/*****************************************************************************
 *
 *   bCreatePersistentDB
 *
 *   Called when persistent DB should be re-created due to corruption
 *   or any other issue (like versions mismatch).
 *
 *****************************************************************************/
static BOOLEAN bCreatePersistentDB (
    SQL_INTERFACE_OBJECT hPersistSQLConnection,
    void *pvArg
        )
{
    BOOLEAN bOk = FALSE,
        bTransactionStarted = FALSE;

    do
    {
        bTransactionStarted =
            SQL_INTERFACE.bStartTransaction(hPersistSQLConnection);

        if (bTransactionStarted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME
                ": failed to start transaction");
            break;
        }

        // Create table to store maps updates
        bOk = SQL_INTERFACE.bExecuteCommand(hPersistSQLConnection,
            MAPS_CREATE_UPDATED_MAPS_TABLE);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME
                ": failed to create updates table in persistant storage");
            break;
        }

        // Create table to store updated image files
        bOk = SQL_INTERFACE.bExecuteCommand(hPersistSQLConnection,
            MAPS_CREATE_UPDATED_IMAGES_TABLE);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME
                ": failed to create updates table in persistant storage");
            break;
        }
    } while (FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        // End transaction
        bTransactionEnded =
            SQL_INTERFACE.bEndTransaction(hPersistSQLConnection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME
                ": failed to end transaction");
            bOk = FALSE;
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bVerifyDBVersion
 *
 *   Verifies the database schema version.
 *
 *****************************************************************************/
static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    MAPS_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    OSAL.bMemSet(&sQueryResult, 0, sizeof(MAPS_DB_QUERY_RESULT_STRUCT));

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
            hSQLConnection, MAPS_SELECT_DB_VERSION,
            bProcessSelectDBVersionResult, &sQueryResult ) ;

    if (bOk == TRUE)
    {
        if (sQueryResult.bResultantRows == TRUE)
        {
            MAPS_VERSION_ROW_STRUCT *psVersionRow =
                &sQueryResult.uDbRow.sVersion;

            // Verify that the db schema version matched
            if (psVersionRow->un8DBVer == MAPS_DATABASE_FILE_VERSION)
            {
                bOk = TRUE;
            }
            else
            {
                // Version mismatch!
                bOk =  FALSE;
            }
        }
    }
    return bOk;
}

/*****************************************************************************
 *
 *   bProcessSelectDBVersionResult
 *
 *****************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       )
{
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_VERSION_FIELDS_ENUM eCurrentField = (DB_VERSION_FIELDS_ENUM)0;
    MAPS_DB_QUERY_RESULT_STRUCT *psQueryResult =
        (MAPS_DB_QUERY_RESULT_STRUCT *)pvArg;

    if (NULL == psQueryResult)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < DB_VERSION_MAX_FIELDS)
    {
        return FALSE;
    }

    do
    {
        psQueryResult->bResultantRows = TRUE;

        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case DB_VERSION_FIELD_DB_VER:
            {
                psQueryResult->uDbRow.sVersion.un8DBVer =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            default:
            {
                // Shouldn't happen
            }
            break;
        }

        ++eCurrentField;

    } while (eCurrentField < DB_VERSION_MAX_FIELDS);

    // Return false since we don't need to iterate through the table
    // should only be one row
    return FALSE;
}

/*****************************************************************************
 *
 *   bPrepareUpdatedMapInsert
 *
 *****************************************************************************/
static BOOLEAN bPrepareUpdatedMapInsert (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    void *pvCallbackArg
        )
{
    BOOLEAN bSuccess = TRUE;
    MAPS_UPDATES_ROW_STRUCT *psMapUpdatesRow =
        (MAPS_UPDATES_ROW_STRUCT *)pvCallbackArg;

    switch (tIndex)
    {
        case DB_MAPS_UPDATED_MAPS_AREA_NAME:
        {
            *peType = SQL_BIND_TYPE_C_STRING;
            *ppvData = psMapUpdatesRow->pcMarketName;
        }
        break;

        case DB_MAPS_UPDATED_MAPS_ZOOM_LEVEL:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psMapUpdatesRow->eLevel;
        }
        break;

        case DB_MAPS_UPDATED_MAPS_DATA:
        {
            *peType = SQL_BIND_TYPE_C_STRING;
            *ppvData = psMapUpdatesRow->pcData;
        }
        break;

        case DB_MAPS_UPDATED_MAPS_VERSION:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psMapUpdatesRow->un32ToVersion;
        }
        break;

        case DB_MAPS_UPDATED_MAPS_COUNT:
        default:
        {
            bSuccess = FALSE;
        }
        break;
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bPrepareUpdatedImageInsert
 *
 *****************************************************************************/
static BOOLEAN bPrepareUpdatedImageInsert (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    void *pvCallbackArg
        )
{
    if (tIndex == DB_MAPS_UPDATED_IMAGES_DATA)
    {
        *peType = SQL_BIND_TYPE_C_STRING;
        *ppvData = (char *)pvCallbackArg;
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
 *
 *   bSelectMapByNameMemoryCallback
 *
 *****************************************************************************/
static BOOLEAN bSelectMapByNameMemoryCallback (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_MAP_BY_NAME_STRUCT *psQueryResult
        )
{
    size_t tPathSize = 0;
    const char *pcColumnData;

    // Verify input
    if (psQueryResult == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < DB_MAPS_SET_COUNT)
    {
        return FALSE;
    }

    // Fill data
    psQueryResult->un32AreaDescriptionID =
        psColumn[DB_MEMORY_MAPS_SET_AREA_DESC_ID].uData.sUN32.un32Data;

    pcColumnData =
        (const char*)psColumn[DB_MEMORY_MAPS_SET_DATA].uData.sCString.pcData;
    tPathSize = strlen(pcColumnData) + 1;
    psQueryResult->pcPath = (char *)OSAL.pvMemoryAllocate("MapImagePath",
                tPathSize * sizeof(char), TRUE);

    if (psQueryResult->pcPath != NULL)
    {
        strncpy(psQueryResult->pcPath,
            pcColumnData,
            tPathSize);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": Unable allocate memory");
    }

    psQueryResult->un32Version =
        psColumn[DB_MEMORY_MAPS_SET_VERSION].uData.sUN32.un32Data;

    psQueryResult->bBaseline = 
        (BOOLEAN)psColumn[DB_MEMORY_MAPS_SET_BASELINE].uData.sUN32.un32Data;

    // Return FALSE, since it should be only one row
    return FALSE;
}

/*****************************************************************************
 *
 *   bSelectImagesVersionMemoryCallback
 *
 *****************************************************************************/
static BOOLEAN bSelectImagesVersionMemoryCallback (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    UN32 *pun32Version
        )
{
    // Verify input
    if (pun32Version == NULL)
    {
        return FALSE;
    }

    *pun32Version = (UN32)psColumn[0].uData.sUN32.un32Data;

    // Return FALSE, since it should be only one row
    return FALSE;
}

/*****************************************************************************
 *
 *   bSelectMapByNamePersistCallback
 *
 *****************************************************************************/
static BOOLEAN bSelectMapByNamePersistCallback (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_MAP_BY_NAME_STRUCT *psQueryResult
        )
{
    size_t tPathSize = 0;
    const char *pcColumnData;

    // Verify input
    if (psQueryResult == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < DB_MAPS_UPDATED_MAPS_COUNT)
    {
        return FALSE;
    }

    // Fill data
    pcColumnData =
        (const char*)psColumn[DB_MAPS_UPDATED_MAPS_DATA].uData.sCString.pcData;
    tPathSize = strlen(pcColumnData) + 1;

    psQueryResult->pcPath = (char *)OSAL.pvMemoryAllocate("MapImagePath",
        tPathSize * sizeof(char), TRUE);

    if (psQueryResult->pcPath != NULL)
    {
        strncpy(psQueryResult->pcPath,
            pcColumnData,
            tPathSize);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": Unable allocate memory");
    }

    psQueryResult->un32Version =
        psColumn[DB_MAPS_UPDATED_MAPS_VERSION].uData.sUN32.un32Data;

    psQueryResult->bBaseline = FALSE;

    // Return FALSE, since it should be only one row
    return FALSE;
}

/*****************************************************************************
 *
 *   bProcessNearestAreaQuery
 *
 *****************************************************************************/
static BOOLEAN bProcessNearestAreaQuery (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_NEAREST_STRUCT *psData
        )
{
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_AREA_DESC_FIELDS_ENUM eCurrentField = (DB_AREA_DESC_FIELDS_ENUM)0;
    OSAL_FIXED_OBJECT hCenterLat = OSAL_FIXED_INVALID_OBJECT,
                      hCenterLon = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA atCenterLat[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT_DATA atCenterLon[OSAL_FIXED_OBJECT_SIZE];
    UN32 un32AreaDescID = UN32_MAX;

    // Verify input
    if (psData == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < DB_MAPS_AREA_DESC_COUNT)
    {
        return FALSE;
    }

    do
    {
        // Grab the current column data
        psCurrentCol = &psColumn[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case DB_MAPS_AREA_DESC_ID:
            {
                un32AreaDescID = (UN32)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case DB_MAPS_AREA_DESC_CENTER_LAT:
            {
                hCenterLat = OSAL_FIXED.hCreateInMemory(
                    (N32)psCurrentCol->uData.sUN32.un32Data,
                    MAPS_FRACTIONAL_BITS, &atCenterLat[0]);
            }
            break;

            case DB_MAPS_AREA_DESC_CENTER_LON:
            {
                hCenterLon = OSAL_FIXED.hCreateInMemory(
                    (N32)psCurrentCol->uData.sUN32.un32Data,
                    MAPS_FRACTIONAL_BITS, &atCenterLon[0]);
            }
            break;

            case DB_MAPS_AREA_DESC_MARKET_NAME:
            {
                strncpy(psData->pacMarketName,
                    (const char *)psCurrentCol->uData.sCString.pcData,
                    MAPS_SUBNAME_LENGHT_MAX);
            }
            break;

            case DB_MAPS_AREA_MARKET_ID:
            case DB_MAPS_AREA_DESC_UPDATE_NAME:
            case DB_MAPS_AREA_DESC_MAX_LAT:
            case DB_MAPS_AREA_DESC_MIN_LAT:
            case DB_MAPS_AREA_DESC_MAX_LON:
            case DB_MAPS_AREA_DESC_MIN_LON:
            case DB_MAPS_AREA_DESC_COUNT:
            {
                // Do nothing
            }
            break;
        }

        ++eCurrentField;

    } while (eCurrentField < DB_MAPS_AREA_DESC_COUNT);

    // Create location for map center if it doesn't exist
    if (psData->hMinDistanceLocation == LOCATION_INVALID_OBJECT)
    {
        psData->hMinDistanceLocation = LOCATION.hCreateForRadius(
            hCenterLat, hCenterLon,
            DISTANCE_INVALID_OBJECT);
        if (psData->hMinDistanceLocation == LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": failed to create intermediate"
                " LOCATION object");
            psData->bSuccess = FALSE;
        }
    }
    else
    {
        psData->bSuccess = LOCATION_bUpdateCoordinatesByFixed(
            psData->hMinDistanceLocation, hCenterLat, hCenterLon);
    }

    if (psData->bSuccess == TRUE)
    {
        DISTANCE_OBJECT hDistance;

        // Get distance
        hDistance = LOCATION.hDistance(
            psData->hUserLocation, psData->hMinDistanceLocation);

        if (hDistance != DISTANCE_INVALID_OBJECT)
        {
            N16 n16Compare;

            n16Compare = DISTANCE.n16Compare(hDistance, psData->hMinDistance);
            if (n16Compare < 0)
            {
                // This map center is nearer
                if (psData->hMinDistance != DISTANCE_INVALID_OBJECT)
                {
                    DISTANCE.vDestroy(psData->hMinDistance);
                }
                psData->hMinDistance = hDistance;
                psData->un32MinDistanceAreaID = un32AreaDescID;
            }
            else
            {
                DISTANCE.vDestroy(hDistance);
            }
        }
    }

    return psData->bSuccess;
}

/*****************************************************************************
 *
 *   bProcessNearestMapQuery
 *
 *****************************************************************************/
static BOOLEAN bProcessNearestMapQuery (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_NEAREST_STRUCT *psData
        )
{
    size_t tPathSize = 0;
    const char *pcColumnData;

    // Verify input
    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if ((psData == NULL) ||
        (n32NumberOfColumns < DB_MEMORY_MAPS_SET_COUNT))
    {
        return FALSE;
    }

    pcColumnData =
        (const char*)psColumn[DB_MEMORY_MAPS_SET_DATA].uData.sCString.pcData;

    // Fill data
    tPathSize = strlen(pcColumnData) + 1;
    psData->pcPath = (char *)OSAL.pvMemoryAllocate("MapImagePath",
                tPathSize * sizeof(char), TRUE);

    if (psData->pcPath != NULL)
    {
        strncpy(psData->pcPath,
            pcColumnData,
            tPathSize);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": Unable allocate memory");
    }

    psData->bBaseline =
        (BOOLEAN)psColumn[DB_MEMORY_MAPS_SET_BASELINE].uData.sUN32.un32Data;

    // Return FALSE, since it should be only one row
    return FALSE;
}

/*****************************************************************************
 *
 *   bUpdateMemoryDB
 *
 *****************************************************************************/
static BOOLEAN bUpdateMemoryDB (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvData
        )
{
    BOOLEAN bOk = TRUE;
    MAPS_MGR_OBJECT_STRUCT *psObj =
        (MAPS_MGR_OBJECT_STRUCT *)pvData;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    DB_MAPS_UPDATED_MAPS_FIELDS_ENUM eCurrentField =
        (DB_MAPS_UPDATED_MAPS_FIELDS_ENUM)0;
    MAPS_UPDATES_ROW_STRUCT sUpdateRow;
    UN8 un8FileAttr = 0;

    // Verify input
    if (n32NumberOfColumns < DB_MAPS_UPDATED_MAPS_COUNT)
    {
        return FALSE;
    }

    sUpdateRow.eLevel = MAPS_SERVICE_DETAIL_LEVEL_UNKNOWN;
    sUpdateRow.pcData = (char *)NULL;
    sUpdateRow.pcMarketName = (char *)NULL;
    sUpdateRow.un32ToVersion = 0;

    do
    {
        // Grab the current column data
        psCurrentCol = &psColumns[(UN8)eCurrentField];

        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case DB_MAPS_UPDATED_MAPS_AREA_NAME:
            {
                sUpdateRow.pcMarketName =
                    (char *)psCurrentCol->uData.sCString.pcData;
            }
            break;

            case DB_MAPS_UPDATED_MAPS_ZOOM_LEVEL:
            {
                sUpdateRow.eLevel = (MAPS_SERVICE_DETAIL_LEVEL_ENUM)
                    psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case DB_MAPS_UPDATED_MAPS_DATA:
            {
                // Get Map name
                sUpdateRow.pcData = (char *)psCurrentCol->uData.sCString.pcData;

                if ((psObj->pcPersistentDBDirPath != NULL) &&
                    (sUpdateRow.pcData != NULL))
                {
                    // Get full path to image file
                    snprintf(&psObj->acBuffer[0],
                        sizeof(psObj->acBuffer),
                        MAPS_PATH_BUILD_FMT,
                        psObj->pcPersistentDBDirPath,
                        sUpdateRow.pcData);

                    bOk = OSAL.bFileSystemGetFileAttributes(
                        &psObj->acBuffer[0],
                        &un8FileAttr);

                    if (bOk != TRUE)
                    {
                        // File doesn't exists.
                        // Don't update the path and continue iteration.
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            MAPS_MGR_OBJECT_NAME": Cannot find image file: %s",
                            &psObj->acBuffer[0]);
                        return TRUE;
                    }
                }
            }
            break;

            case DB_MAPS_UPDATED_MAPS_VERSION:
            {
                sUpdateRow.un32ToVersion =
                    (UN32)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            default:
            {
                // Shouldn't happen
            }
            break;
        }

        ++eCurrentField;
    } while (eCurrentField < DB_MAPS_UPDATED_MAPS_COUNT);

    if ((sUpdateRow.pcMarketName != NULL) &&
        (sUpdateRow.pcData != NULL))
    {
        // Update record in Memory DB.
        bOk = bUpdateMemoryMapSetTable(
            (MAPS_SERVICE_OBJECT)psObj,
            &sUpdateRow);
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bUpdateMemoryMapSetTable
 *
 *****************************************************************************/
static BOOLEAN bUpdateMemoryMapSetTable (
    MAPS_SERVICE_OBJECT hMapsService,
    MAPS_UPDATES_ROW_STRUCT *psUpdateRow
        )
{
    BOOLEAN bOk = FALSE;
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    MAPS_MGR_OBJECT_STRUCT *psObj =
        (MAPS_MGR_OBJECT_STRUCT *)hMapsService;

    if ((psUpdateRow == NULL) ||
        (psUpdateRow->pcMarketName == NULL) ||
        (psUpdateRow->pcData == NULL))
    {
        return FALSE;
    }

    psAppObj = psGetAppFacingObject(hMapsService);

    if (psAppObj != NULL)
    {
        // Create query
        snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
            MAPS_MEMORY_MAP_SET_UPDATE,
            psUpdateRow->pcData,
            psUpdateRow->un32ToVersion,
            FALSE,
            psUpdateRow->eLevel,
            psUpdateRow->pcMarketName);

        bOk = SQL_INTERFACE.bExecuteCommand(
            psAppObj->hSQLMemoryConnection,
            &psObj->acBuffer[0]);

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bRemoveUnusedImageFiles
 *
 *   Remove unused map image files from persistent folder.
 *
 *****************************************************************************/
static BOOLEAN bRemoveUnusedImageFiles (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    void *pvData
        )
{
    const char *pacFileName = (char *)NULL;
    BOOLEAN bOk = TRUE;
    MAPS_MGR_OBJECT_STRUCT *psObj =
        (MAPS_MGR_OBJECT_STRUCT *)pvData;
    N32 n32Result = 0;

    // Verify input
    // Only 1 column (_data) should be in result row
    if (n32NumberOfColumns != 1)
    {
        return FALSE;
    }

    do
    {
        // Get image file name
        pacFileName = (const char *)psColumn[0].uData.sCString.pcData;

        if ((psObj->pcPersistentDBDirPath != NULL) &&
            (pacFileName != NULL))
        {
            // Get full path to image file
            snprintf(&psObj->acBuffer[0],
                sizeof(psObj->acBuffer),
                MAPS_PATH_BUILD_FMT,
                psObj->pcPersistentDBDirPath,
                pacFileName);

            n32Result = (N32)remove(&psObj->acBuffer[0]);

            if (n32Result != 0)
            {
                // File doesn't exists
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MAPS_MGR_OBJECT_NAME": Cannot delete image file: %s",
                    &psObj->acBuffer[0]);
                bOk = FALSE;
            }
        }
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   bFillDescriptorsArray
 *
 *****************************************************************************/
static BOOLEAN bFillDescriptorsArray (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    void *pvData
        )
{
    MAPS_MGR_OBJECT_STRUCT *psObj = 
        (MAPS_MGR_OBJECT_STRUCT *)pvData;
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    BOOLEAN bOk = FALSE;

    // Verify input
    if ((psObj == NULL) ||
        (psObj->psAppObj == NULL) ||
        (n32NumberOfColumns < DB_MAPS_AREA_DESC_COUNT))
    {
        return FALSE;
    }

    // App Facing object was locked before this query
    // so, omit bGetAppFacingObject() call to reduce efforts
    psAppObj = psObj->psAppObj;

    // Fill data
    if (psAppObj->un16MarketsCount < MAPS_MARKETS_COUNT_MAX)
    {
        // Fill Market Name
        strncpy(
            &psAppObj->asDescriptors[psAppObj->un16MarketsCount].acMarketName[0],
            (const char*)psColumn[DB_MAPS_AREA_DESC_MARKET_NAME].uData.sCString.pcData,
            MAPS_SUBNAME_LENGHT_MAX);

        // Fill Update Name
        strncpy(
            &psAppObj->asDescriptors[psAppObj->un16MarketsCount].acUpdateName[0],
            (const char*)psColumn[DB_MAPS_AREA_DESC_UPDATE_NAME].uData.sCString.pcData,
            MAPS_SUBNAME_LENGHT_MAX);

        // Fill Market ID
        psAppObj->asDescriptors[psAppObj->un16MarketsCount].tMarketID =
            (TRAFFIC_MARKET)psColumn[DB_MAPS_AREA_MARKET_ID].uData.sUN32.un32Data;

        // Get version
        bOk = bQueryMarketImagesVersion(psObj,
            &psAppObj->asDescriptors[psAppObj->un16MarketsCount].acMarketName[0],
            &psAppObj->asDescriptors[psAppObj->un16MarketsCount].un32Version);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_MGR_OBJECT_NAME": unable to get map by name.");
        }

        ++psAppObj->un16MarketsCount;
    }

    return bOk;
}

/*****************************************************************************
 *
 *   eBuildFullMapPath
 *
 *   Builds the full path to map image file in persistent storage.
 *
 *   Inputs:
 *       psObj - Pointer to MAPS_MGR_OBJECT_STRUCT.
 *       pacMapName - Map image file name.
 *       pacDirPath - Map image file directory.
 *       tImageFilePathSize - Size of the passed buffer to store path to the
 *            map image.
 *
 *   Outputs:
 *       pacImageFilePath - Full path to map image file.
 *       SMSAPI_RETURN_CODE_SUCCESS code on success, or error code otherwise.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBuildFullMapPath (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    const char *pacMapName,
    const char *pacDirPath,
    char *pacImageFilePath,
    size_t tImageFilePathSize
        )
{
    size_t tLength = 0;

    if ((pacMapName == NULL) ||
        (pacImageFilePath == NULL) ||
        (pacDirPath == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    tLength = snprintf(pacImageFilePath,
                    tImageFilePathSize,
                    MAPS_PATH_BUILD_FMT,
                    pacDirPath,
                    pacMapName);

    if (tLength > tImageFilePathSize)
    {
        // This means that provided buffer is less then full path length
        return SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
 *
 *   bQueryMapByName
 *
 *****************************************************************************/
static BOOLEAN bQueryMapByName (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    const char *pacMapName,
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
    MAPS_DB_QUERY_MAP_BY_NAME_STRUCT *psQueryResult
        )
{
    BOOLEAN bOk = FALSE;
    MAPS_APP_OBJECT_STRUCT *psAppObj;

    if (pacMapName != NULL)
    {
        psAppObj =
            psGetAppFacingObject((MAPS_SERVICE_OBJECT)psObj);

        if (psAppObj != NULL)
        {
            psQueryResult->pcPath = (char *)NULL;

            psAppObj->asBindParams[MAPS_SELECT_BY_NAME_STMT_ATTR_NAME].eType =
                SQL_BIND_TYPE_C_STRING;
            psAppObj->asBindParams[MAPS_SELECT_BY_NAME_STMT_ATTR_NAME].pvData =
                (void *)pacMapName;

            psAppObj->asBindParams[MAPS_SELECT_BY_NAME_STMT_ATTR_LEVEL].eType =
                SQL_BIND_TYPE_UN32;
            psAppObj->asBindParams[MAPS_SELECT_BY_NAME_STMT_ATTR_LEVEL].pvData =
                (void *)(size_t)eDetailLevel;

            bOk = SQL_INTERFACE.bExecutePreparedStatement(
                psAppObj->hSQLMemoryConnection,
                psAppObj->hFindByNameStmt,
                (SQL_QUERY_RESULT_HANDLER)bSelectMapByNameMemoryCallback,
                psQueryResult,
                MAPS_SELECT_BY_NAME_STMT_ATTR_COUNT,
                psAppObj->asBindParams);

            SMSO_vUnlock((SMS_OBJECT)psAppObj);
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bQueryMarketImagesVersion
 *
 *****************************************************************************/
static BOOLEAN bQueryMarketImagesVersion (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    const char *pacMapName,
    UN32 *pun32LatestVersion
        )
{
    BOOLEAN bOk = FALSE;
    MAPS_APP_OBJECT_STRUCT *psAppObj;

    if (pacMapName != NULL)
    {
        psAppObj =
            psGetAppFacingObject((MAPS_SERVICE_OBJECT)psObj);

        if (psAppObj != NULL)
        {
            snprintf(&psObj->acBuffer[0], sizeof(psObj->acBuffer),
                MAPS_MEMORY_SELECT_MARKET_IMAGES_VERSION,
                pacMapName);

            // Perform the SQL query and process the result
            bOk = SQL_INTERFACE.bQuery(
                psAppObj->hSQLMemoryConnection,
                &psObj->acBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bSelectImagesVersionMemoryCallback,
                pun32LatestVersion);

            SMSO_vUnlock((SMS_OBJECT)psAppObj);
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bInitAppFacingObject
 *
 *****************************************************************************/
static BOOLEAN bInitAppFacingObject (
    MAPS_MGR_OBJECT_STRUCT *psObj
        )
{
    // Create object (initially locked)
    psObj->psAppObj = (MAPS_APP_OBJECT_STRUCT *)SMSO_hCreate(
        MAPS_APP_FACING_OBJECT_NAME,
        sizeof(MAPS_APP_OBJECT_STRUCT),
        SMS_INVALID_OBJECT, // No parent
        TRUE);              // Self-locked

    if (psObj->psAppObj == NULL)
    {
        return FALSE;
    }

    // Relinquish ownership
    SMSO_vUnlock((SMS_OBJECT)psObj->psAppObj);

    return TRUE;
}

/*****************************************************************************
 *
 *   vUninitAppFacingObject
 *
 *****************************************************************************/
static void vUninitAppFacingObject(
    MAPS_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bLocked = FALSE;

    if (psAppObj != NULL)
    {
        // Lock the app object
        bLocked =
            SMSO_bLock((SMS_OBJECT)psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MAPS_APP_FACING_OBJECT_NAME": Unable to lock object");
            return;
        }

        vReleaseDatabaseConnectionsLocked(psAppObj);

        // Destroy object itself
        SMSO_vDestroy((SMS_OBJECT)psAppObj);
    }

    return;
}

/*****************************************************************************
 *
 *   psGetAppFacingObject
 *
 *****************************************************************************/
static MAPS_APP_OBJECT_STRUCT *psGetAppFacingObject(
    MAPS_SERVICE_OBJECT hMapsService
        )
{
    MAPS_MGR_OBJECT_STRUCT *psObj =
        (MAPS_MGR_OBJECT_STRUCT *)hMapsService;

    do
    {
        BOOLEAN bValid, bLocked;

        bValid =
            DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hMapsService);

        if (bValid == FALSE)
        {
            break;
        }

        bLocked =
            SMSO_bLock((SMS_OBJECT)psObj->psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            break;
        }

        return psObj->psAppObj;
    } while (FALSE);

    return NULL;
}

/*****************************************************************************
 *
 *   n16GetDescriptor
 *
 *
 *****************************************************************************/
static N16 n16GetDescriptor (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacName,
    MAPS_MARKET_DESCRIPTOR_STRUCT *psDescriptor
        )
{
    MAPS_APP_OBJECT_STRUCT *psAppObj = NULL;
    UN16 un16Top, un16Bottom, un16Index;
    N8 n8Result;
    N16 n16DescriptorIdx = -1;
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eLevel = MAPS_SERVICE_DETAIL_LEVEL_LOW;

    if ((pacName == NULL) ||
        ((eLevel < MAPS_SERVICE_DETAIL_LEVEL_LOW) ||
        (eLevel > MAPS_SERVICE_DETAIL_LEVEL_HIGHEST)))
    {
        // Invalid input
        return -1;
    }

    psAppObj = psGetAppFacingObject(hMapsService);
    if (psAppObj != NULL)
    {
        un16Top = psAppObj->un16MarketsCount;
        un16Bottom = 0;
        do
        {
            un16Index = (un16Bottom + un16Top) / 2;
            n8Result = (N8)strcmp(
                psAppObj->asDescriptors[un16Index].acUpdateName,
                pacName);

            if (n8Result == 0)
            {
                if (psDescriptor != NULL)
                {
                    psDescriptor->tMarketID =
                        psAppObj->asDescriptors[un16Index].tMarketID;

                    strncpy(psDescriptor->acMarketName,
                        psAppObj->asDescriptors[un16Index].acMarketName,
                        MAPS_SUBNAME_LENGHT_MAX);

                    strncpy(psDescriptor->acUpdateName,
                        psAppObj->asDescriptors[un16Index].acUpdateName,
                        MAPS_SUBNAME_LENGHT_MAX);

                    psDescriptor->un32Version =
                        psAppObj->asDescriptors[un16Index].un32Version;
                }

                n16DescriptorIdx = (N16)un16Index;
                break;
            }
            else if (n8Result > 0)
            {
                un16Top = un16Index - 1;
            }
            else
            {
                un16Bottom = un16Index + 1;
            }
        } while (un16Top >= un16Bottom);
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return n16DescriptorIdx;
}

/*****************************************************************************
 *
 *   bSetVersion
 *
 *****************************************************************************/
static BOOLEAN bSetVersion (
    MAPS_SERVICE_OBJECT hMapsService,
    UN16 un16Index,
    UN32 un32Version
        )
{
    BOOLEAN bSuccess = FALSE;

    if (un16Index < MAPS_MARKETS_COUNT_MAX)
    {
        MAPS_APP_OBJECT_STRUCT *psAppObj;

        psAppObj = psGetAppFacingObject(hMapsService);
        if (psAppObj != NULL)
        {
            if (un32Version > psAppObj->asDescriptors[un16Index].un32Version)
            {
                psAppObj->asDescriptors[un16Index].un32Version = un32Version;
            }

            SMSO_vUnlock((SMS_OBJECT)psAppObj);
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   eResolutionToEnum
 *
 *****************************************************************************/
static MAPS_SERVICE_DETAIL_LEVEL_ENUM eResolutionToEnum (
    UN16 un16HorResolution
        )
{
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eLevel;

    switch (un16HorResolution)
    {
        case MAPS_ZOOM_X_RESOLUTION_LOW:
        {
            eLevel = MAPS_SERVICE_DETAIL_LEVEL_LOW;
            break;
        }

        case MAPS_ZOOM_X_RESOLUTION_MEDIUM:
        {
            eLevel = MAPS_SERVICE_DETAIL_LEVEL_MEDIUM;
            break;
        }

        case MAPS_ZOOM_X_RESOLUTION_HIGH:
        {
            eLevel = MAPS_SERVICE_DETAIL_LEVEL_HIGH;
            break;
        }
        case MAPS_ZOOM_X_RESOLUTION_HIGHEST:
        {
            eLevel = MAPS_SERVICE_DETAIL_LEVEL_HIGHEST;
            break;
        }

        default:
        {
            eLevel = MAPS_SERVICE_DETAIL_LEVEL_UNKNOWN;
        }
    }

    return eLevel;
}

/*****************************************************************************
 *
 *   bClearPersistentDirectory
 *
 *****************************************************************************/
static BOOLEAN bClearPersistentDirectory(
    MAPS_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hDirItemsList = OSAL_INVALID_OBJECT_HDL;
    BOOLEAN bOk = FALSE;

    eReturnCode =
        OSAL.eFileSystemGetDirItems(&hDirItemsList,
            psObj ->pcPersistentDBDirPath,
            (OSAL_FS_FILE_QUALIFIER)NULL,
            NULL);

    if ((eReturnCode == OSAL_SUCCESS) &&
        (hDirItemsList != OSAL_INVALID_OBJECT_HDL))
    {
        eReturnCode = OSAL.eLinkedListIterate(
            hDirItemsList,
            (OSAL_LL_ITERATOR_HANDLER)bRemovePersistentFile,
            (void*)psObj);

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eFileSystemReleaseDirItems(hDirItemsList);
                if (eReturnCode == OSAL_SUCCESS)
                {
                    // Successful cleanup!
                    bOk = TRUE;
                }
            }
    }
    return bOk;
}

/*****************************************************************************
 *
 *   bRemovePersistentFile
 *
 *****************************************************************************/
static BOOLEAN bRemovePersistentFile(
    const char *pcDirItem,
    void *pvArg
        )
{
    MAPS_MGR_OBJECT_STRUCT *psObj = (MAPS_MGR_OBJECT_STRUCT *)pvArg;
    BOOLEAN bOk = FALSE;
    N32 n32Result;

    if (pcDirItem == NULL)
    {
        return FALSE;
    }

    snprintf(&psObj->acBuffer[0],
        sizeof(psObj->acBuffer),
        MAPS_PATH_BUILD_FMT,
        psObj->pcPersistentDBDirPath,
        pcDirItem);

    n32Result = (N32)remove(&psObj->acBuffer[0]);
    if (n32Result == 0)
    {
        bOk = TRUE;
    }
    return bOk;
}

/*******************************************************************************
*
*   bCreateDatabaseConnectionsLocked
*
*******************************************************************************/
static BOOLEAN bCreateDatabaseConnectionsLocked (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        )
{
    char *pacRefDatabaseFullPath = (char*)NULL;
    char *pacPersistDatabaseFullPath = (char*)NULL;
    BOOLEAN bOk = FALSE;

    if (psObj == NULL)
    {
        return FALSE;
    }

    do
    {
        // Construct the path to the Reference DB
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pcReferenceDBDirPath,
            NULL,
            MAPS_REF_DATABASE_FILENAME,
            &pacRefDatabaseFullPath);

        if (bOk == FALSE)
        {
            // Error! Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Try to connect to the reference database
        psObj->psAppObj->hSQLRefConnection =
            DB_UTIL_hConnectToReference(
                &pacRefDatabaseFullPath[0],
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyDBVersion,
                NULL,
                peErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);

        // We don't need the database file path anymore
        SMSO_vDestroy((SMS_OBJECT)pacRefDatabaseFullPath);

        // Check the connection
        if (psObj->psAppObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            // Note: Error code was set inside the DB_UTIL_hConnectToReference() call
            bOk = FALSE;
            break;
        }

        // Construct the path to the persistent DB
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pcPersistentDBDirPath,
            NULL,
            MAPS_PERSIST_DATABASE_FILENAME,
            &pacPersistDatabaseFullPath);

        if (bOk == FALSE)
        {
            // Error! Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Try to connect to the persistent DB
        psObj->psAppObj->hSQLPersistConnection =
                    DB_UTIL_hConnectToPersistent(
                        &pacPersistDatabaseFullPath[0],
                        (DB_UTIL_CREATE_DB_HANDLER)bCreatePersistentDB,
                        (void *)psObj,
                        (DB_UTIL_CHECK_VERSION_HANDLER)NULL,
                        NULL,
                        peErrorCode);

        // We don't need the database file path anymore
        SMSO_vDestroy((SMS_OBJECT)pacPersistDatabaseFullPath);

        // Check the connection
        if (psObj->psAppObj->hSQLPersistConnection ==
            SQL_INTERFACE_INVALID_OBJECT)
        {
            // Note: Error code was set inside the DB_UTIL_hConnectToReference() call
            bOk = FALSE;
            break;
        }

        bOk = bCreateMemoryDB(psObj,
            psObj->psAppObj->hSQLRefConnection,
            psObj->psAppObj->hSQLPersistConnection,
            &psObj->psAppObj->hSQLMemoryConnection);

        if (bOk == FALSE)
        {
            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            break;
        }

        bOk = bCreatePreparedStatements(psObj->psAppObj);
        if (bOk == FALSE)
        {
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
        }
    } while (FALSE);

    return bOk;
}

/*******************************************************************************
*
*   bBuildDescriptorsArrayLocked
*
*******************************************************************************/
static BOOLEAN bBuildDescriptorsArrayLocked (
    MAPS_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    if (psObj == NULL)
    {
        return FALSE;
    }

    psObj->psAppObj->un16MarketsCount = 0;

    bOk = SQL_INTERFACE.bQuery(
        psObj->psAppObj->hSQLMemoryConnection,
        MAPS_MEMORY_SELECT_AREA_DESCRIPTION,
        (SQL_QUERY_RESULT_HANDLER)bFillDescriptorsArray,
        (void *)psObj);

    return bOk;
}

/*******************************************************************************
*
*   vReleaseDatabaseConnectionsLocked
*
*******************************************************************************/
static void vReleaseDatabaseConnectionsLocked (
    MAPS_APP_OBJECT_STRUCT *psAppObj
        )
{
    if (psAppObj == NULL)
    {
        return;
    }

    // Disconnect from the reference database
    if (psAppObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psAppObj->hSQLRefConnection);
        psAppObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Disconnect from the persistent database
    if (psAppObj->hSQLPersistConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psAppObj->hSQLPersistConnection);
        psAppObj->hSQLPersistConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Disconnect from the memory database
    if (psAppObj->hSQLMemoryConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        if (psAppObj->hFindByNameStmt !=
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            SQL_INTERFACE.vDestroyPreparedStatement(
                psAppObj->hSQLMemoryConnection,
                psAppObj->hFindByNameStmt);
            psAppObj->hFindByNameStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;
        }

        if (psAppObj->hFindNearestStmt !=
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            SQL_INTERFACE.vDestroyPreparedStatement(
                psAppObj->hSQLMemoryConnection,
                psAppObj->hFindNearestStmt);
            psAppObj->hFindNearestStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;
        }

        if (psAppObj->hFindMapDetailsStmt !=
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            SQL_INTERFACE.vDestroyPreparedStatement(
                psAppObj->hSQLMemoryConnection,
                psAppObj->hFindMapDetailsStmt);
            psAppObj->hFindMapDetailsStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;
        }

        SQL_INTERFACE.vDisconnect(psAppObj->hSQLMemoryConnection);
        psAppObj->hSQLMemoryConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    return;
}

/*******************************************************************************
*
*   bCreatePreparedStatements
*
*******************************************************************************/
static BOOLEAN bCreatePreparedStatements (
    MAPS_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk = FALSE;

    if (psAppObj == NULL)
    {
        return FALSE;
    }

    do
    {
        psAppObj->hFindByNameStmt = 
            SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLMemoryConnection,
                MAPS_MEMORY_SELECT_MAP_BY_NAME);

        if (psAppObj->hFindByNameStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            break;
        }

        psAppObj->hFindNearestStmt = 
            SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLMemoryConnection,
                MAPS_MEMORY_SELECT_NEAREST_AREA_DESCRIPTION);

        if (psAppObj->hFindNearestStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            break;
        }

        psAppObj->hFindMapDetailsStmt = 
            SQL_INTERFACE.hCreatePreparedStatement(
                psAppObj->hSQLMemoryConnection,
                MAPS_MEMORY_SELECT_MAP_DETAILS);

        if (psAppObj->hFindMapDetailsStmt ==
            SQL_PREPARED_STATEMENT_INVALID_HANDLE)
        {
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   eCreateUpdateImageFile
 *
 *   Creates new image file by copying data from RFD Update file.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCreateUpdateImageFile(
    MAPS_SERVICE_OBJECT hMapsService,
    FILE *psRFDFile,
    size_t tFileSize,
    const char *pacMarketName,
    UN16 un16HorResolution,
    UN32 un32Version
        )
{
    MAPS_MGR_OBJECT_STRUCT *psObj =
            (MAPS_MGR_OBJECT_STRUCT *)hMapsService;
    char acFileName[MAPS_IMAGE_FILE_NAME_LENGHT_MAX];
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
    size_t tBytesRead, tBytesWrite;
    FILE *psImageFile;

    if ((pacMarketName == NULL) ||
        (psRFDFile == NULL) ||
        (hMapsService == MAPS_SERVICE_INVALID_OBJECT))
    {
        return SMSAPI_RETURN_CODE_BAD_ARGUMENT;
    }

    do
    {
        // First, create path to new image file
        snprintf(acFileName,
            sizeof(acFileName),
            MAPS_IMAGE_FILE_NAME_FMT,
            pacMarketName,
            un16HorResolution,
            un32Version);

        eErrorCode = eBuildFullMapPath(psObj,
            acFileName,
            psObj->pcPersistentDBDirPath, // Persistent dir for updates.
            &psObj->acBuffer[0],
            sizeof(psObj->acBuffer));

        if (eErrorCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        psImageFile = fopen(&psObj->acBuffer[0], "wb");

        if (psImageFile == NULL)
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        while (tFileSize != 0)
        {
            if (tFileSize > MAPS_MGR_BUFFER_SIZE)
            {
                tBytesRead = fread(&psObj->aun8Buffer[0],
                    sizeof(UN8), MAPS_MGR_BUFFER_SIZE,
                    psRFDFile);
                tFileSize -= MAPS_MGR_BUFFER_SIZE;
            }
            else
            {
                tBytesRead = fread(&psObj->aun8Buffer[0],
                    sizeof(UN8),
                    tFileSize,
                    psRFDFile);
                tFileSize = 0;
            }
            tBytesWrite = fwrite(&psObj->aun8Buffer[0],
                sizeof(UN8),
                tBytesRead,
                psImageFile);

            if (tBytesWrite != tBytesRead)
            {
                eErrorCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }
        }

        // End copying file
        fclose(psImageFile);

        if (eErrorCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Remove created file on error
            remove(&psObj->acBuffer[0]);
        }
    } while (FALSE);

    return eErrorCode;
}

