/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the maps
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef MAPS_DB_CONSTANTS_H_
#define MAPS_DB_CONSTANTS_H_

  /**************/
 /** INCLUDES **/
/**************/
#include "sms_api.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* Database default folders */
#define MAPS_REF_DB_FOLDER_DEFAULT        "maps"
#define MAPS_PERSIST_DB_FOLDER_DEFAULT \
    MAPS_REF_DB_FOLDER_DEFAULT"/updates"

/* Database File Names */
#define MAPS_REF_DATABASE_FILENAME        "maps_ref.db"
#define MAPS_PERSIST_DATABASE_FILENAME    "maps.db"


/* Database Table Names */

// Reference DB tables
#define VERSION_TABLE_NAME "version"

#define MAPS_AREA_TABLE_NAME "maps_area_description"

#define MAPS_SET_TABLE_NAME "map_set"

// Persistent DB tables
#define MAPS_UPDATED_MAPS_TABLE_NAME "maps"

#define MAPS_UPDATED_IMAGES_TABLE_NAME "updated_images"

/* Database Table Attributes */
#define MAPS_DATABASE_FILE_VERSION (3)

/* Reference Database Queries */
// Select all from version
#define MAPS_SELECT_DB_VERSION \
    "SELECT * from "VERSION_TABLE_NAME";"

#define MAPS_SELECT_CURRENT_VERSION \
    "SELECT zoom_level, current_version FROM "MAPS_SET_TABLE_NAME \
    " WHERE area_desc_id = %d;"

#define MAPS_INSERT_AREA_DESCRIPTION \
    "INSERT INTO "MAPS_AREA_TABLE_NAME" " \
    "VALUES(?, ?, ?, ?, ?, ?, ?);"

#define MAPS_INSERT_MAP_SET \
    "INSERT INTO "MAPS_SET_TABLE_NAME" " \
    "VALUES(?, ?, ?, ?, ?, ?);"

#define MAPS_UPDATE_MAP_SET \
      "UPDATE "MAPS_SET_TABLE_NAME" SET " \
      "map_id = ?, " \
      "_data = ?, " \
      "current_version = ? " \
      "where map_id = ?1;"

/* Persistent Database Queries */
#define MAPS_CREATE_UPDATED_MAPS_TABLE \
    "CREATE TABLE "MAPS_UPDATED_MAPS_TABLE_NAME" ( " \
    "market_name VARCHAR, " \
    "zoom_level INT, " \
    "_data VARCHAR, " \
    "version INT, " \
    "PRIMARY KEY (market_name, zoom_level) );"

#define MAPS_CREATE_UPDATED_IMAGES_TABLE \
    "CREATE TABLE "MAPS_UPDATED_IMAGES_TABLE_NAME" ( " \
    "_data VARCHAR );"

#define MAPS_PERSISTENT_SELECT_MAP_BY_NAME \
    "SELECT * FROM "MAPS_UPDATED_MAPS_TABLE_NAME \
    " WHERE market_name = '%s' AND zoom_level = %d;"

#define MAPS_SELECT_UPDATED_MAPS \
    "SELECT * FROM "MAPS_UPDATED_MAPS_TABLE_NAME";"

#define MAPS_SELECT_UPDATED_IMAGE_PATH \
    "SELECT _data FROM "MAPS_UPDATED_IMAGES_TABLE_NAME";"

#define MAPS_INSERT_UPDATED_MAPS \
    "REPLACE INTO "MAPS_UPDATED_MAPS_TABLE_NAME" " \
    "VALUES(?, ?, ?, ?);"

#define MAPS_INSERT_UPDATED_IMAGES \
    "REPLACE INTO "MAPS_UPDATED_IMAGES_TABLE_NAME" " \
    "VALUES(?);"

#define MAPS_DROP_UPDATED_IMAGES_TABLE \
    "DELETE FROM "MAPS_UPDATED_IMAGES_TABLE_NAME";"

/* Memory Database Queries*/
#define MAPS_MEMORY_ADD_COLUMN \
    "ALTER TABLE "MAPS_SET_TABLE_NAME \
    " ADD baseline INT DEFAULT 1;"

#define MAPS_MEMORY_MAP_SET_UPDATE \
    "UPDATE "MAPS_SET_TABLE_NAME" SET " \
    "_data = '%s', " \
    "version = %d, " \
    "baseline = %d " \
    "WHERE zoom_level = %d AND area_desc_id IN " \
    "(SELECT area_id FROM "MAPS_AREA_TABLE_NAME \
    " WHERE market_name = '%s');"

#define MAPS_SELECT_BY_NAME_STMT_ATTR_NAME  (0)
#define MAPS_SELECT_BY_NAME_STMT_ATTR_LEVEL (1)
#define MAPS_SELECT_BY_NAME_STMT_ATTR_COUNT (2)

#define MAPS_MEMORY_SELECT_MAP_BY_NAME \
    "SELECT * FROM "MAPS_SET_TABLE_NAME" WHERE area_desc_id IN " \
    "(SELECT area_id FROM "MAPS_AREA_TABLE_NAME" WHERE market_name =?) AND " \
    "zoom_level =?;"

#define MAPS_MEMORY_SELECT_MARKET_IMAGES_VERSION \
    "SELECT MAX(version) FROM "MAPS_SET_TABLE_NAME" WHERE area_desc_id IN " \
    "(SELECT area_id FROM "MAPS_AREA_TABLE_NAME" WHERE market_name = '%s');"

#define MAPS_SELECT_NEAREST_STMT_ATTR_MIN_LAT (0)
#define MAPS_SELECT_NEAREST_STMT_ATTR_MAX_LAT (1)
#define MAPS_SELECT_NEAREST_STMT_ATTR_MAX_LON (2)
#define MAPS_SELECT_NEAREST_STMT_ATTR_MIN_LON (3)
#define MAPS_SELECT_NEAREST_STMT_ATTR_COUNT   (4)

#define MAPS_MEMORY_SELECT_NEAREST_AREA_DESCRIPTION \
    "SELECT * from "MAPS_AREA_TABLE_NAME" WHERE ? > MIN_LAT AND" \
    " ? < MAX_LAT AND ? < MAX_LON AND ? > MIN_LON;"

#define MAPS_MEMORY_SELECT_AREA_DESCRIPTION \
    "SELECT * FROM "MAPS_AREA_TABLE_NAME" ORDER BY update_name;"

#define MAPS_SELECT_MAP_DETAILS_STMT_ATTR_AREA_ID (0)
#define MAPS_SELECT_MAP_DETAILS_STMT_ATTR_LEVEL   (1)
#define MAPS_SELECT_MAP_DETAILS_STMT_ATTR_COUNT   (2)

#define MAPS_MEMORY_SELECT_MAP_DETAILS \
    "SELECT * FROM "MAPS_SET_TABLE_NAME" WHERE area_desc_id =? AND" \
    " zoom_level =?;"

// Constant which defines the size of the buffer used by
// MAPS service for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define MAPS_SHARED_BUFFER_LENGTH (400)

// Fractional bits used to represent the loc coordinates
#define MAPS_FRACTIONAL_BITS (22)

// Max lenght of Update Name stored in DB
#define MAPS_SUBNAME_LENGHT_MAX (16)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/
typedef UN32 MAPS_VERSION;
#define MAPS_INVALID_VERSION 0xFFFFFFFF

// Enumeration specifying all the available fields
// in the db_version database relation
typedef enum db_version_fields_enum
{
    DB_VERSION_FIELD_DB_VER = 0,
    DB_VERSION_MAX_FIELDS
} DB_VERSION_FIELDS_ENUM;

typedef enum db_area_desc_fields_enum
{
    DB_MAPS_AREA_DESC_ID = 0,
    DB_MAPS_AREA_MARKET_ID,
    DB_MAPS_AREA_DESC_MARKET_NAME,
    DB_MAPS_AREA_DESC_UPDATE_NAME,
    DB_MAPS_AREA_DESC_MAX_LAT,
    DB_MAPS_AREA_DESC_MIN_LAT,
    DB_MAPS_AREA_DESC_MAX_LON,
    DB_MAPS_AREA_DESC_MIN_LON,
    DB_MAPS_AREA_DESC_CENTER_LAT,
    DB_MAPS_AREA_DESC_CENTER_LON,
    DB_MAPS_AREA_DESC_COUNT
} DB_AREA_DESC_FIELDS_ENUM;

typedef enum db_reference_maps_set_fields_enum
{
    DB_MAPS_SET_MAP_ID = 0,
    DB_MAPS_SET_AREA_DESC_ID,
    DB_MAPS_SET_DATA,
    DB_MAPS_SET_ZOOM_LEVEL,
    DB_MAPS_SET_VERSION,
    DB_MAPS_SET_COUNT
} DB_REFERENCE_MAPS_SET_FIELDS_ENUM;

  typedef enum db_memory_maps_set_fields_enum
  {
      DB_MEMORY_MAPS_SET_MAP_ID = 0,
      DB_MEMORY_MAPS_SET_AREA_DESC_ID,
      DB_MEMORY_MAPS_SET_DATA,
      DB_MEMORY_MAPS_SET_ZOOM_LEVEL,
      DB_MEMORY_MAPS_SET_VERSION,
      DB_MEMORY_MAPS_SET_BASELINE,
      DB_MEMORY_MAPS_SET_COUNT
  } DB_MEMORY_MAPS_SET_FIELDS_ENUM;

  typedef enum db_maps_updated_maps_fields_enum
  {
      DB_MAPS_UPDATED_MAPS_AREA_NAME = 0,
      DB_MAPS_UPDATED_MAPS_ZOOM_LEVEL,
      DB_MAPS_UPDATED_MAPS_DATA,
      DB_MAPS_UPDATED_MAPS_VERSION,
      DB_MAPS_UPDATED_MAPS_COUNT
  } DB_MAPS_UPDATED_MAPS_FIELDS_ENUM;

  typedef enum db_maps_updated_images_fields_enum
  {
      DB_MAPS_UPDATED_IMAGES_DATA = 0,
      DB_MAPS_UPDATED_IMAGES_COUNT
  } DB_MAPS_UPDATED_IMAGES_FIELDS_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// Structure which specifies all attributes
// of a single database version row that is
// available within the maps database
// These are the raw values that are retrieved from the database
typedef struct maps_version_row_struct
{
    UN8 un8DBVer;           // Database schema version
    MAPS_VERSION tBaselineVersion;
} MAPS_VERSION_ROW_STRUCT;

// Structure which specifies all attributes
// of a single map area description row that is
// available within the maps database
// These are the raw values that are retrieved from the database
typedef struct maps_area_description_row_struct
{
    UN32 un32MapsAreaDescID;
    STRING_OBJECT hMapName;
    N32 n32MaxLat;
    N32 n32MinLat;
    N32 n32MaxLon;
    N32 n32MinLon;
    N32 n32CenterLat;
    N32 n32CenterLon;
} MAPS_AREA_DESCRIPTION_ROW_STRUCT;

  // Structure which specifies all attributes
  // of a single map set row that is
  // available within the maps database
  // These are the raw values that are retrieved from the database
typedef struct maps_set_row_struct
{
    UN32 un32MapSetID;
    UN32 un32DescID;
    STRING_OBJECT hData;
    UN8 un8DetalizationLevel;
    UN16 un16BaselineVersion;
    UN16 un16CurrentVersion;
} MAPS_SET_ROW_STRUCT;

  // Structure which specifies all attributes
  // of a single map set update row that is
  // available only within the persistent maps database
  // These are the raw values that are retrieved from the database
typedef struct maps_updates_row_struct
{
    char *pcMarketName;
    char *pcData;
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eLevel;
    UN32 un32ToVersion;
} MAPS_UPDATES_ROW_STRUCT;

typedef struct maps_db_query_map_by_name_struct
{
    UN32 un32AreaDescriptionID;
    char *pcPath;
    UN32 un32Version;
    BOOLEAN bBaseline;
} MAPS_DB_QUERY_MAP_BY_NAME_STRUCT;

typedef struct maps_db_query_nearest_struct
{
    BOOLEAN bSuccess;
    char pacMarketName[MAPS_SUBNAME_LENGHT_MAX];
    UN32 un32MinDistanceAreaID;
    DISTANCE_OBJECT hMinDistance;
    LOCATION_OBJECT hUserLocation;
    LOCATION_OBJECT hMinDistanceLocation;
    char *pcPath;
    BOOLEAN bBaseline;
} MAPS_DB_QUERY_NEAREST_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

// Union to group all the row definitions
// in a convenient manner
typedef union maps_db_row_union
{
    MAPS_VERSION_ROW_STRUCT sVersion;

} MAPS_DB_ROW_UNION;

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif // MAPS_DB_CONSTANTS_H_
