/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:LOCID implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "radio.h"
#include "string_obj.h"
#include "traffic_locid_obj.h"
#include "locid_obj.h"
#include "_locid_obj.h"


/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreate
*
* This object interface method is used to create a LOCID object with an ID
* value
*
* Inputs:
*
*   tID - The LOC_ID that is the core component of this LOCID_OBJECT
*
* Outputs:
*
*   A handle to a LOCID_OBJECT
*
*****************************************************************************/
static LOCID_OBJECT hCreate (
    LOC_ID tID
        )
{
    return LOCID_hCreate(tID, LOCID_TYPE_UNKNOWN, NULL);
}

/*****************************************************************************
*
*   tID
*
* This object interface method is used to access the tID field of the
* specifc LOCID_OBJECT
*
* Inputs:
*
*   hLocID - The LOCID_OBJECT
*
* Outputs:
*
*   A LOC_ID value for this LOCID_OBJECT
*
*****************************************************************************/
static LOC_ID tID (
    LOCID_OBJECT hLocID
        )
{
    LOCID_OBJECT_STRUCT *psObj =
        (LOCID_OBJECT_STRUCT *)hLocID;
    BOOLEAN bValid;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hLocID);
    if(bValid == FALSE)
    {
        // Error!
        return LOC_INVALID_ID;
    }

    return psObj->tID;
}

/*****************************************************************************
*
*   eType
*
* This object interface method returns the type of LOCID this is
*
* Inputs:
*
*   hLocID - The LOCID_OBJECT object to get they type of
*
* Outputs:
*
*   A LOCID_TYPE_ENUM indicating the type of LOCID if known
*
*****************************************************************************/
static LOCID_TYPE_ENUM eType (
    LOCID_OBJECT hLocID
        )
{
    LOCID_OBJECT_STRUCT *psObj =
        (LOCID_OBJECT_STRUCT *)hLocID;
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hLocID);
    if(bValid == FALSE )
    {
        // Error!
        return LOCID_TYPE_UNKNOWN;
    }

    return psObj->eType;
}

/*****************************************************************************
*
*   hDuplicate
*
* This object interface method is used to access the do a "deep-copy" of a
* LOCID.
*
* Inputs:
*
*   hLocID - The LOCID_OBJECT object to duplicate
*
* Outputs:
*
*   A LOCID that is the duplicate of hLocID
*
*****************************************************************************/
static LOCID_OBJECT hDuplicate (
    LOCID_OBJECT hLocID
        )
{
    LOCID_OBJECT_STRUCT *psObj =
        (LOCID_OBJECT_STRUCT *)hLocID;
    LOCID_OBJECT hLocIDCopy;
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hLocID);
    if(bValid == FALSE )
    {
        // Error!
        return LOCID_INVALID_OBJECT;
    }

    // if this is a known subtype, pass it on
    if (psObj->psInterface->hDuplicate != NULL)
    {
        hLocIDCopy =
            (LOCID_OBJECT)psObj->psInterface->hDuplicate((void*)psObj);
    }
    else
    {
        hLocIDCopy = LOCID_hCreate(psObj->tID,
                                   psObj->eType,
                                   psObj->psInterface);
    }

    return hLocIDCopy;
}

/*****************************************************************************
*
*   n16Compare
*
* This interface method compares the relationship of two LOCIDs.
* This performs a relationship (lt, gt, eq) compare typically used for sorting.
*
*       Outputs:
*               0   - LOCIDs have the same value (IDs are equal)
*               > 0 - hLocID1 is greater than (after) hLocID2
*               < 0 - hLocID1 is less than (before) hLocID2
*                       (or error)
*
*****************************************************************************/
static N16 n16Compare (
    LOCID_OBJECT hLocID1,
    LOCID_OBJECT hLocID2
        )
{
    LOCID_OBJECT_STRUCT *psObj1 =
        (LOCID_OBJECT_STRUCT *)hLocID1;
    LOCID_OBJECT_STRUCT *psObj2 =
        (LOCID_OBJECT_STRUCT *)hLocID2;
    N16 n16Return = N16_MIN;
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hLocID1);
    if(bValid == TRUE)
    {
        bValid = SMSO_bValid((SMS_OBJECT)hLocID2);
    }
    if(bValid == FALSE)
    {
        // Error!
        return N16_MIN;
    }

    // if this is a known subtype, pass it on
    if (psObj1->psInterface->n16Compare != NULL &&
        psObj1->eType == psObj2->eType)
    {
        n16Return =
            psObj1->psInterface->n16Compare((void*)psObj1, (void*) psObj2);
    }
    else
    {
        // Just compare the ID values
        n16Return = COMPARE(psObj1->tID, psObj2->tID);
    }

    return n16Return;
}

/*****************************************************************************
*
*   n32FWrite
*
* This object interface method is used to write the contents of a LOCID
* (performing a deep write) to a device specified by psFile. The intent of
* this method is to effectively store the contents of a LOCID for later
* retrieval (perhaps after a power cycle). The entire LOCID contents are written
* to the device, with enough information to recreate the exact LOCID
* when the hFRead() method is called.
*
* Inputs:
*
*   hLocID - The LOCID handle the caller wishes to write.
*   psFile - The device to write the LOCID contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FWrite (
    LOCID_OBJECT hLocID,
    FILE *psFile
        )
{
    LOCID_OBJECT_STRUCT *psObj =
        (LOCID_OBJECT_STRUCT *)hLocID;
    N32 n32Return = 0;
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hLocID);
    if((bValid == FALSE)  || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // if this is a known subtype, pass it on
    if (psObj->psInterface->n32FWrite != NULL)
    {
        n32Return = psObj->psInterface->n32FWrite(
                (void*)psObj, psFile);
    }
    else
    {
        n32Return = LOCID_n32FWrite(hLocID, psFile);
    }

    return n32Return;
}

/*****************************************************************************
*
*   hFRead
*
* This object interface method is used to read from a specified device psFile
* and generate a LOCID from that information. The data read from the device
* must have been previously written by the n32FWrite method. Upon
* successful execution of this method a new LOCID is created (created by the
* caller) which may be used to register for events or presented to the
* UI for display, etc. This method allows the caller to effectively read
* a previously stored LOCID regenerating the original LOCID written(saved) at
* an earlier time.
*
* Inputs:
*
*   psFile - The device to read the LOCID contents from.
*
* Outputs:
*
*   A new LOCID on success, otherwise LOCID_INVALID_OBJECT on failure.
*
*****************************************************************************/
static LOCID_OBJECT hFRead (
    FILE *psFile
        )
{
    LOCID_OBJECT hLocID;
    LOC_ID tID = LOC_INVALID_ID;
    UN32 un32Type = 0;

    size_t tTotal, tLen;

    // Verify input
    if(psFile == NULL)
    {
        return LOCID_INVALID_OBJECT;
    }
    // Read LOCID from file...

    // Read tID
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&tID) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(LOC_ID)));

    if(tTotal != sizeof(LOC_ID))
    {
        // End of file or error occurred
        return LOCID_INVALID_OBJECT;
    }

    // Read the type of LOCID
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&un32Type) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(UN32)));

    if(tTotal != sizeof(UN32))
    {
        // End of file or error occurred
        return LOCID_INVALID_OBJECT;
    }

    // Well, we don't have the interface here to call a object-specifc
    // hFread, so we cheat and call a further hFRead based on type
    switch ((LOCID_TYPE_ENUM)un32Type)
    {
        case LOCID_TYPE_TRAFFIC:
            hLocID = TRAFFIC_LOCID_hFRead(tID, psFile);
            break;
        case LOCID_TYPE_UNKNOWN:
        default:
            hLocID = LOCID.hCreate(tID);
            break;
    }

    // Attempt to create a LOCID using the ID
    return hLocID;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a LOCID's contents to a specified file or device.
* This is mainly helpful during debugging of LOCID's but could be used by
* an application for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a LOCID to a file for the
* purposes of later re-generating the LOCID (for storage of the object).
* This API instead sends the LOCID as a verbose formatted output version.
*
* Inputs:
*
*   hLocID - The LOCID handle the caller wishes to write.
*   psFile - The device to write the LOCID contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    LOCID_OBJECT hLocID,
    FILE *psFile
        )
{
    LOCID_OBJECT_STRUCT *psObj =
        (LOCID_OBJECT_STRUCT *)hLocID;
    N32 n32Return = 0;
    BOOLEAN bValid;

    // Determine if the caller owns this resource
    bValid = SMSO_bValid((SMS_OBJECT)hLocID);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bValid == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // if this is a known subtype, pass it on
    if (psObj->psInterface->n32FPrintf != NULL)
    {
        psObj->psInterface->n32FPrintf(
            (void*)hLocID, psFile);
    }
    else
    {
        // Print LOCATION information header
        n32Return += fprintf(psFile, "LOCID: hLocID = 0x%p\n",
            psObj);

        // Print LOCID
        n32Return += fprintf(psFile, "\tID = %d\n",
            psObj->tID);

        // Print LOCID Type
        n32Return += fprintf(psFile, "\tType = %s\n",
            pacTypeText(psObj->eType));
    }

    return n32Return;
}

/*****************************************************************************
*
*   vDestroy
*
* This object interface method is used by the caller to destroy the specified
* LOC_ID object and all members of the object
*

* Inputs:
*
*   hLocID - A handle to a valid LOC_ID object that the caller wants
*                 to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
static void vDestroy (
    LOCID_OBJECT hLocID
        )
{
    BOOLEAN bValid;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hLocID);
    if(bValid == FALSE)
    {
        // Error!
        return;
    }

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hLocID);

    return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       LOCID_hCreate
*
*****************************************************************************/
LOCID_OBJECT LOCID_hCreate(
    LOC_ID tID,
    LOCID_TYPE_ENUM       eType,
    const LOCID_INTERFACE_STRUCT *psInterface
        )
{
    LOCID_OBJECT_STRUCT *psObj;

    // Create an instance of the LOCID object
    psObj = (LOCID_OBJECT_STRUCT *)
        SMSO_hCreate(
            LOCID_OBJECT_NAME,
            sizeof(LOCID_OBJECT_STRUCT),
            SMS_INVALID_OBJECT,
            FALSE);

    if(psObj == NULL)
    {
        // Error!
        return LOCID_INVALID_OBJECT;
    }

    // Initialize object per inputs
    psObj->eType = eType;
    psObj->tID = tID;

    if (psInterface == NULL)
    {
        psObj->psInterface = &gsDefaultInterface;
    }
    else
    {
        psObj->psInterface = psInterface;
    }

    OSAL.bMemSet(&psObj->uLOCID, 0, sizeof(psObj->uLOCID));

    return (LOCID_OBJECT)psObj;
}

/*****************************************************************************
*
*       LOCID_puSubClassData
*
*****************************************************************************/
LOCID_UNION *LOCID_puSubClassData (
    LOCID_OBJECT hLocID
        )
{
    LOCID_OBJECT_STRUCT *psObj =
            (LOCID_OBJECT_STRUCT *)hLocID;
    BOOLEAN bValid;

    // Verify inputs. Object handle must be valid.
    bValid = SMSO_bValid((SMS_OBJECT)hLocID);
    if(bValid == FALSE)
    {
        // Error!
        return LOCID_INVALID_UNION;
    }

    return &psObj->uLOCID;
}

/*****************************************************************************
*
*       LOCID_n32FWrite
*
*****************************************************************************/
N32 LOCID_n32FWrite (
    LOCID_OBJECT hLocID,
    FILE *psFile
        )
{
    LOCID_OBJECT_STRUCT *psObj =
        (LOCID_OBJECT_STRUCT *)hLocID;
    N32 n32Return = 0;
    UN32 un32Type;
    UN8 un8ByteToWrite;

    // Write LOC_ID
    un8ByteToWrite = BYTE0(psObj->tID);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(psObj->tID);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(psObj->tID);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(psObj->tID);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);

    // Write Type
    un32Type = (UN32)psObj->eType;
    un8ByteToWrite = BYTE0(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(un32Type);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);

    return n32Return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       pacTypeText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacTypeText(
    LOCID_TYPE_ENUM        eType
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case LOCID_TYPE_TRAFFIC:
            pacReturnString = MACRO_TO_STRING(LOCID_TYPE_TRAFFIC);
        break;

        case LOCID_TYPE_THEATER:
            pacReturnString = MACRO_TO_STRING(LOCID_TYPE_THEATER);
        break;

        case LOCID_TYPE_FUEL_STATION:
            pacReturnString = MACRO_TO_STRING(LOCID_TYPE_FUEL_STATION);
        break;

        case LOCID_TYPE_WEATHER:
            pacReturnString = MACRO_TO_STRING(LOCID_TYPE_WEATHER);
        break;

        case LOCID_TYPE_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(LOCID_TYPE_UNKNOWN);
        break;
    }

    return pacReturnString;
}

#ifdef SUPPORT_CUNIT
#include <locid_obj.cunit>
#endif
