/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:LOCATION implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <ctype.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "dsrl_target_obj.h"
#include "string_obj.h"
#include "locid_obj.h"
#include "distance_obj.h"
#include "device_obj.h"

#include "location_obj.h"
#include "_location_obj.h"

#include "math.h"
#include "_osal_fixed.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

// Bosch ID#18: Enable the changes for Bosch distance calculation in this file
#define VARIANT_S_FTR_ENABLE_SMSLIB_CUSTOM_DISTANCE

// start demo-implementation with simple formula using double-type

static inline double fDistSqr(double fLat1, double fLon1,
                    double fLat2, double fLon2) {
    double x = (fLon2-fLon1) * cos((fLat1+fLat2)/2);
    double y = (fLat2-fLat1);
    return x*x + y*y;
}


#define DEG_TO_RAD (3.14159265 / 180)
//#define BIN_POINT_VALUE(binPoint)   ((UN32)1 << ((UN8)(binPoint)))

#define fFixedDegToFloatRad(O) ((DEG_TO_RAD) *  ((double)((O)->n32Value) / BIN_POINT_VALUE((O)->un8NumFractionalBits)))



static inline int bClosestFixed_peha(FIXED_OBJECT_STRUCT *hLatT,
                                 FIXED_OBJECT_STRUCT *hLonT,
                                 FIXED_OBJECT_STRUCT *hLat1,
                                 FIXED_OBJECT_STRUCT *hLon1,
                                 FIXED_OBJECT_STRUCT *hLat2,
                                 FIXED_OBJECT_STRUCT *hLon2
                                 ) {
    float fLatT=fFixedDegToFloatRad(hLatT);
    float fLonT=fFixedDegToFloatRad(hLonT);
    float fLat1=fFixedDegToFloatRad(hLat1);
    float fLon1=fFixedDegToFloatRad(hLon1);
    float fLat2=fFixedDegToFloatRad(hLat2);
    float fLon2=fFixedDegToFloatRad(hLon2);
    return fDistSqr(fLatT, fLonT, fLat1, fLon1) < fDistSqr(fLatT, fLonT, fLat2, fLon2);
}

#ifdef VARIANT_S_FTR_ENABLE_SMSLIB_CUSTOM_DISTANCE
// replacement for bComputeDistanceVector
static inline BOOLEAN bComputeDistanceVector(LOCATION_OBJECT_STRUCT *psObjFrom,
                                              OSAL_FIXED_OBJECT hLatTo,
                                              OSAL_FIXED_OBJECT hLonTo,
                                              OSAL_FIXED_OBJECT hRes
                                              ) {
    float fLatFrom=fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)psObjFrom->hLat);
    float fLonFrom=fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)psObjFrom->hLon);
    float fLatTo=  fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)hLatTo);
    float fLonTo=  fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)hLonTo);
    double fDist1=sqrt(fDistSqr(fLatFrom, fLonFrom, fLatTo, fLonTo));
    
    // convert back to fixed
    // assume 16 bit
    N32 n32Val=fDist1 * 0x10000;
    ((FIXED_OBJECT_STRUCT *)hRes)->n32Value = n32Val;
    ((FIXED_OBJECT_STRUCT *)hRes)->un8NumFractionalBits = 16;

    return TRUE; 
}

#define  SMSLIB_R_EARTH 6371;
float LOCATION_fGetLocationDistKm(LOCATION_OBJECT hLocation1,
                                  LOCATION_OBJECT hLocation2) {

    /* Check for Invalid Types */
    if( (hLocation1 == LOCATION_INVALID_OBJECT) || 
        (hLocation2 == LOCATION_INVALID_OBJECT) )
	 {
	 	 //ETG_TRACE_ERR(("hLocation1/hLocation2 invalid object"));                
		 return 0.0;
	 }

    LOCATION_OBJECT_STRUCT *prLoc1=(LOCATION_OBJECT_STRUCT *)hLocation1;
    LOCATION_OBJECT_STRUCT *prLoc2  =(LOCATION_OBJECT_STRUCT *)hLocation2;
    float fLat1=fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)(prLoc1->hLat));
    float fLon1=fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)(prLoc1->hLon));
    float fLat2=  fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)(prLoc2->hLat));
    float fLon2=  fFixedDegToFloatRad((FIXED_OBJECT_STRUCT *)(prLoc2->hLon));

    double x=(fLon2-fLon1) * cos((fLat2+fLat1)/2);
    double y=(fLat2-fLat1);
    double fDistKm = sqrt(x*x + y*y) * SMSLIB_R_EARTH;

#if 0
    if (SMSLIB_bLogDistance) {
        char sDistKm[30];
        char sLat1[30];
        char sLon1[30];
        char sLat2[30];
        char sLon2[30];
        sprintf(sDistKm, "%f", fDistKm);
        sprintf(sLat1, "%f", fLat1);
        sprintf(sLon1, "%f", fLon1);
        sprintf(sLat2, "%f", fLat2);
        sprintf(sLon2, "%f", fLon2);

        ETG_TRACE_ERR(("fc_sxm_fGetLocationDistKm:p1=%10s:%10s p2=%10s:%10s dist=%10s km",
                         sLat1,sLon1, sLat2, sLon2, sDistKm));
    }
#endif

    return fDistKm; 
}
#else
float LOCATION_fGetLocationDistKm(LOCATION_OBJECT hLocation1,
                                  LOCATION_OBJECT hLocation2) {

    DISTANCE_OBJECT hDistance = DISTANCE_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hFixed = OSAL_FIXED_INVALID_OBJECT;

    hDistance = hDistance(hLocation1, hLocation2);

    hFixed = DISTANCE.hValue(hDistance, DISTANCE_UNIT_TYPE_KILOMETERS);
    float fReturnValue=(float)OSAL_FIXED.n32Value(hFixed);
    /* Check for -Ve Val */
    if(f1<0.0) {
        /* Change -Ve sign to +ve sign */
        f1*=-1;
    }

    fReturnValue = f1 / (l1 << OSAL_FIXED.un8NumFractionalBits(hFixed));
    
    /** Destroying the DISTANCE object created to avoid Memory Leak**/
    DISTANCE.vDestroy(hDistance);
    return fValue;
}


}
#endif
/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreateForRadius
*
* This object interface method is used by the caller to create a LOCATION
* object that is based on a lat/lon position and could include a radius
* component.
*
* This method could be used to specify a single point (un32Radius would be
* equal to zero) or an area that is centered around a point (un32Radius
* would be equal to the number of miles around that point that the LOCATION
* should cover).
*
* Inputs:
*
*   hLat - The latitude for the LOCATION as a OSAL_FIXED_OBJECT
*   hLon - The longitude for the LOCATION as a OSAL_FIXED_OBJECT
*   hRadius - The DISTANCE object representing the number of miles/km
*             that would describe the area around tLat,tLon
*             that this LOCATION should represent
*
* Outputs:
*
*   A handle to a LOCATION object
*
*****************************************************************************/
static LOCATION_OBJECT hCreateForRadius (
    OSAL_FIXED_OBJECT  hLat,
    OSAL_FIXED_OBJECT  hLon,
    DISTANCE_OBJECT hRadius
        )
{
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;
    LOCATION_OBJECT_STRUCT *psObj = NULL;
    N32 n32Lat, n32Lon;
    UN8 un8LatBits, un8LonBits;
    BOOLEAN bSuccess;
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;

    // Check provided Radius object
    bSuccess = SMSO_bValid((SMS_OBJECT)hRadius);
    if ((hRadius != DISTANCE_INVALID_OBJECT) && (bSuccess != TRUE))
    {
        // ERROR!
        // Provided pointer is not the valid object
        return LOCATION_INVALID_OBJECT;
    }

    // Get the fixed object components from the caller
    // we do this since psCreateLocation ends up storing fixed data within
    // the LOCATION structure instead of on the heap.
    n32Lat = OSAL_FIXED.n32Value(hLat);
    un8LatBits = OSAL_FIXED.un8NumFractionalBits(hLat);
    n32Lon = OSAL_FIXED.n32Value(hLon);
    un8LonBits = OSAL_FIXED.un8NumFractionalBits(hLon);

    // Set specific properties for circular location
    uProperties.sCircle.hRadius = hRadius;

    // Create location
    psObj = psCreateObject (
        SMS_INVALID_OBJECT,
        LOCATION_POSITION_TYPE_STATIC,
        LOCATION_TYPE_CIRCLE,
        LOCID_INVALID_OBJECT,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits,
        &uProperties, NULL, TRUE );

    if (psObj != NULL)
    {
        hResult = (LOCATION_OBJECT)psObj;
    }

    return hResult;
}

/*****************************************************************************
*
*   hCreateForLocID
*
* This object interface method is used by the caller to create a LOCATION
* object that is based on a lat/lon position and would have a LOCID associated
* with it
*
* This method could be used to tie a position (in lat,lon) to an application-
* specific ID (represent by the LOCID object)
*
* Inputs:
*
*   hLat - The latitude for the LOCATION as a OSAL_FIXED_OBJECT
*   hLon - The longitude for the LOCATION as a OSAL_FIXED_OBJECT
*   hLocID - The handle to the LOCID_OBJECT used to further descrbe this location
*
* Outputs:
*
*   A handle to a LOCATION object
*
*****************************************************************************/
static LOCATION_OBJECT hCreateForLocID (
    OSAL_FIXED_OBJECT  hLat,
    OSAL_FIXED_OBJECT  hLon,
    LOCID_OBJECT    hLocID
        )
{
    LOCATION_OBJECT_STRUCT *psObj;
    N32 n32Lat, n32Lon;
    UN8 un8LatBits, un8LonBits;
    BOOLEAN bOk;
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;

    bOk = SMSO_bValid((SMS_OBJECT)hLocID);
    if (bOk == FALSE)
    {
        // ERROR!
        // Invalid LOCID object
        return LOCATION_INVALID_OBJECT;
    }

    // Invalid coordinates would indicate that the
    // position is known only by the LOC ID
    if (hLat != OSAL_FIXED_INVALID_OBJECT ||
        hLon != OSAL_FIXED_INVALID_OBJECT)
    {
        // Range check the lat/long. 
        bOk = bCheckLatLonRange(hLat, hLon);
        if (bOk == FALSE)
        {
            // ERROR!
            // Out of range lat/lon values
            return LOCATION_INVALID_OBJECT;
        }
    }

    // Get the fixed object components from the caller
    // we do this since psCreateLocation ends up storing fixed data within
    // the LOCATION structure instead of on the heap.
    n32Lat = OSAL_FIXED.n32Value(hLat);
    un8LatBits = OSAL_FIXED.un8NumFractionalBits(hLat);
    n32Lon = OSAL_FIXED.n32Value(hLon);
    un8LonBits = OSAL_FIXED.un8NumFractionalBits(hLon);

    // Clear all specific properties
    uProperties.sCircle.hRadius = DISTANCE_INVALID_OBJECT;

    // Create location
    psObj = psCreateObject (
        SMS_INVALID_OBJECT,
        LOCATION_POSITION_TYPE_STATIC,
        LOCATION_TYPE_CIRCLE,
        hLocID,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits,
        &uProperties, NULL, TRUE);

    return (LOCATION_OBJECT)psObj;
}

/*****************************************************************************
*
*   hCreateRectangle
*
* This object interface method is used by the caller to create a LOCATION
* object via rectangular area.
*
* Inputs:
*
*   hSouthernLat - The latitude for the LOCATION as a OSAL_FIXED_OBJECT
*   hWesternLon - The longitude for the LOCATION as a OSAL_FIXED_OBJECT
*   hDistanceLat - The height of rectangular area
*   hDistanceLon - The width of rectangular area
*
* Outputs:
*
*   A handle to a LOCATION object
*
*****************************************************************************/
static LOCATION_OBJECT hCreateRectangle (
    OSAL_FIXED_OBJECT  hSouthernLat,
    OSAL_FIXED_OBJECT  hWesternLon,
    DISTANCE_OBJECT hDistanceLat,
    DISTANCE_OBJECT hDistanceLon
        )
{
    LOCATION_OBJECT_STRUCT *psRectLocation;
    BOOLEAN bOk, bLatOk, bLonOk;
    N32 n32Lat, n32Lon, n32DistanceLat, n32DistanceLon;
    UN8 un8LatBits, un8LonBits;
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;
    OSAL_FIXED_OBJECT hDistanceLatFixed, hDistanceLonFixed;

    // Check provided distances object
    bLatOk = SMSO_bValid((SMS_OBJECT)hDistanceLat);
    bLonOk = SMSO_bValid((SMS_OBJECT)hDistanceLon);

    if (((hDistanceLat != DISTANCE_INVALID_OBJECT) && (bLatOk != TRUE)) ||
        ((hDistanceLon != DISTANCE_INVALID_OBJECT) && (bLonOk != TRUE)))
    {
        // ERROR!
        // At least one of provided pointers is not the valid object
        return LOCATION_INVALID_OBJECT;
    }

    bOk = bCheckLatLonRange(hSouthernLat, hWesternLon);
    if (bOk == FALSE)
    {
        // ERROR!
        // Out of range lat/lon values
        return LOCATION_INVALID_OBJECT;
    }

    // We allow only point location (both distances are zero value or
    // INVALID objects, or locations having some measurable area.
    // Lines (one valid and one invalid distance) are not allowed

    // Reading distances in  miles to check if they are greater than 0
    hDistanceLatFixed = DISTANCE.hValue(hDistanceLat, 
        DISTANCE_UNIT_TYPE_MILES);
    hDistanceLonFixed = DISTANCE.hValue(hDistanceLon, 
        DISTANCE_UNIT_TYPE_MILES);

    n32DistanceLat = OSAL_FIXED.n32Value(hDistanceLatFixed);
    n32DistanceLon = OSAL_FIXED.n32Value(hDistanceLonFixed);

    if (((n32DistanceLat == 0) && (n32DistanceLon != 0)) ||
        ((n32DistanceLat != 0) && (n32DistanceLon == 0)))
    {
        // ERROR!
        // This is an attempt to create a LINE.
        return LOCATION_INVALID_OBJECT;
    }

    n32Lat = OSAL_FIXED.n32Value(hSouthernLat);
    n32Lon = OSAL_FIXED.n32Value(hWesternLon);

    un8LatBits = OSAL_FIXED.un8NumFractionalBits(hSouthernLat);
    un8LonBits = OSAL_FIXED.un8NumFractionalBits(hWesternLon);

    // Set specific properties for rectangle
    uProperties.sRectangle.hHeight = hDistanceLat;
    uProperties.sRectangle.hWidth = hDistanceLon;

    // Create rectangle location object using bottom-left coordinates
    // and both sides
    psRectLocation = psCreateObject (
        SMS_INVALID_OBJECT,
        LOCATION_POSITION_TYPE_STATIC,
        LOCATION_TYPE_RECTANGLE,
        LOCID_INVALID_OBJECT,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits,
        &uProperties, NULL, TRUE);

    return (LOCATION_OBJECT)psRectLocation;
}

/*****************************************************************************
*
*   hDuplicate
*
* This object interface method is used to access, then do a "deep-copy" of a
* LOCATION.
*
* Inputs:
*
*   hLocation - The LOCATION_OBJECT object to duplicate
*
* Outputs:
*
*   A LOCATION that is the duplicate of hLocation
*
*
*****************************************************************************/
static LOCATION_OBJECT hDuplicate (
    LOCATION_OBJECT hLocation
        )
{
    // Verify inputs
    if(DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation) == FALSE )
    {
        // Error!
        return LOCATION_INVALID_OBJECT;
    }

    return LOCATION_hDuplicate(SMS_INVALID_OBJECT, hLocation);
}

/*****************************************************************************
*
*   hLat
*
* This object interface method is used by the caller to get the latitude
* component (as a OSAL_FIXED_OBJECT) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A OSAL_FIXED_OBJECT representing the latitude of this LOCATION on success.
*   A OSAL_FIXED_INVALID_OBJECT upon error
*
*****************************************************************************/
static OSAL_FIXED_OBJECT hLat (
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bOwner;
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;
        hLat = psObj->hLat;
    }

    return hLat;
}

/*****************************************************************************
*
*   tLon
*
* This object interface method is used by the caller to get the longitude
* component (as a OSAL_FIXED_OBJECT) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A OSAL_FIXED_OBJECT representing the latitude of this LOCATION on success.
*   A OSAL_FIXED_INVALID_OBJECT upon error
*
*****************************************************************************/
static OSAL_FIXED_OBJECT hLon (
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bOwner;
    OSAL_FIXED_OBJECT hLon = OSAL_FIXED_INVALID_OBJECT;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;
        hLon = psObj->hLon;
    }

    return hLon;
}

/*****************************************************************************
*
*   hRadius
*
* This object interface method is used by the caller to get the radius
* (as a DISTANCE) of the circular area that a specified LOCATION object covers
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A DISTANCE_OBJECT representing the radius of the circular area for this
*   LOCATION on success. DISTANCE_INVALID_OBJECT upon error
*
*****************************************************************************/
static DISTANCE_OBJECT hRadius (
    LOCATION_OBJECT hLocation
        )
{
    DISTANCE_OBJECT hRadius = DISTANCE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        // Extracting radius for circular areas only
        if (psObj->psIntf->eType == LOCATION_TYPE_CIRCLE)
        {
            hRadius = psObj->uProperties.sCircle.hRadius;
        }
    }

    return hRadius;
}


/*****************************************************************************
*
*   hHeight
*
* This object interface method is used by the caller to get the height
* (latitude difference) of the rectangular area that a specified
* LOCATION object covers
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A DISTANCE_OBJECT representing the height of the rectangular area for this
*   LOCATION on success. DISTANCE_INVALID_OBJECT upon error
*
*****************************************************************************/
static DISTANCE_OBJECT hHeight (
    LOCATION_OBJECT hLocation
        )
{
    DISTANCE_OBJECT hHeight = DISTANCE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        // Extracting radius for circular areas only
        if (psObj->psIntf->eType == LOCATION_TYPE_RECTANGLE)
        {
            hHeight = psObj->uProperties.sRectangle.hHeight;
        }
    }

    return hHeight;
}

/*****************************************************************************
*
*   hWidth
*
* This object interface method is used by the caller to get the width
* (longitude difference) of the rectangular area that a specified
* LOCATION object covers
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A DISTANCE_OBJECT representing the width of the rectangular area for this
*   LOCATION on success. DISTANCE_INVALID_OBJECT upon error
*
*****************************************************************************/
static DISTANCE_OBJECT hWidth (
    LOCATION_OBJECT hLocation
        )
{
    DISTANCE_OBJECT hWidth = DISTANCE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        // Extracting radius for circular areas only
        if (psObj->psIntf->eType == LOCATION_TYPE_RECTANGLE)
        {
            hWidth = psObj->uProperties.sRectangle.hWidth;
        }
    }

    return hWidth;
}

/*****************************************************************************
*
*   hDescription
*
* This object interface method is used by the caller to get the location's
* description (as a STRING_OBJECT handle) for the specified LOCATION object.
* This description is usually used as the field in which to find a location's
* name.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A STRING_OBJECT containing the string form of the LOCATION's street number
*   if it has it.  If there is an error, or if this LOCATION doesn't contain
*   a street number, then STRING_INVALID_OBJECT is returned.
*
*****************************************************************************/
static STRING_OBJECT hDescription (
    LOCATION_OBJECT hLocation
        )
{
    STRING_OBJECT hDescription = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hDescription = psObj->sDescriptiveAttribs.hDescription;
    }

    return hDescription;
}

/*****************************************************************************
*
*   hStreetNum
*
* This object interface method is used by the caller to get the street number
* (as a STRING_OBJECT handle) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A STRING_OBJECT containing the string form of the LOCATION's street number
*   if it has it.  If there is an error, or if this LOCATION doesn't contain
*   a street number, then STRING_INVALID_OBJECT is returned.
*
*****************************************************************************/
static STRING_OBJECT hStreetNum (
    LOCATION_OBJECT hLocation
        )
{
    STRING_OBJECT hStreetNum = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hStreetNum = psObj->sDescriptiveAttribs.hStreetNum;
    }

    return hStreetNum;
}

/*****************************************************************************
*
*   hStreetName
*
* This object interface method is used by the caller to get the street name
* (as a STRING_OBJECT handle) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A STRING_OBJECT containing the string form of the LOCATION's street name
*   if it has it.  If there is an error, or if this LOCATION doesn't contain
*   a street name, then STRING_INVALID_OBJECT is returned.
*
*****************************************************************************/
static STRING_OBJECT hStreetName (
    LOCATION_OBJECT hLocation
        )
{
    STRING_OBJECT hStreetName = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hStreetName = psObj->sDescriptiveAttribs.hStreetName;
    }

    return hStreetName;
}

/*****************************************************************************
*
*   hCity
*
* This object interface method is used by the caller to get the city name
* (as a STRING_OBJECT handle) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A STRING_OBJECT containing the string form of the LOCATION's city name
*   if it has it.  If there is an error, or if this LOCATION doesn't contain
*   a city name, then STRING_INVALID_OBJECT is returned.
*
*****************************************************************************/
static STRING_OBJECT hCity (
    LOCATION_OBJECT hLocation
        )
{
    STRING_OBJECT hCity = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hCity = psObj->sDescriptiveAttribs.hCity;
    }

    return hCity;
}

/*****************************************************************************
*
*   hState
*
* This object interface method is used by the caller to get the state
* abbreviation (as a STRING_OBJECT handle) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A STRING_OBJECT containing the string form of the LOCATION's state
*   abbreviation  if it has it.  If there is an error, or if this LOCATION
*     doesn't contain a state abbreviation, then STRING_INVALID_OBJECT is
*   returned.
*
*****************************************************************************/
static STRING_OBJECT hState (
    LOCATION_OBJECT hLocation
        )
{
    STRING_OBJECT hState = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hState = psObj->sDescriptiveAttribs.hState;
    }

    return hState;
}

/*****************************************************************************
*
*   hZipCode
*
* This object interface method is used by the caller to get the zip code
* (as a STRING_OBJECT handle) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A STRING_OBJECT containing the string form of the LOCATION's zip code
*   if it has it.  If there is an error, or if this LOCATION doesn't contain
*   a zip code, then STRING_INVALID_OBJECT is returned.
*
*****************************************************************************/
static STRING_OBJECT hZipCode (
    LOCATION_OBJECT hLocation
        )
{
    STRING_OBJECT hZipCode = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hZipCode = psObj->sDescriptiveAttribs.hZipCode;
    }

    return hZipCode;
}

/*****************************************************************************
*
*   hPhone
*
* This object interface method is used by the caller to get the phone number
* (as a STRING_OBJECT handle) for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   A STRING_OBJECT containing the string form of the LOCATION's phone number
*   if it has it.  If there is an error, or if this LOCATION doesn't contain
*   a phone number, then STRING_INVALID_OBJECT is returned.
*
*****************************************************************************/
static STRING_OBJECT hPhone (
    LOCATION_OBJECT hLocation
        )
{
    STRING_OBJECT hPhone = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hPhone = psObj->sDescriptiveAttribs.hPhone;
    }

    return hPhone;
}

/*****************************************************************************
*
*   hLocID
*
* This object interface method is used by the caller to get the LOCID handle
* for the specified LOCATION object.
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   The LOCATION's LOCID handle if it has it.  If there is an error, or if
*   this LOCATION doesn't contain a LOCID handle, then LOCID_INVALID_OBJECT
*   is returned.
*
*****************************************************************************/
static LOCID_OBJECT hLocID (
    LOCATION_OBJECT hLocation
        )
{
    LOCID_OBJECT hLocID = LOCID_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        hLocID = psObj->hLocID;
    }

    return hLocID;
}

/*****************************************************************************
*
*   hDistance
*
* This object interface method is used by the caller to deterime the distance
*  between two LOCATION objects.  This method uses
* "Great Circle Distance" when determining the distance.  Information about
* this type of calculation can be found in numerous places on the internet.
*
* The formula works like this:
* acos(
*        (sin(latAngleFrom) * sin(latAngleTo)) +
*        (cos(latAngleFrom) * cos(latAngleTo) * cos(lonAngleTo-lonAngleFrom))
*     ) * EARTH_RADIUS_IN_UNITS
*
* The final multiplication by the earth's radius is handled by the
* DISTANCE_OBJECT when the caller requests the value in a particular unit type
*
*
* Inputs:
*
*   hLocationFrom - A handle to a valid LOCATION object that represets the
*                     "start point" for the distance equation.
*   hLocationTo - A handle to a valid LOCATION object that represets the
*                     "end point" for the distance equation.
*
* Outputs:
*
*   A DISTANCE representing the distance  between hLocationFrom
*   and hLocationTo.  If there is an error, a DISTANCE_INVALID_OBJECT will
*   be returned.
*
*****************************************************************************/
static DISTANCE_OBJECT hDistance (
    LOCATION_OBJECT hLocationFrom,
    LOCATION_OBJECT hLocationTo
        )
{
    DISTANCE_OBJECT hDistance = DISTANCE_INVALID_OBJECT;
    BOOLEAN bOwnerFrom, bOwnerTo;

    // Verify ownership of these objects
    bOwnerFrom =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocationFrom);
    bOwnerTo =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocationTo);

    if (bOwnerFrom == TRUE && bOwnerTo == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObjFrom =
            (LOCATION_OBJECT_STRUCT *)hLocationFrom;

        LOCATION_OBJECT_STRUCT *psObjTo =
            (LOCATION_OBJECT_STRUCT *)hLocationTo;

        if ((psObjFrom->hLat != OSAL_FIXED_INVALID_OBJECT) &&
            (psObjFrom->hLon != OSAL_FIXED_INVALID_OBJECT) &&
            (psObjTo->hLat != OSAL_FIXED_INVALID_OBJECT) &&
            (psObjTo->hLon != OSAL_FIXED_INVALID_OBJECT))
        {
            OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
            OSAL_FIXED_OBJECT hResultVector;

            // Create the result vector fixed object
            hResultVector = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);
            if (hResultVector != OSAL_FIXED_INVALID_OBJECT)
            {
                BOOLEAN bSuccess;

                // Compute the distance vector
                bSuccess = bComputeDistanceVector(psObjFrom,
                    psObjTo->hLat, psObjTo->hLon, hResultVector);

                if (bSuccess == TRUE)
                {
                    // Create the distance object now
                    hDistance =
                        DISTANCE_hCreateFromDistanceVector(hResultVector);
                }
            }
        }
    }

    return hDistance;
}

/*****************************************************************************
*
*   hLocationAtDistance
*
******************************************************************************/
static LOCATION_OBJECT hLocationAtDistance (
    LOCATION_OBJECT hLocationFrom,
    DISTANCE_OBJECT hDistance,
    OSAL_FIXED_OBJECT hBearing
        )
{
    OSAL_FIXED_OBJECT hLatDest = OSAL_FIXED_INVALID_OBJECT,
                      hLonDest = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
    UN8 un8FixedObjectCount = 0;

    // Our return object, initialized to the invalid state for our error case
    LOCATION_OBJECT hLocationTo = LOCATION_INVALID_OBJECT;

    do
    {
        BOOLEAN bSuccess;

        // Create our fixed objects on the stack
        hLatDest =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE * un8FixedObjectCount++]);

        if (hLatDest == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        hLonDest =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE * un8FixedObjectCount++]);

        if (hLonDest == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // We'll verify inputs in the friend function
        bSuccess = LOCATION_bCalcDestCoordinatesFromLocation (
            hLocationFrom,hDistance, hBearing, hLatDest, hLonDest);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Create a location object for these coordinates
        hLocationTo = LOCATION.hCreateForRadius(hLatDest, hLonDest, DISTANCE_INVALID_OBJECT);

    } while (FALSE);

    return hLocationTo;
}

/*****************************************************************************
*
*   bCoordinatesWithinArea
*
* This object interface method is used by the caller to determine if a
* lat/lon coordinate pair is found within a LOCATION object's described area.
* If the LOCATION object does not have a radius (the DISTANCE object for that
* attribute is invalid), then the coordinate pair must exactly match the
* coordinate pair kept by the LOCATION object.
*
*   hLocation - A handle to a valid LOCATION object that represents the
*                     coordinate pair + radius for a geographical area.
*   hLat - A handle to a valid FIXED object describing a latitude
*   hLon - A handle to a valid FIXED object describing a longitude
*
* Outputs:
*
*   A BOOLEAN indicating if the hLat/hLon pairing is found within the
*   LOCATION object's defined area.  TRUE if yet, FALSE otherwise or on error.
*
*****************************************************************************/
static BOOLEAN bCoordinatesWithinArea (
    LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bOwner, bResult = FALSE;

    // Verify ownership
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        if( (psObj->hLat != OSAL_FIXED_INVALID_OBJECT) &&
            (psObj->hLon != OSAL_FIXED_INVALID_OBJECT) &&
            (hLat != OSAL_FIXED_INVALID_OBJECT) &&
            (hLon != OSAL_FIXED_INVALID_OBJECT) )
        {
            // Use interface function if one exists
            if(psObj->psIntf->bCoordinatesWithinArea != NULL)
            {
                // Determine if coordinates are within area
                bResult = psObj->psIntf->bCoordinatesWithinArea(
                    hLocation, hLat, hLon);
            }
        }
    }
    return bResult;
}

/*****************************************************************************
*
*   hClosest
*
*****************************************************************************/
static LOCATION_OBJECT hClosest (
    LOCATION_OBJECT hTarget,
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        )
{
    LOCATION_OBJECT_STRUCT *psObjTarget =
        (LOCATION_OBJECT_STRUCT *)hTarget;
    LOCATION_OBJECT_STRUCT *psObjLoc1 =
        (LOCATION_OBJECT_STRUCT *)hLocation1;
    LOCATION_OBJECT_STRUCT *psObjLoc2 =
        (LOCATION_OBJECT_STRUCT *)hLocation2;

    LOCATION_OBJECT hClosestLocation = LOCATION_INVALID_OBJECT;

    BOOLEAN bOwnerTarget, bOwnerLoc1, bOwnerLoc2;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
    OSAL_FIXED_OBJECT hVector1, hVector2;
    N16 n16Compare;

    // Verify ownership of these objects
    bOwnerTarget =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hTarget);
    bOwnerLoc1 =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation1);
    bOwnerLoc2 =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation2);

    do
    {
        BOOLEAN bOk;

        // Make sure we are owners
        if ( (bOwnerTarget == FALSE) ||
             (bOwnerLoc1 == FALSE) ||
             (bOwnerLoc2 == FALSE)
           )
        {
            break;
        }

        // Make sure we have valid lat/lon values to do distance comparisons
        if ( (psObjTarget->hLat == OSAL_FIXED_INVALID_OBJECT) ||
             (psObjTarget->hLon == OSAL_FIXED_INVALID_OBJECT) ||
             (psObjLoc1->hLat == OSAL_FIXED_INVALID_OBJECT) ||
             (psObjLoc1->hLon == OSAL_FIXED_INVALID_OBJECT) ||
             (psObjLoc2->hLat == OSAL_FIXED_INVALID_OBJECT) ||
             (psObjLoc2->hLon == OSAL_FIXED_INVALID_OBJECT)
           )
        {
            break;
        }

        // Initilize our fixed objects
        hVector1 = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);

        if (hVector1 == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        hVector2 = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE]);

        if (hVector2 == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Create distance vectors for the two locations
        bOk = bComputeDistanceVector(psObjTarget,
            psObjLoc1->hLat, psObjLoc1->hLon, hVector1);

        if (bOk == FALSE)
        {
            break;
        }

        bOk = bComputeDistanceVector(psObjTarget,
            psObjLoc2->hLat, psObjLoc2->hLon, hVector2);

        if (bOk == FALSE)
        {
            break;
        }

        // Compare the distance vectors
        n16Compare = OSAL_FIXED.n16Compare(hVector1, hVector2);

        if (n16Compare == N16_MIN)
        {
            // Error
            break;
        }

        if (n16Compare <= 0)
        {
            // Location1 was closest or they are equal
            hClosestLocation = hLocation1;
        }
        else
        {
            // Otherwise, our second location was closest
            hClosestLocation = hLocation2;
        }

    } while (FALSE);

    return hClosestLocation;
}

/*****************************************************************************
*
*   n32FWrite
*
* This object interface method is used to write the contents of a LOCATION
* (performing a deep write) to a device specified by psFile. The intent of
* this method is to effectively store the contents of a LOCATION for later
* retrieval (perhaps after a power cycle). The entire LOCATION contents are written
* to the device, with enough information to recreate the exact LOCATION
* when the hFRead() method is called.
*
* Inputs:
*
*   hLocation - The LOCATION handle the caller wishes to write.
*   psFile - The device to write the LOCATION contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FWrite (
    LOCATION_OBJECT hLocation,
    FILE *psFile
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    N32 n32Return = EOF;
    BOOLEAN bOwner;

    // Verify inputs
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if(( bOwner == TRUE) && (psFile != NULL))
    {
        // Use interface function if one exists
        if(psObj->psIntf->n32FWrite != NULL)
        {
            // Write content to file
            n32Return = psObj->psIntf->n32FWrite(hLocation, psFile);
        }
    }

    return n32Return;
}

/*****************************************************************************
*
*   hFRead
*
* This object interface method is used to read from a specified device psFile
* and generate a LOCATION from that information. The data read from the device
* must have been previously written by the n32FWrite method. Upon
* successful execution of this method a new LOCATION is created (created by the
* caller) which may be used to register for events or presented to the
* UI for display, etc. This method allows the caller to effectively read
* a previously stored LOCATION regenerating the original LOCATION written(saved) at
* an earlier time.
*
* Inputs:
*
*   psFile - The device to read the LOCATION contents from.
*
* Outputs:
*
*   A new LOCATION on success, otherwise LOCATION_INVALID_OBJECT on failure.
*
*****************************************************************************/
static LOCATION_OBJECT hFRead (
    FILE *psFile
        )
{
    LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;
    size_t tTotal, tLen;
    UN8 un8Type;

    // Read the type of location area
    tTotal = 0;

    do
    {
        tLen =
            fread(((UN8*)&un8Type) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(UN8)));

    if(tTotal != sizeof(UN8))
    {
        // End of file or error occurred
        return LOCATION_INVALID_OBJECT;
    }

    switch ((LOCATION_TYPE_ENUM)un8Type)
    {
        case LOCATION_TYPE_RECTANGLE:
        {
            hLocation = (LOCATION_OBJECT)GsLocationRectangleIntf.hFRead(psFile);
        }
        break;

        case LOCATION_TYPE_CIRCLE:
        {
            hLocation = (LOCATION_OBJECT)GsLocationCircleIntf.hFRead(psFile);
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LOCATION_OBJECT_NAME": Error: Unknown area type (%d)",
                un8Type);
        }
        break;
    }
    return hLocation;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a LOCATION's contents to a specified file or device.
* This is mainly helpful during debugging of LOCATION's but could be used by
* an application for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a LOCATION to a file for the
* purposes of later re-generating the LOCATION (for storage of the object).
* This API instead sends the LOCATION as a verbose formatted output version.
*
* Inputs:
*
*   hLocation - The LOCATION handle the caller wishes to write.
*   psFile - The device to write the LOCATION contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    LOCATION_OBJECT hLocation,
    FILE *psFile
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    N32 n32Return = 0;
    N32 n32tmp = 0;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);

    // Verify inputs. Object handle must be valid as well as the file handle.
    if((bOwner == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Print LOCATION information header
    n32Return += fprintf(psFile, "LOCATION: hLocation = 0x%p\n",
        psObj);

    // Print LOCATION type
    if (psObj->psIntf->eType == LOCATION_TYPE_RECTANGLE)
    {
        n32Return += fprintf(psFile, "\tRectangle\n");
    }
    else if (psObj->psIntf->eType == LOCATION_TYPE_CIRCLE)
    {
        n32Return += fprintf(psFile, "\tCircle\n");
    }
    else
    {
        n32Return += fprintf(psFile, "\tUnknown area type\n");
    }

    // Print lat, lon
    n32Return += fprintf(psFile, "\tLat = \n\t{\n\t");
    n32tmp = OSAL_FIXED.n32FPrintf(psObj->hLat, psFile, FALSE);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\t}\n");

    n32Return += fprintf(psFile, "\tLon = \n\t{\n\t");
    n32tmp = OSAL_FIXED.n32FPrintf(psObj->hLon, psFile, FALSE);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\t}\n");

    // Print area-specific properties
    if (psObj->psIntf->eType == LOCATION_TYPE_CIRCLE)
    {
        // Print radius
        n32Return += fprintf(psFile, "\tRadius = \n\t{\n\t");
        n32tmp = DISTANCE.n32FPrintf(psObj->uProperties.sCircle.hRadius, psFile);
        n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
        n32Return += fprintf(psFile, "\t}\n");
    }
    else if (psObj->psIntf->eType == LOCATION_TYPE_RECTANGLE)
    {
        // Print width
        n32Return += fprintf(psFile, "\tWidth = \n\t{\n\t");
        n32tmp = DISTANCE.n32FPrintf(psObj->uProperties.sRectangle.hWidth, psFile);
        n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
        n32Return += fprintf(psFile, "\t}\n");

        // Print height
        n32Return += fprintf(psFile, "\tHeight = \n\t{\n\t");
        n32tmp = DISTANCE.n32FPrintf(psObj->uProperties.sRectangle.hHeight, psFile);
        n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
        n32Return += fprintf(psFile, "\t}\n");
    }

    // Print Description
    n32Return += fprintf(psFile, "\tDescription=\n");
    n32tmp = STRING.n32FPrintf(psObj->sDescriptiveAttribs.hDescription, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");

    // Print Street Num
    n32Return += fprintf(psFile, "\tStreetNum =\n");
    n32tmp = STRING.n32FPrintf(psObj->sDescriptiveAttribs.hStreetNum, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");

    // Print Street Name
    n32Return += fprintf(psFile, "\tStreetName =\n");
    n32tmp = STRING.n32FPrintf(psObj->sDescriptiveAttribs.hStreetName, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");

    // Print City
    n32Return += fprintf(psFile, "\tCity =\n");
    n32tmp = STRING.n32FPrintf(psObj->sDescriptiveAttribs.hCity, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");

    // Print State
    n32Return += fprintf(psFile, "\tState =\n");
    n32tmp = STRING.n32FPrintf(psObj->sDescriptiveAttribs.hState, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");

    // Print Zip Code
    n32Return += fprintf(psFile, "\tZip Code =\n");
    n32tmp = STRING.n32FPrintf(psObj->sDescriptiveAttribs.hZipCode, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");

    // Print Phone number
    n32Return += fprintf(psFile, "\tPhone Number =\n");
    n32tmp = STRING.n32FPrintf(psObj->sDescriptiveAttribs.hPhone, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");

    // Print LOCID
    n32Return += fprintf(psFile, "\tLOCID = \n\t{\n\t");
    n32tmp = LOCID.n32FPrintf(psObj->hLocID, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\t}\n");

    return n32Return;
}

/*****************************************************************************
*
*   vDestroy
*
* This object interface method is used by the caller to destroy the specified
* LOCATION object and all members of the object
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object that the caller wants
*                 to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
static void vDestroy (
    LOCATION_OBJECT hLocation
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    if (psObj->hLocID != LOCID_INVALID_OBJECT)
    {
        LOCID.vDestroy(psObj->hLocID);
        psObj->hLocID = LOCID_INVALID_OBJECT;
    }

    // Destroying area-specific object properties
    if (psObj->psIntf != NULL)
    {
        psObj->psIntf->vDestroyProperties(hLocation);
        psObj->psIntf = NULL;
    }

    STRING_vDestroy(psObj->sDescriptiveAttribs.hDescription);
    STRING_vDestroy(psObj->sDescriptiveAttribs.hStreetNum);
    STRING_vDestroy(psObj->sDescriptiveAttribs.hStreetName);
    STRING_vDestroy(psObj->sDescriptiveAttribs.hCity);
    STRING_vDestroy(psObj->sDescriptiveAttribs.hState);
    STRING_vDestroy(psObj->sDescriptiveAttribs.hZipCode);
    STRING_vDestroy(psObj->sDescriptiveAttribs.hPhone);
    OSAL.bMemSet(&psObj->sDescriptiveAttribs, 0, sizeof(LOCATION_ATTRIBUTE_STRUCT));

    OSAL_FIXED.vDestroy(psObj->sPrecalcFixedValues.hCosLat);
    OSAL_FIXED.vDestroy(psObj->sPrecalcFixedValues.hSinLat);
    OSAL_FIXED.vDestroy(psObj->sPrecalcFixedValues.hLatInRadian);
    OSAL_FIXED.vDestroy(psObj->sPrecalcFixedValues.hLonInRadian);

    // Free object instance
    DSRL_TARGET_vDestroy((DSRL_TARGET_OBJECT)hLocation);

    return;
}

/*****************************************************************************
*
*   bIsPoint
*
* This object interface method is used to check whenever provided LOCATION
* object is an area or just the point
*
*
* Inputs:
*
*   hLocation - A handle to a valid LOCATION object
*
* Outputs:
*
*   FALSE if LOCATION is non-zero size area.
*   TRUE is LOCATION is just a single point.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsPoint (
    LOCATION_OBJECT hLocation,
    BOOLEAN *pbIsPoint
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // Verify inputs. Object handle must be valid.
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if((bOwner == TRUE) && (pbIsPoint != NULL))
    {
        *pbIsPoint = psObj->bIsPoint;
        eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturn;
}

/*****************************************************************************
*
*   eType
*
*   This function is used to return an area type (Rectangular vs Circular)
*   of provided LOCATION object
*
*****************************************************************************/
static LOCATION_TYPE_ENUM eType(
    LOCATION_OBJECT hLocation
    )
{
    BOOLEAN bOwner;
    LOCATION_TYPE_ENUM eType = LOCATION_TYPE_UNKNOWN;

    // Verify ownership of this object
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        if (psObj->psIntf != NULL)
        {
            eType = psObj->psIntf->eType;
        }
    }

    return eType;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   LOCATION_hCreate
*
*****************************************************************************/
LOCATION_OBJECT LOCATION_hCreate (
    SMS_OBJECT hOwner,
    LOCID_OBJECT hLocID,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    DISTANCE_OBJECT hRadius,
    LOCATION_ATTRIBUTE_STRUCT *psDescriptiveAttrs,
    BOOLEAN bPerformDeepCopy
        )
{
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;
    LOCATION_OBJECT_STRUCT *psLocation;

    uProperties.sCircle.hRadius = hRadius;

    psLocation = psCreateLocation( hOwner, hLocID,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits,
        &uProperties, psDescriptiveAttrs, bPerformDeepCopy,
        LOCATION_TYPE_CIRCLE );

    return (LOCATION_OBJECT)psLocation;
}

/*****************************************************************************
*
*   LOCATION_hCreateForDeviceArea
*
*****************************************************************************/
LOCATION_OBJECT LOCATION_hCreateForDeviceArea (
    DISTANCE_OBJECT hRadius
        )
{
    LOCATION_OBJECT_STRUCT *psObj;
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;

    uProperties.sCircle.hRadius = hRadius;

    // Create location
    psObj = psCreateObject(
        SMS_INVALID_OBJECT,
        LOCATION_POSITION_TYPE_DEVICE,
        LOCATION_TYPE_CIRCLE,
        LOCID_INVALID_OBJECT,
        0, 0, 0, 0,
        &uProperties, NULL, TRUE);

    return (LOCATION_OBJECT)psObj;
}

/*****************************************************************************
*
*   LOCATION_hDuplicate
*
* This object interface method is a friend function that is used to do a
* "deep-copy" of a LOCATION object.
*
* Inputs:
*
*   hOwner - The SMS_OBJECT that will own the duplicated LOCATION object.
*   hLocation - The LOCATION object do a deep-copy of
*
* Outputs:
*
*   A valid LOCATION_OBJECT handle on success
*   LOCATION_INVALID_OBJECT on failure
*
*****************************************************************************/
LOCATION_OBJECT LOCATION_hDuplicate (
    SMS_OBJECT     hOwner,
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN             bOwner;
    N32                 n32Lat = 0;
    N32                 n32Lon = 0;
    UN8                 un8LatBits = 0;
    UN8                 un8LonBits = 0;
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    LOCATION_OBJECT_STRUCT *psCopy;

    // Verify inputs
    bOwner =  SMSO_bOwner(hOwner);
    if ((hOwner != SMS_INVALID_OBJECT) && (bOwner == FALSE ))
    {
        // Error!
        return LOCATION_INVALID_OBJECT;
    }

    bOwner =  DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if(bOwner == FALSE )
    {
        // Error!
        return LOCATION_INVALID_OBJECT;
    }

    // Extract the fixed values for base point
    switch (psObj->psIntf->eType)
    {
        case LOCATION_TYPE_CIRCLE:
        {
            // For circle the base point is the center
            n32Lat = OSAL_FIXED.n32Value(psObj->hLat);
            un8LatBits = OSAL_FIXED.un8NumFractionalBits(psObj->hLat);
            n32Lon = OSAL_FIXED.n32Value(psObj->hLon);
            un8LonBits = OSAL_FIXED.un8NumFractionalBits(psObj->hLon);
        }
        break;
        case LOCATION_TYPE_RECTANGLE:
        {
            // For rectangle the base point is bottom-left
            // (which is always preset)

            n32Lat = OSAL_FIXED.n32Value(psObj->hBotLeftLat);
            un8LatBits = OSAL_FIXED.un8NumFractionalBits(psObj->hBotLeftLat);
            n32Lon = OSAL_FIXED.n32Value(psObj->hBotLeftLon);
            un8LonBits = OSAL_FIXED.un8NumFractionalBits(psObj->hBotLeftLon);
        }
        break;
        case LOCATION_TYPE_UNKNOWN:
        {
            // Nothing to do
        }
        break;
    }

    // Create a new location object and deep-copy it
    psCopy = psCreateObject (
        hOwner,
        psObj->ePositionType,
        psObj->psIntf->eType,
        psObj->hLocID,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits,
        &psObj->uProperties,
        &psObj->sDescriptiveAttribs,
        TRUE);

    return (LOCATION_OBJECT)psCopy;
}

/*****************************************************************************
*
*   LOCATION_bUpdateCoordinates
*
* This object interface method is a friend function that is used to update
* the geolocation attributes of an existing LOCATION_OBJECT.
*
* Inputs:
*
*   hLocation - The LOCATION object to update
*   n32Lat - The latitude for the LOCATION as a N32 in fixed format
*   un8LatBits - The number of fractional bits used n32Lat
*   n32Lon - The longitude for the LOCATION as a N32 in fixed format
*   un8LonBits - The number of fractional bits used n32Lon
*   bPerformDeepCopy - A flag indicating if the all attributes provided
*       may be used directly or must be copied.
*
* Outputs:
*
*   TRUE on success
*   FALSE on failure
*
*****************************************************************************/
BOOLEAN LOCATION_bUpdateCoordinates (
    LOCATION_OBJECT  hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bOwner, bResult = FALSE;

    // Verify we own the location object
    bOwner =  DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)psObj);
    if(bOwner == TRUE)
    {
        // Use interface function if one exists
        if(psObj->psIntf->bUpdateCoordinates != NULL)
        {
            // update radius
            bResult = psObj->psIntf->bUpdateCoordinates(
                hLocation,
                n32Lat, un8LatBits,
                n32Lon, un8LonBits);
        }
    }
    return bResult;
}

/*****************************************************************************
*
*   LOCATION_bUpdateCoordinatesByFixed
*
* This object interface method is a friend function that is used to update
* the geolocation attributes of an existing LOCATION_OBJECT.
*
* Inputs:
*
*   hLocation - The LOCATION object to update
*   hLat - The latitude for the LOCATION
*   nLon - The longitude for the LOCATION
*
* Outputs:
*
*   TRUE on success
*   FALSE on failure
*
*****************************************************************************/
BOOLEAN LOCATION_bUpdateCoordinatesByFixed (
    LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bReturn = FALSE;

    do 
    {
        BOOLEAN bOwner;

        // Check input parameters
        if ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Make sure we own the location object
        bOwner =  DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
        if (bOwner == FALSE)
        {
            break;
        }

        // Cleanup location parameters
        vUpdateCoordinatesCommonPart(hLocation, 0, 0, 0, 0);

        // Populate them now
        bReturn = OSAL_FIXED.bCopyToMemory(hLat, psObj->hLat);
        if (FALSE == bReturn)
        {
            break;
        }

        bReturn = OSAL_FIXED.bCopyToMemory(hLon, psObj->hLon);

    } while (FALSE);

    return bReturn;
}

/*****************************************************************************
*
*   LOCATION_bUpdateRadius
*
* This object interface method is a friend function that is used to update
* the radius attribute of an existing LOCATION_OBJECT.
*
* Inputs:
*
*   hLocation - The LOCATION object to update
*   hRadius - The distance that describes the area around (lat,lon)
*       that this LOCATION represents.
*   bPerformDeepCopy - A flag indicating if the all attributes provided
*       may be used directly or must be copied.
*
* Outputs:
*
*   TRUE on success
*   FALSE on failure
*
*****************************************************************************/
BOOLEAN LOCATION_bUpdateRadius (
    LOCATION_OBJECT hLocation,
    DISTANCE_OBJECT hRadius,
    BOOLEAN bPerformDeepCopy
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;
        BOOLEAN bOwner;

        // Verify we own the location object
        bOwner =  DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
        if (bOwner == FALSE)
        {
            break;
        }

        // Applicable only for circular locations
        if(psObj->psIntf->eType != LOCATION_TYPE_CIRCLE)
        {
            break;
        }

        // Update radius
        // Clear the left/right square attributes since we have
        // a new radius now
        psObj->hBotLeftLat = OSAL_FIXED_INVALID_OBJECT;
        psObj->hBotLeftLon = OSAL_FIXED_INVALID_OBJECT;
        psObj->hTopRightLat = OSAL_FIXED_INVALID_OBJECT;
        psObj->hTopRightLon = OSAL_FIXED_INVALID_OBJECT;

        // Update the radius DISTANCE object
        DISTANCE.vDestroy(psObj->uProperties.sCircle.hRadius);

        // Resetting the flag. We will set it after calculation
        psObj->bIsPoint = FALSE;

        if (bPerformDeepCopy == TRUE)
        {
            psObj->uProperties.sCircle.hRadius = DISTANCE.hDuplicate(hRadius);
        }
        else
        {
            psObj->uProperties.sCircle.hRadius = hRadius;
        }

        if (psObj->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT)
        {
            // Target radius happens to be a NULL object.
            // If we were given a valid radius then it is an error
            if (hRadius != DISTANCE_INVALID_OBJECT)
            {
                // Failed to copy
                break;
            }
            else
            {
                // Or else we replace it with valid object of 0
                psObj->uProperties.sCircle.hRadius =
                    DISTANCE.hCreate(0, DISTANCE_UNIT_TYPE_MILES);
                if (psObj->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT)
                {
                    // Failed to create
                    break;
                }
                psObj->bIsPoint = TRUE;
            }
        }

        // We are done
        bSuccess = TRUE;

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   LOCATION_bCalcDestCoordinatesFromLocation
*
*****************************************************************************/
BOOLEAN LOCATION_bCalcDestCoordinatesFromLocation (
    LOCATION_OBJECT   hLocationFrom,
    DISTANCE_OBJECT   hDistance,
    OSAL_FIXED_OBJECT hBearing,
    OSAL_FIXED_OBJECT hLatTo,
    OSAL_FIXED_OBJECT hLonTo
        )
{
    LOCATION_OBJECT_STRUCT *psObjFrom =
        (LOCATION_OBJECT_STRUCT *)hLocationFrom;
    BOOLEAN bOwnerFrom, bOwnerDistance;

    // Determine Ownership
    bOwnerFrom =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocationFrom);
    bOwnerDistance =
        SMSO_bOwner((SMS_OBJECT)hDistance);

    do
    {
        // Verify inputs
        if ( (bOwnerFrom == FALSE) ||
             (bOwnerDistance == FALSE)  ||
             (hBearing == OSAL_FIXED_INVALID_OBJECT) ||
             (hLatTo == OSAL_FIXED_INVALID_OBJECT) ||
             (hLonTo == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Make sure this location has a lat/lon
        if ((psObjFrom->hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psObjFrom->hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Return the status from our private function.
        return bCalcDestCoordinatesFromLocation(
            psObjFrom, hDistance, hBearing, hLatTo, hLonTo);

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   LOCATION_bContains
*
*****************************************************************************/
BOOLEAN LOCATION_bContains (
    LOCATION_OBJECT hContainer,
    LOCATION_OBJECT hContained,
    DISTANCE_OBJECT hThreshold
        )
{
    BOOLEAN bContains = FALSE;
    LOCATION_OBJECT_STRUCT *psObjContainer =
            (LOCATION_OBJECT_STRUCT *)hContainer;
    BOOLEAN bOwner;

    // Check object ownership
    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hContainer);
    if (bOwner == TRUE)
    {
        bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hContained);
    }

    if (bOwner == TRUE)
    {
        // Use interface function if one exists
        if(psObjContainer->psIntf->bContains != NULL)
        {
            // check if one location contains another
            bContains = psObjContainer->psIntf->bContains(
                hContainer,
                hContained,
                hThreshold);
        }
    }
    return bContains;
}

/*****************************************************************************
*
*   LOCATION_bTopRight
*
*****************************************************************************/
BOOLEAN LOCATION_bTopRight (
    LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hTopRightLat,
    OSAL_FIXED_OBJECT hTopRightLon
        )
{
    BOOLEAN bReturn = FALSE;

    do
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;
        BOOLEAN bOwner;

        // Check input parameters
        bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
        if (bOwner == FALSE)
        {
            break;
        }

        // Check if the LOCATION itself is valid
        if ((psObj->hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psObj->hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        if((psObj->hTopRightLat != OSAL_FIXED_INVALID_OBJECT) &&
           (psObj->hTopRightLon != OSAL_FIXED_INVALID_OBJECT))
        {
            if (hTopRightLon != OSAL_FIXED_INVALID_OBJECT)
            {
                bReturn = OSAL_FIXED.bCopyToMemory(psObj->hTopRightLon,
                    hTopRightLon);
                if (bReturn == FALSE)
                {
                    break;
                }
            }

            if (hTopRightLat != OSAL_FIXED_INVALID_OBJECT)
            {
                bReturn = OSAL_FIXED.bCopyToMemory(psObj->hTopRightLat,
                    hTopRightLat);
            }
            else
            {
                bReturn = TRUE;
            }
        }
        else
        {
            switch (psObj->psIntf->eType)
            {
                case LOCATION_TYPE_CIRCLE:
                {
                    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
                    OSAL_FIXED_OBJECT hTopRightBearing;

                    psObj->hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atTopRightLatData[0]);
                    psObj->hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atTopRightLonData[0]);

                    // If radius is zero, bottom left is equal to center
                    if (psObj->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hLat,
                            &psObj->atTopRightLatData[0]);
                        if(bReturn == FALSE)
                        {
                            break;
                        }
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hLon,
                            &psObj->atTopRightLonData[0]);
                    }
                    else
                    {
                        // Create our bearing in degrees
                        hTopRightBearing =
                            OSAL_FIXED.hCreateInMemory(LOCATION_TOP_RIGHT_BEARING, 0,
                            &atFixedData[0]);

                        if (hTopRightBearing == OSAL_FIXED_INVALID_OBJECT)
                        {
                            break;
                        }

                        // Return the status from our private function.
                        bReturn = bCalcDestCoordinatesFromLocation(
                            psObj, psObj->uProperties.sCircle.hRadius, hTopRightBearing,
                            psObj->hTopRightLat, psObj->hTopRightLon);
                    }

                    if( bReturn == FALSE )
                    {
                        break;
                    }

                    if (hTopRightLon != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hTopRightLon,
                            hTopRightLon);
                        if (bReturn == FALSE)
                        {
                            break;
                        }
                    }

                    if (hTopRightLat != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hTopRightLat,
                            hTopRightLat);
                    }
                    else
                    {
                        bReturn = TRUE;
                    }
                }
                break;

                case LOCATION_TYPE_RECTANGLE:
                {
                    // Create objects for bottom-left coordinates
                    psObj->hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atTopRightLatData[0]);
                    psObj->hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atTopRightLonData[0]);

                    // Calculate coordinates of the point
                    bReturn = bCalculateHalfwayShift(psObj->hLat, psObj->hLon,
                        psObj->uProperties.sRectangle.hWidth,
                        psObj->uProperties.sRectangle.hHeight,
                        LOCATION_TOP_BEARING, LOCATION_RIGHT_BEARING,
                        psObj->hTopRightLat, psObj->hTopRightLon);
                    if (bReturn == FALSE)
                    {
                        break;
                    }

                    if (hTopRightLon != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hTopRightLon,
                            hTopRightLon);
                        if (bReturn == FALSE)
                        {
                            break;
                        }
                    }

                    if (hTopRightLat != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hTopRightLat,
                            hTopRightLat);
                    }
                    else
                    {
                        bReturn = TRUE;
                    }
                }
                break;

                default:
                {
                    psObj->hTopRightLat = OSAL_FIXED_INVALID_OBJECT;
                    psObj->hTopRightLon = OSAL_FIXED_INVALID_OBJECT;
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        LOCATION_OBJECT_NAME": Error: Invalid area type");
                }
                break;
            }
        }
    } while (FALSE);

    return bReturn;
}

/*****************************************************************************
*
*   LOCATION_bBottomLeft
*
*****************************************************************************/
BOOLEAN LOCATION_bBottomLeft (
    LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hBottomLeftLat,
    OSAL_FIXED_OBJECT hBottomLeftLon
        )
{
    BOOLEAN bReturn = FALSE;

    do
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;
        BOOLEAN bOwner;

        // Check input parameters
        bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
        if (bOwner == FALSE)
        {
            break;
        }

        // Check if the LOCATION itself is valid
        if ((psObj->hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psObj->hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        if(psObj->hBotLeftLat != OSAL_FIXED_INVALID_OBJECT &&
           psObj->hBotLeftLon != OSAL_FIXED_INVALID_OBJECT)
        {
            if (hBottomLeftLon != OSAL_FIXED_INVALID_OBJECT)
            {
                bReturn = OSAL_FIXED.bCopyToMemory(psObj->hBotLeftLon,
                    hBottomLeftLon);
                if (bReturn == FALSE)
                {
                    break;
                }
            }

            if(hBottomLeftLat != OSAL_FIXED_INVALID_OBJECT)
            {
                bReturn = OSAL_FIXED.bCopyToMemory(psObj->hBotLeftLat,
                    hBottomLeftLat);
            }
            else
            {
                bReturn = TRUE;
            }
        }
        else
        {
            switch (psObj->psIntf->eType)
            {
                case LOCATION_TYPE_CIRCLE:
                {
                    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
                    OSAL_FIXED_OBJECT hBottomLeftBearing;

                    psObj->hBotLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atBotLeftLatData[0]);
                    psObj->hBotLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atBotLeftLonData[0]);

                    // If radius is zero, bottom left is equal to center
                    if (psObj->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hLat,
                            &psObj->atBotLeftLatData[0]);
                        if(bReturn == FALSE)
                        {
                            break;
                        }
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hLon,
                            &psObj->atBotLeftLonData[0]);
                    }
                    else
                    {
                        // Create our bearing in degrees
                        hBottomLeftBearing =
                            OSAL_FIXED.hCreateInMemory(LOCATION_BOTTOM_LEFT_BEARING, 0,
                            &atFixedData[0]);

                        if (hBottomLeftBearing == OSAL_FIXED_INVALID_OBJECT)
                        {
                            break;
                        }

                        // Return the status from our private function.
                        bReturn = bCalcDestCoordinatesFromLocation(
                            psObj, psObj->uProperties.sCircle.hRadius, hBottomLeftBearing,
                            psObj->hBotLeftLat, psObj->hBotLeftLon);
                    }

                    if( bReturn == FALSE )
                    {
                        break;
                    }

                    if(hBottomLeftLat != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hBotLeftLat,
                            hBottomLeftLat);
                        if (bReturn == FALSE)
                        {
                            break;
                        }
                    }

                    if (hBottomLeftLon != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hBotLeftLon,
                            hBottomLeftLon);
                    }
                    else
                    {
                        bReturn = TRUE;
                    }
                }
                break;

                case LOCATION_TYPE_RECTANGLE:
                {
                    // Create objects for bottom-left coordinates
                    psObj->hBotLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atBotLeftLatData[0]);
                    psObj->hBotLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
                        &psObj->atBotLeftLonData[0]);

                    // Calculate coordinates of the point
                    bReturn = bCalculateHalfwayShift(psObj->hLat, psObj->hLon,
                        psObj->uProperties.sRectangle.hWidth,
                        psObj->uProperties.sRectangle.hHeight,
                        LOCATION_BOTTOM_BEARING, LOCATION_LEFT_BEARING,
                        psObj->hBotLeftLat, psObj->hBotLeftLon);
                    if (bReturn == FALSE)
                    {
                        break;
                    }

                    if(hBottomLeftLat != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hBotLeftLat,
                            hBottomLeftLat);
                        if (bReturn == FALSE)
                        {
                            break;
                        }
                    }

                    if(hBottomLeftLon != OSAL_FIXED_INVALID_OBJECT)
                    {
                        bReturn = OSAL_FIXED.bCopyToMemory(psObj->hBotLeftLon,
                            hBottomLeftLon);
                    }
                    else
                    {
                        bReturn = TRUE;
                    }
                }
                break;

                default:
                {
                    psObj->hBotLeftLat = OSAL_FIXED_INVALID_OBJECT;
                    psObj->hBotLeftLon = OSAL_FIXED_INVALID_OBJECT;
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        LOCATION_OBJECT_NAME": Error: Invalid area type");
                }
                break;
            }
        }
    } while (FALSE);

    return bReturn;
}

/*****************************************************************************
*
*   LOCATION_bCoordinatesWithinRectangle
*
* This object interface method is a friend function that is used to check
* if a point coordinates are located within rectangle defined by top right
* and bottom left corners
*
* Inputs:
*
*   hLocation - The LOCATION object handle of the target rectangle
*   hLat - the latitude of the point to check
*   hLon - the longitude of the point to check
*
* Outputs:
*
*   TRUE if point located within the rectangle
*   FALSE if failure happened or point outside the rectangle
*
*****************************************************************************/
BOOLEAN LOCATION_bCoordinatesWithinRectangle (
    LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    BOOLEAN bResult = FALSE;

    do 
    {
        N16 n16Compare;
        BOOLEAN bSuccess;
        LOCATION_OBJECT_STRUCT *psObj = 
            (LOCATION_OBJECT_STRUCT *)hLocation;

        if ((hLat == OSAL_FIXED_INVALID_OBJECT) || 
            (hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // calculate bottom left lat and lon
        bSuccess = LOCATION_bBottomLeft(hLocation, 
            OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);
        if (bSuccess == FALSE)
        {
            break;
        }

        // checking bottom left latitude
        n16Compare = OSAL_FIXED.n16Compare(hLat, psObj->hBotLeftLat);
        if (n16Compare < 0)
        {
            break;
        }

        // checking bottom left longitude
        n16Compare = OSAL_FIXED.n16Compare(hLon, psObj->hBotLeftLon);
        if (n16Compare < 0)
        {
            break;
        }

        // calculate the top right lat, lon
        bSuccess = LOCATION_bTopRight(hLocation,
            OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);
        if (bSuccess == FALSE)
        {
            break;
        }

        // checking top right latitude
        n16Compare = OSAL_FIXED.n16Compare(hLat, psObj->hTopRightLat);
        if (n16Compare > 0)
        {
            break;
        }

        // checking top right longitude
        n16Compare = OSAL_FIXED.n16Compare(hLon, psObj->hTopRightLon);
        if (n16Compare > 0)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   LOCATION_bUpdateLocID
*
* This object interface method is a friend function that is used to update
* the LOC_ID of a pre-existing LOCATION_OBJECT.
*
* Inputs:
*
*   hLocation - The LOCATION object to update
*   hLocID - The new LOC_ID to assign to this object
*   bPerformDeepCopy - A flag indicating if the all attributes provided
*       may be used directly or must be copied.
*
* Outputs:
*
*   TRUE on success
*   FALSE on failure
*
*****************************************************************************/
BOOLEAN LOCATION_bUpdateLocID (
    LOCATION_OBJECT hLocation,
    LOCID_OBJECT hLocID,
    BOOLEAN bPerformDeepCopy
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bOwner;

    bOwner =  DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);

    // Verify the attributes pointer is valid
    // and we own the location object
    if (bOwner == FALSE)
    {
        return FALSE;
    }

    // Destroy our old LocID
    LOCID.vDestroy(psObj->hLocID);
    if (bPerformDeepCopy == TRUE)
    {
        psObj->hLocID = LOCID.hDuplicate(hLocID);
    }
    else
    {
        psObj->hLocID = hLocID;
    }

    return TRUE;
}

/*****************************************************************************
*
*   LOCATION_bUpdateDescriptiveAttributes
*
* This object interface method is a friend function that is used to update
* the STRING based descriptive attributes of a location object.
*
* Inputs:
*
*   hLocation - The LOCATION object to update
*   *psAttributes - The new attributes for the LOCATION to utilize
*   bOverwriteAllFields - A flag indicating if all fields the LOCATION
*       object currently maintains are to be updated by the contents
*       of psAttributes or if only those fields which are valid within
*       psAttributes should be updated internally.
*   bPerformDeepCopy - A flag indicating if the all attributes provided
*       may be used directly or must be copied.
*
* Outputs:
*
*   TRUE on success
*   FALSE on failure
*
*****************************************************************************/
BOOLEAN LOCATION_bUpdateDescriptiveAttributes (
    LOCATION_OBJECT hLocation,
    LOCATION_ATTRIBUTE_STRUCT *psAttributes,
    BOOLEAN bOverwriteAllFields,
    BOOLEAN bPerformDeepCopy
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bOwner;

    bOwner =  DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);

    // Verify the attributes pointer is valid
    // and we own the location object
    if ((bOwner == FALSE) || (psAttributes == NULL))
    {
        return FALSE;
    }

    // Update the attributes

    // Description
    vUpdateDescriptiveAttribute(&psObj->sDescriptiveAttribs.hDescription,
        psAttributes->hDescription, bOverwriteAllFields, bPerformDeepCopy);

    // Street num
    vUpdateDescriptiveAttribute(&psObj->sDescriptiveAttribs.hStreetNum,
        psAttributes->hStreetNum, bOverwriteAllFields, bPerformDeepCopy);

    // Street name
    vUpdateDescriptiveAttribute(&psObj->sDescriptiveAttribs.hStreetName,
        psAttributes->hStreetName, bOverwriteAllFields, bPerformDeepCopy);

    // City
    vUpdateDescriptiveAttribute(&psObj->sDescriptiveAttribs.hCity,
        psAttributes->hCity, bOverwriteAllFields, bPerformDeepCopy);

    // Phone
    vUpdateDescriptiveAttribute(&psObj->sDescriptiveAttribs.hPhone,
        psAttributes->hPhone, bOverwriteAllFields, bPerformDeepCopy);

    // State
    vUpdateDescriptiveAttribute(&psObj->sDescriptiveAttribs.hState,
        psAttributes->hState, bOverwriteAllFields, bPerformDeepCopy);

    // Zip Code
    vUpdateDescriptiveAttribute(&psObj->sDescriptiveAttribs.hZipCode,
        psAttributes->hZipCode, bOverwriteAllFields, bPerformDeepCopy);

    return TRUE;
}

/*****************************************************************************
*
*   LOCATION_bCompare
*
* Compares two locations based upon their uniquely identifying attributes.
* This includes the lat/lon pair, the radius (if it exists), and the
* locid (if it exists).
*
*****************************************************************************/
BOOLEAN LOCATION_bCompare (
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        )
{
    BOOLEAN bOwner;

    bOwner = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation1);
    bOwner &= DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation2);

    // Verify we own the location objects
    if (bOwner == FALSE)
    {
        return FALSE;
    }

    // Compare objects
    return bCompare(hLocation1, hLocation2);
}

/*****************************************************************************
*
*   LOCATION_hStateAbbrvForID
*   Returns STRING object containing State Abbreviation for given State ID
*
*****************************************************************************/
STRING_OBJECT LOCATION_hStateAbbrvForID (
    STATE_ID tStateID
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;

    if ( (tStateID >= STATE_ID_MIN) &&
         (tStateID <= STATE_ID_MAX) )
    {
        // Create a constant string using our table
        hString = STRING_hCreateConst(
            gasStateName[tStateID].pcAbbv,
            strlen(gasStateName[tStateID].pcAbbv));
    }

    return hString;
}

/*****************************************************************************
*
*   LOCATION_tStateIdForStateName
*   Returns State id for given State Name Abbreviation
*
*****************************************************************************/
STATE_ID LOCATION_tStateIDForAbbrv (
    STRING_OBJECT hStateAbbrevName
        )
{
    STATE_ID tFoundStateId = 0;

    if (STRING_INVALID_OBJECT != hStateAbbrevName)
    {
        UN16 un16Index;
        const char *pcStateAbbrevName = NULL;
        char *pcStateAbbrevNameUpperCase = NULL;
        size_t tStrLen, i;

        tFoundStateId = STATE_ID_MAX + 1;
        pcStateAbbrevName = STRING.pacCStr(hStateAbbrevName);
        tStrLen = strlen(pcStateAbbrevName);

        pcStateAbbrevNameUpperCase = (char*)OSAL.pvMemoryAllocate(
                    LOCATION_OBJECT_NAME":TempString",
                       tStrLen + 1,
                       FALSE);

        for(i = 0; i < tStrLen; i++)
        {
            pcStateAbbrevNameUpperCase[i] = (char)toupper(pcStateAbbrevName[i]);
        }

        pcStateAbbrevNameUpperCase[tStrLen] = '\0';

        for( un16Index = STATE_ID_MIN; un16Index < STATE_ID_MAX; un16Index++ )
        {
            int iResult;

            iResult = strcmp(gasStateName[un16Index].pcAbbv, pcStateAbbrevNameUpperCase);
            if (iResult == 0) //Equal
            {
                tFoundStateId = (STATE_ID)un16Index;
                break;
            }
        }

        OSAL.vMemoryFree(pcStateAbbrevNameUpperCase);
    }

    return tFoundStateId;
}

/*****************************************************************************
*
*   LOCATION_hGetTargetTag
*
*   Gets a TAG_OBJECT for the location represented by the DSRL_TARGET_OBJECT
*   provided.  If the tag doesn't exist, this function creates that tag
*   and sets the value.
*
*****************************************************************************/
TAG_OBJECT LOCATION_hGetTargetTag (
    TAG_OBJECT hParentTag,
    DSRL_TARGET_OBJECT hTarget
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    LOCATION_OBJECT hLocation;
    LOCID_OBJECT hLocId;
    DSRL_TARGET_TYPE_ENUM eType;
    LOCATION_TAG_SEARCH_STRUCT sSearch;

    // Initialize the search structure
    sSearch.tID = LOC_INVALID_ID;
    sSearch.hTag = TAG_INVALID_OBJECT;

    // Verify target type
    eType = DSRL_TARGET.eType(hTarget);
    if (eType != DSRL_TARGET_TYPE_LOCATION)
    {
        // We can't use this type of target
        return TAG_INVALID_OBJECT;
    }

    // Extract the target info all the
    // way down to the LOC_ID
    hLocation = DSRL_TARGET.hLocation(hTarget);
    hLocId = LOCATION.hLocID(hLocation);
    sSearch.tID = LOCID.tID(hLocId);
    if (sSearch.tID == LOC_INVALID_ID)
    {
        // This isn't a target which can have a tag
        return TAG_INVALID_OBJECT;
    }

    // Iterate the child tags to locate the
    // tag for this location
    eReturnCode = TAG_eIterateChildren(
        hParentTag,
        (TAG_ITERATION_HANDLER)bFindLocationTag,
        &sSearch);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error!
        return TAG_INVALID_OBJECT;
    }

    // Did our search find the appropriate tag?
    if (sSearch.hTag == TAG_INVALID_OBJECT)
    {
        // No -- Attempt to add the tag now
        eReturnCode = TAG_eAdd(
            LOCATION_OBJECT_NAME, hParentTag, &sSearch.hTag, NULL);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Set this tag's value and commit it to the config file
            TAG_eSetTagValue( sSearch.hTag, TAG_TYPE_INTEGER,
                &sSearch.tID, sizeof(LOC_ID), TRUE);
        }
    }

    return sSearch.hTag;
}

/*****************************************************************************
*
*   LOCATION_hCreateTargetFromTag
*
*   Reads a tag and generates a DSRL_TARGET_OBJECT based on the
*   tags contents
*
*****************************************************************************/
DSRL_TARGET_OBJECT LOCATION_hCreateTargetFromTag (
    TAG_OBJECT hTag
        )
{
    DSRL_TARGET_OBJECT hTarget = DSRL_TARGET_INVALID_OBJECT;
    size_t tSize;
    STRING_OBJECT hTagName;

    // get this tag's name
    hTagName = TAG_hTagName(hTag);

    // is this a tag we're interested in?
    if(STRING.n16CompareCStr(LOCATION_OBJECT_NAME, 0, hTagName) == 0)
    {
        // yes, we're interested in this tag
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        LOC_ID tLocId = 0, *ptLocId;

        // get the tag value
        ptLocId = &tLocId;
        tSize = sizeof(LOC_ID);
        eReturnCode = TAG_eGetTagValue(hTag, TAG_TYPE_INTEGER,
            (void **)&ptLocId, &tSize);

        // We were able to get the LocId.  Create
        // the location object now
        if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            LOCID_OBJECT hLocId;
            LOCATION_OBJECT hLocation;

            // Create the Location ID and Location object
            hLocId = LOCID.hCreate(tLocId);
            hLocation = LOCATION.hCreateForLocID(
                OSAL_FIXED_INVALID_OBJECT,
                OSAL_FIXED_INVALID_OBJECT, hLocId);
            LOCID.vDestroy(hLocId);

            // Provide it to the caller as a target object
            hTarget = (DSRL_TARGET_OBJECT)hLocation;
        }
    }

    return hTarget;
}

/*****************************************************************************
*
*   LOCATION_bIsDevicePosition
*
*   This function is used to indicate if a LOCATION object is using data from
*   the DeVICE service or not
*
*****************************************************************************/
BOOLEAN LOCATION_bIsDevicePosition(
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bOwner, bDevBased = FALSE;

    // Verify ownership of this object
    bOwner =
        DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hLocation);
    if (bOwner == TRUE)
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT *)hLocation;

        if (psObj->ePositionType == LOCATION_POSITION_TYPE_DEVICE)
        {
            bDevBased = TRUE;
        }
    }

    return bDevBased;
}

/*****************************************************************************
*
*   LOCATION_bLineClip
*
*   This API is used to determine if the input line is crossing the square
*   polygon. In case of crossing this function will provide the clipped line
*   inside the square polygon. The algorithm is based on Cohen-Sutherland
*   algorithm. It divides the space on 9 different areas. Each area has its own
*   unique binary id. Please look to the table below for details.
*
*   1001 | 1000 | 1010
*   -----|------|-----
*   0001 | 0000 | 0010
*   -----|------|-----
*   0101 | 0100 | 0110
*
*   0000 area is a square polygon which we check for crossings. Each time then a
*   pair of coordinates is passed to the input the algorithm assigns a binary id
*   to the points and computes its bitwise AND. If the result is > 0 it means
*   that the line is situated on the nearby areas and there are no any
*   crossings. If the result is = 0 the algorithm clips the line until the
*   nearest boarder. Then it renews the id's, calculates the bitwise AND and
*   repeats the clipping if necessary. This is done until both codes of line will
*   be = 0 or until the bitwise AND of those codes will be > 0.
*
*   Inputs:
*
*   hX0, hY0 - Line beginning X, Y coordinate.
*   hX1, hY1 - Line ending X, Y coordinate.
*   hLocation - Current location.
*
*   Outputs:
*
*   TRUE if there is a crossing. Otherwise - FALSE.
*
*******************************************************************************/
BOOLEAN LOCATION_bLineClip (
    OSAL_FIXED_OBJECT hX0Original,
    OSAL_FIXED_OBJECT hY0Original,
    OSAL_FIXED_OBJECT hX1Original,
    OSAL_FIXED_OBJECT hY1Original,
    LOCATION_OBJECT  hLocation
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    N32 n32CodeA = 0, n32CodeB = 0, n32Code = 0;
    OSAL_FIXED_OBJECT hX0, hY0, hX1, hY1, hOffset;
    OSAL_FIXED_OBJECT hBotLeftX, hBotLeftY, hTopRightX, hTopRightY, hCurCrdX,
        hCurCrdY, hCenterLat, hCenterLon;
    OSAL_FIXED_OBJECT hDxDy,hDyDx, hDx, hDy, hMultiply, hSubtract;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 16];
    UN8 un8NumFixed = 0;    // Keep track of the fixed objects
    BOOLEAN bIsVertical = FALSE, bIsHorizontal = FALSE;
    BOOLEAN bReturn = FALSE;
    BOOLEAN bZeroRadius = FALSE;
    BOOLEAN bResult = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    do
    {
        // Create working copies of input coordinates
        hX0 = OSAL_FIXED.hDuplicateInMemory(hX0Original,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hX1 = OSAL_FIXED.hDuplicateInMemory(hX1Original,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hY0 = OSAL_FIXED.hDuplicateInMemory(hY0Original,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hY1 = OSAL_FIXED.hDuplicateInMemory(hY1Original,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // This is 0.0003 mi radius, which will be used for
        // expanding zero-radius target location. Assumed to be small
        // enough to represent point area
        hOffset = OSAL_FIXED.hCreate(0,3,4);

        // Check input parameters
        if (hX0 == OSAL_FIXED_INVALID_OBJECT ||
            hY0 == OSAL_FIXED_INVALID_OBJECT ||
            hX1 == OSAL_FIXED_INVALID_OBJECT ||
            hY1 == OSAL_FIXED_INVALID_OBJECT ||
            hLocation == LOCATION_INVALID_OBJECT )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LOCATION_OBJECT_NAME": LOCATION_bLineClip: Invalid input\n");
            break;
        }

        hCenterLat = LOCATION.hLat(hLocation);
        hCenterLon = LOCATION.hLon(hLocation);

        // Bottom left corner of target location's bounding box
        hBotLeftX = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hBotLeftY = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturn = LOCATION.eIsPoint(hLocation, &bZeroRadius);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Location is wrong. Error!
            break;
        }

        if( bZeroRadius == FALSE )
        {
            // Get the bottom left lat, lon
            bResult = LOCATION_bBottomLeft(hLocation,hBotLeftY, hBotLeftX);

            if( bResult == FALSE )
            {
                break;
            }
        }
        else
        {
            OSAL_FIXED.eSubtract(hCenterLat,hOffset,hBotLeftY);
            OSAL_FIXED.eSubtract(hCenterLon,hOffset,hBotLeftX);
        }

        // Top right corner of target location's bounding box
        hTopRightX = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hTopRightY = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        if( bZeroRadius == FALSE )
        {
            // Get the top right lat, lon
            bResult = LOCATION_bTopRight( hLocation, hTopRightY, hTopRightX );

            if( bResult == FALSE )
            {
                break;
            }
        }
        else
        {
            OSAL_FIXED.eAdd(hCenterLat,hOffset,hTopRightY);
            OSAL_FIXED.eAdd(hCenterLon,hOffset,hTopRightX);
        }

#if DEBUG_LOCATION_COORDINATES == 1
        printf("[DEBUG] %s: (%f, %f) - (%f, %f)\n",
            __FUNCTION__,
            LOCATION_DEBUG_fFixed2Float(hY0), LOCATION_DEBUG_fFixed2Float(hX0),
            LOCATION_DEBUG_fFixed2Float(hY1), LOCATION_DEBUG_fFixed2Float(hX1));
        printf("-> (%f, %f) - (%f, %f)\n",
            LOCATION_DEBUG_fFixed2Float(hTopRightY), LOCATION_DEBUG_fFixed2Float(hTopRightX),
            LOCATION_DEBUG_fFixed2Float(hBotLeftY), LOCATION_DEBUG_fFixed2Float(hBotLeftX));
#endif

        n32CodeA = n32ComputeCode(hX0, hY0, hBotLeftX, hBotLeftY, hTopRightX, hTopRightY);
        n32CodeB = n32ComputeCode(hX1, hY1, hBotLeftX, hBotLeftY, hTopRightX, hTopRightY);

        hDx = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hDy = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eSubtract(hX1, hX0, hDx);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }
        eReturnCode = OSAL_FIXED.eSubtract(hY1, hY0, hDy);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        /* Precomputing of the line crossing.
        *
        * P1(hX0,hY0) and P2(hX1,hY1) - points on the line.
        *
        * Line Function is:
        * y = m(x - hX0) + hY0
        * or
        * y = m(x - hX1) + hY1
        *
        * where
        * m = (hY1-hY0)/(hX1-hX0) - line slope.
        *
        * We use this equations to compute a crossing with a square area.
        */
        hDxDy = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hDyDx = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hCurCrdX = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hCurCrdY = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eDivide(hDx, hDy, hDxDy); //computing line slopes here
        if (eReturnCode == OSAL_ERROR_MATH_DIVIDE_BY_ZERO)
        {
            // This is horizontal line (dY/dX = 0)
            bIsHorizontal = TRUE;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eDivide(hDy, hDx, hDyDx);//..and here
        if (eReturnCode == OSAL_ERROR_MATH_DIVIDE_BY_ZERO)
        {
            // This is vertical line (dX/dY = 0)
            bIsVertical = TRUE;
        }
        else if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hSubtract = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hMultiply = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Setting return value to TRUE by default
        bReturn = TRUE;

        while ((n32CodeA | n32CodeB) != 0)
        {
            if ((n32CodeA & n32CodeB) != 0)
            {
                // Both points are on the one side and
                // have no crossings with square area
                bReturn = FALSE;
                break;
            }

            // performing a check here to work only with point that
            // is not inside of the square area
            if (n32CodeA != 0)
            {
                n32Code = n32CodeA;
                OSAL_FIXED.bCopyToMemory(hX0, hCurCrdX);
                OSAL_FIXED.bCopyToMemory(hY0, hCurCrdY);
            }
            else
            {
                n32Code = n32CodeB;
                OSAL_FIXED.bCopyToMemory(hX1, hCurCrdX);
                OSAL_FIXED.bCopyToMemory(hY1, hCurCrdY);
            }

            //computing the crossing with asymptotes.
            if (n32Code & 1)
            {
                // Right line
                if(bIsHorizontal == FALSE)
                {
                    OSAL_FIXED.eSubtract(hBotLeftX , hCurCrdX, hSubtract);
                    OSAL_FIXED.eMultiply(hSubtract, hDyDx, hMultiply);
                    OSAL_FIXED.eAdd(hMultiply, hCurCrdY, hCurCrdY);
                }
                OSAL_FIXED.bCopyToMemory(hBotLeftX, hCurCrdX);
            }
            else if (n32Code & 2)
            {
                // Left line
                if(bIsHorizontal == FALSE)
                {
                    OSAL_FIXED.eSubtract(hTopRightX, hCurCrdX, hSubtract);
                    OSAL_FIXED.eMultiply(hSubtract, hDyDx, hMultiply);
                    OSAL_FIXED.eAdd(hMultiply, hCurCrdY, hCurCrdY);
                }
                OSAL_FIXED.bCopyToMemory(hTopRightX, hCurCrdX);
            }
            else if(n32Code & 4)
            {
                // Bottom line
                if(bIsVertical == FALSE)
                {
                    OSAL_FIXED.eSubtract(hBotLeftY, hCurCrdY, hSubtract);
                    OSAL_FIXED.eMultiply(hSubtract, hDxDy, hMultiply);
                    OSAL_FIXED.eAdd(hMultiply, hCurCrdX, hCurCrdX);
                }
                OSAL_FIXED.bCopyToMemory(hBotLeftY, hCurCrdY);
            }
            else if(n32Code & 8)
            {
                // Top line
                if(bIsVertical == FALSE)
                {
                    OSAL_FIXED.eSubtract(hTopRightY, hCurCrdY, hSubtract);
                    OSAL_FIXED.eMultiply(hSubtract, hDxDy, hMultiply);
                    OSAL_FIXED.eAdd(hMultiply, hCurCrdX, hCurCrdX);
                }
                OSAL_FIXED.bCopyToMemory(hTopRightY, hCurCrdY);
            }

            // renew codes
            if(n32Code == n32CodeA)
            {
                OSAL_FIXED.bCopyToMemory(hCurCrdX, hX0);
                OSAL_FIXED.bCopyToMemory(hCurCrdY, hY0);
                n32CodeA = n32ComputeCode(hX0, hY0, hBotLeftX, hBotLeftY, hTopRightX, hTopRightY);
            }
            else
            {
                OSAL_FIXED.bCopyToMemory(hCurCrdX, hX1);
                OSAL_FIXED.bCopyToMemory(hCurCrdY, hY1);
                n32CodeB = n32ComputeCode(hX1, hY1, hBotLeftX, hBotLeftY, hTopRightX, hTopRightY);
            }
        }
    } while (FALSE);

    OSAL_FIXED.vDestroy(hOffset);

    return bReturn;
}

/*****************************************************************************
*
*   LOCATION_bRayCross
*
*   This API is used to make a horizontal ray from a single point. After that
*   it computes the cross point and if it exist it returns the result. This
*   simple functionality can be very helpful if it is need to determine the
*   point's positions depending of the line.
*   It is very easy to determine if a single point is inside the polygon.
*   This API function should be looped through the lines of the polygon.
*   Depending of the number of the crossings we can clearly determine the
*   position of the point. If it is odd - the point is inside of the polygon.
*   Otherwise it is outside.
*
*   Inputs:
*
*   hX0, hY0 - Line beginning X, Y coordinate.
*   hX1, hY1 - Line ending X, Y coordinate.
*   hLocation - Current location.
*
*   Outputs:
*
*   TRUE if there is a crossing with a ray. Otherwise - FALSE.
*
*****************************************************************************/
BOOLEAN LOCATION_bRayCross (
    OSAL_FIXED_OBJECT hX0,
    OSAL_FIXED_OBJECT hY0,
    OSAL_FIXED_OBJECT hX1,
    OSAL_FIXED_OBJECT hY1,
    LOCATION_OBJECT  hLocation
        )
{
    OSAL_FIXED_OBJECT hLonX, hLatY, hCompRes1, hCompRes2, hDivide;
    N16 n16CompareValue1, n16CompareValue2;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 3];
    UN8 un8NumFixed = 0;

#if DEBUG_LOCATION_COORDINATES == 1
    printf("[DEBUG] %s: (%f, %f) - (%f, %f)\n",
        __FUNCTION__,
        LOCATION_DEBUG_fFixed2Float(hY0), LOCATION_DEBUG_fFixed2Float(hX0),
        LOCATION_DEBUG_fFixed2Float(hY1), LOCATION_DEBUG_fFixed2Float(hX1));
#endif

    hCompRes1 = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
    hCompRes2 = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
    hDivide = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
    hLonX = LOCATION.hLon(hLocation);
    hLatY = LOCATION.hLat(hLocation);

#if DEBUG_LOCATION_COORDINATES == 1
    printf("-> (%f, %f)\n",
        LOCATION_DEBUG_fFixed2Float(hLatY), LOCATION_DEBUG_fFixed2Float(hLonX));
#endif

    n16CompareValue1 = OSAL_FIXED.n16Compare(hY0, hLatY);
    n16CompareValue2 = OSAL_FIXED.n16Compare(hY1, hLatY);
    if (
         (( n16CompareValue1 <= 0 ) && ( n16CompareValue2 > 0 )) ||
         (( n16CompareValue1 > 0 ) && ( n16CompareValue2 <= 0 ))
       )
    {
        OSAL_FIXED.eSubtract(hLatY , hY0, hCompRes1);
        OSAL_FIXED.eSubtract(hY1 , hY0, hCompRes2);
        OSAL_FIXED.eDivide(hCompRes1, hCompRes2, hDivide);

        OSAL_FIXED.eSubtract(hX1 , hX0, hCompRes1);
        OSAL_FIXED.eMultiply(hDivide, hCompRes1, hCompRes2);
        OSAL_FIXED.eAdd(hCompRes2, hX0, hCompRes1); // from now hCompRes1 and hLatY are the coordinates of the crossing
        n16CompareValue1 = OSAL_FIXED.n16Compare(hLonX, hCompRes1);
        if( n16CompareValue1 < 0 )
        {
            return TRUE;
        }
    }

    return FALSE;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   psCreateLocation
*
*****************************************************************************/
static LOCATION_OBJECT_STRUCT *psCreateLocation (
    SMS_OBJECT hOwner,
    LOCID_OBJECT hLocID,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_ATTRIBUTE_STRUCT *psDescriptiveAttrs,
    BOOLEAN bPerformDeepCopy,
    LOCATION_TYPE_ENUM eLocationAreaType
        )
{
    LOCATION_OBJECT_STRUCT *psObj;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner(hOwner);
    if ((bOwner == FALSE) && (hOwner != SMS_INVALID_OBJECT))
    {
        // Error!
        return NULL;
    }

    psObj = psCreateObject(
        hOwner,
        LOCATION_POSITION_TYPE_STATIC,
        eLocationAreaType,
        hLocID,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits, puProperties, psDescriptiveAttrs,
        bPerformDeepCopy);

    return psObj;
}

/*****************************************************************************
*
*   psCreateObject
*
*****************************************************************************/
static LOCATION_OBJECT_STRUCT *psCreateObject (
    SMS_OBJECT hOwner,
    LOCATION_POSITION_TYPE_ENUM ePosType,
    LOCATION_TYPE_ENUM eType,
    LOCID_OBJECT hLocID,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_ATTRIBUTE_STRUCT *psDescriptiveAttrs,
    BOOLEAN bPerformDeepCopy
        )
{
    LOCATION_OBJECT_STRUCT *psObj;
    BOOLEAN bOk = FALSE;

    // Verify object type
    if ((ePosType >= LOCATION_POSITION_TYPE_INVALID) ||
        (eType >= LOCATION_TYPE_UNKNOWN))
    {
        return NULL;
    }

    // Create an instance of the LOCATION object
    psObj = (LOCATION_OBJECT_STRUCT *)
        DSRL_TARGET_hCreate(
            LOCATION_OBJECT_NAME,
            DSRL_TARGET_TYPE_LOCATION,
            sizeof(LOCATION_OBJECT_STRUCT),
            hOwner, // Child of hOwner (if provided, otherwise it is a parent)
            FALSE); // No Lock (ignored if hSMS is valid (parent provided))

    if (psObj == NULL)
    {
        // Error!
        return NULL;
    }

    // Initialize attributes
    psObj->ePositionType = ePosType;
    psObj->bIsPoint = FALSE;

    vLocationInitCommonPart(
        psObj,
        bPerformDeepCopy,
        hLocID,
        psDescriptiveAttrs);

    switch (eType)
    {
        case LOCATION_TYPE_CIRCLE:
        {
            bOk = bCircleInitializeObject(
                (LOCATION_OBJECT)psObj,
                n32Lat, un8LatBits,
                n32Lon, un8LonBits,
                puProperties,
                ePosType,
                bPerformDeepCopy,
                (hLocID == LOCID_INVALID_OBJECT) ? FALSE : TRUE);
        }
        break;
        case LOCATION_TYPE_RECTANGLE:
        {
            bOk = bRectangleInitializeObject(
                (LOCATION_OBJECT)psObj,
                n32Lat, un8LatBits,
                n32Lon, un8LonBits,
                puProperties,
                ePosType,
                bPerformDeepCopy,
                (hLocID == LOCID_INVALID_OBJECT) ? FALSE : TRUE);
        }
        break;
        case LOCATION_TYPE_UNKNOWN:
        {
            // Error. Nothing to do.
        }
        break;
    }

    if(bOk == FALSE)
    {
        LOCATION.vDestroy((LOCATION_OBJECT)psObj);
        psObj = NULL;
    }

    return psObj;
}

/*****************************************************************************
*
*   bRectangleInitializeObject
*
*****************************************************************************/
static BOOLEAN bRectangleInitializeObject (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_POSITION_TYPE_ENUM ePosType,
    BOOLEAN bPerformDeepCopy,
    BOOLEAN bIsLocID
        )
{
    do
    {
        LOCATION_OBJECT_STRUCT *psObj =
            (LOCATION_OBJECT_STRUCT*)hLocation;
        OSAL_FIXED_OBJECT hAddLat, hAddLon, hDeltaLat, hDeltaLon,
                          hDeltaLatRad, hDeltaLatDeg, hDeltaLonDeg,
                          hFixed90, hFixed2;
        OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 7];
        BOOLEAN bReturn;
        N32 n32Width, n32Height;
        N16 n16CompareResult;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // We already have bottom-left coordinates, so let's populate them
        // into the structure and save time on calculations
        psObj->hBotLeftLat = OSAL_FIXED.hCreateInMemory(n32Lat, un8LatBits,
            &psObj->atBotLeftLatData[0]);
        psObj->hBotLeftLon = OSAL_FIXED.hCreateInMemory(n32Lon, un8LonBits,
            &psObj->atBotLeftLonData[0]);

        // Initialize top right coordinate objects to be equal to 
        // bottom left by default (to automatically cover zero deltas case)
        psObj->hTopRightLat = OSAL_FIXED.hCreateInMemory(n32Lat, un8LatBits,
            &psObj->atTopRightLatData[0]);
        psObj->hTopRightLon = OSAL_FIXED.hCreateInMemory(n32Lon, un8LonBits,
            &psObj->atTopRightLonData[0]);

        // Now let's check if provided distances do not drive us outside 
        // of allowed coordinates area

        // Creating fixed objects for calculating rectangle sides in 
        // radians and degrees
        hDeltaLatRad = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);
        hDeltaLatDeg = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);
        hDeltaLonDeg = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 2]);

        // Creating fixed objects for calculating center point
        hAddLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 3]);
        hAddLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 4]);

        // Create fixed constants 
        // 90 degrees - the North Pole latitude
        hFixed90 = OSAL_FIXED.hCreateInMemory(90, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 5]);
        // 2 - denominator for division
        hFixed2 = OSAL_FIXED.hCreateInMemory(2, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 6]);

        // Calculate delta latitude in radians by just dividing
        // the distance by Earth radius
        if (puProperties->sRectangle.hHeight != DISTANCE_INVALID_OBJECT)
        {
            bReturn = DISTANCE_bDistanceOverRadius(
                puProperties->sRectangle.hHeight,
                hDeltaLatRad);
            if (FALSE == bReturn)
            {
                // Error!
                break;
            }
        }

        // Convert it into degrees...
        eReturnCode = OSAL_FIXED.eRadianToDegree(hDeltaLatRad,
            hDeltaLatDeg);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            break;
        }

        // ... and add to latitude (in degrees again) of the bottom-left
        // corner of our rectangle. This is expected to become a latitude 
        // of out top-right corner.
        eReturnCode = OSAL_FIXED.eAdd(psObj->hBotLeftLat, hDeltaLatDeg,
            psObj->hTopRightLat);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            break;
        }

        // Did it wrap over the North pole?
        n16CompareResult = OSAL_FIXED.n16Compare(
            psObj->hTopRightLat, 
            hFixed90);

        if (n16CompareResult > 0)
        {
            // We wrapped over the Pole, which is not permitted
            break;
        }

        // Now calculating the longitude of top-right corner
        if (puProperties->sRectangle.hWidth != DISTANCE_INVALID_OBJECT)
        {
            bReturn = bCalculateLongitudeAtDistance(
                psObj->hBotLeftLat, psObj->hBotLeftLon,
                puProperties->sRectangle.hWidth,
                psObj->hTopRightLon);

            if (FALSE == bReturn)
            {
                // Error!
                break;
            }

            bReturn = bCheckLatLonRange(psObj->hTopRightLat,
                psObj->hTopRightLon);

            if (FALSE == bReturn)
            {
                // Error: the point is outside of allowed area
                break;
            }
        }

        // Initialize interface
        psObj->psIntf = &GsLocationRectangleIntf;

        // Getting coordinate deltas as fixed objects
        hDeltaLat = DISTANCE.hValue(puProperties->sRectangle.hHeight,
            DISTANCE_UNIT_TYPE_MILES);
        hDeltaLon = DISTANCE.hValue(puProperties->sRectangle.hWidth,
            DISTANCE_UNIT_TYPE_MILES);

        // Divide them by 2 and add to base point (bottom left)
        // This is going to be a center of the area
        eReturnCode = OSAL_FIXED.eDivide(hDeltaLatDeg, hFixed2, hAddLat);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            break;
        }

        eReturnCode = OSAL_FIXED.eSubtract(
            psObj->hTopRightLon, psObj->hBotLeftLon, hDeltaLonDeg);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            break;
        }

        eReturnCode = OSAL_FIXED.eDivide(hDeltaLonDeg, hFixed2, hAddLon);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            break;
        }

        // Initialize object per inputs
        psObj->hLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &psObj->atLatData[0]);

        psObj->hLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &psObj->atLonData[0]);

        // Calculate the center of the area
        eReturnCode = OSAL_FIXED.eAdd(psObj->hBotLeftLat, hAddLat,
            psObj->hLat);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            break;
        }

        eReturnCode = OSAL_FIXED.eAdd(psObj->hBotLeftLon, hAddLon,
            psObj->hLon);
        if (OSAL_SUCCESS != eReturnCode)
        {
            // Error!
            break;
        }

        // Set area specific properties
        if (bPerformDeepCopy == FALSE)
        {
            psObj->uProperties.sRectangle = puProperties->sRectangle;
        }
        else
        {
            // DISTANCE.hDuplicate returns DISTANCE_INVALID_OBJECT for
            // invalid source provided, so no bother to check input here
            psObj->uProperties.sRectangle.hHeight =
                DISTANCE.hDuplicate(puProperties->sRectangle.hHeight);
            psObj->uProperties.sRectangle.hWidth =
                DISTANCE.hDuplicate(puProperties->sRectangle.hWidth);
        }

        // If width and height are 0 (either the object itself or DISTANCE object
        // of zero value) we need to mark the area as a single point
        if (psObj->uProperties.sRectangle.hWidth == DISTANCE_INVALID_OBJECT)
        {
            psObj->uProperties.sRectangle.hWidth =
                DISTANCE.hCreate(0, DISTANCE_UNIT_TYPE_MILES);
            if (psObj->uProperties.sRectangle.hWidth == DISTANCE_INVALID_OBJECT)
            {
                // Error! Cleaning up.
                break;
            }
            n32Width = 0;
        }
        else
        {
            n32Width = OSAL_FIXED.n32Value(hDeltaLon);
        }

        if (psObj->uProperties.sRectangle.hHeight == DISTANCE_INVALID_OBJECT)
        {
            psObj->uProperties.sRectangle.hHeight =
                DISTANCE.hCreate(0, DISTANCE_UNIT_TYPE_MILES);
            if (psObj->uProperties.sRectangle.hHeight == DISTANCE_INVALID_OBJECT)
            {
                // Error! Cleaning up.
                break;
            }
            n32Height = 0;
        }
        else
        {
            n32Height = OSAL_FIXED.n32Value(hDeltaLat);
        }

        // If both dimensions are zeros, it is just a point
        if ((n32Width == 0) && (n32Height == 0))
        {
            psObj->bIsPoint = TRUE;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vRectangleDestroyProperties
*
*****************************************************************************/
static void vRectangleDestroyProperties(
    const LOCATION_OBJECT hLocation
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT*)hLocation;

    if (psObj->uProperties.sRectangle.hHeight != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(psObj->uProperties.sRectangle.hHeight);
        psObj->uProperties.sRectangle.hHeight = DISTANCE_INVALID_OBJECT;
    }

    if (psObj->uProperties.sRectangle.hWidth != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(psObj->uProperties.sRectangle.hWidth);
        psObj->uProperties.sRectangle.hWidth = DISTANCE_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bRectangleCoordinatesWithinArea
*
*****************************************************************************/
static BOOLEAN bRectangleCoordinatesWithinArea (
    const LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bCoordsInRect = FALSE;
    N16 n16Compare;

    // Compare both pairs to see if they are the same
    // We can skip out on performing a calculation if it is the case
    n16Compare = OSAL_FIXED.n16Compare(psObj->hLat, hLat);
    if (n16Compare == 0)
    {
        n16Compare = OSAL_FIXED.n16Compare(psObj->hLon, hLon);
    }

    // If they matched, then its the same point
    if (n16Compare == 0)
    {
        bCoordsInRect = TRUE;
    }
    else
    {
       bCoordsInRect = bPointInsideRectangle(psObj, hLat, hLon);
    }

    return bCoordsInRect;
}

/*****************************************************************************
*
*   bRectangleContains
*
*****************************************************************************/
static BOOLEAN bRectangleContains (
    LOCATION_OBJECT hContainer,
    LOCATION_OBJECT hContained,
    DISTANCE_OBJECT hThreshold
        )
{
    BOOLEAN bContains = FALSE;
    LOCATION_OBJECT_STRUCT *psObjContainer =
        (LOCATION_OBJECT_STRUCT *)hContainer;
    LOCATION_OBJECT_STRUCT *psObjContained =
        (LOCATION_OBJECT_STRUCT *)hContained;
    N16 n16Compare = N16_MIN, n16Diff;
    OSAL_FIXED_OBJECT hTopFixed, hBottomFixed, hLeftFixed, hRightFixed;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
    BOOLEAN bOk;

    // Create objects
    hTopFixed = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);
    hBottomFixed = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);
    hLeftFixed = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 2]);
    hRightFixed = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 3]);

    if ((hTopFixed == OSAL_FIXED_INVALID_OBJECT) ||
        (hBottomFixed == OSAL_FIXED_INVALID_OBJECT) ||
        (hRightFixed == OSAL_FIXED_INVALID_OBJECT) ||
        (hLeftFixed == OSAL_FIXED_INVALID_OBJECT))
    {
        //Error!
        return FALSE;
    }

    do
    {
        if (psObjContained->psIntf->eType == LOCATION_TYPE_RECTANGLE)
        {
            // Checking if rectangle contains rectangle

            // Making sure corner coordinates are updated inside the objects
            bOk = LOCATION_bTopRight(hContainer,
                OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);
            bOk &= LOCATION_bBottomLeft(hContainer,
                OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);
            bOk &= LOCATION_bTopRight(hContained,
                OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);
            bOk &= LOCATION_bBottomLeft(hContained,
                OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);

            if (bOk == FALSE)
            {
                // Something is wrong on the way. Error!
                break;
            }

            // Top Right of contained shall be less than top right
            // of container
            n16Compare = OSAL_FIXED.n16Compare(psObjContained->hTopRightLat,
                psObjContainer->hTopRightLat);
            n16Diff = OSAL_FIXED.n16Compare(psObjContained->hTopRightLon,
                psObjContainer->hTopRightLon);
            // Storing bigger value into n16Compare
            n16Compare = (n16Compare < n16Diff) ? n16Diff : n16Compare;

            // Bottom Left of container shall be less than bottom left
            // of contained
            n16Diff = OSAL_FIXED.n16Compare(psObjContainer->hBotLeftLat,
                psObjContained->hBotLeftLat);
            // Storing biggest value into n16Compare
            n16Compare = (n16Compare < n16Diff) ? n16Diff : n16Compare;
            n16Diff = OSAL_FIXED.n16Compare(psObjContainer->hBotLeftLon,
                psObjContained->hBotLeftLon);
            // Storing biggest value into n16Compare
            n16Compare = (n16Compare < n16Diff) ? n16Diff : n16Compare;
        }
        else if (psObjContained->psIntf->eType == LOCATION_TYPE_CIRCLE)
        {
            // Checking if rectangle contains circle
            // Circumscribed square of circle shall be within the rectangle
            LOCATION_OBJECT hTempLocation;
            OSAL_FIXED_OBJECT hValue, hTopBearing, hBottomBearing,
                hLeftBearing, hRightBearing;
            OSAL_FIXED_OBJECT_DATA atData[OSAL_FIXED_OBJECT_SIZE * 4];

            // Preparing fixed objects for bearings
            hTopBearing = OSAL_FIXED.hCreateInMemory(LOCATION_TOP_BEARING, 0,
                &atData[OSAL_FIXED_OBJECT_SIZE * 0]);
            hBottomBearing = OSAL_FIXED.hCreateInMemory(LOCATION_BOTTOM_BEARING, 0,
                &atData[OSAL_FIXED_OBJECT_SIZE * 1]);
            hRightBearing = OSAL_FIXED.hCreateInMemory(LOCATION_RIGHT_BEARING, 0,
                &atData[OSAL_FIXED_OBJECT_SIZE * 2]);
            hLeftBearing = OSAL_FIXED.hCreateInMemory(LOCATION_LEFT_BEARING, 0,
                &atData[OSAL_FIXED_OBJECT_SIZE * 3]);

            // Calculating top/north latitude
            hTempLocation = LOCATION.hLocationAtDistance(
                (LOCATION_OBJECT)psObjContained,
                psObjContained->uProperties.sCircle.hRadius,
                hTopBearing);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hValue = LOCATION.hLat(hTempLocation);
            bOk = OSAL_FIXED.bCopyToMemory(hValue, hTopFixed);
            // Destroying unneeded object
            LOCATION.vDestroy((LOCATION_OBJECT)hTempLocation);
            if (bOk == FALSE)
            {
                break;
            }

            // Calculating bottom/south latitude
            hTempLocation = LOCATION.hLocationAtDistance(
                (LOCATION_OBJECT)psObjContained,
                psObjContained->uProperties.sCircle.hRadius,
                hBottomBearing);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hValue = LOCATION.hLat(hTempLocation);
            bOk = OSAL_FIXED.bCopyToMemory(hValue, hBottomFixed);
            // Destroying unneeded object
            LOCATION.vDestroy((LOCATION_OBJECT)hTempLocation);
            if (bOk == FALSE)
            {
                break;
            }

            // Calculating left/west longitude
            hTempLocation = LOCATION.hLocationAtDistance(
                (LOCATION_OBJECT)psObjContained,
                psObjContained->uProperties.sCircle.hRadius,
                hLeftBearing);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hValue = LOCATION.hLon(hTempLocation);
            bOk = OSAL_FIXED.bCopyToMemory(hValue, hLeftFixed);
            // Destroying unneeded object
            LOCATION.vDestroy((LOCATION_OBJECT)hTempLocation);
            if (bOk == FALSE)
            {
                break;
            }

            // Calculating right/east longitude
            hTempLocation = LOCATION.hLocationAtDistance(
                (LOCATION_OBJECT)psObjContained,
                psObjContained->uProperties.sCircle.hRadius,
                hRightBearing);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hValue = LOCATION.hLon(hTempLocation);
            bOk = OSAL_FIXED.bCopyToMemory(hValue, hRightFixed);
            // Destroying unneeded object
            LOCATION.vDestroy((LOCATION_OBJECT)hTempLocation);
            if (bOk == FALSE)
            {
                break;
            }

            // Now compare coordinates
            // Top Right of contained shall be less than top right
            // of container
            n16Compare = OSAL_FIXED.n16Compare(hTopFixed,
                psObjContainer->hTopRightLat);
            n16Diff = OSAL_FIXED.n16Compare(hRightFixed,
                psObjContainer->hTopRightLon);
            // Storing bigger value into n16Compare
            n16Compare = (n16Compare < n16Diff)? n16Diff : n16Compare;

            // Bottom Left of container shall be less than bottom left
            // of contained
            n16Diff = OSAL_FIXED.n16Compare(psObjContainer->hBotLeftLat,
                hBottomFixed);
            // Storing biggest value into n16Compare
            n16Compare = (n16Compare < n16Diff)? n16Diff : n16Compare;
            n16Diff = OSAL_FIXED.n16Compare(psObjContainer->hBotLeftLon,
                hLeftFixed);
            // Storing biggest value into n16Compare
            n16Compare = (n16Compare < n16Diff)? n16Diff : n16Compare;
        }
        else
        {
            // Unknown area type. Error!
            break;
        }

        // Was a threshold provided?
        if (hThreshold != DISTANCE_INVALID_OBJECT)
        {
            OSAL_FIXED_OBJECT hThresholdFixed;
            N32 n32Threshold;

            // Yes - extract it
            hThresholdFixed = DISTANCE.hValue(
                hThreshold, DISTANCE_UNIT_TYPE_MILES);
            n32Threshold = OSAL_FIXED.n32Value(hThresholdFixed);

            // Adjust by previously calculated difference by
            // the threshold provided
            n16Compare -= (N16)n32Threshold;
        }

        if (n16Compare <= 0)
        {
            bContains = TRUE;
        }
    } while (FALSE);

    return bContains;
}



/*****************************************************************************
*
*   bRectangleCompare
*
*****************************************************************************/
static BOOLEAN bRectangleCompare (
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        )
{
    LOCATION_OBJECT_STRUCT *psObj1 = (LOCATION_OBJECT_STRUCT *)hLocation1;
    LOCATION_OBJECT_STRUCT *psObj2 = (LOCATION_OBJECT_STRUCT *)hLocation2;
    N16 n16Compare = N16_MIN;

    // Compare DISTANCE objects themselves first
    if (psObj1->uProperties.sRectangle.hHeight ==
        psObj2->uProperties.sRectangle.hHeight)
    {
        if (psObj1->uProperties.sRectangle.hWidth ==
            psObj2->uProperties.sRectangle.hWidth)
        {
            n16Compare = 0;
        }
    }

    // If some objects are different then compare values
    if (n16Compare != 0)
    {
        n16Compare = DISTANCE.n16Compare (
            psObj1->uProperties.sRectangle.hHeight,
            psObj2->uProperties.sRectangle.hHeight);
        if (n16Compare == 0)
        {
            n16Compare = DISTANCE.n16Compare (
                psObj1->uProperties.sRectangle.hWidth,
                psObj2->uProperties.sRectangle.hWidth);
        }
    }

    if (n16Compare != 0)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bRectangleUpdateCoordinates
*
*****************************************************************************/
static BOOLEAN bRectangleUpdateCoordinates (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits
        )
{
    BOOLEAN bOk;
    LOCATION_OBJECT_STRUCT *psObj = (LOCATION_OBJECT_STRUCT *)hLocation;
    OSAL_FIXED_OBJECT hFixedLat, hFixedLon, hFixedCenterLat, hFixedCenterLon;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
    N32 n32CenterLat, n32CenterLon;
    UN8 un8CenterLatBits, un8CenterLonBits;

    // Input coordinates are provided for bottom-left corner.
    // So we need to calculate where the new center is.
    hFixedLat = OSAL_FIXED.hCreateInMemory(n32Lat, un8LatBits,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);
    hFixedLon = OSAL_FIXED.hCreateInMemory(n32Lon, un8LonBits,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);

    hFixedCenterLat = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 2]);
    hFixedCenterLon = OSAL_FIXED.hCreateInMemory(0, 0,
        &atFixedData[OSAL_FIXED_OBJECT_SIZE * 3]);

    bOk = bCalculateHalfwayShift(hFixedLat, hFixedLon,
        psObj->uProperties.sRectangle.hWidth,
        psObj->uProperties.sRectangle.hHeight,
        LOCATION_TOP_BEARING, LOCATION_RIGHT_BEARING,
        hFixedCenterLat, hFixedCenterLon);
    if (bOk == FALSE)
    {
        return FALSE;
    }

    n32CenterLat = OSAL_FIXED.n32Value(hFixedCenterLat);
    un8CenterLatBits = OSAL_FIXED.un8NumFractionalBits(hFixedCenterLat);
    n32CenterLon = OSAL_FIXED.n32Value(hFixedCenterLon);
    un8CenterLonBits = OSAL_FIXED.un8NumFractionalBits(hFixedCenterLon);

    vUpdateCoordinatesCommonPart(hLocation,
        n32CenterLat, un8CenterLatBits,
        n32CenterLon, un8CenterLonBits);
    
    // Bottom-left is what we already have, so let's avoid calculating them
    // later again
    psObj->hBotLeftLat = OSAL_FIXED.hCreateInMemory(n32Lat, un8LatBits,
        &psObj->atBotLeftLatData[0]);
    psObj->hBotLeftLon = OSAL_FIXED.hCreateInMemory(n32Lon, un8LonBits,
        &psObj->atBotLeftLonData[0]);

    return TRUE;
}

/*****************************************************************************
*
*   n32RectangleFWrite
*
*****************************************************************************/
static N32 n32RectangleFWrite (
    const LOCATION_OBJECT hLocation,
    FILE *psFile
        )

{
    N32 n32Return = 0, n32Value;
    UN8 un8ByteToWrite, un8Bits;
    LOCATION_OBJECT_STRUCT *psObj = (LOCATION_OBJECT_STRUCT *)hLocation;

    // Write LOCATION to file...

    // Write location area type
    un8ByteToWrite = (UN8)LOCATION_TYPE_RECTANGLE;
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);

    //Write height and width for rectangular location
    if (psObj->uProperties.sRectangle.hHeight == DISTANCE_INVALID_OBJECT)
    {
        // Height is NOT present, so indicate it with the flag
        un8ByteToWrite = FALSE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    }
    else
    {
        // Height is present, so indicate it with the flag
        un8ByteToWrite = TRUE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
        n32Return += DISTANCE.n32FWrite(
            psObj->uProperties.sRectangle.hHeight,
            psFile);
    }

    if (psObj->uProperties.sRectangle.hWidth == DISTANCE_INVALID_OBJECT)
    {
        // Width is NOT present, so indicate it with the flag
        un8ByteToWrite = FALSE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    }
    else
    {
        // Width is present, so indicate it with the flag
        un8ByteToWrite = TRUE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
        n32Return += DISTANCE.n32FWrite(
            psObj->uProperties.sRectangle.hWidth,
            psFile);
    }

    // For rectangle we are storing bottom-left corner, so creation
    // of the object upon FRead would be consistent.
    // But first we want to make sure coordinates are in place.
    LOCATION_bBottomLeft((LOCATION_OBJECT)psObj,
        OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);

    // Write Latitude
    n32Value = OSAL_FIXED.n32Value(psObj->hBotLeftLat);
    un8Bits = OSAL_FIXED.un8NumFractionalBits(psObj->hBotLeftLat);

    un8ByteToWrite = BYTE0(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    n32Return += (N32)fwrite(&un8Bits, sizeof(UN8), 1, psFile);

    // Write Longitude
    n32Value = OSAL_FIXED.n32Value(psObj->hBotLeftLon);
    un8Bits = OSAL_FIXED.un8NumFractionalBits(psObj->hBotLeftLon);

    un8ByteToWrite = BYTE0(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    n32Return += (N32)fwrite(&un8Bits, sizeof(UN8), 1, psFile);

    n32Return += n32WriteCommonPart(psObj, psFile);

    return n32Return;
}

/*****************************************************************************
*
*   hRectangleFRead
*
*****************************************************************************/
static LOCATION_OBJECT hRectangleFRead (
    FILE *psFile
        )
{
    LOCATION_OBJECT_STRUCT *psObj;
    N32 n32Lat = 0, n32Lon = 0;
    UN8 un8LatBits = 0, un8LonBits = 0, un8CurByteRead = 0;
    size_t tTotal;
    LOCATION_ATTRIBUTE_STRUCT sAttrs;
    LOCID_OBJECT hLocID = LOCID_INVALID_OBJECT;
    DISTANCE_OBJECT hRectHeight = DISTANCE_INVALID_OBJECT;
    DISTANCE_OBJECT hRectWidth = DISTANCE_INVALID_OBJECT;
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;

    // Verify input
    if(psFile == NULL)
    {
        return NULL;
    }

    // Read LOCATION from file...

    // Read rectangle height present flag
    tTotal = fread(&un8CurByteRead, 1, 1, psFile);
    if(tTotal != 1)
    {
        // End of file or error occurred
        return LOCATION_INVALID_OBJECT;
    }

    if (un8CurByteRead == TRUE)
    {
        // Read rectangle height
        hRectHeight = DISTANCE.hFRead(psFile);
    }

    // Read rectangle width present flag
    tTotal = fread(&un8CurByteRead, 1, 1, psFile);
    if(tTotal != 1)
    {
        // End of file or error occurred
        return LOCATION_INVALID_OBJECT;
    }
    if (un8CurByteRead == TRUE)
    {
        // Read rectangle width
        hRectWidth = DISTANCE.hFRead(psFile);
    }

    vFReadCommonPart(psFile, &n32Lat, &n32Lon,
        &un8LatBits, &un8LonBits, &sAttrs, &hLocID);

    // Populating area-specific properties
    uProperties.sRectangle.hHeight = hRectHeight;
    uProperties.sRectangle.hWidth = hRectWidth;

    // Create an instance of the LOCATION object
    psObj = psCreateLocation(
        SMS_INVALID_OBJECT, hLocID,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits,
        &uProperties, &sAttrs, FALSE,
        LOCATION_TYPE_RECTANGLE);

    if(psObj == NULL)
    {
        // Error!
        DISTANCE.vDestroy(hRectHeight);
        DISTANCE.vDestroy(hRectWidth);
        LOCID.vDestroy(hLocID);
        STRING_vDestroy(sAttrs.hStreetNum);
        STRING_vDestroy(sAttrs.hStreetName);
        STRING_vDestroy(sAttrs.hCity);
        STRING_vDestroy(sAttrs.hState);
        STRING_vDestroy(sAttrs.hZipCode);
        STRING_vDestroy(sAttrs.hPhone);

        return LOCATION_INVALID_OBJECT;
    }

    return (LOCATION_OBJECT)psObj;
}

/*****************************************************************************
*
*   bCircleInitializeObject
*
*****************************************************************************/
static BOOLEAN bCircleInitializeObject (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_POSITION_TYPE_ENUM ePosType,
    BOOLEAN bPerformDeepCopy,
    BOOLEAN bIsLocID
        )
{
    LOCATION_OBJECT_STRUCT *psObj = (LOCATION_OBJECT_STRUCT *)hLocation;
    OSAL_FIXED_OBJECT hFixedRadius;
    N32 n32Radius;

    // Initialize interface
    psObj->psIntf = &GsLocationCircleIntf;

    // Initialize object per inputs
    if (n32Lat != 0)
    {
        psObj->hLat =
            OSAL_FIXED.hCreateInMemory(n32Lat, un8LatBits,
                &psObj->atLatData[0]);
    }

    if (n32Lon != 0)
    {
        psObj->hLon =
            OSAL_FIXED.hCreateInMemory(n32Lon, un8LonBits,
                &psObj->atLonData[0]);
    }

    if (bPerformDeepCopy == FALSE)
    {
        psObj->uProperties.sCircle = puProperties->sCircle;
    }
    else
    {
        // DISTANCE.hDuplicate returns DISTANCE_INVALID_OBJECT for bad input
        // provided, so do not bother for extra checking
        psObj->uProperties.sCircle.hRadius =
            DISTANCE.hDuplicate(puProperties->sCircle.hRadius);
    }

    // If NULL object is provided as radius, we create fake distance with
    // 0 length for easier processing in the future
    if (psObj->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT)
    {
        psObj->uProperties.sCircle.hRadius =
            DISTANCE.hCreate(0, DISTANCE_UNIT_TYPE_MILES);
        if (psObj->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT)
        {
            // Error!
            return FALSE;
        }
        psObj->bIsPoint = TRUE;

        // Check validity of coordinates-based location
        if ((bIsLocID == FALSE) && (ePosType != LOCATION_POSITION_TYPE_DEVICE))
        {
            BOOLEAN bSuccess;

            bSuccess = bCheckLatLonRange(psObj->hLat, psObj->hLon);
            if (FALSE == bSuccess)
            {
                // ERROR!
                // Out of range lat/lon values
                DISTANCE.vDestroy(psObj->uProperties.sCircle.hRadius);
                return FALSE;
            }
        }
    }
    else
    {
        // else we calculate the radius value
        hFixedRadius = DISTANCE.hValue(psObj->uProperties.sCircle.hRadius,
            DISTANCE_UNIT_TYPE_MILES);
        n32Radius = OSAL_FIXED.n32Value(hFixedRadius);
        if (n32Radius == 0)
        {
            // Oops, it is a single point again
            psObj->bIsPoint = TRUE;
        }

        if ((bIsLocID == FALSE) && (ePosType != LOCATION_POSITION_TYPE_DEVICE))
        {
            // Only do range checks if it is not LocID or DEVICE location
            BOOLEAN bSuccess;
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_FIXED_OBJECT hRadiusDeg, hTopLat, hFixed90;
            OSAL_FIXED_OBJECT_DATA atData[OSAL_FIXED_OBJECT_SIZE * 3];
            N16 n16Compare;

            // Create temporary fixed objects
            hRadiusDeg = OSAL_FIXED.hCreateInMemory(0, 0, 
                &atData[OSAL_FIXED_OBJECT_SIZE * 0]);
            hTopLat = OSAL_FIXED.hCreateInMemory(0, 0, 
                &atData[OSAL_FIXED_OBJECT_SIZE * 1]);
            hFixed90 = OSAL_FIXED.hCreateInMemory(90, 0, 
                &atData[OSAL_FIXED_OBJECT_SIZE * 2]);

            if ((OSAL_FIXED_INVALID_OBJECT == hRadiusDeg) ||
                (OSAL_FIXED_INVALID_OBJECT == hTopLat) ||
                (OSAL_FIXED_INVALID_OBJECT == hFixed90))
            {
                // Error! No memory.
                return FALSE;
            }

            bSuccess = DISTANCE_bDistanceOverRadius(
                psObj->uProperties.sCircle.hRadius, 
                hRadiusDeg);
            
            if (FALSE == bSuccess)
            {
                // error!
                return FALSE;
            }

            eReturnCode = OSAL_FIXED.eAdd(psObj->hLat, hRadiusDeg, hTopLat);
            if (OSAL_SUCCESS != eReturnCode)
            {
                // error!
                return FALSE;
            }

            n16Compare = OSAL_FIXED.n16Compare(hTopLat, hFixed90);
            if (n16Compare > 0)
            {
                // Overflow over North Pole
                return FALSE;
            }

            // Now we are sure that there will be no latitude overflow
            // so it is safe to calculate Top-Right coordinates using
            // trigonometry functions
            bSuccess = LOCATION_bTopRight((LOCATION_OBJECT)psObj,
                OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);
            if (FALSE == bSuccess)
            {
                // error!
                return FALSE;
            }

            // Range check the lat/long
            bSuccess = bCheckLatLonRange(psObj->hTopRightLat,
                psObj->hTopRightLon);
            if (FALSE == bSuccess)
            {
                // ERROR!
                // Out of range lat/lon values
                return FALSE;
            }

            bSuccess = LOCATION_bBottomLeft((LOCATION_OBJECT)psObj,
                OSAL_FIXED_INVALID_OBJECT, OSAL_FIXED_INVALID_OBJECT);
            if (FALSE == bSuccess)
            {
                // error!
                return FALSE;
            }

            // Range check the lat/long
            bSuccess = bCheckLatLonRange(psObj->hBotLeftLat,
                psObj->hBotLeftLon);
            if (FALSE == bSuccess)
            {
                // ERROR!
                // Out of range lat/lon values
                return FALSE;
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   vCircleDestroyProperties
*
*****************************************************************************/
static void vCircleDestroyProperties(
    const LOCATION_OBJECT hLocation
    )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT*)hLocation;

    if (psObj->uProperties.sCircle.hRadius != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(psObj->uProperties.sCircle.hRadius);
        psObj->uProperties.sCircle.hRadius = DISTANCE_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bCircleCoordinatesWithinArea
*
*****************************************************************************/
static BOOLEAN bCircleCoordinatesWithinArea (
    const LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    LOCATION_OBJECT_STRUCT *psObj =
        (LOCATION_OBJECT_STRUCT *)hLocation;
    BOOLEAN bCoordsInRadius = FALSE;
    N16 n16Compare;

    // Compare both pairs to see if they are the same
    // We can skip out on performing a calculation if it is the case
    n16Compare = OSAL_FIXED.n16Compare(psObj->hLat, hLat);
    if (n16Compare == 0)
    {
        n16Compare = OSAL_FIXED.n16Compare(psObj->hLon, hLon);
    }

    // If they matched, then its the same point
    // and coordinates are within the "radius"
    if (n16Compare == 0)
    {
        bCoordsInRadius = TRUE;
    }
    else
    {
        // If the lat/lon wasn't equal we can do a check as long
        // as we have a radius to check with
        if (psObj->bIsPoint == FALSE)
        {
            OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
            OSAL_FIXED_OBJECT hResultVector;
            BOOLEAN bOk;

            // Create the result vector fixed object
            hResultVector = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);

            // Compute the distance vector
            bOk = bComputeDistanceVector(psObj,
                hLat, hLon, hResultVector);

            if (bOk == TRUE)
            {
                N16 n16Result;

                // Get the comparison between the
                // distance vector and the radius
                n16Result = DISTANCE_n16CompareToDistanceVector(
                    psObj->uProperties.sCircle.hRadius, hResultVector);

                if (n16Result >= 0)
                {
                    bCoordsInRadius = TRUE;
                }
            }
        }
    }

    return bCoordsInRadius;
}

/*****************************************************************************
*
*   bCircleContains
*
*****************************************************************************/
static BOOLEAN bCircleContains (
    LOCATION_OBJECT hLocationContainer,
    LOCATION_OBJECT hLocationContained,
    DISTANCE_OBJECT hThreshold
        )
{
    BOOLEAN bContains = FALSE;
    LOCATION_OBJECT_STRUCT *psObjContainer =
        (LOCATION_OBJECT_STRUCT *)hLocationContainer;
    LOCATION_OBJECT_STRUCT *psObjContained =
        (LOCATION_OBJECT_STRUCT *)hLocationContained;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    N16 n16Compare = N16_MIN;
    LOCATION_OBJECT hTempLocation = LOCATION_INVALID_OBJECT;
    DISTANCE_OBJECT hDistance = DISTANCE_INVALID_OBJECT;

    do
    {
        if(psObjContained->psIntf->eType == LOCATION_TYPE_RECTANGLE)
        {
            // Checking whenever circle contains rectangle.
            // For rectangle to be contained within circle, all vertexes
            // shall be within the circle.
            OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
            OSAL_FIXED_OBJECT hTempTopLat, hTempBottomLat,
                hTempLeftLon, hTempRightLon;
            N16 n16Diff;
            BOOLEAN bResult;

            // Create temporary FIXED objects
            hTempTopLat = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);
            hTempRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);
            hTempBottomLat = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE * 2]);
            hTempLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE * 3]);

            if ((hTempTopLat == OSAL_FIXED_INVALID_OBJECT) ||
                (hTempRightLon == OSAL_FIXED_INVALID_OBJECT) ||
                (hTempBottomLat == OSAL_FIXED_INVALID_OBJECT) ||
                (hTempLeftLon == OSAL_FIXED_INVALID_OBJECT))
            {
                break;
            }

            // Getting top/bottom/left/right coordinates of rectangle
            bResult = LOCATION_bTopRight(hLocationContained,
                hTempTopLat, hTempRightLon);
            bResult &= LOCATION_bBottomLeft(hLocationContained,
                hTempBottomLat, hTempLeftLon);
            if (bResult == FALSE)
            {
                break;
            }

            // Now check distances from circle's center to every vertex
            // and compare those to circle's radius

            // Top-left corner
            hTempLocation = LOCATION.hCreateForRadius(
                hTempTopLat,
                hTempLeftLon,
                DISTANCE_INVALID_OBJECT);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hDistance = LOCATION.hDistance(hLocationContainer, hTempLocation);
            if (hDistance == DISTANCE_INVALID_OBJECT)
            {
                break;
            }

            n16Diff = DISTANCE.n16Compare(hDistance,
                psObjContainer->uProperties.sCircle.hRadius);

            // Clean up objects for next calculations
            LOCATION.vDestroy(hTempLocation);
            DISTANCE.vDestroy(hDistance);
            hDistance = DISTANCE_INVALID_OBJECT;

            // Store the maximal extent
            n16Compare = n16Diff;

            // Top-right corner
            hTempLocation = LOCATION.hCreateForRadius(
                hTempTopLat,
                hTempRightLon,
                DISTANCE_INVALID_OBJECT);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hDistance = LOCATION.hDistance(hLocationContainer, hTempLocation);
            if (hDistance == DISTANCE_INVALID_OBJECT)
            {
                break;
            }

            n16Diff = DISTANCE.n16Compare(hDistance,
                psObjContainer->uProperties.sCircle.hRadius);

            // Clean up objects for next calculations
            LOCATION.vDestroy(hTempLocation);
            DISTANCE.vDestroy(hDistance);
            hDistance = DISTANCE_INVALID_OBJECT;

            // Store the maximal extent
            n16Compare = (n16Compare < n16Diff) ? n16Diff : n16Compare;

            // Bottom-right corner
            hTempLocation = LOCATION.hCreateForRadius(
                hTempBottomLat,
                hTempRightLon,
                DISTANCE_INVALID_OBJECT);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hDistance = LOCATION.hDistance(hLocationContainer, hTempLocation);
            if (hDistance == DISTANCE_INVALID_OBJECT)
            {
                break;
            }
            n16Diff = DISTANCE.n16Compare(hDistance,
                psObjContainer->uProperties.sCircle.hRadius);

            // Clean up objects for next calculations
            LOCATION.vDestroy(hTempLocation);
            DISTANCE.vDestroy(hDistance);
            hDistance = DISTANCE_INVALID_OBJECT;

            // Store the maximal extent
            n16Compare = (n16Compare < n16Diff) ? n16Diff : n16Compare;

            // Bottom-left corner
            hTempLocation = LOCATION.hCreateForRadius(
                hTempBottomLat,
                hTempLeftLon,
                DISTANCE_INVALID_OBJECT);
            if (hTempLocation == LOCATION_INVALID_OBJECT)
            {
                break;
            }

            hDistance = LOCATION.hDistance(hLocationContainer, hTempLocation);
            if (hDistance == DISTANCE_INVALID_OBJECT)
            {
                break;
            }
            n16Diff = DISTANCE.n16Compare(hDistance,
                psObjContainer->uProperties.sCircle.hRadius);

            // Clean up objects for next calculations
            LOCATION.vDestroy(hTempLocation);
            hTempLocation = LOCATION_INVALID_OBJECT;
            DISTANCE.vDestroy(hDistance);
            hDistance = DISTANCE_INVALID_OBJECT;

            // Store the maximal extent
            n16Compare = (n16Compare < n16Diff) ? n16Diff : n16Compare;
        }
        else if(psObjContained->psIntf->eType == LOCATION_TYPE_CIRCLE)
        {
            // Checking whenever circle contains circle.
            // Containing circle's radius shall be bigger than contained
            // radius plus distance between centers
            OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE];
            OSAL_FIXED_OBJECT hContainerRadius, hContainedRadius,
                hCenterDistanceFixed, hTotalDistance;

            // Get the distance between the two center points
            hDistance = LOCATION.hDistance(
                hLocationContainer,
                hLocationContained);

            if (hDistance == DISTANCE_INVALID_OBJECT)
            {
                // Something went wrong
                break;
            }

            // Get the fixed object for that
            hCenterDistanceFixed = DISTANCE.hValue(
                hDistance, DISTANCE_UNIT_TYPE_MILES);
            if (hCenterDistanceFixed == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }

            // Get radiuses of two locations
            hContainedRadius = DISTANCE.hValue(
                psObjContained->uProperties.sCircle.hRadius,
                DISTANCE_UNIT_TYPE_MILES);
            hContainerRadius = DISTANCE.hValue(
                psObjContainer->uProperties.sCircle.hRadius,
                DISTANCE_UNIT_TYPE_MILES);

            if ((hContainedRadius == OSAL_FIXED_INVALID_OBJECT) ||
                (hContainerRadius == OSAL_FIXED_INVALID_OBJECT))
            {
                break;
            }

            // Create the fixed objects in memory
            hTotalDistance = OSAL_FIXED.hCreateInMemory( 0, 0,
                &atFixedData[0]);
            if (hTotalDistance == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }

            // Add the contained radius to the distance between center points
            eReturnCode = OSAL_FIXED.eAdd(
                hContainedRadius, hCenterDistanceFixed, hTotalDistance);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            // Now, compare distance with container radius
            n16Compare = OSAL_FIXED.n16Compare(hTotalDistance,
                hContainerRadius);
        }
        else
        {
            // Unknown area type. Error!
            break;
        }

        // Was a threshold provided?
        if (hThreshold != DISTANCE_INVALID_OBJECT)
        {
            OSAL_FIXED_OBJECT hThresholdFixed;
            N32 n32Threshold;

            // Yes - extract it
            hThresholdFixed = DISTANCE.hValue(
                hThreshold, DISTANCE_UNIT_TYPE_MILES);
            n32Threshold = OSAL_FIXED.n32Value(hThresholdFixed);

            // Adjust by previously calculated difference by
            // the threshold provided
            n16Compare -= (N16)n32Threshold;
        }

        if (n16Compare <= 0)
        {
            bContains = TRUE;
        }
    } while (FALSE);

    // Clean up any mess left behind
    if (hDistance != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(hDistance);
    }
    if (hTempLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(hTempLocation);
    }

    return bContains;
}

/*****************************************************************************
*
*   bCircleCompare
*
*****************************************************************************/
static BOOLEAN bCircleCompare (
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        )
{
    LOCATION_OBJECT_STRUCT *psObj1 = (LOCATION_OBJECT_STRUCT *)hLocation1;
    LOCATION_OBJECT_STRUCT *psObj2 = (LOCATION_OBJECT_STRUCT *)hLocation2;
    N16 n16Compare = N16_MIN;

    if (psObj1->uProperties.sCircle.hRadius ==
        psObj2->uProperties.sCircle.hRadius)
    {
        n16Compare = 0;
    }
    else
    {
        if ((psObj1->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT) ||
            (psObj2->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT))
        {
            return FALSE;
        }

        // Perform the comparison of the distance objects
        n16Compare = DISTANCE.n16Compare(
            psObj1->uProperties.sCircle.hRadius,
            psObj2->uProperties.sCircle.hRadius);
    }

    if (n16Compare != 0)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bCircleUpdateCoordinates
*
*****************************************************************************/
static BOOLEAN bCircleUpdateCoordinates (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits
        )
{
    vUpdateCoordinatesCommonPart(hLocation,
        n32Lat, un8LatBits, n32Lon, un8LonBits);

    return TRUE;
}

/*****************************************************************************
*
*   n32CircleFWrite
*
*****************************************************************************/
static N32 n32CircleFWrite (
    const LOCATION_OBJECT hLocation,
    FILE *psFile
        )
{
    LOCATION_OBJECT_STRUCT *psObj = (LOCATION_OBJECT_STRUCT *)hLocation;
    N32 n32Return = 0, n32Value;
    UN8 un8ByteToWrite, un8Bits;

    // Write LOCATION to file...

    // Write location area type
    un8ByteToWrite = (UN8)LOCATION_TYPE_CIRCLE;
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);

    //Write Radius
    if (psObj->uProperties.sCircle.hRadius == DISTANCE_INVALID_OBJECT)
    {
        // Radius is NOT present, so indicate it with the flag
        un8ByteToWrite = FALSE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    }
    else
    {
        // Radius is present, so indicate it with the flag
        un8ByteToWrite = TRUE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
        n32Return += DISTANCE.n32FWrite(psObj->uProperties.sCircle.hRadius, psFile);
    }

    // Write Latitude
    n32Value = OSAL_FIXED.n32Value(psObj->hLat);
    un8Bits = OSAL_FIXED.un8NumFractionalBits(psObj->hLat);

    un8ByteToWrite = BYTE0(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    n32Return += (N32)fwrite(&un8Bits, sizeof(UN8), 1, psFile);

    // Write Longitude
    n32Value = OSAL_FIXED.n32Value(psObj->hLon);
    un8Bits = OSAL_FIXED.un8NumFractionalBits(psObj->hLon);

    un8ByteToWrite = BYTE0(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE1(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE2(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    un8ByteToWrite = BYTE3(n32Value);
    n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    n32Return += (N32)fwrite(&un8Bits, sizeof(UN8), 1, psFile);

    // Now writing remaining common part
    n32Return += n32WriteCommonPart(psObj, psFile);

    return n32Return;
}

/*****************************************************************************
*
*   hCircleFRead
*
*****************************************************************************/
static LOCATION_OBJECT hCircleFRead (
    FILE *psFile
        )
{
    LOCATION_OBJECT_STRUCT *psObj;
    N32 n32Lat = 0, n32Lon = 0;
    UN8 un8LatBits = 0, un8LonBits = 0, un8CurByteRead = 0;
    size_t tTotal;
    LOCATION_ATTRIBUTE_STRUCT sAttrs;
    LOCID_OBJECT hLocID = LOCID_INVALID_OBJECT;
    DISTANCE_OBJECT hRadius = DISTANCE_INVALID_OBJECT;
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;

    // Verify input
    if(psFile == NULL)
    {
        return LOCATION_INVALID_OBJECT;
    }

    // Read LOCATION from file...

    // Read radius present flag
    tTotal = fread(&un8CurByteRead, 1, 1, psFile);
    if(tTotal != sizeof(UN8))
    {
        // End of file or error occurred
        return LOCATION_INVALID_OBJECT;
    }

    if (un8CurByteRead == TRUE)
    {
        // Read radius
        hRadius = DISTANCE.hFRead(psFile);
    }

    vFReadCommonPart(psFile, &n32Lat, &n32Lon, &un8LatBits, &un8LonBits, &sAttrs, &hLocID);

    // Populating area-specific properties
    uProperties.sCircle.hRadius = hRadius;

    // Create an instance of the LOCATION object
    psObj = psCreateLocation(
        SMS_INVALID_OBJECT, hLocID,
        n32Lat, un8LatBits,
        n32Lon, un8LonBits,
        &uProperties, &sAttrs, FALSE,
        LOCATION_TYPE_CIRCLE);

    if(psObj == NULL)
    {
        // Error!
        DISTANCE.vDestroy(hRadius);
        LOCID.vDestroy(hLocID);
        STRING_vDestroy(sAttrs.hStreetNum);
        STRING_vDestroy(sAttrs.hStreetName);
        STRING_vDestroy(sAttrs.hCity);
        STRING_vDestroy(sAttrs.hState);
        STRING_vDestroy(sAttrs.hZipCode);
        STRING_vDestroy(sAttrs.hPhone);

        return LOCATION_INVALID_OBJECT;
    }

    return (LOCATION_OBJECT)psObj;
}

/*****************************************************************************
*
*   bCheckLatLonRange
*
*****************************************************************************/
static BOOLEAN bCheckLatLonRange (
    OSAL_FIXED_OBJECT  hLat,
    OSAL_FIXED_OBJECT  hLon
        )
{
    BOOLEAN bReturn = FALSE;
    N16 n16Compare;

    if (GsLocationCoordLimits.bIsMinMaxCoordinatesInitialized == FALSE)
    {
        vCreateFixedConstants();
    }

    do 
    {
        // Are we less than the min latitude?
        n16Compare = OSAL_FIXED.n16Compare(hLat, 
            GsLocationCoordLimits.hMinLat);
        if (n16Compare < 0)
        {
            break;
        }

        // Are we greater than the max latitude?
        n16Compare = OSAL_FIXED.n16Compare(hLat, 
            GsLocationCoordLimits.hMaxLat);
        if (n16Compare > 0)
        {
            break;
        }

        // Are we less than the min longitude?
        n16Compare = OSAL_FIXED.n16Compare(hLon, 
            GsLocationCoordLimits.hMinLon);
        if (n16Compare < 0)
        {
            break;
        }

        // Are we greater than the max longitude?
        n16Compare = OSAL_FIXED.n16Compare(hLon, 
            GsLocationCoordLimits.hMaxLon);
        if (n16Compare > 0)
        {
            break;
        }     

        // Looks like we are good
        bReturn = TRUE;

    } while (FALSE);

    return bReturn;
}

/*****************************************************************************
*
*   bComputeDistanceVector
*
*  This method uses an "Equirectangular Approximation" to calculate the
*  distance between the center point defined by a LOCATION_OBJECT and a
*  second lat/lon pairing.
*
*  The equirectangular approximation is defined as
*  x = (lon2 - lon1) cos ( (lat2+lat1)/2) )
*  y = (lat2 - lat1)
*  distance = EarthRadius * sqrt(x*x + y*y)
*
*  This function does not multiply by the EarthRadius as it is returning
*  a distance vector.  A distance vector in this object is defined as
*  the distance/EarthRadius.
*
* Inputs:
*
*   psObj - A pointer to a LOCATION_OBJECT_STRUCT containing the center-point
*       details.
*   hLonTo - A valid FIXED object handle indicating the foreign latitude.
*   hLonTo - A valid FIXED object handle indicating the foreign longitude.
*   hResultantVector - A valid FIXED object handle used to store the
*       calculated distance vector.
*
* Outputs:
*
*   TRUE if vector correctly calculated, FALSE otherwise
*
*****************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_SMSLIB_CUSTOM_DISTANCE
static BOOLEAN bComputeDistanceVector(
    LOCATION_OBJECT_STRUCT *psObj,
    OSAL_FIXED_OBJECT hLatTo,
    OSAL_FIXED_OBJECT hLonTo,
    OSAL_FIXED_OBJECT hResultantVector
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_FIXED_OBJECT hLatRadianTo = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLonRadianTo = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hDeltaLat = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hDeltaLon = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hAverageLat = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hTwo = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hX = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hY = OSAL_FIXED_INVALID_OBJECT;
    BOOLEAN bReturn = FALSE;

    // Create an array to hold our 10 fixed object structures
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 10];
    UN8 un8NumFixed = 0;    // Keep track of the fixed objects
    do
    {
        // Convert the locations to radians
        if (psObj->sPrecalcFixedValues.bIsLatLonInitialized == FALSE)
        {
            bReturn = bPrecalculateLatLonRadians(psObj);
            if (bReturn != TRUE)
            {
                break;
            }
        }

        hLatRadianTo =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode =
            OSAL_FIXED.eDegreeToRadian(hLatTo, hLatRadianTo);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLonRadianTo =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode =
            OSAL_FIXED.eDegreeToRadian(hLonTo, hLonRadianTo);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hDeltaLat=
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode = OSAL_FIXED.eSubtract(hLatRadianTo,
                                           psObj->sPrecalcFixedValues.hLatInRadian,
                                           hDeltaLat);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hDeltaLon =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode = OSAL_FIXED.eSubtract(hLonRadianTo,
                                           psObj->sPrecalcFixedValues.hLonInRadian,
                                           hDeltaLon);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hAverageLat =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode = OSAL_FIXED.eAdd(psObj->sPrecalcFixedValues.hLatInRadian,
                                      hLatRadianTo,
                                      hAverageLat);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hTwo = OSAL_FIXED.hCreateInMemory(2, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode = OSAL_FIXED.eDivide(hAverageLat,
                                         hTwo,
                                         hAverageLat);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hX =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode = OSAL_FIXED.eCos(hAverageLat, hX);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eMultiply(hDeltaLon, hX, hX);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eMultiply(hX, hX, hResultantVector);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hY =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode = OSAL_FIXED.eMultiply(hDeltaLat, hDeltaLat, hY);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eAdd(hResultantVector,
                                      hY,
                                      hResultantVector);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        eReturnCode = OSAL_FIXED.eSqrt(hResultantVector,
                                       hResultantVector);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // We were able to compute the vector
        bReturn = TRUE;

    } while (0);

    return bReturn;
}
#endif

/*****************************************************************************
*
*   bCalcNewDestCoordinatesFromLocation
*
* This function that can calculate a destination point given a starting point,
* a distance and a bearing.
*
* lat2: =ASIN(SIN(lat1)*COS(d/R) + COS(lat1)*SIN(d/R)*COS(brng))
* lon2: =lon1 + ATAN2(SIN(brng)*SIN(d/R)*COS(lat1), COS(d/R)-SIN(lat1)*SIN(lat2))
*
* Inputs:
*
*   psObj - A pointer to a LOCATION_OBJECT_STRUCT containing the center-point
*       details (lat1, lat2)
*   hDistance - The distance (d) to travel.  The units for hDistance would be
*       evaluated to determine the correct R to use.
*   hBrearing - The direction (brng) to travel in as a FIXED object in Degrees
*   hLonTo - A valid FIXED object handle indicating the foreign latitude (lat2).
*   hLonTo - A valid FIXED object handle indicating the foreign longitude (lon2).
*
* Outputs:
*
*   TRUE if vector correctly calculated, FALSE otherwise
*
*****************************************************************************/
static BOOLEAN bCalcDestCoordinatesFromLocation(
    LOCATION_OBJECT_STRUCT *psObj,
    DISTANCE_OBJECT hDistance,
    OSAL_FIXED_OBJECT hBearingInDeg,
    OSAL_FIXED_OBJECT hLatTo,
    OSAL_FIXED_OBJECT hLonTo
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    OSAL_FIXED_OBJECT hBearingInRad = OSAL_FIXED_INVALID_OBJECT;

    // Variables used by both lat and lon equations
    OSAL_FIXED_OBJECT hDOverR = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hCosDOverR = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hSinDOverR = OSAL_FIXED_INVALID_OBJECT;

    // Variables used by lat equation
    // lat2: =ASIN(SIN(lat1)*COS(d/R) + COS(lat1)*SIN(d/R)*COS(brng)
    //             [    hLatPartA   ]   [   hLatPartB1   ]
    //                                  [        hLatPartB         ]
    //             [            hLatSumOfAB                        ]
    OSAL_FIXED_OBJECT hCosBrng = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLatPartA = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLatPartB = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLatPartB1 = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLatSumOfAB = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLatToAsRadians = OSAL_FIXED_INVALID_OBJECT;

    // Variables used by lon equation
    // lon2: =lon1 + ATAN2(SIN(brng)*SIN(d/R)*COS(lat1), COS(d/R)-SIN(lat1)*SIN(lat2))
    //                     [   hLonPartA1    ]                    [   hLonPartB1    ]
    //                     [     hLonPartA            ]  [     hLonPartB            ]
    //               [                        hLonATan2Result                        ]

    OSAL_FIXED_OBJECT hSinLatTo = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLonPartA1 = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLonPartA = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hSinBrng = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLonPartB1 = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLonPartB = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLonATan2Result = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLonToAsRadians = OSAL_FIXED_INVALID_OBJECT;

    // Create an array to hold our 22 fixed object structures
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 22];
    UN8 un8NumFixed = 0;    // Keep track of the fixed objects
    BOOLEAN bReturn = FALSE;

    do
    {
        BOOLEAN bSuccess;

        if (hLatTo == OSAL_FIXED_INVALID_OBJECT ||
            hLonTo == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Convert our bearing to radians
        hBearingInRad =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        eReturnCode = OSAL_FIXED.eDegreeToRadian(hBearingInDeg, hBearingInRad);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Calculate D over R
        hDOverR =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        bSuccess = DISTANCE_bDistanceOverRadius(hDistance, hDOverR);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Convert the locations to angles
        if (psObj->sPrecalcFixedValues.bIsLatLonInitialized == FALSE)
        {
            bReturn = bPrecalculateLatLonRadians(psObj);
            if (bReturn != TRUE)
            {
                break;
            }
        }
        
        // Get the components that will be used by both equations in this function
        if (psObj->sPrecalcFixedValues.bIsSinCosLatInitialized == FALSE)
        {
            bReturn = bPrecalculateLatSinCos(psObj);
            if (bReturn != TRUE)
            {
                break;
            }
        }
        

        hCosDOverR =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eCos(hDOverR, hCosDOverR);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hSinDOverR =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eSin(hDOverR, hSinDOverR);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Perform the destination lattitude calculation
        // lat2: =ASIN(SIN(lat1)*COS(d/R) + COS(lat1)*SIN(d/R)*COS(brng)
        //             [    hLatPartA   ]   [   hLatPartB1   ]
        //                                  [        hLatPartB         ]
        //             [            hLatSumOfAB                        ]


        hLatPartA =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eMultiply(psObj->sPrecalcFixedValues.hSinLat,
                                           hCosDOverR, hLatPartA);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLatPartB1 =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eMultiply(psObj->sPrecalcFixedValues.hCosLat,
                                           hSinDOverR, hLatPartB1);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hCosBrng =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eCos(hBearingInRad, hCosBrng);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLatPartB =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eMultiply(hLatPartB1, hCosBrng, hLatPartB);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }


        hLatSumOfAB =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eAdd(hLatPartA, hLatPartB, hLatSumOfAB);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLatToAsRadians =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eASin(hLatSumOfAB, hLatToAsRadians);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Perform the destination longitude calculation
        // lon2: =lon1 + ATAN2(SIN(brng)*SIN(d/R)*COS(lat1), COS(d/R)-SIN(lat1)*SIN(lat2))
        //                     [   hLonPartA1    ]                    [   hLonPartB1    ]
        //                     [     hLonPartA            ]  [     hLonPartB            ]
        //               [                        hLonATan2Result                        ]

        hSinLatTo =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eSin(hLatToAsRadians,
                                      hSinLatTo);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLonPartB1 =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eMultiply(psObj->sPrecalcFixedValues.hSinLat,
                                      hSinLatTo,
                                      hLonPartB1);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLonPartB =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eSubtract(hCosDOverR,
                                      hLonPartB1,
                                      hLonPartB);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hSinBrng =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eSin(hBearingInRad, hSinBrng);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLonPartA1 =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eMultiply(hSinBrng,
                                      hSinDOverR,
                                      hLonPartA1);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLonPartA =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eMultiply(hLonPartA1,
                                      psObj->sPrecalcFixedValues.hCosLat,
                                      hLonPartA);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLonATan2Result =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eATan2(hLonPartA,
                                   hLonPartB,
                                   hLonATan2Result);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        hLonToAsRadians =
            OSAL_FIXED.hCreateInMemory(0, 0,
                &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        eReturnCode = OSAL_FIXED.eAdd(psObj->sPrecalcFixedValues.hLonInRadian,
                                 hLonATan2Result,
                                 hLonToAsRadians);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Convert the destination lattitude from radians to degrees
        eReturnCode = OSAL_FIXED.eRadianToDegree (hLatToAsRadians, hLatTo);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Convert the destination longitude from radians to degrees
        eReturnCode = OSAL_FIXED.eRadianToDegree (hLonToAsRadians, hLonTo);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // We were able to compute the new coordinates
        return TRUE;

    } while (0);

    return FALSE;
}

/*****************************************************************************
*
*   vUpdateDescriptiveAttribute
*
*  This method updates a STRING_OBJECT handle pointer (and its contents)
*  with the STRING_OBJECT found with the handle hNewField.  If we are to
*  just perform a blind overwrite with no care as to the contents of
*  hNewField, just go ahead.  Otherwise, only overwrite a field with valid
*  contents that are different from what is currently found in *phOrigField.
*
*  Perform a deep copy only if specified.
*
*****************************************************************************/
static void vUpdateDescriptiveAttribute (
    STRING_OBJECT *phOrigField,
    STRING_OBJECT hNewField,
    BOOLEAN bBlindOverwrite,
    BOOLEAN bPerformDeepCopy
        )
{
    // If we are not doing a blind overwrite, then
    // update this field if it has been provided and
    // the string has changed
    if ((bBlindOverwrite == FALSE) &&
        (hNewField != STRING_INVALID_OBJECT))
    {
        N16 n16Compare;

        // Perform a binary compare
        n16Compare = STRING.n16Compare(
            *phOrigField, hNewField, TRUE);

        // If this field has changed, we'll update it
        if (n16Compare != 0)
        {
            bBlindOverwrite = TRUE;
        }
    }

    // Just overwrite the original field
    // with whatever is in the new field
    if (bBlindOverwrite == TRUE)
    {
        // Destroy the old field if is contains
        // valid STRING_OBJECT data
        if (*phOrigField != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(*phOrigField);
        }

        // Copy the new data
        if (bPerformDeepCopy == FALSE)
        {
            *phOrigField = hNewField;
        }
        else
        {
            *phOrigField = STRING.hDuplicate(hNewField);
        }
    }

    return;
}

/*****************************************************************************
*
*   bFindLocationTag
*
*****************************************************************************/
static BOOLEAN bFindLocationTag (
    TAG_OBJECT hTag,
    LOCATION_TAG_SEARCH_STRUCT *psSearch
        )
{
    STRING_OBJECT hTagName;

    // get this tag's name
    hTagName = TAG_hTagName(hTag);

    // Make sure this is a location tag
    if (STRING.n16CompareCStr(LOCATION_OBJECT_NAME, 0, hTagName) == 0)
    {
        size_t tSize;
        LOC_ID tLocId = 0, *ptLocId;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Get the tag's value
        ptLocId = &tLocId;
        tSize = sizeof(LOC_ID);
        eReturnCode = TAG_eGetTagValue(hTag, TAG_TYPE_INTEGER,
            (void **)&ptLocId, &tSize);
        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (tLocId == psSearch->tID))
        {
            psSearch->hTag = hTag;

            // Stop here
            return FALSE;
        }
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   n32ComputeCode
*
*****************************************************************************/
static N32 n32ComputeCode (
        OSAL_FIXED_OBJECT hCLineX,
        OSAL_FIXED_OBJECT hCLineY,
        OSAL_FIXED_OBJECT hBotLeftX,
        OSAL_FIXED_OBJECT hBotLeftY,
        OSAL_FIXED_OBJECT hTopRightX,
        OSAL_FIXED_OBJECT hTopRightY
        )
{
    N32 n32Code = 0;
    N16 n16CompareValue = 0;

    n16CompareValue = OSAL_FIXED.n16Compare(hCLineX, hBotLeftX);
    if ( n16CompareValue < 0)
    {
        n32Code++;
    }
    n16CompareValue = OSAL_FIXED.n16Compare(hCLineX, hTopRightX);
    if ( n16CompareValue > 0)
    {
        n32Code += 2;
    }
    n16CompareValue = OSAL_FIXED.n16Compare(hCLineY, hBotLeftY);
    if ( n16CompareValue < 0)
    {
        n32Code += 4;
    }
    n16CompareValue = OSAL_FIXED.n16Compare(hCLineY, hTopRightY);
    if ( n16CompareValue > 0)
    {
        n32Code += 8;
    }

    return n32Code;
}

/*****************************************************************************
 *
 *   bPointInsideRectangle
 *
 *****************************************************************************/
static BOOLEAN bPointInsideRectangle (
    LOCATION_OBJECT_STRUCT *psObj,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    N16 n16CompareLat1, n16CompareLat2, n16CompareLon1, n16CompareLon2;
    do
    {
        if ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Bottom Left
        n16CompareLat1 = OSAL_FIXED.n16Compare(hLat, psObj->hBotLeftLat);
        n16CompareLon1 = OSAL_FIXED.n16Compare(hLon, psObj->hBotLeftLon);

        // Top Right
        n16CompareLat2 = OSAL_FIXED.n16Compare(hLat, psObj->hTopRightLat);
        n16CompareLon2 = OSAL_FIXED.n16Compare(hLon, psObj->hTopRightLon);

        if ( (n16CompareLat1 >= 0) && (n16CompareLat2 <= 0) &&
             (n16CompareLon1 >= 0) && (n16CompareLon2 <= 0) )
        {
            return TRUE;
        }

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
 *
 *   vLocationInitCommonPart
 *
 *****************************************************************************/
static void vLocationInitCommonPart (
    LOCATION_OBJECT_STRUCT *psObj,
    BOOLEAN bPerformDeepCopy,
    LOCID_OBJECT hLocID,
    LOCATION_ATTRIBUTE_STRUCT *psDescriptiveAttrs
        )
{
    if (bPerformDeepCopy == FALSE)
    {
        // Copy the locid and the radius handle values
        psObj->hLocID = hLocID;

        // Copy the attribute values if they were given
        if (psDescriptiveAttrs != NULL)
        {
            psObj->sDescriptiveAttribs = *psDescriptiveAttrs;
        }
    }
    else
    {
        // Perform a deep copy of all the attributes
        if (hLocID != LOCID_INVALID_OBJECT)
        {
            psObj->hLocID = LOCID.hDuplicate(hLocID);
        }

        if (psDescriptiveAttrs != NULL)
        {
            if (psDescriptiveAttrs->hDescription != STRING_INVALID_OBJECT)
            {
                psObj->sDescriptiveAttribs.hDescription =
                    STRING.hDuplicate(psDescriptiveAttrs->hDescription);
            }

            if (psDescriptiveAttrs->hCity != STRING_INVALID_OBJECT)
            {
                psObj->sDescriptiveAttribs.hCity =
                    STRING.hDuplicate(psDescriptiveAttrs->hCity);
            }

            if (psDescriptiveAttrs->hPhone != STRING_INVALID_OBJECT)
            {
                psObj->sDescriptiveAttribs.hPhone =
                    STRING.hDuplicate(psDescriptiveAttrs->hPhone);
            }

            if (psDescriptiveAttrs->hState != STRING_INVALID_OBJECT)
            {
                psObj->sDescriptiveAttribs.hState =
                    STRING.hDuplicate(psDescriptiveAttrs->hState);
            }

            if (psDescriptiveAttrs->hStreetName != STRING_INVALID_OBJECT)
            {
                psObj->sDescriptiveAttribs.hStreetName =
                    STRING.hDuplicate(psDescriptiveAttrs->hStreetName);
            }

            if (psDescriptiveAttrs->hStreetNum != STRING_INVALID_OBJECT)
            {
                psObj->sDescriptiveAttribs.hStreetNum =
                    STRING.hDuplicate(psDescriptiveAttrs->hStreetNum);
            }

            if (psDescriptiveAttrs->hZipCode != STRING_INVALID_OBJECT)
            {
                psObj->sDescriptiveAttribs.hZipCode =
                    STRING.hDuplicate(psDescriptiveAttrs->hZipCode);
            }
        }
    }

    
    psObj->sPrecalcFixedValues.hLatInRadian = OSAL_FIXED_INVALID_OBJECT;
    psObj->sPrecalcFixedValues.hLonInRadian = OSAL_FIXED_INVALID_OBJECT;

    psObj->sPrecalcFixedValues.bIsLatLonInitialized = FALSE;

    psObj->sPrecalcFixedValues.hSinLat = OSAL_FIXED_INVALID_OBJECT;
    psObj->sPrecalcFixedValues.hCosLat = OSAL_FIXED_INVALID_OBJECT;

    psObj->sPrecalcFixedValues.bIsSinCosLatInitialized = FALSE;

    return;
}

/*****************************************************************************
 *
 *   n32WriteCommonPart
 *
 *****************************************************************************/
static N32 n32WriteCommonPart (
    const LOCATION_OBJECT_STRUCT *psObject,
    FILE *psFile
        )
{
    N32 n32Return = 0, n32Temp;
    UN8 un8ByteToWrite;
    const char cChar = '\0';

    // Write Description
    n32Temp = STRING.n32FWrite(psObject->sDescriptiveAttribs.hDescription, psFile);
    if(n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    // Add null terminator
    n32Return += (N32)fwrite(&cChar, sizeof(cChar), 1, psFile);

    // Write Street Number
    n32Temp = STRING.n32FWrite(psObject->sDescriptiveAttribs.hStreetNum, psFile);
    if(n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    // Add null terminator
    n32Return += (N32)fwrite(&cChar, sizeof(cChar), 1, psFile);

    // Write Street Name
    n32Temp = STRING.n32FWrite(psObject->sDescriptiveAttribs.hStreetName, psFile);
    if(n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    // Add null terminator
    n32Return += (N32)fwrite(&cChar, sizeof(cChar), 1, psFile);

    // Write City
    n32Temp = STRING.n32FWrite(psObject->sDescriptiveAttribs.hCity, psFile);
    if(n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    // Add null terminator
    n32Return += (N32)fwrite(&cChar, sizeof(cChar), 1, psFile);

    // Write State
    n32Temp = STRING.n32FWrite(psObject->sDescriptiveAttribs.hState, psFile);
    if(n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    // Add null terminator
    n32Return += (N32)fwrite(&cChar, sizeof(cChar), 1, psFile);

    // Write Zip
    n32Temp = STRING.n32FWrite(psObject->sDescriptiveAttribs.hZipCode, psFile);
    if(n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    // Add null terminator
    n32Return += (N32)fwrite(&cChar, sizeof(cChar), 1, psFile);

    // Write Phone
    n32Temp = STRING.n32FWrite(psObject->sDescriptiveAttribs.hPhone, psFile);
    if(n32Temp > 0)
    {
        n32Return += n32Temp;
    }

    // Add null terminator
    n32Return += (N32)fwrite(&cChar, sizeof(cChar), 1, psFile);

    // Write LOCID
    if (psObject->hLocID == LOCID_INVALID_OBJECT)
    {
        // LocID is NOT present, so indicate it with the flag
        un8ByteToWrite = FALSE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
    }
    else
    {
        // LocID is present, so indicate it with the flag
        un8ByteToWrite = TRUE;
        n32Return += (N32)fwrite(&un8ByteToWrite, sizeof(UN8), 1, psFile);
        n32Return += LOCID.n32FWrite(psObject->hLocID, psFile);
    }

    return n32Return;
}

/*****************************************************************************
 *
 *   vFReadCommonPart
 *
 *****************************************************************************/
static void vFReadCommonPart (
    FILE *psFile,
    N32 *n32Lat,
    N32 *n32Lon,
    UN8 *un8LatBits,
    UN8 *un8LonBits,
    LOCATION_ATTRIBUTE_STRUCT *sAttrs,
    LOCID_OBJECT *hLocID
        )
{
    UN8 un8CurByteRead = 0;
    size_t tTotal, tLen;

    OSAL.bMemSet(sAttrs, 0, sizeof(LOCATION_ATTRIBUTE_STRUCT));

    // Read tLat
    tTotal = 0;
    do
    {
        tLen =
            fread(((UN8*)n32Lat) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(N32)));

    if(tTotal != sizeof(N32))
    {
        // End of file or error occurred
        return;
    }

    tTotal = fread(un8LatBits, 1, 1, psFile);

    if(tTotal != 1)
    {
        // End of file or error occurred
        return;
    }

    // Read tLon
    tTotal = 0;
    do
    {
        tLen =
            fread(((UN8*)n32Lon) + tTotal, 1, 1, psFile);
        tTotal += tLen;
    } while((tLen != 0) && (tTotal <  sizeof(N32)));

    if(tTotal != sizeof(N32))
    {
        // End of file or error occurred
        return;
    }

    tTotal = fread(un8LonBits, 1, 1, psFile);

    if(tTotal != 1)
    {
        // End of file or error occurred
        return;
    }

    // read in Description
    sAttrs->hDescription = STRING.hFRead(psFile);

    // read in Street Number
    sAttrs->hStreetNum = STRING.hFRead(psFile);

    // read in Street Name
    sAttrs->hStreetName = STRING.hFRead(psFile);

    // read in City
    sAttrs->hCity = STRING.hFRead(psFile);

    // read in State
    sAttrs->hState = STRING.hFRead(psFile);

    // read in Zip Code
    sAttrs->hZipCode = STRING.hFRead(psFile);

    // read in Phone number
    sAttrs->hPhone = STRING.hFRead(psFile);

    // Read locid present flag
    tTotal = fread(&un8CurByteRead, 1, 1, psFile);
    if(tTotal != 1)
    {
        // End of file or error occurred
        return;
    }

    if (un8CurByteRead == TRUE)
    {
        // read in LocID
        *hLocID = LOCID.hFRead(psFile);
    }

    return;
}

/*****************************************************************************
 *
 *   bCompare
 *
 *****************************************************************************/
static BOOLEAN bCompare (
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        )
{
    N16 n16Compare = N16_MIN;
    LOCATION_OBJECT_STRUCT *psObject1 = (LOCATION_OBJECT_STRUCT*)hLocation1;
    LOCATION_OBJECT_STRUCT *psObject2 = (LOCATION_OBJECT_STRUCT*)hLocation2;

    // If locations have different types, we do not compare them
    if(psObject1->psIntf != psObject2->psIntf)
    {
        return FALSE;
    }

    // Location objects are equal if they have the same
    // lat / lon information and if their underlying
    // LOCID_OBJECTs compare as equal as well
    if (psObject1->hLat == psObject2->hLat)
    {
        if (psObject1->hLon == psObject2->hLon)
        {
            n16Compare = 0;
        }
    }
    else
    {
        if ((psObject1->hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psObject2->hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psObject1->hLon == OSAL_FIXED_INVALID_OBJECT) ||
            (psObject2->hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            return FALSE;
        }

        n16Compare = OSAL_FIXED.n16Compare(psObject1->hLat, psObject2->hLat);
        if (n16Compare == 0)
        {
            n16Compare = OSAL_FIXED.n16Compare(psObject1->hLon, psObject2->hLon);
        }
    }

    if (n16Compare != 0)
    {
        return FALSE;
    }

    // Compare the LOCID objects if either one is valid
    if (psObject1->hLocID == psObject2->hLocID)
    {
        n16Compare = 0;
    }
    else
    {
        if ((psObject1->hLocID == LOCID_INVALID_OBJECT) ||
        (psObject2->hLocID == LOCID_INVALID_OBJECT))
        {
            return FALSE;
        }

        // Perform the comparison of the Loc ID objects
        n16Compare = LOCID.n16Compare(
            psObject1->hLocID, psObject2->hLocID);
    }
    if (n16Compare != 0)
    {
        return FALSE;
    }

    // Both locations have the same type (we already checked that)
    // so we can call interface method of either one.
    return psObject1->psIntf->bCompare(hLocation1, hLocation2);

}

/*****************************************************************************
 *
 *   vUpdateCoordinatesCommonPart
 *
 *****************************************************************************/
static void vUpdateCoordinatesCommonPart (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits
        )
{
    LOCATION_OBJECT_STRUCT *psObj = (LOCATION_OBJECT_STRUCT *)hLocation;

    // Clear left/right square and pre-calculated attributes
    psObj->hBotLeftLat = OSAL_FIXED_INVALID_OBJECT;
    psObj->hBotLeftLon = OSAL_FIXED_INVALID_OBJECT;
    psObj->hTopRightLat = OSAL_FIXED_INVALID_OBJECT;
    psObj->hTopRightLon = OSAL_FIXED_INVALID_OBJECT;

    psObj->sPrecalcFixedValues.bIsLatLonInitialized = FALSE;
    psObj->sPrecalcFixedValues.bIsSinCosLatInitialized = FALSE;

    // Create and populate lat/lon attributes
    psObj->hLat = OSAL_FIXED.hCreateInMemory(n32Lat, un8LatBits,
        &psObj->atLatData[0]);

    psObj->hLon = OSAL_FIXED.hCreateInMemory(n32Lon, un8LonBits,
        &psObj->atLonData[0]);

    return;
}

/*****************************************************************************
*
*   bCalculateHalfwayShift
*
*  This helper function is used to calculate coordinates out of base point
*  shifted by 1/2 of width to left/right and 1/2 of height to top/bottom.
*
*  The use case is to reuse this for calculating top-right and bottom-left
*  coordinates of base rectangle, given center, as well as calculate center
*  provided bottom-left corner (during rectangular area creation).
*
*  Inputs:
*       {hBaseLat, hBaseLon} - coordinates of the base point to go from
*       hWidth  - width of the area (delta longitude). We will use half of
*                 it to jump
*       hHeight - height of the area (delta latitude). We will use half of
*                 it to jump
*       n32FirstBearing - vertical bearing. Shall be either 0 (up/north) or
*                 180 (down/south)
*       n32SecondBearing - horizontal bearing. Shall be either 90 (left/east)
*                 or 270 (right/west)
*       {hTargetLat, hTargetLon} - calculated target coordinates will be placed
*                 there
*  Return:
*       TRUE on success, with {hTargetLat, hTargetLon} populated with calculated
*       coordinates,
*       FALSE on failure. In this case {hTargetLat, hTargetLon} are not affected
*
*****************************************************************************/
static BOOLEAN bCalculateHalfwayShift (
    OSAL_FIXED_OBJECT hBaseLat,
    OSAL_FIXED_OBJECT hBaseLon,
    DISTANCE_OBJECT hWidth,
    DISTANCE_OBJECT hHeight,
    N32 n32FirstBearing,
    N32 n32SecondBearing,
    OSAL_FIXED_OBJECT hTargetLat,
    OSAL_FIXED_OBJECT hTargetLon
    )
{
    LOCATION_OBJECT hBaseLocation = LOCATION_INVALID_OBJECT;
    DISTANCE_OBJECT hHalfDistanceLat = DISTANCE_INVALID_OBJECT,
        hHalfDistanceLon = DISTANCE_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hFixedWidth, hFixedHeight;
    N32 n32Width, n32Height;
    OSAL_FIXED_OBJECT hLat, hLon, hFirstBearing, hSecondBearing;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
    BOOLEAN bReturn = FALSE;

    do
    {
        if ((hTargetLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hTargetLon == OSAL_FIXED_INVALID_OBJECT) ||
            (hBaseLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hBaseLon == OSAL_FIXED_INVALID_OBJECT))
        {
            // Error! Invalid parameters
            break;
        }

        // Creating location from base point
        // We need only coordinates, so creating it as zero-area circle
        hBaseLocation = LOCATION.hCreateForRadius (hBaseLat, hBaseLon,
            DISTANCE_INVALID_OBJECT);
        if (hBaseLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // Calculate halfway distances
        hFixedWidth = DISTANCE.hValue(hWidth, DISTANCE_UNIT_TYPE_MILES);
        hFixedHeight = DISTANCE.hValue(hHeight, DISTANCE_UNIT_TYPE_MILES);
        n32Width = OSAL_FIXED.n32Value(hFixedWidth);
        n32Height = OSAL_FIXED.n32Value(hFixedHeight);

        hHalfDistanceLat = DISTANCE.hCreate(n32Height / 2,
            DISTANCE_UNIT_TYPE_MILES);
        hHalfDistanceLon = DISTANCE.hCreate(n32Width / 2,
            DISTANCE_UNIT_TYPE_MILES);

        // Create intermediate location coordinates
        hLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);
        if (hLat == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }
        hLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);
        if (hLon == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Create our bearings in degrees
        hFirstBearing = OSAL_FIXED.hCreateInMemory(n32FirstBearing, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 2]);
        if (hFirstBearing == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        hSecondBearing = OSAL_FIXED.hCreateInMemory(n32SecondBearing, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 3]);
        if (hSecondBearing == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Calculating latitude of the target point
        if (n32Height == 0)
        {
            // This is a special case - no vertical shift.
            // So we do not need to calculate anything
            bReturn = OSAL_FIXED.bCopyToMemory(hBaseLat, hTargetLat);
        }
        else
        {
            // Otherwise - do some maths
            bReturn = bCalcDestCoordinatesFromLocation(
                (LOCATION_OBJECT_STRUCT*)hBaseLocation, hHalfDistanceLat,
                hFirstBearing, hTargetLat, hLon);
        }
        if( bReturn == FALSE )
        {
            break;
        }

        // Calculating longitude of the target point
        if (n32Width == 0)
        {
            // This is a special case - no vertical shift.
            // So we do not need to calculate anything
            bReturn = OSAL_FIXED.bCopyToMemory(hBaseLon, hTargetLon);
        }
        else
        {
            // Otherwise - do some maths
            bReturn = bCalcDestCoordinatesFromLocation(
                (LOCATION_OBJECT_STRUCT*)hBaseLocation, hHalfDistanceLon,
                hSecondBearing, hLat, hTargetLon);
        }

        if( bReturn == FALSE )
        {
            break;
        }

        // Finally!
        bReturn = TRUE;

    } while( FALSE );

    // Cleaning up temporary objects
    if (hBaseLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(hBaseLocation);
    }
    if (hHalfDistanceLat != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(hHalfDistanceLat);
    }
    if (hHalfDistanceLon != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(hHalfDistanceLon);
    }

    return bReturn;
}

/*****************************************************************************
*
*   bCalculateLongitudeAtDistance
*
*  This helper function is used to calculate longitude of the point
*  shifted by some given delta to the leftof the base point.
*
*  Inputs:
*       {hBaseLat, hBaseLon} - coordinates of the base point to go from
*       hDeltaLon - delta longitude to shift
*       hTargetLon - calculated target longitude
*  Return:
*       TRUE on success, with hTargetLon populated with calculated
*       coordinate,
*       FALSE on failure. In this case hTargetLon is not affected
*
*****************************************************************************/
static BOOLEAN bCalculateLongitudeAtDistance (
    OSAL_FIXED_OBJECT hBaseLat,
    OSAL_FIXED_OBJECT hBaseLon,
    DISTANCE_OBJECT hDeltaLon,
    OSAL_FIXED_OBJECT hTargetLon
        )
{
    LOCATION_OBJECT hBaseLocation = LOCATION_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLat, hLon, hBearing;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 3];
    BOOLEAN bReturn = FALSE;

    do
    {
        if ((hTargetLon == OSAL_FIXED_INVALID_OBJECT) ||
            (hBaseLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hBaseLon == OSAL_FIXED_INVALID_OBJECT))
        {
            // Error! Invalid parameters
            break;
        }

        // Creating location from base point
        // We need only coordinates, so creating it as zero-area circle
        hBaseLocation = LOCATION.hCreateForRadius (hBaseLat, hBaseLon,
            DISTANCE_INVALID_OBJECT);
        if (hBaseLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // Create intermediate location coordinates
        hLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);

        hLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);

        // Create 90 deg bearing (eastward direction)
        hBearing = OSAL_FIXED.hCreateInMemory(90, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE * 2]);

        if ((hLon == OSAL_FIXED_INVALID_OBJECT) ||
            (hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hBearing == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Calculating longitude of the target point
        bReturn = bCalcDestCoordinatesFromLocation(
            (LOCATION_OBJECT_STRUCT*)hBaseLocation, hDeltaLon,
            hBearing, hLat, hTargetLon);

        if( bReturn == FALSE )
        {
            break;
        }

        // Finally!
        bReturn = TRUE;

    } while( FALSE );

    // Cleaning up temporary objects
    if (hBaseLocation != LOCATION_INVALID_OBJECT)
    {
        LOCATION.vDestroy(hBaseLocation);
    }

    return bReturn;
}

/*****************************************************************************
 *
 *   bPrecalculateLatLonRadians
 *
 *****************************************************************************/
static BOOLEAN bPrecalculateLatLonRadians(
    LOCATION_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    
    if (psObj->sPrecalcFixedValues.hLatInRadian == OSAL_FIXED_INVALID_OBJECT)
    {
        psObj->sPrecalcFixedValues.hLatInRadian = 
            OSAL_FIXED.hCreate(0, 0, 0);
    }
    eReturnCode = OSAL_FIXED.eDegreeToRadian(psObj->hLat, 
        psObj->sPrecalcFixedValues.hLatInRadian);
    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    if (psObj->sPrecalcFixedValues.hLonInRadian == OSAL_FIXED_INVALID_OBJECT)
    {
        psObj->sPrecalcFixedValues.hLonInRadian =
            OSAL_FIXED.hCreate(0, 0, 0);
    }
    eReturnCode = OSAL_FIXED.eDegreeToRadian(psObj->hLon, 
        psObj->sPrecalcFixedValues.hLonInRadian);
    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }
    psObj->sPrecalcFixedValues.bIsLatLonInitialized = TRUE;

    return TRUE;     
}

/*****************************************************************************
 *
 *   bPrecalculateLatSinCos
 *
 *****************************************************************************/
static BOOLEAN bPrecalculateLatSinCos(
    LOCATION_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    
    if (psObj->sPrecalcFixedValues.bIsLatLonInitialized == FALSE)
    {
        BOOLEAN bSuccess;
        
        bSuccess = bPrecalculateLatLonRadians(psObj);
        if (bSuccess == FALSE)
        {
            return FALSE;
        }
    }

    if (psObj->sPrecalcFixedValues.hSinLat == OSAL_FIXED_INVALID_OBJECT)
    {
        psObj->sPrecalcFixedValues.hSinLat = OSAL_FIXED.hCreate(0, 0, 0);
    }

    eReturnCode = OSAL_FIXED.eSin(psObj->sPrecalcFixedValues.hLatInRadian, 
                                  psObj->sPrecalcFixedValues.hSinLat);
    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    if (psObj->sPrecalcFixedValues.hCosLat == OSAL_FIXED_INVALID_OBJECT)
    {
        psObj->sPrecalcFixedValues.hCosLat = OSAL_FIXED.hCreate(0, 0, 0);
    }

    eReturnCode = OSAL_FIXED.eCos(psObj->sPrecalcFixedValues.hLatInRadian,
                                  psObj->sPrecalcFixedValues.hCosLat);
    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }
    psObj->sPrecalcFixedValues.bIsSinCosLatInitialized = TRUE;

    return TRUE;     
}
/*****************************************************************************
 *
 *   vCreateFixedConstants
 *
 *****************************************************************************/
static void vCreateFixedConstants( void )
{
    GsLocationCoordLimits.hMinLat = OSAL_FIXED.hCreateInMemory(
        LOCATION_MIN_LATITUDE, 0,
        &GsLocationCoordLimits.atData[OSAL_FIXED_OBJECT_SIZE * 0]);

    GsLocationCoordLimits.hMaxLat= OSAL_FIXED.hCreateInMemory(
        LOCATION_MAX_LATITUDE, 0,
        &GsLocationCoordLimits.atData[OSAL_FIXED_OBJECT_SIZE * 1]);

    GsLocationCoordLimits.hMinLon = OSAL_FIXED.hCreateInMemory(
        LOCATION_MIN_LONGITUDE, 0,
        &GsLocationCoordLimits.atData[OSAL_FIXED_OBJECT_SIZE * 2]);

    GsLocationCoordLimits.hMaxLon = OSAL_FIXED.hCreateInMemory(
        LOCATION_MAX_LONGITUDE, 0,
        &GsLocationCoordLimits.atData[OSAL_FIXED_OBJECT_SIZE * 3]);

    GsLocationCoordLimits.bIsMinMaxCoordinatesInitialized = TRUE;

    return;
}

#if DEBUG_LOCATION_COORDINATES == 1
/*******************************************************
*
* LOCATION_DEBUG_fFixed2Float
*
* The purpose of this function is to convert OSAL FIXED
* number to float type.
*
* Inputs:
*   hFixed - The handle of OSAL FIXED object
*
* Outputs:
*   float value
*
********************************************************/
static float LOCATION_DEBUG_fFixed2Float(OSAL_FIXED_OBJECT hFixed)
{
    float fReturnValue = 0.0;

    if(hFixed != OSAL_INVALID_OBJECT_HDL)
    {
        fReturnValue = (float)OSAL_FIXED.n32Value(hFixed) / (float)(1UL << OSAL_FIXED.un8NumFractionalBits(hFixed));
    }

    return fReturnValue;
}
#endif

#ifdef SUPPORT_CUNIT
#include <location_obj.cunit>
#endif // SUPPORT_CUNIT
