/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:LOCALIZATION implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"

#include "localization_obj.h"
#include "_localization_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
 *
 *   bSetMeasurementSystem
 *
 *****************************************************************************/
static BOOLEAN bSetMeasurementSystem (
    LOCALIZATION_MEASUREMENT_SYSTEM_ENUM eMeasureSystem
        )
{
    const LOCALIZATION_ITEM_STRUCT *psLocalization = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // Try initializing LOCALIZATION object. If it is already created,
    // ALREADY_INITIALIZED will be returned immediately
    eReturn = eInitialize();
    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS &&
        eReturn != SMSAPI_RETURN_CODE_ALREADY_INITIALIZED)
    {
        // Cannot initialize object
        return FALSE;
    }

    if (eMeasureSystem != GpsLocalization->eSystem)
    {
        GpsLocalization->psCurrentLocal = NULL;
        GpsLocalization->eSystem = eMeasureSystem;
    }

    // Try to get localization to check its validity
    psLocalization = psGetCurrentLocalization();
    if (psLocalization == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            LOCALIZATION_OBJECT_NAME": failed to change localization to %d",
            eMeasureSystem);

        // Setting current to default
        psLocalization = psGetDefaultLocalization();
        if (psLocalization != NULL)
        {
            GpsLocalization->psCurrentLocal = psLocalization;
            GpsLocalization->eSystem = LOCALIZATION_DEFAULT;
        }
        else
        {
            // Error! Very bad - cannot retrieve even default
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LOCALIZATION_OBJECT_NAME": cannot get default Localization");
        }

        // Returning FALSE anyway, even if Localization was successfully
        // reset to default
        return FALSE;
    }

    printf(LOCALIZATION_OBJECT_NAME
        ": new localization is %d\n", psLocalization->eSystem);

    return TRUE;
}

/*****************************************************************************
 *
 *   eGetTemperatureUnits
 *
 *****************************************************************************/
static LOCALIZATION_MEASUREMENT_UNITS_ENUM eGetTemperatureUnits (
       void )
{
    const LOCALIZATION_ITEM_STRUCT *psObj = psGetCurrentLocalization();
    if (psObj != NULL)
    {
        return psObj->eTemperature;
    }
    else
    {
    	SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
    			LOCALIZATION_OBJECT_NAME
                ": failed to get temperature due to localization absence\n");
        return LOCALIZATION_MEASUREMENT_UNITS_UNKNOWN;
    }
}

/*****************************************************************************
 *
 *   eGetSnowRangeUnits
 *
 *****************************************************************************/
static LOCALIZATION_MEASUREMENT_UNITS_ENUM eGetDistanceUnits (
       void )
{
    const LOCALIZATION_ITEM_STRUCT *psObj = psGetCurrentLocalization();
    if (psObj != NULL)
    {
        return psObj->eDistance;
    }
    else
    {
    	SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
    			LOCALIZATION_OBJECT_NAME
                ": failed to get distance due to localization absence\n");
        return LOCALIZATION_MEASUREMENT_UNITS_UNKNOWN;
    }
}


/*****************************************************************************
 *
 *   eGetSpeedUnits
 *
 *****************************************************************************/
static LOCALIZATION_MEASUREMENT_UNITS_ENUM eGetSpeedUnits (
       void )
{
    const LOCALIZATION_ITEM_STRUCT *psObj = psGetCurrentLocalization();
    if (psObj != NULL)
    {
        return psObj->eSpeed;
    }
    else
    {
    	SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
    			LOCALIZATION_OBJECT_NAME
                ": failed to get speed due to localization absence\n");
        return LOCALIZATION_MEASUREMENT_UNITS_UNKNOWN;
    }
}

/*****************************************************************************
 *
 *   eGetPrecipitationUnits
 *
 *****************************************************************************/
static LOCALIZATION_MEASUREMENT_UNITS_ENUM eGetPrecipitationUnits (
       void )
{
    const LOCALIZATION_ITEM_STRUCT *psObj = psGetCurrentLocalization();
    if (psObj != NULL)
    {
        return psObj->ePrecipAmount;
    }
    else
    {
    	SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
    			LOCALIZATION_OBJECT_NAME
                ": failed to get precipitation amount due to localization absence\n");
        return LOCALIZATION_MEASUREMENT_UNITS_UNKNOWN;
    }
}

/*****************************************************************************
 *
 *   eGetLanguage
 *
 *****************************************************************************/
static SMS_LANGUAGE_ENUM eGetLanguage (
       void )
{
    return SMS_LANGUAGE_ENGLISH;
}

/*****************************************************************************
 *                             FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   LOCALIZATION_eInitialize
 *
 *   Creates structure used by LOCALIZATION object and
 *   initializes its contents
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM eInitialize( void )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ALREADY_INITIALIZED;

    if (GpsLocalization == NULL)
    {
        GpsLocalization = (LOCALIZATION_OBJECT_STRUCT*)SMSO_hCreate(
            LOCALIZATION_OBJECT_NAME,
            sizeof(LOCALIZATION_OBJECT_STRUCT),
            SMS_INVALID_OBJECT, // No parent
            FALSE);             // No lock

        if (GpsLocalization == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
        }
        else
        {
            GpsLocalization->eSystem = LOCALIZATION_DEFAULT;
            GpsLocalization->psCurrentLocal =
                GpsLocalization->psDefaultLocal =
                    psGetDefaultLocalization();
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   LOCALIZATION_vUnInitialize
 *
 *   Destroys global Localization structure
 *
 *****************************************************************************/
void LOCALIZATION_vUnInitialize( void )
{
    if (GpsLocalization != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)GpsLocalization);
    }

    return;
}

/*****************************************************************************
 *                             PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   psFindLocalization
 *
 *****************************************************************************/
static const LOCALIZATION_ITEM_STRUCT *psFindLocalization(
    LOCALIZATION_MEASUREMENT_SYSTEM_ENUM eMeasureSystem
        )
{
    UN32 un32Local;
    const LOCALIZATION_ITEM_STRUCT *psLocalization = NULL;

    // Need to find out default localization
    for (un32Local = 0;
         un32Local < (sizeof(GsLocalizationList) / sizeof(GsLocalizationList[0]));
         ++un32Local)
    {
        if (GsLocalizationList[un32Local].eSystem == eMeasureSystem)
        {
            psLocalization = &GsLocalizationList[un32Local];
            break;
        }
    }

    return psLocalization;
}

/*****************************************************************************
 *
 *   psGetCurrentLocalization
 *
 *****************************************************************************/
static const LOCALIZATION_ITEM_STRUCT *psGetCurrentLocalization(
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // Try initializing LOCALIZATION object. If it is already created,
    // ALREADY_INITIALIZED will be returned immediately
    eReturn = eInitialize();
    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS &&
        eReturn != SMSAPI_RETURN_CODE_ALREADY_INITIALIZED)
    {
        // Cannot initialize object
        return NULL;
    }

    // Update current localization if it is absent
    if (GpsLocalization->psCurrentLocal == NULL)
    {
        GpsLocalization->psCurrentLocal =
                psFindLocalization(GpsLocalization->eSystem);
    }

    return GpsLocalization->psCurrentLocal;
}

/*****************************************************************************
 *
 *   psGetDefaultLocalization
 *
 *****************************************************************************/
static const LOCALIZATION_ITEM_STRUCT *psGetDefaultLocalization(
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // Try initializing LOCALIZATION object. If it is already created,
    // ALREADY_INITIALIZED will be returned immediately
    eReturn = eInitialize();
    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS &&
        eReturn != SMSAPI_RETURN_CODE_ALREADY_INITIALIZED)
    {
        // Cannot initialize object
        return NULL;
    }

    // Load default configuration if it is not yet available
    if (GpsLocalization->psDefaultLocal == NULL)
    {
        GpsLocalization->psDefaultLocal =
                psFindLocalization(LOCALIZATION_DEFAULT);
    }

    return GpsLocalization->psDefaultLocal;
}
