/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
*  This module contains the TEAM ID implementation for the
*  Sirius Module Services (SMS)
*
******************************************************************************/

#include <string.h>
#include <stdarg.h>

#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "cme.h"
#include "cdo_obj.h"
#include "sports_obj.h"

// Include TEAM ID module headers
#include "league_team_id.h"
#include "_league_team_id.h"

/*****************************************************************************
PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pvCreate
*
*   Creates a CID object data object (LEAGUE_TEAM_ID) from object data which is an
*   integer.
*
*****************************************************************************/
static const void *pvCreate (
    const void *pvSrcObjectData,
    size_t tMinimumSize,
    SMS_OBJECT hParent,
    BOOLEAN bConstant
    )
{
    // Object data for this type is a LEAGUE_TEAM_ID_STRUCT object
    LEAGUE_TEAM_ID_STRUCT *psLeagueTeamId = NULL;

    // Source object data (raw) is a LEAGUE_TEAM_ID_STRUCT type
    const LEAGUE_TEAM_ID_STRUCT *psSrcLeagueTeamId =
        (const LEAGUE_TEAM_ID_STRUCT *)pvSrcObjectData;

    // If a valid handle was provided, the SMS object create will
    // simply inherit any locking ability that objects has. However
    // if the handle is invalid, we would never use any locking. This
    // is due to the fact that this happens only when making a duplicate.
    // Therefore providing FALSE for the lock feature is always right.

    // Create a TEAM ID object
    psLeagueTeamId = (LEAGUE_TEAM_ID_STRUCT *)SMSO_hCreate(
        "CID_LeagueTeamId",
        sizeof(LEAGUE_TEAM_ID_STRUCT),
        hParent,
        FALSE);

    if (psLeagueTeamId != NULL)
    {
        // Just copy the contents
        psLeagueTeamId->hLeagueId = CID.hDuplicate(psSrcLeagueTeamId->hLeagueId);
        psLeagueTeamId->hTeamId = CID.hDuplicate(psSrcLeagueTeamId->hTeamId);
    }

    return (const void *)psLeagueTeamId;
}

/*****************************************************************************
*
*   vDestroy
*
*   Destroys a CID object data object (LEAGUE_TEAM_ID)
*
*****************************************************************************/
static void vDestroy (
    const void *pvObject
    )
{
    LEAGUE_TEAM_ID_STRUCT const *psLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT const *)pvObject;

    // Get rid of entries
    CID.vDestroy(psLeagueTeamId->hLeagueId);
    CID.vDestroy(psLeagueTeamId->hTeamId);

    // These are SMSO objects
    SMSO_vDestroy((SMS_OBJECT)pvObject);

    return;
}

/*****************************************************************************
*
*   pvDuplicate
*
*   Duplicates a CID object data object (LEAGUE_TEAM_ID)
*
*****************************************************************************/
static const void *pvDuplicate (
    const void *pvObject
    )
{
    // These are TEAM_ID's
    LEAGUE_TEAM_ID_STRUCT const *psLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT const *)pvObject;
    LEAGUE_TEAM_ID_STRUCT const *psDuplicateTeamId;

    psDuplicateTeamId = (LEAGUE_TEAM_ID_STRUCT const *)
        pvCreate(
        (void const *)psLeagueTeamId,
        sizeof(LEAGUE_TEAM_ID_STRUCT),
        SMS_INVALID_OBJECT,
        FALSE
        );

    return (const void *)psDuplicateTeamId;
}

/*****************************************************************************
*
*   bModify
*
*   Modifies a CID object data object (LEAGUE_TEAM_ID)
*
*****************************************************************************/
static BOOLEAN bModify (
    const void **ppvObject,
    const void *pvSrcObjectData
    )
{
    // Object data for this type is a LEAGUE_TEAM_ID_STRUCT object
    LEAGUE_TEAM_ID_STRUCT **ppsLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT **)ppvObject;

    // Source object data (raw) is a LEAGUE_TEAM_ID_STRUCT type
    LEAGUE_TEAM_ID_STRUCT const *psSrcLeagueTeamId =
        (LEAGUE_TEAM_ID_STRUCT const *)pvSrcObjectData;

    // Get rid of old entries
    CID.vDestroy((*ppsLeagueTeamId)->hLeagueId);
    CID.vDestroy((*ppsLeagueTeamId)->hTeamId);

    // Put in new ones
    (*ppsLeagueTeamId)->hLeagueId = psSrcLeagueTeamId->hLeagueId;
    (*ppsLeagueTeamId)->hTeamId = psSrcLeagueTeamId->hTeamId;

    return FALSE;
}

/*****************************************************************************
*
*   bCopy
*
*   Copies a CID object data object (LEAGUE_TEAM_ID) into another
*
*****************************************************************************/
static BOOLEAN bCopy (
    void **ppvDstObject,
    const void *pvSrcObject
    )
{
    // These are TEAM_ID objects
    LEAGUE_TEAM_ID_STRUCT **ppsLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT **)ppvDstObject;
    LEAGUE_TEAM_ID_STRUCT *psLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT *)pvSrcObject;

    // Get rid of old entries
    CID.vDestroy((*ppsLeagueTeamId)->hLeagueId);
    CID.vDestroy((*ppsLeagueTeamId)->hTeamId);

    // Put in new ones
    (*ppsLeagueTeamId)->hLeagueId = psLeagueTeamId->hLeagueId;
    (*ppsLeagueTeamId)->hTeamId = psLeagueTeamId->hTeamId;

    return TRUE;
}

/*****************************************************************************
*
*   tSize
*
*   Retrieves the size a CID object data object (LEAGUE_TEAM_ID) in bytes
*
*****************************************************************************/
static size_t tSize(
    const void *pvObject
    )
{
    return sizeof(LEAGUE_TEAM_ID_STRUCT);
}

/*****************************************************************************
*
*   n16Compare
*
*   Compares a CID object data object (LEAGUE_TEAM_ID) to another CID object data
*   object.
*
*****************************************************************************/
static N16 n16Compare (
    const void *pvObject1,
    const void *pvObject2,
    BOOLEAN bBinary
    )
{
    // These are TEAM_ID objects
    LEAGUE_TEAM_ID_STRUCT const *psLeagueTeamId1 = 
        (LEAGUE_TEAM_ID_STRUCT const *)pvObject1, 
        *psLeagueTeamId2 = (LEAGUE_TEAM_ID_STRUCT const *)pvObject2;
    N16 n16Return;

    if((psLeagueTeamId1->hLeagueId == CID_INVALID_OBJECT) || 
        (psLeagueTeamId2->hLeagueId == CID_INVALID_OBJECT))
    {
        n16Return = 0;
    }
    else
    {
        n16Return = CID.n16Compare(
            psLeagueTeamId1->hLeagueId, psLeagueTeamId2->hLeagueId);
    }

    if(n16Return == 0)
    {
        n16Return = CID.n16Compare(
            psLeagueTeamId1->hTeamId, psLeagueTeamId2->hTeamId);
    }

    return n16Return;
}

/*****************************************************************************
*
*   n32FWrite
*
*   Writes a CID object data object (LEAGUE_TEAM_ID) to a device.
*
*****************************************************************************/
static N32 n32FWrite (
    const void *pvObject,
    FILE *psFile
    )
{
    LEAGUE_TEAM_ID_STRUCT const *psLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT const *)pvObject;
    N32 n32Written = 0;

    // Write the league
    n32Written += CID.n32FWrite(psLeagueTeamId->hLeagueId, psFile);

    // Write the team
    n32Written += CID.n32FWrite(psLeagueTeamId->hTeamId, psFile);

    return n32Written;
}

/*****************************************************************************
*
*   n32FWriteToMemory
*
*   Writes a CID object data object (LEAGUE_TEAM_ID) to memory.
*
*****************************************************************************/
static N32 n32FWriteToMemory (
    const void *pvObject,
    void **ppvMemory
    )
{
    LEAGUE_TEAM_ID_STRUCT const *psLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT const *)pvObject;
    N32 n32Return = 0;

    *((LEAGUE_TEAM_ID_STRUCT *)(*ppvMemory)) = *psLeagueTeamId; 

    return n32Return;
}

/*****************************************************************************
*
*   n32GetValue
*
*   Retreives CID object data value (LEAGUE_TEAM_ID)
*
*****************************************************************************/
static N32 n32GetValue (
    const void *pvObject,
    void **ppvValue
    )
{
    LEAGUE_TEAM_ID_STRUCT const *psLeagueTeamId = 
        (LEAGUE_TEAM_ID_STRUCT const *)pvObject;
    LEAGUE_TEAM_ID_STRUCT const **pptValue = 
        (LEAGUE_TEAM_ID_STRUCT const **)ppvValue;
    N32 n32Return = EOF;

    if (*pptValue != NULL)
    {
        // write the value
        *pptValue =  psLeagueTeamId;

        n32Return = sizeof(LEAGUE_TEAM_ID_STRUCT);
    }

    return n32Return;
}

/*****************************************************************************
*
*   pvRead
*
*   Reads a CID object data object (LEAGUE_TEAM_ID) from a file
*
*****************************************************************************/
static const void *pvRead (
    FILE *psFile
    )
{
    LEAGUE_TEAM_ID_STRUCT const *psNewTeamId;
    LEAGUE_TEAM_ID_STRUCT sTeamId;

    sTeamId.hLeagueId = CID.hFRead(psFile);
    sTeamId.hTeamId = CID.hFRead(psFile);

    psNewTeamId = (LEAGUE_TEAM_ID_STRUCT const *)
        pvCreate(
        (void const *)&sTeamId,
        sizeof(LEAGUE_TEAM_ID_STRUCT),
        SMS_INVALID_OBJECT,
        FALSE
        );

    return psNewTeamId;
}

/*****************************************************************************
*
*   bReadFromMemory
*
*   Create a CID object (LEAGUE_TEAM_ID) from data in memory.
*
*****************************************************************************/
static BOOLEAN bReadFromMemory(
    const void **ppvObjectData,
    void **ppvMemory,
    SMS_OBJECT hParent
    )
{
    LEAGUE_TEAM_ID_STRUCT const *psLeagueTeamId;
    BOOLEAN bOk = FALSE;

    // Verify we have a CID
    if ( (ppvObjectData != NULL) && (ppvMemory != NULL) )
    {
        psLeagueTeamId = (LEAGUE_TEAM_ID_STRUCT const *)*ppvObjectData;

        // Check if this CID has object data we can reuse.
        // If so, we can just modify it. Otherwise we
        // need to create new object data.
        if(*ppvObjectData != NULL)
        {
            // Modify the existing object data, replacing it with the new
            // object data provided by the caller.
            bOk = bModify((const void **)&psLeagueTeamId, *ppvMemory);
        }
        else
        {
            psLeagueTeamId = (LEAGUE_TEAM_ID_STRUCT const *)
                pvCreate(
                (void const *)*ppvMemory,
                sizeof(LEAGUE_TEAM_ID_STRUCT),
                SMS_INVALID_OBJECT,
                FALSE
                );
            if(psLeagueTeamId != NULL)
            {
                bOk = TRUE;
            }
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   n32FPrintf
*
*   Prints a CID object data object (LEAGUE_TEAM_ID) to a file
*
*****************************************************************************/
static N32 n32FPrintf (
    const void *pvObject,
    FILE *psFile
    )
{
    N32 n32NumWritten = EOF;
    if (pvObject != NULL)
    {
        LEAGUE_TEAM_ID_STRUCT *psLeagueTeamId = 
            (LEAGUE_TEAM_ID_STRUCT *)pvObject;

        n32NumWritten = 
            CID.n32FPrintf(psLeagueTeamId->hLeagueId, psFile);
        n32NumWritten += 
            CID.n32FPrintf(psLeagueTeamId->hTeamId, psFile);
    }

    return n32NumWritten;
}

/*****************************************************************************
*
*   bParse
*
*****************************************************************************/
static BOOLEAN bParse (
    CD_OBJECT hCDO,
    const void *pvData
    )
{
    // Nothing to do
    return FALSE;
}

/*****************************************************************************
FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
PRIVATE FUNCTIONS
*****************************************************************************/
