/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
*  This module contains the Object:LEAGUE implementation for the
*  Sirius Module Services (SMS)
*
******************************************************************************/
#include <stdio.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "cid_obj.h"
#include "cid_integer.h"
#include "sql_interface_obj.h"

#include "db_util.h"

#include "sports.h"
#include "team_obj.h"
#include "sports_obj.h"
#include "sms.h"
#include "radio.h"
#include "league_team_id.h"
#include "league_obj.h"
#include "_league_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hName
*
*****************************************************************************/
static STRING_OBJECT hName (
    LEAGUE_OBJECT hLeague
    )
{
    STRING_OBJECT hName = STRING_INVALID_OBJECT;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(psLeagueObject != NULL)
    {
        // Extract name
        hName = psLeagueObject->psLeague->hName;
    }

    return hName;
}

/*****************************************************************************
*
*   hAbbreviation
*
*****************************************************************************/
static STRING_OBJECT hAbbreviation (
    LEAGUE_OBJECT hLeague
    )
{
    STRING_OBJECT hAbbreviation = STRING_INVALID_OBJECT;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(psLeagueObject != NULL)
    {
        // Extract abbreviation
        hAbbreviation = psLeagueObject->psLeague->hAbbreviation;
    }

    return hAbbreviation;
}

/*****************************************************************************
*
*   eLeague
*
*****************************************************************************/
static LEAGUE_ENUM eLeague (
    LEAGUE_OBJECT hLeague
    )
{
    LEAGUE_ENUM eReturnLeague = LEAGUE_UNKNOWN;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(psLeagueObject != NULL)
    {
        // Extract league
        eReturnLeague = psLeagueObject->psLeague->eLeague;
    }

    return eReturnLeague;
}

/*****************************************************************************
*
*   bIterateContent
*
*****************************************************************************/
static BOOLEAN bIterateContent (
    LEAGUE_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
        )
{
    BOOLEAN bLocked, bResult = FALSE;

    // Try to lock master object
    bLocked = bMasterLock();
    if (bLocked == TRUE)
    {
        bResult =
            LEAGUE_bIterateContent(bContentIteratorCallback,
                                   pvContentIteratorCallbackArg);

        vMasterUnlock();
    }

    return bResult;
}

/*****************************************************************************
*
*   hId
*
*****************************************************************************/
static CID_OBJECT hId (
    LEAGUE_OBJECT hLeague
    )
{
    CID_OBJECT hId = CID_INVALID_OBJECT;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(hLeague != LEAGUE_INVALID_OBJECT)
    {
        // Extract id
        hId = psLeagueObject->hId;
    }

    return hId;
}

/*****************************************************************************
*
*   eSport
*
*****************************************************************************/
SPORTS_ENUM eSport (
    LEAGUE_OBJECT hLeague
    )
{

    SPORTS_ENUM eSport = SPORTS_UNKNOWN;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(psLeagueObject != NULL)
    {
        // Extract name
        eSport = psLeagueObject->psLeague->eSport;
    }

    return eSport;
}

/*****************************************************************************
*
*   n32Version
*
*****************************************************************************/
static N32 n32Version (
    LEAGUE_OBJECT hLeague
    )
{
    N32 n32Version = LEAGUE_INVALID_VERSION;

    LEAGUE_OBJECT_STRUCT const *psObj =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(psObj != NULL)
    {
        n32Version = psObj->psLeague->n32Version;
    }

    return n32Version;
}

/*****************************************************************************
*
*   bGetTableVersion
*
*****************************************************************************/
static BOOLEAN bGetTableVersion (
    N32 *pn32Version,
    BOOLEAN *pbComplete
    )
{
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bLocked;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        if(pn32Version != NULL)
        {
            *pn32Version = gpsLeagueCtrl->n32Version;
        }

        if(pbComplete != NULL)
        {
            *pbComplete = gpsLeagueCtrl->bComplete;
        }

        bSuccess = TRUE;

        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   hTeamId
*
*   This API is used to retreive a CID object representing membership
*   of the provided Team object in the corresponding League.
*
*   Where:
*       hLeague - A valid League handle for which the caller wishes to 
*           determine a Team membership.
*       hTeam - A valid Team handle for which the caller wishes to 
*           determine membership in the desired League.
*
*   Return:
*       A valid CID_OBJECT handle is returned on success / relationship
*           existence.
*       Otherwise CID_INVALID_OBJECT is returned.
*
*****************************************************************************/
static CID_OBJECT hTeamId (
    LEAGUE_OBJECT hLeague,
    TEAM_OBJECT hTeam
        )
{
    BOOLEAN bLocked;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    CID_OBJECT hId = CID_INVALID_OBJECT;

    LEAGUE_TEAM_ENTRY_STRUCT sEntry;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if ((hLeague == LEAGUE_INVALID_OBJECT) ||
        (hTeam == TEAM_INVALID_OBJECT))
    {
        return hId;
    }

    // Set a Team to search for
    sEntry.hTeam = hTeam;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        BOOLEAN bSuccess;
        LEAGUE_SEARCH_STRUCT sSearch;

        // Initialize the struct
        sSearch.bFound = FALSE;
        sSearch.hLeagueId = LEAGUE.hId(hLeague);

        // Search for the league membership
        bSuccess = TEAM_bIterateLeagues(hTeam, bSearchForLeague, &sSearch);
        if ((bSuccess == TRUE) && (sSearch.bFound == TRUE))
        {
            eReturnCode = OSAL.eLinkedListSearch(
                psLeagueObject->hTeamList,
                &hEntry,
                (void *)&sEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                LEAGUE_TEAM_ENTRY_STRUCT *psEntry;

                psEntry = (LEAGUE_TEAM_ENTRY_STRUCT *)
                    OSAL.pvLinkedListThis(hEntry);

                hId = psEntry->hLeagueTeamId;
            }
            else if (eReturnCode == OSAL_ERROR_INVALID_HANDLE)
            {
                /* This is a special case for newly added leagues (without valid LeagueID)
                   We do nothing here and suppress error message */
            }
            else if (eReturnCode != OSAL_OBJECT_NOT_FOUND)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    LEAGUE_OBJECT_NAME": Failed to search a Team: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            }
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }

    return hId;
}

/*****************************************************************************
*
*   un8Id
*
*****************************************************************************/
static UN8 un8Id (
    LEAGUE_OBJECT hLeague
        )
{
    UN8 un8Id = UN8_MAX;

    LEAGUE_OBJECT_STRUCT const *psObj =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(psObj != NULL)
    {
        UN32 un32Id = UN32_MAX;
        void *pvId = &un32Id;
        CID_n32GetValue(psObj->hId, &pvId);
        un8Id = (UN8)un32Id;
    }

    return un8Id;
}

/*****************************************************************************
*
*   bIsSportsFlashEnabled
*
*****************************************************************************/
static BOOLEAN bIsSportsFlashEnabled(
    LEAGUE_OBJECT hLeague
        )
{
    BOOLEAN bEnabled = FALSE;

    LEAGUE_OBJECT_STRUCT const *psObj =
        (LEAGUE_OBJECT_STRUCT const *)hLeague;

    if(psObj != NULL)
    {
        bEnabled = psObj->psLeague->bSportsFlashEnabled;
    }

    return bEnabled;
}

/*****************************************************************************
FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   LEAGUE_bInitialize
*
* Initialize the available LEAGUE list. This needs to be called once
* when SMS is started. The LEAGUE objects being initialized here can be
* used by multiple SMS objects.
*
* NOTE: This function assumes it will be called only once,
* and by a single context. Without first uninitializing it again.
*
*****************************************************************************/
BOOLEAN LEAGUE_bInitialize ( SMS_OBJECT hParent )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOk;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;

    // Initialize the known league list.
    // Creates an empty league list to be populated later.

    // Create an SMS object which can be locked to enforce exclusive
    // access to the elements within this structure.
    gpsLeagueCtrl = (LEAGUE_CTRL_STRUCT *)
        SMSO_hCreate(
        LEAGUE_OBJECT_NAME":Ctrl",
        sizeof(LEAGUE_CTRL_STRUCT),
        hParent,
        TRUE // We want the lock feature (inherited)
        );
    if(gpsLeagueCtrl == NULL)
    {
        // Error! Something went wrong.
        return FALSE;
    }

    do
    {
        // Create the league list of entries.
        eReturnCode =
            OSAL.eLinkedListCreate(
            &gpsLeagueCtrl->hLeagueList,
            LEAGUE_OBJECT_NAME":List",
            (OSAL_LL_COMPARE_HANDLER)n16CompareIds,
            OSAL_LL_OPTION_UNIQUE | 
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
            OSAL_LL_OPTION_RBTREE
            );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create the league trash bin.
        eReturnCode =
            OSAL.eLinkedListCreate(
            &gpsLeagueCtrl->hLeagueTrashBin,
            LEAGUE_OBJECT_NAME":TrashBin",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR |
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
            );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize CID
        gpsLeagueCtrl->hWorkingLeagueId = CID_INVALID_OBJECT;

        // Initialize last processed version
        gpsLeagueCtrl->n32Version = LEAGUE_INVALID_VERSION;
        gpsLeagueCtrl->bComplete = FALSE;

        // Initialize filename
        gpsLeagueCtrl->pacFileName = NULL;

        // Initialize Master Lockable object
        gpsLeagueCtrl->hMasterLockableObject = SMS_INVALID_OBJECT;

        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            SPORTS_DB_FOLDER,
            SPORTS_DB_FILENAME,
            &gpsLeagueCtrl->pacFileName);

        if (bOk == TRUE)
        {
            gpsLeagueCtrl->hSQLConnection =
                DB_UTIL_hConnectToPersistent(
                &gpsLeagueCtrl->pacFileName[0],
                bCreateDBTables,
                (void *)NULL,
                bVerifyDBVersion,
                (void *)NULL,
                &eErrorCode);

            if (gpsLeagueCtrl->hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
            {
                if (eErrorCode == DATASERVICE_ERROR_CODE_DATABASE_READONLY)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        LEAGUE_OBJECT_NAME": database is read-only '%s'.",
                        &gpsLeagueCtrl->pacFileName[0]);
                }
                else if (eErrorCode == DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        LEAGUE_OBJECT_NAME": version mismatch with database '%s'.",
                        &gpsLeagueCtrl->pacFileName[0]);
                }
                else if (eErrorCode == DATASERVICE_ERROR_CODE_DATABASE_CORRUPT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        LEAGUE_OBJECT_NAME": corrupt database '%s'.",
                        &gpsLeagueCtrl->pacFileName[0]);
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        LEAGUE_OBJECT_NAME": Could not open connection to the "
                        "database '%s' used to identify sports leagues and teams.",
                        &gpsLeagueCtrl->pacFileName[0]);
                }

                // continue init even if SQL connection fails. We can run without
                // db - we just dont get any persisted info.
            }
        }

        // Relinquish control of the object if we are the parent.
        if(hParent == SMS_INVALID_OBJECT)
        {
            SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
        }

        return TRUE;

    } while (FALSE);

    LEAGUE_vUnInitialize();

    return FALSE;
}

/*****************************************************************************
*
*   LEAGUE_bSetMasterLockableObject
*
* Attaches SMS Object which shall be locked before LEAGUE control structure
* where it is needed.
*
*****************************************************************************/
BOOLEAN LEAGUE_bSetMasterLockableObject (
    SMS_OBJECT hObject
        )
{
    BOOLEAN bLocked;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        gpsLeagueCtrl->hMasterLockableObject = hObject;

        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }

    return bLocked;
}

/*****************************************************************************
*
*   LEAGUE_vUnInitialize
*
* Uninitialize the LEAGUE list (upon SMS shutdown). This also iterates each
* teams within a league and uninitializes each team list as well.
*
* NOTE: This function assumes it will be called only once after
* the object has been initialized, and by a single context.
*
*****************************************************************************/
void LEAGUE_vUnInitialize ( void )
{
    BOOLEAN bLocked;

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error! Something went wrong or no object created in the first place.
        return;
    }

    // Save anything that needs updated
    LEAGUE_bSave(LEAGUE_INVALID_VERSION);

    if (gpsLeagueCtrl->hSQLConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        // close the connection to the db
        SQL_INTERFACE.vDisconnect(gpsLeagueCtrl->hSQLConnection);
    }

    // Check if league list exists.
    if(gpsLeagueCtrl->hLeagueList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy all league map entries first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            gpsLeagueCtrl->hLeagueList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyLeagueObject
            );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy the list itself
            eReturnCode = OSAL.eLinkedListDelete(gpsLeagueCtrl->hLeagueList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gpsLeagueCtrl->hLeagueList = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Check if league trash bin exists.
    if(gpsLeagueCtrl->hLeagueTrashBin != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy all league entries first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            gpsLeagueCtrl->hLeagueTrashBin,
            (OSAL_LL_RELEASE_HANDLER)vDestroyLeagueObject
            );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy the list itself
            eReturnCode = OSAL.eLinkedListDelete(gpsLeagueCtrl->hLeagueTrashBin);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gpsLeagueCtrl->hLeagueTrashBin = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    if (gpsLeagueCtrl->pacFileName != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)gpsLeagueCtrl->pacFileName);
        gpsLeagueCtrl->pacFileName = NULL;
    }

    if(gpsLeagueCtrl->hWorkingLeagueId != CID_INVALID_OBJECT)
    {
        // destroy working CID
        CID_vDestroy(gpsLeagueCtrl->hWorkingLeagueId);
        gpsLeagueCtrl->hWorkingLeagueId = CID_INVALID_OBJECT;
    }

    // Unlock object
    SMSO_vUnlock((SMS_OBJECT) gpsLeagueCtrl);

    // Destroy object itself
    SMSO_vDestroy((SMS_OBJECT)gpsLeagueCtrl);

    gpsLeagueCtrl = NULL;

    return;
}

/*****************************************************************************
*
*   LEAGUE_hAdd
*
*   This function assigns a CID to represent this league content. The CID
*   is used to uniquely describe this league and map it to some
*   textual information about the league (e.g. name, abbr, teams, etc.).
*
*   Inputs
*       eSport - 
*       eLeague - 
*       hId - A CID representing this LEAGUE
*       n32Version -
*       pacName -
*       pacAbbrev -
*
*   Outputs:
*       LEAGUE_OBJECT - A valid LEAGUE_OBJECT if the CID was able to be added.
*       Otherwise returns LEAGUE_INVALID_OBJECT if the request resulted
*       in an error.
*
*****************************************************************************/
LEAGUE_OBJECT LEAGUE_hAdd (
    SPORTS_ENUM eSport,
    LEAGUE_ENUM eLeague,
    CID_OBJECT hLeagueId,
    N32 n32Version,
    char const *pacName,
    char const *pacAbbrev,
    BOOLEAN bSportsFlashEnabled,
    UN8 un8SportsFlashTiers
    )
{
    BOOLEAN bLocked;
    LEAGUE_OBJECT_STRUCT *psNewLeagueObject;

    // Check inputs
    if((hLeagueId == CID_INVALID_OBJECT) || (pacName == NULL) ||
        (pacAbbrev == NULL))
    {
        // Error!
        return LEAGUE_INVALID_OBJECT;
    }

    // Invent a new league entry
    psNewLeagueObject = psCreateLeagueObject (
        eSport, eLeague, hLeagueId, n32Version, pacName, pacAbbrev,
        bSportsFlashEnabled, un8SportsFlashTiers
        );
    if (psNewLeagueObject == NULL)
    {
        // Error! Unable to create
        return LEAGUE_INVALID_OBJECT;
    }

    // Now, add this league to our list.

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hExistingLeagueObject = 
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Find this CID in our LEAGUE list (if we can)
        eReturnCode = OSAL.eLinkedListAdd(
            gpsLeagueCtrl->hLeagueList,
            &hExistingLeagueObject,
            psNewLeagueObject);
        if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
        {
            LEAGUE_OBJECT_STRUCT *psExistingLeagueObject;

            // We already found one in our list that looks just like
            // this one. We cannot enter duplicate items in this list.

            // So we need to remove this element, and add the new
            // one as a way to update it, but removed entry needs to stay
            // persistent so we put it into the trash bin.

            // This is the existing league entry we found
            psExistingLeagueObject = (LEAGUE_OBJECT_STRUCT *)
                OSAL.pvLinkedListThis(hExistingLeagueObject);

            // Did the version change?
            if(psExistingLeagueObject->psLeague->n32Version !=
                psNewLeagueObject->psLeague->n32Version)
            {
                // Remove this entry
                OSAL.eLinkedListRemove(hExistingLeagueObject);

                // Move it to the trash bin
                OSAL.eLinkedListAdd(
                    gpsLeagueCtrl->hLeagueTrashBin, 
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, 
                    psExistingLeagueObject
                    );

                // Put new entry in it's place
                OSAL.eLinkedListAdd(
                    gpsLeagueCtrl->hLeagueList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    psNewLeagueObject
                    );

                // Copy team list over to new entry
                OSAL.eLinkedListDelete(psNewLeagueObject->hTeamList);
                psNewLeagueObject->hTeamList =
                    psExistingLeagueObject->hTeamList;
                psExistingLeagueObject->hTeamList = 
                    OSAL_INVALID_OBJECT_HDL;

                // replaced successfully
                SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                    "Replaced league with %s %s (v=%d).\n", 
                    STRING.pacCStr(psNewLeagueObject->psLeague->hName),
                    STRING.pacCStr(psNewLeagueObject->psLeague->hAbbreviation),
                    psNewLeagueObject->psLeague->n32Version
                    );
            }
            else // Don't add it, just use it
            {
                vDestroyLeagueObject(psNewLeagueObject);
                psNewLeagueObject = psExistingLeagueObject;
            }
        }
        else if(eReturnCode != OSAL_SUCCESS)
        {
            // Something went wrong with adding this element. No point
            // in keeping the structure we just created.
            vDestroyLeagueObject(psNewLeagueObject);
            psNewLeagueObject = NULL;
        }
        else
        {
            // added successfully
            SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                "Added league %s %s (v=%d).\n", 
                STRING.pacCStr(psNewLeagueObject->psLeague->hName),
                STRING.pacCStr(psNewLeagueObject->psLeague->hAbbreviation),
                psNewLeagueObject->psLeague->n32Version
                );
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }
    else
    {
        // Now we destroy the entry we created.
        vDestroyLeagueObject(psNewLeagueObject);
        psNewLeagueObject = NULL;
    }

    return (LEAGUE_OBJECT)psNewLeagueObject;
}

/*****************************************************************************
*
*   LEAGUE_bAddTeam
*
*   This function assigns a team to represent this league content. The team
*   is used to uniquely describe this team and map it to some
*   textual information about the team (e.g. name, abbr, etc.).
*
*   Inputs
*       hLeague - A league object to add this team to
*       hTeam - A team object to add
*
*   Outputs:
*       BOOLEAN - A value of TRUE if the team was able to be added.
*       Otherwise returns FALSE if the request resulted if an error occurred.
*
*****************************************************************************/
BOOLEAN LEAGUE_bAddTeam (
    LEAGUE_OBJECT hLeague,
    TEAM_OBJECT hTeam
        )
{
    BOOLEAN bLocked = FALSE, bSuccess = FALSE;
    LEAGUE_TEAM_ENTRY_STRUCT *psEntry = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        LEAGUE_OBJECT_STRUCT const *psLeagueObject =
            (LEAGUE_OBJECT_STRUCT const *)hLeague;
        LEAGUE_TEAM_ID_STRUCT sLeagueTeamId;

        // Verify a TEAM and LEAGUE was provided.
        // Without both these is no point in calling this.
        if ((hLeague == LEAGUE_INVALID_OBJECT) || 
            (hTeam == TEAM_INVALID_OBJECT))
        {
            // Error!
            break;
        }

        // Now, add this team to our list.
        // Lock this object
        bLocked = SMSO_bLock(
            (SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Allocate Team entry
        psEntry = (LEAGUE_TEAM_ENTRY_STRUCT *)
            SMSO_hCreate(
                LEAGUE_OBJECT_NAME": Team Entry", 
                sizeof(LEAGUE_TEAM_ENTRY_STRUCT), 
                SMS_INVALID_OBJECT, 
                FALSE);

        if (psEntry == NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME": Failed to allocate Team Entry");
            break;
        }

        // Get Team Id
        sLeagueTeamId.hTeamId = TEAM.hId(hTeam);
        if (sLeagueTeamId.hTeamId == CID_INVALID_OBJECT)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": Failed to get Team: %s %s Id",
                STRING.pacCStr(TEAM.hAbbreviation(hTeam)),
                STRING.pacCStr(TEAM.hNickname(hTeam)));
            break;
        }

        // Get League Id
        sLeagueTeamId.hLeagueId = LEAGUE.hId(hLeague);
        if (sLeagueTeamId.hLeagueId == CID_INVALID_OBJECT)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": Failed to get League: %s Id",
                STRING.pacCStr(LEAGUE.hAbbreviation(hLeague)));
            break;
        }

        // Create League Team Id
        psEntry->hLeagueTeamId = 
            CID_hCreate(
                CID_POOL_INVALID_OBJECT, 
                CID_LEAGUE_TEAM_ID, 
                &sLeagueTeamId);

        if (psEntry->hLeagueTeamId == CID_INVALID_OBJECT)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": Failed to create League Team Id");
            break;
        }

        // Set corresponding Team object
        psEntry->hTeam = hTeam;

        eReturnCode =
            OSAL.eLinkedListAdd(
            psLeagueObject->hTeamList,
            &hEntry,
            psEntry);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": Failed to add Team to League: %s (#%d)",
                OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            break;
        }

        // added successfully
        SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
            "Added team %s %s to the league %s.\n", 
            STRING.pacCStr(TEAM.hName(hTeam)),
            STRING.pacCStr(TEAM.hNickname(hTeam)),
            STRING.pacCStr(LEAGUE.hName(hLeague)));

        // Now add this league to the team
        bSuccess = TEAM_bAddLeague(hTeam, hLeague);

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }

    // Clean up the mess
    if (bSuccess == FALSE)
    {
        if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            OSAL.eLinkedListRemove(hEntry);
        }

        if (psEntry != NULL)
        {
            vReleaseTeamEntry(psEntry);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   LEAGUE_bRemoveTeam
*
*****************************************************************************/
BOOLEAN LEAGUE_bRemoveTeam(
    LEAGUE_OBJECT hLeague, 
    TEAM_OBJECT hTeam
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    // Verify a TEAM and LEAGUE was provided.
    // Without both these is no point in calling this.
    if ((hLeague == LEAGUE_INVALID_OBJECT) || 
        (hTeam == TEAM_INVALID_OBJECT))
    {
        // Error!
        return FALSE;
    }

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        LEAGUE_OBJECT_STRUCT const *psLeagueObject =
            (LEAGUE_OBJECT_STRUCT const *)hLeague;
        LEAGUE_TEAM_ENTRY_STRUCT sTeamEntry;

        // Set team for search
        sTeamEntry.hTeam = hTeam;
        sTeamEntry.hLeagueTeamId = CID_INVALID_OBJECT;

        eReturnCode = OSAL.eLinkedListSearch(
            psLeagueObject->hTeamList,
            &hEntry,
            (void *)&sTeamEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
            LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry;
 
            psTeamEntry = (LEAGUE_TEAM_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

#if DEBUG_OBJECT==1
            SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                LEAGUE_OBJECT_NAME
                "Original Team: %s %s\n",
                STRING.pacCStr(TEAM.hAbbreviation(hTeam)), 
                STRING.pacCStr(TEAM.hNickname(hTeam))
                    );

            SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                "Removing Team: %s %s from League: %s\n",
                STRING.pacCStr(TEAM.hAbbreviation(psTeamEntry->hTeam)), 
                STRING.pacCStr(TEAM.hNickname(psTeamEntry->hTeam)),
                STRING.pacCStr(LEAGUE.hAbbreviation(hLeague))
                    );
#endif // DEBUG_OBJECT==1

            // Now, remove this team from our list
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode == OSAL_SUCCESS)
            {
                vReleaseTeamEntry(psTeamEntry);
                bSuccess = TRUE;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    LEAGUE_OBJECT_NAME
                    ": Failed to remove Team from League: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), 
                    eReturnCode);
            }
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            bSuccess = TRUE;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": Failed to search Team: %s (#%d)",
                OSAL.pacGetReturnCodeName(eReturnCode), 
                eReturnCode);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   LEAGUE_hFind
*
* Use provided CID to find a LEAGUE object. This function is used to
* determine if a LEAGUE maps to any known league of sports
* teams. If not that's ok, we just cannot map any found teams then. If one
* is found that league object should be used to search for known teams.
*
* Inputs:
*   hLeagueId - A valid CID which represents the league
*
* Outputs:
*   A valid LEAGUE_OBJECT if found or created, otherwise LEAGUE_INVALID_OBJECT.
*
*****************************************************************************/
LEAGUE_OBJECT LEAGUE_hFind (
    CID_OBJECT hLeagueId,
    BOOLEAN bCreateIfNotExists
        )
{
    LEAGUE_OBJECT hLeague = LEAGUE_INVALID_OBJECT;
    BOOLEAN bLocked;

    // Verify inputs
    if(hLeagueId == CID_INVALID_OBJECT)
    {
        // Error! Cannot find leagues with ids we dont have
        return LEAGUE_INVALID_OBJECT;
    }

    // Lock this object
    bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        LEAGUE_OBJECT_STRUCT sDummyLeagueObject;

        // Populate entry for search
        sDummyLeagueObject.hId = hLeagueId;

        // Find League based on provided league id.
        eReturnCode =
            OSAL.eLinkedListSearch(
            gpsLeagueCtrl->hLeagueList, &hEntry,
            (void *)&sDummyLeagueObject
            );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Found the league, extract information into 
            // provided League object
            hLeague = (LEAGUE_OBJECT)OSAL.pvLinkedListThis(hEntry);
        }
        else if ((eReturnCode == OSAL_OBJECT_NOT_FOUND) && 
            (bCreateIfNotExists == TRUE))
        {
            UN32 un32LeagueId;
            void *pvLeagueId = &un32LeagueId;
            LEAGUE_ENUM eLeague;
            char acLeagueName[32], acLeagueAbbrev[32];

            CID_n32GetValue(hLeagueId, &pvLeagueId);

            snprintf(acLeagueName, sizeof(acLeagueName),
                "League #%u", un32LeagueId);
            snprintf(acLeagueAbbrev, sizeof(acLeagueAbbrev),
                "L#%u", un32LeagueId);

            // Find league enum
            eLeague = RADIO_eMapLeague(un32LeagueId);

            // Add this CID with default mapping to our list
            hLeague = LEAGUE_hAdd(
                SPORTS_UNKNOWN, eLeague, 
                hLeagueId, 
                LEAGUE_INVALID_VERSION,
                acLeagueName, acLeagueAbbrev,
                FALSE, 0
                );
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }

    return hLeague;
}

/*****************************************************************************
*
*   LEAGUE_hFindTeam
*
* Retrieves a TEAM object matching this provided CID in the provided LEAGUE.
*
*****************************************************************************/
TEAM_OBJECT LEAGUE_hFindTeam (
    LEAGUE_OBJECT hLeague,
    CID_OBJECT hTeamId,
    BOOLEAN bCreateIfNotExists
        )
{
    TEAM_OBJECT hTeam = TEAM_INVALID_OBJECT;
    BOOLEAN bLocked;

    // Verify inputs
    if(hTeamId == CID_INVALID_OBJECT)
    {
        // Error! Cannot find teams with ids we dont have
        return TEAM_INVALID_OBJECT;
    }

    // Lock this object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        LEAGUE_OBJECT_STRUCT const *psLeagueObject =
            (LEAGUE_OBJECT_STRUCT const *)hLeague;
        LEAGUE_TEAM_ENTRY_STRUCT sTeamEntry;

        // Extract dummy team
        sTeamEntry.hTeam = TEAM_hGetDummyObj(hTeamId);

        eReturnCode = OSAL.eLinkedListSearch(
            psLeagueObject->hTeamList, 
            &hEntry, 
            (void *)&sTeamEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
            LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry;

            // Extract team object
            psTeamEntry = (LEAGUE_TEAM_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

            hTeam = psTeamEntry->hTeam;
        }
        else if ((eReturnCode == OSAL_OBJECT_NOT_FOUND) && 
            (bCreateIfNotExists == TRUE))
        {
            UN32 un32TeamId, un32LeagueId;
            void *pvLeagueId = &un32LeagueId,
                *pvTeamId = &un32TeamId;
            char acTeamNickname[32], acTeamAbbrev[32];
            CID_OBJECT hLeagueId;
            UN8 un8LeagueId;

            // Extract league-id
            hLeagueId = LEAGUE.hId(hLeague);

            // Extract league-id value
            CID_n32GetValue(hLeagueId, &pvLeagueId);
            un8LeagueId = (UN8)un32LeagueId;

            // Extract team-id value
            CID_n32GetValue(hTeamId, &pvTeamId);

            snprintf(acTeamNickname, sizeof(acTeamNickname),
                "Team #%u", un32TeamId);
            snprintf(acTeamAbbrev, sizeof(acTeamAbbrev),
                "T#%u", un32TeamId);

            // Add this CID with default mapping to our list
            hTeam = TEAM_hAdd(hTeamId, TEAM_INVALID_VERSION,
                "Unknown", acTeamNickname, acTeamAbbrev,
                1, &un8LeagueId, 0, NULL
                    );
        }
        else
        {
            // No dice
            hTeam = TEAM_INVALID_OBJECT;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
    }

    return hTeam;
}

/*****************************************************************************
*
*   LEAGUE_bIterateTeams
*
*****************************************************************************/
BOOLEAN LEAGUE_bIterateTeams (
    LEAGUE_OBJECT hLeague,
    BOOLEAN bLockMaster,
    TEAM_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
        )
{
    BOOLEAN bRetval = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    LEAGUE_TEAM_ITERATOR_STRUCT sIterator;
    BOOLEAN bLocked;

    // Check inputs
    if ((hLeague == LEAGUE_INVALID_OBJECT) || 
        (bContentIteratorCallback == NULL))
    {
        // Bad param
        return FALSE;
    }

    // Populate iterator structure with info needed to call the provided
    // callback to the caller.
    sIterator.bContentIteratorCallback = bContentIteratorCallback;
    sIterator.pvContentIteratorCallbackArg = pvContentIteratorCallbackArg;

    // Try to lock master lockable object first
    bLocked = (bLockMaster == TRUE) ? bMasterLock() : TRUE;
    if (bLocked == TRUE)
    {
        // Lock this object
        bLocked = SMSO_bLock((SMS_OBJECT)
            gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            LEAGUE_OBJECT_STRUCT const *psLeagueObject =
                (LEAGUE_OBJECT_STRUCT const *)hLeague;

            // Iterate the list and execute caller's callback function for
            // each entry in the list.
            eReturnCode =
                OSAL.eLinkedListIterate(
                    psLeagueObject->hTeamList,
                    (OSAL_LL_ITERATOR_HANDLER)bTeamIterator,
                    &sIterator);

            if ((eReturnCode == OSAL_SUCCESS) ||
                (eReturnCode == OSAL_NO_OBJECTS))
            {
                // List was iterated
                bRetval = TRUE;
            }

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
        }

        if (bLockMaster == TRUE)
        {
            vMasterUnlock();
        }
    }

    return bRetval;
}

/*****************************************************************************
*
*   LEAGUE_bIterateContent
*
*****************************************************************************/
BOOLEAN LEAGUE_bIterateContent (
    LEAGUE_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
        )
{
    BOOLEAN bRetval = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    LEAGUE_ITERATOR_STRUCT sIterator;
    BOOLEAN bLocked;

    // Check inputs
    if(bContentIteratorCallback == NULL)
    {
        // Bad param
        return FALSE;
    }

    // Populate iterator structure with info needed to call the provided
    // callback to the caller.
    sIterator.bContentIteratorCallback = bContentIteratorCallback;
    sIterator.pvContentIteratorCallbackArg = pvContentIteratorCallbackArg;

    // Lock this object
   bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
   if(bLocked == TRUE)
   {
       // Iterate the list and execute caller's callback function for
       // each entry in the list.
       eReturnCode =
           OSAL.eLinkedListIterate(
           gpsLeagueCtrl->hLeagueList,
           (OSAL_LL_ITERATOR_HANDLER)bIterator, &sIterator);
       if((eReturnCode == OSAL_SUCCESS) ||
           (eReturnCode == OSAL_NO_OBJECTS))
       {
           // List was iterated
           bRetval = TRUE;
       }

       // Unlock object
       SMSO_vUnlock((SMS_OBJECT)gpsLeagueCtrl);
   }

    return bRetval;
}

/*****************************************************************************
*
*   LEAGUE_bLoad
*
*  This is a function used to load saved leagues from the db.
*
*****************************************************************************/
BOOLEAN LEAGUE_bLoad (
    CID_ENUM eType
    )
{
    BOOLEAN bReturn = FALSE;
    size_t tLeaguesLoaded = 0;
    UN32 un32Seconds;
    UN16 un16Msecs;
    UN32 un32LoadStartMsec, un32LoadEndMsec;

    // set the methods that will be used to create and modify league cids
    gpsLeagueCtrl->eType = eType;

    // Create a working league id
    if(gpsLeagueCtrl->hWorkingLeagueId == CID_INVALID_OBJECT)
    {
        UN32 un32LeagueId = 0;

        gpsLeagueCtrl->hWorkingLeagueId = 
            LEAGUE_hCreateCid(
            CID_POOL_INVALID_OBJECT,
            &un32LeagueId);
        if(gpsLeagueCtrl->hWorkingLeagueId == CID_INVALID_OBJECT)
        {
            // Error!
            return FALSE;
        }
    }

    OSAL.vTimeUp(&un32Seconds, &un16Msecs);
    un32LoadStartMsec = ((un32Seconds * 1000) + un16Msecs);

    bReturn = SQL_INTERFACE.bStartTransaction(gpsLeagueCtrl->hSQLConnection);
    if (bReturn == TRUE)
    {
        // get all leagues. LEAGUEs created in bLeagueTableHandler
        bReturn = SQL_INTERFACE.bQuery (
            gpsLeagueCtrl->hSQLConnection,
            QUERY_FOR_ALL_LEAGUES,
            bLeagueTableHandler,
            &tLeaguesLoaded
            );

        SQL_INTERFACE.bEndTransaction(gpsLeagueCtrl->hSQLConnection);
    }

    OSAL.vTimeUp(&un32Seconds, &un16Msecs);
    un32LoadEndMsec = ((un32Seconds * 1000) + un16Msecs);

    SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
        PC_BLUE"Loaded %u leagues (%u msec).\n"PC_RESET,
        tLeaguesLoaded, un32LoadEndMsec - un32LoadStartMsec);

    return bReturn;
}

/*****************************************************************************
*
* LEAGUE_un32Count
*
*****************************************************************************/
UN32 LEAGUE_un32Count ( 
    N32 n32Version, 
    UN32 *pun32Total, 
    UN32 un32Expected
    )
{
    LEAGUE_VERSION_MATCH_ITERATOR_STRUCT sMatching;
    UN32 un32Total;

    sMatching.n32Version = n32Version;
    sMatching.un32Matching = 0;

    if(pun32Total == NULL)
    {
        // Use local
        pun32Total = &un32Total;
    }

    // Note: There is no need for exclusive access since the only
    // context able to update the list, is the same which wants the count.
    OSAL.eLinkedListIterate(
        gpsLeagueCtrl->hLeagueList,
        (OSAL_LL_ITERATOR_HANDLER)bCountMatchingVersion, &sMatching);

    *pun32Total = 0;
    OSAL.eLinkedListItems(gpsLeagueCtrl->hLeagueList, pun32Total);

    // Check if the number expected matches the number found.
    if(sMatching.un32Matching == un32Expected)
    {
        gpsLeagueCtrl->bComplete = TRUE;
    }
    else
    {
        gpsLeagueCtrl->bComplete = FALSE;
    }

    gpsLeagueCtrl->n32Version = n32Version;

    return sMatching.un32Matching;
}

/*****************************************************************************
*
* LEAGUE_bSave
*
*****************************************************************************/
BOOLEAN LEAGUE_bSave ( N32 n32Version )
{
    BOOLEAN bResult;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    LEAGUE_SAVE_ITERATOR_STRUCT sLeagueSaveIterator;

    if (gpsLeagueCtrl->hSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        // No error. Means, no file system.
        return TRUE;
    }

    bResult = SQL_INTERFACE.bStartTransaction(gpsLeagueCtrl->hSQLConnection);
    if (bResult == TRUE)
    {
        // Get rid of any entries which are not this one.
        bResult = bPrune(n32Version);

        // Iterate leagues and save them
        sLeagueSaveIterator.n32Version = n32Version;
        sLeagueSaveIterator.tInsertCount = 0;
        sLeagueSaveIterator.tReplacedCount = 0;
        sLeagueSaveIterator.tAlreadyUpToDateCount = 0;

        // Iterate leagues and save them

        // Note: There is no need for exclusive access since the only
        // context able to update the list, is the same which wants to save it.
        eReturnCode = OSAL.eLinkedListIterate(
            gpsLeagueCtrl->hLeagueList,
            (OSAL_LL_ITERATOR_HANDLER)bSaveLeague, &sLeagueSaveIterator);
        if((eReturnCode == OSAL_SUCCESS) || (eReturnCode == OSAL_NO_OBJECTS))
        {
            BOOLEAN bSuccess;

            // All saved
            SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                "%u leagues saved with version v%d.\n"
                "%u inserted, %u replaced, %u already up-to-date.\n",
                sLeagueSaveIterator.tInsertCount +
                    sLeagueSaveIterator.tReplacedCount +
                        sLeagueSaveIterator.tAlreadyUpToDateCount,
                n32Version,
                sLeagueSaveIterator.tInsertCount,
                sLeagueSaveIterator.tReplacedCount,
                sLeagueSaveIterator.tAlreadyUpToDateCount);

            // Write version info

            // Build the version string
            snprintf(&gpsLeagueCtrl->acQueryBuffer[0],
                sizeof(gpsLeagueCtrl->acQueryBuffer),
                UPDATE_VERSION,
                gpsLeagueCtrl->n32Version, gpsLeagueCtrl->bComplete
                );

            bSuccess = SQL_INTERFACE.bExecuteCommand(
                gpsLeagueCtrl->hSQLConnection,
                &gpsLeagueCtrl->acQueryBuffer[0]);

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    LEAGUE_OBJECT_NAME
                    ": failed to set league table version");
            }
            else
            {
                bResult = TRUE;
            }
        }

        SQL_INTERFACE.bEndTransaction(gpsLeagueCtrl->hSQLConnection);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            LEAGUE_OBJECT_NAME
            ": failed to start transaction to recreate persistent storage");
    }

    return bResult;
}

/*****************************************************************************
*
* LEAGUE_hCreateCid
*
*****************************************************************************/
CID_OBJECT LEAGUE_hCreateCid(     
    CID_POOL hCidPool,
    const void *pvSrcObjectData
    )
{
    CID_OBJECT hLeague;

    hLeague = 
        CID_hCreate(
        hCidPool, gpsLeagueCtrl->eType, pvSrcObjectData);

    return hLeague;
}

/*****************************************************************************
*
* LEAGUE_bPruneTeams
*
*****************************************************************************/
BOOLEAN LEAGUE_bPruneTeams(
    LEAGUE_OBJECT hLeague, 
    N32 n32Version
        )
{
    N32 n32TeamVersion;
    LEAGUE_OBJECT_STRUCT *psLeague = 
        (LEAGUE_OBJECT_STRUCT *)hLeague;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hCurrent, 
        hRemove = OSAL_INVALID_LINKED_LIST_ENTRY;
    LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry = NULL;

    hCurrent = OSAL.hLinkedListFirst(
        psLeague->hTeamList, (void **)&psTeamEntry);

    while ((hCurrent != OSAL_INVALID_LINKED_LIST_ENTRY) &&
           (psTeamEntry != NULL))
    {
        n32TeamVersion = TEAM.n32Version(psTeamEntry->hTeam);
        if (n32TeamVersion != n32Version)
        {
            SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                "%s - removing Team: %s, v%d\n",
                STRING.pacCStr(LEAGUE.hAbbreviation(hLeague)), 
                STRING.pacCStr(TEAM.hNickname(psTeamEntry->hTeam)),
                n32TeamVersion);

            hRemove = hCurrent;
        }

        hCurrent = OSAL.hLinkedListNext(
            hCurrent, (void **)&psTeamEntry);

        if (hRemove != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            LEAGUE_TEAM_ENTRY_STRUCT *psEntry;

            psEntry = (LEAGUE_TEAM_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hRemove);

            eReturnCode = OSAL.eLinkedListRemove(hRemove);
            if (eReturnCode == OSAL_SUCCESS)
            {
                vReleaseTeamEntry(psEntry);
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    LEAGUE_OBJECT_NAME
                    ": Failed to remove Team from League's Team list");
                return FALSE;
            }

            hRemove = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
* LEAGUE_un8SportsFlashTiers
*
*****************************************************************************/
UN8 LEAGUE_un8SportsFlashTiers(
    LEAGUE_OBJECT hLeague
        )
{
    UN8 un8Tiers = 0;

    LEAGUE_OBJECT_STRUCT *psLeagueObj =
        (LEAGUE_OBJECT_STRUCT *)hLeague;

    if (psLeagueObj != NULL)
    {
        un8Tiers = psLeagueObj->psLeague->un8SportsFlashTiers;
    }

    return un8Tiers;
}

/*****************************************************************************
PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroyLeagueObject
*
* Destroy a league entry (an entry in the list)
*
*****************************************************************************/
static void vDestroyLeagueObject (
    LEAGUE_OBJECT_STRUCT *psLeagueObject
    )
{
    // Check if we have a CID
    if(psLeagueObject->hId != CID_INVALID_OBJECT)
    {
        // Destroy CID
        CID_vDestroy(psLeagueObject->hId);
        psLeagueObject->hId = CID_INVALID_OBJECT;
    }

    if (psLeagueObject->psLeague != NULL)
    {
        LEAGUE_STRUCT *psLeagueStruct = psLeagueObject->psLeague;

        if (psLeagueStruct->hAbbreviation != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psLeagueStruct->hAbbreviation);
            psLeagueStruct->hAbbreviation = STRING_INVALID_OBJECT;
        }

        if (psLeagueStruct->hName != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psLeagueStruct->hName);
            psLeagueStruct->hName = STRING_INVALID_OBJECT;
        }
    }

    // Remove any and all teams
    if(psLeagueObject->hTeamList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Remove all teams first
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psLeagueObject->hTeamList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseTeamEntry
                );
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Delete team list
            eReturnCode = OSAL.eLinkedListDelete(
                psLeagueObject->hTeamList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                psLeagueObject->hTeamList =
                    OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Destroy object
    OSAL.vLinkedListMemoryFree((void *)psLeagueObject);

    return;
}

/*****************************************************************************
*
*   bIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* league iterator structure to call the caller's own iterator function along
* with an argument for each league in the list.
*
*****************************************************************************/
static BOOLEAN bIterator (
    LEAGUE_OBJECT_STRUCT const *psLeagueObject,
    LEAGUE_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bReturn = TRUE;

    // Check inputs
    if ((psLeagueObject == NULL) || (psIterator == NULL))
    {
        // Cease iterating
        return FALSE;
    }

    // Call provided iterator function if one exists
    if (psIterator->bContentIteratorCallback != NULL)
    {
        // Return whatever their callback returns back to the iterator
        bReturn = psIterator->bContentIteratorCallback(
            (LEAGUE_OBJECT)psLeagueObject,
            psIterator->pvContentIteratorCallbackArg
                );
    }

    return bReturn;
}

/*****************************************************************************
*
*   bTeamIterator
*
*****************************************************************************/
static BOOLEAN bTeamIterator (
    LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry,
    LEAGUE_TEAM_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bReturn = TRUE;

    // Check inputs
    if ((psTeamEntry == NULL) || (psIterator == NULL))
    {
        // Cease iterating
        return FALSE;
    }

    // Call provided iterator function if one exists
    if (psIterator->bContentIteratorCallback != NULL)
    {
        // Return whatever their callback returns back to the iterator
        bReturn = psIterator->bContentIteratorCallback(
            psTeamEntry->hTeam,
            psIterator->pvContentIteratorCallbackArg
                );
    }

    return bReturn;
}

/*****************************************************************************
*
*   bCountMatchingVersion
*
*****************************************************************************/
static BOOLEAN bCountMatchingVersion (
    LEAGUE_OBJECT_STRUCT const *psLeagueObject,
    LEAGUE_VERSION_MATCH_ITERATOR_STRUCT *psMatching
    )
{
    // Check inputs
    if((psLeagueObject != NULL) && (psMatching != NULL))
    {
        if(psLeagueObject->psLeague->n32Version == psMatching->n32Version)
        {
            psMatching->un32Matching++;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bSaveLeague
*
*****************************************************************************/
static BOOLEAN bSaveLeague (
    LEAGUE_OBJECT_STRUCT *psLeagueObject,
    LEAGUE_SAVE_ITERATOR_STRUCT *psLeagueSaveIterator
    )
{
    BOOLEAN bSuccess = TRUE;

    // Check inputs
    if((psLeagueObject != NULL) && (psLeagueSaveIterator != NULL))
    {
        N32 n32Version;

        // If the input version is 'invalid', this indicates
        // we want to store everything. However we will only
        // store leagues which have a valid version.

        // Does this league have a valid version? Has it not
        // already been saved?
        if((psLeagueObject->psLeague->n32Version != TEAM_INVALID_VERSION) &&
            (psLeagueObject->bSaved == FALSE))
        {
            n32Version = psLeagueSaveIterator->n32Version;

            if(psLeagueSaveIterator->n32Version == TEAM_INVALID_VERSION)
            {
                // Force save
                n32Version = psLeagueObject->psLeague->n32Version;
            }

            // Save this league if it is the same as the 
            // provided version.
            if(n32Version == psLeagueObject->psLeague->n32Version)
            {
                BOOLEAN bExists;
                LEAGUE_ROW_STRUCT sLeagueRow;
                N32 n32SavedVersion;
                void *pvLeagueId = &sLeagueRow.un32LeagueId;

                // Get raw league id
                CID_n32GetValue(psLeagueObject->hId, &pvLeagueId);

                // populate remaining part of the structure
                sLeagueRow.n32Version = 
                    psLeagueObject->psLeague->n32Version;
                sLeagueRow.un32SportId = 
                    (UN32)psLeagueObject->psLeague->eSport;
                sLeagueRow.pacName = 
                    STRING.pacCStr(psLeagueObject->psLeague->hName);
                sLeagueRow.pacAbbrev =
                    STRING.pacCStr(psLeagueObject->psLeague->hAbbreviation);
                sLeagueRow.un32SportsFlashEnabled =
                    (UN32)psLeagueObject->psLeague->bSportsFlashEnabled;
                sLeagueRow.un32SportsFlashTiers =
                    (UN32)psLeagueObject->psLeague->un8SportsFlashTiers;

                bExists = bQueryLeagueVersion(
                    sLeagueRow.un32LeagueId,
                    &n32SavedVersion, &sLeagueRow.n64RowId);
                if (FALSE == bExists)
                {
                    SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                        "Inserting League: %s(%s)\n",
                        sLeagueRow.pacName,
                        sLeagueRow.pacAbbrev);

                    bSuccess = psLeagueObject->bSaved = 
                        SQL_INTERFACE.bExecutePreparedCommand(
                        gpsLeagueCtrl->hSQLConnection,
                        INSERT_LEAGUE,
                        LEAGUE_MAX_FIELDS,
                        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareLeagueColumn,
                        (void *)&sLeagueRow);
                    if(bSuccess == TRUE)
                    {
                        psLeagueSaveIterator->tInsertCount++;
                    }
                }
                else if (sLeagueRow.n32Version != n32SavedVersion)
                {
                    SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                        "Updating League: %s(%s)\n", 
                        sLeagueRow.pacName,
                        sLeagueRow.pacAbbrev);

                    bSuccess = psLeagueObject->bSaved = 
                        SQL_INTERFACE.bExecutePreparedCommand(
                        gpsLeagueCtrl->hSQLConnection,
                        UPDATE_LEAGUE,
                        LEAGUE_MAX_FIELDS + 1, // +1 because ROWID is used
                        (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareLeagueColumn,
                        (void *)&sLeagueRow);
                    if(bSuccess == TRUE)
                    {
                        psLeagueSaveIterator->tReplacedCount++;
                    }
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                        "League %s(%s) is already up to date\n",
                        sLeagueRow.pacName, sLeagueRow.pacAbbrev);
                    psLeagueSaveIterator->tAlreadyUpToDateCount++;
                }
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
* psCreateLeagueObject
*
*****************************************************************************/
static LEAGUE_OBJECT_STRUCT *psCreateLeagueObject (
    SPORTS_ENUM eSport,
    LEAGUE_ENUM eLeague,
    CID_OBJECT hId,
    N32 n32Version,
    const char *pacName,
    const char *pacAbbreviation,
    BOOLEAN bSportsFlashEnabled,
    UN8 un8SportsFlashTiers
        )
{
    size_t tSize = sizeof(LEAGUE_OBJECT_STRUCT) + sizeof(LEAGUE_STRUCT);
    LEAGUE_OBJECT_STRUCT *psLeagueObject = NULL;
    LEAGUE_STRUCT *psLocalLeagueStruct;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    char acLeagueAbbrev[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;

    snprintf(&acName[0], sizeof(acName), 
        LEAGUE_OBJECT_NAME":%s", pacAbbreviation);

    // Now allocate a structure to hold the CID and market as one mapping.
    // This entry is available for all SMS objects and applications to use.
    psLeagueObject = (LEAGUE_OBJECT_STRUCT *)
        OSAL.pvLinkedListMemoryAllocate(
        &acName[0],
        tSize,
        FALSE);

    if (psLeagueObject == NULL)
    {
        // Error! Unable to create
        return NULL;
    }

    // Object is not saved
    psLeagueObject->bSaved = FALSE;

    // Populate new LEAGUE object with provided inputs.
    psLeagueObject->hId = CID.hDuplicate(hId);

    // Initialize
    psLocalLeagueStruct = (LEAGUE_STRUCT *)(psLeagueObject + 1);

    psLocalLeagueStruct->n32Version = n32Version;
    psLocalLeagueStruct->eLeague = eLeague;
    psLocalLeagueStruct->eSport = eSport;
    psLocalLeagueStruct->hAbbreviation = STRING.hCreate(pacAbbreviation, 0);
    psLocalLeagueStruct->hName = STRING.hCreate(pacName, 0);
    psLocalLeagueStruct->bSportsFlashEnabled = bSportsFlashEnabled;
    psLocalLeagueStruct->un8SportsFlashTiers = un8SportsFlashTiers;
    psLeagueObject->psLeague = psLocalLeagueStruct;

    // Construct team list
    STRING.tCopyToCStr(psLeagueObject->psLeague->hAbbreviation,
        &acLeagueAbbrev[0], sizeof(acLeagueAbbrev));

    // Construct a unique name for this list.
    snprintf(&acName[0],
        sizeof(acName),
        LEAGUE_OBJECT_NAME":%s:TeamList",
        acLeagueAbbrev);

    // Create a team list
    eReturnCode = OSAL.eLinkedListCreate(
        &psLeagueObject->hTeamList,
        acName,
        (OSAL_LL_COMPARE_HANDLER)n16CompareTeams,
        OSAL_LL_OPTION_RBTREE
            );

    if (eReturnCode != OSAL_SUCCESS)
    {
        // We could not create a team list
        vDestroyLeagueObject(psLeagueObject);
        psLeagueObject = NULL;
    }

    return psLeagueObject;
}

/*****************************************************************************
*
* bQueryLeagueVersion
*
*****************************************************************************/
static BOOLEAN bQueryLeagueVersion (
    UN32 un32LeagueId,
    N32 *pn32Version,
    N64 *pn64RowId
    )
{
    LEAGUE_VERSION_QUERY_STRUCT sQuery;

    sQuery.bFound = FALSE;

    snprintf(&gpsLeagueCtrl->acQueryBuffer[0], LEAGUE_MAX_QUERY_LENGTH,
        QUERY_LEAGUE_VERSION, un32LeagueId);

    SQL_INTERFACE.bQuery(
        gpsLeagueCtrl->hSQLConnection,
        &gpsLeagueCtrl->acQueryBuffer[0],
        bLeagueVersionHandler,
        (void *)&sQuery);

    if (TRUE == sQuery.bFound)
    {
        *pn32Version = sQuery.n32Version;
        *pn64RowId = sQuery.n64RowId;
    }

    return sQuery.bFound;
}

/*****************************************************************************
*
* bLeagueVersionHandler
*
*****************************************************************************/
static BOOLEAN bLeagueVersionHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    LEAGUE_VERSION_QUERY_STRUCT *psQuery = 
        (LEAGUE_VERSION_QUERY_STRUCT *)pvArg;

    // Waiting for 2 columns: 0 - Version, 1 - rowid
    // see QUERY_LEAGUE_VERSION statement
    if (2 == n32NumberOfColumns)
    {
        psQuery->n32Version = (N32)psColumns[0].uData.sUN32.un32Data;
        psQuery->n64RowId = psColumns[1].n64NativeInteger;
        psQuery->bFound = TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
* bLeagueTableHandler
*
*****************************************************************************/
static BOOLEAN bLeagueTableHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    size_t *ptNumLeagues = (size_t *)pvArg;
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    char *pacName = NULL, *pacAbbreviation = NULL;
    LEAGUE_ENUM eLeague = LEAGUE_UNKNOWN;
    SPORTS_ENUM eSport = SPORTS_UNKNOWN;
    UN8 un8CurrentField = 0;
    N32 n32Version = 0;
    LEAGUE_OBJECT hLeague;
    BOOLEAN bSportsFlashEnabled = FALSE;
    UN8 un8SportsFlashTiers = 0;

    do
    {
        // Grab the current column data
        psCurrentCol = &psColumns[un8CurrentField];

        // Decode the current field based on it's type
        switch (un8CurrentField)
        {
            case LEAGUE_FIELD_ID:
            {
                CID_bModify(
                    &gpsLeagueCtrl->hWorkingLeagueId,
                    &psCurrentCol->uData.sUN32.un32Data);

                eLeague = RADIO_eMapLeague(
                    psCurrentCol->uData.sUN32.un32Data);
            }
            break;

            case LEAGUE_FIELD_NAME:
            {
                pacName = 
                    (char *)psCurrentCol->uData.sCString.pcData;
            }
            break;

            case LEAGUE_FIELD_ABBREV:
            {
                pacAbbreviation = 
                    (char *)psCurrentCol->uData.sCString.pcData;
            }
            break;

            case LEAGUE_FIELD_SPORT:
            {
                // map the sport to an enum
                eSport = 
                    (SPORTS_ENUM)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case LEAGUE_FIELD_VERSION:
            {
                n32Version = (N32)psCurrentCol->uData.sUN32.un32Data;
                SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
                    "League version: %d\n", n32Version);
            }
            break;

            case LEAGUE_FIELD_SF_ENABLED:
            {
                // SportsFlash Enabled value
                bSportsFlashEnabled =
                    (BOOLEAN)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            case LEAGUE_FIELD_SF_TIERS:
            {
                // SportsFlash Tiers value
                un8SportsFlashTiers =
                    (UN8)psCurrentCol->uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }

    } while ( ++un8CurrentField < LEAGUE_MAX_FIELDS);

    SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
        "Loading %s (%s) v%d league.\n",
        pacName, pacAbbreviation, n32Version);

    // we processed all the fields, now create the league
    hLeague = LEAGUE_hAdd(eSport, eLeague, 
        gpsLeagueCtrl->hWorkingLeagueId, 
        n32Version, pacName, pacAbbreviation,
        bSportsFlashEnabled, un8SportsFlashTiers);

    // Any league added here is by definition 'saved'
    if(hLeague != LEAGUE_INVALID_OBJECT)
    {
        LEAGUE_OBJECT_STRUCT *psLeagueObject = 
            (LEAGUE_OBJECT_STRUCT *)hLeague;
        psLeagueObject->bSaved = TRUE;
        (*ptNumLeagues)++;
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bPrepareLeagueColumn
*
*****************************************************************************/
static BOOLEAN bPrepareLeagueColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    LEAGUE_ROW_STRUCT *psLeagueRow
    )
{
    BOOLEAN bSuccess = TRUE;

    switch (tIndex)
    {
    case LEAGUE_FIELD_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLeagueRow->un32LeagueId;
        }
        break;

    case LEAGUE_FIELD_VERSION:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLeagueRow->n32Version;
        }
        break;

    case LEAGUE_FIELD_ABBREV:
        {
            *peType = SQL_BIND_TYPE_C_STRING;
            *ppvData = (void *)psLeagueRow->pacAbbrev;
        }
        break;

    case LEAGUE_FIELD_NAME:
        {
            *peType = SQL_BIND_TYPE_C_STRING;
            *ppvData = (void *)psLeagueRow->pacName;
        }
        break;

    case LEAGUE_FIELD_SPORT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLeagueRow->un32SportId;
        }
        break;

    case LEAGUE_FIELD_SF_ENABLED:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLeagueRow->un32SportsFlashEnabled;
        }
        break;

    case LEAGUE_FIELD_SF_TIERS:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psLeagueRow->un32SportsFlashTiers;
        }
        break;

    case LEAGUE_MAX_FIELDS:
        {
            // this is a special case for virtual column ROWID
            *peType = SQL_BIND_TYPE_N64;
            *ppvData = (void*)&psLeagueRow->n64RowId;
        }
        break;

    default:
        bSuccess = FALSE;
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bCreateDBTables
*
*****************************************************************************/
static BOOLEAN bCreateDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
    )
{
    BOOLEAN bSuccess = FALSE, bTransactionStarted = FALSE;

    do
    {
        bTransactionStarted = SQL_INTERFACE.bStartTransaction(hSQLConnection);

        if (bTransactionStarted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to start transaction to recreate persistant storage");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to create leagues version table");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            INSERT_VERSION_ROW);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to set initial db league version");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            CREATE_LEAGUE_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to create league table");
            break;
        }

        // Build Teams
        bSuccess = TEAM_bCreateDBTables(hSQLConnection, pvArg);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to create team tables");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            SPORTS_DB_CREATE_SCHEMA_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to create schema version table");
            break;
        }

        // Build the version string
        snprintf(&gpsLeagueCtrl->acQueryBuffer[0],
            sizeof(gpsLeagueCtrl->acQueryBuffer),
            SPORTS_DB_INSERT_VERSION_ROW,
            SPORTS_DB_SCHEMA_VER
            );

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            &gpsLeagueCtrl->acQueryBuffer[0]);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to set initial db schema version");
            break;
        }
    }
    while (FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        bTransactionEnded = SQL_INTERFACE.bEndTransaction(hSQLConnection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                LEAGUE_OBJECT_NAME
                ": failed to end transaction");

            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bVerifyDBVersion
*
*****************************************************************************/
static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
    )
{
    BOOLEAN bOk, bVersionMatched = FALSE;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        SPORTS_DB_SELECT_SCHEMA_VERSION,
        bProcessSelectDBVersionResult,
        &bVersionMatched
        );

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            LEAGUE_OBJECT_NAME
            ": failed to get version row from table");
        return FALSE;
    }

    bOk = SQL_INTERFACE.bQuery(
        hSQLConnection,
        SELECT_VERSION,
        bProcessSelectTableVersionResult,
        NULL
        );

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            LEAGUE_OBJECT_NAME
            ": failed to get table version row from table");
        return FALSE;
    }

    return bVersionMatched;
}

/*******************************************************************************
*
*   bProcessSelectDBVersionResult
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    BOOLEAN *pbVersionMatched = (BOOLEAN *)pvArg;
    UN8 un8Version = 0;

    if (pbVersionMatched == NULL)
    {
        return FALSE;
    }

    // Init our return value
    *pbVersionMatched = FALSE;

    if ( (n32NumberOfColumns == 1) &&
        (psColumns[0].eType == SQL_COLUMN_TYPE_INTEGER) )
    {
        // There is only one column, and it should be a UN32
        un8Version =
            (UN8)psColumns[0].uData.sUN32.un32Data;

        if (un8Version == SPORTS_DB_SCHEMA_VER)
        {
            *pbVersionMatched = TRUE;
        }
    }

    // Return false since we don't need to iterate through the table
    // should only be one row
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectTableVersionResult
*
*******************************************************************************/
static BOOLEAN bProcessSelectTableVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    )
{
    gpsLeagueCtrl->n32Version = LEAGUE_INVALID_VERSION;
    gpsLeagueCtrl->bComplete = FALSE;

    if (n32NumberOfColumns == 2)
    {
        gpsLeagueCtrl->n32Version = (N32)psColumns[0].uData.sUN32.un32Data;
        gpsLeagueCtrl->bComplete = (BOOLEAN)psColumns[1].uData.sUN32.un32Data;
    }

    // Return false since we don't need to iterate through the table
    // should only be one row
    return FALSE;
}

/*****************************************************************************
*
* bPrune
*
*****************************************************************************/
BOOLEAN bPrune(N32 n32Version)
{
    BOOLEAN bOk = FALSE;

    if (n32Version != LEAGUE_INVALID_VERSION)
    {
        snprintf(&gpsLeagueCtrl->acQueryBuffer[0], LEAGUE_MAX_QUERY_LENGTH,
            PRUNE_LEAGUES,  n32Version);

        bOk = SQL_INTERFACE.bExecuteCommand (
            gpsLeagueCtrl->hSQLConnection,
            &gpsLeagueCtrl->acQueryBuffer[0]);
        if(bOk == TRUE)
        {
            gpsLeagueCtrl->n32Version = n32Version;
        }
    }

    // Check purge results
    if (TRUE == bOk)
    {
        SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
            "Leagues with version not equal to #%d"
            " have been purged.\n", n32Version);
    }
    else
    {
        SMSAPI_DEBUG_vPrint(LEAGUE_OBJECT_NAME, 4,
            "Leagues, nothing has been purged.\n");
    }

    return bOk;
}

/*****************************************************************************
*
*   n16CompareIds
*
*       This is the function used to compare existing LEAGUE objects
*       to each other. Specifically this function is used to perform a
*       relational comparison for sorting. This sorter keeps the LEAGUEs
*       in order based on each LEAGUE's id.
*
*       Outputs:
*               0   - LEAGUEs have the same value (equal)
*               > 0 - LEAGUE1 is greater than (after) LEAGUE2
*               < 0 - LEAGUE1 is less than (before) LEAGUE2 or error
*
*****************************************************************************/
static N16 n16CompareIds (
    LEAGUE_OBJECT hLeague1,
    LEAGUE_OBJECT hLeague2
        )
{
    N16 n16Result = N16_MIN;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject1 = 
        (LEAGUE_OBJECT_STRUCT const *)hLeague1;
    LEAGUE_OBJECT_STRUCT const *psLeagueObject2 = 
        (LEAGUE_OBJECT_STRUCT const *)hLeague2;

    // Verify inputs
    if( (psLeagueObject1 != NULL) && (psLeagueObject2 != NULL) )
    {
        // Check CID relationship
        n16Result = CID.n16Compare(psLeagueObject1->hId,
            psLeagueObject2->hId);
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16CompareTeams
*
*****************************************************************************/
static N16 n16CompareTeams(
    void *pvObj1, 
    void *pvObj2
        )
{
    N16 n16Result = N16_MIN;
    LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry1 = 
        (LEAGUE_TEAM_ENTRY_STRUCT *)pvObj1;
    LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry2 = 
        (LEAGUE_TEAM_ENTRY_STRUCT *)pvObj2;

    // Verify inputs
    if ((psTeamEntry1 != NULL) && (psTeamEntry2 != NULL))
    {
        n16Result = TEAM_n16CompareIds(
            psTeamEntry1->hTeam, psTeamEntry2->hTeam);
    }

    return n16Result;
}

/*****************************************************************************
*
*   vReleaseTeamEntry
*
*****************************************************************************/
static void vReleaseTeamEntry(
    void *pvData
        )
{
    LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry = 
        (LEAGUE_TEAM_ENTRY_STRUCT *)pvData;

    if (psTeamEntry != NULL)
    {
        if (psTeamEntry->hLeagueTeamId != CID_INVALID_OBJECT)
        {
            CID_vDestroy(psTeamEntry->hLeagueTeamId);
        }

        // Do not destroy associated with this entry Team object
        // It will be released as part of the global Team list removal

        SMSO_vDestroy((SMS_OBJECT)psTeamEntry);
    }

    return;
}

/*****************************************************************************
*
*   bSearchForLeague
*
*****************************************************************************/
static BOOLEAN bSearchForLeague(
    LEAGUE_OBJECT hLeague,
    void *pvArg
        )
{
    N16 n16Compare;
    CID_OBJECT hLeagueId;
    LEAGUE_SEARCH_STRUCT *psSearch = (LEAGUE_SEARCH_STRUCT *)pvArg;

    hLeagueId = LEAGUE.hId(hLeague);
    if (hLeagueId == CID_INVALID_OBJECT)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            LEAGUE_OBJECT_NAME
            ": Failed to get a CID object from League: %s",
            STRING.pacCStr(LEAGUE.hAbbreviation(hLeague)));

        // Stop immediately
        return FALSE;
    }

    n16Compare = CID.n16Compare(psSearch->hLeagueId, hLeagueId);
    if (n16Compare == 0)
    {
        // Found it!
        psSearch->bFound = TRUE;
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bMasterLock
*
*****************************************************************************/
static BOOLEAN bMasterLock( void )
{
    BOOLEAN bLocked;
    SMS_OBJECT hObject = SMS_INVALID_OBJECT;

    bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        hObject = gpsLeagueCtrl->hMasterLockableObject;

        SMSO_vUnlock((SMS_OBJECT) gpsLeagueCtrl);

        // if master lockable object is absent tell caller we're locked
        if (hObject != SMS_INVALID_OBJECT)
        {
            bLocked =
                SMSO_bLock((SMS_OBJECT) hObject, OSAL_OBJ_TIMEOUT_INFINITE);
        }
    }

    return bLocked;
}

/*****************************************************************************
*
*   vMasterUnlock
*
*****************************************************************************/
static void vMasterUnlock( void )
{
    BOOLEAN bLocked;
    SMS_OBJECT hObject = SMS_INVALID_OBJECT;

    bLocked = SMSO_bLock((SMS_OBJECT)gpsLeagueCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        hObject = gpsLeagueCtrl->hMasterLockableObject;

        SMSO_vUnlock((SMS_OBJECT) gpsLeagueCtrl);

        if (hObject != SMS_INVALID_OBJECT)
        {
            SMSO_vUnlock((SMS_OBJECT) hObject);
        }
    }

    return;
}

