/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:IMAGE implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <standard.h>
#include <string.h>

#include "osal.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "image_obj.h"
#include "_image_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eFormat
*
*   Retrieves the format of this image.
*
*****************************************************************************/
static IMAGE_FORMAT_ENUM eFormat (
    IMAGE_OBJECT hImage
        )
{
    IMAGE_FORMAT_ENUM eFormat = IMAGE_INVALID_FORMAT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hImage);
    if (bOwner == TRUE)
    {
        IMAGE_OBJECT_STRUCT *psObj =
            (IMAGE_OBJECT_STRUCT *)hImage;

        eFormat = psObj->eFormat;
    }

    return eFormat;
}

/*****************************************************************************
*
*   hFileName
*
*   Retrieves the filename of this image.
*
*****************************************************************************/
static STRING_OBJECT hFileName (
    IMAGE_OBJECT hImage
        )
{
    STRING_OBJECT hFileName = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hImage);
    if (bOwner == TRUE)
    {
        IMAGE_OBJECT_STRUCT *psObj =
            (IMAGE_OBJECT_STRUCT *)hImage;

        // Extract the STRING_OBJECT to provide to the caller
        hFileName = psObj->hFilename;
    }

    return hFileName;
}

/*****************************************************************************
*
*   eProperty
*
*   Provides image properties.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_ENUM eProperty,
    IMAGE_PROPERTY_VALUE_STRUCT *psValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (psValue != NULL)
    {
        IMAGE_PROPERTY_INTERNAL_ENUM eInternalProperty;
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sInternalValue;

        // Map internal property to the public one
        eInternalProperty = eMapProperty(eProperty);

        // Pass call to the friend function
        eReturnCode =
            IMAGE_eProperty(hImage, eInternalProperty, &sInternalValue);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Convert internal value into public one
            eReturnCode =
                eMapInternalPropertyValueToPublic(&sInternalValue, psValue);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eImageBackgroundDisplayRules
*
*   Queries the Image to see if it has background color information associated
*   with it.  If so, populate the requested arguments.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eImageBackgroundDisplayRules (
    IMAGE_OBJECT hImage,
    BOOLEAN *pbUseBackgroundColor,
    BOOLEAN *pbUseLineBitmap,
    size_t  *tNumLineBitmapEntries,
    RGB_STRUCT *psBackgroundColor
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    do
    {
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sValue;

        // Check inputs
        if (   (pbUseBackgroundColor == NULL)
            || (pbUseLineBitmap == NULL)
            || (tNumLineBitmapEntries == NULL)
            || (psBackgroundColor == NULL)
           )
        {
            break;
        }

        eReturnCode =
            eGetValidPropertyValue(hImage, IMAGE_PROPERTY_INTERNAL_BACKGROUND,
                IMAGE_PROPERTY_INTERNAL_TYPE_BACKGROUND, &sValue);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get background color info
        *pbUseBackgroundColor = sValue.uData.psBackground->bUseBackgroundColor;
        if (*pbUseBackgroundColor == TRUE)
        {
            // Copy the color (an rgb structure)
            *psBackgroundColor = sValue.uData.psBackground->sBackgroundColor;
        }

        // Get line bitmap info
        *pbUseLineBitmap = sValue.uData.psBackground->bUseLineBitmap;
        if (*pbUseLineBitmap == TRUE)
        {
            // Tell them how large the line bitmap is
            *tNumLineBitmapEntries =
                sValue.uData.psBackground->tNumLineBitmapEntries;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bGetLineBitmap
*
*   Extracts the background line bitmap (if present) into the caller's
*   buffer
*
*****************************************************************************/
static BOOLEAN bGetLineBitmap (
    IMAGE_OBJECT hImage,
    RGB_STRUCT *psLineBitmap
        )
{
    BOOLEAN bSuccess = FALSE;

    if (psLineBitmap != NULL)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sValue;

        eReturnCode =
            eGetValidPropertyValue(hImage, IMAGE_PROPERTY_INTERNAL_BACKGROUND,
                IMAGE_PROPERTY_INTERNAL_TYPE_BACKGROUND, &sValue);
        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (sValue.uData.psBackground->bUseLineBitmap == TRUE))
        {
            IMAGE_BACKGROUND_STRUCT *psBackground;

            psBackground = sValue.uData.psBackground;

            // Copy the line bitmap info into the caller's buffer
            bSuccess = OSAL.bMemCpy(
                (void *)psLineBitmap, psBackground->psLineBitmap,
                psBackground->tNumLineBitmapEntries * sizeof(RGB_STRUCT));
        }
    }

    return bSuccess;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   IMAGE_hCreate
*
*****************************************************************************/
IMAGE_OBJECT IMAGE_hCreate (
    SMS_OBJECT hOwner,
    const char *pacFilePath,
    IMAGE_FORMAT_ENUM eFormat,
    const IMAGE_INTF_STRUCT *psIntf,
    void *pvArg,
    BOOLEAN bFileExists
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    IMAGE_OBJECT_STRUCT *psObj = (IMAGE_OBJECT_STRUCT*) NULL;
    IMAGE_OBJECT hResult = IMAGE_INVALID_OBJECT;
    BOOLEAN bOk;
    size_t tFilenameLen = 0;

    do
    {
        // Check inputs
        if ((pacFilePath == NULL) ||
            (psIntf == NULL) ||
            (psIntf->bCreateImageFileName == NULL) ||
            (psIntf->bCalculateImageFileNameLen == NULL))
        {
            break;
        }

        // Compose object name
        snprintf( &acName[0], sizeof(acName),
                  IMAGE_OBJECT_NAME":%p:%u",
                  psIntf,
                  (UN32)eFormat);

        // Create an instance of this object
        psObj = (IMAGE_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(IMAGE_OBJECT_STRUCT) + psIntf->tSpecificDataSize,
                hOwner,
                FALSE );
        if(psObj == NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_OBJECT_NAME": failed to create object");
            break;
        }

        // Initialize image attributes
        psObj->eFormat = eFormat;
        psObj->tNumReferences = 1;
        psObj->psIntf = psIntf;

        // Find out specific data position if it has been requested
        if (psIntf->tSpecificDataSize > 0)
        {
            psObj->pvSpecificData = (void*)(psObj + 1);

            // Let caller initialize specific data
            if (psIntf->bInitSpecificData == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    IMAGE_OBJECT_NAME
                    ": there is no mandatory method to initialize specific data");
                break;
            }

            // Initialize specific data via provided callback
            bOk = psIntf->bInitSpecificData(
                        (IMAGE_OBJECT) psObj,
                        psObj->pvSpecificData,
                        pvArg);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    IMAGE_OBJECT_NAME
                    ": failed to initialize specific data");
                break;
            }
        }

        // Determine how large our filename buffer needs to be
        bOk = psObj->psIntf->bCalculateImageFileNameLen(
                        (IMAGE_OBJECT) psObj, pacFilePath,
                        psObj->pvSpecificData, &tFilenameLen);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_OBJECT_NAME
                ": failed to calculate file path length");
            break;
        }

        // Allocate memory for the file path

        // Generate a name
        // Compose object name
        snprintf( &acName[0], sizeof(acName),
                  IMAGE_OBJECT_NAME":%p:%u:Filename",
                  psIntf,
                  (UN32)eFormat);

        // Allocate memory for the file name
        psObj->pacFilename =
            (char *)SMSO_hCreate (
                        &acName[0],
                        tFilenameLen,
                        (SMS_OBJECT)psObj,
                        FALSE );

        // Verify file name allocation succeeded
        if (psObj->pacFilename == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_OBJECT_NAME
                ": failed to allocate memory for the file path");
            break;
        }

        // Generate the filename
        bOk = psObj->psIntf->bCreateImageFileName (
                (IMAGE_OBJECT) psObj,
                pacFilePath,
                psObj->pvSpecificData,
                psObj->pacFilename,
                tFilenameLen
                    );
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_OBJECT_NAME
                ": failed to compose file path");
            break;
        }

        // Create a STRING object for public access
        // create a constant string because the contents
        // will never be modified and we don't want
        // two copies of the file name...
        psObj->hFilename = STRING_hCreateConst(
                    psObj->pacFilename,
                    tFilenameLen
                        );
        if (psObj->hFilename == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_OBJECT_NAME
                ": failed to create STRING object for the file path");
            break;
        }

        // Let caller last chance to initialize IMAGE object specific data
        // based on the fully created object
        if ((psObj->psIntf->bCreationPostStep != NULL) &&
            (psObj->pvSpecificData != NULL))
        {
            bOk = psObj->psIntf->bCreationPostStep(
                                        (IMAGE_OBJECT) psObj,
                                        psObj->pvSpecificData);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    IMAGE_OBJECT_NAME
                    ": failed to call post-step interface fuction");
                break;
            }
        }

        // At last step let's check file existence on the file system
        // if the caller said it exists.
        if (bFileExists == TRUE)
        {
            UN8 un8FileAttrs = 0;

            bOk = OSAL.bFileSystemGetFileAttributes(psObj->pacFilename,
                                                    &un8FileAttrs);
            if ((bOk == FALSE) ||
                ((un8FileAttrs & OSAL_FILE_ATTR_DIRECTORY) ==
                    OSAL_FILE_ATTR_DIRECTORY))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    IMAGE_OBJECT_NAME
                    ": despite of the fact that the caller said file exists "
                    "it is absent or failed to be accessed");
                break;
            }
        }

        // We're ok. Populate create object to the caller
        hResult = (IMAGE_OBJECT) psObj;
    } while (FALSE);

    if ((hResult == IMAGE_INVALID_OBJECT) && (psObj != NULL))
    {
        IMAGE_vDestroy( (IMAGE_OBJECT)psObj );
    }

    return hResult;
}

/*****************************************************************************
*
*   IMAGE_hCreateCopy
*
*   This is called when a caller would like an IMAGE_OBJECT which is
*   identical to the provided IMAGE_OBJECT.  Instead of actually copying the
*   data, we'll just up our reference count.
*
*****************************************************************************/
IMAGE_OBJECT IMAGE_hCreateCopy (
    IMAGE_OBJECT hSrcImage
        )
{
    BOOLEAN bOwner;
    IMAGE_OBJECT hImage = IMAGE_INVALID_OBJECT;

    // Verify ownership of the source object
    bOwner = SMSO_bOwner((SMS_OBJECT)hSrcImage);
    if (bOwner == TRUE)
    {
        IMAGE_OBJECT_STRUCT *psObj =
            (IMAGE_OBJECT_STRUCT *)hSrcImage;

        // Update the reference count
        psObj->tNumReferences++;

        // Provide the caller with the handle
        hImage = hSrcImage;
    }

    return hImage;
}

/*****************************************************************************
*
*   IMAGE_eProperty
*
*   This function allows the caller to get an image's non-identifying
*   properties.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM IMAGE_eProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner;
    IMAGE_OBJECT_STRUCT *psObj = (IMAGE_OBJECT_STRUCT*)hImage;

    do
    {
        // Check input
        if ((eProperty < IMAGE_PROPERTY_INTERNAL_WIDTH) ||
            (eProperty >= IMAGE_PROPERTY_INTERNAL_INVALID) ||
            (psValue == NULL))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Is the object valid?
        bOwner = SMSO_bOwner((SMS_OBJECT) hImage);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->psIntf->eProperty != NULL)
        {
            // Pass the call to the certain implementation
            eReturnCode =
                psObj->psIntf->eProperty(hImage, eProperty,
                                         psObj->pvSpecificData, psValue);
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   IMAGE_eUpdateProperty
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM IMAGE_eUpdateProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    const IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue,
    BOOLEAN *pbUpdated
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    IMAGE_OBJECT_STRUCT *psObj;
    BOOLEAN bOwner;
    BOOLEAN bUpdated;

    do
    {
        // Make sure the address is valid
        if (pbUpdated == NULL)
        {
            pbUpdated = &bUpdated;
        }

        // Initialize the value
        *pbUpdated = FALSE;

        // Check input
        if ((eProperty < IMAGE_PROPERTY_INTERNAL_WIDTH) ||
            (eProperty >= IMAGE_PROPERTY_INTERNAL_INVALID))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Verify ownership of this object
        bOwner = SMSO_bOwner((SMS_OBJECT)hImage);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        psObj = (IMAGE_OBJECT_STRUCT*) hImage;

        // Try to pass the request to the certain implementation
        if (psObj->psIntf->eUpdateProperty != NULL)
        {
            eReturnCode =
                psObj->psIntf->eUpdateProperty(hImage, eProperty,
                            psObj->pvSpecificData, psValue, pbUpdated);
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   IMAGE_pvData
*
*   This function allows the caller to get access the image specific data.
*
*****************************************************************************/
void *IMAGE_pvData (
    IMAGE_OBJECT hImage
        )
{
    BOOLEAN bOwner;
    void *pvResult = NULL;

    // Check ownership
    bOwner = SMSO_bOwner((SMS_OBJECT) hImage);
    if (bOwner == TRUE)
    {
        IMAGE_OBJECT_STRUCT *psObj = (IMAGE_OBJECT_STRUCT*)hImage;

        // Populate specific data
        pvResult = psObj->pvSpecificData;
    }

    return pvResult;
}

/*****************************************************************************
*
*   IMAGE_n32FPrintf
*
*****************************************************************************/
N32 IMAGE_n32FPrintf(
    IMAGE_OBJECT hImage,
    FILE *psFile
        )
{
    N32 n32Return = EOF, n32Temp;
    BOOLEAN bOwner;
    IMAGE_OBJECT_STRUCT *psObj = (IMAGE_OBJECT_STRUCT*)hImage;
    IMAGE_PROPERTY_INTERNAL_ENUM eInternalProperty;
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sPropertyValue;
    SMSAPI_RETURN_CODE_ENUM eSmsApiReturnCode;
    const char *pacFormatName;
    const char *pacPropertyName;

    do
    {

        // Check ownership and input
        bOwner = SMSO_bOwner((SMS_OBJECT) hImage);
        if ((bOwner == FALSE) || (psFile == NULL))
        {
            break;
        }

        pacFormatName = pacImageFormatName(psObj->eFormat);

        n32Temp = fprintf(psFile, "IMAGE: hImage = %p\n"
                                     "\tPath: %s\n"
                                     "\thFilename: %p\n"
                                     "\tReferences: %d\n"
                                     "\tFormat: %s\n"
                                     "\tInterface: 0x%p\n"
                                     "\tSpecific Data: %p\n",
                                hImage, psObj->pacFilename,
                                psObj->hFilename,
                                psObj->tNumReferences,
                                pacFormatName,
                                psObj->psIntf,
                                psObj->pvSpecificData
                                    );
        if (n32Temp <= 0)
        {
            break;
        }
        n32Return = n32Temp;

        // Print out all properties which are available
        n32Temp = fprintf(psFile, "Properties:\n");
        if (n32Temp <= 0)
        {
            n32Return = EOF;
            break;
        }
        n32Return += n32Temp;

        for (eInternalProperty = IMAGE_PROPERTY_INTERNAL_WIDTH;
             eInternalProperty < IMAGE_PROPERTY_INTERNAL_INVALID;
             ++eInternalProperty)
        {
            // Request property from the object using friend API
            eSmsApiReturnCode = IMAGE_eProperty(hImage,
                                        eInternalProperty, &sPropertyValue);
            if (eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                pacPropertyName =
                    pacImagePropertyInternalName(eInternalProperty);

                n32Temp = fprintf(psFile, "\t%s = ", pacPropertyName);
                if (n32Temp <= 0)
                {
                    n32Return = EOF;
                    break;
                }
                n32Return += n32Temp;

                switch (sPropertyValue.eType)
                {
                    case IMAGE_PROPERTY_INTERNAL_TYPE_UN32:
                    {
                        n32Temp =
                            fprintf(psFile, "(UN32) %u\n",
                                sPropertyValue.uData.un32Value);
                    }
                    break;
                    case IMAGE_PROPERTY_INTERNAL_TYPE_STRING:
                    {
                        const char *pacString;

                        pacString =
                            STRING.pacCStr(sPropertyValue.uData.hStringValue);

                        n32Temp =
                            fprintf(psFile, "(STRING) '%s'\n", pacString);
                    }
                    break;
                    case IMAGE_PROPERTY_INTERNAL_TYPE_BACKGROUND:
                    {
                        IMAGE_BACKGROUND_STRUCT const *psBackground =
                            sPropertyValue.uData.psBackground;
                        n32Temp =
                            fprintf(psFile,
                                "(BKG) COLOR: %s,(R:%02x, G:%02x, B:%02); "
                                      "BMP: %s, %u, %p\n",
                                ((psBackground->bUseBackgroundColor == TRUE) ?
                                    "YES": "NO"),
                                psBackground->sBackgroundColor.un8Red,
                                psBackground->sBackgroundColor.un8Green,
                                psBackground->sBackgroundColor.un8Blue,
                                ((psBackground->bUseLineBitmap == TRUE) ?
                                    "YES": "NO"),
                                psBackground->tNumLineBitmapEntries,
                                psBackground->psLineBitmap);
                    }
                    break;
                }

                if (n32Temp <= 0)
                {
                    n32Return = EOF;
                    break;
                }
                n32Return += n32Temp;
            }
        }

        n32Temp = fprintf(psFile, "\n\n");
        if (n32Temp <= 0)
        {
            n32Return = EOF;
            break;
        }
        n32Return += n32Temp;
    } while (FALSE);

    return n32Return;
}

/*****************************************************************************
*
*   IMAGE_vDestroy
*
*****************************************************************************/
void IMAGE_vDestroy (
    IMAGE_OBJECT hImage
        )
{
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner = SMSO_bOwner((SMS_OBJECT)hImage);
    if (bOwner == TRUE)
    {
        IMAGE_OBJECT_STRUCT *psObj =
            (IMAGE_OBJECT_STRUCT *)hImage;

        // Reduce the number of outstanding
        // references
        if (psObj->tNumReferences > 0)
        {
            psObj->tNumReferences--;
        }

        // Only delete if nobody else has
        // a reference to this object
        if (psObj->tNumReferences == 0)
        {
            // Let owner un-init specific data
            if ((psObj->psIntf != NULL) &&
                (psObj->psIntf->vUninitSpecificData != NULL) &&
                (psObj->pvSpecificData != NULL))
            {
                psObj->psIntf->vUninitSpecificData(hImage,
                                                   psObj->pvSpecificData);
            }

            // Delete filename if it exists
            if (psObj->pacFilename != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psObj->pacFilename);
                psObj->pacFilename = NULL;
            }

            if (psObj->hFilename != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy( psObj->hFilename );
                psObj->hFilename = STRING_INVALID_OBJECT;
            }

            SMSO_vDestroy((SMS_OBJECT)hImage);
        }
    }

    return;
}

/*****************************************************************************
*
*   IMAGE_psGetLineBitmap
*
*   This function retrieves the image's line bitmap info much like the public
*   .bGetLineBitmap, but this function just copies the pointer value instead
*   of copying it to the caller's buffer.
*
*****************************************************************************/
const RGB_STRUCT *IMAGE_psGetLineBitmap (
    IMAGE_OBJECT hImage
        )
{
    const RGB_STRUCT *psLineBitmap = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sValue;

    eReturnCode =
        eGetValidPropertyValue(hImage, IMAGE_PROPERTY_INTERNAL_BACKGROUND,
            IMAGE_PROPERTY_INTERNAL_TYPE_BACKGROUND, &sValue);
    if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        psLineBitmap = sValue.uData.psBackground->psLineBitmap;
    }

    return psLineBitmap;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eMapProperty
*
*****************************************************************************/
static IMAGE_PROPERTY_INTERNAL_ENUM eMapProperty (
    IMAGE_PROPERTY_ENUM eProperty
        )
{
    IMAGE_PROPERTY_INTERNAL_ENUM eReturn = IMAGE_PROPERTY_INTERNAL_INVALID;

    switch (eProperty)
    {
        case IMAGE_PROPERTY_HEIGHT:
            eReturn = IMAGE_PROPERTY_INTERNAL_HEIGHT;
            break;
        case IMAGE_PROPERTY_WIDTH:
            eReturn = IMAGE_PROPERTY_INTERNAL_WIDTH;
            break;
        case IMAGE_PROPERTY_BITS_PER_PIXEL:
            eReturn = IMAGE_PROPERTY_INTERNAL_BITS_PER_PIXEL;
            break;
        case IMAGE_PROPERTY_CAPTION:
            eReturn = IMAGE_PROPERTY_INTERNAL_CAPTION;
            break;
    }

    return eReturn;
}

/*****************************************************************************
*
*   eMapInternalPropertyValueToPublic
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eMapInternalPropertyValueToPublic (
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT const *psIntenalValue,
    IMAGE_PROPERTY_VALUE_STRUCT *psValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;

    if ((psIntenalValue != NULL) && (psValue != NULL))
    {
        eResult = SMSAPI_RETURN_CODE_SUCCESS;

        // Convert type
        switch (psIntenalValue->eType)
        {
            case IMAGE_PROPERTY_INTERNAL_TYPE_UN32:
            {
                psValue->eType = IMAGE_PROPERTY_TYPE_UN32;
                psValue->uData.un32Value =
                    psIntenalValue->uData.un32Value;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_TYPE_STRING:
            {
                psValue->eType = IMAGE_PROPERTY_INTERNAL_TYPE_STRING;
                psValue->uData.hStringValue =
                    psIntenalValue->uData.hStringValue;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_TYPE_BACKGROUND:
            {
                eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            }
            break;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   eGetValidPropertyValue
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetValidPropertyValue(
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    IMAGE_PROPERTY_INTERNAL_TYPE_ENUM ePropertyType,
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    if (psValue != NULL)
    {
        eReturnCode =
            IMAGE_eProperty(hImage, eProperty, psValue);

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (psValue->eType != ePropertyType))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_OBJECT_NAME
                ": failed to get %s property due to invalid type",
                pacImagePropertyInternalName(eProperty));
        }
        else if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_OBJECT_NAME": failed to get %s property (%s)",
                pacImagePropertyInternalName(eProperty),
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   pacImageFormatName
*
*****************************************************************************/
static const char *pacImageFormatName(
    IMAGE_FORMAT_ENUM eFormat
        )
{
    const char *pacReturnString = "UNKNOWN";

    switch (eFormat)
    {
        case IMAGE_FORMAT_BMP:
            pacReturnString = MACRO_TO_STRING(IMAGE_FORMAT_BMP);
        break;

        case IMAGE_FORMAT_PNG:
            pacReturnString = MACRO_TO_STRING(IMAGE_FORMAT_PNG);
        break;

        case IMAGE_FORMAT_JPEG:
            pacReturnString = MACRO_TO_STRING(IMAGE_FORMAT_JPEG);
        break;

        case IMAGE_FORMAT_RAW:
            pacReturnString = MACRO_TO_STRING(IMAGE_FORMAT_RAW);
        break;

        case IMAGE_INVALID_FORMAT:
            pacReturnString = MACRO_TO_STRING(IMAGE_INVALID_FORMAT);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
*
*   pacImagePropertyInternalName
*
*****************************************************************************/
static const char *pacImagePropertyInternalName(
    IMAGE_PROPERTY_INTERNAL_ENUM eInternalProperty
        )
{
    const char *pacReturnString = "UNKNOWN";

    switch (eInternalProperty)
    {
        case IMAGE_PROPERTY_INTERNAL_WIDTH:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_WIDTH);
        break;

        case IMAGE_PROPERTY_INTERNAL_HEIGHT:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_HEIGHT);
        break;

        case IMAGE_PROPERTY_INTERNAL_BITS_PER_PIXEL:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_BITS_PER_PIXEL);
        break;

        case IMAGE_PROPERTY_INTERNAL_ID:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_ID);
        break;

        case IMAGE_PROPERTY_INTERNAL_VERSION:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_VERSION);
        break;

        case IMAGE_PROPERTY_INTERNAL_BACKGROUND:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_BACKGROUND);
        break;

        case IMAGE_PROPERTY_INTERNAL_CAPTION:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_CAPTION);
        break;

        case IMAGE_PROPERTY_INTERNAL_INVALID:
            pacReturnString =
                MACRO_TO_STRING(IMAGE_PROPERTY_INTERNAL_INVALID);
        break;
    }

    return pacReturnString;
}
