/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:HUMIDITY implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_humidity_obj.h"
#include "humidity_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eRange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRange (
    HUMIDITY_OBJECT hHumidity,
    UN8 *pun8HumidityRangeLow,
    UN8 *pun8HumidityRangeHigh
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    HUMIDITY_OBJECT_STRUCT *psObj = (HUMIDITY_OBJECT_STRUCT *)hHumidity;

    do
    {
        if ((pun8HumidityRangeHigh == NULL) || (pun8HumidityRangeLow == NULL))
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hHumidity);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }

        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *pun8HumidityRangeHigh = psObj->sData.un8High;
        *pun8HumidityRangeLow = psObj->sData.un8Low;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    HUMIDITY_OBJECT hHumidity,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    HUMIDITY_OBJECT_STRUCT *psObj = (HUMIDITY_OBJECT_STRUCT *)hHumidity;
    BOOLEAN bOwner;

    // Determine if we own the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hHumidity);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print HUMIDITY information header
    n32Return += fprintf(psFile, "\nHumidity (Object: 0x%X):\n",
                         hHumidity);

    if (psObj->bAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tRange: %u - %u\n",
            psObj->sData.un8Low, psObj->sData.un8High);
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   HUMIDITY_hCreate
*
*****************************************************************************/
HUMIDITY_OBJECT HUMIDITY_hCreate(
        SMS_OBJECT hParent,
        BOOLEAN bAvailable,
        HUMIDITY_UNPARSED_DATA tData
            )
{
    HUMIDITY_OBJECT_STRUCT *psObj =
        (HUMIDITY_OBJECT_STRUCT *)HUMIDITY_INVALID_OBJECT;

    // Create an instance of the HUMIDITY object
    psObj = (HUMIDITY_OBJECT_STRUCT *)
        SMSO_hCreate(
            HUMIDITY_OBJECT_NAME,
            sizeof(HUMIDITY_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return HUMIDITY_INVALID_OBJECT;
    }

    // Initialize object per inputs
    psObj->bAvailable = bAvailable;


    if (bAvailable == TRUE)
    {
        psObj->bAvailable = GsWeatherIntf.bProcessHumidityData(tData, &psObj->sData);
    }

    return (HUMIDITY_OBJECT)psObj;
}

/*****************************************************************************
*
*   HUMIDITY_vDestroy
*
*****************************************************************************/
void HUMIDITY_vDestroy (
    HUMIDITY_OBJECT hHumidity
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hHumidity);
    if(bOwner == TRUE)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hHumidity);
    }
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
