/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:FUEL_STATION implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "dsrl_entry_obj.h"
#include "location_obj.h"
#include "locid_obj.h"
#include "string_obj.h"
#include "fuel_db_constants.h"
#include "fuel_mgr_obj.h"
#include "_fuel_station_obj.h"
#include "fuel_station_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hBrand
*
*****************************************************************************/
static STRING_OBJECT hBrand (
    FUEL_STATION_OBJECT hFuelStation
        )
{
    BOOLEAN bOwner;
    STRING_OBJECT hBrand = STRING_INVALID_OBJECT;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;

        hBrand = psObj->hBrand;
    }

    return hBrand;
}

/*****************************************************************************
*
*   hBrandLogo
*
*****************************************************************************/
static IMAGE_OBJECT hBrandLogo (
    FUEL_STATION_OBJECT hFuelStation,
    FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM eLogoType
        )
{
    BOOLEAN bOwner;
    IMAGE_OBJECT hLogo = IMAGE_INVALID_OBJECT;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;

        // Get the logo now
        hLogo = FUEL_MGR_hBrandLogo(
            psObj->hOwner, eLogoType, psObj->n16LogoId);

        if (hLogo == IMAGE_INVALID_OBJECT)
        {
            // If we cannot get associated logo, UIRR requires 
            // to provide default (req. EV_CHARGE_DISP_017)
            hLogo = FUEL_MGR_hBrandLogo(
                psObj->hOwner, eLogoType, 0);

        }
    }

    return hLogo;
}

/*****************************************************************************
*
*   hAdditionalInfo
*
*****************************************************************************/
static STRING_OBJECT hAdditionalInfo (
    FUEL_STATION_OBJECT hFuelStation
        )
{
    BOOLEAN bOwner;
    STRING_OBJECT hInfo = STRING_INVALID_OBJECT;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;

        hInfo = psObj->hAdditionalInfo;
    }

    return hInfo;
}

/*****************************************************************************
*
*   hLocation
*
*****************************************************************************/
static LOCATION_OBJECT hLocation (
    FUEL_STATION_OBJECT hFuelStation
        )
{
    BOOLEAN bOwner;
    LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;

        hLocation = psObj->hLocation;
    }

    return hLocation;
}

/*****************************************************************************
*
*   un8NumAvailableFuelTypes
*
*****************************************************************************/
static UN8 un8NumAvailableFuelTypes (
    FUEL_STATION_OBJECT hFuelStation
        )
{
    BOOLEAN bOwner;
    UN8 un8NumFuelTypes = 0;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32Items = 0;

        // Get the number of prices in our list
        eReturnCode = OSAL.eLinkedListItems(
            psObj->hFuelPrices, &un32Items);
        if (eReturnCode == OSAL_SUCCESS)
        {
            if (un32Items > UN8_MAX)
            {
                un8NumFuelTypes = UN8_MAX;
            }
            else
            {
                un8NumFuelTypes = (UN8)un32Items;
            }
        }
    }

    return un8NumFuelTypes;
}

/*****************************************************************************
*
*   eFuelInfo
*
*   Provides more detailed info about a fuel entry (new availability enum,
*   refueling position data, new text data)
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFuelInfo (
    FUEL_STATION_OBJECT hFuelStation,
    UN8 un8FuelIndex,
    FUEL_INFO_STRUCT *psFuelInfo
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bOwner;

    // Check ownership of the object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == FALSE)
    {
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    // Check the info argument
    else if (psFuelInfo == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
        OSAL_LINKED_LIST_ENTRY hEntry;

        // Get the first entry
        hEntry = OSAL.hLinkedListFirst(
            psObj->hFuelPrices, (void **)NULL);

        while (un8FuelIndex-- > 0)
        {
            hEntry = OSAL.hLinkedListNext(hEntry, (void **)NULL);
        }

        // Verify we ended on a valid entry
        if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
        else // Extract requested data now
        {
            FUEL_STATION_PRICE_DESC_STRUCT *psPriceData;

            // Extract the price data from the entry
            psPriceData = (FUEL_STATION_PRICE_DESC_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);

            // Copy the "base" data first
            psFuelInfo->hShortFuelName = psPriceData->hShortFuelName;
            psFuelInfo->hLongFuelName = psPriceData->hLongFuelName;
            psFuelInfo->eFuelType = psPriceData->eFuelType;
            psFuelInfo->eAvailability = psPriceData->eAvailability;
            psFuelInfo->tInfoMask = psPriceData->tInfoMask;

            // Do we know the number of refueling positions?
            if ((FUEL_INFO_NUM_POSITIONS & psPriceData->tInfoMask) == FUEL_INFO_NUM_POSITIONS)
            {
                // Do we have dynamic position data available?
                if ((FUEL_INFO_DYNAMIC_POSITIONS & psPriceData->tInfoMask) == FUEL_INFO_DYNAMIC_POSITIONS)
                {
                    // Report the total number of positions as the sum 
                    // of these values
                    psFuelInfo->un16NumberOfRefuelingPositions = (
                        psPriceData->sDynamic.un16NumberAvailableRefuelingPositions + 
                        psPriceData->sDynamic.un16NumberInUseRefuelingPositions + 
                        psPriceData->sDynamic.un16NumberOfflineRefuelingPositions
                            );

                    // Copy out the dynamic data
                    psFuelInfo->sDynamicPositionInfo = psPriceData->sDynamic;

                    // Copy out the report time
                    psFuelInfo->un32ReportTime = psPriceData->un32ReportTime;
                }
                else
                {
                    // Report the number of positions simply as the number
                    // given to us by the database here
                    psFuelInfo->un16NumberOfRefuelingPositions = 
                        (UN16)psPriceData->un8NumPositions;
                }
            }
            else // this is a price entry
            {
                // Is this fuel available?
                if (psPriceData->eAvailability == FUEL_AVAILABLE)
                {
                    UN32 un32UTCSec = 0;

                    // Get the time now
                    OSAL.eTimeGet(&un32UTCSec);

                    // Calculate the price age in days
                    psFuelInfo->un32ReportTime = (
                        (un32UTCSec - psPriceData->un32ReportTime) 
                        / FUEL_SECONDS_PER_DAY);

                    // Copy out the price
                    psFuelInfo->un32PriceInTenthsOfLowestDenomination =
                        psPriceData->un32FuelPrice;
                }
            }

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   eAmenities
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAmenities (
    FUEL_STATION_OBJECT hFuelStation,
    size_t tNumAmenities,
    AMENITY_STRUCT *pasFuelAmenities
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;
    FUEL_STATION_OBJECT_STRUCT *psObj =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;

    // Ensure the structure provided is large enough
    if ((tNumAmenities < FUEL_STATION_MAX_AMENITY) ||
        (pasFuelAmenities == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Check ownership of the object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == FALSE)
    {
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Process the amenities now
    eReturnCode = FUEL_MGR_eProcessAmenities(
        psObj->hOwner, tNumAmenities,
        pasFuelAmenities, psObj->un32Amenities);

    return eReturnCode;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    FUEL_STATION_OBJECT hFuelStation,
    FILE *psFile
        )
{
    N32 n32Return = 0, n32Verify;
    BOOLEAN bOwner;
    FUEL_STATION_OBJECT_STRUCT *psObj =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
    STRING_OBJECT hString;
    IMAGE_OBJECT hLogo;
    OSAL_FIXED_OBJECT hLat, hLon;
    LOCID_OBJECT hLocID;
    LOC_ID tThisID;

    // Ensure we own this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    // Print the station data
    n32Return += fprintf(psFile, "Fuel Station (Name: ");

    hString = LOCATION.hDescription(psObj->hLocation);
    n32Verify = STRING.n32FWrite(hString, psFile);
    n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;

    hLocID = LOCATION.hLocID(psObj->hLocation);
    tThisID = LOCID.tID(hLocID);
    n32Return += fprintf(psFile, "), \n\t(LOCID: %d)",tThisID);
    n32Return += fprintf(psFile, "\n\t(STATID: %d)",FUEL_LOCID_TO_STATID(tThisID));
    n32Return += fprintf(psFile, "\n\t(REG: %d)",FUEL_LOCID_TO_REG(tThisID));
    n32Return += fprintf(psFile, "\n\t(Brand: ");
    n32Verify = STRING.n32FWrite(psObj->hBrand, psFile);
    n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;

    n32Return += fprintf(psFile, ")\n\tAddress: ");
    hString = LOCATION.hStreetName(psObj->hLocation);
    n32Verify = STRING.n32FWrite(hString, psFile);
    n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;

    n32Return += fprintf(psFile, ", ");
    hString = LOCATION.hCity(psObj->hLocation);
    n32Verify = STRING.n32FWrite(hString, psFile);
    n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;

    n32Return += fprintf(psFile, ", ");
    hString = LOCATION.hState(psObj->hLocation);
    n32Verify = STRING.n32FWrite(hString, psFile);
    n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;

    n32Return += fprintf(psFile, ", ");
    hString = LOCATION.hZipCode(psObj->hLocation);
    n32Verify = STRING.n32FWrite(hString, psFile);
    n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;

    n32Return += fprintf(psFile, "\n\tPhone: ");
    hString = LOCATION.hPhone(psObj->hLocation);
    n32Verify = STRING.n32FWrite(hString, psFile);
    n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;

    if (psObj->hAdditionalInfo != STRING_INVALID_OBJECT)
    {
        n32Return += fprintf(psFile, "\n\tAdditional Info: ");
        n32Verify = STRING.n32FWrite(psObj->hAdditionalInfo, psFile);
        n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;
    }

    // Grab the logos
    hLogo = hBrandLogo((FUEL_STATION_OBJECT)hFuelStation, FUEL_BRAND_LOGO_TYPE_LOW_RES);
    if (hLogo != IMAGE_INVALID_OBJECT)
    {
        hString = IMAGE.hFileName(hLogo);
        n32Return += fprintf(psFile, "\n\tLow Res Logo: ");
        n32Verify = STRING.n32FWrite(hString, psFile);
        n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;
    }

    hLogo = hBrandLogo((FUEL_STATION_OBJECT)hFuelStation, FUEL_BRAND_LOGO_TYPE_MED_RES);
    if (hLogo != IMAGE_INVALID_OBJECT)
    {
        hString = IMAGE.hFileName(hLogo);
        n32Return += fprintf(psFile, "\n\tMed Res Logo: ");
        n32Verify = STRING.n32FWrite(hString, psFile);
        n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;
    }

    hLogo = hBrandLogo((FUEL_STATION_OBJECT)hFuelStation, FUEL_BRAND_LOGO_TYPE_HIGH_RES);
    if (hLogo != IMAGE_INVALID_OBJECT)
    {
        hString = IMAGE.hFileName(hLogo);
        n32Return += fprintf(psFile, "\n\tHigh Res Logo: ");
        n32Verify = STRING.n32FWrite(hString, psFile);
        n32Return += ( n32Verify > 0 ) ? n32Verify : 0 ;
    }

    hLat = LOCATION.hLat(psObj->hLocation);
    hLon = LOCATION.hLon(psObj->hLocation);

    n32Return += fprintf(psFile,
        "\n\tLat: Fixed Val: %10d Bits: %2u\n\tLon: Fixed Val: %9d Bits: %2u\n",
        OSAL_FIXED.n32Value(hLat),
        OSAL_FIXED.un8NumFractionalBits(hLat),
        OSAL_FIXED.n32Value(hLon),
        OSAL_FIXED.un8NumFractionalBits(hLon));

    n32Return += fprintf(psFile, "\nOwner handle: %p\n", psObj->hOwner);

    return n32Return;
}



/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   FUEL_STATION_hCreate
*
*   Create a new object based on the identifiers in the station row.
*   Ownership of all string objects found in the station row is transfered
*   to the newly created object.
*
*****************************************************************************/
FUEL_STATION_OBJECT FUEL_STATION_hCreate(
    SMS_OBJECT hOwner,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    size_t tServiceDescriptorSize
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    size_t tCharsWritten;
    FUEL_STATION_OBJECT_STRUCT *psObj;
    LOCATION_ATTRIBUTE_STRUCT sLocAttrs;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify ownership of parent object
    bOwner = SMSO_bOwner((SMS_OBJECT)hOwner);

    // Validate inputs
    if ((psStationRow == NULL) ||
        (bOwner == FALSE) ||
        (eFuelPriceSortMethod >= FUEL_PRICE_SORT_METHOD_INVALID))
    {
        return FUEL_STATION_INVALID_OBJECT;
    }

    OSAL.bMemSet(&sLocAttrs, 0, sizeof(sLocAttrs));

    // Generate a name for this station
    tCharsWritten = snprintf( &acName[0], sizeof(acName),
              FUEL_STATION_OBJECT_NAME": reg:%u:id:%u",
              psStationRow->tRegion,
              psStationRow->tStationId );

    // Create an instance of this object
    psObj = (FUEL_STATION_OBJECT_STRUCT *)
        DSRL_ENTRY_hCreate(
            &acName[0],
            DSRL_ENTRY_TYPE_FUEL_STATION,
            sizeof(FUEL_STATION_OBJECT_STRUCT),
            tServiceDescriptorSize,
            hOwner, FALSE);
    if(psObj == NULL)
    {
        // Error!
        return FUEL_STATION_INVALID_OBJECT;
    }

    do
    {
        SMS_OBJECT hSMSStation;
        LOCID_OBJECT hLocId;
        LOC_ID tId = FUEL_REG_STATID_TO_LOCID(
            psStationRow->tRegion, psStationRow->tStationId);
        OSAL_LL_COMPARE_HANDLER hPriceCompareHandler = NULL;
        BOOLEAN bOk;

        // Get the underlying SMS Object for this station
        hSMSStation = DSRL_ENTRY_hGetSMSObject((DSRL_ENTRY_OBJECT)psObj);

        // Create a loc id for this station
        hLocId = LOCID_hCreate(tId,
            LOCID_TYPE_FUEL_STATION,
            (const LOCID_INTERFACE_STRUCT *)NULL);

        if (hLocId == LOCID_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Select the appropriate price sort method
        switch (eFuelPriceSortMethod)
        {
            case FUEL_PRICE_SORT_METHOD_FUELTYPE:
            {
                hPriceCompareHandler =
                    (OSAL_LL_COMPARE_HANDLER)n16ComparePriceEntriesByType;
            }
            break;

            // Use sort by price as the catch-all
            case FUEL_PRICE_SORT_METHOD_PRICE:
            default:
            {
                hPriceCompareHandler =
                    (OSAL_LL_COMPARE_HANDLER)n16ComparePriceEntriesByPrice;
            }
            break;
        }

        // Generate a name for the price list
        snprintf( &acName[tCharsWritten], sizeof(acName) - tCharsWritten,
                  "Prices" );

        // Create a linked list for fuel prices at this station
        eReturnCode = OSAL.eLinkedListCreate(
           &psObj->hFuelPrices,
           &acName[0],
           hPriceCompareHandler,
           OSAL_LL_OPTION_LINEAR
               );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize state attributes
        psObj->n16PriceTextVersion = -1;

        // Copy fuel-specific attributes
        psObj->hBrand = psStationRow->hBrand;
        psObj->hAdditionalInfo = psStationRow->hDesc;

        psObj->tIDForCompare = tId;

        // Copy amenities information
        psObj->un32Amenities = psStationRow->un32Amenities;

        // Copy phone info
        psObj->n64RawPhone = psStationRow->n64Phone;

        // Copy state info
        psObj->tStateId = psStationRow->tStateID;

        // Copy logo info
        psObj->n16LogoId = psStationRow->n16LogoId;

        // Copy station owner
        psObj->hOwner = hOwner;

        // Populate the location attribute structure
        sLocAttrs.hDescription = psStationRow->hName;
        sLocAttrs.hStreetNum = STRING_INVALID_OBJECT;
        sLocAttrs.hCity = psStationRow->hCity;
        sLocAttrs.hPhone = FUEL_MGR_hPhoneToString(psStationRow->n64Phone);
        sLocAttrs.hState = LOCATION_hStateAbbrvForID(psStationRow->tStateID);
        sLocAttrs.hStreetName = psStationRow->hAddr;
        sLocAttrs.hZipCode = psStationRow->hZIP;

        // Create the location object
        psObj->hLocation = LOCATION_hCreate(
            hSMSStation, hLocId,
            psStationRow->n32Lat,
            LOCATION_BINPOINT,
            psStationRow->n32Lon,
            LOCATION_BINPOINT,
            DISTANCE_INVALID_OBJECT,
            &sLocAttrs, FALSE );

        if (psObj->hLocation == LOCATION_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Add any refueling positions reported
        bOk = bAddRefuelingPositions(psObj, psStationRow);
        if (bOk == FALSE)
        {
            // Error!
            break;
        }

        // Provide the caller with the object
        return (FUEL_STATION_OBJECT)psObj;

    } while (FALSE);

    FUEL_STATION_vDestroy((FUEL_STATION_OBJECT)psObj);

    return FUEL_STATION_INVALID_OBJECT;
}

/*****************************************************************************
*
*   FUEL_STATION_hCreateDummy
*
*   Create a new object to use for comparison reasons.  Just stores
*   the LOCID.
*
*****************************************************************************/
FUEL_STATION_OBJECT FUEL_STATION_hCreateDummy (
    SMS_OBJECT hOwner
        )
{
    FUEL_STATION_OBJECT_STRUCT *psObj;
    BOOLEAN bOwner;

    // Verify ownership of parent object
    bOwner = SMSO_bOwner((SMS_OBJECT)hOwner);

    // Validate input
    if (bOwner == FALSE)
    {
        return FUEL_STATION_INVALID_OBJECT;
    }

    // Create an instance of this object
    psObj = (FUEL_STATION_OBJECT_STRUCT *)
        DSRL_ENTRY_hCreate(
            FUEL_STATION_OBJECT_NAME": Dummy",
            DSRL_ENTRY_TYPE_FUEL_STATION,
            sizeof(FUEL_STATION_OBJECT_STRUCT),
            0,
            hOwner, FALSE);
    if(psObj == NULL)
    {
        // Error!
        return FUEL_STATION_INVALID_OBJECT;
    }

    do
    {
        LOCID_OBJECT hLocId;
        LOC_ID tId = FUEL_REG_STATID_TO_LOCID(0,0);

        // Create a loc id for this station
        hLocId = LOCID_hCreate(tId,
            LOCID_TYPE_FUEL_STATION,
            (const LOCID_INTERFACE_STRUCT *)NULL);

        if (hLocId == LOCID_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Initialize state attributes
        psObj->n16PriceTextVersion = -1;

        psObj->hLocation = LOCATION.hCreateForLocID(
            OSAL_FIXED_INVALID_OBJECT,
            OSAL_FIXED_INVALID_OBJECT,
            hLocId);

        // Destroy the temporary locid object
        LOCID.vDestroy(hLocId);

        if (psObj->hLocation == LOCATION_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Ensure the LL is invalid
        psObj->hFuelPrices = OSAL_INVALID_OBJECT_HDL;

        // Provide the caller with the object
        return (FUEL_STATION_OBJECT)psObj;

    } while (FALSE);

    FUEL_STATION_vDestroy((FUEL_STATION_OBJECT)psObj);

    return FUEL_STATION_INVALID_OBJECT;
}

/*****************************************************************************
*
*   FUEL_STATION_bSetDummyLocId
*
*****************************************************************************/
BOOLEAN FUEL_STATION_bSetDummyLocId (
    FUEL_STATION_OBJECT hDummyStation,
    LOC_ID tLocId
        )
{
    BOOLEAN bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hDummyStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hDummyStation;

        psObj->tIDForCompare = tLocId;
    }

    return bOwner;
}

/*****************************************************************************
*
*   FUEL_STATION_vDestroy
*
*****************************************************************************/
void FUEL_STATION_vDestroy (
    FUEL_STATION_OBJECT hFuelStation
        )
{
    BOOLEAN bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;

        // Destroy the fuel price list
        if (psObj->hFuelPrices != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Remove all price data
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->hFuelPrices,
                (OSAL_LL_RELEASE_HANDLER)vReleasePriceData);

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode =  OSAL.eLinkedListDelete(
                    psObj->hFuelPrices);
                if (eReturnCode == OSAL_SUCCESS)
                {
                    psObj->hFuelPrices = OSAL_INVALID_OBJECT_HDL;
                }
            }
        }

        if (psObj->hBrand != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hBrand);
            psObj->hBrand = STRING_INVALID_OBJECT;
        }

        if (psObj->hAdditionalInfo != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hAdditionalInfo);
            psObj->hAdditionalInfo = STRING_INVALID_OBJECT;
        }

        if (psObj->hLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hLocation);
            psObj->hLocation = LOCATION_INVALID_OBJECT;
        }

        // Destroy the object itself
        DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT)psObj);
    }

    return;
}

#if 0

// Leaving this code here for future reference

/*****************************************************************************
*
*   FUEL_STATION_bUpdate
*
*   Update a fuel station's attributes with the contents of *psStationRow.
*   If bOverwriteAllFields == TRUE, all station attributes are updated.
*   If bOverwriteAllFields == FALSE, only those attributes in *psStationRow
*   which are valid are utilized.
*
*****************************************************************************/
BOOLEAN FUEL_STATION_bUpdate (
    FUEL_STATION_OBJECT hFuelStation,
    BOOLEAN bOverwriteAllFields,
    BOOLEAN bAmenitiesUpdated,
    FUEL_STATION_ROW_STRUCT *psStationRow
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if (bOwner == TRUE)
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
        LOCATION_ATTRIBUTE_STRUCT sNewLocAttrs;
        N16 n16Compare = !0;

        OSAL.bMemSet( &sNewLocAttrs, 0, sizeof(LOCATION_ATTRIBUTE_STRUCT));

        do
        {
            // If we were given new, valid
            // coordinates then we need
            // to update our LOCATION object
            if ((psStationRow->n32Lat != -1) &&
                (psStationRow->n32Lon != -1))
            {
                // Update the LOCATION object's coordinates
                // with our new information
                bSuccess = LOCATION_bUpdateCoordinates(
                    psObj->hLocation,
                    psStationRow->n32Lat, LOCATION_BINPOINT,
                    psStationRow->n32Lon, LOCATION_BINPOINT);
                if (bSuccess == FALSE)
                {
                    break;
                }
            }

            // Update the location object with the new row attributes
            sNewLocAttrs.hDescription = psStationRow->hName;
            sNewLocAttrs.hCity = psStationRow->hCity;

            if (psStationRow->n64Phone != 0)
            {
                // Copy phone info
                psObj->n64RawPhone = psStationRow->n64Phone;
                sNewLocAttrs.hPhone = FUEL_MGR_hPhoneToString(psStationRow->n64Phone);
            }
            else
            {
                sNewLocAttrs.hPhone = STRING_INVALID_OBJECT;
            }

            if (psStationRow->tStateID != STATE_INVALID_ID)
            {
                // Copy state info
                psObj->tStateId = psStationRow->tStateID;
                sNewLocAttrs.hState = LOCATION_hStateAbbrvForID(psStationRow->tStateID);
            }
            else
            {
                sNewLocAttrs.hState = STRING_INVALID_OBJECT;
            }

            sNewLocAttrs.hStreetName = psStationRow->hAddr;
            sNewLocAttrs.hStreetNum = STRING_INVALID_OBJECT;
            sNewLocAttrs.hZipCode = psStationRow->hZIP;

            // Update the location object's descriptive attributes
            bSuccess = LOCATION_bUpdateDescriptiveAttributes(
                psObj->hLocation, &sNewLocAttrs,
                bOverwriteAllFields, FALSE);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Update fuel-specific attributes

            // Station Brand if present
            if ((bOverwriteAllFields == FALSE) &&
                (psStationRow->hBrand != STRING_INVALID_OBJECT))
            {
                n16Compare = STRING.n16Compare(
                    psStationRow->hBrand, psObj->hBrand, TRUE);
            }
            if (n16Compare != 0)
            {
                STRING_vDestroy(psObj->hBrand);
                psObj->hBrand = STRING.hDuplicate(psStationRow->hBrand);
            }

            if (bOverwriteAllFields == TRUE)
            {
                // If we're overwriting all fields,
                // then be sure to take the amenities updates
                bAmenitiesUpdated = TRUE;
            }

            if (bAmenitiesUpdated == TRUE)
            {
                // Copy amenities information
                psObj->un32Amenities = psStationRow->un32Amenities;
            }

            return TRUE;

        } while (FALSE);
    }
    return FALSE;
}
#endif

/*****************************************************************************
*
*   FUEL_STATION_bUpdatePrices
*
*   Provide the fuel station with new price data.
*
*   Returns TRUE if the station was updated, FALSE if the station was not
*   or if the price info didn't represent changes from the current price data
*
*****************************************************************************/
BOOLEAN FUEL_STATION_bUpdatePrices (
    FUEL_STATION_OBJECT hFuelStation,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    FUEL_TYPE_TEXT_CALLBACK hFuelTextCallback,
    FUEL_TYPE_ENUM_CALLBACK eFuelTypeCallback,
    FUEL_PRICE_ROW_STRUCT *psPriceRow,
    UN8 un8TextVersion,
    void *pvCallbackArg
        )
{
    BOOLEAN bStationUpdated = FALSE;
    FUEL_STATION_OBJECT_STRUCT *psObj =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    size_t tPriceIndex;

    if (psPriceRow == NULL)
    {
        return FALSE;
    }

    if ((psPriceRow->tNumPrices == 0) || (psPriceRow->pasPrices == NULL))
    {
        return FALSE;
    }

    // Is this price text version different
    // from what we currently have?
    if (psObj->n16PriceTextVersion != (N16)un8TextVersion)
    {
        // We need to flush our price entries now
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->hFuelPrices,
            (OSAL_LL_RELEASE_HANDLER)vReleasePriceData);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // We can't access the price list!
            return FALSE;
        }

        // Use this as our price text version now
        psObj->n16PriceTextVersion = (N16)un8TextVersion;

        // We updated all the prices in this station already
        bStationUpdated = TRUE;
    }

    // Apply all the data in the price array
    for (tPriceIndex = 0; tPriceIndex < psPriceRow->tNumPrices; tPriceIndex++)
    {
        // Was the station updated by this?
        bStationUpdated |= bUpdatePriceEntry(
            psObj, eFuelPriceSortMethod, 
            hFuelTextCallback, eFuelTypeCallback, 
            &psPriceRow->pasPrices[tPriceIndex],
            pvCallbackArg);
    }

    return bStationUpdated;
}

/*****************************************************************************
*
*   FUEL_STATION_bFlushPrices
*
*   Flush prices of a certain age from the fuel station.  Return if this
*   station's price list was updated due to this operation.
*
*****************************************************************************/
BOOLEAN FUEL_STATION_bFlushPrices (
    FUEL_STATION_OBJECT hFuelStation,
    UN32 un32PriceExpireAge,
    UN32 *pun32OldestPriceAfterFlush
        )
{
    BOOLEAN bOwner, bFlushed = FALSE;

    // Verify ownership
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hFuelStation);
    if ((bOwner == TRUE) &&
        (pun32OldestPriceAfterFlush != NULL))
    {
        FUEL_STATION_OBJECT_STRUCT *psObj =
            (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        FUEL_STATION_PRICE_FLUSH_STRUCT sFlush;

        // Initialize the flush structure
        sFlush.un32PriceExpireAge = un32PriceExpireAge;
        sFlush.bPricesFlushed = FALSE;
        sFlush.pun32OldestPriceAfterFlush =
            pun32OldestPriceAfterFlush;

        // Iterate the prices to flush them
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->hFuelPrices,
            (OSAL_LL_ITERATOR_HANDLER)bIteratePricesForFlush,
            (void *)(size_t)&sFlush);
        if (eReturnCode == OSAL_SUCCESS)
        {
            bFlushed = sFlush.bPricesFlushed;
        }
    }

    return bFlushed;
}

/*****************************************************************************
*
*   FUEL_STATION_bUpdateRefuelingPositions
*
*****************************************************************************/
BOOLEAN FUEL_STATION_bUpdateRefuelingPositions (
    FUEL_STATION_OBJECT hFuelStation,
    UN16 un16ReportTime,
    FUEL_POSITION_UPDATE_STRUCT *pasPositions,
    size_t tNumPositions
        )
{
    BOOLEAN bStationUpdated = FALSE;
    FUEL_STATION_OBJECT_STRUCT *psObj =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
    size_t tIndex;

    // Are we flushing position data?
    if ((0 == tNumPositions) || (NULL == pasPositions))
    {
        // Yes, Flush dynamic position data out of the station
        // don't worry about return code
        OSAL.eLinkedListIterate(
            psObj->hFuelPrices,
            (OSAL_LL_ITERATOR_HANDLER)bIterateToRemoveDynamicPos,
            &bStationUpdated);

        return bStationUpdated;
    }

    // Update the entries now
    for (tIndex = 0; tIndex < tNumPositions; tIndex++)
    {
        bStationUpdated |= bUpdateDynamicPositions(
            psObj, un16ReportTime, &pasPositions[tIndex]);
    }

    return bStationUpdated;
}

/*****************************************************************************
*
*   FUEL_STATION_n16LogoId
*
*****************************************************************************/
N16 FUEL_STATION_n16LogoId(
    FUEL_STATION_OBJECT hFuelStation
        )
{
    FUEL_STATION_OBJECT_STRUCT *psObj =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;

    return psObj->n16LogoId;
}

/*****************************************************************************
*
*   FUEL_STATION_n16Compare
*
*   Perform a compare between two stations using their LOCIDs
*
*****************************************************************************/
N16 FUEL_STATION_n16Compare (
    FUEL_STATION_OBJECT hFuelStation1,
    FUEL_STATION_OBJECT hFuelStation2
        )
{
    FUEL_STATION_OBJECT_STRUCT *psObj1 =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation1;
    FUEL_STATION_OBJECT_STRUCT *psObj2 =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation2;

    if (psObj1->tIDForCompare == psObj2->tIDForCompare)
    {
        return 0;
    }

    if (psObj1->tIDForCompare > psObj2->tIDForCompare)
    {
        return 1;
    }

    return -1;
}

/*****************************************************************************
*
*   FUEL_STATION_bSerializeData
*
*   NOTE: It is assumed that there is enough memory allocated within
*   psDBUpdate to handle all the entries we have to serialize
*
*****************************************************************************/
BOOLEAN FUEL_STATION_bSerializeData (
    FUEL_STATION_OBJECT hFuelStation,
    FUEL_DB_UPDATE_PRICE_TABLE_STRUCT *psDBUpdate
        )
{
    FUEL_STATION_OBJECT_STRUCT *psObj =
        (FUEL_STATION_OBJECT_STRUCT *)hFuelStation;
    LOCID_OBJECT hLocId;

    if (psDBUpdate == NULL)
    {
        return FALSE;
    }

    // Get our loc Id
    hLocId = LOCATION.hLocID(psObj->hLocation);
    if (hLocId != LOCID_INVALID_OBJECT)
    {
        FUEL_STATION_PRICE_DESC_STRUCT *psCurPrice =
            (FUEL_STATION_PRICE_DESC_STRUCT *)NULL;
        OSAL_LINKED_LIST_ENTRY hEntry;
        LOC_ID tId;
        OSAL_FIXED_OBJECT hFixedLat, hFixedLon;
        UN32 un32UTCsec = 0;

        // Extract the ID value
        tId = LOCID.tID(hLocId);

        psDBUpdate->sStationRow.tRegion = FUEL_LOCID_TO_REG(tId);
        psDBUpdate->sStationRow.tStationId = FUEL_LOCID_TO_STATID(tId);
        psDBUpdate->sStationRow.hBrand = psObj->hBrand;
        psDBUpdate->sStationRow.hDesc = psObj->hAdditionalInfo;
        psDBUpdate->sStationRow.un32Amenities = psObj->un32Amenities;
        psDBUpdate->sStationRow.n16LogoId = psObj->n16LogoId;

        // Copy general location info
        psDBUpdate->sStationRow.hAddr = LOCATION.hStreetName(psObj->hLocation);
        psDBUpdate->sStationRow.hCity = LOCATION.hCity(psObj->hLocation);
        psDBUpdate->sStationRow.hName = LOCATION.hDescription(psObj->hLocation);
        psDBUpdate->sStationRow.n64Phone = psObj->n64RawPhone;
        psDBUpdate->sStationRow.tStateID = psObj->tStateId;
        psDBUpdate->sStationRow.hZIP = LOCATION.hZipCode(psObj->hLocation);

        // Calculate the lat / lon
        hFixedLat = LOCATION.hLat(psObj->hLocation);
        hFixedLon = LOCATION.hLon(psObj->hLocation);

        psDBUpdate->sStationRow.n32Lat =
            OSAL_FIXED.n32ScaledValue(
                hFixedLat, LOCATION_BINPOINT);

        psDBUpdate->sStationRow.n32Lon =
            OSAL_FIXED.n32ScaledValue(
                hFixedLon, LOCATION_BINPOINT);

        // Initialize the price count
        psDBUpdate->sPriceRow.tNumPrices = 0;

        // Initalize the type count
        psDBUpdate->sStationRow.un8NumPermanentTypes = 0;

        psDBUpdate->sPriceRow.un32YoungestPriceAgeUTCSeconds = 0;

        // Get the current time
        OSAL.eTimeGet(&un32UTCsec);

        // Now, copy the prices
        // Get the first entry
        hEntry = OSAL.hLinkedListFirst(
            psObj->hFuelPrices, (void **)&psCurPrice);

        if (hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // We don't have any prices at this time -
            // and that's just fine

            // Report our youngest price age as right now
            psDBUpdate->sPriceRow.un32YoungestPriceAgeUTCSeconds = un32UTCsec;

            return TRUE;
        }

        while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) && (psCurPrice != NULL))
        {
            // Copy the data, but discriminate between actual price entries and
            // position entries
            if ((psCurPrice->tInfoMask & FUEL_INFO_NUM_POSITIONS) == FUEL_INFO_NUM_POSITIONS)
            {
                // This is a refueling type report
                psDBUpdate->sStationRow.psTypes[psDBUpdate->sStationRow.un8NumPermanentTypes].un8FuelType =
                    psCurPrice->un8FuelType;

                psDBUpdate->sStationRow.psTypes[psDBUpdate->sStationRow.un8NumPermanentTypes].un8NumPositions =
                    psCurPrice->un8NumPositions;

                psDBUpdate->sStationRow.un8NumPermanentTypes++;

                // Report our youngest price age as right now
                psDBUpdate->sPriceRow.un32YoungestPriceAgeUTCSeconds = un32UTCsec;
            }
            else
            {
                // This is a price entry
                psDBUpdate->sPriceRow.pasPrices[psDBUpdate->sPriceRow.tNumPrices].un32FuelPrice =
                    psCurPrice->un32FuelPrice;
                psDBUpdate->sPriceRow.pasPrices[psDBUpdate->sPriceRow.tNumPrices].un8FuelType =
                    psCurPrice->un8FuelType;
                psDBUpdate->sPriceRow.pasPrices[psDBUpdate->sPriceRow.tNumPrices].eAvailable =
                    psCurPrice->eAvailability;
                psDBUpdate->sPriceRow.pasPrices[psDBUpdate->sPriceRow.tNumPrices].un32PriceAgeUTCSeconds =
                    psCurPrice->un32ReportTime;

                // Is this price "younger" than the youngest we have so far?
                if (psCurPrice->un32ReportTime >
                        psDBUpdate->sPriceRow.un32YoungestPriceAgeUTCSeconds)
                {
                    // Yes -- use it
                    psDBUpdate->sPriceRow.un32YoungestPriceAgeUTCSeconds =
                        psCurPrice->un32ReportTime;
                }

                psDBUpdate->sPriceRow.tNumPrices++;
            }

            // Get the next entry
            psCurPrice = (FUEL_STATION_PRICE_DESC_STRUCT *)NULL;
            hEntry = OSAL.hLinkedListNext(hEntry, (void **)&psCurPrice);
        }

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   psGetPriceDesc
*
*   Locates a price description given the fuel type.  Creates one if
*   nothing is found.
*
*****************************************************************************/
static FUEL_STATION_PRICE_DESC_STRUCT *psGetPriceDesc(
    FUEL_STATION_OBJECT_STRUCT *psObj,
    UN8 un8FuelType,
    BOOLEAN *pbNewEntry,
    FUEL_INFO_MASK tInitializerMask
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceDesc = 
        (FUEL_STATION_PRICE_DESC_STRUCT *)NULL;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;

    // Search the linked list for this price info
    eReturnCode = OSAL.eLinkedListLinearSearch(
        psObj->hFuelPrices,
        &hEntry,
        (OSAL_LL_COMPARE_HANDLER)n16ComparePriceEntryToType,
        (void *)&un8FuelType);

    // This entry already exists -- grab it
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Extract the price entry
        psPriceDesc = (FUEL_STATION_PRICE_DESC_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }
    // We need to generate a new entry
    else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        static UN32 un32Instance = 0;
        SMS_OBJECT hSMSStation;

        // Get the underlying SMS Object for this station
        hSMSStation = DSRL_ENTRY_hGetSMSObject(
            (DSRL_ENTRY_OBJECT)psObj);

        // Generate a name for the price desc
        snprintf( &acName[0], sizeof(acName),
              FUEL_STATION_OBJECT_NAME": Price %u",
              un32Instance++ );

        // Create an instance of this object
        psPriceDesc = (FUEL_STATION_PRICE_DESC_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(FUEL_STATION_PRICE_DESC_STRUCT),
                hSMSStation,
                FALSE );
        if (psPriceDesc != NULL)
        {
            FUEL_TEXT_STRUCT sFuelType;

            // Initialize mask as requested
            psPriceDesc->tInfoMask = tInitializerMask;

            // Initialize type
            eReturnCode = FUEL_MGR_eGetFuelType(psObj->hOwner, un8FuelType, &sFuelType);
            if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
            {
                // Fuel type is not found. Initialize what we can anyway, but report the issue
                SMSAPI_DEBUG_vPrint(FUEL_STATION_OBJECT_NAME, 2, "Cannot find Fuel Type %u", un8FuelType);
            }

            psPriceDesc->eFuelType = sFuelType.eFuelType;
            psPriceDesc->un8FuelType = sFuelType.un8FuelType;
            psPriceDesc->hShortFuelName = sFuelType.hShortFuelName;
            psPriceDesc->hLongFuelName = sFuelType.hLongFuelName;

            // Add the entry to the list
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->hFuelPrices,
                &psPriceDesc->hEntry,
                (void *)psPriceDesc);

            if (OSAL_SUCCESS == eReturnCode)
            {
                // Tell the caller we created a new entry for them
                *pbNewEntry = TRUE;
            }
            else
            {
                SMSO_vDestroy((SMS_OBJECT)psPriceDesc);
                psPriceDesc = (FUEL_STATION_PRICE_DESC_STRUCT *)NULL;
            }
        }
    }

    return psPriceDesc;
}

/*****************************************************************************
*
*   bUpdatePriceEntry
*
*   Updates a price entry with the provided price information.
*
*   Returns TRUE if the price entry was updated with the new info, FALSE
*   if the info was old news or if an error occurred.
*
*****************************************************************************/
static BOOLEAN bUpdatePriceEntry (
    FUEL_STATION_OBJECT_STRUCT *psObj,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    FUEL_TYPE_TEXT_CALLBACK hFuelTextCallback,
    FUEL_TYPE_ENUM_CALLBACK eFuelTypeCallback,
    FUEL_PRICE_ENTRY_STRUCT *psPriceEntry,
    void *pvCallbackArg
        )
{
    BOOLEAN bEntryCreated = FALSE, bUpdated = FALSE;
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceDesc;

    // Get the price descriptor that matches this entry
    psPriceDesc = psGetPriceDesc(
        psObj, psPriceEntry->un8FuelType, &bEntryCreated,
        FUEL_INFO_NONE);

    // Did that work?
    if ((FUEL_STATION_PRICE_DESC_STRUCT *)NULL == psPriceDesc)
    {
        // Nope!
        return FALSE;
    }

    // Did we create a new price entry?
    if (TRUE == bEntryCreated)
    {
        // Yes, populate the "static" members of the entry
        psPriceDesc->eFuelType = 
            eFuelTypeCallback(psPriceEntry->un8FuelType, pvCallbackArg);
        psPriceDesc->hShortFuelName = 
            hFuelTextCallback(psPriceEntry->un8FuelType, pvCallbackArg);
        psPriceDesc->hLongFuelName = STRING_INVALID_OBJECT;

        // Just say the price has been updated
        bUpdated = TRUE;
    }
    else
    {
        // No, ok well then was anything updated?  
        bUpdated = (
            // Did the availability change?
            (psPriceDesc->eAvailability != psPriceEntry->eAvailable) ||
            // Did the price change?
            (psPriceDesc->un32FuelPrice != psPriceEntry->un32FuelPrice) ||
            // Did the age change?
            (psPriceDesc->un32ReportTime != psPriceEntry->un32PriceAgeUTCSeconds)
            // Don't check the type, type enum, or type text since they will
            // never change with a price update
                );
    }

    if (TRUE == bUpdated)
    {
        // Update the relevant price information
        psPriceDesc->eAvailability = psPriceEntry->eAvailable;

        // Is this fuel available now?
        if (FUEL_AVAILABLE == psPriceDesc->eAvailability)
        {
            // This entry contains price & time data
            psPriceDesc->tInfoMask = (FUEL_INFO_PRICE | FUEL_INFO_TIME);
                
            // Copy out the price data
            psPriceDesc->un32FuelPrice = psPriceEntry->un32FuelPrice;
            psPriceDesc->un32ReportTime =
                psPriceEntry->un32PriceAgeUTCSeconds;
        }
        else
        {
            // This entry does not contain price & time data
            psPriceDesc->tInfoMask &= ~(FUEL_INFO_PRICE | FUEL_INFO_TIME);

            // Clear out the price & time data
            psPriceDesc->un32FuelPrice = 0;
            psPriceDesc->un32ReportTime = 0;
        }

        // We need to re-sort if the price changed
        // and this station sorts by price
        if (FUEL_PRICE_SORT_METHOD_PRICE == eFuelPriceSortMethod)
        {
            // Replace the entry and re-sort the list
            OSAL.eLinkedListReplaceEntry(
                psObj->hFuelPrices,
                psPriceDesc->hEntry, (void *)psPriceDesc);
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*   bAddRefuelingPositions
*
*   Populates a price entry as a refueling position.
*
*   Returns TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bAddRefuelingPositions (
    FUEL_STATION_OBJECT_STRUCT *psObj,
    FUEL_STATION_ROW_STRUCT *psStationRow
        )
{
    BOOLEAN bSuccess = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceDesc = NULL;
    FUEL_REFUELING_TYPE_STRUCT *psCurType;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    static UN32 un32Instance = 0;
    UN8 un8Index;
    SMS_OBJECT hSMSStation;

    if ((psStationRow->un8NumPermanentTypes == 0) ||
        (psStationRow->psTypes == NULL))
    {
        // Nothing to do -- no problem
        return TRUE;
    }

    // Get the underlying SMS Object for this station
    hSMSStation = DSRL_ENTRY_hGetSMSObject(
        (DSRL_ENTRY_OBJECT)psObj);

    for (un8Index = 0; un8Index < psStationRow->un8NumPermanentTypes; un8Index++)
    {
        // Get a convenient pointer
        psCurType = &psStationRow->psTypes[un8Index];

        // Generate a name for the price desc
        snprintf( &acName[0], sizeof(acName),
                FUEL_STATION_OBJECT_NAME": Position %u",
                un32Instance++ );

        // Create an instance of this object
        psPriceDesc = (FUEL_STATION_PRICE_DESC_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(FUEL_STATION_PRICE_DESC_STRUCT),
                hSMSStation,
                FALSE );

        if (psPriceDesc ==  (FUEL_STATION_PRICE_DESC_STRUCT *)NULL)
        {
            bSuccess = FALSE;
            break;
        }

        // The mask for this fuel entry must include
        // the position value
        psPriceDesc->tInfoMask |= FUEL_INFO_NUM_POSITIONS;

        // Refueling positions always use these values
        psPriceDesc->eAvailability = FUEL_AVAILABLE;
        psPriceDesc->un32FuelPrice = 0;
        psPriceDesc->un32ReportTime = 0;

        // Populate the entry with what we have from this structure
        psPriceDesc->un8FuelType = psCurType->un8FuelType;
        psPriceDesc->hShortFuelName = psCurType->hShortFuelName;
        psPriceDesc->hLongFuelName = psCurType->hLongFuelName;
        psPriceDesc->eFuelType = psCurType->eFuelType;
        psPriceDesc->un8NumPositions = psCurType->un8NumPositions;

        // Initialize entry handle
        psPriceDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Add the entry to the list
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hFuelPrices,
            &psPriceDesc->hEntry,
            (void *)psPriceDesc);

        if (eReturnCode == OSAL_SUCCESS)
        {
            // Clear the price description pointer
            psPriceDesc = (FUEL_STATION_PRICE_DESC_STRUCT *)NULL;
        }
        else
        {
            bSuccess = FALSE;
            break;
        }
    }

    // Anything left behind needs to be destroyed
    if (psPriceDesc != (FUEL_STATION_PRICE_DESC_STRUCT *)NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psPriceDesc);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateDynamicPositions
*
*   Updates an entry with the provided position information.
*
*   Returns TRUE if the entry was updated with the new info, FALSE
*   if the info was old news or if an error occurred.
*
*****************************************************************************/
static BOOLEAN bUpdateDynamicPositions (
    FUEL_STATION_OBJECT_STRUCT *psObj,
    UN16 un16ReportTime,
    FUEL_POSITION_UPDATE_STRUCT *psPosition
        )
{
    BOOLEAN bEntryCreated = FALSE, bUpdated = FALSE;
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceDesc;

    // Get the price descriptor that matches this entry
    psPriceDesc = psGetPriceDesc(
        psObj, psPosition->un8FuelType, &bEntryCreated,
        FUEL_INFO_NUM_POSITIONS);

    // Did that work?
    if ((FUEL_STATION_PRICE_DESC_STRUCT *)NULL == psPriceDesc)
    {
        // Nope!
        return FALSE;
    }

    // Did we create a new entry?
    if (TRUE == bEntryCreated)
    {
        // Yes, populate the "static" members of the entry
        psPriceDesc->eFuelType = psPosition->eFuelType;

        // Just say the entry has been updated
        bUpdated = TRUE;
    }
    else
    {
        // Was anything updated?  I have this block of code going 
        // past our regular column limit because I'd rather have long 
        // lines than break up these lines and make this harder to read.
        bUpdated = (
            // Are we not currently marked as dynamic?
            ((FUEL_INFO_DYNAMIC_POSITIONS & psPriceDesc->tInfoMask) != FUEL_INFO_DYNAMIC_POSITIONS) ||
            // Did the number of positions change?
            (psPriceDesc->sDynamic.un16NumberAvailableRefuelingPositions != psPosition->sPosition.un16NumberAvailableRefuelingPositions) ||
            // Did the number in use change?
            (psPriceDesc->sDynamic.un16NumberInUseRefuelingPositions != psPosition->sPosition.un16NumberInUseRefuelingPositions) ||
            // Did the number offline change?
            (psPriceDesc->sDynamic.un16NumberOfflineRefuelingPositions != psPosition->sPosition.un16NumberOfflineRefuelingPositions) ||
            // Did the report time change?
            (psPriceDesc->un32ReportTime != un16ReportTime)
                );
    }

    if (TRUE == bUpdated)
    {
        // Update all necessary fields
        psPriceDesc->sDynamic = psPosition->sPosition;
        psPriceDesc->un32ReportTime = un16ReportTime;

        // Mask needs to indicate we have time & dynamic positon data
        // because we get both with these types of updates
        psPriceDesc->tInfoMask |= 
            (FUEL_INFO_TIME | FUEL_INFO_DYNAMIC_POSITIONS);

        // Update "long" text if necessary
        if (STRING_INVALID_OBJECT != psPosition->hLongText)
        {
            psPriceDesc->hLongFuelName = psPosition->hLongText;
        }

        // Update "short" text if necessary
        if (STRING_INVALID_OBJECT != psPosition->hShortText)
        {
            psPriceDesc->hShortFuelName = psPosition->hShortText;
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*   vReleasePriceData
*
*****************************************************************************/
static void vReleasePriceData (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData
        )
{
    if (psPriceData != NULL)
    {
        psPriceData->tInfoMask = FUEL_INFO_NONE;
        psPriceData->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        SMSO_vDestroy((SMS_OBJECT)psPriceData);
    }

    return;
}

/*****************************************************************************
*
*   bIteratePricesForFlush
*
*****************************************************************************/
static BOOLEAN bIteratePricesForFlush (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData,
    FUEL_STATION_PRICE_FLUSH_STRUCT *psFlush
        )
{
    if ((psPriceData == NULL) || (psFlush == NULL))
    {
        return FALSE;
    }

    // If this price data is at least as old as
    // the flush argument it needs to be removed
    if (psPriceData->un32ReportTime <=
            psFlush->un32PriceExpireAge)
    {
        // Remove this entry
        OSAL.eLinkedListRemove(psPriceData->hEntry);
        vReleasePriceData(psPriceData);

        // Track that we removed something
        psFlush->bPricesFlushed = TRUE;
    }
    // Is this price older than the running oldest?
    else if (psPriceData->un32ReportTime <
                (*psFlush->pun32OldestPriceAfterFlush))
    {
        // Yes -- track it
        *psFlush->pun32OldestPriceAfterFlush =
            psPriceData->un32ReportTime;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateToRemoveDynamicPos
*
*****************************************************************************/
static BOOLEAN bIterateToRemoveDynamicPos (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData,
    BOOLEAN *pbStationUpdated
        )
{
    // Do we have valid inputs?
    if (((FUEL_STATION_PRICE_DESC_STRUCT *)NULL == psPriceData) ||
        ((BOOLEAN *)NULL == pbStationUpdated))
    {
        return FALSE;
    }

    // Did this station have dynamic data available?
    if (FUEL_INFO_DYNAMIC_POSITIONS == 
        (FUEL_INFO_DYNAMIC_POSITIONS & psPriceData->tInfoMask))
    {
        // Yes it did -- update the mask so we don't report
        // dynamic positions/time anymore
        psPriceData->tInfoMask &= 
            ~(FUEL_INFO_DYNAMIC_POSITIONS | FUEL_INFO_TIME);

        *pbStationUpdated = TRUE;
    }

    // Iterate entire list
    return TRUE;
}

/*******************************************************************************
*
*   n16ComparePriceEntriesByPrice
*
*   This function compares two FUEL_STATION_PRICE_DESC_STRUCT and orders
*   them by price.
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16ComparePriceEntriesByPrice (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData1,
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData2
        )
{
    if ((psPriceData1 == NULL) || (psPriceData2 == NULL))
    {
        return N16_MIN;
    }

    // We don't know the price of a fuel type which is unavailable,
    // so make sure we don't compare unavailable types with
    // available ones.  Put all unavailable prices at the end
    // of the list
    if (psPriceData1->eAvailability != psPriceData2->eAvailability)
    {
        if (psPriceData1->eAvailability == FUEL_AVAILABLE)
        {
            // Put available price first
            return -1;
        }

        // Put available prices first
        return 1;
    }

    // We now know that both prices are either available or
    // unavailable.  Only sort types which are available by price
    if (psPriceData1->eAvailability == FUEL_AVAILABLE)
    {
        if (psPriceData1->un32FuelPrice < psPriceData2->un32FuelPrice)
        {
            return -1;
        }

        if (psPriceData1->un32FuelPrice > psPriceData2->un32FuelPrice)
        {
            return 1;
        }

        return 0;
    }
    else // Both types are unavailable
    {
        // Don't care about order -- but we'll use type here to sort this out
        return n16ComparePriceEntriesByType(psPriceData1, psPriceData2);
    }
}

/*******************************************************************************
*
*   n16ComparePriceEntriesByType
*
*   This function compares two FUEL_STATION_PRICE_DESC_STRUCT and orders
*   them by the underlying raw fuel type.
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16ComparePriceEntriesByType (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData1,
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData2
        )
{
    if ((psPriceData1 == NULL) || (psPriceData2 == NULL))
    {
        return N16_MIN;
    }

    if (psPriceData1->un8FuelType < psPriceData2->un8FuelType)
    {
        return -1;
    }

    if (psPriceData1->un8FuelType > psPriceData2->un8FuelType)
    {
        return 1;
    }

    return 0;
}

/*******************************************************************************
*
*   n16ComparePriceEntryToType
*
*   This function compares a FUEL_STATION_PRICE_DESC_STRUCT against a
*   provided fuel type
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16ComparePriceEntryToType (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData,
    UN8 *pun8FuelType
        )
{
    if ((psPriceData == NULL) || (pun8FuelType == NULL))
    {
        return N16_MIN;
    }

    if (psPriceData->un8FuelType < *pun8FuelType)
    {
        return -1;
    }

    if (psPriceData->un8FuelType > *pun8FuelType)
    {
        return 1;
    }

    return 0;
}
