/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:FUEL_PRICE implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"
#include "sql_interface_obj.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "location_obj.h"
#include "fuel_db_constants.h"
#include "locid_obj.h"
#include "db_util.h"
#include "_fuel_price_service.h"
#include "fuel_price_service.h"
#include "fuel_interface.h"
#include "fuel_mgr_obj.h"
#include "fuel_station_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;


/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hConnect
*
*   Connects to the fuel price database and initializes the fuel price
*   database object
*
*   Inputs:
*       hFuelService - A valid handle to a FUEL_SERVICE_OBJECT representing
*           the central fuel service manager.
*       *pacRefDBDirPath - The path to the reference database
*       *peErrorCode - Reports any error encountered
*
*   Output:
*       An object handle representing this instantiation of the
*       FUEL_DB_INTERFACE_OBJECT
*
*****************************************************************************/
static FUEL_DB_INTERFACE_OBJECT hConnect (
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    const char *pacRefDBDirPath,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        )
{
    FUEL_PRICE_DB_OBJ_STRUCT *psObj =
        (FUEL_PRICE_DB_OBJ_STRUCT *)NULL;
    BOOLEAN bOwner;

    if ((hFuelService == FUEL_SERVICE_INVALID_OBJECT) ||
        (peErrorCode == NULL))
    {
        return FUEL_DB_INTERFACE_INVALID_OBJECT;
    }

    // Initialize
    *peErrorCode = DATASERVICE_ERROR_CODE_NONE;

    // Verify we own service handle
    bOwner = SMSO_bOwner(hParent);
    if (bOwner == FALSE)
    {
        return FUEL_DB_INTERFACE_INVALID_OBJECT;
    }

    // Create an instance of this object
    psObj = (FUEL_PRICE_DB_OBJ_STRUCT *)
        SMSO_hCreate(
            FUEL_PRICE_NAME,
            sizeof(FUEL_PRICE_DB_OBJ_STRUCT),
            hParent, FALSE );

    if (psObj == NULL)
    {
        return FUEL_DB_INTERFACE_INVALID_OBJECT;
    }

    // Save the service handle
    psObj->hFuelService = hFuelService;

    // Save this path data
    psObj->pacRefDatabaseDirPath = pacRefDBDirPath;

    // Not in a transaction right now
    psObj->bInTransaction = FALSE;

    // Keep OTa interface
    psObj->psOTAInterface = psOTAInterface;

    // Initialize version information
    psObj->sVerCtrl.n16DBVersion = psDBInterface->n16DBVersion;
    psObj->sVerCtrl.n16DBContentVersion = -1;
    psObj->sVerCtrl.n16TextVer = FUEL_PRICE_TEXT_INITIAL_VALUE;

    do
    {
        BOOLEAN bOk = TRUE, bServiceUpdated = FALSE;
        char *pacDatabaseFilePath,
             *pacDatabaseFilePathB;

        bOk = bInitializeWellKnownFuelTypes(psObj);
        if (bOk == FALSE)
        {
            // Error!

            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Expand our shared buffer to its initial size
        bOk = bExpandBuffer(
            &psObj->pacBuffer, &psObj->tBufferSize,
            FUEL_MAX_SQL_STRING_LENGTH, FALSE);
        if (bOk == FALSE)
        {
            // Error!

            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the path needed for the reference database
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            psDBInterface->pacFolder,
            psDBInterface->pacRefDBNameA, &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Construct the path needed for the reference database
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacRefDatabaseDirPath,
            psDBInterface->pacFolder,
            psDBInterface->pacRefDBNameB, &pacDatabaseFilePathB);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the ref database bank, but
        // never perform an integrity check
        psObj->hSQLRefConnection =
            DB_UTIL_hConnectToReferenceBank(
                &pacDatabaseFilePath[0],
                &pacDatabaseFilePathB[0],
                &psObj->pacCurRefDatabaseFilePath,
                n32ExtractDataVersion, (void*)&psDBInterface->n16DBVersion,
                TRUE, &bServiceUpdated,
                psOTAInterface->tMaxVersionBitlen,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyAndLoadDBVersionFields,
                (void *)&psObj->sVerCtrl, peErrorCode,
                SQL_INTERFACE_OPTIONS_READONLY |
                SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK);

        if (psObj->pacCurRefDatabaseFilePath == pacDatabaseFilePath)
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePathB;
        }
        else
        {
            psObj->pacNextRefDatabaseFilePath = pacDatabaseFilePath;
        }

        // Stop here if the connection failed
        if (psObj->hSQLRefConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        // Build the path to the persistent storage DB
        bOk = DB_UTIL_bCreateFilePath(
            NULL, // We don't know the base path
            psDBInterface->pacFolder,
            psDBInterface->pacPersistDBName,
            &pacDatabaseFilePath);

        if (bOk == FALSE)
        {
            // Error!  Couldn't build database path
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Connect to the persistent storage database
        psObj->hSQLPersistConnection =
            DB_UTIL_hConnectToPersistent(
                &pacDatabaseFilePath[0],
                bCreatePersistDBTables,
                (void *)psObj,
                (DB_UTIL_CHECK_VERSION_HANDLER)bVerifyAndLoadDBVersionFields,
                (void *)&psObj->sVerCtrl,
                peErrorCode);

        // Done with that
        SMSO_vDestroy((SMS_OBJECT)pacDatabaseFilePath);

        if (psObj->hSQLPersistConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        if (bServiceUpdated == TRUE)
        {
            bOk = bFlushPrices(psObj);
        }
        else
        {
            // Age out price entries now
            bOk = bAgeoutPrices(psObj);
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME": Unable to flush prices");
            *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Now, load all other tables that
        // we need to have available at all times
        bOk = bLoadTables( psObj );
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME": Unable to load fuel tables!");
            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            break;
        }

        return (FUEL_DB_INTERFACE_OBJECT)psObj;

    } while (FALSE);

    if (psObj != (FUEL_PRICE_DB_OBJ_STRUCT *)NULL)
    {
        vDisconnect((FUEL_DB_INTERFACE_OBJECT)psObj);
    }

    return FUEL_DB_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDisconnect
*
*   Uninitializes and disconnect from the fuel price db.
*
*   Inputs:
*       hDBInterface - The interface object handle created with a
*           previous call to hConnect.
*
*****************************************************************************/
static void vDisconnect (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDBInterface);
    if (bOwner == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        // End any currently-open transaction
        vEndTransaction(psObj);

        if (psObj->pacCurRefDatabaseFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacCurRefDatabaseFilePath);
            psObj->pacCurRefDatabaseFilePath = NULL;
        }

        if (psObj->pacNextRefDatabaseFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacNextRefDatabaseFilePath);
            psObj->pacNextRefDatabaseFilePath = NULL;
        }

        if (psObj->pacRefDatabaseDirPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacRefDatabaseDirPath);
            psObj->pacRefDatabaseDirPath = NULL;
        }

        // Disconnect from the ref database
        if (psObj->hSQLRefConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect( psObj->hSQLRefConnection );
            psObj->hSQLRefConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Disconnect from the persistent database
        if (psObj->hSQLPersistConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect( psObj->hSQLPersistConnection );
            psObj->hSQLPersistConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Destroy the shared buffer
        if (psObj->pacBuffer != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacBuffer);
            psObj->pacBuffer = NULL;
            psObj->tBufferSize = 0;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bCanProcessPriceMessage
*
*   This function is called by the manager to determine if we're ready
*   to begin processing price messages yet
*
*****************************************************************************/
static BOOLEAN bCanProcessPriceMessage (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8TextVersion
        )
{
    BOOLEAN bValid, bProcess = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        // We don't need any price updates until
        // we have a valid text table
        if (psObj->sVerCtrl.n16TextVer == FUEL_PRICE_TEXT_INITIAL_VALUE)
        {
            return FALSE;
        }

        // Ensure the version matches
        // our current text table version
        if ((UN16)un8TextVersion != psObj->sVerCtrl.n16TextVer)
        {
            // The text version provided to us was valid,
            // but it doesn't match our current text version,
            // so we don't want this message
            return FALSE;
        }

        // We can process this price message
        bProcess = TRUE;
    }

    return bProcess;
}

/*****************************************************************************
*
*   eMatchFuelType
*
*   This function attempts to match a fuel type with a value from
*   the well-known fuel type enumeration
*
*****************************************************************************/
static FUEL_TYPE_ENUM eMatchFuelType (
    UN8 un8TextId
        )
{
    FUEL_TYPE_ENUM eMatchedType = FUEL_TYPE_UNKNOWN;

    switch (un8TextId)
    {
        case REGULAR_FUEL_ID:
        {
            eMatchedType = FUEL_TYPE_REGULAR;
        }
        break;

        case MIDRANGE_FUEL_ID:
        {
            eMatchedType = FUEL_TYPE_MIDRANGE;
        }
        break;

        case PREMIUM_FUEL_ID:
        {
            eMatchedType = FUEL_TYPE_PREMIUM;
        }
        break;

        case DIESEL_FUEL_ID:
        {
            eMatchedType = FUEL_TYPE_DIESEL;
        }
        break;
    }

    return eMatchedType;
}

/*****************************************************************************
*
*   tLoadStations
*
*   This function is called by the manager to extract all stations found
*   in a provided area (defined by (lat/lon+radius, region id) or
*   (region id, station id)
*
*****************************************************************************/
static size_t tLoadStations (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;
        FUEL_PRICE_DB_LOAD_STATION_STRUCT sLoad;
        BOOLEAN bOk;

        // Build a query for a specific region & station
        if (tStationID != FUEL_INVALID_STATION_ID)
        {
            snprintf( &psObj->pacBuffer[0], psObj->tBufferSize,
                FUEL_SELECT_STATION_DESC,
                tRegionID, tStationID );
        }
        // Build a query for a specific region
        else
        {
            BOOLEAN bBuildQuery;

            // Build the query (it's a process!)
            bBuildQuery = bBuildStationLoadQuery(
                psObj, hLocation, tRegionID);
            if (bBuildQuery == FALSE)
            {
                return FALSE;
            }
        }

        // Initialize the result struct
        sLoad.psObj = psObj;
        sLoad.tRegion = tRegionID;
        sLoad.tStation = tStationID;
        sLoad.bSuccess = TRUE;
        sLoad.tNumObjectsCreated = 0;

        // Perform the SQL query and process the result
        bOk = SQL_INTERFACE.bQuery(
                psObj->hSQLRefConnection, &psObj->pacBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectStations,
                &sLoad ) ;

        if ((bOk == TRUE) && (sLoad.bSuccess == TRUE))
        {
            return sLoad.tNumObjectsCreated;
        }
    }

    return 0;
}

/*****************************************************************************
*
*   bLoadPrices
*
*   This function is called by the manager to extract all price data or
*   price data for a given region and station
*
*****************************************************************************/
static BOOLEAN bLoadPrices (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        )
{
    BOOLEAN bValid, bOk = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        FUEL_DB_PRICE_QUERY_RESULT_STRUCT sPriceResult;
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Get the current time (UTC)
        eReturnCode = OSAL.eTimeGet(&sPriceResult.un32CurrentTime);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // We couldn't get the time or it was not set properly
            printf(FUEL_PRICE_NAME": eTimeGet Result: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
            return FALSE;
        }

        if (sPriceResult.un32CurrentTime <= FUEL_ZERO_DATE)
        {
            // Time was not set properly
            puts(FUEL_PRICE_NAME": bad time from OSAL");
            return FALSE;
        }

        // Compute the age limit
        sPriceResult.un32PriceAgeLimit =
            sPriceResult.un32CurrentTime - FUEL_PRICES_AGE_MAX_IN_SECS;

        if (tStationID != FUEL_INVALID_STATION_ID)
        {
            // Build our price query for a single station
            snprintf( &psObj->pacBuffer[0], psObj->tBufferSize,
                FUEL_SELECT_PRICES_STATIONID, sPriceResult.un32PriceAgeLimit,
                tRegionID, tStationID);
        }
        else
        {
            // Read all prices
            // Build our price query
            snprintf( &psObj->pacBuffer[0], psObj->tBufferSize,
                FUEL_SELECT_PRICES, sPriceResult.un32PriceAgeLimit);
        }

        // Initialize the result struct
        sPriceResult.bSuccess = TRUE;
        sPriceResult.psObj = psObj;

        // Perform the SQL query and process the result
        bOk = SQL_INTERFACE.bQuery(
                psObj->hSQLPersistConnection, &psObj->pacBuffer[0],
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectPrices,
                &sPriceResult ) ;

        if (bOk == TRUE)
        {
            // Just use the success flag
            bOk = sPriceResult.bSuccess;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   bTextQuery
*
*   This function is used to determine if a newly received text update
*   needs to be processed.
*
*****************************************************************************/
static BOOLEAN bTextQuery (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8NewTextVersion
        )
{
    BOOLEAN bValid, bUpdateNeeded = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        // Are the versions different?
        if (psObj->sVerCtrl.n16TextVer != (N16)un8NewTextVersion)
        {
            bUpdateNeeded = TRUE;
        }
    }
    return bUpdateNeeded;
}

/*****************************************************************************
*
*   bUpdateTextTable
*
*   This function is called by the manager to begin or finsh updating the
*   text table.
*
*****************************************************************************/
static BOOLEAN bUpdateTextTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin,
    UN8 un8NewTextVersion
        )
{
    BOOLEAN bValid, bOk = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        if (bBegin == TRUE)
        {
            // Flush the prices in the database as well
            bOk = bFlushPrices(psObj);
            if (bOk == TRUE)
            {
                // Start a transaction for this text update
                vStartTransaction(psObj);

                // Reset the text version to the initial value
                snprintf( &psObj->pacBuffer[0], psObj->tBufferSize,
                    FUEL_UPDATE_DB_VERSION_ROW, FUEL_PRICE_TEXT_INITIAL_VALUE,
                    FUEL_PRICE_DB_VERSION_TYPE_TEXT );

                // Perform the update
                bOk = SQL_INTERFACE.bExecuteCommand(
                    psObj->hSQLPersistConnection, &psObj->pacBuffer[0] );
            }

            if (bOk == TRUE)
            {
                // Clear old entries from the text table
                bOk = SQL_INTERFACE.bExecuteCommand(
                    psObj->hSQLPersistConnection, FUEL_CLEAR_TEXT_TABLE );
            }
        }
        else
        {
            // Update the table in memory
            psObj->sVerCtrl.n16TextVer = (N16)un8NewTextVersion;

            // Generate the SQL command to update the version row
            // in the database
            snprintf( &psObj->pacBuffer[0], psObj->tBufferSize,
                FUEL_UPDATE_DB_VERSION_ROW,
                un8NewTextVersion,
                FUEL_PRICE_DB_VERSION_TYPE_TEXT
                    );

            // Update the row
            bOk = SQL_INTERFACE.bExecuteCommand(
                    psObj->hSQLPersistConnection, &psObj->pacBuffer[0] );

            // We're all done with this transaction
            vEndTransaction(psObj);
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   bUpdateTextEntry
*
*   This function is called by the manager to update a text table entry
*
*   Note: hSQLConnection is ignored in this implementation of this
*   interface function
*
*****************************************************************************/
static BOOLEAN bUpdateTextEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    SQL_INTERFACE_OBJECT hSQLConnection,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        )
{
    BOOLEAN bValid, bUpdated = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        bUpdated = SQL_INTERFACE.bExecutePreparedCommand(
            psObj->hSQLPersistConnection,
            FUEL_INSERT_TEXT_ROW,
            FUEL_PRICE_DB_TEXT_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareTextColumn,
            psTextRow);
    }

    return bUpdated;
}

/*****************************************************************************
*
*   bUpdateLogoEntry
*
*****************************************************************************/
static BOOLEAN bUpdateLogoEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bDelete,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        )
{
    // We don't support this
    return FALSE;
}

/*****************************************************************************
*
*   vUpdatePriceTable
*
*   This function is called by the manager to begin or finsh updating the
*   price table.
*
*****************************************************************************/
static void vUpdatePriceTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        if (bBegin == TRUE)
        {
            vStartTransaction(psObj);
        }
        else
        {
            vEndTransaction(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   bUpdatePriceEntry
*
*   This function is called by the manager to update a price table entry
*
*****************************************************************************/
static BOOLEAN bUpdatePriceEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_STATION_OBJECT hStation
        )
{
    UN8 un8NumPrices;
    BOOLEAN bValid, bSuccess = TRUE;
    size_t tSizeNeeded;
    FUEL_PRICE_DB_OBJ_STRUCT *psObj =
        (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;
    FUEL_DB_UPDATE_PRICE_TABLE_STRUCT sUpdate;

    bValid = SMSO_bValid((SMS_OBJECT)hDBInterface);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    do
    {
        // Calculate the space we'll need to handle the prices from
        // this station
        un8NumPrices = FUEL_STATION.un8NumAvailableFuelTypes(hStation);
        tSizeNeeded = un8NumPrices * sizeof(FUEL_PRICE_ENTRY_STRUCT);

        if (tSizeNeeded > psObj->tBufferSize)
        {
            // We need to expand shared buffer to accommodate
            // this data
            bSuccess = bExpandBuffer(
                &psObj->pacBuffer, &psObj->tBufferSize, tSizeNeeded, FALSE );
        }

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME": bUpdatePriceEntry() unable to expand buffer");
            break;
        }

        // Clear the update structure
        OSAL.bMemSet(&sUpdate, 0, sizeof(FUEL_DB_UPDATE_PRICE_TABLE_STRUCT));

        // Provide the update structure with memory for the price list
        sUpdate.sPriceRow.pasPrices = (FUEL_PRICE_ENTRY_STRUCT *)psObj->pacBuffer;

        // We need to serialize this entry
        bSuccess = FUEL_STATION_bSerializeData(hStation, &sUpdate);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME": bUpdatePriceDBEntry() FUEL_STATION_bSerializeData failed");
            break;
        }

        // Update the database
        bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
            psObj->hSQLPersistConnection, FUEL_INSERT_UPDATE_PRICE_ROW,
            FUEL_PRICE_DB_PRICE_MAX_FIELDS,
            (PREPARED_QUERY_COLUMN_CALLBACK)bPreparePriceColumn, &sUpdate);

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   un16DBContentVer
*
*   This function is called by the manager to extract the current DB
*   content version number
*
*****************************************************************************/
static UN16 un16DBContentVer (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        )
{
    BOOLEAN bOwner;
    UN8 un8Ver = 0;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDBInterface);
    if (bOwner == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        // We intentionally clip this to a UN8 for safety; the return type
        // of this f'n needs to be a UN16 to allow for EV's extra-large
        // (up to 999) version numbers.

        un8Ver = (UN8)psObj->sVerCtrl.n16DBContentVersion;
    }

    return (UN16)un8Ver;
}

/*****************************************************************************
*
*   bRefDBBank
*
*****************************************************************************/
static BOOLEAN bRefDBBank (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    if ((phInUseDB == NULL) || (phNextDB == NULL))
    {
        return FALSE;
    }

    // Initialize inputs
    *phInUseDB = *phNextDB = STRING_INVALID_OBJECT;

    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDBInterface, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        FUEL_PRICE_DB_OBJ_STRUCT *psObj =
            (FUEL_PRICE_DB_OBJ_STRUCT *)hDBInterface;

        do
        {
            // Provide the caller with the path info
            *phInUseDB = STRING.hCreate(
                psObj->pacCurRefDatabaseFilePath,
                strlen(psObj->pacCurRefDatabaseFilePath));

            if (*phInUseDB == STRING_INVALID_OBJECT)
            {
                // Error! Stop here
                break;
            }

            *phNextDB = STRING.hCreate(
                psObj->pacNextRefDatabaseFilePath,
                strlen(psObj->pacNextRefDatabaseFilePath));

            if (*phNextDB == STRING_INVALID_OBJECT)
            {
                // Clear the other string now
                STRING_vDestroy(*phInUseDB);
                *phInUseDB = STRING_INVALID_OBJECT;

                break;
            }

            // All went well
            bSuccess = TRUE;
        } while (FALSE);

        SMSO_vUnlock((SMS_OBJECT)hDBInterface);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDBUpdateBegin
*
*****************************************************************************/
static BOOLEAN bDBUpdateBegin (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char *pcSQLCommandBuffer,
   size_t tBufferSize
        )
{
    BOOLEAN bOk;

    // Generate the SQL command to update the version row
    // in the database to the db_util "under construction" value
    snprintf( &pcSQLCommandBuffer[0], tBufferSize,
        FUEL_UPDATE_DB_VERSION_ROW,
        DB_UTIL_DB_UNDER_CONSTRUCTION_VER,
        FUEL_PRICE_DB_VERSION_TYPE_CONTENTS
            );

    // Update the row
    bOk = SQL_INTERFACE.bExecuteCommand(
        hSQLConnection, &pcSQLCommandBuffer[0] );

    return bOk;
}

/*****************************************************************************
*
*   bDBUpdateEnd
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a database update has been completed and
*   a report the new database version number.
*
*****************************************************************************/
static BOOLEAN bDBUpdateEnd (
    SQL_INTERFACE_OBJECT hConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    UN16 un16NewDBVersion
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        N32 n32Result    = 0;

        // While the DB_UTIL function will print its own error, we also print
        // one here to make it easier to trace the DB failure back to a
        // particular service.
        bResult = DB_UTIL_bUpdateTimestamp( hConnection, pcSQLCommandBuffer, tBufferSize );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    FUEL_PRICE_NAME": Unable to set timestamp.");
        }


        // We make sure to clamp the DB version back down to a UN8 for safety.
        // This parameter is passed in as a UN16 as fuel and ev share a common
        // FUEL_DB_INTERFACE_STRUCT; EV needs a UN16 as their version numbers
        // can go as high as 999, while fuel is limited to 99.
        n32Result = snprintf( pcSQLCommandBuffer, tBufferSize,
                              FUEL_UPDATE_DB_VERSION_ROW,
                              (UN8)un16NewDBVersion,
                              FUEL_PRICE_DB_VERSION_TYPE_CONTENTS );
        if ( n32Result <= 0 )
        {
            bResult = FALSE;
            break;
        }

        // Updating the baseline version comes last, as this effectively stamps
        // the DB as "ready."
        bResult = SQL_INTERFACE.bExecuteCommand( hConnection, pcSQLCommandBuffer );
        if ( bResult == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    FUEL_PRICE_NAME": Unable to set baseline version.");
            break;
        }

    } while ( FALSE );

    return bResult;
}

/*****************************************************************************
*
*   bRegionUpdateBegin
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a region update has been received.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       *psNewRegionRow - A pointer to the newly received region information.
*       un8MsgId - The ID of the message bearing this update
*
*   Output: BOOLEAN TRUE on success, FALSE on error or unwanted data
*
*****************************************************************************/
static BOOLEAN bRegionUpdateBegin (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    FUEL_REGION tRegionID
        )
{
    BOOLEAN bSuccess, bRegionExists = FALSE;

    // Generate the command to query for this region
    snprintf( &pcSQLCommandBuffer[0], tBufferSize,
        FUEL_SELECT_REGION, tRegionID );

    // Start a new transaction
    bSuccess = SQL_INTERFACE.bStartTransaction(hSQLConnection);
    if (bSuccess == FALSE)
    {
        return FALSE;
    }

    // Does this region exist?
    bSuccess = SQL_INTERFACE.bQuery(
        hSQLConnection, &pcSQLCommandBuffer[0],
        (SQL_QUERY_RESULT_HANDLER)bProcessSelectRegionExists,
        (void *)&bRegionExists);
    if (bSuccess == TRUE)
    {
        if (bRegionExists == FALSE)
        {
            // We need to add a region to our DB now

            // Insert the region entry
            bSuccess = SQL_INTERFACE.bExecutePreparedCommand(
                hSQLConnection, FUEL_INSERT_REGION_ROW,
                FUEL_DB_REGION_MAX_FIELDS,
                (PREPARED_QUERY_COLUMN_CALLBACK)bPrepareRegionColumn,
                &tRegionID);

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_PRICE_NAME": Unable to insert the region entry");
            }

            // Generate the command to create the table
            snprintf( &pcSQLCommandBuffer[0], tBufferSize,
                FUEL_CREATE_STATION_TABLE, tRegionID );

            // Create the new table
            bSuccess = SQL_INTERFACE.bExecuteCommand(
                hSQLConnection, &pcSQLCommandBuffer[0]);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_PRICE_NAME": Unable to create new stations table");
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bRegionUpdateEnd
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a region update has been completed.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       *psRegion - A pointer to the newly received region information.
*       un8MsgId - The ID of the message bearing this update
*
*   Output: BOOLEAN TRUE on success, FALSE on error or unwanted data
*
*****************************************************************************/
static BOOLEAN bRegionUpdateEnd (
    SQL_INTERFACE_OBJECT hSQLConnection
        )
{
    return SQL_INTERFACE.bEndTransaction(hSQLConnection);
}

/*****************************************************************************
*
*   bStationUpdate
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a station update has been received.
*
*****************************************************************************/
static BOOLEAN bStationUpdate (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char **ppcSQLCommandBuffer,
   size_t *ptBufferSize,
   FUEL_STATION_ROW_STRUCT *psStationRow,
   FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
   BOOLEAN bAmenitiesUpdated
       )
{
    BOOLEAN bOk;

    if (eUpdateType >= FUEL_STATION_UPDATE_MAX_TYPES)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_PRICE_NAME": invalid update type!");

        return FALSE;
    }

    // Update the database
    bOk = bUpdateStationDBEntry(
        hSQLConnection, ppcSQLCommandBuffer, ptBufferSize,
        psStationRow, eUpdateType, bAmenitiesUpdated);

    // Free all strings given to us by the interface

    // ...Except don't destroy the brand text
    // since it is persistent
    psStationRow->hBrand = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hAddr);
    psStationRow->hAddr = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hCity);
    psStationRow->hCity = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hName);
    psStationRow->hName = STRING_INVALID_OBJECT;

    STRING_vDestroy(psStationRow->hZIP);
    psStationRow->hZIP = STRING_INVALID_OBJECT;

    return bOk;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   FUEL_PRICES_hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
FUEL_SERVICE_OBJECT FUEL_PRICES_hStart (
    const char *pacSRHDriverName,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    FUEL_SERVICE_OBJECT hFuelPriceService =
        FUEL_SERVICE_INVALID_OBJECT;
    FUEL_SERVICE_OBJECT hFuelService;
    DATASERVICE_CREATE_STRUCT sCreate;

    // Populate our data service creation structure with attributes
    // specific to the fuel price data service (also populate the common items
    // that we can populate given the input arguments
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);

    // Input arguments
    sCreate.pacSRHDriverName = pacSRHDriverName;

    // Fuel prices specific attributes
    sCreate.pacServiceObjectName = FUEL_PRICE_NAME;
    sCreate.tDataID = GsPriceOTAIntf.tDataID;

    // Start the fuel service using the common
    // underlying hStart function
    hFuelService = FUEL_MGR_hStart(
        &sCreate, DATASERVICE_TYPE_FUEL, FALSE,
        &GsPriceOTAIntf, &GsPriceDBIntf,
        eFuelPriceSortMethod, tEventRequestMask,
        vEventCallback, pvEventCallbackArg,
        psOptions );
    if (hFuelService != FUEL_SERVICE_INVALID_OBJECT)
    {
        // Provide the caller with the fuel service object
        hFuelPriceService = (FUEL_SERVICE_OBJECT)hFuelService;
    }

    return hFuelPriceService;
}

/*****************************************************************************
*
*   FUEL_PRICES_eGetReferenceDataVersion
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM FUEL_PRICES_eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode;

    eErrorCode = eRegularGetReferenceDataVersion(pcContainingDirectoryPath,
        &GsPriceDBIntf, GsPriceOTAIntf.tMaxVersionBitlen,
        ptCurrentRefDataVer, ptNextRefDataVer);
 
    return eErrorCode;
}

/*****************************************************************************
*
*   CANFUEL_PRICES_hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
FUEL_SERVICE_OBJECT CANFUEL_PRICES_hStart (
    const char *pacSRHDriverName,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    FUEL_SERVICE_OBJECT hFuelPriceService =
        FUEL_SERVICE_INVALID_OBJECT;
    FUEL_SERVICE_OBJECT hFuelService;
    DATASERVICE_CREATE_STRUCT sCreate;

    // Populate our data service creation structure with attributes
    // specific to the fuel price data service (also populate the common items
    // that we can populate given the input arguments
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);

    // Input arguments
    sCreate.pacSRHDriverName = pacSRHDriverName;

    // Fuel prices specific attributes
    sCreate.pacServiceObjectName = FUEL_CAN_PRICE_NAME;
    sCreate.tDataID = GsCanadianPriceOTAIntf.tDataID;

    // Start the fuel service using the common
    // underlying hStart function
    hFuelService = FUEL_MGR_hStart(
        &sCreate, DATASERVICE_TYPE_CANFUEL, FALSE,
        &GsCanadianPriceOTAIntf, &GsCanadianPriceDBIntf,
        eFuelPriceSortMethod, tEventRequestMask,
        vEventCallback, pvEventCallbackArg,
        psOptions );
    if (hFuelService != FUEL_SERVICE_INVALID_OBJECT)
    {
        // Provide the caller with the fuel service object
        hFuelPriceService = (FUEL_SERVICE_OBJECT)hFuelService;
    }

    return hFuelPriceService;
}

/*****************************************************************************
*
*   CANFUEL_PRICES_eGetReferenceDataVersion
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM CANFUEL_PRICES_eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode;

    eErrorCode = eRegularGetReferenceDataVersion(pcContainingDirectoryPath,
        &GsCanadianPriceDBIntf, GsCanadianPriceOTAIntf.tMaxVersionBitlen,
        ptCurrentRefDataVer, ptNextRefDataVer);
 
    return eErrorCode;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eRegularGetReferenceDataVersion
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eRegularGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    const FUEL_DB_INTERFACE_STRUCT *psInterface,
    size_t tVersionBitLen, 
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
    BOOLEAN bOk;
    char *pacDatabaseFilePathA = (char *)NULL,
         *pacDatabaseFilePathB = (char *)NULL;

    if (ptCurrentRefDataVer == NULL)
    {
        return DATASERVICE_ERROR_CODE_GENERAL;
    }

    do
    {
        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            psInterface->pacFolder,
            psInterface->pacRefDBNameA,
            &pacDatabaseFilePathA);

        if (bOk != TRUE)
        {
            break;
        }

        bOk = DB_UTIL_bCreateFilePath(
            pcContainingDirectoryPath,
            psInterface->pacFolder,
            psInterface->pacRefDBNameB,
            &pacDatabaseFilePathB);

        if (bOk != TRUE)
        {
            break;
        }

        // Connect to the fuel ref database
        eErrorCode = DB_UTIL_eCheckReferenceBanks(
            &pacDatabaseFilePathA[0],
            &pacDatabaseFilePathB[0],
            n32ExtractDataVersion, (void*)&psInterface->n16DBVersion,
            tVersionBitLen,
            ptCurrentRefDataVer, ptNextRefDataVer );

    } while (FALSE);

    // Remove database path resources
    if (pacDatabaseFilePathA != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathA);
    }

    if (pacDatabaseFilePathB != NULL)
    {
        SMSO_vDestroy(
            (SMS_OBJECT)pacDatabaseFilePathB);
    }

    return eErrorCode;
}

/*****************************************************************************
*
*   bPriceAgeFilter
*
*   This function is called when price information is extracted from
*   the database in order to determine if the age of the price data meets
*   the application's current age-out criteria.
*
*****************************************************************************/
static BOOLEAN bPriceAgeFilter (
    UN32 un32PriceAgeInEpochSeconds,
    UN32 un32CurrentTime
        )
{
    UN32 un32AgeInEpochSeconds = un32CurrentTime;
    BOOLEAN bMeetsCriteria = FALSE;

    // Only compute the age if the time was set
    // correctly
    if (un32AgeInEpochSeconds > un32PriceAgeInEpochSeconds)
    {
        un32AgeInEpochSeconds -= un32PriceAgeInEpochSeconds;

        if (un32AgeInEpochSeconds < FUEL_PRICES_AGE_MAX_IN_SECS)
        {
            bMeetsCriteria = TRUE;
        }
    }

    return bMeetsCriteria;
}

/*****************************************************************************
*
*   bInitializeWellKnownFuelTypes
*
*****************************************************************************/
static BOOLEAN bInitializeWellKnownFuelTypes (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
       )
{
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hFuelType = STRING_INVALID_OBJECT;

    do
    {
        BOOLEAN bTextAdded;
        FUEL_TEXT_ROW_STRUCT sText;

        // These two fields are always set this way
        sText.bBrandText = FALSE;
        sText.hLongText = STRING_INVALID_OBJECT;

        // Create the string for regular
        hFuelType =
            STRING_hCreate( SMS_INVALID_OBJECT,
                REGULAR_FUEL_NAME,
                strlen(REGULAR_FUEL_NAME),
                strlen(REGULAR_FUEL_NAME));

        if (hFuelType == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Populate the structure
        sText.hText = hFuelType;
        sText.un8TextId = REGULAR_FUEL_ID;

        // Provide to the manager
        bTextAdded = GsFuelMgrIntf.bReportTextFromDB(
            psObj->hFuelService, &sText);
        if (bTextAdded == FALSE)
        {
            break;
        }

        // Create the string for mid-range
        hFuelType =
            STRING_hCreate( SMS_INVALID_OBJECT,
                MIDRANGE_FUEL_NAME,
                strlen(MIDRANGE_FUEL_NAME),
                strlen(MIDRANGE_FUEL_NAME));
        if (hFuelType == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Populate the structure
        sText.hText = hFuelType;
        sText.un8TextId = MIDRANGE_FUEL_ID;

        // Provide to the manager
        bTextAdded = GsFuelMgrIntf.bReportTextFromDB(
            psObj->hFuelService, &sText);
        if (bTextAdded == FALSE)
        {
            break;
        }

        // Create the string for premium
        hFuelType =
            STRING_hCreate( SMS_INVALID_OBJECT,
                PREMIUM_FUEL_NAME,
                strlen(PREMIUM_FUEL_NAME),
                strlen(PREMIUM_FUEL_NAME));
        if (hFuelType == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Populate the structure
        sText.hText = hFuelType;
        sText.un8TextId = PREMIUM_FUEL_ID;

        // Provide to the manager
        bTextAdded = GsFuelMgrIntf.bReportTextFromDB(
            psObj->hFuelService, &sText);
        if (bTextAdded == FALSE)
        {
            break;
        }

        // Create the string for diesel
        hFuelType =
            STRING_hCreate( SMS_INVALID_OBJECT,
                DIESEL_FUEL_NAME,
                strlen(DIESEL_FUEL_NAME),
                strlen(DIESEL_FUEL_NAME));
        if (hFuelType == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Populate the structure
        sText.hText = hFuelType;
        sText.un8TextId = DIESEL_FUEL_ID;

        // Provide to the manager
        bTextAdded = GsFuelMgrIntf.bReportTextFromDB(
            psObj->hFuelService, &sText);
        if (bTextAdded == FALSE)
        {
            break;
        }

        // Clear this handle since that worked
        hFuelType = STRING_INVALID_OBJECT;

        // All done!
        bSuccess = TRUE;
    } while (FALSE);

    // Clean up anything we left behind
    if (hFuelType != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hFuelType);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bFlushPrices
*
*   Flush all prices out of the database.
*
*****************************************************************************/
static BOOLEAN bFlushPrices (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;

    printf(FUEL_PRICE_NAME": "FUEL_FLUSH_PRICE_ROWS"\n");

    // Perform the flush
    bSuccess = SQL_INTERFACE.bExecuteCommand(
        psObj->hSQLPersistConnection, FUEL_FLUSH_PRICE_ROWS );

    return bSuccess;
}

/*****************************************************************************
*
*   bAgeoutPrices
*
*   This function removes all old price entries from the database
*
*****************************************************************************/
static BOOLEAN bAgeoutPrices (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = TRUE;
    UN32 un32AgeLimit = 0;

    // Get the current time (UTC)
    eReturnCode = OSAL.eTimeGet(&un32AgeLimit);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_PRICE_NAME
            ": bAgeoutPriceEntries() OSAL.eTimeGet () failure (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));

        return FALSE;
    }

    // Ensure that the clock is set correctly(-ish)
    if (un32AgeLimit <= FUEL_ZERO_DATE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_PRICE_NAME
            ": bHandleAgeout() bad time reported");

        return FALSE;
    }

    // Data which is at least FUEL_AGE_MAX_IN_SECS
    // old must be aged out now.  Compute that
    // in epoch seconds
    un32AgeLimit -= FUEL_PRICES_AGE_MAX_IN_SECS;

    // Generate the command
    snprintf( &psObj->pacBuffer[0], psObj->tBufferSize,
            FUEL_AGEOUT_PRICE_ROWS, un32AgeLimit);

    bSuccess = SQL_INTERFACE.bExecuteCommand(
        psObj->hSQLPersistConnection, &psObj->pacBuffer[0]);

    return bSuccess;
}

/*****************************************************************************
*
*   vShiftPriceEntries
*
*   This function does a lazy shift of price entries to overwrite a price
*   entry which we don't actually want to report to a station
*
*****************************************************************************/
static void vShiftPriceEntries(
    FUEL_PRICE_ENTRY_STRUCT *pasPrices,
    size_t tNumPrices,
    size_t tCurIndex
        )
{
    size_t tStop = tNumPrices - 1;

    while (tCurIndex < tStop)
    {
        pasPrices[tCurIndex] = pasPrices[tCurIndex + 1];
        tCurIndex++;
    }

    return;
}

/*****************************************************************************
*
*   bUpdateStationDBEntry
*
*****************************************************************************/
static BOOLEAN bUpdateStationDBEntry (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char **ppacBuffer,
    size_t *ptBufferSize,
    FUEL_STATION_ROW_STRUCT *psStation,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    BOOLEAN bAmenitiesUpdated
        )
{
    BOOLEAN bOk;

    if (eUpdateType == FUEL_STATION_UPDATE_TYPE_DELETE)
    {
        // Just generate a delete command
        snprintf(
            *ppacBuffer, *ptBufferSize,
            FUEL_DEL_STATION_DESC,
            psStation->tRegion,
            psStation->tStationId);
    }
    else
    {
        size_t tTextLen, tCharsWritten;

        // How long could the text get?
        tTextLen = STRING.tSize(psStation->hAddr);
        tTextLen += STRING.tSize(psStation->hBrand);
        tTextLen += STRING.tSize(psStation->hCity);
        tTextLen += STRING.tSize(psStation->hName);
        tTextLen += STRING.tSize(psStation->hZIP);
        tTextLen += FUEL_INSERT_REPLACE_STATION_ROW_LEN;

        if (tTextLen > *ptBufferSize)
        {
            // We need to expand our shared buffer to accommodate
            // this data
            bOk = bExpandBuffer( ppacBuffer, ptBufferSize, tTextLen, FALSE );
            if (bOk == FALSE)
            {
                // Error! We couldn't expand the buffer!
                return FALSE;
            }
        }

        // If we're processing a modify we need to first see if the
        // station listed exists.  If not, we'll treat this as an add
        if (FUEL_STATION_UPDATE_TYPE_MODIFY_GIVEN_ATTRIBS == eUpdateType)
        {
            BOOLEAN bStationExists = FALSE, bSuccess;

            // Generate the command to query for this station
            snprintf( *ppacBuffer, *ptBufferSize,
                FUEL_SELECT_STATION_EXISTS,
                psStation->tRegion, psStation->tStationId );

            // Does this station exist?
            bSuccess = SQL_INTERFACE.bQuery(
                hSQLConnection, *ppacBuffer,
                (SQL_QUERY_RESULT_HANDLER)bProcessSelectStationExists,
                (void *)&bStationExists);

            if (FALSE == bSuccess)
            {
                // Error!
                return FALSE;
            }

            if (FALSE == bStationExists)
            {
                // This station doesn't exist...so treat this
                // modify command as if it were an add
                eUpdateType = FUEL_STATION_UPDATE_TYPE_NEW;
            }
        }

        // Create our SQL command string based on the modify type
        if (eUpdateType == FUEL_STATION_UPDATE_TYPE_MODIFY_GIVEN_ATTRIBS)
        {
            BOOLEAN bFirstArg = TRUE;

            // Insert first part
            tCharsWritten = snprintf( *ppacBuffer, *ptBufferSize,
                FUEL_UPDATE_ROW_BEGIN, psStation->tRegion );

            // Do we need to update the brand?
            if (psStation->hBrand != STRING_INVALID_OBJECT)
            {
                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_BRAND);

                // Get the brand string to add its contents to the command
                STRING.tCopyToCStr(psStation->hBrand,
                    (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize - tCharsWritten);

                // Ensure this field is correctly formatted
                // for SQL -- if the string gets updated
                // tCharsWritten will be as well
                bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
                if (bOk == FALSE)
                {
                    return FALSE;
                }

                // Move our index past the brand text
                tCharsWritten += STRING.tSize(psStation->hBrand);

                // Add formatting characters
                tCharsWritten += snprintf(
                    (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, "\'");

                bFirstArg = FALSE;
            }

            if (psStation->hName != STRING_INVALID_OBJECT)
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_NAME);

                // Get the name string to add its contents to the command
                STRING.tCopyToCStr(psStation->hName,
                    (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize - tCharsWritten);

                // Ensure this field is correctly formatted
                // for SQL -- if the string gets updated
                // tCharsWritten will be as well
                bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
                if (bOk == FALSE)
                {
                    return FALSE;
                }

                // Move our index past the name text
                tCharsWritten += STRING.tSize(psStation->hName);

                // Add formatting characters
                tCharsWritten += snprintf((*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, "\'");

                bFirstArg = FALSE;
            }

            if (psStation->hAddr != STRING_INVALID_OBJECT)
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_ADDR);

                // Get the addr string to add its contents to the command
                STRING.tCopyToCStr(psStation->hAddr,
                    (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize - tCharsWritten);

                // Ensure this field is correctly formatted
                // for SQL -- if the string gets updated
                // tCharsWritten will be as well
                bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
                if (bOk == FALSE)
                {
                    return FALSE;
                }

                // Move our index past the addr text
                tCharsWritten += STRING.tSize(psStation->hAddr);

                // Add formatting characters
                tCharsWritten += snprintf((*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, "\'");

                bFirstArg = FALSE;
            }

            if (psStation->hCity != STRING_INVALID_OBJECT)
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_CITY);

                // Get the city string to add its contents to the command
                STRING.tCopyToCStr(psStation->hCity,
                    (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize - tCharsWritten);

                // Ensure this field is correctly formatted
                // for SQL -- if the string gets updated
                // tCharsWritten will be as well
                bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
                if (bOk == FALSE)
                {
                    return FALSE;
                }

                // Move our index past the city text
                tCharsWritten += STRING.tSize(psStation->hCity);

                // Add formatting characters
                tCharsWritten += snprintf((*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, "\'");

                bFirstArg = FALSE;
            }

            if (psStation->tStateID != STATE_INVALID_ID)
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                // Insert state id
                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_STATE,
                    psStation->tStateID);

                bFirstArg = FALSE;
            }

            if (psStation->hZIP != STRING_INVALID_OBJECT)
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_ZIP);

                // Get the ZIP string to add its contents to the command
                STRING.tCopyToCStr(psStation->hZIP,
                    (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize - tCharsWritten);

                // Ensure this field is correctly formatted
                // for SQL -- if the string gets updated
                // tCharsWritten will be as well
                bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
                if (bOk == FALSE)
                {
                    return FALSE;
                }

                // Move our index past the ZIP text
                tCharsWritten += STRING.tSize(psStation->hZIP);

                // Add formatting characters
                tCharsWritten += snprintf((*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, "\'");

                bFirstArg = FALSE;
            }

            if (psStation->n64Phone != 0)
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                // Insert state id
                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_PHONE,
                    psStation->n64Phone);

                bFirstArg = FALSE;
            }

            if ((psStation->n32Lat != -1) && (psStation->n32Lon != -1))
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                // Insert lat / lon
                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_LOC,
                    psStation->n32Lat, psStation->n32Lon);

                bFirstArg = FALSE;
            }

            // Have the amenities been updated?
            if (bAmenitiesUpdated == TRUE)
            {
                if (bFirstArg == FALSE)
                {
                    tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                        *ptBufferSize, ",");
                }

                // Insert amenities
                tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                    *ptBufferSize, FUEL_UPDATE_AMENITIES,
                    psStation->un32Amenities);

                bFirstArg = FALSE;
            }

            if (bFirstArg == TRUE)
            {
                // We don't have any changes.  Weird,
                // but not really a problem
                return TRUE;
            }

            // Close the string
            snprintf( (*ppacBuffer) + tCharsWritten,
                *ptBufferSize, FUEL_UPDATE_ROW_END,
                psStation->tStationId);
        }
        else // Overwrite all attributes for this entry
        {
            // Insert first part
            tCharsWritten = snprintf( *ppacBuffer, *ptBufferSize,
                FUEL_INSERT_REPLACE_STATION_ROW_BEGIN,
                psStation->tRegion, psStation->tStationId);

            // Get the brand string to add its contents to the command
            STRING.tCopyToCStr(psStation->hBrand,
                (*ppacBuffer) + tCharsWritten,
                *ptBufferSize - tCharsWritten);

            // Ensure this field is correctly formatted
            // for SQL -- if the string gets updated
            // tCharsWritten will be as well
            bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            // Move our index past the brand text
            tCharsWritten += STRING.tSize(psStation->hBrand);

            // Insert command formatting
            tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                *ptBufferSize, "\', \'");

            // Get the name string to add its contents to the command
            STRING.tCopyToCStr(psStation->hName,
                (*ppacBuffer) + tCharsWritten,
                *ptBufferSize - tCharsWritten);

            // Ensure this field is correctly formatted
            // for SQL -- if the string gets updated
            // tCharsWritten will be as well
            bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            // Move our index past the name text
            tCharsWritten += STRING.tSize(psStation->hName);

            // Insert command formatting
            tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                *ptBufferSize, "\', \'");

            // Get the addr string to add its contents to the command
            STRING.tCopyToCStr(psStation->hAddr,
                (*ppacBuffer) + tCharsWritten,
                *ptBufferSize - tCharsWritten);

            // Ensure this field is correctly formatted
            // for SQL -- if the string gets updated
            // tCharsWritten will be as well
            bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            // Move our index past the address text
            tCharsWritten += STRING.tSize(psStation->hAddr);

            // Insert command formatting
            tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                *ptBufferSize, "\', \'");

            // Get the city string to add its contents to the command
            STRING.tCopyToCStr(psStation->hCity,
                (*ppacBuffer) + tCharsWritten,
                *ptBufferSize - tCharsWritten);

            // Ensure this field is correctly formatted
            // for SQL -- if the string gets updated
            // tCharsWritten will be as well
            bOk = bFormatTextField( ppacBuffer, ptBufferSize, &tCharsWritten);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            // Move our index past the city text
            tCharsWritten += STRING.tSize(psStation->hCity);

            // Insert command formatting, add state id
            tCharsWritten += snprintf( (*ppacBuffer) + tCharsWritten,
                *ptBufferSize, "\', %d, \'", psStation->tStateID);

            // Get the zip string to add its contents to the command
            STRING.tCopyToCStr(psStation->hZIP,
                (*ppacBuffer) + tCharsWritten,
                *ptBufferSize - tCharsWritten);
            tCharsWritten += STRING.tSize(psStation->hZIP);

            // Insert command formatting, lat / lon & amenities 1 & 2
            // and close the command string
            snprintf( (*ppacBuffer) + tCharsWritten,
                *ptBufferSize, "\', %lld, %d, %d, %u);",
                psStation->n64Phone,
                psStation->n32Lat, psStation->n32Lon,
                psStation->un32Amenities);
        }
    }

    printf(FUEL_PRICE_NAME": %s\n", *ppacBuffer);

    // Update the database
    bOk = SQL_INTERFACE.bExecuteCommand(
            hSQLConnection, *ppacBuffer );

    return bOk;
}

/*****************************************************************************
*
*   bBuildStationLoadQuery
*
*   Perform all the math necessary to construct a location-based DB
*   query for a station search.
*
*****************************************************************************/
static BOOLEAN bBuildStationLoadQuery (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegion
        )
{
    BOOLEAN bQueryBuilt = FALSE;
    OSAL_FIXED_OBJECT hTopRightLat,
                      hTopRightLon,
                      hBottomLeftLat,
                      hBottomLeftLon;

    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 4];
    UN8 un8NumFixed = 0;    // Keep track of the fixed objects
    BOOLEAN bOk;
    N32 n32FixedTopRightLat,
        n32FixedTopRightLon,
        n32FixedBottomLeftLat,
        n32FixedBottomLeftLon;

    do
    {
        // Create the objects.  If this fails, LOCATION_bTopRight
        // will result in an error
        hTopRightLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hTopRightLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the top right lat, lon
        bOk = LOCATION_bTopRight(hLocation, hTopRightLat, hTopRightLon);

        if (bOk == FALSE)
        {
            break;
        }

        // Create the objects.  If this fails, LOCATION_bBottomLeft
        // will result in an error
        hBottomLeftLat = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);
        hBottomLeftLon = OSAL_FIXED.hCreateInMemory(0, 0,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE*(un8NumFixed++)]);

        // Get the bottom left lat, lon
        bOk = LOCATION_bBottomLeft(hLocation, hBottomLeftLat, hBottomLeftLon);

        if (bOk == FALSE)
        {
            break;
        }

        n32FixedTopRightLat =
            OSAL_FIXED.n32ScaledValue(hTopRightLat, LOCATION_BINPOINT);
        n32FixedTopRightLon =
            OSAL_FIXED.n32ScaledValue(hTopRightLon, LOCATION_BINPOINT);

        n32FixedBottomLeftLat =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLat, LOCATION_BINPOINT);
        n32FixedBottomLeftLon =
            OSAL_FIXED.n32ScaledValue(hBottomLeftLon, LOCATION_BINPOINT);

        // Specify a search box within the region
        snprintf( &psObj->pacBuffer[0], psObj->tBufferSize,
            FUEL_SELECT_STATION_DESC_BY_LOCATION, tRegion,
            n32FixedBottomLeftLat, n32FixedTopRightLat, // lat min / max
            n32FixedBottomLeftLon, n32FixedTopRightLon); // lon min /max

        bQueryBuilt = TRUE;
    } while (FALSE);

    return bQueryBuilt;
}

/*****************************************************************************
*
*   bLoadTables
*
*****************************************************************************/
static BOOLEAN bLoadTables (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        )
{
    BOOLEAN bSuccess;

    do
    {
        // Load the region table
        bSuccess = bLoadRegionTable(psObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Load the text table
        bSuccess = bLoadTextTable(psObj);
        if (bSuccess == FALSE)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bLoadTextTable
*
*****************************************************************************/
static BOOLEAN bLoadTextTable (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        )
{
    FUEL_PRICE_DB_RESULT_STRUCT sTextResult;
    BOOLEAN bOk;

    // Initialize the result struct
    sTextResult.bSuccess = TRUE;
    sTextResult.psObj = psObj;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
        psObj->hSQLPersistConnection,
        FUEL_SELECT_ALL_TEXT,
        (SQL_QUERY_RESULT_HANDLER)bProcessSelectText,
        &sTextResult ) ;

    if (bOk == TRUE)
    {
        bOk = sTextResult.bSuccess;
    }

    return bOk;
}

/*****************************************************************************
*
*   bLoadRegionTable
*
*****************************************************************************/
static BOOLEAN bLoadRegionTable (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        )
{
    FUEL_PRICE_DB_RESULT_STRUCT sRegionResult;
    BOOLEAN bOk;

    // Initialize the result struct
    sRegionResult.bSuccess = FALSE;
    sRegionResult.psObj = psObj;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
            psObj->hSQLRefConnection, FUEL_SELECT_ALL_REGIONS,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectRegions,
            &sRegionResult ) ;

    if (bOk == TRUE)
    {
        bOk = sRegionResult.bSuccess;
    }

    return bOk;
}

/*****************************************************************************
*
*   bExpandBuffer
*
*   Expand the scratch buffer to the requested size.  If bCopy == TRUE, then
*   copy the contents of the old buffer to the new one
*
*****************************************************************************/
static BOOLEAN bExpandBuffer (
    char **ppacBuffer,
    size_t *ptBufferSize,
    size_t tNewSizeRequired,
    BOOLEAN bCopy
        )
{
    char *pacNewBuffer;

    // Add some margin to the requested size
    tNewSizeRequired += FUEL_PRICES_BUFFER_INCREASE_MARGIN;

    // Now, allocate the new space we want
    pacNewBuffer = (char *)
        SMSO_hCreate(
            FUEL_PRICE_NAME": Buffer",
            tNewSizeRequired,
            SMS_INVALID_OBJECT, FALSE );

    if (pacNewBuffer != NULL)
    {
        // Do we have a previous valid buffer?
        if (*ppacBuffer != NULL)
        {
            // Were we asked to copy its contents
            // into the new buffer?
            if (bCopy == TRUE)
            {
                // Copy the entire contents of the old
                // buffer into the new buffer
                OSAL.bMemCpy(
                    &pacNewBuffer[0], *ppacBuffer,
                    *ptBufferSize);
            }

            // Destroy the old buffer
            SMSO_vDestroy((SMS_OBJECT)*ppacBuffer);
        }

        // Use the new buffer now
        *ppacBuffer = pacNewBuffer;
        *ptBufferSize = tNewSizeRequired;

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bFormatTextField
*
*   Format a text field for the SQL interpreter.  Escapes all instances
*   of the ' character by adding a following ' (so, ' becomes '')
*
*****************************************************************************/
static BOOLEAN bFormatTextField (
    char **ppacBuffer,
    size_t *ptBufferSize,
    size_t *ptBufferIndex
        )
{
    char *pacSingleQuote;
    size_t tSearchIndex = *ptBufferIndex,
           tIndex;
    ptrdiff_t tQuoteIndex;
    BOOLEAN bSkip;
    char *pacBuffer = *ppacBuffer;

    do
    {
        // Does the single-quote appear in this buffer?
        pacSingleQuote = strchr(&pacBuffer[tSearchIndex], '\'');
        if (pacSingleQuote != NULL)
        {
            // Yes, a single quote has been found in the
            // buffer -- we must escape it for the SQL interpreter
            tQuoteIndex = pacSingleQuote - &pacBuffer[0];
            bSkip = FALSE;

            tSearchIndex = tQuoteIndex + 1 + 1;

            // We're going to have to add a character to the
            // buffer -- is there space to do so?
            if (*ptBufferSize < (*ptBufferIndex + 1))
            {
                BOOLEAN bOk;

                // We need to expand the buffer, and copy the
                // contents from the old buffer
                bOk = bExpandBuffer( ppacBuffer, ptBufferSize, (*ptBufferIndex + 1), TRUE );
                if (bOk == FALSE)
                {
                    // Can't expand the buffer!
                    return FALSE;
                }

                pacBuffer = *ppacBuffer;
            }

            // Get the length of this string
            tIndex = strlen(&pacBuffer[0]);

            // Has this single quote already been escaped?
            if (tQuoteIndex + 1 < (ptrdiff_t)tIndex)
            {
                if (pacBuffer[tQuoteIndex + 1] == '\'')
                {
                    bSkip = TRUE;
                }
            }

            if (bSkip == FALSE)
            {
                // Shift all characters over,
                // including the ' character -- we'll
                // end with a copy of the ' character next
                // to the original
                while ((ptrdiff_t)tIndex > tQuoteIndex)
                {
                    pacBuffer[tIndex] = pacBuffer[tIndex-1];
                    tIndex--;
                }

                // We just added a character
                (*ptBufferIndex)++;
            }
        }
    } while (pacSingleQuote != NULL);

    return TRUE;
}

/*****************************************************************************
*
*   vStartTransaction
*
*    This is for starting an SQL transaction
*
*****************************************************************************/
static void vStartTransaction (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        )
{
    if (psObj->bInTransaction == FALSE)
    {
        // Start a transaction for all the upcoming
        // station updates
        psObj->bInTransaction =
            SQL_INTERFACE.bStartTransaction(psObj->hSQLPersistConnection);
    }

    return;
}

/*****************************************************************************
*
*   vEndTransaction
*
*    This is for ending an SQL transaction
*
*****************************************************************************/
static void vEndTransaction (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        )
{
    if (psObj->bInTransaction == TRUE)
    {
        // End it now
        psObj->bInTransaction =
            !SQL_INTERFACE.bEndTransaction(psObj->hSQLPersistConnection);
    }

    return;
}

/*****************************************************************************
*
*   bCreatePersistDBTables
*
*   Creates the default tables in the persistent database
*
*****************************************************************************/
static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        )
{
    BOOLEAN bSuccess = FALSE, bTransactionStarted = FALSE;
    FUEL_PRICE_DB_OBJ_STRUCT *psObj = (FUEL_PRICE_DB_OBJ_STRUCT *)pvArg;

    do
    {
        if (psObj == NULL)
        {
            break;
        }

        // Using direct call instead of vStartTransaction since
        // we don't know we have valid database handle until this
        // call returns
        bTransactionStarted = SQL_INTERFACE.bStartTransaction(hSQLConnection);

        if (bTransactionStarted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME
                ": failed to start transaction");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            FUEL_CREATE_PRICES_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME
                ": failed to create prices table in persistant storage");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            FUEL_CREATE_TEXT_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME
                ": failed to create text table in persistent storage");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
            FUEL_CREATE_VERSION_TABLE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME
                ": failed to create version table in persistent storage");
            break;
        }

        // Build the text version string
        snprintf(psObj->pacBuffer, psObj->tBufferSize,
            FUEL_INSERT_VERSION_ROW,
            FUEL_PRICE_DB_VERSION_TYPE_TEXT,
            FUEL_PRICE_TEXT_INITIAL_VALUE);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection, psObj->pacBuffer);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME
                ": failed to set text version in persistent storage");
            break;
        }

        // Build the version string
        snprintf(psObj->pacBuffer, psObj->tBufferSize,
            FUEL_INSERT_VERSION_ROW,
            FUEL_PRICE_DB_VERSION_TYPE_DB,
            psObj->sVerCtrl.n16DBVersion);

        bSuccess = SQL_INTERFACE.bExecuteCommand(hSQLConnection, psObj->pacBuffer);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME
                ": failed to set db version in persistent storage");
            break;
        }
    }
    while (FALSE);

    if (bTransactionStarted == TRUE)
    {
        BOOLEAN bTransactionEnded;

        // Using direct call instead of vEndTransaction since
        // we don't know we have valid database handle until this
        // call returns
        bTransactionEnded = SQL_INTERFACE.bEndTransaction(hSQLConnection);

        if (bTransactionEnded == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_PRICE_NAME
                ": failed to end transaction");
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareTextColumn
*
*****************************************************************************/
static BOOLEAN bPrepareTextColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        )
{
    BOOLEAN bSuccess = TRUE;
    FUEL_PRICE_DB_TEXT_FIELDS_ENUM eCurrentField =
        (FUEL_PRICE_DB_TEXT_FIELDS_ENUM)tIndex;

    switch (eCurrentField)
    {
        case FUEL_PRICE_DB_TEXT_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTextRow->un8TextId;
        }
        break;

        case FUEL_PRICE_DB_TEXT_BRAND:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psTextRow->bBrandText;
        }
        break;

        case FUEL_PRICE_DB_TEXT_VALUE:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = (void *)(size_t)psTextRow->hText;
        }
        break;

        case FUEL_PRICE_DB_TEXT_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPrepareRegionColumn
*
*****************************************************************************/
static BOOLEAN bPrepareRegionColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_REGION *ptRegionID
        )
{
    BOOLEAN bSuccess = TRUE;
    FUEL_DB_REGION_FIELDS_ENUM eCurrentField =
        (FUEL_DB_REGION_FIELDS_ENUM)tIndex;

    switch (eCurrentField)
    {
        case FUEL_DB_REGION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(*ptRegionID);
        }
        break;

        case FUEL_DB_REGION_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPreparePriceColumn
*
*****************************************************************************/
static BOOLEAN bPreparePriceColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_DB_UPDATE_PRICE_TABLE_STRUCT *psDBUpdate
        )
{
    BOOLEAN bSuccess = TRUE;
    FUEL_PRICE_DB_PRICE_FIELDS_ENUM eCurrentField =
        (FUEL_PRICE_DB_PRICE_FIELDS_ENUM)tIndex;

    switch (eCurrentField)
    {
        case FUEL_PRICE_DB_PRICE_REGION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.tRegion;
        }
        break;

        case FUEL_PRICE_DB_PRICE_STATION_ID:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.tStationId;
        }
        break;

        case FUEL_PRICE_DB_PRICE_BRAND:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hBrand;
        }
        break;

        case FUEL_PRICE_DB_PRICE_NAME:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hName;
        }
        break;

        case FUEL_PRICE_DB_PRICE_ADDR:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hAddr;
        }
        break;

        case FUEL_PRICE_DB_PRICE_CITY:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hCity;
        }
        break;

        case FUEL_PRICE_DB_PRICE_STATE:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)psDBUpdate->sStationRow.tStateID;
        }
        break;

        case FUEL_PRICE_DB_PRICE_ZIP:
        {
            *peType = SQL_BIND_TYPE_STRING_OBJECT;
            *ppvData = psDBUpdate->sStationRow.hZIP;
        }
        break;

        case FUEL_PRICE_DB_PRICE_PHONE:
        {
            *peType = SQL_BIND_TYPE_N64;
            *ppvData = (void *)(&psDBUpdate->sStationRow.n64Phone);
        }
        break;

        case FUEL_PRICE_DB_PRICE_LAT:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sStationRow.n32Lat;
        }
        break;

        case FUEL_PRICE_DB_PRICE_LON:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sStationRow.n32Lon;
        }
        break;

        case FUEL_PRICE_DB_PRICE_AMENITIES:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sStationRow.un32Amenities;
        }
        break;

        case FUEL_PRICE_DB_PRICE_AGE_EPOCH_SECONDS:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sPriceRow.un32YoungestPriceAgeUTCSeconds;
        }
        break;

        case FUEL_PRICE_DB_PRICE_NUM_PRICES:
        {
            *peType = SQL_BIND_TYPE_UN32;
            *ppvData = (void *)(size_t)(UN32)psDBUpdate->sPriceRow.tNumPrices;
        }
        break;

        case FUEL_PRICE_DB_PRICE_PRICE_LIST:
        {
            *peType = SQL_BIND_TYPE_BLOB;

            if (psDBUpdate->sPriceRow.tNumPrices == 0)
            {
                *ppvData = (void *)NULL;
                *ptDataSize = 0;
            }
            else
            {
                *ppvData = (void *)&psDBUpdate->sPriceRow.pasPrices[0];
                *ptDataSize = psDBUpdate->sPriceRow.tNumPrices * sizeof(FUEL_PRICE_ENTRY_STRUCT);
            }
        }
        break;

        case FUEL_PRICE_DB_PRICE_MAX_FIELDS:
        default:
            bSuccess = FALSE;
        break;

    }

    return bSuccess;
}

/*****************************************************************************
*
*   bVerifyAndLoadDBVersionFields
*
*****************************************************************************/
static BOOLEAN bVerifyAndLoadDBVersionFields (
    SQL_INTERFACE_OBJECT hSQLConnection,
    FUEL_PRICE_VER_CTRL_STRUCT *psVerCtrl
        )
{
    FUEL_PRICE_DB_VERSION_RESULT_STRUCT sVersion;
    BOOLEAN bVersionMatched = FALSE, bOk;

    // Initialize the result structure
    sVersion.bSuccess = FALSE;
    sVersion.psVerCtrl = psVerCtrl;

    // Perform the SQL query and process the result
    // (it will provide us with a data row)
    bOk = SQL_INTERFACE.bQuery(
             hSQLConnection, FUEL_SELECT_ALL_DB_VERSION,
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectVersion,
            &sVersion ) ;

    if (bOk == TRUE)
    {
        bVersionMatched = sVersion.bSuccess;
    }

    return bVersionMatched;
}

/*****************************************************************************
*
*   n32ExtractDataVersion
*
*****************************************************************************/
static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        )
{
    BOOLEAN bVerified;
    FUEL_PRICE_VER_CTRL_STRUCT sVerCtrl;
    N32 n32Result = DB_UTIL_DB_UNDER_CONSTRUCTION_VER;

    if (hConnection == SQL_INTERFACE_INVALID_OBJECT)
    {
        return DB_UTIL_DB_UNDER_CONSTRUCTION_VER;
    }

    sVerCtrl.n16DBVersion = *((N16*)pvArg);

    // Get the version fields
    bVerified = bVerifyAndLoadDBVersionFields(
        hConnection, &sVerCtrl);
    if (bVerified == TRUE)
    {
        n32Result = (N32)sVerCtrl.n16DBContentVersion;
    }

    return n32Result;
}

/*******************************************************************************
*
*   bProcessSelectVersion
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database version relation.  It populates
*   a pointer to a FUEL_VERSION_ROW_STRUCT with the information found
*   in the database as well as performs version verification.
*
*******************************************************************************/
static BOOLEAN bProcessSelectVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_VERSION_RESULT_STRUCT *psResult
        )
{
    FUEL_PRICE_DB_VERSION_FIELDS_ENUM eCurrentField =
        (FUEL_PRICE_DB_VERSION_FIELDS_ENUM)0;
    FUEL_PRICE_VERSION_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // Initialize row structure
    sCurrentRow.eType = FUEL_PRICE_DB_VERSION_MAX_TYPES;
    sCurrentRow.n32Version = -1;

    // If we have the correct number of columns, then we have good results
    if ( n32NumberOfColumns == FUEL_PRICE_DB_VERSION_MAX_FIELDS )
    {
        // Start off by marking this a success
        psResult->bSuccess = TRUE;
    }
    else
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case FUEL_PRICE_DB_VERSION_TYPE:
            {
                sCurrentRow.eType = (FUEL_PRICE_DB_VERSION_TYPES_ENUM)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case FUEL_PRICE_DB_VERSION_VER:
            {
                sCurrentRow.n32Version =(N32)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_PRICE_NAME": bad field in ver tabl");
            }
            break;
        }

    } while ( ++eCurrentField < FUEL_PRICE_DB_VERSION_MAX_FIELDS);

    // Now process the version data
    switch(sCurrentRow.eType)
    {
        case FUEL_PRICE_DB_VERSION_TYPE_DB:
        {
            if (sCurrentRow.n32Version != (N32)psResult->psVerCtrl->n16DBVersion)
            {
                printf(FUEL_PRICE_NAME": DB version fail");
                psResult->bSuccess = FALSE;
            }
        }
        break;

        case FUEL_PRICE_DB_VERSION_TYPE_TEXT:
        {
            // Just pull this value in
            psResult->psVerCtrl->n16TextVer =
                (N16)sCurrentRow.n32Version;
        }
        break;

        case FUEL_PRICE_DB_VERSION_TYPE_CONTENTS:
        {
            // Just pull this value in
            psResult->psVerCtrl->n16DBContentVersion =
                (N16)sCurrentRow.n32Version;
        }
        break;

        case FUEL_PRICE_DB_VERSION_MAX_TYPES:
        {
            // Nothing to do
        }
        break;
    }

    // FALSE will stop iteration; TRUE will keep going
    return psResult->bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectText
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database text table.  It populates the
*   manager's text table based upon the data found in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectText (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_RESULT_STRUCT *psResult
        )
{
    FUEL_PRICE_DB_TEXT_FIELDS_ENUM eCurrentField =
        (FUEL_PRICE_DB_TEXT_FIELDS_ENUM)0;
    FUEL_TEXT_ROW_STRUCT sCurrentRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // Initialize the row structure
    OSAL.bMemSet((void *)&sCurrentRow, 0, sizeof(FUEL_TEXT_ROW_STRUCT));

    // We don't support long text
    sCurrentRow.hLongText = STRING_INVALID_OBJECT;

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case FUEL_PRICE_DB_TEXT_ID:
            {
                sCurrentRow.un8TextId = (UN8)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case FUEL_PRICE_DB_TEXT_BRAND:
            {
                sCurrentRow.bBrandText = (BOOLEAN)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case FUEL_PRICE_DB_TEXT_VALUE:
            {
                // Create a string object for this
                sCurrentRow.hText = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sCurrentRow.hText == STRING_INVALID_OBJECT)
                {
                    psResult->bSuccess = FALSE;
                }
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_PRICE_NAME": bad field in text table");
            }
            break;
        }

        // Stop it we experienced an error
        if (psResult->bSuccess == FALSE)
        {
            break;
        }

    } while ( ++eCurrentField < FUEL_PRICE_DB_TEXT_MAX_FIELDS);

    if (psResult->bSuccess == TRUE)
    {
        // Tell the manager about this text
        psResult->bSuccess = GsFuelMgrIntf.bReportTextFromDB(
            psResult->psObj->hFuelService, &sCurrentRow);
    }

    // Stop if we found an error
    if (psResult->bSuccess == FALSE)
    {
        if (sCurrentRow.hText != STRING_INVALID_OBJECT)
        {
            // Free the string
            STRING_vDestroy(sCurrentRow.hText);
            sCurrentRow.hText = STRING_INVALID_OBJECT;
        }

        // Stop iterating
        return FALSE;
    }

    // Keep going
    return TRUE;
}

/*******************************************************************************
*
*   bProcessSelectRegionExists
*
*******************************************************************************/
static BOOLEAN bProcessSelectRegionExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbRegionExists
        )
{
    // Verify input
    if (pbRegionExists == NULL)
    {
        // Stop now
        return FALSE;
    }

    // We should only have 1 column here
    if (n32NumberOfColumns == 1 )
    {
        if ((psColumn[0].eType == SQL_COLUMN_TYPE_INTEGER) &&
            (psColumn[0].uData.sUN32.un32Data == 1))
        {
            // We performed a "select count(1)" on this
            // region ID and we got a "1" in response...good
            *pbRegionExists = TRUE;
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectRegions
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database region table.  It populates the
*   manager's region list based upon the data found in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectRegions (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_RESULT_STRUCT *psResult
        )
{
    FUEL_DB_REGION_FIELDS_ENUM eCurrentField =
        (FUEL_DB_REGION_FIELDS_ENUM)0;
    FUEL_REGION tRegion = FUEL_INVALID_REGION;

    // Verify input
    if (psResult == NULL)
    {
        // Stop now
        return FALSE;
    }

    // If we have the correct number of columns, then we have good results
    if (n32NumberOfColumns == FUEL_DB_REGION_MAX_FIELDS )
    {
        // Start off by marking this a success
        psResult->bSuccess = TRUE;
    }
    else
    {
        // This is definitely not a match
        psResult->bSuccess = FALSE;

        // Stop now
        return FALSE;
    }

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case FUEL_DB_REGION_ID:
            {
                tRegion = (FUEL_REGION)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_PRICE_NAME": bad field in region table");
            }
            break;
        }

    } while ( ++eCurrentField < FUEL_DB_REGION_MAX_FIELDS);

    // Only add this region if we extracted it without error
    if (psResult->bSuccess == TRUE)
    {
        // Add this region to our table
        psResult->bSuccess = GsFuelMgrIntf.bReportRegionFromDB(
            psResult->psObj->hFuelService, tRegion);
    }

    // Stop if we encountered an error
    return psResult->bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectStations
*
*******************************************************************************/
static BOOLEAN bProcessSelectStations (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_LOAD_STATION_STRUCT *psBuild
        )
{
    FUEL_PRICE_DB_STATION_FIELDS_ENUM eCurrentField =
        (FUEL_PRICE_DB_STATION_FIELDS_ENUM)0;
    FUEL_STATION_ROW_STRUCT sStationRow;
    BOOLEAN bStationInUse = FALSE,
            bStationNeeded = TRUE,
            bOk = TRUE,
            bStationOwnsStrings = FALSE,
            bCacheOnly = TRUE;
    FUEL_STATION_OBJECT hStation =
        FUEL_STATION_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT,
                      hLon = OSAL_FIXED_INVALID_OBJECT;

    // Verify input
    if (psBuild == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns (just make sure we have enough),
    // then we have good results
    if (n32NumberOfColumns < FUEL_PRICE_DB_STATION_MAX_FIELDS )
    {
        psBuild->bSuccess = FALSE;

        return FALSE;
    }

    // Initialize the row structure
    OSAL.bMemSet(&sStationRow, 0, sizeof(FUEL_STATION_ROW_STRUCT));

    // Populate the station row's region
    sStationRow.tRegion = psBuild->tRegion;

    // Get the station ID
    sStationRow.tStationId = (FUEL_STATION_ID)
        psColumn[FUEL_PRICE_DB_STATION_ID].uData.sUN32.un32Data;

    // Get the lat/lon of this station
    sStationRow.n32Lat = (N32)
            psColumn[FUEL_PRICE_DB_STATION_LAT].uData.sUN32.un32Data;
    sStationRow.n32Lon = (N32)
            psColumn[FUEL_PRICE_DB_STATION_LON].uData.sUN32.un32Data;

    // Create some fixed objects on the stack if the database
    // has valid values
    if ((sStationRow.n32Lat != -1) && (sStationRow.n32Lon != -1))
    {
        hLat = OSAL_FIXED.hCreateInMemory(
            sStationRow.n32Lat, LOCATION_BINPOINT,
            &atFixedData[0]);
        hLon = OSAL_FIXED.hCreateInMemory(
            sStationRow.n32Lon, LOCATION_BINPOINT,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE]);
    }

    // If this this a not a query on a particular station,
    // then we need to check the location criteria
    if (psBuild->tStation == FUEL_INVALID_STATION_ID)
    {
        BOOLEAN bLocationAccepted;

        // Does the station meet our background location criteria?
        // Note: could we improve performance with r-tree query and remove this call?
        bLocationAccepted = GsFuelMgrIntf.bBackgroundLocationFilter(
            psBuild->psObj->hFuelService, hLat, hLon, &bCacheOnly);

        if (bLocationAccepted == FALSE)
        {
            // We're done now with this station,
            // but keep iterating
            return TRUE;
        }
    }
    else // This is a query for a particular station
    {
        // Is this the one we asked for?
        if (psBuild->tStation != sStationRow.tStationId)
        {
            // This isn't the correct station
            return TRUE;
        }

        // This is a request to place a station into the DSRL
        bCacheOnly = FALSE;
    }

    // Attempt to find this station to see if
    // it's already in service.  We may already be
    // tracking this station, and if so, we can skip
    // the database processing code below
    hStation = GsFuelMgrIntf.hGetStation(
        psBuild->psObj->hFuelService,
        sStationRow.tRegion, sStationRow.tStationId,
        &bStationNeeded);
    if (hStation != FUEL_STATION_INVALID_OBJECT)
    {
        // This station is available...
        // We can skip the database processing below
        // now, so setup the loop to quit immediately
        eCurrentField = FUEL_PRICE_DB_STATION_MAX_FIELDS;

        // This station is in use by the manager already
        bStationInUse = TRUE;
    }

    if (bStationNeeded == FALSE)
    {
        // We don't want this station
        // -- keep iterating
        return TRUE;
    }

    // We got past the location filter -- grab the
    // entire station record and create a station object
    do
    {
        switch (eCurrentField)
        {
            case FUEL_PRICE_DB_STATION_ID:
            {
                // We already got this
            }
            break;

            case FUEL_PRICE_DB_STATION_BRAND:
            {
                sStationRow.hBrand = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hBrand == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case FUEL_PRICE_DB_STATION_NAME:
            {
                sStationRow.hName = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hName == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case FUEL_PRICE_DB_STATION_ADDR:
            {
                sStationRow.hAddr = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hAddr == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case FUEL_PRICE_DB_STATION_CITY:
            {
                sStationRow.hCity = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case FUEL_PRICE_DB_STATION_STATE:
            {
                sStationRow.tStateID = (STATE_ID)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case FUEL_PRICE_DB_STATION_ZIP:
            {
                sStationRow.hZIP = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case FUEL_PRICE_DB_STATION_PHONE:
            {
                sStationRow.n64Phone =
                    psColumn[eCurrentField].n64NativeInteger;
            }
            break;

            case FUEL_PRICE_DB_STATION_LAT:
            case FUEL_PRICE_DB_STATION_LON:
            {
                // We already got these
            }
            break;

            case FUEL_PRICE_DB_STATION_AMENITIES:
            {
                sStationRow.un32Amenities =
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }

        // Stop if we experienced an error
        if (bOk == FALSE)
        {
            break;
        }
    } while (++eCurrentField < FUEL_PRICE_DB_STATION_MAX_FIELDS);

    if (bOk == TRUE)
    {
        if (hStation == FUEL_STATION_INVALID_OBJECT)
        {
            // Now, create a station object and take ownership
            // of the objects in the row
            hStation = GsFuelMgrIntf.hCreateStation(
                psBuild->psObj->hFuelService, &sStationRow);

            // We have created an object now, (well, at least we tried)
            psBuild->tNumObjectsCreated++;
        }

        if (hStation != FUEL_STATION_INVALID_OBJECT)
        {
            // The fuel station object owns the STRING objects now
            bStationOwnsStrings = TRUE;

            // Have the manager put this station into service
            bStationInUse |= GsFuelMgrIntf.bPutStationInService(
                psBuild->psObj->hFuelService,
                hStation, bCacheOnly);

            // Is the station in use now?
            if (FALSE == bStationInUse)
            {
                // We need to destroy this station now
                FUEL_STATION_vDestroy(hStation);
            }
        }
    }
    else
    {
        psBuild->bSuccess = FALSE;

        // We need to destroy all strings directly
        if (bStationOwnsStrings == FALSE)
        {
            if (sStationRow.hAddr != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hAddr);
                sStationRow.hAddr = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hBrand != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hBrand);
                sStationRow.hBrand = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hCity != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hCity);
                sStationRow.hCity = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hName);
                sStationRow.hName = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hZIP != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hZIP);
                sStationRow.hZIP = STRING_INVALID_OBJECT;
            }
        }
    }

    // FALSE will stop iteration; TRUE will keep going
    return psBuild->bSuccess;
}

/*******************************************************************************
*
*   bProcessSelectStationExists
*
*******************************************************************************/
static BOOLEAN bProcessSelectStationExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbStationExists
        )
{
    // Verify input
    if (NULL == pbStationExists)
    {
        // Stop now
        return FALSE;
    }

    // We should only have 1 column here
    if (1 == n32NumberOfColumns)
    {
        if ((SQL_COLUMN_TYPE_INTEGER == psColumn[0].eType) &&
            (1 == psColumn[0].uData.sUN32.un32Data))
        {
            // We performed a "select count(1)" on this
            // station ID and we got a "1" in response...good
            *pbStationExists = TRUE;
        }
    }

    // Stop processing
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectPrices
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select" query made on the database price table.  It creates FUEL_STATION
*   objects (if necessary) for a target and populates those stations with prices
*   (again, if necessary).  These stations are then passed to the target's
*   DSRL to determine if they are wanted.
*
*******************************************************************************/
static BOOLEAN bProcessSelectPrices (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_DB_PRICE_QUERY_RESULT_STRUCT *psResult
        )
{
    FUEL_PRICE_DB_PRICE_FIELDS_ENUM eCurrentField =
        FUEL_PRICE_DB_PRICE_BRAND;
    FUEL_STATION_ROW_STRUCT sStationRow;
    FUEL_PRICE_ROW_STRUCT sPriceRow;
    size_t tNumPrices = 0;
    FUEL_STATION_OBJECT hStation = FUEL_STATION_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
    LOC_ID tLocId;
    BOOLEAN bOk,
            bPriceAccepted,
            bStationNeeded = TRUE,
            bStationInUse = FALSE;
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT,
                      hLon = OSAL_FIXED_INVALID_OBJECT;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there is at least one data row and the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == FUEL_PRICE_DB_PRICE_MAX_FIELDS )
    {
        psResult->bSuccess = TRUE;
    }
    else
    {
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    // Initialize the station & price row
    OSAL.bMemSet(&sStationRow, 0, sizeof(FUEL_STATION_ROW_STRUCT));
    OSAL.bMemSet(&sPriceRow, 0, sizeof(FUEL_PRICE_ROW_STRUCT));

    // First, filter by age
    sPriceRow.un32YoungestPriceAgeUTCSeconds = (UN32)
        psColumn[FUEL_PRICE_DB_PRICE_AGE_EPOCH_SECONDS].uData.sUN32.un32Data;

    // Can we accept the age of this price data?
    bPriceAccepted = bPriceAgeFilter(
        sPriceRow.un32YoungestPriceAgeUTCSeconds,
        psResult->un32CurrentTime );
    if (bPriceAccepted == FALSE)
    {
        // Stop processing this row -- try the next
        return TRUE;
    }

    // Populate the station row's region
    sStationRow.tRegion =
        psColumn[FUEL_PRICE_DB_PRICE_REGION_ID].uData.sUN32.un32Data;

    // Get the station ID
    sStationRow.tStationId =
        psColumn[FUEL_PRICE_DB_PRICE_STATION_ID].uData.sUN32.un32Data;

    // Get the lat/lon of this station
    sStationRow.n32Lat = (N32)
            psColumn[FUEL_PRICE_DB_PRICE_LAT].uData.sUN32.un32Data;
    sStationRow.n32Lon = (N32)
            psColumn[FUEL_PRICE_DB_PRICE_LON].uData.sUN32.un32Data;

    // Create some fixed objects on the stack
    hLat = OSAL_FIXED.hCreateInMemory(
        sStationRow.n32Lat, LOCATION_BINPOINT,
        &atFixedData[0]);
    hLon = OSAL_FIXED.hCreateInMemory(
        sStationRow.n32Lon, LOCATION_BINPOINT,
            &atFixedData[OSAL_FIXED_OBJECT_SIZE]);

    // Generate the loc id
    tLocId = FUEL_REG_STATID_TO_LOCID(
        sStationRow.tRegion, sStationRow.tStationId);

    // Now, filter by location
    bOk = GsFuelMgrIntf.bVerifyQueryResultLocation(
        psResult->psObj->hFuelService, tLocId, hLat, hLon);
    if (bOk == FALSE)
    {
        // We don't want this station...but keep looking
        return TRUE;
    }

    // Is this station already present in the cache?
    hStation = GsFuelMgrIntf.hGetStation(
        psResult->psObj->hFuelService,
        sStationRow.tRegion,
        sStationRow.tStationId,
        &bStationNeeded);

    if (bStationNeeded == FALSE)
    {
        // We don't want this station
        return TRUE;
    }

    if (hStation != FUEL_STATION_INVALID_OBJECT)
    {
        // Skip the row processor
        eCurrentField = FUEL_PRICE_DB_PRICE_MAX_FIELDS;

        // This station is already in use in the cache
        bStationInUse = TRUE;
    }

    // Grab the entire price record
    do
    {
        switch (eCurrentField)
        {
            case FUEL_PRICE_DB_PRICE_REGION_ID:
            case FUEL_PRICE_DB_PRICE_STATION_ID:
            case FUEL_PRICE_DB_PRICE_AGE_EPOCH_SECONDS:
            case FUEL_PRICE_DB_PRICE_LAT:
            case FUEL_PRICE_DB_PRICE_LON:
            {
                // Already done
            }
            break;

            case FUEL_PRICE_DB_PRICE_BRAND:
            {
                sStationRow.hBrand = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hBrand == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case FUEL_PRICE_DB_PRICE_NAME:
            {
                sStationRow.hName = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hName == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case FUEL_PRICE_DB_PRICE_ADDR:
            {
                sStationRow.hAddr = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
                if (sStationRow.hAddr == STRING_INVALID_OBJECT)
                {
                    bOk = FALSE;
                }
            }
            break;

            case FUEL_PRICE_DB_PRICE_CITY:
            {
                sStationRow.hCity = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case FUEL_PRICE_DB_PRICE_STATE:
            {
                sStationRow.tStateID = (STATE_ID)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case FUEL_PRICE_DB_PRICE_ZIP:
            {
                sStationRow.hZIP = STRING.hCreate(
                    (const char *)psColumn[eCurrentField].uData.sCString.pcData,
                    strlen((const char *)psColumn[eCurrentField].uData.sCString.pcData));
            }
            break;

            case FUEL_PRICE_DB_PRICE_PHONE:
            {
                sStationRow.n64Phone =
                    psColumn[eCurrentField].n64NativeInteger;
            }
            break;

            case FUEL_PRICE_DB_PRICE_AMENITIES:
            {
                sStationRow.un32Amenities =
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case FUEL_PRICE_DB_PRICE_NUM_PRICES:
            {
                tNumPrices = (size_t)
                    psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case FUEL_PRICE_DB_PRICE_PRICE_LIST:
            {
                // Verify the size of this blob against the expected size
                size_t tExpectedSize =
                    sizeof(FUEL_PRICE_ENTRY_STRUCT) * tNumPrices;

                if (tNumPrices == 0)
                {
                    sPriceRow.pasPrices = NULL;
                }
                else
                {
                    if (tExpectedSize == psColumn[eCurrentField].uData.sBLOB.tSize)
                    {
                        sPriceRow.pasPrices = (FUEL_PRICE_ENTRY_STRUCT *)
                            psColumn[eCurrentField].uData.sBLOB.pvData;
                    }
                    else
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        FUEL_PRICE_NAME": bProcessSelectPrices() Price data has incorrect size");

                        // Stop processing this now -- try the next
                        return TRUE;
                    }
                }
            }
            break;

            default: // Shouldn't happen
            break;
        }

        if (bOk == FALSE)
        {
            break;
        }
    } while (++eCurrentField < FUEL_PRICE_DB_PRICE_MAX_FIELDS);

    if (bOk == TRUE)
    {
        if (hStation == FUEL_STATION_INVALID_OBJECT)
        {
            size_t tPriceIndex;
            FUEL_PRICE_ENTRY_STRUCT *psCurEntry;

            // Now, create a station object and take ownership
            // of the objects in the row
            hStation = GsFuelMgrIntf.hCreateStation(
                psResult->psObj->hFuelService, &sStationRow);

            for (tPriceIndex = 0; tPriceIndex < tNumPrices; tPriceIndex++)
            {
                // Get the current price entry
                psCurEntry = &sPriceRow.pasPrices[sPriceRow.tNumPrices];

                // Check this entry's age
                bPriceAccepted = bPriceAgeFilter(
                    psCurEntry->un32PriceAgeUTCSeconds,
                    psResult->un32CurrentTime );

                if (bPriceAccepted == TRUE)
                {
                    // Indicate we have another price to report
                    sPriceRow.tNumPrices++;
                }
                else
                {
                    // Shift the prices down to conceal this one
                    vShiftPriceEntries(
                        sPriceRow.pasPrices, tNumPrices, sPriceRow.tNumPrices);
                }
            }

            if (sPriceRow.tNumPrices > 0)
            {
                // Update this station with some prices
                GsFuelMgrIntf.bAddFuelPricesToStation(
                    psResult->psObj->hFuelService,
                    hStation, &sPriceRow,
                    (UN8)psResult->psObj->sVerCtrl.n16TextVer);
            }
        }

        // Have the manager put this station into service
        bStationInUse |= GsFuelMgrIntf.bPutStationInService(
            psResult->psObj->hFuelService,
            hStation, FALSE);

        // Is the station in use now?
        if (bStationInUse == FALSE)
        {
            // Manager didn't want this station and
            // it isn't otherwise in use.
            // We don't need it anymore
            FUEL_STATION_vDestroy(hStation);
            hStation = FUEL_STATION_INVALID_OBJECT;
        }
    }

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_PRICE_NAME": Unable to update station with price from DB.");

        if (hStation == FUEL_STATION_INVALID_OBJECT)
        {
            // Destroy the strings
            if (sStationRow.hAddr != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hAddr);
                sStationRow.hAddr = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hBrand != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hBrand);
                sStationRow.hBrand = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hCity != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hCity);
                sStationRow.hCity = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hName != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hName);
                sStationRow.hName = STRING_INVALID_OBJECT;
            }

            if (sStationRow.hZIP != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(sStationRow.hZIP);
                sStationRow.hZIP = STRING_INVALID_OBJECT;
            }
        }
        else if (bStationInUse == FALSE)
        {
            // Destroy the fuel station
            FUEL_STATION_vDestroy(hStation);
        }

        psResult->bSuccess = FALSE;
    }

    // Keep iterating
    return TRUE;
}
