/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:FUEL_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"
#include "sms_update.h"
#include "sms_event.h"
#include "string_obj.h"
#include "location_obj.h"
#include "dataservice_mgr_impl.h"
#include "dsrl_obj.h"
#include "dsrl_entry_obj.h"
#include "fuel_mgr_obj.h"
#include "_fuel_mgr_obj.h"
#include "fuel_station_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;


/*****************************************************************************
                             COMMON PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   FUEL_MGR_hStart
*
*****************************************************************************/
FUEL_SERVICE_OBJECT FUEL_MGR_hStart (
    DATASERVICE_CREATE_STRUCT *psCreate,
    DATASERVICE_TYPE_ENUM eServiceType,
    BOOLEAN bEnableLogoSupport,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj;
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;
    BOOLEAN bOk;

    // Verify the provided sort method choice
    if (eFuelPriceSortMethod >= FUEL_PRICE_SORT_METHOD_INVALID)
    {
        return FUEL_SERVICE_INVALID_OBJECT;
    }

    bOk = DATASERVICE_IMPL_bProcessOptions(
        FUEL_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
    if (bOk == FALSE)
    {
        // Bad options!
        return FUEL_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    psCreate->tServiceObjectSize = sizeof(FUEL_MGR_OBJECT_STRUCT);

    // This is a multi-DSI service
    psCreate->bMultiDSIRequired = TRUE;
    psCreate->sProductsInfo.bGetDSIForProduct = psOTAInterface->bGetDSIForProduct;
    psCreate->sProductsInfo.eGetNextProductState = psOTAInterface->eNextProductState;
    psCreate->sProductsInfo.peProducts = psOTAInterface->psProducts;
    psCreate->sProductsInfo.un8Count = psOTAInterface->un8NumProducts;

    // Configure the data service's static event attributes
    psCreate->vEventCallback = vEventHandler;
    psCreate->tEventRequestMask = (
        DATASERVICE_EVENT_ALL |
        DATASERVICE_INTERNAL_EVENT_DSRL |
        DATASERVICE_INTERNAL_EVENT_PRODUCT_STATE );

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psObj = (FUEL_MGR_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hCreateNewService( psCreate );
    if (psObj == NULL)
    {
        // Can't create the service, fail out!
        // Free options memory
        DATASERVICE_IMPL_vFreeOptions(&sOptionValues);

        return FUEL_SERVICE_INVALID_OBJECT;
    }

    // We're managing this service
    psObj->eServiceType = eServiceType;

    // Use the provided DB interface
    psObj->psDBInterface = psDBInterface;

    // Use the provided OTA interface
    psObj->psOTAInterface = psOTAInterface;

    do
    {
        DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Update the path for the reference db
        psObj->pacRefDatabaseDirPath = sOptionValues.pcRefDBPath;

        // Create the background radius distance object
        psObj->sWorkCtrl.hBackgroundRadius = DISTANCE.hCreate(
            FUEL_BACKGROUND_PROCESSING_RADIUS_IN_MILES, DISTANCE_UNIT_TYPE_MILES);
        if (psObj->sWorkCtrl.hBackgroundRadius == DISTANCE_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Save the application's sort method
        psObj->sWorkCtrl.eFuelPriceSortMethod = eFuelPriceSortMethod;

        // Initialize our oldest price data attribute
        // to indicate very young data
        psObj->sAgeoutCtrl.un32OldestPriceData = UN32_MAX;

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_NONE,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg);

        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":Targets %u",
            psObj->psOTAInterface->tDataID );

        // Create our list of targets
        eReturnCode = OSAL.eLinkedListCreate(
           &psObj->hTargets,
           &acName[0],
           (OSAL_LL_COMPARE_HANDLER)NULL,
           OSAL_LL_OPTION_NONE
               );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Begin DSRL config
        DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

        // Set service type
        sDSRLConfig.eServiceType = eServiceType;

        // Set parent object size
        sDSRLConfig.tParentObjectSize = sizeof(FUEL_MGR_STATION_OWNER_STRUCT);

        // Set the DSRL descriptor size
        sDSRLConfig.tServiceDataSize = sizeof(FUEL_TARGET_DESC_STRUCT);

        // Configure the favorites feature
        sDSRLConfig.bFavoritesEnabled = TRUE;
        sDSRLConfig.hCreateTargetFromTag = LOCATION_hCreateTargetFromTag;
        sDSRLConfig.hGetTagForTarget = LOCATION_hGetTargetTag;

        // Configure the device feature
        sDSRLConfig.bDeviceEnabled = TRUE;
        sDSRLConfig.un32DeviceNotifyDistance = FUEL_DEVICE_DISTANCE_THRESHOLD;
        sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

        // Create the station owner object
        psObj->sWorkCtrl.psStationOwner = (FUEL_MGR_STATION_OWNER_STRUCT *)
            DATASERVICE_IMPL_hConfigureDSRL(
                (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);
        if ((FUEL_MGR_STATION_OWNER_STRUCT *)NULL == psObj->sWorkCtrl.psStationOwner)
        {
            // Error!
            break;
        }

        // Create a dummy station which provides us the
        // ability to search the station cache efficiently
        psObj->sWorkCtrl.hDummyStation =
            FUEL_STATION_hCreateDummy((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
        if (psObj->sWorkCtrl.hDummyStation == FUEL_STATION_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Get a more convenient pointer for use here
        psOwner = (FUEL_MGR_STATION_OWNER_STRUCT *)psObj->sWorkCtrl.psStationOwner;
        psOwner->psObj = psObj;
        psOwner->pacLogoFileDir = NULL;
        psOwner->pacBaselineLogoFileDir = NULL;
        psOwner->pacLogoFilePath = NULL;
        psOwner->tLogoFilePathLen = 0;

        // Create the name for the text table
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":Text %u",
            psObj->psOTAInterface->tDataID );

        // Create a list of text entries
        eReturnCode = OSAL.eLinkedListCreate(
            &psOwner->hTextEntries,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)n16CompareTextEntries,
            OSAL_LL_OPTION_NONE
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Release our hold on the owner object
        SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);

        // Initialize the OTA state variables
        psObj->sOTACtrl.bDSRL = FALSE;
        psObj->sOTACtrl.psRegion = NULL;
        psObj->sOTACtrl.bInTransaction = FALSE;

        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":Regions %u",
            psObj->psOTAInterface->tDataID );

        // Create a list of region entries
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hRegions,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)n16CompareRegions,
            OSAL_LL_OPTION_BINARY_SEARCH
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":Stations %u",
            psObj->psOTAInterface->tDataID );

        // Create the list of stations maintained as the cache
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->sWorkCtrl.hStations,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)n16CompareStationEntry,
            OSAL_LL_OPTION_RBTREE
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // The service may now start
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

        if (bOk == FALSE)
        {
            break;
        }

        return (FUEL_SERVICE_OBJECT)psObj;
    } while (FALSE);

    // Something bad happened
    vUninitObject( psObj, TRUE );
    DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

    return FUEL_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eFuelTypes
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eFuelTypes (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_TYPE_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bValid, bLocked = FALSE;

    // Verify service object and validate argument
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hFuelService);
    if ((FALSE == bValid) ||
        (NULL == bIterator))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock the station owner
    bLocked = SMSO_bLock((SMS_OBJECT)
        psObj->sWorkCtrl.psStationOwner, OSAL_OBJ_TIMEOUT_INFINITE);

    if (TRUE == bLocked)
    {
        FUEL_TEXT_ITERATOR_STRUCT sIterator;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Populate the iterator struct with the caller's
        // arguments
        sIterator.bIterator = bIterator;
        sIterator.pvIteratorArg = pvIteratorArg;
        sIterator.psDBInterface = psObj->psDBInterface;

        // Iterate the list of fuels in the fuel list
        eOsalReturnCode = OSAL.eLinkedListIterate(
            psObj->sWorkCtrl.psStationOwner->hTextEntries,
            (OSAL_LL_ITERATOR_HANDLER)bIterateTextEntries,
            &sIterator);
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   FUEL_MGR_n64GeneratePhoneFromFields
*
*****************************************************************************/
N64 FUEL_MGR_n64GeneratePhoneFromFields (
    UN16 un16AreaCode,
    UN16 un16Exchange,
    UN16 un16Number
        )
{
    N64 n64Result;

    // Grab the area code
    n64Result = un16AreaCode;

    // Move it out of the way
    n64Result *= FUEL_AREA_CODE_MULTIPLIER;

    // Grab the exchange
    n64Result += un16Exchange;

    // Move all of that out of the way
    n64Result *= FUEL_EXCHANGE_MULTIPLIER;

    // Grab the number
    n64Result += un16Number;

    return n64Result;
}

/*****************************************************************************
*
*   FUEL_MGR_hPhoneToString
*
*****************************************************************************/
STRING_OBJECT FUEL_MGR_hPhoneToString (
    N64 n64Phone
        )
{
    STRING_OBJECT hPhone = STRING_INVALID_OBJECT;

    if (n64Phone != 0)
    {
        char acPhone[FUEL_PHONE_CHAR_ARRAY_SIZE];
        UN16 un16AreaCode;
        UN16 un16Exchange;
        UN16 un16Number;
        UN64 n64Temp;

        // Pull area code out of phone number
        n64Temp = (n64Phone / FUEL_AREA_CODE_DIVISOR);
        un16AreaCode = (UN16)n64Temp;

        // Remove Area code from the number
        n64Temp *= FUEL_AREA_CODE_DIVISOR;
        n64Phone -= n64Temp;

        // Pull exchange out of phone number
        n64Temp = (n64Phone / FUEL_EXCHANGE_MULTIPLIER);
        un16Exchange = (UN16)n64Temp;

        // Remove exchange from the number
        n64Temp *= FUEL_EXCHANGE_MULTIPLIER;
        n64Phone -= n64Temp;

        // Pull the number out now
        un16Number = (UN16)n64Phone;

        // Create a string for the phone number
        snprintf(&acPhone[0], sizeof(acPhone), "(%03u)%03u-%04u",
            un16AreaCode, un16Exchange, un16Number );

        hPhone = STRING.hCreate( &acPhone[0],
                                 FUEL_PHONE_CHAR_ARRAY_SIZE );
    }

    return hPhone;
}

/*****************************************************************************
*
*   FUEL_MGR_eProcessAmenities
*
*   Process amenities for a station using the OTA specification
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM FUEL_MGR_eProcessAmenities (
    SMS_OBJECT hOwner,
    size_t tNumAmenities,
    AMENITY_STRUCT *pasFuelAmenities,
    UN32 un32RawAmenitiesData
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;

    bOwner = SMSO_bOwner(hOwner);
    if (bOwner == TRUE)
    {
        // We may access the OTA interface attribute here since
        // we are guaranteed that it is constant
        eReturnCode = ((FUEL_MGR_STATION_OWNER_STRUCT *)hOwner)->psObj->
            psOTAInterface->eProcessAmenities(
            tNumAmenities,
            pasFuelAmenities,
            un32RawAmenitiesData);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   FUEL_MGR_eGetFuelType
*
*   Fills Fuel Type structure with values from DB by provided fuel type
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM FUEL_MGR_eGetFuelType(
    SMS_OBJECT hOwner,
    UN8 un8FuelType,
    FUEL_TEXT_STRUCT *psFuelText
)
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    bOwner = SMSO_bOwner(hOwner);
    if (bOwner != TRUE)
    {
        eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    else if (psFuelText == NULL)
    {
        eReturnCode = SMSAPI_RETURN_CODE_BAD_ARGUMENT;
    }
    else
    {
        FUEL_MGR_OBJECT_STRUCT *psObj = 
            ((FUEL_MGR_STATION_OWNER_STRUCT *)hOwner)->psObj;
        BOOLEAN bSuccess;

        psFuelText->un8FuelType = un8FuelType;
        psFuelText->eFuelType = psObj->psDBInterface->eMatchFuelType(un8FuelType);
        bSuccess = bGetTextPair(psObj, FALSE, un8FuelType, &psFuelText->hShortFuelName, &psFuelText->hLongFuelName);
        if (bSuccess != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   FUEL_MGR_hBrandLogo
*
*   Get the Brand logo for a station
*
*****************************************************************************/
IMAGE_OBJECT FUEL_MGR_hBrandLogo (
    SMS_OBJECT hOwner,
    FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM eLogoType,
    N16 n16LogoId
        )
{
    IMAGE_OBJECT hLogo = IMAGE_INVALID_OBJECT;

    do
    {
        BOOLEAN bOwner;
        FUEL_LOGO_ENTRY_STRUCT *psLogoEntry;
        FUEL_LOGO_ROW_STRUCT sLogoRow;

        // First, is this logo id valid?
        if (n16LogoId == -1)
        {
            // Nope!
            break;
        }

        bOwner = SMSO_bOwner(hOwner);
        if (bOwner == FALSE)
        {
            break;
        }

        // We're looking for this logo
        sLogoRow.eLogoType = eLogoType;
        sLogoRow.un16LogoId = (UN16)n16LogoId;

        psLogoEntry = psGetLogoEntry((FUEL_MGR_STATION_OWNER_STRUCT *)hOwner,
            &sLogoRow);
        if (psLogoEntry == NULL)
        {
            break;
        }

        hLogo = psLogoEntry->hLogo;

    } while (FALSE);

    return hLogo;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our fuel handle from the callback argument
    psObj = (FUEL_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == FALSE)
    {
        return;
    }

    switch( tCurrentEvent )
    {
        // Handle Fuel Service events here...

        // State has changed
        case DATASERVICE_EVENT_STATE:
        {
            BOOLEAN bStateChanged;
            DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

            // Process the state transition
            bStateChanged = DATASERVICE_IMPL_bStateFSM(
                (DATASERVICE_IMPL_HDL)psObj,
                psStateChange,
                &GsFuelStateHandlers,
                (void *)psObj);

            if (bStateChanged == TRUE)
            {
                // The state has been updated
                tEventMask |= DATASERVICE_EVENT_STATE;

                // Is the service stopped now?
                if (psStateChange->eCurrentState ==
                        DATASERVICE_STATE_STOPPED)
                {
                    bStopEvent = TRUE;
                }
            }
        }
        break;

        // This service has a message to process
        case DATASERVICE_EVENT_NEW_DATA:
        {
            OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
            BOOLEAN bOk = TRUE;

            // Ensure the payload handle is valid
            // If it isn't, there's a problem with
            // SMS
            if (hPayload == OSAL_INVALID_BUFFER_HDL)
            {
                // Set the error
                vSetError( psObj,
                           DATASERVICE_ERROR_CODE_GENERAL);
            }
            else
            {
                printf(FUEL_MGR_OBJECT_NAME": Payload Received (%u)\n",
                    OSAL.tBufferGetSize(hPayload));

                // Process this message now
                bOk = bProcessMessage( psObj, &hPayload );

                if (bOk == TRUE)
                {
                    puts(FUEL_MGR_OBJECT_NAME": Message Payload Processed Ok");
                }
                else
                {
                    DATASERVICE_IMPL_vLog(
                        FUEL_MGR_OBJECT_NAME": Failed to process message\n");
                }
            }

            // We're all done with this payload
            DATASERVICE_IMPL_bFreeDataPayload(hPayload);
        }
        break;

        // We just experienced a price timeout
        case DATASERVICE_EVENT_TIMEOUT:
        {
            BOOLEAN bSuccess = TRUE;
            BOOLEAN bProcessTimer = (BOOLEAN)(size_t)pvEventArg;

            if (bProcessTimer == TRUE)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Iterate the targets to process
                // all that need to be processed
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->hTargets,
                    (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsToProcess,
                    &bSuccess );
                if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
                {
                    // If the OSAL call failed make sure to
                    // indicate failure
                    bSuccess = FALSE;
                }

                if (eReturnCode == OSAL_SUCCESS)
                {
                    OSAL.eLinkedListIterate(
                        psObj->hTargets,
                        (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsForDSRLReady,
                        NULL);
                }

                DATASERVICE_IMPL_vLog(FUEL_MGR_OBJECT_NAME
                    ": Finished processing all targets: %s\n",
                    (bSuccess == TRUE) ? "Success" : "Error");
            }
            else
            {
                // Handle ageout
                bSuccess = bHandleAgeout(psObj);
            }

            if (bSuccess == FALSE)
            {
                // If an error occurred, indicate a state change
                // to the application
                tEventMask |= DATASERVICE_EVENT_STATE;
            }
        }
        break;

        // We need to do some work with our DSRLs
        case DATASERVICE_INTERNAL_EVENT_DSRL:
        {
            BOOLEAN bSuccess = TRUE;
            DSRL_ARG_STRUCT *psDSRLArg =
                (DSRL_ARG_STRUCT *)pvEventArg;
            if (psDSRLArg == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_MGR_OBJECT_NAME": DSRL_EVENT with no targets");
                break;
            }

            switch (psDSRLArg->eAction)
            {
                case DSRL_ACTION_ADD:
                {
                    // Create a new list
                    bSuccess = bHandleCreateList(
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_MODIFY:
                {
                    // Modify a pre-existing list
                    bSuccess = bHandleModifyList (
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_REFRESH:
                {
                    // Refresh a pre-existing list
                    bSuccess = bHandleRefreshList (
                        psObj, psDSRLArg);
                }
                break;

                case DSRL_ACTION_REMOVE:
                {
                    // Handle the deletion
                    vHandleDeleteList(psObj, psDSRLArg);
                    bSuccess = TRUE;
                }
                break;

                case DSRL_ACTION_INVALID:
                default:
                {
                    bSuccess = FALSE;
                }
                break;
            }

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_MGR_OBJECT_NAME": Error processing DSRL_EVENT");

                break;
            }
        }
        break;

        // One of the products has changed state
        case DATASERVICE_INTERNAL_EVENT_PRODUCT_STATE:
        {
            DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState = 
                (DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *)pvEventArg;
            BOOLEAN bSuccess;

            // Have the interface handle the state change
            bSuccess = psObj->psOTAInterface->bProductStateChange(
                psObj->hOTAInterface, psState);

            // Always report a state change so the application
            // knows to inspect the product states now
            tEventMask |= DATASERVICE_EVENT_STATE;

            if (FALSE == bSuccess)
            {
                // Indicate an error
                vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            }
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": Got unknown event (%u)",
                tCurrentEvent);
        }
        break;
    }

    if (bStopEvent == TRUE)
    {
        // Uninitialize the object, but leave
        // enough to be useful to the callback
        vUninitObject( psObj, FALSE );
    }

    // Update event mask with any relevant events which have occurred
    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    // Notify of any change via any registered callback which may be present
    SMSU_bNotify(&psObj->sEvent);

    if (bStopEvent == TRUE)
    {
        // Filter out all further fuel manager updates
        SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_NONE;

    do
    {
        // Register for a timed event
        psObj->sWorkCtrl.hProcessEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)(size_t)TRUE);

        if(psObj->sWorkCtrl.hProcessEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Register for a timed event
        psObj->sAgeoutCtrl.hDataExpireEvent =
            DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)(size_t)FALSE);

        if (psObj->sAgeoutCtrl.hDataExpireEvent ==
            DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Error!
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        if (psObj->hDBInterface == FUEL_DB_INTERFACE_INVALID_OBJECT)
        {
            BOOLEAN bLocked;

            // The DB interface may need to alter values protected
            // by the station owner lock.  So get a hold of that now
            bLocked = SMSO_bLock((SMS_OBJECT)
                psObj->sWorkCtrl.psStationOwner, OSAL_OBJ_TIMEOUT_INFINITE);
            if (FALSE == bLocked)
            {
                // Error!
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }

            // Connect to the DB interface
            psObj->hDBInterface = psObj->psDBInterface->hConnect(
                (FUEL_SERVICE_OBJECT)psObj,
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
                psObj->pacRefDatabaseDirPath,
                psObj->psDBInterface,
                psObj->psOTAInterface,
                &eErrorCode );

            // Unlock now regardless of whatever happened above
            SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);

            if (psObj->hDBInterface == FUEL_DB_INTERFACE_INVALID_OBJECT)
            {
                break;
            }

            // The DB interface now owns this
            psObj->pacRefDatabaseDirPath = NULL;
        }

        if (psObj->hOTAInterface == FUEL_OTA_INTERFACE_INVALID_OBJECT)
        {
            // Start the interface
            psObj->hOTAInterface = psObj->psOTAInterface->hInit(
                (FUEL_SERVICE_OBJECT)psObj,
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
                psObj->hDBInterface, psObj->psDBInterface,
                psObj->psOTAInterface);
            if (psObj->hOTAInterface == FUEL_OTA_INTERFACE_INVALID_OBJECT)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }
        }

        // All is well
        return TRUE;

    } while (FALSE);

    // Set the error we experienced
    vSetError( psObj, eErrorCode );

    // Indicate failure
    return FALSE;
}

/*****************************************************************************
*
*   bHandleServiceError
*
*****************************************************************************/
static BOOLEAN bHandleServiceError (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop any timers we have now 'cause
    // who knows what's going on up there
    vStopAgeoutTimer(psObj);
    vStopProcessTimer(psObj);

    return TRUE;
}

/*****************************************************************************
*
*   bProcessMessage
*
*   This function is called to provide the fuel interface with
*   a newly received message
*
*****************************************************************************/
static BOOLEAN bProcessMessage (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bProcessed = TRUE;

    // Only process this message if
    // the handle is valid
    if (*phPayload != OSAL_INVALID_BUFFER_HDL)
    {
        // Tell the interface it's time to
        // process another message
        bProcessed = psObj->psOTAInterface->bProcessMessage(
            psObj->hOTAInterface, phPayload );
    }

    return bProcessed;
}

/*****************************************************************************
*
*   bHandleAgeout
*
*   This function is called by the event handler when a price timeout
*   occurs.
*
*****************************************************************************/
static BOOLEAN bHandleAgeout (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bSuccess = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32AgeLimitUTC = 0;

    // Get the current time (UTC)
    eReturnCode = OSAL.eTimeGet(&un32AgeLimitUTC);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bHandleAgeout() OSAL.eTimeGet () failure");
        printf("Result: %s\n",
               OSAL.pacGetReturnCodeName(eReturnCode));

        // We can't get the time
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

        return FALSE;
    }

    // Ensure that the clock is set correctly(-ish)
    if (un32AgeLimitUTC <= FUEL_ZERO_DATE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bHandleAgeout() bad time reported");

        // We can't get the time
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

        return FALSE;
    }

    // Data which is at least FUEL_AGE_MAX_IN_SECS
    // old must be aged out now.  Compute that
    // in epoch seconds
    un32AgeLimitUTC -= FUEL_AGE_MAX_IN_SECS;

    // Reset the oldest processed
    psObj->sAgeoutCtrl.un32OldestProcessedPriceData = UN32_MAX;

    // Iterate the DSRLs to remove all price entries
    // using old text versions, get the next
    // oldest price time stamp
    bSuccess = bAgeoutPrices(
        psObj, un32AgeLimitUTC,
        &psObj->sAgeoutCtrl.un32OldestProcessedPriceData);
    if (bSuccess == TRUE)
    {
        // Reset the oldest
        psObj->sAgeoutCtrl.un32OldestPriceData = UN32_MAX;

        // Start the timer again
        vStartAgeoutTimer(psObj);
    }

    if (bSuccess == FALSE)
    {
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwnerLocked = FALSE;

    // Clear the timed event handles, we're done with them
    psObj->sWorkCtrl.hProcessEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;
    psObj->sAgeoutCtrl.hDataExpireEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

    // Destroy the DISTANCE object
    if (psObj->sWorkCtrl.hBackgroundRadius != DISTANCE_INVALID_OBJECT)
    {
        DISTANCE.vDestroy(psObj->sWorkCtrl.hBackgroundRadius);
        psObj->sWorkCtrl.hBackgroundRadius = DISTANCE_INVALID_OBJECT;
    }

    // Destroy the target list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psObj->hTargets,
        (OSAL_LL_RELEASE_HANDLER)vDestroyTarget);

    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->hTargets);
        psObj->hTargets = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->pacRefDatabaseDirPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacRefDatabaseDirPath);
        psObj->pacRefDatabaseDirPath = NULL;
    }

    // Destroy the region list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psObj->hRegions,
        (OSAL_LL_RELEASE_HANDLER)vReleaseRegionEntry);
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psObj->hRegions);
        psObj->hRegions = OSAL_INVALID_OBJECT_HDL;
    }

    // Lock the station owner now if possible
    bOwnerLocked = SMSO_bLock(
        (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
        OSAL_OBJ_TIMEOUT_INFINITE);

    // Destroy station owner attributes if it locked
    if (TRUE == bOwnerLocked)
    {
        // Logo list
        if (psObj->sWorkCtrl.psStationOwner->hLogos != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->sWorkCtrl.psStationOwner->hLogos,
                (OSAL_LL_RELEASE_HANDLER)vReleaseLogoEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                OSAL.eLinkedListDelete(psObj->sWorkCtrl.psStationOwner->hLogos);
                psObj->sWorkCtrl.psStationOwner->hLogos = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Logo path info
        if (psObj->sWorkCtrl.psStationOwner->pacLogoFileDir != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner->pacLogoFileDir);
            psObj->sWorkCtrl.psStationOwner->pacLogoFileDir = NULL;
        }

        if (psObj->sWorkCtrl.psStationOwner->pacBaselineLogoFileDir != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)
                psObj->sWorkCtrl.psStationOwner->pacBaselineLogoFileDir);
            psObj->sWorkCtrl.psStationOwner->pacBaselineLogoFileDir = NULL;
        }

        if (psObj->sWorkCtrl.psStationOwner->pacLogoFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner->pacLogoFilePath);
            psObj->sWorkCtrl.psStationOwner->pacLogoFilePath = NULL;
            psObj->sWorkCtrl.psStationOwner->tLogoFilePathLen = 0;
        }

        // Destroy the text list
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->sWorkCtrl.psStationOwner->hTextEntries,
            (OSAL_LL_RELEASE_HANDLER)vReleaseTextEntry);
        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psObj->sWorkCtrl.psStationOwner->hTextEntries);
            psObj->sWorkCtrl.psStationOwner->hTextEntries = OSAL_INVALID_OBJECT_HDL;
        }

        // Dummy station
        if (psObj->sWorkCtrl.hDummyStation != FUEL_STATION_INVALID_OBJECT)
        {
            FUEL_STATION_vDestroy(psObj->sWorkCtrl.hDummyStation);
            psObj->sWorkCtrl.hDummyStation = FUEL_STATION_INVALID_OBJECT;
        }

        // Destroy the station cache
        vDestroyCache(psObj);

        // Destroy the station owner
        DATASERVICE_IMPL_vDestroyDSRLParent(
            (DATASERVICE_IMPL_HDL)psObj,
            (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
        psObj->sWorkCtrl.psStationOwner = (FUEL_MGR_STATION_OWNER_STRUCT *)NULL;
    }

    // Stop the fuel DB interface
    psObj->psDBInterface->vDisconnect(psObj->hDBInterface);
    psObj->hDBInterface = FUEL_DB_INTERFACE_INVALID_OBJECT;

    // Stop the fuel ota interface
    psObj->psOTAInterface->vUnInit(psObj->hOTAInterface);
    psObj->hOTAInterface = FUEL_OTA_INTERFACE_INVALID_OBJECT;

    if (bFullDelete == TRUE)
    {
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    // Destroy the SMS Update Event object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   bIsRegionNeeded
*
*   This function is part of the manager interface API and is utilized by
*   the interface to check if received fuel region is required by manager.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       tRegion - The price group's region
*       n16TextVersion - The text version used to populate this
*           price group's data
*
*   Output:
*       TRUE - if the manager wants to receive information
*           for this region; FALSE otherwise
*
*****************************************************************************/
static BOOLEAN bIsRegionNeeded (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tRegion,
    UN8 un8TextVersion
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    BOOLEAN bCanProcess;

    // Can we process updates now?
    bCanProcess = psObj->psDBInterface->bCanProcessUpdate(
        psObj->hDBInterface, un8TextVersion);

    if (bCanProcess == FALSE)
    {
        // Can't process this, so gotta stop here
        return FALSE;
    }

    // Update the current price region
    if ((psObj->sOTACtrl.psRegion == NULL) ||
        (psObj->sOTACtrl.psRegion->tRegionId != tRegion))
    {
        // Attempt to locate this region
        psObj->sOTACtrl.psRegion = psGetRegion(
            psObj, tRegion);
        if (psObj->sOTACtrl.psRegion == NULL)
        {
            // We don't know this region
            return FALSE;
        }
    }

    // Is this region in use?
    if (psObj->sOTACtrl.psRegion->tTotalUsageCount == 0)
    {
        // We don't need this region because nobody
        // is using it
        printf(FUEL_MGR_OBJECT_NAME
            ": Price update for region %d ignored -- not interested\n",
            tRegion);
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bRegionUpdateBegin
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a new fuel region is being received.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       un32MsgId - The unique ID of this message, used for filtering
*       tRegion - The price group's region
*       n16TextVersion - The text version used to populate this
*           price group's data
*
*   Output:
*       TRUE - if the manager wants to receive information
*           for this region; FALSE otherwise
*
*****************************************************************************/
static BOOLEAN bRegionUpdateBegin (
    FUEL_SERVICE_OBJECT hFuelService,
    UN32 un32MsgId,
    FUEL_REGION tRegion,
    UN8 un8TextVersion
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    BOOLEAN bMsgFiltered, bBackgroundRegion = FALSE,
            bLocked;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psObj->sOTACtrl.psRegion->tLocationUsageCount == 0)
    {
        // This is only used as a background
        bBackgroundRegion = TRUE;
    }

    // Get the current time (UTC)
    eReturnCode = OSAL.eTimeGet(&psObj->sOTACtrl.un32UTCsec);
    if (eReturnCode != OSAL_SUCCESS)
    {
        printf("Result: %s\n",
               OSAL.pacGetReturnCodeName(eReturnCode));
        // We can't tell the time!
        psObj->sOTACtrl.psRegion = NULL;
        return FALSE;
    }

    // Are we filtering this message?
    bMsgFiltered = bFilterPriceMessage(
        psObj, psObj->sOTACtrl.psRegion,
        un32MsgId, psObj->sOTACtrl.un32UTCsec);
    if (bMsgFiltered == TRUE)
    {
        // We don't want this update
        psObj->sOTACtrl.psRegion = NULL;
        printf(FUEL_MGR_OBJECT_NAME
            ": Price update filtered for region %d\n",
            tRegion);
        DATASERVICE_IMPL_vLog(FUEL_MGR_OBJECT_NAME
            ": Price update filtered for region %d\n",
            tRegion);
        return FALSE;
    }

    // Now, try to lock the station owner
    bLocked = SMSO_bLock(
        (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if (FALSE == bLocked)
    {
        return FALSE;
    }

    // If this region is only used for background processing then
    // we don't have to worry about informing DSRLs of this update
    psObj->sOTACtrl.bDSRL = !bBackgroundRegion;

    // Initialize the oldest processed data
    psObj->sAgeoutCtrl.un32OldestProcessedPriceData = UN32_MAX;

    // Processing messages with this text version now
    psObj->sOTACtrl.un8TextVer = un8TextVersion;

    DATASERVICE_IMPL_vLog(FUEL_MGR_OBJECT_NAME
        ": Price update needed for region %d\n",
        tRegion);

    printf(FUEL_MGR_OBJECT_NAME
        ": Price update needed for region %d\n",
        tRegion);

    return TRUE;
}

/*****************************************************************************
*
*   bPriceUpdate
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a price update has been received.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       tStationId - The ID of the station being changed
*       *psPriceRow - A pointer to the newly received price information.
*
*   Output: BOOLEAN TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bPriceUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_ID tStationId,
    FUEL_PRICE_ROW_STRUCT *psPriceRow
       )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    size_t tEntryIndex;
    FUEL_PRICE_ENTRY_STRUCT *psCurPrice;
    FUEL_STATION_OBJECT hStation;
    BOOLEAN bOk = TRUE, bStationUpdated;

    // Validate the region pointer
    if (psObj->sOTACtrl.psRegion == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": OTA Region pointer null!");
        return FALSE;
    }

    // This station must exist somewhere in the cache
    hStation = hFindStationInCache(
        psObj, NULL, NULL,
        psObj->sOTACtrl.psRegion->tRegionId, tStationId );
    if (hStation == FUEL_STATION_INVALID_OBJECT)
    {
        // We don't care about this station
        return TRUE;
    }

    for (tEntryIndex = 0; tEntryIndex < psPriceRow->tNumPrices; tEntryIndex++)
    {
        // Get the current price entry
        psCurPrice = &psPriceRow->pasPrices[tEntryIndex];

        // Calculate the price age now
        psCurPrice->un32PriceAgeUTCSeconds =
            psObj->sOTACtrl.un32UTCsec -
                (FUEL_SECONDS_PER_DAY * psCurPrice->un32PriceAgeUTCSeconds);

        if (psObj->sAgeoutCtrl.un32OldestProcessedPriceData >
                psCurPrice->un32PriceAgeUTCSeconds)
        {
            // Set this as our oldest price data being processed
            psObj->sAgeoutCtrl.un32OldestProcessedPriceData =
                psCurPrice->un32PriceAgeUTCSeconds;
        }
    }

    // Update the station's fuel price table
    bStationUpdated = FUEL_STATION_bUpdatePrices (
        hStation, psObj->sWorkCtrl.eFuelPriceSortMethod, 
        (FUEL_TYPE_TEXT_CALLBACK)hFuelTextCallback, 
        (FUEL_TYPE_ENUM_CALLBACK)eFuelTypeCallback,
        psPriceRow,
        psObj->sOTACtrl.un8TextVer, psObj);

    // The fuel station was updated, so now
    // we have to make sure that any DSRLs this
    // station is currently a part of indicate
    // that they are updating
    if ((bStationUpdated == TRUE) &&
        (psObj->sOTACtrl.bDSRL == TRUE))
    {
        // Iterate the entries for this station
        // so we can update all DSRLs which use it
        bOk = bIterateStationEntries(
            psObj, hStation,
            (STATION_ITERATOR_CALLBACK)bUpdateDSRLForStationChange);
    }

    // Close out this price group
    vCloseOutUpdateGroup(
        psObj, psObj->sOTACtrl.bDSRL,
        hStation, FALSE);

    // If we're stopping, then clear the price region
    if (bOk == FALSE)
    {
        psObj->sOTACtrl.psRegion = NULL;
    }

    return bOk;
}

/*****************************************************************************
*
*   bPositionUpdate
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a fuel position update has been recieved.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       un16ReportTime - The timestamp for this data
*       tStationId - The ID of the station being changed
*       pasPositions - The array of position data
*       tNumPositions - The number of entries in the array
*
*   Output: BOOLEAN TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bPositionUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    UN16 un16ReportTime,
    FUEL_STATION_ID tStationId,
    FUEL_POSITION_UPDATE_STRUCT *pasPositions,
    size_t tNumPositions
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    FUEL_STATION_OBJECT hStation;
    BOOLEAN bOk = TRUE, bStationUpdated;

    // Validate the region pointer
    if (psObj->sOTACtrl.psRegion == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": OTA Region pointer null!");
        return FALSE;
    }

    // This station must exist somewhere in the cache
    hStation = hFindStationInCache(
        psObj, NULL, NULL,
        psObj->sOTACtrl.psRegion->tRegionId, tStationId );
    if (hStation == FUEL_STATION_INVALID_OBJECT)
    {
        // We don't care about this station
        return TRUE;
    }

    // Update the station's fuel position table
    bStationUpdated = FUEL_STATION_bUpdateRefuelingPositions(
        hStation, un16ReportTime, pasPositions, tNumPositions);

    // The fuel station was updated, so now
    // we have to make sure that any DSRLs this
    // station is currently a part of indicate
    // that they are updating
    if ((bStationUpdated == TRUE) &&
        (psObj->sOTACtrl.bDSRL == TRUE))
    {
        // Iterate the entries for this station
        // so we can update all DSRLs which use it
        bOk = bIterateStationEntries(
            psObj, hStation,
            (STATION_ITERATOR_CALLBACK)bUpdateDSRLForStationChange);
    }

    // Close out this update group
    vCloseOutUpdateGroup(
        psObj, psObj->sOTACtrl.bDSRL,
        hStation, FALSE);

    // If we're stopping, then clear the price region
    if (bOk == FALSE)
    {
        psObj->sOTACtrl.psRegion = NULL;
    }

    return bOk;
}

/*****************************************************************************
*
*   vRegionUpdateComplete
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a region has been completed.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*
*   Output: None
*
*****************************************************************************/
static void vRegionUpdateComplete (
    FUEL_SERVICE_OBJECT hFuelService
       )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;

    // We are no longer working on a price region
    psObj->sOTACtrl.psRegion = NULL;

    // Close out the previous price group if
    // we just finished updating a station
    vCloseOutUpdateGroup(
        psObj, psObj->sOTACtrl.bDSRL,
        FUEL_STATION_INVALID_OBJECT, TRUE);

    // Clear the DSRL flag
    psObj->sOTACtrl.bDSRL = FALSE;

    // Now we can unlock the station owner
    SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);

    // Update the ageout timer for this price data
    vStartAgeoutTimer(psObj);

    return;
}

/*****************************************************************************
*
*   bTextUpdateBegin
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a text update has begun.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       un8NewTextVersion - The version of the new text data.
*
*   Output: BOOLEAN TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bTextUpdateBegin (
    FUEL_SERVICE_OBJECT hFuelService,
    UN8 un8NewTextVersion
        )
{
    BOOLEAN bSuccess, bLocked;
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;

    // Lock the station owner now
    bLocked = SMSO_bLock(
        (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if (TRUE != bLocked)
    {
        return FALSE;
    }

    // Tell the DB interface an update has begun
    bSuccess = psObj->psDBInterface->bUpdateTextTable(
        psObj->hDBInterface, TRUE, un8NewTextVersion);

    if (TRUE == bSuccess)
    {
        DATASERVICE_IMPL_vLog(
            FUEL_MGR_OBJECT_NAME": Update text version %d\n",
            un8NewTextVersion);
    }
    else
    {
        // Error! DB access failure
        vSetError( psObj, DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE );
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bTextUpdate
*
*   This function is part of the manager interface API and is utilized by
*   the interface to report an update to the text table.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       *psTextRow - A pointer to the newly received text information.
*
*   Output: BOOLEAN TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bTextUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    BOOLEAN bOk;

    // Have the DB update this text entry
    bOk = psObj->psDBInterface->bUpdateTextEntry(
        psObj->hDBInterface,
        SQL_INTERFACE_INVALID_OBJECT,
        psTextRow );

    if (bOk == FALSE)
    {
        // Error! DB access failure
        vSetError( psObj, DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE );

        return FALSE;
    }

    // Update the text entry in memory
    bOk = bUpdateTextEntry(hFuelService, psTextRow);

    return bOk;
}

/*****************************************************************************
*
*   vTextUpdateEnd
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a text update has concluded.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       un8NewTextVersion - The version of the new text data.
*
*   Output: None
*
*****************************************************************************/
static void vTextUpdateEnd (
    FUEL_SERVICE_OBJECT hFuelService,
    UN8 un8NewTextVersion
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    BOOLEAN bOk;

    // Update the database now
    bOk = psObj->psDBInterface->bUpdateTextTable(
        psObj->hDBInterface, FALSE, un8NewTextVersion);

    if (bOk == FALSE)
    {
        // Error! DB access failure
        vSetError( psObj, DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE );
    }

    SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);

    return;
}

/*****************************************************************************
*
*   bLogoUpdate
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate a logo update has been received.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*****************************************************************************/
static BOOLEAN bLogoUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_LOGO_ROW_STRUCT *psLogoRow,
    BOOLEAN bDelete,
    OSAL_BUFFER_HDL hLogoData
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    BOOLEAN bLocked, bOk = FALSE, bUpdateNeeded = FALSE;

    // Logo processing must be protected by
    // the application lock
    bLocked = SMSO_bLock(
        (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        FUEL_LOGO_ENTRY_STRUCT *psLogoEntry;

        // Find this logo
        psLogoEntry = psGetLogoEntry(
            psObj->sWorkCtrl.psStationOwner, psLogoRow);

        // Do we need to perform this update?
        if (bDelete == TRUE)
        {
            // We need to perform an update for a delete
            // only if we know about this logo
            if (psLogoEntry != NULL)
            {
                bUpdateNeeded = TRUE;
            }
        }
        else
        {
            // If this is a new or updated logo, we need to
            // peform an update if:
            // 1) We can't find a logo entry (new logo), or
            if (psLogoEntry == NULL)
            {
                bUpdateNeeded = TRUE;
            }
            // 2) The logo entry found is of a different version (updated logo)
            else
            {
                if ((psLogoEntry->sLogoRow.un8LogoTableVer != psLogoRow->un8LogoTableVer) ||
                    (psLogoEntry->sLogoRow.un8LogoVer != psLogoRow->un8LogoVer))
                {
                    bUpdateNeeded = TRUE;
                }
            }
        }

        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

            if (bUpdateNeeded == FALSE)
            {
                // Not a problem
                bOk = TRUE;
                break;
            }

            DATASERVICE_IMPL_vLog(
                FUEL_MGR_OBJECT_NAME": Update Logo - ID: %u Ver: %u Type: %u %s\n",
                psLogoRow->un16LogoId, psLogoRow->un8LogoVer,
                psLogoRow->eLogoType,
                (bDelete == TRUE) ? "(DELETE)":"");

            if (bDelete == TRUE)
            {
                // Delete the logo from memory and the FS
                bOk = bDeleteLogo(psObj->sWorkCtrl.psStationOwner, psLogoEntry);
            }
            else
            {
                // Write the file now
                bOk = bWriteLogoFile(psObj->sWorkCtrl.psStationOwner, hLogoData, psLogoRow);

                if (bOk == TRUE)
                {
                    // Update the logo entry in memory
                    bOk = bUpdateLogoEntry(hFuelService, psLogoRow);
                }

                if (bOk == FALSE)
                {
                    // We failed in our update of this image
                    // so, clean up the mess we made
                    FUEL_LOGO_ENTRY_STRUCT *psEntryToDelete;

                    // Look for the entry
                    psEntryToDelete = psGetLogoEntry(psObj->sWorkCtrl.psStationOwner, psLogoRow);

                    if (psEntryToDelete != NULL)
                    {
                        // Remove it!
                        bDeleteLogo(psObj->sWorkCtrl.psStationOwner, psEntryToDelete);
                    }
                }
            }

            if (bOk == FALSE)
            {
                // Error!
                vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
                break;
            }

            // Update the database now
            bOk = psObj->psDBInterface->bUpdateLogoEntry(
                psObj->hDBInterface, bDelete, psLogoRow);
            if (bOk == FALSE)
            {
                // Error! DB access failure
                vSetError( psObj, DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE );
                break;
            }

            // Update the stations if this is a logo update
            if (psLogoEntry != NULL)
            {
                // We're updating a logo, so iterate the cache
                // for any affected stations
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->sWorkCtrl.hStations,
                    (OSAL_LL_ITERATOR_HANDLER)bIterateCacheForLogoUpdate,
                    &psLogoRow->un16LogoId);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    // iterate dsrls to put them back to ready
                    eReturnCode = OSAL.eLinkedListIterate(
                        psObj->hTargets,
                        (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsForDSRLReady,
                        NULL);
                }
            }

            if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
            {
                bOk = FALSE;
                break;
            }

            // Make this sure this is TRUE
            bOk = TRUE;

        } while (FALSE);

        SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
    }

    // Always free this data when we're done with it
    OSAL.eBufferFree(hLogoData);

    return bOk;
}

/*****************************************************************************
*
*   bLocationFilter
*
*   This function is part of the manager interface API and is utilized by
*   the DB interface to pass a lat/lon pair through the location filter
*   for the current target.
*
*****************************************************************************/
static BOOLEAN bLocationFilter (
    FUEL_SERVICE_OBJECT hFuelService,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon,
    BOOLEAN *pbCacheOnly
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation;
    BOOLEAN bLocationAccepted = FALSE;

    // Initialize the value here
    *pbCacheOnly = TRUE;

    // Grab the current location handle
    psLocation = (FUEL_TARGET_LOCATION_DESC_STRUCT *)
        OSAL.pvLinkedListThis(
            psObj->sWorkCtrl.psCurrentTarget->hCurrentLocation);

    if (psLocation != (FUEL_TARGET_LOCATION_DESC_STRUCT *)NULL)
    {
        // Does the station meet our background location criteria?
        bLocationAccepted = LOCATION.bCoordinatesWithinArea(
            psLocation->hBackground, hLat, hLon);

        // Location was accepted for the background radius
        // what about the actual target radius?
        if (bLocationAccepted == TRUE)
        {
            BOOLEAN bInTargetArea = TRUE;

            // This location will make it into the cache, but
            // should we provide it to the DSRL?

            // Only have to check if these are different
            if (psLocation->hBackground != psLocation->hLocation)
            {
                // Check against the actual location provided
                bInTargetArea = LOCATION.bCoordinatesWithinArea (
                    psLocation->hLocation, hLat, hLon);
            }

            // Is this in the target area?
            if (bInTargetArea == TRUE)
            {
                // Yes! This station doesn't
                // have to be restricted to the cache
                *pbCacheOnly = FALSE;
            }
        }
    }

    return bLocationAccepted;
}

/*****************************************************************************
*
*   hGetStationFromCache
*
*   This function is part of the manager interface API and is utilized by
*   the DB interface to determine if a station found frmo the DB is
*   already present.
*
*****************************************************************************/
static FUEL_STATION_OBJECT hGetStationFromCache (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID,
    BOOLEAN *pbStationNeeded
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    FUEL_STATION_OBJECT hStation;
    FUEL_TARGET_DESC_STRUCT *psTarget =
        psObj->sWorkCtrl.psCurrentTarget;
    BOOLEAN bStationFoundInThisTarget = FALSE;

    // Attempt to find this station in the
    // station cache.  We may already be
    // tracking this station, and if so, we can skip
    // the database processing code below
    hStation = hFindStationInCache(
        psObj, psTarget, &bStationFoundInThisTarget,
        tRegionID, tStationID);
    if (hStation != FUEL_STATION_INVALID_OBJECT)
    {
        if (bStationFoundInThisTarget == TRUE)
        {
            // We don't want this station again
            hStation = FUEL_STATION_INVALID_OBJECT;

            // Tell the DB interface to stop processing
            // this station
            *pbStationNeeded = FALSE;
        }
    }

    return hStation;
}

/*****************************************************************************
*
*   bAddFuelPricesToStation
*
*   This function is part of the manager interface API and is utilized by
*   the DB interface to populate a station with a number of fuel prices.
*
*****************************************************************************/
static BOOLEAN bAddFuelPricesToStation (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_OBJECT hStation,
    FUEL_PRICE_ROW_STRUCT *psPriceRow,
    UN8 un8TextVer
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    size_t tIndex;
    FUEL_PRICE_ENTRY_STRUCT *psCurEntry;
    BOOLEAN bSuccess = TRUE;

    if (psPriceRow->tNumPrices == 0)
    {
        // Nothing to do
        return TRUE;
    }

    // Iterate all prices reported
    for (tIndex = 0; tIndex < psPriceRow->tNumPrices; tIndex++)
    {
        psCurEntry = &psPriceRow->pasPrices[tIndex];

        // Is this the oldest data we've recovered?
        if (psCurEntry->un32PriceAgeUTCSeconds <
                psObj->sAgeoutCtrl.un32OldestProcessedPriceData)
        {
            // Yeah, keep track of this age
            psObj->sAgeoutCtrl.un32OldestProcessedPriceData =
                psCurEntry->un32PriceAgeUTCSeconds;
        }
    }

    // Update the station now if that all worked
    if (bSuccess == TRUE)
    {
        bSuccess = FUEL_STATION_bUpdatePrices (
            hStation, psObj->sWorkCtrl.eFuelPriceSortMethod,
            (FUEL_TYPE_TEXT_CALLBACK)hFuelTextCallback, 
            (FUEL_TYPE_ENUM_CALLBACK)eFuelTypeCallback,
            psPriceRow,
            un8TextVer, psObj);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPutStationInService
*
*   This function is part of the manager interface API and is utilized by
*   the DB interface to tell the manager it may start using a station found
*   in the DB.
*
*****************************************************************************/
static BOOLEAN bPutStationInService (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_OBJECT hStation,
    BOOLEAN bCacheOnly
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    BOOLEAN bAddedToDSRL = FALSE,
            bAddToCache = FALSE,
            bStationInService = FALSE;
    FUEL_TARGET_DESC_STRUCT *psTarget =
        psObj->sWorkCtrl.psCurrentTarget;

    // Add this entry to the dsrl:
    // If we're loading stations, they always get added to the cache
    // if we're loading prices, they get added to the cache if the DSRL wants them
    bAddToCache = (psObj->sWorkCtrl.bLoadingPrices == TRUE) ? FALSE : TRUE;

    // If this station is only meant for the cache then
    // we won't attempt to add it to the DSRL for this target
    if (bCacheOnly == FALSE)
    {
        // Add this station to the DSRL
        bAddedToDSRL = bAddToDSRL(
            psTarget->hDSRL, hStation);

        // If the DSRL took it then we need to add it
        // to the fuel manager's cache
        bAddToCache |= bAddedToDSRL;
    }

    // Do we need to add this station to the cache?
    if (bAddToCache == TRUE)
    {
        // Add this station to the station cache now
        bStationInService = bAddStationToCache(
            psTarget->psObj, hStation, psTarget,
            bAddedToDSRL, bCacheOnly);
    }

    /*
     We need to remove this entry from the DSRL if:
        1. If we were instructed to add this station to the DSRL
        2. It was added to the DSRL
        3. It failed to get inserted into the cache
    */
    if ((FALSE == bStationInService) &&
        (FALSE == bCacheOnly) &&
        (TRUE == bAddedToDSRL))
    {
        DSRL_vRemoveEntry(psTarget->hDSRL, (DSRL_ENTRY_OBJECT)hStation);
    }

    return bStationInService;
}

/*****************************************************************************
*
*   bAddRegionOfInterest
*
*   This function is part of the manager interface API and is utilized by
*   the interface to indicate which regions are of interest to the manager
*   based on LOCATION_OBJECT inputs to GsFuelIntf.bAddRegionsForLocation().
*
*   This API is also called by the manager when adding regions of interest
*   for LOCID-based LOCATION_OBJECTs.
*
*   This API must only be called when already in the
*   context of the fuel manager.
*
*   Inputs:
*       hFuelService - The fuel service object handle provided to
*           the interface.
*       tRegion - The Region Id which should be tracked as a region
*           of interest.
*       *pvArg - An argument provided by the called to the interface function
*           or to this function directly relaying information about how
*           to track this region.
*
*   Output: BOOLEAN TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bAddRegionOfInterest (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tRegion,
    void *pvArg
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL_REGION_INTEREST_STRUCT *psInterest =
        (FUEL_REGION_INTEREST_STRUCT *)pvArg;
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bSuccess = FALSE;
    FUEL_REGION_INTEREST_ENTRY_STRUCT sDummyRegionEntry;

    sDummyRegionEntry.tRegionID = tRegion;

    // Search for this region in our list
    eReturnCode = OSAL.eLinkedListSearch(
        psInterest->psLocation->hRegionsOfInterest,
        &hEntry,
        (void *)&sDummyRegionEntry
            );
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Extract the entry
        psRegionEntry = (FUEL_REGION_INTEREST_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }
    else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":RegOfInterest %u",
            psObj->psOTAInterface->tDataID );

        psRegionEntry = (FUEL_REGION_INTEREST_ENTRY_STRUCT *)
            SMSO_hCreate(&acName[0],
                sizeof(FUEL_REGION_INTEREST_ENTRY_STRUCT), SMS_INVALID_OBJECT, FALSE);

        if (psRegionEntry != NULL)
        {
            psRegionEntry->tRegionID = tRegion;
            psRegionEntry->bBackgroundOnly = TRUE;
            psRegionEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psRegionEntry->psRegion = psGetRegion(psObj, tRegion);

            DATASERVICE_IMPL_vLog(FUEL_MGR_OBJECT_NAME
                ": Adding region %d as background region of interest\n",
                tRegion);

            printf(FUEL_MGR_OBJECT_NAME
                ": Adding region %d as background region of interest\n",
                tRegion);

            // Add the entry to the list
            eReturnCode = OSAL.eLinkedListAdd(
                psInterest->psLocation->hRegionsOfInterest,
                &psRegionEntry->hEntry, psRegionEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSO_vDestroy((SMS_OBJECT)psRegionEntry);
                psRegionEntry = NULL;
            }
        }
    }

    if ((eReturnCode != OSAL_SUCCESS) || (psRegionEntry == NULL))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": Error adding region of interest");
    }

    if (psRegionEntry != NULL)
    {
        // Is this region only used as a background at this time?
        if (psRegionEntry->bBackgroundOnly == TRUE)
        {
            // If we now want to use this as a region for an
            // actual target, indicate that now
            if (psInterest->bBackgroundOnly == FALSE)
            {
                DATASERVICE_IMPL_vLog(FUEL_MGR_OBJECT_NAME
                    ": Marking region %d as region of interest for target location\n",
                    tRegion);

                printf(FUEL_MGR_OBJECT_NAME
                    ": Marking region %d as region of interest for target location\n",
                    tRegion);

                psRegionEntry->bBackgroundOnly = FALSE;
            }
        }

        // Manage the usage counts
        if (psRegionEntry->psRegion != NULL)
        {
            psRegionEntry->psRegion->tTotalUsageCount++;
        }
        if (psRegionEntry->bBackgroundOnly == FALSE)
        {
            // Increment the total number of location users for this region
            if (psRegionEntry->psRegion != NULL)
            {
                psRegionEntry->psRegion->tLocationUsageCount++;
            }
        }

        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bEnableLogoSupport
*
*   This function is called by a DB interface in order to tell the fuel
*   service that it needs to enable logo support.
*
*****************************************************************************/
static BOOLEAN bEnableLogoSupport (
    FUEL_SERVICE_OBJECT hFuelService,
    const char *pacLogoPath,
    const char *pacBaselineLogoPath
        )
{
    do
    {
        FUEL_MGR_OBJECT_STRUCT *psObj =
            (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        BOOLEAN bOk;

        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":Logos %u",
            psObj->psOTAInterface->tDataID );

        // Create the list
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->sWorkCtrl.psStationOwner->hLogos,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)n16CompareLogoEntries,
            OSAL_LL_OPTION_NONE
                );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Build the logo path
        bOk = bBuildLogoDirPath(
           psObj->sWorkCtrl.psStationOwner,
           pacLogoPath, pacBaselineLogoPath);
        if (bOk == FALSE)
        {
            // Error!
            break;
        }

        return TRUE;
    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bUpdateTextEntry
*
*   This function updates a text entry in the text table using the
*   provided parameters.  If the entry doesn't already exist in the list,
*   one is created.  If the entry is already in the list, it is overwritten
*   and the old data is replaced.
*
*****************************************************************************/
static BOOLEAN bUpdateTextEntry (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    FUEL_TEXT_ROW_STRUCT *psLocalText =
        (FUEL_TEXT_ROW_STRUCT *)NULL;
    BOOLEAN bSuccess = FALSE;

    // Search for this entry
    eReturnCode = OSAL.eLinkedListSearch(
        psObj->sWorkCtrl.psStationOwner->hTextEntries,
        &hEntry,
        (void *)psTextRow);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // Extract the text row
        psLocalText = (FUEL_TEXT_ROW_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        if (psLocalText != NULL)
        {
            // Update the text for this entry
            if (psLocalText->hText != STRING_INVALID_OBJECT)
            {
                // Copy the data into the text entry
                STRING.tCopy(psTextRow->hText, psLocalText->hText);

                // Destroy the string provided
                STRING_vDestroy(psTextRow->hText);
            }
            else
            {
                psLocalText->hText = psTextRow->hText;
            }

            if (psLocalText->hLongText != STRING_INVALID_OBJECT)
            {
                // Copy the data into the text entry
                STRING.tCopy(psTextRow->hLongText, psLocalText->hLongText);

                // Destroy the string provided
                STRING_vDestroy(psTextRow->hLongText);
            }
            else
            {
                psLocalText->hLongText = psTextRow->hLongText;
            }

            // That's all we gotta do!
            bSuccess = TRUE;
        }
    }
    else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        // We need to add a new entry
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Create the name for this text entry
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":Text(B:%u ID:%u) dsi %u",
            psTextRow->bBrandText,
            psTextRow->un8TextId,
            psObj->psOTAInterface->tDataID );

        // Allocate the text and add it
        psLocalText = (FUEL_TEXT_ROW_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(FUEL_TEXT_ROW_STRUCT),
                DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj),
                FALSE);

        if (psLocalText != NULL)
        {
            // Populate the entry
            *psLocalText = *psTextRow;

            // Ensure hEntry is invalid
            hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Add the new entry to the database
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->sWorkCtrl.psStationOwner->hTextEntries,
                &hEntry, (void *)psLocalText );

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSO_vDestroy((SMS_OBJECT)psLocalText);
            }
            else
            {
                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateLogoEntry
*
*   This function updates a logo entry in the logo table using the
*   provided parameters.  If the entry doesn't already exist in the list,
*   one is created.  If the entry is already in the list, it is overwritten
*   and the old data is replaced.
*
*****************************************************************************/
static BOOLEAN bUpdateLogoEntry (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
       )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    BOOLEAN bSuccess = FALSE;

    do
    {
        IMAGE_OBJECT hNewLogo;
        FUEL_LOGO_ENTRY_STRUCT sSearchParameters;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hSearchEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        FUEL_LOGO_ENTRY_STRUCT *psLogo =
            (FUEL_LOGO_ENTRY_STRUCT *)NULL;
        const char *pacLogoDir;

        // Is this a baseline logo?
        if (FALSE == psLogoRow->bIsBaselineLogo)
        {
            // Nope, use the regular path
            pacLogoDir =
                psObj->sWorkCtrl.psStationOwner->pacLogoFileDir;
        }
        else
        {
            // Yes, use the baseline path
            pacLogoDir =
                psObj->sWorkCtrl.psStationOwner->pacBaselineLogoFileDir;
        }

        // First, try to create the new image object
        // for the updated logo
        hNewLogo = IMAGE_hCreate(
            (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
            &pacLogoDir[0],
            IMAGE_FORMAT_PNG, // Always a PNG
            &GsFuelImageIntf,
            (void*) psLogoRow,
            TRUE);

        if (hNewLogo == IMAGE_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // Search for this entry now
        sSearchParameters.sLogoRow = *psLogoRow;

        // Search for this entry now
        eReturnCode = OSAL.eLinkedListSearch(
            psObj->sWorkCtrl.psStationOwner->hLogos,
            &hSearchEntry,
            (void *)&sSearchParameters);

        if (eReturnCode == OSAL_SUCCESS)
        {
            // Extract the text row
            psLogo = (FUEL_LOGO_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hSearchEntry);

            if (psLogo == NULL)
            {
                // Error!
                break;
            }

            // Update the info for this entry
            psLogo->sLogoRow.un8LogoTableVer = psLogoRow->un8LogoTableVer;
            psLogo->sLogoRow.un8LogoVer = psLogoRow->un8LogoVer;

            // Update the image object
            if (psLogo->hLogo != IMAGE_INVALID_OBJECT)
            {
                IMAGE_vDestroy(psLogo->hLogo);
            }

            // Save the new logo handle
            psLogo->hLogo = hNewLogo;

            // Don't need to replace the entry - they're
            // sorted by id/type, which never changes
        }
        else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // We need to add a new entry
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

            // Create the name for this text entry
            snprintf( &acName[0],
                OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                FUEL_MGR_OBJECT_NAME":Logo(ID:%u Type: %u) dsi %u",
                psLogoRow->un16LogoId,
                psLogoRow->eLogoType,
                psObj->psOTAInterface->tDataID );

            // Allocate the text and add it
            psLogo = (FUEL_LOGO_ENTRY_STRUCT *)
                SMSO_hCreate(
                    &acName[0],
                    sizeof(FUEL_LOGO_ENTRY_STRUCT),
                    (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
                    FALSE);

            if (psLogo == NULL)
            {
                // Error!
                break;
            }

            // Populate the entry
            psLogo->sLogoRow = *psLogoRow;

            // Store the logo
            psLogo->hLogo = hNewLogo;

            // Ensure hEntry is invalid
            psLogo->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Add the new entry to the database
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->sWorkCtrl.psStationOwner->hLogos,
                &psLogo->hEntry, (void *)psLogo );

            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error!
                SMSO_vDestroy((SMS_OBJECT)psLogo);
                break;
            }
        }
        else
        {
            // Search failed
            break;
        }

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   hGetShortText
*
*   Just reports the short text
*
*****************************************************************************/
static STRING_OBJECT hGetShortText (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bBrandText,
    UN8 un8TextId
        )
{
    BOOLEAN bFound;
    STRING_OBJECT hShortText, hLongText;

    // Search for this text pair
    bFound = bGetTextPair(
        psObj, bBrandText, un8TextId, &hShortText, &hLongText);
    if (bFound == TRUE)
    {
        return hShortText;
    }

    return STRING_INVALID_OBJECT;
}

/*****************************************************************************
*
*   bGetTextPair
*
*   Pulls a text entry from the table based on id and a flag indicating
*   if the requested text needs to be brand text or fuel type text.
*
*****************************************************************************/
static BOOLEAN bGetTextPair (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bBrandText,
    UN8 un8TextId,
    STRING_OBJECT *phShortText,
    STRING_OBJECT *phLongText
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    FUEL_TEXT_ROW_STRUCT sSearchParameters;

    // Clear the output pointers
    *phShortText = STRING_INVALID_OBJECT;
    *phLongText = STRING_INVALID_OBJECT;

    // Searching for this entry
    sSearchParameters.bBrandText = bBrandText;
    sSearchParameters.un8TextId = un8TextId;

    // Search for this entry
    eReturnCode = OSAL.eLinkedListSearch(
        psObj->sWorkCtrl.psStationOwner->hTextEntries,
        &hEntry,
        (void *)&sSearchParameters);

    if (eReturnCode == OSAL_SUCCESS)
    {
        FUEL_TEXT_ROW_STRUCT *psText;

        // Extract the text entry data
        psText = (FUEL_TEXT_ROW_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        // Pull out the string handles
        *phShortText = psText->hText;
        *phLongText = psText->hLongText;
    }

    return (eReturnCode == OSAL_SUCCESS);
}

/*****************************************************************************
*
*   bWriteLogoFile
*
*   Writes a new or updated logo file to the FS
*
*****************************************************************************/
static BOOLEAN bWriteLogoFile (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    OSAL_BUFFER_HDL hLogoData,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        )
{
    FILE *psLogoFile = (FILE *)NULL;
    BOOLEAN bSuccess = FALSE;

    do
    {
        BOOLEAN bOk;
        FUEL_IMAGE_DATA_STRUCT sFuelImageData;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Prepare structure for image path generation
        bOk = bImageInitSpecificData(IMAGE_INVALID_OBJECT,
                        (void*)&sFuelImageData, (void*)psLogoRow);
        if (bOk == FALSE)
        {
            break;
        }

        // Create the image file name
        bOk = bImageFilenameCreateBasedOnFormat(
            IMAGE_FORMAT_PNG,
            &psOwner->pacLogoFileDir[0],
            &sFuelImageData,
            &psOwner->pacLogoFilePath[0],
            psOwner->tLogoFilePathLen);

        if (bOk == FALSE)
        {
            break;
        }

        // Open that file
        psLogoFile = fopen(&psOwner->pacLogoFilePath[0], "wb");

        if (psLogoFile == (FILE *)NULL)
        {
            break;
        }

        eReturnCode = OSAL.eBufferWriteToFile(hLogoData, TRUE, psLogoFile);
        if (eReturnCode != OSAL_SUCCESS)
        {
            // error!
            break;
        }

        bSuccess = TRUE;

    } while (FALSE);

    // Close the file now
    if (psLogoFile != (FILE *)NULL)
    {
        // Close the file
        fclose(psLogoFile);
    }

    if (bSuccess == FALSE)
    {
        // Remove the file if we had a problem
        remove(&psOwner->pacLogoFilePath[0]);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDeleteLogo
*
*   Removes logo data from memory and FS
*
*****************************************************************************/
static BOOLEAN bDeleteLogo (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    FUEL_LOGO_ENTRY_STRUCT *psLogoEntry
        )
{
    BOOLEAN bSuccess = FALSE;

    // Do we have a logo entry to work with?
    if (psLogoEntry == NULL)
    {
        // Nope, just tell the caller we succeeded
        // at deleting non-existent file
        return TRUE;
    }

    do
    {
        BOOLEAN bOk;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        FUEL_IMAGE_DATA_STRUCT sFuelImageData;

        // Initialize specific data
        bOk = bImageInitSpecificData(IMAGE_INVALID_OBJECT,
                    (void*)&sFuelImageData, (void*)&psLogoEntry->sLogoRow);
        if (bOk == FALSE)
        {
            break;
        }

        // Create the image file name
        bOk = bImageFilenameCreateBasedOnFormat(
            IMAGE_FORMAT_PNG,
            &psOwner->pacLogoFileDir[0],
            &sFuelImageData,
            &psOwner->pacLogoFilePath[0],
            psOwner->tLogoFilePathLen);

        if (bOk == FALSE)
        {
            break;
        }

        // Remove this entry from the list first
        eReturnCode = OSAL.eLinkedListRemove(psLogoEntry->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Now, remove the file from the FS
        remove(&psOwner->pacLogoFilePath[0]);

        // Free the logo entry
        vReleaseLogoEntry(psLogoEntry);

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   psGetLogoEntry
*
*   Removes logo data from memory and FS
*
*****************************************************************************/
static FUEL_LOGO_ENTRY_STRUCT *psGetLogoEntry (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        )
{
    BOOLEAN bOwner;
    FUEL_LOGO_ENTRY_STRUCT *psEntry = NULL;

    bOwner = SMSO_bOwner((SMS_OBJECT)psOwner);
    if (bOwner == TRUE)
    {
        FUEL_LOGO_ENTRY_STRUCT sSearchParameters;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Search for this entry
        sSearchParameters.sLogoRow = *psLogoRow;

        eReturnCode = OSAL.eLinkedListSearch(
            psOwner->hLogos,
            &hEntry,
            (void *)&sSearchParameters);

        if (eReturnCode == OSAL_SUCCESS)
        {
            // Extract the text row
            psEntry = (FUEL_LOGO_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hEntry);
        }
    }

    return psEntry;
}

/*****************************************************************************
*
*   bBuildLogoDirPath
*
*   This function generates the logo image file path based upon
*   the SMS file path.  This information is used in order to locate the
*   fuel brand logo files.
*
*****************************************************************************/
static BOOLEAN bBuildLogoDirPath (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    const char *pacLogoPath,
    const char *pacBaselineLogoPath
        )
{
    size_t tLogoPathLen, tBaselinePathLen;
    const char *pacLongest = pacLogoPath;

    // Compute the length of the provided path data
    tLogoPathLen = strlen(pacLogoPath) + 1;

    // Compute the length of the provided baseline path data
    tBaselinePathLen = strlen(pacBaselineLogoPath) + 1;

    // The logo path length is equal to the
    // path of the service directory
    // (and +1 for the necessary path delimiter character)
    tLogoPathLen += 1;

    // The baseline logo path length is equal to the
    // path of the baseline directory
    // (and +1 for the necessary path delimiter character)
    tBaselinePathLen += 1;

    if (tBaselinePathLen > tLogoPathLen)
    {
        pacLongest = pacBaselineLogoPath;
    }

    // Now, allocate space for the logo file path (plus 1 for NULL)
    bImageFilenameLenBasedOnFormat(IMAGE_FORMAT_PNG,
        pacLongest, &psOwner->tLogoFilePathLen);
    (psOwner->tLogoFilePathLen)++;

    psOwner->pacLogoFilePath = (char *)
        SMSO_hCreate(
            FUEL_MGR_OBJECT_NAME": LogoFilePath",
            psOwner->tLogoFilePathLen,
            (SMS_OBJECT)psOwner, FALSE );

    if (psOwner->pacLogoFilePath == NULL)
    {
        return FALSE;
    }

    // Store these base paths
    psOwner->pacLogoFileDir = pacLogoPath;
    psOwner->pacBaselineLogoFileDir = pacBaselineLogoPath;

    return TRUE;
}

/*****************************************************************************
*
*   bImageInitSpecificData
*
*   This function initializes specific data of the IMAGE object
*
*****************************************************************************/
static BOOLEAN bImageInitSpecificData(
    IMAGE_OBJECT hImage,
    void *pvSpecificData,
    void *pvArg
        )
{
    FUEL_IMAGE_DATA_STRUCT *psImageData =
        (FUEL_IMAGE_DATA_STRUCT*) pvSpecificData;
    FUEL_LOGO_ROW_STRUCT *psLogo =
        (FUEL_LOGO_ROW_STRUCT*)pvArg;

    if ((psImageData != NULL) && (psLogo != NULL))
    {
        psImageData->un16LogoId = psLogo->un16LogoId;
        psImageData->un8LogoVer = psLogo->un8LogoVer;
        psImageData->eLogoType = psLogo->eLogoType;

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bImageFilenameLenBasedOnFormat
*
*   This function calculates how long a logo file name will be
*
*****************************************************************************/
static BOOLEAN bImageFilenameLenBasedOnFormat (
    IMAGE_FORMAT_ENUM eFormat,
    const char *pacFilePath,
    size_t *ptLength
        )
{
    // Verify input pointer and the image format
    if ((pacFilePath != NULL) && (ptLength != NULL) &&
        (eFormat == IMAGE_FORMAT_PNG))
    {
        // Add the length of the provided path
        (*ptLength) += strlen(pacFilePath);

        // Add the length of the file name
        (*ptLength) += FUEL_FILE_NAME_ID_LABEL_LEN;

        // Add file extension length including space for NULL character
        (*ptLength) += sizeof(FUEL_PNG_FILE_EXTENSION);

        return TRUE;
    }


    return FALSE;
}

/*****************************************************************************
*
*   bImageFilenameLen
*
*   This function calculates how long a logo file name will be
*
*****************************************************************************/
static BOOLEAN bImageFilenameLen (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    size_t *ptLength
        )
{
    BOOLEAN bOk = FALSE;

    // verify input and check the image format
    if (ptLength != NULL)
    {
        IMAGE_FORMAT_ENUM eFormat;

        // Get image format
        eFormat = IMAGE.eFormat(hImage);

        // Do calculation
        bOk = bImageFilenameLenBasedOnFormat(eFormat, pacFilePath, ptLength);
    }

    return bOk;
}

/*****************************************************************************
*
*   bImageFilenameCreateBasedOnType
*
*   This function creates a logo file name
*
*****************************************************************************/
static BOOLEAN bImageFilenameCreateBasedOnFormat (
    IMAGE_FORMAT_ENUM eFormat,
    const char *pacFilePath,
    FUEL_IMAGE_DATA_STRUCT *psImageData,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    size_t tFilenameLen = 0;
    BOOLEAN bResult;

    // Check input
    if ((pacFilePath == NULL) || (psImageData == NULL) ||
        (pacBuffer == NULL))
    {
        return FALSE;
    }

    // Calculate the name of the file name
    bResult =
        bImageFilenameLenBasedOnFormat(eFormat, pacFilePath, &tFilenameLen);
    if (bResult == FALSE)
    {
        return FALSE;
    }

    // Ensure the destination buffer size can hold the
    // filename which will be generated
    if (tBufferSize < tFilenameLen)
    {
        return FALSE;
    }

    // Generate the filename
    snprintf( pacBuffer,
              tFilenameLen,
              "%s"FUEL_FILE_NAME_ID_LABEL"%s",
              pacFilePath,
              (UN32)psImageData->un16LogoId,
              (UN32)psImageData->eLogoType,
              FUEL_PNG_FILE_EXTENSION );

    return TRUE;
}

/*****************************************************************************
*
*   bImageFilenameCreate
*
*   This function creates a logo file name
*
*****************************************************************************/
static BOOLEAN bImageFilenameCreate (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    IMAGE_FORMAT_ENUM eFormat;
    BOOLEAN bResult;
    FUEL_IMAGE_DATA_STRUCT *psImageData =
        (FUEL_IMAGE_DATA_STRUCT*)pvSpecificData;

    // Get image format
    eFormat = IMAGE.eFormat(hImage);

    // Create the path
    bResult =
        bImageFilenameCreateBasedOnFormat(
            eFormat, pacFilePath, psImageData, pacBuffer, tBufferSize);

    return bResult;
}


/*****************************************************************************
*
*   bPopulateRegionTable
*
*   Add a region entry to the region table based upon data in the db.
*
*****************************************************************************/
static BOOLEAN bPopulateRegionTable (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tID
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegionAdd;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    BOOLEAN bSuccess = TRUE;

    // Create a name for this region
    snprintf( &acName[0], sizeof(acName),
              FUEL_MGR_OBJECT_NAME":reg %u (dsi %u)",
              tID, psObj->psOTAInterface->tDataID );

    // Allocate space for the region
    psRegionAdd = (FUEL_REGION_DESCRIPTOR_STRUCT *)
        SMSO_hCreate(&acName[0],
            sizeof(FUEL_REGION_DESCRIPTOR_STRUCT),
            DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj), FALSE);

    if (psRegionAdd != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Copy the ID
        psRegionAdd->tRegionId = tID;

        // Add this region to the list
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hRegions,
            &hEntry,
            (void *)psRegionAdd);

        if (eReturnCode == OSAL_SUCCESS)
        {
            bSuccess = TRUE;
        }
        else
        {
            OSAL.eLinkedListRemove(hEntry);

            if (psRegionAdd != NULL)
            {
                OSAL.bMemSet(psRegionAdd, 0, sizeof(FUEL_REGION_DESCRIPTOR_STRUCT));
                SMSO_vDestroy((SMS_OBJECT)psRegionAdd);
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   psGetRegion
*
*   Get a region row provided a set of search parameters.
*
*****************************************************************************/
static FUEL_REGION_DESCRIPTOR_STRUCT *psGetRegion (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_REGION tRegionID
        )
{
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL_REGION_DESCRIPTOR_STRUCT sSearchCriteria;

    // init search criteria struct with input data
    sSearchCriteria.tRegionId = tRegionID;

    // Attempt to locate this region
    eReturnCode = OSAL.eLinkedListSearch(psObj->hRegions, &hEntry,
        &sSearchCriteria);

    if (eReturnCode == OSAL_SUCCESS)
    {
        // Extract the region
        psRegion = (FUEL_REGION_DESCRIPTOR_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }

    return psRegion;
}

/*****************************************************************************
*
*   bFilterPriceMessage
*
*****************************************************************************/
static BOOLEAN bFilterPriceMessage (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_REGION_DESCRIPTOR_STRUCT *psPriceRegion,
    UN32 un32PriceMessageHash,
    UN32 un32CurrentTime
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bFiltered = FALSE;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Does this region have a tracking list yet?
    if (psPriceRegion->hPriceUpdates == OSAL_INVALID_OBJECT_HDL)
    {

        // Create the name
        snprintf( &acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                  FUEL_MGR_OBJECT_NAME":PriceHashList %u (dsi %u)",
                  psPriceRegion->tRegionId, 
                  psObj->psOTAInterface->tDataID );

        // Create the list
        eReturnCode = OSAL.eLinkedListCreate(
            &psPriceRegion->hPriceUpdates,
            &acName[0],
            NULL, OSAL_LL_OPTION_NONE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": Unable to create price message list");

            // Ensure the handle is invalid
            psPriceRegion->hPriceUpdates = OSAL_INVALID_OBJECT_HDL;
        }
    }

    if (psPriceRegion->hPriceUpdates != OSAL_INVALID_OBJECT_HDL)
    {
        FUEL_PRICE_HASH_ITERATOR_STRUCT sIterator;

        // Initialize our iterator
        sIterator.bFiltered = FALSE;
        sIterator.bFound = FALSE;
        sIterator.un32Hash = un32PriceMessageHash;
        sIterator.un32CurrentTime = un32CurrentTime;

        // Iterate the list to age out old entries
        // and look for this hash
        eReturnCode = OSAL.eLinkedListIterate(
            psPriceRegion->hPriceUpdates,
            (OSAL_LL_ITERATOR_HANDLER)bIteratePriceMsgs,
            &sIterator);

        if ((eReturnCode == OSAL_SUCCESS) ||
            (eReturnCode == OSAL_NO_OBJECTS))
        {
            if (sIterator.bFound == TRUE)
            {
                // Was this flagged as filtered?
                bFiltered = sIterator.bFiltered;
            }
            else // Hash not found -- add an entry
            {
                FUEL_REGION_PRICE_UPDATE_MSG_STRUCT *psUpdate;

                // Create the name
                snprintf( &acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                          FUEL_MGR_OBJECT_NAME":PriceHashEntry %u",
                          psObj->psOTAInterface->tDataID );

                // Create the entry object
                psUpdate = (FUEL_REGION_PRICE_UPDATE_MSG_STRUCT *)
                    SMSO_hCreate(
                        &acName[0],
                        sizeof(FUEL_REGION_PRICE_UPDATE_MSG_STRUCT),
                        SMS_INVALID_OBJECT, FALSE);
                if (psUpdate == NULL)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        FUEL_MGR_OBJECT_NAME
                        ": Unable to create an entry for the price message list");
                }
                else
                {
                    // Populate the entry
                    psUpdate->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                    psUpdate->un32Hash = un32PriceMessageHash;
                    psUpdate->un32TimeStamp = un32CurrentTime;

                    // Add this entry to the list
                    eReturnCode = OSAL.eLinkedListAdd(
                        psPriceRegion->hPriceUpdates,
                        &psUpdate->hEntry,
                        (void *)psUpdate);
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            FUEL_MGR_OBJECT_NAME": Unable to add to price message list");

                        SMSO_vDestroy((SMS_OBJECT)psUpdate);
                    }
                }
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": Unable to search price message list");
        }
    }

    return bFiltered;
}

/*****************************************************************************
*
*   vCloseOutUpdateGroup
*
*   This function performs all the necessary steps to update a DSRL after
*   an update group has been received.
*
*****************************************************************************/
static void vCloseOutUpdateGroup (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bDSRL,
    FUEL_STATION_OBJECT hStation,
    BOOLEAN bAllGroupsComplete
        )
{
    // Do we have a station to process and do we need to provide
    // it to a DSRL?
    if ((hStation != FUEL_STATION_INVALID_OBJECT) && (bDSRL == TRUE))
    {
        BOOLEAN bSuccess;

        // We just updated a fuel station that may potentially end
        // up in a DSRL.  We're done populating this station with
        // prices, so see if it can go to the DSRLs now

        // Iterate the entries for this station
        // so we can update all DSRLs which use it
        bSuccess = bIterateStationEntries(
            psObj, hStation,
            (STATION_ITERATOR_CALLBACK)bUpdateDSRLForNewStation);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": bUpdateDSRLForNewStation() failed");
            return;
        }
    }

    // Are we done processing for all price groups now?
    if (bAllGroupsComplete == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Bring all necessary DSRLs to the ready state now
        eReturnCode = OSAL.eLinkedListIterate(
                psObj->hTargets,
                (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsForDSRLReady,
                NULL);
        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": Unable to iterate targets to finish price message");
        }
    }

    return;
}

/*****************************************************************************
*
*   hFuelTextCallback
*
*****************************************************************************/
static STRING_OBJECT hFuelTextCallback (
    UN8 un8FuelType,
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    return hGetShortText(psObj, FALSE, un8FuelType);
}

/*****************************************************************************
*
*   eFuelTypeCallback
*
*****************************************************************************/
static FUEL_TYPE_ENUM eFuelTypeCallback (
    UN8 un8FuelType,
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    return psObj->psDBInterface->eMatchFuelType(un8FuelType);
}

/*****************************************************************************
*
*   bAddStationToCache
*
*****************************************************************************/
static BOOLEAN bAddStationToCache (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_STATION_OBJECT hStation,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN bInDSRL,
    BOOLEAN bBackgroundOnly
        )
{
    BOOLEAN bSuccess = FALSE;
    FUEL_STATION_ENTRY_STRUCT *psEntry;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Create the name
    snprintf( &acName[0],
        OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
        FUEL_MGR_OBJECT_NAME":Station %u",
        psObj->psOTAInterface->tDataID );

    // Create a new entry
    psEntry = (FUEL_STATION_ENTRY_STRUCT *)
        SMSO_hCreate(&acName[0],
            sizeof(FUEL_STATION_ENTRY_STRUCT),
            SMS_INVALID_OBJECT, FALSE);
    if (psEntry != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Populate the station entry
        psEntry->hStation = hStation;
        psEntry->psTarget = psTarget;
        psEntry->bInDSRL = bInDSRL;
        psEntry->bBackgroundOnly = bBackgroundOnly;

        // Add the entry to the list
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->sWorkCtrl.hStations,
            &hEntry, psEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
            FUEL_STATION_DSRL_ENTRY_STRUCT *psDSRLEntry;

            // Extract the DSRL struct from this entry now
            psDSRLEntry = (FUEL_STATION_DSRL_ENTRY_STRUCT *)
                DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hStation);

            if (NULL != psDSRLEntry)
            {
                bSuccess = TRUE;

                // Do we need to set the station entry handle?
                if (OSAL_INVALID_LINKED_LIST_ENTRY == 
                        psDSRLEntry->hFirstStationEntry)
                {
                    // Yes, set it now
                    psDSRLEntry->hFirstStationEntry = hEntry;
                }
            }
        }
        else
        {
            SMSO_vDestroy((SMS_OBJECT)psEntry);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   hCreateStationObject
*
*****************************************************************************/
static FUEL_STATION_OBJECT hCreateStationObject (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_ROW_STRUCT *psStation
        )
{
    FUEL_STATION_OBJECT hStation = FUEL_STATION_INVALID_OBJECT;
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;
    
    // Do we need to populate permanent types?
    if ((psStation->un8NumPermanentTypes != 0) &&
        (psStation->psTypes != NULL))
    {
        // We have to iterate these positions to get the associated text entries
        UN8 un8Index;
        FUEL_REFUELING_TYPE_STRUCT *psCurType;

        for (un8Index = 0;
                un8Index < psStation->un8NumPermanentTypes;
                un8Index++)
        {
            psCurType = &psStation->psTypes[un8Index];

            // Grab the text pair, don't worry about the result
            bGetTextPair(psObj, FALSE, psCurType->un8FuelType,
                &psCurType->hShortFuelName,
                &psCurType->hLongFuelName);
        }
    }

    // Create the station using the application-chosen
    // fuel price sort method
    hStation = FUEL_STATION_hCreate(
        (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
        psStation, psObj->sWorkCtrl.eFuelPriceSortMethod,
        sizeof(FUEL_STATION_DSRL_ENTRY_STRUCT));
    if (FUEL_STATION_INVALID_OBJECT != hStation)
    {
        FUEL_STATION_DSRL_ENTRY_STRUCT *psDSRLEntry;

        // Extract the DSRL entry struct from this entry now
        psDSRLEntry = (FUEL_STATION_DSRL_ENTRY_STRUCT *)
            DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hStation);

        if (NULL != psDSRLEntry)
        {
            // Station is not yet in the cache
            psDSRLEntry->hFirstStationEntry = 
                OSAL_INVALID_LINKED_LIST_ENTRY;
        }
    }

    return hStation;
}

/*****************************************************************************
*
*   hFindStationInCache
*
*   This function finds the station using the station & region ID pairing.
*   The caller may specify a target in which to search
*
*****************************************************************************/
static FUEL_STATION_OBJECT hFindStationInCache (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN *pbStationFoundInProvidedTarget,
    FUEL_REGION tRegion,
    FUEL_STATION_ID tStationId
        )
{
    FUEL_STATION_OBJECT hStation =
        FUEL_STATION_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    LOC_ID tIdToFind = FUEL_REG_STATID_TO_LOCID(tRegion,tStationId);
    FUEL_STATION_ENTRY_STRUCT sEntryToFind;
    BOOLEAN bDummyUpdated;
    OSAL_LINKED_LIST_ENTRY hFoundEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;

    // We're looking for a station with this loc id
    bDummyUpdated = FUEL_STATION_bSetDummyLocId(
        psObj->sWorkCtrl.hDummyStation, tIdToFind);
    if (bDummyUpdated == TRUE)
    {
        // Create an entry for the search (station only always)
        sEntryToFind.hStation = psObj->sWorkCtrl.hDummyStation;
        sEntryToFind.psTarget = NULL;

        // Search the station cache for this station
        // using the built-in search criteria
        eReturnCode = OSAL.eLinkedListSearch(
            psObj->sWorkCtrl.hStations,
            &hFoundEntry,
            &sEntryToFind );
        if (eReturnCode == OSAL_SUCCESS)
        {
            // We found it!
            FUEL_STATION_ENTRY_STRUCT *psFoundStation;

            psFoundStation = (FUEL_STATION_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(hFoundEntry);

            // Extract the station we found
            hStation = psFoundStation->hStation;

            // Caller wants to know if this station
            // is a part of a certain target
            if (psTarget != NULL)
            {
                // We are now looking for this station handle
                sEntryToFind.hStation = hStation;
                sEntryToFind.psTarget = psTarget;

                *pbStationFoundInProvidedTarget =
                    bEntryExists(hFoundEntry, &sEntryToFind);
            }
        }
    }

    return hStation;
}

/*****************************************************************************
*
*   bIterateStationEntries
*
*****************************************************************************/
static BOOLEAN bIterateStationEntries (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_STATION_OBJECT hStation,
    STATION_ITERATOR_CALLBACK bIterator
       )
{
    FUEL_STATION_DSRL_ENTRY_STRUCT *psDSRLEntry;
    BOOLEAN bSuccess = TRUE;

    // Grab the first entry in the cache which stores this station
    psDSRLEntry = (FUEL_STATION_DSRL_ENTRY_STRUCT *)
        DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hStation);

    // Update all DSRLs which currently use this station
    if ((FUEL_STATION_DSRL_ENTRY_STRUCT *)NULL != psDSRLEntry)
    {
        FUEL_STATION_ENTRY_STRUCT *psCacheEntry;
        OSAL_LINKED_LIST_ENTRY hCacheEntry = psDSRLEntry->hFirstStationEntry;

        // Extract the cache entry entry from this
        psCacheEntry = (FUEL_STATION_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(hCacheEntry);

        while (((FUEL_STATION_ENTRY_STRUCT *)NULL != psCacheEntry) && 
               (OSAL_INVALID_LINKED_LIST_ENTRY != hCacheEntry))
        {
            // Is this still the station we want?
            if (psCacheEntry->hStation != hStation)
            {
                // No, stop now
                break;
            }

            // Invoke the iterator on this callback
            bSuccess = bIterator(psCacheEntry, hStation);

            if (FALSE == bSuccess)
            {
                // Stop here
                break;
            }

            // Get the next entry
            hCacheEntry = OSAL.hLinkedListNext(
                hCacheEntry, (void **)&psCacheEntry);

            // Did we wrap?
            if (hCacheEntry == psDSRLEntry->hFirstStationEntry)
            {
                // We wrapped around
                break;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bPruneCache
*
*   Whenever a background location changes this function is used to
*   prune all entries which are no longer associated with any given target
*
*****************************************************************************/
static BOOLEAN bPruneCache (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    size_t tNumPruned = 0;
    OSAL_LINKED_LIST_ENTRY hFirst = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_LINKED_LIST_ENTRY hEntry, hRemove = OSAL_INVALID_LINKED_LIST_ENTRY;
    FUEL_STATION_ENTRY_STRUCT *psEntry, *psRemove = NULL;
    FUEL_STATION_DSRL_ENTRY_STRUCT *psDSRLEntry = NULL;
    FUEL_STATION_OBJECT hStation = FUEL_STATION_INVALID_OBJECT;
    size_t tInstances = 0, tNoTargets = 0;
    BOOLEAN bRemove = FALSE;

    // Grab the first entry, ensure it's valid
    hFirst = hEntry = OSAL.hLinkedListFirst(
        psObj->sWorkCtrl.hStations, (void **)&psEntry);
    if ((hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
        (psEntry == NULL))
    {
        // Cache is empty -- fine
        return TRUE;
    }

    // We do stuff when the station changes
    while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
           (psEntry != NULL))
    {
        // Did the station handle change?
        if (psEntry->hStation != hStation)
        {
            // Yes, if the previous station had no valid target
            // assignments then we can destroy it
            if (tInstances == tNoTargets)
            {
                printf(FUEL_MGR_OBJECT_NAME": Prune: destroying station %p (SMS %p)\n",
                    hStation,
                    DSRL_ENTRY_hGetSMSObject((DSRL_ENTRY_OBJECT)hStation)
                        );

                // We can remove this station
                FUEL_STATION_vDestroy(hStation);
            }

            // Get the "current" station
            hStation = psEntry->hStation;

            // Get this station's DSRL entry data
            psDSRLEntry = (FUEL_STATION_DSRL_ENTRY_STRUCT *)
                DSRL_ENTRY_pvServiceData((DSRL_ENTRY_OBJECT)hStation);

            if (NULL == psDSRLEntry)
            {
                // Fatal error. This shall never happen.
                // Setting service into error
                vSetError( psObj, 
                    DATASERVICE_ERROR_CODE_GENERAL);
                return FALSE;
            }
            // Number of instances starts at one since we have an instance here
            tInstances = 1;

            // Haven't come across this yet
            tNoTargets = 0;
        }
        else
        {
            // Another instance of the same station
            tInstances++;
        }

        // If this entry doesn't have a target
        // that means it's slated for removal
        if (psEntry->psTarget == NULL)
        {
            // We have to perform a remove
            bRemove = TRUE;

            // Number of "no target" entries has increased
            tNoTargets++;
        }

        if (bRemove == TRUE)
        {
            // Store the handles to remove
            psRemove = psEntry;
            hRemove = hEntry;
        }

        // Get the next entry
        psEntry = NULL;
        hEntry = OSAL.hLinkedListNext(hEntry, (void **)&psEntry);
        if (hEntry == hFirst)
        {
            // We've wrapped around
            hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psEntry = NULL;
        }

        if (bRemove == TRUE)
        {
            printf(FUEL_MGR_OBJECT_NAME": Prune: Removing Entry %p\n",
                psRemove);

            // Remove the entry
            eReturnCode = OSAL.eLinkedListRemove(hRemove);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Was this the first entry for this station?
                if (hRemove == psDSRLEntry->hFirstStationEntry)
                {
                    // Yes, we now need to update DSRL entry data
                    psDSRLEntry->hFirstStationEntry = hEntry;
                }

                // Destroy the entry
                SMSO_vDestroy((SMS_OBJECT)psRemove);

                // Was that the first entry?
                if (hFirst == hRemove)
                {
                    // Grab the first entry again
                    hFirst = OSAL.hLinkedListFirst(psObj->sWorkCtrl.hStations, NULL);
                }
            }
            else
            {
                // I don't know what to do here
                break;
            }

            // We're done removing
            bRemove = FALSE;
            tNumPruned++;
        }
    }

    // Check the station we left behind
    if (tInstances == tNoTargets)
    {
        printf(FUEL_MGR_OBJECT_NAME": Prune: destroying station %p (SMS %p)\n",
            hStation,
            DSRL_ENTRY_hGetSMSObject((DSRL_ENTRY_OBJECT)hStation)
                );

        // We can remove this station
        FUEL_STATION_vDestroy(hStation);
    }

    printf(FUEL_MGR_OBJECT_NAME": Pruned %u entries\n", tNumPruned);

    return TRUE;
}

/*****************************************************************************
*
*   vDestroyCache
*
*****************************************************************************/
static void vDestroyCache (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL_STATION_OBJECT hStation = FUEL_STATION_INVALID_OBJECT;

    // Iterate the list to destroy the entry contents
    // correctly
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->sWorkCtrl.hStations,
        (OSAL_LL_ITERATOR_HANDLER)bIterateCacheForDestroy,
        &hStation);
    if ((eReturnCode == OSAL_SUCCESS) || (eReturnCode == OSAL_NO_OBJECTS))
    {
        // Destroy the station handle left behind
        FUEL_STATION_vDestroy(hStation);

        // Now remove all the entries
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->sWorkCtrl.hStations,
            (OSAL_LL_RELEASE_HANDLER)vReleaseStations);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // And delete the list
            eReturnCode = OSAL.eLinkedListDelete(psObj->sWorkCtrl.hStations);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_MGR_OBJECT_NAME": vDestroyCache() Failed to delete list (%s)\n",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            psObj->sWorkCtrl.hStations = OSAL_INVALID_OBJECT_HDL;
        }
        else
        {

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": vDestroyCache() Failed to remove all entries from cache (%s)\n",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }

    return;
}

/*****************************************************************************
*
*   bHandleCreateList
*
*   A DSRL creation event has been passed to us and we need to handle it now
*
*****************************************************************************/
static BOOLEAN bHandleCreateList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE,
            bLocked;

    // Lock the owner object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME
            ": bHandleCreateList() Unable to lock station owner");

        return FALSE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        FUEL_TARGET_DESC_STRUCT *psTarget;

        // Create our target descriptor based
        // on the provided arguments
        psTarget = psCreateTarget( psObj, psDSRLArg );
        if (psTarget == NULL)
        {
            // Error!
            break;
        }

        // Add this to our list of tracked targets for this service
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hTargets, &psTarget->hEntry, psTarget);

        if (eReturnCode != OSAL_SUCCESS)
        {
            vDestroyTarget(psTarget);
            break;
        }

        // Populate the new list with the contents of our database
        bSuccess = bInitDSRL(psObj, psTarget);

        // Delete the list if there was a problem
        if (bSuccess == FALSE)
        {
            vHandleDeleteList(psObj, psDSRLArg);
        }

    } while(FALSE);

    SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleModifyList
*
*   A DSRL rebuild event has been passed to us and we need to handle it now
*
*****************************************************************************/
static BOOLEAN bHandleModifyList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE,
            bLocked = FALSE,
            bNeedPrune = TRUE;
    DSRL_OBJECT hDSRL = DSRL_INVALID_OBJECT;
    DSRL_STATE_ENUM eNextState = DSRL_STATE_READY;

    do
    {
        FUEL_TARGET_DESC_STRUCT *psTarget;

        // Lock the owner object
        bLocked = SMSO_bLock(
            (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME
                ": bHandleModifyList() Unable to lock station owner");

            break;
        }

        // Extract the target descriptor
        psTarget = (FUEL_TARGET_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psTarget == NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME
                ": bHandleModifyList() Fuel Target list corrupted");
            break;
        }

        // Extract the DSRL handle
        hDSRL = psTarget->hDSRL;

        // This operation tells us explicitly if
        // we should do this or not
        if (psDSRLArg->uAction.sModify.bForceDSRLStateChange == TRUE)
        {
            // Tell the DSRL it is being updated
            DSRL_vSetState(hDSRL, DSRL_STATE_UPDATING);
        }

        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                // Just add to the target
                bSuccess = bExpandTarget(
                    psObj, psTarget, psDSRLArg);

                // Don't need to prune when adding
                // locations
                bNeedPrune = FALSE;
            }
            break;
            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                // Update the target now that
                // the location has changed
                bSuccess = bUpdateTarget(
                    psObj, psTarget, psDSRLArg);
            }
            break;
            case DSRL_MODIFY_OPERATION_REMOVE:
            case DSRL_MODIFY_OPERATION_REMOVEALL:
            {
                // Pull the listed locations out
                // of the target
                bSuccess = bReduceTarget(
                    psObj, psTarget, psDSRLArg);
            }
            break;
            default:
            {
                // Don't know what to do here
                bSuccess = FALSE;
            }
            break;
        }


    } while (FALSE);

    if (bSuccess == FALSE)
    {
        eNextState = DSRL_STATE_ERROR;

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bHandleModifyList() Error");
    }

    if (bLocked == TRUE)
    {
        if (DSRL_INVALID_OBJECT != hDSRL)
        {
            // Bring the DSRL to its next state
            DSRL_vSetState(hDSRL, eNextState);
        }

        if (bNeedPrune == TRUE)
        {
            // Prune the cache now that we've
            // reported all our changes to the DSRL
            bPruneCache(psObj);
        }

        // Unlock the owner now
        SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bHandleRefreshList
*
*   A DSRL rebuild event has been passed to us and we need to handle it now
*
*****************************************************************************/
static BOOLEAN bHandleRefreshList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE,
            bLocked = FALSE;
    DSRL_OBJECT hDSRL = DSRL_INVALID_OBJECT;
    DSRL_STATE_ENUM eNextState = DSRL_STATE_READY;

    do
    {
        FUEL_TARGET_DESC_STRUCT *psTarget;

        // Lock the owner object
        bLocked = SMSO_bLock(
            (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME
                ": bHandleRefreshList() Unable to lock station owner");

            // Stop processing now
            break;
        }

        // Extract the target descriptor
        psTarget = (FUEL_TARGET_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psTarget == NULL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME
                ": bHandleRefreshList() Fuel Target list corrupted");
            break;
        }

        // Extract the DSRL handle
        hDSRL = psTarget->hDSRL;

        // Iterate the cache for all entries associated with this target
        // pass them through the filter if they make it, add to the DSRL
        bSuccess = bUpdateDSRLForRefresh(psObj, psTarget);

    } while (FALSE);

    if (bSuccess == FALSE)
    {
        eNextState = DSRL_STATE_ERROR;

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bHandleRefreshList() Error");
    }

    if (bLocked == TRUE)
    {
        if (DSRL_INVALID_OBJECT != hDSRL)
        {
            // Bring the DSRL to its next state
            DSRL_vSetState(hDSRL, eNextState);
        }

        // Prune the cache now that we've
        // reported all our changes to the DSRL
        bPruneCache(psObj);

        // Unlock the owner now
        SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vHandleDeleteList
*
*   A DSRL delete event has been passed to us and we need to handle it now
*
*****************************************************************************/
static void vHandleDeleteList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bLocked = FALSE;
    do
    {
        FUEL_TARGET_DESC_STRUCT *psTarget;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Lock the owner object
        bLocked = SMSO_bLock(
            (SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if (FALSE == bLocked)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME
                ": vHandleDeleteList() Unable to lock station owner");

            break;
        }

        // Extract the target descriptor
        psTarget = (FUEL_TARGET_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);

        if ((FUEL_TARGET_DESC_STRUCT *)NULL == psTarget)
        {
            break;
        }

        // Remove this from our list of tracked targets for this service
        eReturnCode = OSAL.eLinkedListRemove(psTarget->hEntry);

        if (OSAL_SUCCESS != eReturnCode)
        {
            break;
        }


        // Clear the target's entry attribute
        psTarget->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Destroy this target
        vDestroyTarget(psTarget);

    } while (FALSE);

    if (TRUE == bLocked)
    {
        SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
    }

    return;
}

/*****************************************************************************
*
*   psCreateTarget
*
*   Create a new target for the application
*
*****************************************************************************/
static FUEL_TARGET_DESC_STRUCT *psCreateTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    FUEL_TARGET_DESC_STRUCT *psTarget = NULL;

    if ((psDSRLArg->eDSRLType != DSRL_TYPE_FAVORITES) &&
        (psDSRLArg->tNumTargets == 0))
    {
        // Can't do much with this!
        return NULL;
    }

    do
    {
        BOOLEAN bOk = TRUE;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Verify the target list is acceptable
        bOk = bVerifyTargetList(psDSRLArg);
        if (bOk == FALSE)
        {
            break;
        }

        // Get the target descriptor from the DSRL
        psTarget = (FUEL_TARGET_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);
       
        if (psTarget == NULL)
        {
            // Error!
            break;
        }

        // Tell the target who the manager is
        psTarget->psObj = psObj;

        // Keep track of the favorites flag value
        psTarget->bFavorite =
            (psDSRLArg->eDSRLType == DSRL_TYPE_FAVORITES);

        // We are not serializing this target's
        // DSRL at this time
        psTarget->bSerializeDSRL = FALSE;

        // Only initialize the entry handle if this is
        // a new target
        psTarget->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // This target needs to be processed
        psTarget->bProcessPending = TRUE;

        // This target's DSRL
        psTarget->hDSRL = psDSRLArg->hDSRL;

        // Create a list for the location list
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":TargetLocations" );

        // Create a list for the locations
        eReturnCode = OSAL.eLinkedListCreate(
            &psTarget->hLocations,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize this target
        psTarget->tCurrentRegion = FUEL_INVALID_REGION;
        psTarget->bCurrentRegionInDB = FALSE;
        psTarget->hCurrentLocation = OSAL_INVALID_LINKED_LIST_ENTRY;
        psTarget->bCurrentLocationIsCoordBased = FALSE;

        // Set the entry finalize function
        bOk = DSRL_bSetFinalizeFunction(
            psTarget->hDSRL,
            (DSRL_FINALIZE_FUNCTION)vDSRLFinalizeStationDelete,
            psTarget);
        if (FALSE == bOk)
        {
            // Error!
            break;
        }

        // Set the default sort function
        bOk = DSRL_bSetDefaultSortFunction (psTarget->hDSRL,
            (DSRL_SORT_FUNCTION)n16SortDSRLByPrice, psTarget);
        if (FALSE == bOk)
        {
            // Error!
            break;
        }

        // Add all the locations to our list now
        bOk = bAddLocationsToTarget(psObj, psTarget, psDSRLArg);
        if (bOk == FALSE)
        {
            // Error!
            break;
        }

        // Provide the caller with the
        // target we've created
        return psTarget;

    } while (FALSE);

    // Destroy this target now
    vDestroyTarget(psTarget);

    return NULL;
}

/*****************************************************************************
*
*   vDestroyTarget
*
*****************************************************************************/
static void vDestroyTarget (
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DSRL_STATE_ENUM eState;
    BOOLEAN bLocked;

    // Validate inputs
    if (psTarget == NULL)
    {
        return;
    }

    // Extract the manager pointer
    psObj = psTarget->psObj;

    // We're not serialize anything now
    psTarget->bSerializeDSRL = FALSE;

    // Lock the station owner
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME
            ": vDestroyTarget() Unable to lock station owner");
        return;
    }

    // Get this DSRL's current state
    eState = DSRL.eState(psTarget->hDSRL);
    if ((eState != DSRL_STATE_ERROR) &&
        (eState != DSRL_STATE_UNKNOWN))
    {
        // If the DSRL isn't in a weird state
        // then we should serialize it as we destroy it
        psTarget->bSerializeDSRL = TRUE;
    }

    // Tell the DB we're gonna start updating the price table
    psObj->psDBInterface->vUpdatePriceTable(psObj->hDBInterface, TRUE);

    // Remove / free all entries in the DSRL
    DSRL_vRemoveAllEntries(psTarget->hDSRL);

    // Done updating the price table now
    psObj->psDBInterface->vUpdatePriceTable(psObj->hDBInterface, FALSE);

    if (psTarget->hLocations != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all target locations
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psTarget->hLocations,
            (OSAL_LL_RELEASE_HANDLER)vDestroyTargetLocation);

        psTarget->hCurrentLocation = OSAL_INVALID_LINKED_LIST_ENTRY;

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psTarget->hLocations);
            psTarget->hLocations = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Reduce the cache now
    bReduceCacheForLocationChange(psObj, psTarget);

    // Destroy the DSRL & target
    DSRL_vDestroy(psTarget->hDSRL);

    // Prune the cache now
    bPruneCache(psObj);

    // Unlock the station owner
    SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);

    return;
}

/*****************************************************************************
*
*   bUpdateTarget
*
*****************************************************************************/
static BOOLEAN bUpdateTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess;

    do
    {
        FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation = NULL;
        BOOLEAN bBackgroundUpdatedInThisCall = FALSE;

        // Update this target's location
        // modify location object contents & update background radius if needed
        bSuccess = bUpdateTargetLocation(
            psObj, psTarget, psDSRLArg, &bBackgroundUpdatedInThisCall, &psLocation);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": bUpdateTarget() unable to update target list\n");
            break;
        }

        // Was the background location updated by our call above?
        if (bBackgroundUpdatedInThisCall == TRUE)
        {
            // Update the cache & this target's DSRL for the location / background change
            bSuccess = bReduceCacheForLocationChange(psObj, psTarget);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_MGR_OBJECT_NAME": bUpdateTarget() unable to update cache\n");
                break;
            }

            // We now need to perform some processing
            // for this target's background area
            psTarget->bProcessPending = TRUE;
            vStartProcessTimer(psObj);
        }

        // Iterate the cache for all entries associated with this target
        // pass them through the filter if they make it, add to the DSRL
        bSuccess = bUpdateDSRLForLocationChange(psObj, psTarget);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": bUpdateTarget() unable to update DSRL\n");
        }

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bExpandTarget
*
*****************************************************************************/
static BOOLEAN bExpandTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess;

    // Ensure we're dealing with a favorites list
    if (psTarget->bFavorite == FALSE)
    {
        return FALSE;
    }

    // Add the locations from the target arg
    bSuccess = bAddLocationsToTarget(psObj, psTarget, psDSRLArg);

    return bSuccess;
}

/*****************************************************************************
*
*   bReduceTarget
*
*   Reduce a target by removing all locations which match the contents
*   of the DSRL target arg
*
*****************************************************************************/
static BOOLEAN bReduceTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    BOOLEAN bSuccess = TRUE;

    // Ensure we're dealing with a favorites list
    if (psTarget->bFavorite == FALSE)
    {
        return FALSE;
    }

    if (psDSRLArg->uAction.sModify.eModifyType == DSRL_MODIFY_OPERATION_REMOVEALL)
    {
        // Remove all target locations
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psTarget->hLocations,
            (OSAL_LL_RELEASE_HANDLER)vDestroyTargetLocation);

        if (eReturnCode == OSAL_SUCCESS)
        {
            psTarget->hCurrentLocation =
                OSAL_INVALID_LINKED_LIST_ENTRY;
        }
    }
    else
    {
        // Iterate the list to remove all targets that match
        eReturnCode = OSAL.eLinkedListIterate(
            psTarget->hLocations,
            (OSAL_LL_ITERATOR_HANDLER)bIterateTargetLocationsForReduce,
            (void *)psDSRLArg);
    }

    if (eReturnCode == OSAL_SUCCESS)
    {
        // Reduce the contents of the DSRL / Cache due to the location change
        bSuccess = bReduceCacheForLocationChange(psObj, psTarget);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": bReduceTarget() unable to update cache\n");
        }
    }
    else if (eReturnCode != OSAL_NO_OBJECTS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bReduceTarget() unable to access target locations (%s)\n",
            OSAL.pacGetReturnCodeName(eReturnCode));

        bSuccess = FALSE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bReduceCacheForLocationChange
*
*****************************************************************************/
static BOOLEAN bReduceCacheForLocationChange (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Iterate the station cache to see if we
    // need to pull some out of this DSRL now
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->sWorkCtrl.hStations,
        (OSAL_LL_ITERATOR_HANDLER)bIterateCacheForBackgroundChange,
        (void *)psTarget);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bReduceCacheForLocationChange() unable to iterate cache");

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateDSRLForLocationChange
*
*****************************************************************************/
static BOOLEAN bUpdateDSRLForLocationChange (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL_TARGET_LOCATION_UPDATE_STRUCT sUpdate;

    sUpdate.psTarget = psTarget;
    sUpdate.bDSRLUpdated = FALSE;

    // Iterate the station cache to see if we
    // need to pull some out of this DSRL now
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->sWorkCtrl.hStations,
        (OSAL_LL_ITERATOR_HANDLER)bIterateCacheForLocationUpdate,
        (void *)&sUpdate);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bUpdateDSRLForLocationChange() unable to iterate cache");

        return FALSE;
    }

    if (sUpdate.bDSRLUpdated == TRUE)
    {
        // Tell the DSRL it is being updated
        DSRL_vSetState(psTarget->hDSRL, DSRL_STATE_UPDATING);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateDSRLForRefresh
*
*****************************************************************************/
static BOOLEAN bUpdateDSRLForRefresh (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    FUEL_REFRESH_DSRL_STRUCT sRefresh;

    // Refresh this target
    sRefresh.psTarget = psTarget;
    sRefresh.bDSRLUpdated = FALSE;

    // Start by flushing old entries
    sRefresh.bFlush = TRUE;

    // Iterate the station cache to update the DSRL contents
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->sWorkCtrl.hStations,
        (OSAL_LL_ITERATOR_HANDLER)bIterateCacheForDSRLRefresh,
        (void *)&sRefresh);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bUpdateDSRLForRefresh() unable to iterate cache");

        return FALSE;
    }

    if (sRefresh.bDSRLUpdated == TRUE)
    {
        // Put the DSRL in the updating state if necessary
        DSRL_vSetState(psTarget->hDSRL, DSRL_STATE_UPDATING);
    }

    // Now we're going to try adding some entries
    sRefresh.bFlush = FALSE;
    sRefresh.bDSRLUpdated = FALSE;

    // Iterate the station cache to update the DSRL contents
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->sWorkCtrl.hStations,
        (OSAL_LL_ITERATOR_HANDLER)bIterateCacheForDSRLRefresh,
        (void *)&sRefresh);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bUpdateDSRLForRefresh() unable to iterate cache");

        return FALSE;
    }

    if (sRefresh.bDSRLUpdated == TRUE)
    {
        // Put the DSRL in the updating state if necessary
        DSRL_vSetState(psTarget->hDSRL, DSRL_STATE_UPDATING);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bVerifyTargetList
*
*****************************************************************************/
static BOOLEAN bVerifyTargetList (
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    DSRL_TARGET_TYPE_ENUM eTargetType;
    size_t tIndex, tNumRadiusLocations = 0;
    BOOLEAN bIsPoint, bOk = TRUE;
    LOCATION_OBJECT hLocation;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // Verify all targets are locations
    for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
    {
        // Get the current type
        eTargetType = DSRL_TARGET.eType(psDSRLArg->ahTargetList[tIndex]);
        if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
        {
            // Only allow location targets in here
            bOk = FALSE;
            break;
        }

        // Grab the location / radius handle
        hLocation = DSRL_TARGET.hLocation(psDSRLArg->ahTargetList[tIndex]);
        if (hLocation == LOCATION_INVALID_OBJECT)
        {
            bOk = FALSE;
            break;
        }

        eReturn = LOCATION.eIsPoint(hLocation, &bIsPoint);
        if ((eReturn == SMSAPI_RETURN_CODE_SUCCESS) &&
            (bIsPoint == FALSE))
        {
            tNumRadiusLocations++;
        }
    }

    if (bOk == TRUE)
    {
        // There can only be one radius location at most
        if (tNumRadiusLocations > 1)
        {
            bOk = FALSE;
        }
        // Otherwise ensure there are no other targets specified when
        // a location / radius is provided
        else if ((tNumRadiusLocations == 1) &&
                 (psDSRLArg->tNumTargets != tNumRadiusLocations))
        {
            bOk = FALSE;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   bAddLocationsToTarget
*
*   Adds the locations contained in the DSRL target arg to the given
*   target descriptor
*
*****************************************************************************/
static BOOLEAN bAddLocationsToTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tIndex;
    LOCATION_OBJECT hLocation;
    LOCID_OBJECT hLocId;
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOk;
    FUEL_REGION_INTEREST_STRUCT sInterest;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
    {
        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":TargetLocation %u",
            psObj->psOTAInterface->tDataID );

        // Create a new location descriptor
        psLocation = (FUEL_TARGET_LOCATION_DESC_STRUCT *)
            SMSO_hCreate(&acName[0],
                sizeof(FUEL_TARGET_LOCATION_DESC_STRUCT),
                (SMS_OBJECT)psObj, FALSE);
        if (psLocation == NULL)
        {
            return FALSE;
        }

        // Create a name for the region list
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            FUEL_MGR_OBJECT_NAME":RegionsOfInterest %u",
            psObj->psOTAInterface->tDataID );

        // Create a list for the regions of interest
        eReturnCode = OSAL.eLinkedListCreate(
            &psLocation->hRegionsOfInterest,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)n16CompareRegionsOfInterestById,
            OSAL_LL_OPTION_LINEAR);
        if (eReturnCode != OSAL_SUCCESS)
        {
            vDestroyTargetLocation(psLocation);
            return FALSE;
        }

        // Get the current location object from the arg
        hLocation = DSRL_TARGET.hLocation(
            psDSRLArg->ahTargetList[tIndex]);

        // Copy the handle
        // This is our own object to use, the DSRL
        // already handled duplicating the target for us
        psLocation->hLocation = hLocation;
        if (psLocation->hLocation == LOCATION_INVALID_OBJECT)
        {
            vDestroyTargetLocation(psLocation);
            return FALSE;
        }

        // We own this handle now
        psDSRLArg->ahTargetList[tIndex] = DSRL_TARGET_INVALID_OBJECT;

        // Create the background location
        bOk = bCreateBackgroundLocation(
            psObj, psDSRLArg, psLocation);
        if (bOk == FALSE)
        {
            vDestroyTargetLocation(psLocation);
            return FALSE;
        }

        // Add this location to the target
        eReturnCode = OSAL.eLinkedListAdd(
                psTarget->hLocations, &psLocation->hEntry,
                (void *)psLocation);

        if (eReturnCode != OSAL_SUCCESS)
        {
            vDestroyTargetLocation(psLocation);
            return FALSE;
        }

        hLocId = LOCATION.hLocID(psLocation->hLocation);
        if (hLocId != LOCID_INVALID_OBJECT)
        {
            LOC_ID tId;
            FUEL_REGION tRegion;
            FUEL_STATION_ID tStation;

            // Grab the ID
            tId = LOCID.tID(hLocId);

            tRegion = FUEL_LOCID_TO_REG(tId);
            tStation = FUEL_LOCID_TO_STATID(tId);

            // Add this as a region of interest
            sInterest.psLocation = psLocation;
            sInterest.bBackgroundOnly = FALSE;
            sInterest.bSuccess = TRUE;

            // Add this as a region of interest
            bOk = bAddRegionOfInterest(
                (FUEL_SERVICE_OBJECT)psObj, tRegion, &sInterest);

            // If that worked then attempt to pull in stations from
            // from the cache
            if (bOk == TRUE)
            {
                BOOLEAN bLoaded;

                // Load this station into this target & DSRL
                bLoaded = bLoadStationIntoDSRLFromCache(
                    psObj, psTarget, tRegion, tStation);
                // If we didn't load the station, let's search the
                // db for the station and try again.
                if ((bLoaded == FALSE) && (psTarget->bFavorite == TRUE))
                {
                    bLoaded = bLoadPricesForFavoriteStationID(psObj,psTarget,tRegion, tStation);
                }

                if (bLoaded == FALSE)
                {
                    // We now need to perform some processing
                    // for this target's background area
                    psTarget->bProcessPending = TRUE;
                    vStartProcessTimer(psObj);
                }
            }
        }
        else
        {
            if (psLocation->hLocation != psLocation->hBackground)
            {
                // Add the regions of interest for this background location
                sInterest.psLocation = psLocation;
                sInterest.bBackgroundOnly = TRUE;
                sInterest.bSuccess = TRUE;
                bOk = psObj->psOTAInterface->bAddRegionsForLocation(
                    psObj->hOTAInterface, psLocation->hBackground, (void *)&sInterest );
                if ((bOk == FALSE) || (sInterest.bSuccess == FALSE))
                {
                    OSAL.eLinkedListRemove(psLocation->hEntry);
                    vDestroyTargetLocation(psLocation);
                    return FALSE;
                }
            }

            // Add the regions of interest for this location
            sInterest.psLocation = psLocation;
            sInterest.bBackgroundOnly = FALSE;
            sInterest.bSuccess = TRUE;
            bOk = psObj->psOTAInterface->bAddRegionsForLocation(
                psObj->hOTAInterface, psLocation->hLocation, (void *)&sInterest );
            if ((bOk == FALSE) || (sInterest.bSuccess == FALSE))
            {
                OSAL.eLinkedListRemove(psLocation->hEntry);
                vDestroyTargetLocation(psLocation);
                return FALSE;
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bCreateBackgroundLocation
*
*****************************************************************************/
static BOOLEAN bCreateBackgroundLocation (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation
        )
{
    BOOLEAN bSuccess = TRUE;

    // Initialize the background as equal to the
    // current operational location
    psLocation->hBackground = psLocation->hLocation;

    // Only perform calculate background
    // processing location on device DSRLs
    if (psDSRLArg->eDSRLType == DSRL_TYPE_DEVICE)
    {
        DISTANCE_OBJECT hRadius;

        // Get this location's radius
        hRadius = LOCATION.hRadius(psLocation->hLocation);

        // Compare to the background radius only if
        // there is a radius available here
        if (hRadius != DISTANCE_INVALID_OBJECT)
        {
            N16 n16Compare;

            // Determine which radius is smaller
            n16Compare = DISTANCE.n16Compare(
                hRadius, psObj->sWorkCtrl.hBackgroundRadius);
            if (n16Compare < 0)
            {
                // Duplicate this location
                psLocation->hBackground = LOCATION.hDuplicate(psLocation->hLocation);

                // Replace with our processing radius
                bSuccess = LOCATION_bUpdateRadius(
                    psLocation->hBackground,
                    psObj->sWorkCtrl.hBackgroundRadius, TRUE);
            }
        }
    }

    if (bSuccess == TRUE)
    {
        // This background location has been updated
        psLocation->bBackgroundLocationUpdated = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bUpdateTargetLocation
*
*****************************************************************************/
static BOOLEAN bUpdateTargetLocation (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg,
    BOOLEAN *pbBackgroundUpdatedInThisCall,
    FUEL_TARGET_LOCATION_DESC_STRUCT **ppsLocation
        )
{
    BOOLEAN bSuccess = FALSE;

    if ((psTarget->bFavorite == FALSE) &&
        (psDSRLArg->tNumTargets == 1))
    {
        // Grab the first location
        FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation = NULL;
        OSAL_LINKED_LIST_ENTRY hEntry;
        LOCATION_OBJECT hLocation;
        DSRL_TARGET_TYPE_ENUM eTargetType;
        BOOLEAN bIsPoint = FALSE;
        BOOLEAN bStillInBackgroundLocation;
        SMSAPI_RETURN_CODE_ENUM eReturn;

        // Get the target type
        eTargetType = DSRL_TARGET.eType(psDSRLArg->ahTargetList[0]);

        if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
        {
            return FALSE;
        }

        // Extract the location object from the target arg
        hLocation = DSRL_TARGET.hLocation(psDSRLArg->ahTargetList[0]);
        if (hLocation == LOCATION_INVALID_OBJECT)
        {
            return FALSE;
        }

        // Checking if it is a location with non-empty area
        eReturn = LOCATION.eIsPoint(hLocation, &bIsPoint);
        if ((eReturn == SMSAPI_RETURN_CODE_SUCCESS) &&
            (bIsPoint == TRUE))
        {
            return FALSE;
        }

        // Grab the first (and only) target entry
        hEntry = OSAL.hLinkedListFirst(
            psTarget->hLocations, (void **)&psLocation);
        if ((hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
            (psLocation == NULL))
        {
            return FALSE;
        }

        // Destroy any old location object present if possible
        if ((psLocation->hLocation != LOCATION_INVALID_OBJECT) &&
            (psLocation->hLocation != psLocation->hBackground))
        {
            LOCATION.vDestroy(psLocation->hLocation);
        }

        // Update the location to use the newly provided target
        psLocation->hLocation = hLocation;

        // We own this handle now
        psDSRLArg->ahTargetList[0] = DSRL_TARGET_INVALID_OBJECT;


        // See if this location is still within the
        // background area (with our given threshold)
        bStillInBackgroundLocation =
            LOCATION_bContains(
                psLocation->hBackground,
                psLocation->hLocation,
                DISTANCE_INVALID_OBJECT);
        if (bStillInBackgroundLocation == FALSE)
        {
            // Destroy the old background location, if necessary
            if (psLocation->hBackground != psLocation->hLocation)
            {
                LOCATION.vDestroy(psLocation->hBackground);
            }

            // Create a new background location
            bSuccess = bCreateBackgroundLocation(
                psObj, psDSRLArg, psLocation);

            // Tell that caller that we just updated this
            // background location
            *pbBackgroundUpdatedInThisCall = TRUE;
        }
        else
        {
            // No problems yet
            bSuccess = TRUE;
        }

        if (bSuccess == TRUE)
        {
            FUEL_REGION_INTEREST_STRUCT sInterest;
            BOOLEAN bRecalcBackgroundRegions = FALSE;
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Only re-process the regions for the background if it changed
            // and if the background is not the same as the location provided
            // by the target arg
            if ((bStillInBackgroundLocation == FALSE) &&
                (psLocation->hLocation != psLocation->hBackground))
            {
                bRecalcBackgroundRegions = TRUE;
            }

            // Remove the old regions of interest which are either
            // background only or both background & location
            eReturnCode = OSAL.eLinkedListIterate(
                psLocation->hRegionsOfInterest,
                (OSAL_LL_ITERATOR_HANDLER)bRemoveOldRegionsOfInterest,
                &bRecalcBackgroundRegions);
            if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
            {
                return FALSE;
            }

            if (bRecalcBackgroundRegions == TRUE)
            {
                // Add the regions of interest for this background location
                sInterest.psLocation = psLocation;
                sInterest.bBackgroundOnly = TRUE;
                sInterest.bSuccess = TRUE;
                bSuccess = psObj->psOTAInterface->bAddRegionsForLocation(
                    psObj->hOTAInterface, psLocation->hBackground, (void *)&sInterest );
                if ((bSuccess == FALSE) || (sInterest.bSuccess == FALSE))
                {
                    return FALSE;
                }
            }

            // Add the regions of interest for this location
            sInterest.psLocation = psLocation;
            sInterest.bBackgroundOnly = FALSE;
            sInterest.bSuccess = TRUE;
            bSuccess = psObj->psOTAInterface->bAddRegionsForLocation(
                psObj->hOTAInterface, psLocation->hLocation, (void *)&sInterest );
            if ((bSuccess == FALSE) || (sInterest.bSuccess == FALSE))
            {
                return FALSE;
            }

            // Provide the caller with the location entry
            *ppsLocation = psLocation;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bVerifyQueryResultLocation
*
*   This manager interface API is used by the DB interface when it needs
*   to verify a location encountered from the DB
*
*****************************************************************************/
static BOOLEAN bVerifyQueryResultLocation (
    FUEL_SERVICE_OBJECT hFuelService,
    LOC_ID tLocId,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    FUEL_MGR_OBJECT_STRUCT *psObj =
        (FUEL_MGR_OBJECT_STRUCT *)hFuelService;

    return bStationMeetsTargetLocationCriteria(
        psObj->sWorkCtrl.psCurrentTarget,
        FALSE, tLocId, hLat, hLon);
}

/*****************************************************************************
*
*   bStationMeetsTargetLocationCriteria
*
*   Test to determine if a given station meets the location criteria for
*   a given target
*
*****************************************************************************/
static BOOLEAN bStationMeetsTargetLocationCriteria (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN bUseBackground,
    LOC_ID tLocID,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    BOOLEAN bLocationAccepted;
    OSAL_LINKED_LIST_ENTRY hEntry;
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation = NULL;
    LOCATION_OBJECT hLocation;
    LOCID_OBJECT hCurLocID;
    LOC_ID tCurLocID;

    // Get the first entry
    hEntry = OSAL.hLinkedListFirst(
        psTarget->hLocations, (void **)&psLocation);
    if (psLocation == NULL)
    {
        // No locations in this list, so it'll never match anything
        return FALSE;
    }

    // Go through the whole list; ensure the contents are valid
    while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
           (psLocation != NULL))
    {
        if (bUseBackground == TRUE)
        {
            hLocation = psLocation->hBackground;
        }
        else
        {
            hLocation = psLocation->hLocation;
        }

        // Get this location's locid
        hCurLocID = LOCATION.hLocID(hLocation);

        // If this location has a LOCID,
        // just use that for a comparison
        if (hCurLocID != LOCID_INVALID_OBJECT)
        {
            // Extract the ID value
            tCurLocID = LOCID.tID(hCurLocID);

            if (tCurLocID == tLocID)
            {
                // Match!
                return TRUE;
            }
        }
        else // No LOCID so test the location criteria
        {
            // Does this station meet the current
            // location criteria?
            bLocationAccepted =
                LOCATION.bCoordinatesWithinArea( hLocation, hLat, hLon );
            if (bLocationAccepted == TRUE)
            {
                // Yes, this station is found within
                // on of the target's locations
                return TRUE;
            }
        }

        // Get the next location
        psLocation = NULL;
        hEntry = OSAL.hLinkedListNext(hEntry, (void **)&psLocation);
    }

    // No, this station is not found within
    // the locations in this target
    return FALSE;
}

/*****************************************************************************
*
*   bMoveToNextLocation
*
*   Move a target to the next fuel location, determined by the application
*   provided LOCATION objects.
*
*****************************************************************************/
static BOOLEAN bMoveToNextLocation (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    BOOLEAN bMoved = FALSE,
            bIterateLocations = TRUE;
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation;
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psCurRegion;

    // Always clear out the current station
    psTarget->tCurrentStation = FUEL_INVALID_STATION_ID;

    // If the current location object is coordinate based,
    // then we need to see if there are any more
    // regions to query
    if (psTarget->bCurrentLocationIsCoordBased == TRUE)
    {
        // Grab the location handle
        psLocation = (FUEL_TARGET_LOCATION_DESC_STRUCT *)
            OSAL.pvLinkedListThis(psTarget->hCurrentLocation);

        // Is there another region of interest for our
        // current location?

        // Move the region entry handle
        psLocation->hCurrentRegion = OSAL.hLinkedListNext(
            psLocation->hCurrentRegion, (void **)&psCurRegion);
        if (psCurRegion != NULL)
        {
            psTarget->tCurrentRegion = psCurRegion->tRegionID;
            if (psCurRegion->psRegion == NULL)
            {
                psTarget->bCurrentRegionInDB = FALSE;
            }
            else
            {
                psTarget->bCurrentRegionInDB = TRUE;
            }

            // Don't go to the next location
            bIterateLocations = FALSE;

            // Tell the caller we moved along
            bMoved = TRUE;
        }
        else
        {
            // Move to the next location
            psTarget->hCurrentLocation =
                OSAL.hLinkedListNext(psTarget->hCurrentLocation, NULL);
        }
    }
    else
    {
        if (psTarget->hCurrentLocation == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Grab the first entry
            psTarget->hCurrentLocation =
                OSAL.hLinkedListFirst(psTarget->hLocations, NULL);
        }
        else
        {
            // Move to the next location
            psTarget->hCurrentLocation =
                OSAL.hLinkedListNext(psTarget->hCurrentLocation, NULL);
        }
    }

    // Do we need to continue to the next location, and
    // are there more locations to inspect?
    if ((bIterateLocations == TRUE) &&
        (psTarget->hCurrentLocation != OSAL_INVALID_LINKED_LIST_ENTRY))
    {
        LOCID_OBJECT hLocId;

        // Reset state variables
        psTarget->tCurrentRegion = FUEL_INVALID_REGION;
        psTarget->bCurrentRegionInDB = FALSE;
        psTarget->tCurrentStation = FUEL_INVALID_STATION_ID;
        psTarget->bCurrentLocationIsCoordBased = FALSE;

        // Now, inspect the next location
        psLocation = (FUEL_TARGET_LOCATION_DESC_STRUCT *)
            OSAL.pvLinkedListThis(psTarget->hCurrentLocation);
        hLocId = LOCATION.hLocID(psLocation->hLocation);

        // Is there a valid hLocId associated with it?
        if (hLocId != LOCID_INVALID_OBJECT)
        {
            LOC_ID tId;

            // Grab the ID
            tId = LOCID.tID(hLocId);

            // Grab the region & station ID directly
            // from the locid
            psTarget->tCurrentRegion = FUEL_LOCID_TO_REG(tId);
            psTarget->bCurrentRegionInDB = TRUE;
            psTarget->tCurrentStation = FUEL_LOCID_TO_STATID(tId);
            bMoved = TRUE;
        }
        else // This is a coordinate based location
        {
            // This location is based on map coordinates
            psTarget->bCurrentLocationIsCoordBased = TRUE;

            // Get the first region
            psCurRegion = NULL;
            psLocation->hCurrentRegion =
                OSAL.hLinkedListFirst(
                    psLocation->hRegionsOfInterest,
                    (void **)&psCurRegion);
            if (psCurRegion != NULL)
            {
                psTarget->tCurrentRegion = psCurRegion->tRegionID;
                if (psCurRegion->psRegion == NULL)
                {
                    psTarget->bCurrentRegionInDB = FALSE;
                }
                else
                {
                    psTarget->bCurrentRegionInDB = TRUE;
                }

                // We have moved
                bMoved = TRUE;
            }
        }
    }

    return bMoved;
}

/*****************************************************************************
*
*   bInitDSRL
*
*****************************************************************************/
static BOOLEAN bInitDSRL (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    BOOLEAN bOk;
    DSRL_STATE_ENUM eNextState = DSRL_STATE_READY;

    // Load the prices from the DB for the DSRL
    bOk = bLoadPricesForDSRL(psObj, psTarget);
    if (bOk == FALSE)
    {
        eNextState = DSRL_STATE_ERROR;
    }

    DSRL_vSetState(psTarget->hDSRL, eNextState);

    if (bOk == TRUE)
    {
        // Start the processing timer
        vStartProcessTimer(psObj);
    }

    return bOk;
}

/*****************************************************************************
*
*   bProcessTarget
*
*   Perform the processing required to extract all necessary stations
*   from the DB according to this target's specifications
*
*****************************************************************************/
static BOOLEAN bProcessTarget (
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    BOOLEAN bSuccess = TRUE,
            bMoved = FALSE;
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation = NULL,
                                     *psPrevLocation = NULL;
    FUEL_MGR_OBJECT_STRUCT *psObj = psTarget->psObj;

    // If the background location(s) changed, update the station cache
    do
    {
        // Move us to the first location to process
        bMoved = bMoveToNextLocation(psObj, psTarget);
        if (bMoved == FALSE)
        {
            // We're all done now.. make sure we mark
            // the last location as processed
            if (psPrevLocation != NULL)
            {
                // The previous location was valid and we're done with it
                psPrevLocation->bBackgroundLocationUpdated = FALSE;
            }
            break;
        }

        // Grab the current location handle
        psLocation = (FUEL_TARGET_LOCATION_DESC_STRUCT *)
            OSAL.pvLinkedListThis(psTarget->hCurrentLocation);
        if (psLocation == NULL)
        {
            bSuccess = FALSE;
            break;
        }

        // Determine if we're done with the last location we processed
        if (psLocation != psPrevLocation)
        {
            // The location changed
            if (psPrevLocation != NULL)
            {
                // The previous location was valid and we're done with it
                psPrevLocation->bBackgroundLocationUpdated = FALSE;
            }

            // Update the previous location
            psPrevLocation = psLocation;
        }

        // Was the background location updated?
        if (psLocation->bBackgroundLocationUpdated == TRUE)
        {
            DATASERVICE_IMPL_vLog(
                FUEL_MGR_OBJECT_NAME": Processing background for target\n");

            // Pull in stations which now match the new background area
            bSuccess = bLoadStationsForDSRL(psObj, psTarget, psLocation);
        }

    } while (bSuccess == TRUE);

    if (bSuccess == FALSE)
    {
        // The manager and the DSRL are now in a bad state
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
    }

    // Reset the attributes used during the build process
    psTarget->bCurrentLocationIsCoordBased = FALSE;
    psTarget->tCurrentRegion = FUEL_INVALID_REGION;
    psTarget->bCurrentRegionInDB = FALSE;
    psTarget->tCurrentStation = FUEL_INVALID_STATION_ID;

    return bSuccess;
}

/*****************************************************************************
*
*   bAgeoutPrices
*
*   Ageout prices -- all prices which exceed the expire age will be removed
*   from all stations in the cache & DSRLs.
*
*****************************************************************************/
static BOOLEAN bAgeoutPrices (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    UN32 un32PriceExpireAge,
    UN32 *pun32OldestPriceAfterFlush
        )
{
    BOOLEAN bLocked;
    BOOLEAN bSuccess = FALSE;

    // Lock the station owner
    bLocked =
        SMSO_bLock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner,
            OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        FUEL_STATION_PRICE_AGEOUT_STRUCT sAgeout;

        puts(FUEL_MGR_OBJECT_NAME": Flush station prices by age");

        // Iterate all DSRLs to lock them
        // if possible
        OSAL.eLinkedListIterate(
            psObj->hTargets,
            (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsForDSRLAccess,
            (void *)(size_t)TRUE);

        // Describe how we're going to update
        // our stations
        sAgeout.psObj = psObj;
        sAgeout.un32PriceExpireAge = un32PriceExpireAge;
        sAgeout.pun32OldestPriceAfterFlush = pun32OldestPriceAfterFlush;
        sAgeout.bLastStationUpdated = FALSE;
        sAgeout.hLastStation = FUEL_STATION_INVALID_OBJECT;

        // Iterate the cache to handle ageout
        // and update DSRL(s) if necessary
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->sWorkCtrl.hStations,
            (OSAL_LL_ITERATOR_HANDLER)bIterateCacheForAgeout,
            (void *)&sAgeout);

        bSuccess = (eReturnCode == OSAL_SUCCESS);

        // Iterate all DSRLs to unlock them and bring them to ready (if possible)
        OSAL.eLinkedListIterate(
            psObj->hTargets,
            (OSAL_LL_ITERATOR_HANDLER)bIterateTargetsForDSRLAccess,
            (void *)(size_t)FALSE);

        // Unlock the station owner
        SMSO_vUnlock((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bLoadStationIntoDSRLFromCache
*
*   A specific region & station pair was provided to a DSRL.  Before we
*   hit the database, attempt to load this specific station from the cache
*   into the destination DSRL
*
*****************************************************************************/
static BOOLEAN bLoadStationIntoDSRLFromCache (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    FUEL_REGION tRegion,
    FUEL_STATION_ID tStationId
        )
{
    BOOLEAN bStationLoadedInThisTarget = FALSE;
    FUEL_STATION_OBJECT hStation;

    // Attempt to find this station in the cache
    hStation = hFindStationInCache(
        psObj, psTarget, &bStationLoadedInThisTarget,
        tRegion, tStationId);
    if (hStation != FUEL_STATION_INVALID_OBJECT)
    {
        // Add this station to the cache for this target
        if (bStationLoadedInThisTarget == FALSE)
        {
            // Add this station to the DSRL
            bStationLoadedInThisTarget = bAddToDSRL(
                psTarget->hDSRL, hStation);
            if (bStationLoadedInThisTarget == TRUE)
            {
                // Add this station to the cache and mark it as placed
                // in the DSRL
                bStationLoadedInThisTarget = bAddStationToCache(
                    psObj, hStation, psTarget, TRUE, FALSE );
            }
        }
    }

    return bStationLoadedInThisTarget;
}

/*****************************************************************************
*
*   bAddToDSRL
*
*****************************************************************************/
static BOOLEAN bAddToDSRL (
    DSRL_OBJECT hDSRL,
    FUEL_STATION_OBJECT hStation
        )
{
    DSRL_ADD_REPLACE_RESULT_ENUM eResult;

    // Attempt to add this station to the DSRL
    eResult = DSRL_eAddEntry(
        hDSRL, (DSRL_ENTRY_OBJECT)hStation);

    if (eResult == DSRL_ADD_REPLACE_OK)
    {
        return TRUE;
    }
    else if (eResult == DSRL_ADD_REPLACE_ERROR)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bAddToDSRL() DSRL add error!");
    }

    return FALSE;
}

/*****************************************************************************
*
*   vDestroyTargetLocation
*
*****************************************************************************/
static void vDestroyTargetLocation (
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation
        )
{
    if (psLocation != NULL)
    {
        BOOLEAN bBackgroundDifferent =
            (psLocation->hBackground != psLocation->hLocation);

        // Destroy the location
        LOCATION.vDestroy(psLocation->hLocation);
        psLocation->hLocation = LOCATION_INVALID_OBJECT;

        // Destroy the background if it is a different object
        if (bBackgroundDifferent == TRUE)
        {
            LOCATION.vDestroy(psLocation->hBackground);
            psLocation->hBackground = LOCATION_INVALID_OBJECT;
        }

        psLocation->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Destroy the list of regions
        if (psLocation->hRegionsOfInterest != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            BOOLEAN bRemoveAll = TRUE;

            // Remove all the entries
            eReturnCode = OSAL.eLinkedListIterate(
                psLocation->hRegionsOfInterest,
                (OSAL_LL_ITERATOR_HANDLER)bRemoveOldRegionsOfInterest,
                &bRemoveAll);

            if (eReturnCode == OSAL_SUCCESS)
            {
                OSAL.eLinkedListDelete(psLocation->hRegionsOfInterest);
                psLocation->hRegionsOfInterest = OSAL_INVALID_OBJECT_HDL;
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psLocation);
    }
    return;
}

/*****************************************************************************
*
*   bIterateTargetLocationsForReduce
*
*****************************************************************************/
static BOOLEAN bIterateTargetLocationsForReduce (
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    size_t tIndex;
    BOOLEAN bMatch;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    LOCID_OBJECT hLocID, hCurLocID;
    N16 n16Compare;

    // Get the loc id of the current location
    hLocID = LOCATION.hLocID(psLocation->hLocation);

    // Compare this location to the contents of the target arg
    // if that location matches, remove it
    for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
    {
        bMatch = FALSE;

        // Only perform this check if the location in psLocation
        // has a valid locid
        if (hLocID != LOCID_INVALID_OBJECT)
        {
            // Get this location's locid
            hCurLocID = LOCATION.hLocID(
                (LOCATION_OBJECT)psDSRLArg->ahTargetList[tIndex]);

            // If both have valid locids, then compare them
            if (hCurLocID != LOCID_INVALID_OBJECT)
            {
                n16Compare = LOCID.n16Compare(hLocID, hCurLocID);
                if (n16Compare == 0)
                {
                    // These two locations match!
                    bMatch = TRUE;
                }
            }
        }

        if (bMatch == FALSE)
        {
            // Compare these two location targets if needed
            bMatch = LOCATION_bCompare(
                psLocation->hLocation,
                (LOCATION_OBJECT)psDSRLArg->ahTargetList[tIndex]);
        }

        if (bMatch == TRUE)
        {
            eReturnCode = OSAL.eLinkedListRemove(psLocation->hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                // Leave the entry here to help track this issue
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_MGR_OBJECT_NAME": Unable to remove location "
                    "entry in bIterateTargetLocationsForReduce()");
            }
            else
            {
                // Destroy the entry
                vDestroyTargetLocation(psLocation);
            }
        }
    }

    // Go through the entire list
    return TRUE;
}

/*****************************************************************************
*
*   bIteratePriceMsgs
*
*****************************************************************************/
static BOOLEAN bIteratePriceMsgs (
    FUEL_REGION_PRICE_UPDATE_MSG_STRUCT *psPriceUpdate,
    FUEL_PRICE_HASH_ITERATOR_STRUCT *psIterator
        )
{
    UN32 un32MsgAge;
    BOOLEAN bHashMatched =
        (psIterator->un32Hash == psPriceUpdate->un32Hash);

    // If the hash value matched, make sure
    // to tell the caller we found the entry
    if (bHashMatched == TRUE)
    {
        psIterator->bFound = TRUE;
    }

    // Calculate this message's age
    un32MsgAge = psIterator->un32CurrentTime - psPriceUpdate->un32TimeStamp;

    // Has this hash reached retirement?
    if (un32MsgAge > FUEL_PRICE_HASH_LIFETIME)
    {
        // Is this the hash we're looking for?
        if (bHashMatched == TRUE)
        {
            // This matched, but we don't filter since
            // the hash in the list has expired.  Just
            // update this entry's reception time
            psPriceUpdate->un32TimeStamp = psIterator->un32CurrentTime;
        }
        else
        {
            // Remove this entry now
            OSAL.eLinkedListRemove(psPriceUpdate->hEntry);
            psPriceUpdate->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psPriceUpdate->un32Hash = 0;
            psPriceUpdate->un32TimeStamp = 0;
            SMSO_vDestroy((SMS_OBJECT)psPriceUpdate);
        }
    }
    // Otherwise, if this hash matches then we don't
    // want this message
    else if (bHashMatched == TRUE)
    {
        psIterator->bFiltered = TRUE;
    }

    // Iterate all entries in this list
    return TRUE;
}

/*****************************************************************************
*
*   bIterateTargetsToProcess
*
*****************************************************************************/
static BOOLEAN bIterateTargetsToProcess (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN *pbSuccess
        )
{
    BOOLEAN bSuccess = TRUE;

    if (psTarget->bProcessPending == TRUE)
    {
        // Reset the "current" values for this target
        psTarget->hCurrentLocation = OSAL_INVALID_LINKED_LIST_ENTRY;
        psTarget->bCurrentLocationIsCoordBased = FALSE;

        DATASERVICE_IMPL_vLog(FUEL_MGR_OBJECT_NAME": Processing Next Target\n");

        // Process all that needs to be processed for this target
        bSuccess = bProcessTarget(psTarget);
        if (bSuccess == FALSE)
        {
            *pbSuccess = FALSE;
        }

        psTarget->bProcessPending = FALSE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bIterateTargetsForDSRLReady
*
*   An OSAL iterator which is used to switch all DSRLs that are in the
*   UPDATING state to the READY state due to the completion of a region
*
*****************************************************************************/
static BOOLEAN bIterateTargetsForDSRLReady (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    void *pvUnused
        )
{
    DSRL_STATE_ENUM eCurrentState;

    if (psTarget == NULL)
    {
        // Stop!
        return FALSE;
    }

    // Get this DSRL's current state
    eCurrentState = DSRL.eState(psTarget->hDSRL);

    // All DSRLs which are currently in the UPDATING
    // state are now ready
    if (eCurrentState == DSRL_STATE_UPDATING)
    {
        BOOLEAN bLocked;
        bLocked = SMSO_bLock(
            (SMS_OBJECT)psTarget->hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            // Set the DSRL to ready
            DSRL_vSetState(psTarget->hDSRL, DSRL_STATE_READY);
        }

        SMSO_vUnlock((SMS_OBJECT)psTarget->hDSRL);
    }

    // Iterate all targets
    return TRUE;
}

/*****************************************************************************
*
*   bIterateTargetsForDSRLAccess
*
*****************************************************************************/
static BOOLEAN bIterateTargetsForDSRLAccess (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    void *pvArg
        )
{
    BOOLEAN bLockRequested = (BOOLEAN)(size_t)pvArg;
    DSRL_STATE_ENUM eCurrentState;

    if (psTarget == NULL)
    {
        // Stop!
        return FALSE;
    }

    // Get this DSRL's current state
    eCurrentState = DSRL.eState(psTarget->hDSRL);

    if (bLockRequested == TRUE)
    {
        // This DSRL is not currently locked for the
        // upcoming ageout procedure
        psTarget->bDSRLLockedForAgeout = FALSE;
    }

    if (bLockRequested == TRUE)
    {
        // Only lock a DSRL if we're actually going
        // to be updating it (we leave DSRLs in
        // error/unknown states alone)
        if ((eCurrentState != DSRL_STATE_ERROR) &&
            (eCurrentState != DSRL_STATE_UNKNOWN))
        {
            psTarget->bDSRLLockedForAgeout = TRUE;
        }
    }
    else if (psTarget->bDSRLLockedForAgeout == TRUE)
    {
        // Bring the DSRL back to ready if its isn't in error
        if ((eCurrentState != DSRL_STATE_ERROR) &&
            (eCurrentState != DSRL_STATE_UNKNOWN))
        {
            DSRL_vSetState(psTarget->hDSRL, DSRL_STATE_READY);
        }

        psTarget->bDSRLLockedForAgeout = FALSE;
    }

    // Iterate all targets
    return TRUE;
}

/*****************************************************************************
*
*   n16CompareTextEntries
*
*****************************************************************************/
static N16 n16CompareTextEntries (
    FUEL_TEXT_ROW_STRUCT *psText1,
    FUEL_TEXT_ROW_STRUCT *psText2
        )
{
    if ((psText1 != NULL) && (psText2 != NULL))
    {
        // Fuel Types come first
        if (psText1->bBrandText != psText2->bBrandText)
        {
            if (psText1->bBrandText == TRUE)
            {
                return -1;
            }

            return 1;
        }

        // Then sort by ID
        if (psText1->un8TextId < psText2->un8TextId)
        {
            return -1;
        }

        if (psText1->un8TextId > psText2->un8TextId)
        {
            return 1;
        }

        return 0;
    }

    return N16_MIN;
}

/*****************************************************************************
*
*   n16CompareLogoEntries
*
*****************************************************************************/
static N16 n16CompareLogoEntries (
    FUEL_LOGO_ENTRY_STRUCT *psLogo1,
    FUEL_LOGO_ENTRY_STRUCT *psLogo2
        )
{
    if ((psLogo1 != NULL) && (psLogo2 != NULL))
    {
        // Logo IDs come first
        if (psLogo1->sLogoRow.un16LogoId < psLogo2->sLogoRow.un16LogoId)
        {
            return -1;
        }

        if (psLogo1->sLogoRow.un16LogoId > psLogo2->sLogoRow.un16LogoId)
        {
            return 1;
        }

        // Now Logo Types
        if (psLogo1->sLogoRow.eLogoType < psLogo2->sLogoRow.eLogoType)
        {
            return -1;
        }

        if (psLogo1->sLogoRow.eLogoType > psLogo2->sLogoRow.eLogoType)
        {
            return 1;
        }

        // Match
        return 0;
    }

    return N16_MIN;
}

/*****************************************************************************
*
*   bIterateTextEntries
*
*****************************************************************************/
static BOOLEAN bIterateTextEntries (
    FUEL_TEXT_ROW_STRUCT *psText,
    void *pvArg
        )
{
    FUEL_TEXT_ITERATOR_STRUCT *psIterator =
        (FUEL_TEXT_ITERATOR_STRUCT *)pvArg;
    BOOLEAN bContinue = FALSE;

    if ((psText == NULL) || (psIterator == NULL))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": bIterateTextEntries() NULL in list or argument\n");
        return FALSE;
    }

    if (psText->bBrandText == FALSE)
    {
        FUEL_TYPE_ENUM eFuelType;

        // Get the enumerated fuel type (if there is one)
        eFuelType = psIterator->psDBInterface->eMatchFuelType(psText->un8TextId);

        // Call the provided callback
        bContinue = psIterator->bIterator(
            psText->hText, eFuelType, psIterator->pvIteratorArg);

        // Do we have a long text to report as well?
        if ((bContinue == TRUE) && (psText->hLongText != STRING_INVALID_OBJECT))
        {
            // Yes - report that as well
            bContinue = psIterator->bIterator(
                psText->hLongText, eFuelType, psIterator->pvIteratorArg);
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   vReleaseTextEntry
*
*****************************************************************************/
static void vReleaseTextEntry (
    FUEL_TEXT_ROW_STRUCT *psText
        )
{
    if (psText != NULL)
    {
        if (psText->hText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psText->hText);
            psText->hText = STRING_INVALID_OBJECT;
        }

        if (psText->hLongText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psText->hLongText);
            psText->hLongText = STRING_INVALID_OBJECT;
        }

        SMSO_vDestroy((SMS_OBJECT)psText);
    }

    return;
}

/*****************************************************************************
*
*   vReleaseLogoEntry
*
*****************************************************************************/
static void vReleaseLogoEntry (
    FUEL_LOGO_ENTRY_STRUCT *psLogo
        )
{
    if (psLogo != NULL)
    {
        if (psLogo->hLogo != IMAGE_INVALID_OBJECT)
        {
            IMAGE_vDestroy(psLogo->hLogo);
            psLogo->hLogo = IMAGE_INVALID_OBJECT;
        }

        psLogo->sLogoRow.eLogoType = FUEL_BRAND_LOGO_INVALID_TYPE;

        SMSO_vDestroy((SMS_OBJECT)psLogo);
    }
    return;
}

/*****************************************************************************
*
*   n16CompareRegionsOfInterestById
*
*****************************************************************************/
static N16 n16CompareRegionsOfInterestById (
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry1,
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry2
        )
{
    N16 n16Result = N16_MIN;

    if ((psRegionEntry1 != NULL) &&
        (psRegionEntry2 != NULL))
    {
        n16Result = 
            COMPARE(psRegionEntry1->tRegionID, psRegionEntry2->tRegionID);
    }

    return n16Result;
}

/*****************************************************************************
*
*   bRemoveOldRegionsOfInterest
*
*****************************************************************************/
static BOOLEAN bRemoveOldRegionsOfInterest (
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry,
    BOOLEAN *pbRemoveAll
        )
{
    // If remove all then just do that
    if (*pbRemoveAll == TRUE)
    {
        vDestroyRegionOfInterest(psRegionEntry);
    }
    else
    {
        // Just update all regions so they are
        // just marked as background regions
        psRegionEntry->bBackgroundOnly = TRUE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   vDestroyRegionOfInterest
*
*****************************************************************************/
static void vDestroyRegionOfInterest (
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Remove this entry
    eReturnCode = OSAL.eLinkedListRemove(
        psRegionEntry->hEntry);
    if (eReturnCode == OSAL_SUCCESS)
    {
        if (psRegionEntry->psRegion != NULL)
        {
            if (psRegionEntry->psRegion->tTotalUsageCount > 0)
            {
                psRegionEntry->psRegion->tTotalUsageCount--;
            }

            if (psRegionEntry->bBackgroundOnly == FALSE)
            {
                if (psRegionEntry->psRegion->tLocationUsageCount > 0)
                {
                    psRegionEntry->psRegion->tLocationUsageCount--;
                }
            }

            psRegionEntry->psRegion = NULL;
        }

        psRegionEntry->tRegionID = FUEL_INVALID_REGION;
        psRegionEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        SMSO_vDestroy((SMS_OBJECT)psRegionEntry);
    }

    return;
}

/*******************************************************************************
*
*   n16CompareRegions
*
*   This function compares two FUEL_REGION_DESCRIPTOR_STRUCT and orders
*   them by ID.
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16CompareRegions (
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion1,
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion2
        )
{
    if ((psRegion1 != NULL) && (psRegion2 != NULL))
    {
        if (psRegion1->tRegionId < psRegion2->tRegionId)
        {
            return -1;
        }

        if (psRegion1->tRegionId > psRegion2->tRegionId)
        {
            return 1;
        }

        return 0;
    }

    return N16_MIN;
}

/*****************************************************************************
*
*   vReleaseRegionEntry
*
*****************************************************************************/
static void vReleaseRegionEntry (
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion
        )
{
    if (psRegion != NULL)
    {
        // Clear the price hash list if it exists
        if (psRegion->hPriceUpdates != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Empty the list
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psRegion->hPriceUpdates,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy);

            if (eReturnCode == OSAL_SUCCESS)
            {
                // Destroy the list
                OSAL.eLinkedListDelete(psRegion->hPriceUpdates);
            }

            psRegion->hPriceUpdates = OSAL_INVALID_OBJECT_HDL;
        }

        SMSO_vDestroy((SMS_OBJECT)psRegion);
    }
    return;
}

/*****************************************************************************
*
*   n16CompareStationEntry
*
*   This function compares two FUEL_STATION_ENTRY_STRUCTs and orders
*   them by their handles while ensuring all
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*****************************************************************************/
static N16 n16CompareStationEntry (
    FUEL_STATION_ENTRY_STRUCT *psEntry1,
    FUEL_STATION_ENTRY_STRUCT *psEntry2
        )
{
    N16 n16Compare;

    if (psEntry1->hStation == FUEL_STATION_INVALID_OBJECT)
    {
        return -1;
    }

    if (psEntry2->hStation == FUEL_STATION_INVALID_OBJECT)
    {
        return 1;
    }

    // Have the stations compare themselves
    n16Compare = FUEL_STATION_n16Compare(
        psEntry1->hStation, psEntry2->hStation);

    // Did the station match?
    if (0 == n16Compare)
    {
        // Compare the targets now, but only if we have targets to compare
        if ((NULL != psEntry1->psTarget) && (NULL != psEntry2->psTarget))
        {
            // Compare the targets now
            if (psEntry1->psTarget == psEntry2->psTarget)
            {
                // Match
                return 0;
            }

            // This forces all new entries which utilize the 
            // same station to be added after the original entry
            return -1;
        }
    }

    return n16Compare;
}

/*****************************************************************************
*
*   bUpdateDSRLForNewStation
*
*****************************************************************************/
static BOOLEAN bUpdateDSRLForNewStation (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_OBJECT hStation
        )
{
    // If this entry's station handle matches that of the new
    // station, and the station is not already in the DSRL,
    // see if we should try to add it
    if ((psEntry->hStation == hStation) &&
        (psEntry->bInDSRL == FALSE))
    {
        // Is this station usable based upon the application
        // provided location criteria?
        if (FALSE == psEntry->bBackgroundOnly)
        {
            DSRL_STATE_ENUM eCurrentState;
            BOOLEAN bStationUsable = FALSE;

            eCurrentState = DSRL.eState(psEntry->psTarget->hDSRL);
            if (eCurrentState != DSRL_STATE_ERROR)
            {
                // Add this entry to the DSRL for this target
                bStationUsable = bAddToDSRL(
                    psEntry->psTarget->hDSRL,
                    hStation );
            }

            if (bStationUsable == TRUE)
            {
                // Update this station's cache entry now that it is
                // in the DSRL
                psEntry->bInDSRL = TRUE;

                // Tell the DSRL that took
                // this that it has been updated
                DSRL_vSetState(
                    psEntry->psTarget->hDSRL, DSRL_STATE_UPDATING);
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateDSRLForStationChange
*
*****************************************************************************/
static BOOLEAN bUpdateDSRLForStationChange (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_OBJECT hStation
        )
{
    // Only operate on entries which are
    // currently in the DSRL
    if (psEntry->bInDSRL == TRUE)
    {
        DSRL_STATE_ENUM eState;
        DSRL_OBJECT hDSRL = psEntry->psTarget->hDSRL;
        DSRL_ADD_REPLACE_RESULT_ENUM eReplaced;

        // Get this DSRL's state
        eState = DSRL.eState(hDSRL);
        if (eState == DSRL_STATE_ERROR)
        {
            // Don't mess with this DSRL
            // it is forever in error
            return TRUE;
        }

        // This DSRL contains the fuel station,
        // so indicate that it is updating
        DSRL_vSetState(hDSRL, DSRL_STATE_UPDATING);

        // Replace the entry in the DSRL
        eReplaced = DSRL_eReplaceEntry (
            hDSRL,
            (DSRL_ENTRY_OBJECT)hStation,
            (DSRL_ENTRY_OBJECT)hStation );
        if (eReplaced == DSRL_ADD_REPLACE_NOP)
        {
            printf(FUEL_MGR_OBJECT_NAME
                ": bUpdateDSRLForStationChange() DSRL.eReplace:NO_OP on %p",
                hStation);

            // Remove this entry from the DSRL
            DSRL_vRemoveEntry( hDSRL, (DSRL_ENTRY_OBJECT)hStation);
        }
        else if (eReplaced == DSRL_ADD_REPLACE_ERROR)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": bUpdateDSRLForStationChange() DSRL replace error!");
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateCacheForLocationUpdate
*
*****************************************************************************/
static BOOLEAN bIterateCacheForLocationUpdate (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_TARGET_LOCATION_UPDATE_STRUCT *psUpdate
        )
{
    FUEL_TARGET_DESC_STRUCT *psTarget;
    if ((psEntry == NULL) || (psUpdate == NULL))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME
            ": bIterateCacheForLocationUpdate() invalid arguments!");
        return FALSE;
    }

    // Extract pointer for ease of use
    psTarget = psUpdate->psTarget;

    // Ensure the station handle is valid and
    // the target descriptor matches
    if ((psEntry->hStation != FUEL_STATION_INVALID_OBJECT) &&
        (psEntry->psTarget == psTarget))
    {
        BOOLEAN bMeetsCriteria;
        OSAL_FIXED_OBJECT hLat, hLon;
        LOCATION_OBJECT hLocation;
        LOCID_OBJECT hLocID;
        LOC_ID tLocID;

        // Get this station's location object and lat / lon
        hLocation = FUEL_STATION.hLocation(psEntry->hStation);
        hLocID = LOCATION.hLocID(hLocation);
        tLocID = LOCID.tID(hLocID);
        hLat = LOCATION.hLat(hLocation);
        hLon = LOCATION.hLon(hLocation);

        // Determine if this station meets the
        // target's location criteria
        bMeetsCriteria = bStationMeetsTargetLocationCriteria(
            psTarget, FALSE, tLocID, hLat, hLon);

        // Does this meet the location criteria anymore?
        if (bMeetsCriteria == FALSE)
        {
            // This station meets only the background criteria
            psEntry->bBackgroundOnly = TRUE;

            // Is the target's DSRL affected?
            if (psEntry->bInDSRL == TRUE)
            {
                // Remove this entry from the DSRL (it no longer
                // makes the cut)
                DSRL_vRemoveEntry(
                    psTarget->hDSRL, (DSRL_ENTRY_OBJECT)psEntry->hStation);

                // This entry is no longer in a DSRL
                psEntry->bInDSRL = FALSE;

                psUpdate->bDSRLUpdated = TRUE;
            }
        }
        else // Passes the location filter
        {
            // Station is not background only
            psEntry->bBackgroundOnly = FALSE;

            // Is this station already in the DSRL?
            if (psEntry->bInDSRL == FALSE)
            {
                BOOLEAN bAdded;

                // Add this entry to the DSRL
                bAdded = bAddToDSRL(
                    psTarget->hDSRL, psEntry->hStation );
                if (bAdded == TRUE)
                {
                    psEntry->bInDSRL = TRUE;
                    psUpdate->bDSRLUpdated = TRUE;
                }
            }
        }
    }

    // Go through the entire cache
    return TRUE;
}

/*****************************************************************************
*
*   bIterateCacheForDSRLRefresh
*
*****************************************************************************/
static BOOLEAN bIterateCacheForDSRLRefresh (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_REFRESH_DSRL_STRUCT *psRefresh
        )
{
    // Ensure the station handle is valid and
    // the target descriptor matches
    if ((psEntry->hStation == FUEL_STATION_INVALID_OBJECT) ||
        (psEntry->psTarget != psRefresh->psTarget))
    {
        return TRUE;
    }

    // Skip accordingly
    if (psRefresh->bFlush == TRUE)
    {
        // If we are flushing, don't worry about
        // looking at entries which aren't in a DSRL
        // since we are only going to remove entries that
        // are in a DSRL
        if (psEntry->bInDSRL == FALSE)
        {
            return TRUE;
        }
    }
    else // (psRefresh->bFlush == FALSE)
    {
        // We are only adding entries at this time
        // So, if this entry is already in a DSRL
        // then just move on
        if (psEntry->bInDSRL == TRUE)
        {
            return TRUE;
        }
    }

    // Is this station useful only for the background
    // location?
    if (FALSE == psEntry->bBackgroundOnly)
    {
        // No, this is good for the target location.
        // We need to add this entry to the DSRL only
        // if we're not currently flushing entries
        if (psRefresh->bFlush == FALSE)
        {
            BOOLEAN bAdded;

            // Try to add this entry to the DSRL since we already know
            // it matches our location area
            bAdded = bAddToDSRL(
                psRefresh->psTarget->hDSRL, psEntry->hStation );

            if (bAdded == TRUE)
            {
                psEntry->bInDSRL = TRUE;
                psRefresh->bDSRLUpdated = TRUE;
            }
        }
    }
    else // (bMeetsCriteria == TRUE)
    {
        // We need to remove this entry only if we're
        // flushing entries
        if (psRefresh->bFlush == TRUE)
        {
            // Remove this entry from the DSRL (it no longer
            // makes the cut)
            DSRL_vRemoveEntry(
                psRefresh->psTarget->hDSRL,
                (DSRL_ENTRY_OBJECT)psEntry->hStation);

            // This entry is no longer in a DSRL
            psEntry->bInDSRL = FALSE;

            // We've updated the DSRL
            psRefresh->bDSRLUpdated = TRUE;
        }
    }

    // Go through the entire cache
    return TRUE;
}

/*****************************************************************************
*
*   bIterateCacheForAgeout
*
*****************************************************************************/
static BOOLEAN bIterateCacheForAgeout (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_PRICE_AGEOUT_STRUCT *psAgeout
        )
{
    // Update the status only when station handle changes
    if (psEntry->hStation != psAgeout->hLastStation)
    {
        psAgeout->hLastStation = psEntry->hStation;

        // Flush prices based on their age
        psAgeout->bLastStationUpdated =
            FUEL_STATION_bFlushPrices(
                psEntry->hStation,
                psAgeout->un32PriceExpireAge,
                psAgeout->pun32OldestPriceAfterFlush
                    );
    }

    if (psAgeout->bLastStationUpdated == FALSE)
    {
        // Nothing to do here
        return TRUE;
    }

    // We only need to operate on entries which
    // reside in their respective DSRLs now
    if ((psEntry->bInDSRL == TRUE) &&
        (psEntry->psTarget->bDSRLLockedForAgeout == TRUE))
    {
        // This DSRL is now updating if it isn't already
        DSRL_vSetState(psEntry->psTarget->hDSRL, DSRL_STATE_UPDATING);

        if (psAgeout->bLastStationUpdated == TRUE)
        {
            DSRL_ADD_REPLACE_RESULT_ENUM eReplaced;

            // Tell the DSRL this entry was updated
            eReplaced = DSRL_eReplaceEntry(
                psEntry->psTarget->hDSRL,
                (DSRL_ENTRY_OBJECT)psEntry->hStation,
                (DSRL_ENTRY_OBJECT)psEntry->hStation );
            if (eReplaced == DSRL_ADD_REPLACE_NOP)
            {
                printf(FUEL_MGR_OBJECT_NAME
                    ": bIterateCacheForAgeout() DSRL.eReplace:NO_OP on %p",
                    psEntry->hStation);

                // Remove this entry 'cause the DSRL don't like it anymore
                DSRL_vRemoveEntry(psEntry->psTarget->hDSRL,
                    (DSRL_ENTRY_OBJECT)psEntry->hStation);
            }
            else if (eReplaced == DSRL_ADD_REPLACE_ERROR)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_MGR_OBJECT_NAME": bIterateCacheForAgeout() DSRL replace error!");
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateCacheForBackgroundChange
*
*****************************************************************************/
static BOOLEAN bIterateCacheForBackgroundChange(
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    BOOLEAN bMatched = FALSE;

    // Only match valid objects
    if (psEntry->hStation != FUEL_STATION_INVALID_OBJECT)
    {
        // If a target was given, just match that
        if (psEntry->psTarget != NULL)
        {
            bMatched = (psEntry->psTarget == psTarget);
        }
    }

    if (bMatched == TRUE)
    {
        BOOLEAN bMeetsCriteria;
        OSAL_FIXED_OBJECT hLat, hLon;
        LOCATION_OBJECT hLocation;
        LOCID_OBJECT hLocID;
        LOC_ID tLocID;

        // Get this station's location object and lat / lon
        hLocation = FUEL_STATION.hLocation(psEntry->hStation);
        hLocID = LOCATION.hLocID(hLocation);
        tLocID = LOCID.tID(hLocID);
        hLat = LOCATION.hLat(hLocation);
        hLon = LOCATION.hLon(hLocation);

        // Determine if this station still meets the
        // target's location criteria
        bMeetsCriteria = bStationMeetsTargetLocationCriteria(
            psTarget, TRUE, tLocID, hLat, hLon);

        // Does this meet the location criteria anymore?
        if (bMeetsCriteria == FALSE)
        {
            // Remove this entry from the DSRL (if present)
            if (psEntry->bInDSRL == TRUE)
            {
                DSRL_vRemoveEntry(
                    psTarget->hDSRL,
                    (DSRL_ENTRY_OBJECT)psEntry->hStation);

                psEntry->bInDSRL = FALSE;
            }

            // Clear this entry's target descriptor
            psEntry->psTarget = NULL;
        }
    }

    // Go through the entire cache
    return TRUE;
}

/*****************************************************************************
*
*   bIterateCacheForDestroy
*
*****************************************************************************/
static BOOLEAN bIterateCacheForDestroy (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_OBJECT *phLastStation
        )
{
    // We only want to destroy a fuel station once,
    // so keep track of the last station encountered
    // and if the current station is different then
    // we're safe to destroy the last one
    if (psEntry->hStation != *phLastStation)
    {
        // Only destroy valid objects
        if (*phLastStation != FUEL_STATION_INVALID_OBJECT)
        {
            FUEL_STATION_vDestroy(*phLastStation);
        }

        // Track the last station encountered
        *phLastStation = psEntry->hStation;
    }

    // Clear this entry's attributes
    psEntry->hStation = FUEL_STATION_INVALID_OBJECT;
    psEntry->bInDSRL = FALSE;

    return TRUE;
}

/*****************************************************************************
*
*   bIterateCacheForLogoUpdate
*
*****************************************************************************/
static BOOLEAN bIterateCacheForLogoUpdate (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    UN16 *pun16LogoId
        )
{
    // Only check stations in DSRLs
    if (psEntry->bInDSRL == TRUE)
    {
        N16 n16StationLogo;

        // Get the logo associated with this station
        n16StationLogo = FUEL_STATION_n16LogoId(psEntry->hStation);

        // Is this the logo that has been updated?
        if (n16StationLogo == (N16)(*pun16LogoId))
        {
            DSRL_ADD_REPLACE_RESULT_ENUM eReplaced;

            // Make sure this DSRL is in the UPDATING state now
            DSRL_vSetState(
                psEntry->psTarget->hDSRL, DSRL_STATE_UPDATING);

            // Replace the entry in the DSRL to indicate change
            eReplaced = DSRL_eReplaceEntry (
                psEntry->psTarget->hDSRL,
                (DSRL_ENTRY_OBJECT)psEntry->hStation,
                (DSRL_ENTRY_OBJECT)psEntry->hStation );
            if (eReplaced == DSRL_ADD_REPLACE_NOP)
            {
                printf(FUEL_MGR_OBJECT_NAME
                    ": bIterateCacheForLogoUpdate() DSRL.eReplace:NO_OP on %p",
                    psEntry->hStation);

                // Remove this entry from the DSRL
                DSRL_vRemoveEntry( psEntry->psTarget->hDSRL,
                    (DSRL_ENTRY_OBJECT)psEntry->hStation);
            }
            else if (eReplaced == DSRL_ADD_REPLACE_ERROR)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    FUEL_MGR_OBJECT_NAME": bIterateCacheForLogoUpdate() DSRL replace error!");
            }
        }
    }

    // Iterate the whole cache
    return TRUE;
}

/*****************************************************************************
*
*   vReleaseStations
*
*****************************************************************************/
static void vReleaseStations(
    FUEL_STATION_ENTRY_STRUCT *psEntry
        )
{
    psEntry->bInDSRL = FALSE;
    psEntry->hStation = FUEL_STATION_INVALID_OBJECT;
    psEntry->psTarget = NULL;
    SMSO_vDestroy((SMS_OBJECT)psEntry);

    return;
}

/*****************************************************************************
*
*   bEntryExists
*
*   From the starting point, attempt to find an entry with a
*   matching station & target
*
*****************************************************************************/
static BOOLEAN bEntryExists (
    OSAL_LINKED_LIST_ENTRY hStartingEntry,
    FUEL_STATION_ENTRY_STRUCT *psEntryToVerify
        )
{
    FUEL_STATION_ENTRY_STRUCT *psStartEntry;
    FUEL_STATION_ENTRY_STRUCT *psCurEntry;
    OSAL_LINKED_LIST_ENTRY hCurEntry;
    BOOLEAN bFound = FALSE;

    psStartEntry = (FUEL_STATION_ENTRY_STRUCT *)
        OSAL.pvLinkedListThis(hStartingEntry);

    if (psStartEntry->psTarget == psEntryToVerify->psTarget)
    {
        // Yep, this is the entry they're looking for
        return TRUE;
    }

    // Search forwards from the starting point for the entry in question

    // Get the next entry
    psCurEntry = NULL;
    hCurEntry = OSAL.hLinkedListNext(
            hStartingEntry, (void*)((void *)&psCurEntry));

    if (hCurEntry == hStartingEntry)
    {
        // List only has one entry, so we already know
        // we won't find what we're looking for
        return FALSE;
    }

    // While that is valid keep looking forward
    while ((hCurEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
           (psCurEntry != NULL))
    {
        if (psCurEntry->hStation != psEntryToVerify->hStation)
        {
            // We're past the entries for this station
            // so we don't need to look any further
            break;
        }

        // Did we match the targets?
        if (psCurEntry->psTarget == psEntryToVerify->psTarget)
        {
            bFound = TRUE;
            break;
        }

        // Move along the list
        psCurEntry = NULL;
        hCurEntry = OSAL.hLinkedListNext(
                hCurEntry, (void*)((void *)&psCurEntry));
    }

    if (bFound == TRUE)
    {
        return TRUE;
    }

    // Search backwards from the starting point for the entry in question

    // Get the prev entry
    psCurEntry = NULL;
    hCurEntry = OSAL.hLinkedListPrev(
            hStartingEntry, (void*)((void *)&psCurEntry));

    // While that is valid keep looking forward
    while ((hCurEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
           (psCurEntry != NULL))
    {
        if (psCurEntry->hStation != psEntryToVerify->hStation)
        {
            // We're past the entries for this station
            // so we don't need to look any further
            break;
        }

        if (psCurEntry->psTarget == psEntryToVerify->psTarget)
        {
            bFound = TRUE;
            break;
        }

        // Move along the list
        psCurEntry = NULL;
        hCurEntry = OSAL.hLinkedListPrev(
                hCurEntry, (void*)((void *)&psCurEntry));
    }

    return bFound;
}

/*****************************************************************************
*
*   psCacheEntryFromDSRLEntry
*
*   Try to find a station/target pair starting with this entry
*
*****************************************************************************/
static FUEL_STATION_ENTRY_STRUCT *psCacheEntryFromDSRLEntry (
    OSAL_LINKED_LIST_ENTRY hStartingEntry,
    FUEL_STATION_OBJECT hStation,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    FUEL_STATION_ENTRY_STRUCT *psCurEntry;
    OSAL_LINKED_LIST_ENTRY hCurEntry = hStartingEntry;
    BOOLEAN bFound = FALSE;

    // Try to find this target starting at this entry
    psCurEntry = (FUEL_STATION_ENTRY_STRUCT *)
        OSAL.pvLinkedListThis(hStartingEntry);

    // Do we have a good starting point?
    if ((OSAL_INVALID_LINKED_LIST_ENTRY == hCurEntry)  ||
        ((FUEL_STATION_ENTRY_STRUCT *)NULL == psCurEntry))
    {
        return (FUEL_STATION_ENTRY_STRUCT *)NULL;
    }

    // Loop through the entries until we find what we're
    // looking for or until we detect we won't find it
    do
    {
        // An entry is a match if both the station & target
        // agree.  If the station doesn't agree it means that
        // we've gone past the portion of the cache which
        // stores data for this station
        // If the target doesn't agree then we just move on
        // to the next entry

        // Does the station match?
        if (psCurEntry->hStation != hStation)
        {
            // Nope and we're not going to find it!
            break;
        }

        // Did we match the targets?
        if (psCurEntry->psTarget == psTarget)
        {
            // Yep we can stop now
            bFound = TRUE;
            break;
        }

        // Move to the next entry
        hCurEntry = OSAL.hLinkedListNext(
            hCurEntry, (void*)((void *)&psCurEntry));

        // Did we wrap or get invalid data?
        if ((hCurEntry == hStartingEntry) ||
            (OSAL_INVALID_LINKED_LIST_ENTRY == hCurEntry) ||
            ((FUEL_STATION_ENTRY_STRUCT *)NULL == psCurEntry))
        {
            break;
        }

    } while (TRUE);
 
    // Did we find what we were looking for?
    if (TRUE == bFound)
    {
        // Yes, provide it to the caller
        return psCurEntry;
    }
    
    // Return NULL since we didn't find anything
    return (FUEL_STATION_ENTRY_STRUCT *)NULL;
}

/*****************************************************************************
*
*   n16SortDSRLByPrice
*
*****************************************************************************/
static N16 n16SortDSRLByPrice (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    FUEL_STATION_OBJECT hStation;
    UN8 un8PriceIndex, un8NumPrices = 0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    UN32 un32Station1Lowest = UN32_MAX,
         un32Station2Lowest = UN32_MAX;
    FUEL_INFO_STRUCT sFuelInfo;

    // Clear out the fuel info struct
    OSAL.bMemSet( &sFuelInfo, (UN8)0, sizeof(sFuelInfo) );

    // Extract the first station
    hStation = (FUEL_STATION_OBJECT)DSRL_ENTRY.hFuelStation(hEntry1);

    // Only query the price list if this is a valid object
    if (FUEL_STATION_INVALID_OBJECT != hStation)
    {
        // Determine the lowest price in the station
        un8NumPrices = FUEL_STATION.un8NumAvailableFuelTypes(hStation);
    }

    for (un8PriceIndex = 0; un8PriceIndex < un8NumPrices; un8PriceIndex++)
    {
        // Grab this entry and extract its price
        eReturnCode = FUEL_STATION.eFuelInfo(
                hStation, un8PriceIndex, &sFuelInfo );
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": Unable extract prices for sort");

            // Push this to the end of the list
            un32Station1Lowest = UN32_MAX;

            break;
        }

        if ( ( sFuelInfo.eAvailability == FUEL_AVAILABLE ) &&
             ( sFuelInfo.un32PriceInTenthsOfLowestDenomination
                     < un32Station1Lowest ) )
        {
            un32Station1Lowest
                = sFuelInfo.un32PriceInTenthsOfLowestDenomination;
        }
    }

    // Extract the second station
    hStation = (FUEL_STATION_OBJECT)DSRL_ENTRY.hFuelStation(hEntry2);

    // Reset price count
    un8NumPrices = 0;

    // Only query the price list if this is a valid object
    if (FUEL_STATION_INVALID_OBJECT != hStation)
    {
        // Determine the lowest price in the station
        un8NumPrices = FUEL_STATION.un8NumAvailableFuelTypes(hStation);
    }

    for (un8PriceIndex = 0; un8PriceIndex < un8NumPrices; un8PriceIndex++)
    {
        // Grab this entry and extract its price
        eReturnCode = FUEL_STATION.eFuelInfo(
                hStation, un8PriceIndex, &sFuelInfo);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": Unable extract prices for sort");

            // Push this to the end of the list
            un32Station2Lowest = UN32_MAX;

            break;
        }

        if ( ( sFuelInfo.eAvailability  == FUEL_AVAILABLE ) &&
             ( sFuelInfo.un32PriceInTenthsOfLowestDenomination
                     < un32Station2Lowest ) )
        {
            un32Station2Lowest
                = sFuelInfo.un32PriceInTenthsOfLowestDenomination;
        }
    }

    // Now, sort these two
    if (un32Station1Lowest < un32Station2Lowest)
    {
        return -1;
    }

    if (un32Station1Lowest > un32Station2Lowest)
    {
        return 1;
    }

    return 0;
}

/*****************************************************************************
*
*   vDSRLFinalizeStationDelete
*
*   This is a callback invoked by the DSRL when it wants to remove and delete
*   an entry (in our case its a FUEL_STATION_OBJECT handle)
*
*****************************************************************************/
static void vDSRLFinalizeStationDelete (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    BOOLEAN bOwner;
    FUEL_STATION_OBJECT hFuelStation;
    FUEL_MGR_OBJECT_STRUCT *psObj = psTarget->psObj;
    FUEL_STATION_ENTRY_STRUCT *psEntry = 
        (FUEL_STATION_ENTRY_STRUCT *)NULL;
    FUEL_STATION_DSRL_ENTRY_STRUCT *psDSRLEntry;

    puts(FUEL_MGR_OBJECT_NAME": DSRL finalizer invoked");

    // Check object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)psObj->sWorkCtrl.psStationOwner);
    if (bOwner == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": vDSRLFinalizeStationDelete() ownership check failed");
        return;
    }

    // Extract the fuel station from this entry
    hFuelStation = DSRL_ENTRY.hFuelStation(hEntry);

    // Are we serializing now?
    if (psTarget->bSerializeDSRL == TRUE)
    {
        BOOLEAN bSerialized;

        // Attempt to serialize this price entry
        bSerialized = psObj->psDBInterface->bUpdatePriceEntry(
            psObj->hDBInterface, hFuelStation);
        if (bSerialized == TRUE)
        {
            puts(FUEL_MGR_OBJECT_NAME": Station serialized");
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                FUEL_MGR_OBJECT_NAME": vDSRLFinalizeStationDelete() unable to serialize station");
        }
    }

    // Clear the DSRL flag to indicate
    // this entry needs to be pruned

    // Grab the first entry in the cache which stores this station
    psDSRLEntry = (FUEL_STATION_DSRL_ENTRY_STRUCT *)
        DSRL_ENTRY_pvServiceData(hEntry);

    if (NULL != psDSRLEntry)
    {
        // Find this exact entry in the cache
        psEntry = psCacheEntryFromDSRLEntry(
            psDSRLEntry->hFirstStationEntry, hFuelStation, psTarget);

        if (psEntry != NULL)
        {
            printf(FUEL_MGR_OBJECT_NAME": Station %p no longer in DSRL\n",
                psEntry->hStation);

            // Update entry to state that it is no longer in the cache
            psEntry->bInDSRL = FALSE;
        }
    }

    return;
}

/*****************************************************************************
*
*   vStartAgeoutTimer
*
*   This function starts the ageout timer based on the oldest
*   data we have in a station.
*
*****************************************************************************/
static void vStartAgeoutTimer (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32NextExpireTime;

    // Don't start the timer if we have no
    // data to time-out
    if ((psObj->sAgeoutCtrl.un32OldestProcessedPriceData == UN32_MAX) &&
       (psObj->sAgeoutCtrl.un32OldestPriceData == UN32_MAX))
    {
        return;
    }

    // Is the newly processed price data older
    // than our current price data?
    if (psObj->sAgeoutCtrl.un32OldestProcessedPriceData <
        psObj->sAgeoutCtrl.un32OldestPriceData)
    {
        // Update the oldest price data time
        psObj->sAgeoutCtrl.un32OldestPriceData =
            psObj->sAgeoutCtrl.un32OldestProcessedPriceData;

        // Stop the timer if it was previously running
        vStopAgeoutTimer(psObj);
    }

    // Get the current time (UTC)
    eReturnCode = OSAL.eTimeGet(&un32NextExpireTime);
    if (eReturnCode != OSAL_SUCCESS)
    {

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": Can't get current time!");
        printf("Result: %s\n",
               OSAL.pacGetReturnCodeName(eReturnCode));
        return;
    }

    // Now, compute when the timer should expire next.
    // The timer should expire when the oldest data reaches
    // the expire age.

    // How old is the oldest data? (subtracting these gives us
    // the age of the oldest data in seconds -- un32OldestPriceData
    // is a time stamp
    un32NextExpireTime -= psObj->sAgeoutCtrl.un32OldestPriceData;

    // If it is old enough to meet our lifetime limit,
    // we need to handle it now
    if (un32NextExpireTime > FUEL_AGE_MAX_IN_SECS)
    {
        // Fire timer now -- this shouldn't happen!
        un32NextExpireTime = 0;

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            FUEL_MGR_OBJECT_NAME": We are using data that is too old!");
    }
    else // Otherwise, we have some time until it expires
    {
        // Prices are aged out once they reach FUEL_AGE_MAX_IN_SECS,
        // so calculate how much time there is before the oldest
        // price data reaches that age
        un32NextExpireTime = FUEL_AGE_MAX_IN_SECS - un32NextExpireTime;

        printf(FUEL_MGR_OBJECT_NAME": Next Ageout timer in %u secs\n",
            un32NextExpireTime);
    }

    // Kick off the event
    bOk = DATASERVICE_IMPL_bSetTimedEvent(
        psObj->sAgeoutCtrl.hDataExpireEvent, FALSE, FALSE, un32NextExpireTime);

    if (bOk == FALSE)
    {
        puts(FUEL_MGR_OBJECT_NAME": Unable to start ageout event");
    }

    return;
}

/*****************************************************************************
*
*   vStopAgeoutTimer
*
*****************************************************************************/
static void vStopAgeoutTimer (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    // Stop the expire event from occurring
    DATASERVICE_IMPL_bStopTimedEvent(psObj->sAgeoutCtrl.hDataExpireEvent);

    return;
}

/*****************************************************************************
*
*   vStartProcessTimer
*
*****************************************************************************/
static void vStartProcessTimer (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    // Kick off the event, override any event already set
    bOk = DATASERVICE_IMPL_bSetTimedEvent(
        psObj->sWorkCtrl.hProcessEvent,
        TRUE, FALSE, FUEL_PROCESS_TARGET_TIMEOUT_SECONDS);

    if (bOk != TRUE)
    {
        puts(FUEL_MGR_OBJECT_NAME": Unable to start process timer");
    }

    return;
}

/*****************************************************************************
*
*   vStopProcessTimer
*
*****************************************************************************/
static void vStopProcessTimer (
    FUEL_MGR_OBJECT_STRUCT *psObj
        )
{
    puts(FUEL_MGR_OBJECT_NAME": Stopping process timer");

    // Stop the process event from occurring
    DATASERVICE_IMPL_bStopTimedEvent(psObj->sWorkCtrl.hProcessEvent);
    return;
}

/*****************************************************************************
*
*   bLoadStationsForDSRL
*
*   Pull background stations from the DB into memory for a provided target.
*
*****************************************************************************/
static BOOLEAN bLoadStationsForDSRL (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation
        )
{
    size_t tNumObjectsCreated = 0;
    BOOLEAN bLocked;

    // Lock the DSRL so we can update it
    bLocked = SMSO_bLock(
        (SMS_OBJECT)psTarget->hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return FALSE;
    }

    // This DSRL is updating now
    DSRL_vSetState(psTarget->hDSRL, DSRL_STATE_UPDATING);

    // Is this region in the DB?
    if (TRUE == psTarget->bCurrentRegionInDB)
    {
        // Yes, try to pull some data out of the DB now

        // This is our current target
        psObj->sWorkCtrl.psCurrentTarget = psTarget;
        psObj->sWorkCtrl.bLoadingPrices = FALSE;

        // Load the stations we need
        tNumObjectsCreated = psObj->psDBInterface->tLoadStations(
            psObj->hDBInterface,
            psLocation->hBackground,
            psTarget->tCurrentRegion,
            psTarget->tCurrentStation);

        // Don't need the current anymore
        psObj->sWorkCtrl.psCurrentTarget = NULL;
    }

    // All done with that
    SMSO_vUnlock((SMS_OBJECT)psTarget->hDSRL);

    // If all has gone well and we've created objects
    // based on data in the DB, then we need to update
    // the message filter for the affected region
    if (tNumObjectsCreated > 0)
    {
        FUEL_REGION_DESCRIPTOR_STRUCT *psRegion;
        psRegion = psGetRegion(psObj, psTarget->tCurrentRegion);
        if (psRegion != NULL)
        {
            // Empty the filter list for this region
            OSAL.eLinkedListRemoveAll(
                psRegion->hPriceUpdates,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bLoadPricesForFavoriteStationID
*
*   Pull prices from the DB into memory for a provided target
*
*****************************************************************************/
static BOOLEAN bLoadPricesForFavoriteStationID (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    FUEL_REGION tRegion,
    FUEL_STATION_ID tStation
        )
{
    BOOLEAN bOk;

    // Reset the oldest processed data value
    psObj->sAgeoutCtrl.un32OldestProcessedPriceData = UN32_MAX;

    // Prepare for DB processing
    psObj->sWorkCtrl.psCurrentTarget = psTarget;
    psObj->sWorkCtrl.bLoadingPrices = TRUE;

    // Get the DB interface to load all prices needed
    bOk = psObj->psDBInterface->bLoadPrices(
        psObj->hDBInterface, tRegion, tStation);

    psObj->sWorkCtrl.psCurrentTarget = NULL;

    if (bOk == TRUE)
    {
        // Start the ageout timer based on
        // the age data we just collected
        vStartAgeoutTimer(psObj);

        // Now load this station into the DSRL
        bOk = bLoadStationIntoDSRLFromCache(
                psObj, psTarget, tRegion, tStation);
    }

    return bOk;
}

/*****************************************************************************
*
*   bLoadPricesForDSRL
*
*   Pull prices from the DB into memory for a provided target
*
*****************************************************************************/
static BOOLEAN bLoadPricesForDSRL (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        )
{
    BOOLEAN bOk;

    // Reset the oldest processed data value
    psObj->sAgeoutCtrl.un32OldestProcessedPriceData = UN32_MAX;

    // Prepare for DB processing
    psObj->sWorkCtrl.psCurrentTarget = psTarget;
    psObj->sWorkCtrl.bLoadingPrices = TRUE;

    // Get the DB interface to load all prices needed
    bOk = psObj->psDBInterface->bLoadPrices(
        psObj->hDBInterface, FUEL_INVALID_REGION,
        FUEL_INVALID_STATION_ID);

    psObj->sWorkCtrl.psCurrentTarget = NULL;

    if (bOk == TRUE)
    {
        // Start the ageout timer based on
        // the age data we just collected
        vStartAgeoutTimer(psObj);
    }

    return bOk;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the world about it
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        FUEL_MGR_OBJECT_NAME": vSetError()");

    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}
