/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Fuel Price Interface header file.
 *
 ******************************************************************************/

#ifndef FUEL_INTERFACE_H_
#define FUEL_INTERFACE_H_

#include "osal.h"
#include "sms_api.h"
#include "dataservice_mgr_impl.h"
#include "fuel_db_constants.h"
#include "sql_interface_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Interface handles
typedef void * FUEL_OTA_INTERFACE_OBJECT;
#define FUEL_OTA_INTERFACE_INVALID_OBJECT (FUEL_OTA_INTERFACE_OBJECT)NULL

typedef void * FUEL_DB_INTERFACE_OBJECT;
#define FUEL_DB_INTERFACE_INVALID_OBJECT (FUEL_DB_INTERFACE_OBJECT)NULL

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// forward-declare typedefs so we can use them as pointers
// within the struct itself.
typedef struct fuel_db_interface_struct FUEL_DB_INTERFACE_STRUCT;
typedef struct fuel_ota_interface_struct FUEL_OTA_INTERFACE_STRUCT;

// The OTA fuel interface APIs
// typedef is forward-declared above
struct fuel_ota_interface_struct
{
    const DSI tDSI;

    const DATASERVICE_ID tDataID;

    const size_t tMaxVersionBitlen;

    const DATASERVICE_PRODUCT_DESCRIPTOR_STRUCT *psProducts;

    const UN8 un8NumProducts;

    const UN8 un8RFDClientId;

    FUEL_OTA_INTERFACE_OBJECT (*hInit) (
        FUEL_SERVICE_OBJECT hFuelService,
        SMS_OBJECT hParent,
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
        const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface
            );

    void (*vUnInit) (
        FUEL_OTA_INTERFACE_OBJECT hInterface
            );

    BOOLEAN (*bProductStateChange) (
        FUEL_OTA_INTERFACE_OBJECT hInterface,
        DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
            );

    BOOLEAN (*bGetDSIForProduct) (
        DATA_PRODUCT_TYPE_ENUM eType,
        DATA_PRODUCT_MASK tMask,
        DATASERVICE_DSI_INFO_STRUCT * psDSIInfo
            );

    DATA_PRODUCT_STATE_ENUM (*eNextProductState) (
        DATASERVICE_IMPL_HDL hServiceImpl,
        DATA_PRODUCT_TYPE_ENUM eType,
        DSI tDSI,
        DATASERVICE_STATE_ENUM eDSIState,
        SXM_DMI *pDMIs,
        UN8 un8DMICount
            );

    BOOLEAN (*bProcessMessage) (
        FUEL_OTA_INTERFACE_OBJECT hInterface,
        OSAL_BUFFER_HDL *phPayload
            );

    BOOLEAN (*bAddRegionsForLocation) (
        FUEL_OTA_INTERFACE_OBJECT hInterface,
        LOCATION_OBJECT hCenterLocation,
        void *pvAddArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eProcessAmenities) (
        size_t tNumAmenities,
        AMENITY_STRUCT *pasFuelAmenities,
        UN32 un32RawAmenitiesData
            );

};

typedef enum fuel_station_update_type_enum
{
    FUEL_STATION_UPDATE_TYPE_NEW,
    FUEL_STATION_UPDATE_TYPE_MODIFY_GIVEN_ATTRIBS,
    FUEL_STATION_UPDATE_TYPE_DELETE,
    FUEL_STATION_UPDATE_MAX_TYPES

} FUEL_STATION_UPDATE_TYPE_ENUM;

typedef struct fuel_position_update_struct
{
    UN8 un8FuelType;
    FUEL_TYPE_ENUM eFuelType;
    STRING_OBJECT hShortText;
    STRING_OBJECT hLongText;
    FUEL_INFO_DYNAMIC_POSITION_STRUCT sPosition;

} FUEL_POSITION_UPDATE_STRUCT;

// The fuel manager's interface APIs
typedef struct fuel_mgr_interface_struct
{
    BOOLEAN (*bRegionUpdateBegin) (
       FUEL_SERVICE_OBJECT hFuelService,
       UN32 un32MsgId,
       FUEL_REGION tRegion,
       UN8 un8TextVersion
           );

    BOOLEAN (*bPriceUpdate) (
       FUEL_SERVICE_OBJECT hFuelService,
       FUEL_STATION_ID tStationId,
       FUEL_PRICE_ROW_STRUCT *psPriceRow
           );

    BOOLEAN (*bPositionUpdate) (
       FUEL_SERVICE_OBJECT hFuelService,
       UN16 un16ReportTime,
       FUEL_STATION_ID tStationId,
       FUEL_POSITION_UPDATE_STRUCT *pasPositions,
       size_t tNumPositions
            ); 

    void (*vRegionUpdateComplete) (
       FUEL_SERVICE_OBJECT hFuelService
           );

    BOOLEAN (*bTextUpdateBegin) (
        FUEL_SERVICE_OBJECT hFuelService,
        UN8 un8NewTextVersion
            );

    BOOLEAN (*bTextUpdate) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_TEXT_ROW_STRUCT *psTextRow
            );

    void (*vTextUpdateEnd) (
        FUEL_SERVICE_OBJECT hFuelService,
        UN8 un8NewTextVersion
            );

    BOOLEAN (*bLogoUpdate) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_LOGO_ROW_STRUCT *psLogoRow,
        BOOLEAN bDeleteLogo,
        OSAL_BUFFER_HDL hLogoData
            );

    BOOLEAN (*bAddRegionOfInterest) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_REGION tRegion,
        void *pvArg
            );

    BOOLEAN (*bEnableLogoSupport) (
        FUEL_SERVICE_OBJECT hFuelService,
        const char *pacLogoPath,
        const char *pacBaselineLogoPath
            );

    BOOLEAN (*bReportTextFromDB) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_TEXT_ROW_STRUCT *psTextRow
            );

    BOOLEAN (*bReportLogoFromDB) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_LOGO_ROW_STRUCT *psLogoRow
            );

    BOOLEAN (*bReportRegionFromDB) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_REGION tID
            );

    BOOLEAN (*bBackgroundLocationFilter) (
        FUEL_SERVICE_OBJECT hFuelService,
        OSAL_FIXED_OBJECT hLat,
        OSAL_FIXED_OBJECT hLon,
        BOOLEAN *pbCacheOnly
            );

    BOOLEAN (*bVerifyQueryResultLocation) (
        FUEL_SERVICE_OBJECT hFuelService,
        LOC_ID tLocId,
        OSAL_FIXED_OBJECT hLat,
        OSAL_FIXED_OBJECT hLon
            );

    FUEL_STATION_OBJECT (*hGetStation) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_REGION tRegionID,
        FUEL_STATION_ID tStationID,
        BOOLEAN *pbStationNeeded
            );

    FUEL_STATION_OBJECT (*hCreateStation) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_STATION_ROW_STRUCT *psStationRow
            );

    BOOLEAN (*bAddFuelPricesToStation) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_STATION_OBJECT hStation,
        FUEL_PRICE_ROW_STRUCT *psPriceRow,
        UN8 un8TextVer
            );

    BOOLEAN (*bPutStationInService) (
        FUEL_SERVICE_OBJECT hFuelService,
        FUEL_STATION_OBJECT hStation,
        BOOLEAN bCacheOnly
            );

    BOOLEAN (*bIsRegionNeeded) (
       FUEL_SERVICE_OBJECT hFuelService,
       FUEL_REGION tRegion,
       UN8 un8TextVersion
            );

} FUEL_MGR_INTERFACE_STRUCT;

// The DB fuel interface APIs
//
// Note: bDBUpdateEnd takes a U16 version number to accommodate
// EV's larger (up to 999) version numbers.

// typedef is forward-declared near the top of this file - BC
struct fuel_db_interface_struct
{
    const char *pacFolder;

    const char *pacRefDBNameA;

    const char *pacRefDBNameB;

    const N16 n16DBVersion;

    const char *pacPersistDBName;

    FUEL_DB_INTERFACE_OBJECT (*hConnect) (
        FUEL_SERVICE_OBJECT hFuelService,
        SMS_OBJECT hParent,
        const char *pacRefDBDirPath,
        const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
        const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface,
        DATASERVICE_ERROR_CODE_ENUM *peErrorCode
            );

    void (*vDisconnect) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface
            );

    BOOLEAN (*bCanProcessUpdate) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        UN8 un8TextVersion
            );

    FUEL_TYPE_ENUM (*eMatchFuelType) (
        UN8 un8TextId
            );

    size_t (*tLoadStations) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        LOCATION_OBJECT hLocation,
        FUEL_REGION tRegionID,
        FUEL_STATION_ID tStationID
           );

    BOOLEAN (*bLoadPrices) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        FUEL_REGION tRegionID,
        FUEL_STATION_ID tStationID
            );

    BOOLEAN (*bTextQuery) (
       FUEL_DB_INTERFACE_OBJECT hDBInterface,
       UN8 un8NewTextVersion
           );

    BOOLEAN (*bUpdateTextTable) (
       FUEL_DB_INTERFACE_OBJECT hDBInterface,
       BOOLEAN bBegin,
       UN8 un8NewTextVersion
           );

    BOOLEAN (*bUpdateTextEntry) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        SQL_INTERFACE_OBJECT hSQLConnection,
        FUEL_TEXT_ROW_STRUCT *psTextRow
            );

    BOOLEAN (*bUpdateLogoEntry) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        BOOLEAN bDelete,
        FUEL_LOGO_ROW_STRUCT *psLogoRow
            );

    void (*vUpdatePriceTable) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        BOOLEAN bBegin
            );

    BOOLEAN (*bUpdatePriceEntry) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        FUEL_STATION_OBJECT hStation
            );

    UN16 (*un16DBContentVer) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface
            );

    BOOLEAN (*bRefDBBank) (
        FUEL_DB_INTERFACE_OBJECT hDBInterface,
        STRING_OBJECT *phInUseDB,
        STRING_OBJECT *phNextDB
            );

    BOOLEAN (*bDBUpdateBegin) (
       SQL_INTERFACE_OBJECT hSQLConnection,
       char *pcSQLCommandBuffer,
       size_t tBufferSize
            );

    BOOLEAN (*bDBUpdateEnd) (
       SQL_INTERFACE_OBJECT hSQLConnection,
       char *pcSQLCommandBuffer,
       size_t tBufferSize,
       UN16 un16NewDBVersion
           );

    BOOLEAN (*bRegionUpdateBegin) (
       SQL_INTERFACE_OBJECT hSQLConnection,
       char *pcSQLCommandBuffer,
       size_t tBufferSize,
       FUEL_REGION tRegion
           );

    BOOLEAN (*bRegionUpdateEnd) (
       SQL_INTERFACE_OBJECT hSQLConnection
           );

    BOOLEAN (*bStationUpdate) (
       SQL_INTERFACE_OBJECT hSQLConnection,
       char **ppcSQLCommandBuffer,
       size_t *ptBufferSize,
       FUEL_STATION_ROW_STRUCT *psStationRow,
       FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
       BOOLEAN bAmenitiesUpdated
           );

};

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

// Interface into the US Fuel Price OTA plugin
extern const FUEL_OTA_INTERFACE_STRUCT GsPriceOTAIntf;

// Interface into the Fuel Price DB plugin
extern const FUEL_DB_INTERFACE_STRUCT GsPriceDBIntf;

// Interface into the Canadian Fuel Price OTA plugin
extern const FUEL_OTA_INTERFACE_STRUCT GsCanadianPriceOTAIntf;

// Interface into the Fuel Price DB plugin
extern const FUEL_DB_INTERFACE_STRUCT GsCanadianPriceDBIntf;

// Interface into the EV Charging OTA plugin
extern const FUEL_OTA_INTERFACE_STRUCT GsEVChargingOTAIntf;

// Interface into the EV Charging DB plugin
extern const FUEL_DB_INTERFACE_STRUCT GsEVChargingDBIntf;

// Interface into the Fuel manager
extern const FUEL_MGR_INTERFACE_STRUCT GsFuelMgrIntf;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

#endif /* FUEL_INTERFACE_H_ */
